/**
 *
 * @file        apm32f411xx.h
 *
 * @brief       CMSIS APM32F411xx Device Peripheral Access Layer Header File.
 *
 * @version     V1.0.1
 *
 * @date        2024-08-01
 *
 * @attention
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * The original code has been modified by Geehy Semiconductor.
 *
 * Copyright (c) 2017 STMicroelectronics.
 * Copyright (C) 2023-2024 Geehy Semiconductor.
 * All rights reserved.
 *
 * This software is licensed under terms that can be found in the LICENSE file
 * in the root directory of this software component.
 * If no LICENSE file comes with this software, it is provided AS-IS.
 *
 */

/** @addtogroup CMSIS_Device
  * @{
  */

/** @addtogroup apm32f411xx
  * @{
  */

#ifndef __APM32F411xx_H
#define __APM32F411xx_H

#ifdef __cplusplus
 extern "C" {
#endif /* __cplusplus */

/** @addtogroup Configuration_section_for_CMSIS
  * @{
  */

/**
  * @brief Configuration of the Cortex-M4 Processor and Core Peripherals
  */
#define __CM4_REV                 0x0001U  /*!< Core revision r0p1                            */
#define __MPU_PRESENT             1U       /*!< APM32F4XX provides an MPU                     */
#define __NVIC_PRIO_BITS          4U       /*!< APM32F4XX uses 4 Bits for the Priority Levels */
#define __Vendor_SysTickConfig    0U       /*!< Set to 1 if different SysTick Config is used  */
#define __FPU_PRESENT             1U       /*!< FPU present                                   */

/**
  * @}
  */

/** @addtogroup Peripheral_interrupt_number_definition
  * @{
  */

/**
 * @brief APM32F4XX Interrupt Number Definition, according to the selected device
 *        in @ref Library_configuration_section
 */
typedef enum
{
/******  Cortex-M4 Processor Exceptions Numbers ****************************************************************/
  NonMaskableInt_IRQn         = -14,    /*!< 2 Non Maskable Interrupt                                          */
  MemoryManagement_IRQn       = -12,    /*!< 4 Cortex-M4 Memory Management Interrupt                           */
  BusFault_IRQn               = -11,    /*!< 5 Cortex-M4 Bus Fault Interrupt                                   */
  UsageFault_IRQn             = -10,    /*!< 6 Cortex-M4 Usage Fault Interrupt                                 */
  SVCall_IRQn                 = -5,     /*!< 11 Cortex-M4 SV Call Interrupt                                    */
  DebugMonitor_IRQn           = -4,     /*!< 12 Cortex-M4 Debug Monitor Interrupt                              */
  PendSV_IRQn                 = -2,     /*!< 14 Cortex-M4 Pend SV Interrupt                                    */
  SysTick_IRQn                = -1,     /*!< 15 Cortex-M4 System Tick Interrupt                                */
/******  APM32 specific Interrupt Numbers **********************************************************************/
  WWDT_IRQn                   = 0,      /*!< Window WatchDog Interrupt                                         */
  PVD_IRQn                    = 1,      /*!< PVD through EINT Line detection Interrupt                         */
  TAMP_STAMP_IRQn             = 2,      /*!< Tamper and TimeStamp interrupts through the EINT line             */
  RTC_WKUP_IRQn               = 3,      /*!< RTC Wakeup interrupt through the EINT line                        */
  FLASH_IRQn                  = 4,      /*!< FLASH global Interrupt                                            */
  RCM_IRQn                    = 5,      /*!< RCM global Interrupt                                              */
  EINT0_IRQn                  = 6,      /*!< EINT Line0 Interrupt                                              */
  EINT1_IRQn                  = 7,      /*!< EINT Line1 Interrupt                                              */
  EINT2_IRQn                  = 8,      /*!< EINT Line2 Interrupt                                              */
  EINT3_IRQn                  = 9,      /*!< EINT Line3 Interrupt                                              */
  EINT4_IRQn                  = 10,     /*!< EINT Line4 Interrupt                                              */
  DMA1_Stream0_IRQn           = 11,     /*!< DMA1 Stream 0 global Interrupt                                    */
  DMA1_Stream1_IRQn           = 12,     /*!< DMA1 Stream 1 global Interrupt                                    */
  DMA1_Stream2_IRQn           = 13,     /*!< DMA1 Stream 2 global Interrupt                                    */
  DMA1_Stream3_IRQn           = 14,     /*!< DMA1 Stream 3 global Interrupt                                    */
  DMA1_Stream4_IRQn           = 15,     /*!< DMA1 Stream 4 global Interrupt                                    */
  DMA1_Stream5_IRQn           = 16,     /*!< DMA1 Stream 5 global Interrupt                                    */
  DMA1_Stream6_IRQn           = 17,     /*!< DMA1 Stream 6 global Interrupt                                    */
  ADC_IRQn                    = 18,     /*!< ADC1 and ADC2 global Interrupts                                   */
  CAN1_TX_IRQn                = 19,     /*!< CAN1 TX Interrupt                                                 */
  CAN1_RX0_IRQn               = 20,     /*!< CAN1 RX0 Interrupt                                                */
  CAN1_RX1_IRQn               = 21,     /*!< CAN1 RX1 Interrupt                                                */
  CAN1_SCE_IRQn               = 22,     /*!< CAN1 SCE Interrupt                                                */
  EINT9_5_IRQn                = 23,     /*!< External Line[9:5] Interrupts                                     */
  TMR1_BRK_TMR9_IRQn          = 24,     /*!< TMR1 Break interrupt and TMR9 global interrupt                    */
  TMR1_UP_TMR10_IRQn          = 25,     /*!< TMR1 Update Interrupt and TMR10 global interrupt                  */
  TMR1_TRG_COM_TMR11_IRQn     = 26,     /*!< TMR1 Trigger and Commutation Interrupt and TMR11 global interrupt */
  TMR1_CC_IRQn                = 27,     /*!< TMR1 Capture Compare Interrupt                                    */
  TMR2_IRQn                   = 28,     /*!< TMR2 global Interrupt                                             */
  TMR3_IRQn                   = 29,     /*!< TMR3 global Interrupt                                             */
  TMR4_IRQn                   = 30,     /*!< TMR4 global Interrupt                                             */
  I2C1_EV_IRQn                = 31,     /*!< I2C1 Event Interrupt                                              */
  I2C1_ER_IRQn                = 32,     /*!< I2C1 Error Interrupt                                              */
  I2C2_EV_IRQn                = 33,     /*!< I2C2 Event Interrupt                                              */
  I2C2_ER_IRQn                = 34,     /*!< I2C2 Error Interrupt                                              */
  SPI1_IRQn                   = 35,     /*!< SPI1 global Interrupt                                             */
  SPI2_IRQn                   = 36,     /*!< SPI2 global Interrupt                                             */
  USART1_IRQn                 = 37,     /*!< USART1 global Interrupt                                           */
  USART2_IRQn                 = 38,     /*!< USART2 global Interrupt                                           */
  USART3_IRQn                 = 39,     /*!< USART3 global Interrupt                                           */
  EINT15_10_IRQn              = 40,     /*!< External Line[15:10] Interrupts                                   */
  RTC_Alarm_IRQn              = 41,     /*!< RTC Alarm (A and B) through EINT Line Interrupt                   */
  OTG_FS_WKUP_IRQn            = 42,     /*!< USB OTG FS Wakeup through EINT line interrupt                     */
  TMR8_BRK_TMR12_IRQn         = 43,     /*!< TMR8 Break Interrupt and TMR12 global interrupt                   */
  TMR8_UP_TMR13_IRQn          = 44,     /*!< TMR8 Update Interrupt and TMR13 global interrupt                  */
  TMR8_TRG_COM_TMR14_IRQn     = 45,     /*!< TMR8 Trigger and Commutation Interrupt and TMR14 global interrupt */
  TMR8_CC_IRQn                = 46,     /*!< TMR8 Capture Compare global interrupt                             */
  DMA1_Stream7_IRQn           = 47,     /*!< DMA1 Stream7 Interrupt                                            */
  SMC_IRQn                    = 48,     /*!< SMC global Interrupt                                              */
  SDIO_IRQn                   = 49,     /*!< SDIO global Interrupt                                             */
  TMR5_IRQn                   = 50,     /*!< TMR5 global Interrupt                                             */
  SPI3_IRQn                   = 51,     /*!< SPI3 global Interrupt                                             */
  UART4_IRQn                  = 52,     /*!< UART4 global Interrupt                                            */
  UART5_IRQn                  = 53,     /*!< UART5 global Interrupt                                            */

  DMA2_Stream0_IRQn           = 56,     /*!< DMA2 Stream 0 global Interrupt                                    */
  DMA2_Stream1_IRQn           = 57,     /*!< DMA2 Stream 1 global Interrupt                                    */
  DMA2_Stream2_IRQn           = 58,     /*!< DMA2 Stream 2 global Interrupt                                    */
  DMA2_Stream3_IRQn           = 59,     /*!< DMA2 Stream 3 global Interrupt                                    */
  DMA2_Stream4_IRQn           = 60,     /*!< DMA2 Stream 4 global Interrupt                                    */

  CAN2_TX_IRQn                = 63,     /*!< CAN2 TX Interrupt                                                 */
  CAN2_RX0_IRQn               = 64,     /*!< CAN2 RX0 Interrupt                                                */
  CAN2_RX1_IRQn               = 65,     /*!< CAN2 RX1 Interrupt                                                */
  CAN2_SCE_IRQn               = 66,     /*!< CAN2 SCE Interrupt                                                */
  OTG_FS_IRQn                 = 67,     /*!< USB OTG FS global Interrupt                                       */
  DMA2_Stream5_IRQn           = 68,     /*!< DMA2 Stream 5 global interrupt                                    */
  DMA2_Stream6_IRQn           = 69,     /*!< DMA2 Stream 6 global interrupt                                    */
  DMA2_Stream7_IRQn           = 70,     /*!< DMA2 Stream 7 global interrupt                                    */
  USART6_IRQn                 = 71,     /*!< USART6 global interrupt                                           */
  I2C3_EV_IRQn                = 72,     /*!< I2C3 event interrupt                                              */
  I2C3_ER_IRQn                = 73,     /*!< I2C3 error interrupt                                              */

  RNG_IRQn                    = 80,     /*!< RNG global Interrupt                                              */
  FPU_IRQn                    = 81,     /*!< FPU global interrupt                                              */

  QSPI_IRQn                   = 83,     /*!< QuadSPI global interrupt                                          */
  SPI4_IRQn                   = 84,     /*!< SPI4 global interrupt                                             */
  SPI5_IRQn                   = 85      /*!< SPI5 global interrupt                                             */
} IRQn_Type;

/**
  * @}
  */

#include "core_cm4.h"             /* Cortex-M4 processor and core peripherals */
#include "system_apm32f4xx.h"
#include <stdint.h>

/** @addtogroup Peripheral_registers_structures
  * @{
  */

/**
  * @brief Analog to Digital Converter
  */

typedef struct
{
  __IO uint32_t STS;      /*!< ADC status register,                         Address offset: 0x00 */
  __IO uint32_t CTRL1;    /*!< ADC control register 1,                      Address offset: 0x04 */
  __IO uint32_t CTRL2;    /*!< ADC control register 2,                      Address offset: 0x08 */
  __IO uint32_t SMPTIM1;  /*!< ADC sample time register 1,                  Address offset: 0x0C */
  __IO uint32_t SMPTIM2;  /*!< ADC sample time register 2,                  Address offset: 0x10 */
  __IO uint32_t INJDOF1;  /*!< ADC injected channel data offset register 1, Address offset: 0x14 */
  __IO uint32_t INJDOF2;  /*!< ADC injected channel data offset register 2, Address offset: 0x18 */
  __IO uint32_t INJDOF3;  /*!< ADC injected channel data offset register 3, Address offset: 0x1C */
  __IO uint32_t INJDOF4;  /*!< ADC injected channel data offset register 4, Address offset: 0x20 */
  __IO uint32_t AWDHT;    /*!< ADC watchdog higher threshold register,      Address offset: 0x24 */
  __IO uint32_t AWDLT;    /*!< ADC watchdog lower threshold register,       Address offset: 0x28 */
  __IO uint32_t REGSEQ1;  /*!< ADC regular sequence register 1,             Address offset: 0x2C */
  __IO uint32_t REGSEQ2;  /*!< ADC regular sequence register 2,             Address offset: 0x30 */
  __IO uint32_t REGSEQ3;  /*!< ADC regular sequence register 3,             Address offset: 0x34 */
  __IO uint32_t INJSEQ;   /*!< ADC injected sequence register,              Address offset: 0x38 */
  __IO uint32_t INJDATA1; /*!< ADC injected data register 1,                Address offset: 0x3C */
  __IO uint32_t INJDATA2; /*!< ADC injected data register 2,                Address offset: 0x40 */
  __IO uint32_t INJDATA3; /*!< ADC injected data register 3,                Address offset: 0x44 */
  __IO uint32_t INJDATA4; /*!< ADC injected data register 4,                Address offset: 0x48 */
  __IO uint32_t REGDATA;  /*!< ADC regular data register,                   Address offset: 0x4C */
} ADC_TypeDef;

typedef struct
{
  __IO uint32_t RESERVED1;  /*!< Reserved, 0x300                                                         */
  __IO uint32_t CCTRL;      /*!< ADC common control register,  Address offset: ADC1 base address + 0x304 */
} ADC_Common_TypeDef;


/**
  * @brief Controller Area Network TxMailBox
  */

typedef struct
{
  __IO uint32_t TXMID;  /*!< CAN TX mailbox identifier register */
  __IO uint32_t TXDLEN; /*!< CAN mailbox data length control and time stamp register */
  __IO uint32_t TXMDL;  /*!< CAN mailbox data low register */
  __IO uint32_t TXMDH;  /*!< CAN mailbox data high register */
} CAN_TxMailBox_TypeDef;

/**
  * @brief Controller Area Network FIFOMailBox
  */

typedef struct
{
  __IO uint32_t RXMID;  /*!< CAN receive FIFO mailbox identifier register */
  __IO uint32_t RXDLEN; /*!< CAN receive FIFO mailbox data length control and time stamp register */
  __IO uint32_t RXMDL;  /*!< CAN receive FIFO mailbox data low register */
  __IO uint32_t RXMDH;  /*!< CAN receive FIFO mailbox data high register */
} CAN_FIFOMailBox_TypeDef;

/**
  * @brief Controller Area Network FilterRegister
  */

typedef struct
{
  __IO uint32_t FBANK1; /*!< CAN Filter bank register 1 */
  __IO uint32_t FBANK2; /*!< CAN Filter bank register 1 */
} CAN_FilterRegister_TypeDef;

/**
  * @brief Controller Area Network
  */

typedef struct
{
  __IO uint32_t              MCTRL;               /*!< CAN master control register,         Address offset: 0x00          */
  __IO uint32_t              MSTS;                /*!< CAN master status register,          Address offset: 0x04          */
  __IO uint32_t              TXSTS;               /*!< CAN transmit status register,        Address offset: 0x08          */
  __IO uint32_t              RXF0;                /*!< CAN receive FIFO 0 register,         Address offset: 0x0C          */
  __IO uint32_t              RXF1;                /*!< CAN receive FIFO 1 register,         Address offset: 0x10          */
  __IO uint32_t              INTEN;               /*!< CAN interrupt enable register,       Address offset: 0x14          */
  __IO uint32_t              ERRSTS;              /*!< CAN error status register,           Address offset: 0x18          */
  __IO uint32_t              BITTIM;              /*!< CAN bit timing register,             Address offset: 0x1C          */
  uint32_t                   RESERVED0[88];       /*!< Reserved, 0x020 - 0x17F                                            */
  CAN_TxMailBox_TypeDef      sTxMailBox[3];       /*!< CAN Tx MailBox,                      Address offset: 0x180 - 0x1AC */
  CAN_FIFOMailBox_TypeDef    sFIFOMailBox[2];     /*!< CAN FIFO MailBox,                    Address offset: 0x1B0 - 0x1CC */
  uint32_t                   RESERVED1[12];       /*!< Reserved, 0x1D0 - 0x1FF                                            */
  __IO uint32_t              FCTRL;               /*!< CAN filter master register,          Address offset: 0x200         */
  __IO uint32_t              FMCFG;               /*!< CAN filter mode register,            Address offset: 0x204         */
  uint32_t                   RESERVED2;           /*!< Reserved, 0x208                                                    */
  __IO uint32_t              FSCFG;               /*!< CAN filter scale register,           Address offset: 0x20C         */
  uint32_t                   RESERVED3;           /*!< Reserved, 0x210                                                    */
  __IO uint32_t              FFASS;               /*!< CAN filter FIFO assignment register, Address offset: 0x214         */
  uint32_t                   RESERVED4;           /*!< Reserved, 0x218                                                    */
  __IO uint32_t              FACT;                /*!< CAN filter activation register,      Address offset: 0x21C         */
  uint32_t                   RESERVED5[8];        /*!< Reserved, 0x220-0x23F                                              */
  CAN_FilterRegister_TypeDef sFilterRegister[28]; /*!< CAN Filter Register,                 Address offset: 0x240-0x31C   */
} CAN_TypeDef;

/**
  * @brief CRC calculation unit
  */

typedef struct
{
  __IO uint32_t DATA;       /*!< CRC Data register,             Address offset: 0x00 */
  __IO uint8_t  INDATA;     /*!< CRC Independent data register, Address offset: 0x04 */
  uint8_t       RESERVED0;  /*!< Reserved, 0x05                                      */
  uint16_t      RESERVED1;  /*!< Reserved, 0x06                                      */
  __IO uint32_t CTRL;       /*!< CRC Control register,          Address offset: 0x08 */
} CRC_TypeDef;

/**
  * @brief Debug MCU
  */

typedef struct
{
  __IO uint32_t IDCODE;   /*!< MCU device ID code,               Address offset: 0x00 */
  __IO uint32_t CFG;      /*!< Debug MCU configuration register, Address offset: 0x04 */
  __IO uint32_t APB1F;    /*!< Debug MCU APB1 freeze register,   Address offset: 0x08 */
  __IO uint32_t APB2F;    /*!< Debug MCU APB2 freeze register,   Address offset: 0x0C */
} DBGMCU_TypeDef;

/**
  * @brief DMA Controller
  */

typedef struct
{
  __IO uint32_t SCFG;   /*!< DMA stream x configuration register      */
  __IO uint32_t NDATA;  /*!< DMA stream x number of data register     */
  __IO uint32_t PADDR;  /*!< DMA stream x peripheral address register */
  __IO uint32_t M0ADDR; /*!< DMA stream x memory 0 address register   */
  __IO uint32_t M1ADDR; /*!< DMA stream x memory 1 address register   */
  __IO uint32_t FCTRL;  /*!< DMA stream x FIFO control register       */
} DMA_Stream_TypeDef;

typedef struct
{
  __IO uint32_t LINTSTS;  /*!< DMA low interrupt status register,      Address offset: 0x00 */
  __IO uint32_t HINTSTS;  /*!< DMA high interrupt status register,     Address offset: 0x04 */
  __IO uint32_t LIFCLR;   /*!< DMA low interrupt flag clear register,  Address offset: 0x08 */
  __IO uint32_t HIFCLR;   /*!< DMA high interrupt flag clear register, Address offset: 0x0C */
} DMA_TypeDef;

/**
  * @brief External Interrupt/Event Controller
  */

typedef struct
{
  __IO uint32_t IMASK;    /*!< EINT Interrupt mask register,            Address offset: 0x00 */
  __IO uint32_t EMASK;    /*!< EINT Event mask register,                Address offset: 0x04 */
  __IO uint32_t RTEN;     /*!< EINT Rising trigger selection register,  Address offset: 0x08 */
  __IO uint32_t FTEN;     /*!< EINT Falling trigger selection register, Address offset: 0x0C */
  __IO uint32_t SWINTE;   /*!< EINT Software interrupt event register,  Address offset: 0x10 */
  __IO uint32_t IPEND;    /*!< EINT Pending register,                   Address offset: 0x14 */
} EINT_TypeDef;

/**
  * @brief FLASH Registers
  */

typedef struct
{
  __IO uint32_t ACCTRL;   /*!< FLASH access control register,   Address offset: 0x00 */
  __IO uint32_t KEY;      /*!< FLASH key register,              Address offset: 0x04 */
  __IO uint32_t OPTKEY;   /*!< FLASH option key register,       Address offset: 0x08 */
  __IO uint32_t STS;      /*!< FLASH status register,           Address offset: 0x0C */
  __IO uint32_t CTRL;     /*!< FLASH control register,          Address offset: 0x10 */
  __IO uint32_t OPTCTRL;  /*!< FLASH option control register ,  Address offset: 0x14 */
} FLASH_TypeDef;

/**
  * @brief Flexible Static Memory Controller
  */

typedef struct
{
  __IO uint32_t CSCTRL[8];    /*!< NOR/PSRAM chip-select control register(CSCTRL) and chip-select timing register(CSTIM), Address offset: 0x00-1C */
} SMC_Bank1_TypeDef;

/**
  * @brief Flexible Static Memory Controller Bank1E
  */

typedef struct
{
  __IO uint32_t WRTTIM[7];    /*!< NOR/PSRAM write timing registers, Address offset: 0x104-0x11C */
} SMC_Bank1E_TypeDef;

/**
  * @brief Flexible Static Memory Controller Bank2
  */

typedef struct
{
  __IO uint32_t CTRL2;        /*!< NAND Flash control register 2,                       Address offset: 0x60 */
  __IO uint32_t STSINT2;      /*!< NAND Flash FIFO status and interrupt register 2,     Address offset: 0x64 */
  __IO uint32_t CMSTIM2;      /*!< NAND Flash Common memory space timing register 2,    Address offset: 0x68 */
  __IO uint32_t AMSTIM2;      /*!< NAND Flash Attribute memory space timing register 2, Address offset: 0x6C */
  uint32_t      RESERVED0;    /*!< Reserved, 0x70                                                            */
  __IO uint32_t ECCRS2;       /*!< NAND Flash ECC result registers 2,                   Address offset: 0x74 */
  uint32_t      RESERVED1;    /*!< Reserved, 0x78                                                            */
  uint32_t      RESERVED2;    /*!< Reserved, 0x7C                                                            */
  __IO uint32_t CTRL3;        /*!< NAND Flash control register 3,                       Address offset: 0x80 */
  __IO uint32_t STSINT3;      /*!< NAND Flash FIFO status and interrupt register 3,     Address offset: 0x84 */
  __IO uint32_t CMSTIM3;      /*!< NAND Flash Common memory space timing register 3,    Address offset: 0x88 */
  __IO uint32_t AMSTIM3;      /*!< NAND Flash Attribute memory space timing register 3, Address offset: 0x8C */
  uint32_t      RESERVED3;    /*!< Reserved, 0x90                                                            */
  __IO uint32_t ECCRS3;       /*!< NAND Flash ECC result registers 3,                   Address offset: 0x94 */
} SMC_Bank2_3_TypeDef;

/**
  * @brief Flexible Static Memory Controller Bank4
  */

typedef struct
{
  __IO uint32_t CTRL4;        /*!< PC Card  control register 4,                       Address offset: 0xA0 */
  __IO uint32_t STSINT4;      /*!< PC Card  FIFO status and interrupt register 4,     Address offset: 0xA4 */
  __IO uint32_t CMSTIM4;      /*!< PC Card  Common memory space timing register 4,    Address offset: 0xA8 */
  __IO uint32_t AMSTIM4;      /*!< PC Card  Attribute memory space timing register 4, Address offset: 0xAC */
  __IO uint32_t IOSTIM4;      /*!< PC Card  I/O space timing register 4,              Address offset: 0xB0 */
} SMC_Bank4_TypeDef;

/**
  * @brief General Purpose I/O
  */

typedef struct
{
  __IO uint32_t MODE;     /*!< GPIO port mode register,               Address offset: 0x00      */
  __IO uint32_t OMODE;    /*!< GPIO port output type register,        Address offset: 0x04      */
  __IO uint32_t OSSEL;    /*!< GPIO port output speed register,       Address offset: 0x08      */
  __IO uint32_t PUPD;     /*!< GPIO port pull-up/pull-down register,  Address offset: 0x0C      */
  __IO uint32_t IDATA;    /*!< GPIO port input data register,         Address offset: 0x10      */
  __IO uint32_t ODATA;    /*!< GPIO port output data register,        Address offset: 0x14      */
  __IO uint32_t BSC;      /*!< GPIO port bit set/reset register,      Address offset: 0x18      */
  __IO uint32_t LOCK;     /*!< GPIO port configuration lock register, Address offset: 0x1C      */
  __IO uint32_t ALF[2];   /*!< GPIO alternate function registers,     Address offset: 0x20-0x24 */
} GPIO_TypeDef;

/**
  * @brief System configuration controller
  */

typedef struct
{
  __IO uint32_t MMSEL;        /*!< SYSCFG memory remap register,                      Address offset: 0x00      */
  __IO uint32_t PMCFG;        /*!< SYSCFG peripheral mode configuration register,     Address offset: 0x04      */
  __IO uint32_t EINTCFG[4];   /*!< SYSCFG external interrupt configuration registers, Address offset: 0x08-0x14 */
  uint32_t      RESERVED[2];  /*!< Reserved, 0x18-0x1C                                                          */
  __IO uint32_t CCCTRL;       /*!< SYSCFG Compensation cell control register,         Address offset: 0x20      */
} SYSCFG_TypeDef;

/**
  * @brief Inter-integrated Circuit Interface
  */

typedef struct
{
  __IO uint32_t CTRL1;      /*!< I2C Control register 1,     Address offset: 0x00 */
  __IO uint32_t CTRL2;      /*!< I2C Control register 2,     Address offset: 0x04 */
  __IO uint32_t SADDR1;     /*!< I2C Own address register 1, Address offset: 0x08 */
  __IO uint32_t SADDR2;     /*!< I2C Own address register 2, Address offset: 0x0C */
  __IO uint32_t DATA;       /*!< I2C Data register,          Address offset: 0x10 */
  __IO uint32_t STS1;       /*!< I2C Status register 1,      Address offset: 0x14 */
  __IO uint32_t STS2;       /*!< I2C Status register 2,      Address offset: 0x18 */
  __IO uint32_t CLKCTRL;    /*!< I2C Clock control register, Address offset: 0x1C */
  __IO uint32_t RISETMAX;   /*!< I2C RISETMAX register,      Address offset: 0x20 */
  __IO uint32_t FILTER;     /*!< I2C Filter register,        Address offset: 0x24 */
} I2C_TypeDef;

/**
  * @brief Independent WATCHDOG
  */

typedef struct
{
  __IO uint32_t KEY;    /*!< IWDT Key register,       Address offset: 0x00 */
  __IO uint32_t PSC;    /*!< IWDT Prescaler register, Address offset: 0x04 */
  __IO uint32_t CNTRLD; /*!< IWDT Reload register,    Address offset: 0x08 */
  __IO uint32_t STS;    /*!< IWDT Status register,    Address offset: 0x0C */
} IWDT_TypeDef;


/**
  * @brief Power Control
  */

typedef struct
{
  __IO uint32_t CTRL; /*!< PMU power control register,        Address offset: 0x00 */
  __IO uint32_t CSTS; /*!< PMU power control/status register, Address offset: 0x04 */
} PMU_TypeDef;

/**
  * @brief Reset and Clock Control
  */

typedef struct
{
  __IO uint32_t CTRL;           /*!< RCM clock control register,                                  Address offset: 0x00 */
  __IO uint32_t PLL1CFG;        /*!< RCM PLL configuration register,                              Address offset: 0x04 */
  __IO uint32_t CFG;            /*!< RCM clock configuration register,                            Address offset: 0x08 */
  __IO uint32_t INT;            /*!< RCM clock interrupt register,                                Address offset: 0x0C */
  __IO uint32_t AHB1RST;        /*!< RCM AHB1 peripheral reset register,                          Address offset: 0x10 */
  __IO uint32_t AHB2RST;        /*!< RCM AHB2 peripheral reset register,                          Address offset: 0x14 */
  __IO uint32_t AHB3RST;        /*!< RCM AHB3 peripheral reset register,                          Address offset: 0x18 */
  uint32_t      RESERVED0;      /*!< Reserved, 0x1C                                                                    */
  __IO uint32_t APB1RST;        /*!< RCM APB1 peripheral reset register,                          Address offset: 0x20 */
  __IO uint32_t APB2RST;        /*!< RCM APB2 peripheral reset register,                          Address offset: 0x24 */
  uint32_t      RESERVED1[2];   /*!< Reserved, 0x28-0x2C                                                               */
  __IO uint32_t AHB1CLKEN;      /*!< RCM AHB1 peripheral clock register,                          Address offset: 0x30 */
  __IO uint32_t AHB2CLKEN;      /*!< RCM AHB2 peripheral clock register,                          Address offset: 0x34 */
  __IO uint32_t AHB3CLKEN;      /*!< RCM AHB3 peripheral clock register,                          Address offset: 0x38 */
  uint32_t      RESERVED2;      /*!< Reserved, 0x3C                                                                    */
  __IO uint32_t APB1CLKEN;      /*!< RCM APB1 peripheral clock enable register,                   Address offset: 0x40 */
  __IO uint32_t APB2CLKEN;      /*!< RCM APB2 peripheral clock enable register,                   Address offset: 0x44 */
  uint32_t      RESERVED3[2];   /*!< Reserved, 0x48-0x4C                                                               */
  __IO uint32_t LPAHB1CLKEN;    /*!< RCM AHB1 peripheral clock enable in low power mode register, Address offset: 0x50 */
  __IO uint32_t LPAHB2CLKEN;    /*!< RCM AHB2 peripheral clock enable in low power mode register, Address offset: 0x54 */
  __IO uint32_t LPAHB3CLKEN;    /*!< RCM AHB3 peripheral clock enable in low power mode register, Address offset: 0x58 */
  uint32_t      RESERVED4;      /*!< Reserved, 0x5C                                                                    */
  __IO uint32_t LPAPB1CLKEN;    /*!< RCM APB1 peripheral clock enable in low power mode register, Address offset: 0x60 */
  __IO uint32_t LPAPB2CLKEN;    /*!< RCM APB2 peripheral clock enable in low power mode register, Address offset: 0x64 */
  uint32_t      RESERVED5[2];   /*!< Reserved, 0x68-0x6C                                                               */
  __IO uint32_t BDCTRL;         /*!< RCM Backup domain control register,                          Address offset: 0x70 */
  __IO uint32_t CSTS;           /*!< RCM clock control & status register,                         Address offset: 0x74 */
  uint32_t      RESERVED6[2];   /*!< Reserved, 0x78-0x7C                                                               */
  __IO uint32_t SSCCFG;         /*!< RCM spread spectrum clock generation register,               Address offset: 0x80 */
  __IO uint32_t PLL2CFG;        /*!< RCM PLLI2S configuration register,                           Address offset: 0x84 */
  uint32_t      RESERVED7;      /*!< Reserved, 0x88                                                                    */
  __IO uint32_t CFGSEL;         /*!< RCM clock configuration select register,                     Address offset: 0x8C */
} RCM_TypeDef;

/**
  * @brief Real-Time Clock
  */

typedef struct
{
  __IO uint32_t TIME;     /*!< RTC time register,                                        Address offset: 0x00 */
  __IO uint32_t DATE;     /*!< RTC date register,                                        Address offset: 0x04 */
  __IO uint32_t CTRL;     /*!< RTC control register,                                     Address offset: 0x08 */
  __IO uint32_t STS;      /*!< RTC initialization and status register,                   Address offset: 0x0C */
  __IO uint32_t PSC;      /*!< RTC prescaler register,                                   Address offset: 0x10 */
  __IO uint32_t AUTORLD;  /*!< RTC wakeup timer register,                                Address offset: 0x14 */
  __IO uint32_t DCAL;     /*!< RTC calibration register,                                 Address offset: 0x18 */
  __IO uint32_t ALRMA;    /*!< RTC alarm A register,                                     Address offset: 0x1C */
  __IO uint32_t ALRMB;    /*!< RTC alarm B register,                                     Address offset: 0x20 */
  __IO uint32_t WRPROT;   /*!< RTC write protection register,                            Address offset: 0x24 */
  __IO uint32_t SUBSEC;   /*!< RTC sub second register,                                  Address offset: 0x28 */
  __IO uint32_t SHIFT;    /*!< RTC shift control register,                               Address offset: 0x2C */
  __IO uint32_t TSTIME;   /*!< RTC time stamp time register,                             Address offset: 0x30 */
  __IO uint32_t TSDATE;   /*!< RTC time stamp date register,                             Address offset: 0x34 */
  __IO uint32_t TSSUBSEC; /*!< RTC time-stamp sub second register,                       Address offset: 0x38 */
  __IO uint32_t CAL;      /*!< RTC calibration register,                                 Address offset: 0x3C */
  __IO uint32_t TACFG;    /*!< RTC tamper and alternate function configuration register, Address offset: 0x40 */
  __IO uint32_t ALRMASS;  /*!< RTC alarm A sub second register,                          Address offset: 0x44 */
  __IO uint32_t ALRMBSS;  /*!< RTC alarm B sub second register,                          Address offset: 0x48 */
  uint32_t RESERVED1;    /*!< Reserved, 0x4C                                                                 */
  __IO uint32_t BAKP0;   /*!< RTC backup register 1,                                    Address offset: 0x50 */
  __IO uint32_t BAKP1;   /*!< RTC backup register 1,                                    Address offset: 0x54 */
  __IO uint32_t BAKP2;   /*!< RTC backup register 2,                                    Address offset: 0x58 */
  __IO uint32_t BAKP3;   /*!< RTC backup register 3,                                    Address offset: 0x5C */
  __IO uint32_t BAKP4;   /*!< RTC backup register 4,                                    Address offset: 0x60 */
  __IO uint32_t BAKP5;   /*!< RTC backup register 5,                                    Address offset: 0x64 */
  __IO uint32_t BAKP6;   /*!< RTC backup register 6,                                    Address offset: 0x68 */
  __IO uint32_t BAKP7;   /*!< RTC backup register 7,                                    Address offset: 0x6C */
  __IO uint32_t BAKP8;   /*!< RTC backup register 8,                                    Address offset: 0x70 */
  __IO uint32_t BAKP9;   /*!< RTC backup register 9,                                    Address offset: 0x74 */
  __IO uint32_t BAKP10;  /*!< RTC backup register 10,                                   Address offset: 0x78 */
  __IO uint32_t BAKP11;  /*!< RTC backup register 11,                                   Address offset: 0x7C */
  __IO uint32_t BAKP12;  /*!< RTC backup register 12,                                   Address offset: 0x80 */
  __IO uint32_t BAKP13;  /*!< RTC backup register 13,                                   Address offset: 0x84 */
  __IO uint32_t BAKP14;  /*!< RTC backup register 14,                                   Address offset: 0x88 */
  __IO uint32_t BAKP15;  /*!< RTC backup register 15,                                   Address offset: 0x8C */
  __IO uint32_t BAKP16;  /*!< RTC backup register 16,                                   Address offset: 0x90 */
  __IO uint32_t BAKP17;  /*!< RTC backup register 17,                                   Address offset: 0x94 */
  __IO uint32_t BAKP18;  /*!< RTC backup register 18,                                   Address offset: 0x98 */
  __IO uint32_t BAKP19;  /*!< RTC backup register 19,                                   Address offset: 0x9C */
} RTC_TypeDef;

/**
  * @brief SD host Interface
  */

typedef struct
{
  __IO uint32_t PWRCTRL;              /*!< SDIO power control register,    Address offset: 0x00 */
  __IO uint32_t CLKCTRL;              /*!< SDI clock control register,     Address offset: 0x04 */
  __IO uint32_t ARG;                  /*!< SDIO argument register,         Address offset: 0x08 */
  __IO uint32_t CMD;                  /*!< SDIO command register,          Address offset: 0x0C */
  __IO const uint32_t  CMDRES;        /*!< SDIO command response register, Address offset: 0x10 */
  __IO const uint32_t  RES1;          /*!< SDIO response 1 register,       Address offset: 0x14 */
  __IO const uint32_t  RES2;          /*!< SDIO response 2 register,       Address offset: 0x18 */
  __IO const uint32_t  RES3;          /*!< SDIO response 3 register,       Address offset: 0x1C */
  __IO const uint32_t  RES4;          /*!< SDIO response 4 register,       Address offset: 0x20 */
  __IO uint32_t DATATIME;             /*!< SDIO data timer register,       Address offset: 0x24 */
  __IO uint32_t DATALEN;              /*!< SDIO data length register,      Address offset: 0x28 */
  __IO uint32_t DCTRL;                /*!< SDIO data control register,     Address offset: 0x2C */
  __IO const uint32_t  DCNT;          /*!< SDIO data counter register,     Address offset: 0x30 */
  __IO const uint32_t  STS;           /*!< SDIO status register,           Address offset: 0x34 */
  __IO uint32_t ICF;                  /*!< SDIO interrupt clear register,  Address offset: 0x38 */
  __IO uint32_t MASK;                 /*!< SDIO mask register,             Address offset: 0x3C */
  uint32_t      RESERVED0[2];         /*!< Reserved, 0x40-0x44                                  */
  __IO const uint32_t  FIFOCNT;       /*!< SDIO FIFO counter register,     Address offset: 0x48 */
  uint32_t      RESERVED1[13];        /*!< Reserved, 0x4C-0x7C                                  */
  __IO uint32_t FIFODATA;             /*!< SDIO data FIFO register,        Address offset: 0x80 */
} SDIO_TypeDef;

/**
  * @brief Serial Peripheral Interface
  */

typedef struct
{
  __IO uint32_t CTRL1;      /*!< SPI control register 1 (not used in I2S mode),      Address offset: 0x00 */
  __IO uint32_t CTRL2;      /*!< SPI control register 2,                             Address offset: 0x04 */
  __IO uint32_t STS;        /*!< SPI status register,                                Address offset: 0x08 */
  __IO uint32_t DATA;       /*!< SPI data register,                                  Address offset: 0x0C */
  __IO uint32_t CRCPOLY;    /*!< SPI CRC polynomial register (not used in I2S mode), Address offset: 0x10 */
  __IO uint32_t RXCRC;      /*!< SPI RX CRC register (not used in I2S mode),         Address offset: 0x14 */
  __IO uint32_t TXCRC;      /*!< SPI TX CRC register (not used in I2S mode),         Address offset: 0x18 */
  __IO uint32_t I2SCFG;     /*!< SPI_I2S configuration register,                     Address offset: 0x1C */
  __IO uint32_t I2SPSC;     /*!< SPI_I2S prescaler register,                         Address offset: 0x20 */
} SPI_TypeDef;


/**
  * @brief TMR
  */

typedef struct
{
  __IO uint32_t CTRL1;        /*!< TMR control register 1,              Address offset: 0x00 */
  __IO uint32_t CTRL2;        /*!< TMR control register 2,              Address offset: 0x04 */
  __IO uint32_t SMCTRL;       /*!< TMR slave mode control register,     Address offset: 0x08 */
  __IO uint32_t DIEN;         /*!< TMR DMA/interrupt enable register,   Address offset: 0x0C */
  __IO uint32_t STS;          /*!< TMR status register,                 Address offset: 0x10 */
  __IO uint32_t CEG;          /*!< TMR event generation register,       Address offset: 0x14 */
  __IO uint32_t CCM1;         /*!< TMR capture/compare mode register 1, Address offset: 0x18 */
  __IO uint32_t CCM2;         /*!< TMR capture/compare mode register 2, Address offset: 0x1C */
  __IO uint32_t CCEN;         /*!< TMR capture/compare enable register, Address offset: 0x20 */
  __IO uint32_t CNT;          /*!< TMR counter register,                Address offset: 0x24 */
  __IO uint32_t PSC;          /*!< TMR prescaler,                       Address offset: 0x28 */
  __IO uint32_t AUTORLD;      /*!< TMR auto-reload register,            Address offset: 0x2C */
  __IO uint32_t REPCNT;       /*!< TMR repetition counter register,     Address offset: 0x30 */
  __IO uint32_t CC1;          /*!< TMR capture/compare register 1,      Address offset: 0x34 */
  __IO uint32_t CC2;          /*!< TMR capture/compare register 2,      Address offset: 0x38 */
  __IO uint32_t CC3;          /*!< TMR capture/compare register 3,      Address offset: 0x3C */
  __IO uint32_t CC4;          /*!< TMR capture/compare register 4,      Address offset: 0x40 */
  __IO uint32_t BDT;          /*!< TMR break and dead-time register,    Address offset: 0x44 */
  __IO uint32_t DCTRL;        /*!< TMR DMA control register,            Address offset: 0x48 */
  __IO uint32_t DMADDR;       /*!< TMR DMA address for full transfer,   Address offset: 0x4C */
  __IO uint32_t OPT;          /*!< TMR option register,                 Address offset: 0x50 */
} TMR_TypeDef;

/**
  * @brief Universal Synchronous Asynchronous Receiver Transmitter
  */

typedef struct
{
  __IO uint32_t STS;        /*!< USART Status register,                   Address offset: 0x00 */
  __IO uint32_t DATA;       /*!< USART Data register,                     Address offset: 0x04 */
  __IO uint32_t BR;         /*!< USART Baud rate register,                Address offset: 0x08 */
  __IO uint32_t CTRL1;      /*!< USART Control register 1,                Address offset: 0x0C */
  __IO uint32_t CTRL2;      /*!< USART Control register 2,                Address offset: 0x10 */
  __IO uint32_t CTRL3;      /*!< USART Control register 3,                Address offset: 0x14 */
  __IO uint32_t GTPSC;      /*!< USART Guard time and prescaler register, Address offset: 0x18 */
} USART_TypeDef;

/**
  * @brief Window WATCHDOG
  */

typedef struct
{
  __IO uint32_t CTRL; /*!< WWDT Control register,       Address offset: 0x00 */
  __IO uint32_t CFG;  /*!< WWDT Configuration register, Address offset: 0x04 */
  __IO uint32_t STS;  /*!< WWDT Status register,        Address offset: 0x08 */
} WWDT_TypeDef;

/**
  * @brief RNG
  */

typedef struct
{
  __IO uint32_t CTRL; /*!< RNG control register, Address offset: 0x00 */
  __IO uint32_t STS;  /*!< RNG status register,  Address offset: 0x04 */
  __IO uint32_t DATA; /*!< RNG data register,    Address offset: 0x08 */
} RNG_TypeDef;

/**
 * @brief Quick Serial peripheral interface
 */

typedef struct
{
    __IO uint32_t CTRL1;            /*!< QSPI control register 1,                               Address offset: 0x00 */
    __IO uint32_t CTRL2;            /*!< QSPI control register 2,                               Address offset: 0x04 */
    __IO uint32_t SSIEN;            /*!< QSPI enable register,                                  Address offset: 0x08 */
    __IO uint32_t RESERVED0;        /*!< Reserved,                                              Address offset: 0x0C */
    __IO uint32_t SLAEN;            /*!< QSPI slave enable register,                            Address offset: 0x10 */
    __IO uint32_t BR;               /*!< QSPI baud rate register,                               Address offset: 0x14 */
    __IO uint32_t TFTL;             /*!< QSPI transmit FIFO threshold level register,           Address offset: 0x18 */
    __IO uint32_t RFTL;             /*!< QSPI receive FIFO threshold level register,            Address offset: 0x1C */
    __IO uint32_t TFL;              /*!< QSPI transmit FIFO level register,                     Address offset: 0x20 */
    __IO uint32_t RFL;              /*!< QSPI receive FIFO level register,                      Address offset: 0x24 */
    __IO uint32_t STS;              /*!< QSPI status register,                                  Address offset: 0x28 */
    __IO uint32_t INTEN;            /*!< QSPI interrupt enable register,                        Address offset: 0x2C */
    __IO uint32_t ISTS;             /*!< QSPI interrupt status register,                        Address offset: 0x30 */
    __IO uint32_t RIS;              /*!< QSPI raw interrupt status register,                    Address offset: 0x34 */
    __IO uint32_t TFOIC;            /*!< QSPI transmit FIFO overflow interrupt clear register,  Address offset: 0x38 */
    __IO uint32_t RFOIC;            /*!< QSPI receive FIFO overflow interrupt clear register,   Address offset: 0x3C */
    __IO uint32_t RFUIC;            /*!< QSPI receive FIFO underflow interrupt clear register,  Address offset: 0x40 */
    __IO uint32_t MIC;              /*!< QSPI master interrupt clear register,                  Address offset: 0x44 */
    __IO uint32_t ICF;              /*!< QSPI interrupt clear flag register,                    Address offset: 0x48 */
    __IO uint32_t DMACTRL;          /*!< QSPI DMA control register,                             Address offset: 0x4C */
    __IO uint32_t DMATDL;           /*!< QSPI DMA transmit data level register,                 Address offset: 0x50 */
    __IO uint32_t DMARDL;           /*!< QSPI DMA receive data level register,                  Address offset: 0x54 */
    __IO uint32_t RESERVED1[2];     /*!< Reserved,                                              Address offset: 0x58-0x5C */
    __IO uint32_t DATA;             /*!< QSPI data register,                                    Address offset: 0x60 */
    __IO uint32_t RESERVED2[35];    /*!< Reserved,                                              Address offset: 0x64-0xEC */
    __IO uint32_t RSD;              /*!< QSPI receive sample delay register,                    Address offset: 0xF0 */
    __IO uint32_t CTRL3;            /*!< QSPI control register 3,                               Address offset: 0xF4 */
    __IO uint32_t RESERVED3[66];    /*!< Reserved,                                              Address offset: 0xF8-0x1FC */
    __IO uint32_t IOSW;             /*!< QSPI IO switch register,                               Address offset: 0x200 */
} QSPI_TypeDef;

/**
  * @brief Comparator
  */

typedef struct
{
    __IO uint32_t CSTS;             /*!< COMP control and status register,              Address offset: 0x00 */
} COMP_TypeDef;

typedef struct
{
    __IO uint32_t CSTS;             /*!< COMP control and status register, used for bits common to several COMP instances, Address offset: 0x00 */
} COMP_Common_TypeDef;

/**
  * @brief USB_OTG_Core_Registers
  */
typedef struct
{
  __IO uint32_t GCTRLSTS;             /*!< USB_OTG Control and Status Register          000h */
  __IO uint32_t GINT;                 /*!< USB_OTG Interrupt Register                   004h */
  __IO uint32_t GAHBCFG;              /*!< Core AHB Configuration Register              008h */
  __IO uint32_t GUSBCFG;              /*!< Core USB Configuration Register              00Ch */
  __IO uint32_t GRSTCTRL;             /*!< Core Reset Register                          010h */
  __IO uint32_t GCINT;                /*!< Core Interrupt Register                      014h */
  __IO uint32_t GINTMASK;             /*!< Core Interrupt Mask Register                 018h */
  __IO uint32_t GRXSTS;               /*!< Receive Sts Q Read Register                  01Ch */
  __IO uint32_t GRXSTSP;              /*!< Receive Sts Q Read & POP Register            020h */
  __IO uint32_t GRXFIFO;              /*!< Receive FIFO Size Register                   024h */
  __IO uint32_t GTXFCFG;              /*!< EP0 / Non Periodic Tx FIFO Size Register     028h */
  __IO uint32_t GNPTXFQSTS;           /*!< Non Periodic Tx FIFO/Queue Sts reg           02Ch */
  uint32_t Reserved30[2];             /*!< Reserved                                     030h */
  __IO uint32_t GGCCFG;               /*!< General Purpose IO Register                  038h */
  __IO uint32_t GCID;                 /*!< User ID Register                             03Ch */
  uint32_t  Reserved40[48];           /*!< Reserved                                0x40-0xFF */
  __IO uint32_t GHPTXFSIZE;           /*!< Host Periodic Tx FIFO Size Reg               100h */
  __IO uint32_t DTXFIFO[0x0F];        /*!< dev Periodic Transmit FIFO                        */
} USB_OTG_GlobalTypeDef;

/**
  * @brief USB_OTG_device_Registers
  */
typedef struct
{
  __IO uint32_t DCFG;           /*!< dev Configuration Register   800h */
  __IO uint32_t DCTRL;          /*!< dev Control Register         804h */
  __IO uint32_t DSTS;           /*!< dev Status Register (RO)     808h */
  uint32_t Reserved0C;          /*!< Reserved                     80Ch */
  __IO uint32_t DINIMASK;       /*!< dev IN Endpoint Mask         810h */
  __IO uint32_t DOUTIMASK;      /*!< dev OUT Endpoint Mask        814h */
  __IO uint32_t DAEPINT;        /*!< dev All Endpoints Itr Reg    818h */
  __IO uint32_t DAEPIMASK;      /*!< dev All Endpoints Itr Mask   81Ch */
  uint32_t  Reserved20;         /*!< Reserved                     820h */
  uint32_t Reserved9;           /*!< Reserved                     824h */
  __IO uint32_t DVBUSDTIM;      /*!< dev VBUS discharge Register  828h */
  __IO uint32_t DVBUSPTIM;      /*!< dev VBUS Pulse Register      82Ch */
  __IO uint32_t DTHCTRL;        /*!< dev threshold                830h */
  __IO uint32_t DIEIMASK;       /*!< dev empty msk                834h */
  __IO uint32_t DEPINT;         /*!< dedicated EP interrupt       838h */
  __IO uint32_t DEPIMASK;       /*!< dedicated EP msk             83Ch */
  uint32_t Reserved40;          /*!< dedicated EP mask            840h */
  __IO uint32_t DIN1IMASK;      /*!< dedicated EP mask            844h */
  uint32_t  Reserved44[15];     /*!< Reserved                 844-87Ch */
  __IO uint32_t DOUT1MASK;      /*!< dedicated EP msk             884h */
} USB_OTG_DeviceTypeDef;

/**
  * @brief USB_OTG_IN_Endpoint-Specific_Register
  */
typedef struct
{
  __IO uint32_t DIEPCTRL;         /*!< dev IN Endpoint Control Reg    900h + (ep_num * 20h) + 00h */
  uint32_t Reserved04;            /*!< Reserved                       900h + (ep_num * 20h) + 04h */
  __IO uint32_t DIEPINT;          /*!< dev IN Endpoint Itr Reg        900h + (ep_num * 20h) + 08h */
  uint32_t Reserved0C;            /*!< Reserved                       900h + (ep_num * 20h) + 0Ch */
  __IO uint32_t DIEPTRS;          /*!< IN Endpoint Txfer Size         900h + (ep_num * 20h) + 10h */
  __IO uint32_t DIEPDMA;          /*!< IN Endpoint DMA Address Reg    900h + (ep_num * 20h) + 14h */
  __IO uint32_t DITXFSTS;         /*!< IN Endpoint Tx FIFO Status Reg 900h + (ep_num * 20h) + 18h */
  uint32_t Reserved18;            /*!< Reserved  900h+(ep_num*20h)+1Ch-900h+ (ep_num * 20h) + 1Ch */
} USB_OTG_INEndpointTypeDef;

/**
  * @brief USB_OTG_OUT_Endpoint-Specific_Registers
  */
typedef struct
{
  __IO uint32_t DOEPCTRL;       /*!< dev OUT Endpoint Control Reg           B00h + (ep_num * 20h) + 00h */
  uint32_t Reserved04;          /*!< Reserved                               B00h + (ep_num * 20h) + 04h */
  __IO uint32_t DOEPINT;        /*!< dev OUT Endpoint Itr Reg               B00h + (ep_num * 20h) + 08h */
  uint32_t Reserved0C;          /*!< Reserved                               B00h + (ep_num * 20h) + 0Ch */
  __IO uint32_t DOEPTRS;        /*!< dev OUT Endpoint Txfer Size            B00h + (ep_num * 20h) + 10h */
  __IO uint32_t DOEPDMA;        /*!< dev OUT Endpoint DMA Address           B00h + (ep_num * 20h) + 14h */
  uint32_t Reserved18[2];       /*!< Reserved B00h + (ep_num * 20h) + 18h - B00h + (ep_num * 20h) + 1Ch */
} USB_OTG_OUTEndpointTypeDef;

/**
  * @brief USB_OTG_Host_Mode_Register_Structures
  */
typedef struct
{
  __IO uint32_t HCFG;             /*!< Host Configuration Register          400h */
  __IO uint32_t HFIVL;            /*!< Host Frame Interval Register         404h */
  __IO uint32_t HFIFM;            /*!< Host Frame Nbr/Frame Remaining       408h */
  uint32_t Reserved40C;           /*!< Reserved                             40Ch */
  __IO uint32_t HPTXSTS;          /*!< Host Periodic Tx FIFO/ Queue Status  410h */
  __IO uint32_t HACHINT;          /*!< Host All Channels Interrupt Register 414h */
  __IO uint32_t HACHIMASK;        /*!< Host All Channels Interrupt Mask     418h */
} USB_OTG_HostTypeDef;

/**
  * @brief USB_OTG_Host_Channel_Specific_Registers
  */
typedef struct
{
  __IO uint32_t HCH;              /*!< Host Channel Characteristics Register    500h */
  __IO uint32_t HCHSCTRL;         /*!< Host Channel Split Control Register      504h */
  __IO uint32_t HCHINT;           /*!< Host Channel Interrupt Register          508h */
  __IO uint32_t HCHIMASK;         /*!< Host Channel Interrupt Mask Register     50Ch */
  __IO uint32_t HCHTSIZE;         /*!< Host Channel Transfer Size Register      510h */
  __IO uint32_t HCHDMA;           /*!< Host Channel DMA Address Register        514h */
  uint32_t Reserved[2];           /*!< Reserved                                      */
} USB_OTG_HostChannelTypeDef;

/**
  * @}
  */

/** @addtogroup Peripheral_memory_map
  * @{
  */
#define FLASH_BASE            0x08000000UL /*!< FLASH(up to 1 MB) base address in the alias region                         */
#define CCMDATARAM_BASE       0x10000000UL /*!< CCM(core coupled memory) data RAM(64 KB) base address in the alias region  */
#define SRAM1_BASE            0x20000000UL /*!< SRAM1(112 KB) base address in the alias region                             */
#define SRAM2_BASE            0x2001C000UL /*!< SRAM2(16 KB) base address in the alias region                              */
#define PERIPH_BASE           0x40000000UL /*!< Peripheral base address in the alias region                                */
#define BKPSRAM_BASE          0x40024000UL /*!< Backup SRAM(4 KB) base address in the alias region                         */
#define SMC_R_BASE            0xA0000000UL /*!< SMC registers base address                                                 */
#define SRAM1_BB_BASE         0x22000000UL /*!< SRAM1(112 KB) base address in the bit-band region                          */
#define SRAM2_BB_BASE         0x22380000UL /*!< SRAM2(16 KB) base address in the bit-band region                           */
#define PERIPH_BB_BASE        0x42000000UL /*!< Peripheral base address in the bit-band region                             */
#define BKPSRAM_BB_BASE       0x42480000UL /*!< Backup SRAM(4 KB) base address in the bit-band region                      */
#define FLASH_END             0x080FFFFFUL /*!< FLASH end address                                                          */
#define FLASH_OTP_BASE        0x1FFF7800UL /*!< Base address of : (up to 528 Bytes) embedded FLASH OTP Area                */
#define FLASH_OTP_END         0x1FFF7A0FUL /*!< End address of : (up to 528 Bytes) embedded FLASH OTP Area                 */
#define CCMDATARAM_END        0x1000FFFFUL /*!< CCM data RAM end address                                                   */

/* Legacy defines */
#define SRAM_BASE             SRAM1_BASE
#define SRAM_BB_BASE          SRAM1_BB_BASE

/*!< Peripheral memory map */
#define APB1PERIPH_BASE       PERIPH_BASE
#define APB2PERIPH_BASE       (PERIPH_BASE + 0x00010000UL)
#define AHB1PERIPH_BASE       (PERIPH_BASE + 0x00020000UL)
#define AHB2PERIPH_BASE       (PERIPH_BASE + 0x10000000UL)

/*!< APB1 peripherals */
#define TMR2_BASE             (APB1PERIPH_BASE + 0x0000UL)
#define TMR3_BASE             (APB1PERIPH_BASE + 0x0400UL)
#define TMR4_BASE             (APB1PERIPH_BASE + 0x0800UL)
#define TMR5_BASE             (APB1PERIPH_BASE + 0x0C00UL)
#define TMR12_BASE            (APB1PERIPH_BASE + 0x1800UL)
#define TMR13_BASE            (APB1PERIPH_BASE + 0x1C00UL)
#define TMR14_BASE            (APB1PERIPH_BASE + 0x2000UL)
#define RTC_BASE              (APB1PERIPH_BASE + 0x2800UL)
#define WWDT_BASE             (APB1PERIPH_BASE + 0x2C00UL)
#define IWDT_BASE             (APB1PERIPH_BASE + 0x3000UL)
#define I2S2ext_BASE          (APB1PERIPH_BASE + 0x3400UL)
#define SPI2_BASE             (APB1PERIPH_BASE + 0x3800UL)
#define SPI3_BASE             (APB1PERIPH_BASE + 0x3C00UL)
#define I2S3ext_BASE          (APB1PERIPH_BASE + 0x4000UL)
#define USART2_BASE           (APB1PERIPH_BASE + 0x4400UL)
#define USART3_BASE           (APB1PERIPH_BASE + 0x4800UL)
#define UART4_BASE            (APB1PERIPH_BASE + 0x4C00UL)
#define UART5_BASE            (APB1PERIPH_BASE + 0x5000UL)
#define I2C1_BASE             (APB1PERIPH_BASE + 0x5400UL)
#define I2C2_BASE             (APB1PERIPH_BASE + 0x5800UL)
#define I2C3_BASE             (APB1PERIPH_BASE + 0x5C00UL)
#define CAN1_BASE             (APB1PERIPH_BASE + 0x6400UL)
#define CAN2_BASE             (APB1PERIPH_BASE + 0x6800UL)
#define PMU_BASE              (APB1PERIPH_BASE + 0x7000UL)

/*!< APB2 peripherals */
#define TMR1_BASE             (APB2PERIPH_BASE + 0x0000UL)
#define TMR8_BASE             (APB2PERIPH_BASE + 0x0400UL)
#define USART1_BASE           (APB2PERIPH_BASE + 0x1000UL)
#define USART6_BASE           (APB2PERIPH_BASE + 0x1400UL)
#define ADC1_BASE             (APB2PERIPH_BASE + 0x2000UL)
#define ADC2_BASE             (APB2PERIPH_BASE + 0x2400UL)
#define ADC1_COMMON_BASE      (APB2PERIPH_BASE + 0x2300UL)
#define ADC2_COMMON_BASE      (APB2PERIPH_BASE + 0x2700UL)
/* Legacy define */
#define SDIO_BASE             (APB2PERIPH_BASE + 0x2C00UL)
#define SPI1_BASE             (APB2PERIPH_BASE + 0x3000UL)
#define SPI4_BASE             (APB2PERIPH_BASE + 0x3400UL)
#define SYSCFG_BASE           (APB2PERIPH_BASE + 0x3800UL)
#define COMP1_BASE            (APB2PERIPH_BASE + 0x3818UL)
#define COMP2_BASE            (APB2PERIPH_BASE + 0x381CUL)
#define EINT_BASE             (APB2PERIPH_BASE + 0x3C00UL)
#define TMR9_BASE             (APB2PERIPH_BASE + 0x4000UL)
#define TMR10_BASE            (APB2PERIPH_BASE + 0x4400UL)
#define TMR11_BASE            (APB2PERIPH_BASE + 0x4800UL)
#define SPI5_BASE             (APB2PERIPH_BASE + 0x5000UL)

/*!< AHB1 peripherals */
#define GPIOA_BASE            (AHB1PERIPH_BASE + 0x0000UL)
#define GPIOB_BASE            (AHB1PERIPH_BASE + 0x0400UL)
#define GPIOC_BASE            (AHB1PERIPH_BASE + 0x0800UL)
#define GPIOD_BASE            (AHB1PERIPH_BASE + 0x0C00UL)
#define GPIOE_BASE            (AHB1PERIPH_BASE + 0x1000UL)
#define GPIOH_BASE            (AHB1PERIPH_BASE + 0x1C00UL)
#define CRC_BASE              (AHB1PERIPH_BASE + 0x3000UL)
#define RCM_BASE              (AHB1PERIPH_BASE + 0x3800UL)
#define FLASH_R_BASE          (AHB1PERIPH_BASE + 0x3C00UL)
#define DMA1_BASE             (AHB1PERIPH_BASE + 0x6000UL)
#define DMA1_Stream0_BASE     (DMA1_BASE + 0x010UL)
#define DMA1_Stream1_BASE     (DMA1_BASE + 0x028UL)
#define DMA1_Stream2_BASE     (DMA1_BASE + 0x040UL)
#define DMA1_Stream3_BASE     (DMA1_BASE + 0x058UL)
#define DMA1_Stream4_BASE     (DMA1_BASE + 0x070UL)
#define DMA1_Stream5_BASE     (DMA1_BASE + 0x088UL)
#define DMA1_Stream6_BASE     (DMA1_BASE + 0x0A0UL)
#define DMA1_Stream7_BASE     (DMA1_BASE + 0x0B8UL)
#define DMA2_BASE             (AHB1PERIPH_BASE + 0x6400UL)
#define DMA2_Stream0_BASE     (DMA2_BASE + 0x010UL)
#define DMA2_Stream1_BASE     (DMA2_BASE + 0x028UL)
#define DMA2_Stream2_BASE     (DMA2_BASE + 0x040UL)
#define DMA2_Stream3_BASE     (DMA2_BASE + 0x058UL)
#define DMA2_Stream4_BASE     (DMA2_BASE + 0x070UL)
#define DMA2_Stream5_BASE     (DMA2_BASE + 0x088UL)
#define DMA2_Stream6_BASE     (DMA2_BASE + 0x0A0UL)
#define DMA2_Stream7_BASE     (DMA2_BASE + 0x0B8UL)

/*!< AHB2 peripherals */
#define RNG_BASE              (AHB2PERIPH_BASE + 0x60800UL)

/*!< SMC Bankx registers base address */
#define SMC_Bank1_R_BASE     (SMC_R_BASE + 0x0000UL)
#define SMC_Bank1E_R_BASE    (SMC_R_BASE + 0x0104UL)
#define SMC_Bank2_3_R_BASE   (SMC_R_BASE + 0x0060UL)
#define SMC_Bank4_R_BASE     (SMC_R_BASE + 0x00A0UL)

/*!< QuadSPI registers base address */
#define QSPI_BASE            0xA0001000UL

/*!< Debug MCU registers base address */
#define DBGMCU_BASE          0xE0042000UL
/*!< USB registers base address */
#define USB_OTG_FS_PERIPH_BASE               0x50000000UL

#define USB_OTG_GLOBAL_BASE                  0x000UL
#define USB_OTG_DEVICE_BASE                  0x800UL
#define USB_OTG_IN_ENDPOINT_BASE             0x900UL
#define USB_OTG_OUT_ENDPOINT_BASE            0xB00UL
#define USB_OTG_EP_REG_SIZE                  0x20UL
#define USB_OTG_HOST_BASE                    0x400UL
#define USB_OTG_HOST_PORT_BASE               0x440UL
#define USB_OTG_HOST_CHANNEL_BASE            0x500UL
#define USB_OTG_HOST_CHANNEL_SIZE            0x20UL
#define USB_OTG_PCGCCTL_BASE                 0xE00UL
#define USB_OTG_FIFO_BASE                    0x1000UL
#define USB_OTG_FIFO_SIZE                    0x1000UL

#define UID_BASE                     0x1FFF7A10UL           /*!< Unique device ID register base address */
#define FLASHSIZE_BASE               0x1FFF7A22UL           /*!< FLASH Size register base address       */
#define PACKAGE_BASE                 0x1FFF7BF0UL           /*!< Package size register base address     */
/**
  * @}
  */

/** @addtogroup Peripheral_declaration
  * @{
  */
#define TMR2                ((TMR_TypeDef *) TMR2_BASE)
#define TMR3                ((TMR_TypeDef *) TMR3_BASE)
#define TMR4                ((TMR_TypeDef *) TMR4_BASE)
#define TMR5                ((TMR_TypeDef *) TMR5_BASE)
#define TMR12               ((TMR_TypeDef *) TMR12_BASE)
#define TMR13               ((TMR_TypeDef *) TMR13_BASE)
#define TMR14               ((TMR_TypeDef *) TMR14_BASE)
#define RTC                 ((RTC_TypeDef *) RTC_BASE)
#define WWDT                ((WWDT_TypeDef *) WWDT_BASE)
#define IWDT                ((IWDT_TypeDef *) IWDT_BASE)
#define I2S2ext             ((SPI_TypeDef *) I2S2ext_BASE)
#define SPI2                ((SPI_TypeDef *) SPI2_BASE)
#define SPI3                ((SPI_TypeDef *) SPI3_BASE)
#define I2S3ext             ((SPI_TypeDef *) I2S3ext_BASE)
#define USART2              ((USART_TypeDef *) USART2_BASE)
#define USART3              ((USART_TypeDef *) USART3_BASE)
#define UART4               ((USART_TypeDef *) UART4_BASE)
#define UART5               ((USART_TypeDef *) UART5_BASE)
#define I2C1                ((I2C_TypeDef *) I2C1_BASE)
#define I2C2                ((I2C_TypeDef *) I2C2_BASE)
#define I2C3                ((I2C_TypeDef *) I2C3_BASE)
#define CAN1                ((CAN_TypeDef *) CAN1_BASE)
#define CAN2                ((CAN_TypeDef *) CAN2_BASE)
#define PMU                 ((PMU_TypeDef *) PMU_BASE)
#define TMR1                ((TMR_TypeDef *) TMR1_BASE)
#define TMR8                ((TMR_TypeDef *) TMR8_BASE)
#define USART1              ((USART_TypeDef *) USART1_BASE)
#define USART6              ((USART_TypeDef *) USART6_BASE)
#define ADC1                ((ADC_TypeDef *) ADC1_BASE)
#define ADC2                ((ADC_TypeDef *) ADC2_BASE)
#define ADC1_COMMON         ((ADC_Common_TypeDef *) ADC1_COMMON_BASE)
#define ADC2_COMMON         ((ADC_Common_TypeDef *) ADC2_COMMON_BASE)
/* Legacy define */
#define ADC1_C              ADC1_COMMON
#define ADC2_C              ADC2_COMMON
#define SDIO                ((SDIO_TypeDef *) SDIO_BASE)
#define SPI1                ((SPI_TypeDef *) SPI1_BASE)
#define SPI4                ((SPI_TypeDef *) SPI4_BASE)
#define SYSCFG              ((SYSCFG_TypeDef *) SYSCFG_BASE)
#define COMP1               ((COMP_TypeDef *) COMP1_BASE)
#define COMP2               ((COMP_TypeDef *) COMP2_BASE)
#define COMP12_COMMON       ((COMP_Common_TypeDef *) COMP1_BASE)
#define EINT                ((EINT_TypeDef *) EINT_BASE)
#define TMR9                ((TMR_TypeDef *) TMR9_BASE)
#define TMR10               ((TMR_TypeDef *) TMR10_BASE)
#define TMR11               ((TMR_TypeDef *) TMR11_BASE)
#define SPI5                ((SPI_TypeDef *) SPI5_BASE)
#define GPIOA               ((GPIO_TypeDef *) GPIOA_BASE)
#define GPIOB               ((GPIO_TypeDef *) GPIOB_BASE)
#define GPIOC               ((GPIO_TypeDef *) GPIOC_BASE)
#define GPIOD               ((GPIO_TypeDef *) GPIOD_BASE)
#define GPIOE               ((GPIO_TypeDef *) GPIOE_BASE)
#define GPIOH               ((GPIO_TypeDef *) GPIOH_BASE)
#define CRC                 ((CRC_TypeDef *) CRC_BASE)
#define RCM                 ((RCM_TypeDef *) RCM_BASE)
#define FLASH               ((FLASH_TypeDef *) FLASH_R_BASE)
#define DMA1                ((DMA_TypeDef *) DMA1_BASE)
#define DMA1_Stream0        ((DMA_Stream_TypeDef *) DMA1_Stream0_BASE)
#define DMA1_Stream1        ((DMA_Stream_TypeDef *) DMA1_Stream1_BASE)
#define DMA1_Stream2        ((DMA_Stream_TypeDef *) DMA1_Stream2_BASE)
#define DMA1_Stream3        ((DMA_Stream_TypeDef *) DMA1_Stream3_BASE)
#define DMA1_Stream4        ((DMA_Stream_TypeDef *) DMA1_Stream4_BASE)
#define DMA1_Stream5        ((DMA_Stream_TypeDef *) DMA1_Stream5_BASE)
#define DMA1_Stream6        ((DMA_Stream_TypeDef *) DMA1_Stream6_BASE)
#define DMA1_Stream7        ((DMA_Stream_TypeDef *) DMA1_Stream7_BASE)
#define DMA2                ((DMA_TypeDef *) DMA2_BASE)
#define DMA2_Stream0        ((DMA_Stream_TypeDef *) DMA2_Stream0_BASE)
#define DMA2_Stream1        ((DMA_Stream_TypeDef *) DMA2_Stream1_BASE)
#define DMA2_Stream2        ((DMA_Stream_TypeDef *) DMA2_Stream2_BASE)
#define DMA2_Stream3        ((DMA_Stream_TypeDef *) DMA2_Stream3_BASE)
#define DMA2_Stream4        ((DMA_Stream_TypeDef *) DMA2_Stream4_BASE)
#define DMA2_Stream5        ((DMA_Stream_TypeDef *) DMA2_Stream5_BASE)
#define DMA2_Stream6        ((DMA_Stream_TypeDef *) DMA2_Stream6_BASE)
#define DMA2_Stream7        ((DMA_Stream_TypeDef *) DMA2_Stream7_BASE)
#define RNG                 ((RNG_TypeDef *) RNG_BASE)
#define QSPI                ((QSPI_TypeDef *) QSPI_BASE)
#define SMC_Bank1           ((SMC_Bank1_TypeDef *) SMC_Bank1_R_BASE)
#define SMC_Bank1E          ((SMC_Bank1E_TypeDef *) SMC_Bank1E_R_BASE)
#define SMC_Bank2_3         ((SMC_Bank2_3_TypeDef *) SMC_Bank2_3_R_BASE)
#define SMC_Bank4           ((SMC_Bank4_TypeDef *) SMC_Bank4_R_BASE)
#define DBGMCU              ((DBGMCU_TypeDef *) DBGMCU_BASE)
#define USB_OTG_FS          ((USB_OTG_GlobalTypeDef *) USB_OTG_FS_PERIPH_BASE)

/**
  * @}
  */

/** @addtogroup Exported_constants
  * @{
  */

/** @addtogroup Hardware_Constant_Definition
  * @{
  */
#define LSI_STARTUP_TIME                40U /*!< LSI Maximum startup time in us */
/**
  * @}
  */

  /** @addtogroup Peripheral_Registers_Bits_Definition
  * @{
  */

/******************************************************************************/
/*                         Peripheral Registers_Bits_Definition               */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*                        Analog to Digital Converter                         */
/*                                                                            */
/******************************************************************************/
/*
 * @brief Specific device feature definitions (not present on all devices in the APM32F4 serie)
 */

/********************  Bit definition for ADC_STS register  ********************/
#define ADC_STS_AWDFLG_Pos                                          (0U)
#define ADC_STS_AWDFLG_Msk                                          (0x1UL << ADC_STS_AWDFLG_Pos)                                  /*!< 0x00000001 */
#define ADC_STS_AWDFLG                                              ADC_STS_AWDFLG_Msk                                             /*!<Analog watchdog flag */
#define ADC_STS_EOCFLG_Pos                                          (1U)
#define ADC_STS_EOCFLG_Msk                                          (0x1UL << ADC_STS_EOCFLG_Pos)                                  /*!< 0x00000002 */
#define ADC_STS_EOCFLG                                              ADC_STS_EOCFLG_Msk                                             /*!<End of conversion */
#define ADC_STS_INJEOCFLG_Pos                                       (2U)
#define ADC_STS_INJEOCFLG_Msk                                       (0x1UL << ADC_STS_INJEOCFLG_Pos)                               /*!< 0x00000004 */
#define ADC_STS_INJEOCFLG                                           ADC_STS_INJEOCFLG_Msk                                          /*!<Injected channel end of conversion */
#define ADC_STS_INJCSFLG_Pos                                        (3U)
#define ADC_STS_INJCSFLG_Msk                                        (0x1UL << ADC_STS_INJCSFLG_Pos)                                /*!< 0x00000008 */
#define ADC_STS_INJCSFLG                                            ADC_STS_INJCSFLG_Msk                                           /*!<Injected channel Start flag */
#define ADC_STS_REGCSFLG_Pos                                        (4U)
#define ADC_STS_REGCSFLG_Msk                                        (0x1UL << ADC_STS_REGCSFLG_Pos)                                /*!< 0x00000010 */
#define ADC_STS_REGCSFLG                                            ADC_STS_REGCSFLG_Msk                                           /*!<Regular channel Start flag */
#define ADC_STS_OVRFLG_Pos                                          (5U)
#define ADC_STS_OVRFLG_Msk                                          (0x1UL << ADC_STS_OVRFLG_Pos)                                  /*!< 0x00000020 */
#define ADC_STS_OVRFLG                                              ADC_STS_OVRFLG_Msk                                             /*!<Overrun flag */

/*******************  Bit definition for ADC_CTRL1 register  ********************/
#define ADC_CTRL1_AWDCHSEL_Pos                                      (0U)
#define ADC_CTRL1_AWDCHSEL_Msk                                      (0x1FUL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x0000001F */
#define ADC_CTRL1_AWDCHSEL                                          ADC_CTRL1_AWDCHSEL_Msk                                         /*!<AWDCH[4:0] bits (Analog watchdog channel select bits) */
#define ADC_CTRL1_AWDCHSEL_0                                        (0x01UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000001 */
#define ADC_CTRL1_AWDCHSEL_1                                        (0x02UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000002 */
#define ADC_CTRL1_AWDCHSEL_2                                        (0x04UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000004 */
#define ADC_CTRL1_AWDCHSEL_3                                        (0x08UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000008 */
#define ADC_CTRL1_AWDCHSEL_4                                        (0x10UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000010 */
#define ADC_CTRL1_EOCIEN_Pos                                        (5U)
#define ADC_CTRL1_EOCIEN_Msk                                        (0x1UL << ADC_CTRL1_EOCIEN_Pos)                                /*!< 0x00000020 */
#define ADC_CTRL1_EOCIEN                                            ADC_CTRL1_EOCIEN_Msk                                           /*!<Interrupt enable for EOC */
#define ADC_CTRL1_AWDIEN_Pos                                        (6U)
#define ADC_CTRL1_AWDIEN_Msk                                        (0x1UL << ADC_CTRL1_AWDIEN_Pos)                                /*!< 0x00000040 */
#define ADC_CTRL1_AWDIEN                                            ADC_CTRL1_AWDIEN_Msk                                           /*!<AAnalog Watchdog interrupt enable */
#define ADC_CTRL1_INJEOCIEN_Pos                                     (7U)
#define ADC_CTRL1_INJEOCIEN_Msk                                     (0x1UL << ADC_CTRL1_INJEOCIEN_Pos)                             /*!< 0x00000080 */
#define ADC_CTRL1_INJEOCIEN                                         ADC_CTRL1_INJEOCIEN_Msk                                        /*!<Interrupt enable for injected channels */
#define ADC_CTRL1_SCANEN_Pos                                        (8U)
#define ADC_CTRL1_SCANEN_Msk                                        (0x1UL << ADC_CTRL1_SCANEN_Pos)                                /*!< 0x00000100 */
#define ADC_CTRL1_SCANEN                                            ADC_CTRL1_SCANEN_Msk                                           /*!<Scan mode */
#define ADC_CTRL1_AWDSGLEN_Pos                                      (9U)
#define ADC_CTRL1_AWDSGLEN_Msk                                      (0x1UL << ADC_CTRL1_AWDSGLEN_Pos)                              /*!< 0x00000200 */
#define ADC_CTRL1_AWDSGLEN                                          ADC_CTRL1_AWDSGLEN_Msk                                         /*!<Enable the watchdog on a single channel in scan mode */
#define ADC_CTRL1_INJGACEN_Pos                                      (10U)
#define ADC_CTRL1_INJGACEN_Msk                                      (0x1UL << ADC_CTRL1_INJGACEN_Pos)                              /*!< 0x00000400 */
#define ADC_CTRL1_INJGACEN                                          ADC_CTRL1_INJGACEN_Msk                                         /*!<Automatic injected group conversion */
#define ADC_CTRL1_REGDISCEN_Pos                                     (11U)
#define ADC_CTRL1_REGDISCEN_Msk                                     (0x1UL << ADC_CTRL1_REGDISCEN_Pos)                             /*!< 0x00000800 */
#define ADC_CTRL1_REGDISCEN                                         ADC_CTRL1_REGDISCEN_Msk                                        /*!<Discontinuous mode on regular channels */
#define ADC_CTRL1_INJDISCEN_Pos                                     (12U)
#define ADC_CTRL1_INJDISCEN_Msk                                     (0x1UL << ADC_CTRL1_INJDISCEN_Pos)                             /*!< 0x00001000 */
#define ADC_CTRL1_INJDISCEN                                         ADC_CTRL1_INJDISCEN_Msk                                        /*!<Discontinuous mode on injected channels */
#define ADC_CTRL1_DISCNUMCFG_Pos                                    (13U)
#define ADC_CTRL1_DISCNUMCFG_Msk                                    (0x7UL << ADC_CTRL1_DISCNUMCFG_Pos)                            /*!< 0x0000E000 */
#define ADC_CTRL1_DISCNUMCFG                                        ADC_CTRL1_DISCNUMCFG_Msk                                       /*!<DISCNUM[2:0] bits (Discontinuous mode channel count) */
#define ADC_CTRL1_DISCNUMCFG_0                                      (0x1UL << ADC_CTRL1_DISCNUMCFG_Pos)                            /*!< 0x00002000 */
#define ADC_CTRL1_DISCNUMCFG_1                                      (0x2UL << ADC_CTRL1_DISCNUMCFG_Pos)                            /*!< 0x00004000 */
#define ADC_CTRL1_DISCNUMCFG_2                                      (0x4UL << ADC_CTRL1_DISCNUMCFG_Pos)                            /*!< 0x00008000 */
#define ADC_CTRL1_INJAWDEN_Pos                                      (22U)
#define ADC_CTRL1_INJAWDEN_Msk                                      (0x1UL << ADC_CTRL1_INJAWDEN_Pos)                              /*!< 0x00400000 */
#define ADC_CTRL1_INJAWDEN                                          ADC_CTRL1_INJAWDEN_Msk                                         /*!<Analog watchdog enable on injected channels */
#define ADC_CTRL1_REGAWDEN_Pos                                      (23U)
#define ADC_CTRL1_REGAWDEN_Msk                                      (0x1UL << ADC_CTRL1_REGAWDEN_Pos)                              /*!< 0x00800000 */
#define ADC_CTRL1_REGAWDEN                                          ADC_CTRL1_REGAWDEN_Msk                                         /*!<Analog watchdog enable on regular channels */
#define ADC_CTRL1_RESSEL_Pos                                        (24U)
#define ADC_CTRL1_RESSEL_Msk                                        (0x3UL << ADC_CTRL1_RESSEL_Pos)                                /*!< 0x03000000 */
#define ADC_CTRL1_RESSEL                                            ADC_CTRL1_RESSEL_Msk                                           /*!<RES[2:0] bits (Resolution) */
#define ADC_CTRL1_RESSEL_0                                          (0x1UL << ADC_CTRL1_RESSEL_Pos)                                /*!< 0x01000000 */
#define ADC_CTRL1_RESSEL_1                                          (0x2UL << ADC_CTRL1_RESSEL_Pos)                                /*!< 0x02000000 */
#define ADC_CTRL1_OVRIEN_Pos                                        (26U)
#define ADC_CTRL1_OVRIEN_Msk                                        (0x1UL << ADC_CTRL1_OVRIEN_Pos)                                /*!< 0x04000000 */
#define ADC_CTRL1_OVRIEN                                            ADC_CTRL1_OVRIEN_Msk                                           /*!<overrun interrupt enable */

/*******************  Bit definition for ADC_CTRL2 register  ********************/
#define ADC_CTRL2_ADCEN_Pos                                         (0U)
#define ADC_CTRL2_ADCEN_Msk                                         (0x1UL << ADC_CTRL2_ADCEN_Pos)                                 /*!< 0x00000001 */
#define ADC_CTRL2_ADCEN                                             ADC_CTRL2_ADCEN_Msk                                            /*!<A/D Converter ON / OFF */
#define ADC_CTRL2_CONTCEN_Pos                                       (1U)
#define ADC_CTRL2_CONTCEN_Msk                                       (0x1UL << ADC_CTRL2_CONTCEN_Pos)                               /*!< 0x00000002 */
#define ADC_CTRL2_CONTCEN                                           ADC_CTRL2_CONTCEN_Msk                                          /*!<Continuous Conversion */
#define ADC_CTRL2_DMAEN_Pos                                         (8U)
#define ADC_CTRL2_DMAEN_Msk                                         (0x1UL << ADC_CTRL2_DMAEN_Pos)                                 /*!< 0x00000100 */
#define ADC_CTRL2_DMAEN                                             ADC_CTRL2_DMAEN_Msk                                            /*!<Direct Memory access mode */
#define ADC_CTRL2_DMADISSEL_Pos                                     (9U)
#define ADC_CTRL2_DMADISSEL_Msk                                     (0x1UL << ADC_CTRL2_DMADISSEL_Pos)                             /*!< 0x00000200 */
#define ADC_CTRL2_DMADISSEL                                         ADC_CTRL2_DMADISSEL_Msk                                        /*!<DMA disable selection (Single ADC) */
#define ADC_CTRL2_EOCSEL_Pos                                        (10U)
#define ADC_CTRL2_EOCSEL_Msk                                        (0x1UL << ADC_CTRL2_EOCSEL_Pos)                                /*!< 0x00000400 */
#define ADC_CTRL2_EOCSEL                                            ADC_CTRL2_EOCSEL_Msk                                           /*!<End of conversion selection */
#define ADC_CTRL2_DALIGNCFG_Pos                                     (11U)
#define ADC_CTRL2_DALIGNCFG_Msk                                     (0x1UL << ADC_CTRL2_DALIGNCFG_Pos)                             /*!< 0x00000800 */
#define ADC_CTRL2_DALIGNCFG                                         ADC_CTRL2_DALIGNCFG_Msk                                        /*!<Data Alignment */
#define ADC_CTRL2_INJGEXTTRGSEL_Pos                                 (16U)
#define ADC_CTRL2_INJGEXTTRGSEL_Msk                                 (0xFUL << ADC_CTRL2_INJGEXTTRGSEL_Pos)                         /*!< 0x000F0000 */
#define ADC_CTRL2_INJGEXTTRGSEL                                     ADC_CTRL2_INJGEXTTRGSEL_Msk                                    /*!<JEXTSEL[3:0] bits (External event select for injected group) */
#define ADC_CTRL2_INJGEXTTRGSEL_0                                   (0x1UL << ADC_CTRL2_INJGEXTTRGSEL_Pos)                         /*!< 0x00010000 */
#define ADC_CTRL2_INJGEXTTRGSEL_1                                   (0x2UL << ADC_CTRL2_INJGEXTTRGSEL_Pos)                         /*!< 0x00020000 */
#define ADC_CTRL2_INJGEXTTRGSEL_2                                   (0x4UL << ADC_CTRL2_INJGEXTTRGSEL_Pos)                         /*!< 0x00040000 */
#define ADC_CTRL2_INJGEXTTRGSEL_3                                   (0x8UL << ADC_CTRL2_INJGEXTTRGSEL_Pos)                         /*!< 0x00080000 */
#define ADC_CTRL2_INJEXTTRGEN_Pos                                   (20U)
#define ADC_CTRL2_INJEXTTRGEN_Msk                                   (0x3UL << ADC_CTRL2_INJEXTTRGEN_Pos)                           /*!< 0x00300000 */
#define ADC_CTRL2_INJEXTTRGEN                                       ADC_CTRL2_INJEXTTRGEN_Msk                                      /*!<JEXTEN[1:0] bits (External Trigger Conversion mode for injected channelsp) */
#define ADC_CTRL2_INJEXTTRGEN_0                                     (0x1UL << ADC_CTRL2_INJEXTTRGEN_Pos)                           /*!< 0x00100000 */
#define ADC_CTRL2_INJEXTTRGEN_1                                     (0x2UL << ADC_CTRL2_INJEXTTRGEN_Pos)                           /*!< 0x00200000 */
#define ADC_CTRL2_INJSWSC_Pos                                       (22U)
#define ADC_CTRL2_INJSWSC_Msk                                       (0x1UL << ADC_CTRL2_INJSWSC_Pos)                               /*!< 0x00400000 */
#define ADC_CTRL2_INJSWSC                                           ADC_CTRL2_INJSWSC_Msk                                          /*!<Start Conversion of injected channels */
#define ADC_CTRL2_REGEXTTRGSEL_Pos                                  (24U)
#define ADC_CTRL2_REGEXTTRGSEL_Msk                                  (0xFUL << ADC_CTRL2_REGEXTTRGSEL_Pos)                          /*!< 0x0F000000 */
#define ADC_CTRL2_REGEXTTRGSEL                                      ADC_CTRL2_REGEXTTRGSEL_Msk                                     /*!<EXTSEL[3:0] bits (External Event Select for regular group) */
#define ADC_CTRL2_REGEXTTRGSEL_0                                    (0x1UL << ADC_CTRL2_REGEXTTRGSEL_Pos)                          /*!< 0x01000000 */
#define ADC_CTRL2_REGEXTTRGSEL_1                                    (0x2UL << ADC_CTRL2_REGEXTTRGSEL_Pos)                          /*!< 0x02000000 */
#define ADC_CTRL2_REGEXTTRGSEL_2                                    (0x4UL << ADC_CTRL2_REGEXTTRGSEL_Pos)                          /*!< 0x04000000 */
#define ADC_CTRL2_REGEXTTRGSEL_3                                    (0x8UL << ADC_CTRL2_REGEXTTRGSEL_Pos)                          /*!< 0x08000000 */
#define ADC_CTRL2_REGEXTTRGEN_Pos                                   (28U)
#define ADC_CTRL2_REGEXTTRGEN_Msk                                   (0x3UL << ADC_CTRL2_REGEXTTRGEN_Pos)                           /*!< 0x30000000 */
#define ADC_CTRL2_REGEXTTRGEN                                       ADC_CTRL2_REGEXTTRGEN_Msk                                      /*!<EXTEN[1:0] bits (External Trigger Conversion mode for regular channelsp) */
#define ADC_CTRL2_REGEXTTRGEN_0                                     (0x1UL << ADC_CTRL2_REGEXTTRGEN_Pos)                           /*!< 0x10000000 */
#define ADC_CTRL2_REGEXTTRGEN_1                                     (0x2UL << ADC_CTRL2_REGEXTTRGEN_Pos)                           /*!< 0x20000000 */
#define ADC_CTRL2_REGSWSC_Pos                                       (30U)
#define ADC_CTRL2_REGSWSC_Msk                                       (0x1UL << ADC_CTRL2_REGSWSC_Pos)                               /*!< 0x40000000 */
#define ADC_CTRL2_REGSWSC                                           ADC_CTRL2_REGSWSC_Msk                                          /*!<Start Conversion of regular channels */

/******************  Bit definition for ADC_SMPTIM1 register  *******************/
#define ADC_SMPTIM1_SMPCYCCFG10_Pos                                 (0U)
#define ADC_SMPTIM1_SMPCYCCFG10_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG10_Pos)                         /*!< 0x00000007 */
#define ADC_SMPTIM1_SMPCYCCFG10                                     ADC_SMPTIM1_SMPCYCCFG10_Msk                                    /*!<SMP10[2:0] bits (Channel 10 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG10_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG10_Pos)                         /*!< 0x00000001 */
#define ADC_SMPTIM1_SMPCYCCFG10_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG10_Pos)                         /*!< 0x00000002 */
#define ADC_SMPTIM1_SMPCYCCFG10_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG10_Pos)                         /*!< 0x00000004 */
#define ADC_SMPTIM1_SMPCYCCFG11_Pos                                 (3U)
#define ADC_SMPTIM1_SMPCYCCFG11_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG11_Pos)                         /*!< 0x00000038 */
#define ADC_SMPTIM1_SMPCYCCFG11                                     ADC_SMPTIM1_SMPCYCCFG11_Msk                                    /*!<SMP11[2:0] bits (Channel 11 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG11_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG11_Pos)                         /*!< 0x00000008 */
#define ADC_SMPTIM1_SMPCYCCFG11_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG11_Pos)                         /*!< 0x00000010 */
#define ADC_SMPTIM1_SMPCYCCFG11_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG11_Pos)                         /*!< 0x00000020 */
#define ADC_SMPTIM1_SMPCYCCFG12_Pos                                 (6U)
#define ADC_SMPTIM1_SMPCYCCFG12_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG12_Pos)                         /*!< 0x000001C0 */
#define ADC_SMPTIM1_SMPCYCCFG12                                     ADC_SMPTIM1_SMPCYCCFG12_Msk                                    /*!<SMP12[2:0] bits (Channel 12 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG12_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG12_Pos)                         /*!< 0x00000040 */
#define ADC_SMPTIM1_SMPCYCCFG12_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG12_Pos)                         /*!< 0x00000080 */
#define ADC_SMPTIM1_SMPCYCCFG12_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG12_Pos)                         /*!< 0x00000100 */
#define ADC_SMPTIM1_SMPCYCCFG13_Pos                                 (9U)
#define ADC_SMPTIM1_SMPCYCCFG13_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG13_Pos)                         /*!< 0x00000E00 */
#define ADC_SMPTIM1_SMPCYCCFG13                                     ADC_SMPTIM1_SMPCYCCFG13_Msk                                    /*!<SMP13[2:0] bits (Channel 13 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG13_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG13_Pos)                         /*!< 0x00000200 */
#define ADC_SMPTIM1_SMPCYCCFG13_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG13_Pos)                         /*!< 0x00000400 */
#define ADC_SMPTIM1_SMPCYCCFG13_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG13_Pos)                         /*!< 0x00000800 */
#define ADC_SMPTIM1_SMPCYCCFG14_Pos                                 (12U)
#define ADC_SMPTIM1_SMPCYCCFG14_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG14_Pos)                         /*!< 0x00007000 */
#define ADC_SMPTIM1_SMPCYCCFG14                                     ADC_SMPTIM1_SMPCYCCFG14_Msk                                    /*!<SMP14[2:0] bits (Channel 14 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG14_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG14_Pos)                         /*!< 0x00001000 */
#define ADC_SMPTIM1_SMPCYCCFG14_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG14_Pos)                         /*!< 0x00002000 */
#define ADC_SMPTIM1_SMPCYCCFG14_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG14_Pos)                         /*!< 0x00004000 */
#define ADC_SMPTIM1_SMPCYCCFG15_Pos                                 (15U)
#define ADC_SMPTIM1_SMPCYCCFG15_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG15_Pos)                         /*!< 0x00038000 */
#define ADC_SMPTIM1_SMPCYCCFG15                                     ADC_SMPTIM1_SMPCYCCFG15_Msk                                    /*!<SMP15[2:0] bits (Channel 15 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG15_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG15_Pos)                         /*!< 0x00008000 */
#define ADC_SMPTIM1_SMPCYCCFG15_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG15_Pos)                         /*!< 0x00010000 */
#define ADC_SMPTIM1_SMPCYCCFG15_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG15_Pos)                         /*!< 0x00020000 */
#define ADC_SMPTIM1_SMPCYCCFG16_Pos                                 (18U)
#define ADC_SMPTIM1_SMPCYCCFG16_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG16_Pos)                         /*!< 0x001C0000 */
#define ADC_SMPTIM1_SMPCYCCFG16                                     ADC_SMPTIM1_SMPCYCCFG16_Msk                                    /*!<SMP16[2:0] bits (Channel 16 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG16_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG16_Pos)                         /*!< 0x00040000 */
#define ADC_SMPTIM1_SMPCYCCFG16_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG16_Pos)                         /*!< 0x00080000 */
#define ADC_SMPTIM1_SMPCYCCFG16_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG16_Pos)                         /*!< 0x00100000 */
#define ADC_SMPTIM1_SMPCYCCFG17_Pos                                 (21U)
#define ADC_SMPTIM1_SMPCYCCFG17_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG17_Pos)                         /*!< 0x00E00000 */
#define ADC_SMPTIM1_SMPCYCCFG17                                     ADC_SMPTIM1_SMPCYCCFG17_Msk                                    /*!<SMP17[2:0] bits (Channel 17 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG17_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG17_Pos)                         /*!< 0x00200000 */
#define ADC_SMPTIM1_SMPCYCCFG17_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG17_Pos)                         /*!< 0x00400000 */
#define ADC_SMPTIM1_SMPCYCCFG17_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG17_Pos)                         /*!< 0x00800000 */
#define ADC_SMPTIM1_SMPCYCCFG18_Pos                                 (24U)
#define ADC_SMPTIM1_SMPCYCCFG18_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG18_Pos)                         /*!< 0x07000000 */
#define ADC_SMPTIM1_SMPCYCCFG18                                     ADC_SMPTIM1_SMPCYCCFG18_Msk                                    /*!<SMP18[2:0] bits (Channel 18 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG18_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG18_Pos)                         /*!< 0x01000000 */
#define ADC_SMPTIM1_SMPCYCCFG18_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG18_Pos)                         /*!< 0x02000000 */
#define ADC_SMPTIM1_SMPCYCCFG18_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG18_Pos)                         /*!< 0x04000000 */

/******************  Bit definition for ADC_SMPTIM2 register  *******************/
#define ADC_SMPTIM2_SMPCYCCFG0_Pos                                  (0U)
#define ADC_SMPTIM2_SMPCYCCFG0_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG0_Pos)                          /*!< 0x00000007 */
#define ADC_SMPTIM2_SMPCYCCFG0                                      ADC_SMPTIM2_SMPCYCCFG0_Msk                                     /*!<SMP0[2:0] bits (Channel 0 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG0_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG0_Pos)                          /*!< 0x00000001 */
#define ADC_SMPTIM2_SMPCYCCFG0_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG0_Pos)                          /*!< 0x00000002 */
#define ADC_SMPTIM2_SMPCYCCFG0_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG0_Pos)                          /*!< 0x00000004 */
#define ADC_SMPTIM2_SMPCYCCFG1_Pos                                  (3U)
#define ADC_SMPTIM2_SMPCYCCFG1_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG1_Pos)                          /*!< 0x00000038 */
#define ADC_SMPTIM2_SMPCYCCFG1                                      ADC_SMPTIM2_SMPCYCCFG1_Msk                                     /*!<SMP1[2:0] bits (Channel 1 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG1_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG1_Pos)                          /*!< 0x00000008 */
#define ADC_SMPTIM2_SMPCYCCFG1_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG1_Pos)                          /*!< 0x00000010 */
#define ADC_SMPTIM2_SMPCYCCFG1_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG1_Pos)                          /*!< 0x00000020 */
#define ADC_SMPTIM2_SMPCYCCFG2_Pos                                  (6U)
#define ADC_SMPTIM2_SMPCYCCFG2_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG2_Pos)                          /*!< 0x000001C0 */
#define ADC_SMPTIM2_SMPCYCCFG2                                      ADC_SMPTIM2_SMPCYCCFG2_Msk                                     /*!<SMP2[2:0] bits (Channel 2 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG2_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG2_Pos)                          /*!< 0x00000040 */
#define ADC_SMPTIM2_SMPCYCCFG2_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG2_Pos)                          /*!< 0x00000080 */
#define ADC_SMPTIM2_SMPCYCCFG2_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG2_Pos)                          /*!< 0x00000100 */
#define ADC_SMPTIM2_SMPCYCCFG3_Pos                                  (9U)
#define ADC_SMPTIM2_SMPCYCCFG3_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG3_Pos)                          /*!< 0x00000E00 */
#define ADC_SMPTIM2_SMPCYCCFG3                                      ADC_SMPTIM2_SMPCYCCFG3_Msk                                     /*!<SMP3[2:0] bits (Channel 3 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG3_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG3_Pos)                          /*!< 0x00000200 */
#define ADC_SMPTIM2_SMPCYCCFG3_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG3_Pos)                          /*!< 0x00000400 */
#define ADC_SMPTIM2_SMPCYCCFG3_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG3_Pos)                          /*!< 0x00000800 */
#define ADC_SMPTIM2_SMPCYCCFG4_Pos                                  (12U)
#define ADC_SMPTIM2_SMPCYCCFG4_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG4_Pos)                          /*!< 0x00007000 */
#define ADC_SMPTIM2_SMPCYCCFG4                                      ADC_SMPTIM2_SMPCYCCFG4_Msk                                     /*!<SMP4[2:0] bits (Channel 4 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG4_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG4_Pos)                          /*!< 0x00001000 */
#define ADC_SMPTIM2_SMPCYCCFG4_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG4_Pos)                          /*!< 0x00002000 */
#define ADC_SMPTIM2_SMPCYCCFG4_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG4_Pos)                          /*!< 0x00004000 */
#define ADC_SMPTIM2_SMPCYCCFG5_Pos                                  (15U)
#define ADC_SMPTIM2_SMPCYCCFG5_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG5_Pos)                          /*!< 0x00038000 */
#define ADC_SMPTIM2_SMPCYCCFG5                                      ADC_SMPTIM2_SMPCYCCFG5_Msk                                     /*!<SMP5[2:0] bits (Channel 5 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG5_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG5_Pos)                          /*!< 0x00008000 */
#define ADC_SMPTIM2_SMPCYCCFG5_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG5_Pos)                          /*!< 0x00010000 */
#define ADC_SMPTIM2_SMPCYCCFG5_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG5_Pos)                          /*!< 0x00020000 */
#define ADC_SMPTIM2_SMPCYCCFG6_Pos                                  (18U)
#define ADC_SMPTIM2_SMPCYCCFG6_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG6_Pos)                          /*!< 0x001C0000 */
#define ADC_SMPTIM2_SMPCYCCFG6                                      ADC_SMPTIM2_SMPCYCCFG6_Msk                                     /*!<SMP6[2:0] bits (Channel 6 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG6_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG6_Pos)                          /*!< 0x00040000 */
#define ADC_SMPTIM2_SMPCYCCFG6_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG6_Pos)                          /*!< 0x00080000 */
#define ADC_SMPTIM2_SMPCYCCFG6_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG6_Pos)                          /*!< 0x00100000 */
#define ADC_SMPTIM2_SMPCYCCFG7_Pos                                  (21U)
#define ADC_SMPTIM2_SMPCYCCFG7_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG7_Pos)                          /*!< 0x00E00000 */
#define ADC_SMPTIM2_SMPCYCCFG7                                      ADC_SMPTIM2_SMPCYCCFG7_Msk                                     /*!<SMP7[2:0] bits (Channel 7 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG7_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG7_Pos)                          /*!< 0x00200000 */
#define ADC_SMPTIM2_SMPCYCCFG7_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG7_Pos)                          /*!< 0x00400000 */
#define ADC_SMPTIM2_SMPCYCCFG7_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG7_Pos)                          /*!< 0x00800000 */
#define ADC_SMPTIM2_SMPCYCCFG8_Pos                                  (24U)
#define ADC_SMPTIM2_SMPCYCCFG8_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG8_Pos)                          /*!< 0x07000000 */
#define ADC_SMPTIM2_SMPCYCCFG8                                      ADC_SMPTIM2_SMPCYCCFG8_Msk                                     /*!<SMP8[2:0] bits (Channel 8 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG8_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG8_Pos)                          /*!< 0x01000000 */
#define ADC_SMPTIM2_SMPCYCCFG8_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG8_Pos)                          /*!< 0x02000000 */
#define ADC_SMPTIM2_SMPCYCCFG8_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG8_Pos)                          /*!< 0x04000000 */
#define ADC_SMPTIM2_SMPCYCCFG9_Pos                                  (27U)
#define ADC_SMPTIM2_SMPCYCCFG9_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG9_Pos)                          /*!< 0x38000000 */
#define ADC_SMPTIM2_SMPCYCCFG9                                      ADC_SMPTIM2_SMPCYCCFG9_Msk                                     /*!<SMP9[2:0] bits (Channel 9 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG9_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG9_Pos)                          /*!< 0x08000000 */
#define ADC_SMPTIM2_SMPCYCCFG9_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG9_Pos)                          /*!< 0x10000000 */
#define ADC_SMPTIM2_SMPCYCCFG9_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG9_Pos)                          /*!< 0x20000000 */

/******************  Bit definition for ADC_INJDOF1 register  *******************/
#define ADC_INJDOF1_INJDOF1_Pos                                     (0U)
#define ADC_INJDOF1_INJDOF1_Msk                                     (0xFFFUL << ADC_INJDOF1_INJDOF1_Pos)                           /*!< 0x00000FFF */
#define ADC_INJDOF1_INJDOF1                                         ADC_INJDOF1_INJDOF1_Msk                                        /*!<Data offset for injected channel 1 */

/******************  Bit definition for ADC_INJDOF2 register  *******************/
#define ADC_INJDOF2_INJDOF2_Pos                                     (0U)
#define ADC_INJDOF2_INJDOF2_Msk                                     (0xFFFUL << ADC_INJDOF2_INJDOF2_Pos)                           /*!< 0x00000FFF */
#define ADC_INJDOF2_INJDOF2                                         ADC_INJDOF2_INJDOF2_Msk                                        /*!<Data offset for injected channel 2 */

/******************  Bit definition for ADC_INJDOF3 register  *******************/
#define ADC_INJDOF3_INJDOF3_Pos                                     (0U)
#define ADC_INJDOF3_INJDOF3_Msk                                     (0xFFFUL << ADC_INJDOF3_INJDOF3_Pos)                           /*!< 0x00000FFF */
#define ADC_INJDOF3_INJDOF3                                         ADC_INJDOF3_INJDOF3_Msk                                        /*!<Data offset for injected channel 3 */

/******************  Bit definition for ADC_INJDOF4 register  *******************/
#define ADC_INJDOF4_INJDOF4_Pos                                     (0U)
#define ADC_INJDOF4_INJDOF4_Msk                                     (0xFFFUL << ADC_INJDOF4_INJDOF4_Pos)                           /*!< 0x00000FFF */
#define ADC_INJDOF4_INJDOF4                                         ADC_INJDOF4_INJDOF4_Msk                                        /*!<Data offset for injected channel 4 */

/*******************  Bit definition for ADC_AWDHT register  ********************/
#define ADC_AWDHT_AWDHT_Pos                                         (0U)
#define ADC_AWDHT_AWDHT_Msk                                         (0xFFFUL << ADC_AWDHT_AWDHT_Pos)                               /*!< 0x00000FFF */
#define ADC_AWDHT_AWDHT                                             ADC_AWDHT_AWDHT_Msk                                            /*!<Analog watchdog high threshold */

/*******************  Bit definition for ADC_AWDLT register  ********************/
#define ADC_AWDLT_AWDLT_Pos                                         (0U)
#define ADC_AWDLT_AWDLT_Msk                                         (0xFFFUL << ADC_AWDLT_AWDLT_Pos)                               /*!< 0x00000FFF */
#define ADC_AWDLT_AWDLT                                             ADC_AWDLT_AWDLT_Msk                                            /*!<Analog watchdog low threshold */

/*******************  Bit definition for ADC_REGSEQ1 register  *******************/
#define ADC_REGSEQ1_REGSEQC13_Pos                                   (0U)
#define ADC_REGSEQ1_REGSEQC13_Msk                                   (0x1FUL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x0000001F */
#define ADC_REGSEQ1_REGSEQC13                                       ADC_REGSEQ1_REGSEQC13_Msk                                      /*!<SQ13[4:0] bits (13th conversion in regular sequence) */
#define ADC_REGSEQ1_REGSEQC13_0                                     (0x01UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000001 */
#define ADC_REGSEQ1_REGSEQC13_1                                     (0x02UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000002 */
#define ADC_REGSEQ1_REGSEQC13_2                                     (0x04UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000004 */
#define ADC_REGSEQ1_REGSEQC13_3                                     (0x08UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000008 */
#define ADC_REGSEQ1_REGSEQC13_4                                     (0x10UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000010 */
#define ADC_REGSEQ1_REGSEQC14_Pos                                   (5U)
#define ADC_REGSEQ1_REGSEQC14_Msk                                   (0x1FUL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x000003E0 */
#define ADC_REGSEQ1_REGSEQC14                                       ADC_REGSEQ1_REGSEQC14_Msk                                      /*!<SQ14[4:0] bits (14th conversion in regular sequence) */
#define ADC_REGSEQ1_REGSEQC14_0                                     (0x01UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000020 */
#define ADC_REGSEQ1_REGSEQC14_1                                     (0x02UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000040 */
#define ADC_REGSEQ1_REGSEQC14_2                                     (0x04UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000080 */
#define ADC_REGSEQ1_REGSEQC14_3                                     (0x08UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000100 */
#define ADC_REGSEQ1_REGSEQC14_4                                     (0x10UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000200 */
#define ADC_REGSEQ1_REGSEQC15_Pos                                   (10U)
#define ADC_REGSEQ1_REGSEQC15_Msk                                   (0x1FUL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00007C00 */
#define ADC_REGSEQ1_REGSEQC15                                       ADC_REGSEQ1_REGSEQC15_Msk                                      /*!<SQ15[4:0] bits (15th conversion in regular sequence) */
#define ADC_REGSEQ1_REGSEQC15_0                                     (0x01UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00000400 */
#define ADC_REGSEQ1_REGSEQC15_1                                     (0x02UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00000800 */
#define ADC_REGSEQ1_REGSEQC15_2                                     (0x04UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00001000 */
#define ADC_REGSEQ1_REGSEQC15_3                                     (0x08UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00002000 */
#define ADC_REGSEQ1_REGSEQC15_4                                     (0x10UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00004000 */
#define ADC_REGSEQ1_REGSEQC16_Pos                                   (15U)
#define ADC_REGSEQ1_REGSEQC16_Msk                                   (0x1FUL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x000F8000 */
#define ADC_REGSEQ1_REGSEQC16                                       ADC_REGSEQ1_REGSEQC16_Msk                                      /*!<SQ16[4:0] bits (16th conversion in regular sequence) */
#define ADC_REGSEQ1_REGSEQC16_0                                     (0x01UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00008000 */
#define ADC_REGSEQ1_REGSEQC16_1                                     (0x02UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00010000 */
#define ADC_REGSEQ1_REGSEQC16_2                                     (0x04UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00020000 */
#define ADC_REGSEQ1_REGSEQC16_3                                     (0x08UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00040000 */
#define ADC_REGSEQ1_REGSEQC16_4                                     (0x10UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00080000 */
#define ADC_REGSEQ1_REGSEQLEN_Pos                                   (20U)
#define ADC_REGSEQ1_REGSEQLEN_Msk                                   (0xFUL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00F00000 */
#define ADC_REGSEQ1_REGSEQLEN                                       ADC_REGSEQ1_REGSEQLEN_Msk                                      /*!<L[3:0] bits (Regular channel sequence length) */
#define ADC_REGSEQ1_REGSEQLEN_0                                     (0x1UL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00100000 */
#define ADC_REGSEQ1_REGSEQLEN_1                                     (0x2UL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00200000 */
#define ADC_REGSEQ1_REGSEQLEN_2                                     (0x4UL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00400000 */
#define ADC_REGSEQ1_REGSEQLEN_3                                     (0x8UL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00800000 */

/*******************  Bit definition for ADC_REGSEQ2 register  *******************/
#define ADC_REGSEQ2_REGSEQC7_Pos                                    (0U)
#define ADC_REGSEQ2_REGSEQC7_Msk                                    (0x1FUL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x0000001F */
#define ADC_REGSEQ2_REGSEQC7                                        ADC_REGSEQ2_REGSEQC7_Msk                                       /*!<SQ7[4:0] bits (7th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC7_0                                      (0x01UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000001 */
#define ADC_REGSEQ2_REGSEQC7_1                                      (0x02UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000002 */
#define ADC_REGSEQ2_REGSEQC7_2                                      (0x04UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000004 */
#define ADC_REGSEQ2_REGSEQC7_3                                      (0x08UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000008 */
#define ADC_REGSEQ2_REGSEQC7_4                                      (0x10UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000010 */
#define ADC_REGSEQ2_REGSEQC8_Pos                                    (5U)
#define ADC_REGSEQ2_REGSEQC8_Msk                                    (0x1FUL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x000003E0 */
#define ADC_REGSEQ2_REGSEQC8                                        ADC_REGSEQ2_REGSEQC8_Msk                                       /*!<SQ8[4:0] bits (8th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC8_0                                      (0x01UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000020 */
#define ADC_REGSEQ2_REGSEQC8_1                                      (0x02UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000040 */
#define ADC_REGSEQ2_REGSEQC8_2                                      (0x04UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000080 */
#define ADC_REGSEQ2_REGSEQC8_3                                      (0x08UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000100 */
#define ADC_REGSEQ2_REGSEQC8_4                                      (0x10UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000200 */
#define ADC_REGSEQ2_REGSEQC9_Pos                                    (10U)
#define ADC_REGSEQ2_REGSEQC9_Msk                                    (0x1FUL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00007C00 */
#define ADC_REGSEQ2_REGSEQC9                                        ADC_REGSEQ2_REGSEQC9_Msk                                       /*!<SQ9[4:0] bits (9th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC9_0                                      (0x01UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00000400 */
#define ADC_REGSEQ2_REGSEQC9_1                                      (0x02UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00000800 */
#define ADC_REGSEQ2_REGSEQC9_2                                      (0x04UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00001000 */
#define ADC_REGSEQ2_REGSEQC9_3                                      (0x08UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00002000 */
#define ADC_REGSEQ2_REGSEQC9_4                                      (0x10UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00004000 */
#define ADC_REGSEQ2_REGSEQC10_Pos                                   (15U)
#define ADC_REGSEQ2_REGSEQC10_Msk                                   (0x1FUL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x000F8000 */
#define ADC_REGSEQ2_REGSEQC10                                       ADC_REGSEQ2_REGSEQC10_Msk                                      /*!<SQ10[4:0] bits (10th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC10_0                                     (0x01UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00008000 */
#define ADC_REGSEQ2_REGSEQC10_1                                     (0x02UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00010000 */
#define ADC_REGSEQ2_REGSEQC10_2                                     (0x04UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00020000 */
#define ADC_REGSEQ2_REGSEQC10_3                                     (0x08UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00040000 */
#define ADC_REGSEQ2_REGSEQC10_4                                     (0x10UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00080000 */
#define ADC_REGSEQ2_REGSEQC11_Pos                                   (20U)
#define ADC_REGSEQ2_REGSEQC11_Msk                                   (0x1FUL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x01F00000 */
#define ADC_REGSEQ2_REGSEQC11                                       ADC_REGSEQ2_REGSEQC11_Msk                                      /*!<SQ11[4:0] bits (11th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC11_0                                     (0x01UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x00100000 */
#define ADC_REGSEQ2_REGSEQC11_1                                     (0x02UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x00200000 */
#define ADC_REGSEQ2_REGSEQC11_2                                     (0x04UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x00400000 */
#define ADC_REGSEQ2_REGSEQC11_3                                     (0x08UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x00800000 */
#define ADC_REGSEQ2_REGSEQC11_4                                     (0x10UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x01000000 */
#define ADC_REGSEQ2_REGSEQC12_Pos                                   (25U)
#define ADC_REGSEQ2_REGSEQC12_Msk                                   (0x1FUL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x3E000000 */
#define ADC_REGSEQ2_REGSEQC12                                       ADC_REGSEQ2_REGSEQC12_Msk                                      /*!<SQ12[4:0] bits (12th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC12_0                                     (0x01UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x02000000 */
#define ADC_REGSEQ2_REGSEQC12_1                                     (0x02UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x04000000 */
#define ADC_REGSEQ2_REGSEQC12_2                                     (0x04UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x08000000 */
#define ADC_REGSEQ2_REGSEQC12_3                                     (0x08UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x10000000 */
#define ADC_REGSEQ2_REGSEQC12_4                                     (0x10UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x20000000 */

/*******************  Bit definition for ADC_REGSEQ3 register  *******************/
#define ADC_REGSEQ3_REGSEQC1_Pos                                    (0U)
#define ADC_REGSEQ3_REGSEQC1_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x0000001F */
#define ADC_REGSEQ3_REGSEQC1                                        ADC_REGSEQ3_REGSEQC1_Msk                                       /*!<SQ1[4:0] bits (1st conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC1_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000001 */
#define ADC_REGSEQ3_REGSEQC1_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000002 */
#define ADC_REGSEQ3_REGSEQC1_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000004 */
#define ADC_REGSEQ3_REGSEQC1_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000008 */
#define ADC_REGSEQ3_REGSEQC1_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000010 */
#define ADC_REGSEQ3_REGSEQC2_Pos                                    (5U)
#define ADC_REGSEQ3_REGSEQC2_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x000003E0 */
#define ADC_REGSEQ3_REGSEQC2                                        ADC_REGSEQ3_REGSEQC2_Msk                                       /*!<SQ2[4:0] bits (2nd conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC2_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000020 */
#define ADC_REGSEQ3_REGSEQC2_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000040 */
#define ADC_REGSEQ3_REGSEQC2_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000080 */
#define ADC_REGSEQ3_REGSEQC2_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000100 */
#define ADC_REGSEQ3_REGSEQC2_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000200 */
#define ADC_REGSEQ3_REGSEQC3_Pos                                    (10U)
#define ADC_REGSEQ3_REGSEQC3_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00007C00 */
#define ADC_REGSEQ3_REGSEQC3                                        ADC_REGSEQ3_REGSEQC3_Msk                                       /*!<SQ3[4:0] bits (3rd conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC3_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00000400 */
#define ADC_REGSEQ3_REGSEQC3_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00000800 */
#define ADC_REGSEQ3_REGSEQC3_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00001000 */
#define ADC_REGSEQ3_REGSEQC3_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00002000 */
#define ADC_REGSEQ3_REGSEQC3_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00004000 */
#define ADC_REGSEQ3_REGSEQC4_Pos                                    (15U)
#define ADC_REGSEQ3_REGSEQC4_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x000F8000 */
#define ADC_REGSEQ3_REGSEQC4                                        ADC_REGSEQ3_REGSEQC4_Msk                                       /*!<SQ4[4:0] bits (4th conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC4_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00008000 */
#define ADC_REGSEQ3_REGSEQC4_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00010000 */
#define ADC_REGSEQ3_REGSEQC4_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00020000 */
#define ADC_REGSEQ3_REGSEQC4_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00040000 */
#define ADC_REGSEQ3_REGSEQC4_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00080000 */
#define ADC_REGSEQ3_REGSEQC5_Pos                                    (20U)
#define ADC_REGSEQ3_REGSEQC5_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x01F00000 */
#define ADC_REGSEQ3_REGSEQC5                                        ADC_REGSEQ3_REGSEQC5_Msk                                       /*!<SQ5[4:0] bits (5th conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC5_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x00100000 */
#define ADC_REGSEQ3_REGSEQC5_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x00200000 */
#define ADC_REGSEQ3_REGSEQC5_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x00400000 */
#define ADC_REGSEQ3_REGSEQC5_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x00800000 */
#define ADC_REGSEQ3_REGSEQC5_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x01000000 */
#define ADC_REGSEQ3_REGSEQC6_Pos                                    (25U)
#define ADC_REGSEQ3_REGSEQC6_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x3E000000 */
#define ADC_REGSEQ3_REGSEQC6                                        ADC_REGSEQ3_REGSEQC6_Msk                                       /*!<SQ6[4:0] bits (6th conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC6_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x02000000 */
#define ADC_REGSEQ3_REGSEQC6_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x04000000 */
#define ADC_REGSEQ3_REGSEQC6_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x08000000 */
#define ADC_REGSEQ3_REGSEQC6_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x10000000 */
#define ADC_REGSEQ3_REGSEQC6_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x20000000 */

/*******************  Bit definition for ADC_INJSEQ register  *******************/
#define ADC_INJSEQ_INJSEQC1_Pos                                     (0U)
#define ADC_INJSEQ_INJSEQC1_Msk                                     (0x1FUL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x0000001F */
#define ADC_INJSEQ_INJSEQC1                                         ADC_INJSEQ_INJSEQC1_Msk                                        /*!<JSQ1[4:0] bits (1st conversion in injected sequence) */
#define ADC_INJSEQ_INJSEQC1_0                                       (0x01UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000001 */
#define ADC_INJSEQ_INJSEQC1_1                                       (0x02UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000002 */
#define ADC_INJSEQ_INJSEQC1_2                                       (0x04UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000004 */
#define ADC_INJSEQ_INJSEQC1_3                                       (0x08UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000008 */
#define ADC_INJSEQ_INJSEQC1_4                                       (0x10UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000010 */
#define ADC_INJSEQ_INJSEQC2_Pos                                     (5U)
#define ADC_INJSEQ_INJSEQC2_Msk                                     (0x1FUL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x000003E0 */
#define ADC_INJSEQ_INJSEQC2                                         ADC_INJSEQ_INJSEQC2_Msk                                        /*!<JSQ2[4:0] bits (2nd conversion in injected sequence) */
#define ADC_INJSEQ_INJSEQC2_0                                       (0x01UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000020 */
#define ADC_INJSEQ_INJSEQC2_1                                       (0x02UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000040 */
#define ADC_INJSEQ_INJSEQC2_2                                       (0x04UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000080 */
#define ADC_INJSEQ_INJSEQC2_3                                       (0x08UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000100 */
#define ADC_INJSEQ_INJSEQC2_4                                       (0x10UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000200 */
#define ADC_INJSEQ_INJSEQC3_Pos                                     (10U)
#define ADC_INJSEQ_INJSEQC3_Msk                                     (0x1FUL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00007C00 */
#define ADC_INJSEQ_INJSEQC3                                         ADC_INJSEQ_INJSEQC3_Msk                                        /*!<JSQ3[4:0] bits (3rd conversion in injected sequence) */
#define ADC_INJSEQ_INJSEQC3_0                                       (0x01UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00000400 */
#define ADC_INJSEQ_INJSEQC3_1                                       (0x02UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00000800 */
#define ADC_INJSEQ_INJSEQC3_2                                       (0x04UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00001000 */
#define ADC_INJSEQ_INJSEQC3_3                                       (0x08UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00002000 */
#define ADC_INJSEQ_INJSEQC3_4                                       (0x10UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00004000 */
#define ADC_INJSEQ_INJSEQC4_Pos                                     (15U)
#define ADC_INJSEQ_INJSEQC4_Msk                                     (0x1FUL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x000F8000 */
#define ADC_INJSEQ_INJSEQC4                                         ADC_INJSEQ_INJSEQC4_Msk                                        /*!<JSQ4[4:0] bits (4th conversion in injected sequence) */
#define ADC_INJSEQ_INJSEQC4_0                                       (0x01UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00008000 */
#define ADC_INJSEQ_INJSEQC4_1                                       (0x02UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00010000 */
#define ADC_INJSEQ_INJSEQC4_2                                       (0x04UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00020000 */
#define ADC_INJSEQ_INJSEQC4_3                                       (0x08UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00040000 */
#define ADC_INJSEQ_INJSEQC4_4                                       (0x10UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00080000 */
#define ADC_INJSEQ_INJSEQLEN_Pos                                    (20U)
#define ADC_INJSEQ_INJSEQLEN_Msk                                    (0x3UL << ADC_INJSEQ_INJSEQLEN_Pos)                            /*!< 0x00300000 */
#define ADC_INJSEQ_INJSEQLEN                                        ADC_INJSEQ_INJSEQLEN_Msk                                       /*!<JL[1:0] bits (Injected Sequence length) */
#define ADC_INJSEQ_INJSEQLEN_0                                      (0x1UL << ADC_INJSEQ_INJSEQLEN_Pos)                            /*!< 0x00100000 */
#define ADC_INJSEQ_INJSEQLEN_1                                      (0x2UL << ADC_INJSEQ_INJSEQLEN_Pos)                            /*!< 0x00200000 */

/*******************  Bit definition for ADC_INJDATA1 register  *******************/
#define ADC_INJDATA1_INJDATA_Pos                                    (0U)
#define ADC_INJDATA1_INJDATA_Msk                                    (0xFFFFUL << ADC_INJDATA1_INJDATA_Pos)                         /*!< 0x0000FFFF */
#define ADC_INJDATA1_INJDATA                                        ADC_INJDATA1_INJDATA_Msk                                       /*!<Injected data */

/*******************  Bit definition for ADC_INJDATA2 register  *******************/
#define ADC_INJDATA2_INJDATA_Pos                                    (0U)
#define ADC_INJDATA2_INJDATA_Msk                                    (0xFFFFUL << ADC_INJDATA2_INJDATA_Pos)                         /*!< 0x0000FFFF */
#define ADC_INJDATA2_INJDATA                                        ADC_INJDATA2_INJDATA_Msk                                       /*!<Injected data */

/*******************  Bit definition for ADC_INJDATA3 register  *******************/
#define ADC_INJDATA3_INJDATA_Pos                                    (0U)
#define ADC_INJDATA3_INJDATA_Msk                                    (0xFFFFUL << ADC_INJDATA3_INJDATA_Pos)                         /*!< 0x0000FFFF */
#define ADC_INJDATA3_INJDATA                                        ADC_INJDATA3_INJDATA_Msk                                       /*!<Injected data */

/*******************  Bit definition for ADC_INJDATA4 register  *******************/
#define ADC_INJDATA4_INJDATA_Pos                                    (0U)
#define ADC_INJDATA4_INJDATA_Msk                                    (0xFFFFUL << ADC_INJDATA4_INJDATA_Pos)                         /*!< 0x0000FFFF */
#define ADC_INJDATA4_INJDATA                                        ADC_INJDATA4_INJDATA_Msk                                       /*!<Injected data */

/********************  Bit definition for ADC_REGDATA register  ********************/
#define ADC_REGDATA_REGDATA_Pos                                     (0U)
#define ADC_REGDATA_REGDATA_Msk                                     (0xFFFFUL << ADC_REGDATA_REGDATA_Pos)                          /*!< 0x0000FFFF */
#define ADC_REGDATA_REGDATA                                         ADC_REGDATA_REGDATA_Msk                                        /*!<Regular data */

/*******************  Bit definition for ADC_CCTRL register  ********************/
#define ADC_CCTRL_ADCMSEL_Pos                                       (0U)
#define ADC_CCTRL_ADCMSEL_Msk                                       (0x1FUL << ADC_CCTRL_ADCMSEL_Pos)                              /*!< 0x0000001F */
#define ADC_CCTRL_ADCMSEL                                           ADC_CCTRL_ADCMSEL_Msk                                          /*!<MULTI[4:0] bits (Multi-ADC mode selection) */
#define ADC_CCTRL_ADCMSEL_0                                         (0x01UL << ADC_CCTRL_ADCMSEL_Pos)                              /*!< 0x00000001 */
#define ADC_CCTRL_ADCMSEL_1                                         (0x02UL << ADC_CCTRL_ADCMSEL_Pos)                              /*!< 0x00000002 */
#define ADC_CCTRL_ADCMSEL_2                                         (0x04UL << ADC_CCTRL_ADCMSEL_Pos)                              /*!< 0x00000004 */
#define ADC_CCTRL_ADCMSEL_3                                         (0x08UL << ADC_CCTRL_ADCMSEL_Pos)                              /*!< 0x00000008 */
#define ADC_CCTRL_ADCMSEL_4                                         (0x10UL << ADC_CCTRL_ADCMSEL_Pos)                              /*!< 0x00000010 */
#define ADC_CCTRL_SMPDEL2_Pos                                       (8U)
#define ADC_CCTRL_SMPDEL2_Msk                                       (0xFUL << ADC_CCTRL_SMPDEL2_Pos)                               /*!< 0x00000F00 */
#define ADC_CCTRL_SMPDEL2                                           ADC_CCTRL_SMPDEL2_Msk                                          /*!<DELAY[3:0] bits (Delay between 2 sampling phases) */
#define ADC_CCTRL_SMPDEL2_0                                         (0x1UL << ADC_CCTRL_SMPDEL2_Pos)                               /*!< 0x00000100 */
#define ADC_CCTRL_SMPDEL2_1                                         (0x2UL << ADC_CCTRL_SMPDEL2_Pos)                               /*!< 0x00000200 */
#define ADC_CCTRL_SMPDEL2_2                                         (0x4UL << ADC_CCTRL_SMPDEL2_Pos)                               /*!< 0x00000400 */
#define ADC_CCTRL_SMPDEL2_3                                         (0x8UL << ADC_CCTRL_SMPDEL2_Pos)                               /*!< 0x00000800 */
#define ADC_CCTRL_DMAMODEDISSEL_Pos                                 (13U)
#define ADC_CCTRL_DMAMODEDISSEL_Msk                                 (0x1UL << ADC_CCTRL_DMAMODEDISSEL_Pos)                         /*!< 0x00002000 */
#define ADC_CCTRL_DMAMODEDISSEL                                     ADC_CCTRL_DMAMODEDISSEL_Msk                                    /*!<DMA disable selection (Multi-ADC mode) */
#define ADC_CCTRL_DMAMODE_Pos                                       (14U)
#define ADC_CCTRL_DMAMODE_Msk                                       (0x3UL << ADC_CCTRL_DMAMODE_Pos)                               /*!< 0x0000C000 */
#define ADC_CCTRL_DMAMODE                                           ADC_CCTRL_DMAMODE_Msk                                          /*!<DMA[1:0] bits (Direct Memory Access mode for multimode) */
#define ADC_CCTRL_DMAMODE_0                                         (0x1UL << ADC_CCTRL_DMAMODE_Pos)                               /*!< 0x00004000 */
#define ADC_CCTRL_DMAMODE_1                                         (0x2UL << ADC_CCTRL_DMAMODE_Pos)                               /*!< 0x00008000 */
#define ADC_CCTRL_ADCPRE_Pos                                        (16U)
#define ADC_CCTRL_ADCPRE_Msk                                        (0x3UL << ADC_CCTRL_ADCPRE_Pos)                                /*!< 0x00030000 */
#define ADC_CCTRL_ADCPRE                                            ADC_CCTRL_ADCPRE_Msk                                           /*!<ADCPRE[1:0] bits (ADC prescaler) */
#define ADC_CCTRL_ADCPRE_0                                          (0x1UL << ADC_CCTRL_ADCPRE_Pos)                                /*!< 0x00010000 */
#define ADC_CCTRL_ADCPRE_1                                          (0x2UL << ADC_CCTRL_ADCPRE_Pos)                                /*!< 0x00020000 */
#define ADC_CCTRL_VBATEN_Pos                                        (22U)
#define ADC_CCTRL_VBATEN_Msk                                        (0x1UL << ADC_CCTRL_VBATEN_Pos)                                /*!< 0x00400000 */
#define ADC_CCTRL_VBATEN                                            ADC_CCTRL_VBATEN_Msk                                           /*!<VBAT Enable */
#define ADC_CCTRL_TSVREFEN_Pos                                      (23U)
#define ADC_CCTRL_TSVREFEN_Msk                                      (0x1UL << ADC_CCTRL_TSVREFEN_Pos)                              /*!< 0x00800000 */
#define ADC_CCTRL_TSVREFEN                                          ADC_CCTRL_TSVREFEN_Msk                                         /*!<Temperature Sensor and VREFINT Enable */

/******************************************************************************/
/*                                                                            */
/*                         Controller Area Network                            */
/*                                                                            */
/******************************************************************************/
/*!<CAN control and status registers */
/*******************  Bit definition for CAN_MCTRL register  ********************/
#define CAN_MCTRL_INITREQ_Pos                                       (0U)
#define CAN_MCTRL_INITREQ_Msk                                       (0x1UL << CAN_MCTRL_INITREQ_Pos)                               /*!< 0x00000001 */
#define CAN_MCTRL_INITREQ                                           CAN_MCTRL_INITREQ_Msk                                          /*!<Initialization Request */
#define CAN_MCTRL_SLEEPREQ_Pos                                      (1U)
#define CAN_MCTRL_SLEEPREQ_Msk                                      (0x1UL << CAN_MCTRL_SLEEPREQ_Pos)                              /*!< 0x00000002 */
#define CAN_MCTRL_SLEEPREQ                                          CAN_MCTRL_SLEEPREQ_Msk                                         /*!<Sleep Mode Request */
#define CAN_MCTRL_TXFPCFG_Pos                                       (2U)
#define CAN_MCTRL_TXFPCFG_Msk                                       (0x1UL << CAN_MCTRL_TXFPCFG_Pos)                               /*!< 0x00000004 */
#define CAN_MCTRL_TXFPCFG                                           CAN_MCTRL_TXFPCFG_Msk                                          /*!<Transmit FIFO Priority */
#define CAN_MCTRL_RXFLOCK_Pos                                       (3U)
#define CAN_MCTRL_RXFLOCK_Msk                                       (0x1UL << CAN_MCTRL_RXFLOCK_Pos)                               /*!< 0x00000008 */
#define CAN_MCTRL_RXFLOCK                                           CAN_MCTRL_RXFLOCK_Msk                                          /*!<Receive FIFO Locked Mode */
#define CAN_MCTRL_ARTXMD_Pos                                        (4U)
#define CAN_MCTRL_ARTXMD_Msk                                        (0x1UL << CAN_MCTRL_ARTXMD_Pos)                                /*!< 0x00000010 */
#define CAN_MCTRL_ARTXMD                                            CAN_MCTRL_ARTXMD_Msk                                           /*!<No Automatic Retransmission */
#define CAN_MCTRL_AWUPCFG_Pos                                       (5U)
#define CAN_MCTRL_AWUPCFG_Msk                                       (0x1UL << CAN_MCTRL_AWUPCFG_Pos)                               /*!< 0x00000020 */
#define CAN_MCTRL_AWUPCFG                                           CAN_MCTRL_AWUPCFG_Msk                                          /*!<Automatic Wakeup Mode */
#define CAN_MCTRL_ALBOFFM_Pos                                       (6U)
#define CAN_MCTRL_ALBOFFM_Msk                                       (0x1UL << CAN_MCTRL_ALBOFFM_Pos)                               /*!< 0x00000040 */
#define CAN_MCTRL_ALBOFFM                                           CAN_MCTRL_ALBOFFM_Msk                                          /*!<Automatic Bus-Off Management */
#define CAN_MCTRL_TTCM_Pos                                          (7U)
#define CAN_MCTRL_TTCM_Msk                                          (0x1UL << CAN_MCTRL_TTCM_Pos)                                  /*!< 0x00000080 */
#define CAN_MCTRL_TTCM                                              CAN_MCTRL_TTCM_Msk                                             /*!<Time Triggered Communication Mode */
#define CAN_MCTRL_SWRST_Pos                                         (15U)
#define CAN_MCTRL_SWRST_Msk                                         (0x1UL << CAN_MCTRL_SWRST_Pos)                                 /*!< 0x00008000 */
#define CAN_MCTRL_SWRST                                             CAN_MCTRL_SWRST_Msk                                            /*!<bxCAN software master reset */
#define CAN_MCTRL_DBGFRZE_Pos                                       (16U)
#define CAN_MCTRL_DBGFRZE_Msk                                       (0x1UL << CAN_MCTRL_DBGFRZE_Pos)                               /*!< 0x00010000 */
#define CAN_MCTRL_DBGFRZE                                           CAN_MCTRL_DBGFRZE_Msk                                          /*!<bxCAN Debug freeze */
/*******************  Bit definition for CAN_MSTS register  ********************/
#define CAN_MSTS_INITFLG_Pos                                        (0U)
#define CAN_MSTS_INITFLG_Msk                                        (0x1UL << CAN_MSTS_INITFLG_Pos)                                /*!< 0x00000001 */
#define CAN_MSTS_INITFLG                                            CAN_MSTS_INITFLG_Msk                                           /*!<Initialization Acknowledge */
#define CAN_MSTS_SLEEPFLG_Pos                                       (1U)
#define CAN_MSTS_SLEEPFLG_Msk                                       (0x1UL << CAN_MSTS_SLEEPFLG_Pos)                               /*!< 0x00000002 */
#define CAN_MSTS_SLEEPFLG                                           CAN_MSTS_SLEEPFLG_Msk                                          /*!<Sleep Acknowledge */
#define CAN_MSTS_ERRIFLG_Pos                                        (2U)
#define CAN_MSTS_ERRIFLG_Msk                                        (0x1UL << CAN_MSTS_ERRIFLG_Pos)                                /*!< 0x00000004 */
#define CAN_MSTS_ERRIFLG                                            CAN_MSTS_ERRIFLG_Msk                                           /*!<Error Interrupt */
#define CAN_MSTS_WUPIFLG_Pos                                        (3U)
#define CAN_MSTS_WUPIFLG_Msk                                        (0x1UL << CAN_MSTS_WUPIFLG_Pos)                                /*!< 0x00000008 */
#define CAN_MSTS_WUPIFLG                                            CAN_MSTS_WUPIFLG_Msk                                           /*!<Wakeup Interrupt */
#define CAN_MSTS_SLEEPIFLG_Pos                                      (4U)
#define CAN_MSTS_SLEEPIFLG_Msk                                      (0x1UL << CAN_MSTS_SLEEPIFLG_Pos)                              /*!< 0x00000010 */
#define CAN_MSTS_SLEEPIFLG                                          CAN_MSTS_SLEEPIFLG_Msk                                         /*!<Sleep Acknowledge Interrupt */
#define CAN_MSTS_TXMFLG_Pos                                         (8U)
#define CAN_MSTS_TXMFLG_Msk                                         (0x1UL << CAN_MSTS_TXMFLG_Pos)                                 /*!< 0x00000100 */
#define CAN_MSTS_TXMFLG                                             CAN_MSTS_TXMFLG_Msk                                            /*!<Transmit Mode */
#define CAN_MSTS_RXSIGLMFLG_Pos                                     (9U)
#define CAN_MSTS_RXSIGLMFLG_Msk                                     (0x1UL << CAN_MSTS_RXSIGLMFLG_Pos)                             /*!< 0x00000200 */
#define CAN_MSTS_RXSIGLMFLG                                         CAN_MSTS_RXSIGLMFLG_Msk                                        /*!<Receive Mode */
#define CAN_MSTS_LSAMVALUE_Pos                                      (10U)
#define CAN_MSTS_LSAMVALUE_Msk                                      (0x1UL << CAN_MSTS_LSAMVALUE_Pos)                              /*!< 0x00000400 */
#define CAN_MSTS_LSAMVALUE                                          CAN_MSTS_LSAMVALUE_Msk                                         /*!<Last Sample Point */
#define CAN_MSTS_RXSIGL_Pos                                         (11U)
#define CAN_MSTS_RXSIGL_Msk                                         (0x1UL << CAN_MSTS_RXSIGL_Pos)                                 /*!< 0x00000800 */
#define CAN_MSTS_RXSIGL                                             CAN_MSTS_RXSIGL_Msk                                            /*!<CAN Rx Signal */

/*******************  Bit definition for CAN_TXSTS register  ********************/
#define CAN_TXSTS_REQCFLG0_Pos                                      (0U)
#define CAN_TXSTS_REQCFLG0_Msk                                      (0x1UL << CAN_TXSTS_REQCFLG0_Pos)                              /*!< 0x00000001 */
#define CAN_TXSTS_REQCFLG0                                          CAN_TXSTS_REQCFLG0_Msk                                         /*!<Request Completed Mailbox0 */
#define CAN_TXSTS_TXSUSFLG0_Pos                                     (1U)
#define CAN_TXSTS_TXSUSFLG0_Msk                                     (0x1UL << CAN_TXSTS_TXSUSFLG0_Pos)                             /*!< 0x00000002 */
#define CAN_TXSTS_TXSUSFLG0                                         CAN_TXSTS_TXSUSFLG0_Msk                                        /*!<Transmission OK of Mailbox0 */
#define CAN_TXSTS_ARBLSTFLG0_Pos                                    (2U)
#define CAN_TXSTS_ARBLSTFLG0_Msk                                    (0x1UL << CAN_TXSTS_ARBLSTFLG0_Pos)                            /*!< 0x00000004 */
#define CAN_TXSTS_ARBLSTFLG0                                        CAN_TXSTS_ARBLSTFLG0_Msk                                       /*!<Arbitration Lost for Mailbox0 */
#define CAN_TXSTS_TXERRFLG0_Pos                                     (3U)
#define CAN_TXSTS_TXERRFLG0_Msk                                     (0x1UL << CAN_TXSTS_TXERRFLG0_Pos)                             /*!< 0x00000008 */
#define CAN_TXSTS_TXERRFLG0                                         CAN_TXSTS_TXERRFLG0_Msk                                        /*!<Transmission Error of Mailbox0 */
#define CAN_TXSTS_ABREQFLG0_Pos                                     (7U)
#define CAN_TXSTS_ABREQFLG0_Msk                                     (0x1UL << CAN_TXSTS_ABREQFLG0_Pos)                             /*!< 0x00000080 */
#define CAN_TXSTS_ABREQFLG0                                         CAN_TXSTS_ABREQFLG0_Msk                                        /*!<Abort Request for Mailbox0 */
#define CAN_TXSTS_REQCFLG1_Pos                                      (8U)
#define CAN_TXSTS_REQCFLG1_Msk                                      (0x1UL << CAN_TXSTS_REQCFLG1_Pos)                              /*!< 0x00000100 */
#define CAN_TXSTS_REQCFLG1                                          CAN_TXSTS_REQCFLG1_Msk                                         /*!<Request Completed Mailbox1 */
#define CAN_TXSTS_TXSUSFLG1_Pos                                     (9U)
#define CAN_TXSTS_TXSUSFLG1_Msk                                     (0x1UL << CAN_TXSTS_TXSUSFLG1_Pos)                             /*!< 0x00000200 */
#define CAN_TXSTS_TXSUSFLG1                                         CAN_TXSTS_TXSUSFLG1_Msk                                        /*!<Transmission OK of Mailbox1 */
#define CAN_TXSTS_ARBLSTFLG1_Pos                                    (10U)
#define CAN_TXSTS_ARBLSTFLG1_Msk                                    (0x1UL << CAN_TXSTS_ARBLSTFLG1_Pos)                            /*!< 0x00000400 */
#define CAN_TXSTS_ARBLSTFLG1                                        CAN_TXSTS_ARBLSTFLG1_Msk                                       /*!<Arbitration Lost for Mailbox1 */
#define CAN_TXSTS_TXERRFLG1_Pos                                     (11U)
#define CAN_TXSTS_TXERRFLG1_Msk                                     (0x1UL << CAN_TXSTS_TXERRFLG1_Pos)                             /*!< 0x00000800 */
#define CAN_TXSTS_TXERRFLG1                                         CAN_TXSTS_TXERRFLG1_Msk                                        /*!<Transmission Error of Mailbox1 */
#define CAN_TXSTS_ABREQFLG1_Pos                                     (15U)
#define CAN_TXSTS_ABREQFLG1_Msk                                     (0x1UL << CAN_TXSTS_ABREQFLG1_Pos)                             /*!< 0x00008000 */
#define CAN_TXSTS_ABREQFLG1                                         CAN_TXSTS_ABREQFLG1_Msk                                        /*!<Abort Request for Mailbox 1 */
#define CAN_TXSTS_REQCFLG2_Pos                                      (16U)
#define CAN_TXSTS_REQCFLG2_Msk                                      (0x1UL << CAN_TXSTS_REQCFLG2_Pos)                              /*!< 0x00010000 */
#define CAN_TXSTS_REQCFLG2                                          CAN_TXSTS_REQCFLG2_Msk                                         /*!<Request Completed Mailbox2 */
#define CAN_TXSTS_TXSUSFLG2_Pos                                     (17U)
#define CAN_TXSTS_TXSUSFLG2_Msk                                     (0x1UL << CAN_TXSTS_TXSUSFLG2_Pos)                             /*!< 0x00020000 */
#define CAN_TXSTS_TXSUSFLG2                                         CAN_TXSTS_TXSUSFLG2_Msk                                        /*!<Transmission OK of Mailbox 2 */
#define CAN_TXSTS_ARBLSTFLG2_Pos                                    (18U)
#define CAN_TXSTS_ARBLSTFLG2_Msk                                    (0x1UL << CAN_TXSTS_ARBLSTFLG2_Pos)                            /*!< 0x00040000 */
#define CAN_TXSTS_ARBLSTFLG2                                        CAN_TXSTS_ARBLSTFLG2_Msk                                       /*!<Arbitration Lost for mailbox 2 */
#define CAN_TXSTS_TXERRFLG2_Pos                                     (19U)
#define CAN_TXSTS_TXERRFLG2_Msk                                     (0x1UL << CAN_TXSTS_TXERRFLG2_Pos)                             /*!< 0x00080000 */
#define CAN_TXSTS_TXERRFLG2                                         CAN_TXSTS_TXERRFLG2_Msk                                        /*!<Transmission Error of Mailbox 2 */
#define CAN_TXSTS_ABREQFLG2_Pos                                     (23U)
#define CAN_TXSTS_ABREQFLG2_Msk                                     (0x1UL << CAN_TXSTS_ABREQFLG2_Pos)                             /*!< 0x00800000 */
#define CAN_TXSTS_ABREQFLG2                                         CAN_TXSTS_ABREQFLG2_Msk                                        /*!<Abort Request for Mailbox 2 */
#define CAN_TXSTS_EMNUM_Pos                                         (24U)
#define CAN_TXSTS_EMNUM_Msk                                         (0x3UL << CAN_TXSTS_EMNUM_Pos)                                 /*!< 0x03000000 */
#define CAN_TXSTS_EMNUM                                             CAN_TXSTS_EMNUM_Msk                                            /*!<Mailbox Code */

#define CAN_TXSTS_TXMEFLG_Pos                                       (26U)
#define CAN_TXSTS_TXMEFLG_Msk                                       (0x7UL << CAN_TXSTS_TXMEFLG_Pos)                               /*!< 0x1C000000 */
#define CAN_TXSTS_TXMEFLG                                           CAN_TXSTS_TXMEFLG_Msk                                          /*!<TME[2:0] bits */
#define CAN_TXSTS_TXMEFLG0_Pos                                      (26U)
#define CAN_TXSTS_TXMEFLG0_Msk                                      (0x1UL << CAN_TXSTS_TXMEFLG0_Pos)                              /*!< 0x04000000 */
#define CAN_TXSTS_TXMEFLG0                                          CAN_TXSTS_TXMEFLG0_Msk                                         /*!<Transmit Mailbox 0 Empty */
#define CAN_TXSTS_TXMEFLG1_Pos                                      (27U)
#define CAN_TXSTS_TXMEFLG1_Msk                                      (0x1UL << CAN_TXSTS_TXMEFLG1_Pos)                              /*!< 0x08000000 */
#define CAN_TXSTS_TXMEFLG1                                          CAN_TXSTS_TXMEFLG1_Msk                                         /*!<Transmit Mailbox 1 Empty */
#define CAN_TXSTS_TXMEFLG2_Pos                                      (28U)
#define CAN_TXSTS_TXMEFLG2_Msk                                      (0x1UL << CAN_TXSTS_TXMEFLG2_Pos)                              /*!< 0x10000000 */
#define CAN_TXSTS_TXMEFLG2                                          CAN_TXSTS_TXMEFLG2_Msk                                         /*!<Transmit Mailbox 2 Empty */

#define CAN_TXSTS_LOWESTP_Pos                                       (29U)
#define CAN_TXSTS_LOWESTP_Msk                                       (0x7UL << CAN_TXSTS_LOWESTP_Pos)                               /*!< 0xE0000000 */
#define CAN_TXSTS_LOWESTP                                           CAN_TXSTS_LOWESTP_Msk                                          /*!<LOW[2:0] bits */
#define CAN_TXSTS_LOWESTP0_Pos                                      (29U)
#define CAN_TXSTS_LOWESTP0_Msk                                      (0x1UL << CAN_TXSTS_LOWESTP0_Pos)                              /*!< 0x20000000 */
#define CAN_TXSTS_LOWESTP0                                          CAN_TXSTS_LOWESTP0_Msk                                         /*!<Lowest Priority Flag for Mailbox 0 */
#define CAN_TXSTS_LOWESTP1_Pos                                      (30U)
#define CAN_TXSTS_LOWESTP1_Msk                                      (0x1UL << CAN_TXSTS_LOWESTP1_Pos)                              /*!< 0x40000000 */
#define CAN_TXSTS_LOWESTP1                                          CAN_TXSTS_LOWESTP1_Msk                                         /*!<Lowest Priority Flag for Mailbox 1 */
#define CAN_TXSTS_LOWESTP2_Pos                                      (31U)
#define CAN_TXSTS_LOWESTP2_Msk                                      (0x1UL << CAN_TXSTS_LOWESTP2_Pos)                              /*!< 0x80000000 */
#define CAN_TXSTS_LOWESTP2                                          CAN_TXSTS_LOWESTP2_Msk                                         /*!<Lowest Priority Flag for Mailbox 2 */

/*******************  Bit definition for CAN_RXF0 register  *******************/
#define CAN_RXF0_FMNUM0_Pos                                         (0U)
#define CAN_RXF0_FMNUM0_Msk                                         (0x3UL << CAN_RXF0_FMNUM0_Pos)                                 /*!< 0x00000003 */
#define CAN_RXF0_FMNUM0                                             CAN_RXF0_FMNUM0_Msk                                            /*!<FIFO 0 Message Pending */
#define CAN_RXF0_FFULLFLG0_Pos                                      (3U)
#define CAN_RXF0_FFULLFLG0_Msk                                      (0x1UL << CAN_RXF0_FFULLFLG0_Pos)                              /*!< 0x00000008 */
#define CAN_RXF0_FFULLFLG0                                          CAN_RXF0_FFULLFLG0_Msk                                         /*!<FIFO 0 Full */
#define CAN_RXF0_FOVRFLG0_Pos                                       (4U)
#define CAN_RXF0_FOVRFLG0_Msk                                       (0x1UL << CAN_RXF0_FOVRFLG0_Pos)                               /*!< 0x00000010 */
#define CAN_RXF0_FOVRFLG0                                           CAN_RXF0_FOVRFLG0_Msk                                          /*!<FIFO 0 Overrun */
#define CAN_RXF0_RFOM0_Pos                                          (5U)
#define CAN_RXF0_RFOM0_Msk                                          (0x1UL << CAN_RXF0_RFOM0_Pos)                                  /*!< 0x00000020 */
#define CAN_RXF0_RFOM0                                              CAN_RXF0_RFOM0_Msk                                             /*!<Release FIFO 0 Output Mailbox */

/*******************  Bit definition for CAN_RXF1 register  *******************/
#define CAN_RXF1_FMNUM1_Pos                                         (0U)
#define CAN_RXF1_FMNUM1_Msk                                         (0x3UL << CAN_RXF1_FMNUM1_Pos)                                 /*!< 0x00000003 */
#define CAN_RXF1_FMNUM1                                             CAN_RXF1_FMNUM1_Msk                                            /*!<FIFO 1 Message Pending */
#define CAN_RXF1_FFULLFLG1_Pos                                      (3U)
#define CAN_RXF1_FFULLFLG1_Msk                                      (0x1UL << CAN_RXF1_FFULLFLG1_Pos)                              /*!< 0x00000008 */
#define CAN_RXF1_FFULLFLG1                                          CAN_RXF1_FFULLFLG1_Msk                                         /*!<FIFO 1 Full */
#define CAN_RXF1_FOVRFLG1_Pos                                       (4U)
#define CAN_RXF1_FOVRFLG1_Msk                                       (0x1UL << CAN_RXF1_FOVRFLG1_Pos)                               /*!< 0x00000010 */
#define CAN_RXF1_FOVRFLG1                                           CAN_RXF1_FOVRFLG1_Msk                                          /*!<FIFO 1 Overrun */
#define CAN_RXF1_RFOM1_Pos                                          (5U)
#define CAN_RXF1_RFOM1_Msk                                          (0x1UL << CAN_RXF1_RFOM1_Pos)                                  /*!< 0x00000020 */
#define CAN_RXF1_RFOM1                                              CAN_RXF1_RFOM1_Msk                                             /*!<Release FIFO 1 Output Mailbox */

/********************  Bit definition for CAN_INTEN register  *******************/
#define CAN_INTEN_TXMEIEN_Pos                                       (0U)
#define CAN_INTEN_TXMEIEN_Msk                                       (0x1UL << CAN_INTEN_TXMEIEN_Pos)                               /*!< 0x00000001 */
#define CAN_INTEN_TXMEIEN                                           CAN_INTEN_TXMEIEN_Msk                                          /*!<Transmit Mailbox Empty Interrupt Enable */
#define CAN_INTEN_FMIEN0_Pos                                        (1U)
#define CAN_INTEN_FMIEN0_Msk                                        (0x1UL << CAN_INTEN_FMIEN0_Pos)                                /*!< 0x00000002 */
#define CAN_INTEN_FMIEN0                                            CAN_INTEN_FMIEN0_Msk                                           /*!<FIFO Message Pending Interrupt Enable */
#define CAN_INTEN_FFULLIEN0_Pos                                     (2U)
#define CAN_INTEN_FFULLIEN0_Msk                                     (0x1UL << CAN_INTEN_FFULLIEN0_Pos)                             /*!< 0x00000004 */
#define CAN_INTEN_FFULLIEN0                                         CAN_INTEN_FFULLIEN0_Msk                                        /*!<FIFO Full Interrupt Enable */
#define CAN_INTEN_FOVRIEN0_Pos                                      (3U)
#define CAN_INTEN_FOVRIEN0_Msk                                      (0x1UL << CAN_INTEN_FOVRIEN0_Pos)                              /*!< 0x00000008 */
#define CAN_INTEN_FOVRIEN0                                          CAN_INTEN_FOVRIEN0_Msk                                         /*!<FIFO Overrun Interrupt Enable */
#define CAN_INTEN_FMIEN1_Pos                                        (4U)
#define CAN_INTEN_FMIEN1_Msk                                        (0x1UL << CAN_INTEN_FMIEN1_Pos)                                /*!< 0x00000010 */
#define CAN_INTEN_FMIEN1                                            CAN_INTEN_FMIEN1_Msk                                           /*!<FIFO Message Pending Interrupt Enable */
#define CAN_INTEN_FFULLIEN1_Pos                                     (5U)
#define CAN_INTEN_FFULLIEN1_Msk                                     (0x1UL << CAN_INTEN_FFULLIEN1_Pos)                             /*!< 0x00000020 */
#define CAN_INTEN_FFULLIEN1                                         CAN_INTEN_FFULLIEN1_Msk                                        /*!<FIFO Full Interrupt Enable */
#define CAN_INTEN_FOVRIEN1_Pos                                      (6U)
#define CAN_INTEN_FOVRIEN1_Msk                                      (0x1UL << CAN_INTEN_FOVRIEN1_Pos)                              /*!< 0x00000040 */
#define CAN_INTEN_FOVRIEN1                                          CAN_INTEN_FOVRIEN1_Msk                                         /*!<FIFO Overrun Interrupt Enable */
#define CAN_INTEN_ERRWIEN_Pos                                       (8U)
#define CAN_INTEN_ERRWIEN_Msk                                       (0x1UL << CAN_INTEN_ERRWIEN_Pos)                               /*!< 0x00000100 */
#define CAN_INTEN_ERRWIEN                                           CAN_INTEN_ERRWIEN_Msk                                          /*!<Error Warning Interrupt Enable */
#define CAN_INTEN_ERRPIEN_Pos                                       (9U)
#define CAN_INTEN_ERRPIEN_Msk                                       (0x1UL << CAN_INTEN_ERRPIEN_Pos)                               /*!< 0x00000200 */
#define CAN_INTEN_ERRPIEN                                           CAN_INTEN_ERRPIEN_Msk                                          /*!<Error Passive Interrupt Enable */
#define CAN_INTEN_BOFFIEN_Pos                                       (10U)
#define CAN_INTEN_BOFFIEN_Msk                                       (0x1UL << CAN_INTEN_BOFFIEN_Pos)                               /*!< 0x00000400 */
#define CAN_INTEN_BOFFIEN                                           CAN_INTEN_BOFFIEN_Msk                                          /*!<Bus-Off Interrupt Enable */
#define CAN_INTEN_LECIEN_Pos                                        (11U)
#define CAN_INTEN_LECIEN_Msk                                        (0x1UL << CAN_INTEN_LECIEN_Pos)                                /*!< 0x00000800 */
#define CAN_INTEN_LECIEN                                            CAN_INTEN_LECIEN_Msk                                           /*!<Last Error Code Interrupt Enable */
#define CAN_INTEN_ERRIEN_Pos                                        (15U)
#define CAN_INTEN_ERRIEN_Msk                                        (0x1UL << CAN_INTEN_ERRIEN_Pos)                                /*!< 0x00008000 */
#define CAN_INTEN_ERRIEN                                            CAN_INTEN_ERRIEN_Msk                                           /*!<Error Interrupt Enable */
#define CAN_INTEN_WUPIEN_Pos                                        (16U)
#define CAN_INTEN_WUPIEN_Msk                                        (0x1UL << CAN_INTEN_WUPIEN_Pos)                                /*!< 0x00010000 */
#define CAN_INTEN_WUPIEN                                            CAN_INTEN_WUPIEN_Msk                                           /*!<Wakeup Interrupt Enable */
#define CAN_INTEN_SLEEPIEN_Pos                                      (17U)
#define CAN_INTEN_SLEEPIEN_Msk                                      (0x1UL << CAN_INTEN_SLEEPIEN_Pos)                              /*!< 0x00020000 */
#define CAN_INTEN_SLEEPIEN                                          CAN_INTEN_SLEEPIEN_Msk                                         /*!<Sleep Interrupt Enable */

/********************  Bit definition for CAN_ERRSTS register  *******************/
#define CAN_ERRSTS_ERRWFLG_Pos                                      (0U)
#define CAN_ERRSTS_ERRWFLG_Msk                                      (0x1UL << CAN_ERRSTS_ERRWFLG_Pos)                              /*!< 0x00000001 */
#define CAN_ERRSTS_ERRWFLG                                          CAN_ERRSTS_ERRWFLG_Msk                                         /*!<Error Warning Flag */
#define CAN_ERRSTS_ERRPFLG_Pos                                      (1U)
#define CAN_ERRSTS_ERRPFLG_Msk                                      (0x1UL << CAN_ERRSTS_ERRPFLG_Pos)                              /*!< 0x00000002 */
#define CAN_ERRSTS_ERRPFLG                                          CAN_ERRSTS_ERRPFLG_Msk                                         /*!<Error Passive Flag */
#define CAN_ERRSTS_BOFLG_Pos                                        (2U)
#define CAN_ERRSTS_BOFLG_Msk                                        (0x1UL << CAN_ERRSTS_BOFLG_Pos)                                /*!< 0x00000004 */
#define CAN_ERRSTS_BOFLG                                            CAN_ERRSTS_BOFLG_Msk                                           /*!<Bus-Off Flag */

#define CAN_ERRSTS_LERRC_Pos                                        (4U)
#define CAN_ERRSTS_LERRC_Msk                                        (0x7UL << CAN_ERRSTS_LERRC_Pos)                                /*!< 0x00000070 */
#define CAN_ERRSTS_LERRC                                            CAN_ERRSTS_LERRC_Msk                                           /*!<LEC[2:0] bits (Last Error Code) */
#define CAN_ERRSTS_LERRC_0                                          (0x1UL << CAN_ERRSTS_LERRC_Pos)                                /*!< 0x00000010 */
#define CAN_ERRSTS_LERRC_1                                          (0x2UL << CAN_ERRSTS_LERRC_Pos)                                /*!< 0x00000020 */
#define CAN_ERRSTS_LERRC_2                                          (0x4UL << CAN_ERRSTS_LERRC_Pos)                                /*!< 0x00000040 */

#define CAN_ERRSTS_TXERRCNT_Pos                                     (16U)
#define CAN_ERRSTS_TXERRCNT_Msk                                     (0xFFUL << CAN_ERRSTS_TXERRCNT_Pos)                            /*!< 0x00FF0000 */
#define CAN_ERRSTS_TXERRCNT                                         CAN_ERRSTS_TXERRCNT_Msk                                        /*!<Least significant byte of the 9-bit Transmit Error Counter */
#define CAN_ERRSTS_RXERRCNT_Pos                                     (24U)
#define CAN_ERRSTS_RXERRCNT_Msk                                     (0xFFUL << CAN_ERRSTS_RXERRCNT_Pos)                            /*!< 0xFF000000 */
#define CAN_ERRSTS_RXERRCNT                                         CAN_ERRSTS_RXERRCNT_Msk                                        /*!<Receive Error Counter */

/*******************  Bit definition for CAN_BITTIM register  ********************/
#define CAN_BITTIM_BRPSC_Pos                                        (0U)
#define CAN_BITTIM_BRPSC_Msk                                        (0x3FFUL << CAN_BITTIM_BRPSC_Pos)                              /*!< 0x000003FF */
#define CAN_BITTIM_BRPSC                                            CAN_BITTIM_BRPSC_Msk                                           /*!<Baud Rate Prescaler */
#define CAN_BITTIM_TIMSEG1_Pos                                      (16U)
#define CAN_BITTIM_TIMSEG1_Msk                                      (0xFUL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x000F0000 */
#define CAN_BITTIM_TIMSEG1                                          CAN_BITTIM_TIMSEG1_Msk                                         /*!<Time Segment 1 */
#define CAN_BITTIM_TIMSEG1_0                                        (0x1UL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x00010000 */
#define CAN_BITTIM_TIMSEG1_1                                        (0x2UL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x00020000 */
#define CAN_BITTIM_TIMSEG1_2                                        (0x4UL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x00040000 */
#define CAN_BITTIM_TIMSEG1_3                                        (0x8UL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x00080000 */
#define CAN_BITTIM_TIMSEG2_Pos                                      (20U)
#define CAN_BITTIM_TIMSEG2_Msk                                      (0x7UL << CAN_BITTIM_TIMSEG2_Pos)                              /*!< 0x00700000 */
#define CAN_BITTIM_TIMSEG2                                          CAN_BITTIM_TIMSEG2_Msk                                         /*!<Time Segment 2 */
#define CAN_BITTIM_TIMSEG2_0                                        (0x1UL << CAN_BITTIM_TIMSEG2_Pos)                              /*!< 0x00100000 */
#define CAN_BITTIM_TIMSEG2_1                                        (0x2UL << CAN_BITTIM_TIMSEG2_Pos)                              /*!< 0x00200000 */
#define CAN_BITTIM_TIMSEG2_2                                        (0x4UL << CAN_BITTIM_TIMSEG2_Pos)                              /*!< 0x00400000 */
#define CAN_BITTIM_RSYNJW_Pos                                       (24U)
#define CAN_BITTIM_RSYNJW_Msk                                       (0x3UL << CAN_BITTIM_RSYNJW_Pos)                               /*!< 0x03000000 */
#define CAN_BITTIM_RSYNJW                                           CAN_BITTIM_RSYNJW_Msk                                          /*!<Resynchronization Jump Width */
#define CAN_BITTIM_RSYNJW_0                                         (0x1UL << CAN_BITTIM_RSYNJW_Pos)                               /*!< 0x01000000 */
#define CAN_BITTIM_RSYNJW_1                                         (0x2UL << CAN_BITTIM_RSYNJW_Pos)                               /*!< 0x02000000 */
#define CAN_BITTIM_LBKMEN_Pos                                       (30U)
#define CAN_BITTIM_LBKMEN_Msk                                       (0x1UL << CAN_BITTIM_LBKMEN_Pos)                               /*!< 0x40000000 */
#define CAN_BITTIM_LBKMEN                                           CAN_BITTIM_LBKMEN_Msk                                          /*!<Loop Back Mode (Debug) */
#define CAN_BITTIM_SILMEN_Pos                                       (31U)
#define CAN_BITTIM_SILMEN_Msk                                       (0x1UL << CAN_BITTIM_SILMEN_Pos)                               /*!< 0x80000000 */
#define CAN_BITTIM_SILMEN                                           CAN_BITTIM_SILMEN_Msk                                          /*!<Silent Mode */


/*!<Mailbox registers */
/******************  Bit definition for CAN_TXMID0 register  ********************/
#define CAN_TXMID0_TXMREQ_Pos                                       (0U)
#define CAN_TXMID0_TXMREQ_Msk                                       (0x1UL << CAN_TXMID0_TXMREQ_Pos)                               /*!< 0x00000001 */
#define CAN_TXMID0_TXMREQ                                           CAN_TXMID0_TXMREQ_Msk                                          /*!<Transmit Mailbox Request */
#define CAN_TXMID0_TXRFREQ_Pos                                      (1U)
#define CAN_TXMID0_TXRFREQ_Msk                                      (0x1UL << CAN_TXMID0_TXRFREQ_Pos)                              /*!< 0x00000002 */
#define CAN_TXMID0_TXRFREQ                                          CAN_TXMID0_TXRFREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_TXMID0_IDTYPESEL_Pos                                    (2U)
#define CAN_TXMID0_IDTYPESEL_Msk                                    (0x1UL << CAN_TXMID0_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_TXMID0_IDTYPESEL                                        CAN_TXMID0_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_TXMID0_EXTID_Pos                                        (3U)
#define CAN_TXMID0_EXTID_Msk                                        (0x3FFFFUL << CAN_TXMID0_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_TXMID0_EXTID                                            CAN_TXMID0_EXTID_Msk                                           /*!<Extended Identifier */
#define CAN_TXMID0_STDID_Pos                                        (21U)
#define CAN_TXMID0_STDID_Msk                                        (0x7FFUL << CAN_TXMID0_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_TXMID0_STDID                                            CAN_TXMID0_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/******************  Bit definition for CAN_TXDLEN0 register  *******************/
#define CAN_TXDLEN0_DLCODE_Pos                                      (0U)
#define CAN_TXDLEN0_DLCODE_Msk                                      (0xFUL << CAN_TXDLEN0_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_TXDLEN0_DLCODE                                          CAN_TXDLEN0_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_TXDLEN0_TXTS_Pos                                        (8U)
#define CAN_TXDLEN0_TXTS_Msk                                        (0x1UL << CAN_TXDLEN0_TXTS_Pos)                                /*!< 0x00000100 */
#define CAN_TXDLEN0_TXTS                                            CAN_TXDLEN0_TXTS_Msk                                           /*!<Transmit Global Time */
#define CAN_TXDLEN0_MTS_Pos                                         (16U)
#define CAN_TXDLEN0_MTS_Msk                                         (0xFFFFUL << CAN_TXDLEN0_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_TXDLEN0_MTS                                             CAN_TXDLEN0_MTS_Msk                                            /*!<Message Time Stamp */

/******************  Bit definition for CAN_TXMDL0 register  *******************/
#define CAN_TXMDL0_DATABYTE0_Pos                                    (0U)
#define CAN_TXMDL0_DATABYTE0_Msk                                    (0xFFUL << CAN_TXMDL0_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDL0_DATABYTE0                                        CAN_TXMDL0_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_TXMDL0_DATABYTE1_Pos                                    (8U)
#define CAN_TXMDL0_DATABYTE1_Msk                                    (0xFFUL << CAN_TXMDL0_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDL0_DATABYTE1                                        CAN_TXMDL0_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_TXMDL0_DATABYTE2_Pos                                    (16U)
#define CAN_TXMDL0_DATABYTE2_Msk                                    (0xFFUL << CAN_TXMDL0_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDL0_DATABYTE2                                        CAN_TXMDL0_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_TXMDL0_DATABYTE3_Pos                                    (24U)
#define CAN_TXMDL0_DATABYTE3_Msk                                    (0xFFUL << CAN_TXMDL0_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDL0_DATABYTE3                                        CAN_TXMDL0_DATABYTE3_Msk                                       /*!<Data byte 3 */

/******************  Bit definition for CAN_TXMDH0 register  *******************/
#define CAN_TXMDH0_DATABYTE4_Pos                                    (0U)
#define CAN_TXMDH0_DATABYTE4_Msk                                    (0xFFUL << CAN_TXMDH0_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDH0_DATABYTE4                                        CAN_TXMDH0_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_TXMDH0_DATABYTE5_Pos                                    (8U)
#define CAN_TXMDH0_DATABYTE5_Msk                                    (0xFFUL << CAN_TXMDH0_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDH0_DATABYTE5                                        CAN_TXMDH0_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_TXMDH0_DATABYTE6_Pos                                    (16U)
#define CAN_TXMDH0_DATABYTE6_Msk                                    (0xFFUL << CAN_TXMDH0_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDH0_DATABYTE6                                        CAN_TXMDH0_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_TXMDH0_DATABYTE7_Pos                                    (24U)
#define CAN_TXMDH0_DATABYTE7_Msk                                    (0xFFUL << CAN_TXMDH0_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDH0_DATABYTE7                                        CAN_TXMDH0_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*******************  Bit definition for CAN_TXMID1 register  *******************/
#define CAN_TXMID1_TXMREQ_Pos                                       (0U)
#define CAN_TXMID1_TXMREQ_Msk                                       (0x1UL << CAN_TXMID1_TXMREQ_Pos)                               /*!< 0x00000001 */
#define CAN_TXMID1_TXMREQ                                           CAN_TXMID1_TXMREQ_Msk                                          /*!<Transmit Mailbox Request */
#define CAN_TXMID1_TXRFREQ_Pos                                      (1U)
#define CAN_TXMID1_TXRFREQ_Msk                                      (0x1UL << CAN_TXMID1_TXRFREQ_Pos)                              /*!< 0x00000002 */
#define CAN_TXMID1_TXRFREQ                                          CAN_TXMID1_TXRFREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_TXMID1_IDTYPESEL_Pos                                    (2U)
#define CAN_TXMID1_IDTYPESEL_Msk                                    (0x1UL << CAN_TXMID1_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_TXMID1_IDTYPESEL                                        CAN_TXMID1_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_TXMID1_EXTID_Pos                                        (3U)
#define CAN_TXMID1_EXTID_Msk                                        (0x3FFFFUL << CAN_TXMID1_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_TXMID1_EXTID                                            CAN_TXMID1_EXTID_Msk                                           /*!<Extended Identifier */
#define CAN_TXMID1_STDID_Pos                                        (21U)
#define CAN_TXMID1_STDID_Msk                                        (0x7FFUL << CAN_TXMID1_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_TXMID1_STDID                                            CAN_TXMID1_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_TXDLEN1 register  ******************/
#define CAN_TXDLEN1_DLCODE_Pos                                      (0U)
#define CAN_TXDLEN1_DLCODE_Msk                                      (0xFUL << CAN_TXDLEN1_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_TXDLEN1_DLCODE                                          CAN_TXDLEN1_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_TXDLEN1_TXTS_Pos                                        (8U)
#define CAN_TXDLEN1_TXTS_Msk                                        (0x1UL << CAN_TXDLEN1_TXTS_Pos)                                /*!< 0x00000100 */
#define CAN_TXDLEN1_TXTS                                            CAN_TXDLEN1_TXTS_Msk                                           /*!<Transmit Global Time */
#define CAN_TXDLEN1_MTS_Pos                                         (16U)
#define CAN_TXDLEN1_MTS_Msk                                         (0xFFFFUL << CAN_TXDLEN1_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_TXDLEN1_MTS                                             CAN_TXDLEN1_MTS_Msk                                            /*!<Message Time Stamp */

/*******************  Bit definition for CAN_TXMDL1 register  ******************/
#define CAN_TXMDL1_DATABYTE0_Pos                                    (0U)
#define CAN_TXMDL1_DATABYTE0_Msk                                    (0xFFUL << CAN_TXMDL1_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDL1_DATABYTE0                                        CAN_TXMDL1_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_TXMDL1_DATABYTE1_Pos                                    (8U)
#define CAN_TXMDL1_DATABYTE1_Msk                                    (0xFFUL << CAN_TXMDL1_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDL1_DATABYTE1                                        CAN_TXMDL1_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_TXMDL1_DATABYTE2_Pos                                    (16U)
#define CAN_TXMDL1_DATABYTE2_Msk                                    (0xFFUL << CAN_TXMDL1_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDL1_DATABYTE2                                        CAN_TXMDL1_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_TXMDL1_DATABYTE3_Pos                                    (24U)
#define CAN_TXMDL1_DATABYTE3_Msk                                    (0xFFUL << CAN_TXMDL1_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDL1_DATABYTE3                                        CAN_TXMDL1_DATABYTE3_Msk                                       /*!<Data byte 3 */

/*******************  Bit definition for CAN_TXMDH1 register  ******************/
#define CAN_TXMDH1_DATABYTE4_Pos                                    (0U)
#define CAN_TXMDH1_DATABYTE4_Msk                                    (0xFFUL << CAN_TXMDH1_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDH1_DATABYTE4                                        CAN_TXMDH1_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_TXMDH1_DATABYTE5_Pos                                    (8U)
#define CAN_TXMDH1_DATABYTE5_Msk                                    (0xFFUL << CAN_TXMDH1_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDH1_DATABYTE5                                        CAN_TXMDH1_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_TXMDH1_DATABYTE6_Pos                                    (16U)
#define CAN_TXMDH1_DATABYTE6_Msk                                    (0xFFUL << CAN_TXMDH1_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDH1_DATABYTE6                                        CAN_TXMDH1_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_TXMDH1_DATABYTE7_Pos                                    (24U)
#define CAN_TXMDH1_DATABYTE7_Msk                                    (0xFFUL << CAN_TXMDH1_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDH1_DATABYTE7                                        CAN_TXMDH1_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*******************  Bit definition for CAN_TXMID2 register  *******************/
#define CAN_TXMID2_TXMREQ_Pos                                       (0U)
#define CAN_TXMID2_TXMREQ_Msk                                       (0x1UL << CAN_TXMID2_TXMREQ_Pos)                               /*!< 0x00000001 */
#define CAN_TXMID2_TXMREQ                                           CAN_TXMID2_TXMREQ_Msk                                          /*!<Transmit Mailbox Request */
#define CAN_TXMID2_TXRFREQ_Pos                                      (1U)
#define CAN_TXMID2_TXRFREQ_Msk                                      (0x1UL << CAN_TXMID2_TXRFREQ_Pos)                              /*!< 0x00000002 */
#define CAN_TXMID2_TXRFREQ                                          CAN_TXMID2_TXRFREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_TXMID2_IDTYPESEL_Pos                                    (2U)
#define CAN_TXMID2_IDTYPESEL_Msk                                    (0x1UL << CAN_TXMID2_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_TXMID2_IDTYPESEL                                        CAN_TXMID2_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_TXMID2_EXTID_Pos                                        (3U)
#define CAN_TXMID2_EXTID_Msk                                        (0x3FFFFUL << CAN_TXMID2_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_TXMID2_EXTID                                            CAN_TXMID2_EXTID_Msk                                           /*!<Extended identifier */
#define CAN_TXMID2_STDID_Pos                                        (21U)
#define CAN_TXMID2_STDID_Msk                                        (0x7FFUL << CAN_TXMID2_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_TXMID2_STDID                                            CAN_TXMID2_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_TXDLEN2 register  ******************/
#define CAN_TXDLEN2_DLCODE_Pos                                      (0U)
#define CAN_TXDLEN2_DLCODE_Msk                                      (0xFUL << CAN_TXDLEN2_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_TXDLEN2_DLCODE                                          CAN_TXDLEN2_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_TXDLEN2_TXTS_Pos                                        (8U)
#define CAN_TXDLEN2_TXTS_Msk                                        (0x1UL << CAN_TXDLEN2_TXTS_Pos)                                /*!< 0x00000100 */
#define CAN_TXDLEN2_TXTS                                            CAN_TXDLEN2_TXTS_Msk                                           /*!<Transmit Global Time */
#define CAN_TXDLEN2_MTS_Pos                                         (16U)
#define CAN_TXDLEN2_MTS_Msk                                         (0xFFFFUL << CAN_TXDLEN2_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_TXDLEN2_MTS                                             CAN_TXDLEN2_MTS_Msk                                            /*!<Message Time Stamp */

/*******************  Bit definition for CAN_TXMDL2 register  ******************/
#define CAN_TXMDL2_DATABYTE0_Pos                                    (0U)
#define CAN_TXMDL2_DATABYTE0_Msk                                    (0xFFUL << CAN_TXMDL2_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDL2_DATABYTE0                                        CAN_TXMDL2_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_TXMDL2_DATABYTE1_Pos                                    (8U)
#define CAN_TXMDL2_DATABYTE1_Msk                                    (0xFFUL << CAN_TXMDL2_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDL2_DATABYTE1                                        CAN_TXMDL2_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_TXMDL2_DATABYTE2_Pos                                    (16U)
#define CAN_TXMDL2_DATABYTE2_Msk                                    (0xFFUL << CAN_TXMDL2_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDL2_DATABYTE2                                        CAN_TXMDL2_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_TXMDL2_DATABYTE3_Pos                                    (24U)
#define CAN_TXMDL2_DATABYTE3_Msk                                    (0xFFUL << CAN_TXMDL2_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDL2_DATABYTE3                                        CAN_TXMDL2_DATABYTE3_Msk                                       /*!<Data byte 3 */

/*******************  Bit definition for CAN_TXMDH2 register  ******************/
#define CAN_TXMDH2_DATABYTE4_Pos                                    (0U)
#define CAN_TXMDH2_DATABYTE4_Msk                                    (0xFFUL << CAN_TXMDH2_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDH2_DATABYTE4                                        CAN_TXMDH2_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_TXMDH2_DATABYTE5_Pos                                    (8U)
#define CAN_TXMDH2_DATABYTE5_Msk                                    (0xFFUL << CAN_TXMDH2_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDH2_DATABYTE5                                        CAN_TXMDH2_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_TXMDH2_DATABYTE6_Pos                                    (16U)
#define CAN_TXMDH2_DATABYTE6_Msk                                    (0xFFUL << CAN_TXMDH2_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDH2_DATABYTE6                                        CAN_TXMDH2_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_TXMDH2_DATABYTE7_Pos                                    (24U)
#define CAN_TXMDH2_DATABYTE7_Msk                                    (0xFFUL << CAN_TXMDH2_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDH2_DATABYTE7                                        CAN_TXMDH2_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*******************  Bit definition for CAN_RXMID0 register  *******************/
#define CAN_RXMID0_RFTXREQ_Pos                                      (1U)
#define CAN_RXMID0_RFTXREQ_Msk                                      (0x1UL << CAN_RXMID0_RFTXREQ_Pos)                              /*!< 0x00000002 */
#define CAN_RXMID0_RFTXREQ                                          CAN_RXMID0_RFTXREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_RXMID0_IDTYPESEL_Pos                                    (2U)
#define CAN_RXMID0_IDTYPESEL_Msk                                    (0x1UL << CAN_RXMID0_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_RXMID0_IDTYPESEL                                        CAN_RXMID0_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_RXMID0_EXTID_Pos                                        (3U)
#define CAN_RXMID0_EXTID_Msk                                        (0x3FFFFUL << CAN_RXMID0_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_RXMID0_EXTID                                            CAN_RXMID0_EXTID_Msk                                           /*!<Extended Identifier */
#define CAN_RXMID0_STDID_Pos                                        (21U)
#define CAN_RXMID0_STDID_Msk                                        (0x7FFUL << CAN_RXMID0_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_RXMID0_STDID                                            CAN_RXMID0_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_RXDLEN0 register  ******************/
#define CAN_RXDLEN0_DLCODE_Pos                                      (0U)
#define CAN_RXDLEN0_DLCODE_Msk                                      (0xFUL << CAN_RXDLEN0_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_RXDLEN0_DLCODE                                          CAN_RXDLEN0_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_RXDLEN0_FMIDX_Pos                                       (8U)
#define CAN_RXDLEN0_FMIDX_Msk                                       (0xFFUL << CAN_RXDLEN0_FMIDX_Pos)                              /*!< 0x0000FF00 */
#define CAN_RXDLEN0_FMIDX                                           CAN_RXDLEN0_FMIDX_Msk                                          /*!<Filter Match Index */
#define CAN_RXDLEN0_MTS_Pos                                         (16U)
#define CAN_RXDLEN0_MTS_Msk                                         (0xFFFFUL << CAN_RXDLEN0_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_RXDLEN0_MTS                                             CAN_RXDLEN0_MTS_Msk                                            /*!<Message Time Stamp */

/*******************  Bit definition for CAN_RXMDL0 register  ******************/
#define CAN_RXMDL0_DATABYTE0_Pos                                    (0U)
#define CAN_RXMDL0_DATABYTE0_Msk                                    (0xFFUL << CAN_RXMDL0_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_RXMDL0_DATABYTE0                                        CAN_RXMDL0_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_RXMDL0_DATABYTE1_Pos                                    (8U)
#define CAN_RXMDL0_DATABYTE1_Msk                                    (0xFFUL << CAN_RXMDL0_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_RXMDL0_DATABYTE1                                        CAN_RXMDL0_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_RXMDL0_DATABYTE2_Pos                                    (16U)
#define CAN_RXMDL0_DATABYTE2_Msk                                    (0xFFUL << CAN_RXMDL0_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_RXMDL0_DATABYTE2                                        CAN_RXMDL0_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_RXMDL0_DATABYTE3_Pos                                    (24U)
#define CAN_RXMDL0_DATABYTE3_Msk                                    (0xFFUL << CAN_RXMDL0_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_RXMDL0_DATABYTE3                                        CAN_RXMDL0_DATABYTE3_Msk                                       /*!<Data byte 3 */

/*******************  Bit definition for CAN_RXMDH0 register  ******************/
#define CAN_RXMDH0_DATABYTE4_Pos                                    (0U)
#define CAN_RXMDH0_DATABYTE4_Msk                                    (0xFFUL << CAN_RXMDH0_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_RXMDH0_DATABYTE4                                        CAN_RXMDH0_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_RXMDH0_DATABYTE5_Pos                                    (8U)
#define CAN_RXMDH0_DATABYTE5_Msk                                    (0xFFUL << CAN_RXMDH0_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_RXMDH0_DATABYTE5                                        CAN_RXMDH0_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_RXMDH0_DATABYTE6_Pos                                    (16U)
#define CAN_RXMDH0_DATABYTE6_Msk                                    (0xFFUL << CAN_RXMDH0_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_RXMDH0_DATABYTE6                                        CAN_RXMDH0_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_RXMDH0_DATABYTE7_Pos                                    (24U)
#define CAN_RXMDH0_DATABYTE7_Msk                                    (0xFFUL << CAN_RXMDH0_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_RXMDH0_DATABYTE7                                        CAN_RXMDH0_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*******************  Bit definition for CAN_RXMID1 register  *******************/
#define CAN_RXMID1_RFTXREQ_Pos                                      (1U)
#define CAN_RXMID1_RFTXREQ_Msk                                      (0x1UL << CAN_RXMID1_RFTXREQ_Pos)                              /*!< 0x00000002 */
#define CAN_RXMID1_RFTXREQ                                          CAN_RXMID1_RFTXREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_RXMID1_IDTYPESEL_Pos                                    (2U)
#define CAN_RXMID1_IDTYPESEL_Msk                                    (0x1UL << CAN_RXMID1_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_RXMID1_IDTYPESEL                                        CAN_RXMID1_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_RXMID1_EXTID_Pos                                        (3U)
#define CAN_RXMID1_EXTID_Msk                                        (0x3FFFFUL << CAN_RXMID1_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_RXMID1_EXTID                                            CAN_RXMID1_EXTID_Msk                                           /*!<Extended identifier */
#define CAN_RXMID1_STDID_Pos                                        (21U)
#define CAN_RXMID1_STDID_Msk                                        (0x7FFUL << CAN_RXMID1_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_RXMID1_STDID                                            CAN_RXMID1_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_RXDLEN1 register  ******************/
#define CAN_RXDLEN1_DLCODE_Pos                                      (0U)
#define CAN_RXDLEN1_DLCODE_Msk                                      (0xFUL << CAN_RXDLEN1_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_RXDLEN1_DLCODE                                          CAN_RXDLEN1_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_RXDLEN1_FMIDX_Pos                                       (8U)
#define CAN_RXDLEN1_FMIDX_Msk                                       (0xFFUL << CAN_RXDLEN1_FMIDX_Pos)                              /*!< 0x0000FF00 */
#define CAN_RXDLEN1_FMIDX                                           CAN_RXDLEN1_FMIDX_Msk                                          /*!<Filter Match Index */
#define CAN_RXDLEN1_MTS_Pos                                         (16U)
#define CAN_RXDLEN1_MTS_Msk                                         (0xFFFFUL << CAN_RXDLEN1_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_RXDLEN1_MTS                                             CAN_RXDLEN1_MTS_Msk                                            /*!<Message Time Stamp */

/*******************  Bit definition for CAN_RXMDL1 register  ******************/
#define CAN_RXMDL1_DATABYTE0_Pos                                    (0U)
#define CAN_RXMDL1_DATABYTE0_Msk                                    (0xFFUL << CAN_RXMDL1_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_RXMDL1_DATABYTE0                                        CAN_RXMDL1_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_RXMDL1_DATABYTE1_Pos                                    (8U)
#define CAN_RXMDL1_DATABYTE1_Msk                                    (0xFFUL << CAN_RXMDL1_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_RXMDL1_DATABYTE1                                        CAN_RXMDL1_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_RXMDL1_DATABYTE2_Pos                                    (16U)
#define CAN_RXMDL1_DATABYTE2_Msk                                    (0xFFUL << CAN_RXMDL1_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_RXMDL1_DATABYTE2                                        CAN_RXMDL1_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_RXMDL1_DATABYTE3_Pos                                    (24U)
#define CAN_RXMDL1_DATABYTE3_Msk                                    (0xFFUL << CAN_RXMDL1_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_RXMDL1_DATABYTE3                                        CAN_RXMDL1_DATABYTE3_Msk                                       /*!<Data byte 3 */

/*******************  Bit definition for CAN_RXMDH1 register  ******************/
#define CAN_RXMDH1_DATABYTE4_Pos                                    (0U)
#define CAN_RXMDH1_DATABYTE4_Msk                                    (0xFFUL << CAN_RXMDH1_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_RXMDH1_DATABYTE4                                        CAN_RXMDH1_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_RXMDH1_DATABYTE5_Pos                                    (8U)
#define CAN_RXMDH1_DATABYTE5_Msk                                    (0xFFUL << CAN_RXMDH1_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_RXMDH1_DATABYTE5                                        CAN_RXMDH1_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_RXMDH1_DATABYTE6_Pos                                    (16U)
#define CAN_RXMDH1_DATABYTE6_Msk                                    (0xFFUL << CAN_RXMDH1_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_RXMDH1_DATABYTE6                                        CAN_RXMDH1_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_RXMDH1_DATABYTE7_Pos                                    (24U)
#define CAN_RXMDH1_DATABYTE7_Msk                                    (0xFFUL << CAN_RXMDH1_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_RXMDH1_DATABYTE7                                        CAN_RXMDH1_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*!<CAN filter registers */
/*******************  Bit definition for CAN_FCTRL register  ********************/
#define CAN_FCTRL_FINITEN_Pos                                       (0U)
#define CAN_FCTRL_FINITEN_Msk                                       (0x1UL << CAN_FCTRL_FINITEN_Pos)                               /*!< 0x00000001 */
#define CAN_FCTRL_FINITEN                                           CAN_FCTRL_FINITEN_Msk                                          /*!<Filter Init Mode */
#define CAN_FCTRL_CAN2SB_Pos                                        (8U)
#define CAN_FCTRL_CAN2SB_Msk                                        (0x3FUL << CAN_FCTRL_CAN2SB_Pos)                               /*!< 0x00003F00 */
#define CAN_FCTRL_CAN2SB                                            CAN_FCTRL_CAN2SB_Msk                                           /*!<CAN2 start bank */

/*******************  Bit definition for CAN_FMCFG register  *******************/
#define CAN_FMCFG_FMCFG_Pos                                         (0U)
#define CAN_FMCFG_FMCFG_Msk                                         (0xFFFFFFFUL << CAN_FMCFG_FMCFG_Pos)                           /*!< 0x0FFFFFFF */
#define CAN_FMCFG_FMCFG                                             CAN_FMCFG_FMCFG_Msk                                            /*!<Filter Mode */
#define CAN_FMCFG_FMCFG0_Pos                                        (0U)
#define CAN_FMCFG_FMCFG0_Msk                                        (0x1UL << CAN_FMCFG_FMCFG0_Pos)                                /*!< 0x00000001 */
#define CAN_FMCFG_FMCFG0                                            CAN_FMCFG_FMCFG0_Msk                                           /*!<Filter Init Mode bit 0 */
#define CAN_FMCFG_FMCFG1_Pos                                        (1U)
#define CAN_FMCFG_FMCFG1_Msk                                        (0x1UL << CAN_FMCFG_FMCFG1_Pos)                                /*!< 0x00000002 */
#define CAN_FMCFG_FMCFG1                                            CAN_FMCFG_FMCFG1_Msk                                           /*!<Filter Init Mode bit 1 */
#define CAN_FMCFG_FMCFG2_Pos                                        (2U)
#define CAN_FMCFG_FMCFG2_Msk                                        (0x1UL << CAN_FMCFG_FMCFG2_Pos)                                /*!< 0x00000004 */
#define CAN_FMCFG_FMCFG2                                            CAN_FMCFG_FMCFG2_Msk                                           /*!<Filter Init Mode bit 2 */
#define CAN_FMCFG_FMCFG3_Pos                                        (3U)
#define CAN_FMCFG_FMCFG3_Msk                                        (0x1UL << CAN_FMCFG_FMCFG3_Pos)                                /*!< 0x00000008 */
#define CAN_FMCFG_FMCFG3                                            CAN_FMCFG_FMCFG3_Msk                                           /*!<Filter Init Mode bit 3 */
#define CAN_FMCFG_FMCFG4_Pos                                        (4U)
#define CAN_FMCFG_FMCFG4_Msk                                        (0x1UL << CAN_FMCFG_FMCFG4_Pos)                                /*!< 0x00000010 */
#define CAN_FMCFG_FMCFG4                                            CAN_FMCFG_FMCFG4_Msk                                           /*!<Filter Init Mode bit 4 */
#define CAN_FMCFG_FMCFG5_Pos                                        (5U)
#define CAN_FMCFG_FMCFG5_Msk                                        (0x1UL << CAN_FMCFG_FMCFG5_Pos)                                /*!< 0x00000020 */
#define CAN_FMCFG_FMCFG5                                            CAN_FMCFG_FMCFG5_Msk                                           /*!<Filter Init Mode bit 5 */
#define CAN_FMCFG_FMCFG6_Pos                                        (6U)
#define CAN_FMCFG_FMCFG6_Msk                                        (0x1UL << CAN_FMCFG_FMCFG6_Pos)                                /*!< 0x00000040 */
#define CAN_FMCFG_FMCFG6                                            CAN_FMCFG_FMCFG6_Msk                                           /*!<Filter Init Mode bit 6 */
#define CAN_FMCFG_FMCFG7_Pos                                        (7U)
#define CAN_FMCFG_FMCFG7_Msk                                        (0x1UL << CAN_FMCFG_FMCFG7_Pos)                                /*!< 0x00000080 */
#define CAN_FMCFG_FMCFG7                                            CAN_FMCFG_FMCFG7_Msk                                           /*!<Filter Init Mode bit 7 */
#define CAN_FMCFG_FMCFG8_Pos                                        (8U)
#define CAN_FMCFG_FMCFG8_Msk                                        (0x1UL << CAN_FMCFG_FMCFG8_Pos)                                /*!< 0x00000100 */
#define CAN_FMCFG_FMCFG8                                            CAN_FMCFG_FMCFG8_Msk                                           /*!<Filter Init Mode bit 8 */
#define CAN_FMCFG_FMCFG9_Pos                                        (9U)
#define CAN_FMCFG_FMCFG9_Msk                                        (0x1UL << CAN_FMCFG_FMCFG9_Pos)                                /*!< 0x00000200 */
#define CAN_FMCFG_FMCFG9                                            CAN_FMCFG_FMCFG9_Msk                                           /*!<Filter Init Mode bit 9 */
#define CAN_FMCFG_FMCFG10_Pos                                       (10U)
#define CAN_FMCFG_FMCFG10_Msk                                       (0x1UL << CAN_FMCFG_FMCFG10_Pos)                               /*!< 0x00000400 */
#define CAN_FMCFG_FMCFG10                                           CAN_FMCFG_FMCFG10_Msk                                          /*!<Filter Init Mode bit 10 */
#define CAN_FMCFG_FMCFG11_Pos                                       (11U)
#define CAN_FMCFG_FMCFG11_Msk                                       (0x1UL << CAN_FMCFG_FMCFG11_Pos)                               /*!< 0x00000800 */
#define CAN_FMCFG_FMCFG11                                           CAN_FMCFG_FMCFG11_Msk                                          /*!<Filter Init Mode bit 11 */
#define CAN_FMCFG_FMCFG12_Pos                                       (12U)
#define CAN_FMCFG_FMCFG12_Msk                                       (0x1UL << CAN_FMCFG_FMCFG12_Pos)                               /*!< 0x00001000 */
#define CAN_FMCFG_FMCFG12                                           CAN_FMCFG_FMCFG12_Msk                                          /*!<Filter Init Mode bit 12 */
#define CAN_FMCFG_FMCFG13_Pos                                       (13U)
#define CAN_FMCFG_FMCFG13_Msk                                       (0x1UL << CAN_FMCFG_FMCFG13_Pos)                               /*!< 0x00002000 */
#define CAN_FMCFG_FMCFG13                                           CAN_FMCFG_FMCFG13_Msk                                          /*!<Filter Init Mode bit 13 */
#define CAN_FMCFG_FMCFG14_Pos                                       (14U)
#define CAN_FMCFG_FMCFG14_Msk                                       (0x1UL << CAN_FMCFG_FMCFG14_Pos)                               /*!< 0x00004000 */
#define CAN_FMCFG_FMCFG14                                           CAN_FMCFG_FMCFG14_Msk                                          /*!<Filter Init Mode bit 14 */
#define CAN_FMCFG_FMCFG15_Pos                                       (15U)
#define CAN_FMCFG_FMCFG15_Msk                                       (0x1UL << CAN_FMCFG_FMCFG15_Pos)                               /*!< 0x00008000 */
#define CAN_FMCFG_FMCFG15                                           CAN_FMCFG_FMCFG15_Msk                                          /*!<Filter Init Mode bit 15 */
#define CAN_FMCFG_FMCFG16_Pos                                       (16U)
#define CAN_FMCFG_FMCFG16_Msk                                       (0x1UL << CAN_FMCFG_FMCFG16_Pos)                               /*!< 0x00010000 */
#define CAN_FMCFG_FMCFG16                                           CAN_FMCFG_FMCFG16_Msk                                          /*!<Filter Init Mode bit 16 */
#define CAN_FMCFG_FMCFG17_Pos                                       (17U)
#define CAN_FMCFG_FMCFG17_Msk                                       (0x1UL << CAN_FMCFG_FMCFG17_Pos)                               /*!< 0x00020000 */
#define CAN_FMCFG_FMCFG17                                           CAN_FMCFG_FMCFG17_Msk                                          /*!<Filter Init Mode bit 17 */
#define CAN_FMCFG_FMCFG18_Pos                                       (18U)
#define CAN_FMCFG_FMCFG18_Msk                                       (0x1UL << CAN_FMCFG_FMCFG18_Pos)                               /*!< 0x00040000 */
#define CAN_FMCFG_FMCFG18                                           CAN_FMCFG_FMCFG18_Msk                                          /*!<Filter Init Mode bit 18 */
#define CAN_FMCFG_FMCFG19_Pos                                       (19U)
#define CAN_FMCFG_FMCFG19_Msk                                       (0x1UL << CAN_FMCFG_FMCFG19_Pos)                               /*!< 0x00080000 */
#define CAN_FMCFG_FMCFG19                                           CAN_FMCFG_FMCFG19_Msk                                          /*!<Filter Init Mode bit 19 */
#define CAN_FMCFG_FMCFG20_Pos                                       (20U)
#define CAN_FMCFG_FMCFG20_Msk                                       (0x1UL << CAN_FMCFG_FMCFG20_Pos)                               /*!< 0x00100000 */
#define CAN_FMCFG_FMCFG20                                           CAN_FMCFG_FMCFG20_Msk                                          /*!<Filter Init Mode bit 20 */
#define CAN_FMCFG_FMCFG21_Pos                                       (21U)
#define CAN_FMCFG_FMCFG21_Msk                                       (0x1UL << CAN_FMCFG_FMCFG21_Pos)                               /*!< 0x00200000 */
#define CAN_FMCFG_FMCFG21                                           CAN_FMCFG_FMCFG21_Msk                                          /*!<Filter Init Mode bit 21 */
#define CAN_FMCFG_FMCFG22_Pos                                       (22U)
#define CAN_FMCFG_FMCFG22_Msk                                       (0x1UL << CAN_FMCFG_FMCFG22_Pos)                               /*!< 0x00400000 */
#define CAN_FMCFG_FMCFG22                                           CAN_FMCFG_FMCFG22_Msk                                          /*!<Filter Init Mode bit 22 */
#define CAN_FMCFG_FMCFG23_Pos                                       (23U)
#define CAN_FMCFG_FMCFG23_Msk                                       (0x1UL << CAN_FMCFG_FMCFG23_Pos)                               /*!< 0x00800000 */
#define CAN_FMCFG_FMCFG23                                           CAN_FMCFG_FMCFG23_Msk                                          /*!<Filter Init Mode bit 23 */
#define CAN_FMCFG_FMCFG24_Pos                                       (24U)
#define CAN_FMCFG_FMCFG24_Msk                                       (0x1UL << CAN_FMCFG_FMCFG24_Pos)                               /*!< 0x01000000 */
#define CAN_FMCFG_FMCFG24                                           CAN_FMCFG_FMCFG24_Msk                                          /*!<Filter Init Mode bit 24 */
#define CAN_FMCFG_FMCFG25_Pos                                       (25U)
#define CAN_FMCFG_FMCFG25_Msk                                       (0x1UL << CAN_FMCFG_FMCFG25_Pos)                               /*!< 0x02000000 */
#define CAN_FMCFG_FMCFG25                                           CAN_FMCFG_FMCFG25_Msk                                          /*!<Filter Init Mode bit 25 */
#define CAN_FMCFG_FMCFG26_Pos                                       (26U)
#define CAN_FMCFG_FMCFG26_Msk                                       (0x1UL << CAN_FMCFG_FMCFG26_Pos)                               /*!< 0x04000000 */
#define CAN_FMCFG_FMCFG26                                           CAN_FMCFG_FMCFG26_Msk                                          /*!<Filter Init Mode bit 26 */
#define CAN_FMCFG_FMCFG27_Pos                                       (27U)
#define CAN_FMCFG_FMCFG27_Msk                                       (0x1UL << CAN_FMCFG_FMCFG27_Pos)                               /*!< 0x08000000 */
#define CAN_FMCFG_FMCFG27                                           CAN_FMCFG_FMCFG27_Msk                                          /*!<Filter Init Mode bit 27 */

/*******************  Bit definition for CAN_FSCFG register  *******************/
#define CAN_FSCFG_FSCFG_Pos                                         (0U)
#define CAN_FSCFG_FSCFG_Msk                                         (0xFFFFFFFUL << CAN_FSCFG_FSCFG_Pos)                           /*!< 0x0FFFFFFF */
#define CAN_FSCFG_FSCFG                                             CAN_FSCFG_FSCFG_Msk                                            /*!<Filter Scale Configuration */
#define CAN_FSCFG_FSCFG0_Pos                                        (0U)
#define CAN_FSCFG_FSCFG0_Msk                                        (0x1UL << CAN_FSCFG_FSCFG0_Pos)                                /*!< 0x00000001 */
#define CAN_FSCFG_FSCFG0                                            CAN_FSCFG_FSCFG0_Msk                                           /*!<Filter Scale Configuration bit 0 */
#define CAN_FSCFG_FSCFG1_Pos                                        (1U)
#define CAN_FSCFG_FSCFG1_Msk                                        (0x1UL << CAN_FSCFG_FSCFG1_Pos)                                /*!< 0x00000002 */
#define CAN_FSCFG_FSCFG1                                            CAN_FSCFG_FSCFG1_Msk                                           /*!<Filter Scale Configuration bit 1 */
#define CAN_FSCFG_FSCFG2_Pos                                        (2U)
#define CAN_FSCFG_FSCFG2_Msk                                        (0x1UL << CAN_FSCFG_FSCFG2_Pos)                                /*!< 0x00000004 */
#define CAN_FSCFG_FSCFG2                                            CAN_FSCFG_FSCFG2_Msk                                           /*!<Filter Scale Configuration bit 2 */
#define CAN_FSCFG_FSCFG3_Pos                                        (3U)
#define CAN_FSCFG_FSCFG3_Msk                                        (0x1UL << CAN_FSCFG_FSCFG3_Pos)                                /*!< 0x00000008 */
#define CAN_FSCFG_FSCFG3                                            CAN_FSCFG_FSCFG3_Msk                                           /*!<Filter Scale Configuration bit 3 */
#define CAN_FSCFG_FSCFG4_Pos                                        (4U)
#define CAN_FSCFG_FSCFG4_Msk                                        (0x1UL << CAN_FSCFG_FSCFG4_Pos)                                /*!< 0x00000010 */
#define CAN_FSCFG_FSCFG4                                            CAN_FSCFG_FSCFG4_Msk                                           /*!<Filter Scale Configuration bit 4 */
#define CAN_FSCFG_FSCFG5_Pos                                        (5U)
#define CAN_FSCFG_FSCFG5_Msk                                        (0x1UL << CAN_FSCFG_FSCFG5_Pos)                                /*!< 0x00000020 */
#define CAN_FSCFG_FSCFG5                                            CAN_FSCFG_FSCFG5_Msk                                           /*!<Filter Scale Configuration bit 5 */
#define CAN_FSCFG_FSCFG6_Pos                                        (6U)
#define CAN_FSCFG_FSCFG6_Msk                                        (0x1UL << CAN_FSCFG_FSCFG6_Pos)                                /*!< 0x00000040 */
#define CAN_FSCFG_FSCFG6                                            CAN_FSCFG_FSCFG6_Msk                                           /*!<Filter Scale Configuration bit 6 */
#define CAN_FSCFG_FSCFG7_Pos                                        (7U)
#define CAN_FSCFG_FSCFG7_Msk                                        (0x1UL << CAN_FSCFG_FSCFG7_Pos)                                /*!< 0x00000080 */
#define CAN_FSCFG_FSCFG7                                            CAN_FSCFG_FSCFG7_Msk                                           /*!<Filter Scale Configuration bit 7 */
#define CAN_FSCFG_FSCFG8_Pos                                        (8U)
#define CAN_FSCFG_FSCFG8_Msk                                        (0x1UL << CAN_FSCFG_FSCFG8_Pos)                                /*!< 0x00000100 */
#define CAN_FSCFG_FSCFG8                                            CAN_FSCFG_FSCFG8_Msk                                           /*!<Filter Scale Configuration bit 8 */
#define CAN_FSCFG_FSCFG9_Pos                                        (9U)
#define CAN_FSCFG_FSCFG9_Msk                                        (0x1UL << CAN_FSCFG_FSCFG9_Pos)                                /*!< 0x00000200 */
#define CAN_FSCFG_FSCFG9                                            CAN_FSCFG_FSCFG9_Msk                                           /*!<Filter Scale Configuration bit 9 */
#define CAN_FSCFG_FSCFG10_Pos                                       (10U)
#define CAN_FSCFG_FSCFG10_Msk                                       (0x1UL << CAN_FSCFG_FSCFG10_Pos)                               /*!< 0x00000400 */
#define CAN_FSCFG_FSCFG10                                           CAN_FSCFG_FSCFG10_Msk                                          /*!<Filter Scale Configuration bit 10 */
#define CAN_FSCFG_FSCFG11_Pos                                       (11U)
#define CAN_FSCFG_FSCFG11_Msk                                       (0x1UL << CAN_FSCFG_FSCFG11_Pos)                               /*!< 0x00000800 */
#define CAN_FSCFG_FSCFG11                                           CAN_FSCFG_FSCFG11_Msk                                          /*!<Filter Scale Configuration bit 11 */
#define CAN_FSCFG_FSCFG12_Pos                                       (12U)
#define CAN_FSCFG_FSCFG12_Msk                                       (0x1UL << CAN_FSCFG_FSCFG12_Pos)                               /*!< 0x00001000 */
#define CAN_FSCFG_FSCFG12                                           CAN_FSCFG_FSCFG12_Msk                                          /*!<Filter Scale Configuration bit 12 */
#define CAN_FSCFG_FSCFG13_Pos                                       (13U)
#define CAN_FSCFG_FSCFG13_Msk                                       (0x1UL << CAN_FSCFG_FSCFG13_Pos)                               /*!< 0x00002000 */
#define CAN_FSCFG_FSCFG13                                           CAN_FSCFG_FSCFG13_Msk                                          /*!<Filter Scale Configuration bit 13 */
#define CAN_FSCFG_FSCFG14_Pos                                       (14U)
#define CAN_FSCFG_FSCFG14_Msk                                       (0x1UL << CAN_FSCFG_FSCFG14_Pos)                               /*!< 0x00004000 */
#define CAN_FSCFG_FSCFG14                                           CAN_FSCFG_FSCFG14_Msk                                          /*!<Filter Scale Configuration bit 14 */
#define CAN_FSCFG_FSCFG15_Pos                                       (15U)
#define CAN_FSCFG_FSCFG15_Msk                                       (0x1UL << CAN_FSCFG_FSCFG15_Pos)                               /*!< 0x00008000 */
#define CAN_FSCFG_FSCFG15                                           CAN_FSCFG_FSCFG15_Msk                                          /*!<Filter Scale Configuration bit 15 */
#define CAN_FSCFG_FSCFG16_Pos                                       (16U)
#define CAN_FSCFG_FSCFG16_Msk                                       (0x1UL << CAN_FSCFG_FSCFG16_Pos)                               /*!< 0x00010000 */
#define CAN_FSCFG_FSCFG16                                           CAN_FSCFG_FSCFG16_Msk                                          /*!<Filter Scale Configuration bit 16 */
#define CAN_FSCFG_FSCFG17_Pos                                       (17U)
#define CAN_FSCFG_FSCFG17_Msk                                       (0x1UL << CAN_FSCFG_FSCFG17_Pos)                               /*!< 0x00020000 */
#define CAN_FSCFG_FSCFG17                                           CAN_FSCFG_FSCFG17_Msk                                          /*!<Filter Scale Configuration bit 17 */
#define CAN_FSCFG_FSCFG18_Pos                                       (18U)
#define CAN_FSCFG_FSCFG18_Msk                                       (0x1UL << CAN_FSCFG_FSCFG18_Pos)                               /*!< 0x00040000 */
#define CAN_FSCFG_FSCFG18                                           CAN_FSCFG_FSCFG18_Msk                                          /*!<Filter Scale Configuration bit 18 */
#define CAN_FSCFG_FSCFG19_Pos                                       (19U)
#define CAN_FSCFG_FSCFG19_Msk                                       (0x1UL << CAN_FSCFG_FSCFG19_Pos)                               /*!< 0x00080000 */
#define CAN_FSCFG_FSCFG19                                           CAN_FSCFG_FSCFG19_Msk                                          /*!<Filter Scale Configuration bit 19 */
#define CAN_FSCFG_FSCFG20_Pos                                       (20U)
#define CAN_FSCFG_FSCFG20_Msk                                       (0x1UL << CAN_FSCFG_FSCFG20_Pos)                               /*!< 0x00100000 */
#define CAN_FSCFG_FSCFG20                                           CAN_FSCFG_FSCFG20_Msk                                          /*!<Filter Scale Configuration bit 20 */
#define CAN_FSCFG_FSCFG21_Pos                                       (21U)
#define CAN_FSCFG_FSCFG21_Msk                                       (0x1UL << CAN_FSCFG_FSCFG21_Pos)                               /*!< 0x00200000 */
#define CAN_FSCFG_FSCFG21                                           CAN_FSCFG_FSCFG21_Msk                                          /*!<Filter Scale Configuration bit 21 */
#define CAN_FSCFG_FSCFG22_Pos                                       (22U)
#define CAN_FSCFG_FSCFG22_Msk                                       (0x1UL << CAN_FSCFG_FSCFG22_Pos)                               /*!< 0x00400000 */
#define CAN_FSCFG_FSCFG22                                           CAN_FSCFG_FSCFG22_Msk                                          /*!<Filter Scale Configuration bit 22 */
#define CAN_FSCFG_FSCFG23_Pos                                       (23U)
#define CAN_FSCFG_FSCFG23_Msk                                       (0x1UL << CAN_FSCFG_FSCFG23_Pos)                               /*!< 0x00800000 */
#define CAN_FSCFG_FSCFG23                                           CAN_FSCFG_FSCFG23_Msk                                          /*!<Filter Scale Configuration bit 23 */
#define CAN_FSCFG_FSCFG24_Pos                                       (24U)
#define CAN_FSCFG_FSCFG24_Msk                                       (0x1UL << CAN_FSCFG_FSCFG24_Pos)                               /*!< 0x01000000 */
#define CAN_FSCFG_FSCFG24                                           CAN_FSCFG_FSCFG24_Msk                                          /*!<Filter Scale Configuration bit 24 */
#define CAN_FSCFG_FSCFG25_Pos                                       (25U)
#define CAN_FSCFG_FSCFG25_Msk                                       (0x1UL << CAN_FSCFG_FSCFG25_Pos)                               /*!< 0x02000000 */
#define CAN_FSCFG_FSCFG25                                           CAN_FSCFG_FSCFG25_Msk                                          /*!<Filter Scale Configuration bit 25 */
#define CAN_FSCFG_FSCFG26_Pos                                       (26U)
#define CAN_FSCFG_FSCFG26_Msk                                       (0x1UL << CAN_FSCFG_FSCFG26_Pos)                               /*!< 0x04000000 */
#define CAN_FSCFG_FSCFG26                                           CAN_FSCFG_FSCFG26_Msk                                          /*!<Filter Scale Configuration bit 26 */
#define CAN_FSCFG_FSCFG27_Pos                                       (27U)
#define CAN_FSCFG_FSCFG27_Msk                                       (0x1UL << CAN_FSCFG_FSCFG27_Pos)                               /*!< 0x08000000 */
#define CAN_FSCFG_FSCFG27                                           CAN_FSCFG_FSCFG27_Msk                                          /*!<Filter Scale Configuration bit 27 */

/******************  Bit definition for CAN_FFASS register  *******************/
#define CAN_FFASS_FFASS_Pos                                         (0U)
#define CAN_FFASS_FFASS_Msk                                         (0xFFFFFFFUL << CAN_FFASS_FFASS_Pos)                           /*!< 0x0FFFFFFF */
#define CAN_FFASS_FFASS                                             CAN_FFASS_FFASS_Msk                                            /*!<Filter FIFO Assignment */
#define CAN_FFASS_FFASS0_Pos                                        (0U)
#define CAN_FFASS_FFASS0_Msk                                        (0x1UL << CAN_FFASS_FFASS0_Pos)                                /*!< 0x00000001 */
#define CAN_FFASS_FFASS0                                            CAN_FFASS_FFASS0_Msk                                           /*!<Filter FIFO Assignment bit 0 */
#define CAN_FFASS_FFASS1_Pos                                        (1U)
#define CAN_FFASS_FFASS1_Msk                                        (0x1UL << CAN_FFASS_FFASS1_Pos)                                /*!< 0x00000002 */
#define CAN_FFASS_FFASS1                                            CAN_FFASS_FFASS1_Msk                                           /*!<Filter FIFO Assignment bit 1 */
#define CAN_FFASS_FFASS2_Pos                                        (2U)
#define CAN_FFASS_FFASS2_Msk                                        (0x1UL << CAN_FFASS_FFASS2_Pos)                                /*!< 0x00000004 */
#define CAN_FFASS_FFASS2                                            CAN_FFASS_FFASS2_Msk                                           /*!<Filter FIFO Assignment bit 2 */
#define CAN_FFASS_FFASS3_Pos                                        (3U)
#define CAN_FFASS_FFASS3_Msk                                        (0x1UL << CAN_FFASS_FFASS3_Pos)                                /*!< 0x00000008 */
#define CAN_FFASS_FFASS3                                            CAN_FFASS_FFASS3_Msk                                           /*!<Filter FIFO Assignment bit 3 */
#define CAN_FFASS_FFASS4_Pos                                        (4U)
#define CAN_FFASS_FFASS4_Msk                                        (0x1UL << CAN_FFASS_FFASS4_Pos)                                /*!< 0x00000010 */
#define CAN_FFASS_FFASS4                                            CAN_FFASS_FFASS4_Msk                                           /*!<Filter FIFO Assignment bit 4 */
#define CAN_FFASS_FFASS5_Pos                                        (5U)
#define CAN_FFASS_FFASS5_Msk                                        (0x1UL << CAN_FFASS_FFASS5_Pos)                                /*!< 0x00000020 */
#define CAN_FFASS_FFASS5                                            CAN_FFASS_FFASS5_Msk                                           /*!<Filter FIFO Assignment bit 5 */
#define CAN_FFASS_FFASS6_Pos                                        (6U)
#define CAN_FFASS_FFASS6_Msk                                        (0x1UL << CAN_FFASS_FFASS6_Pos)                                /*!< 0x00000040 */
#define CAN_FFASS_FFASS6                                            CAN_FFASS_FFASS6_Msk                                           /*!<Filter FIFO Assignment bit 6 */
#define CAN_FFASS_FFASS7_Pos                                        (7U)
#define CAN_FFASS_FFASS7_Msk                                        (0x1UL << CAN_FFASS_FFASS7_Pos)                                /*!< 0x00000080 */
#define CAN_FFASS_FFASS7                                            CAN_FFASS_FFASS7_Msk                                           /*!<Filter FIFO Assignment bit 7 */
#define CAN_FFASS_FFASS8_Pos                                        (8U)
#define CAN_FFASS_FFASS8_Msk                                        (0x1UL << CAN_FFASS_FFASS8_Pos)                                /*!< 0x00000100 */
#define CAN_FFASS_FFASS8                                            CAN_FFASS_FFASS8_Msk                                           /*!<Filter FIFO Assignment bit 8 */
#define CAN_FFASS_FFASS9_Pos                                        (9U)
#define CAN_FFASS_FFASS9_Msk                                        (0x1UL << CAN_FFASS_FFASS9_Pos)                                /*!< 0x00000200 */
#define CAN_FFASS_FFASS9                                            CAN_FFASS_FFASS9_Msk                                           /*!<Filter FIFO Assignment bit 9 */
#define CAN_FFASS_FFASS10_Pos                                       (10U)
#define CAN_FFASS_FFASS10_Msk                                       (0x1UL << CAN_FFASS_FFASS10_Pos)                               /*!< 0x00000400 */
#define CAN_FFASS_FFASS10                                           CAN_FFASS_FFASS10_Msk                                          /*!<Filter FIFO Assignment bit 10 */
#define CAN_FFASS_FFASS11_Pos                                       (11U)
#define CAN_FFASS_FFASS11_Msk                                       (0x1UL << CAN_FFASS_FFASS11_Pos)                               /*!< 0x00000800 */
#define CAN_FFASS_FFASS11                                           CAN_FFASS_FFASS11_Msk                                          /*!<Filter FIFO Assignment bit 11 */
#define CAN_FFASS_FFASS12_Pos                                       (12U)
#define CAN_FFASS_FFASS12_Msk                                       (0x1UL << CAN_FFASS_FFASS12_Pos)                               /*!< 0x00001000 */
#define CAN_FFASS_FFASS12                                           CAN_FFASS_FFASS12_Msk                                          /*!<Filter FIFO Assignment bit 12 */
#define CAN_FFASS_FFASS13_Pos                                       (13U)
#define CAN_FFASS_FFASS13_Msk                                       (0x1UL << CAN_FFASS_FFASS13_Pos)                               /*!< 0x00002000 */
#define CAN_FFASS_FFASS13                                           CAN_FFASS_FFASS13_Msk                                          /*!<Filter FIFO Assignment bit 13 */
#define CAN_FFASS_FFASS14_Pos                                       (14U)
#define CAN_FFASS_FFASS14_Msk                                       (0x1UL << CAN_FFASS_FFASS14_Pos)                               /*!< 0x00004000 */
#define CAN_FFASS_FFASS14                                           CAN_FFASS_FFASS14_Msk                                          /*!<Filter FIFO Assignment bit 14 */
#define CAN_FFASS_FFASS15_Pos                                       (15U)
#define CAN_FFASS_FFASS15_Msk                                       (0x1UL << CAN_FFASS_FFASS15_Pos)                               /*!< 0x00008000 */
#define CAN_FFASS_FFASS15                                           CAN_FFASS_FFASS15_Msk                                          /*!<Filter FIFO Assignment bit 15 */
#define CAN_FFASS_FFASS16_Pos                                       (16U)
#define CAN_FFASS_FFASS16_Msk                                       (0x1UL << CAN_FFASS_FFASS16_Pos)                               /*!< 0x00010000 */
#define CAN_FFASS_FFASS16                                           CAN_FFASS_FFASS16_Msk                                          /*!<Filter FIFO Assignment bit 16 */
#define CAN_FFASS_FFASS17_Pos                                       (17U)
#define CAN_FFASS_FFASS17_Msk                                       (0x1UL << CAN_FFASS_FFASS17_Pos)                               /*!< 0x00020000 */
#define CAN_FFASS_FFASS17                                           CAN_FFASS_FFASS17_Msk                                          /*!<Filter FIFO Assignment bit 17 */
#define CAN_FFASS_FFASS18_Pos                                       (18U)
#define CAN_FFASS_FFASS18_Msk                                       (0x1UL << CAN_FFASS_FFASS18_Pos)                               /*!< 0x00040000 */
#define CAN_FFASS_FFASS18                                           CAN_FFASS_FFASS18_Msk                                          /*!<Filter FIFO Assignment bit 18 */
#define CAN_FFASS_FFASS19_Pos                                       (19U)
#define CAN_FFASS_FFASS19_Msk                                       (0x1UL << CAN_FFASS_FFASS19_Pos)                               /*!< 0x00080000 */
#define CAN_FFASS_FFASS19                                           CAN_FFASS_FFASS19_Msk                                          /*!<Filter FIFO Assignment bit 19 */
#define CAN_FFASS_FFASS20_Pos                                       (20U)
#define CAN_FFASS_FFASS20_Msk                                       (0x1UL << CAN_FFASS_FFASS20_Pos)                               /*!< 0x00100000 */
#define CAN_FFASS_FFASS20                                           CAN_FFASS_FFASS20_Msk                                          /*!<Filter FIFO Assignment bit 20 */
#define CAN_FFASS_FFASS21_Pos                                       (21U)
#define CAN_FFASS_FFASS21_Msk                                       (0x1UL << CAN_FFASS_FFASS21_Pos)                               /*!< 0x00200000 */
#define CAN_FFASS_FFASS21                                           CAN_FFASS_FFASS21_Msk                                          /*!<Filter FIFO Assignment bit 21 */
#define CAN_FFASS_FFASS22_Pos                                       (22U)
#define CAN_FFASS_FFASS22_Msk                                       (0x1UL << CAN_FFASS_FFASS22_Pos)                               /*!< 0x00400000 */
#define CAN_FFASS_FFASS22                                           CAN_FFASS_FFASS22_Msk                                          /*!<Filter FIFO Assignment bit 22 */
#define CAN_FFASS_FFASS23_Pos                                       (23U)
#define CAN_FFASS_FFASS23_Msk                                       (0x1UL << CAN_FFASS_FFASS23_Pos)                               /*!< 0x00800000 */
#define CAN_FFASS_FFASS23                                           CAN_FFASS_FFASS23_Msk                                          /*!<Filter FIFO Assignment bit 23 */
#define CAN_FFASS_FFASS24_Pos                                       (24U)
#define CAN_FFASS_FFASS24_Msk                                       (0x1UL << CAN_FFASS_FFASS24_Pos)                               /*!< 0x01000000 */
#define CAN_FFASS_FFASS24                                           CAN_FFASS_FFASS24_Msk                                          /*!<Filter FIFO Assignment bit 24 */
#define CAN_FFASS_FFASS25_Pos                                       (25U)
#define CAN_FFASS_FFASS25_Msk                                       (0x1UL << CAN_FFASS_FFASS25_Pos)                               /*!< 0x02000000 */
#define CAN_FFASS_FFASS25                                           CAN_FFASS_FFASS25_Msk                                          /*!<Filter FIFO Assignment bit 25 */
#define CAN_FFASS_FFASS26_Pos                                       (26U)
#define CAN_FFASS_FFASS26_Msk                                       (0x1UL << CAN_FFASS_FFASS26_Pos)                               /*!< 0x04000000 */
#define CAN_FFASS_FFASS26                                           CAN_FFASS_FFASS26_Msk                                          /*!<Filter FIFO Assignment bit 26 */
#define CAN_FFASS_FFASS27_Pos                                       (27U)
#define CAN_FFASS_FFASS27_Msk                                       (0x1UL << CAN_FFASS_FFASS27_Pos)                               /*!< 0x08000000 */
#define CAN_FFASS_FFASS27                                           CAN_FFASS_FFASS27_Msk                                          /*!<Filter FIFO Assignment bit 27 */

/*******************  Bit definition for CAN_FACT register  *******************/
#define CAN_FACT_FACT_Pos                                           (0U)
#define CAN_FACT_FACT_Msk                                           (0xFFFFFFFUL << CAN_FACT_FACT_Pos)                             /*!< 0x0FFFFFFF */
#define CAN_FACT_FACT                                               CAN_FACT_FACT_Msk                                              /*!<Filter Active */
#define CAN_FACT_FACT0_Pos                                          (0U)
#define CAN_FACT_FACT0_Msk                                          (0x1UL << CAN_FACT_FACT0_Pos)                                  /*!< 0x00000001 */
#define CAN_FACT_FACT0                                              CAN_FACT_FACT0_Msk                                             /*!<Filter Active bit 0 */
#define CAN_FACT_FACT1_Pos                                          (1U)
#define CAN_FACT_FACT1_Msk                                          (0x1UL << CAN_FACT_FACT1_Pos)                                  /*!< 0x00000002 */
#define CAN_FACT_FACT1                                              CAN_FACT_FACT1_Msk                                             /*!<Filter Active bit 1 */
#define CAN_FACT_FACT2_Pos                                          (2U)
#define CAN_FACT_FACT2_Msk                                          (0x1UL << CAN_FACT_FACT2_Pos)                                  /*!< 0x00000004 */
#define CAN_FACT_FACT2                                              CAN_FACT_FACT2_Msk                                             /*!<Filter Active bit 2 */
#define CAN_FACT_FACT3_Pos                                          (3U)
#define CAN_FACT_FACT3_Msk                                          (0x1UL << CAN_FACT_FACT3_Pos)                                  /*!< 0x00000008 */
#define CAN_FACT_FACT3                                              CAN_FACT_FACT3_Msk                                             /*!<Filter Active bit 3 */
#define CAN_FACT_FACT4_Pos                                          (4U)
#define CAN_FACT_FACT4_Msk                                          (0x1UL << CAN_FACT_FACT4_Pos)                                  /*!< 0x00000010 */
#define CAN_FACT_FACT4                                              CAN_FACT_FACT4_Msk                                             /*!<Filter Active bit 4 */
#define CAN_FACT_FACT5_Pos                                          (5U)
#define CAN_FACT_FACT5_Msk                                          (0x1UL << CAN_FACT_FACT5_Pos)                                  /*!< 0x00000020 */
#define CAN_FACT_FACT5                                              CAN_FACT_FACT5_Msk                                             /*!<Filter Active bit 5 */
#define CAN_FACT_FACT6_Pos                                          (6U)
#define CAN_FACT_FACT6_Msk                                          (0x1UL << CAN_FACT_FACT6_Pos)                                  /*!< 0x00000040 */
#define CAN_FACT_FACT6                                              CAN_FACT_FACT6_Msk                                             /*!<Filter Active bit 6 */
#define CAN_FACT_FACT7_Pos                                          (7U)
#define CAN_FACT_FACT7_Msk                                          (0x1UL << CAN_FACT_FACT7_Pos)                                  /*!< 0x00000080 */
#define CAN_FACT_FACT7                                              CAN_FACT_FACT7_Msk                                             /*!<Filter Active bit 7 */
#define CAN_FACT_FACT8_Pos                                          (8U)
#define CAN_FACT_FACT8_Msk                                          (0x1UL << CAN_FACT_FACT8_Pos)                                  /*!< 0x00000100 */
#define CAN_FACT_FACT8                                              CAN_FACT_FACT8_Msk                                             /*!<Filter Active bit 8 */
#define CAN_FACT_FACT9_Pos                                          (9U)
#define CAN_FACT_FACT9_Msk                                          (0x1UL << CAN_FACT_FACT9_Pos)                                  /*!< 0x00000200 */
#define CAN_FACT_FACT9                                              CAN_FACT_FACT9_Msk                                             /*!<Filter Active bit 9 */
#define CAN_FACT_FACT10_Pos                                         (10U)
#define CAN_FACT_FACT10_Msk                                         (0x1UL << CAN_FACT_FACT10_Pos)                                 /*!< 0x00000400 */
#define CAN_FACT_FACT10                                             CAN_FACT_FACT10_Msk                                            /*!<Filter Active bit 10 */
#define CAN_FACT_FACT11_Pos                                         (11U)
#define CAN_FACT_FACT11_Msk                                         (0x1UL << CAN_FACT_FACT11_Pos)                                 /*!< 0x00000800 */
#define CAN_FACT_FACT11                                             CAN_FACT_FACT11_Msk                                            /*!<Filter Active bit 11 */
#define CAN_FACT_FACT12_Pos                                         (12U)
#define CAN_FACT_FACT12_Msk                                         (0x1UL << CAN_FACT_FACT12_Pos)                                 /*!< 0x00001000 */
#define CAN_FACT_FACT12                                             CAN_FACT_FACT12_Msk                                            /*!<Filter Active bit 12 */
#define CAN_FACT_FACT13_Pos                                         (13U)
#define CAN_FACT_FACT13_Msk                                         (0x1UL << CAN_FACT_FACT13_Pos)                                 /*!< 0x00002000 */
#define CAN_FACT_FACT13                                             CAN_FACT_FACT13_Msk                                            /*!<Filter Active bit 13 */
#define CAN_FACT_FACT14_Pos                                         (14U)
#define CAN_FACT_FACT14_Msk                                         (0x1UL << CAN_FACT_FACT14_Pos)                                 /*!< 0x00004000 */
#define CAN_FACT_FACT14                                             CAN_FACT_FACT14_Msk                                            /*!<Filter Active bit 14 */
#define CAN_FACT_FACT15_Pos                                         (15U)
#define CAN_FACT_FACT15_Msk                                         (0x1UL << CAN_FACT_FACT15_Pos)                                 /*!< 0x00008000 */
#define CAN_FACT_FACT15                                             CAN_FACT_FACT15_Msk                                            /*!<Filter Active bit 15 */
#define CAN_FACT_FACT16_Pos                                         (16U)
#define CAN_FACT_FACT16_Msk                                         (0x1UL << CAN_FACT_FACT16_Pos)                                 /*!< 0x00010000 */
#define CAN_FACT_FACT16                                             CAN_FACT_FACT16_Msk                                            /*!<Filter Active bit 16 */
#define CAN_FACT_FACT17_Pos                                         (17U)
#define CAN_FACT_FACT17_Msk                                         (0x1UL << CAN_FACT_FACT17_Pos)                                 /*!< 0x00020000 */
#define CAN_FACT_FACT17                                             CAN_FACT_FACT17_Msk                                            /*!<Filter Active bit 17 */
#define CAN_FACT_FACT18_Pos                                         (18U)
#define CAN_FACT_FACT18_Msk                                         (0x1UL << CAN_FACT_FACT18_Pos)                                 /*!< 0x00040000 */
#define CAN_FACT_FACT18                                             CAN_FACT_FACT18_Msk                                            /*!<Filter Active bit 18 */
#define CAN_FACT_FACT19_Pos                                         (19U)
#define CAN_FACT_FACT19_Msk                                         (0x1UL << CAN_FACT_FACT19_Pos)                                 /*!< 0x00080000 */
#define CAN_FACT_FACT19                                             CAN_FACT_FACT19_Msk                                            /*!<Filter Active bit 19 */
#define CAN_FACT_FACT20_Pos                                         (20U)
#define CAN_FACT_FACT20_Msk                                         (0x1UL << CAN_FACT_FACT20_Pos)                                 /*!< 0x00100000 */
#define CAN_FACT_FACT20                                             CAN_FACT_FACT20_Msk                                            /*!<Filter Active bit 20 */
#define CAN_FACT_FACT21_Pos                                         (21U)
#define CAN_FACT_FACT21_Msk                                         (0x1UL << CAN_FACT_FACT21_Pos)                                 /*!< 0x00200000 */
#define CAN_FACT_FACT21                                             CAN_FACT_FACT21_Msk                                            /*!<Filter Active bit 21 */
#define CAN_FACT_FACT22_Pos                                         (22U)
#define CAN_FACT_FACT22_Msk                                         (0x1UL << CAN_FACT_FACT22_Pos)                                 /*!< 0x00400000 */
#define CAN_FACT_FACT22                                             CAN_FACT_FACT22_Msk                                            /*!<Filter Active bit 22 */
#define CAN_FACT_FACT23_Pos                                         (23U)
#define CAN_FACT_FACT23_Msk                                         (0x1UL << CAN_FACT_FACT23_Pos)                                 /*!< 0x00800000 */
#define CAN_FACT_FACT23                                             CAN_FACT_FACT23_Msk                                            /*!<Filter Active bit 23 */
#define CAN_FACT_FACT24_Pos                                         (24U)
#define CAN_FACT_FACT24_Msk                                         (0x1UL << CAN_FACT_FACT24_Pos)                                 /*!< 0x01000000 */
#define CAN_FACT_FACT24                                             CAN_FACT_FACT24_Msk                                            /*!<Filter Active bit 24 */
#define CAN_FACT_FACT25_Pos                                         (25U)
#define CAN_FACT_FACT25_Msk                                         (0x1UL << CAN_FACT_FACT25_Pos)                                 /*!< 0x02000000 */
#define CAN_FACT_FACT25                                             CAN_FACT_FACT25_Msk                                            /*!<Filter Active bit 25 */
#define CAN_FACT_FACT26_Pos                                         (26U)
#define CAN_FACT_FACT26_Msk                                         (0x1UL << CAN_FACT_FACT26_Pos)                                 /*!< 0x04000000 */
#define CAN_FACT_FACT26                                             CAN_FACT_FACT26_Msk                                            /*!<Filter Active bit 26 */
#define CAN_FACT_FACT27_Pos                                         (27U)
#define CAN_FACT_FACT27_Msk                                         (0x1UL << CAN_FACT_FACT27_Pos)                                 /*!< 0x08000000 */
#define CAN_FACT_FACT27                                             CAN_FACT_FACT27_Msk                                            /*!<Filter Active bit 27 */


/*******************  Bit definition for CAN_F0BANK1 register  *******************/
#define CAN_F0BANK1_FBIT0_Pos                                       (0U)
#define CAN_F0BANK1_FBIT0_Msk                                       (0x1UL << CAN_F0BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F0BANK1_FBIT0                                           CAN_F0BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F0BANK1_FBIT1_Pos                                       (1U)
#define CAN_F0BANK1_FBIT1_Msk                                       (0x1UL << CAN_F0BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F0BANK1_FBIT1                                           CAN_F0BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F0BANK1_FBIT2_Pos                                       (2U)
#define CAN_F0BANK1_FBIT2_Msk                                       (0x1UL << CAN_F0BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F0BANK1_FBIT2                                           CAN_F0BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F0BANK1_FBIT3_Pos                                       (3U)
#define CAN_F0BANK1_FBIT3_Msk                                       (0x1UL << CAN_F0BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F0BANK1_FBIT3                                           CAN_F0BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F0BANK1_FBIT4_Pos                                       (4U)
#define CAN_F0BANK1_FBIT4_Msk                                       (0x1UL << CAN_F0BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F0BANK1_FBIT4                                           CAN_F0BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F0BANK1_FBIT5_Pos                                       (5U)
#define CAN_F0BANK1_FBIT5_Msk                                       (0x1UL << CAN_F0BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F0BANK1_FBIT5                                           CAN_F0BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F0BANK1_FBIT6_Pos                                       (6U)
#define CAN_F0BANK1_FBIT6_Msk                                       (0x1UL << CAN_F0BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F0BANK1_FBIT6                                           CAN_F0BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F0BANK1_FBIT7_Pos                                       (7U)
#define CAN_F0BANK1_FBIT7_Msk                                       (0x1UL << CAN_F0BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F0BANK1_FBIT7                                           CAN_F0BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F0BANK1_FBIT8_Pos                                       (8U)
#define CAN_F0BANK1_FBIT8_Msk                                       (0x1UL << CAN_F0BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F0BANK1_FBIT8                                           CAN_F0BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F0BANK1_FBIT9_Pos                                       (9U)
#define CAN_F0BANK1_FBIT9_Msk                                       (0x1UL << CAN_F0BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F0BANK1_FBIT9                                           CAN_F0BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F0BANK1_FBIT10_Pos                                      (10U)
#define CAN_F0BANK1_FBIT10_Msk                                      (0x1UL << CAN_F0BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F0BANK1_FBIT10                                          CAN_F0BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F0BANK1_FBIT11_Pos                                      (11U)
#define CAN_F0BANK1_FBIT11_Msk                                      (0x1UL << CAN_F0BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F0BANK1_FBIT11                                          CAN_F0BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F0BANK1_FBIT12_Pos                                      (12U)
#define CAN_F0BANK1_FBIT12_Msk                                      (0x1UL << CAN_F0BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F0BANK1_FBIT12                                          CAN_F0BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F0BANK1_FBIT13_Pos                                      (13U)
#define CAN_F0BANK1_FBIT13_Msk                                      (0x1UL << CAN_F0BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F0BANK1_FBIT13                                          CAN_F0BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F0BANK1_FBIT14_Pos                                      (14U)
#define CAN_F0BANK1_FBIT14_Msk                                      (0x1UL << CAN_F0BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F0BANK1_FBIT14                                          CAN_F0BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F0BANK1_FBIT15_Pos                                      (15U)
#define CAN_F0BANK1_FBIT15_Msk                                      (0x1UL << CAN_F0BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F0BANK1_FBIT15                                          CAN_F0BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F0BANK1_FBIT16_Pos                                      (16U)
#define CAN_F0BANK1_FBIT16_Msk                                      (0x1UL << CAN_F0BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F0BANK1_FBIT16                                          CAN_F0BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F0BANK1_FBIT17_Pos                                      (17U)
#define CAN_F0BANK1_FBIT17_Msk                                      (0x1UL << CAN_F0BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F0BANK1_FBIT17                                          CAN_F0BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F0BANK1_FBIT18_Pos                                      (18U)
#define CAN_F0BANK1_FBIT18_Msk                                      (0x1UL << CAN_F0BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F0BANK1_FBIT18                                          CAN_F0BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F0BANK1_FBIT19_Pos                                      (19U)
#define CAN_F0BANK1_FBIT19_Msk                                      (0x1UL << CAN_F0BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F0BANK1_FBIT19                                          CAN_F0BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F0BANK1_FBIT20_Pos                                      (20U)
#define CAN_F0BANK1_FBIT20_Msk                                      (0x1UL << CAN_F0BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F0BANK1_FBIT20                                          CAN_F0BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F0BANK1_FBIT21_Pos                                      (21U)
#define CAN_F0BANK1_FBIT21_Msk                                      (0x1UL << CAN_F0BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F0BANK1_FBIT21                                          CAN_F0BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F0BANK1_FBIT22_Pos                                      (22U)
#define CAN_F0BANK1_FBIT22_Msk                                      (0x1UL << CAN_F0BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F0BANK1_FBIT22                                          CAN_F0BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F0BANK1_FBIT23_Pos                                      (23U)
#define CAN_F0BANK1_FBIT23_Msk                                      (0x1UL << CAN_F0BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F0BANK1_FBIT23                                          CAN_F0BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F0BANK1_FBIT24_Pos                                      (24U)
#define CAN_F0BANK1_FBIT24_Msk                                      (0x1UL << CAN_F0BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F0BANK1_FBIT24                                          CAN_F0BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F0BANK1_FBIT25_Pos                                      (25U)
#define CAN_F0BANK1_FBIT25_Msk                                      (0x1UL << CAN_F0BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F0BANK1_FBIT25                                          CAN_F0BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F0BANK1_FBIT26_Pos                                      (26U)
#define CAN_F0BANK1_FBIT26_Msk                                      (0x1UL << CAN_F0BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F0BANK1_FBIT26                                          CAN_F0BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F0BANK1_FBIT27_Pos                                      (27U)
#define CAN_F0BANK1_FBIT27_Msk                                      (0x1UL << CAN_F0BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F0BANK1_FBIT27                                          CAN_F0BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F0BANK1_FBIT28_Pos                                      (28U)
#define CAN_F0BANK1_FBIT28_Msk                                      (0x1UL << CAN_F0BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F0BANK1_FBIT28                                          CAN_F0BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F0BANK1_FBIT29_Pos                                      (29U)
#define CAN_F0BANK1_FBIT29_Msk                                      (0x1UL << CAN_F0BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F0BANK1_FBIT29                                          CAN_F0BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F0BANK1_FBIT30_Pos                                      (30U)
#define CAN_F0BANK1_FBIT30_Msk                                      (0x1UL << CAN_F0BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F0BANK1_FBIT30                                          CAN_F0BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F0BANK1_FBIT31_Pos                                      (31U)
#define CAN_F0BANK1_FBIT31_Msk                                      (0x1UL << CAN_F0BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F0BANK1_FBIT31                                          CAN_F0BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F1BANK1 register  *******************/
#define CAN_F1BANK1_FBIT0_Pos                                       (0U)
#define CAN_F1BANK1_FBIT0_Msk                                       (0x1UL << CAN_F1BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F1BANK1_FBIT0                                           CAN_F1BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F1BANK1_FBIT1_Pos                                       (1U)
#define CAN_F1BANK1_FBIT1_Msk                                       (0x1UL << CAN_F1BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F1BANK1_FBIT1                                           CAN_F1BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F1BANK1_FBIT2_Pos                                       (2U)
#define CAN_F1BANK1_FBIT2_Msk                                       (0x1UL << CAN_F1BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F1BANK1_FBIT2                                           CAN_F1BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F1BANK1_FBIT3_Pos                                       (3U)
#define CAN_F1BANK1_FBIT3_Msk                                       (0x1UL << CAN_F1BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F1BANK1_FBIT3                                           CAN_F1BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F1BANK1_FBIT4_Pos                                       (4U)
#define CAN_F1BANK1_FBIT4_Msk                                       (0x1UL << CAN_F1BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F1BANK1_FBIT4                                           CAN_F1BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F1BANK1_FBIT5_Pos                                       (5U)
#define CAN_F1BANK1_FBIT5_Msk                                       (0x1UL << CAN_F1BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F1BANK1_FBIT5                                           CAN_F1BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F1BANK1_FBIT6_Pos                                       (6U)
#define CAN_F1BANK1_FBIT6_Msk                                       (0x1UL << CAN_F1BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F1BANK1_FBIT6                                           CAN_F1BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F1BANK1_FBIT7_Pos                                       (7U)
#define CAN_F1BANK1_FBIT7_Msk                                       (0x1UL << CAN_F1BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F1BANK1_FBIT7                                           CAN_F1BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F1BANK1_FBIT8_Pos                                       (8U)
#define CAN_F1BANK1_FBIT8_Msk                                       (0x1UL << CAN_F1BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F1BANK1_FBIT8                                           CAN_F1BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F1BANK1_FBIT9_Pos                                       (9U)
#define CAN_F1BANK1_FBIT9_Msk                                       (0x1UL << CAN_F1BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F1BANK1_FBIT9                                           CAN_F1BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F1BANK1_FBIT10_Pos                                      (10U)
#define CAN_F1BANK1_FBIT10_Msk                                      (0x1UL << CAN_F1BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F1BANK1_FBIT10                                          CAN_F1BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F1BANK1_FBIT11_Pos                                      (11U)
#define CAN_F1BANK1_FBIT11_Msk                                      (0x1UL << CAN_F1BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F1BANK1_FBIT11                                          CAN_F1BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F1BANK1_FBIT12_Pos                                      (12U)
#define CAN_F1BANK1_FBIT12_Msk                                      (0x1UL << CAN_F1BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F1BANK1_FBIT12                                          CAN_F1BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F1BANK1_FBIT13_Pos                                      (13U)
#define CAN_F1BANK1_FBIT13_Msk                                      (0x1UL << CAN_F1BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F1BANK1_FBIT13                                          CAN_F1BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F1BANK1_FBIT14_Pos                                      (14U)
#define CAN_F1BANK1_FBIT14_Msk                                      (0x1UL << CAN_F1BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F1BANK1_FBIT14                                          CAN_F1BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F1BANK1_FBIT15_Pos                                      (15U)
#define CAN_F1BANK1_FBIT15_Msk                                      (0x1UL << CAN_F1BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F1BANK1_FBIT15                                          CAN_F1BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F1BANK1_FBIT16_Pos                                      (16U)
#define CAN_F1BANK1_FBIT16_Msk                                      (0x1UL << CAN_F1BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F1BANK1_FBIT16                                          CAN_F1BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F1BANK1_FBIT17_Pos                                      (17U)
#define CAN_F1BANK1_FBIT17_Msk                                      (0x1UL << CAN_F1BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F1BANK1_FBIT17                                          CAN_F1BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F1BANK1_FBIT18_Pos                                      (18U)
#define CAN_F1BANK1_FBIT18_Msk                                      (0x1UL << CAN_F1BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F1BANK1_FBIT18                                          CAN_F1BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F1BANK1_FBIT19_Pos                                      (19U)
#define CAN_F1BANK1_FBIT19_Msk                                      (0x1UL << CAN_F1BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F1BANK1_FBIT19                                          CAN_F1BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F1BANK1_FBIT20_Pos                                      (20U)
#define CAN_F1BANK1_FBIT20_Msk                                      (0x1UL << CAN_F1BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F1BANK1_FBIT20                                          CAN_F1BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F1BANK1_FBIT21_Pos                                      (21U)
#define CAN_F1BANK1_FBIT21_Msk                                      (0x1UL << CAN_F1BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F1BANK1_FBIT21                                          CAN_F1BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F1BANK1_FBIT22_Pos                                      (22U)
#define CAN_F1BANK1_FBIT22_Msk                                      (0x1UL << CAN_F1BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F1BANK1_FBIT22                                          CAN_F1BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F1BANK1_FBIT23_Pos                                      (23U)
#define CAN_F1BANK1_FBIT23_Msk                                      (0x1UL << CAN_F1BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F1BANK1_FBIT23                                          CAN_F1BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F1BANK1_FBIT24_Pos                                      (24U)
#define CAN_F1BANK1_FBIT24_Msk                                      (0x1UL << CAN_F1BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F1BANK1_FBIT24                                          CAN_F1BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F1BANK1_FBIT25_Pos                                      (25U)
#define CAN_F1BANK1_FBIT25_Msk                                      (0x1UL << CAN_F1BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F1BANK1_FBIT25                                          CAN_F1BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F1BANK1_FBIT26_Pos                                      (26U)
#define CAN_F1BANK1_FBIT26_Msk                                      (0x1UL << CAN_F1BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F1BANK1_FBIT26                                          CAN_F1BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F1BANK1_FBIT27_Pos                                      (27U)
#define CAN_F1BANK1_FBIT27_Msk                                      (0x1UL << CAN_F1BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F1BANK1_FBIT27                                          CAN_F1BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F1BANK1_FBIT28_Pos                                      (28U)
#define CAN_F1BANK1_FBIT28_Msk                                      (0x1UL << CAN_F1BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F1BANK1_FBIT28                                          CAN_F1BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F1BANK1_FBIT29_Pos                                      (29U)
#define CAN_F1BANK1_FBIT29_Msk                                      (0x1UL << CAN_F1BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F1BANK1_FBIT29                                          CAN_F1BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F1BANK1_FBIT30_Pos                                      (30U)
#define CAN_F1BANK1_FBIT30_Msk                                      (0x1UL << CAN_F1BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F1BANK1_FBIT30                                          CAN_F1BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F1BANK1_FBIT31_Pos                                      (31U)
#define CAN_F1BANK1_FBIT31_Msk                                      (0x1UL << CAN_F1BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F1BANK1_FBIT31                                          CAN_F1BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F2BANK1 register  *******************/
#define CAN_F2BANK1_FBIT0_Pos                                       (0U)
#define CAN_F2BANK1_FBIT0_Msk                                       (0x1UL << CAN_F2BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F2BANK1_FBIT0                                           CAN_F2BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F2BANK1_FBIT1_Pos                                       (1U)
#define CAN_F2BANK1_FBIT1_Msk                                       (0x1UL << CAN_F2BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F2BANK1_FBIT1                                           CAN_F2BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F2BANK1_FBIT2_Pos                                       (2U)
#define CAN_F2BANK1_FBIT2_Msk                                       (0x1UL << CAN_F2BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F2BANK1_FBIT2                                           CAN_F2BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F2BANK1_FBIT3_Pos                                       (3U)
#define CAN_F2BANK1_FBIT3_Msk                                       (0x1UL << CAN_F2BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F2BANK1_FBIT3                                           CAN_F2BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F2BANK1_FBIT4_Pos                                       (4U)
#define CAN_F2BANK1_FBIT4_Msk                                       (0x1UL << CAN_F2BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F2BANK1_FBIT4                                           CAN_F2BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F2BANK1_FBIT5_Pos                                       (5U)
#define CAN_F2BANK1_FBIT5_Msk                                       (0x1UL << CAN_F2BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F2BANK1_FBIT5                                           CAN_F2BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F2BANK1_FBIT6_Pos                                       (6U)
#define CAN_F2BANK1_FBIT6_Msk                                       (0x1UL << CAN_F2BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F2BANK1_FBIT6                                           CAN_F2BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F2BANK1_FBIT7_Pos                                       (7U)
#define CAN_F2BANK1_FBIT7_Msk                                       (0x1UL << CAN_F2BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F2BANK1_FBIT7                                           CAN_F2BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F2BANK1_FBIT8_Pos                                       (8U)
#define CAN_F2BANK1_FBIT8_Msk                                       (0x1UL << CAN_F2BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F2BANK1_FBIT8                                           CAN_F2BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F2BANK1_FBIT9_Pos                                       (9U)
#define CAN_F2BANK1_FBIT9_Msk                                       (0x1UL << CAN_F2BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F2BANK1_FBIT9                                           CAN_F2BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F2BANK1_FBIT10_Pos                                      (10U)
#define CAN_F2BANK1_FBIT10_Msk                                      (0x1UL << CAN_F2BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F2BANK1_FBIT10                                          CAN_F2BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F2BANK1_FBIT11_Pos                                      (11U)
#define CAN_F2BANK1_FBIT11_Msk                                      (0x1UL << CAN_F2BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F2BANK1_FBIT11                                          CAN_F2BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F2BANK1_FBIT12_Pos                                      (12U)
#define CAN_F2BANK1_FBIT12_Msk                                      (0x1UL << CAN_F2BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F2BANK1_FBIT12                                          CAN_F2BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F2BANK1_FBIT13_Pos                                      (13U)
#define CAN_F2BANK1_FBIT13_Msk                                      (0x1UL << CAN_F2BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F2BANK1_FBIT13                                          CAN_F2BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F2BANK1_FBIT14_Pos                                      (14U)
#define CAN_F2BANK1_FBIT14_Msk                                      (0x1UL << CAN_F2BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F2BANK1_FBIT14                                          CAN_F2BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F2BANK1_FBIT15_Pos                                      (15U)
#define CAN_F2BANK1_FBIT15_Msk                                      (0x1UL << CAN_F2BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F2BANK1_FBIT15                                          CAN_F2BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F2BANK1_FBIT16_Pos                                      (16U)
#define CAN_F2BANK1_FBIT16_Msk                                      (0x1UL << CAN_F2BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F2BANK1_FBIT16                                          CAN_F2BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F2BANK1_FBIT17_Pos                                      (17U)
#define CAN_F2BANK1_FBIT17_Msk                                      (0x1UL << CAN_F2BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F2BANK1_FBIT17                                          CAN_F2BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F2BANK1_FBIT18_Pos                                      (18U)
#define CAN_F2BANK1_FBIT18_Msk                                      (0x1UL << CAN_F2BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F2BANK1_FBIT18                                          CAN_F2BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F2BANK1_FBIT19_Pos                                      (19U)
#define CAN_F2BANK1_FBIT19_Msk                                      (0x1UL << CAN_F2BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F2BANK1_FBIT19                                          CAN_F2BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F2BANK1_FBIT20_Pos                                      (20U)
#define CAN_F2BANK1_FBIT20_Msk                                      (0x1UL << CAN_F2BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F2BANK1_FBIT20                                          CAN_F2BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F2BANK1_FBIT21_Pos                                      (21U)
#define CAN_F2BANK1_FBIT21_Msk                                      (0x1UL << CAN_F2BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F2BANK1_FBIT21                                          CAN_F2BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F2BANK1_FBIT22_Pos                                      (22U)
#define CAN_F2BANK1_FBIT22_Msk                                      (0x1UL << CAN_F2BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F2BANK1_FBIT22                                          CAN_F2BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F2BANK1_FBIT23_Pos                                      (23U)
#define CAN_F2BANK1_FBIT23_Msk                                      (0x1UL << CAN_F2BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F2BANK1_FBIT23                                          CAN_F2BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F2BANK1_FBIT24_Pos                                      (24U)
#define CAN_F2BANK1_FBIT24_Msk                                      (0x1UL << CAN_F2BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F2BANK1_FBIT24                                          CAN_F2BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F2BANK1_FBIT25_Pos                                      (25U)
#define CAN_F2BANK1_FBIT25_Msk                                      (0x1UL << CAN_F2BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F2BANK1_FBIT25                                          CAN_F2BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F2BANK1_FBIT26_Pos                                      (26U)
#define CAN_F2BANK1_FBIT26_Msk                                      (0x1UL << CAN_F2BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F2BANK1_FBIT26                                          CAN_F2BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F2BANK1_FBIT27_Pos                                      (27U)
#define CAN_F2BANK1_FBIT27_Msk                                      (0x1UL << CAN_F2BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F2BANK1_FBIT27                                          CAN_F2BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F2BANK1_FBIT28_Pos                                      (28U)
#define CAN_F2BANK1_FBIT28_Msk                                      (0x1UL << CAN_F2BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F2BANK1_FBIT28                                          CAN_F2BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F2BANK1_FBIT29_Pos                                      (29U)
#define CAN_F2BANK1_FBIT29_Msk                                      (0x1UL << CAN_F2BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F2BANK1_FBIT29                                          CAN_F2BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F2BANK1_FBIT30_Pos                                      (30U)
#define CAN_F2BANK1_FBIT30_Msk                                      (0x1UL << CAN_F2BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F2BANK1_FBIT30                                          CAN_F2BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F2BANK1_FBIT31_Pos                                      (31U)
#define CAN_F2BANK1_FBIT31_Msk                                      (0x1UL << CAN_F2BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F2BANK1_FBIT31                                          CAN_F2BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F3BANK1 register  *******************/
#define CAN_F3BANK1_FBIT0_Pos                                       (0U)
#define CAN_F3BANK1_FBIT0_Msk                                       (0x1UL << CAN_F3BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F3BANK1_FBIT0                                           CAN_F3BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F3BANK1_FBIT1_Pos                                       (1U)
#define CAN_F3BANK1_FBIT1_Msk                                       (0x1UL << CAN_F3BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F3BANK1_FBIT1                                           CAN_F3BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F3BANK1_FBIT2_Pos                                       (2U)
#define CAN_F3BANK1_FBIT2_Msk                                       (0x1UL << CAN_F3BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F3BANK1_FBIT2                                           CAN_F3BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F3BANK1_FBIT3_Pos                                       (3U)
#define CAN_F3BANK1_FBIT3_Msk                                       (0x1UL << CAN_F3BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F3BANK1_FBIT3                                           CAN_F3BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F3BANK1_FBIT4_Pos                                       (4U)
#define CAN_F3BANK1_FBIT4_Msk                                       (0x1UL << CAN_F3BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F3BANK1_FBIT4                                           CAN_F3BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F3BANK1_FBIT5_Pos                                       (5U)
#define CAN_F3BANK1_FBIT5_Msk                                       (0x1UL << CAN_F3BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F3BANK1_FBIT5                                           CAN_F3BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F3BANK1_FBIT6_Pos                                       (6U)
#define CAN_F3BANK1_FBIT6_Msk                                       (0x1UL << CAN_F3BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F3BANK1_FBIT6                                           CAN_F3BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F3BANK1_FBIT7_Pos                                       (7U)
#define CAN_F3BANK1_FBIT7_Msk                                       (0x1UL << CAN_F3BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F3BANK1_FBIT7                                           CAN_F3BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F3BANK1_FBIT8_Pos                                       (8U)
#define CAN_F3BANK1_FBIT8_Msk                                       (0x1UL << CAN_F3BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F3BANK1_FBIT8                                           CAN_F3BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F3BANK1_FBIT9_Pos                                       (9U)
#define CAN_F3BANK1_FBIT9_Msk                                       (0x1UL << CAN_F3BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F3BANK1_FBIT9                                           CAN_F3BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F3BANK1_FBIT10_Pos                                      (10U)
#define CAN_F3BANK1_FBIT10_Msk                                      (0x1UL << CAN_F3BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F3BANK1_FBIT10                                          CAN_F3BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F3BANK1_FBIT11_Pos                                      (11U)
#define CAN_F3BANK1_FBIT11_Msk                                      (0x1UL << CAN_F3BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F3BANK1_FBIT11                                          CAN_F3BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F3BANK1_FBIT12_Pos                                      (12U)
#define CAN_F3BANK1_FBIT12_Msk                                      (0x1UL << CAN_F3BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F3BANK1_FBIT12                                          CAN_F3BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F3BANK1_FBIT13_Pos                                      (13U)
#define CAN_F3BANK1_FBIT13_Msk                                      (0x1UL << CAN_F3BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F3BANK1_FBIT13                                          CAN_F3BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F3BANK1_FBIT14_Pos                                      (14U)
#define CAN_F3BANK1_FBIT14_Msk                                      (0x1UL << CAN_F3BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F3BANK1_FBIT14                                          CAN_F3BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F3BANK1_FBIT15_Pos                                      (15U)
#define CAN_F3BANK1_FBIT15_Msk                                      (0x1UL << CAN_F3BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F3BANK1_FBIT15                                          CAN_F3BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F3BANK1_FBIT16_Pos                                      (16U)
#define CAN_F3BANK1_FBIT16_Msk                                      (0x1UL << CAN_F3BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F3BANK1_FBIT16                                          CAN_F3BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F3BANK1_FBIT17_Pos                                      (17U)
#define CAN_F3BANK1_FBIT17_Msk                                      (0x1UL << CAN_F3BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F3BANK1_FBIT17                                          CAN_F3BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F3BANK1_FBIT18_Pos                                      (18U)
#define CAN_F3BANK1_FBIT18_Msk                                      (0x1UL << CAN_F3BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F3BANK1_FBIT18                                          CAN_F3BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F3BANK1_FBIT19_Pos                                      (19U)
#define CAN_F3BANK1_FBIT19_Msk                                      (0x1UL << CAN_F3BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F3BANK1_FBIT19                                          CAN_F3BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F3BANK1_FBIT20_Pos                                      (20U)
#define CAN_F3BANK1_FBIT20_Msk                                      (0x1UL << CAN_F3BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F3BANK1_FBIT20                                          CAN_F3BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F3BANK1_FBIT21_Pos                                      (21U)
#define CAN_F3BANK1_FBIT21_Msk                                      (0x1UL << CAN_F3BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F3BANK1_FBIT21                                          CAN_F3BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F3BANK1_FBIT22_Pos                                      (22U)
#define CAN_F3BANK1_FBIT22_Msk                                      (0x1UL << CAN_F3BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F3BANK1_FBIT22                                          CAN_F3BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F3BANK1_FBIT23_Pos                                      (23U)
#define CAN_F3BANK1_FBIT23_Msk                                      (0x1UL << CAN_F3BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F3BANK1_FBIT23                                          CAN_F3BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F3BANK1_FBIT24_Pos                                      (24U)
#define CAN_F3BANK1_FBIT24_Msk                                      (0x1UL << CAN_F3BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F3BANK1_FBIT24                                          CAN_F3BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F3BANK1_FBIT25_Pos                                      (25U)
#define CAN_F3BANK1_FBIT25_Msk                                      (0x1UL << CAN_F3BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F3BANK1_FBIT25                                          CAN_F3BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F3BANK1_FBIT26_Pos                                      (26U)
#define CAN_F3BANK1_FBIT26_Msk                                      (0x1UL << CAN_F3BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F3BANK1_FBIT26                                          CAN_F3BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F3BANK1_FBIT27_Pos                                      (27U)
#define CAN_F3BANK1_FBIT27_Msk                                      (0x1UL << CAN_F3BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F3BANK1_FBIT27                                          CAN_F3BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F3BANK1_FBIT28_Pos                                      (28U)
#define CAN_F3BANK1_FBIT28_Msk                                      (0x1UL << CAN_F3BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F3BANK1_FBIT28                                          CAN_F3BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F3BANK1_FBIT29_Pos                                      (29U)
#define CAN_F3BANK1_FBIT29_Msk                                      (0x1UL << CAN_F3BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F3BANK1_FBIT29                                          CAN_F3BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F3BANK1_FBIT30_Pos                                      (30U)
#define CAN_F3BANK1_FBIT30_Msk                                      (0x1UL << CAN_F3BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F3BANK1_FBIT30                                          CAN_F3BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F3BANK1_FBIT31_Pos                                      (31U)
#define CAN_F3BANK1_FBIT31_Msk                                      (0x1UL << CAN_F3BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F3BANK1_FBIT31                                          CAN_F3BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F4BANK1 register  *******************/
#define CAN_F4BANK1_FBIT0_Pos                                       (0U)
#define CAN_F4BANK1_FBIT0_Msk                                       (0x1UL << CAN_F4BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F4BANK1_FBIT0                                           CAN_F4BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F4BANK1_FBIT1_Pos                                       (1U)
#define CAN_F4BANK1_FBIT1_Msk                                       (0x1UL << CAN_F4BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F4BANK1_FBIT1                                           CAN_F4BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F4BANK1_FBIT2_Pos                                       (2U)
#define CAN_F4BANK1_FBIT2_Msk                                       (0x1UL << CAN_F4BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F4BANK1_FBIT2                                           CAN_F4BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F4BANK1_FBIT3_Pos                                       (3U)
#define CAN_F4BANK1_FBIT3_Msk                                       (0x1UL << CAN_F4BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F4BANK1_FBIT3                                           CAN_F4BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F4BANK1_FBIT4_Pos                                       (4U)
#define CAN_F4BANK1_FBIT4_Msk                                       (0x1UL << CAN_F4BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F4BANK1_FBIT4                                           CAN_F4BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F4BANK1_FBIT5_Pos                                       (5U)
#define CAN_F4BANK1_FBIT5_Msk                                       (0x1UL << CAN_F4BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F4BANK1_FBIT5                                           CAN_F4BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F4BANK1_FBIT6_Pos                                       (6U)
#define CAN_F4BANK1_FBIT6_Msk                                       (0x1UL << CAN_F4BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F4BANK1_FBIT6                                           CAN_F4BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F4BANK1_FBIT7_Pos                                       (7U)
#define CAN_F4BANK1_FBIT7_Msk                                       (0x1UL << CAN_F4BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F4BANK1_FBIT7                                           CAN_F4BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F4BANK1_FBIT8_Pos                                       (8U)
#define CAN_F4BANK1_FBIT8_Msk                                       (0x1UL << CAN_F4BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F4BANK1_FBIT8                                           CAN_F4BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F4BANK1_FBIT9_Pos                                       (9U)
#define CAN_F4BANK1_FBIT9_Msk                                       (0x1UL << CAN_F4BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F4BANK1_FBIT9                                           CAN_F4BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F4BANK1_FBIT10_Pos                                      (10U)
#define CAN_F4BANK1_FBIT10_Msk                                      (0x1UL << CAN_F4BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F4BANK1_FBIT10                                          CAN_F4BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F4BANK1_FBIT11_Pos                                      (11U)
#define CAN_F4BANK1_FBIT11_Msk                                      (0x1UL << CAN_F4BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F4BANK1_FBIT11                                          CAN_F4BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F4BANK1_FBIT12_Pos                                      (12U)
#define CAN_F4BANK1_FBIT12_Msk                                      (0x1UL << CAN_F4BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F4BANK1_FBIT12                                          CAN_F4BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F4BANK1_FBIT13_Pos                                      (13U)
#define CAN_F4BANK1_FBIT13_Msk                                      (0x1UL << CAN_F4BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F4BANK1_FBIT13                                          CAN_F4BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F4BANK1_FBIT14_Pos                                      (14U)
#define CAN_F4BANK1_FBIT14_Msk                                      (0x1UL << CAN_F4BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F4BANK1_FBIT14                                          CAN_F4BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F4BANK1_FBIT15_Pos                                      (15U)
#define CAN_F4BANK1_FBIT15_Msk                                      (0x1UL << CAN_F4BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F4BANK1_FBIT15                                          CAN_F4BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F4BANK1_FBIT16_Pos                                      (16U)
#define CAN_F4BANK1_FBIT16_Msk                                      (0x1UL << CAN_F4BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F4BANK1_FBIT16                                          CAN_F4BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F4BANK1_FBIT17_Pos                                      (17U)
#define CAN_F4BANK1_FBIT17_Msk                                      (0x1UL << CAN_F4BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F4BANK1_FBIT17                                          CAN_F4BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F4BANK1_FBIT18_Pos                                      (18U)
#define CAN_F4BANK1_FBIT18_Msk                                      (0x1UL << CAN_F4BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F4BANK1_FBIT18                                          CAN_F4BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F4BANK1_FBIT19_Pos                                      (19U)
#define CAN_F4BANK1_FBIT19_Msk                                      (0x1UL << CAN_F4BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F4BANK1_FBIT19                                          CAN_F4BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F4BANK1_FBIT20_Pos                                      (20U)
#define CAN_F4BANK1_FBIT20_Msk                                      (0x1UL << CAN_F4BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F4BANK1_FBIT20                                          CAN_F4BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F4BANK1_FBIT21_Pos                                      (21U)
#define CAN_F4BANK1_FBIT21_Msk                                      (0x1UL << CAN_F4BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F4BANK1_FBIT21                                          CAN_F4BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F4BANK1_FBIT22_Pos                                      (22U)
#define CAN_F4BANK1_FBIT22_Msk                                      (0x1UL << CAN_F4BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F4BANK1_FBIT22                                          CAN_F4BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F4BANK1_FBIT23_Pos                                      (23U)
#define CAN_F4BANK1_FBIT23_Msk                                      (0x1UL << CAN_F4BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F4BANK1_FBIT23                                          CAN_F4BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F4BANK1_FBIT24_Pos                                      (24U)
#define CAN_F4BANK1_FBIT24_Msk                                      (0x1UL << CAN_F4BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F4BANK1_FBIT24                                          CAN_F4BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F4BANK1_FBIT25_Pos                                      (25U)
#define CAN_F4BANK1_FBIT25_Msk                                      (0x1UL << CAN_F4BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F4BANK1_FBIT25                                          CAN_F4BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F4BANK1_FBIT26_Pos                                      (26U)
#define CAN_F4BANK1_FBIT26_Msk                                      (0x1UL << CAN_F4BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F4BANK1_FBIT26                                          CAN_F4BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F4BANK1_FBIT27_Pos                                      (27U)
#define CAN_F4BANK1_FBIT27_Msk                                      (0x1UL << CAN_F4BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F4BANK1_FBIT27                                          CAN_F4BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F4BANK1_FBIT28_Pos                                      (28U)
#define CAN_F4BANK1_FBIT28_Msk                                      (0x1UL << CAN_F4BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F4BANK1_FBIT28                                          CAN_F4BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F4BANK1_FBIT29_Pos                                      (29U)
#define CAN_F4BANK1_FBIT29_Msk                                      (0x1UL << CAN_F4BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F4BANK1_FBIT29                                          CAN_F4BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F4BANK1_FBIT30_Pos                                      (30U)
#define CAN_F4BANK1_FBIT30_Msk                                      (0x1UL << CAN_F4BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F4BANK1_FBIT30                                          CAN_F4BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F4BANK1_FBIT31_Pos                                      (31U)
#define CAN_F4BANK1_FBIT31_Msk                                      (0x1UL << CAN_F4BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F4BANK1_FBIT31                                          CAN_F4BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F5BANK1 register  *******************/
#define CAN_F5BANK1_FBIT0_Pos                                       (0U)
#define CAN_F5BANK1_FBIT0_Msk                                       (0x1UL << CAN_F5BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F5BANK1_FBIT0                                           CAN_F5BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F5BANK1_FBIT1_Pos                                       (1U)
#define CAN_F5BANK1_FBIT1_Msk                                       (0x1UL << CAN_F5BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F5BANK1_FBIT1                                           CAN_F5BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F5BANK1_FBIT2_Pos                                       (2U)
#define CAN_F5BANK1_FBIT2_Msk                                       (0x1UL << CAN_F5BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F5BANK1_FBIT2                                           CAN_F5BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F5BANK1_FBIT3_Pos                                       (3U)
#define CAN_F5BANK1_FBIT3_Msk                                       (0x1UL << CAN_F5BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F5BANK1_FBIT3                                           CAN_F5BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F5BANK1_FBIT4_Pos                                       (4U)
#define CAN_F5BANK1_FBIT4_Msk                                       (0x1UL << CAN_F5BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F5BANK1_FBIT4                                           CAN_F5BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F5BANK1_FBIT5_Pos                                       (5U)
#define CAN_F5BANK1_FBIT5_Msk                                       (0x1UL << CAN_F5BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F5BANK1_FBIT5                                           CAN_F5BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F5BANK1_FBIT6_Pos                                       (6U)
#define CAN_F5BANK1_FBIT6_Msk                                       (0x1UL << CAN_F5BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F5BANK1_FBIT6                                           CAN_F5BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F5BANK1_FBIT7_Pos                                       (7U)
#define CAN_F5BANK1_FBIT7_Msk                                       (0x1UL << CAN_F5BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F5BANK1_FBIT7                                           CAN_F5BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F5BANK1_FBIT8_Pos                                       (8U)
#define CAN_F5BANK1_FBIT8_Msk                                       (0x1UL << CAN_F5BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F5BANK1_FBIT8                                           CAN_F5BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F5BANK1_FBIT9_Pos                                       (9U)
#define CAN_F5BANK1_FBIT9_Msk                                       (0x1UL << CAN_F5BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F5BANK1_FBIT9                                           CAN_F5BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F5BANK1_FBIT10_Pos                                      (10U)
#define CAN_F5BANK1_FBIT10_Msk                                      (0x1UL << CAN_F5BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F5BANK1_FBIT10                                          CAN_F5BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F5BANK1_FBIT11_Pos                                      (11U)
#define CAN_F5BANK1_FBIT11_Msk                                      (0x1UL << CAN_F5BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F5BANK1_FBIT11                                          CAN_F5BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F5BANK1_FBIT12_Pos                                      (12U)
#define CAN_F5BANK1_FBIT12_Msk                                      (0x1UL << CAN_F5BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F5BANK1_FBIT12                                          CAN_F5BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F5BANK1_FBIT13_Pos                                      (13U)
#define CAN_F5BANK1_FBIT13_Msk                                      (0x1UL << CAN_F5BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F5BANK1_FBIT13                                          CAN_F5BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F5BANK1_FBIT14_Pos                                      (14U)
#define CAN_F5BANK1_FBIT14_Msk                                      (0x1UL << CAN_F5BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F5BANK1_FBIT14                                          CAN_F5BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F5BANK1_FBIT15_Pos                                      (15U)
#define CAN_F5BANK1_FBIT15_Msk                                      (0x1UL << CAN_F5BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F5BANK1_FBIT15                                          CAN_F5BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F5BANK1_FBIT16_Pos                                      (16U)
#define CAN_F5BANK1_FBIT16_Msk                                      (0x1UL << CAN_F5BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F5BANK1_FBIT16                                          CAN_F5BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F5BANK1_FBIT17_Pos                                      (17U)
#define CAN_F5BANK1_FBIT17_Msk                                      (0x1UL << CAN_F5BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F5BANK1_FBIT17                                          CAN_F5BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F5BANK1_FBIT18_Pos                                      (18U)
#define CAN_F5BANK1_FBIT18_Msk                                      (0x1UL << CAN_F5BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F5BANK1_FBIT18                                          CAN_F5BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F5BANK1_FBIT19_Pos                                      (19U)
#define CAN_F5BANK1_FBIT19_Msk                                      (0x1UL << CAN_F5BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F5BANK1_FBIT19                                          CAN_F5BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F5BANK1_FBIT20_Pos                                      (20U)
#define CAN_F5BANK1_FBIT20_Msk                                      (0x1UL << CAN_F5BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F5BANK1_FBIT20                                          CAN_F5BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F5BANK1_FBIT21_Pos                                      (21U)
#define CAN_F5BANK1_FBIT21_Msk                                      (0x1UL << CAN_F5BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F5BANK1_FBIT21                                          CAN_F5BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F5BANK1_FBIT22_Pos                                      (22U)
#define CAN_F5BANK1_FBIT22_Msk                                      (0x1UL << CAN_F5BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F5BANK1_FBIT22                                          CAN_F5BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F5BANK1_FBIT23_Pos                                      (23U)
#define CAN_F5BANK1_FBIT23_Msk                                      (0x1UL << CAN_F5BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F5BANK1_FBIT23                                          CAN_F5BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F5BANK1_FBIT24_Pos                                      (24U)
#define CAN_F5BANK1_FBIT24_Msk                                      (0x1UL << CAN_F5BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F5BANK1_FBIT24                                          CAN_F5BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F5BANK1_FBIT25_Pos                                      (25U)
#define CAN_F5BANK1_FBIT25_Msk                                      (0x1UL << CAN_F5BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F5BANK1_FBIT25                                          CAN_F5BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F5BANK1_FBIT26_Pos                                      (26U)
#define CAN_F5BANK1_FBIT26_Msk                                      (0x1UL << CAN_F5BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F5BANK1_FBIT26                                          CAN_F5BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F5BANK1_FBIT27_Pos                                      (27U)
#define CAN_F5BANK1_FBIT27_Msk                                      (0x1UL << CAN_F5BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F5BANK1_FBIT27                                          CAN_F5BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F5BANK1_FBIT28_Pos                                      (28U)
#define CAN_F5BANK1_FBIT28_Msk                                      (0x1UL << CAN_F5BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F5BANK1_FBIT28                                          CAN_F5BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F5BANK1_FBIT29_Pos                                      (29U)
#define CAN_F5BANK1_FBIT29_Msk                                      (0x1UL << CAN_F5BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F5BANK1_FBIT29                                          CAN_F5BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F5BANK1_FBIT30_Pos                                      (30U)
#define CAN_F5BANK1_FBIT30_Msk                                      (0x1UL << CAN_F5BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F5BANK1_FBIT30                                          CAN_F5BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F5BANK1_FBIT31_Pos                                      (31U)
#define CAN_F5BANK1_FBIT31_Msk                                      (0x1UL << CAN_F5BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F5BANK1_FBIT31                                          CAN_F5BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F6BANK1 register  *******************/
#define CAN_F6BANK1_FBIT0_Pos                                       (0U)
#define CAN_F6BANK1_FBIT0_Msk                                       (0x1UL << CAN_F6BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F6BANK1_FBIT0                                           CAN_F6BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F6BANK1_FBIT1_Pos                                       (1U)
#define CAN_F6BANK1_FBIT1_Msk                                       (0x1UL << CAN_F6BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F6BANK1_FBIT1                                           CAN_F6BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F6BANK1_FBIT2_Pos                                       (2U)
#define CAN_F6BANK1_FBIT2_Msk                                       (0x1UL << CAN_F6BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F6BANK1_FBIT2                                           CAN_F6BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F6BANK1_FBIT3_Pos                                       (3U)
#define CAN_F6BANK1_FBIT3_Msk                                       (0x1UL << CAN_F6BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F6BANK1_FBIT3                                           CAN_F6BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F6BANK1_FBIT4_Pos                                       (4U)
#define CAN_F6BANK1_FBIT4_Msk                                       (0x1UL << CAN_F6BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F6BANK1_FBIT4                                           CAN_F6BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F6BANK1_FBIT5_Pos                                       (5U)
#define CAN_F6BANK1_FBIT5_Msk                                       (0x1UL << CAN_F6BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F6BANK1_FBIT5                                           CAN_F6BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F6BANK1_FBIT6_Pos                                       (6U)
#define CAN_F6BANK1_FBIT6_Msk                                       (0x1UL << CAN_F6BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F6BANK1_FBIT6                                           CAN_F6BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F6BANK1_FBIT7_Pos                                       (7U)
#define CAN_F6BANK1_FBIT7_Msk                                       (0x1UL << CAN_F6BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F6BANK1_FBIT7                                           CAN_F6BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F6BANK1_FBIT8_Pos                                       (8U)
#define CAN_F6BANK1_FBIT8_Msk                                       (0x1UL << CAN_F6BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F6BANK1_FBIT8                                           CAN_F6BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F6BANK1_FBIT9_Pos                                       (9U)
#define CAN_F6BANK1_FBIT9_Msk                                       (0x1UL << CAN_F6BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F6BANK1_FBIT9                                           CAN_F6BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F6BANK1_FBIT10_Pos                                      (10U)
#define CAN_F6BANK1_FBIT10_Msk                                      (0x1UL << CAN_F6BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F6BANK1_FBIT10                                          CAN_F6BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F6BANK1_FBIT11_Pos                                      (11U)
#define CAN_F6BANK1_FBIT11_Msk                                      (0x1UL << CAN_F6BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F6BANK1_FBIT11                                          CAN_F6BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F6BANK1_FBIT12_Pos                                      (12U)
#define CAN_F6BANK1_FBIT12_Msk                                      (0x1UL << CAN_F6BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F6BANK1_FBIT12                                          CAN_F6BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F6BANK1_FBIT13_Pos                                      (13U)
#define CAN_F6BANK1_FBIT13_Msk                                      (0x1UL << CAN_F6BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F6BANK1_FBIT13                                          CAN_F6BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F6BANK1_FBIT14_Pos                                      (14U)
#define CAN_F6BANK1_FBIT14_Msk                                      (0x1UL << CAN_F6BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F6BANK1_FBIT14                                          CAN_F6BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F6BANK1_FBIT15_Pos                                      (15U)
#define CAN_F6BANK1_FBIT15_Msk                                      (0x1UL << CAN_F6BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F6BANK1_FBIT15                                          CAN_F6BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F6BANK1_FBIT16_Pos                                      (16U)
#define CAN_F6BANK1_FBIT16_Msk                                      (0x1UL << CAN_F6BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F6BANK1_FBIT16                                          CAN_F6BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F6BANK1_FBIT17_Pos                                      (17U)
#define CAN_F6BANK1_FBIT17_Msk                                      (0x1UL << CAN_F6BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F6BANK1_FBIT17                                          CAN_F6BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F6BANK1_FBIT18_Pos                                      (18U)
#define CAN_F6BANK1_FBIT18_Msk                                      (0x1UL << CAN_F6BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F6BANK1_FBIT18                                          CAN_F6BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F6BANK1_FBIT19_Pos                                      (19U)
#define CAN_F6BANK1_FBIT19_Msk                                      (0x1UL << CAN_F6BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F6BANK1_FBIT19                                          CAN_F6BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F6BANK1_FBIT20_Pos                                      (20U)
#define CAN_F6BANK1_FBIT20_Msk                                      (0x1UL << CAN_F6BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F6BANK1_FBIT20                                          CAN_F6BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F6BANK1_FBIT21_Pos                                      (21U)
#define CAN_F6BANK1_FBIT21_Msk                                      (0x1UL << CAN_F6BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F6BANK1_FBIT21                                          CAN_F6BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F6BANK1_FBIT22_Pos                                      (22U)
#define CAN_F6BANK1_FBIT22_Msk                                      (0x1UL << CAN_F6BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F6BANK1_FBIT22                                          CAN_F6BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F6BANK1_FBIT23_Pos                                      (23U)
#define CAN_F6BANK1_FBIT23_Msk                                      (0x1UL << CAN_F6BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F6BANK1_FBIT23                                          CAN_F6BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F6BANK1_FBIT24_Pos                                      (24U)
#define CAN_F6BANK1_FBIT24_Msk                                      (0x1UL << CAN_F6BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F6BANK1_FBIT24                                          CAN_F6BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F6BANK1_FBIT25_Pos                                      (25U)
#define CAN_F6BANK1_FBIT25_Msk                                      (0x1UL << CAN_F6BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F6BANK1_FBIT25                                          CAN_F6BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F6BANK1_FBIT26_Pos                                      (26U)
#define CAN_F6BANK1_FBIT26_Msk                                      (0x1UL << CAN_F6BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F6BANK1_FBIT26                                          CAN_F6BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F6BANK1_FBIT27_Pos                                      (27U)
#define CAN_F6BANK1_FBIT27_Msk                                      (0x1UL << CAN_F6BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F6BANK1_FBIT27                                          CAN_F6BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F6BANK1_FBIT28_Pos                                      (28U)
#define CAN_F6BANK1_FBIT28_Msk                                      (0x1UL << CAN_F6BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F6BANK1_FBIT28                                          CAN_F6BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F6BANK1_FBIT29_Pos                                      (29U)
#define CAN_F6BANK1_FBIT29_Msk                                      (0x1UL << CAN_F6BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F6BANK1_FBIT29                                          CAN_F6BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F6BANK1_FBIT30_Pos                                      (30U)
#define CAN_F6BANK1_FBIT30_Msk                                      (0x1UL << CAN_F6BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F6BANK1_FBIT30                                          CAN_F6BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F6BANK1_FBIT31_Pos                                      (31U)
#define CAN_F6BANK1_FBIT31_Msk                                      (0x1UL << CAN_F6BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F6BANK1_FBIT31                                          CAN_F6BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F7BANK1 register  *******************/
#define CAN_F7BANK1_FBIT0_Pos                                       (0U)
#define CAN_F7BANK1_FBIT0_Msk                                       (0x1UL << CAN_F7BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F7BANK1_FBIT0                                           CAN_F7BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F7BANK1_FBIT1_Pos                                       (1U)
#define CAN_F7BANK1_FBIT1_Msk                                       (0x1UL << CAN_F7BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F7BANK1_FBIT1                                           CAN_F7BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F7BANK1_FBIT2_Pos                                       (2U)
#define CAN_F7BANK1_FBIT2_Msk                                       (0x1UL << CAN_F7BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F7BANK1_FBIT2                                           CAN_F7BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F7BANK1_FBIT3_Pos                                       (3U)
#define CAN_F7BANK1_FBIT3_Msk                                       (0x1UL << CAN_F7BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F7BANK1_FBIT3                                           CAN_F7BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F7BANK1_FBIT4_Pos                                       (4U)
#define CAN_F7BANK1_FBIT4_Msk                                       (0x1UL << CAN_F7BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F7BANK1_FBIT4                                           CAN_F7BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F7BANK1_FBIT5_Pos                                       (5U)
#define CAN_F7BANK1_FBIT5_Msk                                       (0x1UL << CAN_F7BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F7BANK1_FBIT5                                           CAN_F7BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F7BANK1_FBIT6_Pos                                       (6U)
#define CAN_F7BANK1_FBIT6_Msk                                       (0x1UL << CAN_F7BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F7BANK1_FBIT6                                           CAN_F7BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F7BANK1_FBIT7_Pos                                       (7U)
#define CAN_F7BANK1_FBIT7_Msk                                       (0x1UL << CAN_F7BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F7BANK1_FBIT7                                           CAN_F7BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F7BANK1_FBIT8_Pos                                       (8U)
#define CAN_F7BANK1_FBIT8_Msk                                       (0x1UL << CAN_F7BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F7BANK1_FBIT8                                           CAN_F7BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F7BANK1_FBIT9_Pos                                       (9U)
#define CAN_F7BANK1_FBIT9_Msk                                       (0x1UL << CAN_F7BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F7BANK1_FBIT9                                           CAN_F7BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F7BANK1_FBIT10_Pos                                      (10U)
#define CAN_F7BANK1_FBIT10_Msk                                      (0x1UL << CAN_F7BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F7BANK1_FBIT10                                          CAN_F7BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F7BANK1_FBIT11_Pos                                      (11U)
#define CAN_F7BANK1_FBIT11_Msk                                      (0x1UL << CAN_F7BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F7BANK1_FBIT11                                          CAN_F7BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F7BANK1_FBIT12_Pos                                      (12U)
#define CAN_F7BANK1_FBIT12_Msk                                      (0x1UL << CAN_F7BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F7BANK1_FBIT12                                          CAN_F7BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F7BANK1_FBIT13_Pos                                      (13U)
#define CAN_F7BANK1_FBIT13_Msk                                      (0x1UL << CAN_F7BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F7BANK1_FBIT13                                          CAN_F7BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F7BANK1_FBIT14_Pos                                      (14U)
#define CAN_F7BANK1_FBIT14_Msk                                      (0x1UL << CAN_F7BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F7BANK1_FBIT14                                          CAN_F7BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F7BANK1_FBIT15_Pos                                      (15U)
#define CAN_F7BANK1_FBIT15_Msk                                      (0x1UL << CAN_F7BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F7BANK1_FBIT15                                          CAN_F7BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F7BANK1_FBIT16_Pos                                      (16U)
#define CAN_F7BANK1_FBIT16_Msk                                      (0x1UL << CAN_F7BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F7BANK1_FBIT16                                          CAN_F7BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F7BANK1_FBIT17_Pos                                      (17U)
#define CAN_F7BANK1_FBIT17_Msk                                      (0x1UL << CAN_F7BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F7BANK1_FBIT17                                          CAN_F7BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F7BANK1_FBIT18_Pos                                      (18U)
#define CAN_F7BANK1_FBIT18_Msk                                      (0x1UL << CAN_F7BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F7BANK1_FBIT18                                          CAN_F7BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F7BANK1_FBIT19_Pos                                      (19U)
#define CAN_F7BANK1_FBIT19_Msk                                      (0x1UL << CAN_F7BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F7BANK1_FBIT19                                          CAN_F7BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F7BANK1_FBIT20_Pos                                      (20U)
#define CAN_F7BANK1_FBIT20_Msk                                      (0x1UL << CAN_F7BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F7BANK1_FBIT20                                          CAN_F7BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F7BANK1_FBIT21_Pos                                      (21U)
#define CAN_F7BANK1_FBIT21_Msk                                      (0x1UL << CAN_F7BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F7BANK1_FBIT21                                          CAN_F7BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F7BANK1_FBIT22_Pos                                      (22U)
#define CAN_F7BANK1_FBIT22_Msk                                      (0x1UL << CAN_F7BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F7BANK1_FBIT22                                          CAN_F7BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F7BANK1_FBIT23_Pos                                      (23U)
#define CAN_F7BANK1_FBIT23_Msk                                      (0x1UL << CAN_F7BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F7BANK1_FBIT23                                          CAN_F7BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F7BANK1_FBIT24_Pos                                      (24U)
#define CAN_F7BANK1_FBIT24_Msk                                      (0x1UL << CAN_F7BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F7BANK1_FBIT24                                          CAN_F7BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F7BANK1_FBIT25_Pos                                      (25U)
#define CAN_F7BANK1_FBIT25_Msk                                      (0x1UL << CAN_F7BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F7BANK1_FBIT25                                          CAN_F7BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F7BANK1_FBIT26_Pos                                      (26U)
#define CAN_F7BANK1_FBIT26_Msk                                      (0x1UL << CAN_F7BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F7BANK1_FBIT26                                          CAN_F7BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F7BANK1_FBIT27_Pos                                      (27U)
#define CAN_F7BANK1_FBIT27_Msk                                      (0x1UL << CAN_F7BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F7BANK1_FBIT27                                          CAN_F7BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F7BANK1_FBIT28_Pos                                      (28U)
#define CAN_F7BANK1_FBIT28_Msk                                      (0x1UL << CAN_F7BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F7BANK1_FBIT28                                          CAN_F7BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F7BANK1_FBIT29_Pos                                      (29U)
#define CAN_F7BANK1_FBIT29_Msk                                      (0x1UL << CAN_F7BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F7BANK1_FBIT29                                          CAN_F7BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F7BANK1_FBIT30_Pos                                      (30U)
#define CAN_F7BANK1_FBIT30_Msk                                      (0x1UL << CAN_F7BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F7BANK1_FBIT30                                          CAN_F7BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F7BANK1_FBIT31_Pos                                      (31U)
#define CAN_F7BANK1_FBIT31_Msk                                      (0x1UL << CAN_F7BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F7BANK1_FBIT31                                          CAN_F7BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F8BANK1 register  *******************/
#define CAN_F8BANK1_FBIT0_Pos                                       (0U)
#define CAN_F8BANK1_FBIT0_Msk                                       (0x1UL << CAN_F8BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F8BANK1_FBIT0                                           CAN_F8BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F8BANK1_FBIT1_Pos                                       (1U)
#define CAN_F8BANK1_FBIT1_Msk                                       (0x1UL << CAN_F8BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F8BANK1_FBIT1                                           CAN_F8BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F8BANK1_FBIT2_Pos                                       (2U)
#define CAN_F8BANK1_FBIT2_Msk                                       (0x1UL << CAN_F8BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F8BANK1_FBIT2                                           CAN_F8BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F8BANK1_FBIT3_Pos                                       (3U)
#define CAN_F8BANK1_FBIT3_Msk                                       (0x1UL << CAN_F8BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F8BANK1_FBIT3                                           CAN_F8BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F8BANK1_FBIT4_Pos                                       (4U)
#define CAN_F8BANK1_FBIT4_Msk                                       (0x1UL << CAN_F8BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F8BANK1_FBIT4                                           CAN_F8BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F8BANK1_FBIT5_Pos                                       (5U)
#define CAN_F8BANK1_FBIT5_Msk                                       (0x1UL << CAN_F8BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F8BANK1_FBIT5                                           CAN_F8BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F8BANK1_FBIT6_Pos                                       (6U)
#define CAN_F8BANK1_FBIT6_Msk                                       (0x1UL << CAN_F8BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F8BANK1_FBIT6                                           CAN_F8BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F8BANK1_FBIT7_Pos                                       (7U)
#define CAN_F8BANK1_FBIT7_Msk                                       (0x1UL << CAN_F8BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F8BANK1_FBIT7                                           CAN_F8BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F8BANK1_FBIT8_Pos                                       (8U)
#define CAN_F8BANK1_FBIT8_Msk                                       (0x1UL << CAN_F8BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F8BANK1_FBIT8                                           CAN_F8BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F8BANK1_FBIT9_Pos                                       (9U)
#define CAN_F8BANK1_FBIT9_Msk                                       (0x1UL << CAN_F8BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F8BANK1_FBIT9                                           CAN_F8BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F8BANK1_FBIT10_Pos                                      (10U)
#define CAN_F8BANK1_FBIT10_Msk                                      (0x1UL << CAN_F8BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F8BANK1_FBIT10                                          CAN_F8BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F8BANK1_FBIT11_Pos                                      (11U)
#define CAN_F8BANK1_FBIT11_Msk                                      (0x1UL << CAN_F8BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F8BANK1_FBIT11                                          CAN_F8BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F8BANK1_FBIT12_Pos                                      (12U)
#define CAN_F8BANK1_FBIT12_Msk                                      (0x1UL << CAN_F8BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F8BANK1_FBIT12                                          CAN_F8BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F8BANK1_FBIT13_Pos                                      (13U)
#define CAN_F8BANK1_FBIT13_Msk                                      (0x1UL << CAN_F8BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F8BANK1_FBIT13                                          CAN_F8BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F8BANK1_FBIT14_Pos                                      (14U)
#define CAN_F8BANK1_FBIT14_Msk                                      (0x1UL << CAN_F8BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F8BANK1_FBIT14                                          CAN_F8BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F8BANK1_FBIT15_Pos                                      (15U)
#define CAN_F8BANK1_FBIT15_Msk                                      (0x1UL << CAN_F8BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F8BANK1_FBIT15                                          CAN_F8BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F8BANK1_FBIT16_Pos                                      (16U)
#define CAN_F8BANK1_FBIT16_Msk                                      (0x1UL << CAN_F8BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F8BANK1_FBIT16                                          CAN_F8BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F8BANK1_FBIT17_Pos                                      (17U)
#define CAN_F8BANK1_FBIT17_Msk                                      (0x1UL << CAN_F8BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F8BANK1_FBIT17                                          CAN_F8BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F8BANK1_FBIT18_Pos                                      (18U)
#define CAN_F8BANK1_FBIT18_Msk                                      (0x1UL << CAN_F8BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F8BANK1_FBIT18                                          CAN_F8BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F8BANK1_FBIT19_Pos                                      (19U)
#define CAN_F8BANK1_FBIT19_Msk                                      (0x1UL << CAN_F8BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F8BANK1_FBIT19                                          CAN_F8BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F8BANK1_FBIT20_Pos                                      (20U)
#define CAN_F8BANK1_FBIT20_Msk                                      (0x1UL << CAN_F8BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F8BANK1_FBIT20                                          CAN_F8BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F8BANK1_FBIT21_Pos                                      (21U)
#define CAN_F8BANK1_FBIT21_Msk                                      (0x1UL << CAN_F8BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F8BANK1_FBIT21                                          CAN_F8BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F8BANK1_FBIT22_Pos                                      (22U)
#define CAN_F8BANK1_FBIT22_Msk                                      (0x1UL << CAN_F8BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F8BANK1_FBIT22                                          CAN_F8BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F8BANK1_FBIT23_Pos                                      (23U)
#define CAN_F8BANK1_FBIT23_Msk                                      (0x1UL << CAN_F8BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F8BANK1_FBIT23                                          CAN_F8BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F8BANK1_FBIT24_Pos                                      (24U)
#define CAN_F8BANK1_FBIT24_Msk                                      (0x1UL << CAN_F8BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F8BANK1_FBIT24                                          CAN_F8BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F8BANK1_FBIT25_Pos                                      (25U)
#define CAN_F8BANK1_FBIT25_Msk                                      (0x1UL << CAN_F8BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F8BANK1_FBIT25                                          CAN_F8BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F8BANK1_FBIT26_Pos                                      (26U)
#define CAN_F8BANK1_FBIT26_Msk                                      (0x1UL << CAN_F8BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F8BANK1_FBIT26                                          CAN_F8BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F8BANK1_FBIT27_Pos                                      (27U)
#define CAN_F8BANK1_FBIT27_Msk                                      (0x1UL << CAN_F8BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F8BANK1_FBIT27                                          CAN_F8BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F8BANK1_FBIT28_Pos                                      (28U)
#define CAN_F8BANK1_FBIT28_Msk                                      (0x1UL << CAN_F8BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F8BANK1_FBIT28                                          CAN_F8BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F8BANK1_FBIT29_Pos                                      (29U)
#define CAN_F8BANK1_FBIT29_Msk                                      (0x1UL << CAN_F8BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F8BANK1_FBIT29                                          CAN_F8BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F8BANK1_FBIT30_Pos                                      (30U)
#define CAN_F8BANK1_FBIT30_Msk                                      (0x1UL << CAN_F8BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F8BANK1_FBIT30                                          CAN_F8BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F8BANK1_FBIT31_Pos                                      (31U)
#define CAN_F8BANK1_FBIT31_Msk                                      (0x1UL << CAN_F8BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F8BANK1_FBIT31                                          CAN_F8BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F9BANK1 register  *******************/
#define CAN_F9BANK1_FBIT0_Pos                                       (0U)
#define CAN_F9BANK1_FBIT0_Msk                                       (0x1UL << CAN_F9BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F9BANK1_FBIT0                                           CAN_F9BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F9BANK1_FBIT1_Pos                                       (1U)
#define CAN_F9BANK1_FBIT1_Msk                                       (0x1UL << CAN_F9BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F9BANK1_FBIT1                                           CAN_F9BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F9BANK1_FBIT2_Pos                                       (2U)
#define CAN_F9BANK1_FBIT2_Msk                                       (0x1UL << CAN_F9BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F9BANK1_FBIT2                                           CAN_F9BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F9BANK1_FBIT3_Pos                                       (3U)
#define CAN_F9BANK1_FBIT3_Msk                                       (0x1UL << CAN_F9BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F9BANK1_FBIT3                                           CAN_F9BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F9BANK1_FBIT4_Pos                                       (4U)
#define CAN_F9BANK1_FBIT4_Msk                                       (0x1UL << CAN_F9BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F9BANK1_FBIT4                                           CAN_F9BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F9BANK1_FBIT5_Pos                                       (5U)
#define CAN_F9BANK1_FBIT5_Msk                                       (0x1UL << CAN_F9BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F9BANK1_FBIT5                                           CAN_F9BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F9BANK1_FBIT6_Pos                                       (6U)
#define CAN_F9BANK1_FBIT6_Msk                                       (0x1UL << CAN_F9BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F9BANK1_FBIT6                                           CAN_F9BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F9BANK1_FBIT7_Pos                                       (7U)
#define CAN_F9BANK1_FBIT7_Msk                                       (0x1UL << CAN_F9BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F9BANK1_FBIT7                                           CAN_F9BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F9BANK1_FBIT8_Pos                                       (8U)
#define CAN_F9BANK1_FBIT8_Msk                                       (0x1UL << CAN_F9BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F9BANK1_FBIT8                                           CAN_F9BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F9BANK1_FBIT9_Pos                                       (9U)
#define CAN_F9BANK1_FBIT9_Msk                                       (0x1UL << CAN_F9BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F9BANK1_FBIT9                                           CAN_F9BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F9BANK1_FBIT10_Pos                                      (10U)
#define CAN_F9BANK1_FBIT10_Msk                                      (0x1UL << CAN_F9BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F9BANK1_FBIT10                                          CAN_F9BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F9BANK1_FBIT11_Pos                                      (11U)
#define CAN_F9BANK1_FBIT11_Msk                                      (0x1UL << CAN_F9BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F9BANK1_FBIT11                                          CAN_F9BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F9BANK1_FBIT12_Pos                                      (12U)
#define CAN_F9BANK1_FBIT12_Msk                                      (0x1UL << CAN_F9BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F9BANK1_FBIT12                                          CAN_F9BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F9BANK1_FBIT13_Pos                                      (13U)
#define CAN_F9BANK1_FBIT13_Msk                                      (0x1UL << CAN_F9BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F9BANK1_FBIT13                                          CAN_F9BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F9BANK1_FBIT14_Pos                                      (14U)
#define CAN_F9BANK1_FBIT14_Msk                                      (0x1UL << CAN_F9BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F9BANK1_FBIT14                                          CAN_F9BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F9BANK1_FBIT15_Pos                                      (15U)
#define CAN_F9BANK1_FBIT15_Msk                                      (0x1UL << CAN_F9BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F9BANK1_FBIT15                                          CAN_F9BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F9BANK1_FBIT16_Pos                                      (16U)
#define CAN_F9BANK1_FBIT16_Msk                                      (0x1UL << CAN_F9BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F9BANK1_FBIT16                                          CAN_F9BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F9BANK1_FBIT17_Pos                                      (17U)
#define CAN_F9BANK1_FBIT17_Msk                                      (0x1UL << CAN_F9BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F9BANK1_FBIT17                                          CAN_F9BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F9BANK1_FBIT18_Pos                                      (18U)
#define CAN_F9BANK1_FBIT18_Msk                                      (0x1UL << CAN_F9BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F9BANK1_FBIT18                                          CAN_F9BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F9BANK1_FBIT19_Pos                                      (19U)
#define CAN_F9BANK1_FBIT19_Msk                                      (0x1UL << CAN_F9BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F9BANK1_FBIT19                                          CAN_F9BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F9BANK1_FBIT20_Pos                                      (20U)
#define CAN_F9BANK1_FBIT20_Msk                                      (0x1UL << CAN_F9BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F9BANK1_FBIT20                                          CAN_F9BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F9BANK1_FBIT21_Pos                                      (21U)
#define CAN_F9BANK1_FBIT21_Msk                                      (0x1UL << CAN_F9BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F9BANK1_FBIT21                                          CAN_F9BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F9BANK1_FBIT22_Pos                                      (22U)
#define CAN_F9BANK1_FBIT22_Msk                                      (0x1UL << CAN_F9BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F9BANK1_FBIT22                                          CAN_F9BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F9BANK1_FBIT23_Pos                                      (23U)
#define CAN_F9BANK1_FBIT23_Msk                                      (0x1UL << CAN_F9BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F9BANK1_FBIT23                                          CAN_F9BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F9BANK1_FBIT24_Pos                                      (24U)
#define CAN_F9BANK1_FBIT24_Msk                                      (0x1UL << CAN_F9BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F9BANK1_FBIT24                                          CAN_F9BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F9BANK1_FBIT25_Pos                                      (25U)
#define CAN_F9BANK1_FBIT25_Msk                                      (0x1UL << CAN_F9BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F9BANK1_FBIT25                                          CAN_F9BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F9BANK1_FBIT26_Pos                                      (26U)
#define CAN_F9BANK1_FBIT26_Msk                                      (0x1UL << CAN_F9BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F9BANK1_FBIT26                                          CAN_F9BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F9BANK1_FBIT27_Pos                                      (27U)
#define CAN_F9BANK1_FBIT27_Msk                                      (0x1UL << CAN_F9BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F9BANK1_FBIT27                                          CAN_F9BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F9BANK1_FBIT28_Pos                                      (28U)
#define CAN_F9BANK1_FBIT28_Msk                                      (0x1UL << CAN_F9BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F9BANK1_FBIT28                                          CAN_F9BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F9BANK1_FBIT29_Pos                                      (29U)
#define CAN_F9BANK1_FBIT29_Msk                                      (0x1UL << CAN_F9BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F9BANK1_FBIT29                                          CAN_F9BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F9BANK1_FBIT30_Pos                                      (30U)
#define CAN_F9BANK1_FBIT30_Msk                                      (0x1UL << CAN_F9BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F9BANK1_FBIT30                                          CAN_F9BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F9BANK1_FBIT31_Pos                                      (31U)
#define CAN_F9BANK1_FBIT31_Msk                                      (0x1UL << CAN_F9BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F9BANK1_FBIT31                                          CAN_F9BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F10BANK1 register  ******************/
#define CAN_F10BANK1_FBIT0_Pos                                      (0U)
#define CAN_F10BANK1_FBIT0_Msk                                      (0x1UL << CAN_F10BANK1_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F10BANK1_FBIT0                                          CAN_F10BANK1_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F10BANK1_FBIT1_Pos                                      (1U)
#define CAN_F10BANK1_FBIT1_Msk                                      (0x1UL << CAN_F10BANK1_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F10BANK1_FBIT1                                          CAN_F10BANK1_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F10BANK1_FBIT2_Pos                                      (2U)
#define CAN_F10BANK1_FBIT2_Msk                                      (0x1UL << CAN_F10BANK1_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F10BANK1_FBIT2                                          CAN_F10BANK1_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F10BANK1_FBIT3_Pos                                      (3U)
#define CAN_F10BANK1_FBIT3_Msk                                      (0x1UL << CAN_F10BANK1_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F10BANK1_FBIT3                                          CAN_F10BANK1_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F10BANK1_FBIT4_Pos                                      (4U)
#define CAN_F10BANK1_FBIT4_Msk                                      (0x1UL << CAN_F10BANK1_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F10BANK1_FBIT4                                          CAN_F10BANK1_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F10BANK1_FBIT5_Pos                                      (5U)
#define CAN_F10BANK1_FBIT5_Msk                                      (0x1UL << CAN_F10BANK1_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F10BANK1_FBIT5                                          CAN_F10BANK1_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F10BANK1_FBIT6_Pos                                      (6U)
#define CAN_F10BANK1_FBIT6_Msk                                      (0x1UL << CAN_F10BANK1_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F10BANK1_FBIT6                                          CAN_F10BANK1_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F10BANK1_FBIT7_Pos                                      (7U)
#define CAN_F10BANK1_FBIT7_Msk                                      (0x1UL << CAN_F10BANK1_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F10BANK1_FBIT7                                          CAN_F10BANK1_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F10BANK1_FBIT8_Pos                                      (8U)
#define CAN_F10BANK1_FBIT8_Msk                                      (0x1UL << CAN_F10BANK1_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F10BANK1_FBIT8                                          CAN_F10BANK1_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F10BANK1_FBIT9_Pos                                      (9U)
#define CAN_F10BANK1_FBIT9_Msk                                      (0x1UL << CAN_F10BANK1_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F10BANK1_FBIT9                                          CAN_F10BANK1_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F10BANK1_FBIT10_Pos                                     (10U)
#define CAN_F10BANK1_FBIT10_Msk                                     (0x1UL << CAN_F10BANK1_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F10BANK1_FBIT10                                         CAN_F10BANK1_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F10BANK1_FBIT11_Pos                                     (11U)
#define CAN_F10BANK1_FBIT11_Msk                                     (0x1UL << CAN_F10BANK1_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F10BANK1_FBIT11                                         CAN_F10BANK1_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F10BANK1_FBIT12_Pos                                     (12U)
#define CAN_F10BANK1_FBIT12_Msk                                     (0x1UL << CAN_F10BANK1_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F10BANK1_FBIT12                                         CAN_F10BANK1_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F10BANK1_FBIT13_Pos                                     (13U)
#define CAN_F10BANK1_FBIT13_Msk                                     (0x1UL << CAN_F10BANK1_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F10BANK1_FBIT13                                         CAN_F10BANK1_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F10BANK1_FBIT14_Pos                                     (14U)
#define CAN_F10BANK1_FBIT14_Msk                                     (0x1UL << CAN_F10BANK1_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F10BANK1_FBIT14                                         CAN_F10BANK1_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F10BANK1_FBIT15_Pos                                     (15U)
#define CAN_F10BANK1_FBIT15_Msk                                     (0x1UL << CAN_F10BANK1_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F10BANK1_FBIT15                                         CAN_F10BANK1_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F10BANK1_FBIT16_Pos                                     (16U)
#define CAN_F10BANK1_FBIT16_Msk                                     (0x1UL << CAN_F10BANK1_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F10BANK1_FBIT16                                         CAN_F10BANK1_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F10BANK1_FBIT17_Pos                                     (17U)
#define CAN_F10BANK1_FBIT17_Msk                                     (0x1UL << CAN_F10BANK1_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F10BANK1_FBIT17                                         CAN_F10BANK1_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F10BANK1_FBIT18_Pos                                     (18U)
#define CAN_F10BANK1_FBIT18_Msk                                     (0x1UL << CAN_F10BANK1_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F10BANK1_FBIT18                                         CAN_F10BANK1_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F10BANK1_FBIT19_Pos                                     (19U)
#define CAN_F10BANK1_FBIT19_Msk                                     (0x1UL << CAN_F10BANK1_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F10BANK1_FBIT19                                         CAN_F10BANK1_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F10BANK1_FBIT20_Pos                                     (20U)
#define CAN_F10BANK1_FBIT20_Msk                                     (0x1UL << CAN_F10BANK1_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F10BANK1_FBIT20                                         CAN_F10BANK1_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F10BANK1_FBIT21_Pos                                     (21U)
#define CAN_F10BANK1_FBIT21_Msk                                     (0x1UL << CAN_F10BANK1_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F10BANK1_FBIT21                                         CAN_F10BANK1_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F10BANK1_FBIT22_Pos                                     (22U)
#define CAN_F10BANK1_FBIT22_Msk                                     (0x1UL << CAN_F10BANK1_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F10BANK1_FBIT22                                         CAN_F10BANK1_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F10BANK1_FBIT23_Pos                                     (23U)
#define CAN_F10BANK1_FBIT23_Msk                                     (0x1UL << CAN_F10BANK1_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F10BANK1_FBIT23                                         CAN_F10BANK1_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F10BANK1_FBIT24_Pos                                     (24U)
#define CAN_F10BANK1_FBIT24_Msk                                     (0x1UL << CAN_F10BANK1_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F10BANK1_FBIT24                                         CAN_F10BANK1_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F10BANK1_FBIT25_Pos                                     (25U)
#define CAN_F10BANK1_FBIT25_Msk                                     (0x1UL << CAN_F10BANK1_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F10BANK1_FBIT25                                         CAN_F10BANK1_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F10BANK1_FBIT26_Pos                                     (26U)
#define CAN_F10BANK1_FBIT26_Msk                                     (0x1UL << CAN_F10BANK1_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F10BANK1_FBIT26                                         CAN_F10BANK1_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F10BANK1_FBIT27_Pos                                     (27U)
#define CAN_F10BANK1_FBIT27_Msk                                     (0x1UL << CAN_F10BANK1_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F10BANK1_FBIT27                                         CAN_F10BANK1_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F10BANK1_FBIT28_Pos                                     (28U)
#define CAN_F10BANK1_FBIT28_Msk                                     (0x1UL << CAN_F10BANK1_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F10BANK1_FBIT28                                         CAN_F10BANK1_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F10BANK1_FBIT29_Pos                                     (29U)
#define CAN_F10BANK1_FBIT29_Msk                                     (0x1UL << CAN_F10BANK1_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F10BANK1_FBIT29                                         CAN_F10BANK1_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F10BANK1_FBIT30_Pos                                     (30U)
#define CAN_F10BANK1_FBIT30_Msk                                     (0x1UL << CAN_F10BANK1_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F10BANK1_FBIT30                                         CAN_F10BANK1_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F10BANK1_FBIT31_Pos                                     (31U)
#define CAN_F10BANK1_FBIT31_Msk                                     (0x1UL << CAN_F10BANK1_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F10BANK1_FBIT31                                         CAN_F10BANK1_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F11BANK1 register  ******************/
#define CAN_F11BANK1_FBIT0_Pos                                      (0U)
#define CAN_F11BANK1_FBIT0_Msk                                      (0x1UL << CAN_F11BANK1_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F11BANK1_FBIT0                                          CAN_F11BANK1_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F11BANK1_FBIT1_Pos                                      (1U)
#define CAN_F11BANK1_FBIT1_Msk                                      (0x1UL << CAN_F11BANK1_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F11BANK1_FBIT1                                          CAN_F11BANK1_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F11BANK1_FBIT2_Pos                                      (2U)
#define CAN_F11BANK1_FBIT2_Msk                                      (0x1UL << CAN_F11BANK1_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F11BANK1_FBIT2                                          CAN_F11BANK1_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F11BANK1_FBIT3_Pos                                      (3U)
#define CAN_F11BANK1_FBIT3_Msk                                      (0x1UL << CAN_F11BANK1_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F11BANK1_FBIT3                                          CAN_F11BANK1_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F11BANK1_FBIT4_Pos                                      (4U)
#define CAN_F11BANK1_FBIT4_Msk                                      (0x1UL << CAN_F11BANK1_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F11BANK1_FBIT4                                          CAN_F11BANK1_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F11BANK1_FBIT5_Pos                                      (5U)
#define CAN_F11BANK1_FBIT5_Msk                                      (0x1UL << CAN_F11BANK1_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F11BANK1_FBIT5                                          CAN_F11BANK1_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F11BANK1_FBIT6_Pos                                      (6U)
#define CAN_F11BANK1_FBIT6_Msk                                      (0x1UL << CAN_F11BANK1_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F11BANK1_FBIT6                                          CAN_F11BANK1_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F11BANK1_FBIT7_Pos                                      (7U)
#define CAN_F11BANK1_FBIT7_Msk                                      (0x1UL << CAN_F11BANK1_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F11BANK1_FBIT7                                          CAN_F11BANK1_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F11BANK1_FBIT8_Pos                                      (8U)
#define CAN_F11BANK1_FBIT8_Msk                                      (0x1UL << CAN_F11BANK1_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F11BANK1_FBIT8                                          CAN_F11BANK1_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F11BANK1_FBIT9_Pos                                      (9U)
#define CAN_F11BANK1_FBIT9_Msk                                      (0x1UL << CAN_F11BANK1_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F11BANK1_FBIT9                                          CAN_F11BANK1_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F11BANK1_FBIT10_Pos                                     (10U)
#define CAN_F11BANK1_FBIT10_Msk                                     (0x1UL << CAN_F11BANK1_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F11BANK1_FBIT10                                         CAN_F11BANK1_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F11BANK1_FBIT11_Pos                                     (11U)
#define CAN_F11BANK1_FBIT11_Msk                                     (0x1UL << CAN_F11BANK1_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F11BANK1_FBIT11                                         CAN_F11BANK1_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F11BANK1_FBIT12_Pos                                     (12U)
#define CAN_F11BANK1_FBIT12_Msk                                     (0x1UL << CAN_F11BANK1_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F11BANK1_FBIT12                                         CAN_F11BANK1_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F11BANK1_FBIT13_Pos                                     (13U)
#define CAN_F11BANK1_FBIT13_Msk                                     (0x1UL << CAN_F11BANK1_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F11BANK1_FBIT13                                         CAN_F11BANK1_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F11BANK1_FBIT14_Pos                                     (14U)
#define CAN_F11BANK1_FBIT14_Msk                                     (0x1UL << CAN_F11BANK1_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F11BANK1_FBIT14                                         CAN_F11BANK1_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F11BANK1_FBIT15_Pos                                     (15U)
#define CAN_F11BANK1_FBIT15_Msk                                     (0x1UL << CAN_F11BANK1_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F11BANK1_FBIT15                                         CAN_F11BANK1_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F11BANK1_FBIT16_Pos                                     (16U)
#define CAN_F11BANK1_FBIT16_Msk                                     (0x1UL << CAN_F11BANK1_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F11BANK1_FBIT16                                         CAN_F11BANK1_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F11BANK1_FBIT17_Pos                                     (17U)
#define CAN_F11BANK1_FBIT17_Msk                                     (0x1UL << CAN_F11BANK1_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F11BANK1_FBIT17                                         CAN_F11BANK1_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F11BANK1_FBIT18_Pos                                     (18U)
#define CAN_F11BANK1_FBIT18_Msk                                     (0x1UL << CAN_F11BANK1_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F11BANK1_FBIT18                                         CAN_F11BANK1_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F11BANK1_FBIT19_Pos                                     (19U)
#define CAN_F11BANK1_FBIT19_Msk                                     (0x1UL << CAN_F11BANK1_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F11BANK1_FBIT19                                         CAN_F11BANK1_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F11BANK1_FBIT20_Pos                                     (20U)
#define CAN_F11BANK1_FBIT20_Msk                                     (0x1UL << CAN_F11BANK1_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F11BANK1_FBIT20                                         CAN_F11BANK1_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F11BANK1_FBIT21_Pos                                     (21U)
#define CAN_F11BANK1_FBIT21_Msk                                     (0x1UL << CAN_F11BANK1_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F11BANK1_FBIT21                                         CAN_F11BANK1_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F11BANK1_FBIT22_Pos                                     (22U)
#define CAN_F11BANK1_FBIT22_Msk                                     (0x1UL << CAN_F11BANK1_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F11BANK1_FBIT22                                         CAN_F11BANK1_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F11BANK1_FBIT23_Pos                                     (23U)
#define CAN_F11BANK1_FBIT23_Msk                                     (0x1UL << CAN_F11BANK1_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F11BANK1_FBIT23                                         CAN_F11BANK1_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F11BANK1_FBIT24_Pos                                     (24U)
#define CAN_F11BANK1_FBIT24_Msk                                     (0x1UL << CAN_F11BANK1_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F11BANK1_FBIT24                                         CAN_F11BANK1_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F11BANK1_FBIT25_Pos                                     (25U)
#define CAN_F11BANK1_FBIT25_Msk                                     (0x1UL << CAN_F11BANK1_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F11BANK1_FBIT25                                         CAN_F11BANK1_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F11BANK1_FBIT26_Pos                                     (26U)
#define CAN_F11BANK1_FBIT26_Msk                                     (0x1UL << CAN_F11BANK1_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F11BANK1_FBIT26                                         CAN_F11BANK1_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F11BANK1_FBIT27_Pos                                     (27U)
#define CAN_F11BANK1_FBIT27_Msk                                     (0x1UL << CAN_F11BANK1_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F11BANK1_FBIT27                                         CAN_F11BANK1_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F11BANK1_FBIT28_Pos                                     (28U)
#define CAN_F11BANK1_FBIT28_Msk                                     (0x1UL << CAN_F11BANK1_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F11BANK1_FBIT28                                         CAN_F11BANK1_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F11BANK1_FBIT29_Pos                                     (29U)
#define CAN_F11BANK1_FBIT29_Msk                                     (0x1UL << CAN_F11BANK1_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F11BANK1_FBIT29                                         CAN_F11BANK1_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F11BANK1_FBIT30_Pos                                     (30U)
#define CAN_F11BANK1_FBIT30_Msk                                     (0x1UL << CAN_F11BANK1_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F11BANK1_FBIT30                                         CAN_F11BANK1_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F11BANK1_FBIT31_Pos                                     (31U)
#define CAN_F11BANK1_FBIT31_Msk                                     (0x1UL << CAN_F11BANK1_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F11BANK1_FBIT31                                         CAN_F11BANK1_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F12BANK1 register  ******************/
#define CAN_F12BANK1_FBIT0_Pos                                      (0U)
#define CAN_F12BANK1_FBIT0_Msk                                      (0x1UL << CAN_F12BANK1_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F12BANK1_FBIT0                                          CAN_F12BANK1_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F12BANK1_FBIT1_Pos                                      (1U)
#define CAN_F12BANK1_FBIT1_Msk                                      (0x1UL << CAN_F12BANK1_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F12BANK1_FBIT1                                          CAN_F12BANK1_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F12BANK1_FBIT2_Pos                                      (2U)
#define CAN_F12BANK1_FBIT2_Msk                                      (0x1UL << CAN_F12BANK1_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F12BANK1_FBIT2                                          CAN_F12BANK1_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F12BANK1_FBIT3_Pos                                      (3U)
#define CAN_F12BANK1_FBIT3_Msk                                      (0x1UL << CAN_F12BANK1_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F12BANK1_FBIT3                                          CAN_F12BANK1_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F12BANK1_FBIT4_Pos                                      (4U)
#define CAN_F12BANK1_FBIT4_Msk                                      (0x1UL << CAN_F12BANK1_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F12BANK1_FBIT4                                          CAN_F12BANK1_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F12BANK1_FBIT5_Pos                                      (5U)
#define CAN_F12BANK1_FBIT5_Msk                                      (0x1UL << CAN_F12BANK1_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F12BANK1_FBIT5                                          CAN_F12BANK1_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F12BANK1_FBIT6_Pos                                      (6U)
#define CAN_F12BANK1_FBIT6_Msk                                      (0x1UL << CAN_F12BANK1_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F12BANK1_FBIT6                                          CAN_F12BANK1_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F12BANK1_FBIT7_Pos                                      (7U)
#define CAN_F12BANK1_FBIT7_Msk                                      (0x1UL << CAN_F12BANK1_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F12BANK1_FBIT7                                          CAN_F12BANK1_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F12BANK1_FBIT8_Pos                                      (8U)
#define CAN_F12BANK1_FBIT8_Msk                                      (0x1UL << CAN_F12BANK1_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F12BANK1_FBIT8                                          CAN_F12BANK1_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F12BANK1_FBIT9_Pos                                      (9U)
#define CAN_F12BANK1_FBIT9_Msk                                      (0x1UL << CAN_F12BANK1_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F12BANK1_FBIT9                                          CAN_F12BANK1_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F12BANK1_FBIT10_Pos                                     (10U)
#define CAN_F12BANK1_FBIT10_Msk                                     (0x1UL << CAN_F12BANK1_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F12BANK1_FBIT10                                         CAN_F12BANK1_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F12BANK1_FBIT11_Pos                                     (11U)
#define CAN_F12BANK1_FBIT11_Msk                                     (0x1UL << CAN_F12BANK1_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F12BANK1_FBIT11                                         CAN_F12BANK1_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F12BANK1_FBIT12_Pos                                     (12U)
#define CAN_F12BANK1_FBIT12_Msk                                     (0x1UL << CAN_F12BANK1_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F12BANK1_FBIT12                                         CAN_F12BANK1_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F12BANK1_FBIT13_Pos                                     (13U)
#define CAN_F12BANK1_FBIT13_Msk                                     (0x1UL << CAN_F12BANK1_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F12BANK1_FBIT13                                         CAN_F12BANK1_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F12BANK1_FBIT14_Pos                                     (14U)
#define CAN_F12BANK1_FBIT14_Msk                                     (0x1UL << CAN_F12BANK1_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F12BANK1_FBIT14                                         CAN_F12BANK1_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F12BANK1_FBIT15_Pos                                     (15U)
#define CAN_F12BANK1_FBIT15_Msk                                     (0x1UL << CAN_F12BANK1_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F12BANK1_FBIT15                                         CAN_F12BANK1_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F12BANK1_FBIT16_Pos                                     (16U)
#define CAN_F12BANK1_FBIT16_Msk                                     (0x1UL << CAN_F12BANK1_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F12BANK1_FBIT16                                         CAN_F12BANK1_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F12BANK1_FBIT17_Pos                                     (17U)
#define CAN_F12BANK1_FBIT17_Msk                                     (0x1UL << CAN_F12BANK1_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F12BANK1_FBIT17                                         CAN_F12BANK1_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F12BANK1_FBIT18_Pos                                     (18U)
#define CAN_F12BANK1_FBIT18_Msk                                     (0x1UL << CAN_F12BANK1_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F12BANK1_FBIT18                                         CAN_F12BANK1_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F12BANK1_FBIT19_Pos                                     (19U)
#define CAN_F12BANK1_FBIT19_Msk                                     (0x1UL << CAN_F12BANK1_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F12BANK1_FBIT19                                         CAN_F12BANK1_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F12BANK1_FBIT20_Pos                                     (20U)
#define CAN_F12BANK1_FBIT20_Msk                                     (0x1UL << CAN_F12BANK1_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F12BANK1_FBIT20                                         CAN_F12BANK1_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F12BANK1_FBIT21_Pos                                     (21U)
#define CAN_F12BANK1_FBIT21_Msk                                     (0x1UL << CAN_F12BANK1_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F12BANK1_FBIT21                                         CAN_F12BANK1_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F12BANK1_FBIT22_Pos                                     (22U)
#define CAN_F12BANK1_FBIT22_Msk                                     (0x1UL << CAN_F12BANK1_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F12BANK1_FBIT22                                         CAN_F12BANK1_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F12BANK1_FBIT23_Pos                                     (23U)
#define CAN_F12BANK1_FBIT23_Msk                                     (0x1UL << CAN_F12BANK1_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F12BANK1_FBIT23                                         CAN_F12BANK1_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F12BANK1_FBIT24_Pos                                     (24U)
#define CAN_F12BANK1_FBIT24_Msk                                     (0x1UL << CAN_F12BANK1_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F12BANK1_FBIT24                                         CAN_F12BANK1_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F12BANK1_FBIT25_Pos                                     (25U)
#define CAN_F12BANK1_FBIT25_Msk                                     (0x1UL << CAN_F12BANK1_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F12BANK1_FBIT25                                         CAN_F12BANK1_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F12BANK1_FBIT26_Pos                                     (26U)
#define CAN_F12BANK1_FBIT26_Msk                                     (0x1UL << CAN_F12BANK1_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F12BANK1_FBIT26                                         CAN_F12BANK1_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F12BANK1_FBIT27_Pos                                     (27U)
#define CAN_F12BANK1_FBIT27_Msk                                     (0x1UL << CAN_F12BANK1_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F12BANK1_FBIT27                                         CAN_F12BANK1_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F12BANK1_FBIT28_Pos                                     (28U)
#define CAN_F12BANK1_FBIT28_Msk                                     (0x1UL << CAN_F12BANK1_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F12BANK1_FBIT28                                         CAN_F12BANK1_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F12BANK1_FBIT29_Pos                                     (29U)
#define CAN_F12BANK1_FBIT29_Msk                                     (0x1UL << CAN_F12BANK1_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F12BANK1_FBIT29                                         CAN_F12BANK1_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F12BANK1_FBIT30_Pos                                     (30U)
#define CAN_F12BANK1_FBIT30_Msk                                     (0x1UL << CAN_F12BANK1_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F12BANK1_FBIT30                                         CAN_F12BANK1_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F12BANK1_FBIT31_Pos                                     (31U)
#define CAN_F12BANK1_FBIT31_Msk                                     (0x1UL << CAN_F12BANK1_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F12BANK1_FBIT31                                         CAN_F12BANK1_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F13BANK1 register  ******************/
#define CAN_F13BANK1_FBIT0_Pos                                      (0U)
#define CAN_F13BANK1_FBIT0_Msk                                      (0x1UL << CAN_F13BANK1_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F13BANK1_FBIT0                                          CAN_F13BANK1_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F13BANK1_FBIT1_Pos                                      (1U)
#define CAN_F13BANK1_FBIT1_Msk                                      (0x1UL << CAN_F13BANK1_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F13BANK1_FBIT1                                          CAN_F13BANK1_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F13BANK1_FBIT2_Pos                                      (2U)
#define CAN_F13BANK1_FBIT2_Msk                                      (0x1UL << CAN_F13BANK1_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F13BANK1_FBIT2                                          CAN_F13BANK1_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F13BANK1_FBIT3_Pos                                      (3U)
#define CAN_F13BANK1_FBIT3_Msk                                      (0x1UL << CAN_F13BANK1_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F13BANK1_FBIT3                                          CAN_F13BANK1_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F13BANK1_FBIT4_Pos                                      (4U)
#define CAN_F13BANK1_FBIT4_Msk                                      (0x1UL << CAN_F13BANK1_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F13BANK1_FBIT4                                          CAN_F13BANK1_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F13BANK1_FBIT5_Pos                                      (5U)
#define CAN_F13BANK1_FBIT5_Msk                                      (0x1UL << CAN_F13BANK1_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F13BANK1_FBIT5                                          CAN_F13BANK1_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F13BANK1_FBIT6_Pos                                      (6U)
#define CAN_F13BANK1_FBIT6_Msk                                      (0x1UL << CAN_F13BANK1_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F13BANK1_FBIT6                                          CAN_F13BANK1_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F13BANK1_FBIT7_Pos                                      (7U)
#define CAN_F13BANK1_FBIT7_Msk                                      (0x1UL << CAN_F13BANK1_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F13BANK1_FBIT7                                          CAN_F13BANK1_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F13BANK1_FBIT8_Pos                                      (8U)
#define CAN_F13BANK1_FBIT8_Msk                                      (0x1UL << CAN_F13BANK1_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F13BANK1_FBIT8                                          CAN_F13BANK1_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F13BANK1_FBIT9_Pos                                      (9U)
#define CAN_F13BANK1_FBIT9_Msk                                      (0x1UL << CAN_F13BANK1_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F13BANK1_FBIT9                                          CAN_F13BANK1_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F13BANK1_FBIT10_Pos                                     (10U)
#define CAN_F13BANK1_FBIT10_Msk                                     (0x1UL << CAN_F13BANK1_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F13BANK1_FBIT10                                         CAN_F13BANK1_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F13BANK1_FBIT11_Pos                                     (11U)
#define CAN_F13BANK1_FBIT11_Msk                                     (0x1UL << CAN_F13BANK1_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F13BANK1_FBIT11                                         CAN_F13BANK1_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F13BANK1_FBIT12_Pos                                     (12U)
#define CAN_F13BANK1_FBIT12_Msk                                     (0x1UL << CAN_F13BANK1_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F13BANK1_FBIT12                                         CAN_F13BANK1_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F13BANK1_FBIT13_Pos                                     (13U)
#define CAN_F13BANK1_FBIT13_Msk                                     (0x1UL << CAN_F13BANK1_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F13BANK1_FBIT13                                         CAN_F13BANK1_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F13BANK1_FBIT14_Pos                                     (14U)
#define CAN_F13BANK1_FBIT14_Msk                                     (0x1UL << CAN_F13BANK1_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F13BANK1_FBIT14                                         CAN_F13BANK1_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F13BANK1_FBIT15_Pos                                     (15U)
#define CAN_F13BANK1_FBIT15_Msk                                     (0x1UL << CAN_F13BANK1_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F13BANK1_FBIT15                                         CAN_F13BANK1_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F13BANK1_FBIT16_Pos                                     (16U)
#define CAN_F13BANK1_FBIT16_Msk                                     (0x1UL << CAN_F13BANK1_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F13BANK1_FBIT16                                         CAN_F13BANK1_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F13BANK1_FBIT17_Pos                                     (17U)
#define CAN_F13BANK1_FBIT17_Msk                                     (0x1UL << CAN_F13BANK1_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F13BANK1_FBIT17                                         CAN_F13BANK1_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F13BANK1_FBIT18_Pos                                     (18U)
#define CAN_F13BANK1_FBIT18_Msk                                     (0x1UL << CAN_F13BANK1_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F13BANK1_FBIT18                                         CAN_F13BANK1_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F13BANK1_FBIT19_Pos                                     (19U)
#define CAN_F13BANK1_FBIT19_Msk                                     (0x1UL << CAN_F13BANK1_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F13BANK1_FBIT19                                         CAN_F13BANK1_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F13BANK1_FBIT20_Pos                                     (20U)
#define CAN_F13BANK1_FBIT20_Msk                                     (0x1UL << CAN_F13BANK1_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F13BANK1_FBIT20                                         CAN_F13BANK1_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F13BANK1_FBIT21_Pos                                     (21U)
#define CAN_F13BANK1_FBIT21_Msk                                     (0x1UL << CAN_F13BANK1_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F13BANK1_FBIT21                                         CAN_F13BANK1_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F13BANK1_FBIT22_Pos                                     (22U)
#define CAN_F13BANK1_FBIT22_Msk                                     (0x1UL << CAN_F13BANK1_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F13BANK1_FBIT22                                         CAN_F13BANK1_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F13BANK1_FBIT23_Pos                                     (23U)
#define CAN_F13BANK1_FBIT23_Msk                                     (0x1UL << CAN_F13BANK1_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F13BANK1_FBIT23                                         CAN_F13BANK1_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F13BANK1_FBIT24_Pos                                     (24U)
#define CAN_F13BANK1_FBIT24_Msk                                     (0x1UL << CAN_F13BANK1_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F13BANK1_FBIT24                                         CAN_F13BANK1_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F13BANK1_FBIT25_Pos                                     (25U)
#define CAN_F13BANK1_FBIT25_Msk                                     (0x1UL << CAN_F13BANK1_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F13BANK1_FBIT25                                         CAN_F13BANK1_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F13BANK1_FBIT26_Pos                                     (26U)
#define CAN_F13BANK1_FBIT26_Msk                                     (0x1UL << CAN_F13BANK1_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F13BANK1_FBIT26                                         CAN_F13BANK1_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F13BANK1_FBIT27_Pos                                     (27U)
#define CAN_F13BANK1_FBIT27_Msk                                     (0x1UL << CAN_F13BANK1_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F13BANK1_FBIT27                                         CAN_F13BANK1_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F13BANK1_FBIT28_Pos                                     (28U)
#define CAN_F13BANK1_FBIT28_Msk                                     (0x1UL << CAN_F13BANK1_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F13BANK1_FBIT28                                         CAN_F13BANK1_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F13BANK1_FBIT29_Pos                                     (29U)
#define CAN_F13BANK1_FBIT29_Msk                                     (0x1UL << CAN_F13BANK1_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F13BANK1_FBIT29                                         CAN_F13BANK1_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F13BANK1_FBIT30_Pos                                     (30U)
#define CAN_F13BANK1_FBIT30_Msk                                     (0x1UL << CAN_F13BANK1_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F13BANK1_FBIT30                                         CAN_F13BANK1_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F13BANK1_FBIT31_Pos                                     (31U)
#define CAN_F13BANK1_FBIT31_Msk                                     (0x1UL << CAN_F13BANK1_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F13BANK1_FBIT31                                         CAN_F13BANK1_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F0BANK2 register  *******************/
#define CAN_F0BANK2_FBIT0_Pos                                       (0U)
#define CAN_F0BANK2_FBIT0_Msk                                       (0x1UL << CAN_F0BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F0BANK2_FBIT0                                           CAN_F0BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F0BANK2_FBIT1_Pos                                       (1U)
#define CAN_F0BANK2_FBIT1_Msk                                       (0x1UL << CAN_F0BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F0BANK2_FBIT1                                           CAN_F0BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F0BANK2_FBIT2_Pos                                       (2U)
#define CAN_F0BANK2_FBIT2_Msk                                       (0x1UL << CAN_F0BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F0BANK2_FBIT2                                           CAN_F0BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F0BANK2_FBIT3_Pos                                       (3U)
#define CAN_F0BANK2_FBIT3_Msk                                       (0x1UL << CAN_F0BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F0BANK2_FBIT3                                           CAN_F0BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F0BANK2_FBIT4_Pos                                       (4U)
#define CAN_F0BANK2_FBIT4_Msk                                       (0x1UL << CAN_F0BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F0BANK2_FBIT4                                           CAN_F0BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F0BANK2_FBIT5_Pos                                       (5U)
#define CAN_F0BANK2_FBIT5_Msk                                       (0x1UL << CAN_F0BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F0BANK2_FBIT5                                           CAN_F0BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F0BANK2_FBIT6_Pos                                       (6U)
#define CAN_F0BANK2_FBIT6_Msk                                       (0x1UL << CAN_F0BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F0BANK2_FBIT6                                           CAN_F0BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F0BANK2_FBIT7_Pos                                       (7U)
#define CAN_F0BANK2_FBIT7_Msk                                       (0x1UL << CAN_F0BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F0BANK2_FBIT7                                           CAN_F0BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F0BANK2_FBIT8_Pos                                       (8U)
#define CAN_F0BANK2_FBIT8_Msk                                       (0x1UL << CAN_F0BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F0BANK2_FBIT8                                           CAN_F0BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F0BANK2_FBIT9_Pos                                       (9U)
#define CAN_F0BANK2_FBIT9_Msk                                       (0x1UL << CAN_F0BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F0BANK2_FBIT9                                           CAN_F0BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F0BANK2_FBIT10_Pos                                      (10U)
#define CAN_F0BANK2_FBIT10_Msk                                      (0x1UL << CAN_F0BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F0BANK2_FBIT10                                          CAN_F0BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F0BANK2_FBIT11_Pos                                      (11U)
#define CAN_F0BANK2_FBIT11_Msk                                      (0x1UL << CAN_F0BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F0BANK2_FBIT11                                          CAN_F0BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F0BANK2_FBIT12_Pos                                      (12U)
#define CAN_F0BANK2_FBIT12_Msk                                      (0x1UL << CAN_F0BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F0BANK2_FBIT12                                          CAN_F0BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F0BANK2_FBIT13_Pos                                      (13U)
#define CAN_F0BANK2_FBIT13_Msk                                      (0x1UL << CAN_F0BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F0BANK2_FBIT13                                          CAN_F0BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F0BANK2_FBIT14_Pos                                      (14U)
#define CAN_F0BANK2_FBIT14_Msk                                      (0x1UL << CAN_F0BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F0BANK2_FBIT14                                          CAN_F0BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F0BANK2_FBIT15_Pos                                      (15U)
#define CAN_F0BANK2_FBIT15_Msk                                      (0x1UL << CAN_F0BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F0BANK2_FBIT15                                          CAN_F0BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F0BANK2_FBIT16_Pos                                      (16U)
#define CAN_F0BANK2_FBIT16_Msk                                      (0x1UL << CAN_F0BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F0BANK2_FBIT16                                          CAN_F0BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F0BANK2_FBIT17_Pos                                      (17U)
#define CAN_F0BANK2_FBIT17_Msk                                      (0x1UL << CAN_F0BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F0BANK2_FBIT17                                          CAN_F0BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F0BANK2_FBIT18_Pos                                      (18U)
#define CAN_F0BANK2_FBIT18_Msk                                      (0x1UL << CAN_F0BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F0BANK2_FBIT18                                          CAN_F0BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F0BANK2_FBIT19_Pos                                      (19U)
#define CAN_F0BANK2_FBIT19_Msk                                      (0x1UL << CAN_F0BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F0BANK2_FBIT19                                          CAN_F0BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F0BANK2_FBIT20_Pos                                      (20U)
#define CAN_F0BANK2_FBIT20_Msk                                      (0x1UL << CAN_F0BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F0BANK2_FBIT20                                          CAN_F0BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F0BANK2_FBIT21_Pos                                      (21U)
#define CAN_F0BANK2_FBIT21_Msk                                      (0x1UL << CAN_F0BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F0BANK2_FBIT21                                          CAN_F0BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F0BANK2_FBIT22_Pos                                      (22U)
#define CAN_F0BANK2_FBIT22_Msk                                      (0x1UL << CAN_F0BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F0BANK2_FBIT22                                          CAN_F0BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F0BANK2_FBIT23_Pos                                      (23U)
#define CAN_F0BANK2_FBIT23_Msk                                      (0x1UL << CAN_F0BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F0BANK2_FBIT23                                          CAN_F0BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F0BANK2_FBIT24_Pos                                      (24U)
#define CAN_F0BANK2_FBIT24_Msk                                      (0x1UL << CAN_F0BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F0BANK2_FBIT24                                          CAN_F0BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F0BANK2_FBIT25_Pos                                      (25U)
#define CAN_F0BANK2_FBIT25_Msk                                      (0x1UL << CAN_F0BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F0BANK2_FBIT25                                          CAN_F0BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F0BANK2_FBIT26_Pos                                      (26U)
#define CAN_F0BANK2_FBIT26_Msk                                      (0x1UL << CAN_F0BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F0BANK2_FBIT26                                          CAN_F0BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F0BANK2_FBIT27_Pos                                      (27U)
#define CAN_F0BANK2_FBIT27_Msk                                      (0x1UL << CAN_F0BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F0BANK2_FBIT27                                          CAN_F0BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F0BANK2_FBIT28_Pos                                      (28U)
#define CAN_F0BANK2_FBIT28_Msk                                      (0x1UL << CAN_F0BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F0BANK2_FBIT28                                          CAN_F0BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F0BANK2_FBIT29_Pos                                      (29U)
#define CAN_F0BANK2_FBIT29_Msk                                      (0x1UL << CAN_F0BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F0BANK2_FBIT29                                          CAN_F0BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F0BANK2_FBIT30_Pos                                      (30U)
#define CAN_F0BANK2_FBIT30_Msk                                      (0x1UL << CAN_F0BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F0BANK2_FBIT30                                          CAN_F0BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F0BANK2_FBIT31_Pos                                      (31U)
#define CAN_F0BANK2_FBIT31_Msk                                      (0x1UL << CAN_F0BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F0BANK2_FBIT31                                          CAN_F0BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F1BANK2 register  *******************/
#define CAN_F1BANK2_FBIT0_Pos                                       (0U)
#define CAN_F1BANK2_FBIT0_Msk                                       (0x1UL << CAN_F1BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F1BANK2_FBIT0                                           CAN_F1BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F1BANK2_FBIT1_Pos                                       (1U)
#define CAN_F1BANK2_FBIT1_Msk                                       (0x1UL << CAN_F1BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F1BANK2_FBIT1                                           CAN_F1BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F1BANK2_FBIT2_Pos                                       (2U)
#define CAN_F1BANK2_FBIT2_Msk                                       (0x1UL << CAN_F1BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F1BANK2_FBIT2                                           CAN_F1BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F1BANK2_FBIT3_Pos                                       (3U)
#define CAN_F1BANK2_FBIT3_Msk                                       (0x1UL << CAN_F1BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F1BANK2_FBIT3                                           CAN_F1BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F1BANK2_FBIT4_Pos                                       (4U)
#define CAN_F1BANK2_FBIT4_Msk                                       (0x1UL << CAN_F1BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F1BANK2_FBIT4                                           CAN_F1BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F1BANK2_FBIT5_Pos                                       (5U)
#define CAN_F1BANK2_FBIT5_Msk                                       (0x1UL << CAN_F1BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F1BANK2_FBIT5                                           CAN_F1BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F1BANK2_FBIT6_Pos                                       (6U)
#define CAN_F1BANK2_FBIT6_Msk                                       (0x1UL << CAN_F1BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F1BANK2_FBIT6                                           CAN_F1BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F1BANK2_FBIT7_Pos                                       (7U)
#define CAN_F1BANK2_FBIT7_Msk                                       (0x1UL << CAN_F1BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F1BANK2_FBIT7                                           CAN_F1BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F1BANK2_FBIT8_Pos                                       (8U)
#define CAN_F1BANK2_FBIT8_Msk                                       (0x1UL << CAN_F1BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F1BANK2_FBIT8                                           CAN_F1BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F1BANK2_FBIT9_Pos                                       (9U)
#define CAN_F1BANK2_FBIT9_Msk                                       (0x1UL << CAN_F1BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F1BANK2_FBIT9                                           CAN_F1BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F1BANK2_FBIT10_Pos                                      (10U)
#define CAN_F1BANK2_FBIT10_Msk                                      (0x1UL << CAN_F1BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F1BANK2_FBIT10                                          CAN_F1BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F1BANK2_FBIT11_Pos                                      (11U)
#define CAN_F1BANK2_FBIT11_Msk                                      (0x1UL << CAN_F1BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F1BANK2_FBIT11                                          CAN_F1BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F1BANK2_FBIT12_Pos                                      (12U)
#define CAN_F1BANK2_FBIT12_Msk                                      (0x1UL << CAN_F1BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F1BANK2_FBIT12                                          CAN_F1BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F1BANK2_FBIT13_Pos                                      (13U)
#define CAN_F1BANK2_FBIT13_Msk                                      (0x1UL << CAN_F1BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F1BANK2_FBIT13                                          CAN_F1BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F1BANK2_FBIT14_Pos                                      (14U)
#define CAN_F1BANK2_FBIT14_Msk                                      (0x1UL << CAN_F1BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F1BANK2_FBIT14                                          CAN_F1BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F1BANK2_FBIT15_Pos                                      (15U)
#define CAN_F1BANK2_FBIT15_Msk                                      (0x1UL << CAN_F1BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F1BANK2_FBIT15                                          CAN_F1BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F1BANK2_FBIT16_Pos                                      (16U)
#define CAN_F1BANK2_FBIT16_Msk                                      (0x1UL << CAN_F1BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F1BANK2_FBIT16                                          CAN_F1BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F1BANK2_FBIT17_Pos                                      (17U)
#define CAN_F1BANK2_FBIT17_Msk                                      (0x1UL << CAN_F1BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F1BANK2_FBIT17                                          CAN_F1BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F1BANK2_FBIT18_Pos                                      (18U)
#define CAN_F1BANK2_FBIT18_Msk                                      (0x1UL << CAN_F1BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F1BANK2_FBIT18                                          CAN_F1BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F1BANK2_FBIT19_Pos                                      (19U)
#define CAN_F1BANK2_FBIT19_Msk                                      (0x1UL << CAN_F1BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F1BANK2_FBIT19                                          CAN_F1BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F1BANK2_FBIT20_Pos                                      (20U)
#define CAN_F1BANK2_FBIT20_Msk                                      (0x1UL << CAN_F1BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F1BANK2_FBIT20                                          CAN_F1BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F1BANK2_FBIT21_Pos                                      (21U)
#define CAN_F1BANK2_FBIT21_Msk                                      (0x1UL << CAN_F1BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F1BANK2_FBIT21                                          CAN_F1BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F1BANK2_FBIT22_Pos                                      (22U)
#define CAN_F1BANK2_FBIT22_Msk                                      (0x1UL << CAN_F1BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F1BANK2_FBIT22                                          CAN_F1BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F1BANK2_FBIT23_Pos                                      (23U)
#define CAN_F1BANK2_FBIT23_Msk                                      (0x1UL << CAN_F1BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F1BANK2_FBIT23                                          CAN_F1BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F1BANK2_FBIT24_Pos                                      (24U)
#define CAN_F1BANK2_FBIT24_Msk                                      (0x1UL << CAN_F1BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F1BANK2_FBIT24                                          CAN_F1BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F1BANK2_FBIT25_Pos                                      (25U)
#define CAN_F1BANK2_FBIT25_Msk                                      (0x1UL << CAN_F1BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F1BANK2_FBIT25                                          CAN_F1BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F1BANK2_FBIT26_Pos                                      (26U)
#define CAN_F1BANK2_FBIT26_Msk                                      (0x1UL << CAN_F1BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F1BANK2_FBIT26                                          CAN_F1BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F1BANK2_FBIT27_Pos                                      (27U)
#define CAN_F1BANK2_FBIT27_Msk                                      (0x1UL << CAN_F1BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F1BANK2_FBIT27                                          CAN_F1BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F1BANK2_FBIT28_Pos                                      (28U)
#define CAN_F1BANK2_FBIT28_Msk                                      (0x1UL << CAN_F1BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F1BANK2_FBIT28                                          CAN_F1BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F1BANK2_FBIT29_Pos                                      (29U)
#define CAN_F1BANK2_FBIT29_Msk                                      (0x1UL << CAN_F1BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F1BANK2_FBIT29                                          CAN_F1BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F1BANK2_FBIT30_Pos                                      (30U)
#define CAN_F1BANK2_FBIT30_Msk                                      (0x1UL << CAN_F1BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F1BANK2_FBIT30                                          CAN_F1BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F1BANK2_FBIT31_Pos                                      (31U)
#define CAN_F1BANK2_FBIT31_Msk                                      (0x1UL << CAN_F1BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F1BANK2_FBIT31                                          CAN_F1BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F2BANK2 register  *******************/
#define CAN_F2BANK2_FBIT0_Pos                                       (0U)
#define CAN_F2BANK2_FBIT0_Msk                                       (0x1UL << CAN_F2BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F2BANK2_FBIT0                                           CAN_F2BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F2BANK2_FBIT1_Pos                                       (1U)
#define CAN_F2BANK2_FBIT1_Msk                                       (0x1UL << CAN_F2BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F2BANK2_FBIT1                                           CAN_F2BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F2BANK2_FBIT2_Pos                                       (2U)
#define CAN_F2BANK2_FBIT2_Msk                                       (0x1UL << CAN_F2BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F2BANK2_FBIT2                                           CAN_F2BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F2BANK2_FBIT3_Pos                                       (3U)
#define CAN_F2BANK2_FBIT3_Msk                                       (0x1UL << CAN_F2BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F2BANK2_FBIT3                                           CAN_F2BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F2BANK2_FBIT4_Pos                                       (4U)
#define CAN_F2BANK2_FBIT4_Msk                                       (0x1UL << CAN_F2BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F2BANK2_FBIT4                                           CAN_F2BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F2BANK2_FBIT5_Pos                                       (5U)
#define CAN_F2BANK2_FBIT5_Msk                                       (0x1UL << CAN_F2BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F2BANK2_FBIT5                                           CAN_F2BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F2BANK2_FBIT6_Pos                                       (6U)
#define CAN_F2BANK2_FBIT6_Msk                                       (0x1UL << CAN_F2BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F2BANK2_FBIT6                                           CAN_F2BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F2BANK2_FBIT7_Pos                                       (7U)
#define CAN_F2BANK2_FBIT7_Msk                                       (0x1UL << CAN_F2BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F2BANK2_FBIT7                                           CAN_F2BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F2BANK2_FBIT8_Pos                                       (8U)
#define CAN_F2BANK2_FBIT8_Msk                                       (0x1UL << CAN_F2BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F2BANK2_FBIT8                                           CAN_F2BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F2BANK2_FBIT9_Pos                                       (9U)
#define CAN_F2BANK2_FBIT9_Msk                                       (0x1UL << CAN_F2BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F2BANK2_FBIT9                                           CAN_F2BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F2BANK2_FBIT10_Pos                                      (10U)
#define CAN_F2BANK2_FBIT10_Msk                                      (0x1UL << CAN_F2BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F2BANK2_FBIT10                                          CAN_F2BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F2BANK2_FBIT11_Pos                                      (11U)
#define CAN_F2BANK2_FBIT11_Msk                                      (0x1UL << CAN_F2BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F2BANK2_FBIT11                                          CAN_F2BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F2BANK2_FBIT12_Pos                                      (12U)
#define CAN_F2BANK2_FBIT12_Msk                                      (0x1UL << CAN_F2BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F2BANK2_FBIT12                                          CAN_F2BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F2BANK2_FBIT13_Pos                                      (13U)
#define CAN_F2BANK2_FBIT13_Msk                                      (0x1UL << CAN_F2BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F2BANK2_FBIT13                                          CAN_F2BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F2BANK2_FBIT14_Pos                                      (14U)
#define CAN_F2BANK2_FBIT14_Msk                                      (0x1UL << CAN_F2BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F2BANK2_FBIT14                                          CAN_F2BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F2BANK2_FBIT15_Pos                                      (15U)
#define CAN_F2BANK2_FBIT15_Msk                                      (0x1UL << CAN_F2BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F2BANK2_FBIT15                                          CAN_F2BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F2BANK2_FBIT16_Pos                                      (16U)
#define CAN_F2BANK2_FBIT16_Msk                                      (0x1UL << CAN_F2BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F2BANK2_FBIT16                                          CAN_F2BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F2BANK2_FBIT17_Pos                                      (17U)
#define CAN_F2BANK2_FBIT17_Msk                                      (0x1UL << CAN_F2BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F2BANK2_FBIT17                                          CAN_F2BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F2BANK2_FBIT18_Pos                                      (18U)
#define CAN_F2BANK2_FBIT18_Msk                                      (0x1UL << CAN_F2BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F2BANK2_FBIT18                                          CAN_F2BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F2BANK2_FBIT19_Pos                                      (19U)
#define CAN_F2BANK2_FBIT19_Msk                                      (0x1UL << CAN_F2BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F2BANK2_FBIT19                                          CAN_F2BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F2BANK2_FBIT20_Pos                                      (20U)
#define CAN_F2BANK2_FBIT20_Msk                                      (0x1UL << CAN_F2BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F2BANK2_FBIT20                                          CAN_F2BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F2BANK2_FBIT21_Pos                                      (21U)
#define CAN_F2BANK2_FBIT21_Msk                                      (0x1UL << CAN_F2BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F2BANK2_FBIT21                                          CAN_F2BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F2BANK2_FBIT22_Pos                                      (22U)
#define CAN_F2BANK2_FBIT22_Msk                                      (0x1UL << CAN_F2BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F2BANK2_FBIT22                                          CAN_F2BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F2BANK2_FBIT23_Pos                                      (23U)
#define CAN_F2BANK2_FBIT23_Msk                                      (0x1UL << CAN_F2BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F2BANK2_FBIT23                                          CAN_F2BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F2BANK2_FBIT24_Pos                                      (24U)
#define CAN_F2BANK2_FBIT24_Msk                                      (0x1UL << CAN_F2BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F2BANK2_FBIT24                                          CAN_F2BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F2BANK2_FBIT25_Pos                                      (25U)
#define CAN_F2BANK2_FBIT25_Msk                                      (0x1UL << CAN_F2BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F2BANK2_FBIT25                                          CAN_F2BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F2BANK2_FBIT26_Pos                                      (26U)
#define CAN_F2BANK2_FBIT26_Msk                                      (0x1UL << CAN_F2BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F2BANK2_FBIT26                                          CAN_F2BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F2BANK2_FBIT27_Pos                                      (27U)
#define CAN_F2BANK2_FBIT27_Msk                                      (0x1UL << CAN_F2BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F2BANK2_FBIT27                                          CAN_F2BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F2BANK2_FBIT28_Pos                                      (28U)
#define CAN_F2BANK2_FBIT28_Msk                                      (0x1UL << CAN_F2BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F2BANK2_FBIT28                                          CAN_F2BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F2BANK2_FBIT29_Pos                                      (29U)
#define CAN_F2BANK2_FBIT29_Msk                                      (0x1UL << CAN_F2BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F2BANK2_FBIT29                                          CAN_F2BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F2BANK2_FBIT30_Pos                                      (30U)
#define CAN_F2BANK2_FBIT30_Msk                                      (0x1UL << CAN_F2BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F2BANK2_FBIT30                                          CAN_F2BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F2BANK2_FBIT31_Pos                                      (31U)
#define CAN_F2BANK2_FBIT31_Msk                                      (0x1UL << CAN_F2BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F2BANK2_FBIT31                                          CAN_F2BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F3BANK2 register  *******************/
#define CAN_F3BANK2_FBIT0_Pos                                       (0U)
#define CAN_F3BANK2_FBIT0_Msk                                       (0x1UL << CAN_F3BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F3BANK2_FBIT0                                           CAN_F3BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F3BANK2_FBIT1_Pos                                       (1U)
#define CAN_F3BANK2_FBIT1_Msk                                       (0x1UL << CAN_F3BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F3BANK2_FBIT1                                           CAN_F3BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F3BANK2_FBIT2_Pos                                       (2U)
#define CAN_F3BANK2_FBIT2_Msk                                       (0x1UL << CAN_F3BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F3BANK2_FBIT2                                           CAN_F3BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F3BANK2_FBIT3_Pos                                       (3U)
#define CAN_F3BANK2_FBIT3_Msk                                       (0x1UL << CAN_F3BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F3BANK2_FBIT3                                           CAN_F3BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F3BANK2_FBIT4_Pos                                       (4U)
#define CAN_F3BANK2_FBIT4_Msk                                       (0x1UL << CAN_F3BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F3BANK2_FBIT4                                           CAN_F3BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F3BANK2_FBIT5_Pos                                       (5U)
#define CAN_F3BANK2_FBIT5_Msk                                       (0x1UL << CAN_F3BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F3BANK2_FBIT5                                           CAN_F3BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F3BANK2_FBIT6_Pos                                       (6U)
#define CAN_F3BANK2_FBIT6_Msk                                       (0x1UL << CAN_F3BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F3BANK2_FBIT6                                           CAN_F3BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F3BANK2_FBIT7_Pos                                       (7U)
#define CAN_F3BANK2_FBIT7_Msk                                       (0x1UL << CAN_F3BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F3BANK2_FBIT7                                           CAN_F3BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F3BANK2_FBIT8_Pos                                       (8U)
#define CAN_F3BANK2_FBIT8_Msk                                       (0x1UL << CAN_F3BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F3BANK2_FBIT8                                           CAN_F3BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F3BANK2_FBIT9_Pos                                       (9U)
#define CAN_F3BANK2_FBIT9_Msk                                       (0x1UL << CAN_F3BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F3BANK2_FBIT9                                           CAN_F3BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F3BANK2_FBIT10_Pos                                      (10U)
#define CAN_F3BANK2_FBIT10_Msk                                      (0x1UL << CAN_F3BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F3BANK2_FBIT10                                          CAN_F3BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F3BANK2_FBIT11_Pos                                      (11U)
#define CAN_F3BANK2_FBIT11_Msk                                      (0x1UL << CAN_F3BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F3BANK2_FBIT11                                          CAN_F3BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F3BANK2_FBIT12_Pos                                      (12U)
#define CAN_F3BANK2_FBIT12_Msk                                      (0x1UL << CAN_F3BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F3BANK2_FBIT12                                          CAN_F3BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F3BANK2_FBIT13_Pos                                      (13U)
#define CAN_F3BANK2_FBIT13_Msk                                      (0x1UL << CAN_F3BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F3BANK2_FBIT13                                          CAN_F3BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F3BANK2_FBIT14_Pos                                      (14U)
#define CAN_F3BANK2_FBIT14_Msk                                      (0x1UL << CAN_F3BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F3BANK2_FBIT14                                          CAN_F3BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F3BANK2_FBIT15_Pos                                      (15U)
#define CAN_F3BANK2_FBIT15_Msk                                      (0x1UL << CAN_F3BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F3BANK2_FBIT15                                          CAN_F3BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F3BANK2_FBIT16_Pos                                      (16U)
#define CAN_F3BANK2_FBIT16_Msk                                      (0x1UL << CAN_F3BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F3BANK2_FBIT16                                          CAN_F3BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F3BANK2_FBIT17_Pos                                      (17U)
#define CAN_F3BANK2_FBIT17_Msk                                      (0x1UL << CAN_F3BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F3BANK2_FBIT17                                          CAN_F3BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F3BANK2_FBIT18_Pos                                      (18U)
#define CAN_F3BANK2_FBIT18_Msk                                      (0x1UL << CAN_F3BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F3BANK2_FBIT18                                          CAN_F3BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F3BANK2_FBIT19_Pos                                      (19U)
#define CAN_F3BANK2_FBIT19_Msk                                      (0x1UL << CAN_F3BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F3BANK2_FBIT19                                          CAN_F3BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F3BANK2_FBIT20_Pos                                      (20U)
#define CAN_F3BANK2_FBIT20_Msk                                      (0x1UL << CAN_F3BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F3BANK2_FBIT20                                          CAN_F3BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F3BANK2_FBIT21_Pos                                      (21U)
#define CAN_F3BANK2_FBIT21_Msk                                      (0x1UL << CAN_F3BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F3BANK2_FBIT21                                          CAN_F3BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F3BANK2_FBIT22_Pos                                      (22U)
#define CAN_F3BANK2_FBIT22_Msk                                      (0x1UL << CAN_F3BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F3BANK2_FBIT22                                          CAN_F3BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F3BANK2_FBIT23_Pos                                      (23U)
#define CAN_F3BANK2_FBIT23_Msk                                      (0x1UL << CAN_F3BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F3BANK2_FBIT23                                          CAN_F3BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F3BANK2_FBIT24_Pos                                      (24U)
#define CAN_F3BANK2_FBIT24_Msk                                      (0x1UL << CAN_F3BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F3BANK2_FBIT24                                          CAN_F3BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F3BANK2_FBIT25_Pos                                      (25U)
#define CAN_F3BANK2_FBIT25_Msk                                      (0x1UL << CAN_F3BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F3BANK2_FBIT25                                          CAN_F3BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F3BANK2_FBIT26_Pos                                      (26U)
#define CAN_F3BANK2_FBIT26_Msk                                      (0x1UL << CAN_F3BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F3BANK2_FBIT26                                          CAN_F3BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F3BANK2_FBIT27_Pos                                      (27U)
#define CAN_F3BANK2_FBIT27_Msk                                      (0x1UL << CAN_F3BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F3BANK2_FBIT27                                          CAN_F3BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F3BANK2_FBIT28_Pos                                      (28U)
#define CAN_F3BANK2_FBIT28_Msk                                      (0x1UL << CAN_F3BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F3BANK2_FBIT28                                          CAN_F3BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F3BANK2_FBIT29_Pos                                      (29U)
#define CAN_F3BANK2_FBIT29_Msk                                      (0x1UL << CAN_F3BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F3BANK2_FBIT29                                          CAN_F3BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F3BANK2_FBIT30_Pos                                      (30U)
#define CAN_F3BANK2_FBIT30_Msk                                      (0x1UL << CAN_F3BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F3BANK2_FBIT30                                          CAN_F3BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F3BANK2_FBIT31_Pos                                      (31U)
#define CAN_F3BANK2_FBIT31_Msk                                      (0x1UL << CAN_F3BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F3BANK2_FBIT31                                          CAN_F3BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F4BANK2 register  *******************/
#define CAN_F4BANK2_FBIT0_Pos                                       (0U)
#define CAN_F4BANK2_FBIT0_Msk                                       (0x1UL << CAN_F4BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F4BANK2_FBIT0                                           CAN_F4BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F4BANK2_FBIT1_Pos                                       (1U)
#define CAN_F4BANK2_FBIT1_Msk                                       (0x1UL << CAN_F4BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F4BANK2_FBIT1                                           CAN_F4BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F4BANK2_FBIT2_Pos                                       (2U)
#define CAN_F4BANK2_FBIT2_Msk                                       (0x1UL << CAN_F4BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F4BANK2_FBIT2                                           CAN_F4BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F4BANK2_FBIT3_Pos                                       (3U)
#define CAN_F4BANK2_FBIT3_Msk                                       (0x1UL << CAN_F4BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F4BANK2_FBIT3                                           CAN_F4BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F4BANK2_FBIT4_Pos                                       (4U)
#define CAN_F4BANK2_FBIT4_Msk                                       (0x1UL << CAN_F4BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F4BANK2_FBIT4                                           CAN_F4BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F4BANK2_FBIT5_Pos                                       (5U)
#define CAN_F4BANK2_FBIT5_Msk                                       (0x1UL << CAN_F4BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F4BANK2_FBIT5                                           CAN_F4BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F4BANK2_FBIT6_Pos                                       (6U)
#define CAN_F4BANK2_FBIT6_Msk                                       (0x1UL << CAN_F4BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F4BANK2_FBIT6                                           CAN_F4BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F4BANK2_FBIT7_Pos                                       (7U)
#define CAN_F4BANK2_FBIT7_Msk                                       (0x1UL << CAN_F4BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F4BANK2_FBIT7                                           CAN_F4BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F4BANK2_FBIT8_Pos                                       (8U)
#define CAN_F4BANK2_FBIT8_Msk                                       (0x1UL << CAN_F4BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F4BANK2_FBIT8                                           CAN_F4BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F4BANK2_FBIT9_Pos                                       (9U)
#define CAN_F4BANK2_FBIT9_Msk                                       (0x1UL << CAN_F4BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F4BANK2_FBIT9                                           CAN_F4BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F4BANK2_FBIT10_Pos                                      (10U)
#define CAN_F4BANK2_FBIT10_Msk                                      (0x1UL << CAN_F4BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F4BANK2_FBIT10                                          CAN_F4BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F4BANK2_FBIT11_Pos                                      (11U)
#define CAN_F4BANK2_FBIT11_Msk                                      (0x1UL << CAN_F4BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F4BANK2_FBIT11                                          CAN_F4BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F4BANK2_FBIT12_Pos                                      (12U)
#define CAN_F4BANK2_FBIT12_Msk                                      (0x1UL << CAN_F4BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F4BANK2_FBIT12                                          CAN_F4BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F4BANK2_FBIT13_Pos                                      (13U)
#define CAN_F4BANK2_FBIT13_Msk                                      (0x1UL << CAN_F4BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F4BANK2_FBIT13                                          CAN_F4BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F4BANK2_FBIT14_Pos                                      (14U)
#define CAN_F4BANK2_FBIT14_Msk                                      (0x1UL << CAN_F4BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F4BANK2_FBIT14                                          CAN_F4BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F4BANK2_FBIT15_Pos                                      (15U)
#define CAN_F4BANK2_FBIT15_Msk                                      (0x1UL << CAN_F4BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F4BANK2_FBIT15                                          CAN_F4BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F4BANK2_FBIT16_Pos                                      (16U)
#define CAN_F4BANK2_FBIT16_Msk                                      (0x1UL << CAN_F4BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F4BANK2_FBIT16                                          CAN_F4BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F4BANK2_FBIT17_Pos                                      (17U)
#define CAN_F4BANK2_FBIT17_Msk                                      (0x1UL << CAN_F4BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F4BANK2_FBIT17                                          CAN_F4BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F4BANK2_FBIT18_Pos                                      (18U)
#define CAN_F4BANK2_FBIT18_Msk                                      (0x1UL << CAN_F4BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F4BANK2_FBIT18                                          CAN_F4BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F4BANK2_FBIT19_Pos                                      (19U)
#define CAN_F4BANK2_FBIT19_Msk                                      (0x1UL << CAN_F4BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F4BANK2_FBIT19                                          CAN_F4BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F4BANK2_FBIT20_Pos                                      (20U)
#define CAN_F4BANK2_FBIT20_Msk                                      (0x1UL << CAN_F4BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F4BANK2_FBIT20                                          CAN_F4BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F4BANK2_FBIT21_Pos                                      (21U)
#define CAN_F4BANK2_FBIT21_Msk                                      (0x1UL << CAN_F4BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F4BANK2_FBIT21                                          CAN_F4BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F4BANK2_FBIT22_Pos                                      (22U)
#define CAN_F4BANK2_FBIT22_Msk                                      (0x1UL << CAN_F4BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F4BANK2_FBIT22                                          CAN_F4BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F4BANK2_FBIT23_Pos                                      (23U)
#define CAN_F4BANK2_FBIT23_Msk                                      (0x1UL << CAN_F4BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F4BANK2_FBIT23                                          CAN_F4BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F4BANK2_FBIT24_Pos                                      (24U)
#define CAN_F4BANK2_FBIT24_Msk                                      (0x1UL << CAN_F4BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F4BANK2_FBIT24                                          CAN_F4BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F4BANK2_FBIT25_Pos                                      (25U)
#define CAN_F4BANK2_FBIT25_Msk                                      (0x1UL << CAN_F4BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F4BANK2_FBIT25                                          CAN_F4BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F4BANK2_FBIT26_Pos                                      (26U)
#define CAN_F4BANK2_FBIT26_Msk                                      (0x1UL << CAN_F4BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F4BANK2_FBIT26                                          CAN_F4BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F4BANK2_FBIT27_Pos                                      (27U)
#define CAN_F4BANK2_FBIT27_Msk                                      (0x1UL << CAN_F4BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F4BANK2_FBIT27                                          CAN_F4BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F4BANK2_FBIT28_Pos                                      (28U)
#define CAN_F4BANK2_FBIT28_Msk                                      (0x1UL << CAN_F4BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F4BANK2_FBIT28                                          CAN_F4BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F4BANK2_FBIT29_Pos                                      (29U)
#define CAN_F4BANK2_FBIT29_Msk                                      (0x1UL << CAN_F4BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F4BANK2_FBIT29                                          CAN_F4BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F4BANK2_FBIT30_Pos                                      (30U)
#define CAN_F4BANK2_FBIT30_Msk                                      (0x1UL << CAN_F4BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F4BANK2_FBIT30                                          CAN_F4BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F4BANK2_FBIT31_Pos                                      (31U)
#define CAN_F4BANK2_FBIT31_Msk                                      (0x1UL << CAN_F4BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F4BANK2_FBIT31                                          CAN_F4BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F5BANK2 register  *******************/
#define CAN_F5BANK2_FBIT0_Pos                                       (0U)
#define CAN_F5BANK2_FBIT0_Msk                                       (0x1UL << CAN_F5BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F5BANK2_FBIT0                                           CAN_F5BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F5BANK2_FBIT1_Pos                                       (1U)
#define CAN_F5BANK2_FBIT1_Msk                                       (0x1UL << CAN_F5BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F5BANK2_FBIT1                                           CAN_F5BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F5BANK2_FBIT2_Pos                                       (2U)
#define CAN_F5BANK2_FBIT2_Msk                                       (0x1UL << CAN_F5BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F5BANK2_FBIT2                                           CAN_F5BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F5BANK2_FBIT3_Pos                                       (3U)
#define CAN_F5BANK2_FBIT3_Msk                                       (0x1UL << CAN_F5BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F5BANK2_FBIT3                                           CAN_F5BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F5BANK2_FBIT4_Pos                                       (4U)
#define CAN_F5BANK2_FBIT4_Msk                                       (0x1UL << CAN_F5BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F5BANK2_FBIT4                                           CAN_F5BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F5BANK2_FBIT5_Pos                                       (5U)
#define CAN_F5BANK2_FBIT5_Msk                                       (0x1UL << CAN_F5BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F5BANK2_FBIT5                                           CAN_F5BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F5BANK2_FBIT6_Pos                                       (6U)
#define CAN_F5BANK2_FBIT6_Msk                                       (0x1UL << CAN_F5BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F5BANK2_FBIT6                                           CAN_F5BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F5BANK2_FBIT7_Pos                                       (7U)
#define CAN_F5BANK2_FBIT7_Msk                                       (0x1UL << CAN_F5BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F5BANK2_FBIT7                                           CAN_F5BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F5BANK2_FBIT8_Pos                                       (8U)
#define CAN_F5BANK2_FBIT8_Msk                                       (0x1UL << CAN_F5BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F5BANK2_FBIT8                                           CAN_F5BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F5BANK2_FBIT9_Pos                                       (9U)
#define CAN_F5BANK2_FBIT9_Msk                                       (0x1UL << CAN_F5BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F5BANK2_FBIT9                                           CAN_F5BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F5BANK2_FBIT10_Pos                                      (10U)
#define CAN_F5BANK2_FBIT10_Msk                                      (0x1UL << CAN_F5BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F5BANK2_FBIT10                                          CAN_F5BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F5BANK2_FBIT11_Pos                                      (11U)
#define CAN_F5BANK2_FBIT11_Msk                                      (0x1UL << CAN_F5BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F5BANK2_FBIT11                                          CAN_F5BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F5BANK2_FBIT12_Pos                                      (12U)
#define CAN_F5BANK2_FBIT12_Msk                                      (0x1UL << CAN_F5BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F5BANK2_FBIT12                                          CAN_F5BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F5BANK2_FBIT13_Pos                                      (13U)
#define CAN_F5BANK2_FBIT13_Msk                                      (0x1UL << CAN_F5BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F5BANK2_FBIT13                                          CAN_F5BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F5BANK2_FBIT14_Pos                                      (14U)
#define CAN_F5BANK2_FBIT14_Msk                                      (0x1UL << CAN_F5BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F5BANK2_FBIT14                                          CAN_F5BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F5BANK2_FBIT15_Pos                                      (15U)
#define CAN_F5BANK2_FBIT15_Msk                                      (0x1UL << CAN_F5BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F5BANK2_FBIT15                                          CAN_F5BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F5BANK2_FBIT16_Pos                                      (16U)
#define CAN_F5BANK2_FBIT16_Msk                                      (0x1UL << CAN_F5BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F5BANK2_FBIT16                                          CAN_F5BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F5BANK2_FBIT17_Pos                                      (17U)
#define CAN_F5BANK2_FBIT17_Msk                                      (0x1UL << CAN_F5BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F5BANK2_FBIT17                                          CAN_F5BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F5BANK2_FBIT18_Pos                                      (18U)
#define CAN_F5BANK2_FBIT18_Msk                                      (0x1UL << CAN_F5BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F5BANK2_FBIT18                                          CAN_F5BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F5BANK2_FBIT19_Pos                                      (19U)
#define CAN_F5BANK2_FBIT19_Msk                                      (0x1UL << CAN_F5BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F5BANK2_FBIT19                                          CAN_F5BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F5BANK2_FBIT20_Pos                                      (20U)
#define CAN_F5BANK2_FBIT20_Msk                                      (0x1UL << CAN_F5BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F5BANK2_FBIT20                                          CAN_F5BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F5BANK2_FBIT21_Pos                                      (21U)
#define CAN_F5BANK2_FBIT21_Msk                                      (0x1UL << CAN_F5BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F5BANK2_FBIT21                                          CAN_F5BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F5BANK2_FBIT22_Pos                                      (22U)
#define CAN_F5BANK2_FBIT22_Msk                                      (0x1UL << CAN_F5BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F5BANK2_FBIT22                                          CAN_F5BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F5BANK2_FBIT23_Pos                                      (23U)
#define CAN_F5BANK2_FBIT23_Msk                                      (0x1UL << CAN_F5BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F5BANK2_FBIT23                                          CAN_F5BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F5BANK2_FBIT24_Pos                                      (24U)
#define CAN_F5BANK2_FBIT24_Msk                                      (0x1UL << CAN_F5BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F5BANK2_FBIT24                                          CAN_F5BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F5BANK2_FBIT25_Pos                                      (25U)
#define CAN_F5BANK2_FBIT25_Msk                                      (0x1UL << CAN_F5BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F5BANK2_FBIT25                                          CAN_F5BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F5BANK2_FBIT26_Pos                                      (26U)
#define CAN_F5BANK2_FBIT26_Msk                                      (0x1UL << CAN_F5BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F5BANK2_FBIT26                                          CAN_F5BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F5BANK2_FBIT27_Pos                                      (27U)
#define CAN_F5BANK2_FBIT27_Msk                                      (0x1UL << CAN_F5BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F5BANK2_FBIT27                                          CAN_F5BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F5BANK2_FBIT28_Pos                                      (28U)
#define CAN_F5BANK2_FBIT28_Msk                                      (0x1UL << CAN_F5BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F5BANK2_FBIT28                                          CAN_F5BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F5BANK2_FBIT29_Pos                                      (29U)
#define CAN_F5BANK2_FBIT29_Msk                                      (0x1UL << CAN_F5BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F5BANK2_FBIT29                                          CAN_F5BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F5BANK2_FBIT30_Pos                                      (30U)
#define CAN_F5BANK2_FBIT30_Msk                                      (0x1UL << CAN_F5BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F5BANK2_FBIT30                                          CAN_F5BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F5BANK2_FBIT31_Pos                                      (31U)
#define CAN_F5BANK2_FBIT31_Msk                                      (0x1UL << CAN_F5BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F5BANK2_FBIT31                                          CAN_F5BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F6BANK2 register  *******************/
#define CAN_F6BANK2_FBIT0_Pos                                       (0U)
#define CAN_F6BANK2_FBIT0_Msk                                       (0x1UL << CAN_F6BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F6BANK2_FBIT0                                           CAN_F6BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F6BANK2_FBIT1_Pos                                       (1U)
#define CAN_F6BANK2_FBIT1_Msk                                       (0x1UL << CAN_F6BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F6BANK2_FBIT1                                           CAN_F6BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F6BANK2_FBIT2_Pos                                       (2U)
#define CAN_F6BANK2_FBIT2_Msk                                       (0x1UL << CAN_F6BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F6BANK2_FBIT2                                           CAN_F6BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F6BANK2_FBIT3_Pos                                       (3U)
#define CAN_F6BANK2_FBIT3_Msk                                       (0x1UL << CAN_F6BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F6BANK2_FBIT3                                           CAN_F6BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F6BANK2_FBIT4_Pos                                       (4U)
#define CAN_F6BANK2_FBIT4_Msk                                       (0x1UL << CAN_F6BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F6BANK2_FBIT4                                           CAN_F6BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F6BANK2_FBIT5_Pos                                       (5U)
#define CAN_F6BANK2_FBIT5_Msk                                       (0x1UL << CAN_F6BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F6BANK2_FBIT5                                           CAN_F6BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F6BANK2_FBIT6_Pos                                       (6U)
#define CAN_F6BANK2_FBIT6_Msk                                       (0x1UL << CAN_F6BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F6BANK2_FBIT6                                           CAN_F6BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F6BANK2_FBIT7_Pos                                       (7U)
#define CAN_F6BANK2_FBIT7_Msk                                       (0x1UL << CAN_F6BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F6BANK2_FBIT7                                           CAN_F6BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F6BANK2_FBIT8_Pos                                       (8U)
#define CAN_F6BANK2_FBIT8_Msk                                       (0x1UL << CAN_F6BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F6BANK2_FBIT8                                           CAN_F6BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F6BANK2_FBIT9_Pos                                       (9U)
#define CAN_F6BANK2_FBIT9_Msk                                       (0x1UL << CAN_F6BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F6BANK2_FBIT9                                           CAN_F6BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F6BANK2_FBIT10_Pos                                      (10U)
#define CAN_F6BANK2_FBIT10_Msk                                      (0x1UL << CAN_F6BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F6BANK2_FBIT10                                          CAN_F6BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F6BANK2_FBIT11_Pos                                      (11U)
#define CAN_F6BANK2_FBIT11_Msk                                      (0x1UL << CAN_F6BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F6BANK2_FBIT11                                          CAN_F6BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F6BANK2_FBIT12_Pos                                      (12U)
#define CAN_F6BANK2_FBIT12_Msk                                      (0x1UL << CAN_F6BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F6BANK2_FBIT12                                          CAN_F6BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F6BANK2_FBIT13_Pos                                      (13U)
#define CAN_F6BANK2_FBIT13_Msk                                      (0x1UL << CAN_F6BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F6BANK2_FBIT13                                          CAN_F6BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F6BANK2_FBIT14_Pos                                      (14U)
#define CAN_F6BANK2_FBIT14_Msk                                      (0x1UL << CAN_F6BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F6BANK2_FBIT14                                          CAN_F6BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F6BANK2_FBIT15_Pos                                      (15U)
#define CAN_F6BANK2_FBIT15_Msk                                      (0x1UL << CAN_F6BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F6BANK2_FBIT15                                          CAN_F6BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F6BANK2_FBIT16_Pos                                      (16U)
#define CAN_F6BANK2_FBIT16_Msk                                      (0x1UL << CAN_F6BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F6BANK2_FBIT16                                          CAN_F6BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F6BANK2_FBIT17_Pos                                      (17U)
#define CAN_F6BANK2_FBIT17_Msk                                      (0x1UL << CAN_F6BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F6BANK2_FBIT17                                          CAN_F6BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F6BANK2_FBIT18_Pos                                      (18U)
#define CAN_F6BANK2_FBIT18_Msk                                      (0x1UL << CAN_F6BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F6BANK2_FBIT18                                          CAN_F6BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F6BANK2_FBIT19_Pos                                      (19U)
#define CAN_F6BANK2_FBIT19_Msk                                      (0x1UL << CAN_F6BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F6BANK2_FBIT19                                          CAN_F6BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F6BANK2_FBIT20_Pos                                      (20U)
#define CAN_F6BANK2_FBIT20_Msk                                      (0x1UL << CAN_F6BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F6BANK2_FBIT20                                          CAN_F6BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F6BANK2_FBIT21_Pos                                      (21U)
#define CAN_F6BANK2_FBIT21_Msk                                      (0x1UL << CAN_F6BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F6BANK2_FBIT21                                          CAN_F6BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F6BANK2_FBIT22_Pos                                      (22U)
#define CAN_F6BANK2_FBIT22_Msk                                      (0x1UL << CAN_F6BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F6BANK2_FBIT22                                          CAN_F6BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F6BANK2_FBIT23_Pos                                      (23U)
#define CAN_F6BANK2_FBIT23_Msk                                      (0x1UL << CAN_F6BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F6BANK2_FBIT23                                          CAN_F6BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F6BANK2_FBIT24_Pos                                      (24U)
#define CAN_F6BANK2_FBIT24_Msk                                      (0x1UL << CAN_F6BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F6BANK2_FBIT24                                          CAN_F6BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F6BANK2_FBIT25_Pos                                      (25U)
#define CAN_F6BANK2_FBIT25_Msk                                      (0x1UL << CAN_F6BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F6BANK2_FBIT25                                          CAN_F6BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F6BANK2_FBIT26_Pos                                      (26U)
#define CAN_F6BANK2_FBIT26_Msk                                      (0x1UL << CAN_F6BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F6BANK2_FBIT26                                          CAN_F6BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F6BANK2_FBIT27_Pos                                      (27U)
#define CAN_F6BANK2_FBIT27_Msk                                      (0x1UL << CAN_F6BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F6BANK2_FBIT27                                          CAN_F6BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F6BANK2_FBIT28_Pos                                      (28U)
#define CAN_F6BANK2_FBIT28_Msk                                      (0x1UL << CAN_F6BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F6BANK2_FBIT28                                          CAN_F6BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F6BANK2_FBIT29_Pos                                      (29U)
#define CAN_F6BANK2_FBIT29_Msk                                      (0x1UL << CAN_F6BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F6BANK2_FBIT29                                          CAN_F6BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F6BANK2_FBIT30_Pos                                      (30U)
#define CAN_F6BANK2_FBIT30_Msk                                      (0x1UL << CAN_F6BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F6BANK2_FBIT30                                          CAN_F6BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F6BANK2_FBIT31_Pos                                      (31U)
#define CAN_F6BANK2_FBIT31_Msk                                      (0x1UL << CAN_F6BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F6BANK2_FBIT31                                          CAN_F6BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F7BANK2 register  *******************/
#define CAN_F7BANK2_FBIT0_Pos                                       (0U)
#define CAN_F7BANK2_FBIT0_Msk                                       (0x1UL << CAN_F7BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F7BANK2_FBIT0                                           CAN_F7BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F7BANK2_FBIT1_Pos                                       (1U)
#define CAN_F7BANK2_FBIT1_Msk                                       (0x1UL << CAN_F7BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F7BANK2_FBIT1                                           CAN_F7BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F7BANK2_FBIT2_Pos                                       (2U)
#define CAN_F7BANK2_FBIT2_Msk                                       (0x1UL << CAN_F7BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F7BANK2_FBIT2                                           CAN_F7BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F7BANK2_FBIT3_Pos                                       (3U)
#define CAN_F7BANK2_FBIT3_Msk                                       (0x1UL << CAN_F7BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F7BANK2_FBIT3                                           CAN_F7BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F7BANK2_FBIT4_Pos                                       (4U)
#define CAN_F7BANK2_FBIT4_Msk                                       (0x1UL << CAN_F7BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F7BANK2_FBIT4                                           CAN_F7BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F7BANK2_FBIT5_Pos                                       (5U)
#define CAN_F7BANK2_FBIT5_Msk                                       (0x1UL << CAN_F7BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F7BANK2_FBIT5                                           CAN_F7BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F7BANK2_FBIT6_Pos                                       (6U)
#define CAN_F7BANK2_FBIT6_Msk                                       (0x1UL << CAN_F7BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F7BANK2_FBIT6                                           CAN_F7BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F7BANK2_FBIT7_Pos                                       (7U)
#define CAN_F7BANK2_FBIT7_Msk                                       (0x1UL << CAN_F7BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F7BANK2_FBIT7                                           CAN_F7BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F7BANK2_FBIT8_Pos                                       (8U)
#define CAN_F7BANK2_FBIT8_Msk                                       (0x1UL << CAN_F7BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F7BANK2_FBIT8                                           CAN_F7BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F7BANK2_FBIT9_Pos                                       (9U)
#define CAN_F7BANK2_FBIT9_Msk                                       (0x1UL << CAN_F7BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F7BANK2_FBIT9                                           CAN_F7BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F7BANK2_FBIT10_Pos                                      (10U)
#define CAN_F7BANK2_FBIT10_Msk                                      (0x1UL << CAN_F7BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F7BANK2_FBIT10                                          CAN_F7BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F7BANK2_FBIT11_Pos                                      (11U)
#define CAN_F7BANK2_FBIT11_Msk                                      (0x1UL << CAN_F7BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F7BANK2_FBIT11                                          CAN_F7BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F7BANK2_FBIT12_Pos                                      (12U)
#define CAN_F7BANK2_FBIT12_Msk                                      (0x1UL << CAN_F7BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F7BANK2_FBIT12                                          CAN_F7BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F7BANK2_FBIT13_Pos                                      (13U)
#define CAN_F7BANK2_FBIT13_Msk                                      (0x1UL << CAN_F7BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F7BANK2_FBIT13                                          CAN_F7BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F7BANK2_FBIT14_Pos                                      (14U)
#define CAN_F7BANK2_FBIT14_Msk                                      (0x1UL << CAN_F7BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F7BANK2_FBIT14                                          CAN_F7BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F7BANK2_FBIT15_Pos                                      (15U)
#define CAN_F7BANK2_FBIT15_Msk                                      (0x1UL << CAN_F7BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F7BANK2_FBIT15                                          CAN_F7BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F7BANK2_FBIT16_Pos                                      (16U)
#define CAN_F7BANK2_FBIT16_Msk                                      (0x1UL << CAN_F7BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F7BANK2_FBIT16                                          CAN_F7BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F7BANK2_FBIT17_Pos                                      (17U)
#define CAN_F7BANK2_FBIT17_Msk                                      (0x1UL << CAN_F7BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F7BANK2_FBIT17                                          CAN_F7BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F7BANK2_FBIT18_Pos                                      (18U)
#define CAN_F7BANK2_FBIT18_Msk                                      (0x1UL << CAN_F7BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F7BANK2_FBIT18                                          CAN_F7BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F7BANK2_FBIT19_Pos                                      (19U)
#define CAN_F7BANK2_FBIT19_Msk                                      (0x1UL << CAN_F7BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F7BANK2_FBIT19                                          CAN_F7BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F7BANK2_FBIT20_Pos                                      (20U)
#define CAN_F7BANK2_FBIT20_Msk                                      (0x1UL << CAN_F7BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F7BANK2_FBIT20                                          CAN_F7BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F7BANK2_FBIT21_Pos                                      (21U)
#define CAN_F7BANK2_FBIT21_Msk                                      (0x1UL << CAN_F7BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F7BANK2_FBIT21                                          CAN_F7BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F7BANK2_FBIT22_Pos                                      (22U)
#define CAN_F7BANK2_FBIT22_Msk                                      (0x1UL << CAN_F7BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F7BANK2_FBIT22                                          CAN_F7BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F7BANK2_FBIT23_Pos                                      (23U)
#define CAN_F7BANK2_FBIT23_Msk                                      (0x1UL << CAN_F7BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F7BANK2_FBIT23                                          CAN_F7BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F7BANK2_FBIT24_Pos                                      (24U)
#define CAN_F7BANK2_FBIT24_Msk                                      (0x1UL << CAN_F7BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F7BANK2_FBIT24                                          CAN_F7BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F7BANK2_FBIT25_Pos                                      (25U)
#define CAN_F7BANK2_FBIT25_Msk                                      (0x1UL << CAN_F7BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F7BANK2_FBIT25                                          CAN_F7BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F7BANK2_FBIT26_Pos                                      (26U)
#define CAN_F7BANK2_FBIT26_Msk                                      (0x1UL << CAN_F7BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F7BANK2_FBIT26                                          CAN_F7BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F7BANK2_FBIT27_Pos                                      (27U)
#define CAN_F7BANK2_FBIT27_Msk                                      (0x1UL << CAN_F7BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F7BANK2_FBIT27                                          CAN_F7BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F7BANK2_FBIT28_Pos                                      (28U)
#define CAN_F7BANK2_FBIT28_Msk                                      (0x1UL << CAN_F7BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F7BANK2_FBIT28                                          CAN_F7BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F7BANK2_FBIT29_Pos                                      (29U)
#define CAN_F7BANK2_FBIT29_Msk                                      (0x1UL << CAN_F7BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F7BANK2_FBIT29                                          CAN_F7BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F7BANK2_FBIT30_Pos                                      (30U)
#define CAN_F7BANK2_FBIT30_Msk                                      (0x1UL << CAN_F7BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F7BANK2_FBIT30                                          CAN_F7BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F7BANK2_FBIT31_Pos                                      (31U)
#define CAN_F7BANK2_FBIT31_Msk                                      (0x1UL << CAN_F7BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F7BANK2_FBIT31                                          CAN_F7BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F8BANK2 register  *******************/
#define CAN_F8BANK2_FBIT0_Pos                                       (0U)
#define CAN_F8BANK2_FBIT0_Msk                                       (0x1UL << CAN_F8BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F8BANK2_FBIT0                                           CAN_F8BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F8BANK2_FBIT1_Pos                                       (1U)
#define CAN_F8BANK2_FBIT1_Msk                                       (0x1UL << CAN_F8BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F8BANK2_FBIT1                                           CAN_F8BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F8BANK2_FBIT2_Pos                                       (2U)
#define CAN_F8BANK2_FBIT2_Msk                                       (0x1UL << CAN_F8BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F8BANK2_FBIT2                                           CAN_F8BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F8BANK2_FBIT3_Pos                                       (3U)
#define CAN_F8BANK2_FBIT3_Msk                                       (0x1UL << CAN_F8BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F8BANK2_FBIT3                                           CAN_F8BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F8BANK2_FBIT4_Pos                                       (4U)
#define CAN_F8BANK2_FBIT4_Msk                                       (0x1UL << CAN_F8BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F8BANK2_FBIT4                                           CAN_F8BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F8BANK2_FBIT5_Pos                                       (5U)
#define CAN_F8BANK2_FBIT5_Msk                                       (0x1UL << CAN_F8BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F8BANK2_FBIT5                                           CAN_F8BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F8BANK2_FBIT6_Pos                                       (6U)
#define CAN_F8BANK2_FBIT6_Msk                                       (0x1UL << CAN_F8BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F8BANK2_FBIT6                                           CAN_F8BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F8BANK2_FBIT7_Pos                                       (7U)
#define CAN_F8BANK2_FBIT7_Msk                                       (0x1UL << CAN_F8BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F8BANK2_FBIT7                                           CAN_F8BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F8BANK2_FBIT8_Pos                                       (8U)
#define CAN_F8BANK2_FBIT8_Msk                                       (0x1UL << CAN_F8BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F8BANK2_FBIT8                                           CAN_F8BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F8BANK2_FBIT9_Pos                                       (9U)
#define CAN_F8BANK2_FBIT9_Msk                                       (0x1UL << CAN_F8BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F8BANK2_FBIT9                                           CAN_F8BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F8BANK2_FBIT10_Pos                                      (10U)
#define CAN_F8BANK2_FBIT10_Msk                                      (0x1UL << CAN_F8BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F8BANK2_FBIT10                                          CAN_F8BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F8BANK2_FBIT11_Pos                                      (11U)
#define CAN_F8BANK2_FBIT11_Msk                                      (0x1UL << CAN_F8BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F8BANK2_FBIT11                                          CAN_F8BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F8BANK2_FBIT12_Pos                                      (12U)
#define CAN_F8BANK2_FBIT12_Msk                                      (0x1UL << CAN_F8BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F8BANK2_FBIT12                                          CAN_F8BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F8BANK2_FBIT13_Pos                                      (13U)
#define CAN_F8BANK2_FBIT13_Msk                                      (0x1UL << CAN_F8BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F8BANK2_FBIT13                                          CAN_F8BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F8BANK2_FBIT14_Pos                                      (14U)
#define CAN_F8BANK2_FBIT14_Msk                                      (0x1UL << CAN_F8BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F8BANK2_FBIT14                                          CAN_F8BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F8BANK2_FBIT15_Pos                                      (15U)
#define CAN_F8BANK2_FBIT15_Msk                                      (0x1UL << CAN_F8BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F8BANK2_FBIT15                                          CAN_F8BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F8BANK2_FBIT16_Pos                                      (16U)
#define CAN_F8BANK2_FBIT16_Msk                                      (0x1UL << CAN_F8BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F8BANK2_FBIT16                                          CAN_F8BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F8BANK2_FBIT17_Pos                                      (17U)
#define CAN_F8BANK2_FBIT17_Msk                                      (0x1UL << CAN_F8BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F8BANK2_FBIT17                                          CAN_F8BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F8BANK2_FBIT18_Pos                                      (18U)
#define CAN_F8BANK2_FBIT18_Msk                                      (0x1UL << CAN_F8BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F8BANK2_FBIT18                                          CAN_F8BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F8BANK2_FBIT19_Pos                                      (19U)
#define CAN_F8BANK2_FBIT19_Msk                                      (0x1UL << CAN_F8BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F8BANK2_FBIT19                                          CAN_F8BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F8BANK2_FBIT20_Pos                                      (20U)
#define CAN_F8BANK2_FBIT20_Msk                                      (0x1UL << CAN_F8BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F8BANK2_FBIT20                                          CAN_F8BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F8BANK2_FBIT21_Pos                                      (21U)
#define CAN_F8BANK2_FBIT21_Msk                                      (0x1UL << CAN_F8BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F8BANK2_FBIT21                                          CAN_F8BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F8BANK2_FBIT22_Pos                                      (22U)
#define CAN_F8BANK2_FBIT22_Msk                                      (0x1UL << CAN_F8BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F8BANK2_FBIT22                                          CAN_F8BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F8BANK2_FBIT23_Pos                                      (23U)
#define CAN_F8BANK2_FBIT23_Msk                                      (0x1UL << CAN_F8BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F8BANK2_FBIT23                                          CAN_F8BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F8BANK2_FBIT24_Pos                                      (24U)
#define CAN_F8BANK2_FBIT24_Msk                                      (0x1UL << CAN_F8BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F8BANK2_FBIT24                                          CAN_F8BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F8BANK2_FBIT25_Pos                                      (25U)
#define CAN_F8BANK2_FBIT25_Msk                                      (0x1UL << CAN_F8BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F8BANK2_FBIT25                                          CAN_F8BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F8BANK2_FBIT26_Pos                                      (26U)
#define CAN_F8BANK2_FBIT26_Msk                                      (0x1UL << CAN_F8BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F8BANK2_FBIT26                                          CAN_F8BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F8BANK2_FBIT27_Pos                                      (27U)
#define CAN_F8BANK2_FBIT27_Msk                                      (0x1UL << CAN_F8BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F8BANK2_FBIT27                                          CAN_F8BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F8BANK2_FBIT28_Pos                                      (28U)
#define CAN_F8BANK2_FBIT28_Msk                                      (0x1UL << CAN_F8BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F8BANK2_FBIT28                                          CAN_F8BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F8BANK2_FBIT29_Pos                                      (29U)
#define CAN_F8BANK2_FBIT29_Msk                                      (0x1UL << CAN_F8BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F8BANK2_FBIT29                                          CAN_F8BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F8BANK2_FBIT30_Pos                                      (30U)
#define CAN_F8BANK2_FBIT30_Msk                                      (0x1UL << CAN_F8BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F8BANK2_FBIT30                                          CAN_F8BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F8BANK2_FBIT31_Pos                                      (31U)
#define CAN_F8BANK2_FBIT31_Msk                                      (0x1UL << CAN_F8BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F8BANK2_FBIT31                                          CAN_F8BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F9BANK2 register  *******************/
#define CAN_F9BANK2_FBIT0_Pos                                       (0U)
#define CAN_F9BANK2_FBIT0_Msk                                       (0x1UL << CAN_F9BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F9BANK2_FBIT0                                           CAN_F9BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F9BANK2_FBIT1_Pos                                       (1U)
#define CAN_F9BANK2_FBIT1_Msk                                       (0x1UL << CAN_F9BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F9BANK2_FBIT1                                           CAN_F9BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F9BANK2_FBIT2_Pos                                       (2U)
#define CAN_F9BANK2_FBIT2_Msk                                       (0x1UL << CAN_F9BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F9BANK2_FBIT2                                           CAN_F9BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F9BANK2_FBIT3_Pos                                       (3U)
#define CAN_F9BANK2_FBIT3_Msk                                       (0x1UL << CAN_F9BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F9BANK2_FBIT3                                           CAN_F9BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F9BANK2_FBIT4_Pos                                       (4U)
#define CAN_F9BANK2_FBIT4_Msk                                       (0x1UL << CAN_F9BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F9BANK2_FBIT4                                           CAN_F9BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F9BANK2_FBIT5_Pos                                       (5U)
#define CAN_F9BANK2_FBIT5_Msk                                       (0x1UL << CAN_F9BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F9BANK2_FBIT5                                           CAN_F9BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F9BANK2_FBIT6_Pos                                       (6U)
#define CAN_F9BANK2_FBIT6_Msk                                       (0x1UL << CAN_F9BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F9BANK2_FBIT6                                           CAN_F9BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F9BANK2_FBIT7_Pos                                       (7U)
#define CAN_F9BANK2_FBIT7_Msk                                       (0x1UL << CAN_F9BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F9BANK2_FBIT7                                           CAN_F9BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F9BANK2_FBIT8_Pos                                       (8U)
#define CAN_F9BANK2_FBIT8_Msk                                       (0x1UL << CAN_F9BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F9BANK2_FBIT8                                           CAN_F9BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F9BANK2_FBIT9_Pos                                       (9U)
#define CAN_F9BANK2_FBIT9_Msk                                       (0x1UL << CAN_F9BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F9BANK2_FBIT9                                           CAN_F9BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F9BANK2_FBIT10_Pos                                      (10U)
#define CAN_F9BANK2_FBIT10_Msk                                      (0x1UL << CAN_F9BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F9BANK2_FBIT10                                          CAN_F9BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F9BANK2_FBIT11_Pos                                      (11U)
#define CAN_F9BANK2_FBIT11_Msk                                      (0x1UL << CAN_F9BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F9BANK2_FBIT11                                          CAN_F9BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F9BANK2_FBIT12_Pos                                      (12U)
#define CAN_F9BANK2_FBIT12_Msk                                      (0x1UL << CAN_F9BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F9BANK2_FBIT12                                          CAN_F9BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F9BANK2_FBIT13_Pos                                      (13U)
#define CAN_F9BANK2_FBIT13_Msk                                      (0x1UL << CAN_F9BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F9BANK2_FBIT13                                          CAN_F9BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F9BANK2_FBIT14_Pos                                      (14U)
#define CAN_F9BANK2_FBIT14_Msk                                      (0x1UL << CAN_F9BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F9BANK2_FBIT14                                          CAN_F9BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F9BANK2_FBIT15_Pos                                      (15U)
#define CAN_F9BANK2_FBIT15_Msk                                      (0x1UL << CAN_F9BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F9BANK2_FBIT15                                          CAN_F9BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F9BANK2_FBIT16_Pos                                      (16U)
#define CAN_F9BANK2_FBIT16_Msk                                      (0x1UL << CAN_F9BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F9BANK2_FBIT16                                          CAN_F9BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F9BANK2_FBIT17_Pos                                      (17U)
#define CAN_F9BANK2_FBIT17_Msk                                      (0x1UL << CAN_F9BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F9BANK2_FBIT17                                          CAN_F9BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F9BANK2_FBIT18_Pos                                      (18U)
#define CAN_F9BANK2_FBIT18_Msk                                      (0x1UL << CAN_F9BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F9BANK2_FBIT18                                          CAN_F9BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F9BANK2_FBIT19_Pos                                      (19U)
#define CAN_F9BANK2_FBIT19_Msk                                      (0x1UL << CAN_F9BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F9BANK2_FBIT19                                          CAN_F9BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F9BANK2_FBIT20_Pos                                      (20U)
#define CAN_F9BANK2_FBIT20_Msk                                      (0x1UL << CAN_F9BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F9BANK2_FBIT20                                          CAN_F9BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F9BANK2_FBIT21_Pos                                      (21U)
#define CAN_F9BANK2_FBIT21_Msk                                      (0x1UL << CAN_F9BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F9BANK2_FBIT21                                          CAN_F9BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F9BANK2_FBIT22_Pos                                      (22U)
#define CAN_F9BANK2_FBIT22_Msk                                      (0x1UL << CAN_F9BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F9BANK2_FBIT22                                          CAN_F9BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F9BANK2_FBIT23_Pos                                      (23U)
#define CAN_F9BANK2_FBIT23_Msk                                      (0x1UL << CAN_F9BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F9BANK2_FBIT23                                          CAN_F9BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F9BANK2_FBIT24_Pos                                      (24U)
#define CAN_F9BANK2_FBIT24_Msk                                      (0x1UL << CAN_F9BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F9BANK2_FBIT24                                          CAN_F9BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F9BANK2_FBIT25_Pos                                      (25U)
#define CAN_F9BANK2_FBIT25_Msk                                      (0x1UL << CAN_F9BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F9BANK2_FBIT25                                          CAN_F9BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F9BANK2_FBIT26_Pos                                      (26U)
#define CAN_F9BANK2_FBIT26_Msk                                      (0x1UL << CAN_F9BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F9BANK2_FBIT26                                          CAN_F9BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F9BANK2_FBIT27_Pos                                      (27U)
#define CAN_F9BANK2_FBIT27_Msk                                      (0x1UL << CAN_F9BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F9BANK2_FBIT27                                          CAN_F9BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F9BANK2_FBIT28_Pos                                      (28U)
#define CAN_F9BANK2_FBIT28_Msk                                      (0x1UL << CAN_F9BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F9BANK2_FBIT28                                          CAN_F9BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F9BANK2_FBIT29_Pos                                      (29U)
#define CAN_F9BANK2_FBIT29_Msk                                      (0x1UL << CAN_F9BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F9BANK2_FBIT29                                          CAN_F9BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F9BANK2_FBIT30_Pos                                      (30U)
#define CAN_F9BANK2_FBIT30_Msk                                      (0x1UL << CAN_F9BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F9BANK2_FBIT30                                          CAN_F9BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F9BANK2_FBIT31_Pos                                      (31U)
#define CAN_F9BANK2_FBIT31_Msk                                      (0x1UL << CAN_F9BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F9BANK2_FBIT31                                          CAN_F9BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F10BANK2 register  ******************/
#define CAN_F10BANK2_FBIT0_Pos                                      (0U)
#define CAN_F10BANK2_FBIT0_Msk                                      (0x1UL << CAN_F10BANK2_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F10BANK2_FBIT0                                          CAN_F10BANK2_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F10BANK2_FBIT1_Pos                                      (1U)
#define CAN_F10BANK2_FBIT1_Msk                                      (0x1UL << CAN_F10BANK2_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F10BANK2_FBIT1                                          CAN_F10BANK2_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F10BANK2_FBIT2_Pos                                      (2U)
#define CAN_F10BANK2_FBIT2_Msk                                      (0x1UL << CAN_F10BANK2_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F10BANK2_FBIT2                                          CAN_F10BANK2_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F10BANK2_FBIT3_Pos                                      (3U)
#define CAN_F10BANK2_FBIT3_Msk                                      (0x1UL << CAN_F10BANK2_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F10BANK2_FBIT3                                          CAN_F10BANK2_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F10BANK2_FBIT4_Pos                                      (4U)
#define CAN_F10BANK2_FBIT4_Msk                                      (0x1UL << CAN_F10BANK2_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F10BANK2_FBIT4                                          CAN_F10BANK2_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F10BANK2_FBIT5_Pos                                      (5U)
#define CAN_F10BANK2_FBIT5_Msk                                      (0x1UL << CAN_F10BANK2_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F10BANK2_FBIT5                                          CAN_F10BANK2_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F10BANK2_FBIT6_Pos                                      (6U)
#define CAN_F10BANK2_FBIT6_Msk                                      (0x1UL << CAN_F10BANK2_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F10BANK2_FBIT6                                          CAN_F10BANK2_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F10BANK2_FBIT7_Pos                                      (7U)
#define CAN_F10BANK2_FBIT7_Msk                                      (0x1UL << CAN_F10BANK2_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F10BANK2_FBIT7                                          CAN_F10BANK2_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F10BANK2_FBIT8_Pos                                      (8U)
#define CAN_F10BANK2_FBIT8_Msk                                      (0x1UL << CAN_F10BANK2_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F10BANK2_FBIT8                                          CAN_F10BANK2_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F10BANK2_FBIT9_Pos                                      (9U)
#define CAN_F10BANK2_FBIT9_Msk                                      (0x1UL << CAN_F10BANK2_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F10BANK2_FBIT9                                          CAN_F10BANK2_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F10BANK2_FBIT10_Pos                                     (10U)
#define CAN_F10BANK2_FBIT10_Msk                                     (0x1UL << CAN_F10BANK2_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F10BANK2_FBIT10                                         CAN_F10BANK2_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F10BANK2_FBIT11_Pos                                     (11U)
#define CAN_F10BANK2_FBIT11_Msk                                     (0x1UL << CAN_F10BANK2_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F10BANK2_FBIT11                                         CAN_F10BANK2_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F10BANK2_FBIT12_Pos                                     (12U)
#define CAN_F10BANK2_FBIT12_Msk                                     (0x1UL << CAN_F10BANK2_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F10BANK2_FBIT12                                         CAN_F10BANK2_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F10BANK2_FBIT13_Pos                                     (13U)
#define CAN_F10BANK2_FBIT13_Msk                                     (0x1UL << CAN_F10BANK2_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F10BANK2_FBIT13                                         CAN_F10BANK2_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F10BANK2_FBIT14_Pos                                     (14U)
#define CAN_F10BANK2_FBIT14_Msk                                     (0x1UL << CAN_F10BANK2_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F10BANK2_FBIT14                                         CAN_F10BANK2_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F10BANK2_FBIT15_Pos                                     (15U)
#define CAN_F10BANK2_FBIT15_Msk                                     (0x1UL << CAN_F10BANK2_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F10BANK2_FBIT15                                         CAN_F10BANK2_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F10BANK2_FBIT16_Pos                                     (16U)
#define CAN_F10BANK2_FBIT16_Msk                                     (0x1UL << CAN_F10BANK2_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F10BANK2_FBIT16                                         CAN_F10BANK2_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F10BANK2_FBIT17_Pos                                     (17U)
#define CAN_F10BANK2_FBIT17_Msk                                     (0x1UL << CAN_F10BANK2_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F10BANK2_FBIT17                                         CAN_F10BANK2_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F10BANK2_FBIT18_Pos                                     (18U)
#define CAN_F10BANK2_FBIT18_Msk                                     (0x1UL << CAN_F10BANK2_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F10BANK2_FBIT18                                         CAN_F10BANK2_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F10BANK2_FBIT19_Pos                                     (19U)
#define CAN_F10BANK2_FBIT19_Msk                                     (0x1UL << CAN_F10BANK2_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F10BANK2_FBIT19                                         CAN_F10BANK2_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F10BANK2_FBIT20_Pos                                     (20U)
#define CAN_F10BANK2_FBIT20_Msk                                     (0x1UL << CAN_F10BANK2_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F10BANK2_FBIT20                                         CAN_F10BANK2_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F10BANK2_FBIT21_Pos                                     (21U)
#define CAN_F10BANK2_FBIT21_Msk                                     (0x1UL << CAN_F10BANK2_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F10BANK2_FBIT21                                         CAN_F10BANK2_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F10BANK2_FBIT22_Pos                                     (22U)
#define CAN_F10BANK2_FBIT22_Msk                                     (0x1UL << CAN_F10BANK2_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F10BANK2_FBIT22                                         CAN_F10BANK2_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F10BANK2_FBIT23_Pos                                     (23U)
#define CAN_F10BANK2_FBIT23_Msk                                     (0x1UL << CAN_F10BANK2_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F10BANK2_FBIT23                                         CAN_F10BANK2_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F10BANK2_FBIT24_Pos                                     (24U)
#define CAN_F10BANK2_FBIT24_Msk                                     (0x1UL << CAN_F10BANK2_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F10BANK2_FBIT24                                         CAN_F10BANK2_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F10BANK2_FBIT25_Pos                                     (25U)
#define CAN_F10BANK2_FBIT25_Msk                                     (0x1UL << CAN_F10BANK2_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F10BANK2_FBIT25                                         CAN_F10BANK2_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F10BANK2_FBIT26_Pos                                     (26U)
#define CAN_F10BANK2_FBIT26_Msk                                     (0x1UL << CAN_F10BANK2_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F10BANK2_FBIT26                                         CAN_F10BANK2_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F10BANK2_FBIT27_Pos                                     (27U)
#define CAN_F10BANK2_FBIT27_Msk                                     (0x1UL << CAN_F10BANK2_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F10BANK2_FBIT27                                         CAN_F10BANK2_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F10BANK2_FBIT28_Pos                                     (28U)
#define CAN_F10BANK2_FBIT28_Msk                                     (0x1UL << CAN_F10BANK2_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F10BANK2_FBIT28                                         CAN_F10BANK2_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F10BANK2_FBIT29_Pos                                     (29U)
#define CAN_F10BANK2_FBIT29_Msk                                     (0x1UL << CAN_F10BANK2_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F10BANK2_FBIT29                                         CAN_F10BANK2_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F10BANK2_FBIT30_Pos                                     (30U)
#define CAN_F10BANK2_FBIT30_Msk                                     (0x1UL << CAN_F10BANK2_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F10BANK2_FBIT30                                         CAN_F10BANK2_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F10BANK2_FBIT31_Pos                                     (31U)
#define CAN_F10BANK2_FBIT31_Msk                                     (0x1UL << CAN_F10BANK2_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F10BANK2_FBIT31                                         CAN_F10BANK2_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F11BANK2 register  ******************/
#define CAN_F11BANK2_FBIT0_Pos                                      (0U)
#define CAN_F11BANK2_FBIT0_Msk                                      (0x1UL << CAN_F11BANK2_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F11BANK2_FBIT0                                          CAN_F11BANK2_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F11BANK2_FBIT1_Pos                                      (1U)
#define CAN_F11BANK2_FBIT1_Msk                                      (0x1UL << CAN_F11BANK2_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F11BANK2_FBIT1                                          CAN_F11BANK2_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F11BANK2_FBIT2_Pos                                      (2U)
#define CAN_F11BANK2_FBIT2_Msk                                      (0x1UL << CAN_F11BANK2_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F11BANK2_FBIT2                                          CAN_F11BANK2_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F11BANK2_FBIT3_Pos                                      (3U)
#define CAN_F11BANK2_FBIT3_Msk                                      (0x1UL << CAN_F11BANK2_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F11BANK2_FBIT3                                          CAN_F11BANK2_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F11BANK2_FBIT4_Pos                                      (4U)
#define CAN_F11BANK2_FBIT4_Msk                                      (0x1UL << CAN_F11BANK2_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F11BANK2_FBIT4                                          CAN_F11BANK2_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F11BANK2_FBIT5_Pos                                      (5U)
#define CAN_F11BANK2_FBIT5_Msk                                      (0x1UL << CAN_F11BANK2_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F11BANK2_FBIT5                                          CAN_F11BANK2_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F11BANK2_FBIT6_Pos                                      (6U)
#define CAN_F11BANK2_FBIT6_Msk                                      (0x1UL << CAN_F11BANK2_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F11BANK2_FBIT6                                          CAN_F11BANK2_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F11BANK2_FBIT7_Pos                                      (7U)
#define CAN_F11BANK2_FBIT7_Msk                                      (0x1UL << CAN_F11BANK2_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F11BANK2_FBIT7                                          CAN_F11BANK2_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F11BANK2_FBIT8_Pos                                      (8U)
#define CAN_F11BANK2_FBIT8_Msk                                      (0x1UL << CAN_F11BANK2_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F11BANK2_FBIT8                                          CAN_F11BANK2_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F11BANK2_FBIT9_Pos                                      (9U)
#define CAN_F11BANK2_FBIT9_Msk                                      (0x1UL << CAN_F11BANK2_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F11BANK2_FBIT9                                          CAN_F11BANK2_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F11BANK2_FBIT10_Pos                                     (10U)
#define CAN_F11BANK2_FBIT10_Msk                                     (0x1UL << CAN_F11BANK2_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F11BANK2_FBIT10                                         CAN_F11BANK2_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F11BANK2_FBIT11_Pos                                     (11U)
#define CAN_F11BANK2_FBIT11_Msk                                     (0x1UL << CAN_F11BANK2_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F11BANK2_FBIT11                                         CAN_F11BANK2_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F11BANK2_FBIT12_Pos                                     (12U)
#define CAN_F11BANK2_FBIT12_Msk                                     (0x1UL << CAN_F11BANK2_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F11BANK2_FBIT12                                         CAN_F11BANK2_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F11BANK2_FBIT13_Pos                                     (13U)
#define CAN_F11BANK2_FBIT13_Msk                                     (0x1UL << CAN_F11BANK2_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F11BANK2_FBIT13                                         CAN_F11BANK2_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F11BANK2_FBIT14_Pos                                     (14U)
#define CAN_F11BANK2_FBIT14_Msk                                     (0x1UL << CAN_F11BANK2_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F11BANK2_FBIT14                                         CAN_F11BANK2_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F11BANK2_FBIT15_Pos                                     (15U)
#define CAN_F11BANK2_FBIT15_Msk                                     (0x1UL << CAN_F11BANK2_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F11BANK2_FBIT15                                         CAN_F11BANK2_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F11BANK2_FBIT16_Pos                                     (16U)
#define CAN_F11BANK2_FBIT16_Msk                                     (0x1UL << CAN_F11BANK2_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F11BANK2_FBIT16                                         CAN_F11BANK2_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F11BANK2_FBIT17_Pos                                     (17U)
#define CAN_F11BANK2_FBIT17_Msk                                     (0x1UL << CAN_F11BANK2_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F11BANK2_FBIT17                                         CAN_F11BANK2_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F11BANK2_FBIT18_Pos                                     (18U)
#define CAN_F11BANK2_FBIT18_Msk                                     (0x1UL << CAN_F11BANK2_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F11BANK2_FBIT18                                         CAN_F11BANK2_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F11BANK2_FBIT19_Pos                                     (19U)
#define CAN_F11BANK2_FBIT19_Msk                                     (0x1UL << CAN_F11BANK2_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F11BANK2_FBIT19                                         CAN_F11BANK2_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F11BANK2_FBIT20_Pos                                     (20U)
#define CAN_F11BANK2_FBIT20_Msk                                     (0x1UL << CAN_F11BANK2_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F11BANK2_FBIT20                                         CAN_F11BANK2_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F11BANK2_FBIT21_Pos                                     (21U)
#define CAN_F11BANK2_FBIT21_Msk                                     (0x1UL << CAN_F11BANK2_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F11BANK2_FBIT21                                         CAN_F11BANK2_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F11BANK2_FBIT22_Pos                                     (22U)
#define CAN_F11BANK2_FBIT22_Msk                                     (0x1UL << CAN_F11BANK2_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F11BANK2_FBIT22                                         CAN_F11BANK2_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F11BANK2_FBIT23_Pos                                     (23U)
#define CAN_F11BANK2_FBIT23_Msk                                     (0x1UL << CAN_F11BANK2_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F11BANK2_FBIT23                                         CAN_F11BANK2_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F11BANK2_FBIT24_Pos                                     (24U)
#define CAN_F11BANK2_FBIT24_Msk                                     (0x1UL << CAN_F11BANK2_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F11BANK2_FBIT24                                         CAN_F11BANK2_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F11BANK2_FBIT25_Pos                                     (25U)
#define CAN_F11BANK2_FBIT25_Msk                                     (0x1UL << CAN_F11BANK2_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F11BANK2_FBIT25                                         CAN_F11BANK2_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F11BANK2_FBIT26_Pos                                     (26U)
#define CAN_F11BANK2_FBIT26_Msk                                     (0x1UL << CAN_F11BANK2_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F11BANK2_FBIT26                                         CAN_F11BANK2_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F11BANK2_FBIT27_Pos                                     (27U)
#define CAN_F11BANK2_FBIT27_Msk                                     (0x1UL << CAN_F11BANK2_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F11BANK2_FBIT27                                         CAN_F11BANK2_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F11BANK2_FBIT28_Pos                                     (28U)
#define CAN_F11BANK2_FBIT28_Msk                                     (0x1UL << CAN_F11BANK2_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F11BANK2_FBIT28                                         CAN_F11BANK2_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F11BANK2_FBIT29_Pos                                     (29U)
#define CAN_F11BANK2_FBIT29_Msk                                     (0x1UL << CAN_F11BANK2_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F11BANK2_FBIT29                                         CAN_F11BANK2_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F11BANK2_FBIT30_Pos                                     (30U)
#define CAN_F11BANK2_FBIT30_Msk                                     (0x1UL << CAN_F11BANK2_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F11BANK2_FBIT30                                         CAN_F11BANK2_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F11BANK2_FBIT31_Pos                                     (31U)
#define CAN_F11BANK2_FBIT31_Msk                                     (0x1UL << CAN_F11BANK2_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F11BANK2_FBIT31                                         CAN_F11BANK2_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F12BANK2 register  ******************/
#define CAN_F12BANK2_FBIT0_Pos                                      (0U)
#define CAN_F12BANK2_FBIT0_Msk                                      (0x1UL << CAN_F12BANK2_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F12BANK2_FBIT0                                          CAN_F12BANK2_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F12BANK2_FBIT1_Pos                                      (1U)
#define CAN_F12BANK2_FBIT1_Msk                                      (0x1UL << CAN_F12BANK2_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F12BANK2_FBIT1                                          CAN_F12BANK2_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F12BANK2_FBIT2_Pos                                      (2U)
#define CAN_F12BANK2_FBIT2_Msk                                      (0x1UL << CAN_F12BANK2_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F12BANK2_FBIT2                                          CAN_F12BANK2_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F12BANK2_FBIT3_Pos                                      (3U)
#define CAN_F12BANK2_FBIT3_Msk                                      (0x1UL << CAN_F12BANK2_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F12BANK2_FBIT3                                          CAN_F12BANK2_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F12BANK2_FBIT4_Pos                                      (4U)
#define CAN_F12BANK2_FBIT4_Msk                                      (0x1UL << CAN_F12BANK2_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F12BANK2_FBIT4                                          CAN_F12BANK2_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F12BANK2_FBIT5_Pos                                      (5U)
#define CAN_F12BANK2_FBIT5_Msk                                      (0x1UL << CAN_F12BANK2_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F12BANK2_FBIT5                                          CAN_F12BANK2_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F12BANK2_FBIT6_Pos                                      (6U)
#define CAN_F12BANK2_FBIT6_Msk                                      (0x1UL << CAN_F12BANK2_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F12BANK2_FBIT6                                          CAN_F12BANK2_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F12BANK2_FBIT7_Pos                                      (7U)
#define CAN_F12BANK2_FBIT7_Msk                                      (0x1UL << CAN_F12BANK2_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F12BANK2_FBIT7                                          CAN_F12BANK2_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F12BANK2_FBIT8_Pos                                      (8U)
#define CAN_F12BANK2_FBIT8_Msk                                      (0x1UL << CAN_F12BANK2_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F12BANK2_FBIT8                                          CAN_F12BANK2_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F12BANK2_FBIT9_Pos                                      (9U)
#define CAN_F12BANK2_FBIT9_Msk                                      (0x1UL << CAN_F12BANK2_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F12BANK2_FBIT9                                          CAN_F12BANK2_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F12BANK2_FBIT10_Pos                                     (10U)
#define CAN_F12BANK2_FBIT10_Msk                                     (0x1UL << CAN_F12BANK2_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F12BANK2_FBIT10                                         CAN_F12BANK2_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F12BANK2_FBIT11_Pos                                     (11U)
#define CAN_F12BANK2_FBIT11_Msk                                     (0x1UL << CAN_F12BANK2_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F12BANK2_FBIT11                                         CAN_F12BANK2_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F12BANK2_FBIT12_Pos                                     (12U)
#define CAN_F12BANK2_FBIT12_Msk                                     (0x1UL << CAN_F12BANK2_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F12BANK2_FBIT12                                         CAN_F12BANK2_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F12BANK2_FBIT13_Pos                                     (13U)
#define CAN_F12BANK2_FBIT13_Msk                                     (0x1UL << CAN_F12BANK2_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F12BANK2_FBIT13                                         CAN_F12BANK2_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F12BANK2_FBIT14_Pos                                     (14U)
#define CAN_F12BANK2_FBIT14_Msk                                     (0x1UL << CAN_F12BANK2_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F12BANK2_FBIT14                                         CAN_F12BANK2_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F12BANK2_FBIT15_Pos                                     (15U)
#define CAN_F12BANK2_FBIT15_Msk                                     (0x1UL << CAN_F12BANK2_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F12BANK2_FBIT15                                         CAN_F12BANK2_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F12BANK2_FBIT16_Pos                                     (16U)
#define CAN_F12BANK2_FBIT16_Msk                                     (0x1UL << CAN_F12BANK2_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F12BANK2_FBIT16                                         CAN_F12BANK2_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F12BANK2_FBIT17_Pos                                     (17U)
#define CAN_F12BANK2_FBIT17_Msk                                     (0x1UL << CAN_F12BANK2_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F12BANK2_FBIT17                                         CAN_F12BANK2_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F12BANK2_FBIT18_Pos                                     (18U)
#define CAN_F12BANK2_FBIT18_Msk                                     (0x1UL << CAN_F12BANK2_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F12BANK2_FBIT18                                         CAN_F12BANK2_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F12BANK2_FBIT19_Pos                                     (19U)
#define CAN_F12BANK2_FBIT19_Msk                                     (0x1UL << CAN_F12BANK2_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F12BANK2_FBIT19                                         CAN_F12BANK2_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F12BANK2_FBIT20_Pos                                     (20U)
#define CAN_F12BANK2_FBIT20_Msk                                     (0x1UL << CAN_F12BANK2_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F12BANK2_FBIT20                                         CAN_F12BANK2_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F12BANK2_FBIT21_Pos                                     (21U)
#define CAN_F12BANK2_FBIT21_Msk                                     (0x1UL << CAN_F12BANK2_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F12BANK2_FBIT21                                         CAN_F12BANK2_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F12BANK2_FBIT22_Pos                                     (22U)
#define CAN_F12BANK2_FBIT22_Msk                                     (0x1UL << CAN_F12BANK2_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F12BANK2_FBIT22                                         CAN_F12BANK2_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F12BANK2_FBIT23_Pos                                     (23U)
#define CAN_F12BANK2_FBIT23_Msk                                     (0x1UL << CAN_F12BANK2_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F12BANK2_FBIT23                                         CAN_F12BANK2_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F12BANK2_FBIT24_Pos                                     (24U)
#define CAN_F12BANK2_FBIT24_Msk                                     (0x1UL << CAN_F12BANK2_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F12BANK2_FBIT24                                         CAN_F12BANK2_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F12BANK2_FBIT25_Pos                                     (25U)
#define CAN_F12BANK2_FBIT25_Msk                                     (0x1UL << CAN_F12BANK2_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F12BANK2_FBIT25                                         CAN_F12BANK2_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F12BANK2_FBIT26_Pos                                     (26U)
#define CAN_F12BANK2_FBIT26_Msk                                     (0x1UL << CAN_F12BANK2_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F12BANK2_FBIT26                                         CAN_F12BANK2_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F12BANK2_FBIT27_Pos                                     (27U)
#define CAN_F12BANK2_FBIT27_Msk                                     (0x1UL << CAN_F12BANK2_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F12BANK2_FBIT27                                         CAN_F12BANK2_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F12BANK2_FBIT28_Pos                                     (28U)
#define CAN_F12BANK2_FBIT28_Msk                                     (0x1UL << CAN_F12BANK2_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F12BANK2_FBIT28                                         CAN_F12BANK2_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F12BANK2_FBIT29_Pos                                     (29U)
#define CAN_F12BANK2_FBIT29_Msk                                     (0x1UL << CAN_F12BANK2_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F12BANK2_FBIT29                                         CAN_F12BANK2_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F12BANK2_FBIT30_Pos                                     (30U)
#define CAN_F12BANK2_FBIT30_Msk                                     (0x1UL << CAN_F12BANK2_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F12BANK2_FBIT30                                         CAN_F12BANK2_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F12BANK2_FBIT31_Pos                                     (31U)
#define CAN_F12BANK2_FBIT31_Msk                                     (0x1UL << CAN_F12BANK2_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F12BANK2_FBIT31                                         CAN_F12BANK2_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F13BANK2 register  ******************/
#define CAN_F13BANK2_FBIT0_Pos                                      (0U)
#define CAN_F13BANK2_FBIT0_Msk                                      (0x1UL << CAN_F13BANK2_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F13BANK2_FBIT0                                          CAN_F13BANK2_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F13BANK2_FBIT1_Pos                                      (1U)
#define CAN_F13BANK2_FBIT1_Msk                                      (0x1UL << CAN_F13BANK2_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F13BANK2_FBIT1                                          CAN_F13BANK2_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F13BANK2_FBIT2_Pos                                      (2U)
#define CAN_F13BANK2_FBIT2_Msk                                      (0x1UL << CAN_F13BANK2_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F13BANK2_FBIT2                                          CAN_F13BANK2_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F13BANK2_FBIT3_Pos                                      (3U)
#define CAN_F13BANK2_FBIT3_Msk                                      (0x1UL << CAN_F13BANK2_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F13BANK2_FBIT3                                          CAN_F13BANK2_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F13BANK2_FBIT4_Pos                                      (4U)
#define CAN_F13BANK2_FBIT4_Msk                                      (0x1UL << CAN_F13BANK2_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F13BANK2_FBIT4                                          CAN_F13BANK2_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F13BANK2_FBIT5_Pos                                      (5U)
#define CAN_F13BANK2_FBIT5_Msk                                      (0x1UL << CAN_F13BANK2_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F13BANK2_FBIT5                                          CAN_F13BANK2_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F13BANK2_FBIT6_Pos                                      (6U)
#define CAN_F13BANK2_FBIT6_Msk                                      (0x1UL << CAN_F13BANK2_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F13BANK2_FBIT6                                          CAN_F13BANK2_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F13BANK2_FBIT7_Pos                                      (7U)
#define CAN_F13BANK2_FBIT7_Msk                                      (0x1UL << CAN_F13BANK2_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F13BANK2_FBIT7                                          CAN_F13BANK2_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F13BANK2_FBIT8_Pos                                      (8U)
#define CAN_F13BANK2_FBIT8_Msk                                      (0x1UL << CAN_F13BANK2_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F13BANK2_FBIT8                                          CAN_F13BANK2_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F13BANK2_FBIT9_Pos                                      (9U)
#define CAN_F13BANK2_FBIT9_Msk                                      (0x1UL << CAN_F13BANK2_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F13BANK2_FBIT9                                          CAN_F13BANK2_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F13BANK2_FBIT10_Pos                                     (10U)
#define CAN_F13BANK2_FBIT10_Msk                                     (0x1UL << CAN_F13BANK2_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F13BANK2_FBIT10                                         CAN_F13BANK2_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F13BANK2_FBIT11_Pos                                     (11U)
#define CAN_F13BANK2_FBIT11_Msk                                     (0x1UL << CAN_F13BANK2_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F13BANK2_FBIT11                                         CAN_F13BANK2_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F13BANK2_FBIT12_Pos                                     (12U)
#define CAN_F13BANK2_FBIT12_Msk                                     (0x1UL << CAN_F13BANK2_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F13BANK2_FBIT12                                         CAN_F13BANK2_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F13BANK2_FBIT13_Pos                                     (13U)
#define CAN_F13BANK2_FBIT13_Msk                                     (0x1UL << CAN_F13BANK2_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F13BANK2_FBIT13                                         CAN_F13BANK2_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F13BANK2_FBIT14_Pos                                     (14U)
#define CAN_F13BANK2_FBIT14_Msk                                     (0x1UL << CAN_F13BANK2_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F13BANK2_FBIT14                                         CAN_F13BANK2_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F13BANK2_FBIT15_Pos                                     (15U)
#define CAN_F13BANK2_FBIT15_Msk                                     (0x1UL << CAN_F13BANK2_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F13BANK2_FBIT15                                         CAN_F13BANK2_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F13BANK2_FBIT16_Pos                                     (16U)
#define CAN_F13BANK2_FBIT16_Msk                                     (0x1UL << CAN_F13BANK2_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F13BANK2_FBIT16                                         CAN_F13BANK2_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F13BANK2_FBIT17_Pos                                     (17U)
#define CAN_F13BANK2_FBIT17_Msk                                     (0x1UL << CAN_F13BANK2_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F13BANK2_FBIT17                                         CAN_F13BANK2_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F13BANK2_FBIT18_Pos                                     (18U)
#define CAN_F13BANK2_FBIT18_Msk                                     (0x1UL << CAN_F13BANK2_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F13BANK2_FBIT18                                         CAN_F13BANK2_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F13BANK2_FBIT19_Pos                                     (19U)
#define CAN_F13BANK2_FBIT19_Msk                                     (0x1UL << CAN_F13BANK2_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F13BANK2_FBIT19                                         CAN_F13BANK2_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F13BANK2_FBIT20_Pos                                     (20U)
#define CAN_F13BANK2_FBIT20_Msk                                     (0x1UL << CAN_F13BANK2_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F13BANK2_FBIT20                                         CAN_F13BANK2_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F13BANK2_FBIT21_Pos                                     (21U)
#define CAN_F13BANK2_FBIT21_Msk                                     (0x1UL << CAN_F13BANK2_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F13BANK2_FBIT21                                         CAN_F13BANK2_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F13BANK2_FBIT22_Pos                                     (22U)
#define CAN_F13BANK2_FBIT22_Msk                                     (0x1UL << CAN_F13BANK2_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F13BANK2_FBIT22                                         CAN_F13BANK2_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F13BANK2_FBIT23_Pos                                     (23U)
#define CAN_F13BANK2_FBIT23_Msk                                     (0x1UL << CAN_F13BANK2_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F13BANK2_FBIT23                                         CAN_F13BANK2_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F13BANK2_FBIT24_Pos                                     (24U)
#define CAN_F13BANK2_FBIT24_Msk                                     (0x1UL << CAN_F13BANK2_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F13BANK2_FBIT24                                         CAN_F13BANK2_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F13BANK2_FBIT25_Pos                                     (25U)
#define CAN_F13BANK2_FBIT25_Msk                                     (0x1UL << CAN_F13BANK2_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F13BANK2_FBIT25                                         CAN_F13BANK2_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F13BANK2_FBIT26_Pos                                     (26U)
#define CAN_F13BANK2_FBIT26_Msk                                     (0x1UL << CAN_F13BANK2_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F13BANK2_FBIT26                                         CAN_F13BANK2_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F13BANK2_FBIT27_Pos                                     (27U)
#define CAN_F13BANK2_FBIT27_Msk                                     (0x1UL << CAN_F13BANK2_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F13BANK2_FBIT27                                         CAN_F13BANK2_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F13BANK2_FBIT28_Pos                                     (28U)
#define CAN_F13BANK2_FBIT28_Msk                                     (0x1UL << CAN_F13BANK2_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F13BANK2_FBIT28                                         CAN_F13BANK2_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F13BANK2_FBIT29_Pos                                     (29U)
#define CAN_F13BANK2_FBIT29_Msk                                     (0x1UL << CAN_F13BANK2_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F13BANK2_FBIT29                                         CAN_F13BANK2_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F13BANK2_FBIT30_Pos                                     (30U)
#define CAN_F13BANK2_FBIT30_Msk                                     (0x1UL << CAN_F13BANK2_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F13BANK2_FBIT30                                         CAN_F13BANK2_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F13BANK2_FBIT31_Pos                                     (31U)
#define CAN_F13BANK2_FBIT31_Msk                                     (0x1UL << CAN_F13BANK2_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F13BANK2_FBIT31                                         CAN_F13BANK2_FBIT31_Msk                                        /*!<Filter bit 31 */

/******************************************************************************/
/*                                                                            */
/*                          CRC calculation unit                              */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for CRC_DATA register  *********************/
#define CRC_DATA_DATA_Pos                                           (0U)
#define CRC_DATA_DATA_Msk                                           (0xFFFFFFFFUL << CRC_DATA_DATA_Pos)                            /*!< 0xFFFFFFFF */
#define CRC_DATA_DATA                                               CRC_DATA_DATA_Msk                                              /*!< Data register bits */


/*******************  Bit definition for CRC_INDATA register  ********************/
#define CRC_INDATA_INDATA_Pos                                       (0U)
#define CRC_INDATA_INDATA_Msk                                       (0xFFUL << CRC_INDATA_INDATA_Pos)                              /*!< 0x000000FF */
#define CRC_INDATA_INDATA                                           CRC_INDATA_INDATA_Msk                                          /*!< General-purpose 8-bit data register bits */


/********************  Bit definition for CRC_CTRL register  ********************/
#define CRC_CTRL_RST_Pos                                            (0U)
#define CRC_CTRL_RST_Msk                                            (0x1UL << CRC_CTRL_RST_Pos)                                    /*!< 0x00000001 */
#define CRC_CTRL_RST                                                CRC_CTRL_RST_Msk                                               /*!< RESET bit */

/******************************************************************************/
/*                                                                            */
/*                             DMA Controller                                 */
/*                                                                            */
/******************************************************************************/
/********************  Bits definition for DMA_SCFGx register  *****************/
#define DMA_SCFGx_CHSEL_Pos                                         (25U)
#define DMA_SCFGx_CHSEL_Msk                                         (0x7UL << DMA_SCFGx_CHSEL_Pos)                                 /*!< 0x0E000000 */
#define DMA_SCFGx_CHSEL                                             DMA_SCFGx_CHSEL_Msk
#define DMA_SCFGx_CHSEL_0                                           0x02000000U
#define DMA_SCFGx_CHSEL_1                                           0x04000000U
#define DMA_SCFGx_CHSEL_2                                           0x08000000U
#define DMA_SCFGx_MBCFG_Pos                                         (23U)
#define DMA_SCFGx_MBCFG_Msk                                         (0x3UL << DMA_SCFGx_MBCFG_Pos)                                 /*!< 0x01800000 */
#define DMA_SCFGx_MBCFG                                             DMA_SCFGx_MBCFG_Msk
#define DMA_SCFGx_MBCFG_0                                           (0x1UL << DMA_SCFGx_MBCFG_Pos)                                 /*!< 0x00800000 */
#define DMA_SCFGx_MBCFG_1                                           (0x2UL << DMA_SCFGx_MBCFG_Pos)                                 /*!< 0x01000000 */
#define DMA_SCFGx_PBCFG_Pos                                         (21U)
#define DMA_SCFGx_PBCFG_Msk                                         (0x3UL << DMA_SCFGx_PBCFG_Pos)                                 /*!< 0x00600000 */
#define DMA_SCFGx_PBCFG                                             DMA_SCFGx_PBCFG_Msk
#define DMA_SCFGx_PBCFG_0                                           (0x1UL << DMA_SCFGx_PBCFG_Pos)                                 /*!< 0x00200000 */
#define DMA_SCFGx_PBCFG_1                                           (0x2UL << DMA_SCFGx_PBCFG_Pos)                                 /*!< 0x00400000 */
#define DMA_SCFGx_CTARG_Pos                                         (19U)
#define DMA_SCFGx_CTARG_Msk                                         (0x1UL << DMA_SCFGx_CTARG_Pos)                                 /*!< 0x00080000 */
#define DMA_SCFGx_CTARG                                             DMA_SCFGx_CTARG_Msk
#define DMA_SCFGx_DBM_Pos                                           (18U)
#define DMA_SCFGx_DBM_Msk                                           (0x1UL << DMA_SCFGx_DBM_Pos)                                   /*!< 0x00040000 */
#define DMA_SCFGx_DBM                                               DMA_SCFGx_DBM_Msk
#define DMA_SCFGx_PRILCFG_Pos                                       (16U)
#define DMA_SCFGx_PRILCFG_Msk                                       (0x3UL << DMA_SCFGx_PRILCFG_Pos)                               /*!< 0x00030000 */
#define DMA_SCFGx_PRILCFG                                           DMA_SCFGx_PRILCFG_Msk
#define DMA_SCFGx_PRILCFG_0                                         (0x1UL << DMA_SCFGx_PRILCFG_Pos)                               /*!< 0x00010000 */
#define DMA_SCFGx_PRILCFG_1                                         (0x2UL << DMA_SCFGx_PRILCFG_Pos)                               /*!< 0x00020000 */
#define DMA_SCFGx_PERIOSIZE_Pos                                     (15U)
#define DMA_SCFGx_PERIOSIZE_Msk                                     (0x1UL << DMA_SCFGx_PERIOSIZE_Pos)                             /*!< 0x00008000 */
#define DMA_SCFGx_PERIOSIZE                                         DMA_SCFGx_PERIOSIZE_Msk
#define DMA_SCFGx_MEMSIZECFG_Pos                                    (13U)
#define DMA_SCFGx_MEMSIZECFG_Msk                                    (0x3UL << DMA_SCFGx_MEMSIZECFG_Pos)                            /*!< 0x00006000 */
#define DMA_SCFGx_MEMSIZECFG                                        DMA_SCFGx_MEMSIZECFG_Msk
#define DMA_SCFGx_MEMSIZECFG_0                                      (0x1UL << DMA_SCFGx_MEMSIZECFG_Pos)                            /*!< 0x00002000 */
#define DMA_SCFGx_MEMSIZECFG_1                                      (0x2UL << DMA_SCFGx_MEMSIZECFG_Pos)                            /*!< 0x00004000 */
#define DMA_SCFGx_PERSIZECFG_Pos                                    (11U)
#define DMA_SCFGx_PERSIZECFG_Msk                                    (0x3UL << DMA_SCFGx_PERSIZECFG_Pos)                            /*!< 0x00001800 */
#define DMA_SCFGx_PERSIZECFG                                        DMA_SCFGx_PERSIZECFG_Msk
#define DMA_SCFGx_PERSIZECFG_0                                      (0x1UL << DMA_SCFGx_PERSIZECFG_Pos)                            /*!< 0x00000800 */
#define DMA_SCFGx_PERSIZECFG_1                                      (0x2UL << DMA_SCFGx_PERSIZECFG_Pos)                            /*!< 0x00001000 */
#define DMA_SCFGx_MEMIM_Pos                                         (10U)
#define DMA_SCFGx_MEMIM_Msk                                         (0x1UL << DMA_SCFGx_MEMIM_Pos)                                 /*!< 0x00000400 */
#define DMA_SCFGx_MEMIM                                             DMA_SCFGx_MEMIM_Msk
#define DMA_SCFGx_PERIM_Pos                                         (9U)
#define DMA_SCFGx_PERIM_Msk                                         (0x1UL << DMA_SCFGx_PERIM_Pos)                                 /*!< 0x00000200 */
#define DMA_SCFGx_PERIM                                             DMA_SCFGx_PERIM_Msk
#define DMA_SCFGx_CIRCMEN_Pos                                       (8U)
#define DMA_SCFGx_CIRCMEN_Msk                                       (0x1UL << DMA_SCFGx_CIRCMEN_Pos)                               /*!< 0x00000100 */
#define DMA_SCFGx_CIRCMEN                                           DMA_SCFGx_CIRCMEN_Msk
#define DMA_SCFGx_DIRCFG_Pos                                        (6U)
#define DMA_SCFGx_DIRCFG_Msk                                        (0x3UL << DMA_SCFGx_DIRCFG_Pos)                                /*!< 0x000000C0 */
#define DMA_SCFGx_DIRCFG                                            DMA_SCFGx_DIRCFG_Msk
#define DMA_SCFGx_DIRCFG_0                                          (0x1UL << DMA_SCFGx_DIRCFG_Pos)                                /*!< 0x00000040 */
#define DMA_SCFGx_DIRCFG_1                                          (0x2UL << DMA_SCFGx_DIRCFG_Pos)                                /*!< 0x00000080 */
#define DMA_SCFGx_PERFC_Pos                                         (5U)
#define DMA_SCFGx_PERFC_Msk                                         (0x1UL << DMA_SCFGx_PERFC_Pos)                                 /*!< 0x00000020 */
#define DMA_SCFGx_PERFC                                             DMA_SCFGx_PERFC_Msk
#define DMA_SCFGx_TXCIEN_Pos                                        (4U)
#define DMA_SCFGx_TXCIEN_Msk                                        (0x1UL << DMA_SCFGx_TXCIEN_Pos)                                /*!< 0x00000010 */
#define DMA_SCFGx_TXCIEN                                            DMA_SCFGx_TXCIEN_Msk
#define DMA_SCFGx_HTXIEN_Pos                                        (3U)
#define DMA_SCFGx_HTXIEN_Msk                                        (0x1UL << DMA_SCFGx_HTXIEN_Pos)                                /*!< 0x00000008 */
#define DMA_SCFGx_HTXIEN                                            DMA_SCFGx_HTXIEN_Msk
#define DMA_SCFGx_TXEIEN_Pos                                        (2U)
#define DMA_SCFGx_TXEIEN_Msk                                        (0x1UL << DMA_SCFGx_TXEIEN_Pos)                                /*!< 0x00000004 */
#define DMA_SCFGx_TXEIEN                                            DMA_SCFGx_TXEIEN_Msk
#define DMA_SCFGx_DMEIEN_Pos                                        (1U)
#define DMA_SCFGx_DMEIEN_Msk                                        (0x1UL << DMA_SCFGx_DMEIEN_Pos)                                /*!< 0x00000002 */
#define DMA_SCFGx_DMEIEN                                            DMA_SCFGx_DMEIEN_Msk
#define DMA_SCFGx_EN_Pos                                            (0U)
#define DMA_SCFGx_EN_Msk                                            (0x1UL << DMA_SCFGx_EN_Pos)                                    /*!< 0x00000001 */
#define DMA_SCFGx_EN                                                DMA_SCFGx_EN_Msk

/* Legacy defines */
#define DMA_SCFGx_ACK_Pos                                           (20U)
#define DMA_SCFGx_ACK_Msk                                           (0x1UL << DMA_SCFGx_ACK_Pos)                                   /*!< 0x00100000 */
#define DMA_SCFGx_ACK                                               DMA_SCFGx_ACK_Msk

/********************  Bits definition for DMA_NDATAx register  **************/
#define DMA_NDATAx_Pos                                              (0U)
#define DMA_NDATAx_Msk                                              (0xFFFFUL << DMA_NDATAx_Pos)                                   /*!< 0x0000FFFF */
#define DMA_NDATAx                                                  DMA_NDATAx_Msk
#define DMA_NDATAx_0                                                (0x0001UL << DMA_NDATAx_Pos)                                   /*!< 0x00000001 */
#define DMA_NDATAx_1                                                (0x0002UL << DMA_NDATAx_Pos)                                   /*!< 0x00000002 */
#define DMA_NDATAx_2                                                (0x0004UL << DMA_NDATAx_Pos)                                   /*!< 0x00000004 */
#define DMA_NDATAx_3                                                (0x0008UL << DMA_NDATAx_Pos)                                   /*!< 0x00000008 */
#define DMA_NDATAx_4                                                (0x0010UL << DMA_NDATAx_Pos)                                   /*!< 0x00000010 */
#define DMA_NDATAx_5                                                (0x0020UL << DMA_NDATAx_Pos)                                   /*!< 0x00000020 */
#define DMA_NDATAx_6                                                (0x0040UL << DMA_NDATAx_Pos)                                   /*!< 0x00000040 */
#define DMA_NDATAx_7                                                (0x0080UL << DMA_NDATAx_Pos)                                   /*!< 0x00000080 */
#define DMA_NDATAx_8                                                (0x0100UL << DMA_NDATAx_Pos)                                   /*!< 0x00000100 */
#define DMA_NDATAx_9                                                (0x0200UL << DMA_NDATAx_Pos)                                   /*!< 0x00000200 */
#define DMA_NDATAx_10                                               (0x0400UL << DMA_NDATAx_Pos)                                   /*!< 0x00000400 */
#define DMA_NDATAx_11                                               (0x0800UL << DMA_NDATAx_Pos)                                   /*!< 0x00000800 */
#define DMA_NDATAx_12                                               (0x1000UL << DMA_NDATAx_Pos)                                   /*!< 0x00001000 */
#define DMA_NDATAx_13                                               (0x2000UL << DMA_NDATAx_Pos)                                   /*!< 0x00002000 */
#define DMA_NDATAx_14                                               (0x4000UL << DMA_NDATAx_Pos)                                   /*!< 0x00004000 */
#define DMA_NDATAx_15                                               (0x8000UL << DMA_NDATAx_Pos)                                   /*!< 0x00008000 */

/********************  Bits definition for DMA_FCTRLx register  ****************/
#define DMA_FCTRLx_FEIEN_Pos                                        (7U)
#define DMA_FCTRLx_FEIEN_Msk                                        (0x1UL << DMA_FCTRLx_FEIEN_Pos)                                /*!< 0x00000080 */
#define DMA_FCTRLx_FEIEN                                            DMA_FCTRLx_FEIEN_Msk
#define DMA_FCTRLx_FSTS_Pos                                         (3U)
#define DMA_FCTRLx_FSTS_Msk                                         (0x7UL << DMA_FCTRLx_FSTS_Pos)                                 /*!< 0x00000038 */
#define DMA_FCTRLx_FSTS                                             DMA_FCTRLx_FSTS_Msk
#define DMA_FCTRLx_FSTS_0                                           (0x1UL << DMA_FCTRLx_FSTS_Pos)                                 /*!< 0x00000008 */
#define DMA_FCTRLx_FSTS_1                                           (0x2UL << DMA_FCTRLx_FSTS_Pos)                                 /*!< 0x00000010 */
#define DMA_FCTRLx_FSTS_2                                           (0x4UL << DMA_FCTRLx_FSTS_Pos)                                 /*!< 0x00000020 */
#define DMA_FCTRLx_DMDEN_Pos                                        (2U)
#define DMA_FCTRLx_DMDEN_Msk                                        (0x1UL << DMA_FCTRLx_DMDEN_Pos)                                /*!< 0x00000004 */
#define DMA_FCTRLx_DMDEN                                            DMA_FCTRLx_DMDEN_Msk
#define DMA_FCTRLx_FTHSEL_Pos                                       (0U)
#define DMA_FCTRLx_FTHSEL_Msk                                       (0x3UL << DMA_FCTRLx_FTHSEL_Pos)                               /*!< 0x00000003 */
#define DMA_FCTRLx_FTHSEL                                           DMA_FCTRLx_FTHSEL_Msk
#define DMA_FCTRLx_FTHSEL_0                                         (0x1UL << DMA_FCTRLx_FTHSEL_Pos)                               /*!< 0x00000001 */
#define DMA_FCTRLx_FTHSEL_1                                         (0x2UL << DMA_FCTRLx_FTHSEL_Pos)                               /*!< 0x00000002 */

/********************  Bits definition for DMA_LINTSTS register  *****************/
#define DMA_LINTSTS_TXCIFLG3_Pos                                    (27U)
#define DMA_LINTSTS_TXCIFLG3_Msk                                    (0x1UL << DMA_LINTSTS_TXCIFLG3_Pos)                            /*!< 0x08000000 */
#define DMA_LINTSTS_TXCIFLG3                                        DMA_LINTSTS_TXCIFLG3_Msk
#define DMA_LINTSTS_HTXIFLG3_Pos                                    (26U)
#define DMA_LINTSTS_HTXIFLG3_Msk                                    (0x1UL << DMA_LINTSTS_HTXIFLG3_Pos)                            /*!< 0x04000000 */
#define DMA_LINTSTS_HTXIFLG3                                        DMA_LINTSTS_HTXIFLG3_Msk
#define DMA_LINTSTS_TXEIFLG3_Pos                                    (25U)
#define DMA_LINTSTS_TXEIFLG3_Msk                                    (0x1UL << DMA_LINTSTS_TXEIFLG3_Pos)                            /*!< 0x02000000 */
#define DMA_LINTSTS_TXEIFLG3                                        DMA_LINTSTS_TXEIFLG3_Msk
#define DMA_LINTSTS_DMEIFLG3_Pos                                    (24U)
#define DMA_LINTSTS_DMEIFLG3_Msk                                    (0x1UL << DMA_LINTSTS_DMEIFLG3_Pos)                            /*!< 0x01000000 */
#define DMA_LINTSTS_DMEIFLG3                                        DMA_LINTSTS_DMEIFLG3_Msk
#define DMA_LINTSTS_FEIFLG3_Pos                                     (22U)
#define DMA_LINTSTS_FEIFLG3_Msk                                     (0x1UL << DMA_LINTSTS_FEIFLG3_Pos)                             /*!< 0x00400000 */
#define DMA_LINTSTS_FEIFLG3                                         DMA_LINTSTS_FEIFLG3_Msk
#define DMA_LINTSTS_TXCIFLG2_Pos                                    (21U)
#define DMA_LINTSTS_TXCIFLG2_Msk                                    (0x1UL << DMA_LINTSTS_TXCIFLG2_Pos)                            /*!< 0x00200000 */
#define DMA_LINTSTS_TXCIFLG2                                        DMA_LINTSTS_TXCIFLG2_Msk
#define DMA_LINTSTS_HTXIFLG2_Pos                                    (20U)
#define DMA_LINTSTS_HTXIFLG2_Msk                                    (0x1UL << DMA_LINTSTS_HTXIFLG2_Pos)                            /*!< 0x00100000 */
#define DMA_LINTSTS_HTXIFLG2                                        DMA_LINTSTS_HTXIFLG2_Msk
#define DMA_LINTSTS_TXEIFLG2_Pos                                    (19U)
#define DMA_LINTSTS_TXEIFLG2_Msk                                    (0x1UL << DMA_LINTSTS_TXEIFLG2_Pos)                            /*!< 0x00080000 */
#define DMA_LINTSTS_TXEIFLG2                                        DMA_LINTSTS_TXEIFLG2_Msk
#define DMA_LINTSTS_DMEIFLG2_Pos                                    (18U)
#define DMA_LINTSTS_DMEIFLG2_Msk                                    (0x1UL << DMA_LINTSTS_DMEIFLG2_Pos)                            /*!< 0x00040000 */
#define DMA_LINTSTS_DMEIFLG2                                        DMA_LINTSTS_DMEIFLG2_Msk
#define DMA_LINTSTS_FEIFLG2_Pos                                     (16U)
#define DMA_LINTSTS_FEIFLG2_Msk                                     (0x1UL << DMA_LINTSTS_FEIFLG2_Pos)                             /*!< 0x00010000 */
#define DMA_LINTSTS_FEIFLG2                                         DMA_LINTSTS_FEIFLG2_Msk
#define DMA_LINTSTS_TXCIFLG1_Pos                                    (11U)
#define DMA_LINTSTS_TXCIFLG1_Msk                                    (0x1UL << DMA_LINTSTS_TXCIFLG1_Pos)                            /*!< 0x00000800 */
#define DMA_LINTSTS_TXCIFLG1                                        DMA_LINTSTS_TXCIFLG1_Msk
#define DMA_LINTSTS_HTXIFLG1_Pos                                    (10U)
#define DMA_LINTSTS_HTXIFLG1_Msk                                    (0x1UL << DMA_LINTSTS_HTXIFLG1_Pos)                            /*!< 0x00000400 */
#define DMA_LINTSTS_HTXIFLG1                                        DMA_LINTSTS_HTXIFLG1_Msk
#define DMA_LINTSTS_TXEIFLG1_Pos                                    (9U)
#define DMA_LINTSTS_TXEIFLG1_Msk                                    (0x1UL << DMA_LINTSTS_TXEIFLG1_Pos)                            /*!< 0x00000200 */
#define DMA_LINTSTS_TXEIFLG1                                        DMA_LINTSTS_TXEIFLG1_Msk
#define DMA_LINTSTS_DMEIFLG1_Pos                                    (8U)
#define DMA_LINTSTS_DMEIFLG1_Msk                                    (0x1UL << DMA_LINTSTS_DMEIFLG1_Pos)                            /*!< 0x00000100 */
#define DMA_LINTSTS_DMEIFLG1                                        DMA_LINTSTS_DMEIFLG1_Msk
#define DMA_LINTSTS_FEIFLG1_Pos                                     (6U)
#define DMA_LINTSTS_FEIFLG1_Msk                                     (0x1UL << DMA_LINTSTS_FEIFLG1_Pos)                             /*!< 0x00000040 */
#define DMA_LINTSTS_FEIFLG1                                         DMA_LINTSTS_FEIFLG1_Msk
#define DMA_LINTSTS_TXCIFLG0_Pos                                    (5U)
#define DMA_LINTSTS_TXCIFLG0_Msk                                    (0x1UL << DMA_LINTSTS_TXCIFLG0_Pos)                            /*!< 0x00000020 */
#define DMA_LINTSTS_TXCIFLG0                                        DMA_LINTSTS_TXCIFLG0_Msk
#define DMA_LINTSTS_HTXIFLG0_Pos                                    (4U)
#define DMA_LINTSTS_HTXIFLG0_Msk                                    (0x1UL << DMA_LINTSTS_HTXIFLG0_Pos)                            /*!< 0x00000010 */
#define DMA_LINTSTS_HTXIFLG0                                        DMA_LINTSTS_HTXIFLG0_Msk
#define DMA_LINTSTS_TXEIFLG0_Pos                                    (3U)
#define DMA_LINTSTS_TXEIFLG0_Msk                                    (0x1UL << DMA_LINTSTS_TXEIFLG0_Pos)                            /*!< 0x00000008 */
#define DMA_LINTSTS_TXEIFLG0                                        DMA_LINTSTS_TXEIFLG0_Msk
#define DMA_LINTSTS_DMEIFLG0_Pos                                    (2U)
#define DMA_LINTSTS_DMEIFLG0_Msk                                    (0x1UL << DMA_LINTSTS_DMEIFLG0_Pos)                            /*!< 0x00000004 */
#define DMA_LINTSTS_DMEIFLG0                                        DMA_LINTSTS_DMEIFLG0_Msk
#define DMA_LINTSTS_FEIFLG0_Pos                                     (0U)
#define DMA_LINTSTS_FEIFLG0_Msk                                     (0x1UL << DMA_LINTSTS_FEIFLG0_Pos)                             /*!< 0x00000001 */
#define DMA_LINTSTS_FEIFLG0                                         DMA_LINTSTS_FEIFLG0_Msk

/********************  Bits definition for DMA_HINTSTS register  *****************/
#define DMA_HINTSTS_TXCIFLG7_Pos                                    (27U)
#define DMA_HINTSTS_TXCIFLG7_Msk                                    (0x1UL << DMA_HINTSTS_TXCIFLG7_Pos)                            /*!< 0x08000000 */
#define DMA_HINTSTS_TXCIFLG7                                        DMA_HINTSTS_TXCIFLG7_Msk
#define DMA_HINTSTS_HTXIFLG7_Pos                                    (26U)
#define DMA_HINTSTS_HTXIFLG7_Msk                                    (0x1UL << DMA_HINTSTS_HTXIFLG7_Pos)                            /*!< 0x04000000 */
#define DMA_HINTSTS_HTXIFLG7                                        DMA_HINTSTS_HTXIFLG7_Msk
#define DMA_HINTSTS_TXEIFLG7_Pos                                    (25U)
#define DMA_HINTSTS_TXEIFLG7_Msk                                    (0x1UL << DMA_HINTSTS_TXEIFLG7_Pos)                            /*!< 0x02000000 */
#define DMA_HINTSTS_TXEIFLG7                                        DMA_HINTSTS_TXEIFLG7_Msk
#define DMA_HINTSTS_DMEIFLG7_Pos                                    (24U)
#define DMA_HINTSTS_DMEIFLG7_Msk                                    (0x1UL << DMA_HINTSTS_DMEIFLG7_Pos)                            /*!< 0x01000000 */
#define DMA_HINTSTS_DMEIFLG7                                        DMA_HINTSTS_DMEIFLG7_Msk
#define DMA_HINTSTS_FEIFLG7_Pos                                     (22U)
#define DMA_HINTSTS_FEIFLG7_Msk                                     (0x1UL << DMA_HINTSTS_FEIFLG7_Pos)                             /*!< 0x00400000 */
#define DMA_HINTSTS_FEIFLG7                                         DMA_HINTSTS_FEIFLG7_Msk
#define DMA_HINTSTS_TXCIFLG6_Pos                                    (21U)
#define DMA_HINTSTS_TXCIFLG6_Msk                                    (0x1UL << DMA_HINTSTS_TXCIFLG6_Pos)                            /*!< 0x00200000 */
#define DMA_HINTSTS_TXCIFLG6                                        DMA_HINTSTS_TXCIFLG6_Msk
#define DMA_HINTSTS_HTXIFLG6_Pos                                    (20U)
#define DMA_HINTSTS_HTXIFLG6_Msk                                    (0x1UL << DMA_HINTSTS_HTXIFLG6_Pos)                            /*!< 0x00100000 */
#define DMA_HINTSTS_HTXIFLG6                                        DMA_HINTSTS_HTXIFLG6_Msk
#define DMA_HINTSTS_TXEIFLG6_Pos                                    (19U)
#define DMA_HINTSTS_TXEIFLG6_Msk                                    (0x1UL << DMA_HINTSTS_TXEIFLG6_Pos)                            /*!< 0x00080000 */
#define DMA_HINTSTS_TXEIFLG6                                        DMA_HINTSTS_TXEIFLG6_Msk
#define DMA_HINTSTS_DMEIFLG6_Pos                                    (18U)
#define DMA_HINTSTS_DMEIFLG6_Msk                                    (0x1UL << DMA_HINTSTS_DMEIFLG6_Pos)                            /*!< 0x00040000 */
#define DMA_HINTSTS_DMEIFLG6                                        DMA_HINTSTS_DMEIFLG6_Msk
#define DMA_HINTSTS_FEIFLG6_Pos                                     (16U)
#define DMA_HINTSTS_FEIFLG6_Msk                                     (0x1UL << DMA_HINTSTS_FEIFLG6_Pos)                             /*!< 0x00010000 */
#define DMA_HINTSTS_FEIFLG6                                         DMA_HINTSTS_FEIFLG6_Msk
#define DMA_HINTSTS_TXCIFLG5_Pos                                    (11U)
#define DMA_HINTSTS_TXCIFLG5_Msk                                    (0x1UL << DMA_HINTSTS_TXCIFLG5_Pos)                            /*!< 0x00000800 */
#define DMA_HINTSTS_TXCIFLG5                                        DMA_HINTSTS_TXCIFLG5_Msk
#define DMA_HINTSTS_HTXIFLG5_Pos                                    (10U)
#define DMA_HINTSTS_HTXIFLG5_Msk                                    (0x1UL << DMA_HINTSTS_HTXIFLG5_Pos)                            /*!< 0x00000400 */
#define DMA_HINTSTS_HTXIFLG5                                        DMA_HINTSTS_HTXIFLG5_Msk
#define DMA_HINTSTS_TXEIFLG5_Pos                                    (9U)
#define DMA_HINTSTS_TXEIFLG5_Msk                                    (0x1UL << DMA_HINTSTS_TXEIFLG5_Pos)                            /*!< 0x00000200 */
#define DMA_HINTSTS_TXEIFLG5                                        DMA_HINTSTS_TXEIFLG5_Msk
#define DMA_HINTSTS_DMEIFLG5_Pos                                    (8U)
#define DMA_HINTSTS_DMEIFLG5_Msk                                    (0x1UL << DMA_HINTSTS_DMEIFLG5_Pos)                            /*!< 0x00000100 */
#define DMA_HINTSTS_DMEIFLG5                                        DMA_HINTSTS_DMEIFLG5_Msk
#define DMA_HINTSTS_FEIFLG5_Pos                                     (6U)
#define DMA_HINTSTS_FEIFLG5_Msk                                     (0x1UL << DMA_HINTSTS_FEIFLG5_Pos)                             /*!< 0x00000040 */
#define DMA_HINTSTS_FEIFLG5                                         DMA_HINTSTS_FEIFLG5_Msk
#define DMA_HINTSTS_TXCIFLG4_Pos                                    (5U)
#define DMA_HINTSTS_TXCIFLG4_Msk                                    (0x1UL << DMA_HINTSTS_TXCIFLG4_Pos)                            /*!< 0x00000020 */
#define DMA_HINTSTS_TXCIFLG4                                        DMA_HINTSTS_TXCIFLG4_Msk
#define DMA_HINTSTS_HTXIFLG4_Pos                                    (4U)
#define DMA_HINTSTS_HTXIFLG4_Msk                                    (0x1UL << DMA_HINTSTS_HTXIFLG4_Pos)                            /*!< 0x00000010 */
#define DMA_HINTSTS_HTXIFLG4                                        DMA_HINTSTS_HTXIFLG4_Msk
#define DMA_HINTSTS_TXEIFLG4_Pos                                    (3U)
#define DMA_HINTSTS_TXEIFLG4_Msk                                    (0x1UL << DMA_HINTSTS_TXEIFLG4_Pos)                            /*!< 0x00000008 */
#define DMA_HINTSTS_TXEIFLG4                                        DMA_HINTSTS_TXEIFLG4_Msk
#define DMA_HINTSTS_DMEIFLG4_Pos                                    (2U)
#define DMA_HINTSTS_DMEIFLG4_Msk                                    (0x1UL << DMA_HINTSTS_DMEIFLG4_Pos)                            /*!< 0x00000004 */
#define DMA_HINTSTS_DMEIFLG4                                        DMA_HINTSTS_DMEIFLG4_Msk
#define DMA_HINTSTS_FEIFLG4_Pos                                     (0U)
#define DMA_HINTSTS_FEIFLG4_Msk                                     (0x1UL << DMA_HINTSTS_FEIFLG4_Pos)                             /*!< 0x00000001 */
#define DMA_HINTSTS_FEIFLG4                                         DMA_HINTSTS_FEIFLG4_Msk

/********************  Bits definition for DMA_LIFCLR register  ****************/
#define DMA_LIFCLR_CTXCIFLG3_Pos                                    (27U)
#define DMA_LIFCLR_CTXCIFLG3_Msk                                    (0x1UL << DMA_LIFCLR_CTXCIFLG3_Pos)                            /*!< 0x08000000 */
#define DMA_LIFCLR_CTXCIFLG3                                        DMA_LIFCLR_CTXCIFLG3_Msk
#define DMA_LIFCLR_CHTXIFLG3_Pos                                    (26U)
#define DMA_LIFCLR_CHTXIFLG3_Msk                                    (0x1UL << DMA_LIFCLR_CHTXIFLG3_Pos)                            /*!< 0x04000000 */
#define DMA_LIFCLR_CHTXIFLG3                                        DMA_LIFCLR_CHTXIFLG3_Msk
#define DMA_LIFCLR_CTXEIFLG3_Pos                                    (25U)
#define DMA_LIFCLR_CTXEIFLG3_Msk                                    (0x1UL << DMA_LIFCLR_CTXEIFLG3_Pos)                            /*!< 0x02000000 */
#define DMA_LIFCLR_CTXEIFLG3                                        DMA_LIFCLR_CTXEIFLG3_Msk
#define DMA_LIFCLR_CDMEIFLG3_Pos                                    (24U)
#define DMA_LIFCLR_CDMEIFLG3_Msk                                    (0x1UL << DMA_LIFCLR_CDMEIFLG3_Pos)                            /*!< 0x01000000 */
#define DMA_LIFCLR_CDMEIFLG3                                        DMA_LIFCLR_CDMEIFLG3_Msk
#define DMA_LIFCLR_CFEIFLG3_Pos                                     (22U)
#define DMA_LIFCLR_CFEIFLG3_Msk                                     (0x1UL << DMA_LIFCLR_CFEIFLG3_Pos)                             /*!< 0x00400000 */
#define DMA_LIFCLR_CFEIFLG3                                         DMA_LIFCLR_CFEIFLG3_Msk
#define DMA_LIFCLR_CTXCIFLG2_Pos                                    (21U)
#define DMA_LIFCLR_CTXCIFLG2_Msk                                    (0x1UL << DMA_LIFCLR_CTXCIFLG2_Pos)                            /*!< 0x00200000 */
#define DMA_LIFCLR_CTXCIFLG2                                        DMA_LIFCLR_CTXCIFLG2_Msk
#define DMA_LIFCLR_CHTXIFLG2_Pos                                    (20U)
#define DMA_LIFCLR_CHTXIFLG2_Msk                                    (0x1UL << DMA_LIFCLR_CHTXIFLG2_Pos)                            /*!< 0x00100000 */
#define DMA_LIFCLR_CHTXIFLG2                                        DMA_LIFCLR_CHTXIFLG2_Msk
#define DMA_LIFCLR_CTXEIFLG2_Pos                                    (19U)
#define DMA_LIFCLR_CTXEIFLG2_Msk                                    (0x1UL << DMA_LIFCLR_CTXEIFLG2_Pos)                            /*!< 0x00080000 */
#define DMA_LIFCLR_CTXEIFLG2                                        DMA_LIFCLR_CTXEIFLG2_Msk
#define DMA_LIFCLR_CDMEIFLG2_Pos                                    (18U)
#define DMA_LIFCLR_CDMEIFLG2_Msk                                    (0x1UL << DMA_LIFCLR_CDMEIFLG2_Pos)                            /*!< 0x00040000 */
#define DMA_LIFCLR_CDMEIFLG2                                        DMA_LIFCLR_CDMEIFLG2_Msk
#define DMA_LIFCLR_CFEIFLG2_Pos                                     (16U)
#define DMA_LIFCLR_CFEIFLG2_Msk                                     (0x1UL << DMA_LIFCLR_CFEIFLG2_Pos)                             /*!< 0x00010000 */
#define DMA_LIFCLR_CFEIFLG2                                         DMA_LIFCLR_CFEIFLG2_Msk
#define DMA_LIFCLR_CTXCIFLG1_Pos                                    (11U)
#define DMA_LIFCLR_CTXCIFLG1_Msk                                    (0x1UL << DMA_LIFCLR_CTXCIFLG1_Pos)                            /*!< 0x00000800 */
#define DMA_LIFCLR_CTXCIFLG1                                        DMA_LIFCLR_CTXCIFLG1_Msk
#define DMA_LIFCLR_CHTXIFLG1_Pos                                    (10U)
#define DMA_LIFCLR_CHTXIFLG1_Msk                                    (0x1UL << DMA_LIFCLR_CHTXIFLG1_Pos)                            /*!< 0x00000400 */
#define DMA_LIFCLR_CHTXIFLG1                                        DMA_LIFCLR_CHTXIFLG1_Msk
#define DMA_LIFCLR_CTXEIFLG1_Pos                                    (9U)
#define DMA_LIFCLR_CTXEIFLG1_Msk                                    (0x1UL << DMA_LIFCLR_CTXEIFLG1_Pos)                            /*!< 0x00000200 */
#define DMA_LIFCLR_CTXEIFLG1                                        DMA_LIFCLR_CTXEIFLG1_Msk
#define DMA_LIFCLR_CDMEIFLG1_Pos                                    (8U)
#define DMA_LIFCLR_CDMEIFLG1_Msk                                    (0x1UL << DMA_LIFCLR_CDMEIFLG1_Pos)                            /*!< 0x00000100 */
#define DMA_LIFCLR_CDMEIFLG1                                        DMA_LIFCLR_CDMEIFLG1_Msk
#define DMA_LIFCLR_CFEIFLG1_Pos                                     (6U)
#define DMA_LIFCLR_CFEIFLG1_Msk                                     (0x1UL << DMA_LIFCLR_CFEIFLG1_Pos)                             /*!< 0x00000040 */
#define DMA_LIFCLR_CFEIFLG1                                         DMA_LIFCLR_CFEIFLG1_Msk
#define DMA_LIFCLR_CTXCIFLG0_Pos                                    (5U)
#define DMA_LIFCLR_CTXCIFLG0_Msk                                    (0x1UL << DMA_LIFCLR_CTXCIFLG0_Pos)                            /*!< 0x00000020 */
#define DMA_LIFCLR_CTXCIFLG0                                        DMA_LIFCLR_CTXCIFLG0_Msk
#define DMA_LIFCLR_CHTXIFLG0_Pos                                    (4U)
#define DMA_LIFCLR_CHTXIFLG0_Msk                                    (0x1UL << DMA_LIFCLR_CHTXIFLG0_Pos)                            /*!< 0x00000010 */
#define DMA_LIFCLR_CHTXIFLG0                                        DMA_LIFCLR_CHTXIFLG0_Msk
#define DMA_LIFCLR_CTXEIFLG0_Pos                                    (3U)
#define DMA_LIFCLR_CTXEIFLG0_Msk                                    (0x1UL << DMA_LIFCLR_CTXEIFLG0_Pos)                            /*!< 0x00000008 */
#define DMA_LIFCLR_CTXEIFLG0                                        DMA_LIFCLR_CTXEIFLG0_Msk
#define DMA_LIFCLR_CDMEIFLG0_Pos                                    (2U)
#define DMA_LIFCLR_CDMEIFLG0_Msk                                    (0x1UL << DMA_LIFCLR_CDMEIFLG0_Pos)                            /*!< 0x00000004 */
#define DMA_LIFCLR_CDMEIFLG0                                        DMA_LIFCLR_CDMEIFLG0_Msk
#define DMA_LIFCLR_CFEIFLG0_Pos                                     (0U)
#define DMA_LIFCLR_CFEIFLG0_Msk                                     (0x1UL << DMA_LIFCLR_CFEIFLG0_Pos)                             /*!< 0x00000001 */
#define DMA_LIFCLR_CFEIFLG0                                         DMA_LIFCLR_CFEIFLG0_Msk

/********************  Bits definition for DMA_HIFCLR  register  ****************/
#define DMA_HIFCLR_CTXCIFLG7_Pos                                    (27U)
#define DMA_HIFCLR_CTXCIFLG7_Msk                                    (0x1UL << DMA_HIFCLR_CTXCIFLG7_Pos)                            /*!< 0x08000000 */
#define DMA_HIFCLR_CTXCIFLG7                                        DMA_HIFCLR_CTXCIFLG7_Msk
#define DMA_HIFCLR_CHTXIFLG7_Pos                                    (26U)
#define DMA_HIFCLR_CHTXIFLG7_Msk                                    (0x1UL << DMA_HIFCLR_CHTXIFLG7_Pos)                            /*!< 0x04000000 */
#define DMA_HIFCLR_CHTXIFLG7                                        DMA_HIFCLR_CHTXIFLG7_Msk
#define DMA_HIFCLR_CTXEIFLG7_Pos                                    (25U)
#define DMA_HIFCLR_CTXEIFLG7_Msk                                    (0x1UL << DMA_HIFCLR_CTXEIFLG7_Pos)                            /*!< 0x02000000 */
#define DMA_HIFCLR_CTXEIFLG7                                        DMA_HIFCLR_CTXEIFLG7_Msk
#define DMA_HIFCLR_CDMEIFLG7_Pos                                    (24U)
#define DMA_HIFCLR_CDMEIFLG7_Msk                                    (0x1UL << DMA_HIFCLR_CDMEIFLG7_Pos)                            /*!< 0x01000000 */
#define DMA_HIFCLR_CDMEIFLG7                                        DMA_HIFCLR_CDMEIFLG7_Msk
#define DMA_HIFCLR_CFEIFLG7_Pos                                     (22U)
#define DMA_HIFCLR_CFEIFLG7_Msk                                     (0x1UL << DMA_HIFCLR_CFEIFLG7_Pos)                             /*!< 0x00400000 */
#define DMA_HIFCLR_CFEIFLG7                                         DMA_HIFCLR_CFEIFLG7_Msk
#define DMA_HIFCLR_CTXCIFLG6_Pos                                    (21U)
#define DMA_HIFCLR_CTXCIFLG6_Msk                                    (0x1UL << DMA_HIFCLR_CTXCIFLG6_Pos)                            /*!< 0x00200000 */
#define DMA_HIFCLR_CTXCIFLG6                                        DMA_HIFCLR_CTXCIFLG6_Msk
#define DMA_HIFCLR_CHTXIFLG6_Pos                                    (20U)
#define DMA_HIFCLR_CHTXIFLG6_Msk                                    (0x1UL << DMA_HIFCLR_CHTXIFLG6_Pos)                            /*!< 0x00100000 */
#define DMA_HIFCLR_CHTXIFLG6                                        DMA_HIFCLR_CHTXIFLG6_Msk
#define DMA_HIFCLR_CTXEIFLG6_Pos                                    (19U)
#define DMA_HIFCLR_CTXEIFLG6_Msk                                    (0x1UL << DMA_HIFCLR_CTXEIFLG6_Pos)                            /*!< 0x00080000 */
#define DMA_HIFCLR_CTXEIFLG6                                        DMA_HIFCLR_CTXEIFLG6_Msk
#define DMA_HIFCLR_CDMEIFLG6_Pos                                    (18U)
#define DMA_HIFCLR_CDMEIFLG6_Msk                                    (0x1UL << DMA_HIFCLR_CDMEIFLG6_Pos)                            /*!< 0x00040000 */
#define DMA_HIFCLR_CDMEIFLG6                                        DMA_HIFCLR_CDMEIFLG6_Msk
#define DMA_HIFCLR_CFEIFLG6_Pos                                     (16U)
#define DMA_HIFCLR_CFEIFLG6_Msk                                     (0x1UL << DMA_HIFCLR_CFEIFLG6_Pos)                             /*!< 0x00010000 */
#define DMA_HIFCLR_CFEIFLG6                                         DMA_HIFCLR_CFEIFLG6_Msk
#define DMA_HIFCLR_CTXCIFLG5_Pos                                    (11U)
#define DMA_HIFCLR_CTXCIFLG5_Msk                                    (0x1UL << DMA_HIFCLR_CTXCIFLG5_Pos)                            /*!< 0x00000800 */
#define DMA_HIFCLR_CTXCIFLG5                                        DMA_HIFCLR_CTXCIFLG5_Msk
#define DMA_HIFCLR_CHTXIFLG5_Pos                                    (10U)
#define DMA_HIFCLR_CHTXIFLG5_Msk                                    (0x1UL << DMA_HIFCLR_CHTXIFLG5_Pos)                            /*!< 0x00000400 */
#define DMA_HIFCLR_CHTXIFLG5                                        DMA_HIFCLR_CHTXIFLG5_Msk
#define DMA_HIFCLR_CTXEIFLG5_Pos                                    (9U)
#define DMA_HIFCLR_CTXEIFLG5_Msk                                    (0x1UL << DMA_HIFCLR_CTXEIFLG5_Pos)                            /*!< 0x00000200 */
#define DMA_HIFCLR_CTXEIFLG5                                        DMA_HIFCLR_CTXEIFLG5_Msk
#define DMA_HIFCLR_CDMEIFLG5_Pos                                    (8U)
#define DMA_HIFCLR_CDMEIFLG5_Msk                                    (0x1UL << DMA_HIFCLR_CDMEIFLG5_Pos)                            /*!< 0x00000100 */
#define DMA_HIFCLR_CDMEIFLG5                                        DMA_HIFCLR_CDMEIFLG5_Msk
#define DMA_HIFCLR_CFEIFLG5_Pos                                     (6U)
#define DMA_HIFCLR_CFEIFLG5_Msk                                     (0x1UL << DMA_HIFCLR_CFEIFLG5_Pos)                             /*!< 0x00000040 */
#define DMA_HIFCLR_CFEIFLG5                                         DMA_HIFCLR_CFEIFLG5_Msk
#define DMA_HIFCLR_CTXCIFLG4_Pos                                    (5U)
#define DMA_HIFCLR_CTXCIFLG4_Msk                                    (0x1UL << DMA_HIFCLR_CTXCIFLG4_Pos)                            /*!< 0x00000020 */
#define DMA_HIFCLR_CTXCIFLG4                                        DMA_HIFCLR_CTXCIFLG4_Msk
#define DMA_HIFCLR_CHTXIFLG4_Pos                                    (4U)
#define DMA_HIFCLR_CHTXIFLG4_Msk                                    (0x1UL << DMA_HIFCLR_CHTXIFLG4_Pos)                            /*!< 0x00000010 */
#define DMA_HIFCLR_CHTXIFLG4                                        DMA_HIFCLR_CHTXIFLG4_Msk
#define DMA_HIFCLR_CTXEIFLG4_Pos                                    (3U)
#define DMA_HIFCLR_CTXEIFLG4_Msk                                    (0x1UL << DMA_HIFCLR_CTXEIFLG4_Pos)                            /*!< 0x00000008 */
#define DMA_HIFCLR_CTXEIFLG4                                        DMA_HIFCLR_CTXEIFLG4_Msk
#define DMA_HIFCLR_CDMEIFLG4_Pos                                    (2U)
#define DMA_HIFCLR_CDMEIFLG4_Msk                                    (0x1UL << DMA_HIFCLR_CDMEIFLG4_Pos)                            /*!< 0x00000004 */
#define DMA_HIFCLR_CDMEIFLG4                                        DMA_HIFCLR_CDMEIFLG4_Msk
#define DMA_HIFCLR_CFEIFLG4_Pos                                     (0U)
#define DMA_HIFCLR_CFEIFLG4_Msk                                     (0x1UL << DMA_HIFCLR_CFEIFLG4_Pos)                             /*!< 0x00000001 */
#define DMA_HIFCLR_CFEIFLG4                                         DMA_HIFCLR_CFEIFLG4_Msk

/******************  Bit definition for DMA_SxPAR register  ********************/
#define DMA_PADDRx_PADDR_Pos                                        (0U)
#define DMA_PADDRx_PADDR_Msk                                        (0xFFFFFFFFUL << DMA_PADDRx_PADDR_Pos)                         /*!< 0xFFFFFFFF */
#define DMA_PADDRx_PADDR                                            DMA_PADDRx_PADDR_Msk                                           /*!< Peripheral Address */

/******************  Bit definition for DMA_SxM0AR register  ********************/
#define DMA_M0ADDRx_M0ADDR_Pos                                      (0U)
#define DMA_M0ADDRx_M0ADDR_Msk                                      (0xFFFFFFFFUL << DMA_M0ADDRx_M0ADDR_Pos)                       /*!< 0xFFFFFFFF */
#define DMA_M0ADDRx_M0ADDR                                          DMA_M0ADDRx_M0ADDR_Msk                                         /*!< Memory Address */

/******************  Bit definition for DMA_SxM1AR register  ********************/
#define DMA_M1ADDRx_M1ADDR_Pos                                      (0U)
#define DMA_M1ADDRx_M1ADDR_Msk                                      (0xFFFFFFFFUL << DMA_M1ADDRx_M1ADDR_Pos)                       /*!< 0xFFFFFFFF */
#define DMA_M1ADDRx_M1ADDR                                          DMA_M1ADDRx_M1ADDR_Msk                                         /*!< Memory Address */


/******************************************************************************/
/*                                                                            */
/*                    External Interrupt/Event Controller                     */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for EINT_IMASK register  *******************/
#define EINT_IMASK_IMASK0_Pos                                       (0U)
#define EINT_IMASK_IMASK0_Msk                                       (0x1UL << EINT_IMASK_IMASK0_Pos)                               /*!< 0x00000001 */
#define EINT_IMASK_IMASK0                                           EINT_IMASK_IMASK0_Msk                                          /*!< Interrupt Mask on line 0 */
#define EINT_IMASK_IMASK1_Pos                                       (1U)
#define EINT_IMASK_IMASK1_Msk                                       (0x1UL << EINT_IMASK_IMASK1_Pos)                               /*!< 0x00000002 */
#define EINT_IMASK_IMASK1                                           EINT_IMASK_IMASK1_Msk                                          /*!< Interrupt Mask on line 1 */
#define EINT_IMASK_IMASK2_Pos                                       (2U)
#define EINT_IMASK_IMASK2_Msk                                       (0x1UL << EINT_IMASK_IMASK2_Pos)                               /*!< 0x00000004 */
#define EINT_IMASK_IMASK2                                           EINT_IMASK_IMASK2_Msk                                          /*!< Interrupt Mask on line 2 */
#define EINT_IMASK_IMASK3_Pos                                       (3U)
#define EINT_IMASK_IMASK3_Msk                                       (0x1UL << EINT_IMASK_IMASK3_Pos)                               /*!< 0x00000008 */
#define EINT_IMASK_IMASK3                                           EINT_IMASK_IMASK3_Msk                                          /*!< Interrupt Mask on line 3 */
#define EINT_IMASK_IMASK4_Pos                                       (4U)
#define EINT_IMASK_IMASK4_Msk                                       (0x1UL << EINT_IMASK_IMASK4_Pos)                               /*!< 0x00000010 */
#define EINT_IMASK_IMASK4                                           EINT_IMASK_IMASK4_Msk                                          /*!< Interrupt Mask on line 4 */
#define EINT_IMASK_IMASK5_Pos                                       (5U)
#define EINT_IMASK_IMASK5_Msk                                       (0x1UL << EINT_IMASK_IMASK5_Pos)                               /*!< 0x00000020 */
#define EINT_IMASK_IMASK5                                           EINT_IMASK_IMASK5_Msk                                          /*!< Interrupt Mask on line 5 */
#define EINT_IMASK_IMASK6_Pos                                       (6U)
#define EINT_IMASK_IMASK6_Msk                                       (0x1UL << EINT_IMASK_IMASK6_Pos)                               /*!< 0x00000040 */
#define EINT_IMASK_IMASK6                                           EINT_IMASK_IMASK6_Msk                                          /*!< Interrupt Mask on line 6 */
#define EINT_IMASK_IMASK7_Pos                                       (7U)
#define EINT_IMASK_IMASK7_Msk                                       (0x1UL << EINT_IMASK_IMASK7_Pos)                               /*!< 0x00000080 */
#define EINT_IMASK_IMASK7                                           EINT_IMASK_IMASK7_Msk                                          /*!< Interrupt Mask on line 7 */
#define EINT_IMASK_IMASK8_Pos                                       (8U)
#define EINT_IMASK_IMASK8_Msk                                       (0x1UL << EINT_IMASK_IMASK8_Pos)                               /*!< 0x00000100 */
#define EINT_IMASK_IMASK8                                           EINT_IMASK_IMASK8_Msk                                          /*!< Interrupt Mask on line 8 */
#define EINT_IMASK_IMASK9_Pos                                       (9U)
#define EINT_IMASK_IMASK9_Msk                                       (0x1UL << EINT_IMASK_IMASK9_Pos)                               /*!< 0x00000200 */
#define EINT_IMASK_IMASK9                                           EINT_IMASK_IMASK9_Msk                                          /*!< Interrupt Mask on line 9 */
#define EINT_IMASK_IMASK10_Pos                                      (10U)
#define EINT_IMASK_IMASK10_Msk                                      (0x1UL << EINT_IMASK_IMASK10_Pos)                              /*!< 0x00000400 */
#define EINT_IMASK_IMASK10                                          EINT_IMASK_IMASK10_Msk                                         /*!< Interrupt Mask on line 10 */
#define EINT_IMASK_IMASK11_Pos                                      (11U)
#define EINT_IMASK_IMASK11_Msk                                      (0x1UL << EINT_IMASK_IMASK11_Pos)                              /*!< 0x00000800 */
#define EINT_IMASK_IMASK11                                          EINT_IMASK_IMASK11_Msk                                         /*!< Interrupt Mask on line 11 */
#define EINT_IMASK_IMASK12_Pos                                      (12U)
#define EINT_IMASK_IMASK12_Msk                                      (0x1UL << EINT_IMASK_IMASK12_Pos)                              /*!< 0x00001000 */
#define EINT_IMASK_IMASK12                                          EINT_IMASK_IMASK12_Msk                                         /*!< Interrupt Mask on line 12 */
#define EINT_IMASK_IMASK13_Pos                                      (13U)
#define EINT_IMASK_IMASK13_Msk                                      (0x1UL << EINT_IMASK_IMASK13_Pos)                              /*!< 0x00002000 */
#define EINT_IMASK_IMASK13                                          EINT_IMASK_IMASK13_Msk                                         /*!< Interrupt Mask on line 13 */
#define EINT_IMASK_IMASK14_Pos                                      (14U)
#define EINT_IMASK_IMASK14_Msk                                      (0x1UL << EINT_IMASK_IMASK14_Pos)                              /*!< 0x00004000 */
#define EINT_IMASK_IMASK14                                          EINT_IMASK_IMASK14_Msk                                         /*!< Interrupt Mask on line 14 */
#define EINT_IMASK_IMASK15_Pos                                      (15U)
#define EINT_IMASK_IMASK15_Msk                                      (0x1UL << EINT_IMASK_IMASK15_Pos)                              /*!< 0x00008000 */
#define EINT_IMASK_IMASK15                                          EINT_IMASK_IMASK15_Msk                                         /*!< Interrupt Mask on line 15 */
#define EINT_IMASK_IMASK16_Pos                                      (16U)
#define EINT_IMASK_IMASK16_Msk                                      (0x1UL << EINT_IMASK_IMASK16_Pos)                              /*!< 0x00010000 */
#define EINT_IMASK_IMASK16                                          EINT_IMASK_IMASK16_Msk                                         /*!< Interrupt Mask on line 16 */
#define EINT_IMASK_IMASK17_Pos                                      (17U)
#define EINT_IMASK_IMASK17_Msk                                      (0x1UL << EINT_IMASK_IMASK17_Pos)                              /*!< 0x00020000 */
#define EINT_IMASK_IMASK17                                          EINT_IMASK_IMASK17_Msk                                         /*!< Interrupt Mask on line 17 */
#define EINT_IMASK_IMASK18_Pos                                      (18U)
#define EINT_IMASK_IMASK18_Msk                                      (0x1UL << EINT_IMASK_IMASK18_Pos)                              /*!< 0x00040000 */
#define EINT_IMASK_IMASK18                                          EINT_IMASK_IMASK18_Msk                                         /*!< Interrupt Mask on line 18 */
#define EINT_IMASK_IMASK19_Pos                                      (19U)
#define EINT_IMASK_IMASK19_Msk                                      (0x1UL << EINT_IMASK_IMASK19_Pos)                              /*!< 0x00080000 */
#define EINT_IMASK_IMASK19                                          EINT_IMASK_IMASK19_Msk                                         /*!< Interrupt Mask on line 19 */
#define EINT_IMASK_IMASK20_Pos                                      (20U)
#define EINT_IMASK_IMASK20_Msk                                      (0x1UL << EINT_IMASK_IMASK20_Pos)                              /*!< 0x00100000 */
#define EINT_IMASK_IMASK20                                          EINT_IMASK_IMASK20_Msk                                         /*!< Interrupt Mask on line 20 */
#define EINT_IMASK_IMASK21_Pos                                      (21U)
#define EINT_IMASK_IMASK21_Msk                                      (0x1UL << EINT_IMASK_IMASK21_Pos)                              /*!< 0x00200000 */
#define EINT_IMASK_IMASK21                                          EINT_IMASK_IMASK21_Msk                                         /*!< Interrupt Mask on line 21 */
#define EINT_IMASK_IMASK22_Pos                                      (22U)
#define EINT_IMASK_IMASK22_Msk                                      (0x1UL << EINT_IMASK_IMASK22_Pos)                              /*!< 0x00400000 */
#define EINT_IMASK_IMASK22                                          EINT_IMASK_IMASK22_Msk                                         /*!< Interrupt Mask on line 22 */

/* Reference Defines */
#define EINT_IMASK_IM0                                              EINT_IMASK_IMASK0
#define EINT_IMASK_IM1                                              EINT_IMASK_IMASK1
#define EINT_IMASK_IM2                                              EINT_IMASK_IMASK2
#define EINT_IMASK_IM3                                              EINT_IMASK_IMASK3
#define EINT_IMASK_IM4                                              EINT_IMASK_IMASK4
#define EINT_IMASK_IM5                                              EINT_IMASK_IMASK5
#define EINT_IMASK_IM6                                              EINT_IMASK_IMASK6
#define EINT_IMASK_IM7                                              EINT_IMASK_IMASK7
#define EINT_IMASK_IM8                                              EINT_IMASK_IMASK8
#define EINT_IMASK_IM9                                              EINT_IMASK_IMASK9
#define EINT_IMASK_IM10                                             EINT_IMASK_IMASK10
#define EINT_IMASK_IM11                                             EINT_IMASK_IMASK11
#define EINT_IMASK_IM12                                             EINT_IMASK_IMASK12
#define EINT_IMASK_IM13                                             EINT_IMASK_IMASK13
#define EINT_IMASK_IM14                                             EINT_IMASK_IMASK14
#define EINT_IMASK_IM15                                             EINT_IMASK_IMASK15
#define EINT_IMASK_IM16                                             EINT_IMASK_IMASK16
#define EINT_IMASK_IM17                                             EINT_IMASK_IMASK17
#define EINT_IMASK_IM18                                             EINT_IMASK_IMASK18
#define EINT_IMASK_IM19                                             EINT_IMASK_IMASK19
#define EINT_IMASK_IM20                                             EINT_IMASK_IMASK20
#define EINT_IMASK_IM21                                             EINT_IMASK_IMASK21
#define EINT_IMASK_IM22                                             EINT_IMASK_IMASK22
#define EINT_IMASK_IM_Pos                                           (0U)
#define EINT_IMASK_IM_Msk                                           (0x7FFFFFUL << EINT_IMASK_IM_Pos)                              /*!< 0x007FFFFF */
#define EINT_IMASK_IM                                               EINT_IMASK_IM_Msk                                              /*!< Interrupt Mask All */

/*******************  Bit definition for EINT_EMASK register  *******************/
#define EINT_EMASK_EMASK0_Pos                                       (0U)
#define EINT_EMASK_EMASK0_Msk                                       (0x1UL << EINT_EMASK_EMASK0_Pos)                               /*!< 0x00000001 */
#define EINT_EMASK_EMASK0                                           EINT_EMASK_EMASK0_Msk                                          /*!< Event Mask on line 0 */
#define EINT_EMASK_EMASK1_Pos                                       (1U)
#define EINT_EMASK_EMASK1_Msk                                       (0x1UL << EINT_EMASK_EMASK1_Pos)                               /*!< 0x00000002 */
#define EINT_EMASK_EMASK1                                           EINT_EMASK_EMASK1_Msk                                          /*!< Event Mask on line 1 */
#define EINT_EMASK_EMASK2_Pos                                       (2U)
#define EINT_EMASK_EMASK2_Msk                                       (0x1UL << EINT_EMASK_EMASK2_Pos)                               /*!< 0x00000004 */
#define EINT_EMASK_EMASK2                                           EINT_EMASK_EMASK2_Msk                                          /*!< Event Mask on line 2 */
#define EINT_EMASK_EMASK3_Pos                                       (3U)
#define EINT_EMASK_EMASK3_Msk                                       (0x1UL << EINT_EMASK_EMASK3_Pos)                               /*!< 0x00000008 */
#define EINT_EMASK_EMASK3                                           EINT_EMASK_EMASK3_Msk                                          /*!< Event Mask on line 3 */
#define EINT_EMASK_EMASK4_Pos                                       (4U)
#define EINT_EMASK_EMASK4_Msk                                       (0x1UL << EINT_EMASK_EMASK4_Pos)                               /*!< 0x00000010 */
#define EINT_EMASK_EMASK4                                           EINT_EMASK_EMASK4_Msk                                          /*!< Event Mask on line 4 */
#define EINT_EMASK_EMASK5_Pos                                       (5U)
#define EINT_EMASK_EMASK5_Msk                                       (0x1UL << EINT_EMASK_EMASK5_Pos)                               /*!< 0x00000020 */
#define EINT_EMASK_EMASK5                                           EINT_EMASK_EMASK5_Msk                                          /*!< Event Mask on line 5 */
#define EINT_EMASK_EMASK6_Pos                                       (6U)
#define EINT_EMASK_EMASK6_Msk                                       (0x1UL << EINT_EMASK_EMASK6_Pos)                               /*!< 0x00000040 */
#define EINT_EMASK_EMASK6                                           EINT_EMASK_EMASK6_Msk                                          /*!< Event Mask on line 6 */
#define EINT_EMASK_EMASK7_Pos                                       (7U)
#define EINT_EMASK_EMASK7_Msk                                       (0x1UL << EINT_EMASK_EMASK7_Pos)                               /*!< 0x00000080 */
#define EINT_EMASK_EMASK7                                           EINT_EMASK_EMASK7_Msk                                          /*!< Event Mask on line 7 */
#define EINT_EMASK_EMASK8_Pos                                       (8U)
#define EINT_EMASK_EMASK8_Msk                                       (0x1UL << EINT_EMASK_EMASK8_Pos)                               /*!< 0x00000100 */
#define EINT_EMASK_EMASK8                                           EINT_EMASK_EMASK8_Msk                                          /*!< Event Mask on line 8 */
#define EINT_EMASK_EMASK9_Pos                                       (9U)
#define EINT_EMASK_EMASK9_Msk                                       (0x1UL << EINT_EMASK_EMASK9_Pos)                               /*!< 0x00000200 */
#define EINT_EMASK_EMASK9                                           EINT_EMASK_EMASK9_Msk                                          /*!< Event Mask on line 9 */
#define EINT_EMASK_EMASK10_Pos                                      (10U)
#define EINT_EMASK_EMASK10_Msk                                      (0x1UL << EINT_EMASK_EMASK10_Pos)                              /*!< 0x00000400 */
#define EINT_EMASK_EMASK10                                          EINT_EMASK_EMASK10_Msk                                         /*!< Event Mask on line 10 */
#define EINT_EMASK_EMASK11_Pos                                      (11U)
#define EINT_EMASK_EMASK11_Msk                                      (0x1UL << EINT_EMASK_EMASK11_Pos)                              /*!< 0x00000800 */
#define EINT_EMASK_EMASK11                                          EINT_EMASK_EMASK11_Msk                                         /*!< Event Mask on line 11 */
#define EINT_EMASK_EMASK12_Pos                                      (12U)
#define EINT_EMASK_EMASK12_Msk                                      (0x1UL << EINT_EMASK_EMASK12_Pos)                              /*!< 0x00001000 */
#define EINT_EMASK_EMASK12                                          EINT_EMASK_EMASK12_Msk                                         /*!< Event Mask on line 12 */
#define EINT_EMASK_EMASK13_Pos                                      (13U)
#define EINT_EMASK_EMASK13_Msk                                      (0x1UL << EINT_EMASK_EMASK13_Pos)                              /*!< 0x00002000 */
#define EINT_EMASK_EMASK13                                          EINT_EMASK_EMASK13_Msk                                         /*!< Event Mask on line 13 */
#define EINT_EMASK_EMASK14_Pos                                      (14U)
#define EINT_EMASK_EMASK14_Msk                                      (0x1UL << EINT_EMASK_EMASK14_Pos)                              /*!< 0x00004000 */
#define EINT_EMASK_EMASK14                                          EINT_EMASK_EMASK14_Msk                                         /*!< Event Mask on line 14 */
#define EINT_EMASK_EMASK15_Pos                                      (15U)
#define EINT_EMASK_EMASK15_Msk                                      (0x1UL << EINT_EMASK_EMASK15_Pos)                              /*!< 0x00008000 */
#define EINT_EMASK_EMASK15                                          EINT_EMASK_EMASK15_Msk                                         /*!< Event Mask on line 15 */
#define EINT_EMASK_EMASK16_Pos                                      (16U)
#define EINT_EMASK_EMASK16_Msk                                      (0x1UL << EINT_EMASK_EMASK16_Pos)                              /*!< 0x00010000 */
#define EINT_EMASK_EMASK16                                          EINT_EMASK_EMASK16_Msk                                         /*!< Event Mask on line 16 */
#define EINT_EMASK_EMASK17_Pos                                      (17U)
#define EINT_EMASK_EMASK17_Msk                                      (0x1UL << EINT_EMASK_EMASK17_Pos)                              /*!< 0x00020000 */
#define EINT_EMASK_EMASK17                                          EINT_EMASK_EMASK17_Msk                                         /*!< Event Mask on line 17 */
#define EINT_EMASK_EMASK18_Pos                                      (18U)
#define EINT_EMASK_EMASK18_Msk                                      (0x1UL << EINT_EMASK_EMASK18_Pos)                              /*!< 0x00040000 */
#define EINT_EMASK_EMASK18                                          EINT_EMASK_EMASK18_Msk                                         /*!< Event Mask on line 18 */
#define EINT_EMASK_EMASK19_Pos                                      (19U)
#define EINT_EMASK_EMASK19_Msk                                      (0x1UL << EINT_EMASK_EMASK19_Pos)                              /*!< 0x00080000 */
#define EINT_EMASK_EMASK19                                          EINT_EMASK_EMASK19_Msk                                         /*!< Event Mask on line 19 */
#define EINT_EMASK_EMASK20_Pos                                      (20U)
#define EINT_EMASK_EMASK20_Msk                                      (0x1UL << EINT_EMASK_EMASK20_Pos)                              /*!< 0x00100000 */
#define EINT_EMASK_EMASK20                                          EINT_EMASK_EMASK20_Msk                                         /*!< Event Mask on line 20 */
#define EINT_EMASK_EMASK21_Pos                                      (21U)
#define EINT_EMASK_EMASK21_Msk                                      (0x1UL << EINT_EMASK_EMASK21_Pos)                              /*!< 0x00200000 */
#define EINT_EMASK_EMASK21                                          EINT_EMASK_EMASK21_Msk                                         /*!< Event Mask on line 21 */
#define EINT_EMASK_EMASK22_Pos                                      (22U)
#define EINT_EMASK_EMASK22_Msk                                      (0x1UL << EINT_EMASK_EMASK22_Pos)                              /*!< 0x00400000 */
#define EINT_EMASK_EMASK22                                          EINT_EMASK_EMASK22_Msk                                         /*!< Event Mask on line 22 */

/* Reference Defines */
#define EINT_EMASK_EM0                                              EINT_EMASK_EMASK0
#define EINT_EMASK_EM1                                              EINT_EMASK_EMASK1
#define EINT_EMASK_EM2                                              EINT_EMASK_EMASK2
#define EINT_EMASK_EM3                                              EINT_EMASK_EMASK3
#define EINT_EMASK_EM4                                              EINT_EMASK_EMASK4
#define EINT_EMASK_EM5                                              EINT_EMASK_EMASK5
#define EINT_EMASK_EM6                                              EINT_EMASK_EMASK6
#define EINT_EMASK_EM7                                              EINT_EMASK_EMASK7
#define EINT_EMASK_EM8                                              EINT_EMASK_EMASK8
#define EINT_EMASK_EM9                                              EINT_EMASK_EMASK9
#define EINT_EMASK_EM10                                             EINT_EMASK_EMASK10
#define EINT_EMASK_EM11                                             EINT_EMASK_EMASK11
#define EINT_EMASK_EM12                                             EINT_EMASK_EMASK12
#define EINT_EMASK_EM13                                             EINT_EMASK_EMASK13
#define EINT_EMASK_EM14                                             EINT_EMASK_EMASK14
#define EINT_EMASK_EM15                                             EINT_EMASK_EMASK15
#define EINT_EMASK_EM16                                             EINT_EMASK_EMASK16
#define EINT_EMASK_EM17                                             EINT_EMASK_EMASK17
#define EINT_EMASK_EM18                                             EINT_EMASK_EMASK18
#define EINT_EMASK_EM19                                             EINT_EMASK_EMASK19
#define EINT_EMASK_EM20                                             EINT_EMASK_EMASK20
#define EINT_EMASK_EM21                                             EINT_EMASK_EMASK21
#define EINT_EMASK_EM22                                             EINT_EMASK_EMASK22

/******************  Bit definition for EINT_RTEN register  *******************/
#define EINT_RTEN_RTEN0_Pos                                         (0U)
#define EINT_RTEN_RTEN0_Msk                                         (0x1UL << EINT_RTEN_RTEN0_Pos)                                 /*!< 0x00000001 */
#define EINT_RTEN_RTEN0                                             EINT_RTEN_RTEN0_Msk                                            /*!< Rising trigger event configuration bit of line 0 */
#define EINT_RTEN_RTEN1_Pos                                         (1U)
#define EINT_RTEN_RTEN1_Msk                                         (0x1UL << EINT_RTEN_RTEN1_Pos)                                 /*!< 0x00000002 */
#define EINT_RTEN_RTEN1                                             EINT_RTEN_RTEN1_Msk                                            /*!< Rising trigger event configuration bit of line 1 */
#define EINT_RTEN_RTEN2_Pos                                         (2U)
#define EINT_RTEN_RTEN2_Msk                                         (0x1UL << EINT_RTEN_RTEN2_Pos)                                 /*!< 0x00000004 */
#define EINT_RTEN_RTEN2                                             EINT_RTEN_RTEN2_Msk                                            /*!< Rising trigger event configuration bit of line 2 */
#define EINT_RTEN_RTEN3_Pos                                         (3U)
#define EINT_RTEN_RTEN3_Msk                                         (0x1UL << EINT_RTEN_RTEN3_Pos)                                 /*!< 0x00000008 */
#define EINT_RTEN_RTEN3                                             EINT_RTEN_RTEN3_Msk                                            /*!< Rising trigger event configuration bit of line 3 */
#define EINT_RTEN_RTEN4_Pos                                         (4U)
#define EINT_RTEN_RTEN4_Msk                                         (0x1UL << EINT_RTEN_RTEN4_Pos)                                 /*!< 0x00000010 */
#define EINT_RTEN_RTEN4                                             EINT_RTEN_RTEN4_Msk                                            /*!< Rising trigger event configuration bit of line 4 */
#define EINT_RTEN_RTEN5_Pos                                         (5U)
#define EINT_RTEN_RTEN5_Msk                                         (0x1UL << EINT_RTEN_RTEN5_Pos)                                 /*!< 0x00000020 */
#define EINT_RTEN_RTEN5                                             EINT_RTEN_RTEN5_Msk                                            /*!< Rising trigger event configuration bit of line 5 */
#define EINT_RTEN_RTEN6_Pos                                         (6U)
#define EINT_RTEN_RTEN6_Msk                                         (0x1UL << EINT_RTEN_RTEN6_Pos)                                 /*!< 0x00000040 */
#define EINT_RTEN_RTEN6                                             EINT_RTEN_RTEN6_Msk                                            /*!< Rising trigger event configuration bit of line 6 */
#define EINT_RTEN_RTEN7_Pos                                         (7U)
#define EINT_RTEN_RTEN7_Msk                                         (0x1UL << EINT_RTEN_RTEN7_Pos)                                 /*!< 0x00000080 */
#define EINT_RTEN_RTEN7                                             EINT_RTEN_RTEN7_Msk                                            /*!< Rising trigger event configuration bit of line 7 */
#define EINT_RTEN_RTEN8_Pos                                         (8U)
#define EINT_RTEN_RTEN8_Msk                                         (0x1UL << EINT_RTEN_RTEN8_Pos)                                 /*!< 0x00000100 */
#define EINT_RTEN_RTEN8                                             EINT_RTEN_RTEN8_Msk                                            /*!< Rising trigger event configuration bit of line 8 */
#define EINT_RTEN_RTEN9_Pos                                         (9U)
#define EINT_RTEN_RTEN9_Msk                                         (0x1UL << EINT_RTEN_RTEN9_Pos)                                 /*!< 0x00000200 */
#define EINT_RTEN_RTEN9                                             EINT_RTEN_RTEN9_Msk                                            /*!< Rising trigger event configuration bit of line 9 */
#define EINT_RTEN_RTEN10_Pos                                        (10U)
#define EINT_RTEN_RTEN10_Msk                                        (0x1UL << EINT_RTEN_RTEN10_Pos)                                /*!< 0x00000400 */
#define EINT_RTEN_RTEN10                                            EINT_RTEN_RTEN10_Msk                                           /*!< Rising trigger event configuration bit of line 10 */
#define EINT_RTEN_RTEN11_Pos                                        (11U)
#define EINT_RTEN_RTEN11_Msk                                        (0x1UL << EINT_RTEN_RTEN11_Pos)                                /*!< 0x00000800 */
#define EINT_RTEN_RTEN11                                            EINT_RTEN_RTEN11_Msk                                           /*!< Rising trigger event configuration bit of line 11 */
#define EINT_RTEN_RTEN12_Pos                                        (12U)
#define EINT_RTEN_RTEN12_Msk                                        (0x1UL << EINT_RTEN_RTEN12_Pos)                                /*!< 0x00001000 */
#define EINT_RTEN_RTEN12                                            EINT_RTEN_RTEN12_Msk                                           /*!< Rising trigger event configuration bit of line 12 */
#define EINT_RTEN_RTEN13_Pos                                        (13U)
#define EINT_RTEN_RTEN13_Msk                                        (0x1UL << EINT_RTEN_RTEN13_Pos)                                /*!< 0x00002000 */
#define EINT_RTEN_RTEN13                                            EINT_RTEN_RTEN13_Msk                                           /*!< Rising trigger event configuration bit of line 13 */
#define EINT_RTEN_RTEN14_Pos                                        (14U)
#define EINT_RTEN_RTEN14_Msk                                        (0x1UL << EINT_RTEN_RTEN14_Pos)                                /*!< 0x00004000 */
#define EINT_RTEN_RTEN14                                            EINT_RTEN_RTEN14_Msk                                           /*!< Rising trigger event configuration bit of line 14 */
#define EINT_RTEN_RTEN15_Pos                                        (15U)
#define EINT_RTEN_RTEN15_Msk                                        (0x1UL << EINT_RTEN_RTEN15_Pos)                                /*!< 0x00008000 */
#define EINT_RTEN_RTEN15                                            EINT_RTEN_RTEN15_Msk                                           /*!< Rising trigger event configuration bit of line 15 */
#define EINT_RTEN_RTEN16_Pos                                        (16U)
#define EINT_RTEN_RTEN16_Msk                                        (0x1UL << EINT_RTEN_RTEN16_Pos)                                /*!< 0x00010000 */
#define EINT_RTEN_RTEN16                                            EINT_RTEN_RTEN16_Msk                                           /*!< Rising trigger event configuration bit of line 16 */
#define EINT_RTEN_RTEN17_Pos                                        (17U)
#define EINT_RTEN_RTEN17_Msk                                        (0x1UL << EINT_RTEN_RTEN17_Pos)                                /*!< 0x00020000 */
#define EINT_RTEN_RTEN17                                            EINT_RTEN_RTEN17_Msk                                           /*!< Rising trigger event configuration bit of line 17 */
#define EINT_RTEN_RTEN18_Pos                                        (18U)
#define EINT_RTEN_RTEN18_Msk                                        (0x1UL << EINT_RTEN_RTEN18_Pos)                                /*!< 0x00040000 */
#define EINT_RTEN_RTEN18                                            EINT_RTEN_RTEN18_Msk                                           /*!< Rising trigger event configuration bit of line 18 */
#define EINT_RTEN_RTEN19_Pos                                        (19U)
#define EINT_RTEN_RTEN19_Msk                                        (0x1UL << EINT_RTEN_RTEN19_Pos)                                /*!< 0x00080000 */
#define EINT_RTEN_RTEN19                                            EINT_RTEN_RTEN19_Msk                                           /*!< Rising trigger event configuration bit of line 19 */
#define EINT_RTEN_RTEN20_Pos                                        (20U)
#define EINT_RTEN_RTEN20_Msk                                        (0x1UL << EINT_RTEN_RTEN20_Pos)                                /*!< 0x00100000 */
#define EINT_RTEN_RTEN20                                            EINT_RTEN_RTEN20_Msk                                           /*!< Rising trigger event configuration bit of line 20 */
#define EINT_RTEN_RTEN21_Pos                                        (21U)
#define EINT_RTEN_RTEN21_Msk                                        (0x1UL << EINT_RTEN_RTEN21_Pos)                                /*!< 0x00200000 */
#define EINT_RTEN_RTEN21                                            EINT_RTEN_RTEN21_Msk                                           /*!< Rising trigger event configuration bit of line 21 */
#define EINT_RTEN_RTEN22_Pos                                        (22U)
#define EINT_RTEN_RTEN22_Msk                                        (0x1UL << EINT_RTEN_RTEN22_Pos)                                /*!< 0x00400000 */
#define EINT_RTEN_RTEN22                                            EINT_RTEN_RTEN22_Msk                                           /*!< Rising trigger event configuration bit of line 22 */

/******************  Bit definition for EINT_FTEN register  *******************/
#define EINT_FTEN_FTEN0_Pos                                         (0U)
#define EINT_FTEN_FTEN0_Msk                                         (0x1UL << EINT_FTEN_FTEN0_Pos)                                 /*!< 0x00000001 */
#define EINT_FTEN_FTEN0                                             EINT_FTEN_FTEN0_Msk                                            /*!< Falling trigger event configuration bit of line 0 */
#define EINT_FTEN_FTEN1_Pos                                         (1U)
#define EINT_FTEN_FTEN1_Msk                                         (0x1UL << EINT_FTEN_FTEN1_Pos)                                 /*!< 0x00000002 */
#define EINT_FTEN_FTEN1                                             EINT_FTEN_FTEN1_Msk                                            /*!< Falling trigger event configuration bit of line 1 */
#define EINT_FTEN_FTEN2_Pos                                         (2U)
#define EINT_FTEN_FTEN2_Msk                                         (0x1UL << EINT_FTEN_FTEN2_Pos)                                 /*!< 0x00000004 */
#define EINT_FTEN_FTEN2                                             EINT_FTEN_FTEN2_Msk                                            /*!< Falling trigger event configuration bit of line 2 */
#define EINT_FTEN_FTEN3_Pos                                         (3U)
#define EINT_FTEN_FTEN3_Msk                                         (0x1UL << EINT_FTEN_FTEN3_Pos)                                 /*!< 0x00000008 */
#define EINT_FTEN_FTEN3                                             EINT_FTEN_FTEN3_Msk                                            /*!< Falling trigger event configuration bit of line 3 */
#define EINT_FTEN_FTEN4_Pos                                         (4U)
#define EINT_FTEN_FTEN4_Msk                                         (0x1UL << EINT_FTEN_FTEN4_Pos)                                 /*!< 0x00000010 */
#define EINT_FTEN_FTEN4                                             EINT_FTEN_FTEN4_Msk                                            /*!< Falling trigger event configuration bit of line 4 */
#define EINT_FTEN_FTEN5_Pos                                         (5U)
#define EINT_FTEN_FTEN5_Msk                                         (0x1UL << EINT_FTEN_FTEN5_Pos)                                 /*!< 0x00000020 */
#define EINT_FTEN_FTEN5                                             EINT_FTEN_FTEN5_Msk                                            /*!< Falling trigger event configuration bit of line 5 */
#define EINT_FTEN_FTEN6_Pos                                         (6U)
#define EINT_FTEN_FTEN6_Msk                                         (0x1UL << EINT_FTEN_FTEN6_Pos)                                 /*!< 0x00000040 */
#define EINT_FTEN_FTEN6                                             EINT_FTEN_FTEN6_Msk                                            /*!< Falling trigger event configuration bit of line 6 */
#define EINT_FTEN_FTEN7_Pos                                         (7U)
#define EINT_FTEN_FTEN7_Msk                                         (0x1UL << EINT_FTEN_FTEN7_Pos)                                 /*!< 0x00000080 */
#define EINT_FTEN_FTEN7                                             EINT_FTEN_FTEN7_Msk                                            /*!< Falling trigger event configuration bit of line 7 */
#define EINT_FTEN_FTEN8_Pos                                         (8U)
#define EINT_FTEN_FTEN8_Msk                                         (0x1UL << EINT_FTEN_FTEN8_Pos)                                 /*!< 0x00000100 */
#define EINT_FTEN_FTEN8                                             EINT_FTEN_FTEN8_Msk                                            /*!< Falling trigger event configuration bit of line 8 */
#define EINT_FTEN_FTEN9_Pos                                         (9U)
#define EINT_FTEN_FTEN9_Msk                                         (0x1UL << EINT_FTEN_FTEN9_Pos)                                 /*!< 0x00000200 */
#define EINT_FTEN_FTEN9                                             EINT_FTEN_FTEN9_Msk                                            /*!< Falling trigger event configuration bit of line 9 */
#define EINT_FTEN_FTEN10_Pos                                        (10U)
#define EINT_FTEN_FTEN10_Msk                                        (0x1UL << EINT_FTEN_FTEN10_Pos)                                /*!< 0x00000400 */
#define EINT_FTEN_FTEN10                                            EINT_FTEN_FTEN10_Msk                                           /*!< Falling trigger event configuration bit of line 10 */
#define EINT_FTEN_FTEN11_Pos                                        (11U)
#define EINT_FTEN_FTEN11_Msk                                        (0x1UL << EINT_FTEN_FTEN11_Pos)                                /*!< 0x00000800 */
#define EINT_FTEN_FTEN11                                            EINT_FTEN_FTEN11_Msk                                           /*!< Falling trigger event configuration bit of line 11 */
#define EINT_FTEN_FTEN12_Pos                                        (12U)
#define EINT_FTEN_FTEN12_Msk                                        (0x1UL << EINT_FTEN_FTEN12_Pos)                                /*!< 0x00001000 */
#define EINT_FTEN_FTEN12                                            EINT_FTEN_FTEN12_Msk                                           /*!< Falling trigger event configuration bit of line 12 */
#define EINT_FTEN_FTEN13_Pos                                        (13U)
#define EINT_FTEN_FTEN13_Msk                                        (0x1UL << EINT_FTEN_FTEN13_Pos)                                /*!< 0x00002000 */
#define EINT_FTEN_FTEN13                                            EINT_FTEN_FTEN13_Msk                                           /*!< Falling trigger event configuration bit of line 13 */
#define EINT_FTEN_FTEN14_Pos                                        (14U)
#define EINT_FTEN_FTEN14_Msk                                        (0x1UL << EINT_FTEN_FTEN14_Pos)                                /*!< 0x00004000 */
#define EINT_FTEN_FTEN14                                            EINT_FTEN_FTEN14_Msk                                           /*!< Falling trigger event configuration bit of line 14 */
#define EINT_FTEN_FTEN15_Pos                                        (15U)
#define EINT_FTEN_FTEN15_Msk                                        (0x1UL << EINT_FTEN_FTEN15_Pos)                                /*!< 0x00008000 */
#define EINT_FTEN_FTEN15                                            EINT_FTEN_FTEN15_Msk                                           /*!< Falling trigger event configuration bit of line 15 */
#define EINT_FTEN_FTEN16_Pos                                        (16U)
#define EINT_FTEN_FTEN16_Msk                                        (0x1UL << EINT_FTEN_FTEN16_Pos)                                /*!< 0x00010000 */
#define EINT_FTEN_FTEN16                                            EINT_FTEN_FTEN16_Msk                                           /*!< Falling trigger event configuration bit of line 16 */
#define EINT_FTEN_FTEN17_Pos                                        (17U)
#define EINT_FTEN_FTEN17_Msk                                        (0x1UL << EINT_FTEN_FTEN17_Pos)                                /*!< 0x00020000 */
#define EINT_FTEN_FTEN17                                            EINT_FTEN_FTEN17_Msk                                           /*!< Falling trigger event configuration bit of line 17 */
#define EINT_FTEN_FTEN18_Pos                                        (18U)
#define EINT_FTEN_FTEN18_Msk                                        (0x1UL << EINT_FTEN_FTEN18_Pos)                                /*!< 0x00040000 */
#define EINT_FTEN_FTEN18                                            EINT_FTEN_FTEN18_Msk                                           /*!< Falling trigger event configuration bit of line 18 */
#define EINT_FTEN_FTEN19_Pos                                        (19U)
#define EINT_FTEN_FTEN19_Msk                                        (0x1UL << EINT_FTEN_FTEN19_Pos)                                /*!< 0x00080000 */
#define EINT_FTEN_FTEN19                                            EINT_FTEN_FTEN19_Msk                                           /*!< Falling trigger event configuration bit of line 19 */
#define EINT_FTEN_FTEN20_Pos                                        (20U)
#define EINT_FTEN_FTEN20_Msk                                        (0x1UL << EINT_FTEN_FTEN20_Pos)                                /*!< 0x00100000 */
#define EINT_FTEN_FTEN20                                            EINT_FTEN_FTEN20_Msk                                           /*!< Falling trigger event configuration bit of line 20 */
#define EINT_FTEN_FTEN21_Pos                                        (21U)
#define EINT_FTEN_FTEN21_Msk                                        (0x1UL << EINT_FTEN_FTEN21_Pos)                                /*!< 0x00200000 */
#define EINT_FTEN_FTEN21                                            EINT_FTEN_FTEN21_Msk                                           /*!< Falling trigger event configuration bit of line 21 */
#define EINT_FTEN_FTEN22_Pos                                        (22U)
#define EINT_FTEN_FTEN22_Msk                                        (0x1UL << EINT_FTEN_FTEN22_Pos)                                /*!< 0x00400000 */
#define EINT_FTEN_FTEN22                                            EINT_FTEN_FTEN22_Msk                                           /*!< Falling trigger event configuration bit of line 22 */

/******************  Bit definition for EINT_SWINTE register  ******************/
#define EINT_SWINTE_SWINTE0_Pos                                     (0U)
#define EINT_SWINTE_SWINTE0_Msk                                     (0x1UL << EINT_SWINTE_SWINTE0_Pos)                             /*!< 0x00000001 */
#define EINT_SWINTE_SWINTE0                                         EINT_SWINTE_SWINTE0_Msk                                        /*!< Software Interrupt on line 0 */
#define EINT_SWINTE_SWINTE1_Pos                                     (1U)
#define EINT_SWINTE_SWINTE1_Msk                                     (0x1UL << EINT_SWINTE_SWINTE1_Pos)                             /*!< 0x00000002 */
#define EINT_SWINTE_SWINTE1                                         EINT_SWINTE_SWINTE1_Msk                                        /*!< Software Interrupt on line 1 */
#define EINT_SWINTE_SWINTE2_Pos                                     (2U)
#define EINT_SWINTE_SWINTE2_Msk                                     (0x1UL << EINT_SWINTE_SWINTE2_Pos)                             /*!< 0x00000004 */
#define EINT_SWINTE_SWINTE2                                         EINT_SWINTE_SWINTE2_Msk                                        /*!< Software Interrupt on line 2 */
#define EINT_SWINTE_SWINTE3_Pos                                     (3U)
#define EINT_SWINTE_SWINTE3_Msk                                     (0x1UL << EINT_SWINTE_SWINTE3_Pos)                             /*!< 0x00000008 */
#define EINT_SWINTE_SWINTE3                                         EINT_SWINTE_SWINTE3_Msk                                        /*!< Software Interrupt on line 3 */
#define EINT_SWINTE_SWINTE4_Pos                                     (4U)
#define EINT_SWINTE_SWINTE4_Msk                                     (0x1UL << EINT_SWINTE_SWINTE4_Pos)                             /*!< 0x00000010 */
#define EINT_SWINTE_SWINTE4                                         EINT_SWINTE_SWINTE4_Msk                                        /*!< Software Interrupt on line 4 */
#define EINT_SWINTE_SWINTE5_Pos                                     (5U)
#define EINT_SWINTE_SWINTE5_Msk                                     (0x1UL << EINT_SWINTE_SWINTE5_Pos)                             /*!< 0x00000020 */
#define EINT_SWINTE_SWINTE5                                         EINT_SWINTE_SWINTE5_Msk                                        /*!< Software Interrupt on line 5 */
#define EINT_SWINTE_SWINTE6_Pos                                     (6U)
#define EINT_SWINTE_SWINTE6_Msk                                     (0x1UL << EINT_SWINTE_SWINTE6_Pos)                             /*!< 0x00000040 */
#define EINT_SWINTE_SWINTE6                                         EINT_SWINTE_SWINTE6_Msk                                        /*!< Software Interrupt on line 6 */
#define EINT_SWINTE_SWINTE7_Pos                                     (7U)
#define EINT_SWINTE_SWINTE7_Msk                                     (0x1UL << EINT_SWINTE_SWINTE7_Pos)                             /*!< 0x00000080 */
#define EINT_SWINTE_SWINTE7                                         EINT_SWINTE_SWINTE7_Msk                                        /*!< Software Interrupt on line 7 */
#define EINT_SWINTE_SWINTE8_Pos                                     (8U)
#define EINT_SWINTE_SWINTE8_Msk                                     (0x1UL << EINT_SWINTE_SWINTE8_Pos)                             /*!< 0x00000100 */
#define EINT_SWINTE_SWINTE8                                         EINT_SWINTE_SWINTE8_Msk                                        /*!< Software Interrupt on line 8 */
#define EINT_SWINTE_SWINTE9_Pos                                     (9U)
#define EINT_SWINTE_SWINTE9_Msk                                     (0x1UL << EINT_SWINTE_SWINTE9_Pos)                             /*!< 0x00000200 */
#define EINT_SWINTE_SWINTE9                                         EINT_SWINTE_SWINTE9_Msk                                        /*!< Software Interrupt on line 9 */
#define EINT_SWINTE_SWINTE10_Pos                                    (10U)
#define EINT_SWINTE_SWINTE10_Msk                                    (0x1UL << EINT_SWINTE_SWINTE10_Pos)                            /*!< 0x00000400 */
#define EINT_SWINTE_SWINTE10                                        EINT_SWINTE_SWINTE10_Msk                                       /*!< Software Interrupt on line 10 */
#define EINT_SWINTE_SWINTE11_Pos                                    (11U)
#define EINT_SWINTE_SWINTE11_Msk                                    (0x1UL << EINT_SWINTE_SWINTE11_Pos)                            /*!< 0x00000800 */
#define EINT_SWINTE_SWINTE11                                        EINT_SWINTE_SWINTE11_Msk                                       /*!< Software Interrupt on line 11 */
#define EINT_SWINTE_SWINTE12_Pos                                    (12U)
#define EINT_SWINTE_SWINTE12_Msk                                    (0x1UL << EINT_SWINTE_SWINTE12_Pos)                            /*!< 0x00001000 */
#define EINT_SWINTE_SWINTE12                                        EINT_SWINTE_SWINTE12_Msk                                       /*!< Software Interrupt on line 12 */
#define EINT_SWINTE_SWINTE13_Pos                                    (13U)
#define EINT_SWINTE_SWINTE13_Msk                                    (0x1UL << EINT_SWINTE_SWINTE13_Pos)                            /*!< 0x00002000 */
#define EINT_SWINTE_SWINTE13                                        EINT_SWINTE_SWINTE13_Msk                                       /*!< Software Interrupt on line 13 */
#define EINT_SWINTE_SWINTE14_Pos                                    (14U)
#define EINT_SWINTE_SWINTE14_Msk                                    (0x1UL << EINT_SWINTE_SWINTE14_Pos)                            /*!< 0x00004000 */
#define EINT_SWINTE_SWINTE14                                        EINT_SWINTE_SWINTE14_Msk                                       /*!< Software Interrupt on line 14 */
#define EINT_SWINTE_SWINTE15_Pos                                    (15U)
#define EINT_SWINTE_SWINTE15_Msk                                    (0x1UL << EINT_SWINTE_SWINTE15_Pos)                            /*!< 0x00008000 */
#define EINT_SWINTE_SWINTE15                                        EINT_SWINTE_SWINTE15_Msk                                       /*!< Software Interrupt on line 15 */
#define EINT_SWINTE_SWINTE16_Pos                                    (16U)
#define EINT_SWINTE_SWINTE16_Msk                                    (0x1UL << EINT_SWINTE_SWINTE16_Pos)                            /*!< 0x00010000 */
#define EINT_SWINTE_SWINTE16                                        EINT_SWINTE_SWINTE16_Msk                                       /*!< Software Interrupt on line 16 */
#define EINT_SWINTE_SWINTE17_Pos                                    (17U)
#define EINT_SWINTE_SWINTE17_Msk                                    (0x1UL << EINT_SWINTE_SWINTE17_Pos)                            /*!< 0x00020000 */
#define EINT_SWINTE_SWINTE17                                        EINT_SWINTE_SWINTE17_Msk                                       /*!< Software Interrupt on line 17 */
#define EINT_SWINTE_SWINTE18_Pos                                    (18U)
#define EINT_SWINTE_SWINTE18_Msk                                    (0x1UL << EINT_SWINTE_SWINTE18_Pos)                            /*!< 0x00040000 */
#define EINT_SWINTE_SWINTE18                                        EINT_SWINTE_SWINTE18_Msk                                       /*!< Software Interrupt on line 18 */
#define EINT_SWINTE_SWINTE19_Pos                                    (19U)
#define EINT_SWINTE_SWINTE19_Msk                                    (0x1UL << EINT_SWINTE_SWINTE19_Pos)                            /*!< 0x00080000 */
#define EINT_SWINTE_SWINTE19                                        EINT_SWINTE_SWINTE19_Msk                                       /*!< Software Interrupt on line 19 */
#define EINT_SWINTE_SWINTE20_Pos                                    (20U)
#define EINT_SWINTE_SWINTE20_Msk                                    (0x1UL << EINT_SWINTE_SWINTE20_Pos)                            /*!< 0x00100000 */
#define EINT_SWINTE_SWINTE20                                        EINT_SWINTE_SWINTE20_Msk                                       /*!< Software Interrupt on line 20 */
#define EINT_SWINTE_SWINTE21_Pos                                    (21U)
#define EINT_SWINTE_SWINTE21_Msk                                    (0x1UL << EINT_SWINTE_SWINTE21_Pos)                            /*!< 0x00200000 */
#define EINT_SWINTE_SWINTE21                                        EINT_SWINTE_SWINTE21_Msk                                       /*!< Software Interrupt on line 21 */
#define EINT_SWINTE_SWINTE22_Pos                                    (22U)
#define EINT_SWINTE_SWINTE22_Msk                                    (0x1UL << EINT_SWINTE_SWINTE22_Pos)                            /*!< 0x00400000 */
#define EINT_SWINTE_SWINTE22                                        EINT_SWINTE_SWINTE22_Msk                                       /*!< Software Interrupt on line 22 */

/*******************  Bit definition for EINT_IPEND register  ********************/
#define EINT_IPEND_IPEND0_Pos                                       (0U)
#define EINT_IPEND_IPEND0_Msk                                       (0x1UL << EINT_IPEND_IPEND0_Pos)                               /*!< 0x00000001 */
#define EINT_IPEND_IPEND0                                           EINT_IPEND_IPEND0_Msk                                          /*!< Pending bit for line 0 */
#define EINT_IPEND_IPEND1_Pos                                       (1U)
#define EINT_IPEND_IPEND1_Msk                                       (0x1UL << EINT_IPEND_IPEND1_Pos)                               /*!< 0x00000002 */
#define EINT_IPEND_IPEND1                                           EINT_IPEND_IPEND1_Msk                                          /*!< Pending bit for line 1 */
#define EINT_IPEND_IPEND2_Pos                                       (2U)
#define EINT_IPEND_IPEND2_Msk                                       (0x1UL << EINT_IPEND_IPEND2_Pos)                               /*!< 0x00000004 */
#define EINT_IPEND_IPEND2                                           EINT_IPEND_IPEND2_Msk                                          /*!< Pending bit for line 2 */
#define EINT_IPEND_IPEND3_Pos                                       (3U)
#define EINT_IPEND_IPEND3_Msk                                       (0x1UL << EINT_IPEND_IPEND3_Pos)                               /*!< 0x00000008 */
#define EINT_IPEND_IPEND3                                           EINT_IPEND_IPEND3_Msk                                          /*!< Pending bit for line 3 */
#define EINT_IPEND_IPEND4_Pos                                       (4U)
#define EINT_IPEND_IPEND4_Msk                                       (0x1UL << EINT_IPEND_IPEND4_Pos)                               /*!< 0x00000010 */
#define EINT_IPEND_IPEND4                                           EINT_IPEND_IPEND4_Msk                                          /*!< Pending bit for line 4 */
#define EINT_IPEND_IPEND5_Pos                                       (5U)
#define EINT_IPEND_IPEND5_Msk                                       (0x1UL << EINT_IPEND_IPEND5_Pos)                               /*!< 0x00000020 */
#define EINT_IPEND_IPEND5                                           EINT_IPEND_IPEND5_Msk                                          /*!< Pending bit for line 5 */
#define EINT_IPEND_IPEND6_Pos                                       (6U)
#define EINT_IPEND_IPEND6_Msk                                       (0x1UL << EINT_IPEND_IPEND6_Pos)                               /*!< 0x00000040 */
#define EINT_IPEND_IPEND6                                           EINT_IPEND_IPEND6_Msk                                          /*!< Pending bit for line 6 */
#define EINT_IPEND_IPEND7_Pos                                       (7U)
#define EINT_IPEND_IPEND7_Msk                                       (0x1UL << EINT_IPEND_IPEND7_Pos)                               /*!< 0x00000080 */
#define EINT_IPEND_IPEND7                                           EINT_IPEND_IPEND7_Msk                                          /*!< Pending bit for line 7 */
#define EINT_IPEND_IPEND8_Pos                                       (8U)
#define EINT_IPEND_IPEND8_Msk                                       (0x1UL << EINT_IPEND_IPEND8_Pos)                               /*!< 0x00000100 */
#define EINT_IPEND_IPEND8                                           EINT_IPEND_IPEND8_Msk                                          /*!< Pending bit for line 8 */
#define EINT_IPEND_IPEND9_Pos                                       (9U)
#define EINT_IPEND_IPEND9_Msk                                       (0x1UL << EINT_IPEND_IPEND9_Pos)                               /*!< 0x00000200 */
#define EINT_IPEND_IPEND9                                           EINT_IPEND_IPEND9_Msk                                          /*!< Pending bit for line 9 */
#define EINT_IPEND_IPEND10_Pos                                      (10U)
#define EINT_IPEND_IPEND10_Msk                                      (0x1UL << EINT_IPEND_IPEND10_Pos)                              /*!< 0x00000400 */
#define EINT_IPEND_IPEND10                                          EINT_IPEND_IPEND10_Msk                                         /*!< Pending bit for line 10 */
#define EINT_IPEND_IPEND11_Pos                                      (11U)
#define EINT_IPEND_IPEND11_Msk                                      (0x1UL << EINT_IPEND_IPEND11_Pos)                              /*!< 0x00000800 */
#define EINT_IPEND_IPEND11                                          EINT_IPEND_IPEND11_Msk                                         /*!< Pending bit for line 11 */
#define EINT_IPEND_IPEND12_Pos                                      (12U)
#define EINT_IPEND_IPEND12_Msk                                      (0x1UL << EINT_IPEND_IPEND12_Pos)                              /*!< 0x00001000 */
#define EINT_IPEND_IPEND12                                          EINT_IPEND_IPEND12_Msk                                         /*!< Pending bit for line 12 */
#define EINT_IPEND_IPEND13_Pos                                      (13U)
#define EINT_IPEND_IPEND13_Msk                                      (0x1UL << EINT_IPEND_IPEND13_Pos)                              /*!< 0x00002000 */
#define EINT_IPEND_IPEND13                                          EINT_IPEND_IPEND13_Msk                                         /*!< Pending bit for line 13 */
#define EINT_IPEND_IPEND14_Pos                                      (14U)
#define EINT_IPEND_IPEND14_Msk                                      (0x1UL << EINT_IPEND_IPEND14_Pos)                              /*!< 0x00004000 */
#define EINT_IPEND_IPEND14                                          EINT_IPEND_IPEND14_Msk                                         /*!< Pending bit for line 14 */
#define EINT_IPEND_IPEND15_Pos                                      (15U)
#define EINT_IPEND_IPEND15_Msk                                      (0x1UL << EINT_IPEND_IPEND15_Pos)                              /*!< 0x00008000 */
#define EINT_IPEND_IPEND15                                          EINT_IPEND_IPEND15_Msk                                         /*!< Pending bit for line 15 */
#define EINT_IPEND_IPEND16_Pos                                      (16U)
#define EINT_IPEND_IPEND16_Msk                                      (0x1UL << EINT_IPEND_IPEND16_Pos)                              /*!< 0x00010000 */
#define EINT_IPEND_IPEND16                                          EINT_IPEND_IPEND16_Msk                                         /*!< Pending bit for line 16 */
#define EINT_IPEND_IPEND17_Pos                                      (17U)
#define EINT_IPEND_IPEND17_Msk                                      (0x1UL << EINT_IPEND_IPEND17_Pos)                              /*!< 0x00020000 */
#define EINT_IPEND_IPEND17                                          EINT_IPEND_IPEND17_Msk                                         /*!< Pending bit for line 17 */
#define EINT_IPEND_IPEND18_Pos                                      (18U)
#define EINT_IPEND_IPEND18_Msk                                      (0x1UL << EINT_IPEND_IPEND18_Pos)                              /*!< 0x00040000 */
#define EINT_IPEND_IPEND18                                          EINT_IPEND_IPEND18_Msk                                         /*!< Pending bit for line 18 */
#define EINT_IPEND_IPEND19_Pos                                      (19U)
#define EINT_IPEND_IPEND19_Msk                                      (0x1UL << EINT_IPEND_IPEND19_Pos)                              /*!< 0x00080000 */
#define EINT_IPEND_IPEND19                                          EINT_IPEND_IPEND19_Msk                                         /*!< Pending bit for line 19 */
#define EINT_IPEND_IPEND20_Pos                                      (20U)
#define EINT_IPEND_IPEND20_Msk                                      (0x1UL << EINT_IPEND_IPEND20_Pos)                              /*!< 0x00100000 */
#define EINT_IPEND_IPEND20                                          EINT_IPEND_IPEND20_Msk                                         /*!< Pending bit for line 20 */
#define EINT_IPEND_IPEND21_Pos                                      (21U)
#define EINT_IPEND_IPEND21_Msk                                      (0x1UL << EINT_IPEND_IPEND21_Pos)                              /*!< 0x00200000 */
#define EINT_IPEND_IPEND21                                          EINT_IPEND_IPEND21_Msk                                         /*!< Pending bit for line 21 */
#define EINT_IPEND_IPEND22_Pos                                      (22U)
#define EINT_IPEND_IPEND22_Msk                                      (0x1UL << EINT_IPEND_IPEND22_Pos)                              /*!< 0x00400000 */
#define EINT_IPEND_IPEND22                                          EINT_IPEND_IPEND22_Msk                                         /*!< Pending bit for line 22 */

/******************************************************************************/
/*                                                                            */
/*                                    FLASH                                   */
/*                                                                            */
/******************************************************************************/
/*******************  Bits definition for FMC_ACCTRL register  *****************/
#define FLASH_ACCTRL_WAITP_Pos                                      (0U)
#define FLASH_ACCTRL_WAITP_Msk                                      (0xFUL << FLASH_ACCTRL_WAITP_Pos)                              /*!< 0x0000000F */
#define FLASH_ACCTRL_WAITP                                          FLASH_ACCTRL_WAITP_Msk
#define FLASH_ACCTRL_WAITP_0WS                                      0x00000000U
#define FLASH_ACCTRL_WAITP_1WS                                      0x00000001U
#define FLASH_ACCTRL_WAITP_2WS                                      0x00000002U
#define FLASH_ACCTRL_WAITP_3WS                                      0x00000003U
#define FLASH_ACCTRL_WAITP_4WS                                      0x00000004U
#define FLASH_ACCTRL_WAITP_5WS                                      0x00000005U
#define FLASH_ACCTRL_WAITP_6WS                                      0x00000006U
#define FLASH_ACCTRL_WAITP_7WS                                      0x00000007U
#define FLASH_ACCTRL_WAITP_8WS                                      0x00000008U
#define FLASH_ACCTRL_WAITP_9WS                                      0x00000009U
#define FLASH_ACCTRL_WAITP_10WS                                     0x0000000AU
#define FLASH_ACCTRL_WAITP_11WS                                     0x0000000BU
#define FLASH_ACCTRL_WAITP_12WS                                     0x0000000CU
#define FLASH_ACCTRL_WAITP_13WS                                     0x0000000DU
#define FLASH_ACCTRL_WAITP_14WS                                     0x0000000EU
#define FLASH_ACCTRL_WAITP_15WS                                     0x0000000FU

#define FLASH_ACCTRL_PREFEN_Pos                                     (8U)
#define FLASH_ACCTRL_PREFEN_Msk                                     (0x1UL << FLASH_ACCTRL_PREFEN_Pos)                             /*!< 0x00000100 */
#define FLASH_ACCTRL_PREFEN                                         FLASH_ACCTRL_PREFEN_Msk
#define FLASH_ACCTRL_ICACHEEN_Pos                                   (9U)
#define FLASH_ACCTRL_ICACHEEN_Msk                                   (0x1UL << FLASH_ACCTRL_ICACHEEN_Pos)                           /*!< 0x00000200 */
#define FLASH_ACCTRL_ICACHEEN                                       FLASH_ACCTRL_ICACHEEN_Msk
#define FLASH_ACCTRL_DCACHEEN_Pos                                   (10U)
#define FLASH_ACCTRL_DCACHEEN_Msk                                   (0x1UL << FLASH_ACCTRL_DCACHEEN_Pos)                           /*!< 0x00000400 */
#define FLASH_ACCTRL_DCACHEEN                                       FLASH_ACCTRL_DCACHEEN_Msk
#define FLASH_ACCTRL_ICACHERST_Pos                                  (11U)
#define FLASH_ACCTRL_ICACHERST_Msk                                  (0x1UL << FLASH_ACCTRL_ICACHERST_Pos)                          /*!< 0x00000800 */
#define FLASH_ACCTRL_ICACHERST                                      FLASH_ACCTRL_ICACHERST_Msk
#define FLASH_ACCTRL_DCACHERST_Pos                                  (12U)
#define FLASH_ACCTRL_DCACHERST_Msk                                  (0x1UL << FLASH_ACCTRL_DCACHERST_Pos)                          /*!< 0x00001000 */
#define FLASH_ACCTRL_DCACHERST                                      FLASH_ACCTRL_DCACHERST_Msk
#define FLASH_ACCTRL_BYTE0_ADDRESS_Pos                              (10U)
#define FLASH_ACCTRL_BYTE0_ADDRESS_Msk                              (0x10008FUL << FLASH_ACCTRL_BYTE0_ADDRESS_Pos)                 /*!< 0x40023C00 */
#define FLASH_ACCTRL_BYTE0_ADDRESS                                  FLASH_ACCTRL_BYTE0_ADDRESS_Msk
#define FLASH_ACCTRL_BYTE2_ADDRESS_Pos                              (0U)
#define FLASH_ACCTRL_BYTE2_ADDRESS_Msk                              (0x40023C03UL << FLASH_ACCTRL_BYTE2_ADDRESS_Pos)               /*!< 0x40023C03 */
#define FLASH_ACCTRL_BYTE2_ADDRESS                                  FLASH_ACCTRL_BYTE2_ADDRESS_Msk

/*******************  Bits definition for FMC_STS register  ******************/
#define FLASH_STS_OPRCMP_Pos                                        (0U)
#define FLASH_STS_OPRCMP_Msk                                        (0x1UL << FLASH_STS_OPRCMP_Pos)                                /*!< 0x00000001 */
#define FLASH_STS_OPRCMP                                            FLASH_STS_OPRCMP_Msk
#define FLASH_STS_OPRERR_Pos                                        (1U)
#define FLASH_STS_OPRERR_Msk                                        (0x1UL << FLASH_STS_OPRERR_Pos)                                /*!< 0x00000002 */
#define FLASH_STS_OPRERR                                            FLASH_STS_OPRERR_Msk
#define FLASH_STS_WPROTERR_Pos                                      (4U)
#define FLASH_STS_WPROTERR_Msk                                      (0x1UL << FLASH_STS_WPROTERR_Pos)                              /*!< 0x00000010 */
#define FLASH_STS_WPROTERR                                          FLASH_STS_WPROTERR_Msk
#define FLASH_STS_PGALGERR_Pos                                      (5U)
#define FLASH_STS_PGALGERR_Msk                                      (0x1UL << FLASH_STS_PGALGERR_Pos)                              /*!< 0x00000020 */
#define FLASH_STS_PGALGERR                                          FLASH_STS_PGALGERR_Msk
#define FLASH_STS_PGPRLERR_Pos                                      (6U)
#define FLASH_STS_PGPRLERR_Msk                                      (0x1UL << FLASH_STS_PGPRLERR_Pos)                              /*!< 0x00000040 */
#define FLASH_STS_PGPRLERR                                          FLASH_STS_PGPRLERR_Msk
#define FLASH_STS_PGSEQERR_Pos                                      (7U)
#define FLASH_STS_PGSEQERR_Msk                                      (0x1UL << FLASH_STS_PGSEQERR_Pos)                              /*!< 0x00000080 */
#define FLASH_STS_PGSEQERR                                          FLASH_STS_PGSEQERR_Msk
#define FLASH_STS_RPROERR_Pos                                       (8U)
#define FLASH_STS_RPROERR_Msk                                       (0x1UL << FLASH_STS_RPROERR_Pos)                               /*!< 0x00000100 */
#define FLASH_STS_RPROERR                                           FLASH_STS_RPROERR_Msk
#define FLASH_STS_BUSY_Pos                                          (16U)
#define FLASH_STS_BUSY_Msk                                          (0x1UL << FLASH_STS_BUSY_Pos)                                  /*!< 0x00010000 */
#define FLASH_STS_BUSY                                              FLASH_STS_BUSY_Msk

/*******************  Bits definition for FMC_CTRL register  ******************/
#define FLASH_CTRL_PG_Pos                                           (0U)
#define FLASH_CTRL_PG_Msk                                           (0x1UL << FLASH_CTRL_PG_Pos)                                   /*!< 0x00000001 */
#define FLASH_CTRL_PG                                               FLASH_CTRL_PG_Msk
#define FLASH_CTRL_SERS_Pos                                         (1U)
#define FLASH_CTRL_SERS_Msk                                         (0x1UL << FLASH_CTRL_SERS_Pos)                                 /*!< 0x00000002 */
#define FLASH_CTRL_SERS                                             FLASH_CTRL_SERS_Msk
#define FLASH_CTRL_MERS_Pos                                         (2U)
#define FLASH_CTRL_MERS_Msk                                         (0x1UL << FLASH_CTRL_MERS_Pos)                                 /*!< 0x00000004 */
#define FLASH_CTRL_MERS                                             FLASH_CTRL_MERS_Msk
#define FLASH_CTRL_SNUM_Pos                                         (3U)
#define FLASH_CTRL_SNUM_Msk                                         (0x1FUL << FLASH_CTRL_SNUM_Pos)                                /*!< 0x000000F8 */
#define FLASH_CTRL_SNUM                                             FLASH_CTRL_SNUM_Msk
#define FLASH_CTRL_SNUM_0                                           (0x01UL << FLASH_CTRL_SNUM_Pos)                                /*!< 0x00000008 */
#define FLASH_CTRL_SNUM_1                                           (0x02UL << FLASH_CTRL_SNUM_Pos)                                /*!< 0x00000010 */
#define FLASH_CTRL_SNUM_2                                           (0x04UL << FLASH_CTRL_SNUM_Pos)                                /*!< 0x00000020 */
#define FLASH_CTRL_SNUM_3                                           (0x08UL << FLASH_CTRL_SNUM_Pos)                                /*!< 0x00000040 */
#define FLASH_CTRL_SNUM_4                                           (0x10UL << FLASH_CTRL_SNUM_Pos)                                /*!< 0x00000080 */
#define FLASH_CTRL_PGSIZE_Pos                                       (8U)
#define FLASH_CTRL_PGSIZE_Msk                                       (0x3UL << FLASH_CTRL_PGSIZE_Pos)                               /*!< 0x00000300 */
#define FLASH_CTRL_PGSIZE                                           FLASH_CTRL_PGSIZE_Msk
#define FLASH_CTRL_PGSIZE_0                                         (0x1UL << FLASH_CTRL_PGSIZE_Pos)                               /*!< 0x00000100 */
#define FLASH_CTRL_PGSIZE_1                                         (0x2UL << FLASH_CTRL_PGSIZE_Pos)                               /*!< 0x00000200 */
#define FLASH_CTRL_START_Pos                                        (16U)
#define FLASH_CTRL_START_Msk                                        (0x1UL << FLASH_CTRL_START_Pos)                                /*!< 0x00010000 */
#define FLASH_CTRL_START                                            FLASH_CTRL_START_Msk
#define FLASH_CTRL_OPCINTEN_Pos                                     (24U)
#define FLASH_CTRL_OPCINTEN_Msk                                     (0x1UL << FLASH_CTRL_OPCINTEN_Pos)                             /*!< 0x01000000 */
#define FLASH_CTRL_OPCINTEN                                         FLASH_CTRL_OPCINTEN_Msk
#define FLASH_CTRL_ERRINTEN_Pos                                     (25U)
#define FLASH_CTRL_ERRINTEN_Msk                                     (0x1UL << FLASH_CTRL_ERRINTEN_Pos)
#define FLASH_CTRL_ERRINTEN                                         FLASH_CTRL_ERRINTEN_Msk
#define FLASH_CTRL_LOCK_Pos                                         (31U)
#define FLASH_CTRL_LOCK_Msk                                         (0x1UL << FLASH_CTRL_LOCK_Pos)                                 /*!< 0x80000000 */
#define FLASH_CTRL_LOCK                                             FLASH_CTRL_LOCK_Msk

/*******************  Bits definition for FMC_OPTCTRL register  ***************/
#define FLASH_OPTCTRL_OPTLOCK_Pos                                   (0U)
#define FLASH_OPTCTRL_OPTLOCK_Msk                                   (0x1UL << FLASH_OPTCTRL_OPTLOCK_Pos)                           /*!< 0x00000001 */
#define FLASH_OPTCTRL_OPTLOCK                                       FLASH_OPTCTRL_OPTLOCK_Msk
#define FLASH_OPTCTRL_OPTSTART_Pos                                  (1U)
#define FLASH_OPTCTRL_OPTSTART_Msk                                  (0x1UL << FLASH_OPTCTRL_OPTSTART_Pos)                          /*!< 0x00000002 */
#define FLASH_OPTCTRL_OPTSTART                                      FLASH_OPTCTRL_OPTSTART_Msk

#define FLASH_OPTCTRL_BORLVL_0                                      0x00000004U
#define FLASH_OPTCTRL_BORLVL_1                                      0x00000008U
#define FLASH_OPTCTRL_BORLVL_Pos                                    (2U)
#define FLASH_OPTCTRL_BORLVL_Msk                                    (0x3UL << FLASH_OPTCTRL_BORLVL_Pos)                            /*!< 0x0000000C */
#define FLASH_OPTCTRL_BORLVL                                        FLASH_OPTCTRL_BORLVL_Msk
#define FLASH_OPTCTRL_WDTSEL_Pos                                    (5U)
#define FLASH_OPTCTRL_WDTSEL_Msk                                    (0x1UL << FLASH_OPTCTRL_WDTSEL_Pos)                            /*!< 0x00000020 */
#define FLASH_OPTCTRL_WDTSEL                                        FLASH_OPTCTRL_WDTSEL_Msk
#define FLASH_OPTCTRL_RSTSTOP_Pos                                   (6U)
#define FLASH_OPTCTRL_RSTSTOP_Msk                                   (0x1UL << FLASH_OPTCTRL_RSTSTOP_Pos)                           /*!< 0x00000040 */
#define FLASH_OPTCTRL_RSTSTOP                                       FLASH_OPTCTRL_RSTSTOP_Msk
#define FLASH_OPTCTRL_RSTSTDB_Pos                                   (7U)
#define FLASH_OPTCTRL_RSTSTDB_Msk                                   (0x1UL << FLASH_OPTCTRL_RSTSTDB_Pos)                           /*!< 0x00000080 */
#define FLASH_OPTCTRL_RSTSTDB                                       FLASH_OPTCTRL_RSTSTDB_Msk
#define FLASH_OPTCTRL_RPROT_Pos                                     (8U)
#define FLASH_OPTCTRL_RPROT_Msk                                     (0xFFUL << FLASH_OPTCTRL_RPROT_Pos)                            /*!< 0x0000FF00 */
#define FLASH_OPTCTRL_RPROT                                         FLASH_OPTCTRL_RPROT_Msk
#define FLASH_OPTCTRL_RPROT_0                                       (0x01UL << FLASH_OPTCTRL_RPROT_Pos)                            /*!< 0x00000100 */
#define FLASH_OPTCTRL_RPROT_1                                       (0x02UL << FLASH_OPTCTRL_RPROT_Pos)                            /*!< 0x00000200 */
#define FLASH_OPTCTRL_RPROT_2                                       (0x04UL << FLASH_OPTCTRL_RPROT_Pos)                            /*!< 0x00000400 */
#define FLASH_OPTCTRL_RPROT_3                                       (0x08UL << FLASH_OPTCTRL_RPROT_Pos)                            /*!< 0x00000800 */
#define FLASH_OPTCTRL_RPROT_4                                       (0x10UL << FLASH_OPTCTRL_RPROT_Pos)                            /*!< 0x00001000 */
#define FLASH_OPTCTRL_RPROT_5                                       (0x20UL << FLASH_OPTCTRL_RPROT_Pos)                            /*!< 0x00002000 */
#define FLASH_OPTCTRL_RPROT_6                                       (0x40UL << FLASH_OPTCTRL_RPROT_Pos)                            /*!< 0x00004000 */
#define FLASH_OPTCTRL_RPROT_7                                       (0x80UL << FLASH_OPTCTRL_RPROT_Pos)                            /*!< 0x00008000 */
#define FLASH_OPTCTRL_NWPROT_Pos                                    (16U)
#define FLASH_OPTCTRL_NWPROT_Msk                                    (0xFFUL << FLASH_OPTCTRL_NWPROT_Pos)                          /*!< 0x0FFF0000 */
#define FLASH_OPTCTRL_NWPROT                                        FLASH_OPTCTRL_NWPROT_Msk
#define FLASH_OPTCTRL_NWPROT_0                                      0x00010000U
#define FLASH_OPTCTRL_NWPROT_1                                      0x00020000U
#define FLASH_OPTCTRL_NWPROT_2                                      0x00040000U
#define FLASH_OPTCTRL_NWPROT_3                                      0x00080000U
#define FLASH_OPTCTRL_NWPROT_4                                      0x00100000U
#define FLASH_OPTCTRL_NWPROT_5                                      0x00200000U
#define FLASH_OPTCTRL_NWPROT_6                                      0x00400000U
#define FLASH_OPTCTRL_NWPROT_7                                      0x00800000U
#define FLASH_OPTCTRL_PCROPEN_Pos                                   (31U)
#define FLASH_OPTCTRL_PCROPEN_Msk                                   (0x1UL << FLASH_OPTCTRL_PCROPEN_Pos)                           /*!< 0x80000000 */
#define FLASH_OPTCTRL_PCROPEN                                       FLASH_OPTCTRL_PCROPEN_Msk

/******************  Bits definition for FMC_OPTCTRL1 register  ***************/
#define FLASH_OPTCTRL1_NWPROT_Pos                                   (16U)
#define FLASH_OPTCTRL1_NWPROT_Msk                                   (0xFFFUL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x0FFF0000 */
#define FLASH_OPTCTRL1_NWPROT                                       FLASH_OPTCTRL1_NWPROT_Msk
#define FLASH_OPTCTRL1_NWPROT_0                                     (0x001UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x00010000 */
#define FLASH_OPTCTRL1_NWPROT_1                                     (0x002UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x00020000 */
#define FLASH_OPTCTRL1_NWPROT_2                                     (0x004UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x00040000 */
#define FLASH_OPTCTRL1_NWPROT_3                                     (0x008UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x00080000 */
#define FLASH_OPTCTRL1_NWPROT_4                                     (0x010UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x00100000 */
#define FLASH_OPTCTRL1_NWPROT_5                                     (0x020UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x00200000 */
#define FLASH_OPTCTRL1_NWPROT_6                                     (0x040UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x00400000 */
#define FLASH_OPTCTRL1_NWPROT_7                                     (0x080UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x00800000 */
#define FLASH_OPTCTRL1_NWPROT_8                                     (0x100UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x01000000 */
#define FLASH_OPTCTRL1_NWPROT_9                                     (0x200UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x02000000 */
#define FLASH_OPTCTRL1_NWPROT_10                                    (0x400UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x04000000 */
#define FLASH_OPTCTRL1_NWPROT_11                                    (0x800UL << FLASH_OPTCTRL1_NWPROT_Pos)                         /*!< 0x08000000 */

/******************************************************************************/
/*                                                                            */
/*                   Flexible Static Memory Controller                        */
/*                                                                            */
/******************************************************************************/
/******************  Bit definition for SMC_CSCTRL1 register  *******************/
#define SMC_CSCTRL1_MBKEN_Pos                                       (0U)
#define SMC_CSCTRL1_MBKEN_Msk                                       (0x1UL << SMC_CSCTRL1_MBKEN_Pos)                               /*!< 0x00000001 */
#define SMC_CSCTRL1_MBKEN                                           SMC_CSCTRL1_MBKEN_Msk                                          /*!<Memory bank enable bit                 */
#define SMC_CSCTRL1_ADMUXEN_Pos                                     (1U)
#define SMC_CSCTRL1_ADMUXEN_Msk                                     (0x1UL << SMC_CSCTRL1_ADMUXEN_Pos)                             /*!< 0x00000002 */
#define SMC_CSCTRL1_ADMUXEN                                         SMC_CSCTRL1_ADMUXEN_Msk                                        /*!<Address/data multiplexing enable bit   */

#define SMC_CSCTRL1_MTYPECFG_Pos                                    (2U)
#define SMC_CSCTRL1_MTYPECFG_Msk                                    (0x3UL << SMC_CSCTRL1_MTYPECFG_Pos)                            /*!< 0x0000000C */
#define SMC_CSCTRL1_MTYPECFG                                        SMC_CSCTRL1_MTYPECFG_Msk                                       /*!<MTYP[1:0] bits (Memory type)           */
#define SMC_CSCTRL1_MTYPECFG_0                                      (0x1UL << SMC_CSCTRL1_MTYPECFG_Pos)                            /*!< 0x00000004 */
#define SMC_CSCTRL1_MTYPECFG_1                                      (0x2UL << SMC_CSCTRL1_MTYPECFG_Pos)                            /*!< 0x00000008 */

#define SMC_CSCTRL1_MDBWIDCFG_Pos                                   (4U)
#define SMC_CSCTRL1_MDBWIDCFG_Msk                                   (0x3UL << SMC_CSCTRL1_MDBWIDCFG_Pos)                           /*!< 0x00000030 */
#define SMC_CSCTRL1_MDBWIDCFG                                       SMC_CSCTRL1_MDBWIDCFG_Msk                                      /*!<MWID[1:0] bits (Memory data bus width) */
#define SMC_CSCTRL1_MDBWIDCFG_0                                     (0x1UL << SMC_CSCTRL1_MDBWIDCFG_Pos)                           /*!< 0x00000010 */
#define SMC_CSCTRL1_MDBWIDCFG_1                                     (0x2UL << SMC_CSCTRL1_MDBWIDCFG_Pos)                           /*!< 0x00000020 */

#define SMC_CSCTRL1_NORFMACCEN_Pos                                  (6U)
#define SMC_CSCTRL1_NORFMACCEN_Msk                                  (0x1UL << SMC_CSCTRL1_NORFMACCEN_Pos)                          /*!< 0x00000040 */
#define SMC_CSCTRL1_NORFMACCEN                                      SMC_CSCTRL1_NORFMACCEN_Msk                                     /*!<Flash access enable                    */
#define SMC_CSCTRL1_BURSTEN_Pos                                     (8U)
#define SMC_CSCTRL1_BURSTEN_Msk                                     (0x1UL << SMC_CSCTRL1_BURSTEN_Pos)                             /*!< 0x00000100 */
#define SMC_CSCTRL1_BURSTEN                                         SMC_CSCTRL1_BURSTEN_Msk                                        /*!<Burst enable bit                       */
#define SMC_CSCTRL1_WSPOLCFG_Pos                                    (9U)
#define SMC_CSCTRL1_WSPOLCFG_Msk                                    (0x1UL << SMC_CSCTRL1_WSPOLCFG_Pos)                            /*!< 0x00000200 */
#define SMC_CSCTRL1_WSPOLCFG                                        SMC_CSCTRL1_WSPOLCFG_Msk                                       /*!<Wait signal polarity bit               */
#define SMC_CSCTRL1_WRAPBEN_Pos                                     (10U)
#define SMC_CSCTRL1_WRAPBEN_Msk                                     (0x1UL << SMC_CSCTRL1_WRAPBEN_Pos)                             /*!< 0x00000400 */
#define SMC_CSCTRL1_WRAPBEN                                         SMC_CSCTRL1_WRAPBEN_Msk                                        /*!<Wrapped burst mode support             */
#define SMC_CSCTRL1_WTIMCFG_Pos                                     (11U)
#define SMC_CSCTRL1_WTIMCFG_Msk                                     (0x1UL << SMC_CSCTRL1_WTIMCFG_Pos)                             /*!< 0x00000800 */
#define SMC_CSCTRL1_WTIMCFG                                         SMC_CSCTRL1_WTIMCFG_Msk                                        /*!<Wait timing configuration              */
#define SMC_CSCTRL1_WREN_Pos                                        (12U)
#define SMC_CSCTRL1_WREN_Msk                                        (0x1UL << SMC_CSCTRL1_WREN_Pos)                                /*!< 0x00001000 */
#define SMC_CSCTRL1_WREN                                            SMC_CSCTRL1_WREN_Msk                                           /*!<Write enable bit                       */
#define SMC_CSCTRL1_WAITEN_Pos                                      (13U)
#define SMC_CSCTRL1_WAITEN_Msk                                      (0x1UL << SMC_CSCTRL1_WAITEN_Pos)                              /*!< 0x00002000 */
#define SMC_CSCTRL1_WAITEN                                          SMC_CSCTRL1_WAITEN_Msk                                         /*!<Wait enable bit                        */
#define SMC_CSCTRL1_EXTMODEEN_Pos                                   (14U)
#define SMC_CSCTRL1_EXTMODEEN_Msk                                   (0x1UL << SMC_CSCTRL1_EXTMODEEN_Pos)                           /*!< 0x00004000 */
#define SMC_CSCTRL1_EXTMODEEN                                       SMC_CSCTRL1_EXTMODEEN_Msk                                      /*!<Extended mode enable                   */
#define SMC_CSCTRL1_WSASYNCEN_Pos                                   (15U)
#define SMC_CSCTRL1_WSASYNCEN_Msk                                   (0x1UL << SMC_CSCTRL1_WSASYNCEN_Pos)                           /*!< 0x00008000 */
#define SMC_CSCTRL1_WSASYNCEN                                       SMC_CSCTRL1_WSASYNCEN_Msk                                      /*!<Asynchronous wait                      */
#define SMC_CSCTRL1_CRAMPSIZECFG_Pos                                (16U)
#define SMC_CSCTRL1_CRAMPSIZECFG_Msk                                (0x7UL << SMC_CSCTRL1_CRAMPSIZECFG_Pos)                        /*!< 0x00070000 */
#define SMC_CSCTRL1_CRAMPSIZECFG                                    SMC_CSCTRL1_CRAMPSIZECFG_Msk                                   /*!<CRAM page size */
#define SMC_CSCTRL1_CRAMPSIZECFG_0                                  (0x1UL << SMC_CSCTRL1_CRAMPSIZECFG_Pos)                        /*!< 0x00010000 */
#define SMC_CSCTRL1_CRAMPSIZECFG_1                                  (0x2UL << SMC_CSCTRL1_CRAMPSIZECFG_Pos)                        /*!< 0x00020000 */
#define SMC_CSCTRL1_CRAMPSIZECFG_2                                  (0x4UL << SMC_CSCTRL1_CRAMPSIZECFG_Pos)                        /*!< 0x00040000 */
#define SMC_CSCTRL1_WRBURSTEN_Pos                                   (19U)
#define SMC_CSCTRL1_WRBURSTEN_Msk                                   (0x1UL << SMC_CSCTRL1_WRBURSTEN_Pos)                           /*!< 0x00080000 */
#define SMC_CSCTRL1_WRBURSTEN                                       SMC_CSCTRL1_WRBURSTEN_Msk                                      /*!<Write burst enable                     */

/******************  Bit definition for SMC_CSCTRL2 register  *******************/
#define SMC_CSCTRL2_MBKEN_Pos                                       (0U)
#define SMC_CSCTRL2_MBKEN_Msk                                       (0x1UL << SMC_CSCTRL2_MBKEN_Pos)                               /*!< 0x00000001 */
#define SMC_CSCTRL2_MBKEN                                           SMC_CSCTRL2_MBKEN_Msk                                          /*!<Memory bank enable bit                */
#define SMC_CSCTRL2_ADMUXEN_Pos                                     (1U)
#define SMC_CSCTRL2_ADMUXEN_Msk                                     (0x1UL << SMC_CSCTRL2_ADMUXEN_Pos)                             /*!< 0x00000002 */
#define SMC_CSCTRL2_ADMUXEN                                         SMC_CSCTRL2_ADMUXEN_Msk                                        /*!<Address/data multiplexing enable bit   */

#define SMC_CSCTRL2_MTYPECFG_Pos                                    (2U)
#define SMC_CSCTRL2_MTYPECFG_Msk                                    (0x3UL << SMC_CSCTRL2_MTYPECFG_Pos)                            /*!< 0x0000000C */
#define SMC_CSCTRL2_MTYPECFG                                        SMC_CSCTRL2_MTYPECFG_Msk                                       /*!<MTYP[1:0] bits (Memory type)           */
#define SMC_CSCTRL2_MTYPECFG_0                                      (0x1UL << SMC_CSCTRL2_MTYPECFG_Pos)                            /*!< 0x00000004 */
#define SMC_CSCTRL2_MTYPECFG_1                                      (0x2UL << SMC_CSCTRL2_MTYPECFG_Pos)                            /*!< 0x00000008 */

#define SMC_CSCTRL2_MDBWIDCFG_Pos                                   (4U)
#define SMC_CSCTRL2_MDBWIDCFG_Msk                                   (0x3UL << SMC_CSCTRL2_MDBWIDCFG_Pos)                           /*!< 0x00000030 */
#define SMC_CSCTRL2_MDBWIDCFG                                       SMC_CSCTRL2_MDBWIDCFG_Msk                                      /*!<MWID[1:0] bits (Memory data bus width) */
#define SMC_CSCTRL2_MDBWIDCFG_0                                     (0x1UL << SMC_CSCTRL2_MDBWIDCFG_Pos)                           /*!< 0x00000010 */
#define SMC_CSCTRL2_MDBWIDCFG_1                                     (0x2UL << SMC_CSCTRL2_MDBWIDCFG_Pos)                           /*!< 0x00000020 */

#define SMC_CSCTRL2_NORFMACCEN_Pos                                  (6U)
#define SMC_CSCTRL2_NORFMACCEN_Msk                                  (0x1UL << SMC_CSCTRL2_NORFMACCEN_Pos)                          /*!< 0x00000040 */
#define SMC_CSCTRL2_NORFMACCEN                                      SMC_CSCTRL2_NORFMACCEN_Msk                                     /*!<Flash access enable                    */
#define SMC_CSCTRL2_BURSTEN_Pos                                     (8U)
#define SMC_CSCTRL2_BURSTEN_Msk                                     (0x1UL << SMC_CSCTRL2_BURSTEN_Pos)                             /*!< 0x00000100 */
#define SMC_CSCTRL2_BURSTEN                                         SMC_CSCTRL2_BURSTEN_Msk                                        /*!<Burst enable bit                       */
#define SMC_CSCTRL2_WSPOLCFG_Pos                                    (9U)
#define SMC_CSCTRL2_WSPOLCFG_Msk                                    (0x1UL << SMC_CSCTRL2_WSPOLCFG_Pos)                            /*!< 0x00000200 */
#define SMC_CSCTRL2_WSPOLCFG                                        SMC_CSCTRL2_WSPOLCFG_Msk                                       /*!<Wait signal polarity bit               */
#define SMC_CSCTRL2_WRAPBEN_Pos                                     (10U)
#define SMC_CSCTRL2_WRAPBEN_Msk                                     (0x1UL << SMC_CSCTRL2_WRAPBEN_Pos)                             /*!< 0x00000400 */
#define SMC_CSCTRL2_WRAPBEN                                         SMC_CSCTRL2_WRAPBEN_Msk                                        /*!<Wrapped burst mode support             */
#define SMC_CSCTRL2_WTIMCFG_Pos                                     (11U)
#define SMC_CSCTRL2_WTIMCFG_Msk                                     (0x1UL << SMC_CSCTRL2_WTIMCFG_Pos)                             /*!< 0x00000800 */
#define SMC_CSCTRL2_WTIMCFG                                         SMC_CSCTRL2_WTIMCFG_Msk                                        /*!<Wait timing configuration              */
#define SMC_CSCTRL2_WREN_Pos                                        (12U)
#define SMC_CSCTRL2_WREN_Msk                                        (0x1UL << SMC_CSCTRL2_WREN_Pos)                                /*!< 0x00001000 */
#define SMC_CSCTRL2_WREN                                            SMC_CSCTRL2_WREN_Msk                                           /*!<Write enable bit                       */
#define SMC_CSCTRL2_WAITEN_Pos                                      (13U)
#define SMC_CSCTRL2_WAITEN_Msk                                      (0x1UL << SMC_CSCTRL2_WAITEN_Pos)                              /*!< 0x00002000 */
#define SMC_CSCTRL2_WAITEN                                          SMC_CSCTRL2_WAITEN_Msk                                         /*!<Wait enable bit                        */
#define SMC_CSCTRL2_EXTMODEEN_Pos                                   (14U)
#define SMC_CSCTRL2_EXTMODEEN_Msk                                   (0x1UL << SMC_CSCTRL2_EXTMODEEN_Pos)                           /*!< 0x00004000 */
#define SMC_CSCTRL2_EXTMODEEN                                       SMC_CSCTRL2_EXTMODEEN_Msk                                      /*!<Extended mode enable                   */
#define SMC_CSCTRL2_WSASYNCEN_Pos                                   (15U)
#define SMC_CSCTRL2_WSASYNCEN_Msk                                   (0x1UL << SMC_CSCTRL2_WSASYNCEN_Pos)                           /*!< 0x00008000 */
#define SMC_CSCTRL2_WSASYNCEN                                       SMC_CSCTRL2_WSASYNCEN_Msk                                      /*!<Asynchronous wait                      */
#define SMC_CSCTRL2_CRAMPSIZECFG_Pos                                (16U)
#define SMC_CSCTRL2_CRAMPSIZECFG_Msk                                (0x7UL << SMC_CSCTRL2_CRAMPSIZECFG_Pos)                        /*!< 0x00070000 */
#define SMC_CSCTRL2_CRAMPSIZECFG                                    SMC_CSCTRL2_CRAMPSIZECFG_Msk                                   /*!<CRAM page size */
#define SMC_CSCTRL2_CRAMPSIZECFG_0                                  (0x1UL << SMC_CSCTRL2_CRAMPSIZECFG_Pos)                        /*!< 0x00010000 */
#define SMC_CSCTRL2_CRAMPSIZECFG_1                                  (0x2UL << SMC_CSCTRL2_CRAMPSIZECFG_Pos)                        /*!< 0x00020000 */
#define SMC_CSCTRL2_CRAMPSIZECFG_2                                  (0x4UL << SMC_CSCTRL2_CRAMPSIZECFG_Pos)                        /*!< 0x00040000 */
#define SMC_CSCTRL2_WRBURSTEN_Pos                                   (19U)
#define SMC_CSCTRL2_WRBURSTEN_Msk                                   (0x1UL << SMC_CSCTRL2_WRBURSTEN_Pos)                           /*!< 0x00080000 */
#define SMC_CSCTRL2_WRBURSTEN                                       SMC_CSCTRL2_WRBURSTEN_Msk                                      /*!<Write burst enable                     */

/******************  Bit definition for SMC_CSCTRL3 register  *******************/
#define SMC_CSCTRL3_MBKEN_Pos                                       (0U)
#define SMC_CSCTRL3_MBKEN_Msk                                       (0x1UL << SMC_CSCTRL3_MBKEN_Pos)                               /*!< 0x00000001 */
#define SMC_CSCTRL3_MBKEN                                           SMC_CSCTRL3_MBKEN_Msk                                          /*!<Memory bank enable bit                 */
#define SMC_CSCTRL3_ADMUXEN_Pos                                     (1U)
#define SMC_CSCTRL3_ADMUXEN_Msk                                     (0x1UL << SMC_CSCTRL3_ADMUXEN_Pos)                             /*!< 0x00000002 */
#define SMC_CSCTRL3_ADMUXEN                                         SMC_CSCTRL3_ADMUXEN_Msk                                        /*!<Address/data multiplexing enable bit   */

#define SMC_CSCTRL3_MTYPECFG_Pos                                    (2U)
#define SMC_CSCTRL3_MTYPECFG_Msk                                    (0x3UL << SMC_CSCTRL3_MTYPECFG_Pos)                            /*!< 0x0000000C */
#define SMC_CSCTRL3_MTYPECFG                                        SMC_CSCTRL3_MTYPECFG_Msk                                       /*!<MTYP[1:0] bits (Memory type)           */
#define SMC_CSCTRL3_MTYPECFG_0                                      (0x1UL << SMC_CSCTRL3_MTYPECFG_Pos)                            /*!< 0x00000004 */
#define SMC_CSCTRL3_MTYPECFG_1                                      (0x2UL << SMC_CSCTRL3_MTYPECFG_Pos)                            /*!< 0x00000008 */

#define SMC_CSCTRL3_MDBWIDCFG_Pos                                   (4U)
#define SMC_CSCTRL3_MDBWIDCFG_Msk                                   (0x3UL << SMC_CSCTRL3_MDBWIDCFG_Pos)                           /*!< 0x00000030 */
#define SMC_CSCTRL3_MDBWIDCFG                                       SMC_CSCTRL3_MDBWIDCFG_Msk                                      /*!<MWID[1:0] bits (Memory data bus width) */
#define SMC_CSCTRL3_MDBWIDCFG_0                                     (0x1UL << SMC_CSCTRL3_MDBWIDCFG_Pos)                           /*!< 0x00000010 */
#define SMC_CSCTRL3_MDBWIDCFG_1                                     (0x2UL << SMC_CSCTRL3_MDBWIDCFG_Pos)                           /*!< 0x00000020 */

#define SMC_CSCTRL3_NORFMACCEN_Pos                                  (6U)
#define SMC_CSCTRL3_NORFMACCEN_Msk                                  (0x1UL << SMC_CSCTRL3_NORFMACCEN_Pos)                          /*!< 0x00000040 */
#define SMC_CSCTRL3_NORFMACCEN                                      SMC_CSCTRL3_NORFMACCEN_Msk                                     /*!<Flash access enable                    */
#define SMC_CSCTRL3_BURSTEN_Pos                                     (8U)
#define SMC_CSCTRL3_BURSTEN_Msk                                     (0x1UL << SMC_CSCTRL3_BURSTEN_Pos)                             /*!< 0x00000100 */
#define SMC_CSCTRL3_BURSTEN                                         SMC_CSCTRL3_BURSTEN_Msk                                        /*!<Burst enable bit                       */
#define SMC_CSCTRL3_WSPOLCFG_Pos                                    (9U)
#define SMC_CSCTRL3_WSPOLCFG_Msk                                    (0x1UL << SMC_CSCTRL3_WSPOLCFG_Pos)                            /*!< 0x00000200 */
#define SMC_CSCTRL3_WSPOLCFG                                        SMC_CSCTRL3_WSPOLCFG_Msk                                       /*!<Wait signal polarity bit               */
#define SMC_CSCTRL3_WRAPBEN_Pos                                     (10U)
#define SMC_CSCTRL3_WRAPBEN_Msk                                     (0x1UL << SMC_CSCTRL3_WRAPBEN_Pos)                             /*!< 0x00000400 */
#define SMC_CSCTRL3_WRAPBEN                                         SMC_CSCTRL3_WRAPBEN_Msk                                        /*!<Wrapped burst mode support             */
#define SMC_CSCTRL3_WTIMCFG_Pos                                     (11U)
#define SMC_CSCTRL3_WTIMCFG_Msk                                     (0x1UL << SMC_CSCTRL3_WTIMCFG_Pos)                             /*!< 0x00000800 */
#define SMC_CSCTRL3_WTIMCFG                                         SMC_CSCTRL3_WTIMCFG_Msk                                        /*!<Wait timing configuration              */
#define SMC_CSCTRL3_WREN_Pos                                        (12U)
#define SMC_CSCTRL3_WREN_Msk                                        (0x1UL << SMC_CSCTRL3_WREN_Pos)                                /*!< 0x00001000 */
#define SMC_CSCTRL3_WREN                                            SMC_CSCTRL3_WREN_Msk                                           /*!<Write enable bit                       */
#define SMC_CSCTRL3_WAITEN_Pos                                      (13U)
#define SMC_CSCTRL3_WAITEN_Msk                                      (0x1UL << SMC_CSCTRL3_WAITEN_Pos)                              /*!< 0x00002000 */
#define SMC_CSCTRL3_WAITEN                                          SMC_CSCTRL3_WAITEN_Msk                                         /*!<Wait enable bit                        */
#define SMC_CSCTRL3_EXTMODEEN_Pos                                   (14U)
#define SMC_CSCTRL3_EXTMODEEN_Msk                                   (0x1UL << SMC_CSCTRL3_EXTMODEEN_Pos)                           /*!< 0x00004000 */
#define SMC_CSCTRL3_EXTMODEEN                                       SMC_CSCTRL3_EXTMODEEN_Msk                                      /*!<Extended mode enable                   */
#define SMC_CSCTRL3_WSASYNCEN_Pos                                   (15U)
#define SMC_CSCTRL3_WSASYNCEN_Msk                                   (0x1UL << SMC_CSCTRL3_WSASYNCEN_Pos)                           /*!< 0x00008000 */
#define SMC_CSCTRL3_WSASYNCEN                                       SMC_CSCTRL3_WSASYNCEN_Msk                                      /*!<Asynchronous wait                      */
#define SMC_CSCTRL3_CRAMPSIZECFG_Pos                                (16U)
#define SMC_CSCTRL3_CRAMPSIZECFG_Msk                                (0x7UL << SMC_CSCTRL3_CRAMPSIZECFG_Pos)                        /*!< 0x00070000 */
#define SMC_CSCTRL3_CRAMPSIZECFG                                    SMC_CSCTRL3_CRAMPSIZECFG_Msk                                   /*!<CRAM page size */
#define SMC_CSCTRL3_CRAMPSIZECFG_0                                  (0x1UL << SMC_CSCTRL3_CRAMPSIZECFG_Pos)                        /*!< 0x00010000 */
#define SMC_CSCTRL3_CRAMPSIZECFG_1                                  (0x2UL << SMC_CSCTRL3_CRAMPSIZECFG_Pos)                        /*!< 0x00020000 */
#define SMC_CSCTRL3_CRAMPSIZECFG_2                                  (0x4UL << SMC_CSCTRL3_CRAMPSIZECFG_Pos)                        /*!< 0x00040000 */
#define SMC_CSCTRL3_WRBURSTEN_Pos                                   (19U)
#define SMC_CSCTRL3_WRBURSTEN_Msk                                   (0x1UL << SMC_CSCTRL3_WRBURSTEN_Pos)                           /*!< 0x00080000 */
#define SMC_CSCTRL3_WRBURSTEN                                       SMC_CSCTRL3_WRBURSTEN_Msk                                      /*!<Write burst enable                     */

/******************  Bit definition for SMC_CSCTRL4 register  *******************/
#define SMC_CSCTRL4_MBKEN_Pos                                       (0U)
#define SMC_CSCTRL4_MBKEN_Msk                                       (0x1UL << SMC_CSCTRL4_MBKEN_Pos)                               /*!< 0x00000001 */
#define SMC_CSCTRL4_MBKEN                                           SMC_CSCTRL4_MBKEN_Msk                                          /*!<Memory bank enable bit */
#define SMC_CSCTRL4_ADMUXEN_Pos                                     (1U)
#define SMC_CSCTRL4_ADMUXEN_Msk                                     (0x1UL << SMC_CSCTRL4_ADMUXEN_Pos)                             /*!< 0x00000002 */
#define SMC_CSCTRL4_ADMUXEN                                         SMC_CSCTRL4_ADMUXEN_Msk                                        /*!<Address/data multiplexing enable bit   */

#define SMC_CSCTRL4_MTYPECFG_Pos                                    (2U)
#define SMC_CSCTRL4_MTYPECFG_Msk                                    (0x3UL << SMC_CSCTRL4_MTYPECFG_Pos)                            /*!< 0x0000000C */
#define SMC_CSCTRL4_MTYPECFG                                        SMC_CSCTRL4_MTYPECFG_Msk                                       /*!<MTYP[1:0] bits (Memory type)           */
#define SMC_CSCTRL4_MTYPECFG_0                                      (0x1UL << SMC_CSCTRL4_MTYPECFG_Pos)                            /*!< 0x00000004 */
#define SMC_CSCTRL4_MTYPECFG_1                                      (0x2UL << SMC_CSCTRL4_MTYPECFG_Pos)                            /*!< 0x00000008 */

#define SMC_CSCTRL4_MDBWIDCFG_Pos                                   (4U)
#define SMC_CSCTRL4_MDBWIDCFG_Msk                                   (0x3UL << SMC_CSCTRL4_MDBWIDCFG_Pos)                           /*!< 0x00000030 */
#define SMC_CSCTRL4_MDBWIDCFG                                       SMC_CSCTRL4_MDBWIDCFG_Msk                                      /*!<MWID[1:0] bits (Memory data bus width) */
#define SMC_CSCTRL4_MDBWIDCFG_0                                     (0x1UL << SMC_CSCTRL4_MDBWIDCFG_Pos)                           /*!< 0x00000010 */
#define SMC_CSCTRL4_MDBWIDCFG_1                                     (0x2UL << SMC_CSCTRL4_MDBWIDCFG_Pos)                           /*!< 0x00000020 */

#define SMC_CSCTRL4_NORFMACCEN_Pos                                  (6U)
#define SMC_CSCTRL4_NORFMACCEN_Msk                                  (0x1UL << SMC_CSCTRL4_NORFMACCEN_Pos)                          /*!< 0x00000040 */
#define SMC_CSCTRL4_NORFMACCEN                                      SMC_CSCTRL4_NORFMACCEN_Msk                                     /*!<Flash access enable                    */
#define SMC_CSCTRL4_BURSTEN_Pos                                     (8U)
#define SMC_CSCTRL4_BURSTEN_Msk                                     (0x1UL << SMC_CSCTRL4_BURSTEN_Pos)                             /*!< 0x00000100 */
#define SMC_CSCTRL4_BURSTEN                                         SMC_CSCTRL4_BURSTEN_Msk                                        /*!<Burst enable bit                       */
#define SMC_CSCTRL4_WSPOLCFG_Pos                                    (9U)
#define SMC_CSCTRL4_WSPOLCFG_Msk                                    (0x1UL << SMC_CSCTRL4_WSPOLCFG_Pos)                            /*!< 0x00000200 */
#define SMC_CSCTRL4_WSPOLCFG                                        SMC_CSCTRL4_WSPOLCFG_Msk                                       /*!<Wait signal polarity bit               */
#define SMC_CSCTRL4_WRAPBEN_Pos                                     (10U)
#define SMC_CSCTRL4_WRAPBEN_Msk                                     (0x1UL << SMC_CSCTRL4_WRAPBEN_Pos)                             /*!< 0x00000400 */
#define SMC_CSCTRL4_WRAPBEN                                         SMC_CSCTRL4_WRAPBEN_Msk                                        /*!<Wrapped burst mode support             */
#define SMC_CSCTRL4_WTIMCFG_Pos                                     (11U)
#define SMC_CSCTRL4_WTIMCFG_Msk                                     (0x1UL << SMC_CSCTRL4_WTIMCFG_Pos)                             /*!< 0x00000800 */
#define SMC_CSCTRL4_WTIMCFG                                         SMC_CSCTRL4_WTIMCFG_Msk                                        /*!<Wait timing configuration              */
#define SMC_CSCTRL4_WREN_Pos                                        (12U)
#define SMC_CSCTRL4_WREN_Msk                                        (0x1UL << SMC_CSCTRL4_WREN_Pos)                                /*!< 0x00001000 */
#define SMC_CSCTRL4_WREN                                            SMC_CSCTRL4_WREN_Msk                                           /*!<Write enable bit                       */
#define SMC_CSCTRL4_WAITEN_Pos                                      (13U)
#define SMC_CSCTRL4_WAITEN_Msk                                      (0x1UL << SMC_CSCTRL4_WAITEN_Pos)                              /*!< 0x00002000 */
#define SMC_CSCTRL4_WAITEN                                          SMC_CSCTRL4_WAITEN_Msk                                         /*!<Wait enable bit                        */
#define SMC_CSCTRL4_EXTMODEEN_Pos                                   (14U)
#define SMC_CSCTRL4_EXTMODEEN_Msk                                   (0x1UL << SMC_CSCTRL4_EXTMODEEN_Pos)                           /*!< 0x00004000 */
#define SMC_CSCTRL4_EXTMODEEN                                       SMC_CSCTRL4_EXTMODEEN_Msk                                      /*!<Extended mode enable                   */
#define SMC_CSCTRL4_WSASYNCEN_Pos                                   (15U)
#define SMC_CSCTRL4_WSASYNCEN_Msk                                   (0x1UL << SMC_CSCTRL4_WSASYNCEN_Pos)                           /*!< 0x00008000 */
#define SMC_CSCTRL4_WSASYNCEN                                       SMC_CSCTRL4_WSASYNCEN_Msk                                      /*!<Asynchronous wait                      */
#define SMC_CSCTRL4_CRAMPSIZECFG_Pos                                (16U)
#define SMC_CSCTRL4_CRAMPSIZECFG_Msk                                (0x7UL << SMC_CSCTRL4_CRAMPSIZECFG_Pos)                        /*!< 0x00070000 */
#define SMC_CSCTRL4_CRAMPSIZECFG                                    SMC_CSCTRL4_CRAMPSIZECFG_Msk                                   /*!<CRAM page size */
#define SMC_CSCTRL4_CRAMPSIZECFG_0                                  (0x1UL << SMC_CSCTRL4_CRAMPSIZECFG_Pos)                        /*!< 0x00010000 */
#define SMC_CSCTRL4_CRAMPSIZECFG_1                                  (0x2UL << SMC_CSCTRL4_CRAMPSIZECFG_Pos)                        /*!< 0x00020000 */
#define SMC_CSCTRL4_CRAMPSIZECFG_2                                  (0x4UL << SMC_CSCTRL4_CRAMPSIZECFG_Pos)                        /*!< 0x00040000 */
#define SMC_CSCTRL4_WRBURSTEN_Pos                                   (19U)
#define SMC_CSCTRL4_WRBURSTEN_Msk                                   (0x1UL << SMC_CSCTRL4_WRBURSTEN_Pos)                           /*!< 0x00080000 */
#define SMC_CSCTRL4_WRBURSTEN                                       SMC_CSCTRL4_WRBURSTEN_Msk                                      /*!<Write burst enable                     */

/******************  Bit definition for SMC_CSTIM1 register  ******************/
#define SMC_CSTIM1_ADDRSETCFG_Pos                                   (0U)
#define SMC_CSTIM1_ADDRSETCFG_Msk                                   (0xFUL << SMC_CSTIM1_ADDRSETCFG_Pos)                           /*!< 0x0000000F */
#define SMC_CSTIM1_ADDRSETCFG                                       SMC_CSTIM1_ADDRSETCFG_Msk                                      /*!<ADDSET[3:0] bits (Address setup phase duration) */
#define SMC_CSTIM1_ADDRSETCFG_0                                     (0x1UL << SMC_CSTIM1_ADDRSETCFG_Pos)                           /*!< 0x00000001 */
#define SMC_CSTIM1_ADDRSETCFG_1                                     (0x2UL << SMC_CSTIM1_ADDRSETCFG_Pos)                           /*!< 0x00000002 */
#define SMC_CSTIM1_ADDRSETCFG_2                                     (0x4UL << SMC_CSTIM1_ADDRSETCFG_Pos)                           /*!< 0x00000004 */
#define SMC_CSTIM1_ADDRSETCFG_3                                     (0x8UL << SMC_CSTIM1_ADDRSETCFG_Pos)                           /*!< 0x00000008 */

#define SMC_CSTIM1_ADDRHLDCFG_Pos                                   (4U)
#define SMC_CSTIM1_ADDRHLDCFG_Msk                                   (0xFUL << SMC_CSTIM1_ADDRHLDCFG_Pos)                           /*!< 0x000000F0 */
#define SMC_CSTIM1_ADDRHLDCFG                                       SMC_CSTIM1_ADDRHLDCFG_Msk                                      /*!<ADDHLD[3:0] bits (Address-hold phase duration) */
#define SMC_CSTIM1_ADDRHLDCFG_0                                     (0x1UL << SMC_CSTIM1_ADDRHLDCFG_Pos)                           /*!< 0x00000010 */
#define SMC_CSTIM1_ADDRHLDCFG_1                                     (0x2UL << SMC_CSTIM1_ADDRHLDCFG_Pos)                           /*!< 0x00000020 */
#define SMC_CSTIM1_ADDRHLDCFG_2                                     (0x4UL << SMC_CSTIM1_ADDRHLDCFG_Pos)                           /*!< 0x00000040 */
#define SMC_CSTIM1_ADDRHLDCFG_3                                     (0x8UL << SMC_CSTIM1_ADDRHLDCFG_Pos)                           /*!< 0x00000080 */

#define SMC_CSTIM1_DATASETCFG_Pos                                   (8U)
#define SMC_CSTIM1_DATASETCFG_Msk                                   (0xFFUL << SMC_CSTIM1_DATASETCFG_Pos)                          /*!< 0x0000FF00 */
#define SMC_CSTIM1_DATASETCFG                                       SMC_CSTIM1_DATASETCFG_Msk                                      /*!<DATAST [7:0] bits (Data-phase duration) */
#define SMC_CSTIM1_DATASETCFG_0                                     (0x01UL << SMC_CSTIM1_DATASETCFG_Pos)                          /*!< 0x00000100 */
#define SMC_CSTIM1_DATASETCFG_1                                     (0x02UL << SMC_CSTIM1_DATASETCFG_Pos)                          /*!< 0x00000200 */
#define SMC_CSTIM1_DATASETCFG_2                                     (0x04UL << SMC_CSTIM1_DATASETCFG_Pos)                          /*!< 0x00000400 */
#define SMC_CSTIM1_DATASETCFG_3                                     (0x08UL << SMC_CSTIM1_DATASETCFG_Pos)                          /*!< 0x00000800 */
#define SMC_CSTIM1_DATASETCFG_4                                     (0x10UL << SMC_CSTIM1_DATASETCFG_Pos)                          /*!< 0x00001000 */
#define SMC_CSTIM1_DATASETCFG_5                                     (0x20UL << SMC_CSTIM1_DATASETCFG_Pos)                          /*!< 0x00002000 */
#define SMC_CSTIM1_DATASETCFG_6                                     (0x40UL << SMC_CSTIM1_DATASETCFG_Pos)                          /*!< 0x00004000 */
#define SMC_CSTIM1_DATASETCFG_7                                     (0x80UL << SMC_CSTIM1_DATASETCFG_Pos)                          /*!< 0x00008000 */

#define SMC_CSTIM1_BUSTURNCFG_Pos                                   (16U)
#define SMC_CSTIM1_BUSTURNCFG_Msk                                   (0xFUL << SMC_CSTIM1_BUSTURNCFG_Pos)                           /*!< 0x000F0000 */
#define SMC_CSTIM1_BUSTURNCFG                                       SMC_CSTIM1_BUSTURNCFG_Msk                                      /*!<BUSTURN[3:0] bits (Bus turnaround phase duration) */
#define SMC_CSTIM1_BUSTURNCFG_0                                     (0x1UL << SMC_CSTIM1_BUSTURNCFG_Pos)                           /*!< 0x00010000 */
#define SMC_CSTIM1_BUSTURNCFG_1                                     (0x2UL << SMC_CSTIM1_BUSTURNCFG_Pos)                           /*!< 0x00020000 */
#define SMC_CSTIM1_BUSTURNCFG_2                                     (0x4UL << SMC_CSTIM1_BUSTURNCFG_Pos)                           /*!< 0x00040000 */
#define SMC_CSTIM1_BUSTURNCFG_3                                     (0x8UL << SMC_CSTIM1_BUSTURNCFG_Pos)                           /*!< 0x00080000 */

#define SMC_CSTIM1_CLKDIVCFG_Pos                                    (20U)
#define SMC_CSTIM1_CLKDIVCFG_Msk                                    (0xFUL << SMC_CSTIM1_CLKDIVCFG_Pos)                            /*!< 0x00F00000 */
#define SMC_CSTIM1_CLKDIVCFG                                        SMC_CSTIM1_CLKDIVCFG_Msk                                       /*!<CLKDIV[3:0] bits (Clock divide ratio) */
#define SMC_CSTIM1_CLKDIVCFG_0                                      (0x1UL << SMC_CSTIM1_CLKDIVCFG_Pos)                            /*!< 0x00100000 */
#define SMC_CSTIM1_CLKDIVCFG_1                                      (0x2UL << SMC_CSTIM1_CLKDIVCFG_Pos)                            /*!< 0x00200000 */
#define SMC_CSTIM1_CLKDIVCFG_2                                      (0x4UL << SMC_CSTIM1_CLKDIVCFG_Pos)                            /*!< 0x00400000 */
#define SMC_CSTIM1_CLKDIVCFG_3                                      (0x8UL << SMC_CSTIM1_CLKDIVCFG_Pos)                            /*!< 0x00800000 */

#define SMC_CSTIM1_DATALATCFG_Pos                                   (24U)
#define SMC_CSTIM1_DATALATCFG_Msk                                   (0xFUL << SMC_CSTIM1_DATALATCFG_Pos)                           /*!< 0x0F000000 */
#define SMC_CSTIM1_DATALATCFG                                       SMC_CSTIM1_DATALATCFG_Msk                                      /*!<DATLA[3:0] bits (Data latency) */
#define SMC_CSTIM1_DATALATCFG_0                                     (0x1UL << SMC_CSTIM1_DATALATCFG_Pos)                           /*!< 0x01000000 */
#define SMC_CSTIM1_DATALATCFG_1                                     (0x2UL << SMC_CSTIM1_DATALATCFG_Pos)                           /*!< 0x02000000 */
#define SMC_CSTIM1_DATALATCFG_2                                     (0x4UL << SMC_CSTIM1_DATALATCFG_Pos)                           /*!< 0x04000000 */
#define SMC_CSTIM1_DATALATCFG_3                                     (0x8UL << SMC_CSTIM1_DATALATCFG_Pos)                           /*!< 0x08000000 */

#define SMC_CSTIM1_ASYNCACCCFG_Pos                                  (28U)
#define SMC_CSTIM1_ASYNCACCCFG_Msk                                  (0x3UL << SMC_CSTIM1_ASYNCACCCFG_Pos)                          /*!< 0x30000000 */
#define SMC_CSTIM1_ASYNCACCCFG                                      SMC_CSTIM1_ASYNCACCCFG_Msk                                     /*!<ACCMOD[1:0] bits (Access mode) */
#define SMC_CSTIM1_ASYNCACCCFG_0                                    (0x1UL << SMC_CSTIM1_ASYNCACCCFG_Pos)                          /*!< 0x10000000 */
#define SMC_CSTIM1_ASYNCACCCFG_1                                    (0x2UL << SMC_CSTIM1_ASYNCACCCFG_Pos)                          /*!< 0x20000000 */

/******************  Bit definition for SMC_CSTIM2 register  *******************/
#define SMC_CSTIM2_ADDRSETCFG_Pos                                   (0U)
#define SMC_CSTIM2_ADDRSETCFG_Msk                                   (0xFUL << SMC_CSTIM2_ADDRSETCFG_Pos)                           /*!< 0x0000000F */
#define SMC_CSTIM2_ADDRSETCFG                                       SMC_CSTIM2_ADDRSETCFG_Msk                                      /*!<ADDSET[3:0] bits (Address setup phase duration) */
#define SMC_CSTIM2_ADDRSETCFG_0                                     (0x1UL << SMC_CSTIM2_ADDRSETCFG_Pos)                           /*!< 0x00000001 */
#define SMC_CSTIM2_ADDRSETCFG_1                                     (0x2UL << SMC_CSTIM2_ADDRSETCFG_Pos)                           /*!< 0x00000002 */
#define SMC_CSTIM2_ADDRSETCFG_2                                     (0x4UL << SMC_CSTIM2_ADDRSETCFG_Pos)                           /*!< 0x00000004 */
#define SMC_CSTIM2_ADDRSETCFG_3                                     (0x8UL << SMC_CSTIM2_ADDRSETCFG_Pos)                           /*!< 0x00000008 */

#define SMC_CSTIM2_ADDRHLDCFG_Pos                                   (4U)
#define SMC_CSTIM2_ADDRHLDCFG_Msk                                   (0xFUL << SMC_CSTIM2_ADDRHLDCFG_Pos)                           /*!< 0x000000F0 */
#define SMC_CSTIM2_ADDRHLDCFG                                       SMC_CSTIM2_ADDRHLDCFG_Msk                                      /*!<ADDHLD[3:0] bits (Address-hold phase duration) */
#define SMC_CSTIM2_ADDRHLDCFG_0                                     (0x1UL << SMC_CSTIM2_ADDRHLDCFG_Pos)                           /*!< 0x00000010 */
#define SMC_CSTIM2_ADDRHLDCFG_1                                     (0x2UL << SMC_CSTIM2_ADDRHLDCFG_Pos)                           /*!< 0x00000020 */
#define SMC_CSTIM2_ADDRHLDCFG_2                                     (0x4UL << SMC_CSTIM2_ADDRHLDCFG_Pos)                           /*!< 0x00000040 */
#define SMC_CSTIM2_ADDRHLDCFG_3                                     (0x8UL << SMC_CSTIM2_ADDRHLDCFG_Pos)                           /*!< 0x00000080 */

#define SMC_CSTIM2_DATASETCFG_Pos                                   (8U)
#define SMC_CSTIM2_DATASETCFG_Msk                                   (0xFFUL << SMC_CSTIM2_DATASETCFG_Pos)                          /*!< 0x0000FF00 */
#define SMC_CSTIM2_DATASETCFG                                       SMC_CSTIM2_DATASETCFG_Msk                                      /*!<DATAST [7:0] bits (Data-phase duration) */
#define SMC_CSTIM2_DATASETCFG_0                                     (0x01UL << SMC_CSTIM2_DATASETCFG_Pos)                          /*!< 0x00000100 */
#define SMC_CSTIM2_DATASETCFG_1                                     (0x02UL << SMC_CSTIM2_DATASETCFG_Pos)                          /*!< 0x00000200 */
#define SMC_CSTIM2_DATASETCFG_2                                     (0x04UL << SMC_CSTIM2_DATASETCFG_Pos)                          /*!< 0x00000400 */
#define SMC_CSTIM2_DATASETCFG_3                                     (0x08UL << SMC_CSTIM2_DATASETCFG_Pos)                          /*!< 0x00000800 */
#define SMC_CSTIM2_DATASETCFG_4                                     (0x10UL << SMC_CSTIM2_DATASETCFG_Pos)                          /*!< 0x00001000 */
#define SMC_CSTIM2_DATASETCFG_5                                     (0x20UL << SMC_CSTIM2_DATASETCFG_Pos)                          /*!< 0x00002000 */
#define SMC_CSTIM2_DATASETCFG_6                                     (0x40UL << SMC_CSTIM2_DATASETCFG_Pos)                          /*!< 0x00004000 */
#define SMC_CSTIM2_DATASETCFG_7                                     (0x80UL << SMC_CSTIM2_DATASETCFG_Pos)                          /*!< 0x00008000 */

#define SMC_CSTIM2_BUSTURNCFG_Pos                                   (16U)
#define SMC_CSTIM2_BUSTURNCFG_Msk                                   (0xFUL << SMC_CSTIM2_BUSTURNCFG_Pos)                           /*!< 0x000F0000 */
#define SMC_CSTIM2_BUSTURNCFG                                       SMC_CSTIM2_BUSTURNCFG_Msk                                      /*!<BUSTURN[3:0] bits (Bus turnaround phase duration) */
#define SMC_CSTIM2_BUSTURNCFG_0                                     (0x1UL << SMC_CSTIM2_BUSTURNCFG_Pos)                           /*!< 0x00010000 */
#define SMC_CSTIM2_BUSTURNCFG_1                                     (0x2UL << SMC_CSTIM2_BUSTURNCFG_Pos)                           /*!< 0x00020000 */
#define SMC_CSTIM2_BUSTURNCFG_2                                     (0x4UL << SMC_CSTIM2_BUSTURNCFG_Pos)                           /*!< 0x00040000 */
#define SMC_CSTIM2_BUSTURNCFG_3                                     (0x8UL << SMC_CSTIM2_BUSTURNCFG_Pos)                           /*!< 0x00080000 */

#define SMC_CSTIM2_CLKDIVCFG_Pos                                    (20U)
#define SMC_CSTIM2_CLKDIVCFG_Msk                                    (0xFUL << SMC_CSTIM2_CLKDIVCFG_Pos)                            /*!< 0x00F00000 */
#define SMC_CSTIM2_CLKDIVCFG                                        SMC_CSTIM2_CLKDIVCFG_Msk                                       /*!<CLKDIV[3:0] bits (Clock divide ratio) */
#define SMC_CSTIM2_CLKDIVCFG_0                                      (0x1UL << SMC_CSTIM2_CLKDIVCFG_Pos)                            /*!< 0x00100000 */
#define SMC_CSTIM2_CLKDIVCFG_1                                      (0x2UL << SMC_CSTIM2_CLKDIVCFG_Pos)                            /*!< 0x00200000 */
#define SMC_CSTIM2_CLKDIVCFG_2                                      (0x4UL << SMC_CSTIM2_CLKDIVCFG_Pos)                            /*!< 0x00400000 */
#define SMC_CSTIM2_CLKDIVCFG_3                                      (0x8UL << SMC_CSTIM2_CLKDIVCFG_Pos)                            /*!< 0x00800000 */

#define SMC_CSTIM2_DATALATCFG_Pos                                   (24U)
#define SMC_CSTIM2_DATALATCFG_Msk                                   (0xFUL << SMC_CSTIM2_DATALATCFG_Pos)                           /*!< 0x0F000000 */
#define SMC_CSTIM2_DATALATCFG                                       SMC_CSTIM2_DATALATCFG_Msk                                      /*!<DATLA[3:0] bits (Data latency) */
#define SMC_CSTIM2_DATALATCFG_0                                     (0x1UL << SMC_CSTIM2_DATALATCFG_Pos)                           /*!< 0x01000000 */
#define SMC_CSTIM2_DATALATCFG_1                                     (0x2UL << SMC_CSTIM2_DATALATCFG_Pos)                           /*!< 0x02000000 */
#define SMC_CSTIM2_DATALATCFG_2                                     (0x4UL << SMC_CSTIM2_DATALATCFG_Pos)                           /*!< 0x04000000 */
#define SMC_CSTIM2_DATALATCFG_3                                     (0x8UL << SMC_CSTIM2_DATALATCFG_Pos)                           /*!< 0x08000000 */

#define SMC_CSTIM2_ASYNCACCCFG_Pos                                  (28U)
#define SMC_CSTIM2_ASYNCACCCFG_Msk                                  (0x3UL << SMC_CSTIM2_ASYNCACCCFG_Pos)                          /*!< 0x30000000 */
#define SMC_CSTIM2_ASYNCACCCFG                                      SMC_CSTIM2_ASYNCACCCFG_Msk                                     /*!<ACCMOD[1:0] bits (Access mode) */
#define SMC_CSTIM2_ASYNCACCCFG_0                                    (0x1UL << SMC_CSTIM2_ASYNCACCCFG_Pos)                          /*!< 0x10000000 */
#define SMC_CSTIM2_ASYNCACCCFG_1                                    (0x2UL << SMC_CSTIM2_ASYNCACCCFG_Pos)                          /*!< 0x20000000 */

/*******************  Bit definition for SMC_CSTIM3 register  *******************/
#define SMC_CSTIM3_ADDRSETCFG_Pos                                   (0U)
#define SMC_CSTIM3_ADDRSETCFG_Msk                                   (0xFUL << SMC_CSTIM3_ADDRSETCFG_Pos)                           /*!< 0x0000000F */
#define SMC_CSTIM3_ADDRSETCFG                                       SMC_CSTIM3_ADDRSETCFG_Msk                                      /*!<ADDSET[3:0] bits (Address setup phase duration) */
#define SMC_CSTIM3_ADDRSETCFG_0                                     (0x1UL << SMC_CSTIM3_ADDRSETCFG_Pos)                           /*!< 0x00000001 */
#define SMC_CSTIM3_ADDRSETCFG_1                                     (0x2UL << SMC_CSTIM3_ADDRSETCFG_Pos)                           /*!< 0x00000002 */
#define SMC_CSTIM3_ADDRSETCFG_2                                     (0x4UL << SMC_CSTIM3_ADDRSETCFG_Pos)                           /*!< 0x00000004 */
#define SMC_CSTIM3_ADDRSETCFG_3                                     (0x8UL << SMC_CSTIM3_ADDRSETCFG_Pos)                           /*!< 0x00000008 */

#define SMC_CSTIM3_ADDRHLDCFG_Pos                                   (4U)
#define SMC_CSTIM3_ADDRHLDCFG_Msk                                   (0xFUL << SMC_CSTIM3_ADDRHLDCFG_Pos)                           /*!< 0x000000F0 */
#define SMC_CSTIM3_ADDRHLDCFG                                       SMC_CSTIM3_ADDRHLDCFG_Msk                                      /*!<ADDHLD[3:0] bits (Address-hold phase duration) */
#define SMC_CSTIM3_ADDRHLDCFG_0                                     (0x1UL << SMC_CSTIM3_ADDRHLDCFG_Pos)                           /*!< 0x00000010 */
#define SMC_CSTIM3_ADDRHLDCFG_1                                     (0x2UL << SMC_CSTIM3_ADDRHLDCFG_Pos)                           /*!< 0x00000020 */
#define SMC_CSTIM3_ADDRHLDCFG_2                                     (0x4UL << SMC_CSTIM3_ADDRHLDCFG_Pos)                           /*!< 0x00000040 */
#define SMC_CSTIM3_ADDRHLDCFG_3                                     (0x8UL << SMC_CSTIM3_ADDRHLDCFG_Pos)                           /*!< 0x00000080 */

#define SMC_CSTIM3_DATASETCFG_Pos                                   (8U)
#define SMC_CSTIM3_DATASETCFG_Msk                                   (0xFFUL << SMC_CSTIM3_DATASETCFG_Pos)                          /*!< 0x0000FF00 */
#define SMC_CSTIM3_DATASETCFG                                       SMC_CSTIM3_DATASETCFG_Msk                                      /*!<DATAST [7:0] bits (Data-phase duration) */
#define SMC_CSTIM3_DATASETCFG_0                                     (0x01UL << SMC_CSTIM3_DATASETCFG_Pos)                          /*!< 0x00000100 */
#define SMC_CSTIM3_DATASETCFG_1                                     (0x02UL << SMC_CSTIM3_DATASETCFG_Pos)                          /*!< 0x00000200 */
#define SMC_CSTIM3_DATASETCFG_2                                     (0x04UL << SMC_CSTIM3_DATASETCFG_Pos)                          /*!< 0x00000400 */
#define SMC_CSTIM3_DATASETCFG_3                                     (0x08UL << SMC_CSTIM3_DATASETCFG_Pos)                          /*!< 0x00000800 */
#define SMC_CSTIM3_DATASETCFG_4                                     (0x10UL << SMC_CSTIM3_DATASETCFG_Pos)                          /*!< 0x00001000 */
#define SMC_CSTIM3_DATASETCFG_5                                     (0x20UL << SMC_CSTIM3_DATASETCFG_Pos)                          /*!< 0x00002000 */
#define SMC_CSTIM3_DATASETCFG_6                                     (0x40UL << SMC_CSTIM3_DATASETCFG_Pos)                          /*!< 0x00004000 */
#define SMC_CSTIM3_DATASETCFG_7                                     (0x80UL << SMC_CSTIM3_DATASETCFG_Pos)                          /*!< 0x00008000 */

#define SMC_CSTIM3_BUSTURNCFG_Pos                                   (16U)
#define SMC_CSTIM3_BUSTURNCFG_Msk                                   (0xFUL << SMC_CSTIM3_BUSTURNCFG_Pos)                           /*!< 0x000F0000 */
#define SMC_CSTIM3_BUSTURNCFG                                       SMC_CSTIM3_BUSTURNCFG_Msk                                      /*!<BUSTURN[3:0] bits (Bus turnaround phase duration) */
#define SMC_CSTIM3_BUSTURNCFG_0                                     (0x1UL << SMC_CSTIM3_BUSTURNCFG_Pos)                           /*!< 0x00010000 */
#define SMC_CSTIM3_BUSTURNCFG_1                                     (0x2UL << SMC_CSTIM3_BUSTURNCFG_Pos)                           /*!< 0x00020000 */
#define SMC_CSTIM3_BUSTURNCFG_2                                     (0x4UL << SMC_CSTIM3_BUSTURNCFG_Pos)                           /*!< 0x00040000 */
#define SMC_CSTIM3_BUSTURNCFG_3                                     (0x8UL << SMC_CSTIM3_BUSTURNCFG_Pos)                           /*!< 0x00080000 */

#define SMC_CSTIM3_CLKDIVCFG_Pos                                    (20U)
#define SMC_CSTIM3_CLKDIVCFG_Msk                                    (0xFUL << SMC_CSTIM3_CLKDIVCFG_Pos)                            /*!< 0x00F00000 */
#define SMC_CSTIM3_CLKDIVCFG                                        SMC_CSTIM3_CLKDIVCFG_Msk                                       /*!<CLKDIV[3:0] bits (Clock divide ratio) */
#define SMC_CSTIM3_CLKDIVCFG_0                                      (0x1UL << SMC_CSTIM3_CLKDIVCFG_Pos)                            /*!< 0x00100000 */
#define SMC_CSTIM3_CLKDIVCFG_1                                      (0x2UL << SMC_CSTIM3_CLKDIVCFG_Pos)                            /*!< 0x00200000 */
#define SMC_CSTIM3_CLKDIVCFG_2                                      (0x4UL << SMC_CSTIM3_CLKDIVCFG_Pos)                            /*!< 0x00400000 */
#define SMC_CSTIM3_CLKDIVCFG_3                                      (0x8UL << SMC_CSTIM3_CLKDIVCFG_Pos)                            /*!< 0x00800000 */

#define SMC_CSTIM3_DATALATCFG_Pos                                   (24U)
#define SMC_CSTIM3_DATALATCFG_Msk                                   (0xFUL << SMC_CSTIM3_DATALATCFG_Pos)                           /*!< 0x0F000000 */
#define SMC_CSTIM3_DATALATCFG                                       SMC_CSTIM3_DATALATCFG_Msk                                      /*!<DATLA[3:0] bits (Data latency) */
#define SMC_CSTIM3_DATALATCFG_0                                     (0x1UL << SMC_CSTIM3_DATALATCFG_Pos)                           /*!< 0x01000000 */
#define SMC_CSTIM3_DATALATCFG_1                                     (0x2UL << SMC_CSTIM3_DATALATCFG_Pos)                           /*!< 0x02000000 */
#define SMC_CSTIM3_DATALATCFG_2                                     (0x4UL << SMC_CSTIM3_DATALATCFG_Pos)                           /*!< 0x04000000 */
#define SMC_CSTIM3_DATALATCFG_3                                     (0x8UL << SMC_CSTIM3_DATALATCFG_Pos)                           /*!< 0x08000000 */

#define SMC_CSTIM3_ASYNCACCCFG_Pos                                  (28U)
#define SMC_CSTIM3_ASYNCACCCFG_Msk                                  (0x3UL << SMC_CSTIM3_ASYNCACCCFG_Pos)                          /*!< 0x30000000 */
#define SMC_CSTIM3_ASYNCACCCFG                                      SMC_CSTIM3_ASYNCACCCFG_Msk                                     /*!<ACCMOD[1:0] bits (Access mode) */
#define SMC_CSTIM3_ASYNCACCCFG_0                                    (0x1UL << SMC_CSTIM3_ASYNCACCCFG_Pos)                          /*!< 0x10000000 */
#define SMC_CSTIM3_ASYNCACCCFG_1                                    (0x2UL << SMC_CSTIM3_ASYNCACCCFG_Pos)                          /*!< 0x20000000 */

/******************  Bit definition for SMC_CSTIM4 register  *******************/
#define SMC_CSTIM4_ADDRSETCFG_Pos                                   (0U)
#define SMC_CSTIM4_ADDRSETCFG_Msk                                   (0xFUL << SMC_CSTIM4_ADDRSETCFG_Pos)                           /*!< 0x0000000F */
#define SMC_CSTIM4_ADDRSETCFG                                       SMC_CSTIM4_ADDRSETCFG_Msk                                      /*!<ADDSET[3:0] bits (Address setup phase duration) */
#define SMC_CSTIM4_ADDRSETCFG_0                                     (0x1UL << SMC_CSTIM4_ADDRSETCFG_Pos)                           /*!< 0x00000001 */
#define SMC_CSTIM4_ADDRSETCFG_1                                     (0x2UL << SMC_CSTIM4_ADDRSETCFG_Pos)                           /*!< 0x00000002 */
#define SMC_CSTIM4_ADDRSETCFG_2                                     (0x4UL << SMC_CSTIM4_ADDRSETCFG_Pos)                           /*!< 0x00000004 */
#define SMC_CSTIM4_ADDRSETCFG_3                                     (0x8UL << SMC_CSTIM4_ADDRSETCFG_Pos)                           /*!< 0x00000008 */

#define SMC_CSTIM4_ADDRHLDCFG_Pos                                   (4U)
#define SMC_CSTIM4_ADDRHLDCFG_Msk                                   (0xFUL << SMC_CSTIM4_ADDRHLDCFG_Pos)                           /*!< 0x000000F0 */
#define SMC_CSTIM4_ADDRHLDCFG                                       SMC_CSTIM4_ADDRHLDCFG_Msk                                      /*!<ADDHLD[3:0] bits (Address-hold phase duration) */
#define SMC_CSTIM4_ADDRHLDCFG_0                                     (0x1UL << SMC_CSTIM4_ADDRHLDCFG_Pos)                           /*!< 0x00000010 */
#define SMC_CSTIM4_ADDRHLDCFG_1                                     (0x2UL << SMC_CSTIM4_ADDRHLDCFG_Pos)                           /*!< 0x00000020 */
#define SMC_CSTIM4_ADDRHLDCFG_2                                     (0x4UL << SMC_CSTIM4_ADDRHLDCFG_Pos)                           /*!< 0x00000040 */
#define SMC_CSTIM4_ADDRHLDCFG_3                                     (0x8UL << SMC_CSTIM4_ADDRHLDCFG_Pos)                           /*!< 0x00000080 */

#define SMC_CSTIM4_DATASETCFG_Pos                                   (8U)
#define SMC_CSTIM4_DATASETCFG_Msk                                   (0xFFUL << SMC_CSTIM4_DATASETCFG_Pos)                          /*!< 0x0000FF00 */
#define SMC_CSTIM4_DATASETCFG                                       SMC_CSTIM4_DATASETCFG_Msk                                      /*!<DATAST [7:0] bits (Data-phase duration) */
#define SMC_CSTIM4_DATASETCFG_0                                     (0x01UL << SMC_CSTIM4_DATASETCFG_Pos)                          /*!< 0x00000100 */
#define SMC_CSTIM4_DATASETCFG_1                                     (0x02UL << SMC_CSTIM4_DATASETCFG_Pos)                          /*!< 0x00000200 */
#define SMC_CSTIM4_DATASETCFG_2                                     (0x04UL << SMC_CSTIM4_DATASETCFG_Pos)                          /*!< 0x00000400 */
#define SMC_CSTIM4_DATASETCFG_3                                     (0x08UL << SMC_CSTIM4_DATASETCFG_Pos)                          /*!< 0x00000800 */
#define SMC_CSTIM4_DATASETCFG_4                                     (0x10UL << SMC_CSTIM4_DATASETCFG_Pos)                          /*!< 0x00001000 */
#define SMC_CSTIM4_DATASETCFG_5                                     (0x20UL << SMC_CSTIM4_DATASETCFG_Pos)                          /*!< 0x00002000 */
#define SMC_CSTIM4_DATASETCFG_6                                     (0x40UL << SMC_CSTIM4_DATASETCFG_Pos)                          /*!< 0x00004000 */
#define SMC_CSTIM4_DATASETCFG_7                                     (0x80UL << SMC_CSTIM4_DATASETCFG_Pos)                          /*!< 0x00008000 */

#define SMC_CSTIM4_BUSTURNCFG_Pos                                   (16U)
#define SMC_CSTIM4_BUSTURNCFG_Msk                                   (0xFUL << SMC_CSTIM4_BUSTURNCFG_Pos)                           /*!< 0x000F0000 */
#define SMC_CSTIM4_BUSTURNCFG                                       SMC_CSTIM4_BUSTURNCFG_Msk                                      /*!<BUSTURN[3:0] bits (Bus turnaround phase duration) */
#define SMC_CSTIM4_BUSTURNCFG_0                                     (0x1UL << SMC_CSTIM4_BUSTURNCFG_Pos)                           /*!< 0x00010000 */
#define SMC_CSTIM4_BUSTURNCFG_1                                     (0x2UL << SMC_CSTIM4_BUSTURNCFG_Pos)                           /*!< 0x00020000 */
#define SMC_CSTIM4_BUSTURNCFG_2                                     (0x4UL << SMC_CSTIM4_BUSTURNCFG_Pos)                           /*!< 0x00040000 */
#define SMC_CSTIM4_BUSTURNCFG_3                                     (0x8UL << SMC_CSTIM4_BUSTURNCFG_Pos)                           /*!< 0x00080000 */

#define SMC_CSTIM4_CLKDIVCFG_Pos                                    (20U)
#define SMC_CSTIM4_CLKDIVCFG_Msk                                    (0xFUL << SMC_CSTIM4_CLKDIVCFG_Pos)                            /*!< 0x00F00000 */
#define SMC_CSTIM4_CLKDIVCFG                                        SMC_CSTIM4_CLKDIVCFG_Msk                                       /*!<CLKDIV[3:0] bits (Clock divide ratio) */
#define SMC_CSTIM4_CLKDIVCFG_0                                      (0x1UL << SMC_CSTIM4_CLKDIVCFG_Pos)                            /*!< 0x00100000 */
#define SMC_CSTIM4_CLKDIVCFG_1                                      (0x2UL << SMC_CSTIM4_CLKDIVCFG_Pos)                            /*!< 0x00200000 */
#define SMC_CSTIM4_CLKDIVCFG_2                                      (0x4UL << SMC_CSTIM4_CLKDIVCFG_Pos)                            /*!< 0x00400000 */
#define SMC_CSTIM4_CLKDIVCFG_3                                      (0x8UL << SMC_CSTIM4_CLKDIVCFG_Pos)                            /*!< 0x00800000 */

#define SMC_CSTIM4_DATALATCFG_Pos                                   (24U)
#define SMC_CSTIM4_DATALATCFG_Msk                                   (0xFUL << SMC_CSTIM4_DATALATCFG_Pos)                           /*!< 0x0F000000 */
#define SMC_CSTIM4_DATALATCFG                                       SMC_CSTIM4_DATALATCFG_Msk                                      /*!<DATLA[3:0] bits (Data latency) */
#define SMC_CSTIM4_DATALATCFG_0                                     (0x1UL << SMC_CSTIM4_DATALATCFG_Pos)                           /*!< 0x01000000 */
#define SMC_CSTIM4_DATALATCFG_1                                     (0x2UL << SMC_CSTIM4_DATALATCFG_Pos)                           /*!< 0x02000000 */
#define SMC_CSTIM4_DATALATCFG_2                                     (0x4UL << SMC_CSTIM4_DATALATCFG_Pos)                           /*!< 0x04000000 */
#define SMC_CSTIM4_DATALATCFG_3                                     (0x8UL << SMC_CSTIM4_DATALATCFG_Pos)                           /*!< 0x08000000 */

#define SMC_CSTIM4_ASYNCACCCFG_Pos                                  (28U)
#define SMC_CSTIM4_ASYNCACCCFG_Msk                                  (0x3UL << SMC_CSTIM4_ASYNCACCCFG_Pos)                          /*!< 0x30000000 */
#define SMC_CSTIM4_ASYNCACCCFG                                      SMC_CSTIM4_ASYNCACCCFG_Msk                                     /*!<ACCMOD[1:0] bits (Access mode) */
#define SMC_CSTIM4_ASYNCACCCFG_0                                    (0x1UL << SMC_CSTIM4_ASYNCACCCFG_Pos)                          /*!< 0x10000000 */
#define SMC_CSTIM4_ASYNCACCCFG_1                                    (0x2UL << SMC_CSTIM4_ASYNCACCCFG_Pos)                          /*!< 0x20000000 */

/******************  Bit definition for SMC_WRTTIM1 register  ******************/
#define SMC_WRTTIM1_ADDRSETCFG_Pos                                  (0U)
#define SMC_WRTTIM1_ADDRSETCFG_Msk                                  (0xFUL << SMC_WRTTIM1_ADDRSETCFG_Pos)                          /*!< 0x0000000F */
#define SMC_WRTTIM1_ADDRSETCFG                                      SMC_WRTTIM1_ADDRSETCFG_Msk                                     /*!<ADDSET[3:0] bits (Address setup phase duration) */
#define SMC_WRTTIM1_ADDRSETCFG_0                                    (0x1UL << SMC_WRTTIM1_ADDRSETCFG_Pos)                          /*!< 0x00000001 */
#define SMC_WRTTIM1_ADDRSETCFG_1                                    (0x2UL << SMC_WRTTIM1_ADDRSETCFG_Pos)                          /*!< 0x00000002 */
#define SMC_WRTTIM1_ADDRSETCFG_2                                    (0x4UL << SMC_WRTTIM1_ADDRSETCFG_Pos)                          /*!< 0x00000004 */
#define SMC_WRTTIM1_ADDRSETCFG_3                                    (0x8UL << SMC_WRTTIM1_ADDRSETCFG_Pos)                          /*!< 0x00000008 */

#define SMC_WRTTIM1_ADDRHLDCFG_Pos                                  (4U)
#define SMC_WRTTIM1_ADDRHLDCFG_Msk                                  (0xFUL << SMC_WRTTIM1_ADDRHLDCFG_Pos)                          /*!< 0x000000F0 */
#define SMC_WRTTIM1_ADDRHLDCFG                                      SMC_WRTTIM1_ADDRHLDCFG_Msk                                     /*!<ADDHLD[3:0] bits (Address-hold phase duration) */
#define SMC_WRTTIM1_ADDRHLDCFG_0                                    (0x1UL << SMC_WRTTIM1_ADDRHLDCFG_Pos)                          /*!< 0x00000010 */
#define SMC_WRTTIM1_ADDRHLDCFG_1                                    (0x2UL << SMC_WRTTIM1_ADDRHLDCFG_Pos)                          /*!< 0x00000020 */
#define SMC_WRTTIM1_ADDRHLDCFG_2                                    (0x4UL << SMC_WRTTIM1_ADDRHLDCFG_Pos)                          /*!< 0x00000040 */
#define SMC_WRTTIM1_ADDRHLDCFG_3                                    (0x8UL << SMC_WRTTIM1_ADDRHLDCFG_Pos)                          /*!< 0x00000080 */

#define SMC_WRTTIM1_DATASETCFG_Pos                                  (8U)
#define SMC_WRTTIM1_DATASETCFG_Msk                                  (0xFFUL << SMC_WRTTIM1_DATASETCFG_Pos)                         /*!< 0x0000FF00 */
#define SMC_WRTTIM1_DATASETCFG                                      SMC_WRTTIM1_DATASETCFG_Msk                                     /*!<DATAST [7:0] bits (Data-phase duration) */
#define SMC_WRTTIM1_DATASETCFG_0                                    (0x01UL << SMC_WRTTIM1_DATASETCFG_Pos)                         /*!< 0x00000100 */
#define SMC_WRTTIM1_DATASETCFG_1                                    (0x02UL << SMC_WRTTIM1_DATASETCFG_Pos)                         /*!< 0x00000200 */
#define SMC_WRTTIM1_DATASETCFG_2                                    (0x04UL << SMC_WRTTIM1_DATASETCFG_Pos)                         /*!< 0x00000400 */
#define SMC_WRTTIM1_DATASETCFG_3                                    (0x08UL << SMC_WRTTIM1_DATASETCFG_Pos)                         /*!< 0x00000800 */
#define SMC_WRTTIM1_DATASETCFG_4                                    (0x10UL << SMC_WRTTIM1_DATASETCFG_Pos)                         /*!< 0x00001000 */
#define SMC_WRTTIM1_DATASETCFG_5                                    (0x20UL << SMC_WRTTIM1_DATASETCFG_Pos)                         /*!< 0x00002000 */
#define SMC_WRTTIM1_DATASETCFG_6                                    (0x40UL << SMC_WRTTIM1_DATASETCFG_Pos)                         /*!< 0x00004000 */
#define SMC_WRTTIM1_DATASETCFG_7                                    (0x80UL << SMC_WRTTIM1_DATASETCFG_Pos)                         /*!< 0x00008000 */

#define SMC_WRTTIM1_BUSTURNCFG_Pos                                  (16U)
#define SMC_WRTTIM1_BUSTURNCFG_Msk                                  (0xFUL << SMC_WRTTIM1_BUSTURNCFG_Pos)                          /*!< 0x000F0000 */
#define SMC_WRTTIM1_BUSTURNCFG                                      SMC_WRTTIM1_BUSTURNCFG_Msk                                     /*!<BUSTURN[3:0] bits (Bus turnaround duration) */
#define SMC_WRTTIM1_BUSTURNCFG_0                                    (0x1UL << SMC_WRTTIM1_BUSTURNCFG_Pos)                          /*!< 0x00010000 */
#define SMC_WRTTIM1_BUSTURNCFG_1                                    (0x2UL << SMC_WRTTIM1_BUSTURNCFG_Pos)                          /*!< 0x00020000 */
#define SMC_WRTTIM1_BUSTURNCFG_2                                    (0x4UL << SMC_WRTTIM1_BUSTURNCFG_Pos)                          /*!< 0x00040000 */
#define SMC_WRTTIM1_BUSTURNCFG_3                                    (0x8UL << SMC_WRTTIM1_BUSTURNCFG_Pos)                          /*!< 0x00080000 */

#define SMC_WRTTIM1_ASYNCACCCFG_Pos                                 (28U)
#define SMC_WRTTIM1_ASYNCACCCFG_Msk                                 (0x3UL << SMC_WRTTIM1_ASYNCACCCFG_Pos)                         /*!< 0x30000000 */
#define SMC_WRTTIM1_ASYNCACCCFG                                     SMC_WRTTIM1_ASYNCACCCFG_Msk                                    /*!<ACCMOD[1:0] bits (Access mode) */
#define SMC_WRTTIM1_ASYNCACCCFG_0                                   (0x1UL << SMC_WRTTIM1_ASYNCACCCFG_Pos)                         /*!< 0x10000000 */
#define SMC_WRTTIM1_ASYNCACCCFG_1                                   (0x2UL << SMC_WRTTIM1_ASYNCACCCFG_Pos)                         /*!< 0x20000000 */

/******************  Bit definition for SMC_WRTTIM2 register  ******************/
#define SMC_WRTTIM2_ADDRSETCFG_Pos                                  (0U)
#define SMC_WRTTIM2_ADDRSETCFG_Msk                                  (0xFUL << SMC_WRTTIM2_ADDRSETCFG_Pos)                          /*!< 0x0000000F */
#define SMC_WRTTIM2_ADDRSETCFG                                      SMC_WRTTIM2_ADDRSETCFG_Msk                                     /*!<ADDSET[3:0] bits (Address setup phase duration) */
#define SMC_WRTTIM2_ADDRSETCFG_0                                    (0x1UL << SMC_WRTTIM2_ADDRSETCFG_Pos)                          /*!< 0x00000001 */
#define SMC_WRTTIM2_ADDRSETCFG_1                                    (0x2UL << SMC_WRTTIM2_ADDRSETCFG_Pos)                          /*!< 0x00000002 */
#define SMC_WRTTIM2_ADDRSETCFG_2                                    (0x4UL << SMC_WRTTIM2_ADDRSETCFG_Pos)                          /*!< 0x00000004 */
#define SMC_WRTTIM2_ADDRSETCFG_3                                    (0x8UL << SMC_WRTTIM2_ADDRSETCFG_Pos)                          /*!< 0x00000008 */

#define SMC_WRTTIM2_ADDRHLDCFG_Pos                                  (4U)
#define SMC_WRTTIM2_ADDRHLDCFG_Msk                                  (0xFUL << SMC_WRTTIM2_ADDRHLDCFG_Pos)                          /*!< 0x000000F0 */
#define SMC_WRTTIM2_ADDRHLDCFG                                      SMC_WRTTIM2_ADDRHLDCFG_Msk                                     /*!<ADDHLD[3:0] bits (Address-hold phase duration) */
#define SMC_WRTTIM2_ADDRHLDCFG_0                                    (0x1UL << SMC_WRTTIM2_ADDRHLDCFG_Pos)                          /*!< 0x00000010 */
#define SMC_WRTTIM2_ADDRHLDCFG_1                                    (0x2UL << SMC_WRTTIM2_ADDRHLDCFG_Pos)                          /*!< 0x00000020 */
#define SMC_WRTTIM2_ADDRHLDCFG_2                                    (0x4UL << SMC_WRTTIM2_ADDRHLDCFG_Pos)                          /*!< 0x00000040 */
#define SMC_WRTTIM2_ADDRHLDCFG_3                                    (0x8UL << SMC_WRTTIM2_ADDRHLDCFG_Pos)                          /*!< 0x00000080 */

#define SMC_WRTTIM2_DATASETCFG_Pos                                  (8U)
#define SMC_WRTTIM2_DATASETCFG_Msk                                  (0xFFUL << SMC_WRTTIM2_DATASETCFG_Pos)                         /*!< 0x0000FF00 */
#define SMC_WRTTIM2_DATASETCFG                                      SMC_WRTTIM2_DATASETCFG_Msk                                     /*!<DATAST [7:0] bits (Data-phase duration) */
#define SMC_WRTTIM2_DATASETCFG_0                                    (0x01UL << SMC_WRTTIM2_DATASETCFG_Pos)                         /*!< 0x00000100 */
#define SMC_WRTTIM2_DATASETCFG_1                                    (0x02UL << SMC_WRTTIM2_DATASETCFG_Pos)                         /*!< 0x00000200 */
#define SMC_WRTTIM2_DATASETCFG_2                                    (0x04UL << SMC_WRTTIM2_DATASETCFG_Pos)                         /*!< 0x00000400 */
#define SMC_WRTTIM2_DATASETCFG_3                                    (0x08UL << SMC_WRTTIM2_DATASETCFG_Pos)                         /*!< 0x00000800 */
#define SMC_WRTTIM2_DATASETCFG_4                                    (0x10UL << SMC_WRTTIM2_DATASETCFG_Pos)                         /*!< 0x00001000 */
#define SMC_WRTTIM2_DATASETCFG_5                                    (0x20UL << SMC_WRTTIM2_DATASETCFG_Pos)                         /*!< 0x00002000 */
#define SMC_WRTTIM2_DATASETCFG_6                                    (0x40UL << SMC_WRTTIM2_DATASETCFG_Pos)                         /*!< 0x00004000 */
#define SMC_WRTTIM2_DATASETCFG_7                                    (0x80UL << SMC_WRTTIM2_DATASETCFG_Pos)                         /*!< 0x00008000 */

#define SMC_WRTTIM2_BUSTURNCFG_Pos                                  (16U)
#define SMC_WRTTIM2_BUSTURNCFG_Msk                                  (0xFUL << SMC_WRTTIM2_BUSTURNCFG_Pos)                          /*!< 0x000F0000 */
#define SMC_WRTTIM2_BUSTURNCFG                                      SMC_WRTTIM2_BUSTURNCFG_Msk                                     /*!<BUSTURN[3:0] bits (Bus turnaround duration) */
#define SMC_WRTTIM2_BUSTURNCFG_0                                    (0x1UL << SMC_WRTTIM2_BUSTURNCFG_Pos)                          /*!< 0x00010000 */
#define SMC_WRTTIM2_BUSTURNCFG_1                                    (0x2UL << SMC_WRTTIM2_BUSTURNCFG_Pos)                          /*!< 0x00020000 */
#define SMC_WRTTIM2_BUSTURNCFG_2                                    (0x4UL << SMC_WRTTIM2_BUSTURNCFG_Pos)                          /*!< 0x00040000 */
#define SMC_WRTTIM2_BUSTURNCFG_3                                    (0x8UL << SMC_WRTTIM2_BUSTURNCFG_Pos)                          /*!< 0x00080000 */

#define SMC_WRTTIM2_ASYNCACCCFG_Pos                                 (28U)
#define SMC_WRTTIM2_ASYNCACCCFG_Msk                                 (0x3UL << SMC_WRTTIM2_ASYNCACCCFG_Pos)                         /*!< 0x30000000 */
#define SMC_WRTTIM2_ASYNCACCCFG                                     SMC_WRTTIM2_ASYNCACCCFG_Msk                                    /*!<ACCMOD[1:0] bits (Access mode) */
#define SMC_WRTTIM2_ASYNCACCCFG_0                                   (0x1UL << SMC_WRTTIM2_ASYNCACCCFG_Pos)                         /*!< 0x10000000 */
#define SMC_WRTTIM2_ASYNCACCCFG_1                                   (0x2UL << SMC_WRTTIM2_ASYNCACCCFG_Pos)                         /*!< 0x20000000 */

/******************  Bit definition for SMC_WRTTIM3 register  ******************/
#define SMC_WRTTIM3_ADDRSETCFG_Pos                                  (0U)
#define SMC_WRTTIM3_ADDRSETCFG_Msk                                  (0xFUL << SMC_WRTTIM3_ADDRSETCFG_Pos)                          /*!< 0x0000000F */
#define SMC_WRTTIM3_ADDRSETCFG                                      SMC_WRTTIM3_ADDRSETCFG_Msk                                     /*!<ADDSET[3:0] bits (Address setup phase duration) */
#define SMC_WRTTIM3_ADDRSETCFG_0                                    (0x1UL << SMC_WRTTIM3_ADDRSETCFG_Pos)                          /*!< 0x00000001 */
#define SMC_WRTTIM3_ADDRSETCFG_1                                    (0x2UL << SMC_WRTTIM3_ADDRSETCFG_Pos)                          /*!< 0x00000002 */
#define SMC_WRTTIM3_ADDRSETCFG_2                                    (0x4UL << SMC_WRTTIM3_ADDRSETCFG_Pos)                          /*!< 0x00000004 */
#define SMC_WRTTIM3_ADDRSETCFG_3                                    (0x8UL << SMC_WRTTIM3_ADDRSETCFG_Pos)                          /*!< 0x00000008 */

#define SMC_WRTTIM3_ADDRHLDCFG_Pos                                  (4U)
#define SMC_WRTTIM3_ADDRHLDCFG_Msk                                  (0xFUL << SMC_WRTTIM3_ADDRHLDCFG_Pos)                          /*!< 0x000000F0 */
#define SMC_WRTTIM3_ADDRHLDCFG                                      SMC_WRTTIM3_ADDRHLDCFG_Msk                                     /*!<ADDHLD[3:0] bits (Address-hold phase duration) */
#define SMC_WRTTIM3_ADDRHLDCFG_0                                    (0x1UL << SMC_WRTTIM3_ADDRHLDCFG_Pos)                          /*!< 0x00000010 */
#define SMC_WRTTIM3_ADDRHLDCFG_1                                    (0x2UL << SMC_WRTTIM3_ADDRHLDCFG_Pos)                          /*!< 0x00000020 */
#define SMC_WRTTIM3_ADDRHLDCFG_2                                    (0x4UL << SMC_WRTTIM3_ADDRHLDCFG_Pos)                          /*!< 0x00000040 */
#define SMC_WRTTIM3_ADDRHLDCFG_3                                    (0x8UL << SMC_WRTTIM3_ADDRHLDCFG_Pos)                          /*!< 0x00000080 */

#define SMC_WRTTIM3_DATASETCFG_Pos                                  (8U)
#define SMC_WRTTIM3_DATASETCFG_Msk                                  (0xFFUL << SMC_WRTTIM3_DATASETCFG_Pos)                         /*!< 0x0000FF00 */
#define SMC_WRTTIM3_DATASETCFG                                      SMC_WRTTIM3_DATASETCFG_Msk                                     /*!<DATAST [7:0] bits (Data-phase duration) */
#define SMC_WRTTIM3_DATASETCFG_0                                    (0x01UL << SMC_WRTTIM3_DATASETCFG_Pos)                         /*!< 0x00000100 */
#define SMC_WRTTIM3_DATASETCFG_1                                    (0x02UL << SMC_WRTTIM3_DATASETCFG_Pos)                         /*!< 0x00000200 */
#define SMC_WRTTIM3_DATASETCFG_2                                    (0x04UL << SMC_WRTTIM3_DATASETCFG_Pos)                         /*!< 0x00000400 */
#define SMC_WRTTIM3_DATASETCFG_3                                    (0x08UL << SMC_WRTTIM3_DATASETCFG_Pos)                         /*!< 0x00000800 */
#define SMC_WRTTIM3_DATASETCFG_4                                    (0x10UL << SMC_WRTTIM3_DATASETCFG_Pos)                         /*!< 0x00001000 */
#define SMC_WRTTIM3_DATASETCFG_5                                    (0x20UL << SMC_WRTTIM3_DATASETCFG_Pos)                         /*!< 0x00002000 */
#define SMC_WRTTIM3_DATASETCFG_6                                    (0x40UL << SMC_WRTTIM3_DATASETCFG_Pos)                         /*!< 0x00004000 */
#define SMC_WRTTIM3_DATASETCFG_7                                    (0x80UL << SMC_WRTTIM3_DATASETCFG_Pos)                         /*!< 0x00008000 */

#define SMC_WRTTIM3_BUSTURNCFG_Pos                                  (16U)
#define SMC_WRTTIM3_BUSTURNCFG_Msk                                  (0xFUL << SMC_WRTTIM3_BUSTURNCFG_Pos)                          /*!< 0x000F0000 */
#define SMC_WRTTIM3_BUSTURNCFG                                      SMC_WRTTIM3_BUSTURNCFG_Msk                                     /*!<BUSTURN[3:0] bits (Bus turnaround duration) */
#define SMC_WRTTIM3_BUSTURNCFG_0                                    (0x1UL << SMC_WRTTIM3_BUSTURNCFG_Pos)                          /*!< 0x00010000 */
#define SMC_WRTTIM3_BUSTURNCFG_1                                    (0x2UL << SMC_WRTTIM3_BUSTURNCFG_Pos)                          /*!< 0x00020000 */
#define SMC_WRTTIM3_BUSTURNCFG_2                                    (0x4UL << SMC_WRTTIM3_BUSTURNCFG_Pos)                          /*!< 0x00040000 */
#define SMC_WRTTIM3_BUSTURNCFG_3                                    (0x8UL << SMC_WRTTIM3_BUSTURNCFG_Pos)                          /*!< 0x00080000 */

#define SMC_WRTTIM3_ASYNCACCCFG_Pos                                 (28U)
#define SMC_WRTTIM3_ASYNCACCCFG_Msk                                 (0x3UL << SMC_WRTTIM3_ASYNCACCCFG_Pos)                         /*!< 0x30000000 */
#define SMC_WRTTIM3_ASYNCACCCFG                                     SMC_WRTTIM3_ASYNCACCCFG_Msk                                    /*!<ACCMOD[1:0] bits (Access mode) */
#define SMC_WRTTIM3_ASYNCACCCFG_0                                   (0x1UL << SMC_WRTTIM3_ASYNCACCCFG_Pos)                         /*!< 0x10000000 */
#define SMC_WRTTIM3_ASYNCACCCFG_1                                   (0x2UL << SMC_WRTTIM3_ASYNCACCCFG_Pos)                         /*!< 0x20000000 */

/******************  Bit definition for SMC_WRTTIM4 register  ******************/
#define SMC_WRTTIM4_ADDRSETCFG_Pos                                  (0U)
#define SMC_WRTTIM4_ADDRSETCFG_Msk                                  (0xFUL << SMC_WRTTIM4_ADDRSETCFG_Pos)                          /*!< 0x0000000F */
#define SMC_WRTTIM4_ADDRSETCFG                                      SMC_WRTTIM4_ADDRSETCFG_Msk                                     /*!<ADDSET[3:0] bits (Address setup phase duration) */
#define SMC_WRTTIM4_ADDRSETCFG_0                                    (0x1UL << SMC_WRTTIM4_ADDRSETCFG_Pos)                          /*!< 0x00000001 */
#define SMC_WRTTIM4_ADDRSETCFG_1                                    (0x2UL << SMC_WRTTIM4_ADDRSETCFG_Pos)                          /*!< 0x00000002 */
#define SMC_WRTTIM4_ADDRSETCFG_2                                    (0x4UL << SMC_WRTTIM4_ADDRSETCFG_Pos)                          /*!< 0x00000004 */
#define SMC_WRTTIM4_ADDRSETCFG_3                                    (0x8UL << SMC_WRTTIM4_ADDRSETCFG_Pos)                          /*!< 0x00000008 */

#define SMC_WRTTIM4_ADDRHLDCFG_Pos                                  (4U)
#define SMC_WRTTIM4_ADDRHLDCFG_Msk                                  (0xFUL << SMC_WRTTIM4_ADDRHLDCFG_Pos)                          /*!< 0x000000F0 */
#define SMC_WRTTIM4_ADDRHLDCFG                                      SMC_WRTTIM4_ADDRHLDCFG_Msk                                     /*!<ADDHLD[3:0] bits (Address-hold phase duration) */
#define SMC_WRTTIM4_ADDRHLDCFG_0                                    (0x1UL << SMC_WRTTIM4_ADDRHLDCFG_Pos)                          /*!< 0x00000010 */
#define SMC_WRTTIM4_ADDRHLDCFG_1                                    (0x2UL << SMC_WRTTIM4_ADDRHLDCFG_Pos)                          /*!< 0x00000020 */
#define SMC_WRTTIM4_ADDRHLDCFG_2                                    (0x4UL << SMC_WRTTIM4_ADDRHLDCFG_Pos)                          /*!< 0x00000040 */
#define SMC_WRTTIM4_ADDRHLDCFG_3                                    (0x8UL << SMC_WRTTIM4_ADDRHLDCFG_Pos)                          /*!< 0x00000080 */

#define SMC_WRTTIM4_DATASETCFG_Pos                                  (8U)
#define SMC_WRTTIM4_DATASETCFG_Msk                                  (0xFFUL << SMC_WRTTIM4_DATASETCFG_Pos)                         /*!< 0x0000FF00 */
#define SMC_WRTTIM4_DATASETCFG                                      SMC_WRTTIM4_DATASETCFG_Msk                                     /*!<DATAST [3:0] bits (Data-phase duration) */
#define SMC_WRTTIM4_DATASETCFG_0                                    0x00000100U                                                    /*!<Bit 0 */
#define SMC_WRTTIM4_DATASETCFG_1                                    0x00000200U                                                    /*!<Bit 1 */
#define SMC_WRTTIM4_DATASETCFG_2                                    0x00000400U                                                    /*!<Bit 2 */
#define SMC_WRTTIM4_DATASETCFG_3                                    0x00000800U                                                    /*!<Bit 3 */
#define SMC_WRTTIM4_DATASETCFG_4                                    0x00001000U                                                    /*!<Bit 4 */
#define SMC_WRTTIM4_DATASETCFG_5                                    0x00002000U                                                    /*!<Bit 5 */
#define SMC_WRTTIM4_DATASETCFG_6                                    0x00004000U                                                    /*!<Bit 6 */
#define SMC_WRTTIM4_DATASETCFG_7                                    0x00008000U                                                    /*!<Bit 7 */

#define SMC_WRTTIM4_BUSTURNCFG_Pos                                  (16U)
#define SMC_WRTTIM4_BUSTURNCFG_Msk                                  (0xFUL << SMC_WRTTIM4_BUSTURNCFG_Pos)                          /*!< 0x000F0000 */
#define SMC_WRTTIM4_BUSTURNCFG                                      SMC_WRTTIM4_BUSTURNCFG_Msk                                     /*!<BUSTURN[3:0] bits (Bus turnaround duration) */
#define SMC_WRTTIM4_BUSTURNCFG_0                                    (0x1UL << SMC_WRTTIM4_BUSTURNCFG_Pos)                          /*!< 0x00010000 */
#define SMC_WRTTIM4_BUSTURNCFG_1                                    (0x2UL << SMC_WRTTIM4_BUSTURNCFG_Pos)                          /*!< 0x00020000 */
#define SMC_WRTTIM4_BUSTURNCFG_2                                    (0x4UL << SMC_WRTTIM4_BUSTURNCFG_Pos)                          /*!< 0x00040000 */
#define SMC_WRTTIM4_BUSTURNCFG_3                                    (0x8UL << SMC_WRTTIM4_BUSTURNCFG_Pos)                          /*!< 0x00080000 */

#define SMC_WRTTIM4_ASYNCACCCFG_Pos                                 (28U)
#define SMC_WRTTIM4_ASYNCACCCFG_Msk                                 (0x3UL << SMC_WRTTIM4_ASYNCACCCFG_Pos)                         /*!< 0x30000000 */
#define SMC_WRTTIM4_ASYNCACCCFG                                     SMC_WRTTIM4_ASYNCACCCFG_Msk                                    /*!<ACCMOD[1:0] bits (Access mode) */
#define SMC_WRTTIM4_ASYNCACCCFG_0                                   (0x1UL << SMC_WRTTIM4_ASYNCACCCFG_Pos)                         /*!< 0x10000000 */
#define SMC_WRTTIM4_ASYNCACCCFG_1                                   (0x2UL << SMC_WRTTIM4_ASYNCACCCFG_Pos)                         /*!< 0x20000000 */

/******************  Bit definition for SMC_CTRL2 register  *******************/
#define SMC_CTRL2_WAITFEN_Pos                                       (1U)
#define SMC_CTRL2_WAITFEN_Msk                                       (0x1UL << SMC_CTRL2_WAITFEN_Pos)                               /*!< 0x00000002 */
#define SMC_CTRL2_WAITFEN                                           SMC_CTRL2_WAITFEN_Msk                                          /*!<Wait feature enable bit */
#define SMC_CTRL2_MBKEN_Pos                                         (2U)
#define SMC_CTRL2_MBKEN_Msk                                         (0x1UL << SMC_CTRL2_MBKEN_Pos)                                 /*!< 0x00000004 */
#define SMC_CTRL2_MBKEN                                             SMC_CTRL2_MBKEN_Msk                                            /*!<PC Card/NAND Flash memory bank enable bit */
#define SMC_CTRL2_MTYPECFG_Pos                                      (3U)
#define SMC_CTRL2_MTYPECFG_Msk                                      (0x1UL << SMC_CTRL2_MTYPECFG_Pos)                              /*!< 0x00000008 */
#define SMC_CTRL2_MTYPECFG                                          SMC_CTRL2_MTYPECFG_Msk                                         /*!<Memory type */

#define SMC_CTRL2_DBWIDCFG_Pos                                      (4U)
#define SMC_CTRL2_DBWIDCFG_Msk                                      (0x3UL << SMC_CTRL2_DBWIDCFG_Pos)                              /*!< 0x00000030 */
#define SMC_CTRL2_DBWIDCFG                                          SMC_CTRL2_DBWIDCFG_Msk                                         /*!<PWID[1:0] bits (NAND Flash databus width) */
#define SMC_CTRL2_DBWIDCFG_0                                        (0x1UL << SMC_CTRL2_DBWIDCFG_Pos)                              /*!< 0x00000010 */
#define SMC_CTRL2_DBWIDCFG_1                                        (0x2UL << SMC_CTRL2_DBWIDCFG_Pos)                              /*!< 0x00000020 */

#define SMC_CTRL2_ECCEN_Pos                                         (6U)
#define SMC_CTRL2_ECCEN_Msk                                         (0x1UL << SMC_CTRL2_ECCEN_Pos)                                 /*!< 0x00000040 */
#define SMC_CTRL2_ECCEN                                             SMC_CTRL2_ECCEN_Msk                                            /*!<ECC computation logic enable bit */

#define SMC_CTRL2_C2RDCFG_Pos                                       (9U)
#define SMC_CTRL2_C2RDCFG_Msk                                       (0xFUL << SMC_CTRL2_C2RDCFG_Pos)                               /*!< 0x00001E00 */
#define SMC_CTRL2_C2RDCFG                                           SMC_CTRL2_C2RDCFG_Msk                                          /*!<TCLR[3:0] bits (CLE to RE delay) */
#define SMC_CTRL2_C2RDCFG_0                                         (0x1UL << SMC_CTRL2_C2RDCFG_Pos)                               /*!< 0x00000200 */
#define SMC_CTRL2_C2RDCFG_1                                         (0x2UL << SMC_CTRL2_C2RDCFG_Pos)                               /*!< 0x00000400 */
#define SMC_CTRL2_C2RDCFG_2                                         (0x4UL << SMC_CTRL2_C2RDCFG_Pos)                               /*!< 0x00000800 */
#define SMC_CTRL2_C2RDCFG_3                                         (0x8UL << SMC_CTRL2_C2RDCFG_Pos)                               /*!< 0x00001000 */

#define SMC_CTRL2_A2RDCFG_Pos                                       (13U)
#define SMC_CTRL2_A2RDCFG_Msk                                       (0xFUL << SMC_CTRL2_A2RDCFG_Pos)                               /*!< 0x0001E000 */
#define SMC_CTRL2_A2RDCFG                                           SMC_CTRL2_A2RDCFG_Msk                                          /*!<TAR[3:0] bits (ALE to RE delay) */
#define SMC_CTRL2_A2RDCFG_0                                         (0x1UL << SMC_CTRL2_A2RDCFG_Pos)                               /*!< 0x00002000 */
#define SMC_CTRL2_A2RDCFG_1                                         (0x2UL << SMC_CTRL2_A2RDCFG_Pos)                               /*!< 0x00004000 */
#define SMC_CTRL2_A2RDCFG_2                                         (0x4UL << SMC_CTRL2_A2RDCFG_Pos)                               /*!< 0x00008000 */
#define SMC_CTRL2_A2RDCFG_3                                         (0x8UL << SMC_CTRL2_A2RDCFG_Pos)                               /*!< 0x00010000 */

#define SMC_CTRL2_ECCPSCFG_Pos                                      (17U)
#define SMC_CTRL2_ECCPSCFG_Msk                                      (0x7UL << SMC_CTRL2_ECCPSCFG_Pos)                              /*!< 0x000E0000 */
#define SMC_CTRL2_ECCPSCFG                                          SMC_CTRL2_ECCPSCFG_Msk                                         /*!<ECCPS[1:0] bits (ECC page size) */
#define SMC_CTRL2_ECCPSCFG_0                                        (0x1UL << SMC_CTRL2_ECCPSCFG_Pos)                              /*!< 0x00020000 */
#define SMC_CTRL2_ECCPSCFG_1                                        (0x2UL << SMC_CTRL2_ECCPSCFG_Pos)                              /*!< 0x00040000 */
#define SMC_CTRL2_ECCPSCFG_2                                        (0x4UL << SMC_CTRL2_ECCPSCFG_Pos)                              /*!< 0x00080000 */

/******************  Bit definition for SMC_CTRL3 register  *******************/
#define SMC_CTRL3_WAITFEN_Pos                                       (1U)
#define SMC_CTRL3_WAITFEN_Msk                                       (0x1UL << SMC_CTRL3_WAITFEN_Pos)                               /*!< 0x00000002 */
#define SMC_CTRL3_WAITFEN                                           SMC_CTRL3_WAITFEN_Msk                                          /*!<Wait feature enable bit */
#define SMC_CTRL3_MBKEN_Pos                                         (2U)
#define SMC_CTRL3_MBKEN_Msk                                         (0x1UL << SMC_CTRL3_MBKEN_Pos)                                 /*!< 0x00000004 */
#define SMC_CTRL3_MBKEN                                             SMC_CTRL3_MBKEN_Msk                                            /*!<PC Card/NAND Flash memory bank enable bit */
#define SMC_CTRL3_MTYPECFG_Pos                                      (3U)
#define SMC_CTRL3_MTYPECFG_Msk                                      (0x1UL << SMC_CTRL3_MTYPECFG_Pos)                              /*!< 0x00000008 */
#define SMC_CTRL3_MTYPECFG                                          SMC_CTRL3_MTYPECFG_Msk                                         /*!<Memory type */

#define SMC_CTRL3_DBWIDCFG_Pos                                      (4U)
#define SMC_CTRL3_DBWIDCFG_Msk                                      (0x3UL << SMC_CTRL3_DBWIDCFG_Pos)                              /*!< 0x00000030 */
#define SMC_CTRL3_DBWIDCFG                                          SMC_CTRL3_DBWIDCFG_Msk                                         /*!<PWID[1:0] bits (NAND Flash databus width) */
#define SMC_CTRL3_DBWIDCFG_0                                        (0x1UL << SMC_CTRL3_DBWIDCFG_Pos)                              /*!< 0x00000010 */
#define SMC_CTRL3_DBWIDCFG_1                                        (0x2UL << SMC_CTRL3_DBWIDCFG_Pos)                              /*!< 0x00000020 */

#define SMC_CTRL3_ECCEN_Pos                                         (6U)
#define SMC_CTRL3_ECCEN_Msk                                         (0x1UL << SMC_CTRL3_ECCEN_Pos)                                 /*!< 0x00000040 */
#define SMC_CTRL3_ECCEN                                             SMC_CTRL3_ECCEN_Msk                                            /*!<ECC computation logic enable bit */

#define SMC_CTRL3_C2RDCFG_Pos                                       (9U)
#define SMC_CTRL3_C2RDCFG_Msk                                       (0xFUL << SMC_CTRL3_C2RDCFG_Pos)                               /*!< 0x00001E00 */
#define SMC_CTRL3_C2RDCFG                                           SMC_CTRL3_C2RDCFG_Msk                                          /*!<TCLR[3:0] bits (CLE to RE delay) */
#define SMC_CTRL3_C2RDCFG_0                                         (0x1UL << SMC_CTRL3_C2RDCFG_Pos)                               /*!< 0x00000200 */
#define SMC_CTRL3_C2RDCFG_1                                         (0x2UL << SMC_CTRL3_C2RDCFG_Pos)                               /*!< 0x00000400 */
#define SMC_CTRL3_C2RDCFG_2                                         (0x4UL << SMC_CTRL3_C2RDCFG_Pos)                               /*!< 0x00000800 */
#define SMC_CTRL3_C2RDCFG_3                                         (0x8UL << SMC_CTRL3_C2RDCFG_Pos)                               /*!< 0x00001000 */

#define SMC_CTRL3_A2RDCFG_Pos                                       (13U)
#define SMC_CTRL3_A2RDCFG_Msk                                       (0xFUL << SMC_CTRL3_A2RDCFG_Pos)                               /*!< 0x0001E000 */
#define SMC_CTRL3_A2RDCFG                                           SMC_CTRL3_A2RDCFG_Msk                                          /*!<TAR[3:0] bits (ALE to RE delay) */
#define SMC_CTRL3_A2RDCFG_0                                         (0x1UL << SMC_CTRL3_A2RDCFG_Pos)                               /*!< 0x00002000 */
#define SMC_CTRL3_A2RDCFG_1                                         (0x2UL << SMC_CTRL3_A2RDCFG_Pos)                               /*!< 0x00004000 */
#define SMC_CTRL3_A2RDCFG_2                                         (0x4UL << SMC_CTRL3_A2RDCFG_Pos)                               /*!< 0x00008000 */
#define SMC_CTRL3_A2RDCFG_3                                         (0x8UL << SMC_CTRL3_A2RDCFG_Pos)                               /*!< 0x00010000 */

#define SMC_CTRL3_ECCPSCFG_Pos                                      (17U)
#define SMC_CTRL3_ECCPSCFG_Msk                                      (0x7UL << SMC_CTRL3_ECCPSCFG_Pos)                              /*!< 0x000E0000 */
#define SMC_CTRL3_ECCPSCFG                                          SMC_CTRL3_ECCPSCFG_Msk                                         /*!<ECCPS[2:0] bits (ECC page size) */
#define SMC_CTRL3_ECCPSCFG_0                                        (0x1UL << SMC_CTRL3_ECCPSCFG_Pos)                              /*!< 0x00020000 */
#define SMC_CTRL3_ECCPSCFG_1                                        (0x2UL << SMC_CTRL3_ECCPSCFG_Pos)                              /*!< 0x00040000 */
#define SMC_CTRL3_ECCPSCFG_2                                        (0x4UL << SMC_CTRL3_ECCPSCFG_Pos)                              /*!< 0x00080000 */

/******************  Bit definition for SMC_CTRL4 register  *******************/
#define SMC_CTRL4_WAITFEN_Pos                                       (1U)
#define SMC_CTRL4_WAITFEN_Msk                                       (0x1UL << SMC_CTRL4_WAITFEN_Pos)                               /*!< 0x00000002 */
#define SMC_CTRL4_WAITFEN                                           SMC_CTRL4_WAITFEN_Msk                                          /*!<Wait feature enable bit */
#define SMC_CTRL4_MBKEN_Pos                                         (2U)
#define SMC_CTRL4_MBKEN_Msk                                         (0x1UL << SMC_CTRL4_MBKEN_Pos)                                 /*!< 0x00000004 */
#define SMC_CTRL4_MBKEN                                             SMC_CTRL4_MBKEN_Msk                                            /*!<PC Card/NAND Flash memory bank enable bit */
#define SMC_CTRL4_MTYPECFG_Pos                                      (3U)
#define SMC_CTRL4_MTYPECFG_Msk                                      (0x1UL << SMC_CTRL4_MTYPECFG_Pos)                              /*!< 0x00000008 */
#define SMC_CTRL4_MTYPECFG                                          SMC_CTRL4_MTYPECFG_Msk                                         /*!<Memory type */

#define SMC_CTRL4_DBWIDCFG_Pos                                      (4U)
#define SMC_CTRL4_DBWIDCFG_Msk                                      (0x3UL << SMC_CTRL4_DBWIDCFG_Pos)                              /*!< 0x00000030 */
#define SMC_CTRL4_DBWIDCFG                                          SMC_CTRL4_DBWIDCFG_Msk                                         /*!<PWID[1:0] bits (NAND Flash databus width) */
#define SMC_CTRL4_DBWIDCFG_0                                        (0x1UL << SMC_CTRL4_DBWIDCFG_Pos)                              /*!< 0x00000010 */
#define SMC_CTRL4_DBWIDCFG_1                                        (0x2UL << SMC_CTRL4_DBWIDCFG_Pos)                              /*!< 0x00000020 */

#define SMC_CTRL4_ECCEN_Pos                                         (6U)
#define SMC_CTRL4_ECCEN_Msk                                         (0x1UL << SMC_CTRL4_ECCEN_Pos)                                 /*!< 0x00000040 */
#define SMC_CTRL4_ECCEN                                             SMC_CTRL4_ECCEN_Msk                                            /*!<ECC computation logic enable bit */

#define SMC_CTRL4_C2RDCFG_Pos                                       (9U)
#define SMC_CTRL4_C2RDCFG_Msk                                       (0xFUL << SMC_CTRL4_C2RDCFG_Pos)                               /*!< 0x00001E00 */
#define SMC_CTRL4_C2RDCFG                                           SMC_CTRL4_C2RDCFG_Msk                                          /*!<TCLR[3:0] bits (CLE to RE delay) */
#define SMC_CTRL4_C2RDCFG_0                                         (0x1UL << SMC_CTRL4_C2RDCFG_Pos)                               /*!< 0x00000200 */
#define SMC_CTRL4_C2RDCFG_1                                         (0x2UL << SMC_CTRL4_C2RDCFG_Pos)                               /*!< 0x00000400 */
#define SMC_CTRL4_C2RDCFG_2                                         (0x4UL << SMC_CTRL4_C2RDCFG_Pos)                               /*!< 0x00000800 */
#define SMC_CTRL4_C2RDCFG_3                                         (0x8UL << SMC_CTRL4_C2RDCFG_Pos)                               /*!< 0x00001000 */

#define SMC_CTRL4_A2RDCFG_Pos                                       (13U)
#define SMC_CTRL4_A2RDCFG_Msk                                       (0xFUL << SMC_CTRL4_A2RDCFG_Pos)                               /*!< 0x0001E000 */
#define SMC_CTRL4_A2RDCFG                                           SMC_CTRL4_A2RDCFG_Msk                                          /*!<TAR[3:0] bits (ALE to RE delay) */
#define SMC_CTRL4_A2RDCFG_0                                         (0x1UL << SMC_CTRL4_A2RDCFG_Pos)                               /*!< 0x00002000 */
#define SMC_CTRL4_A2RDCFG_1                                         (0x2UL << SMC_CTRL4_A2RDCFG_Pos)                               /*!< 0x00004000 */
#define SMC_CTRL4_A2RDCFG_2                                         (0x4UL << SMC_CTRL4_A2RDCFG_Pos)                               /*!< 0x00008000 */
#define SMC_CTRL4_A2RDCFG_3                                         (0x8UL << SMC_CTRL4_A2RDCFG_Pos)                               /*!< 0x00010000 */

#define SMC_CTRL4_ECCPSCFG_Pos                                      (17U)
#define SMC_CTRL4_ECCPSCFG_Msk                                      (0x7UL << SMC_CTRL4_ECCPSCFG_Pos)                              /*!< 0x000E0000 */
#define SMC_CTRL4_ECCPSCFG                                          SMC_CTRL4_ECCPSCFG_Msk                                         /*!<ECCPS[2:0] bits (ECC page size) */
#define SMC_CTRL4_ECCPSCFG_0                                        (0x1UL << SMC_CTRL4_ECCPSCFG_Pos)                              /*!< 0x00020000 */
#define SMC_CTRL4_ECCPSCFG_1                                        (0x2UL << SMC_CTRL4_ECCPSCFG_Pos)                              /*!< 0x00040000 */
#define SMC_CTRL4_ECCPSCFG_2                                        (0x4UL << SMC_CTRL4_ECCPSCFG_Pos)                              /*!< 0x00080000 */

/*******************  Bit definition for SMC_STSINT2 register  *******************/
#define SMC_STSINT2_IREFLG_Pos                                      (0U)
#define SMC_STSINT2_IREFLG_Msk                                      (0x1UL << SMC_STSINT2_IREFLG_Pos)                              /*!< 0x00000001 */
#define SMC_STSINT2_IREFLG                                          SMC_STSINT2_IREFLG_Msk                                         /*!<Interrupt Rising Edge status                */
#define SMC_STSINT2_IHLFLG_Pos                                      (1U)
#define SMC_STSINT2_IHLFLG_Msk                                      (0x1UL << SMC_STSINT2_IHLFLG_Pos)                              /*!< 0x00000002 */
#define SMC_STSINT2_IHLFLG                                          SMC_STSINT2_IHLFLG_Msk                                         /*!<Interrupt Level status                      */
#define SMC_STSINT2_IFEFLG_Pos                                      (2U)
#define SMC_STSINT2_IFEFLG_Msk                                      (0x1UL << SMC_STSINT2_IFEFLG_Pos)                              /*!< 0x00000004 */
#define SMC_STSINT2_IFEFLG                                          SMC_STSINT2_IFEFLG_Msk                                         /*!<Interrupt Falling Edge status               */
#define SMC_STSINT2_IREDEN_Pos                                      (3U)
#define SMC_STSINT2_IREDEN_Msk                                      (0x1UL << SMC_STSINT2_IREDEN_Pos)                              /*!< 0x00000008 */
#define SMC_STSINT2_IREDEN                                          SMC_STSINT2_IREDEN_Msk                                         /*!<Interrupt Rising Edge detection Enable bit  */
#define SMC_STSINT2_IHLDEN_Pos                                      (4U)
#define SMC_STSINT2_IHLDEN_Msk                                      (0x1UL << SMC_STSINT2_IHLDEN_Pos)                              /*!< 0x00000010 */
#define SMC_STSINT2_IHLDEN                                          SMC_STSINT2_IHLDEN_Msk                                         /*!<Interrupt Level detection Enable bit        */
#define SMC_STSINT2_IFEDEN_Pos                                      (5U)
#define SMC_STSINT2_IFEDEN_Msk                                      (0x1UL << SMC_STSINT2_IFEDEN_Pos)                              /*!< 0x00000020 */
#define SMC_STSINT2_IFEDEN                                          SMC_STSINT2_IFEDEN_Msk                                         /*!<Interrupt Falling Edge detection Enable bit */
#define SMC_STSINT2_FEFLG_Pos                                       (6U)
#define SMC_STSINT2_FEFLG_Msk                                       (0x1UL << SMC_STSINT2_FEFLG_Pos)                               /*!< 0x00000040 */
#define SMC_STSINT2_FEFLG                                           SMC_STSINT2_FEFLG_Msk                                          /*!<FIFO empty */

/*******************  Bit definition for SMC_STSINT3 register  *******************/
#define SMC_STSINT3_IREFLG_Pos                                      (0U)
#define SMC_STSINT3_IREFLG_Msk                                      (0x1UL << SMC_STSINT3_IREFLG_Pos)                              /*!< 0x00000001 */
#define SMC_STSINT3_IREFLG                                          SMC_STSINT3_IREFLG_Msk                                         /*!<Interrupt Rising Edge status                */
#define SMC_STSINT3_IHLFLG_Pos                                      (1U)
#define SMC_STSINT3_IHLFLG_Msk                                      (0x1UL << SMC_STSINT3_IHLFLG_Pos)                              /*!< 0x00000002 */
#define SMC_STSINT3_IHLFLG                                          SMC_STSINT3_IHLFLG_Msk                                         /*!<Interrupt Level status                      */
#define SMC_STSINT3_IFEFLG_Pos                                      (2U)
#define SMC_STSINT3_IFEFLG_Msk                                      (0x1UL << SMC_STSINT3_IFEFLG_Pos)                              /*!< 0x00000004 */
#define SMC_STSINT3_IFEFLG                                          SMC_STSINT3_IFEFLG_Msk                                         /*!<Interrupt Falling Edge status               */
#define SMC_STSINT3_IREDEN_Pos                                      (3U)
#define SMC_STSINT3_IREDEN_Msk                                      (0x1UL << SMC_STSINT3_IREDEN_Pos)                              /*!< 0x00000008 */
#define SMC_STSINT3_IREDEN                                          SMC_STSINT3_IREDEN_Msk                                         /*!<Interrupt Rising Edge detection Enable bit  */
#define SMC_STSINT3_IHLDEN_Pos                                      (4U)
#define SMC_STSINT3_IHLDEN_Msk                                      (0x1UL << SMC_STSINT3_IHLDEN_Pos)                              /*!< 0x00000010 */
#define SMC_STSINT3_IHLDEN                                          SMC_STSINT3_IHLDEN_Msk                                         /*!<Interrupt Level detection Enable bit        */
#define SMC_STSINT3_IFEDEN_Pos                                      (5U)
#define SMC_STSINT3_IFEDEN_Msk                                      (0x1UL << SMC_STSINT3_IFEDEN_Pos)                              /*!< 0x00000020 */
#define SMC_STSINT3_IFEDEN                                          SMC_STSINT3_IFEDEN_Msk                                         /*!<Interrupt Falling Edge detection Enable bit */
#define SMC_STSINT3_FEFLG_Pos                                       (6U)
#define SMC_STSINT3_FEFLG_Msk                                       (0x1UL << SMC_STSINT3_FEFLG_Pos)                               /*!< 0x00000040 */
#define SMC_STSINT3_FEFLG                                           SMC_STSINT3_FEFLG_Msk                                          /*!<FIFO empty */

/*******************  Bit definition for SMC_STSINT4 register  *******************/
#define SMC_STSINT4_IREFLG_Pos                                      (0U)
#define SMC_STSINT4_IREFLG_Msk                                      (0x1UL << SMC_STSINT4_IREFLG_Pos)                              /*!< 0x00000001 */
#define SMC_STSINT4_IREFLG                                          SMC_STSINT4_IREFLG_Msk                                         /*!<Interrupt Rising Edge status                 */
#define SMC_STSINT4_IHLFLG_Pos                                      (1U)
#define SMC_STSINT4_IHLFLG_Msk                                      (0x1UL << SMC_STSINT4_IHLFLG_Pos)                              /*!< 0x00000002 */
#define SMC_STSINT4_IHLFLG                                          SMC_STSINT4_IHLFLG_Msk                                         /*!<Interrupt Level status                       */
#define SMC_STSINT4_IFEFLG_Pos                                      (2U)
#define SMC_STSINT4_IFEFLG_Msk                                      (0x1UL << SMC_STSINT4_IFEFLG_Pos)                              /*!< 0x00000004 */
#define SMC_STSINT4_IFEFLG                                          SMC_STSINT4_IFEFLG_Msk                                         /*!<Interrupt Falling Edge status                */
#define SMC_STSINT4_IREDEN_Pos                                      (3U)
#define SMC_STSINT4_IREDEN_Msk                                      (0x1UL << SMC_STSINT4_IREDEN_Pos)                              /*!< 0x00000008 */
#define SMC_STSINT4_IREDEN                                          SMC_STSINT4_IREDEN_Msk                                         /*!<Interrupt Rising Edge detection Enable bit   */
#define SMC_STSINT4_IHLDEN_Pos                                      (4U)
#define SMC_STSINT4_IHLDEN_Msk                                      (0x1UL << SMC_STSINT4_IHLDEN_Pos)                              /*!< 0x00000010 */
#define SMC_STSINT4_IHLDEN                                          SMC_STSINT4_IHLDEN_Msk                                         /*!<Interrupt Level detection Enable bit         */
#define SMC_STSINT4_IFEDEN_Pos                                      (5U)
#define SMC_STSINT4_IFEDEN_Msk                                      (0x1UL << SMC_STSINT4_IFEDEN_Pos)                              /*!< 0x00000020 */
#define SMC_STSINT4_IFEDEN                                          SMC_STSINT4_IFEDEN_Msk                                         /*!<Interrupt Falling Edge detection Enable bit  */
#define SMC_STSINT4_FEFLG_Pos                                       (6U)
#define SMC_STSINT4_FEFLG_Msk                                       (0x1UL << SMC_STSINT4_FEFLG_Pos)                               /*!< 0x00000040 */
#define SMC_STSINT4_FEFLG                                           SMC_STSINT4_FEFLG_Msk                                          /*!<FIFO empty */

/******************  Bit definition for SMC_CMSTIM2 register  ******************/
#define SMC_CMSTIM2_SET2_Pos                                        (0U)
#define SMC_CMSTIM2_SET2_Msk                                        (0xFFUL << SMC_CMSTIM2_SET2_Pos)                               /*!< 0x000000FF */
#define SMC_CMSTIM2_SET2                                            SMC_CMSTIM2_SET2_Msk                                           /*!<MEMSET2[7:0] bits (Common memory 2 setup time) */
#define SMC_CMSTIM2_SET2_0                                          (0x01UL << SMC_CMSTIM2_SET2_Pos)                               /*!< 0x00000001 */
#define SMC_CMSTIM2_SET2_1                                          (0x02UL << SMC_CMSTIM2_SET2_Pos)                               /*!< 0x00000002 */
#define SMC_CMSTIM2_SET2_2                                          (0x04UL << SMC_CMSTIM2_SET2_Pos)                               /*!< 0x00000004 */
#define SMC_CMSTIM2_SET2_3                                          (0x08UL << SMC_CMSTIM2_SET2_Pos)                               /*!< 0x00000008 */
#define SMC_CMSTIM2_SET2_4                                          (0x10UL << SMC_CMSTIM2_SET2_Pos)                               /*!< 0x00000010 */
#define SMC_CMSTIM2_SET2_5                                          (0x20UL << SMC_CMSTIM2_SET2_Pos)                               /*!< 0x00000020 */
#define SMC_CMSTIM2_SET2_6                                          (0x40UL << SMC_CMSTIM2_SET2_Pos)                               /*!< 0x00000040 */
#define SMC_CMSTIM2_SET2_7                                          (0x80UL << SMC_CMSTIM2_SET2_Pos)                               /*!< 0x00000080 */

#define SMC_CMSTIM2_WAIT2_Pos                                       (8U)
#define SMC_CMSTIM2_WAIT2_Msk                                       (0xFFUL << SMC_CMSTIM2_WAIT2_Pos)                              /*!< 0x0000FF00 */
#define SMC_CMSTIM2_WAIT2                                           SMC_CMSTIM2_WAIT2_Msk                                          /*!<MEMWAIT2[7:0] bits (Common memory 2 wait time) */
#define SMC_CMSTIM2_WAIT2_0                                         (0x01UL << SMC_CMSTIM2_WAIT2_Pos)                              /*!< 0x00000100 */
#define SMC_CMSTIM2_WAIT2_1                                         (0x02UL << SMC_CMSTIM2_WAIT2_Pos)                              /*!< 0x00000200 */
#define SMC_CMSTIM2_WAIT2_2                                         (0x04UL << SMC_CMSTIM2_WAIT2_Pos)                              /*!< 0x00000400 */
#define SMC_CMSTIM2_WAIT2_3                                         (0x08UL << SMC_CMSTIM2_WAIT2_Pos)                              /*!< 0x00000800 */
#define SMC_CMSTIM2_WAIT2_4                                         (0x10UL << SMC_CMSTIM2_WAIT2_Pos)                              /*!< 0x00001000 */
#define SMC_CMSTIM2_WAIT2_5                                         (0x20UL << SMC_CMSTIM2_WAIT2_Pos)                              /*!< 0x00002000 */
#define SMC_CMSTIM2_WAIT2_6                                         (0x40UL << SMC_CMSTIM2_WAIT2_Pos)                              /*!< 0x00004000 */
#define SMC_CMSTIM2_WAIT2_7                                         (0x80UL << SMC_CMSTIM2_WAIT2_Pos)                              /*!< 0x00008000 */

#define SMC_CMSTIM2_HLD2_Pos                                        (16U)
#define SMC_CMSTIM2_HLD2_Msk                                        (0xFFUL << SMC_CMSTIM2_HLD2_Pos)                               /*!< 0x00FF0000 */
#define SMC_CMSTIM2_HLD2                                            SMC_CMSTIM2_HLD2_Msk                                           /*!<MEMHOLD2[7:0] bits (Common memory 2 hold time) */
#define SMC_CMSTIM2_HLD2_0                                          (0x01UL << SMC_CMSTIM2_HLD2_Pos)                               /*!< 0x00010000 */
#define SMC_CMSTIM2_HLD2_1                                          (0x02UL << SMC_CMSTIM2_HLD2_Pos)                               /*!< 0x00020000 */
#define SMC_CMSTIM2_HLD2_2                                          (0x04UL << SMC_CMSTIM2_HLD2_Pos)                               /*!< 0x00040000 */
#define SMC_CMSTIM2_HLD2_3                                          (0x08UL << SMC_CMSTIM2_HLD2_Pos)                               /*!< 0x00080000 */
#define SMC_CMSTIM2_HLD2_4                                          (0x10UL << SMC_CMSTIM2_HLD2_Pos)                               /*!< 0x00100000 */
#define SMC_CMSTIM2_HLD2_5                                          (0x20UL << SMC_CMSTIM2_HLD2_Pos)                               /*!< 0x00200000 */
#define SMC_CMSTIM2_HLD2_6                                          (0x40UL << SMC_CMSTIM2_HLD2_Pos)                               /*!< 0x00400000 */
#define SMC_CMSTIM2_HLD2_7                                          (0x80UL << SMC_CMSTIM2_HLD2_Pos)                               /*!< 0x00800000 */

#define SMC_CMSTIM2_HIZ2_Pos                                        (24U)
#define SMC_CMSTIM2_HIZ2_Msk                                        (0xFFUL << SMC_CMSTIM2_HIZ2_Pos)                               /*!< 0xFF000000 */
#define SMC_CMSTIM2_HIZ2                                            SMC_CMSTIM2_HIZ2_Msk                                           /*!<MEMHIZ2[7:0] bits (Common memory 2 databus HiZ time) */
#define SMC_CMSTIM2_HIZ2_0                                          (0x01UL << SMC_CMSTIM2_HIZ2_Pos)                               /*!< 0x01000000 */
#define SMC_CMSTIM2_HIZ2_1                                          (0x02UL << SMC_CMSTIM2_HIZ2_Pos)                               /*!< 0x02000000 */
#define SMC_CMSTIM2_HIZ2_2                                          (0x04UL << SMC_CMSTIM2_HIZ2_Pos)                               /*!< 0x04000000 */
#define SMC_CMSTIM2_HIZ2_3                                          (0x08UL << SMC_CMSTIM2_HIZ2_Pos)                               /*!< 0x08000000 */
#define SMC_CMSTIM2_HIZ2_4                                          (0x10UL << SMC_CMSTIM2_HIZ2_Pos)                               /*!< 0x10000000 */
#define SMC_CMSTIM2_HIZ2_5                                          (0x20UL << SMC_CMSTIM2_HIZ2_Pos)                               /*!< 0x20000000 */
#define SMC_CMSTIM2_HIZ2_6                                          (0x40UL << SMC_CMSTIM2_HIZ2_Pos)                               /*!< 0x40000000 */
#define SMC_CMSTIM2_HIZ2_7                                          (0x80UL << SMC_CMSTIM2_HIZ2_Pos)                               /*!< 0x80000000 */

/******************  Bit definition for SMC_CMSTIM3 register  ******************/
#define SMC_CMSTIM3_SET3_Pos                                        (0U)
#define SMC_CMSTIM3_SET3_Msk                                        (0xFFUL << SMC_CMSTIM3_SET3_Pos)                               /*!< 0x000000FF */
#define SMC_CMSTIM3_SET3                                            SMC_CMSTIM3_SET3_Msk                                           /*!<MEMSET3[7:0] bits (Common memory 3 setup time) */
#define SMC_CMSTIM3_SET3_0                                          (0x01UL << SMC_CMSTIM3_SET3_Pos)                               /*!< 0x00000001 */
#define SMC_CMSTIM3_SET3_1                                          (0x02UL << SMC_CMSTIM3_SET3_Pos)                               /*!< 0x00000002 */
#define SMC_CMSTIM3_SET3_2                                          (0x04UL << SMC_CMSTIM3_SET3_Pos)                               /*!< 0x00000004 */
#define SMC_CMSTIM3_SET3_3                                          (0x08UL << SMC_CMSTIM3_SET3_Pos)                               /*!< 0x00000008 */
#define SMC_CMSTIM3_SET3_4                                          (0x10UL << SMC_CMSTIM3_SET3_Pos)                               /*!< 0x00000010 */
#define SMC_CMSTIM3_SET3_5                                          (0x20UL << SMC_CMSTIM3_SET3_Pos)                               /*!< 0x00000020 */
#define SMC_CMSTIM3_SET3_6                                          (0x40UL << SMC_CMSTIM3_SET3_Pos)                               /*!< 0x00000040 */
#define SMC_CMSTIM3_SET3_7                                          (0x80UL << SMC_CMSTIM3_SET3_Pos)                               /*!< 0x00000080 */

#define SMC_CMSTIM3_WAIT3_Pos                                       (8U)
#define SMC_CMSTIM3_WAIT3_Msk                                       (0xFFUL << SMC_CMSTIM3_WAIT3_Pos)                              /*!< 0x0000FF00 */
#define SMC_CMSTIM3_WAIT3                                           SMC_CMSTIM3_WAIT3_Msk                                          /*!<MEMWAIT3[7:0] bits (Common memory 3 wait time) */
#define SMC_CMSTIM3_WAIT3_0                                         (0x01UL << SMC_CMSTIM3_WAIT3_Pos)                              /*!< 0x00000100 */
#define SMC_CMSTIM3_WAIT3_1                                         (0x02UL << SMC_CMSTIM3_WAIT3_Pos)                              /*!< 0x00000200 */
#define SMC_CMSTIM3_WAIT3_2                                         (0x04UL << SMC_CMSTIM3_WAIT3_Pos)                              /*!< 0x00000400 */
#define SMC_CMSTIM3_WAIT3_3                                         (0x08UL << SMC_CMSTIM3_WAIT3_Pos)                              /*!< 0x00000800 */
#define SMC_CMSTIM3_WAIT3_4                                         (0x10UL << SMC_CMSTIM3_WAIT3_Pos)                              /*!< 0x00001000 */
#define SMC_CMSTIM3_WAIT3_5                                         (0x20UL << SMC_CMSTIM3_WAIT3_Pos)                              /*!< 0x00002000 */
#define SMC_CMSTIM3_WAIT3_6                                         (0x40UL << SMC_CMSTIM3_WAIT3_Pos)                              /*!< 0x00004000 */
#define SMC_CMSTIM3_WAIT3_7                                         (0x80UL << SMC_CMSTIM3_WAIT3_Pos)                              /*!< 0x00008000 */

#define SMC_CMSTIM3_HLD3_Pos                                        (16U)
#define SMC_CMSTIM3_HLD3_Msk                                        (0xFFUL << SMC_CMSTIM3_HLD3_Pos)                               /*!< 0x00FF0000 */
#define SMC_CMSTIM3_HLD3                                            SMC_CMSTIM3_HLD3_Msk                                           /*!<MEMHOLD3[7:0] bits (Common memory 3 hold time) */
#define SMC_CMSTIM3_HLD3_0                                          (0x01UL << SMC_CMSTIM3_HLD3_Pos)                               /*!< 0x00010000 */
#define SMC_CMSTIM3_HLD3_1                                          (0x02UL << SMC_CMSTIM3_HLD3_Pos)                               /*!< 0x00020000 */
#define SMC_CMSTIM3_HLD3_2                                          (0x04UL << SMC_CMSTIM3_HLD3_Pos)                               /*!< 0x00040000 */
#define SMC_CMSTIM3_HLD3_3                                          (0x08UL << SMC_CMSTIM3_HLD3_Pos)                               /*!< 0x00080000 */
#define SMC_CMSTIM3_HLD3_4                                          (0x10UL << SMC_CMSTIM3_HLD3_Pos)                               /*!< 0x00100000 */
#define SMC_CMSTIM3_HLD3_5                                          (0x20UL << SMC_CMSTIM3_HLD3_Pos)                               /*!< 0x00200000 */
#define SMC_CMSTIM3_HLD3_6                                          (0x40UL << SMC_CMSTIM3_HLD3_Pos)                               /*!< 0x00400000 */
#define SMC_CMSTIM3_HLD3_7                                          (0x80UL << SMC_CMSTIM3_HLD3_Pos)                               /*!< 0x00800000 */

#define SMC_CMSTIM3_HIZ3_Pos                                        (24U)
#define SMC_CMSTIM3_HIZ3_Msk                                        (0xFFUL << SMC_CMSTIM3_HIZ3_Pos)                               /*!< 0xFF000000 */
#define SMC_CMSTIM3_HIZ3                                            SMC_CMSTIM3_HIZ3_Msk                                           /*!<MEMHIZ3[7:0] bits (Common memory 3 databus HiZ time) */
#define SMC_CMSTIM3_HIZ3_0                                          (0x01UL << SMC_CMSTIM3_HIZ3_Pos)                               /*!< 0x01000000 */
#define SMC_CMSTIM3_HIZ3_1                                          (0x02UL << SMC_CMSTIM3_HIZ3_Pos)                               /*!< 0x02000000 */
#define SMC_CMSTIM3_HIZ3_2                                          (0x04UL << SMC_CMSTIM3_HIZ3_Pos)                               /*!< 0x04000000 */
#define SMC_CMSTIM3_HIZ3_3                                          (0x08UL << SMC_CMSTIM3_HIZ3_Pos)                               /*!< 0x08000000 */
#define SMC_CMSTIM3_HIZ3_4                                          (0x10UL << SMC_CMSTIM3_HIZ3_Pos)                               /*!< 0x10000000 */
#define SMC_CMSTIM3_HIZ3_5                                          (0x20UL << SMC_CMSTIM3_HIZ3_Pos)                               /*!< 0x20000000 */
#define SMC_CMSTIM3_HIZ3_6                                          (0x40UL << SMC_CMSTIM3_HIZ3_Pos)                               /*!< 0x40000000 */
#define SMC_CMSTIM3_HIZ3_7                                          (0x80UL << SMC_CMSTIM3_HIZ3_Pos)                               /*!< 0x80000000 */

/******************  Bit definition for SMC_CMSTIM4 register  ******************/
#define SMC_CMSTIM4_SET4_Pos                                        (0U)
#define SMC_CMSTIM4_SET4_Msk                                        (0xFFUL << SMC_CMSTIM4_SET4_Pos)                               /*!< 0x000000FF */
#define SMC_CMSTIM4_SET4                                            SMC_CMSTIM4_SET4_Msk                                           /*!<MEMSET4[7:0] bits (Common memory 4 setup time) */
#define SMC_CMSTIM4_SET4_0                                          (0x01UL << SMC_CMSTIM4_SET4_Pos)                               /*!< 0x00000001 */
#define SMC_CMSTIM4_SET4_1                                          (0x02UL << SMC_CMSTIM4_SET4_Pos)                               /*!< 0x00000002 */
#define SMC_CMSTIM4_SET4_2                                          (0x04UL << SMC_CMSTIM4_SET4_Pos)                               /*!< 0x00000004 */
#define SMC_CMSTIM4_SET4_3                                          (0x08UL << SMC_CMSTIM4_SET4_Pos)                               /*!< 0x00000008 */
#define SMC_CMSTIM4_SET4_4                                          (0x10UL << SMC_CMSTIM4_SET4_Pos)                               /*!< 0x00000010 */
#define SMC_CMSTIM4_SET4_5                                          (0x20UL << SMC_CMSTIM4_SET4_Pos)                               /*!< 0x00000020 */
#define SMC_CMSTIM4_SET4_6                                          (0x40UL << SMC_CMSTIM4_SET4_Pos)                               /*!< 0x00000040 */
#define SMC_CMSTIM4_SET4_7                                          (0x80UL << SMC_CMSTIM4_SET4_Pos)                               /*!< 0x00000080 */

#define SMC_CMSTIM4_WAIT4_Pos                                       (8U)
#define SMC_CMSTIM4_WAIT4_Msk                                       (0xFFUL << SMC_CMSTIM4_WAIT4_Pos)                              /*!< 0x0000FF00 */
#define SMC_CMSTIM4_WAIT4                                           SMC_CMSTIM4_WAIT4_Msk                                          /*!<MEMWAIT4[7:0] bits (Common memory 4 wait time) */
#define SMC_CMSTIM4_WAIT4_0                                         (0x01UL << SMC_CMSTIM4_WAIT4_Pos)                              /*!< 0x00000100 */
#define SMC_CMSTIM4_WAIT4_1                                         (0x02UL << SMC_CMSTIM4_WAIT4_Pos)                              /*!< 0x00000200 */
#define SMC_CMSTIM4_WAIT4_2                                         (0x04UL << SMC_CMSTIM4_WAIT4_Pos)                              /*!< 0x00000400 */
#define SMC_CMSTIM4_WAIT4_3                                         (0x08UL << SMC_CMSTIM4_WAIT4_Pos)                              /*!< 0x00000800 */
#define SMC_CMSTIM4_WAIT4_4                                         (0x10UL << SMC_CMSTIM4_WAIT4_Pos)                              /*!< 0x00001000 */
#define SMC_CMSTIM4_WAIT4_5                                         (0x20UL << SMC_CMSTIM4_WAIT4_Pos)                              /*!< 0x00002000 */
#define SMC_CMSTIM4_WAIT4_6                                         (0x40UL << SMC_CMSTIM4_WAIT4_Pos)                              /*!< 0x00004000 */
#define SMC_CMSTIM4_WAIT4_7                                         (0x80UL << SMC_CMSTIM4_WAIT4_Pos)                              /*!< 0x00008000 */

#define SMC_CMSTIM4_HLD4_Pos                                        (16U)
#define SMC_CMSTIM4_HLD4_Msk                                        (0xFFUL << SMC_CMSTIM4_HLD4_Pos)                               /*!< 0x00FF0000 */
#define SMC_CMSTIM4_HLD4                                            SMC_CMSTIM4_HLD4_Msk                                           /*!<MEMHOLD4[7:0] bits (Common memory 4 hold time) */
#define SMC_CMSTIM4_HLD4_0                                          (0x01UL << SMC_CMSTIM4_HLD4_Pos)                               /*!< 0x00010000 */
#define SMC_CMSTIM4_HLD4_1                                          (0x02UL << SMC_CMSTIM4_HLD4_Pos)                               /*!< 0x00020000 */
#define SMC_CMSTIM4_HLD4_2                                          (0x04UL << SMC_CMSTIM4_HLD4_Pos)                               /*!< 0x00040000 */
#define SMC_CMSTIM4_HLD4_3                                          (0x08UL << SMC_CMSTIM4_HLD4_Pos)                               /*!< 0x00080000 */
#define SMC_CMSTIM4_HLD4_4                                          (0x10UL << SMC_CMSTIM4_HLD4_Pos)                               /*!< 0x00100000 */
#define SMC_CMSTIM4_HLD4_5                                          (0x20UL << SMC_CMSTIM4_HLD4_Pos)                               /*!< 0x00200000 */
#define SMC_CMSTIM4_HLD4_6                                          (0x40UL << SMC_CMSTIM4_HLD4_Pos)                               /*!< 0x00400000 */
#define SMC_CMSTIM4_HLD4_7                                          (0x80UL << SMC_CMSTIM4_HLD4_Pos)                               /*!< 0x00800000 */

#define SMC_CMSTIM4_HIZ4_Pos                                        (24U)
#define SMC_CMSTIM4_HIZ4_Msk                                        (0xFFUL << SMC_CMSTIM4_HIZ4_Pos)                               /*!< 0xFF000000 */
#define SMC_CMSTIM4_HIZ4                                            SMC_CMSTIM4_HIZ4_Msk                                           /*!<MEMHIZ4[7:0] bits (Common memory 4 databus HiZ time) */
#define SMC_CMSTIM4_HIZ4_0                                          (0x01UL << SMC_CMSTIM4_HIZ4_Pos)                               /*!< 0x01000000 */
#define SMC_CMSTIM4_HIZ4_1                                          (0x02UL << SMC_CMSTIM4_HIZ4_Pos)                               /*!< 0x02000000 */
#define SMC_CMSTIM4_HIZ4_2                                          (0x04UL << SMC_CMSTIM4_HIZ4_Pos)                               /*!< 0x04000000 */
#define SMC_CMSTIM4_HIZ4_3                                          (0x08UL << SMC_CMSTIM4_HIZ4_Pos)                               /*!< 0x08000000 */
#define SMC_CMSTIM4_HIZ4_4                                          (0x10UL << SMC_CMSTIM4_HIZ4_Pos)                               /*!< 0x10000000 */
#define SMC_CMSTIM4_HIZ4_5                                          (0x20UL << SMC_CMSTIM4_HIZ4_Pos)                               /*!< 0x20000000 */
#define SMC_CMSTIM4_HIZ4_6                                          (0x40UL << SMC_CMSTIM4_HIZ4_Pos)                               /*!< 0x40000000 */
#define SMC_CMSTIM4_HIZ4_7                                          (0x80UL << SMC_CMSTIM4_HIZ4_Pos)                               /*!< 0x80000000 */

/******************  Bit definition for SMC_AMSTIM2 register  ******************/
#define SMC_AMSTIM2_SET2_Pos                                        (0U)
#define SMC_AMSTIM2_SET2_Msk                                        (0xFFUL << SMC_AMSTIM2_SET2_Pos)                               /*!< 0x000000FF */
#define SMC_AMSTIM2_SET2                                            SMC_AMSTIM2_SET2_Msk                                           /*!<ATTSET2[7:0] bits (Attribute memory 2 setup time) */
#define SMC_AMSTIM2_SET2_0                                          (0x01UL << SMC_AMSTIM2_SET2_Pos)                               /*!< 0x00000001 */
#define SMC_AMSTIM2_SET2_1                                          (0x02UL << SMC_AMSTIM2_SET2_Pos)                               /*!< 0x00000002 */
#define SMC_AMSTIM2_SET2_2                                          (0x04UL << SMC_AMSTIM2_SET2_Pos)                               /*!< 0x00000004 */
#define SMC_AMSTIM2_SET2_3                                          (0x08UL << SMC_AMSTIM2_SET2_Pos)                               /*!< 0x00000008 */
#define SMC_AMSTIM2_SET2_4                                          (0x10UL << SMC_AMSTIM2_SET2_Pos)                               /*!< 0x00000010 */
#define SMC_AMSTIM2_SET2_5                                          (0x20UL << SMC_AMSTIM2_SET2_Pos)                               /*!< 0x00000020 */
#define SMC_AMSTIM2_SET2_6                                          (0x40UL << SMC_AMSTIM2_SET2_Pos)                               /*!< 0x00000040 */
#define SMC_AMSTIM2_SET2_7                                          (0x80UL << SMC_AMSTIM2_SET2_Pos)                               /*!< 0x00000080 */

#define SMC_AMSTIM2_WAIT2_Pos                                       (8U)
#define SMC_AMSTIM2_WAIT2_Msk                                       (0xFFUL << SMC_AMSTIM2_WAIT2_Pos)                              /*!< 0x0000FF00 */
#define SMC_AMSTIM2_WAIT2                                           SMC_AMSTIM2_WAIT2_Msk                                          /*!<ATTWAIT2[7:0] bits (Attribute memory 2 wait time) */
#define SMC_AMSTIM2_WAIT2_0                                         (0x01UL << SMC_AMSTIM2_WAIT2_Pos)                              /*!< 0x00000100 */
#define SMC_AMSTIM2_WAIT2_1                                         (0x02UL << SMC_AMSTIM2_WAIT2_Pos)                              /*!< 0x00000200 */
#define SMC_AMSTIM2_WAIT2_2                                         (0x04UL << SMC_AMSTIM2_WAIT2_Pos)                              /*!< 0x00000400 */
#define SMC_AMSTIM2_WAIT2_3                                         (0x08UL << SMC_AMSTIM2_WAIT2_Pos)                              /*!< 0x00000800 */
#define SMC_AMSTIM2_WAIT2_4                                         (0x10UL << SMC_AMSTIM2_WAIT2_Pos)                              /*!< 0x00001000 */
#define SMC_AMSTIM2_WAIT2_5                                         (0x20UL << SMC_AMSTIM2_WAIT2_Pos)                              /*!< 0x00002000 */
#define SMC_AMSTIM2_WAIT2_6                                         (0x40UL << SMC_AMSTIM2_WAIT2_Pos)                              /*!< 0x00004000 */
#define SMC_AMSTIM2_WAIT2_7                                         (0x80UL << SMC_AMSTIM2_WAIT2_Pos)                              /*!< 0x00008000 */

#define SMC_AMSTIM2_HLD2_Pos                                        (16U)
#define SMC_AMSTIM2_HLD2_Msk                                        (0xFFUL << SMC_AMSTIM2_HLD2_Pos)                               /*!< 0x00FF0000 */
#define SMC_AMSTIM2_HLD2                                            SMC_AMSTIM2_HLD2_Msk                                           /*!<ATTHOLD2[7:0] bits (Attribute memory 2 hold time) */
#define SMC_AMSTIM2_HLD2_0                                          (0x01UL << SMC_AMSTIM2_HLD2_Pos)                               /*!< 0x00010000 */
#define SMC_AMSTIM2_HLD2_1                                          (0x02UL << SMC_AMSTIM2_HLD2_Pos)                               /*!< 0x00020000 */
#define SMC_AMSTIM2_HLD2_2                                          (0x04UL << SMC_AMSTIM2_HLD2_Pos)                               /*!< 0x00040000 */
#define SMC_AMSTIM2_HLD2_3                                          (0x08UL << SMC_AMSTIM2_HLD2_Pos)                               /*!< 0x00080000 */
#define SMC_AMSTIM2_HLD2_4                                          (0x10UL << SMC_AMSTIM2_HLD2_Pos)                               /*!< 0x00100000 */
#define SMC_AMSTIM2_HLD2_5                                          (0x20UL << SMC_AMSTIM2_HLD2_Pos)                               /*!< 0x00200000 */
#define SMC_AMSTIM2_HLD2_6                                          (0x40UL << SMC_AMSTIM2_HLD2_Pos)                               /*!< 0x00400000 */
#define SMC_AMSTIM2_HLD2_7                                          (0x80UL << SMC_AMSTIM2_HLD2_Pos)                               /*!< 0x00800000 */

#define SMC_AMSTIM2_HIZ2_Pos                                        (24U)
#define SMC_AMSTIM2_HIZ2_Msk                                        (0xFFUL << SMC_AMSTIM2_HIZ2_Pos)                               /*!< 0xFF000000 */
#define SMC_AMSTIM2_HIZ2                                            SMC_AMSTIM2_HIZ2_Msk                                           /*!<ATTHIZ2[7:0] bits (Attribute memory 2 databus HiZ time) */
#define SMC_AMSTIM2_HIZ2_0                                          (0x01UL << SMC_AMSTIM2_HIZ2_Pos)                               /*!< 0x01000000 */
#define SMC_AMSTIM2_HIZ2_1                                          (0x02UL << SMC_AMSTIM2_HIZ2_Pos)                               /*!< 0x02000000 */
#define SMC_AMSTIM2_HIZ2_2                                          (0x04UL << SMC_AMSTIM2_HIZ2_Pos)                               /*!< 0x04000000 */
#define SMC_AMSTIM2_HIZ2_3                                          (0x08UL << SMC_AMSTIM2_HIZ2_Pos)                               /*!< 0x08000000 */
#define SMC_AMSTIM2_HIZ2_4                                          (0x10UL << SMC_AMSTIM2_HIZ2_Pos)                               /*!< 0x10000000 */
#define SMC_AMSTIM2_HIZ2_5                                          (0x20UL << SMC_AMSTIM2_HIZ2_Pos)                               /*!< 0x20000000 */
#define SMC_AMSTIM2_HIZ2_6                                          (0x40UL << SMC_AMSTIM2_HIZ2_Pos)                               /*!< 0x40000000 */
#define SMC_AMSTIM2_HIZ2_7                                          (0x80UL << SMC_AMSTIM2_HIZ2_Pos)                               /*!< 0x80000000 */

/******************  Bit definition for SMC_AMSTIM3 register  ******************/
#define SMC_AMSTIM3_SET3_Pos                                        (0U)
#define SMC_AMSTIM3_SET3_Msk                                        (0xFFUL << SMC_AMSTIM3_SET3_Pos)                               /*!< 0x000000FF */
#define SMC_AMSTIM3_SET3                                            SMC_AMSTIM3_SET3_Msk                                           /*!<ATTSET3[7:0] bits (Attribute memory 3 setup time) */
#define SMC_AMSTIM3_SET3_0                                          (0x01UL << SMC_AMSTIM3_SET3_Pos)                               /*!< 0x00000001 */
#define SMC_AMSTIM3_SET3_1                                          (0x02UL << SMC_AMSTIM3_SET3_Pos)                               /*!< 0x00000002 */
#define SMC_AMSTIM3_SET3_2                                          (0x04UL << SMC_AMSTIM3_SET3_Pos)                               /*!< 0x00000004 */
#define SMC_AMSTIM3_SET3_3                                          (0x08UL << SMC_AMSTIM3_SET3_Pos)                               /*!< 0x00000008 */
#define SMC_AMSTIM3_SET3_4                                          (0x10UL << SMC_AMSTIM3_SET3_Pos)                               /*!< 0x00000010 */
#define SMC_AMSTIM3_SET3_5                                          (0x20UL << SMC_AMSTIM3_SET3_Pos)                               /*!< 0x00000020 */
#define SMC_AMSTIM3_SET3_6                                          (0x40UL << SMC_AMSTIM3_SET3_Pos)                               /*!< 0x00000040 */
#define SMC_AMSTIM3_SET3_7                                          (0x80UL << SMC_AMSTIM3_SET3_Pos)                               /*!< 0x00000080 */

#define SMC_AMSTIM3_WAIT3_Pos                                       (8U)
#define SMC_AMSTIM3_WAIT3_Msk                                       (0xFFUL << SMC_AMSTIM3_WAIT3_Pos)                              /*!< 0x0000FF00 */
#define SMC_AMSTIM3_WAIT3                                           SMC_AMSTIM3_WAIT3_Msk                                          /*!<ATTWAIT3[7:0] bits (Attribute memory 3 wait time) */
#define SMC_AMSTIM3_WAIT3_0                                         (0x01UL << SMC_AMSTIM3_WAIT3_Pos)                              /*!< 0x00000100 */
#define SMC_AMSTIM3_WAIT3_1                                         (0x02UL << SMC_AMSTIM3_WAIT3_Pos)                              /*!< 0x00000200 */
#define SMC_AMSTIM3_WAIT3_2                                         (0x04UL << SMC_AMSTIM3_WAIT3_Pos)                              /*!< 0x00000400 */
#define SMC_AMSTIM3_WAIT3_3                                         (0x08UL << SMC_AMSTIM3_WAIT3_Pos)                              /*!< 0x00000800 */
#define SMC_AMSTIM3_WAIT3_4                                         (0x10UL << SMC_AMSTIM3_WAIT3_Pos)                              /*!< 0x00001000 */
#define SMC_AMSTIM3_WAIT3_5                                         (0x20UL << SMC_AMSTIM3_WAIT3_Pos)                              /*!< 0x00002000 */
#define SMC_AMSTIM3_WAIT3_6                                         (0x40UL << SMC_AMSTIM3_WAIT3_Pos)                              /*!< 0x00004000 */
#define SMC_AMSTIM3_WAIT3_7                                         (0x80UL << SMC_AMSTIM3_WAIT3_Pos)                              /*!< 0x00008000 */

#define SMC_AMSTIM3_HLD3_Pos                                        (16U)
#define SMC_AMSTIM3_HLD3_Msk                                        (0xFFUL << SMC_AMSTIM3_HLD3_Pos)                               /*!< 0x00FF0000 */
#define SMC_AMSTIM3_HLD3                                            SMC_AMSTIM3_HLD3_Msk                                           /*!<ATTHOLD3[7:0] bits (Attribute memory 3 hold time) */
#define SMC_AMSTIM3_HLD3_0                                          (0x01UL << SMC_AMSTIM3_HLD3_Pos)                               /*!< 0x00010000 */
#define SMC_AMSTIM3_HLD3_1                                          (0x02UL << SMC_AMSTIM3_HLD3_Pos)                               /*!< 0x00020000 */
#define SMC_AMSTIM3_HLD3_2                                          (0x04UL << SMC_AMSTIM3_HLD3_Pos)                               /*!< 0x00040000 */
#define SMC_AMSTIM3_HLD3_3                                          (0x08UL << SMC_AMSTIM3_HLD3_Pos)                               /*!< 0x00080000 */
#define SMC_AMSTIM3_HLD3_4                                          (0x10UL << SMC_AMSTIM3_HLD3_Pos)                               /*!< 0x00100000 */
#define SMC_AMSTIM3_HLD3_5                                          (0x20UL << SMC_AMSTIM3_HLD3_Pos)                               /*!< 0x00200000 */
#define SMC_AMSTIM3_HLD3_6                                          (0x40UL << SMC_AMSTIM3_HLD3_Pos)                               /*!< 0x00400000 */
#define SMC_AMSTIM3_HLD3_7                                          (0x80UL << SMC_AMSTIM3_HLD3_Pos)                               /*!< 0x00800000 */

#define SMC_AMSTIM3_HIZ3_Pos                                        (24U)
#define SMC_AMSTIM3_HIZ3_Msk                                        (0xFFUL << SMC_AMSTIM3_HIZ3_Pos)                               /*!< 0xFF000000 */
#define SMC_AMSTIM3_HIZ3                                            SMC_AMSTIM3_HIZ3_Msk                                           /*!<ATTHIZ3[7:0] bits (Attribute memory 3 databus HiZ time) */
#define SMC_AMSTIM3_HIZ3_0                                          (0x01UL << SMC_AMSTIM3_HIZ3_Pos)                               /*!< 0x01000000 */
#define SMC_AMSTIM3_HIZ3_1                                          (0x02UL << SMC_AMSTIM3_HIZ3_Pos)                               /*!< 0x02000000 */
#define SMC_AMSTIM3_HIZ3_2                                          (0x04UL << SMC_AMSTIM3_HIZ3_Pos)                               /*!< 0x04000000 */
#define SMC_AMSTIM3_HIZ3_3                                          (0x08UL << SMC_AMSTIM3_HIZ3_Pos)                               /*!< 0x08000000 */
#define SMC_AMSTIM3_HIZ3_4                                          (0x10UL << SMC_AMSTIM3_HIZ3_Pos)                               /*!< 0x10000000 */
#define SMC_AMSTIM3_HIZ3_5                                          (0x20UL << SMC_AMSTIM3_HIZ3_Pos)                               /*!< 0x20000000 */
#define SMC_AMSTIM3_HIZ3_6                                          (0x40UL << SMC_AMSTIM3_HIZ3_Pos)                               /*!< 0x40000000 */
#define SMC_AMSTIM3_HIZ3_7                                          (0x80UL << SMC_AMSTIM3_HIZ3_Pos)                               /*!< 0x80000000 */

/******************  Bit definition for SMC_AMSTIM4 register  ******************/
#define SMC_AMSTIM4_SET4_Pos                                        (0U)
#define SMC_AMSTIM4_SET4_Msk                                        (0xFFUL << SMC_AMSTIM4_SET4_Pos)                               /*!< 0x000000FF */
#define SMC_AMSTIM4_SET4                                            SMC_AMSTIM4_SET4_Msk                                           /*!<ATTSET4[7:0] bits (Attribute memory 4 setup time) */
#define SMC_AMSTIM4_SET4_0                                          (0x01UL << SMC_AMSTIM4_SET4_Pos)                               /*!< 0x00000001 */
#define SMC_AMSTIM4_SET4_1                                          (0x02UL << SMC_AMSTIM4_SET4_Pos)                               /*!< 0x00000002 */
#define SMC_AMSTIM4_SET4_2                                          (0x04UL << SMC_AMSTIM4_SET4_Pos)                               /*!< 0x00000004 */
#define SMC_AMSTIM4_SET4_3                                          (0x08UL << SMC_AMSTIM4_SET4_Pos)                               /*!< 0x00000008 */
#define SMC_AMSTIM4_SET4_4                                          (0x10UL << SMC_AMSTIM4_SET4_Pos)                               /*!< 0x00000010 */
#define SMC_AMSTIM4_SET4_5                                          (0x20UL << SMC_AMSTIM4_SET4_Pos)                               /*!< 0x00000020 */
#define SMC_AMSTIM4_SET4_6                                          (0x40UL << SMC_AMSTIM4_SET4_Pos)                               /*!< 0x00000040 */
#define SMC_AMSTIM4_SET4_7                                          (0x80UL << SMC_AMSTIM4_SET4_Pos)                               /*!< 0x00000080 */

#define SMC_AMSTIM4_WAIT4_Pos                                       (8U)
#define SMC_AMSTIM4_WAIT4_Msk                                       (0xFFUL << SMC_AMSTIM4_WAIT4_Pos)                              /*!< 0x0000FF00 */
#define SMC_AMSTIM4_WAIT4                                           SMC_AMSTIM4_WAIT4_Msk                                          /*!<ATTWAIT4[7:0] bits (Attribute memory 4 wait time) */
#define SMC_AMSTIM4_WAIT4_0                                         (0x01UL << SMC_AMSTIM4_WAIT4_Pos)                              /*!< 0x00000100 */
#define SMC_AMSTIM4_WAIT4_1                                         (0x02UL << SMC_AMSTIM4_WAIT4_Pos)                              /*!< 0x00000200 */
#define SMC_AMSTIM4_WAIT4_2                                         (0x04UL << SMC_AMSTIM4_WAIT4_Pos)                              /*!< 0x00000400 */
#define SMC_AMSTIM4_WAIT4_3                                         (0x08UL << SMC_AMSTIM4_WAIT4_Pos)                              /*!< 0x00000800 */
#define SMC_AMSTIM4_WAIT4_4                                         (0x10UL << SMC_AMSTIM4_WAIT4_Pos)                              /*!< 0x00001000 */
#define SMC_AMSTIM4_WAIT4_5                                         (0x20UL << SMC_AMSTIM4_WAIT4_Pos)                              /*!< 0x00002000 */
#define SMC_AMSTIM4_WAIT4_6                                         (0x40UL << SMC_AMSTIM4_WAIT4_Pos)                              /*!< 0x00004000 */
#define SMC_AMSTIM4_WAIT4_7                                         (0x80UL << SMC_AMSTIM4_WAIT4_Pos)                              /*!< 0x00008000 */

#define SMC_AMSTIM4_HLD4_Pos                                        (16U)
#define SMC_AMSTIM4_HLD4_Msk                                        (0xFFUL << SMC_AMSTIM4_HLD4_Pos)                               /*!< 0x00FF0000 */
#define SMC_AMSTIM4_HLD4                                            SMC_AMSTIM4_HLD4_Msk                                           /*!<ATTHOLD4[7:0] bits (Attribute memory 4 hold time) */
#define SMC_AMSTIM4_HLD4_0                                          (0x01UL << SMC_AMSTIM4_HLD4_Pos)                               /*!< 0x00010000 */
#define SMC_AMSTIM4_HLD4_1                                          (0x02UL << SMC_AMSTIM4_HLD4_Pos)                               /*!< 0x00020000 */
#define SMC_AMSTIM4_HLD4_2                                          (0x04UL << SMC_AMSTIM4_HLD4_Pos)                               /*!< 0x00040000 */
#define SMC_AMSTIM4_HLD4_3                                          (0x08UL << SMC_AMSTIM4_HLD4_Pos)                               /*!< 0x00080000 */
#define SMC_AMSTIM4_HLD4_4                                          (0x10UL << SMC_AMSTIM4_HLD4_Pos)                               /*!< 0x00100000 */
#define SMC_AMSTIM4_HLD4_5                                          (0x20UL << SMC_AMSTIM4_HLD4_Pos)                               /*!< 0x00200000 */
#define SMC_AMSTIM4_HLD4_6                                          (0x40UL << SMC_AMSTIM4_HLD4_Pos)                               /*!< 0x00400000 */
#define SMC_AMSTIM4_HLD4_7                                          (0x80UL << SMC_AMSTIM4_HLD4_Pos)                               /*!< 0x00800000 */

#define SMC_AMSTIM4_HIZ4_Pos                                        (24U)
#define SMC_AMSTIM4_HIZ4_Msk                                        (0xFFUL << SMC_AMSTIM4_HIZ4_Pos)                               /*!< 0xFF000000 */
#define SMC_AMSTIM4_HIZ4                                            SMC_AMSTIM4_HIZ4_Msk                                           /*!<ATTHIZ4[7:0] bits (Attribute memory 4 databus HiZ time) */
#define SMC_AMSTIM4_HIZ4_0                                          (0x01UL << SMC_AMSTIM4_HIZ4_Pos)                               /*!< 0x01000000 */
#define SMC_AMSTIM4_HIZ4_1                                          (0x02UL << SMC_AMSTIM4_HIZ4_Pos)                               /*!< 0x02000000 */
#define SMC_AMSTIM4_HIZ4_2                                          (0x04UL << SMC_AMSTIM4_HIZ4_Pos)                               /*!< 0x04000000 */
#define SMC_AMSTIM4_HIZ4_3                                          (0x08UL << SMC_AMSTIM4_HIZ4_Pos)                               /*!< 0x08000000 */
#define SMC_AMSTIM4_HIZ4_4                                          (0x10UL << SMC_AMSTIM4_HIZ4_Pos)                               /*!< 0x10000000 */
#define SMC_AMSTIM4_HIZ4_5                                          (0x20UL << SMC_AMSTIM4_HIZ4_Pos)                               /*!< 0x20000000 */
#define SMC_AMSTIM4_HIZ4_6                                          (0x40UL << SMC_AMSTIM4_HIZ4_Pos)                               /*!< 0x40000000 */
#define SMC_AMSTIM4_HIZ4_7                                          (0x80UL << SMC_AMSTIM4_HIZ4_Pos)                               /*!< 0x80000000 */

/******************  Bit definition for SMC_IOSTIM4 register  *******************/
#define SMC_IOSTIM4_SET4_Pos                                        (0U)
#define SMC_IOSTIM4_SET4_Msk                                        (0xFFUL << SMC_IOSTIM4_SET4_Pos)                               /*!< 0x000000FF */
#define SMC_IOSTIM4_SET4                                            SMC_IOSTIM4_SET4_Msk                                           /*!<IOSET4[7:0] bits (I/O 4 setup time) */
#define SMC_IOSTIM4_SET4_0                                          (0x01UL << SMC_IOSTIM4_SET4_Pos)                               /*!< 0x00000001 */
#define SMC_IOSTIM4_SET4_1                                          (0x02UL << SMC_IOSTIM4_SET4_Pos)                               /*!< 0x00000002 */
#define SMC_IOSTIM4_SET4_2                                          (0x04UL << SMC_IOSTIM4_SET4_Pos)                               /*!< 0x00000004 */
#define SMC_IOSTIM4_SET4_3                                          (0x08UL << SMC_IOSTIM4_SET4_Pos)                               /*!< 0x00000008 */
#define SMC_IOSTIM4_SET4_4                                          (0x10UL << SMC_IOSTIM4_SET4_Pos)                               /*!< 0x00000010 */
#define SMC_IOSTIM4_SET4_5                                          (0x20UL << SMC_IOSTIM4_SET4_Pos)                               /*!< 0x00000020 */
#define SMC_IOSTIM4_SET4_6                                          (0x40UL << SMC_IOSTIM4_SET4_Pos)                               /*!< 0x00000040 */
#define SMC_IOSTIM4_SET4_7                                          (0x80UL << SMC_IOSTIM4_SET4_Pos)                               /*!< 0x00000080 */

#define SMC_IOSTIM4_WAIT4_Pos                                       (8U)
#define SMC_IOSTIM4_WAIT4_Msk                                       (0xFFUL << SMC_IOSTIM4_WAIT4_Pos)                              /*!< 0x0000FF00 */
#define SMC_IOSTIM4_WAIT4                                           SMC_IOSTIM4_WAIT4_Msk                                          /*!<IOWAIT4[7:0] bits (I/O 4 wait time) */
#define SMC_IOSTIM4_WAIT4_0                                         (0x01UL << SMC_IOSTIM4_WAIT4_Pos)                              /*!< 0x00000100 */
#define SMC_IOSTIM4_WAIT4_1                                         (0x02UL << SMC_IOSTIM4_WAIT4_Pos)                              /*!< 0x00000200 */
#define SMC_IOSTIM4_WAIT4_2                                         (0x04UL << SMC_IOSTIM4_WAIT4_Pos)                              /*!< 0x00000400 */
#define SMC_IOSTIM4_WAIT4_3                                         (0x08UL << SMC_IOSTIM4_WAIT4_Pos)                              /*!< 0x00000800 */
#define SMC_IOSTIM4_WAIT4_4                                         (0x10UL << SMC_IOSTIM4_WAIT4_Pos)                              /*!< 0x00001000 */
#define SMC_IOSTIM4_WAIT4_5                                         (0x20UL << SMC_IOSTIM4_WAIT4_Pos)                              /*!< 0x00002000 */
#define SMC_IOSTIM4_WAIT4_6                                         (0x40UL << SMC_IOSTIM4_WAIT4_Pos)                              /*!< 0x00004000 */
#define SMC_IOSTIM4_WAIT4_7                                         (0x80UL << SMC_IOSTIM4_WAIT4_Pos)                              /*!< 0x00008000 */

#define SMC_IOSTIM4_HLD4_Pos                                        (16U)
#define SMC_IOSTIM4_HLD4_Msk                                        (0xFFUL << SMC_IOSTIM4_HLD4_Pos)                               /*!< 0x00FF0000 */
#define SMC_IOSTIM4_HLD4                                            SMC_IOSTIM4_HLD4_Msk                                           /*!<IOHOLD4[7:0] bits (I/O 4 hold time) */
#define SMC_IOSTIM4_HLD4_0                                          (0x01UL << SMC_IOSTIM4_HLD4_Pos)                               /*!< 0x00010000 */
#define SMC_IOSTIM4_HLD4_1                                          (0x02UL << SMC_IOSTIM4_HLD4_Pos)                               /*!< 0x00020000 */
#define SMC_IOSTIM4_HLD4_2                                          (0x04UL << SMC_IOSTIM4_HLD4_Pos)                               /*!< 0x00040000 */
#define SMC_IOSTIM4_HLD4_3                                          (0x08UL << SMC_IOSTIM4_HLD4_Pos)                               /*!< 0x00080000 */
#define SMC_IOSTIM4_HLD4_4                                          (0x10UL << SMC_IOSTIM4_HLD4_Pos)                               /*!< 0x00100000 */
#define SMC_IOSTIM4_HLD4_5                                          (0x20UL << SMC_IOSTIM4_HLD4_Pos)                               /*!< 0x00200000 */
#define SMC_IOSTIM4_HLD4_6                                          (0x40UL << SMC_IOSTIM4_HLD4_Pos)                               /*!< 0x00400000 */
#define SMC_IOSTIM4_HLD4_7                                          (0x80UL << SMC_IOSTIM4_HLD4_Pos)                               /*!< 0x00800000 */

#define SMC_IOSTIM4_HIZ4_Pos                                        (24U)
#define SMC_IOSTIM4_HIZ4_Msk                                        (0xFFUL << SMC_IOSTIM4_HIZ4_Pos)                               /*!< 0xFF000000 */
#define SMC_IOSTIM4_HIZ4                                            SMC_IOSTIM4_HIZ4_Msk                                           /*!<IOHIZ4[7:0] bits (I/O 4 databus HiZ time) */
#define SMC_IOSTIM4_HIZ4_0                                          (0x01UL << SMC_IOSTIM4_HIZ4_Pos)                               /*!< 0x01000000 */
#define SMC_IOSTIM4_HIZ4_1                                          (0x02UL << SMC_IOSTIM4_HIZ4_Pos)                               /*!< 0x02000000 */
#define SMC_IOSTIM4_HIZ4_2                                          (0x04UL << SMC_IOSTIM4_HIZ4_Pos)                               /*!< 0x04000000 */
#define SMC_IOSTIM4_HIZ4_3                                          (0x08UL << SMC_IOSTIM4_HIZ4_Pos)                               /*!< 0x08000000 */
#define SMC_IOSTIM4_HIZ4_4                                          (0x10UL << SMC_IOSTIM4_HIZ4_Pos)                               /*!< 0x10000000 */
#define SMC_IOSTIM4_HIZ4_5                                          (0x20UL << SMC_IOSTIM4_HIZ4_Pos)                               /*!< 0x20000000 */
#define SMC_IOSTIM4_HIZ4_6                                          (0x40UL << SMC_IOSTIM4_HIZ4_Pos)                               /*!< 0x40000000 */
#define SMC_IOSTIM4_HIZ4_7                                          (0x80UL << SMC_IOSTIM4_HIZ4_Pos)                               /*!< 0x80000000 */

/******************  Bit definition for SMC_ECCRS2 register  ******************/
#define SMC_ECCRS2_ECCRS2_Pos                                       (0U)
#define SMC_ECCRS2_ECCRS2_Msk                                       (0xFFFFFFFFUL << SMC_ECCRS2_ECCRS2_Pos)                        /*!< 0xFFFFFFFF */
#define SMC_ECCRS2_ECCRS2                                           SMC_ECCRS2_ECCRS2_Msk                                          /*!<ECC result */

/******************  Bit definition for SMC_ECCRS3 register  ******************/
#define SMC_ECCRS3_ECCRS3_Pos                                       (0U)
#define SMC_ECCRS3_ECCRS3_Msk                                       (0xFFFFFFFFUL << SMC_ECCRS3_ECCRS3_Pos)                        /*!< 0xFFFFFFFF */
#define SMC_ECCRS3_ECCRS3                                           SMC_ECCRS3_ECCRS3_Msk                                          /*!<ECC result */

/******************************************************************************/
/*                                                                            */
/*                            General Purpose I/O                             */
/*                                                                            */
/******************************************************************************/
/******************  Bits definition for GPIO_MODE register  *****************/
#define GPIO_MODE_MODE0_Pos                                         (0U)
#define GPIO_MODE_MODE0_Msk                                         (0x3UL << GPIO_MODE_MODE0_Pos)                                 /*!< 0x00000003 */
#define GPIO_MODE_MODE0                                             GPIO_MODE_MODE0_Msk
#define GPIO_MODE_MODE0_0                                           (0x1UL << GPIO_MODE_MODE0_Pos)                                 /*!< 0x00000001 */
#define GPIO_MODE_MODE0_1                                           (0x2UL << GPIO_MODE_MODE0_Pos)                                 /*!< 0x00000002 */
#define GPIO_MODE_MODE1_Pos                                         (2U)
#define GPIO_MODE_MODE1_Msk                                         (0x3UL << GPIO_MODE_MODE1_Pos)                                 /*!< 0x0000000C */
#define GPIO_MODE_MODE1                                             GPIO_MODE_MODE1_Msk
#define GPIO_MODE_MODE1_0                                           (0x1UL << GPIO_MODE_MODE1_Pos)                                 /*!< 0x00000004 */
#define GPIO_MODE_MODE1_1                                           (0x2UL << GPIO_MODE_MODE1_Pos)                                 /*!< 0x00000008 */
#define GPIO_MODE_MODE2_Pos                                         (4U)
#define GPIO_MODE_MODE2_Msk                                         (0x3UL << GPIO_MODE_MODE2_Pos)                                 /*!< 0x00000030 */
#define GPIO_MODE_MODE2                                             GPIO_MODE_MODE2_Msk
#define GPIO_MODE_MODE2_0                                           (0x1UL << GPIO_MODE_MODE2_Pos)                                 /*!< 0x00000010 */
#define GPIO_MODE_MODE2_1                                           (0x2UL << GPIO_MODE_MODE2_Pos)                                 /*!< 0x00000020 */
#define GPIO_MODE_MODE3_Pos                                         (6U)
#define GPIO_MODE_MODE3_Msk                                         (0x3UL << GPIO_MODE_MODE3_Pos)                                 /*!< 0x000000C0 */
#define GPIO_MODE_MODE3                                             GPIO_MODE_MODE3_Msk
#define GPIO_MODE_MODE3_0                                           (0x1UL << GPIO_MODE_MODE3_Pos)                                 /*!< 0x00000040 */
#define GPIO_MODE_MODE3_1                                           (0x2UL << GPIO_MODE_MODE3_Pos)                                 /*!< 0x00000080 */
#define GPIO_MODE_MODE4_Pos                                         (8U)
#define GPIO_MODE_MODE4_Msk                                         (0x3UL << GPIO_MODE_MODE4_Pos)                                 /*!< 0x00000300 */
#define GPIO_MODE_MODE4                                             GPIO_MODE_MODE4_Msk
#define GPIO_MODE_MODE4_0                                           (0x1UL << GPIO_MODE_MODE4_Pos)                                 /*!< 0x00000100 */
#define GPIO_MODE_MODE4_1                                           (0x2UL << GPIO_MODE_MODE4_Pos)                                 /*!< 0x00000200 */
#define GPIO_MODE_MODE5_Pos                                         (10U)
#define GPIO_MODE_MODE5_Msk                                         (0x3UL << GPIO_MODE_MODE5_Pos)                                 /*!< 0x00000C00 */
#define GPIO_MODE_MODE5                                             GPIO_MODE_MODE5_Msk
#define GPIO_MODE_MODE5_0                                           (0x1UL << GPIO_MODE_MODE5_Pos)                                 /*!< 0x00000400 */
#define GPIO_MODE_MODE5_1                                           (0x2UL << GPIO_MODE_MODE5_Pos)                                 /*!< 0x00000800 */
#define GPIO_MODE_MODE6_Pos                                         (12U)
#define GPIO_MODE_MODE6_Msk                                         (0x3UL << GPIO_MODE_MODE6_Pos)                                 /*!< 0x00003000 */
#define GPIO_MODE_MODE6                                             GPIO_MODE_MODE6_Msk
#define GPIO_MODE_MODE6_0                                           (0x1UL << GPIO_MODE_MODE6_Pos)                                 /*!< 0x00001000 */
#define GPIO_MODE_MODE6_1                                           (0x2UL << GPIO_MODE_MODE6_Pos)                                 /*!< 0x00002000 */
#define GPIO_MODE_MODE7_Pos                                         (14U)
#define GPIO_MODE_MODE7_Msk                                         (0x3UL << GPIO_MODE_MODE7_Pos)                                 /*!< 0x0000C000 */
#define GPIO_MODE_MODE7                                             GPIO_MODE_MODE7_Msk
#define GPIO_MODE_MODE7_0                                           (0x1UL << GPIO_MODE_MODE7_Pos)                                 /*!< 0x00004000 */
#define GPIO_MODE_MODE7_1                                           (0x2UL << GPIO_MODE_MODE7_Pos)                                 /*!< 0x00008000 */
#define GPIO_MODE_MODE8_Pos                                         (16U)
#define GPIO_MODE_MODE8_Msk                                         (0x3UL << GPIO_MODE_MODE8_Pos)                                 /*!< 0x00030000 */
#define GPIO_MODE_MODE8                                             GPIO_MODE_MODE8_Msk
#define GPIO_MODE_MODE8_0                                           (0x1UL << GPIO_MODE_MODE8_Pos)                                 /*!< 0x00010000 */
#define GPIO_MODE_MODE8_1                                           (0x2UL << GPIO_MODE_MODE8_Pos)                                 /*!< 0x00020000 */
#define GPIO_MODE_MODE9_Pos                                         (18U)
#define GPIO_MODE_MODE9_Msk                                         (0x3UL << GPIO_MODE_MODE9_Pos)                                 /*!< 0x000C0000 */
#define GPIO_MODE_MODE9                                             GPIO_MODE_MODE9_Msk
#define GPIO_MODE_MODE9_0                                           (0x1UL << GPIO_MODE_MODE9_Pos)                                 /*!< 0x00040000 */
#define GPIO_MODE_MODE9_1                                           (0x2UL << GPIO_MODE_MODE9_Pos)                                 /*!< 0x00080000 */
#define GPIO_MODE_MODE10_Pos                                        (20U)
#define GPIO_MODE_MODE10_Msk                                        (0x3UL << GPIO_MODE_MODE10_Pos)                                /*!< 0x00300000 */
#define GPIO_MODE_MODE10                                            GPIO_MODE_MODE10_Msk
#define GPIO_MODE_MODE10_0                                          (0x1UL << GPIO_MODE_MODE10_Pos)                                /*!< 0x00100000 */
#define GPIO_MODE_MODE10_1                                          (0x2UL << GPIO_MODE_MODE10_Pos)                                /*!< 0x00200000 */
#define GPIO_MODE_MODE11_Pos                                        (22U)
#define GPIO_MODE_MODE11_Msk                                        (0x3UL << GPIO_MODE_MODE11_Pos)                                /*!< 0x00C00000 */
#define GPIO_MODE_MODE11                                            GPIO_MODE_MODE11_Msk
#define GPIO_MODE_MODE11_0                                          (0x1UL << GPIO_MODE_MODE11_Pos)                                /*!< 0x00400000 */
#define GPIO_MODE_MODE11_1                                          (0x2UL << GPIO_MODE_MODE11_Pos)                                /*!< 0x00800000 */
#define GPIO_MODE_MODE12_Pos                                        (24U)
#define GPIO_MODE_MODE12_Msk                                        (0x3UL << GPIO_MODE_MODE12_Pos)                                /*!< 0x03000000 */
#define GPIO_MODE_MODE12                                            GPIO_MODE_MODE12_Msk
#define GPIO_MODE_MODE12_0                                          (0x1UL << GPIO_MODE_MODE12_Pos)                                /*!< 0x01000000 */
#define GPIO_MODE_MODE12_1                                          (0x2UL << GPIO_MODE_MODE12_Pos)                                /*!< 0x02000000 */
#define GPIO_MODE_MODE13_Pos                                        (26U)
#define GPIO_MODE_MODE13_Msk                                        (0x3UL << GPIO_MODE_MODE13_Pos)                                /*!< 0x0C000000 */
#define GPIO_MODE_MODE13                                            GPIO_MODE_MODE13_Msk
#define GPIO_MODE_MODE13_0                                          (0x1UL << GPIO_MODE_MODE13_Pos)                                /*!< 0x04000000 */
#define GPIO_MODE_MODE13_1                                          (0x2UL << GPIO_MODE_MODE13_Pos)                                /*!< 0x08000000 */
#define GPIO_MODE_MODE14_Pos                                        (28U)
#define GPIO_MODE_MODE14_Msk                                        (0x3UL << GPIO_MODE_MODE14_Pos)                                /*!< 0x30000000 */
#define GPIO_MODE_MODE14                                            GPIO_MODE_MODE14_Msk
#define GPIO_MODE_MODE14_0                                          (0x1UL << GPIO_MODE_MODE14_Pos)                                /*!< 0x10000000 */
#define GPIO_MODE_MODE14_1                                          (0x2UL << GPIO_MODE_MODE14_Pos)                                /*!< 0x20000000 */
#define GPIO_MODE_MODE15_Pos                                        (30U)
#define GPIO_MODE_MODE15_Msk                                        (0x3UL << GPIO_MODE_MODE15_Pos)                                /*!< 0xC0000000 */
#define GPIO_MODE_MODE15                                            GPIO_MODE_MODE15_Msk
#define GPIO_MODE_MODE15_0                                          (0x1UL << GPIO_MODE_MODE15_Pos)                                /*!< 0x40000000 */
#define GPIO_MODE_MODE15_1                                          (0x2UL << GPIO_MODE_MODE15_Pos)                                /*!< 0x80000000 */

/******************  Bits definition for GPIO_OMODE register  ****************/
#define GPIO_OMODE_OMODE0_Pos                                       (0U)
#define GPIO_OMODE_OMODE0_Msk                                       (0x1UL << GPIO_OMODE_OMODE0_Pos)                               /*!< 0x00000001 */
#define GPIO_OMODE_OMODE0                                           GPIO_OMODE_OMODE0_Msk
#define GPIO_OMODE_OMODE1_Pos                                       (1U)
#define GPIO_OMODE_OMODE1_Msk                                       (0x1UL << GPIO_OMODE_OMODE1_Pos)                               /*!< 0x00000002 */
#define GPIO_OMODE_OMODE1                                           GPIO_OMODE_OMODE1_Msk
#define GPIO_OMODE_OMODE2_Pos                                       (2U)
#define GPIO_OMODE_OMODE2_Msk                                       (0x1UL << GPIO_OMODE_OMODE2_Pos)                               /*!< 0x00000004 */
#define GPIO_OMODE_OMODE2                                           GPIO_OMODE_OMODE2_Msk
#define GPIO_OMODE_OMODE3_Pos                                       (3U)
#define GPIO_OMODE_OMODE3_Msk                                       (0x1UL << GPIO_OMODE_OMODE3_Pos)                               /*!< 0x00000008 */
#define GPIO_OMODE_OMODE3                                           GPIO_OMODE_OMODE3_Msk
#define GPIO_OMODE_OMODE4_Pos                                       (4U)
#define GPIO_OMODE_OMODE4_Msk                                       (0x1UL << GPIO_OMODE_OMODE4_Pos)                               /*!< 0x00000010 */
#define GPIO_OMODE_OMODE4                                           GPIO_OMODE_OMODE4_Msk
#define GPIO_OMODE_OMODE5_Pos                                       (5U)
#define GPIO_OMODE_OMODE5_Msk                                       (0x1UL << GPIO_OMODE_OMODE5_Pos)                               /*!< 0x00000020 */
#define GPIO_OMODE_OMODE5                                           GPIO_OMODE_OMODE5_Msk
#define GPIO_OMODE_OMODE6_Pos                                       (6U)
#define GPIO_OMODE_OMODE6_Msk                                       (0x1UL << GPIO_OMODE_OMODE6_Pos)                               /*!< 0x00000040 */
#define GPIO_OMODE_OMODE6                                           GPIO_OMODE_OMODE6_Msk
#define GPIO_OMODE_OMODE7_Pos                                       (7U)
#define GPIO_OMODE_OMODE7_Msk                                       (0x1UL << GPIO_OMODE_OMODE7_Pos)                               /*!< 0x00000080 */
#define GPIO_OMODE_OMODE7                                           GPIO_OMODE_OMODE7_Msk
#define GPIO_OMODE_OMODE8_Pos                                       (8U)
#define GPIO_OMODE_OMODE8_Msk                                       (0x1UL << GPIO_OMODE_OMODE8_Pos)                               /*!< 0x00000100 */
#define GPIO_OMODE_OMODE8                                           GPIO_OMODE_OMODE8_Msk
#define GPIO_OMODE_OMODE9_Pos                                       (9U)
#define GPIO_OMODE_OMODE9_Msk                                       (0x1UL << GPIO_OMODE_OMODE9_Pos)                               /*!< 0x00000200 */
#define GPIO_OMODE_OMODE9                                           GPIO_OMODE_OMODE9_Msk
#define GPIO_OMODE_OMODE10_Pos                                      (10U)
#define GPIO_OMODE_OMODE10_Msk                                      (0x1UL << GPIO_OMODE_OMODE10_Pos)                              /*!< 0x00000400 */
#define GPIO_OMODE_OMODE10                                          GPIO_OMODE_OMODE10_Msk
#define GPIO_OMODE_OMODE11_Pos                                      (11U)
#define GPIO_OMODE_OMODE11_Msk                                      (0x1UL << GPIO_OMODE_OMODE11_Pos)                              /*!< 0x00000800 */
#define GPIO_OMODE_OMODE11                                          GPIO_OMODE_OMODE11_Msk
#define GPIO_OMODE_OMODE12_Pos                                      (12U)
#define GPIO_OMODE_OMODE12_Msk                                      (0x1UL << GPIO_OMODE_OMODE12_Pos)                              /*!< 0x00001000 */
#define GPIO_OMODE_OMODE12                                          GPIO_OMODE_OMODE12_Msk
#define GPIO_OMODE_OMODE13_Pos                                      (13U)
#define GPIO_OMODE_OMODE13_Msk                                      (0x1UL << GPIO_OMODE_OMODE13_Pos)                              /*!< 0x00002000 */
#define GPIO_OMODE_OMODE13                                          GPIO_OMODE_OMODE13_Msk
#define GPIO_OMODE_OMODE14_Pos                                      (14U)
#define GPIO_OMODE_OMODE14_Msk                                      (0x1UL << GPIO_OMODE_OMODE14_Pos)                              /*!< 0x00004000 */
#define GPIO_OMODE_OMODE14                                          GPIO_OMODE_OMODE14_Msk
#define GPIO_OMODE_OMODE15_Pos                                      (15U)
#define GPIO_OMODE_OMODE15_Msk                                      (0x1UL << GPIO_OMODE_OMODE15_Pos)                              /*!< 0x00008000 */
#define GPIO_OMODE_OMODE15                                          GPIO_OMODE_OMODE15_Msk

/* Legacy defines */
#define GPIO_OMODE_OT_0                                             GPIO_OMODE_OMODE0
#define GPIO_OMODE_OT_1                                             GPIO_OMODE_OMODE1
#define GPIO_OMODE_OT_2                                             GPIO_OMODE_OMODE2
#define GPIO_OMODE_OT_3                                             GPIO_OMODE_OMODE3
#define GPIO_OMODE_OT_4                                             GPIO_OMODE_OMODE4
#define GPIO_OMODE_OT_5                                             GPIO_OMODE_OMODE5
#define GPIO_OMODE_OT_6                                             GPIO_OMODE_OMODE6
#define GPIO_OMODE_OT_7                                             GPIO_OMODE_OMODE7
#define GPIO_OMODE_OT_8                                             GPIO_OMODE_OMODE8
#define GPIO_OMODE_OT_9                                             GPIO_OMODE_OMODE9
#define GPIO_OMODE_OT_10                                            GPIO_OMODE_OMODE10
#define GPIO_OMODE_OT_11                                            GPIO_OMODE_OMODE11
#define GPIO_OMODE_OT_12                                            GPIO_OMODE_OMODE12
#define GPIO_OMODE_OT_13                                            GPIO_OMODE_OMODE13
#define GPIO_OMODE_OT_14                                            GPIO_OMODE_OMODE14
#define GPIO_OMODE_OT_15                                            GPIO_OMODE_OMODE15

/******************  Bits definition for GPIO_OSSEL register  ***************/
#define GPIO_OSSEL_OSSEL0_Pos                                       (0U)
#define GPIO_OSSEL_OSSEL0_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL0_Pos)                               /*!< 0x00000003 */
#define GPIO_OSSEL_OSSEL0                                           GPIO_OSSEL_OSSEL0_Msk
#define GPIO_OSSEL_OSSEL0_0                                         (0x1UL << GPIO_OSSEL_OSSEL0_Pos)                               /*!< 0x00000001 */
#define GPIO_OSSEL_OSSEL0_1                                         (0x2UL << GPIO_OSSEL_OSSEL0_Pos)                               /*!< 0x00000002 */
#define GPIO_OSSEL_OSSEL1_Pos                                       (2U)
#define GPIO_OSSEL_OSSEL1_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL1_Pos)                               /*!< 0x0000000C */
#define GPIO_OSSEL_OSSEL1                                           GPIO_OSSEL_OSSEL1_Msk
#define GPIO_OSSEL_OSSEL1_0                                         (0x1UL << GPIO_OSSEL_OSSEL1_Pos)                               /*!< 0x00000004 */
#define GPIO_OSSEL_OSSEL1_1                                         (0x2UL << GPIO_OSSEL_OSSEL1_Pos)                               /*!< 0x00000008 */
#define GPIO_OSSEL_OSSEL2_Pos                                       (4U)
#define GPIO_OSSEL_OSSEL2_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL2_Pos)                               /*!< 0x00000030 */
#define GPIO_OSSEL_OSSEL2                                           GPIO_OSSEL_OSSEL2_Msk
#define GPIO_OSSEL_OSSEL2_0                                         (0x1UL << GPIO_OSSEL_OSSEL2_Pos)                               /*!< 0x00000010 */
#define GPIO_OSSEL_OSSEL2_1                                         (0x2UL << GPIO_OSSEL_OSSEL2_Pos)                               /*!< 0x00000020 */
#define GPIO_OSSEL_OSSEL3_Pos                                       (6U)
#define GPIO_OSSEL_OSSEL3_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL3_Pos)                               /*!< 0x000000C0 */
#define GPIO_OSSEL_OSSEL3                                           GPIO_OSSEL_OSSEL3_Msk
#define GPIO_OSSEL_OSSEL3_0                                         (0x1UL << GPIO_OSSEL_OSSEL3_Pos)                               /*!< 0x00000040 */
#define GPIO_OSSEL_OSSEL3_1                                         (0x2UL << GPIO_OSSEL_OSSEL3_Pos)                               /*!< 0x00000080 */
#define GPIO_OSSEL_OSSEL4_Pos                                       (8U)
#define GPIO_OSSEL_OSSEL4_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL4_Pos)                               /*!< 0x00000300 */
#define GPIO_OSSEL_OSSEL4                                           GPIO_OSSEL_OSSEL4_Msk
#define GPIO_OSSEL_OSSEL4_0                                         (0x1UL << GPIO_OSSEL_OSSEL4_Pos)                               /*!< 0x00000100 */
#define GPIO_OSSEL_OSSEL4_1                                         (0x2UL << GPIO_OSSEL_OSSEL4_Pos)                               /*!< 0x00000200 */
#define GPIO_OSSEL_OSSEL5_Pos                                       (10U)
#define GPIO_OSSEL_OSSEL5_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL5_Pos)                               /*!< 0x00000C00 */
#define GPIO_OSSEL_OSSEL5                                           GPIO_OSSEL_OSSEL5_Msk
#define GPIO_OSSEL_OSSEL5_0                                         (0x1UL << GPIO_OSSEL_OSSEL5_Pos)                               /*!< 0x00000400 */
#define GPIO_OSSEL_OSSEL5_1                                         (0x2UL << GPIO_OSSEL_OSSEL5_Pos)                               /*!< 0x00000800 */
#define GPIO_OSSEL_OSSEL6_Pos                                       (12U)
#define GPIO_OSSEL_OSSEL6_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL6_Pos)                               /*!< 0x00003000 */
#define GPIO_OSSEL_OSSEL6                                           GPIO_OSSEL_OSSEL6_Msk
#define GPIO_OSSEL_OSSEL6_0                                         (0x1UL << GPIO_OSSEL_OSSEL6_Pos)                               /*!< 0x00001000 */
#define GPIO_OSSEL_OSSEL6_1                                         (0x2UL << GPIO_OSSEL_OSSEL6_Pos)                               /*!< 0x00002000 */
#define GPIO_OSSEL_OSSEL7_Pos                                       (14U)
#define GPIO_OSSEL_OSSEL7_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL7_Pos)                               /*!< 0x0000C000 */
#define GPIO_OSSEL_OSSEL7                                           GPIO_OSSEL_OSSEL7_Msk
#define GPIO_OSSEL_OSSEL7_0                                         (0x1UL << GPIO_OSSEL_OSSEL7_Pos)                               /*!< 0x00004000 */
#define GPIO_OSSEL_OSSEL7_1                                         (0x2UL << GPIO_OSSEL_OSSEL7_Pos)                               /*!< 0x00008000 */
#define GPIO_OSSEL_OSSEL8_Pos                                       (16U)
#define GPIO_OSSEL_OSSEL8_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL8_Pos)                               /*!< 0x00030000 */
#define GPIO_OSSEL_OSSEL8                                           GPIO_OSSEL_OSSEL8_Msk
#define GPIO_OSSEL_OSSEL8_0                                         (0x1UL << GPIO_OSSEL_OSSEL8_Pos)                               /*!< 0x00010000 */
#define GPIO_OSSEL_OSSEL8_1                                         (0x2UL << GPIO_OSSEL_OSSEL8_Pos)                               /*!< 0x00020000 */
#define GPIO_OSSEL_OSSEL9_Pos                                       (18U)
#define GPIO_OSSEL_OSSEL9_Msk                                       (0x3UL << GPIO_OSSEL_OSSEL9_Pos)                               /*!< 0x000C0000 */
#define GPIO_OSSEL_OSSEL9                                           GPIO_OSSEL_OSSEL9_Msk
#define GPIO_OSSEL_OSSEL9_0                                         (0x1UL << GPIO_OSSEL_OSSEL9_Pos)                               /*!< 0x00040000 */
#define GPIO_OSSEL_OSSEL9_1                                         (0x2UL << GPIO_OSSEL_OSSEL9_Pos)                               /*!< 0x00080000 */
#define GPIO_OSSEL_OSSEL10_Pos                                      (20U)
#define GPIO_OSSEL_OSSEL10_Msk                                      (0x3UL << GPIO_OSSEL_OSSEL10_Pos)                              /*!< 0x00300000 */
#define GPIO_OSSEL_OSSEL10                                          GPIO_OSSEL_OSSEL10_Msk
#define GPIO_OSSEL_OSSEL10_0                                        (0x1UL << GPIO_OSSEL_OSSEL10_Pos)                              /*!< 0x00100000 */
#define GPIO_OSSEL_OSSEL10_1                                        (0x2UL << GPIO_OSSEL_OSSEL10_Pos)                              /*!< 0x00200000 */
#define GPIO_OSSEL_OSSEL11_Pos                                      (22U)
#define GPIO_OSSEL_OSSEL11_Msk                                      (0x3UL << GPIO_OSSEL_OSSEL11_Pos)                              /*!< 0x00C00000 */
#define GPIO_OSSEL_OSSEL11                                          GPIO_OSSEL_OSSEL11_Msk
#define GPIO_OSSEL_OSSEL11_0                                        (0x1UL << GPIO_OSSEL_OSSEL11_Pos)                              /*!< 0x00400000 */
#define GPIO_OSSEL_OSSEL11_1                                        (0x2UL << GPIO_OSSEL_OSSEL11_Pos)                              /*!< 0x00800000 */
#define GPIO_OSSEL_OSSEL12_Pos                                      (24U)
#define GPIO_OSSEL_OSSEL12_Msk                                      (0x3UL << GPIO_OSSEL_OSSEL12_Pos)                              /*!< 0x03000000 */
#define GPIO_OSSEL_OSSEL12                                          GPIO_OSSEL_OSSEL12_Msk
#define GPIO_OSSEL_OSSEL12_0                                        (0x1UL << GPIO_OSSEL_OSSEL12_Pos)                              /*!< 0x01000000 */
#define GPIO_OSSEL_OSSEL12_1                                        (0x2UL << GPIO_OSSEL_OSSEL12_Pos)                              /*!< 0x02000000 */
#define GPIO_OSSEL_OSSEL13_Pos                                      (26U)
#define GPIO_OSSEL_OSSEL13_Msk                                      (0x3UL << GPIO_OSSEL_OSSEL13_Pos)                              /*!< 0x0C000000 */
#define GPIO_OSSEL_OSSEL13                                          GPIO_OSSEL_OSSEL13_Msk
#define GPIO_OSSEL_OSSEL13_0                                        (0x1UL << GPIO_OSSEL_OSSEL13_Pos)                              /*!< 0x04000000 */
#define GPIO_OSSEL_OSSEL13_1                                        (0x2UL << GPIO_OSSEL_OSSEL13_Pos)                              /*!< 0x08000000 */
#define GPIO_OSSEL_OSSEL14_Pos                                      (28U)
#define GPIO_OSSEL_OSSEL14_Msk                                      (0x3UL << GPIO_OSSEL_OSSEL14_Pos)                              /*!< 0x30000000 */
#define GPIO_OSSEL_OSSEL14                                          GPIO_OSSEL_OSSEL14_Msk
#define GPIO_OSSEL_OSSEL14_0                                        (0x1UL << GPIO_OSSEL_OSSEL14_Pos)                              /*!< 0x10000000 */
#define GPIO_OSSEL_OSSEL14_1                                        (0x2UL << GPIO_OSSEL_OSSEL14_Pos)                              /*!< 0x20000000 */
#define GPIO_OSSEL_OSSEL15_Pos                                      (30U)
#define GPIO_OSSEL_OSSEL15_Msk                                      (0x3UL << GPIO_OSSEL_OSSEL15_Pos)                              /*!< 0xC0000000 */
#define GPIO_OSSEL_OSSEL15                                          GPIO_OSSEL_OSSEL15_Msk
#define GPIO_OSSEL_OSSEL15_0                                        (0x1UL << GPIO_OSSEL_OSSEL15_Pos)                              /*!< 0x40000000 */
#define GPIO_OSSEL_OSSEL15_1                                        (0x2UL << GPIO_OSSEL_OSSEL15_Pos)                              /*!< 0x80000000 */

/* Legacy defines */
#define GPIO_OSPEEDER_OSPEEDR0                                      GPIO_OSSEL_OSSEL0
#define GPIO_OSPEEDER_OSPEEDR0_0                                    GPIO_OSSEL_OSSEL0_0
#define GPIO_OSPEEDER_OSPEEDR0_1                                    GPIO_OSSEL_OSSEL0_1
#define GPIO_OSPEEDER_OSPEEDR1                                      GPIO_OSSEL_OSSEL1
#define GPIO_OSPEEDER_OSPEEDR1_0                                    GPIO_OSSEL_OSSEL1_0
#define GPIO_OSPEEDER_OSPEEDR1_1                                    GPIO_OSSEL_OSSEL1_1
#define GPIO_OSPEEDER_OSPEEDR2                                      GPIO_OSSEL_OSSEL2
#define GPIO_OSPEEDER_OSPEEDR2_0                                    GPIO_OSSEL_OSSEL2_0
#define GPIO_OSPEEDER_OSPEEDR2_1                                    GPIO_OSSEL_OSSEL2_1
#define GPIO_OSPEEDER_OSPEEDR3                                      GPIO_OSSEL_OSSEL3
#define GPIO_OSPEEDER_OSPEEDR3_0                                    GPIO_OSSEL_OSSEL3_0
#define GPIO_OSPEEDER_OSPEEDR3_1                                    GPIO_OSSEL_OSSEL3_1
#define GPIO_OSPEEDER_OSPEEDR4                                      GPIO_OSSEL_OSSEL4
#define GPIO_OSPEEDER_OSPEEDR4_0                                    GPIO_OSSEL_OSSEL4_0
#define GPIO_OSPEEDER_OSPEEDR4_1                                    GPIO_OSSEL_OSSEL4_1
#define GPIO_OSPEEDER_OSPEEDR5                                      GPIO_OSSEL_OSSEL5
#define GPIO_OSPEEDER_OSPEEDR5_0                                    GPIO_OSSEL_OSSEL5_0
#define GPIO_OSPEEDER_OSPEEDR5_1                                    GPIO_OSSEL_OSSEL5_1
#define GPIO_OSPEEDER_OSPEEDR6                                      GPIO_OSSEL_OSSEL6
#define GPIO_OSPEEDER_OSPEEDR6_0                                    GPIO_OSSEL_OSSEL6_0
#define GPIO_OSPEEDER_OSPEEDR6_1                                    GPIO_OSSEL_OSSEL6_1
#define GPIO_OSPEEDER_OSPEEDR7                                      GPIO_OSSEL_OSSEL7
#define GPIO_OSPEEDER_OSPEEDR7_0                                    GPIO_OSSEL_OSSEL7_0
#define GPIO_OSPEEDER_OSPEEDR7_1                                    GPIO_OSSEL_OSSEL7_1
#define GPIO_OSPEEDER_OSPEEDR8                                      GPIO_OSSEL_OSSEL8
#define GPIO_OSPEEDER_OSPEEDR8_0                                    GPIO_OSSEL_OSSEL8_0
#define GPIO_OSPEEDER_OSPEEDR8_1                                    GPIO_OSSEL_OSSEL8_1
#define GPIO_OSPEEDER_OSPEEDR9                                      GPIO_OSSEL_OSSEL9
#define GPIO_OSPEEDER_OSPEEDR9_0                                    GPIO_OSSEL_OSSEL9_0
#define GPIO_OSPEEDER_OSPEEDR9_1                                    GPIO_OSSEL_OSSEL9_1
#define GPIO_OSPEEDER_OSPEEDR10                                     GPIO_OSSEL_OSSEL10
#define GPIO_OSPEEDER_OSPEEDR10_0                                   GPIO_OSSEL_OSSEL10_0
#define GPIO_OSPEEDER_OSPEEDR10_1                                   GPIO_OSSEL_OSSEL10_1
#define GPIO_OSPEEDER_OSPEEDR11                                     GPIO_OSSEL_OSSEL11
#define GPIO_OSPEEDER_OSPEEDR11_0                                   GPIO_OSSEL_OSSEL11_0
#define GPIO_OSPEEDER_OSPEEDR11_1                                   GPIO_OSSEL_OSSEL11_1
#define GPIO_OSPEEDER_OSPEEDR12                                     GPIO_OSSEL_OSSEL12
#define GPIO_OSPEEDER_OSPEEDR12_0                                   GPIO_OSSEL_OSSEL12_0
#define GPIO_OSPEEDER_OSPEEDR12_1                                   GPIO_OSSEL_OSSEL12_1
#define GPIO_OSPEEDER_OSPEEDR13                                     GPIO_OSSEL_OSSEL13
#define GPIO_OSPEEDER_OSPEEDR13_0                                   GPIO_OSSEL_OSSEL13_0
#define GPIO_OSPEEDER_OSPEEDR13_1                                   GPIO_OSSEL_OSSEL13_1
#define GPIO_OSPEEDER_OSPEEDR14                                     GPIO_OSSEL_OSSEL14
#define GPIO_OSPEEDER_OSPEEDR14_0                                   GPIO_OSSEL_OSSEL14_0
#define GPIO_OSPEEDER_OSPEEDR14_1                                   GPIO_OSSEL_OSSEL14_1
#define GPIO_OSPEEDER_OSPEEDR15                                     GPIO_OSSEL_OSSEL15
#define GPIO_OSPEEDER_OSPEEDR15_0                                   GPIO_OSSEL_OSSEL15_0
#define GPIO_OSPEEDER_OSPEEDR15_1                                   GPIO_OSSEL_OSSEL15_1

/******************  Bits definition for GPIO_PUPD register  *****************/
#define GPIO_PUPD_PUPD0_Pos                                         (0U)
#define GPIO_PUPD_PUPD0_Msk                                         (0x3UL << GPIO_PUPD_PUPD0_Pos)                                 /*!< 0x00000003 */
#define GPIO_PUPD_PUPD0                                             GPIO_PUPD_PUPD0_Msk
#define GPIO_PUPD_PUPD0_0                                           (0x1UL << GPIO_PUPD_PUPD0_Pos)                                 /*!< 0x00000001 */
#define GPIO_PUPD_PUPD0_1                                           (0x2UL << GPIO_PUPD_PUPD0_Pos)                                 /*!< 0x00000002 */
#define GPIO_PUPD_PUPD1_Pos                                         (2U)
#define GPIO_PUPD_PUPD1_Msk                                         (0x3UL << GPIO_PUPD_PUPD1_Pos)                                 /*!< 0x0000000C */
#define GPIO_PUPD_PUPD1                                             GPIO_PUPD_PUPD1_Msk
#define GPIO_PUPD_PUPD1_0                                           (0x1UL << GPIO_PUPD_PUPD1_Pos)                                 /*!< 0x00000004 */
#define GPIO_PUPD_PUPD1_1                                           (0x2UL << GPIO_PUPD_PUPD1_Pos)                                 /*!< 0x00000008 */
#define GPIO_PUPD_PUPD2_Pos                                         (4U)
#define GPIO_PUPD_PUPD2_Msk                                         (0x3UL << GPIO_PUPD_PUPD2_Pos)                                 /*!< 0x00000030 */
#define GPIO_PUPD_PUPD2                                             GPIO_PUPD_PUPD2_Msk
#define GPIO_PUPD_PUPD2_0                                           (0x1UL << GPIO_PUPD_PUPD2_Pos)                                 /*!< 0x00000010 */
#define GPIO_PUPD_PUPD2_1                                           (0x2UL << GPIO_PUPD_PUPD2_Pos)                                 /*!< 0x00000020 */
#define GPIO_PUPD_PUPD3_Pos                                         (6U)
#define GPIO_PUPD_PUPD3_Msk                                         (0x3UL << GPIO_PUPD_PUPD3_Pos)                                 /*!< 0x000000C0 */
#define GPIO_PUPD_PUPD3                                             GPIO_PUPD_PUPD3_Msk
#define GPIO_PUPD_PUPD3_0                                           (0x1UL << GPIO_PUPD_PUPD3_Pos)                                 /*!< 0x00000040 */
#define GPIO_PUPD_PUPD3_1                                           (0x2UL << GPIO_PUPD_PUPD3_Pos)                                 /*!< 0x00000080 */
#define GPIO_PUPD_PUPD4_Pos                                         (8U)
#define GPIO_PUPD_PUPD4_Msk                                         (0x3UL << GPIO_PUPD_PUPD4_Pos)                                 /*!< 0x00000300 */
#define GPIO_PUPD_PUPD4                                             GPIO_PUPD_PUPD4_Msk
#define GPIO_PUPD_PUPD4_0                                           (0x1UL << GPIO_PUPD_PUPD4_Pos)                                 /*!< 0x00000100 */
#define GPIO_PUPD_PUPD4_1                                           (0x2UL << GPIO_PUPD_PUPD4_Pos)                                 /*!< 0x00000200 */
#define GPIO_PUPD_PUPD5_Pos                                         (10U)
#define GPIO_PUPD_PUPD5_Msk                                         (0x3UL << GPIO_PUPD_PUPD5_Pos)                                 /*!< 0x00000C00 */
#define GPIO_PUPD_PUPD5                                             GPIO_PUPD_PUPD5_Msk
#define GPIO_PUPD_PUPD5_0                                           (0x1UL << GPIO_PUPD_PUPD5_Pos)                                 /*!< 0x00000400 */
#define GPIO_PUPD_PUPD5_1                                           (0x2UL << GPIO_PUPD_PUPD5_Pos)                                 /*!< 0x00000800 */
#define GPIO_PUPD_PUPD6_Pos                                         (12U)
#define GPIO_PUPD_PUPD6_Msk                                         (0x3UL << GPIO_PUPD_PUPD6_Pos)                                 /*!< 0x00003000 */
#define GPIO_PUPD_PUPD6                                             GPIO_PUPD_PUPD6_Msk
#define GPIO_PUPD_PUPD6_0                                           (0x1UL << GPIO_PUPD_PUPD6_Pos)                                 /*!< 0x00001000 */
#define GPIO_PUPD_PUPD6_1                                           (0x2UL << GPIO_PUPD_PUPD6_Pos)                                 /*!< 0x00002000 */
#define GPIO_PUPD_PUPD7_Pos                                         (14U)
#define GPIO_PUPD_PUPD7_Msk                                         (0x3UL << GPIO_PUPD_PUPD7_Pos)                                 /*!< 0x0000C000 */
#define GPIO_PUPD_PUPD7                                             GPIO_PUPD_PUPD7_Msk
#define GPIO_PUPD_PUPD7_0                                           (0x1UL << GPIO_PUPD_PUPD7_Pos)                                 /*!< 0x00004000 */
#define GPIO_PUPD_PUPD7_1                                           (0x2UL << GPIO_PUPD_PUPD7_Pos)                                 /*!< 0x00008000 */
#define GPIO_PUPD_PUPD8_Pos                                         (16U)
#define GPIO_PUPD_PUPD8_Msk                                         (0x3UL << GPIO_PUPD_PUPD8_Pos)                                 /*!< 0x00030000 */
#define GPIO_PUPD_PUPD8                                             GPIO_PUPD_PUPD8_Msk
#define GPIO_PUPD_PUPD8_0                                           (0x1UL << GPIO_PUPD_PUPD8_Pos)                                 /*!< 0x00010000 */
#define GPIO_PUPD_PUPD8_1                                           (0x2UL << GPIO_PUPD_PUPD8_Pos)                                 /*!< 0x00020000 */
#define GPIO_PUPD_PUPD9_Pos                                         (18U)
#define GPIO_PUPD_PUPD9_Msk                                         (0x3UL << GPIO_PUPD_PUPD9_Pos)                                 /*!< 0x000C0000 */
#define GPIO_PUPD_PUPD9                                             GPIO_PUPD_PUPD9_Msk
#define GPIO_PUPD_PUPD9_0                                           (0x1UL << GPIO_PUPD_PUPD9_Pos)                                 /*!< 0x00040000 */
#define GPIO_PUPD_PUPD9_1                                           (0x2UL << GPIO_PUPD_PUPD9_Pos)                                 /*!< 0x00080000 */
#define GPIO_PUPD_PUPD10_Pos                                        (20U)
#define GPIO_PUPD_PUPD10_Msk                                        (0x3UL << GPIO_PUPD_PUPD10_Pos)                                /*!< 0x00300000 */
#define GPIO_PUPD_PUPD10                                            GPIO_PUPD_PUPD10_Msk
#define GPIO_PUPD_PUPD10_0                                          (0x1UL << GPIO_PUPD_PUPD10_Pos)                                /*!< 0x00100000 */
#define GPIO_PUPD_PUPD10_1                                          (0x2UL << GPIO_PUPD_PUPD10_Pos)                                /*!< 0x00200000 */
#define GPIO_PUPD_PUPD11_Pos                                        (22U)
#define GPIO_PUPD_PUPD11_Msk                                        (0x3UL << GPIO_PUPD_PUPD11_Pos)                                /*!< 0x00C00000 */
#define GPIO_PUPD_PUPD11                                            GPIO_PUPD_PUPD11_Msk
#define GPIO_PUPD_PUPD11_0                                          (0x1UL << GPIO_PUPD_PUPD11_Pos)                                /*!< 0x00400000 */
#define GPIO_PUPD_PUPD11_1                                          (0x2UL << GPIO_PUPD_PUPD11_Pos)                                /*!< 0x00800000 */
#define GPIO_PUPD_PUPD12_Pos                                        (24U)
#define GPIO_PUPD_PUPD12_Msk                                        (0x3UL << GPIO_PUPD_PUPD12_Pos)                                /*!< 0x03000000 */
#define GPIO_PUPD_PUPD12                                            GPIO_PUPD_PUPD12_Msk
#define GPIO_PUPD_PUPD12_0                                          (0x1UL << GPIO_PUPD_PUPD12_Pos)                                /*!< 0x01000000 */
#define GPIO_PUPD_PUPD12_1                                          (0x2UL << GPIO_PUPD_PUPD12_Pos)                                /*!< 0x02000000 */
#define GPIO_PUPD_PUPD13_Pos                                        (26U)
#define GPIO_PUPD_PUPD13_Msk                                        (0x3UL << GPIO_PUPD_PUPD13_Pos)                                /*!< 0x0C000000 */
#define GPIO_PUPD_PUPD13                                            GPIO_PUPD_PUPD13_Msk
#define GPIO_PUPD_PUPD13_0                                          (0x1UL << GPIO_PUPD_PUPD13_Pos)                                /*!< 0x04000000 */
#define GPIO_PUPD_PUPD13_1                                          (0x2UL << GPIO_PUPD_PUPD13_Pos)                                /*!< 0x08000000 */
#define GPIO_PUPD_PUPD14_Pos                                        (28U)
#define GPIO_PUPD_PUPD14_Msk                                        (0x3UL << GPIO_PUPD_PUPD14_Pos)                                /*!< 0x30000000 */
#define GPIO_PUPD_PUPD14                                            GPIO_PUPD_PUPD14_Msk
#define GPIO_PUPD_PUPD14_0                                          (0x1UL << GPIO_PUPD_PUPD14_Pos)                                /*!< 0x10000000 */
#define GPIO_PUPD_PUPD14_1                                          (0x2UL << GPIO_PUPD_PUPD14_Pos)                                /*!< 0x20000000 */
#define GPIO_PUPD_PUPD15_Pos                                        (30U)
#define GPIO_PUPD_PUPD15_Msk                                        (0x3UL << GPIO_PUPD_PUPD15_Pos)                                /*!< 0xC0000000 */
#define GPIO_PUPD_PUPD15                                            GPIO_PUPD_PUPD15_Msk
#define GPIO_PUPD_PUPD15_0                                          (0x1UL << GPIO_PUPD_PUPD15_Pos)                                /*!< 0x40000000 */
#define GPIO_PUPD_PUPD15_1                                          (0x2UL << GPIO_PUPD_PUPD15_Pos)                                /*!< 0x80000000 */

/* Legacy defines */
#define GPIO_PUPD_PUPDR0                                            GPIO_PUPD_PUPD0
#define GPIO_PUPD_PUPDR0_0                                          GPIO_PUPD_PUPD0_0
#define GPIO_PUPD_PUPDR0_1                                          GPIO_PUPD_PUPD0_1
#define GPIO_PUPD_PUPDR1                                            GPIO_PUPD_PUPD1
#define GPIO_PUPD_PUPDR1_0                                          GPIO_PUPD_PUPD1_0
#define GPIO_PUPD_PUPDR1_1                                          GPIO_PUPD_PUPD1_1
#define GPIO_PUPD_PUPDR2                                            GPIO_PUPD_PUPD2
#define GPIO_PUPD_PUPDR2_0                                          GPIO_PUPD_PUPD2_0
#define GPIO_PUPD_PUPDR2_1                                          GPIO_PUPD_PUPD2_1
#define GPIO_PUPD_PUPDR3                                            GPIO_PUPD_PUPD3
#define GPIO_PUPD_PUPDR3_0                                          GPIO_PUPD_PUPD3_0
#define GPIO_PUPD_PUPDR3_1                                          GPIO_PUPD_PUPD3_1
#define GPIO_PUPD_PUPDR4                                            GPIO_PUPD_PUPD4
#define GPIO_PUPD_PUPDR4_0                                          GPIO_PUPD_PUPD4_0
#define GPIO_PUPD_PUPDR4_1                                          GPIO_PUPD_PUPD4_1
#define GPIO_PUPD_PUPDR5                                            GPIO_PUPD_PUPD5
#define GPIO_PUPD_PUPDR5_0                                          GPIO_PUPD_PUPD5_0
#define GPIO_PUPD_PUPDR5_1                                          GPIO_PUPD_PUPD5_1
#define GPIO_PUPD_PUPDR6                                            GPIO_PUPD_PUPD6
#define GPIO_PUPD_PUPDR6_0                                          GPIO_PUPD_PUPD6_0
#define GPIO_PUPD_PUPDR6_1                                          GPIO_PUPD_PUPD6_1
#define GPIO_PUPD_PUPDR7                                            GPIO_PUPD_PUPD7
#define GPIO_PUPD_PUPDR7_0                                          GPIO_PUPD_PUPD7_0
#define GPIO_PUPD_PUPDR7_1                                          GPIO_PUPD_PUPD7_1
#define GPIO_PUPD_PUPDR8                                            GPIO_PUPD_PUPD8
#define GPIO_PUPD_PUPDR8_0                                          GPIO_PUPD_PUPD8_0
#define GPIO_PUPD_PUPDR8_1                                          GPIO_PUPD_PUPD8_1
#define GPIO_PUPD_PUPDR9                                            GPIO_PUPD_PUPD9
#define GPIO_PUPD_PUPDR9_0                                          GPIO_PUPD_PUPD9_0
#define GPIO_PUPD_PUPDR9_1                                          GPIO_PUPD_PUPD9_1
#define GPIO_PUPD_PUPDR10                                           GPIO_PUPD_PUPD10
#define GPIO_PUPD_PUPDR10_0                                         GPIO_PUPD_PUPD10_0
#define GPIO_PUPD_PUPDR10_1                                         GPIO_PUPD_PUPD10_1
#define GPIO_PUPD_PUPDR11                                           GPIO_PUPD_PUPD11
#define GPIO_PUPD_PUPDR11_0                                         GPIO_PUPD_PUPD11_0
#define GPIO_PUPD_PUPDR11_1                                         GPIO_PUPD_PUPD11_1
#define GPIO_PUPD_PUPDR12                                           GPIO_PUPD_PUPD12
#define GPIO_PUPD_PUPDR12_0                                         GPIO_PUPD_PUPD12_0
#define GPIO_PUPD_PUPDR12_1                                         GPIO_PUPD_PUPD12_1
#define GPIO_PUPD_PUPDR13                                           GPIO_PUPD_PUPD13
#define GPIO_PUPD_PUPDR13_0                                         GPIO_PUPD_PUPD13_0
#define GPIO_PUPD_PUPDR13_1                                         GPIO_PUPD_PUPD13_1
#define GPIO_PUPD_PUPDR14                                           GPIO_PUPD_PUPD14
#define GPIO_PUPD_PUPDR14_0                                         GPIO_PUPD_PUPD14_0
#define GPIO_PUPD_PUPDR14_1                                         GPIO_PUPD_PUPD14_1
#define GPIO_PUPD_PUPDR15                                           GPIO_PUPD_PUPD15
#define GPIO_PUPD_PUPDR15_0                                         GPIO_PUPD_PUPD15_0
#define GPIO_PUPD_PUPDR15_1                                         GPIO_PUPD_PUPD15_1

/******************  Bits definition for GPIO_IDATA register  *******************/
#define GPIO_IDATA_IDATA0_Pos                                       (0U)
#define GPIO_IDATA_IDATA0_Msk                                       (0x1UL << GPIO_IDATA_IDATA0_Pos)                               /*!< 0x00000001 */
#define GPIO_IDATA_IDATA0                                           GPIO_IDATA_IDATA0_Msk
#define GPIO_IDATA_IDATA1_Pos                                       (1U)
#define GPIO_IDATA_IDATA1_Msk                                       (0x1UL << GPIO_IDATA_IDATA1_Pos)                               /*!< 0x00000002 */
#define GPIO_IDATA_IDATA1                                           GPIO_IDATA_IDATA1_Msk
#define GPIO_IDATA_IDATA2_Pos                                       (2U)
#define GPIO_IDATA_IDATA2_Msk                                       (0x1UL << GPIO_IDATA_IDATA2_Pos)                               /*!< 0x00000004 */
#define GPIO_IDATA_IDATA2                                           GPIO_IDATA_IDATA2_Msk
#define GPIO_IDATA_IDATA3_Pos                                       (3U)
#define GPIO_IDATA_IDATA3_Msk                                       (0x1UL << GPIO_IDATA_IDATA3_Pos)                               /*!< 0x00000008 */
#define GPIO_IDATA_IDATA3                                           GPIO_IDATA_IDATA3_Msk
#define GPIO_IDATA_IDATA4_Pos                                       (4U)
#define GPIO_IDATA_IDATA4_Msk                                       (0x1UL << GPIO_IDATA_IDATA4_Pos)                               /*!< 0x00000010 */
#define GPIO_IDATA_IDATA4                                           GPIO_IDATA_IDATA4_Msk
#define GPIO_IDATA_IDATA5_Pos                                       (5U)
#define GPIO_IDATA_IDATA5_Msk                                       (0x1UL << GPIO_IDATA_IDATA5_Pos)                               /*!< 0x00000020 */
#define GPIO_IDATA_IDATA5                                           GPIO_IDATA_IDATA5_Msk
#define GPIO_IDATA_IDATA6_Pos                                       (6U)
#define GPIO_IDATA_IDATA6_Msk                                       (0x1UL << GPIO_IDATA_IDATA6_Pos)                               /*!< 0x00000040 */
#define GPIO_IDATA_IDATA6                                           GPIO_IDATA_IDATA6_Msk
#define GPIO_IDATA_IDATA7_Pos                                       (7U)
#define GPIO_IDATA_IDATA7_Msk                                       (0x1UL << GPIO_IDATA_IDATA7_Pos)                               /*!< 0x00000080 */
#define GPIO_IDATA_IDATA7                                           GPIO_IDATA_IDATA7_Msk
#define GPIO_IDATA_IDATA8_Pos                                       (8U)
#define GPIO_IDATA_IDATA8_Msk                                       (0x1UL << GPIO_IDATA_IDATA8_Pos)                               /*!< 0x00000100 */
#define GPIO_IDATA_IDATA8                                           GPIO_IDATA_IDATA8_Msk
#define GPIO_IDATA_IDATA9_Pos                                       (9U)
#define GPIO_IDATA_IDATA9_Msk                                       (0x1UL << GPIO_IDATA_IDATA9_Pos)                               /*!< 0x00000200 */
#define GPIO_IDATA_IDATA9                                           GPIO_IDATA_IDATA9_Msk
#define GPIO_IDATA_IDATA10_Pos                                      (10U)
#define GPIO_IDATA_IDATA10_Msk                                      (0x1UL << GPIO_IDATA_IDATA10_Pos)                              /*!< 0x00000400 */
#define GPIO_IDATA_IDATA10                                          GPIO_IDATA_IDATA10_Msk
#define GPIO_IDATA_IDATA11_Pos                                      (11U)
#define GPIO_IDATA_IDATA11_Msk                                      (0x1UL << GPIO_IDATA_IDATA11_Pos)                              /*!< 0x00000800 */
#define GPIO_IDATA_IDATA11                                          GPIO_IDATA_IDATA11_Msk
#define GPIO_IDATA_IDATA12_Pos                                      (12U)
#define GPIO_IDATA_IDATA12_Msk                                      (0x1UL << GPIO_IDATA_IDATA12_Pos)                              /*!< 0x00001000 */
#define GPIO_IDATA_IDATA12                                          GPIO_IDATA_IDATA12_Msk
#define GPIO_IDATA_IDATA13_Pos                                      (13U)
#define GPIO_IDATA_IDATA13_Msk                                      (0x1UL << GPIO_IDATA_IDATA13_Pos)                              /*!< 0x00002000 */
#define GPIO_IDATA_IDATA13                                          GPIO_IDATA_IDATA13_Msk
#define GPIO_IDATA_IDATA14_Pos                                      (14U)
#define GPIO_IDATA_IDATA14_Msk                                      (0x1UL << GPIO_IDATA_IDATA14_Pos)                              /*!< 0x00004000 */
#define GPIO_IDATA_IDATA14                                          GPIO_IDATA_IDATA14_Msk
#define GPIO_IDATA_IDATA15_Pos                                      (15U)
#define GPIO_IDATA_IDATA15_Msk                                      (0x1UL << GPIO_IDATA_IDATA15_Pos)                              /*!< 0x00008000 */
#define GPIO_IDATA_IDATA15                                          GPIO_IDATA_IDATA15_Msk

/* Legacy defines */
#define GPIO_IDATA_IDR_0                                            GPIO_IDATA_IDATA0
#define GPIO_IDATA_IDR_1                                            GPIO_IDATA_IDATA1
#define GPIO_IDATA_IDR_2                                            GPIO_IDATA_IDATA2
#define GPIO_IDATA_IDR_3                                            GPIO_IDATA_IDATA3
#define GPIO_IDATA_IDR_4                                            GPIO_IDATA_IDATA4
#define GPIO_IDATA_IDR_5                                            GPIO_IDATA_IDATA5
#define GPIO_IDATA_IDR_6                                            GPIO_IDATA_IDATA6
#define GPIO_IDATA_IDR_7                                            GPIO_IDATA_IDATA7
#define GPIO_IDATA_IDR_8                                            GPIO_IDATA_IDATA8
#define GPIO_IDATA_IDR_9                                            GPIO_IDATA_IDATA9
#define GPIO_IDATA_IDR_10                                           GPIO_IDATA_IDATA10
#define GPIO_IDATA_IDR_11                                           GPIO_IDATA_IDATA11
#define GPIO_IDATA_IDR_12                                           GPIO_IDATA_IDATA12
#define GPIO_IDATA_IDR_13                                           GPIO_IDATA_IDATA13
#define GPIO_IDATA_IDR_14                                           GPIO_IDATA_IDATA14
#define GPIO_IDATA_IDR_15                                           GPIO_IDATA_IDATA15

/******************  Bits definition for GPIO_ODATA register  *******************/
#define GPIO_ODATA_ODATA0_Pos                                       (0U)
#define GPIO_ODATA_ODATA0_Msk                                       (0x1UL << GPIO_ODATA_ODATA0_Pos)                               /*!< 0x00000001 */
#define GPIO_ODATA_ODATA0                                           GPIO_ODATA_ODATA0_Msk
#define GPIO_ODATA_ODATA1_Pos                                       (1U)
#define GPIO_ODATA_ODATA1_Msk                                       (0x1UL << GPIO_ODATA_ODATA1_Pos)                               /*!< 0x00000002 */
#define GPIO_ODATA_ODATA1                                           GPIO_ODATA_ODATA1_Msk
#define GPIO_ODATA_ODATA2_Pos                                       (2U)
#define GPIO_ODATA_ODATA2_Msk                                       (0x1UL << GPIO_ODATA_ODATA2_Pos)                               /*!< 0x00000004 */
#define GPIO_ODATA_ODATA2                                           GPIO_ODATA_ODATA2_Msk
#define GPIO_ODATA_ODATA3_Pos                                       (3U)
#define GPIO_ODATA_ODATA3_Msk                                       (0x1UL << GPIO_ODATA_ODATA3_Pos)                               /*!< 0x00000008 */
#define GPIO_ODATA_ODATA3                                           GPIO_ODATA_ODATA3_Msk
#define GPIO_ODATA_ODATA4_Pos                                       (4U)
#define GPIO_ODATA_ODATA4_Msk                                       (0x1UL << GPIO_ODATA_ODATA4_Pos)                               /*!< 0x00000010 */
#define GPIO_ODATA_ODATA4                                           GPIO_ODATA_ODATA4_Msk
#define GPIO_ODATA_ODATA5_Pos                                       (5U)
#define GPIO_ODATA_ODATA5_Msk                                       (0x1UL << GPIO_ODATA_ODATA5_Pos)                               /*!< 0x00000020 */
#define GPIO_ODATA_ODATA5                                           GPIO_ODATA_ODATA5_Msk
#define GPIO_ODATA_ODATA6_Pos                                       (6U)
#define GPIO_ODATA_ODATA6_Msk                                       (0x1UL << GPIO_ODATA_ODATA6_Pos)                               /*!< 0x00000040 */
#define GPIO_ODATA_ODATA6                                           GPIO_ODATA_ODATA6_Msk
#define GPIO_ODATA_ODATA7_Pos                                       (7U)
#define GPIO_ODATA_ODATA7_Msk                                       (0x1UL << GPIO_ODATA_ODATA7_Pos)                               /*!< 0x00000080 */
#define GPIO_ODATA_ODATA7                                           GPIO_ODATA_ODATA7_Msk
#define GPIO_ODATA_ODATA8_Pos                                       (8U)
#define GPIO_ODATA_ODATA8_Msk                                       (0x1UL << GPIO_ODATA_ODATA8_Pos)                               /*!< 0x00000100 */
#define GPIO_ODATA_ODATA8                                           GPIO_ODATA_ODATA8_Msk
#define GPIO_ODATA_ODATA9_Pos                                       (9U)
#define GPIO_ODATA_ODATA9_Msk                                       (0x1UL << GPIO_ODATA_ODATA9_Pos)                               /*!< 0x00000200 */
#define GPIO_ODATA_ODATA9                                           GPIO_ODATA_ODATA9_Msk
#define GPIO_ODATA_ODATA10_Pos                                      (10U)
#define GPIO_ODATA_ODATA10_Msk                                      (0x1UL << GPIO_ODATA_ODATA10_Pos)                              /*!< 0x00000400 */
#define GPIO_ODATA_ODATA10                                          GPIO_ODATA_ODATA10_Msk
#define GPIO_ODATA_ODATA11_Pos                                      (11U)
#define GPIO_ODATA_ODATA11_Msk                                      (0x1UL << GPIO_ODATA_ODATA11_Pos)                              /*!< 0x00000800 */
#define GPIO_ODATA_ODATA11                                          GPIO_ODATA_ODATA11_Msk
#define GPIO_ODATA_ODATA12_Pos                                      (12U)
#define GPIO_ODATA_ODATA12_Msk                                      (0x1UL << GPIO_ODATA_ODATA12_Pos)                              /*!< 0x00001000 */
#define GPIO_ODATA_ODATA12                                          GPIO_ODATA_ODATA12_Msk
#define GPIO_ODATA_ODATA13_Pos                                      (13U)
#define GPIO_ODATA_ODATA13_Msk                                      (0x1UL << GPIO_ODATA_ODATA13_Pos)                              /*!< 0x00002000 */
#define GPIO_ODATA_ODATA13                                          GPIO_ODATA_ODATA13_Msk
#define GPIO_ODATA_ODATA14_Pos                                      (14U)
#define GPIO_ODATA_ODATA14_Msk                                      (0x1UL << GPIO_ODATA_ODATA14_Pos)                              /*!< 0x00004000 */
#define GPIO_ODATA_ODATA14                                          GPIO_ODATA_ODATA14_Msk
#define GPIO_ODATA_ODATA15_Pos                                      (15U)
#define GPIO_ODATA_ODATA15_Msk                                      (0x1UL << GPIO_ODATA_ODATA15_Pos)                              /*!< 0x00008000 */
#define GPIO_ODATA_ODATA15                                          GPIO_ODATA_ODATA15_Msk
/* Legacy defines */
#define GPIO_ODATA_ODR_0                                            GPIO_ODATA_ODATA0
#define GPIO_ODATA_ODR_1                                            GPIO_ODATA_ODATA1
#define GPIO_ODATA_ODR_2                                            GPIO_ODATA_ODATA2
#define GPIO_ODATA_ODR_3                                            GPIO_ODATA_ODATA3
#define GPIO_ODATA_ODR_4                                            GPIO_ODATA_ODATA4
#define GPIO_ODATA_ODR_5                                            GPIO_ODATA_ODATA5
#define GPIO_ODATA_ODR_6                                            GPIO_ODATA_ODATA6
#define GPIO_ODATA_ODR_7                                            GPIO_ODATA_ODATA7
#define GPIO_ODATA_ODR_8                                            GPIO_ODATA_ODATA8
#define GPIO_ODATA_ODR_9                                            GPIO_ODATA_ODATA9
#define GPIO_ODATA_ODR_10                                           GPIO_ODATA_ODATA10
#define GPIO_ODATA_ODR_11                                           GPIO_ODATA_ODATA11
#define GPIO_ODATA_ODR_12                                           GPIO_ODATA_ODATA12
#define GPIO_ODATA_ODR_13                                           GPIO_ODATA_ODATA13
#define GPIO_ODATA_ODR_14                                           GPIO_ODATA_ODATA14
#define GPIO_ODATA_ODR_15                                           GPIO_ODATA_ODATA15

/******************  Bits definition for GPIO_BSC register  ******************/
#define GPIO_BSC_BS0_Pos                                            (0U)
#define GPIO_BSC_BS0_Msk                                            (0x1UL << GPIO_BSC_BS0_Pos)                                    /*!< 0x00000001 */
#define GPIO_BSC_BS0                                                GPIO_BSC_BS0_Msk
#define GPIO_BSC_BS1_Pos                                            (1U)
#define GPIO_BSC_BS1_Msk                                            (0x1UL << GPIO_BSC_BS1_Pos)                                    /*!< 0x00000002 */
#define GPIO_BSC_BS1                                                GPIO_BSC_BS1_Msk
#define GPIO_BSC_BS2_Pos                                            (2U)
#define GPIO_BSC_BS2_Msk                                            (0x1UL << GPIO_BSC_BS2_Pos)                                    /*!< 0x00000004 */
#define GPIO_BSC_BS2                                                GPIO_BSC_BS2_Msk
#define GPIO_BSC_BS3_Pos                                            (3U)
#define GPIO_BSC_BS3_Msk                                            (0x1UL << GPIO_BSC_BS3_Pos)                                    /*!< 0x00000008 */
#define GPIO_BSC_BS3                                                GPIO_BSC_BS3_Msk
#define GPIO_BSC_BS4_Pos                                            (4U)
#define GPIO_BSC_BS4_Msk                                            (0x1UL << GPIO_BSC_BS4_Pos)                                    /*!< 0x00000010 */
#define GPIO_BSC_BS4                                                GPIO_BSC_BS4_Msk
#define GPIO_BSC_BS5_Pos                                            (5U)
#define GPIO_BSC_BS5_Msk                                            (0x1UL << GPIO_BSC_BS5_Pos)                                    /*!< 0x00000020 */
#define GPIO_BSC_BS5                                                GPIO_BSC_BS5_Msk
#define GPIO_BSC_BS6_Pos                                            (6U)
#define GPIO_BSC_BS6_Msk                                            (0x1UL << GPIO_BSC_BS6_Pos)                                    /*!< 0x00000040 */
#define GPIO_BSC_BS6                                                GPIO_BSC_BS6_Msk
#define GPIO_BSC_BS7_Pos                                            (7U)
#define GPIO_BSC_BS7_Msk                                            (0x1UL << GPIO_BSC_BS7_Pos)                                    /*!< 0x00000080 */
#define GPIO_BSC_BS7                                                GPIO_BSC_BS7_Msk
#define GPIO_BSC_BS8_Pos                                            (8U)
#define GPIO_BSC_BS8_Msk                                            (0x1UL << GPIO_BSC_BS8_Pos)                                    /*!< 0x00000100 */
#define GPIO_BSC_BS8                                                GPIO_BSC_BS8_Msk
#define GPIO_BSC_BS9_Pos                                            (9U)
#define GPIO_BSC_BS9_Msk                                            (0x1UL << GPIO_BSC_BS9_Pos)                                    /*!< 0x00000200 */
#define GPIO_BSC_BS9                                                GPIO_BSC_BS9_Msk
#define GPIO_BSC_BS10_Pos                                           (10U)
#define GPIO_BSC_BS10_Msk                                           (0x1UL << GPIO_BSC_BS10_Pos)                                   /*!< 0x00000400 */
#define GPIO_BSC_BS10                                               GPIO_BSC_BS10_Msk
#define GPIO_BSC_BS11_Pos                                           (11U)
#define GPIO_BSC_BS11_Msk                                           (0x1UL << GPIO_BSC_BS11_Pos)                                   /*!< 0x00000800 */
#define GPIO_BSC_BS11                                               GPIO_BSC_BS11_Msk
#define GPIO_BSC_BS12_Pos                                           (12U)
#define GPIO_BSC_BS12_Msk                                           (0x1UL << GPIO_BSC_BS12_Pos)                                   /*!< 0x00001000 */
#define GPIO_BSC_BS12                                               GPIO_BSC_BS12_Msk
#define GPIO_BSC_BS13_Pos                                           (13U)
#define GPIO_BSC_BS13_Msk                                           (0x1UL << GPIO_BSC_BS13_Pos)                                   /*!< 0x00002000 */
#define GPIO_BSC_BS13                                               GPIO_BSC_BS13_Msk
#define GPIO_BSC_BS14_Pos                                           (14U)
#define GPIO_BSC_BS14_Msk                                           (0x1UL << GPIO_BSC_BS14_Pos)                                   /*!< 0x00004000 */
#define GPIO_BSC_BS14                                               GPIO_BSC_BS14_Msk
#define GPIO_BSC_BS15_Pos                                           (15U)
#define GPIO_BSC_BS15_Msk                                           (0x1UL << GPIO_BSC_BS15_Pos)                                   /*!< 0x00008000 */
#define GPIO_BSC_BS15                                               GPIO_BSC_BS15_Msk
#define GPIO_BSC_BC0_Pos                                            (16U)
#define GPIO_BSC_BC0_Msk                                            (0x1UL << GPIO_BSC_BC0_Pos)                                    /*!< 0x00010000 */
#define GPIO_BSC_BC0                                                GPIO_BSC_BC0_Msk
#define GPIO_BSC_BC1_Pos                                            (17U)
#define GPIO_BSC_BC1_Msk                                            (0x1UL << GPIO_BSC_BC1_Pos)                                    /*!< 0x00020000 */
#define GPIO_BSC_BC1                                                GPIO_BSC_BC1_Msk
#define GPIO_BSC_BC2_Pos                                            (18U)
#define GPIO_BSC_BC2_Msk                                            (0x1UL << GPIO_BSC_BC2_Pos)                                    /*!< 0x00040000 */
#define GPIO_BSC_BC2                                                GPIO_BSC_BC2_Msk
#define GPIO_BSC_BC3_Pos                                            (19U)
#define GPIO_BSC_BC3_Msk                                            (0x1UL << GPIO_BSC_BC3_Pos)                                    /*!< 0x00080000 */
#define GPIO_BSC_BC3                                                GPIO_BSC_BC3_Msk
#define GPIO_BSC_BC4_Pos                                            (20U)
#define GPIO_BSC_BC4_Msk                                            (0x1UL << GPIO_BSC_BC4_Pos)                                    /*!< 0x00100000 */
#define GPIO_BSC_BC4                                                GPIO_BSC_BC4_Msk
#define GPIO_BSC_BC5_Pos                                            (21U)
#define GPIO_BSC_BC5_Msk                                            (0x1UL << GPIO_BSC_BC5_Pos)                                    /*!< 0x00200000 */
#define GPIO_BSC_BC5                                                GPIO_BSC_BC5_Msk
#define GPIO_BSC_BC6_Pos                                            (22U)
#define GPIO_BSC_BC6_Msk                                            (0x1UL << GPIO_BSC_BC6_Pos)                                    /*!< 0x00400000 */
#define GPIO_BSC_BC6                                                GPIO_BSC_BC6_Msk
#define GPIO_BSC_BC7_Pos                                            (23U)
#define GPIO_BSC_BC7_Msk                                            (0x1UL << GPIO_BSC_BC7_Pos)                                    /*!< 0x00800000 */
#define GPIO_BSC_BC7                                                GPIO_BSC_BC7_Msk
#define GPIO_BSC_BC8_Pos                                            (24U)
#define GPIO_BSC_BC8_Msk                                            (0x1UL << GPIO_BSC_BC8_Pos)                                    /*!< 0x01000000 */
#define GPIO_BSC_BC8                                                GPIO_BSC_BC8_Msk
#define GPIO_BSC_BC9_Pos                                            (25U)
#define GPIO_BSC_BC9_Msk                                            (0x1UL << GPIO_BSC_BC9_Pos)                                    /*!< 0x02000000 */
#define GPIO_BSC_BC9                                                GPIO_BSC_BC9_Msk
#define GPIO_BSC_BC10_Pos                                           (26U)
#define GPIO_BSC_BC10_Msk                                           (0x1UL << GPIO_BSC_BC10_Pos)                                   /*!< 0x04000000 */
#define GPIO_BSC_BC10                                               GPIO_BSC_BC10_Msk
#define GPIO_BSC_BC11_Pos                                           (27U)
#define GPIO_BSC_BC11_Msk                                           (0x1UL << GPIO_BSC_BC11_Pos)                                   /*!< 0x08000000 */
#define GPIO_BSC_BC11                                               GPIO_BSC_BC11_Msk
#define GPIO_BSC_BC12_Pos                                           (28U)
#define GPIO_BSC_BC12_Msk                                           (0x1UL << GPIO_BSC_BC12_Pos)                                   /*!< 0x10000000 */
#define GPIO_BSC_BC12                                               GPIO_BSC_BC12_Msk
#define GPIO_BSC_BC13_Pos                                           (29U)
#define GPIO_BSC_BC13_Msk                                           (0x1UL << GPIO_BSC_BC13_Pos)                                   /*!< 0x20000000 */
#define GPIO_BSC_BC13                                               GPIO_BSC_BC13_Msk
#define GPIO_BSC_BC14_Pos                                           (30U)
#define GPIO_BSC_BC14_Msk                                           (0x1UL << GPIO_BSC_BC14_Pos)                                   /*!< 0x40000000 */
#define GPIO_BSC_BC14                                               GPIO_BSC_BC14_Msk
#define GPIO_BSC_BC15_Pos                                           (31U)
#define GPIO_BSC_BC15_Msk                                           (0x1UL << GPIO_BSC_BC15_Pos)                                   /*!< 0x80000000 */
#define GPIO_BSC_BC15                                               GPIO_BSC_BC15_Msk

/* Legacy defines */
#define GPIO_BSC_BS_0                                               GPIO_BSC_BS0
#define GPIO_BSC_BS_1                                               GPIO_BSC_BS1
#define GPIO_BSC_BS_2                                               GPIO_BSC_BS2
#define GPIO_BSC_BS_3                                               GPIO_BSC_BS3
#define GPIO_BSC_BS_4                                               GPIO_BSC_BS4
#define GPIO_BSC_BS_5                                               GPIO_BSC_BS5
#define GPIO_BSC_BS_6                                               GPIO_BSC_BS6
#define GPIO_BSC_BS_7                                               GPIO_BSC_BS7
#define GPIO_BSC_BS_8                                               GPIO_BSC_BS8
#define GPIO_BSC_BS_9                                               GPIO_BSC_BS9
#define GPIO_BSC_BS_10                                              GPIO_BSC_BS10
#define GPIO_BSC_BS_11                                              GPIO_BSC_BS11
#define GPIO_BSC_BS_12                                              GPIO_BSC_BS12
#define GPIO_BSC_BS_13                                              GPIO_BSC_BS13
#define GPIO_BSC_BS_14                                              GPIO_BSC_BS14
#define GPIO_BSC_BS_15                                              GPIO_BSC_BS15
#define GPIO_BSC_BR_0                                               GPIO_BSC_BC0
#define GPIO_BSC_BR_1                                               GPIO_BSC_BC1
#define GPIO_BSC_BR_2                                               GPIO_BSC_BC2
#define GPIO_BSC_BR_3                                               GPIO_BSC_BC3
#define GPIO_BSC_BR_4                                               GPIO_BSC_BC4
#define GPIO_BSC_BR_5                                               GPIO_BSC_BC5
#define GPIO_BSC_BR_6                                               GPIO_BSC_BC6
#define GPIO_BSC_BR_7                                               GPIO_BSC_BC7
#define GPIO_BSC_BR_8                                               GPIO_BSC_BC8
#define GPIO_BSC_BR_9                                               GPIO_BSC_BC9
#define GPIO_BSC_BR_10                                              GPIO_BSC_BC10
#define GPIO_BSC_BR_11                                              GPIO_BSC_BC11
#define GPIO_BSC_BR_12                                              GPIO_BSC_BC12
#define GPIO_BSC_BR_13                                              GPIO_BSC_BC13
#define GPIO_BSC_BR_14                                              GPIO_BSC_BC14
#define GPIO_BSC_BR_15                                              GPIO_BSC_BC15
#define GPIO_BRR_BR0                                                GPIO_BSC_BC0
#define GPIO_BRR_BR0_Pos                                            GPIO_BSC_BC0_Pos
#define GPIO_BRR_BR0_Msk                                            GPIO_BSC_BC0_Msk
#define GPIO_BRR_BR1                                                GPIO_BSC_BC1
#define GPIO_BRR_BR1_Pos                                            GPIO_BSC_BC1_Pos
#define GPIO_BRR_BR1_Msk                                            GPIO_BSC_BC1_Msk
#define GPIO_BRR_BR2                                                GPIO_BSC_BC2
#define GPIO_BRR_BR2_Pos                                            GPIO_BSC_BC2_Pos
#define GPIO_BRR_BR2_Msk                                            GPIO_BSC_BC2_Msk
#define GPIO_BRR_BR3                                                GPIO_BSC_BC3
#define GPIO_BRR_BR3_Pos                                            GPIO_BSC_BC3_Pos
#define GPIO_BRR_BR3_Msk                                            GPIO_BSC_BC3_Msk
#define GPIO_BRR_BR4                                                GPIO_BSC_BC4
#define GPIO_BRR_BR4_Pos                                            GPIO_BSC_BC4_Pos
#define GPIO_BRR_BR4_Msk                                            GPIO_BSC_BC4_Msk
#define GPIO_BRR_BR5                                                GPIO_BSC_BC5
#define GPIO_BRR_BR5_Pos                                            GPIO_BSC_BC5_Pos
#define GPIO_BRR_BR5_Msk                                            GPIO_BSC_BC5_Msk
#define GPIO_BRR_BR6                                                GPIO_BSC_BC6
#define GPIO_BRR_BR6_Pos                                            GPIO_BSC_BC6_Pos
#define GPIO_BRR_BR6_Msk                                            GPIO_BSC_BC6_Msk
#define GPIO_BRR_BR7                                                GPIO_BSC_BC7
#define GPIO_BRR_BR7_Pos                                            GPIO_BSC_BC7_Pos
#define GPIO_BRR_BR7_Msk                                            GPIO_BSC_BC7_Msk
#define GPIO_BRR_BR8                                                GPIO_BSC_BC8
#define GPIO_BRR_BR8_Pos                                            GPIO_BSC_BC8_Pos
#define GPIO_BRR_BR8_Msk                                            GPIO_BSC_BC8_Msk
#define GPIO_BRR_BR9                                                GPIO_BSC_BC9
#define GPIO_BRR_BR9_Pos                                            GPIO_BSC_BC9_Pos
#define GPIO_BRR_BR9_Msk                                            GPIO_BSC_BC9_Msk
#define GPIO_BRR_BR10                                               GPIO_BSC_BC10
#define GPIO_BRR_BR10_Pos                                           GPIO_BSC_BC10_Pos
#define GPIO_BRR_BR10_Msk                                           GPIO_BSC_BC10_Msk
#define GPIO_BRR_BR11                                               GPIO_BSC_BC11
#define GPIO_BRR_BR11_Pos                                           GPIO_BSC_BC11_Pos
#define GPIO_BRR_BR11_Msk                                           GPIO_BSC_BC11_Msk
#define GPIO_BRR_BR12                                               GPIO_BSC_BC12
#define GPIO_BRR_BR12_Pos                                           GPIO_BSC_BC12_Pos
#define GPIO_BRR_BR12_Msk                                           GPIO_BSC_BC12_Msk
#define GPIO_BRR_BR13                                               GPIO_BSC_BC13
#define GPIO_BRR_BR13_Pos                                           GPIO_BSC_BC13_Pos
#define GPIO_BRR_BR13_Msk                                           GPIO_BSC_BC13_Msk
#define GPIO_BRR_BR14                                               GPIO_BSC_BC14
#define GPIO_BRR_BR14_Pos                                           GPIO_BSC_BC14_Pos
#define GPIO_BRR_BR14_Msk                                           GPIO_BSC_BC14_Msk
#define GPIO_BRR_BR15                                               GPIO_BSC_BC15
#define GPIO_BRR_BR15_Pos                                           GPIO_BSC_BC15_Pos
#define GPIO_BRR_BR15_Msk                                           GPIO_BSC_BC15_Msk
/****************** Bit definition for GPIO_LOCK register *********************/
#define GPIO_LOCK_LOCK0_Pos                                         (0U)
#define GPIO_LOCK_LOCK0_Msk                                         (0x1UL << GPIO_LOCK_LOCK0_Pos)                                 /*!< 0x00000001 */
#define GPIO_LOCK_LOCK0                                             GPIO_LOCK_LOCK0_Msk
#define GPIO_LOCK_LOCK1_Pos                                         (1U)
#define GPIO_LOCK_LOCK1_Msk                                         (0x1UL << GPIO_LOCK_LOCK1_Pos)                                 /*!< 0x00000002 */
#define GPIO_LOCK_LOCK1                                             GPIO_LOCK_LOCK1_Msk
#define GPIO_LOCK_LOCK2_Pos                                         (2U)
#define GPIO_LOCK_LOCK2_Msk                                         (0x1UL << GPIO_LOCK_LOCK2_Pos)                                 /*!< 0x00000004 */
#define GPIO_LOCK_LOCK2                                             GPIO_LOCK_LOCK2_Msk
#define GPIO_LOCK_LOCK3_Pos                                         (3U)
#define GPIO_LOCK_LOCK3_Msk                                         (0x1UL << GPIO_LOCK_LOCK3_Pos)                                 /*!< 0x00000008 */
#define GPIO_LOCK_LOCK3                                             GPIO_LOCK_LOCK3_Msk
#define GPIO_LOCK_LOCK4_Pos                                         (4U)
#define GPIO_LOCK_LOCK4_Msk                                         (0x1UL << GPIO_LOCK_LOCK4_Pos)                                 /*!< 0x00000010 */
#define GPIO_LOCK_LOCK4                                             GPIO_LOCK_LOCK4_Msk
#define GPIO_LOCK_LOCK5_Pos                                         (5U)
#define GPIO_LOCK_LOCK5_Msk                                         (0x1UL << GPIO_LOCK_LOCK5_Pos)                                 /*!< 0x00000020 */
#define GPIO_LOCK_LOCK5                                             GPIO_LOCK_LOCK5_Msk
#define GPIO_LOCK_LOCK6_Pos                                         (6U)
#define GPIO_LOCK_LOCK6_Msk                                         (0x1UL << GPIO_LOCK_LOCK6_Pos)                                 /*!< 0x00000040 */
#define GPIO_LOCK_LOCK6                                             GPIO_LOCK_LOCK6_Msk
#define GPIO_LOCK_LOCK7_Pos                                         (7U)
#define GPIO_LOCK_LOCK7_Msk                                         (0x1UL << GPIO_LOCK_LOCK7_Pos)                                 /*!< 0x00000080 */
#define GPIO_LOCK_LOCK7                                             GPIO_LOCK_LOCK7_Msk
#define GPIO_LOCK_LOCK8_Pos                                         (8U)
#define GPIO_LOCK_LOCK8_Msk                                         (0x1UL << GPIO_LOCK_LOCK8_Pos)                                 /*!< 0x00000100 */
#define GPIO_LOCK_LOCK8                                             GPIO_LOCK_LOCK8_Msk
#define GPIO_LOCK_LOCK9_Pos                                         (9U)
#define GPIO_LOCK_LOCK9_Msk                                         (0x1UL << GPIO_LOCK_LOCK9_Pos)                                 /*!< 0x00000200 */
#define GPIO_LOCK_LOCK9                                             GPIO_LOCK_LOCK9_Msk
#define GPIO_LOCK_LOCK10_Pos                                        (10U)
#define GPIO_LOCK_LOCK10_Msk                                        (0x1UL << GPIO_LOCK_LOCK10_Pos)                                /*!< 0x00000400 */
#define GPIO_LOCK_LOCK10                                            GPIO_LOCK_LOCK10_Msk
#define GPIO_LOCK_LOCK11_Pos                                        (11U)
#define GPIO_LOCK_LOCK11_Msk                                        (0x1UL << GPIO_LOCK_LOCK11_Pos)                                /*!< 0x00000800 */
#define GPIO_LOCK_LOCK11                                            GPIO_LOCK_LOCK11_Msk
#define GPIO_LOCK_LOCK12_Pos                                        (12U)
#define GPIO_LOCK_LOCK12_Msk                                        (0x1UL << GPIO_LOCK_LOCK12_Pos)                                /*!< 0x00001000 */
#define GPIO_LOCK_LOCK12                                            GPIO_LOCK_LOCK12_Msk
#define GPIO_LOCK_LOCK13_Pos                                        (13U)
#define GPIO_LOCK_LOCK13_Msk                                        (0x1UL << GPIO_LOCK_LOCK13_Pos)                                /*!< 0x00002000 */
#define GPIO_LOCK_LOCK13                                            GPIO_LOCK_LOCK13_Msk
#define GPIO_LOCK_LOCK14_Pos                                        (14U)
#define GPIO_LOCK_LOCK14_Msk                                        (0x1UL << GPIO_LOCK_LOCK14_Pos)                                /*!< 0x00004000 */
#define GPIO_LOCK_LOCK14                                            GPIO_LOCK_LOCK14_Msk
#define GPIO_LOCK_LOCK15_Pos                                        (15U)
#define GPIO_LOCK_LOCK15_Msk                                        (0x1UL << GPIO_LOCK_LOCK15_Pos)                                /*!< 0x00008000 */
#define GPIO_LOCK_LOCK15                                            GPIO_LOCK_LOCK15_Msk
#define GPIO_LOCK_LOCKKEY_Pos                                       (16U)
#define GPIO_LOCK_LOCKKEY_Msk                                       (0x1UL << GPIO_LOCK_LOCKKEY_Pos)                               /*!< 0x00010000 */
#define GPIO_LOCK_LOCKKEY                                           GPIO_LOCK_LOCKKEY_Msk
/****************** Bit definition for GPIO_ALFL register *********************/
#define GPIO_ALFL_ALFSEL0_Pos                                       (0U)
#define GPIO_ALFL_ALFSEL0_Msk                                       (0xFUL << GPIO_ALFL_ALFSEL0_Pos)                               /*!< 0x0000000F */
#define GPIO_ALFL_ALFSEL0                                           GPIO_ALFL_ALFSEL0_Msk
#define GPIO_ALFL_ALFSEL0_0                                         (0x1UL << GPIO_ALFL_ALFSEL0_Pos)                               /*!< 0x00000001 */
#define GPIO_ALFL_ALFSEL0_1                                         (0x2UL << GPIO_ALFL_ALFSEL0_Pos)                               /*!< 0x00000002 */
#define GPIO_ALFL_ALFSEL0_2                                         (0x4UL << GPIO_ALFL_ALFSEL0_Pos)                               /*!< 0x00000004 */
#define GPIO_ALFL_ALFSEL0_3                                         (0x8UL << GPIO_ALFL_ALFSEL0_Pos)                               /*!< 0x00000008 */
#define GPIO_ALFL_ALFSEL1_Pos                                       (4U)
#define GPIO_ALFL_ALFSEL1_Msk                                       (0xFUL << GPIO_ALFL_ALFSEL1_Pos)                               /*!< 0x000000F0 */
#define GPIO_ALFL_ALFSEL1                                           GPIO_ALFL_ALFSEL1_Msk
#define GPIO_ALFL_ALFSEL1_0                                         (0x1UL << GPIO_ALFL_ALFSEL1_Pos)                               /*!< 0x00000010 */
#define GPIO_ALFL_ALFSEL1_1                                         (0x2UL << GPIO_ALFL_ALFSEL1_Pos)                               /*!< 0x00000020 */
#define GPIO_ALFL_ALFSEL1_2                                         (0x4UL << GPIO_ALFL_ALFSEL1_Pos)                               /*!< 0x00000040 */
#define GPIO_ALFL_ALFSEL1_3                                         (0x8UL << GPIO_ALFL_ALFSEL1_Pos)                               /*!< 0x00000080 */
#define GPIO_ALFL_ALFSEL2_Pos                                       (8U)
#define GPIO_ALFL_ALFSEL2_Msk                                       (0xFUL << GPIO_ALFL_ALFSEL2_Pos)                               /*!< 0x00000F00 */
#define GPIO_ALFL_ALFSEL2                                           GPIO_ALFL_ALFSEL2_Msk
#define GPIO_ALFL_ALFSEL2_0                                         (0x1UL << GPIO_ALFL_ALFSEL2_Pos)                               /*!< 0x00000100 */
#define GPIO_ALFL_ALFSEL2_1                                         (0x2UL << GPIO_ALFL_ALFSEL2_Pos)                               /*!< 0x00000200 */
#define GPIO_ALFL_ALFSEL2_2                                         (0x4UL << GPIO_ALFL_ALFSEL2_Pos)                               /*!< 0x00000400 */
#define GPIO_ALFL_ALFSEL2_3                                         (0x8UL << GPIO_ALFL_ALFSEL2_Pos)                               /*!< 0x00000800 */
#define GPIO_ALFL_ALFSEL3_Pos                                       (12U)
#define GPIO_ALFL_ALFSEL3_Msk                                       (0xFUL << GPIO_ALFL_ALFSEL3_Pos)                               /*!< 0x0000F000 */
#define GPIO_ALFL_ALFSEL3                                           GPIO_ALFL_ALFSEL3_Msk
#define GPIO_ALFL_ALFSEL3_0                                         (0x1UL << GPIO_ALFL_ALFSEL3_Pos)                               /*!< 0x00001000 */
#define GPIO_ALFL_ALFSEL3_1                                         (0x2UL << GPIO_ALFL_ALFSEL3_Pos)                               /*!< 0x00002000 */
#define GPIO_ALFL_ALFSEL3_2                                         (0x4UL << GPIO_ALFL_ALFSEL3_Pos)                               /*!< 0x00004000 */
#define GPIO_ALFL_ALFSEL3_3                                         (0x8UL << GPIO_ALFL_ALFSEL3_Pos)                               /*!< 0x00008000 */
#define GPIO_ALFL_ALFSEL4_Pos                                       (16U)
#define GPIO_ALFL_ALFSEL4_Msk                                       (0xFUL << GPIO_ALFL_ALFSEL4_Pos)                               /*!< 0x000F0000 */
#define GPIO_ALFL_ALFSEL4                                           GPIO_ALFL_ALFSEL4_Msk
#define GPIO_ALFL_ALFSEL4_0                                         (0x1UL << GPIO_ALFL_ALFSEL4_Pos)                               /*!< 0x00010000 */
#define GPIO_ALFL_ALFSEL4_1                                         (0x2UL << GPIO_ALFL_ALFSEL4_Pos)                               /*!< 0x00020000 */
#define GPIO_ALFL_ALFSEL4_2                                         (0x4UL << GPIO_ALFL_ALFSEL4_Pos)                               /*!< 0x00040000 */
#define GPIO_ALFL_ALFSEL4_3                                         (0x8UL << GPIO_ALFL_ALFSEL4_Pos)                               /*!< 0x00080000 */
#define GPIO_ALFL_ALFSEL5_Pos                                       (20U)
#define GPIO_ALFL_ALFSEL5_Msk                                       (0xFUL << GPIO_ALFL_ALFSEL5_Pos)                               /*!< 0x00F00000 */
#define GPIO_ALFL_ALFSEL5                                           GPIO_ALFL_ALFSEL5_Msk
#define GPIO_ALFL_ALFSEL5_0                                         (0x1UL << GPIO_ALFL_ALFSEL5_Pos)                               /*!< 0x00100000 */
#define GPIO_ALFL_ALFSEL5_1                                         (0x2UL << GPIO_ALFL_ALFSEL5_Pos)                               /*!< 0x00200000 */
#define GPIO_ALFL_ALFSEL5_2                                         (0x4UL << GPIO_ALFL_ALFSEL5_Pos)                               /*!< 0x00400000 */
#define GPIO_ALFL_ALFSEL5_3                                         (0x8UL << GPIO_ALFL_ALFSEL5_Pos)                               /*!< 0x00800000 */
#define GPIO_ALFL_ALFSEL6_Pos                                       (24U)
#define GPIO_ALFL_ALFSEL6_Msk                                       (0xFUL << GPIO_ALFL_ALFSEL6_Pos)                               /*!< 0x0F000000 */
#define GPIO_ALFL_ALFSEL6                                           GPIO_ALFL_ALFSEL6_Msk
#define GPIO_ALFL_ALFSEL6_0                                         (0x1UL << GPIO_ALFL_ALFSEL6_Pos)                               /*!< 0x01000000 */
#define GPIO_ALFL_ALFSEL6_1                                         (0x2UL << GPIO_ALFL_ALFSEL6_Pos)                               /*!< 0x02000000 */
#define GPIO_ALFL_ALFSEL6_2                                         (0x4UL << GPIO_ALFL_ALFSEL6_Pos)                               /*!< 0x04000000 */
#define GPIO_ALFL_ALFSEL6_3                                         (0x8UL << GPIO_ALFL_ALFSEL6_Pos)                               /*!< 0x08000000 */
#define GPIO_ALFL_ALFSEL7_Pos                                       (28U)
#define GPIO_ALFL_ALFSEL7_Msk                                       (0xFUL << GPIO_ALFL_ALFSEL7_Pos)                               /*!< 0xF0000000 */
#define GPIO_ALFL_ALFSEL7                                           GPIO_ALFL_ALFSEL7_Msk
#define GPIO_ALFL_ALFSEL7_0                                         (0x1UL << GPIO_ALFL_ALFSEL7_Pos)                               /*!< 0x10000000 */
#define GPIO_ALFL_ALFSEL7_1                                         (0x2UL << GPIO_ALFL_ALFSEL7_Pos)                               /*!< 0x20000000 */
#define GPIO_ALFL_ALFSEL7_2                                         (0x4UL << GPIO_ALFL_ALFSEL7_Pos)                               /*!< 0x40000000 */
#define GPIO_ALFL_ALFSEL7_3                                         (0x8UL << GPIO_ALFL_ALFSEL7_Pos)                               /*!< 0x80000000 */

/* Legacy defines */
#define GPIO_ALFL_AFRL0                                             GPIO_ALFL_ALFSEL0
#define GPIO_ALFL_AFRL0_0                                           GPIO_ALFL_ALFSEL0_0
#define GPIO_ALFL_AFRL0_1                                           GPIO_ALFL_ALFSEL0_1
#define GPIO_ALFL_AFRL0_2                                           GPIO_ALFL_ALFSEL0_2
#define GPIO_ALFL_AFRL0_3                                           GPIO_ALFL_ALFSEL0_3
#define GPIO_ALFL_AFRL1                                             GPIO_ALFL_ALFSEL1
#define GPIO_ALFL_AFRL1_0                                           GPIO_ALFL_ALFSEL1_0
#define GPIO_ALFL_AFRL1_1                                           GPIO_ALFL_ALFSEL1_1
#define GPIO_ALFL_AFRL1_2                                           GPIO_ALFL_ALFSEL1_2
#define GPIO_ALFL_AFRL1_3                                           GPIO_ALFL_ALFSEL1_3
#define GPIO_ALFL_AFRL2                                             GPIO_ALFL_ALFSEL2
#define GPIO_ALFL_AFRL2_0                                           GPIO_ALFL_ALFSEL2_0
#define GPIO_ALFL_AFRL2_1                                           GPIO_ALFL_ALFSEL2_1
#define GPIO_ALFL_AFRL2_2                                           GPIO_ALFL_ALFSEL2_2
#define GPIO_ALFL_AFRL2_3                                           GPIO_ALFL_ALFSEL2_3
#define GPIO_ALFL_AFRL3                                             GPIO_ALFL_ALFSEL3
#define GPIO_ALFL_AFRL3_0                                           GPIO_ALFL_ALFSEL3_0
#define GPIO_ALFL_AFRL3_1                                           GPIO_ALFL_ALFSEL3_1
#define GPIO_ALFL_AFRL3_2                                           GPIO_ALFL_ALFSEL3_2
#define GPIO_ALFL_AFRL3_3                                           GPIO_ALFL_ALFSEL3_3
#define GPIO_ALFL_AFRL4                                             GPIO_ALFL_ALFSEL4
#define GPIO_ALFL_AFRL4_0                                           GPIO_ALFL_ALFSEL4_0
#define GPIO_ALFL_AFRL4_1                                           GPIO_ALFL_ALFSEL4_1
#define GPIO_ALFL_AFRL4_2                                           GPIO_ALFL_ALFSEL4_2
#define GPIO_ALFL_AFRL4_3                                           GPIO_ALFL_ALFSEL4_3
#define GPIO_ALFL_AFRL5                                             GPIO_ALFL_ALFSEL5
#define GPIO_ALFL_AFRL5_0                                           GPIO_ALFL_ALFSEL5_0
#define GPIO_ALFL_AFRL5_1                                           GPIO_ALFL_ALFSEL5_1
#define GPIO_ALFL_AFRL5_2                                           GPIO_ALFL_ALFSEL5_2
#define GPIO_ALFL_AFRL5_3                                           GPIO_ALFL_ALFSEL5_3
#define GPIO_ALFL_AFRL6                                             GPIO_ALFL_ALFSEL6
#define GPIO_ALFL_AFRL6_0                                           GPIO_ALFL_ALFSEL6_0
#define GPIO_ALFL_AFRL6_1                                           GPIO_ALFL_ALFSEL6_1
#define GPIO_ALFL_AFRL6_2                                           GPIO_ALFL_ALFSEL6_2
#define GPIO_ALFL_AFRL6_3                                           GPIO_ALFL_ALFSEL6_3
#define GPIO_ALFL_AFRL7                                             GPIO_ALFL_ALFSEL7
#define GPIO_ALFL_AFRL7_0                                           GPIO_ALFL_ALFSEL7_0
#define GPIO_ALFL_AFRL7_1                                           GPIO_ALFL_ALFSEL7_1
#define GPIO_ALFL_AFRL7_2                                           GPIO_ALFL_ALFSEL7_2
#define GPIO_ALFL_AFRL7_3                                           GPIO_ALFL_ALFSEL7_3

/****************** Bit definition for GPIO_ALFH register *********************/
#define GPIO_ALFH_ALFSEL8_Pos                                       (0U)
#define GPIO_ALFH_ALFSEL8_Msk                                       (0xFUL << GPIO_ALFH_ALFSEL8_Pos)                               /*!< 0x0000000F */
#define GPIO_ALFH_ALFSEL8                                           GPIO_ALFH_ALFSEL8_Msk
#define GPIO_ALFH_ALFSEL8_0                                         (0x1UL << GPIO_ALFH_ALFSEL8_Pos)                               /*!< 0x00000001 */
#define GPIO_ALFH_ALFSEL8_1                                         (0x2UL << GPIO_ALFH_ALFSEL8_Pos)                               /*!< 0x00000002 */
#define GPIO_ALFH_ALFSEL8_2                                         (0x4UL << GPIO_ALFH_ALFSEL8_Pos)                               /*!< 0x00000004 */
#define GPIO_ALFH_ALFSEL8_3                                         (0x8UL << GPIO_ALFH_ALFSEL8_Pos)                               /*!< 0x00000008 */
#define GPIO_ALFH_ALFSEL9_Pos                                       (4U)
#define GPIO_ALFH_ALFSEL9_Msk                                       (0xFUL << GPIO_ALFH_ALFSEL9_Pos)                               /*!< 0x000000F0 */
#define GPIO_ALFH_ALFSEL9                                           GPIO_ALFH_ALFSEL9_Msk
#define GPIO_ALFH_ALFSEL9_0                                         (0x1UL << GPIO_ALFH_ALFSEL9_Pos)                               /*!< 0x00000010 */
#define GPIO_ALFH_ALFSEL9_1                                         (0x2UL << GPIO_ALFH_ALFSEL9_Pos)                               /*!< 0x00000020 */
#define GPIO_ALFH_ALFSEL9_2                                         (0x4UL << GPIO_ALFH_ALFSEL9_Pos)                               /*!< 0x00000040 */
#define GPIO_ALFH_ALFSEL9_3                                         (0x8UL << GPIO_ALFH_ALFSEL9_Pos)                               /*!< 0x00000080 */
#define GPIO_ALFH_ALFSEL10_Pos                                      (8U)
#define GPIO_ALFH_ALFSEL10_Msk                                      (0xFUL << GPIO_ALFH_ALFSEL10_Pos)                              /*!< 0x00000F00 */
#define GPIO_ALFH_ALFSEL10                                          GPIO_ALFH_ALFSEL10_Msk
#define GPIO_ALFH_ALFSEL10_0                                        (0x1UL << GPIO_ALFH_ALFSEL10_Pos)                              /*!< 0x00000100 */
#define GPIO_ALFH_ALFSEL10_1                                        (0x2UL << GPIO_ALFH_ALFSEL10_Pos)                              /*!< 0x00000200 */
#define GPIO_ALFH_ALFSEL10_2                                        (0x4UL << GPIO_ALFH_ALFSEL10_Pos)                              /*!< 0x00000400 */
#define GPIO_ALFH_ALFSEL10_3                                        (0x8UL << GPIO_ALFH_ALFSEL10_Pos)                              /*!< 0x00000800 */
#define GPIO_ALFH_ALFSEL11_Pos                                      (12U)
#define GPIO_ALFH_ALFSEL11_Msk                                      (0xFUL << GPIO_ALFH_ALFSEL11_Pos)                              /*!< 0x0000F000 */
#define GPIO_ALFH_ALFSEL11                                          GPIO_ALFH_ALFSEL11_Msk
#define GPIO_ALFH_ALFSEL11_0                                        (0x1UL << GPIO_ALFH_ALFSEL11_Pos)                              /*!< 0x00001000 */
#define GPIO_ALFH_ALFSEL11_1                                        (0x2UL << GPIO_ALFH_ALFSEL11_Pos)                              /*!< 0x00002000 */
#define GPIO_ALFH_ALFSEL11_2                                        (0x4UL << GPIO_ALFH_ALFSEL11_Pos)                              /*!< 0x00004000 */
#define GPIO_ALFH_ALFSEL11_3                                        (0x8UL << GPIO_ALFH_ALFSEL11_Pos)                              /*!< 0x00008000 */
#define GPIO_ALFH_ALFSEL12_Pos                                      (16U)
#define GPIO_ALFH_ALFSEL12_Msk                                      (0xFUL << GPIO_ALFH_ALFSEL12_Pos)                              /*!< 0x000F0000 */
#define GPIO_ALFH_ALFSEL12                                          GPIO_ALFH_ALFSEL12_Msk
#define GPIO_ALFH_ALFSEL12_0                                        (0x1UL << GPIO_ALFH_ALFSEL12_Pos)                              /*!< 0x00010000 */
#define GPIO_ALFH_ALFSEL12_1                                        (0x2UL << GPIO_ALFH_ALFSEL12_Pos)                              /*!< 0x00020000 */
#define GPIO_ALFH_ALFSEL12_2                                        (0x4UL << GPIO_ALFH_ALFSEL12_Pos)                              /*!< 0x00040000 */
#define GPIO_ALFH_ALFSEL12_3                                        (0x8UL << GPIO_ALFH_ALFSEL12_Pos)                              /*!< 0x00080000 */
#define GPIO_ALFH_ALFSEL13_Pos                                      (20U)
#define GPIO_ALFH_ALFSEL13_Msk                                      (0xFUL << GPIO_ALFH_ALFSEL13_Pos)                              /*!< 0x00F00000 */
#define GPIO_ALFH_ALFSEL13                                          GPIO_ALFH_ALFSEL13_Msk
#define GPIO_ALFH_ALFSEL13_0                                        (0x1UL << GPIO_ALFH_ALFSEL13_Pos)                              /*!< 0x00100000 */
#define GPIO_ALFH_ALFSEL13_1                                        (0x2UL << GPIO_ALFH_ALFSEL13_Pos)                              /*!< 0x00200000 */
#define GPIO_ALFH_ALFSEL13_2                                        (0x4UL << GPIO_ALFH_ALFSEL13_Pos)                              /*!< 0x00400000 */
#define GPIO_ALFH_ALFSEL13_3                                        (0x8UL << GPIO_ALFH_ALFSEL13_Pos)                              /*!< 0x00800000 */
#define GPIO_ALFH_ALFSEL14_Pos                                      (24U)
#define GPIO_ALFH_ALFSEL14_Msk                                      (0xFUL << GPIO_ALFH_ALFSEL14_Pos)                              /*!< 0x0F000000 */
#define GPIO_ALFH_ALFSEL14                                          GPIO_ALFH_ALFSEL14_Msk
#define GPIO_ALFH_ALFSEL14_0                                        (0x1UL << GPIO_ALFH_ALFSEL14_Pos)                              /*!< 0x01000000 */
#define GPIO_ALFH_ALFSEL14_1                                        (0x2UL << GPIO_ALFH_ALFSEL14_Pos)                              /*!< 0x02000000 */
#define GPIO_ALFH_ALFSEL14_2                                        (0x4UL << GPIO_ALFH_ALFSEL14_Pos)                              /*!< 0x04000000 */
#define GPIO_ALFH_ALFSEL14_3                                        (0x8UL << GPIO_ALFH_ALFSEL14_Pos)                              /*!< 0x08000000 */
#define GPIO_ALFH_ALFSEL15_Pos                                      (28U)
#define GPIO_ALFH_ALFSEL15_Msk                                      (0xFUL << GPIO_ALFH_ALFSEL15_Pos)                              /*!< 0xF0000000 */
#define GPIO_ALFH_ALFSEL15                                          GPIO_ALFH_ALFSEL15_Msk
#define GPIO_ALFH_ALFSEL15_0                                        (0x1UL << GPIO_ALFH_ALFSEL15_Pos)                              /*!< 0x10000000 */
#define GPIO_ALFH_ALFSEL15_1                                        (0x2UL << GPIO_ALFH_ALFSEL15_Pos)                              /*!< 0x20000000 */
#define GPIO_ALFH_ALFSEL15_2                                        (0x4UL << GPIO_ALFH_ALFSEL15_Pos)                              /*!< 0x40000000 */
#define GPIO_ALFH_ALFSEL15_3                                        (0x8UL << GPIO_ALFH_ALFSEL15_Pos)                              /*!< 0x80000000 */

/* Legacy defines */
#define GPIO_ALFH_AFRH0                                             GPIO_ALFH_ALFSEL8
#define GPIO_ALFH_AFRH0_0                                           GPIO_ALFH_ALFSEL8_0
#define GPIO_ALFH_AFRH0_1                                           GPIO_ALFH_ALFSEL8_1
#define GPIO_ALFH_AFRH0_2                                           GPIO_ALFH_ALFSEL8_2
#define GPIO_ALFH_AFRH0_3                                           GPIO_ALFH_ALFSEL8_3
#define GPIO_ALFH_AFRH1                                             GPIO_ALFH_ALFSEL9
#define GPIO_ALFH_AFRH1_0                                           GPIO_ALFH_ALFSEL9_0
#define GPIO_ALFH_AFRH1_1                                           GPIO_ALFH_ALFSEL9_1
#define GPIO_ALFH_AFRH1_2                                           GPIO_ALFH_ALFSEL9_2
#define GPIO_ALFH_AFRH1_3                                           GPIO_ALFH_ALFSEL9_3
#define GPIO_ALFH_AFRH2                                             GPIO_ALFH_ALFSEL10
#define GPIO_ALFH_AFRH2_0                                           GPIO_ALFH_ALFSEL10_0
#define GPIO_ALFH_AFRH2_1                                           GPIO_ALFH_ALFSEL10_1
#define GPIO_ALFH_AFRH2_2                                           GPIO_ALFH_ALFSEL10_2
#define GPIO_ALFH_AFRH2_3                                           GPIO_ALFH_ALFSEL10_3
#define GPIO_ALFH_AFRH3                                             GPIO_ALFH_ALFSEL11
#define GPIO_ALFH_AFRH3_0                                           GPIO_ALFH_ALFSEL11_0
#define GPIO_ALFH_AFRH3_1                                           GPIO_ALFH_ALFSEL11_1
#define GPIO_ALFH_AFRH3_2                                           GPIO_ALFH_ALFSEL11_2
#define GPIO_ALFH_AFRH3_3                                           GPIO_ALFH_ALFSEL11_3
#define GPIO_ALFH_AFRH4                                             GPIO_ALFH_ALFSEL12
#define GPIO_ALFH_AFRH4_0                                           GPIO_ALFH_ALFSEL12_0
#define GPIO_ALFH_AFRH4_1                                           GPIO_ALFH_ALFSEL12_1
#define GPIO_ALFH_AFRH4_2                                           GPIO_ALFH_ALFSEL12_2
#define GPIO_ALFH_AFRH4_3                                           GPIO_ALFH_ALFSEL12_3
#define GPIO_ALFH_AFRH5                                             GPIO_ALFH_ALFSEL13
#define GPIO_ALFH_AFRH5_0                                           GPIO_ALFH_ALFSEL13_0
#define GPIO_ALFH_AFRH5_1                                           GPIO_ALFH_ALFSEL13_1
#define GPIO_ALFH_AFRH5_2                                           GPIO_ALFH_ALFSEL13_2
#define GPIO_ALFH_AFRH5_3                                           GPIO_ALFH_ALFSEL13_3
#define GPIO_ALFH_AFRH6                                             GPIO_ALFH_ALFSEL14
#define GPIO_ALFH_AFRH6_0                                           GPIO_ALFH_ALFSEL14_0
#define GPIO_ALFH_AFRH6_1                                           GPIO_ALFH_ALFSEL14_1
#define GPIO_ALFH_AFRH6_2                                           GPIO_ALFH_ALFSEL14_2
#define GPIO_ALFH_AFRH6_3                                           GPIO_ALFH_ALFSEL14_3
#define GPIO_ALFH_AFRH7                                             GPIO_ALFH_ALFSEL15
#define GPIO_ALFH_AFRH7_0                                           GPIO_ALFH_ALFSEL15_0
#define GPIO_ALFH_AFRH7_1                                           GPIO_ALFH_ALFSEL15_1
#define GPIO_ALFH_AFRH7_2                                           GPIO_ALFH_ALFSEL15_2
#define GPIO_ALFH_AFRH7_3                                           GPIO_ALFH_ALFSEL15_3


/******************************************************************************/
/*                                                                            */
/*                      Inter-integrated Circuit Interface                    */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for I2C_CTRL1 register  ********************/
#define I2C_CTRL1_I2CEN_Pos                                         (0U)
#define I2C_CTRL1_I2CEN_Msk                                         (0x1UL << I2C_CTRL1_I2CEN_Pos)                                 /*!< 0x00000001 */
#define I2C_CTRL1_I2CEN                                             I2C_CTRL1_I2CEN_Msk                                            /*!<Peripheral Enable                             */
#define I2C_CTRL1_SMBEN_Pos                                         (1U)
#define I2C_CTRL1_SMBEN_Msk                                         (0x1UL << I2C_CTRL1_SMBEN_Pos)                                 /*!< 0x00000002 */
#define I2C_CTRL1_SMBEN                                             I2C_CTRL1_SMBEN_Msk                                            /*!<SMBus Mode                                    */
#define I2C_CTRL1_SMBTCFG_Pos                                       (3U)
#define I2C_CTRL1_SMBTCFG_Msk                                       (0x1UL << I2C_CTRL1_SMBTCFG_Pos)                               /*!< 0x00000008 */
#define I2C_CTRL1_SMBTCFG                                           I2C_CTRL1_SMBTCFG_Msk                                          /*!<SMBus Type                                    */
#define I2C_CTRL1_ARPEN_Pos                                         (4U)
#define I2C_CTRL1_ARPEN_Msk                                         (0x1UL << I2C_CTRL1_ARPEN_Pos)                                 /*!< 0x00000010 */
#define I2C_CTRL1_ARPEN                                             I2C_CTRL1_ARPEN_Msk                                            /*!<ARP Enable                                    */
#define I2C_CTRL1_PECEN_Pos                                         (5U)
#define I2C_CTRL1_PECEN_Msk                                         (0x1UL << I2C_CTRL1_PECEN_Pos)                                 /*!< 0x00000020 */
#define I2C_CTRL1_PECEN                                             I2C_CTRL1_PECEN_Msk                                            /*!<PEC Enable                                    */
#define I2C_CTRL1_SRBEN_Pos                                         (6U)
#define I2C_CTRL1_SRBEN_Msk                                         (0x1UL << I2C_CTRL1_SRBEN_Pos)                                 /*!< 0x00000040 */
#define I2C_CTRL1_SRBEN                                             I2C_CTRL1_SRBEN_Msk                                            /*!<General Call Enable                           */
#define I2C_CTRL1_CLKSTRETCHD_Pos                                   (7U)
#define I2C_CTRL1_CLKSTRETCHD_Msk                                   (0x1UL << I2C_CTRL1_CLKSTRETCHD_Pos)                           /*!< 0x00000080 */
#define I2C_CTRL1_CLKSTRETCHD                                       I2C_CTRL1_CLKSTRETCHD_Msk                                      /*!<Clock Stretching Disable (Slave mode)         */
#define I2C_CTRL1_START_Pos                                         (8U)
#define I2C_CTRL1_START_Msk                                         (0x1UL << I2C_CTRL1_START_Pos)                                 /*!< 0x00000100 */
#define I2C_CTRL1_START                                             I2C_CTRL1_START_Msk                                            /*!<Start Generation                              */
#define I2C_CTRL1_STOP_Pos                                          (9U)
#define I2C_CTRL1_STOP_Msk                                          (0x1UL << I2C_CTRL1_STOP_Pos)                                  /*!< 0x00000200 */
#define I2C_CTRL1_STOP                                              I2C_CTRL1_STOP_Msk                                             /*!<Stop Generation                               */
#define I2C_CTRL1_ACKEN_Pos                                         (10U)
#define I2C_CTRL1_ACKEN_Msk                                         (0x1UL << I2C_CTRL1_ACKEN_Pos)                                 /*!< 0x00000400 */
#define I2C_CTRL1_ACKEN                                             I2C_CTRL1_ACKEN_Msk                                            /*!<Acknowledge Enable                            */
#define I2C_CTRL1_ACKPOS_Pos                                        (11U)
#define I2C_CTRL1_ACKPOS_Msk                                        (0x1UL << I2C_CTRL1_ACKPOS_Pos)                                /*!< 0x00000800 */
#define I2C_CTRL1_ACKPOS                                            I2C_CTRL1_ACKPOS_Msk                                           /*!<Acknowledge/PEC Position (for data reception) */
#define I2C_CTRL1_PEC_Pos                                           (12U)
#define I2C_CTRL1_PEC_Msk                                           (0x1UL << I2C_CTRL1_PEC_Pos)                                   /*!< 0x00001000 */
#define I2C_CTRL1_PEC                                               I2C_CTRL1_PEC_Msk                                              /*!<Packet Error Checking                         */
#define I2C_CTRL1_ALERTEN_Pos                                       (13U)
#define I2C_CTRL1_ALERTEN_Msk                                       (0x1UL << I2C_CTRL1_ALERTEN_Pos)                               /*!< 0x00002000 */
#define I2C_CTRL1_ALERTEN                                           I2C_CTRL1_ALERTEN_Msk                                          /*!<SMBus Alert                                   */
#define I2C_CTRL1_SWRST_Pos                                         (15U)
#define I2C_CTRL1_SWRST_Msk                                         (0x1UL << I2C_CTRL1_SWRST_Pos)                                 /*!< 0x00008000 */
#define I2C_CTRL1_SWRST                                             I2C_CTRL1_SWRST_Msk                                            /*!<Software Reset                                */

/*******************  Bit definition for I2C_CTRL2 register  ********************/
#define I2C_CTRL2_CLKFCFG_Pos                                       (0U)
#define I2C_CTRL2_CLKFCFG_Msk                                       (0x3FUL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x0000003F */
#define I2C_CTRL2_CLKFCFG                                           I2C_CTRL2_CLKFCFG_Msk                                          /*!<FREQ[5:0] bits (Peripheral Clock Frequency)   */
#define I2C_CTRL2_CLKFCFG_0                                         (0x01UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000001 */
#define I2C_CTRL2_CLKFCFG_1                                         (0x02UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000002 */
#define I2C_CTRL2_CLKFCFG_2                                         (0x04UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000004 */
#define I2C_CTRL2_CLKFCFG_3                                         (0x08UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000008 */
#define I2C_CTRL2_CLKFCFG_4                                         (0x10UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000010 */
#define I2C_CTRL2_CLKFCFG_5                                         (0x20UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000020 */

#define I2C_CTRL2_ERRIEN_Pos                                        (8U)
#define I2C_CTRL2_ERRIEN_Msk                                        (0x1UL << I2C_CTRL2_ERRIEN_Pos)                                /*!< 0x00000100 */
#define I2C_CTRL2_ERRIEN                                            I2C_CTRL2_ERRIEN_Msk                                           /*!<Error Interrupt Enable  */
#define I2C_CTRL2_EVIEN_Pos                                         (9U)
#define I2C_CTRL2_EVIEN_Msk                                         (0x1UL << I2C_CTRL2_EVIEN_Pos)                                 /*!< 0x00000200 */
#define I2C_CTRL2_EVIEN                                             I2C_CTRL2_EVIEN_Msk                                            /*!<Event Interrupt Enable  */
#define I2C_CTRL2_BUFIEN_Pos                                        (10U)
#define I2C_CTRL2_BUFIEN_Msk                                        (0x1UL << I2C_CTRL2_BUFIEN_Pos)                                /*!< 0x00000400 */
#define I2C_CTRL2_BUFIEN                                            I2C_CTRL2_BUFIEN_Msk                                           /*!<Buffer Interrupt Enable */
#define I2C_CTRL2_DMAEN_Pos                                         (11U)
#define I2C_CTRL2_DMAEN_Msk                                         (0x1UL << I2C_CTRL2_DMAEN_Pos)                                 /*!< 0x00000800 */
#define I2C_CTRL2_DMAEN                                             I2C_CTRL2_DMAEN_Msk                                            /*!<DMA Requests Enable     */
#define I2C_CTRL2_LTCFG_Pos                                         (12U)
#define I2C_CTRL2_LTCFG_Msk                                         (0x1UL << I2C_CTRL2_LTCFG_Pos)                                 /*!< 0x00001000 */
#define I2C_CTRL2_LTCFG                                             I2C_CTRL2_LTCFG_Msk                                            /*!<DMA Last Transfer       */

/*******************  Bit definition for I2C_SADDR1 register  *******************/
#define I2C_SADDR1_ADDR1_7                                          0x000000FEU                                                    /*!<Interface Address */
#define I2C_SADDR1_ADDR8_9                                          0x00000300U                                                    /*!<Interface Address */

#define I2C_SADDR1_ADDR0_Pos                                        (0U)
#define I2C_SADDR1_ADDR0_Msk                                        (0x1UL << I2C_SADDR1_ADDR0_Pos)                                /*!< 0x00000001 */
#define I2C_SADDR1_ADDR0                                            I2C_SADDR1_ADDR0_Msk                                           /*!<Bit 0 */
#define I2C_SADDR1_ADDR1_Pos                                        (1U)
#define I2C_SADDR1_ADDR1_Msk                                        (0x1UL << I2C_SADDR1_ADDR1_Pos)                                /*!< 0x00000002 */
#define I2C_SADDR1_ADDR1                                            I2C_SADDR1_ADDR1_Msk                                           /*!<Bit 1 */
#define I2C_SADDR1_ADDR2_Pos                                        (2U)
#define I2C_SADDR1_ADDR2_Msk                                        (0x1UL << I2C_SADDR1_ADDR2_Pos)                                /*!< 0x00000004 */
#define I2C_SADDR1_ADDR2                                            I2C_SADDR1_ADDR2_Msk                                           /*!<Bit 2 */
#define I2C_SADDR1_ADDR3_Pos                                        (3U)
#define I2C_SADDR1_ADDR3_Msk                                        (0x1UL << I2C_SADDR1_ADDR3_Pos)                                /*!< 0x00000008 */
#define I2C_SADDR1_ADDR3                                            I2C_SADDR1_ADDR3_Msk                                           /*!<Bit 3 */
#define I2C_SADDR1_ADDR4_Pos                                        (4U)
#define I2C_SADDR1_ADDR4_Msk                                        (0x1UL << I2C_SADDR1_ADDR4_Pos)                                /*!< 0x00000010 */
#define I2C_SADDR1_ADDR4                                            I2C_SADDR1_ADDR4_Msk                                           /*!<Bit 4 */
#define I2C_SADDR1_ADDR5_Pos                                        (5U)
#define I2C_SADDR1_ADDR5_Msk                                        (0x1UL << I2C_SADDR1_ADDR5_Pos)                                /*!< 0x00000020 */
#define I2C_SADDR1_ADDR5                                            I2C_SADDR1_ADDR5_Msk                                           /*!<Bit 5 */
#define I2C_SADDR1_ADDR6_Pos                                        (6U)
#define I2C_SADDR1_ADDR6_Msk                                        (0x1UL << I2C_SADDR1_ADDR6_Pos)                                /*!< 0x00000040 */
#define I2C_SADDR1_ADDR6                                            I2C_SADDR1_ADDR6_Msk                                           /*!<Bit 6 */
#define I2C_SADDR1_ADDR7_Pos                                        (7U)
#define I2C_SADDR1_ADDR7_Msk                                        (0x1UL << I2C_SADDR1_ADDR7_Pos)                                /*!< 0x00000080 */
#define I2C_SADDR1_ADDR7                                            I2C_SADDR1_ADDR7_Msk                                           /*!<Bit 7 */
#define I2C_SADDR1_ADDR8_Pos                                        (8U)
#define I2C_SADDR1_ADDR8_Msk                                        (0x1UL << I2C_SADDR1_ADDR8_Pos)                                /*!< 0x00000100 */
#define I2C_SADDR1_ADDR8                                            I2C_SADDR1_ADDR8_Msk                                           /*!<Bit 8 */
#define I2C_SADDR1_ADDR9_Pos                                        (9U)
#define I2C_SADDR1_ADDR9_Msk                                        (0x1UL << I2C_SADDR1_ADDR9_Pos)                                /*!< 0x00000200 */
#define I2C_SADDR1_ADDR9                                            I2C_SADDR1_ADDR9_Msk                                           /*!<Bit 9 */

#define I2C_SADDR1_ADDRLEN_Pos                                      (15U)
#define I2C_SADDR1_ADDRLEN_Msk                                      (0x1UL << I2C_SADDR1_ADDRLEN_Pos)                              /*!< 0x00008000 */
#define I2C_SADDR1_ADDRLEN                                          I2C_SADDR1_ADDRLEN_Msk                                         /*!<Addressing Mode (Slave mode) */

/*******************  Bit definition for I2C_SADDR2 register  *******************/
#define I2C_SADDR2_ADDRNUM_Pos                                      (0U)
#define I2C_SADDR2_ADDRNUM_Msk                                      (0x1UL << I2C_SADDR2_ADDRNUM_Pos)                              /*!< 0x00000001 */
#define I2C_SADDR2_ADDRNUM                                          I2C_SADDR2_ADDRNUM_Msk                                         /*!<Dual addressing mode enable */
#define I2C_SADDR2_ADDR2_Pos                                        (1U)
#define I2C_SADDR2_ADDR2_Msk                                        (0x7FUL << I2C_SADDR2_ADDR2_Pos)                               /*!< 0x000000FE */
#define I2C_SADDR2_ADDR2                                            I2C_SADDR2_ADDR2_Msk                                           /*!<Interface address           */

/********************  Bit definition for I2C_DATA register  ********************/
#define I2C_DATA_DATA_Pos                                           (0U)
#define I2C_DATA_DATA_Msk                                           (0xFFUL << I2C_DATA_DATA_Pos)                                  /*!< 0x000000FF */
#define I2C_DATA_DATA                                               I2C_DATA_DATA_Msk                                              /*!<8-bit Data Register         */

/*******************  Bit definition for I2C_STS1 register  ********************/
#define I2C_STS1_STARTFLG_Pos                                       (0U)
#define I2C_STS1_STARTFLG_Msk                                       (0x1UL << I2C_STS1_STARTFLG_Pos)                               /*!< 0x00000001 */
#define I2C_STS1_STARTFLG                                           I2C_STS1_STARTFLG_Msk                                          /*!<Start Bit (Master mode)                         */
#define I2C_STS1_ADDRFLG_Pos                                        (1U)
#define I2C_STS1_ADDRFLG_Msk                                        (0x1UL << I2C_STS1_ADDRFLG_Pos)                                /*!< 0x00000002 */
#define I2C_STS1_ADDRFLG                                            I2C_STS1_ADDRFLG_Msk                                           /*!<Address sent (master mode)/matched (slave mode) */
#define I2C_STS1_BTCFLG_Pos                                         (2U)
#define I2C_STS1_BTCFLG_Msk                                         (0x1UL << I2C_STS1_BTCFLG_Pos)                                 /*!< 0x00000004 */
#define I2C_STS1_BTCFLG                                             I2C_STS1_BTCFLG_Msk                                            /*!<Byte Transfer Finished                          */
#define I2C_STS1_ADDR10FLG_Pos                                      (3U)
#define I2C_STS1_ADDR10FLG_Msk                                      (0x1UL << I2C_STS1_ADDR10FLG_Pos)                              /*!< 0x00000008 */
#define I2C_STS1_ADDR10FLG                                          I2C_STS1_ADDR10FLG_Msk                                         /*!<10-bit header sent (Master mode)                */
#define I2C_STS1_STOPFLG_Pos                                        (4U)
#define I2C_STS1_STOPFLG_Msk                                        (0x1UL << I2C_STS1_STOPFLG_Pos)                                /*!< 0x00000010 */
#define I2C_STS1_STOPFLG                                            I2C_STS1_STOPFLG_Msk                                           /*!<Stop detection (Slave mode)                     */
#define I2C_STS1_RXBNEFLG_Pos                                       (6U)
#define I2C_STS1_RXBNEFLG_Msk                                       (0x1UL << I2C_STS1_RXBNEFLG_Pos)                               /*!< 0x00000040 */
#define I2C_STS1_RXBNEFLG                                           I2C_STS1_RXBNEFLG_Msk                                          /*!<Data Register not Empty (receivers)             */
#define I2C_STS1_TXBEFLG_Pos                                        (7U)
#define I2C_STS1_TXBEFLG_Msk                                        (0x1UL << I2C_STS1_TXBEFLG_Pos)                                /*!< 0x00000080 */
#define I2C_STS1_TXBEFLG                                            I2C_STS1_TXBEFLG_Msk                                           /*!<Data Register Empty (transmitters)              */
#define I2C_STS1_BERRFLG_Pos                                        (8U)
#define I2C_STS1_BERRFLG_Msk                                        (0x1UL << I2C_STS1_BERRFLG_Pos)                                /*!< 0x00000100 */
#define I2C_STS1_BERRFLG                                            I2C_STS1_BERRFLG_Msk                                           /*!<Bus Error                                       */
#define I2C_STS1_ALFLG_Pos                                          (9U)
#define I2C_STS1_ALFLG_Msk                                          (0x1UL << I2C_STS1_ALFLG_Pos)                                  /*!< 0x00000200 */
#define I2C_STS1_ALFLG                                              I2C_STS1_ALFLG_Msk                                             /*!<Arbitration Lost (master mode)                  */
#define I2C_STS1_AEFLG_Pos                                          (10U)
#define I2C_STS1_AEFLG_Msk                                          (0x1UL << I2C_STS1_AEFLG_Pos)                                  /*!< 0x00000400 */
#define I2C_STS1_AEFLG                                              I2C_STS1_AEFLG_Msk                                             /*!<Acknowledge Failure                             */
#define I2C_STS1_OVRURFLG_Pos                                       (11U)
#define I2C_STS1_OVRURFLG_Msk                                       (0x1UL << I2C_STS1_OVRURFLG_Pos)                               /*!< 0x00000800 */
#define I2C_STS1_OVRURFLG                                           I2C_STS1_OVRURFLG_Msk                                          /*!<Overrun/Underrun                                */
#define I2C_STS1_PECEFLG_Pos                                        (12U)
#define I2C_STS1_PECEFLG_Msk                                        (0x1UL << I2C_STS1_PECEFLG_Pos)                                /*!< 0x00001000 */
#define I2C_STS1_PECEFLG                                            I2C_STS1_PECEFLG_Msk                                           /*!<PEC Error in reception                          */
#define I2C_STS1_TTEFLG_Pos                                         (14U)
#define I2C_STS1_TTEFLG_Msk                                         (0x1UL << I2C_STS1_TTEFLG_Pos)                                 /*!< 0x00004000 */
#define I2C_STS1_TTEFLG                                             I2C_STS1_TTEFLG_Msk                                            /*!<Timeout or Tlow Error                           */
#define I2C_STS1_SMBALTFLG_Pos                                      (15U)
#define I2C_STS1_SMBALTFLG_Msk                                      (0x1UL << I2C_STS1_SMBALTFLG_Pos)                              /*!< 0x00008000 */
#define I2C_STS1_SMBALTFLG                                          I2C_STS1_SMBALTFLG_Msk                                         /*!<SMBus Alert                                     */

/*******************  Bit definition for I2C_STS2 register  ********************/
#define I2C_STS2_MSFLG_Pos                                          (0U)
#define I2C_STS2_MSFLG_Msk                                          (0x1UL << I2C_STS2_MSFLG_Pos)                                  /*!< 0x00000001 */
#define I2C_STS2_MSFLG                                              I2C_STS2_MSFLG_Msk                                             /*!<Master/Slave                                    */
#define I2C_STS2_BUSBSYFLG_Pos                                      (1U)
#define I2C_STS2_BUSBSYFLG_Msk                                      (0x1UL << I2C_STS2_BUSBSYFLG_Pos)                              /*!< 0x00000002 */
#define I2C_STS2_BUSBSYFLG                                          I2C_STS2_BUSBSYFLG_Msk                                         /*!<Bus Busy                                        */
#define I2C_STS2_TRFLG_Pos                                          (2U)
#define I2C_STS2_TRFLG_Msk                                          (0x1UL << I2C_STS2_TRFLG_Pos)                                  /*!< 0x00000004 */
#define I2C_STS2_TRFLG                                              I2C_STS2_TRFLG_Msk                                             /*!<Transmitter/Receiver                            */
#define I2C_STS2_GENCALLFLG_Pos                                     (4U)
#define I2C_STS2_GENCALLFLG_Msk                                     (0x1UL << I2C_STS2_GENCALLFLG_Pos)                             /*!< 0x00000010 */
#define I2C_STS2_GENCALLFLG                                         I2C_STS2_GENCALLFLG_Msk                                        /*!<General Call Address (Slave mode)               */
#define I2C_STS2_SMBDADDRFLG_Pos                                    (5U)
#define I2C_STS2_SMBDADDRFLG_Msk                                    (0x1UL << I2C_STS2_SMBDADDRFLG_Pos)                            /*!< 0x00000020 */
#define I2C_STS2_SMBDADDRFLG                                        I2C_STS2_SMBDADDRFLG_Msk                                       /*!<SMBus Device Default Address (Slave mode)       */
#define I2C_STS2_SMMHADDR_Pos                                       (6U)
#define I2C_STS2_SMMHADDR_Msk                                       (0x1UL << I2C_STS2_SMMHADDR_Pos)                               /*!< 0x00000040 */
#define I2C_STS2_SMMHADDR                                           I2C_STS2_SMMHADDR_Msk                                          /*!<SMBus Host Header (Slave mode)                  */
#define I2C_STS2_DUALADDRFLG_Pos                                    (7U)
#define I2C_STS2_DUALADDRFLG_Msk                                    (0x1UL << I2C_STS2_DUALADDRFLG_Pos)                            /*!< 0x00000080 */
#define I2C_STS2_DUALADDRFLG                                        I2C_STS2_DUALADDRFLG_Msk                                       /*!<Dual Flag (Slave mode)                          */
#define I2C_STS2_PECVALUE_Pos                                       (8U)
#define I2C_STS2_PECVALUE_Msk                                       (0xFFUL << I2C_STS2_PECVALUE_Pos)                              /*!< 0x0000FF00 */
#define I2C_STS2_PECVALUE                                           I2C_STS2_PECVALUE_Msk                                          /*!<Packet Error Checking Register                  */

/*******************  Bit definition for I2C_CLKCTRL register  ********************/
#define I2C_CLKCTRL_CLKS_Pos                                        (0U)
#define I2C_CLKCTRL_CLKS_Msk                                        (0xFFFUL << I2C_CLKCTRL_CLKS_Pos)                              /*!< 0x00000FFF */
#define I2C_CLKCTRL_CLKS                                            I2C_CLKCTRL_CLKS_Msk                                           /*!<Clock Control Register in Fast/Standard mode (Master mode) */
#define I2C_CLKCTRL_FDUTYCFG_Pos                                    (14U)
#define I2C_CLKCTRL_FDUTYCFG_Msk                                    (0x1UL << I2C_CLKCTRL_FDUTYCFG_Pos)                            /*!< 0x00004000 */
#define I2C_CLKCTRL_FDUTYCFG                                        I2C_CLKCTRL_FDUTYCFG_Msk                                       /*!<Fast Mode Duty Cycle                                       */
#define I2C_CLKCTRL_SPEEDCFG_Pos                                    (15U)
#define I2C_CLKCTRL_SPEEDCFG_Msk                                    (0x1UL << I2C_CLKCTRL_SPEEDCFG_Pos)                            /*!< 0x00008000 */
#define I2C_CLKCTRL_SPEEDCFG                                        I2C_CLKCTRL_SPEEDCFG_Msk                                       /*!<I2C Master Mode Selection                                  */

/******************  Bit definition for I2C_RISETMAX register  *******************/
#define I2C_RISETMAX_RISETMAX_Pos                                   (0U)
#define I2C_RISETMAX_RISETMAX_Msk                                   (0x3FUL << I2C_RISETMAX_RISETMAX_Pos)                          /*!< 0x0000003F */
#define I2C_RISETMAX_RISETMAX                                       I2C_RISETMAX_RISETMAX_Msk                                      /*!<Maximum Rise Time in Fast/Standard mode (Master mode) */

/******************  Bit definition for I2C_FILTER register  *******************/
#define I2C_FILTER_DNFCFG_Pos                                       (0U)
#define I2C_FILTER_DNFCFG_Msk                                       (0xFUL << I2C_FILTER_DNFCFG_Pos)                               /*!< 0x0000000F */
#define I2C_FILTER_DNFCFG                                           I2C_FILTER_DNFCFG_Msk                                          /*!<Digital Noise Filter */
#define I2C_FILTER_ANFDIS_Pos                                       (4U)
#define I2C_FILTER_ANFDIS_Msk                                       (0x1UL << I2C_FILTER_ANFDIS_Pos)                               /*!< 0x00000010 */
#define I2C_FILTER_ANFDIS                                           I2C_FILTER_ANFDIS_Msk                                          /*!<Analog Noise Filter */

/******************************************************************************/
/*                                                                            */
/*                           Independent WATCHDOG                             */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for IWDT_KEY register  ********************/
#define IWDT_KEY_KEY_Pos                                            (0U)
#define IWDT_KEY_KEY_Msk                                            (0xFFFFUL << IWDT_KEY_KEY_Pos)                                 /*!< 0x0000FFFF */
#define IWDT_KEY_KEY                                                IWDT_KEY_KEY_Msk                                               /*!<Key value (write only, read 0000h)  */

/*******************  Bit definition for IWDT_PSC register  ********************/
#define IWDT_PSC_PSC_Pos                                            (0U)
#define IWDT_PSC_PSC_Msk                                            (0x7UL << IWDT_PSC_PSC_Pos)                                    /*!< 0x00000007 */
#define IWDT_PSC_PSC                                                IWDT_PSC_PSC_Msk                                               /*!<PR[2:0] (Prescaler divider)         */
#define IWDT_PSC_PSC_0                                              (0x1UL << IWDT_PSC_PSC_Pos)                                    /*!< 0x01 */
#define IWDT_PSC_PSC_1                                              (0x2UL << IWDT_PSC_PSC_Pos)                                    /*!< 0x02 */
#define IWDT_PSC_PSC_2                                              (0x4UL << IWDT_PSC_PSC_Pos)                                    /*!< 0x04 */

/*******************  Bit definition for IWDT_CNTRLD register  *******************/
#define IWDT_CNTRLD_CNTRLD_Pos                                      (0U)
#define IWDT_CNTRLD_CNTRLD_Msk                                      (0xFFFUL << IWDT_CNTRLD_CNTRLD_Pos)                            /*!< 0x00000FFF */
#define IWDT_CNTRLD_CNTRLD                                          IWDT_CNTRLD_CNTRLD_Msk                                         /*!<Watchdog counter reload value        */

/*******************  Bit definition for IWDT_STS register  ********************/
#define IWDT_STS_PSCUFLG_Pos                                        (0U)
#define IWDT_STS_PSCUFLG_Msk                                        (0x1UL << IWDT_STS_PSCUFLG_Pos)                                /*!< 0x00000001 */
#define IWDT_STS_PSCUFLG                                            IWDT_STS_PSCUFLG_Msk                                           /*!<Watchdog prescaler value update      */
#define IWDT_STS_CNTUFLG_Pos                                        (1U)
#define IWDT_STS_CNTUFLG_Msk                                        (0x1UL << IWDT_STS_CNTUFLG_Pos)                                /*!< 0x00000002 */
#define IWDT_STS_CNTUFLG                                            IWDT_STS_CNTUFLG_Msk                                           /*!<Watchdog counter reload value update */



/******************************************************************************/
/*                                                                            */
/*                             Power Control                                  */
/*                                                                            */
/******************************************************************************/
/********************  Bit definition for PMU_CTRL register  ********************/
#define PMU_CTRL_LPDSCFG_Pos                                        (0U)
#define PMU_CTRL_LPDSCFG_Msk                                        (0x1UL << PMU_CTRL_LPDSCFG_Pos)                                /*!< 0x00000001 */
#define PMU_CTRL_LPDSCFG                                            PMU_CTRL_LPDSCFG_Msk                                           /*!< Low-Power Deepsleep                 */
#define PMU_CTRL_PDDSCFG_Pos                                        (1U)
#define PMU_CTRL_PDDSCFG_Msk                                        (0x1UL << PMU_CTRL_PDDSCFG_Pos)                                /*!< 0x00000002 */
#define PMU_CTRL_PDDSCFG                                            PMU_CTRL_PDDSCFG_Msk                                           /*!< Power Down Deepsleep                */
#define PMU_CTRL_WUFLGCLR_Pos                                       (2U)
#define PMU_CTRL_WUFLGCLR_Msk                                       (0x1UL << PMU_CTRL_WUFLGCLR_Pos)                               /*!< 0x00000004 */
#define PMU_CTRL_WUFLGCLR                                           PMU_CTRL_WUFLGCLR_Msk                                          /*!< Clear Wakeup Flag                   */
#define PMU_CTRL_SBFLGCLR_Pos                                       (3U)
#define PMU_CTRL_SBFLGCLR_Msk                                       (0x1UL << PMU_CTRL_SBFLGCLR_Pos)                               /*!< 0x00000008 */
#define PMU_CTRL_SBFLGCLR                                           PMU_CTRL_SBFLGCLR_Msk                                          /*!< Clear Standby Flag                  */
#define PMU_CTRL_PVDEN_Pos                                          (4U)
#define PMU_CTRL_PVDEN_Msk                                          (0x1UL << PMU_CTRL_PVDEN_Pos)                                  /*!< 0x00000010 */
#define PMU_CTRL_PVDEN                                              PMU_CTRL_PVDEN_Msk                                             /*!< Power Voltage Detector Enable       */

#define PMU_CTRL_PLSEL_Pos                                          (5U)
#define PMU_CTRL_PLSEL_Msk                                          (0x7UL << PMU_CTRL_PLSEL_Pos)                                  /*!< 0x000000E0 */
#define PMU_CTRL_PLSEL                                              PMU_CTRL_PLSEL_Msk                                             /*!< PLS[2:0] bits (PVD Level Selection) */
#define PMU_CTRL_PLSEL_0                                            (0x1UL << PMU_CTRL_PLSEL_Pos)                                  /*!< 0x00000020 */
#define PMU_CTRL_PLSEL_1                                            (0x2UL << PMU_CTRL_PLSEL_Pos)                                  /*!< 0x00000040 */
#define PMU_CTRL_PLSEL_2                                            (0x4UL << PMU_CTRL_PLSEL_Pos)                                  /*!< 0x00000080 */

/*!< PVD level configuration */
#define PMU_CTRL_PLSEL_LEV0                                         0x00000000U                                                    /*!< PVD level 0 */
#define PMU_CTRL_PLSEL_LEV1                                         0x00000020U                                                    /*!< PVD level 1 */
#define PMU_CTRL_PLSEL_LEV2                                         0x00000040U                                                    /*!< PVD level 2 */
#define PMU_CTRL_PLSEL_LEV3                                         0x00000060U                                                    /*!< PVD level 3 */
#define PMU_CTRL_PLSEL_LEV4                                         0x00000080U                                                    /*!< PVD level 4 */
#define PMU_CTRL_PLSEL_LEV5                                         0x000000A0U                                                    /*!< PVD level 5 */
#define PMU_CTRL_PLSEL_LEV6                                         0x000000C0U                                                    /*!< PVD level 6 */
#define PMU_CTRL_PLSEL_LEV7                                         0x000000E0U                                                    /*!< PVD level 7 */
#define PMU_CTRL_BPWEN_Pos                                          (8U)
#define PMU_CTRL_BPWEN_Msk                                          (0x1UL << PMU_CTRL_BPWEN_Pos)                                  /*!< 0x00000100 */
#define PMU_CTRL_BPWEN                                              PMU_CTRL_BPWEN_Msk                                             /*!< Disable Backup Domain write protection                     */
#define PMU_CTRL_FPDSM_Pos                                          (9U)
#define PMU_CTRL_FPDSM_Msk                                          (0x1UL << PMU_CTRL_FPDSM_Pos)                                  /*!< 0x00000200 */
#define PMU_CTRL_FPDSM                                              PMU_CTRL_FPDSM_Msk                                             /*!< Flash power down in Stop mode                              */
#define PMU_CTRL_LPRLV_Pos                                          (10U)
#define PMU_CTRL_LPRLV_Msk                                          (0x1UL << PMU_CTRL_LPRLV_Pos)                                  /*!< 0x00000400 */
#define PMU_CTRL_LPRLV                                              PMU_CTRL_LPRLV_Msk                                             /*!< Low Power Regulator Low Voltage                            */
#define PMU_CTRL_MRLV_Pos                                           (11U)
#define PMU_CTRL_MRLV_Msk                                           (0x1UL << PMU_CTRL_MRLV_Pos)                                   /*!< 0x00000800 */
#define PMU_CTRL_MRLV                                               PMU_CTRL_MRLV_Msk                                              /*!< Main Regulator Low Voltage                                 */
#define PMU_CTRL_ADCO1EN_Pos                                        (13U)
#define PMU_CTRL_ADCO1EN_Msk                                        (0x1UL << PMU_CTRL_ADCO1EN_Pos)                                /*!< 0x00002000 */
#define PMU_CTRL_ADCO1EN                                            PMU_CTRL_ADCO1EN_Msk                                           /*!< ADC1 clock enable                                          */
#define PMU_CTRL_VOSSEL_Pos                                         (14U)
#define PMU_CTRL_VOSSEL_Msk                                         (0x3UL << PMU_CTRL_VOSSEL_Pos)                                 /*!< 0x0000C000 */
#define PMU_CTRL_VOSSEL                                             PMU_CTRL_VOSSEL_Msk                                            /*!< VOS bit (Regulator voltage scaling output selection) */
#define PMU_CTRL_VOSSEL_0                                           (0x1UL << PMU_CTRL_VOSSEL_Pos)                                 /*!< 0x00004000 */
#define PMU_CTRL_VOSSEL_1                                           (0x2UL << PMU_CTRL_VOSSEL_Pos)                                 /*!< 0x00008000 */
#define PMU_CTRL_FSMODE_Pos                                         (20U)
#define PMU_CTRL_FSMODE_Msk                                         (0x1UL << PMU_CTRL_FSMODE_Pos)                                 /*!< 0x00100000 */
#define PMU_CTRL_FSMODE                                             PMU_CTRL_FSMODE_Msk                                            /*!< Flash power down in Deepsleep mode                        */
#define PMU_CTRL_FLASHEN_Pos                                        (21U)
#define PMU_CTRL_FLASHEN_Msk                                        (0x1UL << PMU_CTRL_FLASHEN_Pos)                                /*!< 0x00200000 */
#define PMU_CTRL_FLASHEN                                            PMU_CTRL_FLASHEN_Msk                                           /*!< Flash memory interface clock enable                       */

/* Legacy define */
#define PMU_CTRL_PMODE                                              PMU_CTRL_VOSSEL

/*******************  Bit definition for PMU_CSTS register  ********************/
#define PMU_CSTS_WUEFLG_Pos                                         (0U)
#define PMU_CSTS_WUEFLG_Msk                                         (0x1UL << PMU_CSTS_WUEFLG_Pos)                                 /*!< 0x00000001 */
#define PMU_CSTS_WUEFLG                                             PMU_CSTS_WUEFLG_Msk                                            /*!< Wakeup Flag                                      */
#define PMU_CSTS_SBFLG_Pos                                          (1U)
#define PMU_CSTS_SBFLG_Msk                                          (0x1UL << PMU_CSTS_SBFLG_Pos)                                  /*!< 0x00000002 */
#define PMU_CSTS_SBFLG                                              PMU_CSTS_SBFLG_Msk                                             /*!< Standby Flag                                     */
#define PMU_CSTS_PVDOFLG_Pos                                        (2U)
#define PMU_CSTS_PVDOFLG_Msk                                        (0x1UL << PMU_CSTS_PVDOFLG_Pos)                                /*!< 0x00000004 */
#define PMU_CSTS_PVDOFLG                                            PMU_CSTS_PVDOFLG_Msk                                           /*!< PVD Output                                       */
#define PMU_CSTS_BKPRFLG_Pos                                        (3U)
#define PMU_CSTS_BKPRFLG_Msk                                        (0x1UL << PMU_CSTS_BKPRFLG_Pos)                                /*!< 0x00000008 */
#define PMU_CSTS_BKPRFLG                                            PMU_CSTS_BKPRFLG_Msk                                           /*!< Backup regulator ready                           */
#define PMU_CSTS_WKUPCFG_Pos                                        (8U)
#define PMU_CSTS_WKUPCFG_Msk                                        (0x1UL << PMU_CSTS_WKUPCFG_Pos)                                /*!< 0x00000100 */
#define PMU_CSTS_WKUPCFG                                            PMU_CSTS_WKUPCFG_Msk                                           /*!< Enable WKUP pin                                  */
#define PMU_CSTS_BKPREN_Pos                                         (9U)
#define PMU_CSTS_BKPREN_Msk                                         (0x1UL << PMU_CSTS_BKPREN_Pos)                                 /*!< 0x00000200 */
#define PMU_CSTS_BKPREN                                             PMU_CSTS_BKPREN_Msk                                            /*!< Backup regulator enable                          */
#define PMU_CSTS_VOSRFLG_Pos                                        (14U)
#define PMU_CSTS_VOSRFLG_Msk                                        (0x1UL << PMU_CSTS_VOSRFLG_Pos)                                /*!< 0x00004000 */
#define PMU_CSTS_VOSRFLG                                            PMU_CSTS_VOSRFLG_Msk                                           /*!< Regulator voltage scaling output selection ready */

/* Legacy define */
#define PMU_CSTS_REGRDY                                             PMU_CSTS_VOSRFLG

/******************************************************************************/
/*                                                                            */
/*                         Reset and Clock Control                            */
/*                                                                            */
/******************************************************************************/
/********************  Bit definition for RCM_CTRL register  ********************/
#define RCM_CTRL_HSIEN_Pos                                          (0U)
#define RCM_CTRL_HSIEN_Msk                                          (0x1UL << RCM_CTRL_HSIEN_Pos)                                  /*!< 0x00000001 */
#define RCM_CTRL_HSIEN                                              RCM_CTRL_HSIEN_Msk
#define RCM_CTRL_HSIRDYFLG_Pos                                      (1U)
#define RCM_CTRL_HSIRDYFLG_Msk                                      (0x1UL << RCM_CTRL_HSIRDYFLG_Pos)                              /*!< 0x00000002 */
#define RCM_CTRL_HS