/**
 *
 * @file        apm32f402xx.h
 *
 * @brief       CMSIS APM32F402xx Device Peripheral Access Layer Header File.
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * The original code has been modified by Geehy Semiconductor.
 *
 * Copyright (c) 2017 STMicroelectronics.
 * Copyright (C) 2024-2025 Geehy Semiconductor.
 * All rights reserved.
 *
 * This software is licensed under terms that can be found in the LICENSE file
 * in the root directory of this software component.
 * If no LICENSE file comes with this software, it is provided AS-IS.
 *
 */

/** @addtogroup CMSIS_Device
  * @{
  */

/** @addtogroup apm32f402xx
  * @{
  */

#ifndef __APM32F402xx_H
#define __APM32F402xx_H

#ifdef __cplusplus
 extern "C" {
#endif /* __cplusplus */

/** @addtogroup Configuration_section_for_CMSIS
  * @{
  */

/**
  * @brief Configuration of the Cortex-M4 Processor and Core Peripherals
  */
#define __CM4_REV                 0x0001U  /*!< Core revision r0p1                            */
#define __MPU_PRESENT             1U       /*!< APM32F4XX provides an MPU                     */
#define __NVIC_PRIO_BITS          4U       /*!< APM32F4XX uses 4 Bits for the Priority Levels */
#define __Vendor_SysTickConfig    0U       /*!< Set to 1 if different SysTick Config is used  */
#define __FPU_PRESENT             1U       /*!< FPU present                                   */

/**
  * @}
  */

/** @addtogroup Peripheral_interrupt_number_definition
  * @{
  */

/**
 * @brief APM32F4XX Interrupt Number Definition, according to the selected device
 *        in @ref Library_configuration_section
 */
typedef enum
{
/******  Cortex-M4 Processor Exceptions Numbers ****************************************************************/
  NonMaskableInt_IRQn         = -14,    /*!< 2 Non Maskable Interrupt                                          */
  MemoryManagement_IRQn       = -12,    /*!< 4 Cortex-M4 Memory Management Interrupt                           */
  BusFault_IRQn               = -11,    /*!< 5 Cortex-M4 Bus Fault Interrupt                                   */
  UsageFault_IRQn             = -10,    /*!< 6 Cortex-M4 Usage Fault Interrupt                                 */
  SVCall_IRQn                 = -5,     /*!< 11 Cortex-M4 SV Call Interrupt                                    */
  DebugMonitor_IRQn           = -4,     /*!< 12 Cortex-M4 Debug Monitor Interrupt                              */
  PendSV_IRQn                 = -2,     /*!< 14 Cortex-M4 Pend SV Interrupt                                    */
  SysTick_IRQn                = -1,     /*!< 15 Cortex-M4 System Tick Interrupt                                */
/******  APM32 specific Interrupt Numbers **********************************************************************/
  WWDT_IRQn                   = 0,      /*!< Window WatchDog Interrupt                                         */
  PVD_IRQn                    = 1,      /*!< PVD through EINT Line detection Interrupt                         */
  TAMPER_IRQn                 = 2,      /*!< Tamper Interrupts                                                 */
  RTC_IRQn                    = 3,      /*!< RTC global Interrupt                                              */
  FLASH_IRQn                  = 4,      /*!< FLASH global Interrupt                                            */
  RCM_IRQn                    = 5,      /*!< RCM global Interrupt                                              */
  EINT0_IRQn                  = 6,      /*!< EINT Line0 Interrupt                                              */
  EINT1_IRQn                  = 7,      /*!< EINT Line1 Interrupt                                              */
  EINT2_IRQn                  = 8,      /*!< EINT Line2 Interrupt                                              */
  EINT3_IRQn                  = 9,      /*!< EINT Line3 Interrupt                                              */
  EINT4_IRQn                  = 10,     /*!< EINT Line4 Interrupt                                              */
  DMA1_Channel1_IRQn          = 11,     /*!< DMA1 Channel 1 global Interrupt                                   */
  DMA1_Channel2_IRQn          = 12,     /*!< DMA1 Channel 2 global Interrupt                                   */
  DMA1_Channel3_IRQn          = 13,     /*!< DMA1 Channel 3 global Interrupt                                   */
  DMA1_Channel4_IRQn          = 14,     /*!< DMA1 Channel 4 global Interrupt                                   */
  DMA1_Channel5_IRQn          = 15,     /*!< DMA1 Channel 5 global Interrupt                                   */
  DMA1_Channel6_IRQn          = 16,     /*!< DMA1 Channel 6 global Interrupt                                   */
  DMA1_Channel7_IRQn          = 17,     /*!< DMA1 Channel 7 global Interrupt                                   */
  ADC1_2_IRQn                 = 18,     /*!< ADC1 and ADC2 global Interrupt                                    */
  CAN1_TX_IRQn                = 19,     /*!< CAN1 TX Interrupt                                                 */
  CAN1_RX0_IRQn               = 20,     /*!< CAN1 RX0 Interrupt                                                */
  CAN1_RX1_IRQn               = 21,     /*!< CAN1 RX1 Interrupt                                                */
  CAN1_SCE_IRQn               = 22,     /*!< CAN1 SCE Interrupt                                                */
  EINT9_5_IRQn                = 23,     /*!< External Line[9:5] Interrupts                                     */
  TMR1_BRK_IRQn               = 24,     /*!< TMR1 Break Interrupt                                              */
  TMR1_UP_IRQn                = 25,     /*!< TMR1 Update Interrupt                                             */
  TMR1_TRG_COM_IRQn           = 26,     /*!< TMR1 Trigger and Commutation Interrupt                            */
  TMR1_CC_IRQn                = 27,     /*!< TMR1 Capture Compare Interrupt                                    */
  TMR2_IRQn                   = 28,     /*!< TMR2 global Interrupt                                             */
  TMR3_IRQn                   = 29,     /*!< TMR3 global Interrupt                                             */
  TMR4_IRQn                   = 30,     /*!< TMR4 global Interrupt                                             */
  I2C1_EV_IRQn                = 31,     /*!< I2C1 Event Interrupt                                              */
  I2C1_ER_IRQn                = 32,     /*!< I2C1 Error Interrupt                                              */
  SPI1_IRQn                   = 35,     /*!< SPI1 global Interrupt                                             */
  SPI2_IRQn                   = 36,     /*!< SPI2 global Interrupt                                             */
  USART1_IRQn                 = 37,     /*!< USART1 global Interrupt                                           */
  USART2_IRQn                 = 38,     /*!< USART2 global Interrupt                                           */
  USART3_IRQn                 = 39,     /*!< USART3 global Interrupt                                           */
  EINT15_10_IRQn              = 40,     /*!< External Line[15:10] Interrupts                                   */
  RTC_Alarm_IRQn              = 41,     /*!< RTC Alarm through EINT line interrupt                             */
  OTG_FS_WKUP_IRQn            = 42,     /*!< USB OTG FS Wakeup through EINT line interrupt                     */
  TMR8_BRK_IRQn               = 43,     /*!< TMR8 Break Interrupt                                              */
  TMR8_UP_IRQn                = 44,     /*!< TMR8 Update Interrupt                                             */
  TMR8_TRG_COM_IRQn           = 45,     /*!< TMR8 Trigger and Commutation Interrupt                            */
  TMR8_CC_IRQn                = 46,     /*!< TMR8 Capture Compare global interrupt                             */
  TMR5_IRQn                   = 50,     /*!< TMR5 global Interrupt                                             */
  UART4_IRQn                  = 52,     /*!< UART4 global Interrupt                                            */
  DMA2_Channel1_IRQn          = 56,     /*!< DMA2 Channel 1 global Interrupt                                   */
  DMA2_Channel2_IRQn          = 57,     /*!< DMA2 Channel 2 global Interrupt                                   */
  DMA2_Channel3_IRQn          = 58,     /*!< DMA2 Channel 3 global Interrupt                                   */
  DMA2_Channel4_5_IRQn        = 59,     /*!< DMA2 Channel 4 and Channel 5 global Interrupt                     */
  FPU_IRQn                    = 60,     /*!< FPU global interrupt                                              */
  CAN2_TX_IRQn                = 61,     /*!< CAN2 TX Interrupt                                                 */
  CAN2_RX0_IRQn               = 62,     /*!< CAN2 RX0 Interrupt                                                */
  CAN2_RX1_IRQn               = 63,     /*!< CAN2 RX1 Interrupt                                                */
  CAN2_SCE_IRQn               = 64,     /*!< CAN2 SCE Interrupt                                                */
  OTG_FS_IRQn                 = 65,     /*!< USB OTG FS global Interrupt                                       */
} IRQn_Type;

/**
  * @}
  */

#include "core_cm4.h"             /* Cortex-M4 processor and core peripherals */
#include "system_apm32f4xx.h"
#include <stdint.h>

/** @addtogroup Peripheral_registers_structures
  * @{
  */

/**
  * @brief Analog to Digital Converter
  */

typedef struct
{
  __IO uint32_t STS;      /*!< ADC status register,                         Address offset: 0x00 */
  __IO uint32_t CTRL1;    /*!< ADC control register 1,                      Address offset: 0x04 */
  __IO uint32_t CTRL2;    /*!< ADC control register 2,                      Address offset: 0x08 */
  __IO uint32_t SMPTIM1;  /*!< ADC sample time register 1,                  Address offset: 0x0C */
  __IO uint32_t SMPTIM2;  /*!< ADC sample time register 2,                  Address offset: 0x10 */
  __IO uint32_t INJDOF1;  /*!< ADC injected channel data offset register 1, Address offset: 0x14 */
  __IO uint32_t INJDOF2;  /*!< ADC injected channel data offset register 2, Address offset: 0x18 */
  __IO uint32_t INJDOF3;  /*!< ADC injected channel data offset register 3, Address offset: 0x1C */
  __IO uint32_t INJDOF4;  /*!< ADC injected channel data offset register 4, Address offset: 0x20 */
  __IO uint32_t AWDHT;    /*!< ADC watchdog higher threshold register,      Address offset: 0x24 */
  __IO uint32_t AWDLT;    /*!< ADC watchdog lower threshold register,       Address offset: 0x28 */
  __IO uint32_t REGSEQ1;  /*!< ADC regular sequence register 1,             Address offset: 0x2C */
  __IO uint32_t REGSEQ2;  /*!< ADC regular sequence register 2,             Address offset: 0x30 */
  __IO uint32_t REGSEQ3;  /*!< ADC regular sequence register 3,             Address offset: 0x34 */
  __IO uint32_t INJSEQ;   /*!< ADC injected sequence register,              Address offset: 0x38 */
  __IO uint32_t INJDATA1; /*!< ADC injected data register 1,                Address offset: 0x3C */
  __IO uint32_t INJDATA2; /*!< ADC injected data register 2,                Address offset: 0x40 */
  __IO uint32_t INJDATA3; /*!< ADC injected data register 3,                Address offset: 0x44 */
  __IO uint32_t INJDATA4; /*!< ADC injected data register 4,                Address offset: 0x48 */
  __IO uint32_t REGDATA;  /*!< ADC regular data register,                   Address offset: 0x4C */
} ADC_TypeDef;

typedef struct
{
  __IO uint32_t STS;      /*!< ADC status register,                         Address offset: 0x00, used for ADC multimode */
  __IO uint32_t CTRL1;    /*!< ADC control register 1,                      Address offset: 0x04, used for ADC multimode */
  __IO uint32_t CTRL2;    /*!< ADC control register 2,                      Address offset: 0x08, used for ADC multimode */
  uint32_t RESERVED[16];  /*!< Reserved, 0x0C-0x48 */
  __IO uint32_t REGDATA;  /*!< ADC regular data register,                   Address offset: 0x4C, used for ADC multimode */
} ADC_Common_TypeDef;

/**
  * @brief B
  */

typedef struct
{
  uint32_t RESERVED0;     /*!< Reserved, 0x00 */
  __IO uint32_t DATA1;    /*!< BKP data register 1, Address offset: 0x04 */
  __IO uint32_t DATA2;    /*!< BKP data register 2, Address offset: 0x08 */
  __IO uint32_t DATA3;    /*!< BKP data register 3, Address offset: 0x0C */
  __IO uint32_t DATA4;    /*!< BKP data register 4, Address offset: 0x10 */
  __IO uint32_t DATA5;    /*!< BKP data register 5, Address offset: 0x14 */
  __IO uint32_t DATA6;    /*!< BKP data register 6, Address offset: 0x18 */
  __IO uint32_t DATA7;    /*!< BKP data register 7, Address offset: 0x1C */
  __IO uint32_t DATA8;    /*!< BKP data register 8, Address offset: 0x20 */
  __IO uint32_t DATA9;    /*!< BKP data register 9, Address offset: 0x24 */
  __IO uint32_t DATA10;   /*!< BKP data register 10, Address offset: 0x28 */
  __IO uint32_t CLKCAL;   /*!< BKP clock calibration register, Address offset: 0x2C */
  __IO uint32_t CTRL;     /*!< BKP control register, Address offset: 0x30 */
  __IO uint32_t CSTS;     /*!< BKP control/status register, Address offset: 0x34 */
  uint32_t RESERVED1[2];  /*!< Reserved, 0x38-0x3C */
  __IO uint32_t DATA11;   /*!< BKP data register 11, Address offset: 0x40 */
  __IO uint32_t DATA12;   /*!< BKP data register 12, Address offset: 0x44 */
  __IO uint32_t DATA13;   /*!< BKP data register 13, Address offset: 0x48 */
  __IO uint32_t DATA14;   /*!< BKP data register 14, Address offset: 0x4C */
  __IO uint32_t DATA15;   /*!< BKP data register 15, Address offset: 0x50 */
  __IO uint32_t DATA16;   /*!< BKP data register 16, Address offset: 0x54 */
  __IO uint32_t DATA17;   /*!< BKP data register 17, Address offset: 0x58 */
  __IO uint32_t DATA18;   /*!< BKP data register 18, Address offset: 0x5C */
  __IO uint32_t DATA19;   /*!< BKP data register 19, Address offset: 0x60 */
  __IO uint32_t DATA20;   /*!< BKP data register 20, Address offset: 0x64 */
  __IO uint32_t DATA21;   /*!< BKP data register 21, Address offset: 0x68 */
  __IO uint32_t DATA22;   /*!< BKP data register 22, Address offset: 0x6C */
  __IO uint32_t DATA23;   /*!< BKP data register 23, Address offset: 0x70 */
  __IO uint32_t DATA24;   /*!< BKP data register 24, Address offset: 0x74 */
  __IO uint32_t DATA25;   /*!< BKP data register 25, Address offset: 0x78 */
  __IO uint32_t DATA26;   /*!< BKP data register 26, Address offset: 0x7C */
  __IO uint32_t DATA27;   /*!< BKP data register 27, Address offset: 0x80 */
  __IO uint32_t DATA28;   /*!< BKP data register 28, Address offset: 0x84 */
  __IO uint32_t DATA29;   /*!< BKP data register 29, Address offset: 0x88 */
  __IO uint32_t DATA30;   /*!< BKP data register 30, Address offset: 0x8C */
  __IO uint32_t DATA31;   /*!< BKP data register 31, Address offset: 0x90 */
  __IO uint32_t DATA32;   /*!< BKP data register 32, Address offset: 0x94 */
  __IO uint32_t DATA33;   /*!< BKP data register 33, Address offset: 0x98 */
  __IO uint32_t DATA34;   /*!< BKP data register 34, Address offset: 0x9C */
  __IO uint32_t DATA35;   /*!< BKP data register 35, Address offset: 0xA0 */
  __IO uint32_t DATA36;   /*!< BKP data register 36, Address offset: 0xA4 */
  __IO uint32_t DATA37;   /*!< BKP data register 37, Address offset: 0xA8 */
  __IO uint32_t DATA38;   /*!< BKP data register 38, Address offset: 0xAC */
  __IO uint32_t DATA39;   /*!< BKP data register 39, Address offset: 0xB0 */
  __IO uint32_t DATA40;   /*!< BKP data register 40, Address offset: 0xB4 */
  __IO uint32_t DATA41;   /*!< BKP data register 41, Address offset: 0xB8 */
  __IO uint32_t DATA42;   /*!< BKP data register 42, Address offset: 0xBC */
} BAKPR_TypeDef;

/**
  * @brief Controller Area Network TxMailBox
  */

typedef struct
{
  __IO uint32_t TXMID;  /*!< CAN TX mailbox identifier register */
  __IO uint32_t TXDLEN; /*!< CAN mailbox data length control and time stamp register */
  __IO uint32_t TXMDL;  /*!< CAN mailbox data low register */
  __IO uint32_t TXMDH;  /*!< CAN mailbox data high register */
} CAN_TxMailBox_TypeDef;

/**
  * @brief Controller Area Network FIFOMailBox
  */

typedef struct
{
  __IO uint32_t RXMID;  /*!< CAN receive FIFO mailbox identifier register */
  __IO uint32_t RXDLEN; /*!< CAN receive FIFO mailbox data length control and time stamp register */
  __IO uint32_t RXMDL;  /*!< CAN receive FIFO mailbox data low register */
  __IO uint32_t RXMDH;  /*!< CAN receive FIFO mailbox data high register */
} CAN_FIFOMailBox_TypeDef;

/**
  * @brief Controller Area Network FilterRegister
  */

typedef struct
{
  __IO uint32_t FBANK1; /*!< CAN Filter bank register 1 */
  __IO uint32_t FBANK2; /*!< CAN Filter bank register 1 */
} CAN_FilterRegister_TypeDef;

/**
  * @brief Controller Area Network
  */

typedef struct
{
  __IO uint32_t              MCTRL;               /*!< CAN master control register,         Address offset: 0x00          */
  __IO uint32_t              MSTS;                /*!< CAN master status register,          Address offset: 0x04          */
  __IO uint32_t              TXSTS;               /*!< CAN transmit status register,        Address offset: 0x08          */
  __IO uint32_t              RXF0;                /*!< CAN receive FIFO 0 register,         Address offset: 0x0C          */
  __IO uint32_t              RXF1;                /*!< CAN receive FIFO 1 register,         Address offset: 0x10          */
  __IO uint32_t              INTEN;               /*!< CAN interrupt enable register,       Address offset: 0x14          */
  __IO uint32_t              ERRSTS;              /*!< CAN error status register,           Address offset: 0x18          */
  __IO uint32_t              BITTIM;              /*!< CAN bit timing register,             Address offset: 0x1C          */
  uint32_t                   RESERVED0[88];       /*!< Reserved, 0x020 - 0x17F                                            */
  CAN_TxMailBox_TypeDef      sTxMailBox[3];       /*!< CAN Tx MailBox,                      Address offset: 0x180 - 0x1AC */
  CAN_FIFOMailBox_TypeDef    sFIFOMailBox[2];     /*!< CAN FIFO MailBox,                    Address offset: 0x1B0 - 0x1CC */
  uint32_t                   RESERVED1[12];       /*!< Reserved, 0x1D0 - 0x1FF                                            */
  __IO uint32_t              FCTRL;               /*!< CAN filter master register,          Address offset: 0x200         */
  __IO uint32_t              FMCFG;               /*!< CAN filter mode register,            Address offset: 0x204         */
  uint32_t                   RESERVED2;           /*!< Reserved, 0x208                                                    */
  __IO uint32_t              FSCFG;               /*!< CAN filter scale register,           Address offset: 0x20C         */
  uint32_t                   RESERVED3;           /*!< Reserved, 0x210                                                    */
  __IO uint32_t              FFASS;               /*!< CAN filter FIFO assignment register, Address offset: 0x214         */
  uint32_t                   RESERVED4;           /*!< Reserved, 0x218                                                    */
  __IO uint32_t              FACT;                /*!< CAN filter activation register,      Address offset: 0x21C         */
  uint32_t                   RESERVED5[8];        /*!< Reserved, 0x220-0x23F                                              */
  CAN_FilterRegister_TypeDef sFilterRegister[28]; /*!< CAN Filter Register,                 Address offset: 0x240-0x31C   */
} CAN_TypeDef;

/**
  * @brief CRC calculation unit
  */

typedef struct
{
  __IO uint32_t DATA;       /*!< CRC Data register,             Address offset: 0x00 */
  __IO uint8_t  INDATA;     /*!< CRC Independent data register, Address offset: 0x04 */
  uint8_t       RESERVED0;  /*!< Reserved, 0x05                                      */
  uint16_t      RESERVED1;  /*!< Reserved, 0x06                                      */
  __IO uint32_t CTRL;       /*!< CRC Control register,          Address offset: 0x08 */
} CRC_TypeDef;

/**
  * @brief Debug MCU
  */

typedef struct
{
  __IO uint32_t IDCODE;   /*!< MCU device ID code,               Address offset: 0x00 */
  __IO uint32_t CFG;      /*!< Debug MCU configuration register, Address offset: 0x04 */
} DBGMCU_TypeDef;


/**
  * @brief DMA Controller
  */

typedef struct
{
  __IO uint32_t CHCFG;    /*!< DMA channel x configuration register */
  __IO uint32_t CHNDATA;  /*!< DMA channel x number of data register */
  __IO uint32_t CHPADDR;  /*!< DMA channel x peripheral address register */
  __IO uint32_t CHMADDR;  /*!< DMA channel x memory address register */
} DMA_Channel_TypeDef;

typedef struct
{
  __IO uint32_t INTSTS;   /*!< DMA interrupt status register, Address offset: 0x00 */
  __IO uint32_t INTFCLR;  /*!< DMA interrupt flag clear register, Address offset: 0x04 */
} DMA_TypeDef;

/**
  * @brief External Interrupt/Event Controller
  */

typedef struct
{
  __IO uint32_t IMASK;    /*!< EINT Interrupt mask register,            Address offset: 0x00 */
  __IO uint32_t EMASK;    /*!< EINT Event mask register,                Address offset: 0x04 */
  __IO uint32_t RTEN;     /*!< EINT Rising trigger selection register,  Address offset: 0x08 */
  __IO uint32_t FTEN;     /*!< EINT Falling trigger selection register, Address offset: 0x0C */
  __IO uint32_t SWINTE;   /*!< EINT Software interrupt event register,  Address offset: 0x10 */
  __IO uint32_t IPEND;    /*!< EINT Pending register,                   Address offset: 0x14 */
} EINT_TypeDef;

/**
  * @brief FLASH Registers
  */

typedef struct
{
  __IO uint32_t CTRL1;    /*!< FLASH access control register 1,     Address offset: 0x00 */
  __IO uint32_t KEY;      /*!< FLASH key register,                  Address offset: 0x04 */
  __IO uint32_t OPTKEY;   /*!< FLASH option key register,           Address offset: 0x08 */
  __IO uint32_t STS;      /*!< FLASH status register,               Address offset: 0x0C */
  __IO uint32_t CTRL2;    /*!< FLASH control register 2,            Address offset: 0x10 */
  __IO uint32_t ADDR;     /*!< FLASH address register,              Address offset: 0x14 */
  uint32_t      RESERVED; /*!< Reserved,                            Address offset: 0x18 */
  __IO uint32_t OBCS;     /*!< FLASH option byte control register,  Address offset: 0x1C */
  __IO uint32_t WRTPROT;  /*!< FLASH write protection register,     Address offset: 0x20 */
} FLASH_TypeDef;

/**
  * @brief Option Bytes Registers
  */

typedef struct
{
  __IO uint16_t READPROT;   /*!< FLASH option byte read protection register */
  __IO uint16_t UOB;        /*!< FLASH user option byte register */
  __IO uint16_t Data0;      /*!< FLASH option byte data 0 register */
  __IO uint16_t Data1;      /*!< FLASH option byte data 1 register */
  __IO uint16_t WRP0;       /*!< FLASH option byte write protection 0 register */
  __IO uint16_t WRP1;       /*!< FLASH option byte write protection 1 register */
  __IO uint16_t WRP2;       /*!< FLASH option byte write protection 2 register */
  __IO uint16_t WRP3;       /*!< FLASH option byte write protection 3 register */
} OB_TypeDef;

/**
  * @brief General Purpose I/O
  */

typedef struct
{
  __IO uint32_t CFGLOW;   /*!< GPIO port configuration low register,  Address offset: 0x00 */
  __IO uint32_t CFGHIG;   /*!< GPIO port configuration high register, Address offset: 0x04 */
  __IO uint32_t IDATA;    /*!< GPIO port input data register,         Address offset: 0x08 */
  __IO uint32_t ODATA;    /*!< GPIO port output data register,        Address offset: 0x0C */
  __IO uint32_t BSC;      /*!< GPIO port bit set/clear register,      Address offset: 0x10 */
  __IO uint32_t BC;       /*!< GPIO port bit clear register,          Address offset: 0x14 */
  __IO uint32_t LOCK;     /*!< GPIO port configuration lock register, Address offset: 0x18 */
} GPIO_TypeDef;

/**
  * @brief Alternate Function I/O
  */

typedef struct
{
  __IO uint32_t EVCTRL;     /*!< AFIO Event control register, Address offset: 0x00 */
  __IO uint32_t REMAP1;     /*!< AFIO remap register 1,       Address offset: 0x04 */
  __IO uint32_t EINTSEL[4]; /*!< AFIO external interrupt selection register 1, Address offset: 0x08 - 0x14 */
  uint32_t      RESERVED0;  /*!< Reserved, 0x18 */
} AFIO_TypeDef;

/**
  * @brief Inter-integrated Circuit Interface
  */

typedef struct
{
  __IO uint32_t CTRL1;      /*!< I2C Control register 1,     Address offset: 0x00 */
  __IO uint32_t CTRL2;      /*!< I2C Control register 2,     Address offset: 0x04 */
  __IO uint32_t SADDR1;     /*!< I2C Own address register 1, Address offset: 0x08 */
  __IO uint32_t SADDR2;     /*!< I2C Own address register 2, Address offset: 0x0C */
  __IO uint32_t DATA;       /*!< I2C Data register,          Address offset: 0x10 */
  __IO uint32_t STS1;       /*!< I2C Status register 1,      Address offset: 0x14 */
  __IO uint32_t STS2;       /*!< I2C Status register 2,      Address offset: 0x18 */
  __IO uint32_t CLKCTRL;    /*!< I2C Clock control register, Address offset: 0x1C */
  __IO uint32_t RISETMAX;   /*!< I2C RISETMAX register,      Address offset: 0x20 */
} I2C_TypeDef;

/**
  * @brief Independent WATCHDOG
  */

typedef struct
{
  __IO uint32_t KEY;    /*!< IWDT Key register,       Address offset: 0x00 */
  __IO uint32_t PSC;    /*!< IWDT Prescaler register, Address offset: 0x04 */
  __IO uint32_t CNTRLD; /*!< IWDT Reload register,    Address offset: 0x08 */
  __IO uint32_t STS;    /*!< IWDT Status register,    Address offset: 0x0C */
} IWDT_TypeDef;


/**
  * @brief Power Control
  */

typedef struct
{
  __IO uint32_t CTRL; /*!< PMU power control register,        Address offset: 0x00 */
  __IO uint32_t CSTS; /*!< PMU power control/status register, Address offset: 0x04 */
} PMU_TypeDef;

/**
  * @brief Reset and Clock Control
  */

typedef struct
{
  __IO uint32_t CTRL;           /*!< RCM clock control register,                                  Address offset: 0x00 */
  __IO uint32_t CFG;            /*!< RCM clock configuration register,                            Address offset: 0x04 */
  __IO uint32_t INT;            /*!< RCM clock interrupt register,                                Address offset: 0x08 */
  __IO uint32_t APB2RST;        /*!< RCM APB2 peripheral reset register,                          Address offset: 0x0C */
  __IO uint32_t APB1RST;        /*!< RCM APB1 peripheral reset register,                          Address offset: 0x10 */
  __IO uint32_t AHB1CLKEN;      /*!< RCM AHB1 peripheral clock enable register,                   Address offset: 0x14 */
  __IO uint32_t APB2CLKEN;      /*!< RCM APB2 peripheral clock enable register,                   Address offset: 0x18 */
  __IO uint32_t APB1CLKEN;      /*!< RCM APB1 peripheral clock enable register,                   Address offset: 0x1C */
  __IO uint32_t BDCTRL;         /*!< RCM backup domain control register,                          Address offset: 0x20 */
  __IO uint32_t CSTS;           /*!< RCM clock control/status register,                           Address offset: 0x24 */
  __IO uint32_t AHB1RST;        /*!< RCM AHB1 peripheral reset register,                          Address offset: 0x28 */
} RCM_TypeDef;

/**
  * @brief Real-Time Clock
  */

typedef struct
{
  __IO uint32_t CTRL;     /*!< RTC control register,    Address offset: 0x00 */
  __IO uint32_t CSTS;     /*!< RTC status register,     Address offset: 0x04 */
  __IO uint32_t PSCRLDH;  /*!< RTC prescaler high register, Address offset: 0x08 */
  __IO uint32_t PSCRLDL;  /*!< RTC prescaler low register,  Address offset: 0x0C */
  __IO uint32_t PSCH;     /*!< RTC prescaler high register, Address offset: 0x10 */
  __IO uint32_t PSCL;     /*!< RTC prescaler low register,  Address offset: 0x14 */
  __IO uint32_t CNTH;     /*!< RTC counter high register,   Address offset: 0x18 */
  __IO uint32_t CNTL;     /*!< RTC counter low register,    Address offset: 0x1C */
  __IO uint32_t ALRH;     /*!< RTC alarm high register,     Address offset: 0x20 */
  __IO uint32_t ALRL;     /*!< RTC alarm low register,      Address offset: 0x24 */
} RTC_TypeDef;

/**
  * @brief Serial Peripheral Interface
  */

typedef struct
{
  __IO uint32_t CTRL1;      /*!< SPI control register 1 (not used in I2S mode),      Address offset: 0x00 */
  __IO uint32_t CTRL2;      /*!< SPI control register 2,                             Address offset: 0x04 */
  __IO uint32_t STS;        /*!< SPI status register,                                Address offset: 0x08 */
  __IO uint32_t DATA;       /*!< SPI data register,                                  Address offset: 0x0C */
  __IO uint32_t CRCPOLY;    /*!< SPI CRC polynomial register (not used in I2S mode), Address offset: 0x10 */
  __IO uint32_t RXCRC;      /*!< SPI RX CRC register (not used in I2S mode),         Address offset: 0x14 */
  __IO uint32_t TXCRC;      /*!< SPI TX CRC register (not used in I2S mode),         Address offset: 0x18 */
  __IO uint32_t I2SCFG;     /*!< SPI_I2S configuration register,                     Address offset: 0x1C */
  __IO uint32_t I2SPSC;     /*!< SPI_I2S prescaler register,                         Address offset: 0x20 */
} SPI_TypeDef;


/**
  * @brief TMR
  */

typedef struct
{
  __IO uint32_t CTRL1;        /*!< TMR control register 1,              Address offset: 0x00 */
  __IO uint32_t CTRL2;        /*!< TMR control register 2,              Address offset: 0x04 */
  __IO uint32_t SMCTRL;       /*!< TMR slave mode control register,     Address offset: 0x08 */
  __IO uint32_t DIEN;         /*!< TMR DMA/interrupt enable register,   Address offset: 0x0C */
  __IO uint32_t STS;          /*!< TMR status register,                 Address offset: 0x10 */
  __IO uint32_t CEG;          /*!< TMR event generation register,       Address offset: 0x14 */
  __IO uint32_t CCM1;         /*!< TMR capture/compare mode register 1, Address offset: 0x18 */
  __IO uint32_t CCM2;         /*!< TMR capture/compare mode register 2, Address offset: 0x1C */
  __IO uint32_t CCEN;         /*!< TMR capture/compare enable register, Address offset: 0x20 */
  __IO uint32_t CNT;          /*!< TMR counter register,                Address offset: 0x24 */
  __IO uint32_t PSC;          /*!< TMR prescaler,                       Address offset: 0x28 */
  __IO uint32_t AUTORLD;      /*!< TMR auto-reload register,            Address offset: 0x2C */
  __IO uint32_t REPCNT;       /*!< TMR repetition counter register,     Address offset: 0x30 */
  __IO uint32_t CC1;          /*!< TMR capture/compare register 1,      Address offset: 0x34 */
  __IO uint32_t CC2;          /*!< TMR capture/compare register 2,      Address offset: 0x38 */
  __IO uint32_t CC3;          /*!< TMR capture/compare register 3,      Address offset: 0x3C */
  __IO uint32_t CC4;          /*!< TMR capture/compare register 4,      Address offset: 0x40 */
  __IO uint32_t BDT;          /*!< TMR break and dead-time register,    Address offset: 0x44 */
  __IO uint32_t DCTRL;        /*!< TMR DMA control register,            Address offset: 0x48 */
  __IO uint32_t DMADDR;       /*!< TMR DMA address for full transfer,   Address offset: 0x4C */
  __IO uint32_t OPT;          /*!< TMR option register,                 Address offset: 0x50 */
} TMR_TypeDef;

/**
  * @brief Universal Synchronous Asynchronous Receiver Transmitter
  */

typedef struct
{
  __IO uint32_t STS;        /*!< USART Status register,                   Address offset: 0x00 */
  __IO uint32_t DATA;       /*!< USART Data register,                     Address offset: 0x04 */
  __IO uint32_t BR;         /*!< USART Baud rate register,                Address offset: 0x08 */
  __IO uint32_t CTRL1;      /*!< USART Control register 1,                Address offset: 0x0C */
  __IO uint32_t CTRL2;      /*!< USART Control register 2,                Address offset: 0x10 */
  __IO uint32_t CTRL3;      /*!< USART Control register 3,                Address offset: 0x14 */
  __IO uint32_t GTPSC;      /*!< USART Guard time and prescaler register, Address offset: 0x18 */
} USART_TypeDef;

/**
  * @brief Window WATCHDOG
  */

typedef struct
{
  __IO uint32_t CTRL; /*!< WWDT Control register,       Address offset: 0x00 */
  __IO uint32_t CFG;  /*!< WWDT Configuration register, Address offset: 0x04 */
  __IO uint32_t STS;  /*!< WWDT Status register,        Address offset: 0x08 */
} WWDT_TypeDef;

/**
  * @brief USB_OTG_Core_Registers
  */
typedef struct
{
  __IO uint32_t GCTRLSTS;             /*!< USB_OTG Control and Status Register          000h */
  __IO uint32_t GINT;                 /*!< USB_OTG Interrupt Register                   004h */
  __IO uint32_t GAHBCFG;              /*!< Core AHB Configuration Register              008h */
  __IO uint32_t GUSBCFG;              /*!< Core USB Configuration Register              00Ch */
  __IO uint32_t GRSTCTRL;             /*!< Core Reset Register                          010h */
  __IO uint32_t GCINT;                /*!< Core Interrupt Register                      014h */
  __IO uint32_t GINTMASK;             /*!< Core Interrupt Mask Register                 018h */
  __IO uint32_t GRXSTS;               /*!< Receive Sts Q Read Register                  01Ch */
  __IO uint32_t GRXSTSP;              /*!< Receive Sts Q Read & POP Register            020h */
  __IO uint32_t GRXFIFO;              /*!< Receive FIFO Size Register                   024h */
  __IO uint32_t GTXFCFG;              /*!< EP0 / Non Periodic Tx FIFO Size Register     028h */
  __IO uint32_t GNPTXFQSTS;           /*!< Non Periodic Tx FIFO/Queue Sts reg           02Ch */
  uint32_t Reserved30[2];             /*!< Reserved                                     030h */
  __IO uint32_t GGCCFG;               /*!< General Purpose IO Register                  038h */
  __IO uint32_t GCID;                 /*!< User ID Register                             03Ch */
  uint32_t  Reserved40[48];           /*!< Reserved                                0x40-0xFF */
  __IO uint32_t GHPTXFSIZE;           /*!< Host Periodic Tx FIFO Size Reg               100h */
  __IO uint32_t DTXFIFO[0x0F];        /*!< dev Periodic Transmit FIFO                        */
} USB_OTG_GlobalTypeDef;

/**
  * @brief USB_OTG_device_Registers
  */
typedef struct
{
  __IO uint32_t DCFG;           /*!< dev Configuration Register   800h */
  __IO uint32_t DCTRL;          /*!< dev Control Register         804h */
  __IO uint32_t DSTS;           /*!< dev Status Register (RO)     808h */
  uint32_t Reserved0C;          /*!< Reserved                     80Ch */
  __IO uint32_t DINIMASK;       /*!< dev IN Endpoint Mask         810h */
  __IO uint32_t DOUTIMASK;      /*!< dev OUT Endpoint Mask        814h */
  __IO uint32_t DAEPINT;        /*!< dev All Endpoints Itr Reg    818h */
  __IO uint32_t DAEPIMASK;      /*!< dev All Endpoints Itr Mask   81Ch */
  uint32_t  Reserved20;         /*!< Reserved                     820h */
  uint32_t Reserved9;           /*!< Reserved                     824h */
  __IO uint32_t DVBUSDTIM;      /*!< dev VBUS discharge Register  828h */
  __IO uint32_t DVBUSPTIM;      /*!< dev VBUS Pulse Register      82Ch */
  __IO uint32_t DTHCTRL;        /*!< dev threshold                830h */
  __IO uint32_t DIEIMASK;       /*!< dev empty msk                834h */
  __IO uint32_t DEPINT;         /*!< dedicated EP interrupt       838h */
  __IO uint32_t DEPIMASK;       /*!< dedicated EP msk             83Ch */
  uint32_t Reserved40;          /*!< dedicated EP mask            840h */
  __IO uint32_t DIN1IMASK;      /*!< dedicated EP mask            844h */
  uint32_t  Reserved44[15];     /*!< Reserved                 844-87Ch */
  __IO uint32_t DOUT1MASK;      /*!< dedicated EP msk             884h */
} USB_OTG_DeviceTypeDef;

/**
  * @brief USB_OTG_IN_Endpoint-Specific_Register
  */
typedef struct
{
  __IO uint32_t DIEPCTRL;         /*!< dev IN Endpoint Control Reg    900h + (ep_num * 20h) + 00h */
  uint32_t Reserved04;            /*!< Reserved                       900h + (ep_num * 20h) + 04h */
  __IO uint32_t DIEPINT;          /*!< dev IN Endpoint Itr Reg        900h + (ep_num * 20h) + 08h */
  uint32_t Reserved0C;            /*!< Reserved                       900h + (ep_num * 20h) + 0Ch */
  __IO uint32_t DIEPTRS;          /*!< IN Endpoint Txfer Size         900h + (ep_num * 20h) + 10h */
  __IO uint32_t DIEPDMA;          /*!< IN Endpoint DMA Address Reg    900h + (ep_num * 20h) + 14h */
  __IO uint32_t DITXFSTS;         /*!< IN Endpoint Tx FIFO Status Reg 900h + (ep_num * 20h) + 18h */
  uint32_t Reserved18;            /*!< Reserved  900h+(ep_num*20h)+1Ch-900h+ (ep_num * 20h) + 1Ch */
} USB_OTG_INEndpointTypeDef;

/**
  * @brief USB_OTG_OUT_Endpoint-Specific_Registers
  */
typedef struct
{
  __IO uint32_t DOEPCTRL;       /*!< dev OUT Endpoint Control Reg           B00h + (ep_num * 20h) + 00h */
  uint32_t Reserved04;          /*!< Reserved                               B00h + (ep_num * 20h) + 04h */
  __IO uint32_t DOEPINT;        /*!< dev OUT Endpoint Itr Reg               B00h + (ep_num * 20h) + 08h */
  uint32_t Reserved0C;          /*!< Reserved                               B00h + (ep_num * 20h) + 0Ch */
  __IO uint32_t DOEPTRS;        /*!< dev OUT Endpoint Txfer Size            B00h + (ep_num * 20h) + 10h */
  __IO uint32_t DOEPDMA;        /*!< dev OUT Endpoint DMA Address           B00h + (ep_num * 20h) + 14h */
  uint32_t Reserved18[2];       /*!< Reserved B00h + (ep_num * 20h) + 18h - B00h + (ep_num * 20h) + 1Ch */
} USB_OTG_OUTEndpointTypeDef;

/**
  * @brief USB_OTG_Host_Mode_Register_Structures
  */
typedef struct
{
  __IO uint32_t HCFG;             /*!< Host Configuration Register          400h */
  __IO uint32_t HFIVL;            /*!< Host Frame Interval Register         404h */
  __IO uint32_t HFIFM;            /*!< Host Frame Nbr/Frame Remaining       408h */
  uint32_t Reserved40C;           /*!< Reserved                             40Ch */
  __IO uint32_t HPTXSTS;          /*!< Host Periodic Tx FIFO/ Queue Status  410h */
  __IO uint32_t HACHINT;          /*!< Host All Channels Interrupt Register 414h */
  __IO uint32_t HACHIMASK;        /*!< Host All Channels Interrupt Mask     418h */
} USB_OTG_HostTypeDef;

/**
  * @brief USB_OTG_Host_Channel_Specific_Registers
  */
typedef struct
{
  __IO uint32_t HCH;              /*!< Host Channel Characteristics Register    500h */
  __IO uint32_t HCHSCTRL;         /*!< Host Channel Split Control Register      504h */
  __IO uint32_t HCHINT;           /*!< Host Channel Interrupt Register          508h */
  __IO uint32_t HCHIMASK;         /*!< Host Channel Interrupt Mask Register     50Ch */
  __IO uint32_t HCHTSIZE;         /*!< Host Channel Transfer Size Register      510h */
  __IO uint32_t HCHDMA;           /*!< Host Channel DMA Address Register        514h */
  uint32_t Reserved[2];           /*!< Reserved                                      */
} USB_OTG_HostChannelTypeDef;

/**
  * @}
  */

/** @addtogroup Peripheral_memory_map
  * @{
  */
#define FLASH_BASE            0x08000000UL /*!< FLASH base address in the alias region                                     */
#define FLASH_END             0x0801FFFFUL /*!< FLASH end address                                                          */
#define SRAM_BASE             0x20000000UL /*!< SRAM base address in the alias region                                      */
#define PERIPH_BASE           0x40000000UL /*!< Peripheral base address in the alias region                                */

#define SRAM_BB_BASE          0x22000000UL /*!< SRAM base address in the bit-band region                                   */
#define PERIPH_BB_BASE        0x42000000UL /*!< Peripheral base address in the bit-band region                             */
#define BKPSRAM_BB_BASE       0x42480000UL /*!< Backup SRAM(4 KB) base address in the bit-band region                      */

/*!< Peripheral memory map */
#define APB1PERIPH_BASE       PERIPH_BASE
#define APB2PERIPH_BASE       (PERIPH_BASE + 0x00010000UL)
#define AHBPERIPH_BASE        (PERIPH_BASE + 0x00020000UL)

/*!< APB1 peripherals */
#define TMR2_BASE             (APB1PERIPH_BASE + 0x0000UL)
#define TMR3_BASE             (APB1PERIPH_BASE + 0x0400UL)
#define TMR4_BASE             (APB1PERIPH_BASE + 0x0800UL)
#define TMR5_BASE             (APB1PERIPH_BASE + 0x0C00UL)
#define RTC_BASE              (APB1PERIPH_BASE + 0x2800UL)
#define WWDT_BASE             (APB1PERIPH_BASE + 0x2C00UL)
#define IWDT_BASE             (APB1PERIPH_BASE + 0x3000UL)
#define SPI2_BASE             (APB1PERIPH_BASE + 0x3800UL)
#define USART2_BASE           (APB1PERIPH_BASE + 0x4400UL)
#define USART3_BASE           (APB1PERIPH_BASE + 0x4800UL)
#define UART4_BASE            (APB1PERIPH_BASE + 0x4C00UL)
#define I2C1_BASE             (APB1PERIPH_BASE + 0x5400UL)
#define CAN1_BASE             (APB1PERIPH_BASE + 0x6400UL)
#define CAN2_BASE             (APB1PERIPH_BASE + 0x6800UL)
#define BAKPR_BASE            (APB1PERIPH_BASE + 0x6C00UL)
#define PMU_BASE              (APB1PERIPH_BASE + 0x7000UL)

/*!< APB2 peripherals */
#define AFIO_BASE             (APB2PERIPH_BASE + 0x0000UL)
#define EINT_BASE             (APB2PERIPH_BASE + 0x0400UL)
#define GPIOA_BASE            (APB2PERIPH_BASE + 0x0800UL)
#define GPIOB_BASE            (APB2PERIPH_BASE + 0x0C00UL)
#define GPIOC_BASE            (APB2PERIPH_BASE + 0x1000UL)
#define GPIOD_BASE            (APB2PERIPH_BASE + 0x1400UL)
#define ADC1_BASE             (APB2PERIPH_BASE + 0x2400UL)
#define ADC2_BASE             (APB2PERIPH_BASE + 0x2800UL)
#define TMR1_BASE             (APB2PERIPH_BASE + 0x2C00UL)
#define SPI1_BASE             (APB2PERIPH_BASE + 0x3000UL)
#define TMR8_BASE             (APB2PERIPH_BASE + 0x3400UL)
#define USART1_BASE           (APB2PERIPH_BASE + 0x3800UL)

/*!< AHB peripherals */
#define DMA1_BASE             (AHBPERIPH_BASE + 0x0000UL)
#define DMA1_Channel1_BASE    (AHBPERIPH_BASE + 0x0008UL)
#define DMA1_Channel2_BASE    (AHBPERIPH_BASE + 0x001CUL)
#define DMA1_Channel3_BASE    (AHBPERIPH_BASE + 0x0030UL)
#define DMA1_Channel4_BASE    (AHBPERIPH_BASE + 0x0044UL)
#define DMA1_Channel5_BASE    (AHBPERIPH_BASE + 0x0058UL)
#define DMA1_Channel6_BASE    (AHBPERIPH_BASE + 0x006CUL)
#define DMA1_Channel7_BASE    (AHBPERIPH_BASE + 0x0080UL)
#define DMA2_BASE             (AHBPERIPH_BASE + 0x0400UL)
#define DMA2_Channel1_BASE    (AHBPERIPH_BASE + 0x0408UL)
#define DMA2_Channel2_BASE    (AHBPERIPH_BASE + 0x041CUL)
#define DMA2_Channel3_BASE    (AHBPERIPH_BASE + 0x0430UL)
#define DMA2_Channel4_BASE    (AHBPERIPH_BASE + 0x0444UL)
#define DMA2_Channel5_BASE    (AHBPERIPH_BASE + 0x0458UL)
#define RCM_BASE              (AHBPERIPH_BASE + 0x1000UL)
#define CRC_BASE              (AHBPERIPH_BASE + 0x3000UL)
#define FLASH_R_BASE          (AHBPERIPH_BASE + 0x2000UL)   /*!< FLASH registers base address */

/*!< Debug MCU registers base address */
#define DBGMCU_BASE           0xE0042000UL
/*!< USB registers base address */
#define USB_OTG_FS_PERIPH_BASE               0x50000000UL

#define USB_OTG_GLOBAL_BASE                  0x000UL
#define USB_OTG_DEVICE_BASE                  0x800UL
#define USB_OTG_IN_ENDPOINT_BASE             0x900UL
#define USB_OTG_OUT_ENDPOINT_BASE            0xB00UL
#define USB_OTG_EP_REG_SIZE                  0x20UL
#define USB_OTG_HOST_BASE                    0x400UL
#define USB_OTG_HOST_PORT_BASE               0x440UL
#define USB_OTG_HOST_CHANNEL_BASE            0x500UL
#define USB_OTG_HOST_CHANNEL_SIZE            0x20UL
#define USB_OTG_PCGCCTL_BASE                 0xE00UL
#define USB_OTG_FIFO_BASE                    0x1000UL
#define USB_OTG_FIFO_SIZE                    0x1000UL

#define FLASHSIZE_BASE        0x1FFFF7E0UL                  /*!< FLASH Size register base address */
#define UID_BASE              0x1FFFF7E8UL                  /*!< Unique device ID register base address */
#define OB_BASE               0x1FFFF800UL                  /*!< FLASH Option Bytes base address */

/**
  * @}
  */

/** @addtogroup Peripheral_declaration
  * @{
  */
#define TMR2                ((TMR_TypeDef *) TMR2_BASE)
#define TMR3                ((TMR_TypeDef *) TMR3_BASE)
#define TMR4                ((TMR_TypeDef *) TMR4_BASE)
#define TMR5                ((TMR_TypeDef *) TMR5_BASE)
#define RTC                 ((RTC_TypeDef *) RTC_BASE)
#define WWDT                ((WWDT_TypeDef *) WWDT_BASE)
#define IWDT                ((IWDT_TypeDef *) IWDT_BASE)
#define SPI2                ((SPI_TypeDef *) SPI2_BASE)
#define USART2              ((USART_TypeDef *) USART2_BASE)
#define USART3              ((USART_TypeDef *) USART3_BASE)
#define UART4               ((USART_TypeDef *) UART4_BASE)
#define I2C1                ((I2C_TypeDef *) I2C1_BASE)
#define CAN1                ((CAN_TypeDef *) CAN1_BASE)
#define CAN2                ((CAN_TypeDef *) CAN2_BASE)
#define BAKPR               ((BAKPR_TypeDef *) BAKPR_BASE)
#define PMU                 ((PMU_TypeDef *) PMU_BASE)

#define AFIO                ((AFIO_TypeDef *) AFIO_BASE)
#define EINT                ((EINT_TypeDef *) EINT_BASE)
#define GPIOA               ((GPIO_TypeDef *) GPIOA_BASE)
#define GPIOB               ((GPIO_TypeDef *) GPIOB_BASE)
#define GPIOC               ((GPIO_TypeDef *) GPIOC_BASE)
#define GPIOD               ((GPIO_TypeDef *) GPIOD_BASE)
#define ADC1                ((ADC_TypeDef *) ADC1_BASE)
#define ADC2                ((ADC_TypeDef *) ADC2_BASE)
#define ADC12_COMMON        ((ADC_Common_TypeDef *) ADC1_BASE)
#define TMR1                ((TMR_TypeDef *) TMR1_BASE)
#define SPI1                ((SPI_TypeDef *) SPI1_BASE)
#define TMR8                ((TMR_TypeDef *) TMR8_BASE)
#define USART1              ((USART_TypeDef *) USART1_BASE)
#define DMA1                ((DMA_TypeDef *) DMA1_BASE)
#define DMA1_Channel1       ((DMA_Channel_TypeDef *) DMA1_Channel1_BASE)
#define DMA1_Channel2       ((DMA_Channel_TypeDef *) DMA1_Channel2_BASE)
#define DMA1_Channel3       ((DMA_Channel_TypeDef *) DMA1_Channel3_BASE)
#define DMA1_Channel4       ((DMA_Channel_TypeDef *) DMA1_Channel4_BASE)
#define DMA1_Channel5       ((DMA_Channel_TypeDef *) DMA1_Channel5_BASE)
#define DMA1_Channel6       ((DMA_Channel_TypeDef *) DMA1_Channel6_BASE)
#define DMA1_Channel7       ((DMA_Channel_TypeDef *) DMA1_Channel7_BASE)
#define DMA2                ((DMA_TypeDef *) DMA2_BASE)
#define DMA2_Channel1       ((DMA_Channel_TypeDef *) DMA2_Channel1_BASE)
#define DMA2_Channel2       ((DMA_Channel_TypeDef *) DMA2_Channel2_BASE)
#define DMA2_Channel3       ((DMA_Channel_TypeDef *) DMA2_Channel3_BASE)
#define DMA2_Channel4       ((DMA_Channel_TypeDef *) DMA2_Channel4_BASE)
#define DMA2_Channel5       ((DMA_Channel_TypeDef *) DMA2_Channel5_BASE)
#define RCM                 ((RCM_TypeDef *) RCM_BASE)
#define CRC                 ((CRC_TypeDef *) CRC_BASE)
#define FLASH               ((FLASH_TypeDef *) FLASH_R_BASE)
#define OB                  ((OB_TypeDef *) OB_BASE)
#define DBGMCU              ((DBGMCU_TypeDef *) DBGMCU_BASE)
#define USB_OTG_FS          ((USB_OTG_GlobalTypeDef *) USB_OTG_FS_PERIPH_BASE)


/**
  * @}
  */

/** @addtogroup Exported_constants
  * @{
  */

/** @addtogroup Hardware_Constant_Definition
  * @{
  */
#define LSI_STARTUP_TIME                85U /*!< LSI Maximum startup time in us */
/**
  * @}
  */

  /** @addtogroup Peripheral_Registers_Bits_Definition
  * @{
  */

/******************************************************************************/
/*                         Peripheral Registers_Bits_Definition               */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*                        Analog to Digital Converter                         */
/*                                                                            */
/******************************************************************************/
/*
 * @brief Specific device feature definitions (not present on all devices in the APM32F4 serie)
 */
#define ADC_MULTIMODE_SUPPORT                                                                                                      /*!<ADC Multimode feature available on specific devices */

/********************  Bit definition for ADC_STS register  ********************/
#define ADC_STS_AWDFLG_Pos                                          (0U)
#define ADC_STS_AWDFLG_Msk                                          (0x1UL << ADC_STS_AWDFLG_Pos)                                  /*!< 0x00000001 */
#define ADC_STS_AWDFLG                                              ADC_STS_AWDFLG_Msk                                             /*!<Analog watchdog flag */
#define ADC_STS_EOCFLG_Pos                                          (1U)
#define ADC_STS_EOCFLG_Msk                                          (0x1UL << ADC_STS_EOCFLG_Pos)                                  /*!< 0x00000002 */
#define ADC_STS_EOCFLG                                              ADC_STS_EOCFLG_Msk                                             /*!<End of conversion */
#define ADC_STS_INJEOCFLG_Pos                                       (2U)
#define ADC_STS_INJEOCFLG_Msk                                       (0x1UL << ADC_STS_INJEOCFLG_Pos)                               /*!< 0x00000004 */
#define ADC_STS_INJEOCFLG                                           ADC_STS_INJEOCFLG_Msk                                          /*!<Injected channel end of conversion */
#define ADC_STS_INJCSFLG_Pos                                        (3U)
#define ADC_STS_INJCSFLG_Msk                                        (0x1UL << ADC_STS_INJCSFLG_Pos)                                /*!< 0x00000008 */
#define ADC_STS_INJCSFLG                                            ADC_STS_INJCSFLG_Msk                                           /*!<Injected channel Start flag */
#define ADC_STS_REGCSFLG_Pos                                        (4U)
#define ADC_STS_REGCSFLG_Msk                                        (0x1UL << ADC_STS_REGCSFLG_Pos)                                /*!< 0x00000010 */
#define ADC_STS_REGCSFLG                                            ADC_STS_REGCSFLG_Msk                                           /*!<Regular channel Start flag */

/*******************  Bit definition for ADC_CTRL1 register  ********************/
#define ADC_CTRL1_AWDCHSEL_Pos                                      (0U)
#define ADC_CTRL1_AWDCHSEL_Msk                                      (0x1FUL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x0000001F */
#define ADC_CTRL1_AWDCHSEL                                          ADC_CTRL1_AWDCHSEL_Msk                                         /*!<AWDCH[4:0] bits (Analog watchdog channel select bits) */
#define ADC_CTRL1_AWDCHSEL_0                                        (0x01UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000001 */
#define ADC_CTRL1_AWDCHSEL_1                                        (0x02UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000002 */
#define ADC_CTRL1_AWDCHSEL_2                                        (0x04UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000004 */
#define ADC_CTRL1_AWDCHSEL_3                                        (0x08UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000008 */
#define ADC_CTRL1_AWDCHSEL_4                                        (0x10UL << ADC_CTRL1_AWDCHSEL_Pos)                             /*!< 0x00000010 */
#define ADC_CTRL1_EOCIEN_Pos                                        (5U)
#define ADC_CTRL1_EOCIEN_Msk                                        (0x1UL << ADC_CTRL1_EOCIEN_Pos)                                /*!< 0x00000020 */
#define ADC_CTRL1_EOCIEN                                            ADC_CTRL1_EOCIEN_Msk                                           /*!<Interrupt enable for EOC */
#define ADC_CTRL1_AWDIEN_Pos                                        (6U)
#define ADC_CTRL1_AWDIEN_Msk                                        (0x1UL << ADC_CTRL1_AWDIEN_Pos)                                /*!< 0x00000040 */
#define ADC_CTRL1_AWDIEN                                            ADC_CTRL1_AWDIEN_Msk                                           /*!<AAnalog Watchdog interrupt enable */
#define ADC_CTRL1_INJEOCIEN_Pos                                     (7U)
#define ADC_CTRL1_INJEOCIEN_Msk                                     (0x1UL << ADC_CTRL1_INJEOCIEN_Pos)                             /*!< 0x00000080 */
#define ADC_CTRL1_INJEOCIEN                                         ADC_CTRL1_INJEOCIEN_Msk                                        /*!<Interrupt enable for injected channels */
#define ADC_CTRL1_SCANEN_Pos                                        (8U)
#define ADC_CTRL1_SCANEN_Msk                                        (0x1UL << ADC_CTRL1_SCANEN_Pos)                                /*!< 0x00000100 */
#define ADC_CTRL1_SCANEN                                            ADC_CTRL1_SCANEN_Msk                                           /*!<Scan mode */
#define ADC_CTRL1_AWDSGLEN_Pos                                      (9U)
#define ADC_CTRL1_AWDSGLEN_Msk                                      (0x1UL << ADC_CTRL1_AWDSGLEN_Pos)                              /*!< 0x00000200 */
#define ADC_CTRL1_AWDSGLEN                                          ADC_CTRL1_AWDSGLEN_Msk                                         /*!<Enable the watchdog on a single channel in scan mode */
#define ADC_CTRL1_INJGACEN_Pos                                      (10U)
#define ADC_CTRL1_INJGACEN_Msk                                      (0x1UL << ADC_CTRL1_INJGACEN_Pos)                              /*!< 0x00000400 */
#define ADC_CTRL1_INJGACEN                                          ADC_CTRL1_INJGACEN_Msk                                         /*!<Automatic injected group conversion */
#define ADC_CTRL1_REGDISCEN_Pos                                     (11U)
#define ADC_CTRL1_REGDISCEN_Msk                                     (0x1UL << ADC_CTRL1_REGDISCEN_Pos)                             /*!< 0x00000800 */
#define ADC_CTRL1_REGDISCEN                                         ADC_CTRL1_REGDISCEN_Msk                                        /*!<Discontinuous mode on regular channels */
#define ADC_CTRL1_INJDISCEN_Pos                                     (12U)
#define ADC_CTRL1_INJDISCEN_Msk                                     (0x1UL << ADC_CTRL1_INJDISCEN_Pos)                             /*!< 0x00001000 */
#define ADC_CTRL1_INJDISCEN                                         ADC_CTRL1_INJDISCEN_Msk                                        /*!<Discontinuous mode on injected channels */
#define ADC_CTRL1_DISCNUMCFG_Pos                                    (13U)
#define ADC_CTRL1_DISCNUMCFG_Msk                                    (0x7UL << ADC_CTRL1_DISCNUMCFG_Pos)                            /*!< 0x0000E000 */
#define ADC_CTRL1_DISCNUMCFG                                        ADC_CTRL1_DISCNUMCFG_Msk                                       /*!<DISCNUM[2:0] bits (Discontinuous mode channel count) */
#define ADC_CTRL1_DISCNUMCFG_0                                      (0x1UL << ADC_CTRL1_DISCNUMCFG_Pos)                            /*!< 0x00002000 */
#define ADC_CTRL1_DISCNUMCFG_1                                      (0x2UL << ADC_CTRL1_DISCNUMCFG_Pos)                            /*!< 0x00004000 */
#define ADC_CTRL1_DISCNUMCFG_2                                      (0x4UL << ADC_CTRL1_DISCNUMCFG_Pos)                            /*!< 0x00008000 */
#define ADC_CTRL1_DUALMCFG_Pos                                      (16U)
#define ADC_CTRL1_DUALMCFG_Msk                                      (0xFUL << ADC_CTRL1_DUALMCFG_Pos)                              /*!< 0x000F0000 */
#define ADC_CTRL1_DUALMCFG                                          ADC_CTRL1_DUALMCFG_Msk                                         /*!<ADC multimode mode selection */
#define ADC_CTRL1_DUALMCFG_0                                        (0x1UL << ADC_CTRL1_DUALMCFG_Pos)                              /*!< 0x00010000 */
#define ADC_CTRL1_DUALMCFG_1                                        (0x2UL << ADC_CTRL1_DUALMCFG_Pos)                              /*!< 0x00020000 */
#define ADC_CTRL1_DUALMCFG_2                                        (0x4UL << ADC_CTRL1_DUALMCFG_Pos)                              /*!< 0x00040000 */
#define ADC_CTRL1_DUALMCFG_3                                        (0x8UL << ADC_CTRL1_DUALMCFG_Pos)                              /*!< 0x00080000 */
#define ADC_CTRL1_INJAWDEN_Pos                                      (22U)
#define ADC_CTRL1_INJAWDEN_Msk                                      (0x1UL << ADC_CTRL1_INJAWDEN_Pos)                              /*!< 0x00400000 */
#define ADC_CTRL1_INJAWDEN                                          ADC_CTRL1_INJAWDEN_Msk                                         /*!<Analog watchdog enable on injected channels */
#define ADC_CTRL1_REGAWDEN_Pos                                      (23U)
#define ADC_CTRL1_REGAWDEN_Msk                                      (0x1UL << ADC_CTRL1_REGAWDEN_Pos)                              /*!< 0x00800000 */
#define ADC_CTRL1_REGAWDEN                                          ADC_CTRL1_REGAWDEN_Msk                                         /*!<Analog watchdog enable on regular channels */

/*******************  Bit definition for ADC_CTRL2 register  ********************/
#define ADC_CTRL2_ADCEN_Pos                                         (0U)
#define ADC_CTRL2_ADCEN_Msk                                         (0x1UL << ADC_CTRL2_ADCEN_Pos)                                 /*!< 0x00000001 */
#define ADC_CTRL2_ADCEN                                             ADC_CTRL2_ADCEN_Msk                                            /*!<A/D Converter ON / OFF */
#define ADC_CTRL2_CONTCEN_Pos                                       (1U)
#define ADC_CTRL2_CONTCEN_Msk                                       (0x1UL << ADC_CTRL2_CONTCEN_Pos)                               /*!< 0x00000002 */
#define ADC_CTRL2_CONTCEN                                           ADC_CTRL2_CONTCEN_Msk                                          /*!<Continuous Conversion */
#define ADC_CTRL2_CAL_Pos                                           (2U)
#define ADC_CTRL2_CAL_Msk                                           (0x1UL << ADC_CTRL2_CAL_Pos)                                   /*!< 0x00000004 */
#define ADC_CTRL2_CAL                                               ADC_CTRL2_CAL_Msk                                              /*!<A/D Calibration start */
#define ADC_CTRL2_CALRST_Pos                                        (3U)
#define ADC_CTRL2_CALRST_Msk                                        (0x1UL << ADC_CTRL2_CALRST_Pos)                                /*!< 0x00000008 */
#define ADC_CTRL2_CALRST                                            ADC_CTRL2_CALRST_Msk                                           /*!<A/D Calibration reset */
#define ADC_CTRL2_DMAEN_Pos                                         (8U)
#define ADC_CTRL2_DMAEN_Msk                                         (0x1UL << ADC_CTRL2_DMAEN_Pos)                                 /*!< 0x00000100 */
#define ADC_CTRL2_DMAEN                                             ADC_CTRL2_DMAEN_Msk                                            /*!<Direct Memory access mode */
#define ADC_CTRL2_DALIGNCFG_Pos                                     (11U)
#define ADC_CTRL2_DALIGNCFG_Msk                                     (0x1UL << ADC_CTRL2_DALIGNCFG_Pos)                             /*!< 0x00000800 */
#define ADC_CTRL2_DALIGNCFG                                         ADC_CTRL2_DALIGNCFG_Msk                                        /*!<Data Alignment */
#define ADC_CTRL2_INJGEXTTRGSEL_Pos                                 (12U)
#define ADC_CTRL2_INJGEXTTRGSEL_Msk                                 (0x7UL << ADC_CTRL2_INJGEXTTRGSEL_Pos)                         /*!< 0x00007000 */
#define ADC_CTRL2_INJGEXTTRGSEL                                     ADC_CTRL2_INJGEXTTRGSEL_Msk                                    /*!<JEXTSEL[2:0] bits (External event select for injected group) */
#define ADC_CTRL2_INJGEXTTRGSEL_0                                   (0x1UL << ADC_CTRL2_INJGEXTTRGSEL_Pos)                         /*!< 0x00001000 */
#define ADC_CTRL2_INJGEXTTRGSEL_1                                   (0x2UL << ADC_CTRL2_INJGEXTTRGSEL_Pos)                         /*!< 0x00002000 */
#define ADC_CTRL2_INJGEXTTRGSEL_2                                   (0x4UL << ADC_CTRL2_INJGEXTTRGSEL_Pos)                         /*!< 0x00004000 */
#define ADC_CTRL2_INJEXTTRGEN_Pos                                   (15U)
#define ADC_CTRL2_INJEXTTRGEN_Msk                                   (0x1UL << ADC_CTRL2_INJEXTTRGEN_Pos)                           /*!< 0x00008000 */
#define ADC_CTRL2_INJEXTTRGEN                                       ADC_CTRL2_INJEXTTRGEN_Msk                                      /*!<JEXTEN bit (External Trigger Conversion mode for injected channels) */
#define ADC_CTRL2_INJEXTTRGEN_0                                     ADC_CTRL2_INJEXTTRGEN_Msk                                      /*!<JEXTEN bit (External Trigger Conversion mode for injected channels) */
#define ADC_CTRL2_REGEXTTRGSEL_Pos                                  (17U)
#define ADC_CTRL2_REGEXTTRGSEL_Msk                                  (0x7UL << ADC_CTRL2_REGEXTTRGSEL_Pos)                          /*!< 0x000E0000 */
#define ADC_CTRL2_REGEXTTRGSEL                                      ADC_CTRL2_REGEXTTRGSEL_Msk                                     /*!<EXTSEL[2:0] bits (External Event Select for regular group) */
#define ADC_CTRL2_REGEXTTRGSEL_0                                    (0x1UL << ADC_CTRL2_REGEXTTRGSEL_Pos)                          /*!< 0x00020000 */
#define ADC_CTRL2_REGEXTTRGSEL_1                                    (0x2UL << ADC_CTRL2_REGEXTTRGSEL_Pos)                          /*!< 0x00040000 */
#define ADC_CTRL2_REGEXTTRGSEL_2                                    (0x4UL << ADC_CTRL2_REGEXTTRGSEL_Pos)                          /*!< 0x00080000 */
#define ADC_CTRL2_REGEXTTRGEN_Pos                                   (20U)
#define ADC_CTRL2_REGEXTTRGEN_Msk                                   (0x1UL << ADC_CTRL2_REGEXTTRGEN_Pos)                           /*!< 0x00100000 */
#define ADC_CTRL2_REGEXTTRGEN                                       ADC_CTRL2_REGEXTTRGEN_Msk                                      /*!<EXTEN bit (External Trigger Conversion mode for regular channels) */
#define ADC_CTRL2_REGEXTTRGEN_0                                     ADC_CTRL2_REGEXTTRGEN_Msk                                      /*!<EXTEN bit (External Trigger Conversion mode for regular channels) */
#define ADC_CTRL2_INJSWSC_Pos                                       (21U)
#define ADC_CTRL2_INJSWSC_Msk                                       (0x1UL << ADC_CTRL2_INJSWSC_Pos)                               /*!< 0x00200000 */
#define ADC_CTRL2_INJSWSC                                           ADC_CTRL2_INJSWSC_Msk                                          /*!<Start Conversion of injected channels */
#define ADC_CTRL2_REGSWSC_Pos                                       (22U)
#define ADC_CTRL2_REGSWSC_Msk                                       (0x1UL << ADC_CTRL2_REGSWSC_Pos)                               /*!< 0x00400000 */
#define ADC_CTRL2_REGSWSC                                           ADC_CTRL2_REGSWSC_Msk                                          /*!<Start Conversion of regular channels */
#define ADC_CTRL2_TSVREFEN_Pos                                      (23U)
#define ADC_CTRL2_TSVREFEN_Msk                                      (0x1UL << ADC_CTRL2_TSVREFEN_Pos)                              /*!< 0x00800000 */
#define ADC_CTRL2_TSVREFEN                                          ADC_CTRL2_TSVREFEN_Msk                                         /*!<ADC internal path to VrefInt and temperature sensor enable */

/******************  Bit definition for ADC_SMPTIM1 register  *******************/
#define ADC_SMPTIM1_SMPCYCCFG10_Pos                                 (0U)
#define ADC_SMPTIM1_SMPCYCCFG10_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG10_Pos)                         /*!< 0x00000007 */
#define ADC_SMPTIM1_SMPCYCCFG10                                     ADC_SMPTIM1_SMPCYCCFG10_Msk                                    /*!<SMP10[2:0] bits (Channel 10 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG10_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG10_Pos)                         /*!< 0x00000001 */
#define ADC_SMPTIM1_SMPCYCCFG10_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG10_Pos)                         /*!< 0x00000002 */
#define ADC_SMPTIM1_SMPCYCCFG10_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG10_Pos)                         /*!< 0x00000004 */
#define ADC_SMPTIM1_SMPCYCCFG11_Pos                                 (3U)
#define ADC_SMPTIM1_SMPCYCCFG11_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG11_Pos)                         /*!< 0x00000038 */
#define ADC_SMPTIM1_SMPCYCCFG11                                     ADC_SMPTIM1_SMPCYCCFG11_Msk                                    /*!<SMP11[2:0] bits (Channel 11 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG11_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG11_Pos)                         /*!< 0x00000008 */
#define ADC_SMPTIM1_SMPCYCCFG11_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG11_Pos)                         /*!< 0x00000010 */
#define ADC_SMPTIM1_SMPCYCCFG11_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG11_Pos)                         /*!< 0x00000020 */
#define ADC_SMPTIM1_SMPCYCCFG12_Pos                                 (6U)
#define ADC_SMPTIM1_SMPCYCCFG12_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG12_Pos)                         /*!< 0x000001C0 */
#define ADC_SMPTIM1_SMPCYCCFG12                                     ADC_SMPTIM1_SMPCYCCFG12_Msk                                    /*!<SMP12[2:0] bits (Channel 12 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG12_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG12_Pos)                         /*!< 0x00000040 */
#define ADC_SMPTIM1_SMPCYCCFG12_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG12_Pos)                         /*!< 0x00000080 */
#define ADC_SMPTIM1_SMPCYCCFG12_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG12_Pos)                         /*!< 0x00000100 */
#define ADC_SMPTIM1_SMPCYCCFG13_Pos                                 (9U)
#define ADC_SMPTIM1_SMPCYCCFG13_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG13_Pos)                         /*!< 0x00000E00 */
#define ADC_SMPTIM1_SMPCYCCFG13                                     ADC_SMPTIM1_SMPCYCCFG13_Msk                                    /*!<SMP13[2:0] bits (Channel 13 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG13_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG13_Pos)                         /*!< 0x00000200 */
#define ADC_SMPTIM1_SMPCYCCFG13_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG13_Pos)                         /*!< 0x00000400 */
#define ADC_SMPTIM1_SMPCYCCFG13_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG13_Pos)                         /*!< 0x00000800 */
#define ADC_SMPTIM1_SMPCYCCFG14_Pos                                 (12U)
#define ADC_SMPTIM1_SMPCYCCFG14_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG14_Pos)                         /*!< 0x00007000 */
#define ADC_SMPTIM1_SMPCYCCFG14                                     ADC_SMPTIM1_SMPCYCCFG14_Msk                                    /*!<SMP14[2:0] bits (Channel 14 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG14_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG14_Pos)                         /*!< 0x00001000 */
#define ADC_SMPTIM1_SMPCYCCFG14_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG14_Pos)                         /*!< 0x00002000 */
#define ADC_SMPTIM1_SMPCYCCFG14_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG14_Pos)                         /*!< 0x00004000 */
#define ADC_SMPTIM1_SMPCYCCFG15_Pos                                 (15U)
#define ADC_SMPTIM1_SMPCYCCFG15_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG15_Pos)                         /*!< 0x00038000 */
#define ADC_SMPTIM1_SMPCYCCFG15                                     ADC_SMPTIM1_SMPCYCCFG15_Msk                                    /*!<SMP15[2:0] bits (Channel 15 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG15_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG15_Pos)                         /*!< 0x00008000 */
#define ADC_SMPTIM1_SMPCYCCFG15_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG15_Pos)                         /*!< 0x00010000 */
#define ADC_SMPTIM1_SMPCYCCFG15_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG15_Pos)                         /*!< 0x00020000 */
#define ADC_SMPTIM1_SMPCYCCFG16_Pos                                 (18U)
#define ADC_SMPTIM1_SMPCYCCFG16_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG16_Pos)                         /*!< 0x001C0000 */
#define ADC_SMPTIM1_SMPCYCCFG16                                     ADC_SMPTIM1_SMPCYCCFG16_Msk                                    /*!<SMP16[2:0] bits (Channel 16 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG16_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG16_Pos)                         /*!< 0x00040000 */
#define ADC_SMPTIM1_SMPCYCCFG16_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG16_Pos)                         /*!< 0x00080000 */
#define ADC_SMPTIM1_SMPCYCCFG16_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG16_Pos)                         /*!< 0x00100000 */
#define ADC_SMPTIM1_SMPCYCCFG17_Pos                                 (21U)
#define ADC_SMPTIM1_SMPCYCCFG17_Msk                                 (0x7UL << ADC_SMPTIM1_SMPCYCCFG17_Pos)                         /*!< 0x00E00000 */
#define ADC_SMPTIM1_SMPCYCCFG17                                     ADC_SMPTIM1_SMPCYCCFG17_Msk                                    /*!<SMP17[2:0] bits (Channel 17 Sample time selection) */
#define ADC_SMPTIM1_SMPCYCCFG17_0                                   (0x1UL << ADC_SMPTIM1_SMPCYCCFG17_Pos)                         /*!< 0x00200000 */
#define ADC_SMPTIM1_SMPCYCCFG17_1                                   (0x2UL << ADC_SMPTIM1_SMPCYCCFG17_Pos)                         /*!< 0x00400000 */
#define ADC_SMPTIM1_SMPCYCCFG17_2                                   (0x4UL << ADC_SMPTIM1_SMPCYCCFG17_Pos)                         /*!< 0x00800000 */

/******************  Bit definition for ADC_SMPTIM2 register  *******************/
#define ADC_SMPTIM2_SMPCYCCFG0_Pos                                  (0U)
#define ADC_SMPTIM2_SMPCYCCFG0_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG0_Pos)                          /*!< 0x00000007 */
#define ADC_SMPTIM2_SMPCYCCFG0                                      ADC_SMPTIM2_SMPCYCCFG0_Msk                                     /*!<SMP0[2:0] bits (Channel 0 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG0_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG0_Pos)                          /*!< 0x00000001 */
#define ADC_SMPTIM2_SMPCYCCFG0_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG0_Pos)                          /*!< 0x00000002 */
#define ADC_SMPTIM2_SMPCYCCFG0_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG0_Pos)                          /*!< 0x00000004 */
#define ADC_SMPTIM2_SMPCYCCFG1_Pos                                  (3U)
#define ADC_SMPTIM2_SMPCYCCFG1_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG1_Pos)                          /*!< 0x00000038 */
#define ADC_SMPTIM2_SMPCYCCFG1                                      ADC_SMPTIM2_SMPCYCCFG1_Msk                                     /*!<SMP1[2:0] bits (Channel 1 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG1_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG1_Pos)                          /*!< 0x00000008 */
#define ADC_SMPTIM2_SMPCYCCFG1_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG1_Pos)                          /*!< 0x00000010 */
#define ADC_SMPTIM2_SMPCYCCFG1_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG1_Pos)                          /*!< 0x00000020 */
#define ADC_SMPTIM2_SMPCYCCFG2_Pos                                  (6U)
#define ADC_SMPTIM2_SMPCYCCFG2_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG2_Pos)                          /*!< 0x000001C0 */
#define ADC_SMPTIM2_SMPCYCCFG2                                      ADC_SMPTIM2_SMPCYCCFG2_Msk                                     /*!<SMP2[2:0] bits (Channel 2 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG2_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG2_Pos)                          /*!< 0x00000040 */
#define ADC_SMPTIM2_SMPCYCCFG2_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG2_Pos)                          /*!< 0x00000080 */
#define ADC_SMPTIM2_SMPCYCCFG2_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG2_Pos)                          /*!< 0x00000100 */
#define ADC_SMPTIM2_SMPCYCCFG3_Pos                                  (9U)
#define ADC_SMPTIM2_SMPCYCCFG3_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG3_Pos)                          /*!< 0x00000E00 */
#define ADC_SMPTIM2_SMPCYCCFG3                                      ADC_SMPTIM2_SMPCYCCFG3_Msk                                     /*!<SMP3[2:0] bits (Channel 3 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG3_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG3_Pos)                          /*!< 0x00000200 */
#define ADC_SMPTIM2_SMPCYCCFG3_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG3_Pos)                          /*!< 0x00000400 */
#define ADC_SMPTIM2_SMPCYCCFG3_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG3_Pos)                          /*!< 0x00000800 */
#define ADC_SMPTIM2_SMPCYCCFG4_Pos                                  (12U)
#define ADC_SMPTIM2_SMPCYCCFG4_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG4_Pos)                          /*!< 0x00007000 */
#define ADC_SMPTIM2_SMPCYCCFG4                                      ADC_SMPTIM2_SMPCYCCFG4_Msk                                     /*!<SMP4[2:0] bits (Channel 4 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG4_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG4_Pos)                          /*!< 0x00001000 */
#define ADC_SMPTIM2_SMPCYCCFG4_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG4_Pos)                          /*!< 0x00002000 */
#define ADC_SMPTIM2_SMPCYCCFG4_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG4_Pos)                          /*!< 0x00004000 */
#define ADC_SMPTIM2_SMPCYCCFG5_Pos                                  (15U)
#define ADC_SMPTIM2_SMPCYCCFG5_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG5_Pos)                          /*!< 0x00038000 */
#define ADC_SMPTIM2_SMPCYCCFG5                                      ADC_SMPTIM2_SMPCYCCFG5_Msk                                     /*!<SMP5[2:0] bits (Channel 5 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG5_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG5_Pos)                          /*!< 0x00008000 */
#define ADC_SMPTIM2_SMPCYCCFG5_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG5_Pos)                          /*!< 0x00010000 */
#define ADC_SMPTIM2_SMPCYCCFG5_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG5_Pos)                          /*!< 0x00020000 */
#define ADC_SMPTIM2_SMPCYCCFG6_Pos                                  (18U)
#define ADC_SMPTIM2_SMPCYCCFG6_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG6_Pos)                          /*!< 0x001C0000 */
#define ADC_SMPTIM2_SMPCYCCFG6                                      ADC_SMPTIM2_SMPCYCCFG6_Msk                                     /*!<SMP6[2:0] bits (Channel 6 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG6_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG6_Pos)                          /*!< 0x00040000 */
#define ADC_SMPTIM2_SMPCYCCFG6_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG6_Pos)                          /*!< 0x00080000 */
#define ADC_SMPTIM2_SMPCYCCFG6_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG6_Pos)                          /*!< 0x00100000 */
#define ADC_SMPTIM2_SMPCYCCFG7_Pos                                  (21U)
#define ADC_SMPTIM2_SMPCYCCFG7_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG7_Pos)                          /*!< 0x00E00000 */
#define ADC_SMPTIM2_SMPCYCCFG7                                      ADC_SMPTIM2_SMPCYCCFG7_Msk                                     /*!<SMP7[2:0] bits (Channel 7 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG7_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG7_Pos)                          /*!< 0x00200000 */
#define ADC_SMPTIM2_SMPCYCCFG7_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG7_Pos)                          /*!< 0x00400000 */
#define ADC_SMPTIM2_SMPCYCCFG7_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG7_Pos)                          /*!< 0x00800000 */
#define ADC_SMPTIM2_SMPCYCCFG8_Pos                                  (24U)
#define ADC_SMPTIM2_SMPCYCCFG8_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG8_Pos)                          /*!< 0x07000000 */
#define ADC_SMPTIM2_SMPCYCCFG8                                      ADC_SMPTIM2_SMPCYCCFG8_Msk                                     /*!<SMP8[2:0] bits (Channel 8 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG8_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG8_Pos)                          /*!< 0x01000000 */
#define ADC_SMPTIM2_SMPCYCCFG8_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG8_Pos)                          /*!< 0x02000000 */
#define ADC_SMPTIM2_SMPCYCCFG8_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG8_Pos)                          /*!< 0x04000000 */
#define ADC_SMPTIM2_SMPCYCCFG9_Pos                                  (27U)
#define ADC_SMPTIM2_SMPCYCCFG9_Msk                                  (0x7UL << ADC_SMPTIM2_SMPCYCCFG9_Pos)                          /*!< 0x38000000 */
#define ADC_SMPTIM2_SMPCYCCFG9                                      ADC_SMPTIM2_SMPCYCCFG9_Msk                                     /*!<SMP9[2:0] bits (Channel 9 Sample time selection) */
#define ADC_SMPTIM2_SMPCYCCFG9_0                                    (0x1UL << ADC_SMPTIM2_SMPCYCCFG9_Pos)                          /*!< 0x08000000 */
#define ADC_SMPTIM2_SMPCYCCFG9_1                                    (0x2UL << ADC_SMPTIM2_SMPCYCCFG9_Pos)                          /*!< 0x10000000 */
#define ADC_SMPTIM2_SMPCYCCFG9_2                                    (0x4UL << ADC_SMPTIM2_SMPCYCCFG9_Pos)                          /*!< 0x20000000 */

/******************  Bit definition for ADC_INJDOF1 register  *******************/
#define ADC_INJDOF1_INJDOF1_Pos                                     (0U)
#define ADC_INJDOF1_INJDOF1_Msk                                     (0xFFFUL << ADC_INJDOF1_INJDOF1_Pos)                           /*!< 0x00000FFF */
#define ADC_INJDOF1_INJDOF1                                         ADC_INJDOF1_INJDOF1_Msk                                        /*!<Data offset for injected channel 1 */

/******************  Bit definition for ADC_INJDOF2 register  *******************/
#define ADC_INJDOF2_INJDOF2_Pos                                     (0U)
#define ADC_INJDOF2_INJDOF2_Msk                                     (0xFFFUL << ADC_INJDOF2_INJDOF2_Pos)                           /*!< 0x00000FFF */
#define ADC_INJDOF2_INJDOF2                                         ADC_INJDOF2_INJDOF2_Msk                                        /*!<Data offset for injected channel 2 */

/******************  Bit definition for ADC_INJDOF3 register  *******************/
#define ADC_INJDOF3_INJDOF3_Pos                                     (0U)
#define ADC_INJDOF3_INJDOF3_Msk                                     (0xFFFUL << ADC_INJDOF3_INJDOF3_Pos)                           /*!< 0x00000FFF */
#define ADC_INJDOF3_INJDOF3                                         ADC_INJDOF3_INJDOF3_Msk                                        /*!<Data offset for injected channel 3 */

/******************  Bit definition for ADC_INJDOF4 register  *******************/
#define ADC_INJDOF4_INJDOF4_Pos                                     (0U)
#define ADC_INJDOF4_INJDOF4_Msk                                     (0xFFFUL << ADC_INJDOF4_INJDOF4_Pos)                           /*!< 0x00000FFF */
#define ADC_INJDOF4_INJDOF4                                         ADC_INJDOF4_INJDOF4_Msk                                        /*!<Data offset for injected channel 4 */

/*******************  Bit definition for ADC_AWDHT register  ********************/
#define ADC_AWDHT_AWDHT_Pos                                         (0U)
#define ADC_AWDHT_AWDHT_Msk                                         (0xFFFUL << ADC_AWDHT_AWDHT_Pos)                               /*!< 0x00000FFF */
#define ADC_AWDHT_AWDHT                                             ADC_AWDHT_AWDHT_Msk                                            /*!<Analog watchdog high threshold */

/*******************  Bit definition for ADC_AWDLT register  ********************/
#define ADC_AWDLT_AWDLT_Pos                                         (0U)
#define ADC_AWDLT_AWDLT_Msk                                         (0xFFFUL << ADC_AWDLT_AWDLT_Pos)                               /*!< 0x00000FFF */
#define ADC_AWDLT_AWDLT                                             ADC_AWDLT_AWDLT_Msk                                            /*!<Analog watchdog low threshold */

/*******************  Bit definition for ADC_REGSEQ1 register  *******************/
#define ADC_REGSEQ1_REGSEQC13_Pos                                   (0U)
#define ADC_REGSEQ1_REGSEQC13_Msk                                   (0x1FUL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x0000001F */
#define ADC_REGSEQ1_REGSEQC13                                       ADC_REGSEQ1_REGSEQC13_Msk                                      /*!<SQ13[4:0] bits (13th conversion in regular sequence) */
#define ADC_REGSEQ1_REGSEQC13_0                                     (0x01UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000001 */
#define ADC_REGSEQ1_REGSEQC13_1                                     (0x02UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000002 */
#define ADC_REGSEQ1_REGSEQC13_2                                     (0x04UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000004 */
#define ADC_REGSEQ1_REGSEQC13_3                                     (0x08UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000008 */
#define ADC_REGSEQ1_REGSEQC13_4                                     (0x10UL << ADC_REGSEQ1_REGSEQC13_Pos)                          /*!< 0x00000010 */
#define ADC_REGSEQ1_REGSEQC14_Pos                                   (5U)
#define ADC_REGSEQ1_REGSEQC14_Msk                                   (0x1FUL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x000003E0 */
#define ADC_REGSEQ1_REGSEQC14                                       ADC_REGSEQ1_REGSEQC14_Msk                                      /*!<SQ14[4:0] bits (14th conversion in regular sequence) */
#define ADC_REGSEQ1_REGSEQC14_0                                     (0x01UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000020 */
#define ADC_REGSEQ1_REGSEQC14_1                                     (0x02UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000040 */
#define ADC_REGSEQ1_REGSEQC14_2                                     (0x04UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000080 */
#define ADC_REGSEQ1_REGSEQC14_3                                     (0x08UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000100 */
#define ADC_REGSEQ1_REGSEQC14_4                                     (0x10UL << ADC_REGSEQ1_REGSEQC14_Pos)                          /*!< 0x00000200 */
#define ADC_REGSEQ1_REGSEQC15_Pos                                   (10U)
#define ADC_REGSEQ1_REGSEQC15_Msk                                   (0x1FUL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00007C00 */
#define ADC_REGSEQ1_REGSEQC15                                       ADC_REGSEQ1_REGSEQC15_Msk                                      /*!<SQ15[4:0] bits (15th conversion in regular sequence) */
#define ADC_REGSEQ1_REGSEQC15_0                                     (0x01UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00000400 */
#define ADC_REGSEQ1_REGSEQC15_1                                     (0x02UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00000800 */
#define ADC_REGSEQ1_REGSEQC15_2                                     (0x04UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00001000 */
#define ADC_REGSEQ1_REGSEQC15_3                                     (0x08UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00002000 */
#define ADC_REGSEQ1_REGSEQC15_4                                     (0x10UL << ADC_REGSEQ1_REGSEQC15_Pos)                          /*!< 0x00004000 */
#define ADC_REGSEQ1_REGSEQC16_Pos                                   (15U)
#define ADC_REGSEQ1_REGSEQC16_Msk                                   (0x1FUL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x000F8000 */
#define ADC_REGSEQ1_REGSEQC16                                       ADC_REGSEQ1_REGSEQC16_Msk                                      /*!<SQ16[4:0] bits (16th conversion in regular sequence) */
#define ADC_REGSEQ1_REGSEQC16_0                                     (0x01UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00008000 */
#define ADC_REGSEQ1_REGSEQC16_1                                     (0x02UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00010000 */
#define ADC_REGSEQ1_REGSEQC16_2                                     (0x04UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00020000 */
#define ADC_REGSEQ1_REGSEQC16_3                                     (0x08UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00040000 */
#define ADC_REGSEQ1_REGSEQC16_4                                     (0x10UL << ADC_REGSEQ1_REGSEQC16_Pos)                          /*!< 0x00080000 */
#define ADC_REGSEQ1_REGSEQLEN_Pos                                   (20U)
#define ADC_REGSEQ1_REGSEQLEN_Msk                                   (0xFUL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00F00000 */
#define ADC_REGSEQ1_REGSEQLEN                                       ADC_REGSEQ1_REGSEQLEN_Msk                                      /*!<L[3:0] bits (Regular channel sequence length) */
#define ADC_REGSEQ1_REGSEQLEN_0                                     (0x1UL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00100000 */
#define ADC_REGSEQ1_REGSEQLEN_1                                     (0x2UL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00200000 */
#define ADC_REGSEQ1_REGSEQLEN_2                                     (0x4UL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00400000 */
#define ADC_REGSEQ1_REGSEQLEN_3                                     (0x8UL << ADC_REGSEQ1_REGSEQLEN_Pos)                           /*!< 0x00800000 */

/*******************  Bit definition for ADC_REGSEQ2 register  *******************/
#define ADC_REGSEQ2_REGSEQC7_Pos                                    (0U)
#define ADC_REGSEQ2_REGSEQC7_Msk                                    (0x1FUL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x0000001F */
#define ADC_REGSEQ2_REGSEQC7                                        ADC_REGSEQ2_REGSEQC7_Msk                                       /*!<SQ7[4:0] bits (7th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC7_0                                      (0x01UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000001 */
#define ADC_REGSEQ2_REGSEQC7_1                                      (0x02UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000002 */
#define ADC_REGSEQ2_REGSEQC7_2                                      (0x04UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000004 */
#define ADC_REGSEQ2_REGSEQC7_3                                      (0x08UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000008 */
#define ADC_REGSEQ2_REGSEQC7_4                                      (0x10UL << ADC_REGSEQ2_REGSEQC7_Pos)                           /*!< 0x00000010 */
#define ADC_REGSEQ2_REGSEQC8_Pos                                    (5U)
#define ADC_REGSEQ2_REGSEQC8_Msk                                    (0x1FUL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x000003E0 */
#define ADC_REGSEQ2_REGSEQC8                                        ADC_REGSEQ2_REGSEQC8_Msk                                       /*!<SQ8[4:0] bits (8th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC8_0                                      (0x01UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000020 */
#define ADC_REGSEQ2_REGSEQC8_1                                      (0x02UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000040 */
#define ADC_REGSEQ2_REGSEQC8_2                                      (0x04UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000080 */
#define ADC_REGSEQ2_REGSEQC8_3                                      (0x08UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000100 */
#define ADC_REGSEQ2_REGSEQC8_4                                      (0x10UL << ADC_REGSEQ2_REGSEQC8_Pos)                           /*!< 0x00000200 */
#define ADC_REGSEQ2_REGSEQC9_Pos                                    (10U)
#define ADC_REGSEQ2_REGSEQC9_Msk                                    (0x1FUL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00007C00 */
#define ADC_REGSEQ2_REGSEQC9                                        ADC_REGSEQ2_REGSEQC9_Msk                                       /*!<SQ9[4:0] bits (9th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC9_0                                      (0x01UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00000400 */
#define ADC_REGSEQ2_REGSEQC9_1                                      (0x02UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00000800 */
#define ADC_REGSEQ2_REGSEQC9_2                                      (0x04UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00001000 */
#define ADC_REGSEQ2_REGSEQC9_3                                      (0x08UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00002000 */
#define ADC_REGSEQ2_REGSEQC9_4                                      (0x10UL << ADC_REGSEQ2_REGSEQC9_Pos)                           /*!< 0x00004000 */
#define ADC_REGSEQ2_REGSEQC10_Pos                                   (15U)
#define ADC_REGSEQ2_REGSEQC10_Msk                                   (0x1FUL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x000F8000 */
#define ADC_REGSEQ2_REGSEQC10                                       ADC_REGSEQ2_REGSEQC10_Msk                                      /*!<SQ10[4:0] bits (10th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC10_0                                     (0x01UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00008000 */
#define ADC_REGSEQ2_REGSEQC10_1                                     (0x02UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00010000 */
#define ADC_REGSEQ2_REGSEQC10_2                                     (0x04UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00020000 */
#define ADC_REGSEQ2_REGSEQC10_3                                     (0x08UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00040000 */
#define ADC_REGSEQ2_REGSEQC10_4                                     (0x10UL << ADC_REGSEQ2_REGSEQC10_Pos)                          /*!< 0x00080000 */
#define ADC_REGSEQ2_REGSEQC11_Pos                                   (20U)
#define ADC_REGSEQ2_REGSEQC11_Msk                                   (0x1FUL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x01F00000 */
#define ADC_REGSEQ2_REGSEQC11                                       ADC_REGSEQ2_REGSEQC11_Msk                                      /*!<SQ11[4:0] bits (11th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC11_0                                     (0x01UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x00100000 */
#define ADC_REGSEQ2_REGSEQC11_1                                     (0x02UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x00200000 */
#define ADC_REGSEQ2_REGSEQC11_2                                     (0x04UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x00400000 */
#define ADC_REGSEQ2_REGSEQC11_3                                     (0x08UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x00800000 */
#define ADC_REGSEQ2_REGSEQC11_4                                     (0x10UL << ADC_REGSEQ2_REGSEQC11_Pos)                          /*!< 0x01000000 */
#define ADC_REGSEQ2_REGSEQC12_Pos                                   (25U)
#define ADC_REGSEQ2_REGSEQC12_Msk                                   (0x1FUL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x3E000000 */
#define ADC_REGSEQ2_REGSEQC12                                       ADC_REGSEQ2_REGSEQC12_Msk                                      /*!<SQ12[4:0] bits (12th conversion in regular sequence) */
#define ADC_REGSEQ2_REGSEQC12_0                                     (0x01UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x02000000 */
#define ADC_REGSEQ2_REGSEQC12_1                                     (0x02UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x04000000 */
#define ADC_REGSEQ2_REGSEQC12_2                                     (0x04UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x08000000 */
#define ADC_REGSEQ2_REGSEQC12_3                                     (0x08UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x10000000 */
#define ADC_REGSEQ2_REGSEQC12_4                                     (0x10UL << ADC_REGSEQ2_REGSEQC12_Pos)                          /*!< 0x20000000 */

/*******************  Bit definition for ADC_REGSEQ3 register  *******************/
#define ADC_REGSEQ3_REGSEQC1_Pos                                    (0U)
#define ADC_REGSEQ3_REGSEQC1_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x0000001F */
#define ADC_REGSEQ3_REGSEQC1                                        ADC_REGSEQ3_REGSEQC1_Msk                                       /*!<SQ1[4:0] bits (1st conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC1_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000001 */
#define ADC_REGSEQ3_REGSEQC1_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000002 */
#define ADC_REGSEQ3_REGSEQC1_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000004 */
#define ADC_REGSEQ3_REGSEQC1_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000008 */
#define ADC_REGSEQ3_REGSEQC1_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC1_Pos)                           /*!< 0x00000010 */
#define ADC_REGSEQ3_REGSEQC2_Pos                                    (5U)
#define ADC_REGSEQ3_REGSEQC2_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x000003E0 */
#define ADC_REGSEQ3_REGSEQC2                                        ADC_REGSEQ3_REGSEQC2_Msk                                       /*!<SQ2[4:0] bits (2nd conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC2_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000020 */
#define ADC_REGSEQ3_REGSEQC2_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000040 */
#define ADC_REGSEQ3_REGSEQC2_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000080 */
#define ADC_REGSEQ3_REGSEQC2_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000100 */
#define ADC_REGSEQ3_REGSEQC2_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC2_Pos)                           /*!< 0x00000200 */
#define ADC_REGSEQ3_REGSEQC3_Pos                                    (10U)
#define ADC_REGSEQ3_REGSEQC3_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00007C00 */
#define ADC_REGSEQ3_REGSEQC3                                        ADC_REGSEQ3_REGSEQC3_Msk                                       /*!<SQ3[4:0] bits (3rd conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC3_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00000400 */
#define ADC_REGSEQ3_REGSEQC3_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00000800 */
#define ADC_REGSEQ3_REGSEQC3_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00001000 */
#define ADC_REGSEQ3_REGSEQC3_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00002000 */
#define ADC_REGSEQ3_REGSEQC3_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC3_Pos)                           /*!< 0x00004000 */
#define ADC_REGSEQ3_REGSEQC4_Pos                                    (15U)
#define ADC_REGSEQ3_REGSEQC4_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x000F8000 */
#define ADC_REGSEQ3_REGSEQC4                                        ADC_REGSEQ3_REGSEQC4_Msk                                       /*!<SQ4[4:0] bits (4th conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC4_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00008000 */
#define ADC_REGSEQ3_REGSEQC4_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00010000 */
#define ADC_REGSEQ3_REGSEQC4_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00020000 */
#define ADC_REGSEQ3_REGSEQC4_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00040000 */
#define ADC_REGSEQ3_REGSEQC4_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC4_Pos)                           /*!< 0x00080000 */
#define ADC_REGSEQ3_REGSEQC5_Pos                                    (20U)
#define ADC_REGSEQ3_REGSEQC5_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x01F00000 */
#define ADC_REGSEQ3_REGSEQC5                                        ADC_REGSEQ3_REGSEQC5_Msk                                       /*!<SQ5[4:0] bits (5th conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC5_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x00100000 */
#define ADC_REGSEQ3_REGSEQC5_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x00200000 */
#define ADC_REGSEQ3_REGSEQC5_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x00400000 */
#define ADC_REGSEQ3_REGSEQC5_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x00800000 */
#define ADC_REGSEQ3_REGSEQC5_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC5_Pos)                           /*!< 0x01000000 */
#define ADC_REGSEQ3_REGSEQC6_Pos                                    (25U)
#define ADC_REGSEQ3_REGSEQC6_Msk                                    (0x1FUL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x3E000000 */
#define ADC_REGSEQ3_REGSEQC6                                        ADC_REGSEQ3_REGSEQC6_Msk                                       /*!<SQ6[4:0] bits (6th conversion in regular sequence) */
#define ADC_REGSEQ3_REGSEQC6_0                                      (0x01UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x02000000 */
#define ADC_REGSEQ3_REGSEQC6_1                                      (0x02UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x04000000 */
#define ADC_REGSEQ3_REGSEQC6_2                                      (0x04UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x08000000 */
#define ADC_REGSEQ3_REGSEQC6_3                                      (0x08UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x10000000 */
#define ADC_REGSEQ3_REGSEQC6_4                                      (0x10UL << ADC_REGSEQ3_REGSEQC6_Pos)                           /*!< 0x20000000 */

/*******************  Bit definition for ADC_INJSEQ register  *******************/
#define ADC_INJSEQ_INJSEQC1_Pos                                     (0U)
#define ADC_INJSEQ_INJSEQC1_Msk                                     (0x1FUL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x0000001F */
#define ADC_INJSEQ_INJSEQC1                                         ADC_INJSEQ_INJSEQC1_Msk                                        /*!<JSQ1[4:0] bits (1st conversion in injected sequence) */
#define ADC_INJSEQ_INJSEQC1_0                                       (0x01UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000001 */
#define ADC_INJSEQ_INJSEQC1_1                                       (0x02UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000002 */
#define ADC_INJSEQ_INJSEQC1_2                                       (0x04UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000004 */
#define ADC_INJSEQ_INJSEQC1_3                                       (0x08UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000008 */
#define ADC_INJSEQ_INJSEQC1_4                                       (0x10UL << ADC_INJSEQ_INJSEQC1_Pos)                            /*!< 0x00000010 */
#define ADC_INJSEQ_INJSEQC2_Pos                                     (5U)
#define ADC_INJSEQ_INJSEQC2_Msk                                     (0x1FUL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x000003E0 */
#define ADC_INJSEQ_INJSEQC2                                         ADC_INJSEQ_INJSEQC2_Msk                                        /*!<JSQ2[4:0] bits (2nd conversion in injected sequence) */
#define ADC_INJSEQ_INJSEQC2_0                                       (0x01UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000020 */
#define ADC_INJSEQ_INJSEQC2_1                                       (0x02UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000040 */
#define ADC_INJSEQ_INJSEQC2_2                                       (0x04UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000080 */
#define ADC_INJSEQ_INJSEQC2_3                                       (0x08UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000100 */
#define ADC_INJSEQ_INJSEQC2_4                                       (0x10UL << ADC_INJSEQ_INJSEQC2_Pos)                            /*!< 0x00000200 */
#define ADC_INJSEQ_INJSEQC3_Pos                                     (10U)
#define ADC_INJSEQ_INJSEQC3_Msk                                     (0x1FUL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00007C00 */
#define ADC_INJSEQ_INJSEQC3                                         ADC_INJSEQ_INJSEQC3_Msk                                        /*!<JSQ3[4:0] bits (3rd conversion in injected sequence) */
#define ADC_INJSEQ_INJSEQC3_0                                       (0x01UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00000400 */
#define ADC_INJSEQ_INJSEQC3_1                                       (0x02UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00000800 */
#define ADC_INJSEQ_INJSEQC3_2                                       (0x04UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00001000 */
#define ADC_INJSEQ_INJSEQC3_3                                       (0x08UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00002000 */
#define ADC_INJSEQ_INJSEQC3_4                                       (0x10UL << ADC_INJSEQ_INJSEQC3_Pos)                            /*!< 0x00004000 */
#define ADC_INJSEQ_INJSEQC4_Pos                                     (15U)
#define ADC_INJSEQ_INJSEQC4_Msk                                     (0x1FUL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x000F8000 */
#define ADC_INJSEQ_INJSEQC4                                         ADC_INJSEQ_INJSEQC4_Msk                                        /*!<JSQ4[4:0] bits (4th conversion in injected sequence) */
#define ADC_INJSEQ_INJSEQC4_0                                       (0x01UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00008000 */
#define ADC_INJSEQ_INJSEQC4_1                                       (0x02UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00010000 */
#define ADC_INJSEQ_INJSEQC4_2                                       (0x04UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00020000 */
#define ADC_INJSEQ_INJSEQC4_3                                       (0x08UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00040000 */
#define ADC_INJSEQ_INJSEQC4_4                                       (0x10UL << ADC_INJSEQ_INJSEQC4_Pos)                            /*!< 0x00080000 */
#define ADC_INJSEQ_INJSEQLEN_Pos                                    (20U)
#define ADC_INJSEQ_INJSEQLEN_Msk                                    (0x3UL << ADC_INJSEQ_INJSEQLEN_Pos)                            /*!< 0x00300000 */
#define ADC_INJSEQ_INJSEQLEN                                        ADC_INJSEQ_INJSEQLEN_Msk                                       /*!<JL[1:0] bits (Injected Sequence length) */
#define ADC_INJSEQ_INJSEQLEN_0                                      (0x1UL << ADC_INJSEQ_INJSEQLEN_Pos)                            /*!< 0x00100000 */
#define ADC_INJSEQ_INJSEQLEN_1                                      (0x2UL << ADC_INJSEQ_INJSEQLEN_Pos)                            /*!< 0x00200000 */

/*******************  Bit definition for ADC_INJDATA1 register  *******************/
#define ADC_INJDATA1_INJDATA_Pos                                    (0U)
#define ADC_INJDATA1_INJDATA_Msk                                    (0xFFFFUL << ADC_INJDATA1_INJDATA_Pos)                         /*!< 0x0000FFFF */
#define ADC_INJDATA1_INJDATA                                        ADC_INJDATA1_INJDATA_Msk                                       /*!<Injected data */

/*******************  Bit definition for ADC_INJDATA2 register  *******************/
#define ADC_INJDATA2_INJDATA_Pos                                    (0U)
#define ADC_INJDATA2_INJDATA_Msk                                    (0xFFFFUL << ADC_INJDATA2_INJDATA_Pos)                         /*!< 0x0000FFFF */
#define ADC_INJDATA2_INJDATA                                        ADC_INJDATA2_INJDATA_Msk                                       /*!<Injected data */

/*******************  Bit definition for ADC_INJDATA3 register  *******************/
#define ADC_INJDATA3_INJDATA_Pos                                    (0U)
#define ADC_INJDATA3_INJDATA_Msk                                    (0xFFFFUL << ADC_INJDATA3_INJDATA_Pos)                         /*!< 0x0000FFFF */
#define ADC_INJDATA3_INJDATA                                        ADC_INJDATA3_INJDATA_Msk                                       /*!<Injected data */

/*******************  Bit definition for ADC_INJDATA4 register  *******************/
#define ADC_INJDATA4_INJDATA_Pos                                    (0U)
#define ADC_INJDATA4_INJDATA_Msk                                    (0xFFFFUL << ADC_INJDATA4_INJDATA_Pos)                         /*!< 0x0000FFFF */
#define ADC_INJDATA4_INJDATA                                        ADC_INJDATA4_INJDATA_Msk                                       /*!<Injected data */

/********************  Bit definition for ADC_REGDATA register  ********************/
#define ADC_REGDATA_REGDATA_Pos                                     (0U)
#define ADC_REGDATA_REGDATA_Msk                                     (0xFFFFUL << ADC_REGDATA_REGDATA_Pos)                          /*!< 0x0000FFFF */
#define ADC_REGDATA_REGDATA                                         ADC_REGDATA_REGDATA_Msk                                        /*!<Regular data */
#define ADC_REGDATA_ADC2DATA_Pos                                    (16U)
#define ADC_REGDATA_ADC2DATA_Msk                                    (0xFFFFUL << ADC_REGDATA_ADC2DATA_Pos)                         /*!< 0xFFFF0000 */
#define ADC_REGDATA_ADC2DATA                                        ADC_REGDATA_ADC2DATA_Msk                                       /*!<ADC2 data */

/******************************************************************************/
/*                                                                            */
/*                         Controller Area Network                            */
/*                                                                            */
/******************************************************************************/
/*!<CAN control and status registers */
/*******************  Bit definition for CAN_MCTRL register  ********************/
#define CAN_MCTRL_INITREQ_Pos                                       (0U)
#define CAN_MCTRL_INITREQ_Msk                                       (0x1UL << CAN_MCTRL_INITREQ_Pos)                               /*!< 0x00000001 */
#define CAN_MCTRL_INITREQ                                           CAN_MCTRL_INITREQ_Msk                                          /*!<Initialization Request */
#define CAN_MCTRL_SLEEPREQ_Pos                                      (1U)
#define CAN_MCTRL_SLEEPREQ_Msk                                      (0x1UL << CAN_MCTRL_SLEEPREQ_Pos)                              /*!< 0x00000002 */
#define CAN_MCTRL_SLEEPREQ                                          CAN_MCTRL_SLEEPREQ_Msk                                         /*!<Sleep Mode Request */
#define CAN_MCTRL_TXFPCFG_Pos                                       (2U)
#define CAN_MCTRL_TXFPCFG_Msk                                       (0x1UL << CAN_MCTRL_TXFPCFG_Pos)                               /*!< 0x00000004 */
#define CAN_MCTRL_TXFPCFG                                           CAN_MCTRL_TXFPCFG_Msk                                          /*!<Transmit FIFO Priority */
#define CAN_MCTRL_RXFLOCK_Pos                                       (3U)
#define CAN_MCTRL_RXFLOCK_Msk                                       (0x1UL << CAN_MCTRL_RXFLOCK_Pos)                               /*!< 0x00000008 */
#define CAN_MCTRL_RXFLOCK                                           CAN_MCTRL_RXFLOCK_Msk                                          /*!<Receive FIFO Locked Mode */
#define CAN_MCTRL_ARTXMD_Pos                                        (4U)
#define CAN_MCTRL_ARTXMD_Msk                                        (0x1UL << CAN_MCTRL_ARTXMD_Pos)                                /*!< 0x00000010 */
#define CAN_MCTRL_ARTXMD                                            CAN_MCTRL_ARTXMD_Msk                                           /*!<No Automatic Retransmission */
#define CAN_MCTRL_AWUPCFG_Pos                                       (5U)
#define CAN_MCTRL_AWUPCFG_Msk                                       (0x1UL << CAN_MCTRL_AWUPCFG_Pos)                               /*!< 0x00000020 */
#define CAN_MCTRL_AWUPCFG                                           CAN_MCTRL_AWUPCFG_Msk                                          /*!<Automatic Wakeup Mode */
#define CAN_MCTRL_ALBOFFM_Pos                                       (6U)
#define CAN_MCTRL_ALBOFFM_Msk                                       (0x1UL << CAN_MCTRL_ALBOFFM_Pos)                               /*!< 0x00000040 */
#define CAN_MCTRL_ALBOFFM                                           CAN_MCTRL_ALBOFFM_Msk                                          /*!<Automatic Bus-Off Management */
#define CAN_MCTRL_TTCM_Pos                                          (7U)
#define CAN_MCTRL_TTCM_Msk                                          (0x1UL << CAN_MCTRL_TTCM_Pos)                                  /*!< 0x00000080 */
#define CAN_MCTRL_TTCM                                              CAN_MCTRL_TTCM_Msk                                             /*!<Time Triggered Communication Mode */
#define CAN_MCTRL_SWRST_Pos                                         (15U)
#define CAN_MCTRL_SWRST_Msk                                         (0x1UL << CAN_MCTRL_SWRST_Pos)                                 /*!< 0x00008000 */
#define CAN_MCTRL_SWRST                                             CAN_MCTRL_SWRST_Msk                                            /*!<bxCAN software master reset */
#define CAN_MCTRL_DBGFRZE_Pos                                       (16U)
#define CAN_MCTRL_DBGFRZE_Msk                                       (0x1UL << CAN_MCTRL_DBGFRZE_Pos)                               /*!< 0x00010000 */
#define CAN_MCTRL_DBGFRZE                                           CAN_MCTRL_DBGFRZE_Msk                                          /*!<bxCAN Debug freeze */
/*******************  Bit definition for CAN_MSTS register  ********************/
#define CAN_MSTS_INITFLG_Pos                                        (0U)
#define CAN_MSTS_INITFLG_Msk                                        (0x1UL << CAN_MSTS_INITFLG_Pos)                                /*!< 0x00000001 */
#define CAN_MSTS_INITFLG                                            CAN_MSTS_INITFLG_Msk                                           /*!<Initialization Acknowledge */
#define CAN_MSTS_SLEEPFLG_Pos                                       (1U)
#define CAN_MSTS_SLEEPFLG_Msk                                       (0x1UL << CAN_MSTS_SLEEPFLG_Pos)                               /*!< 0x00000002 */
#define CAN_MSTS_SLEEPFLG                                           CAN_MSTS_SLEEPFLG_Msk                                          /*!<Sleep Acknowledge */
#define CAN_MSTS_ERRIFLG_Pos                                        (2U)
#define CAN_MSTS_ERRIFLG_Msk                                        (0x1UL << CAN_MSTS_ERRIFLG_Pos)                                /*!< 0x00000004 */
#define CAN_MSTS_ERRIFLG                                            CAN_MSTS_ERRIFLG_Msk                                           /*!<Error Interrupt */
#define CAN_MSTS_WUPIFLG_Pos                                        (3U)
#define CAN_MSTS_WUPIFLG_Msk                                        (0x1UL << CAN_MSTS_WUPIFLG_Pos)                                /*!< 0x00000008 */
#define CAN_MSTS_WUPIFLG                                            CAN_MSTS_WUPIFLG_Msk                                           /*!<Wakeup Interrupt */
#define CAN_MSTS_SLEEPIFLG_Pos                                      (4U)
#define CAN_MSTS_SLEEPIFLG_Msk                                      (0x1UL << CAN_MSTS_SLEEPIFLG_Pos)                              /*!< 0x00000010 */
#define CAN_MSTS_SLEEPIFLG                                          CAN_MSTS_SLEEPIFLG_Msk                                         /*!<Sleep Acknowledge Interrupt */
#define CAN_MSTS_TXMFLG_Pos                                         (8U)
#define CAN_MSTS_TXMFLG_Msk                                         (0x1UL << CAN_MSTS_TXMFLG_Pos)                                 /*!< 0x00000100 */
#define CAN_MSTS_TXMFLG                                             CAN_MSTS_TXMFLG_Msk                                            /*!<Transmit Mode */
#define CAN_MSTS_RXSIGLMFLG_Pos                                     (9U)
#define CAN_MSTS_RXSIGLMFLG_Msk                                     (0x1UL << CAN_MSTS_RXSIGLMFLG_Pos)                             /*!< 0x00000200 */
#define CAN_MSTS_RXSIGLMFLG                                         CAN_MSTS_RXSIGLMFLG_Msk                                        /*!<Receive Mode */
#define CAN_MSTS_LSAMVALUE_Pos                                      (10U)
#define CAN_MSTS_LSAMVALUE_Msk                                      (0x1UL << CAN_MSTS_LSAMVALUE_Pos)                              /*!< 0x00000400 */
#define CAN_MSTS_LSAMVALUE                                          CAN_MSTS_LSAMVALUE_Msk                                         /*!<Last Sample Point */
#define CAN_MSTS_RXSIGL_Pos                                         (11U)
#define CAN_MSTS_RXSIGL_Msk                                         (0x1UL << CAN_MSTS_RXSIGL_Pos)                                 /*!< 0x00000800 */
#define CAN_MSTS_RXSIGL                                             CAN_MSTS_RXSIGL_Msk                                            /*!<CAN Rx Signal */

/*******************  Bit definition for CAN_TXSTS register  ********************/
#define CAN_TXSTS_REQCFLG0_Pos                                      (0U)
#define CAN_TXSTS_REQCFLG0_Msk                                      (0x1UL << CAN_TXSTS_REQCFLG0_Pos)                              /*!< 0x00000001 */
#define CAN_TXSTS_REQCFLG0                                          CAN_TXSTS_REQCFLG0_Msk                                         /*!<Request Completed Mailbox0 */
#define CAN_TXSTS_TXSUSFLG0_Pos                                     (1U)
#define CAN_TXSTS_TXSUSFLG0_Msk                                     (0x1UL << CAN_TXSTS_TXSUSFLG0_Pos)                             /*!< 0x00000002 */
#define CAN_TXSTS_TXSUSFLG0                                         CAN_TXSTS_TXSUSFLG0_Msk                                        /*!<Transmission OK of Mailbox0 */
#define CAN_TXSTS_ARBLSTFLG0_Pos                                    (2U)
#define CAN_TXSTS_ARBLSTFLG0_Msk                                    (0x1UL << CAN_TXSTS_ARBLSTFLG0_Pos)                            /*!< 0x00000004 */
#define CAN_TXSTS_ARBLSTFLG0                                        CAN_TXSTS_ARBLSTFLG0_Msk                                       /*!<Arbitration Lost for Mailbox0 */
#define CAN_TXSTS_TXERRFLG0_Pos                                     (3U)
#define CAN_TXSTS_TXERRFLG0_Msk                                     (0x1UL << CAN_TXSTS_TXERRFLG0_Pos)                             /*!< 0x00000008 */
#define CAN_TXSTS_TXERRFLG0                                         CAN_TXSTS_TXERRFLG0_Msk                                        /*!<Transmission Error of Mailbox0 */
#define CAN_TXSTS_ABREQFLG0_Pos                                     (7U)
#define CAN_TXSTS_ABREQFLG0_Msk                                     (0x1UL << CAN_TXSTS_ABREQFLG0_Pos)                             /*!< 0x00000080 */
#define CAN_TXSTS_ABREQFLG0                                         CAN_TXSTS_ABREQFLG0_Msk                                        /*!<Abort Request for Mailbox0 */
#define CAN_TXSTS_REQCFLG1_Pos                                      (8U)
#define CAN_TXSTS_REQCFLG1_Msk                                      (0x1UL << CAN_TXSTS_REQCFLG1_Pos)                              /*!< 0x00000100 */
#define CAN_TXSTS_REQCFLG1                                          CAN_TXSTS_REQCFLG1_Msk                                         /*!<Request Completed Mailbox1 */
#define CAN_TXSTS_TXSUSFLG1_Pos                                     (9U)
#define CAN_TXSTS_TXSUSFLG1_Msk                                     (0x1UL << CAN_TXSTS_TXSUSFLG1_Pos)                             /*!< 0x00000200 */
#define CAN_TXSTS_TXSUSFLG1                                         CAN_TXSTS_TXSUSFLG1_Msk                                        /*!<Transmission OK of Mailbox1 */
#define CAN_TXSTS_ARBLSTFLG1_Pos                                    (10U)
#define CAN_TXSTS_ARBLSTFLG1_Msk                                    (0x1UL << CAN_TXSTS_ARBLSTFLG1_Pos)                            /*!< 0x00000400 */
#define CAN_TXSTS_ARBLSTFLG1                                        CAN_TXSTS_ARBLSTFLG1_Msk                                       /*!<Arbitration Lost for Mailbox1 */
#define CAN_TXSTS_TXERRFLG1_Pos                                     (11U)
#define CAN_TXSTS_TXERRFLG1_Msk                                     (0x1UL << CAN_TXSTS_TXERRFLG1_Pos)                             /*!< 0x00000800 */
#define CAN_TXSTS_TXERRFLG1                                         CAN_TXSTS_TXERRFLG1_Msk                                        /*!<Transmission Error of Mailbox1 */
#define CAN_TXSTS_ABREQFLG1_Pos                                     (15U)
#define CAN_TXSTS_ABREQFLG1_Msk                                     (0x1UL << CAN_TXSTS_ABREQFLG1_Pos)                             /*!< 0x00008000 */
#define CAN_TXSTS_ABREQFLG1                                         CAN_TXSTS_ABREQFLG1_Msk                                        /*!<Abort Request for Mailbox 1 */
#define CAN_TXSTS_REQCFLG2_Pos                                      (16U)
#define CAN_TXSTS_REQCFLG2_Msk                                      (0x1UL << CAN_TXSTS_REQCFLG2_Pos)                              /*!< 0x00010000 */
#define CAN_TXSTS_REQCFLG2                                          CAN_TXSTS_REQCFLG2_Msk                                         /*!<Request Completed Mailbox2 */
#define CAN_TXSTS_TXSUSFLG2_Pos                                     (17U)
#define CAN_TXSTS_TXSUSFLG2_Msk                                     (0x1UL << CAN_TXSTS_TXSUSFLG2_Pos)                             /*!< 0x00020000 */
#define CAN_TXSTS_TXSUSFLG2                                         CAN_TXSTS_TXSUSFLG2_Msk                                        /*!<Transmission OK of Mailbox 2 */
#define CAN_TXSTS_ARBLSTFLG2_Pos                                    (18U)
#define CAN_TXSTS_ARBLSTFLG2_Msk                                    (0x1UL << CAN_TXSTS_ARBLSTFLG2_Pos)                            /*!< 0x00040000 */
#define CAN_TXSTS_ARBLSTFLG2                                        CAN_TXSTS_ARBLSTFLG2_Msk                                       /*!<Arbitration Lost for mailbox 2 */
#define CAN_TXSTS_TXERRFLG2_Pos                                     (19U)
#define CAN_TXSTS_TXERRFLG2_Msk                                     (0x1UL << CAN_TXSTS_TXERRFLG2_Pos)                             /*!< 0x00080000 */
#define CAN_TXSTS_TXERRFLG2                                         CAN_TXSTS_TXERRFLG2_Msk                                        /*!<Transmission Error of Mailbox 2 */
#define CAN_TXSTS_ABREQFLG2_Pos                                     (23U)
#define CAN_TXSTS_ABREQFLG2_Msk                                     (0x1UL << CAN_TXSTS_ABREQFLG2_Pos)                             /*!< 0x00800000 */
#define CAN_TXSTS_ABREQFLG2                                         CAN_TXSTS_ABREQFLG2_Msk                                        /*!<Abort Request for Mailbox 2 */
#define CAN_TXSTS_EMNUM_Pos                                         (24U)
#define CAN_TXSTS_EMNUM_Msk                                         (0x3UL << CAN_TXSTS_EMNUM_Pos)                                 /*!< 0x03000000 */
#define CAN_TXSTS_EMNUM                                             CAN_TXSTS_EMNUM_Msk                                            /*!<Mailbox Code */

#define CAN_TXSTS_TXMEFLG_Pos                                       (26U)
#define CAN_TXSTS_TXMEFLG_Msk                                       (0x7UL << CAN_TXSTS_TXMEFLG_Pos)                               /*!< 0x1C000000 */
#define CAN_TXSTS_TXMEFLG                                           CAN_TXSTS_TXMEFLG_Msk                                          /*!<TME[2:0] bits */
#define CAN_TXSTS_TXMEFLG0_Pos                                      (26U)
#define CAN_TXSTS_TXMEFLG0_Msk                                      (0x1UL << CAN_TXSTS_TXMEFLG0_Pos)                              /*!< 0x04000000 */
#define CAN_TXSTS_TXMEFLG0                                          CAN_TXSTS_TXMEFLG0_Msk                                         /*!<Transmit Mailbox 0 Empty */
#define CAN_TXSTS_TXMEFLG1_Pos                                      (27U)
#define CAN_TXSTS_TXMEFLG1_Msk                                      (0x1UL << CAN_TXSTS_TXMEFLG1_Pos)                              /*!< 0x08000000 */
#define CAN_TXSTS_TXMEFLG1                                          CAN_TXSTS_TXMEFLG1_Msk                                         /*!<Transmit Mailbox 1 Empty */
#define CAN_TXSTS_TXMEFLG2_Pos                                      (28U)
#define CAN_TXSTS_TXMEFLG2_Msk                                      (0x1UL << CAN_TXSTS_TXMEFLG2_Pos)                              /*!< 0x10000000 */
#define CAN_TXSTS_TXMEFLG2                                          CAN_TXSTS_TXMEFLG2_Msk                                         /*!<Transmit Mailbox 2 Empty */

#define CAN_TXSTS_LOWESTP_Pos                                       (29U)
#define CAN_TXSTS_LOWESTP_Msk                                       (0x7UL << CAN_TXSTS_LOWESTP_Pos)                               /*!< 0xE0000000 */
#define CAN_TXSTS_LOWESTP                                           CAN_TXSTS_LOWESTP_Msk                                          /*!<LOW[2:0] bits */
#define CAN_TXSTS_LOWESTP0_Pos                                      (29U)
#define CAN_TXSTS_LOWESTP0_Msk                                      (0x1UL << CAN_TXSTS_LOWESTP0_Pos)                              /*!< 0x20000000 */
#define CAN_TXSTS_LOWESTP0                                          CAN_TXSTS_LOWESTP0_Msk                                         /*!<Lowest Priority Flag for Mailbox 0 */
#define CAN_TXSTS_LOWESTP1_Pos                                      (30U)
#define CAN_TXSTS_LOWESTP1_Msk                                      (0x1UL << CAN_TXSTS_LOWESTP1_Pos)                              /*!< 0x40000000 */
#define CAN_TXSTS_LOWESTP1                                          CAN_TXSTS_LOWESTP1_Msk                                         /*!<Lowest Priority Flag for Mailbox 1 */
#define CAN_TXSTS_LOWESTP2_Pos                                      (31U)
#define CAN_TXSTS_LOWESTP2_Msk                                      (0x1UL << CAN_TXSTS_LOWESTP2_Pos)                              /*!< 0x80000000 */
#define CAN_TXSTS_LOWESTP2                                          CAN_TXSTS_LOWESTP2_Msk                                         /*!<Lowest Priority Flag for Mailbox 2 */

/*******************  Bit definition for CAN_RXF0 register  *******************/
#define CAN_RXF0_FMNUM0_Pos                                         (0U)
#define CAN_RXF0_FMNUM0_Msk                                         (0x3UL << CAN_RXF0_FMNUM0_Pos)                                 /*!< 0x00000003 */
#define CAN_RXF0_FMNUM0                                             CAN_RXF0_FMNUM0_Msk                                            /*!<FIFO 0 Message Pending */
#define CAN_RXF0_FFULLFLG0_Pos                                      (3U)
#define CAN_RXF0_FFULLFLG0_Msk                                      (0x1UL << CAN_RXF0_FFULLFLG0_Pos)                              /*!< 0x00000008 */
#define CAN_RXF0_FFULLFLG0                                          CAN_RXF0_FFULLFLG0_Msk                                         /*!<FIFO 0 Full */
#define CAN_RXF0_FOVRFLG0_Pos                                       (4U)
#define CAN_RXF0_FOVRFLG0_Msk                                       (0x1UL << CAN_RXF0_FOVRFLG0_Pos)                               /*!< 0x00000010 */
#define CAN_RXF0_FOVRFLG0                                           CAN_RXF0_FOVRFLG0_Msk                                          /*!<FIFO 0 Overrun */
#define CAN_RXF0_RFOM0_Pos                                          (5U)
#define CAN_RXF0_RFOM0_Msk                                          (0x1UL << CAN_RXF0_RFOM0_Pos)                                  /*!< 0x00000020 */
#define CAN_RXF0_RFOM0                                              CAN_RXF0_RFOM0_Msk                                             /*!<Release FIFO 0 Output Mailbox */

/*******************  Bit definition for CAN_RXF1 register  *******************/
#define CAN_RXF1_FMNUM1_Pos                                         (0U)
#define CAN_RXF1_FMNUM1_Msk                                         (0x3UL << CAN_RXF1_FMNUM1_Pos)                                 /*!< 0x00000003 */
#define CAN_RXF1_FMNUM1                                             CAN_RXF1_FMNUM1_Msk                                            /*!<FIFO 1 Message Pending */
#define CAN_RXF1_FFULLFLG1_Pos                                      (3U)
#define CAN_RXF1_FFULLFLG1_Msk                                      (0x1UL << CAN_RXF1_FFULLFLG1_Pos)                              /*!< 0x00000008 */
#define CAN_RXF1_FFULLFLG1                                          CAN_RXF1_FFULLFLG1_Msk                                         /*!<FIFO 1 Full */
#define CAN_RXF1_FOVRFLG1_Pos                                       (4U)
#define CAN_RXF1_FOVRFLG1_Msk                                       (0x1UL << CAN_RXF1_FOVRFLG1_Pos)                               /*!< 0x00000010 */
#define CAN_RXF1_FOVRFLG1                                           CAN_RXF1_FOVRFLG1_Msk                                          /*!<FIFO 1 Overrun */
#define CAN_RXF1_RFOM1_Pos                                          (5U)
#define CAN_RXF1_RFOM1_Msk                                          (0x1UL << CAN_RXF1_RFOM1_Pos)                                  /*!< 0x00000020 */
#define CAN_RXF1_RFOM1                                              CAN_RXF1_RFOM1_Msk                                             /*!<Release FIFO 1 Output Mailbox */

/********************  Bit definition for CAN_INTEN register  *******************/
#define CAN_INTEN_TXMEIEN_Pos                                       (0U)
#define CAN_INTEN_TXMEIEN_Msk                                       (0x1UL << CAN_INTEN_TXMEIEN_Pos)                               /*!< 0x00000001 */
#define CAN_INTEN_TXMEIEN                                           CAN_INTEN_TXMEIEN_Msk                                          /*!<Transmit Mailbox Empty Interrupt Enable */
#define CAN_INTEN_FMIEN0_Pos                                        (1U)
#define CAN_INTEN_FMIEN0_Msk                                        (0x1UL << CAN_INTEN_FMIEN0_Pos)                                /*!< 0x00000002 */
#define CAN_INTEN_FMIEN0                                            CAN_INTEN_FMIEN0_Msk                                           /*!<FIFO Message Pending Interrupt Enable */
#define CAN_INTEN_FFULLIEN0_Pos                                     (2U)
#define CAN_INTEN_FFULLIEN0_Msk                                     (0x1UL << CAN_INTEN_FFULLIEN0_Pos)                             /*!< 0x00000004 */
#define CAN_INTEN_FFULLIEN0                                         CAN_INTEN_FFULLIEN0_Msk                                        /*!<FIFO Full Interrupt Enable */
#define CAN_INTEN_FOVRIEN0_Pos                                      (3U)
#define CAN_INTEN_FOVRIEN0_Msk                                      (0x1UL << CAN_INTEN_FOVRIEN0_Pos)                              /*!< 0x00000008 */
#define CAN_INTEN_FOVRIEN0                                          CAN_INTEN_FOVRIEN0_Msk                                         /*!<FIFO Overrun Interrupt Enable */
#define CAN_INTEN_FMIEN1_Pos                                        (4U)
#define CAN_INTEN_FMIEN1_Msk                                        (0x1UL << CAN_INTEN_FMIEN1_Pos)                                /*!< 0x00000010 */
#define CAN_INTEN_FMIEN1                                            CAN_INTEN_FMIEN1_Msk                                           /*!<FIFO Message Pending Interrupt Enable */
#define CAN_INTEN_FFULLIEN1_Pos                                     (5U)
#define CAN_INTEN_FFULLIEN1_Msk                                     (0x1UL << CAN_INTEN_FFULLIEN1_Pos)                             /*!< 0x00000020 */
#define CAN_INTEN_FFULLIEN1                                         CAN_INTEN_FFULLIEN1_Msk                                        /*!<FIFO Full Interrupt Enable */
#define CAN_INTEN_FOVRIEN1_Pos                                      (6U)
#define CAN_INTEN_FOVRIEN1_Msk                                      (0x1UL << CAN_INTEN_FOVRIEN1_Pos)                              /*!< 0x00000040 */
#define CAN_INTEN_FOVRIEN1                                          CAN_INTEN_FOVRIEN1_Msk                                         /*!<FIFO Overrun Interrupt Enable */
#define CAN_INTEN_ERRWIEN_Pos                                       (8U)
#define CAN_INTEN_ERRWIEN_Msk                                       (0x1UL << CAN_INTEN_ERRWIEN_Pos)                               /*!< 0x00000100 */
#define CAN_INTEN_ERRWIEN                                           CAN_INTEN_ERRWIEN_Msk                                          /*!<Error Warning Interrupt Enable */
#define CAN_INTEN_ERRPIEN_Pos                                       (9U)
#define CAN_INTEN_ERRPIEN_Msk                                       (0x1UL << CAN_INTEN_ERRPIEN_Pos)                               /*!< 0x00000200 */
#define CAN_INTEN_ERRPIEN                                           CAN_INTEN_ERRPIEN_Msk                                          /*!<Error Passive Interrupt Enable */
#define CAN_INTEN_BOFFIEN_Pos                                       (10U)
#define CAN_INTEN_BOFFIEN_Msk                                       (0x1UL << CAN_INTEN_BOFFIEN_Pos)                               /*!< 0x00000400 */
#define CAN_INTEN_BOFFIEN                                           CAN_INTEN_BOFFIEN_Msk                                          /*!<Bus-Off Interrupt Enable */
#define CAN_INTEN_LECIEN_Pos                                        (11U)
#define CAN_INTEN_LECIEN_Msk                                        (0x1UL << CAN_INTEN_LECIEN_Pos)                                /*!< 0x00000800 */
#define CAN_INTEN_LECIEN                                            CAN_INTEN_LECIEN_Msk                                           /*!<Last Error Code Interrupt Enable */
#define CAN_INTEN_ERRIEN_Pos                                        (15U)
#define CAN_INTEN_ERRIEN_Msk                                        (0x1UL << CAN_INTEN_ERRIEN_Pos)                                /*!< 0x00008000 */
#define CAN_INTEN_ERRIEN                                            CAN_INTEN_ERRIEN_Msk                                           /*!<Error Interrupt Enable */
#define CAN_INTEN_WUPIEN_Pos                                        (16U)
#define CAN_INTEN_WUPIEN_Msk                                        (0x1UL << CAN_INTEN_WUPIEN_Pos)                                /*!< 0x00010000 */
#define CAN_INTEN_WUPIEN                                            CAN_INTEN_WUPIEN_Msk                                           /*!<Wakeup Interrupt Enable */
#define CAN_INTEN_SLEEPIEN_Pos                                      (17U)
#define CAN_INTEN_SLEEPIEN_Msk                                      (0x1UL << CAN_INTEN_SLEEPIEN_Pos)                              /*!< 0x00020000 */
#define CAN_INTEN_SLEEPIEN                                          CAN_INTEN_SLEEPIEN_Msk                                         /*!<Sleep Interrupt Enable */

/********************  Bit definition for CAN_ERRSTS register  *******************/
#define CAN_ERRSTS_ERRWFLG_Pos                                      (0U)
#define CAN_ERRSTS_ERRWFLG_Msk                                      (0x1UL << CAN_ERRSTS_ERRWFLG_Pos)                              /*!< 0x00000001 */
#define CAN_ERRSTS_ERRWFLG                                          CAN_ERRSTS_ERRWFLG_Msk                                         /*!<Error Warning Flag */
#define CAN_ERRSTS_ERRPFLG_Pos                                      (1U)
#define CAN_ERRSTS_ERRPFLG_Msk                                      (0x1UL << CAN_ERRSTS_ERRPFLG_Pos)                              /*!< 0x00000002 */
#define CAN_ERRSTS_ERRPFLG                                          CAN_ERRSTS_ERRPFLG_Msk                                         /*!<Error Passive Flag */
#define CAN_ERRSTS_BOFLG_Pos                                        (2U)
#define CAN_ERRSTS_BOFLG_Msk                                        (0x1UL << CAN_ERRSTS_BOFLG_Pos)                                /*!< 0x00000004 */
#define CAN_ERRSTS_BOFLG                                            CAN_ERRSTS_BOFLG_Msk                                           /*!<Bus-Off Flag */

#define CAN_ERRSTS_LERRC_Pos                                        (4U)
#define CAN_ERRSTS_LERRC_Msk                                        (0x7UL << CAN_ERRSTS_LERRC_Pos)                                /*!< 0x00000070 */
#define CAN_ERRSTS_LERRC                                            CAN_ERRSTS_LERRC_Msk                                           /*!<LEC[2:0] bits (Last Error Code) */
#define CAN_ERRSTS_LERRC_0                                          (0x1UL << CAN_ERRSTS_LERRC_Pos)                                /*!< 0x00000010 */
#define CAN_ERRSTS_LERRC_1                                          (0x2UL << CAN_ERRSTS_LERRC_Pos)                                /*!< 0x00000020 */
#define CAN_ERRSTS_LERRC_2                                          (0x4UL << CAN_ERRSTS_LERRC_Pos)                                /*!< 0x00000040 */

#define CAN_ERRSTS_TXERRCNT_Pos                                     (16U)
#define CAN_ERRSTS_TXERRCNT_Msk                                     (0xFFUL << CAN_ERRSTS_TXERRCNT_Pos)                            /*!< 0x00FF0000 */
#define CAN_ERRSTS_TXERRCNT                                         CAN_ERRSTS_TXERRCNT_Msk                                        /*!<Least significant byte of the 9-bit Transmit Error Counter */
#define CAN_ERRSTS_RXERRCNT_Pos                                     (24U)
#define CAN_ERRSTS_RXERRCNT_Msk                                     (0xFFUL << CAN_ERRSTS_RXERRCNT_Pos)                            /*!< 0xFF000000 */
#define CAN_ERRSTS_RXERRCNT                                         CAN_ERRSTS_RXERRCNT_Msk                                        /*!<Receive Error Counter */

/*******************  Bit definition for CAN_BITTIM register  ********************/
#define CAN_BITTIM_BRPSC_Pos                                        (0U)
#define CAN_BITTIM_BRPSC_Msk                                        (0x3FFUL << CAN_BITTIM_BRPSC_Pos)                              /*!< 0x000003FF */
#define CAN_BITTIM_BRPSC                                            CAN_BITTIM_BRPSC_Msk                                           /*!<Baud Rate Prescaler */
#define CAN_BITTIM_TIMSEG1_Pos                                      (16U)
#define CAN_BITTIM_TIMSEG1_Msk                                      (0xFUL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x000F0000 */
#define CAN_BITTIM_TIMSEG1                                          CAN_BITTIM_TIMSEG1_Msk                                         /*!<Time Segment 1 */
#define CAN_BITTIM_TIMSEG1_0                                        (0x1UL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x00010000 */
#define CAN_BITTIM_TIMSEG1_1                                        (0x2UL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x00020000 */
#define CAN_BITTIM_TIMSEG1_2                                        (0x4UL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x00040000 */
#define CAN_BITTIM_TIMSEG1_3                                        (0x8UL << CAN_BITTIM_TIMSEG1_Pos)                              /*!< 0x00080000 */
#define CAN_BITTIM_TIMSEG2_Pos                                      (20U)
#define CAN_BITTIM_TIMSEG2_Msk                                      (0x7UL << CAN_BITTIM_TIMSEG2_Pos)                              /*!< 0x00700000 */
#define CAN_BITTIM_TIMSEG2                                          CAN_BITTIM_TIMSEG2_Msk                                         /*!<Time Segment 2 */
#define CAN_BITTIM_TIMSEG2_0                                        (0x1UL << CAN_BITTIM_TIMSEG2_Pos)                              /*!< 0x00100000 */
#define CAN_BITTIM_TIMSEG2_1                                        (0x2UL << CAN_BITTIM_TIMSEG2_Pos)                              /*!< 0x00200000 */
#define CAN_BITTIM_TIMSEG2_2                                        (0x4UL << CAN_BITTIM_TIMSEG2_Pos)                              /*!< 0x00400000 */
#define CAN_BITTIM_RSYNJW_Pos                                       (24U)
#define CAN_BITTIM_RSYNJW_Msk                                       (0x3UL << CAN_BITTIM_RSYNJW_Pos)                               /*!< 0x03000000 */
#define CAN_BITTIM_RSYNJW                                           CAN_BITTIM_RSYNJW_Msk                                          /*!<Resynchronization Jump Width */
#define CAN_BITTIM_RSYNJW_0                                         (0x1UL << CAN_BITTIM_RSYNJW_Pos)                               /*!< 0x01000000 */
#define CAN_BITTIM_RSYNJW_1                                         (0x2UL << CAN_BITTIM_RSYNJW_Pos)                               /*!< 0x02000000 */
#define CAN_BITTIM_LBKMEN_Pos                                       (30U)
#define CAN_BITTIM_LBKMEN_Msk                                       (0x1UL << CAN_BITTIM_LBKMEN_Pos)                               /*!< 0x40000000 */
#define CAN_BITTIM_LBKMEN                                           CAN_BITTIM_LBKMEN_Msk                                          /*!<Loop Back Mode (Debug) */
#define CAN_BITTIM_SILMEN_Pos                                       (31U)
#define CAN_BITTIM_SILMEN_Msk                                       (0x1UL << CAN_BITTIM_SILMEN_Pos)                               /*!< 0x80000000 */
#define CAN_BITTIM_SILMEN                                           CAN_BITTIM_SILMEN_Msk                                          /*!<Silent Mode */


/*!<Mailbox registers */
/******************  Bit definition for CAN_TXMID0 register  ********************/
#define CAN_TXMID0_TXMREQ_Pos                                       (0U)
#define CAN_TXMID0_TXMREQ_Msk                                       (0x1UL << CAN_TXMID0_TXMREQ_Pos)                               /*!< 0x00000001 */
#define CAN_TXMID0_TXMREQ                                           CAN_TXMID0_TXMREQ_Msk                                          /*!<Transmit Mailbox Request */
#define CAN_TXMID0_TXRFREQ_Pos                                      (1U)
#define CAN_TXMID0_TXRFREQ_Msk                                      (0x1UL << CAN_TXMID0_TXRFREQ_Pos)                              /*!< 0x00000002 */
#define CAN_TXMID0_TXRFREQ                                          CAN_TXMID0_TXRFREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_TXMID0_IDTYPESEL_Pos                                    (2U)
#define CAN_TXMID0_IDTYPESEL_Msk                                    (0x1UL << CAN_TXMID0_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_TXMID0_IDTYPESEL                                        CAN_TXMID0_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_TXMID0_EXTID_Pos                                        (3U)
#define CAN_TXMID0_EXTID_Msk                                        (0x3FFFFUL << CAN_TXMID0_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_TXMID0_EXTID                                            CAN_TXMID0_EXTID_Msk                                           /*!<Extended Identifier */
#define CAN_TXMID0_STDID_Pos                                        (21U)
#define CAN_TXMID0_STDID_Msk                                        (0x7FFUL << CAN_TXMID0_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_TXMID0_STDID                                            CAN_TXMID0_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/******************  Bit definition for CAN_TXDLEN0 register  *******************/
#define CAN_TXDLEN0_DLCODE_Pos                                      (0U)
#define CAN_TXDLEN0_DLCODE_Msk                                      (0xFUL << CAN_TXDLEN0_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_TXDLEN0_DLCODE                                          CAN_TXDLEN0_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_TXDLEN0_TXTS_Pos                                        (8U)
#define CAN_TXDLEN0_TXTS_Msk                                        (0x1UL << CAN_TXDLEN0_TXTS_Pos)                                /*!< 0x00000100 */
#define CAN_TXDLEN0_TXTS                                            CAN_TXDLEN0_TXTS_Msk                                           /*!<Transmit Global Time */
#define CAN_TXDLEN0_MTS_Pos                                         (16U)
#define CAN_TXDLEN0_MTS_Msk                                         (0xFFFFUL << CAN_TXDLEN0_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_TXDLEN0_MTS                                             CAN_TXDLEN0_MTS_Msk                                            /*!<Message Time Stamp */

/******************  Bit definition for CAN_TXMDL0 register  *******************/
#define CAN_TXMDL0_DATABYTE0_Pos                                    (0U)
#define CAN_TXMDL0_DATABYTE0_Msk                                    (0xFFUL << CAN_TXMDL0_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDL0_DATABYTE0                                        CAN_TXMDL0_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_TXMDL0_DATABYTE1_Pos                                    (8U)
#define CAN_TXMDL0_DATABYTE1_Msk                                    (0xFFUL << CAN_TXMDL0_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDL0_DATABYTE1                                        CAN_TXMDL0_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_TXMDL0_DATABYTE2_Pos                                    (16U)
#define CAN_TXMDL0_DATABYTE2_Msk                                    (0xFFUL << CAN_TXMDL0_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDL0_DATABYTE2                                        CAN_TXMDL0_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_TXMDL0_DATABYTE3_Pos                                    (24U)
#define CAN_TXMDL0_DATABYTE3_Msk                                    (0xFFUL << CAN_TXMDL0_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDL0_DATABYTE3                                        CAN_TXMDL0_DATABYTE3_Msk                                       /*!<Data byte 3 */

/******************  Bit definition for CAN_TXMDH0 register  *******************/
#define CAN_TXMDH0_DATABYTE4_Pos                                    (0U)
#define CAN_TXMDH0_DATABYTE4_Msk                                    (0xFFUL << CAN_TXMDH0_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDH0_DATABYTE4                                        CAN_TXMDH0_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_TXMDH0_DATABYTE5_Pos                                    (8U)
#define CAN_TXMDH0_DATABYTE5_Msk                                    (0xFFUL << CAN_TXMDH0_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDH0_DATABYTE5                                        CAN_TXMDH0_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_TXMDH0_DATABYTE6_Pos                                    (16U)
#define CAN_TXMDH0_DATABYTE6_Msk                                    (0xFFUL << CAN_TXMDH0_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDH0_DATABYTE6                                        CAN_TXMDH0_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_TXMDH0_DATABYTE7_Pos                                    (24U)
#define CAN_TXMDH0_DATABYTE7_Msk                                    (0xFFUL << CAN_TXMDH0_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDH0_DATABYTE7                                        CAN_TXMDH0_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*******************  Bit definition for CAN_TXMID1 register  *******************/
#define CAN_TXMID1_TXMREQ_Pos                                       (0U)
#define CAN_TXMID1_TXMREQ_Msk                                       (0x1UL << CAN_TXMID1_TXMREQ_Pos)                               /*!< 0x00000001 */
#define CAN_TXMID1_TXMREQ                                           CAN_TXMID1_TXMREQ_Msk                                          /*!<Transmit Mailbox Request */
#define CAN_TXMID1_TXRFREQ_Pos                                      (1U)
#define CAN_TXMID1_TXRFREQ_Msk                                      (0x1UL << CAN_TXMID1_TXRFREQ_Pos)                              /*!< 0x00000002 */
#define CAN_TXMID1_TXRFREQ                                          CAN_TXMID1_TXRFREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_TXMID1_IDTYPESEL_Pos                                    (2U)
#define CAN_TXMID1_IDTYPESEL_Msk                                    (0x1UL << CAN_TXMID1_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_TXMID1_IDTYPESEL                                        CAN_TXMID1_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_TXMID1_EXTID_Pos                                        (3U)
#define CAN_TXMID1_EXTID_Msk                                        (0x3FFFFUL << CAN_TXMID1_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_TXMID1_EXTID                                            CAN_TXMID1_EXTID_Msk                                           /*!<Extended Identifier */
#define CAN_TXMID1_STDID_Pos                                        (21U)
#define CAN_TXMID1_STDID_Msk                                        (0x7FFUL << CAN_TXMID1_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_TXMID1_STDID                                            CAN_TXMID1_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_TXDLEN1 register  ******************/
#define CAN_TXDLEN1_DLCODE_Pos                                      (0U)
#define CAN_TXDLEN1_DLCODE_Msk                                      (0xFUL << CAN_TXDLEN1_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_TXDLEN1_DLCODE                                          CAN_TXDLEN1_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_TXDLEN1_TXTS_Pos                                        (8U)
#define CAN_TXDLEN1_TXTS_Msk                                        (0x1UL << CAN_TXDLEN1_TXTS_Pos)                                /*!< 0x00000100 */
#define CAN_TXDLEN1_TXTS                                            CAN_TXDLEN1_TXTS_Msk                                           /*!<Transmit Global Time */
#define CAN_TXDLEN1_MTS_Pos                                         (16U)
#define CAN_TXDLEN1_MTS_Msk                                         (0xFFFFUL << CAN_TXDLEN1_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_TXDLEN1_MTS                                             CAN_TXDLEN1_MTS_Msk                                            /*!<Message Time Stamp */

/*******************  Bit definition for CAN_TXMDL1 register  ******************/
#define CAN_TXMDL1_DATABYTE0_Pos                                    (0U)
#define CAN_TXMDL1_DATABYTE0_Msk                                    (0xFFUL << CAN_TXMDL1_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDL1_DATABYTE0                                        CAN_TXMDL1_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_TXMDL1_DATABYTE1_Pos                                    (8U)
#define CAN_TXMDL1_DATABYTE1_Msk                                    (0xFFUL << CAN_TXMDL1_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDL1_DATABYTE1                                        CAN_TXMDL1_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_TXMDL1_DATABYTE2_Pos                                    (16U)
#define CAN_TXMDL1_DATABYTE2_Msk                                    (0xFFUL << CAN_TXMDL1_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDL1_DATABYTE2                                        CAN_TXMDL1_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_TXMDL1_DATABYTE3_Pos                                    (24U)
#define CAN_TXMDL1_DATABYTE3_Msk                                    (0xFFUL << CAN_TXMDL1_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDL1_DATABYTE3                                        CAN_TXMDL1_DATABYTE3_Msk                                       /*!<Data byte 3 */

/*******************  Bit definition for CAN_TXMDH1 register  ******************/
#define CAN_TXMDH1_DATABYTE4_Pos                                    (0U)
#define CAN_TXMDH1_DATABYTE4_Msk                                    (0xFFUL << CAN_TXMDH1_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDH1_DATABYTE4                                        CAN_TXMDH1_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_TXMDH1_DATABYTE5_Pos                                    (8U)
#define CAN_TXMDH1_DATABYTE5_Msk                                    (0xFFUL << CAN_TXMDH1_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDH1_DATABYTE5                                        CAN_TXMDH1_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_TXMDH1_DATABYTE6_Pos                                    (16U)
#define CAN_TXMDH1_DATABYTE6_Msk                                    (0xFFUL << CAN_TXMDH1_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDH1_DATABYTE6                                        CAN_TXMDH1_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_TXMDH1_DATABYTE7_Pos                                    (24U)
#define CAN_TXMDH1_DATABYTE7_Msk                                    (0xFFUL << CAN_TXMDH1_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDH1_DATABYTE7                                        CAN_TXMDH1_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*******************  Bit definition for CAN_TXMID2 register  *******************/
#define CAN_TXMID2_TXMREQ_Pos                                       (0U)
#define CAN_TXMID2_TXMREQ_Msk                                       (0x1UL << CAN_TXMID2_TXMREQ_Pos)                               /*!< 0x00000001 */
#define CAN_TXMID2_TXMREQ                                           CAN_TXMID2_TXMREQ_Msk                                          /*!<Transmit Mailbox Request */
#define CAN_TXMID2_TXRFREQ_Pos                                      (1U)
#define CAN_TXMID2_TXRFREQ_Msk                                      (0x1UL << CAN_TXMID2_TXRFREQ_Pos)                              /*!< 0x00000002 */
#define CAN_TXMID2_TXRFREQ                                          CAN_TXMID2_TXRFREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_TXMID2_IDTYPESEL_Pos                                    (2U)
#define CAN_TXMID2_IDTYPESEL_Msk                                    (0x1UL << CAN_TXMID2_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_TXMID2_IDTYPESEL                                        CAN_TXMID2_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_TXMID2_EXTID_Pos                                        (3U)
#define CAN_TXMID2_EXTID_Msk                                        (0x3FFFFUL << CAN_TXMID2_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_TXMID2_EXTID                                            CAN_TXMID2_EXTID_Msk                                           /*!<Extended identifier */
#define CAN_TXMID2_STDID_Pos                                        (21U)
#define CAN_TXMID2_STDID_Msk                                        (0x7FFUL << CAN_TXMID2_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_TXMID2_STDID                                            CAN_TXMID2_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_TXDLEN2 register  ******************/
#define CAN_TXDLEN2_DLCODE_Pos                                      (0U)
#define CAN_TXDLEN2_DLCODE_Msk                                      (0xFUL << CAN_TXDLEN2_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_TXDLEN2_DLCODE                                          CAN_TXDLEN2_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_TXDLEN2_TXTS_Pos                                        (8U)
#define CAN_TXDLEN2_TXTS_Msk                                        (0x1UL << CAN_TXDLEN2_TXTS_Pos)                                /*!< 0x00000100 */
#define CAN_TXDLEN2_TXTS                                            CAN_TXDLEN2_TXTS_Msk                                           /*!<Transmit Global Time */
#define CAN_TXDLEN2_MTS_Pos                                         (16U)
#define CAN_TXDLEN2_MTS_Msk                                         (0xFFFFUL << CAN_TXDLEN2_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_TXDLEN2_MTS                                             CAN_TXDLEN2_MTS_Msk                                            /*!<Message Time Stamp */

/*******************  Bit definition for CAN_TXMDL2 register  ******************/
#define CAN_TXMDL2_DATABYTE0_Pos                                    (0U)
#define CAN_TXMDL2_DATABYTE0_Msk                                    (0xFFUL << CAN_TXMDL2_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDL2_DATABYTE0                                        CAN_TXMDL2_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_TXMDL2_DATABYTE1_Pos                                    (8U)
#define CAN_TXMDL2_DATABYTE1_Msk                                    (0xFFUL << CAN_TXMDL2_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDL2_DATABYTE1                                        CAN_TXMDL2_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_TXMDL2_DATABYTE2_Pos                                    (16U)
#define CAN_TXMDL2_DATABYTE2_Msk                                    (0xFFUL << CAN_TXMDL2_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDL2_DATABYTE2                                        CAN_TXMDL2_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_TXMDL2_DATABYTE3_Pos                                    (24U)
#define CAN_TXMDL2_DATABYTE3_Msk                                    (0xFFUL << CAN_TXMDL2_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDL2_DATABYTE3                                        CAN_TXMDL2_DATABYTE3_Msk                                       /*!<Data byte 3 */

/*******************  Bit definition for CAN_TXMDH2 register  ******************/
#define CAN_TXMDH2_DATABYTE4_Pos                                    (0U)
#define CAN_TXMDH2_DATABYTE4_Msk                                    (0xFFUL << CAN_TXMDH2_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_TXMDH2_DATABYTE4                                        CAN_TXMDH2_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_TXMDH2_DATABYTE5_Pos                                    (8U)
#define CAN_TXMDH2_DATABYTE5_Msk                                    (0xFFUL << CAN_TXMDH2_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_TXMDH2_DATABYTE5                                        CAN_TXMDH2_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_TXMDH2_DATABYTE6_Pos                                    (16U)
#define CAN_TXMDH2_DATABYTE6_Msk                                    (0xFFUL << CAN_TXMDH2_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_TXMDH2_DATABYTE6                                        CAN_TXMDH2_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_TXMDH2_DATABYTE7_Pos                                    (24U)
#define CAN_TXMDH2_DATABYTE7_Msk                                    (0xFFUL << CAN_TXMDH2_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_TXMDH2_DATABYTE7                                        CAN_TXMDH2_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*******************  Bit definition for CAN_RXMID0 register  *******************/
#define CAN_RXMID0_RFTXREQ_Pos                                      (1U)
#define CAN_RXMID0_RFTXREQ_Msk                                      (0x1UL << CAN_RXMID0_RFTXREQ_Pos)                              /*!< 0x00000002 */
#define CAN_RXMID0_RFTXREQ                                          CAN_RXMID0_RFTXREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_RXMID0_IDTYPESEL_Pos                                    (2U)
#define CAN_RXMID0_IDTYPESEL_Msk                                    (0x1UL << CAN_RXMID0_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_RXMID0_IDTYPESEL                                        CAN_RXMID0_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_RXMID0_EXTID_Pos                                        (3U)
#define CAN_RXMID0_EXTID_Msk                                        (0x3FFFFUL << CAN_RXMID0_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_RXMID0_EXTID                                            CAN_RXMID0_EXTID_Msk                                           /*!<Extended Identifier */
#define CAN_RXMID0_STDID_Pos                                        (21U)
#define CAN_RXMID0_STDID_Msk                                        (0x7FFUL << CAN_RXMID0_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_RXMID0_STDID                                            CAN_RXMID0_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_RXDLEN0 register  ******************/
#define CAN_RXDLEN0_DLCODE_Pos                                      (0U)
#define CAN_RXDLEN0_DLCODE_Msk                                      (0xFUL << CAN_RXDLEN0_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_RXDLEN0_DLCODE                                          CAN_RXDLEN0_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_RXDLEN0_FMIDX_Pos                                       (8U)
#define CAN_RXDLEN0_FMIDX_Msk                                       (0xFFUL << CAN_RXDLEN0_FMIDX_Pos)                              /*!< 0x0000FF00 */
#define CAN_RXDLEN0_FMIDX                                           CAN_RXDLEN0_FMIDX_Msk                                          /*!<Filter Match Index */
#define CAN_RXDLEN0_MTS_Pos                                         (16U)
#define CAN_RXDLEN0_MTS_Msk                                         (0xFFFFUL << CAN_RXDLEN0_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_RXDLEN0_MTS                                             CAN_RXDLEN0_MTS_Msk                                            /*!<Message Time Stamp */

/*******************  Bit definition for CAN_RXMDL0 register  ******************/
#define CAN_RXMDL0_DATABYTE0_Pos                                    (0U)
#define CAN_RXMDL0_DATABYTE0_Msk                                    (0xFFUL << CAN_RXMDL0_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_RXMDL0_DATABYTE0                                        CAN_RXMDL0_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_RXMDL0_DATABYTE1_Pos                                    (8U)
#define CAN_RXMDL0_DATABYTE1_Msk                                    (0xFFUL << CAN_RXMDL0_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_RXMDL0_DATABYTE1                                        CAN_RXMDL0_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_RXMDL0_DATABYTE2_Pos                                    (16U)
#define CAN_RXMDL0_DATABYTE2_Msk                                    (0xFFUL << CAN_RXMDL0_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_RXMDL0_DATABYTE2                                        CAN_RXMDL0_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_RXMDL0_DATABYTE3_Pos                                    (24U)
#define CAN_RXMDL0_DATABYTE3_Msk                                    (0xFFUL << CAN_RXMDL0_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_RXMDL0_DATABYTE3                                        CAN_RXMDL0_DATABYTE3_Msk                                       /*!<Data byte 3 */

/*******************  Bit definition for CAN_RXMDH0 register  ******************/
#define CAN_RXMDH0_DATABYTE4_Pos                                    (0U)
#define CAN_RXMDH0_DATABYTE4_Msk                                    (0xFFUL << CAN_RXMDH0_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_RXMDH0_DATABYTE4                                        CAN_RXMDH0_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_RXMDH0_DATABYTE5_Pos                                    (8U)
#define CAN_RXMDH0_DATABYTE5_Msk                                    (0xFFUL << CAN_RXMDH0_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_RXMDH0_DATABYTE5                                        CAN_RXMDH0_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_RXMDH0_DATABYTE6_Pos                                    (16U)
#define CAN_RXMDH0_DATABYTE6_Msk                                    (0xFFUL << CAN_RXMDH0_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_RXMDH0_DATABYTE6                                        CAN_RXMDH0_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_RXMDH0_DATABYTE7_Pos                                    (24U)
#define CAN_RXMDH0_DATABYTE7_Msk                                    (0xFFUL << CAN_RXMDH0_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_RXMDH0_DATABYTE7                                        CAN_RXMDH0_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*******************  Bit definition for CAN_RXMID1 register  *******************/
#define CAN_RXMID1_RFTXREQ_Pos                                      (1U)
#define CAN_RXMID1_RFTXREQ_Msk                                      (0x1UL << CAN_RXMID1_RFTXREQ_Pos)                              /*!< 0x00000002 */
#define CAN_RXMID1_RFTXREQ                                          CAN_RXMID1_RFTXREQ_Msk                                         /*!<Remote Transmission Request */
#define CAN_RXMID1_IDTYPESEL_Pos                                    (2U)
#define CAN_RXMID1_IDTYPESEL_Msk                                    (0x1UL << CAN_RXMID1_IDTYPESEL_Pos)                            /*!< 0x00000004 */
#define CAN_RXMID1_IDTYPESEL                                        CAN_RXMID1_IDTYPESEL_Msk                                       /*!<Identifier Extension */
#define CAN_RXMID1_EXTID_Pos                                        (3U)
#define CAN_RXMID1_EXTID_Msk                                        (0x3FFFFUL << CAN_RXMID1_EXTID_Pos)                            /*!< 0x001FFFF8 */
#define CAN_RXMID1_EXTID                                            CAN_RXMID1_EXTID_Msk                                           /*!<Extended identifier */
#define CAN_RXMID1_STDID_Pos                                        (21U)
#define CAN_RXMID1_STDID_Msk                                        (0x7FFUL << CAN_RXMID1_STDID_Pos)                              /*!< 0xFFE00000 */
#define CAN_RXMID1_STDID                                            CAN_RXMID1_STDID_Msk                                           /*!<Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_RXDLEN1 register  ******************/
#define CAN_RXDLEN1_DLCODE_Pos                                      (0U)
#define CAN_RXDLEN1_DLCODE_Msk                                      (0xFUL << CAN_RXDLEN1_DLCODE_Pos)                              /*!< 0x0000000F */
#define CAN_RXDLEN1_DLCODE                                          CAN_RXDLEN1_DLCODE_Msk                                         /*!<Data Length Code */
#define CAN_RXDLEN1_FMIDX_Pos                                       (8U)
#define CAN_RXDLEN1_FMIDX_Msk                                       (0xFFUL << CAN_RXDLEN1_FMIDX_Pos)                              /*!< 0x0000FF00 */
#define CAN_RXDLEN1_FMIDX                                           CAN_RXDLEN1_FMIDX_Msk                                          /*!<Filter Match Index */
#define CAN_RXDLEN1_MTS_Pos                                         (16U)
#define CAN_RXDLEN1_MTS_Msk                                         (0xFFFFUL << CAN_RXDLEN1_MTS_Pos)                              /*!< 0xFFFF0000 */
#define CAN_RXDLEN1_MTS                                             CAN_RXDLEN1_MTS_Msk                                            /*!<Message Time Stamp */

/*******************  Bit definition for CAN_RXMDL1 register  ******************/
#define CAN_RXMDL1_DATABYTE0_Pos                                    (0U)
#define CAN_RXMDL1_DATABYTE0_Msk                                    (0xFFUL << CAN_RXMDL1_DATABYTE0_Pos)                           /*!< 0x000000FF */
#define CAN_RXMDL1_DATABYTE0                                        CAN_RXMDL1_DATABYTE0_Msk                                       /*!<Data byte 0 */
#define CAN_RXMDL1_DATABYTE1_Pos                                    (8U)
#define CAN_RXMDL1_DATABYTE1_Msk                                    (0xFFUL << CAN_RXMDL1_DATABYTE1_Pos)                           /*!< 0x0000FF00 */
#define CAN_RXMDL1_DATABYTE1                                        CAN_RXMDL1_DATABYTE1_Msk                                       /*!<Data byte 1 */
#define CAN_RXMDL1_DATABYTE2_Pos                                    (16U)
#define CAN_RXMDL1_DATABYTE2_Msk                                    (0xFFUL << CAN_RXMDL1_DATABYTE2_Pos)                           /*!< 0x00FF0000 */
#define CAN_RXMDL1_DATABYTE2                                        CAN_RXMDL1_DATABYTE2_Msk                                       /*!<Data byte 2 */
#define CAN_RXMDL1_DATABYTE3_Pos                                    (24U)
#define CAN_RXMDL1_DATABYTE3_Msk                                    (0xFFUL << CAN_RXMDL1_DATABYTE3_Pos)                           /*!< 0xFF000000 */
#define CAN_RXMDL1_DATABYTE3                                        CAN_RXMDL1_DATABYTE3_Msk                                       /*!<Data byte 3 */

/*******************  Bit definition for CAN_RXMDH1 register  ******************/
#define CAN_RXMDH1_DATABYTE4_Pos                                    (0U)
#define CAN_RXMDH1_DATABYTE4_Msk                                    (0xFFUL << CAN_RXMDH1_DATABYTE4_Pos)                           /*!< 0x000000FF */
#define CAN_RXMDH1_DATABYTE4                                        CAN_RXMDH1_DATABYTE4_Msk                                       /*!<Data byte 4 */
#define CAN_RXMDH1_DATABYTE5_Pos                                    (8U)
#define CAN_RXMDH1_DATABYTE5_Msk                                    (0xFFUL << CAN_RXMDH1_DATABYTE5_Pos)                           /*!< 0x0000FF00 */
#define CAN_RXMDH1_DATABYTE5                                        CAN_RXMDH1_DATABYTE5_Msk                                       /*!<Data byte 5 */
#define CAN_RXMDH1_DATABYTE6_Pos                                    (16U)
#define CAN_RXMDH1_DATABYTE6_Msk                                    (0xFFUL << CAN_RXMDH1_DATABYTE6_Pos)                           /*!< 0x00FF0000 */
#define CAN_RXMDH1_DATABYTE6                                        CAN_RXMDH1_DATABYTE6_Msk                                       /*!<Data byte 6 */
#define CAN_RXMDH1_DATABYTE7_Pos                                    (24U)
#define CAN_RXMDH1_DATABYTE7_Msk                                    (0xFFUL << CAN_RXMDH1_DATABYTE7_Pos)                           /*!< 0xFF000000 */
#define CAN_RXMDH1_DATABYTE7                                        CAN_RXMDH1_DATABYTE7_Msk                                       /*!<Data byte 7 */

/*!<CAN filter registers */
/*******************  Bit definition for CAN_FCTRL register  ********************/
#define CAN_FCTRL_FINITEN_Pos                                       (0U)
#define CAN_FCTRL_FINITEN_Msk                                       (0x1UL << CAN_FCTRL_FINITEN_Pos)                                 /*!< 0x00000001 */
#define CAN_FCTRL_FINITEN                                           CAN_FCTRL_FINITEN_Msk                                            /*!<Filter Init Mode */
#define CAN_FCTRL_CAN2SB_Pos                                        (8U)
#define CAN_FCTRL_CAN2SB_Msk                                        (0x3FUL << CAN_FCTRL_CAN2SB_Pos)                               /*!< 0x00003F00 */
#define CAN_FCTRL_CAN2SB                                            CAN_FCTRL_CAN2SB_Msk                                           /*!<CAN2 start bank */

/*******************  Bit definition for CAN_FMCFG register  *******************/
#define CAN_FMCFG_FMCFG_Pos                                         (0U)
#define CAN_FMCFG_FMCFG_Msk                                         (0xFFFFFFFUL << CAN_FMCFG_FMCFG_Pos)                           /*!< 0x0FFFFFFF */
#define CAN_FMCFG_FMCFG                                             CAN_FMCFG_FMCFG_Msk                                            /*!<Filter Mode */
#define CAN_FMCFG_FMCFG0_Pos                                        (0U)
#define CAN_FMCFG_FMCFG0_Msk                                        (0x1UL << CAN_FMCFG_FMCFG0_Pos)                                /*!< 0x00000001 */
#define CAN_FMCFG_FMCFG0                                            CAN_FMCFG_FMCFG0_Msk                                           /*!<Filter Init Mode bit 0 */
#define CAN_FMCFG_FMCFG1_Pos                                        (1U)
#define CAN_FMCFG_FMCFG1_Msk                                        (0x1UL << CAN_FMCFG_FMCFG1_Pos)                                /*!< 0x00000002 */
#define CAN_FMCFG_FMCFG1                                            CAN_FMCFG_FMCFG1_Msk                                           /*!<Filter Init Mode bit 1 */
#define CAN_FMCFG_FMCFG2_Pos                                        (2U)
#define CAN_FMCFG_FMCFG2_Msk                                        (0x1UL << CAN_FMCFG_FMCFG2_Pos)                                /*!< 0x00000004 */
#define CAN_FMCFG_FMCFG2                                            CAN_FMCFG_FMCFG2_Msk                                           /*!<Filter Init Mode bit 2 */
#define CAN_FMCFG_FMCFG3_Pos                                        (3U)
#define CAN_FMCFG_FMCFG3_Msk                                        (0x1UL << CAN_FMCFG_FMCFG3_Pos)                                /*!< 0x00000008 */
#define CAN_FMCFG_FMCFG3                                            CAN_FMCFG_FMCFG3_Msk                                           /*!<Filter Init Mode bit 3 */
#define CAN_FMCFG_FMCFG4_Pos                                        (4U)
#define CAN_FMCFG_FMCFG4_Msk                                        (0x1UL << CAN_FMCFG_FMCFG4_Pos)                                /*!< 0x00000010 */
#define CAN_FMCFG_FMCFG4                                            CAN_FMCFG_FMCFG4_Msk                                           /*!<Filter Init Mode bit 4 */
#define CAN_FMCFG_FMCFG5_Pos                                        (5U)
#define CAN_FMCFG_FMCFG5_Msk                                        (0x1UL << CAN_FMCFG_FMCFG5_Pos)                                /*!< 0x00000020 */
#define CAN_FMCFG_FMCFG5                                            CAN_FMCFG_FMCFG5_Msk                                           /*!<Filter Init Mode bit 5 */
#define CAN_FMCFG_FMCFG6_Pos                                        (6U)
#define CAN_FMCFG_FMCFG6_Msk                                        (0x1UL << CAN_FMCFG_FMCFG6_Pos)                                /*!< 0x00000040 */
#define CAN_FMCFG_FMCFG6                                            CAN_FMCFG_FMCFG6_Msk                                           /*!<Filter Init Mode bit 6 */
#define CAN_FMCFG_FMCFG7_Pos                                        (7U)
#define CAN_FMCFG_FMCFG7_Msk                                        (0x1UL << CAN_FMCFG_FMCFG7_Pos)                                /*!< 0x00000080 */
#define CAN_FMCFG_FMCFG7                                            CAN_FMCFG_FMCFG7_Msk                                           /*!<Filter Init Mode bit 7 */
#define CAN_FMCFG_FMCFG8_Pos                                        (8U)
#define CAN_FMCFG_FMCFG8_Msk                                        (0x1UL << CAN_FMCFG_FMCFG8_Pos)                                /*!< 0x00000100 */
#define CAN_FMCFG_FMCFG8                                            CAN_FMCFG_FMCFG8_Msk                                           /*!<Filter Init Mode bit 8 */
#define CAN_FMCFG_FMCFG9_Pos                                        (9U)
#define CAN_FMCFG_FMCFG9_Msk                                        (0x1UL << CAN_FMCFG_FMCFG9_Pos)                                /*!< 0x00000200 */
#define CAN_FMCFG_FMCFG9                                            CAN_FMCFG_FMCFG9_Msk                                           /*!<Filter Init Mode bit 9 */
#define CAN_FMCFG_FMCFG10_Pos                                       (10U)
#define CAN_FMCFG_FMCFG10_Msk                                       (0x1UL << CAN_FMCFG_FMCFG10_Pos)                               /*!< 0x00000400 */
#define CAN_FMCFG_FMCFG10                                           CAN_FMCFG_FMCFG10_Msk                                          /*!<Filter Init Mode bit 10 */
#define CAN_FMCFG_FMCFG11_Pos                                       (11U)
#define CAN_FMCFG_FMCFG11_Msk                                       (0x1UL << CAN_FMCFG_FMCFG11_Pos)                               /*!< 0x00000800 */
#define CAN_FMCFG_FMCFG11                                           CAN_FMCFG_FMCFG11_Msk                                          /*!<Filter Init Mode bit 11 */
#define CAN_FMCFG_FMCFG12_Pos                                       (12U)
#define CAN_FMCFG_FMCFG12_Msk                                       (0x1UL << CAN_FMCFG_FMCFG12_Pos)                               /*!< 0x00001000 */
#define CAN_FMCFG_FMCFG12                                           CAN_FMCFG_FMCFG12_Msk                                          /*!<Filter Init Mode bit 12 */
#define CAN_FMCFG_FMCFG13_Pos                                       (13U)
#define CAN_FMCFG_FMCFG13_Msk                                       (0x1UL << CAN_FMCFG_FMCFG13_Pos)                               /*!< 0x00002000 */
#define CAN_FMCFG_FMCFG13                                           CAN_FMCFG_FMCFG13_Msk                                          /*!<Filter Init Mode bit 13 */
#define CAN_FMCFG_FMCFG14_Pos                                       (14U)
#define CAN_FMCFG_FMCFG14_Msk                                       (0x1UL << CAN_FMCFG_FMCFG14_Pos)                               /*!< 0x00004000 */
#define CAN_FMCFG_FMCFG14                                           CAN_FMCFG_FMCFG14_Msk                                          /*!<Filter Init Mode bit 14 */
#define CAN_FMCFG_FMCFG15_Pos                                       (15U)
#define CAN_FMCFG_FMCFG15_Msk                                       (0x1UL << CAN_FMCFG_FMCFG15_Pos)                               /*!< 0x00008000 */
#define CAN_FMCFG_FMCFG15                                           CAN_FMCFG_FMCFG15_Msk                                          /*!<Filter Init Mode bit 15 */
#define CAN_FMCFG_FMCFG16_Pos                                       (16U)
#define CAN_FMCFG_FMCFG16_Msk                                       (0x1UL << CAN_FMCFG_FMCFG16_Pos)                               /*!< 0x00010000 */
#define CAN_FMCFG_FMCFG16                                           CAN_FMCFG_FMCFG16_Msk                                          /*!<Filter Init Mode bit 16 */
#define CAN_FMCFG_FMCFG17_Pos                                       (17U)
#define CAN_FMCFG_FMCFG17_Msk                                       (0x1UL << CAN_FMCFG_FMCFG17_Pos)                               /*!< 0x00020000 */
#define CAN_FMCFG_FMCFG17                                           CAN_FMCFG_FMCFG17_Msk                                          /*!<Filter Init Mode bit 17 */
#define CAN_FMCFG_FMCFG18_Pos                                       (18U)
#define CAN_FMCFG_FMCFG18_Msk                                       (0x1UL << CAN_FMCFG_FMCFG18_Pos)                               /*!< 0x00040000 */
#define CAN_FMCFG_FMCFG18                                           CAN_FMCFG_FMCFG18_Msk                                          /*!<Filter Init Mode bit 18 */
#define CAN_FMCFG_FMCFG19_Pos                                       (19U)
#define CAN_FMCFG_FMCFG19_Msk                                       (0x1UL << CAN_FMCFG_FMCFG19_Pos)                               /*!< 0x00080000 */
#define CAN_FMCFG_FMCFG19                                           CAN_FMCFG_FMCFG19_Msk                                          /*!<Filter Init Mode bit 19 */
#define CAN_FMCFG_FMCFG20_Pos                                       (20U)
#define CAN_FMCFG_FMCFG20_Msk                                       (0x1UL << CAN_FMCFG_FMCFG20_Pos)                               /*!< 0x00100000 */
#define CAN_FMCFG_FMCFG20                                           CAN_FMCFG_FMCFG20_Msk                                          /*!<Filter Init Mode bit 20 */
#define CAN_FMCFG_FMCFG21_Pos                                       (21U)
#define CAN_FMCFG_FMCFG21_Msk                                       (0x1UL << CAN_FMCFG_FMCFG21_Pos)                               /*!< 0x00200000 */
#define CAN_FMCFG_FMCFG21                                           CAN_FMCFG_FMCFG21_Msk                                          /*!<Filter Init Mode bit 21 */
#define CAN_FMCFG_FMCFG22_Pos                                       (22U)
#define CAN_FMCFG_FMCFG22_Msk                                       (0x1UL << CAN_FMCFG_FMCFG22_Pos)                               /*!< 0x00400000 */
#define CAN_FMCFG_FMCFG22                                           CAN_FMCFG_FMCFG22_Msk                                          /*!<Filter Init Mode bit 22 */
#define CAN_FMCFG_FMCFG23_Pos                                       (23U)
#define CAN_FMCFG_FMCFG23_Msk                                       (0x1UL << CAN_FMCFG_FMCFG23_Pos)                               /*!< 0x00800000 */
#define CAN_FMCFG_FMCFG23                                           CAN_FMCFG_FMCFG23_Msk                                          /*!<Filter Init Mode bit 23 */
#define CAN_FMCFG_FMCFG24_Pos                                       (24U)
#define CAN_FMCFG_FMCFG24_Msk                                       (0x1UL << CAN_FMCFG_FMCFG24_Pos)                               /*!< 0x01000000 */
#define CAN_FMCFG_FMCFG24                                           CAN_FMCFG_FMCFG24_Msk                                          /*!<Filter Init Mode bit 24 */
#define CAN_FMCFG_FMCFG25_Pos                                       (25U)
#define CAN_FMCFG_FMCFG25_Msk                                       (0x1UL << CAN_FMCFG_FMCFG25_Pos)                               /*!< 0x02000000 */
#define CAN_FMCFG_FMCFG25                                           CAN_FMCFG_FMCFG25_Msk                                          /*!<Filter Init Mode bit 25 */
#define CAN_FMCFG_FMCFG26_Pos                                       (26U)
#define CAN_FMCFG_FMCFG26_Msk                                       (0x1UL << CAN_FMCFG_FMCFG26_Pos)                               /*!< 0x04000000 */
#define CAN_FMCFG_FMCFG26                                           CAN_FMCFG_FMCFG26_Msk                                          /*!<Filter Init Mode bit 26 */
#define CAN_FMCFG_FMCFG27_Pos                                       (27U)
#define CAN_FMCFG_FMCFG27_Msk                                       (0x1UL << CAN_FMCFG_FMCFG27_Pos)                               /*!< 0x08000000 */
#define CAN_FMCFG_FMCFG27                                           CAN_FMCFG_FMCFG27_Msk                                          /*!<Filter Init Mode bit 27 */

/*******************  Bit definition for CAN_FSCFG register  *******************/
#define CAN_FSCFG_FSCFG_Pos                                         (0U)
#define CAN_FSCFG_FSCFG_Msk                                         (0xFFFFFFFUL << CAN_FSCFG_FSCFG_Pos)                           /*!< 0x0FFFFFFF */
#define CAN_FSCFG_FSCFG                                             CAN_FSCFG_FSCFG_Msk                                            /*!<Filter Scale Configuration */
#define CAN_FSCFG_FSCFG0_Pos                                        (0U)
#define CAN_FSCFG_FSCFG0_Msk                                        (0x1UL << CAN_FSCFG_FSCFG0_Pos)                                /*!< 0x00000001 */
#define CAN_FSCFG_FSCFG0                                            CAN_FSCFG_FSCFG0_Msk                                           /*!<Filter Scale Configuration bit 0 */
#define CAN_FSCFG_FSCFG1_Pos                                        (1U)
#define CAN_FSCFG_FSCFG1_Msk                                        (0x1UL << CAN_FSCFG_FSCFG1_Pos)                                /*!< 0x00000002 */
#define CAN_FSCFG_FSCFG1                                            CAN_FSCFG_FSCFG1_Msk                                           /*!<Filter Scale Configuration bit 1 */
#define CAN_FSCFG_FSCFG2_Pos                                        (2U)
#define CAN_FSCFG_FSCFG2_Msk                                        (0x1UL << CAN_FSCFG_FSCFG2_Pos)                                /*!< 0x00000004 */
#define CAN_FSCFG_FSCFG2                                            CAN_FSCFG_FSCFG2_Msk                                           /*!<Filter Scale Configuration bit 2 */
#define CAN_FSCFG_FSCFG3_Pos                                        (3U)
#define CAN_FSCFG_FSCFG3_Msk                                        (0x1UL << CAN_FSCFG_FSCFG3_Pos)                                /*!< 0x00000008 */
#define CAN_FSCFG_FSCFG3                                            CAN_FSCFG_FSCFG3_Msk                                           /*!<Filter Scale Configuration bit 3 */
#define CAN_FSCFG_FSCFG4_Pos                                        (4U)
#define CAN_FSCFG_FSCFG4_Msk                                        (0x1UL << CAN_FSCFG_FSCFG4_Pos)                                /*!< 0x00000010 */
#define CAN_FSCFG_FSCFG4                                            CAN_FSCFG_FSCFG4_Msk                                           /*!<Filter Scale Configuration bit 4 */
#define CAN_FSCFG_FSCFG5_Pos                                        (5U)
#define CAN_FSCFG_FSCFG5_Msk                                        (0x1UL << CAN_FSCFG_FSCFG5_Pos)                                /*!< 0x00000020 */
#define CAN_FSCFG_FSCFG5                                            CAN_FSCFG_FSCFG5_Msk                                           /*!<Filter Scale Configuration bit 5 */
#define CAN_FSCFG_FSCFG6_Pos                                        (6U)
#define CAN_FSCFG_FSCFG6_Msk                                        (0x1UL << CAN_FSCFG_FSCFG6_Pos)                                /*!< 0x00000040 */
#define CAN_FSCFG_FSCFG6                                            CAN_FSCFG_FSCFG6_Msk                                           /*!<Filter Scale Configuration bit 6 */
#define CAN_FSCFG_FSCFG7_Pos                                        (7U)
#define CAN_FSCFG_FSCFG7_Msk                                        (0x1UL << CAN_FSCFG_FSCFG7_Pos)                                /*!< 0x00000080 */
#define CAN_FSCFG_FSCFG7                                            CAN_FSCFG_FSCFG7_Msk                                           /*!<Filter Scale Configuration bit 7 */
#define CAN_FSCFG_FSCFG8_Pos                                        (8U)
#define CAN_FSCFG_FSCFG8_Msk                                        (0x1UL << CAN_FSCFG_FSCFG8_Pos)                                /*!< 0x00000100 */
#define CAN_FSCFG_FSCFG8                                            CAN_FSCFG_FSCFG8_Msk                                           /*!<Filter Scale Configuration bit 8 */
#define CAN_FSCFG_FSCFG9_Pos                                        (9U)
#define CAN_FSCFG_FSCFG9_Msk                                        (0x1UL << CAN_FSCFG_FSCFG9_Pos)                                /*!< 0x00000200 */
#define CAN_FSCFG_FSCFG9                                            CAN_FSCFG_FSCFG9_Msk                                           /*!<Filter Scale Configuration bit 9 */
#define CAN_FSCFG_FSCFG10_Pos                                       (10U)
#define CAN_FSCFG_FSCFG10_Msk                                       (0x1UL << CAN_FSCFG_FSCFG10_Pos)                               /*!< 0x00000400 */
#define CAN_FSCFG_FSCFG10                                           CAN_FSCFG_FSCFG10_Msk                                          /*!<Filter Scale Configuration bit 10 */
#define CAN_FSCFG_FSCFG11_Pos                                       (11U)
#define CAN_FSCFG_FSCFG11_Msk                                       (0x1UL << CAN_FSCFG_FSCFG11_Pos)                               /*!< 0x00000800 */
#define CAN_FSCFG_FSCFG11                                           CAN_FSCFG_FSCFG11_Msk                                          /*!<Filter Scale Configuration bit 11 */
#define CAN_FSCFG_FSCFG12_Pos                                       (12U)
#define CAN_FSCFG_FSCFG12_Msk                                       (0x1UL << CAN_FSCFG_FSCFG12_Pos)                               /*!< 0x00001000 */
#define CAN_FSCFG_FSCFG12                                           CAN_FSCFG_FSCFG12_Msk                                          /*!<Filter Scale Configuration bit 12 */
#define CAN_FSCFG_FSCFG13_Pos                                       (13U)
#define CAN_FSCFG_FSCFG13_Msk                                       (0x1UL << CAN_FSCFG_FSCFG13_Pos)                               /*!< 0x00002000 */
#define CAN_FSCFG_FSCFG13                                           CAN_FSCFG_FSCFG13_Msk                                          /*!<Filter Scale Configuration bit 13 */
#define CAN_FSCFG_FSCFG14_Pos                                       (14U)
#define CAN_FSCFG_FSCFG14_Msk                                       (0x1UL << CAN_FSCFG_FSCFG14_Pos)                               /*!< 0x00004000 */
#define CAN_FSCFG_FSCFG14                                           CAN_FSCFG_FSCFG14_Msk                                          /*!<Filter Scale Configuration bit 14 */
#define CAN_FSCFG_FSCFG15_Pos                                       (15U)
#define CAN_FSCFG_FSCFG15_Msk                                       (0x1UL << CAN_FSCFG_FSCFG15_Pos)                               /*!< 0x00008000 */
#define CAN_FSCFG_FSCFG15                                           CAN_FSCFG_FSCFG15_Msk                                          /*!<Filter Scale Configuration bit 15 */
#define CAN_FSCFG_FSCFG16_Pos                                       (16U)
#define CAN_FSCFG_FSCFG16_Msk                                       (0x1UL << CAN_FSCFG_FSCFG16_Pos)                               /*!< 0x00010000 */
#define CAN_FSCFG_FSCFG16                                           CAN_FSCFG_FSCFG16_Msk                                          /*!<Filter Scale Configuration bit 16 */
#define CAN_FSCFG_FSCFG17_Pos                                       (17U)
#define CAN_FSCFG_FSCFG17_Msk                                       (0x1UL << CAN_FSCFG_FSCFG17_Pos)                               /*!< 0x00020000 */
#define CAN_FSCFG_FSCFG17                                           CAN_FSCFG_FSCFG17_Msk                                          /*!<Filter Scale Configuration bit 17 */
#define CAN_FSCFG_FSCFG18_Pos                                       (18U)
#define CAN_FSCFG_FSCFG18_Msk                                       (0x1UL << CAN_FSCFG_FSCFG18_Pos)                               /*!< 0x00040000 */
#define CAN_FSCFG_FSCFG18                                           CAN_FSCFG_FSCFG18_Msk                                          /*!<Filter Scale Configuration bit 18 */
#define CAN_FSCFG_FSCFG19_Pos                                       (19U)
#define CAN_FSCFG_FSCFG19_Msk                                       (0x1UL << CAN_FSCFG_FSCFG19_Pos)                               /*!< 0x00080000 */
#define CAN_FSCFG_FSCFG19                                           CAN_FSCFG_FSCFG19_Msk                                          /*!<Filter Scale Configuration bit 19 */
#define CAN_FSCFG_FSCFG20_Pos                                       (20U)
#define CAN_FSCFG_FSCFG20_Msk                                       (0x1UL << CAN_FSCFG_FSCFG20_Pos)                               /*!< 0x00100000 */
#define CAN_FSCFG_FSCFG20                                           CAN_FSCFG_FSCFG20_Msk                                          /*!<Filter Scale Configuration bit 20 */
#define CAN_FSCFG_FSCFG21_Pos                                       (21U)
#define CAN_FSCFG_FSCFG21_Msk                                       (0x1UL << CAN_FSCFG_FSCFG21_Pos)                               /*!< 0x00200000 */
#define CAN_FSCFG_FSCFG21                                           CAN_FSCFG_FSCFG21_Msk                                          /*!<Filter Scale Configuration bit 21 */
#define CAN_FSCFG_FSCFG22_Pos                                       (22U)
#define CAN_FSCFG_FSCFG22_Msk                                       (0x1UL << CAN_FSCFG_FSCFG22_Pos)                               /*!< 0x00400000 */
#define CAN_FSCFG_FSCFG22                                           CAN_FSCFG_FSCFG22_Msk                                          /*!<Filter Scale Configuration bit 22 */
#define CAN_FSCFG_FSCFG23_Pos                                       (23U)
#define CAN_FSCFG_FSCFG23_Msk                                       (0x1UL << CAN_FSCFG_FSCFG23_Pos)                               /*!< 0x00800000 */
#define CAN_FSCFG_FSCFG23                                           CAN_FSCFG_FSCFG23_Msk                                          /*!<Filter Scale Configuration bit 23 */
#define CAN_FSCFG_FSCFG24_Pos                                       (24U)
#define CAN_FSCFG_FSCFG24_Msk                                       (0x1UL << CAN_FSCFG_FSCFG24_Pos)                               /*!< 0x01000000 */
#define CAN_FSCFG_FSCFG24                                           CAN_FSCFG_FSCFG24_Msk                                          /*!<Filter Scale Configuration bit 24 */
#define CAN_FSCFG_FSCFG25_Pos                                       (25U)
#define CAN_FSCFG_FSCFG25_Msk                                       (0x1UL << CAN_FSCFG_FSCFG25_Pos)                               /*!< 0x02000000 */
#define CAN_FSCFG_FSCFG25                                           CAN_FSCFG_FSCFG25_Msk                                          /*!<Filter Scale Configuration bit 25 */
#define CAN_FSCFG_FSCFG26_Pos                                       (26U)
#define CAN_FSCFG_FSCFG26_Msk                                       (0x1UL << CAN_FSCFG_FSCFG26_Pos)                               /*!< 0x04000000 */
#define CAN_FSCFG_FSCFG26                                           CAN_FSCFG_FSCFG26_Msk                                          /*!<Filter Scale Configuration bit 26 */
#define CAN_FSCFG_FSCFG27_Pos                                       (27U)
#define CAN_FSCFG_FSCFG27_Msk                                       (0x1UL << CAN_FSCFG_FSCFG27_Pos)                               /*!< 0x08000000 */
#define CAN_FSCFG_FSCFG27                                           CAN_FSCFG_FSCFG27_Msk                                          /*!<Filter Scale Configuration bit 27 */

/******************  Bit definition for CAN_FFASS register  *******************/
#define CAN_FFASS_FFASS_Pos                                         (0U)
#define CAN_FFASS_FFASS_Msk                                         (0xFFFFFFFUL << CAN_FFASS_FFASS_Pos)                           /*!< 0x0FFFFFFF */
#define CAN_FFASS_FFASS                                             CAN_FFASS_FFASS_Msk                                            /*!<Filter FIFO Assignment */
#define CAN_FFASS_FFASS0_Pos                                        (0U)
#define CAN_FFASS_FFASS0_Msk                                        (0x1UL << CAN_FFASS_FFASS0_Pos)                                /*!< 0x00000001 */
#define CAN_FFASS_FFASS0                                            CAN_FFASS_FFASS0_Msk                                           /*!<Filter FIFO Assignment bit 0 */
#define CAN_FFASS_FFASS1_Pos                                        (1U)
#define CAN_FFASS_FFASS1_Msk                                        (0x1UL << CAN_FFASS_FFASS1_Pos)                                /*!< 0x00000002 */
#define CAN_FFASS_FFASS1                                            CAN_FFASS_FFASS1_Msk                                           /*!<Filter FIFO Assignment bit 1 */
#define CAN_FFASS_FFASS2_Pos                                        (2U)
#define CAN_FFASS_FFASS2_Msk                                        (0x1UL << CAN_FFASS_FFASS2_Pos)                                /*!< 0x00000004 */
#define CAN_FFASS_FFASS2                                            CAN_FFASS_FFASS2_Msk                                           /*!<Filter FIFO Assignment bit 2 */
#define CAN_FFASS_FFASS3_Pos                                        (3U)
#define CAN_FFASS_FFASS3_Msk                                        (0x1UL << CAN_FFASS_FFASS3_Pos)                                /*!< 0x00000008 */
#define CAN_FFASS_FFASS3                                            CAN_FFASS_FFASS3_Msk                                           /*!<Filter FIFO Assignment bit 3 */
#define CAN_FFASS_FFASS4_Pos                                        (4U)
#define CAN_FFASS_FFASS4_Msk                                        (0x1UL << CAN_FFASS_FFASS4_Pos)                                /*!< 0x00000010 */
#define CAN_FFASS_FFASS4                                            CAN_FFASS_FFASS4_Msk                                           /*!<Filter FIFO Assignment bit 4 */
#define CAN_FFASS_FFASS5_Pos                                        (5U)
#define CAN_FFASS_FFASS5_Msk                                        (0x1UL << CAN_FFASS_FFASS5_Pos)                                /*!< 0x00000020 */
#define CAN_FFASS_FFASS5                                            CAN_FFASS_FFASS5_Msk                                           /*!<Filter FIFO Assignment bit 5 */
#define CAN_FFASS_FFASS6_Pos                                        (6U)
#define CAN_FFASS_FFASS6_Msk                                        (0x1UL << CAN_FFASS_FFASS6_Pos)                                /*!< 0x00000040 */
#define CAN_FFASS_FFASS6                                            CAN_FFASS_FFASS6_Msk                                           /*!<Filter FIFO Assignment bit 6 */
#define CAN_FFASS_FFASS7_Pos                                        (7U)
#define CAN_FFASS_FFASS7_Msk                                        (0x1UL << CAN_FFASS_FFASS7_Pos)                                /*!< 0x00000080 */
#define CAN_FFASS_FFASS7                                            CAN_FFASS_FFASS7_Msk                                           /*!<Filter FIFO Assignment bit 7 */
#define CAN_FFASS_FFASS8_Pos                                        (8U)
#define CAN_FFASS_FFASS8_Msk                                        (0x1UL << CAN_FFASS_FFASS8_Pos)                                /*!< 0x00000100 */
#define CAN_FFASS_FFASS8                                            CAN_FFASS_FFASS8_Msk                                           /*!<Filter FIFO Assignment bit 8 */
#define CAN_FFASS_FFASS9_Pos                                        (9U)
#define CAN_FFASS_FFASS9_Msk                                        (0x1UL << CAN_FFASS_FFASS9_Pos)                                /*!< 0x00000200 */
#define CAN_FFASS_FFASS9                                            CAN_FFASS_FFASS9_Msk                                           /*!<Filter FIFO Assignment bit 9 */
#define CAN_FFASS_FFASS10_Pos                                       (10U)
#define CAN_FFASS_FFASS10_Msk                                       (0x1UL << CAN_FFASS_FFASS10_Pos)                               /*!< 0x00000400 */
#define CAN_FFASS_FFASS10                                           CAN_FFASS_FFASS10_Msk                                          /*!<Filter FIFO Assignment bit 10 */
#define CAN_FFASS_FFASS11_Pos                                       (11U)
#define CAN_FFASS_FFASS11_Msk                                       (0x1UL << CAN_FFASS_FFASS11_Pos)                               /*!< 0x00000800 */
#define CAN_FFASS_FFASS11                                           CAN_FFASS_FFASS11_Msk                                          /*!<Filter FIFO Assignment bit 11 */
#define CAN_FFASS_FFASS12_Pos                                       (12U)
#define CAN_FFASS_FFASS12_Msk                                       (0x1UL << CAN_FFASS_FFASS12_Pos)                               /*!< 0x00001000 */
#define CAN_FFASS_FFASS12                                           CAN_FFASS_FFASS12_Msk                                          /*!<Filter FIFO Assignment bit 12 */
#define CAN_FFASS_FFASS13_Pos                                       (13U)
#define CAN_FFASS_FFASS13_Msk                                       (0x1UL << CAN_FFASS_FFASS13_Pos)                               /*!< 0x00002000 */
#define CAN_FFASS_FFASS13                                           CAN_FFASS_FFASS13_Msk                                          /*!<Filter FIFO Assignment bit 13 */
#define CAN_FFASS_FFASS14_Pos                                       (14U)
#define CAN_FFASS_FFASS14_Msk                                       (0x1UL << CAN_FFASS_FFASS14_Pos)                               /*!< 0x00004000 */
#define CAN_FFASS_FFASS14                                           CAN_FFASS_FFASS14_Msk                                          /*!<Filter FIFO Assignment bit 14 */
#define CAN_FFASS_FFASS15_Pos                                       (15U)
#define CAN_FFASS_FFASS15_Msk                                       (0x1UL << CAN_FFASS_FFASS15_Pos)                               /*!< 0x00008000 */
#define CAN_FFASS_FFASS15                                           CAN_FFASS_FFASS15_Msk                                          /*!<Filter FIFO Assignment bit 15 */
#define CAN_FFASS_FFASS16_Pos                                       (16U)
#define CAN_FFASS_FFASS16_Msk                                       (0x1UL << CAN_FFASS_FFASS16_Pos)                               /*!< 0x00010000 */
#define CAN_FFASS_FFASS16                                           CAN_FFASS_FFASS16_Msk                                          /*!<Filter FIFO Assignment bit 16 */
#define CAN_FFASS_FFASS17_Pos                                       (17U)
#define CAN_FFASS_FFASS17_Msk                                       (0x1UL << CAN_FFASS_FFASS17_Pos)                               /*!< 0x00020000 */
#define CAN_FFASS_FFASS17                                           CAN_FFASS_FFASS17_Msk                                          /*!<Filter FIFO Assignment bit 17 */
#define CAN_FFASS_FFASS18_Pos                                       (18U)
#define CAN_FFASS_FFASS18_Msk                                       (0x1UL << CAN_FFASS_FFASS18_Pos)                               /*!< 0x00040000 */
#define CAN_FFASS_FFASS18                                           CAN_FFASS_FFASS18_Msk                                          /*!<Filter FIFO Assignment bit 18 */
#define CAN_FFASS_FFASS19_Pos                                       (19U)
#define CAN_FFASS_FFASS19_Msk                                       (0x1UL << CAN_FFASS_FFASS19_Pos)                               /*!< 0x00080000 */
#define CAN_FFASS_FFASS19                                           CAN_FFASS_FFASS19_Msk                                          /*!<Filter FIFO Assignment bit 19 */
#define CAN_FFASS_FFASS20_Pos                                       (20U)
#define CAN_FFASS_FFASS20_Msk                                       (0x1UL << CAN_FFASS_FFASS20_Pos)                               /*!< 0x00100000 */
#define CAN_FFASS_FFASS20                                           CAN_FFASS_FFASS20_Msk                                          /*!<Filter FIFO Assignment bit 20 */
#define CAN_FFASS_FFASS21_Pos                                       (21U)
#define CAN_FFASS_FFASS21_Msk                                       (0x1UL << CAN_FFASS_FFASS21_Pos)                               /*!< 0x00200000 */
#define CAN_FFASS_FFASS21                                           CAN_FFASS_FFASS21_Msk                                          /*!<Filter FIFO Assignment bit 21 */
#define CAN_FFASS_FFASS22_Pos                                       (22U)
#define CAN_FFASS_FFASS22_Msk                                       (0x1UL << CAN_FFASS_FFASS22_Pos)                               /*!< 0x00400000 */
#define CAN_FFASS_FFASS22                                           CAN_FFASS_FFASS22_Msk                                          /*!<Filter FIFO Assignment bit 22 */
#define CAN_FFASS_FFASS23_Pos                                       (23U)
#define CAN_FFASS_FFASS23_Msk                                       (0x1UL << CAN_FFASS_FFASS23_Pos)                               /*!< 0x00800000 */
#define CAN_FFASS_FFASS23                                           CAN_FFASS_FFASS23_Msk                                          /*!<Filter FIFO Assignment bit 23 */
#define CAN_FFASS_FFASS24_Pos                                       (24U)
#define CAN_FFASS_FFASS24_Msk                                       (0x1UL << CAN_FFASS_FFASS24_Pos)                               /*!< 0x01000000 */
#define CAN_FFASS_FFASS24                                           CAN_FFASS_FFASS24_Msk                                          /*!<Filter FIFO Assignment bit 24 */
#define CAN_FFASS_FFASS25_Pos                                       (25U)
#define CAN_FFASS_FFASS25_Msk                                       (0x1UL << CAN_FFASS_FFASS25_Pos)                               /*!< 0x02000000 */
#define CAN_FFASS_FFASS25                                           CAN_FFASS_FFASS25_Msk                                          /*!<Filter FIFO Assignment bit 25 */
#define CAN_FFASS_FFASS26_Pos                                       (26U)
#define CAN_FFASS_FFASS26_Msk                                       (0x1UL << CAN_FFASS_FFASS26_Pos)                               /*!< 0x04000000 */
#define CAN_FFASS_FFASS26                                           CAN_FFASS_FFASS26_Msk                                          /*!<Filter FIFO Assignment bit 26 */
#define CAN_FFASS_FFASS27_Pos                                       (27U)
#define CAN_FFASS_FFASS27_Msk                                       (0x1UL << CAN_FFASS_FFASS27_Pos)                               /*!< 0x08000000 */
#define CAN_FFASS_FFASS27                                           CAN_FFASS_FFASS27_Msk                                          /*!<Filter FIFO Assignment bit 27 */

/*******************  Bit definition for CAN_FACT register  *******************/
#define CAN_FACT_FACT_Pos                                           (0U)
#define CAN_FACT_FACT_Msk                                           (0xFFFFFFFUL << CAN_FACT_FACT_Pos)                             /*!< 0x0FFFFFFF */
#define CAN_FACT_FACT                                               CAN_FACT_FACT_Msk                                              /*!<Filter Active */
#define CAN_FACT_FACT0_Pos                                          (0U)
#define CAN_FACT_FACT0_Msk                                          (0x1UL << CAN_FACT_FACT0_Pos)                                  /*!< 0x00000001 */
#define CAN_FACT_FACT0                                              CAN_FACT_FACT0_Msk                                             /*!<Filter Active bit 0 */
#define CAN_FACT_FACT1_Pos                                          (1U)
#define CAN_FACT_FACT1_Msk                                          (0x1UL << CAN_FACT_FACT1_Pos)                                  /*!< 0x00000002 */
#define CAN_FACT_FACT1                                              CAN_FACT_FACT1_Msk                                             /*!<Filter Active bit 1 */
#define CAN_FACT_FACT2_Pos                                          (2U)
#define CAN_FACT_FACT2_Msk                                          (0x1UL << CAN_FACT_FACT2_Pos)                                  /*!< 0x00000004 */
#define CAN_FACT_FACT2                                              CAN_FACT_FACT2_Msk                                             /*!<Filter Active bit 2 */
#define CAN_FACT_FACT3_Pos                                          (3U)
#define CAN_FACT_FACT3_Msk                                          (0x1UL << CAN_FACT_FACT3_Pos)                                  /*!< 0x00000008 */
#define CAN_FACT_FACT3                                              CAN_FACT_FACT3_Msk                                             /*!<Filter Active bit 3 */
#define CAN_FACT_FACT4_Pos                                          (4U)
#define CAN_FACT_FACT4_Msk                                          (0x1UL << CAN_FACT_FACT4_Pos)                                  /*!< 0x00000010 */
#define CAN_FACT_FACT4                                              CAN_FACT_FACT4_Msk                                             /*!<Filter Active bit 4 */
#define CAN_FACT_FACT5_Pos                                          (5U)
#define CAN_FACT_FACT5_Msk                                          (0x1UL << CAN_FACT_FACT5_Pos)                                  /*!< 0x00000020 */
#define CAN_FACT_FACT5                                              CAN_FACT_FACT5_Msk                                             /*!<Filter Active bit 5 */
#define CAN_FACT_FACT6_Pos                                          (6U)
#define CAN_FACT_FACT6_Msk                                          (0x1UL << CAN_FACT_FACT6_Pos)                                  /*!< 0x00000040 */
#define CAN_FACT_FACT6                                              CAN_FACT_FACT6_Msk                                             /*!<Filter Active bit 6 */
#define CAN_FACT_FACT7_Pos                                          (7U)
#define CAN_FACT_FACT7_Msk                                          (0x1UL << CAN_FACT_FACT7_Pos)                                  /*!< 0x00000080 */
#define CAN_FACT_FACT7                                              CAN_FACT_FACT7_Msk                                             /*!<Filter Active bit 7 */
#define CAN_FACT_FACT8_Pos                                          (8U)
#define CAN_FACT_FACT8_Msk                                          (0x1UL << CAN_FACT_FACT8_Pos)                                  /*!< 0x00000100 */
#define CAN_FACT_FACT8                                              CAN_FACT_FACT8_Msk                                             /*!<Filter Active bit 8 */
#define CAN_FACT_FACT9_Pos                                          (9U)
#define CAN_FACT_FACT9_Msk                                          (0x1UL << CAN_FACT_FACT9_Pos)                                  /*!< 0x00000200 */
#define CAN_FACT_FACT9                                              CAN_FACT_FACT9_Msk                                             /*!<Filter Active bit 9 */
#define CAN_FACT_FACT10_Pos                                         (10U)
#define CAN_FACT_FACT10_Msk                                         (0x1UL << CAN_FACT_FACT10_Pos)                                 /*!< 0x00000400 */
#define CAN_FACT_FACT10                                             CAN_FACT_FACT10_Msk                                            /*!<Filter Active bit 10 */
#define CAN_FACT_FACT11_Pos                                         (11U)
#define CAN_FACT_FACT11_Msk                                         (0x1UL << CAN_FACT_FACT11_Pos)                                 /*!< 0x00000800 */
#define CAN_FACT_FACT11                                             CAN_FACT_FACT11_Msk                                            /*!<Filter Active bit 11 */
#define CAN_FACT_FACT12_Pos                                         (12U)
#define CAN_FACT_FACT12_Msk                                         (0x1UL << CAN_FACT_FACT12_Pos)                                 /*!< 0x00001000 */
#define CAN_FACT_FACT12                                             CAN_FACT_FACT12_Msk                                            /*!<Filter Active bit 12 */
#define CAN_FACT_FACT13_Pos                                         (13U)
#define CAN_FACT_FACT13_Msk                                         (0x1UL << CAN_FACT_FACT13_Pos)                                 /*!< 0x00002000 */
#define CAN_FACT_FACT13                                             CAN_FACT_FACT13_Msk                                            /*!<Filter Active bit 13 */
#define CAN_FACT_FACT14_Pos                                         (14U)
#define CAN_FACT_FACT14_Msk                                         (0x1UL << CAN_FACT_FACT14_Pos)                                 /*!< 0x00004000 */
#define CAN_FACT_FACT14                                             CAN_FACT_FACT14_Msk                                            /*!<Filter Active bit 14 */
#define CAN_FACT_FACT15_Pos                                         (15U)
#define CAN_FACT_FACT15_Msk                                         (0x1UL << CAN_FACT_FACT15_Pos)                                 /*!< 0x00008000 */
#define CAN_FACT_FACT15                                             CAN_FACT_FACT15_Msk                                            /*!<Filter Active bit 15 */
#define CAN_FACT_FACT16_Pos                                         (16U)
#define CAN_FACT_FACT16_Msk                                         (0x1UL << CAN_FACT_FACT16_Pos)                                 /*!< 0x00010000 */
#define CAN_FACT_FACT16                                             CAN_FACT_FACT16_Msk                                            /*!<Filter Active bit 16 */
#define CAN_FACT_FACT17_Pos                                         (17U)
#define CAN_FACT_FACT17_Msk                                         (0x1UL << CAN_FACT_FACT17_Pos)                                 /*!< 0x00020000 */
#define CAN_FACT_FACT17                                             CAN_FACT_FACT17_Msk                                            /*!<Filter Active bit 17 */
#define CAN_FACT_FACT18_Pos                                         (18U)
#define CAN_FACT_FACT18_Msk                                         (0x1UL << CAN_FACT_FACT18_Pos)                                 /*!< 0x00040000 */
#define CAN_FACT_FACT18                                             CAN_FACT_FACT18_Msk                                            /*!<Filter Active bit 18 */
#define CAN_FACT_FACT19_Pos                                         (19U)
#define CAN_FACT_FACT19_Msk                                         (0x1UL << CAN_FACT_FACT19_Pos)                                 /*!< 0x00080000 */
#define CAN_FACT_FACT19                                             CAN_FACT_FACT19_Msk                                            /*!<Filter Active bit 19 */
#define CAN_FACT_FACT20_Pos                                         (20U)
#define CAN_FACT_FACT20_Msk                                         (0x1UL << CAN_FACT_FACT20_Pos)                                 /*!< 0x00100000 */
#define CAN_FACT_FACT20                                             CAN_FACT_FACT20_Msk                                            /*!<Filter Active bit 20 */
#define CAN_FACT_FACT21_Pos                                         (21U)
#define CAN_FACT_FACT21_Msk                                         (0x1UL << CAN_FACT_FACT21_Pos)                                 /*!< 0x00200000 */
#define CAN_FACT_FACT21                                             CAN_FACT_FACT21_Msk                                            /*!<Filter Active bit 21 */
#define CAN_FACT_FACT22_Pos                                         (22U)
#define CAN_FACT_FACT22_Msk                                         (0x1UL << CAN_FACT_FACT22_Pos)                                 /*!< 0x00400000 */
#define CAN_FACT_FACT22                                             CAN_FACT_FACT22_Msk                                            /*!<Filter Active bit 22 */
#define CAN_FACT_FACT23_Pos                                         (23U)
#define CAN_FACT_FACT23_Msk                                         (0x1UL << CAN_FACT_FACT23_Pos)                                 /*!< 0x00800000 */
#define CAN_FACT_FACT23                                             CAN_FACT_FACT23_Msk                                            /*!<Filter Active bit 23 */
#define CAN_FACT_FACT24_Pos                                         (24U)
#define CAN_FACT_FACT24_Msk                                         (0x1UL << CAN_FACT_FACT24_Pos)                                 /*!< 0x01000000 */
#define CAN_FACT_FACT24                                             CAN_FACT_FACT24_Msk                                            /*!<Filter Active bit 24 */
#define CAN_FACT_FACT25_Pos                                         (25U)
#define CAN_FACT_FACT25_Msk                                         (0x1UL << CAN_FACT_FACT25_Pos)                                 /*!< 0x02000000 */
#define CAN_FACT_FACT25                                             CAN_FACT_FACT25_Msk                                            /*!<Filter Active bit 25 */
#define CAN_FACT_FACT26_Pos                                         (26U)
#define CAN_FACT_FACT26_Msk                                         (0x1UL << CAN_FACT_FACT26_Pos)                                 /*!< 0x04000000 */
#define CAN_FACT_FACT26                                             CAN_FACT_FACT26_Msk                                            /*!<Filter Active bit 26 */
#define CAN_FACT_FACT27_Pos                                         (27U)
#define CAN_FACT_FACT27_Msk                                         (0x1UL << CAN_FACT_FACT27_Pos)                                 /*!< 0x08000000 */
#define CAN_FACT_FACT27                                             CAN_FACT_FACT27_Msk                                            /*!<Filter Active bit 27 */


/*******************  Bit definition for CAN_F0BANK1 register  *******************/
#define CAN_F0BANK1_FBIT0_Pos                                       (0U)
#define CAN_F0BANK1_FBIT0_Msk                                       (0x1UL << CAN_F0BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F0BANK1_FBIT0                                           CAN_F0BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F0BANK1_FBIT1_Pos                                       (1U)
#define CAN_F0BANK1_FBIT1_Msk                                       (0x1UL << CAN_F0BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F0BANK1_FBIT1                                           CAN_F0BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F0BANK1_FBIT2_Pos                                       (2U)
#define CAN_F0BANK1_FBIT2_Msk                                       (0x1UL << CAN_F0BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F0BANK1_FBIT2                                           CAN_F0BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F0BANK1_FBIT3_Pos                                       (3U)
#define CAN_F0BANK1_FBIT3_Msk                                       (0x1UL << CAN_F0BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F0BANK1_FBIT3                                           CAN_F0BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F0BANK1_FBIT4_Pos                                       (4U)
#define CAN_F0BANK1_FBIT4_Msk                                       (0x1UL << CAN_F0BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F0BANK1_FBIT4                                           CAN_F0BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F0BANK1_FBIT5_Pos                                       (5U)
#define CAN_F0BANK1_FBIT5_Msk                                       (0x1UL << CAN_F0BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F0BANK1_FBIT5                                           CAN_F0BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F0BANK1_FBIT6_Pos                                       (6U)
#define CAN_F0BANK1_FBIT6_Msk                                       (0x1UL << CAN_F0BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F0BANK1_FBIT6                                           CAN_F0BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F0BANK1_FBIT7_Pos                                       (7U)
#define CAN_F0BANK1_FBIT7_Msk                                       (0x1UL << CAN_F0BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F0BANK1_FBIT7                                           CAN_F0BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F0BANK1_FBIT8_Pos                                       (8U)
#define CAN_F0BANK1_FBIT8_Msk                                       (0x1UL << CAN_F0BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F0BANK1_FBIT8                                           CAN_F0BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F0BANK1_FBIT9_Pos                                       (9U)
#define CAN_F0BANK1_FBIT9_Msk                                       (0x1UL << CAN_F0BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F0BANK1_FBIT9                                           CAN_F0BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F0BANK1_FBIT10_Pos                                      (10U)
#define CAN_F0BANK1_FBIT10_Msk                                      (0x1UL << CAN_F0BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F0BANK1_FBIT10                                          CAN_F0BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F0BANK1_FBIT11_Pos                                      (11U)
#define CAN_F0BANK1_FBIT11_Msk                                      (0x1UL << CAN_F0BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F0BANK1_FBIT11                                          CAN_F0BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F0BANK1_FBIT12_Pos                                      (12U)
#define CAN_F0BANK1_FBIT12_Msk                                      (0x1UL << CAN_F0BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F0BANK1_FBIT12                                          CAN_F0BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F0BANK1_FBIT13_Pos                                      (13U)
#define CAN_F0BANK1_FBIT13_Msk                                      (0x1UL << CAN_F0BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F0BANK1_FBIT13                                          CAN_F0BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F0BANK1_FBIT14_Pos                                      (14U)
#define CAN_F0BANK1_FBIT14_Msk                                      (0x1UL << CAN_F0BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F0BANK1_FBIT14                                          CAN_F0BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F0BANK1_FBIT15_Pos                                      (15U)
#define CAN_F0BANK1_FBIT15_Msk                                      (0x1UL << CAN_F0BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F0BANK1_FBIT15                                          CAN_F0BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F0BANK1_FBIT16_Pos                                      (16U)
#define CAN_F0BANK1_FBIT16_Msk                                      (0x1UL << CAN_F0BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F0BANK1_FBIT16                                          CAN_F0BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F0BANK1_FBIT17_Pos                                      (17U)
#define CAN_F0BANK1_FBIT17_Msk                                      (0x1UL << CAN_F0BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F0BANK1_FBIT17                                          CAN_F0BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F0BANK1_FBIT18_Pos                                      (18U)
#define CAN_F0BANK1_FBIT18_Msk                                      (0x1UL << CAN_F0BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F0BANK1_FBIT18                                          CAN_F0BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F0BANK1_FBIT19_Pos                                      (19U)
#define CAN_F0BANK1_FBIT19_Msk                                      (0x1UL << CAN_F0BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F0BANK1_FBIT19                                          CAN_F0BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F0BANK1_FBIT20_Pos                                      (20U)
#define CAN_F0BANK1_FBIT20_Msk                                      (0x1UL << CAN_F0BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F0BANK1_FBIT20                                          CAN_F0BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F0BANK1_FBIT21_Pos                                      (21U)
#define CAN_F0BANK1_FBIT21_Msk                                      (0x1UL << CAN_F0BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F0BANK1_FBIT21                                          CAN_F0BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F0BANK1_FBIT22_Pos                                      (22U)
#define CAN_F0BANK1_FBIT22_Msk                                      (0x1UL << CAN_F0BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F0BANK1_FBIT22                                          CAN_F0BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F0BANK1_FBIT23_Pos                                      (23U)
#define CAN_F0BANK1_FBIT23_Msk                                      (0x1UL << CAN_F0BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F0BANK1_FBIT23                                          CAN_F0BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F0BANK1_FBIT24_Pos                                      (24U)
#define CAN_F0BANK1_FBIT24_Msk                                      (0x1UL << CAN_F0BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F0BANK1_FBIT24                                          CAN_F0BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F0BANK1_FBIT25_Pos                                      (25U)
#define CAN_F0BANK1_FBIT25_Msk                                      (0x1UL << CAN_F0BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F0BANK1_FBIT25                                          CAN_F0BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F0BANK1_FBIT26_Pos                                      (26U)
#define CAN_F0BANK1_FBIT26_Msk                                      (0x1UL << CAN_F0BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F0BANK1_FBIT26                                          CAN_F0BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F0BANK1_FBIT27_Pos                                      (27U)
#define CAN_F0BANK1_FBIT27_Msk                                      (0x1UL << CAN_F0BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F0BANK1_FBIT27                                          CAN_F0BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F0BANK1_FBIT28_Pos                                      (28U)
#define CAN_F0BANK1_FBIT28_Msk                                      (0x1UL << CAN_F0BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F0BANK1_FBIT28                                          CAN_F0BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F0BANK1_FBIT29_Pos                                      (29U)
#define CAN_F0BANK1_FBIT29_Msk                                      (0x1UL << CAN_F0BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F0BANK1_FBIT29                                          CAN_F0BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F0BANK1_FBIT30_Pos                                      (30U)
#define CAN_F0BANK1_FBIT30_Msk                                      (0x1UL << CAN_F0BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F0BANK1_FBIT30                                          CAN_F0BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F0BANK1_FBIT31_Pos                                      (31U)
#define CAN_F0BANK1_FBIT31_Msk                                      (0x1UL << CAN_F0BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F0BANK1_FBIT31                                          CAN_F0BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F1BANK1 register  *******************/
#define CAN_F1BANK1_FBIT0_Pos                                       (0U)
#define CAN_F1BANK1_FBIT0_Msk                                       (0x1UL << CAN_F1BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F1BANK1_FBIT0                                           CAN_F1BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F1BANK1_FBIT1_Pos                                       (1U)
#define CAN_F1BANK1_FBIT1_Msk                                       (0x1UL << CAN_F1BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F1BANK1_FBIT1                                           CAN_F1BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F1BANK1_FBIT2_Pos                                       (2U)
#define CAN_F1BANK1_FBIT2_Msk                                       (0x1UL << CAN_F1BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F1BANK1_FBIT2                                           CAN_F1BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F1BANK1_FBIT3_Pos                                       (3U)
#define CAN_F1BANK1_FBIT3_Msk                                       (0x1UL << CAN_F1BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F1BANK1_FBIT3                                           CAN_F1BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F1BANK1_FBIT4_Pos                                       (4U)
#define CAN_F1BANK1_FBIT4_Msk                                       (0x1UL << CAN_F1BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F1BANK1_FBIT4                                           CAN_F1BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F1BANK1_FBIT5_Pos                                       (5U)
#define CAN_F1BANK1_FBIT5_Msk                                       (0x1UL << CAN_F1BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F1BANK1_FBIT5                                           CAN_F1BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F1BANK1_FBIT6_Pos                                       (6U)
#define CAN_F1BANK1_FBIT6_Msk                                       (0x1UL << CAN_F1BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F1BANK1_FBIT6                                           CAN_F1BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F1BANK1_FBIT7_Pos                                       (7U)
#define CAN_F1BANK1_FBIT7_Msk                                       (0x1UL << CAN_F1BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F1BANK1_FBIT7                                           CAN_F1BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F1BANK1_FBIT8_Pos                                       (8U)
#define CAN_F1BANK1_FBIT8_Msk                                       (0x1UL << CAN_F1BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F1BANK1_FBIT8                                           CAN_F1BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F1BANK1_FBIT9_Pos                                       (9U)
#define CAN_F1BANK1_FBIT9_Msk                                       (0x1UL << CAN_F1BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F1BANK1_FBIT9                                           CAN_F1BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F1BANK1_FBIT10_Pos                                      (10U)
#define CAN_F1BANK1_FBIT10_Msk                                      (0x1UL << CAN_F1BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F1BANK1_FBIT10                                          CAN_F1BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F1BANK1_FBIT11_Pos                                      (11U)
#define CAN_F1BANK1_FBIT11_Msk                                      (0x1UL << CAN_F1BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F1BANK1_FBIT11                                          CAN_F1BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F1BANK1_FBIT12_Pos                                      (12U)
#define CAN_F1BANK1_FBIT12_Msk                                      (0x1UL << CAN_F1BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F1BANK1_FBIT12                                          CAN_F1BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F1BANK1_FBIT13_Pos                                      (13U)
#define CAN_F1BANK1_FBIT13_Msk                                      (0x1UL << CAN_F1BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F1BANK1_FBIT13                                          CAN_F1BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F1BANK1_FBIT14_Pos                                      (14U)
#define CAN_F1BANK1_FBIT14_Msk                                      (0x1UL << CAN_F1BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F1BANK1_FBIT14                                          CAN_F1BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F1BANK1_FBIT15_Pos                                      (15U)
#define CAN_F1BANK1_FBIT15_Msk                                      (0x1UL << CAN_F1BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F1BANK1_FBIT15                                          CAN_F1BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F1BANK1_FBIT16_Pos                                      (16U)
#define CAN_F1BANK1_FBIT16_Msk                                      (0x1UL << CAN_F1BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F1BANK1_FBIT16                                          CAN_F1BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F1BANK1_FBIT17_Pos                                      (17U)
#define CAN_F1BANK1_FBIT17_Msk                                      (0x1UL << CAN_F1BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F1BANK1_FBIT17                                          CAN_F1BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F1BANK1_FBIT18_Pos                                      (18U)
#define CAN_F1BANK1_FBIT18_Msk                                      (0x1UL << CAN_F1BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F1BANK1_FBIT18                                          CAN_F1BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F1BANK1_FBIT19_Pos                                      (19U)
#define CAN_F1BANK1_FBIT19_Msk                                      (0x1UL << CAN_F1BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F1BANK1_FBIT19                                          CAN_F1BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F1BANK1_FBIT20_Pos                                      (20U)
#define CAN_F1BANK1_FBIT20_Msk                                      (0x1UL << CAN_F1BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F1BANK1_FBIT20                                          CAN_F1BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F1BANK1_FBIT21_Pos                                      (21U)
#define CAN_F1BANK1_FBIT21_Msk                                      (0x1UL << CAN_F1BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F1BANK1_FBIT21                                          CAN_F1BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F1BANK1_FBIT22_Pos                                      (22U)
#define CAN_F1BANK1_FBIT22_Msk                                      (0x1UL << CAN_F1BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F1BANK1_FBIT22                                          CAN_F1BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F1BANK1_FBIT23_Pos                                      (23U)
#define CAN_F1BANK1_FBIT23_Msk                                      (0x1UL << CAN_F1BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F1BANK1_FBIT23                                          CAN_F1BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F1BANK1_FBIT24_Pos                                      (24U)
#define CAN_F1BANK1_FBIT24_Msk                                      (0x1UL << CAN_F1BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F1BANK1_FBIT24                                          CAN_F1BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F1BANK1_FBIT25_Pos                                      (25U)
#define CAN_F1BANK1_FBIT25_Msk                                      (0x1UL << CAN_F1BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F1BANK1_FBIT25                                          CAN_F1BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F1BANK1_FBIT26_Pos                                      (26U)
#define CAN_F1BANK1_FBIT26_Msk                                      (0x1UL << CAN_F1BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F1BANK1_FBIT26                                          CAN_F1BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F1BANK1_FBIT27_Pos                                      (27U)
#define CAN_F1BANK1_FBIT27_Msk                                      (0x1UL << CAN_F1BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F1BANK1_FBIT27                                          CAN_F1BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F1BANK1_FBIT28_Pos                                      (28U)
#define CAN_F1BANK1_FBIT28_Msk                                      (0x1UL << CAN_F1BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F1BANK1_FBIT28                                          CAN_F1BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F1BANK1_FBIT29_Pos                                      (29U)
#define CAN_F1BANK1_FBIT29_Msk                                      (0x1UL << CAN_F1BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F1BANK1_FBIT29                                          CAN_F1BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F1BANK1_FBIT30_Pos                                      (30U)
#define CAN_F1BANK1_FBIT30_Msk                                      (0x1UL << CAN_F1BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F1BANK1_FBIT30                                          CAN_F1BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F1BANK1_FBIT31_Pos                                      (31U)
#define CAN_F1BANK1_FBIT31_Msk                                      (0x1UL << CAN_F1BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F1BANK1_FBIT31                                          CAN_F1BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F2BANK1 register  *******************/
#define CAN_F2BANK1_FBIT0_Pos                                       (0U)
#define CAN_F2BANK1_FBIT0_Msk                                       (0x1UL << CAN_F2BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F2BANK1_FBIT0                                           CAN_F2BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F2BANK1_FBIT1_Pos                                       (1U)
#define CAN_F2BANK1_FBIT1_Msk                                       (0x1UL << CAN_F2BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F2BANK1_FBIT1                                           CAN_F2BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F2BANK1_FBIT2_Pos                                       (2U)
#define CAN_F2BANK1_FBIT2_Msk                                       (0x1UL << CAN_F2BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F2BANK1_FBIT2                                           CAN_F2BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F2BANK1_FBIT3_Pos                                       (3U)
#define CAN_F2BANK1_FBIT3_Msk                                       (0x1UL << CAN_F2BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F2BANK1_FBIT3                                           CAN_F2BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F2BANK1_FBIT4_Pos                                       (4U)
#define CAN_F2BANK1_FBIT4_Msk                                       (0x1UL << CAN_F2BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F2BANK1_FBIT4                                           CAN_F2BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F2BANK1_FBIT5_Pos                                       (5U)
#define CAN_F2BANK1_FBIT5_Msk                                       (0x1UL << CAN_F2BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F2BANK1_FBIT5                                           CAN_F2BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F2BANK1_FBIT6_Pos                                       (6U)
#define CAN_F2BANK1_FBIT6_Msk                                       (0x1UL << CAN_F2BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F2BANK1_FBIT6                                           CAN_F2BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F2BANK1_FBIT7_Pos                                       (7U)
#define CAN_F2BANK1_FBIT7_Msk                                       (0x1UL << CAN_F2BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F2BANK1_FBIT7                                           CAN_F2BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F2BANK1_FBIT8_Pos                                       (8U)
#define CAN_F2BANK1_FBIT8_Msk                                       (0x1UL << CAN_F2BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F2BANK1_FBIT8                                           CAN_F2BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F2BANK1_FBIT9_Pos                                       (9U)
#define CAN_F2BANK1_FBIT9_Msk                                       (0x1UL << CAN_F2BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F2BANK1_FBIT9                                           CAN_F2BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F2BANK1_FBIT10_Pos                                      (10U)
#define CAN_F2BANK1_FBIT10_Msk                                      (0x1UL << CAN_F2BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F2BANK1_FBIT10                                          CAN_F2BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F2BANK1_FBIT11_Pos                                      (11U)
#define CAN_F2BANK1_FBIT11_Msk                                      (0x1UL << CAN_F2BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F2BANK1_FBIT11                                          CAN_F2BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F2BANK1_FBIT12_Pos                                      (12U)
#define CAN_F2BANK1_FBIT12_Msk                                      (0x1UL << CAN_F2BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F2BANK1_FBIT12                                          CAN_F2BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F2BANK1_FBIT13_Pos                                      (13U)
#define CAN_F2BANK1_FBIT13_Msk                                      (0x1UL << CAN_F2BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F2BANK1_FBIT13                                          CAN_F2BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F2BANK1_FBIT14_Pos                                      (14U)
#define CAN_F2BANK1_FBIT14_Msk                                      (0x1UL << CAN_F2BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F2BANK1_FBIT14                                          CAN_F2BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F2BANK1_FBIT15_Pos                                      (15U)
#define CAN_F2BANK1_FBIT15_Msk                                      (0x1UL << CAN_F2BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F2BANK1_FBIT15                                          CAN_F2BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F2BANK1_FBIT16_Pos                                      (16U)
#define CAN_F2BANK1_FBIT16_Msk                                      (0x1UL << CAN_F2BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F2BANK1_FBIT16                                          CAN_F2BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F2BANK1_FBIT17_Pos                                      (17U)
#define CAN_F2BANK1_FBIT17_Msk                                      (0x1UL << CAN_F2BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F2BANK1_FBIT17                                          CAN_F2BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F2BANK1_FBIT18_Pos                                      (18U)
#define CAN_F2BANK1_FBIT18_Msk                                      (0x1UL << CAN_F2BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F2BANK1_FBIT18                                          CAN_F2BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F2BANK1_FBIT19_Pos                                      (19U)
#define CAN_F2BANK1_FBIT19_Msk                                      (0x1UL << CAN_F2BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F2BANK1_FBIT19                                          CAN_F2BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F2BANK1_FBIT20_Pos                                      (20U)
#define CAN_F2BANK1_FBIT20_Msk                                      (0x1UL << CAN_F2BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F2BANK1_FBIT20                                          CAN_F2BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F2BANK1_FBIT21_Pos                                      (21U)
#define CAN_F2BANK1_FBIT21_Msk                                      (0x1UL << CAN_F2BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F2BANK1_FBIT21                                          CAN_F2BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F2BANK1_FBIT22_Pos                                      (22U)
#define CAN_F2BANK1_FBIT22_Msk                                      (0x1UL << CAN_F2BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F2BANK1_FBIT22                                          CAN_F2BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F2BANK1_FBIT23_Pos                                      (23U)
#define CAN_F2BANK1_FBIT23_Msk                                      (0x1UL << CAN_F2BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F2BANK1_FBIT23                                          CAN_F2BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F2BANK1_FBIT24_Pos                                      (24U)
#define CAN_F2BANK1_FBIT24_Msk                                      (0x1UL << CAN_F2BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F2BANK1_FBIT24                                          CAN_F2BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F2BANK1_FBIT25_Pos                                      (25U)
#define CAN_F2BANK1_FBIT25_Msk                                      (0x1UL << CAN_F2BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F2BANK1_FBIT25                                          CAN_F2BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F2BANK1_FBIT26_Pos                                      (26U)
#define CAN_F2BANK1_FBIT26_Msk                                      (0x1UL << CAN_F2BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F2BANK1_FBIT26                                          CAN_F2BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F2BANK1_FBIT27_Pos                                      (27U)
#define CAN_F2BANK1_FBIT27_Msk                                      (0x1UL << CAN_F2BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F2BANK1_FBIT27                                          CAN_F2BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F2BANK1_FBIT28_Pos                                      (28U)
#define CAN_F2BANK1_FBIT28_Msk                                      (0x1UL << CAN_F2BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F2BANK1_FBIT28                                          CAN_F2BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F2BANK1_FBIT29_Pos                                      (29U)
#define CAN_F2BANK1_FBIT29_Msk                                      (0x1UL << CAN_F2BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F2BANK1_FBIT29                                          CAN_F2BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F2BANK1_FBIT30_Pos                                      (30U)
#define CAN_F2BANK1_FBIT30_Msk                                      (0x1UL << CAN_F2BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F2BANK1_FBIT30                                          CAN_F2BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F2BANK1_FBIT31_Pos                                      (31U)
#define CAN_F2BANK1_FBIT31_Msk                                      (0x1UL << CAN_F2BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F2BANK1_FBIT31                                          CAN_F2BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F3BANK1 register  *******************/
#define CAN_F3BANK1_FBIT0_Pos                                       (0U)
#define CAN_F3BANK1_FBIT0_Msk                                       (0x1UL << CAN_F3BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F3BANK1_FBIT0                                           CAN_F3BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F3BANK1_FBIT1_Pos                                       (1U)
#define CAN_F3BANK1_FBIT1_Msk                                       (0x1UL << CAN_F3BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F3BANK1_FBIT1                                           CAN_F3BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F3BANK1_FBIT2_Pos                                       (2U)
#define CAN_F3BANK1_FBIT2_Msk                                       (0x1UL << CAN_F3BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F3BANK1_FBIT2                                           CAN_F3BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F3BANK1_FBIT3_Pos                                       (3U)
#define CAN_F3BANK1_FBIT3_Msk                                       (0x1UL << CAN_F3BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F3BANK1_FBIT3                                           CAN_F3BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F3BANK1_FBIT4_Pos                                       (4U)
#define CAN_F3BANK1_FBIT4_Msk                                       (0x1UL << CAN_F3BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F3BANK1_FBIT4                                           CAN_F3BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F3BANK1_FBIT5_Pos                                       (5U)
#define CAN_F3BANK1_FBIT5_Msk                                       (0x1UL << CAN_F3BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F3BANK1_FBIT5                                           CAN_F3BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F3BANK1_FBIT6_Pos                                       (6U)
#define CAN_F3BANK1_FBIT6_Msk                                       (0x1UL << CAN_F3BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F3BANK1_FBIT6                                           CAN_F3BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F3BANK1_FBIT7_Pos                                       (7U)
#define CAN_F3BANK1_FBIT7_Msk                                       (0x1UL << CAN_F3BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F3BANK1_FBIT7                                           CAN_F3BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F3BANK1_FBIT8_Pos                                       (8U)
#define CAN_F3BANK1_FBIT8_Msk                                       (0x1UL << CAN_F3BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F3BANK1_FBIT8                                           CAN_F3BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F3BANK1_FBIT9_Pos                                       (9U)
#define CAN_F3BANK1_FBIT9_Msk                                       (0x1UL << CAN_F3BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F3BANK1_FBIT9                                           CAN_F3BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F3BANK1_FBIT10_Pos                                      (10U)
#define CAN_F3BANK1_FBIT10_Msk                                      (0x1UL << CAN_F3BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F3BANK1_FBIT10                                          CAN_F3BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F3BANK1_FBIT11_Pos                                      (11U)
#define CAN_F3BANK1_FBIT11_Msk                                      (0x1UL << CAN_F3BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F3BANK1_FBIT11                                          CAN_F3BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F3BANK1_FBIT12_Pos                                      (12U)
#define CAN_F3BANK1_FBIT12_Msk                                      (0x1UL << CAN_F3BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F3BANK1_FBIT12                                          CAN_F3BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F3BANK1_FBIT13_Pos                                      (13U)
#define CAN_F3BANK1_FBIT13_Msk                                      (0x1UL << CAN_F3BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F3BANK1_FBIT13                                          CAN_F3BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F3BANK1_FBIT14_Pos                                      (14U)
#define CAN_F3BANK1_FBIT14_Msk                                      (0x1UL << CAN_F3BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F3BANK1_FBIT14                                          CAN_F3BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F3BANK1_FBIT15_Pos                                      (15U)
#define CAN_F3BANK1_FBIT15_Msk                                      (0x1UL << CAN_F3BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F3BANK1_FBIT15                                          CAN_F3BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F3BANK1_FBIT16_Pos                                      (16U)
#define CAN_F3BANK1_FBIT16_Msk                                      (0x1UL << CAN_F3BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F3BANK1_FBIT16                                          CAN_F3BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F3BANK1_FBIT17_Pos                                      (17U)
#define CAN_F3BANK1_FBIT17_Msk                                      (0x1UL << CAN_F3BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F3BANK1_FBIT17                                          CAN_F3BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F3BANK1_FBIT18_Pos                                      (18U)
#define CAN_F3BANK1_FBIT18_Msk                                      (0x1UL << CAN_F3BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F3BANK1_FBIT18                                          CAN_F3BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F3BANK1_FBIT19_Pos                                      (19U)
#define CAN_F3BANK1_FBIT19_Msk                                      (0x1UL << CAN_F3BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F3BANK1_FBIT19                                          CAN_F3BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F3BANK1_FBIT20_Pos                                      (20U)
#define CAN_F3BANK1_FBIT20_Msk                                      (0x1UL << CAN_F3BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F3BANK1_FBIT20                                          CAN_F3BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F3BANK1_FBIT21_Pos                                      (21U)
#define CAN_F3BANK1_FBIT21_Msk                                      (0x1UL << CAN_F3BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F3BANK1_FBIT21                                          CAN_F3BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F3BANK1_FBIT22_Pos                                      (22U)
#define CAN_F3BANK1_FBIT22_Msk                                      (0x1UL << CAN_F3BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F3BANK1_FBIT22                                          CAN_F3BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F3BANK1_FBIT23_Pos                                      (23U)
#define CAN_F3BANK1_FBIT23_Msk                                      (0x1UL << CAN_F3BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F3BANK1_FBIT23                                          CAN_F3BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F3BANK1_FBIT24_Pos                                      (24U)
#define CAN_F3BANK1_FBIT24_Msk                                      (0x1UL << CAN_F3BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F3BANK1_FBIT24                                          CAN_F3BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F3BANK1_FBIT25_Pos                                      (25U)
#define CAN_F3BANK1_FBIT25_Msk                                      (0x1UL << CAN_F3BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F3BANK1_FBIT25                                          CAN_F3BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F3BANK1_FBIT26_Pos                                      (26U)
#define CAN_F3BANK1_FBIT26_Msk                                      (0x1UL << CAN_F3BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F3BANK1_FBIT26                                          CAN_F3BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F3BANK1_FBIT27_Pos                                      (27U)
#define CAN_F3BANK1_FBIT27_Msk                                      (0x1UL << CAN_F3BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F3BANK1_FBIT27                                          CAN_F3BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F3BANK1_FBIT28_Pos                                      (28U)
#define CAN_F3BANK1_FBIT28_Msk                                      (0x1UL << CAN_F3BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F3BANK1_FBIT28                                          CAN_F3BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F3BANK1_FBIT29_Pos                                      (29U)
#define CAN_F3BANK1_FBIT29_Msk                                      (0x1UL << CAN_F3BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F3BANK1_FBIT29                                          CAN_F3BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F3BANK1_FBIT30_Pos                                      (30U)
#define CAN_F3BANK1_FBIT30_Msk                                      (0x1UL << CAN_F3BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F3BANK1_FBIT30                                          CAN_F3BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F3BANK1_FBIT31_Pos                                      (31U)
#define CAN_F3BANK1_FBIT31_Msk                                      (0x1UL << CAN_F3BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F3BANK1_FBIT31                                          CAN_F3BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F4BANK1 register  *******************/
#define CAN_F4BANK1_FBIT0_Pos                                       (0U)
#define CAN_F4BANK1_FBIT0_Msk                                       (0x1UL << CAN_F4BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F4BANK1_FBIT0                                           CAN_F4BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F4BANK1_FBIT1_Pos                                       (1U)
#define CAN_F4BANK1_FBIT1_Msk                                       (0x1UL << CAN_F4BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F4BANK1_FBIT1                                           CAN_F4BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F4BANK1_FBIT2_Pos                                       (2U)
#define CAN_F4BANK1_FBIT2_Msk                                       (0x1UL << CAN_F4BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F4BANK1_FBIT2                                           CAN_F4BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F4BANK1_FBIT3_Pos                                       (3U)
#define CAN_F4BANK1_FBIT3_Msk                                       (0x1UL << CAN_F4BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F4BANK1_FBIT3                                           CAN_F4BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F4BANK1_FBIT4_Pos                                       (4U)
#define CAN_F4BANK1_FBIT4_Msk                                       (0x1UL << CAN_F4BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F4BANK1_FBIT4                                           CAN_F4BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F4BANK1_FBIT5_Pos                                       (5U)
#define CAN_F4BANK1_FBIT5_Msk                                       (0x1UL << CAN_F4BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F4BANK1_FBIT5                                           CAN_F4BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F4BANK1_FBIT6_Pos                                       (6U)
#define CAN_F4BANK1_FBIT6_Msk                                       (0x1UL << CAN_F4BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F4BANK1_FBIT6                                           CAN_F4BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F4BANK1_FBIT7_Pos                                       (7U)
#define CAN_F4BANK1_FBIT7_Msk                                       (0x1UL << CAN_F4BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F4BANK1_FBIT7                                           CAN_F4BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F4BANK1_FBIT8_Pos                                       (8U)
#define CAN_F4BANK1_FBIT8_Msk                                       (0x1UL << CAN_F4BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F4BANK1_FBIT8                                           CAN_F4BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F4BANK1_FBIT9_Pos                                       (9U)
#define CAN_F4BANK1_FBIT9_Msk                                       (0x1UL << CAN_F4BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F4BANK1_FBIT9                                           CAN_F4BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F4BANK1_FBIT10_Pos                                      (10U)
#define CAN_F4BANK1_FBIT10_Msk                                      (0x1UL << CAN_F4BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F4BANK1_FBIT10                                          CAN_F4BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F4BANK1_FBIT11_Pos                                      (11U)
#define CAN_F4BANK1_FBIT11_Msk                                      (0x1UL << CAN_F4BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F4BANK1_FBIT11                                          CAN_F4BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F4BANK1_FBIT12_Pos                                      (12U)
#define CAN_F4BANK1_FBIT12_Msk                                      (0x1UL << CAN_F4BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F4BANK1_FBIT12                                          CAN_F4BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F4BANK1_FBIT13_Pos                                      (13U)
#define CAN_F4BANK1_FBIT13_Msk                                      (0x1UL << CAN_F4BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F4BANK1_FBIT13                                          CAN_F4BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F4BANK1_FBIT14_Pos                                      (14U)
#define CAN_F4BANK1_FBIT14_Msk                                      (0x1UL << CAN_F4BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F4BANK1_FBIT14                                          CAN_F4BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F4BANK1_FBIT15_Pos                                      (15U)
#define CAN_F4BANK1_FBIT15_Msk                                      (0x1UL << CAN_F4BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F4BANK1_FBIT15                                          CAN_F4BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F4BANK1_FBIT16_Pos                                      (16U)
#define CAN_F4BANK1_FBIT16_Msk                                      (0x1UL << CAN_F4BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F4BANK1_FBIT16                                          CAN_F4BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F4BANK1_FBIT17_Pos                                      (17U)
#define CAN_F4BANK1_FBIT17_Msk                                      (0x1UL << CAN_F4BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F4BANK1_FBIT17                                          CAN_F4BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F4BANK1_FBIT18_Pos                                      (18U)
#define CAN_F4BANK1_FBIT18_Msk                                      (0x1UL << CAN_F4BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F4BANK1_FBIT18                                          CAN_F4BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F4BANK1_FBIT19_Pos                                      (19U)
#define CAN_F4BANK1_FBIT19_Msk                                      (0x1UL << CAN_F4BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F4BANK1_FBIT19                                          CAN_F4BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F4BANK1_FBIT20_Pos                                      (20U)
#define CAN_F4BANK1_FBIT20_Msk                                      (0x1UL << CAN_F4BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F4BANK1_FBIT20                                          CAN_F4BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F4BANK1_FBIT21_Pos                                      (21U)
#define CAN_F4BANK1_FBIT21_Msk                                      (0x1UL << CAN_F4BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F4BANK1_FBIT21                                          CAN_F4BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F4BANK1_FBIT22_Pos                                      (22U)
#define CAN_F4BANK1_FBIT22_Msk                                      (0x1UL << CAN_F4BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F4BANK1_FBIT22                                          CAN_F4BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F4BANK1_FBIT23_Pos                                      (23U)
#define CAN_F4BANK1_FBIT23_Msk                                      (0x1UL << CAN_F4BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F4BANK1_FBIT23                                          CAN_F4BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F4BANK1_FBIT24_Pos                                      (24U)
#define CAN_F4BANK1_FBIT24_Msk                                      (0x1UL << CAN_F4BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F4BANK1_FBIT24                                          CAN_F4BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F4BANK1_FBIT25_Pos                                      (25U)
#define CAN_F4BANK1_FBIT25_Msk                                      (0x1UL << CAN_F4BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F4BANK1_FBIT25                                          CAN_F4BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F4BANK1_FBIT26_Pos                                      (26U)
#define CAN_F4BANK1_FBIT26_Msk                                      (0x1UL << CAN_F4BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F4BANK1_FBIT26                                          CAN_F4BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F4BANK1_FBIT27_Pos                                      (27U)
#define CAN_F4BANK1_FBIT27_Msk                                      (0x1UL << CAN_F4BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F4BANK1_FBIT27                                          CAN_F4BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F4BANK1_FBIT28_Pos                                      (28U)
#define CAN_F4BANK1_FBIT28_Msk                                      (0x1UL << CAN_F4BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F4BANK1_FBIT28                                          CAN_F4BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F4BANK1_FBIT29_Pos                                      (29U)
#define CAN_F4BANK1_FBIT29_Msk                                      (0x1UL << CAN_F4BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F4BANK1_FBIT29                                          CAN_F4BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F4BANK1_FBIT30_Pos                                      (30U)
#define CAN_F4BANK1_FBIT30_Msk                                      (0x1UL << CAN_F4BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F4BANK1_FBIT30                                          CAN_F4BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F4BANK1_FBIT31_Pos                                      (31U)
#define CAN_F4BANK1_FBIT31_Msk                                      (0x1UL << CAN_F4BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F4BANK1_FBIT31                                          CAN_F4BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F5BANK1 register  *******************/
#define CAN_F5BANK1_FBIT0_Pos                                       (0U)
#define CAN_F5BANK1_FBIT0_Msk                                       (0x1UL << CAN_F5BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F5BANK1_FBIT0                                           CAN_F5BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F5BANK1_FBIT1_Pos                                       (1U)
#define CAN_F5BANK1_FBIT1_Msk                                       (0x1UL << CAN_F5BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F5BANK1_FBIT1                                           CAN_F5BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F5BANK1_FBIT2_Pos                                       (2U)
#define CAN_F5BANK1_FBIT2_Msk                                       (0x1UL << CAN_F5BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F5BANK1_FBIT2                                           CAN_F5BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F5BANK1_FBIT3_Pos                                       (3U)
#define CAN_F5BANK1_FBIT3_Msk                                       (0x1UL << CAN_F5BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F5BANK1_FBIT3                                           CAN_F5BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F5BANK1_FBIT4_Pos                                       (4U)
#define CAN_F5BANK1_FBIT4_Msk                                       (0x1UL << CAN_F5BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F5BANK1_FBIT4                                           CAN_F5BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F5BANK1_FBIT5_Pos                                       (5U)
#define CAN_F5BANK1_FBIT5_Msk                                       (0x1UL << CAN_F5BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F5BANK1_FBIT5                                           CAN_F5BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F5BANK1_FBIT6_Pos                                       (6U)
#define CAN_F5BANK1_FBIT6_Msk                                       (0x1UL << CAN_F5BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F5BANK1_FBIT6                                           CAN_F5BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F5BANK1_FBIT7_Pos                                       (7U)
#define CAN_F5BANK1_FBIT7_Msk                                       (0x1UL << CAN_F5BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F5BANK1_FBIT7                                           CAN_F5BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F5BANK1_FBIT8_Pos                                       (8U)
#define CAN_F5BANK1_FBIT8_Msk                                       (0x1UL << CAN_F5BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F5BANK1_FBIT8                                           CAN_F5BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F5BANK1_FBIT9_Pos                                       (9U)
#define CAN_F5BANK1_FBIT9_Msk                                       (0x1UL << CAN_F5BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F5BANK1_FBIT9                                           CAN_F5BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F5BANK1_FBIT10_Pos                                      (10U)
#define CAN_F5BANK1_FBIT10_Msk                                      (0x1UL << CAN_F5BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F5BANK1_FBIT10                                          CAN_F5BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F5BANK1_FBIT11_Pos                                      (11U)
#define CAN_F5BANK1_FBIT11_Msk                                      (0x1UL << CAN_F5BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F5BANK1_FBIT11                                          CAN_F5BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F5BANK1_FBIT12_Pos                                      (12U)
#define CAN_F5BANK1_FBIT12_Msk                                      (0x1UL << CAN_F5BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F5BANK1_FBIT12                                          CAN_F5BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F5BANK1_FBIT13_Pos                                      (13U)
#define CAN_F5BANK1_FBIT13_Msk                                      (0x1UL << CAN_F5BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F5BANK1_FBIT13                                          CAN_F5BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F5BANK1_FBIT14_Pos                                      (14U)
#define CAN_F5BANK1_FBIT14_Msk                                      (0x1UL << CAN_F5BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F5BANK1_FBIT14                                          CAN_F5BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F5BANK1_FBIT15_Pos                                      (15U)
#define CAN_F5BANK1_FBIT15_Msk                                      (0x1UL << CAN_F5BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F5BANK1_FBIT15                                          CAN_F5BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F5BANK1_FBIT16_Pos                                      (16U)
#define CAN_F5BANK1_FBIT16_Msk                                      (0x1UL << CAN_F5BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F5BANK1_FBIT16                                          CAN_F5BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F5BANK1_FBIT17_Pos                                      (17U)
#define CAN_F5BANK1_FBIT17_Msk                                      (0x1UL << CAN_F5BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F5BANK1_FBIT17                                          CAN_F5BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F5BANK1_FBIT18_Pos                                      (18U)
#define CAN_F5BANK1_FBIT18_Msk                                      (0x1UL << CAN_F5BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F5BANK1_FBIT18                                          CAN_F5BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F5BANK1_FBIT19_Pos                                      (19U)
#define CAN_F5BANK1_FBIT19_Msk                                      (0x1UL << CAN_F5BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F5BANK1_FBIT19                                          CAN_F5BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F5BANK1_FBIT20_Pos                                      (20U)
#define CAN_F5BANK1_FBIT20_Msk                                      (0x1UL << CAN_F5BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F5BANK1_FBIT20                                          CAN_F5BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F5BANK1_FBIT21_Pos                                      (21U)
#define CAN_F5BANK1_FBIT21_Msk                                      (0x1UL << CAN_F5BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F5BANK1_FBIT21                                          CAN_F5BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F5BANK1_FBIT22_Pos                                      (22U)
#define CAN_F5BANK1_FBIT22_Msk                                      (0x1UL << CAN_F5BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F5BANK1_FBIT22                                          CAN_F5BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F5BANK1_FBIT23_Pos                                      (23U)
#define CAN_F5BANK1_FBIT23_Msk                                      (0x1UL << CAN_F5BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F5BANK1_FBIT23                                          CAN_F5BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F5BANK1_FBIT24_Pos                                      (24U)
#define CAN_F5BANK1_FBIT24_Msk                                      (0x1UL << CAN_F5BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F5BANK1_FBIT24                                          CAN_F5BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F5BANK1_FBIT25_Pos                                      (25U)
#define CAN_F5BANK1_FBIT25_Msk                                      (0x1UL << CAN_F5BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F5BANK1_FBIT25                                          CAN_F5BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F5BANK1_FBIT26_Pos                                      (26U)
#define CAN_F5BANK1_FBIT26_Msk                                      (0x1UL << CAN_F5BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F5BANK1_FBIT26                                          CAN_F5BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F5BANK1_FBIT27_Pos                                      (27U)
#define CAN_F5BANK1_FBIT27_Msk                                      (0x1UL << CAN_F5BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F5BANK1_FBIT27                                          CAN_F5BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F5BANK1_FBIT28_Pos                                      (28U)
#define CAN_F5BANK1_FBIT28_Msk                                      (0x1UL << CAN_F5BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F5BANK1_FBIT28                                          CAN_F5BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F5BANK1_FBIT29_Pos                                      (29U)
#define CAN_F5BANK1_FBIT29_Msk                                      (0x1UL << CAN_F5BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F5BANK1_FBIT29                                          CAN_F5BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F5BANK1_FBIT30_Pos                                      (30U)
#define CAN_F5BANK1_FBIT30_Msk                                      (0x1UL << CAN_F5BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F5BANK1_FBIT30                                          CAN_F5BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F5BANK1_FBIT31_Pos                                      (31U)
#define CAN_F5BANK1_FBIT31_Msk                                      (0x1UL << CAN_F5BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F5BANK1_FBIT31                                          CAN_F5BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F6BANK1 register  *******************/
#define CAN_F6BANK1_FBIT0_Pos                                       (0U)
#define CAN_F6BANK1_FBIT0_Msk                                       (0x1UL << CAN_F6BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F6BANK1_FBIT0                                           CAN_F6BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F6BANK1_FBIT1_Pos                                       (1U)
#define CAN_F6BANK1_FBIT1_Msk                                       (0x1UL << CAN_F6BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F6BANK1_FBIT1                                           CAN_F6BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F6BANK1_FBIT2_Pos                                       (2U)
#define CAN_F6BANK1_FBIT2_Msk                                       (0x1UL << CAN_F6BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F6BANK1_FBIT2                                           CAN_F6BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F6BANK1_FBIT3_Pos                                       (3U)
#define CAN_F6BANK1_FBIT3_Msk                                       (0x1UL << CAN_F6BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F6BANK1_FBIT3                                           CAN_F6BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F6BANK1_FBIT4_Pos                                       (4U)
#define CAN_F6BANK1_FBIT4_Msk                                       (0x1UL << CAN_F6BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F6BANK1_FBIT4                                           CAN_F6BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F6BANK1_FBIT5_Pos                                       (5U)
#define CAN_F6BANK1_FBIT5_Msk                                       (0x1UL << CAN_F6BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F6BANK1_FBIT5                                           CAN_F6BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F6BANK1_FBIT6_Pos                                       (6U)
#define CAN_F6BANK1_FBIT6_Msk                                       (0x1UL << CAN_F6BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F6BANK1_FBIT6                                           CAN_F6BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F6BANK1_FBIT7_Pos                                       (7U)
#define CAN_F6BANK1_FBIT7_Msk                                       (0x1UL << CAN_F6BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F6BANK1_FBIT7                                           CAN_F6BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F6BANK1_FBIT8_Pos                                       (8U)
#define CAN_F6BANK1_FBIT8_Msk                                       (0x1UL << CAN_F6BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F6BANK1_FBIT8                                           CAN_F6BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F6BANK1_FBIT9_Pos                                       (9U)
#define CAN_F6BANK1_FBIT9_Msk                                       (0x1UL << CAN_F6BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F6BANK1_FBIT9                                           CAN_F6BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F6BANK1_FBIT10_Pos                                      (10U)
#define CAN_F6BANK1_FBIT10_Msk                                      (0x1UL << CAN_F6BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F6BANK1_FBIT10                                          CAN_F6BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F6BANK1_FBIT11_Pos                                      (11U)
#define CAN_F6BANK1_FBIT11_Msk                                      (0x1UL << CAN_F6BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F6BANK1_FBIT11                                          CAN_F6BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F6BANK1_FBIT12_Pos                                      (12U)
#define CAN_F6BANK1_FBIT12_Msk                                      (0x1UL << CAN_F6BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F6BANK1_FBIT12                                          CAN_F6BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F6BANK1_FBIT13_Pos                                      (13U)
#define CAN_F6BANK1_FBIT13_Msk                                      (0x1UL << CAN_F6BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F6BANK1_FBIT13                                          CAN_F6BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F6BANK1_FBIT14_Pos                                      (14U)
#define CAN_F6BANK1_FBIT14_Msk                                      (0x1UL << CAN_F6BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F6BANK1_FBIT14                                          CAN_F6BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F6BANK1_FBIT15_Pos                                      (15U)
#define CAN_F6BANK1_FBIT15_Msk                                      (0x1UL << CAN_F6BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F6BANK1_FBIT15                                          CAN_F6BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F6BANK1_FBIT16_Pos                                      (16U)
#define CAN_F6BANK1_FBIT16_Msk                                      (0x1UL << CAN_F6BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F6BANK1_FBIT16                                          CAN_F6BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F6BANK1_FBIT17_Pos                                      (17U)
#define CAN_F6BANK1_FBIT17_Msk                                      (0x1UL << CAN_F6BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F6BANK1_FBIT17                                          CAN_F6BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F6BANK1_FBIT18_Pos                                      (18U)
#define CAN_F6BANK1_FBIT18_Msk                                      (0x1UL << CAN_F6BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F6BANK1_FBIT18                                          CAN_F6BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F6BANK1_FBIT19_Pos                                      (19U)
#define CAN_F6BANK1_FBIT19_Msk                                      (0x1UL << CAN_F6BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F6BANK1_FBIT19                                          CAN_F6BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F6BANK1_FBIT20_Pos                                      (20U)
#define CAN_F6BANK1_FBIT20_Msk                                      (0x1UL << CAN_F6BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F6BANK1_FBIT20                                          CAN_F6BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F6BANK1_FBIT21_Pos                                      (21U)
#define CAN_F6BANK1_FBIT21_Msk                                      (0x1UL << CAN_F6BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F6BANK1_FBIT21                                          CAN_F6BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F6BANK1_FBIT22_Pos                                      (22U)
#define CAN_F6BANK1_FBIT22_Msk                                      (0x1UL << CAN_F6BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F6BANK1_FBIT22                                          CAN_F6BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F6BANK1_FBIT23_Pos                                      (23U)
#define CAN_F6BANK1_FBIT23_Msk                                      (0x1UL << CAN_F6BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F6BANK1_FBIT23                                          CAN_F6BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F6BANK1_FBIT24_Pos                                      (24U)
#define CAN_F6BANK1_FBIT24_Msk                                      (0x1UL << CAN_F6BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F6BANK1_FBIT24                                          CAN_F6BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F6BANK1_FBIT25_Pos                                      (25U)
#define CAN_F6BANK1_FBIT25_Msk                                      (0x1UL << CAN_F6BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F6BANK1_FBIT25                                          CAN_F6BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F6BANK1_FBIT26_Pos                                      (26U)
#define CAN_F6BANK1_FBIT26_Msk                                      (0x1UL << CAN_F6BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F6BANK1_FBIT26                                          CAN_F6BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F6BANK1_FBIT27_Pos                                      (27U)
#define CAN_F6BANK1_FBIT27_Msk                                      (0x1UL << CAN_F6BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F6BANK1_FBIT27                                          CAN_F6BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F6BANK1_FBIT28_Pos                                      (28U)
#define CAN_F6BANK1_FBIT28_Msk                                      (0x1UL << CAN_F6BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F6BANK1_FBIT28                                          CAN_F6BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F6BANK1_FBIT29_Pos                                      (29U)
#define CAN_F6BANK1_FBIT29_Msk                                      (0x1UL << CAN_F6BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F6BANK1_FBIT29                                          CAN_F6BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F6BANK1_FBIT30_Pos                                      (30U)
#define CAN_F6BANK1_FBIT30_Msk                                      (0x1UL << CAN_F6BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F6BANK1_FBIT30                                          CAN_F6BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F6BANK1_FBIT31_Pos                                      (31U)
#define CAN_F6BANK1_FBIT31_Msk                                      (0x1UL << CAN_F6BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F6BANK1_FBIT31                                          CAN_F6BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F7BANK1 register  *******************/
#define CAN_F7BANK1_FBIT0_Pos                                       (0U)
#define CAN_F7BANK1_FBIT0_Msk                                       (0x1UL << CAN_F7BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F7BANK1_FBIT0                                           CAN_F7BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F7BANK1_FBIT1_Pos                                       (1U)
#define CAN_F7BANK1_FBIT1_Msk                                       (0x1UL << CAN_F7BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F7BANK1_FBIT1                                           CAN_F7BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F7BANK1_FBIT2_Pos                                       (2U)
#define CAN_F7BANK1_FBIT2_Msk                                       (0x1UL << CAN_F7BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F7BANK1_FBIT2                                           CAN_F7BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F7BANK1_FBIT3_Pos                                       (3U)
#define CAN_F7BANK1_FBIT3_Msk                                       (0x1UL << CAN_F7BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F7BANK1_FBIT3                                           CAN_F7BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F7BANK1_FBIT4_Pos                                       (4U)
#define CAN_F7BANK1_FBIT4_Msk                                       (0x1UL << CAN_F7BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F7BANK1_FBIT4                                           CAN_F7BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F7BANK1_FBIT5_Pos                                       (5U)
#define CAN_F7BANK1_FBIT5_Msk                                       (0x1UL << CAN_F7BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F7BANK1_FBIT5                                           CAN_F7BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F7BANK1_FBIT6_Pos                                       (6U)
#define CAN_F7BANK1_FBIT6_Msk                                       (0x1UL << CAN_F7BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F7BANK1_FBIT6                                           CAN_F7BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F7BANK1_FBIT7_Pos                                       (7U)
#define CAN_F7BANK1_FBIT7_Msk                                       (0x1UL << CAN_F7BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F7BANK1_FBIT7                                           CAN_F7BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F7BANK1_FBIT8_Pos                                       (8U)
#define CAN_F7BANK1_FBIT8_Msk                                       (0x1UL << CAN_F7BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F7BANK1_FBIT8                                           CAN_F7BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F7BANK1_FBIT9_Pos                                       (9U)
#define CAN_F7BANK1_FBIT9_Msk                                       (0x1UL << CAN_F7BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F7BANK1_FBIT9                                           CAN_F7BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F7BANK1_FBIT10_Pos                                      (10U)
#define CAN_F7BANK1_FBIT10_Msk                                      (0x1UL << CAN_F7BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F7BANK1_FBIT10                                          CAN_F7BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F7BANK1_FBIT11_Pos                                      (11U)
#define CAN_F7BANK1_FBIT11_Msk                                      (0x1UL << CAN_F7BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F7BANK1_FBIT11                                          CAN_F7BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F7BANK1_FBIT12_Pos                                      (12U)
#define CAN_F7BANK1_FBIT12_Msk                                      (0x1UL << CAN_F7BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F7BANK1_FBIT12                                          CAN_F7BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F7BANK1_FBIT13_Pos                                      (13U)
#define CAN_F7BANK1_FBIT13_Msk                                      (0x1UL << CAN_F7BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F7BANK1_FBIT13                                          CAN_F7BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F7BANK1_FBIT14_Pos                                      (14U)
#define CAN_F7BANK1_FBIT14_Msk                                      (0x1UL << CAN_F7BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F7BANK1_FBIT14                                          CAN_F7BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F7BANK1_FBIT15_Pos                                      (15U)
#define CAN_F7BANK1_FBIT15_Msk                                      (0x1UL << CAN_F7BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F7BANK1_FBIT15                                          CAN_F7BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F7BANK1_FBIT16_Pos                                      (16U)
#define CAN_F7BANK1_FBIT16_Msk                                      (0x1UL << CAN_F7BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F7BANK1_FBIT16                                          CAN_F7BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F7BANK1_FBIT17_Pos                                      (17U)
#define CAN_F7BANK1_FBIT17_Msk                                      (0x1UL << CAN_F7BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F7BANK1_FBIT17                                          CAN_F7BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F7BANK1_FBIT18_Pos                                      (18U)
#define CAN_F7BANK1_FBIT18_Msk                                      (0x1UL << CAN_F7BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F7BANK1_FBIT18                                          CAN_F7BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F7BANK1_FBIT19_Pos                                      (19U)
#define CAN_F7BANK1_FBIT19_Msk                                      (0x1UL << CAN_F7BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F7BANK1_FBIT19                                          CAN_F7BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F7BANK1_FBIT20_Pos                                      (20U)
#define CAN_F7BANK1_FBIT20_Msk                                      (0x1UL << CAN_F7BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F7BANK1_FBIT20                                          CAN_F7BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F7BANK1_FBIT21_Pos                                      (21U)
#define CAN_F7BANK1_FBIT21_Msk                                      (0x1UL << CAN_F7BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F7BANK1_FBIT21                                          CAN_F7BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F7BANK1_FBIT22_Pos                                      (22U)
#define CAN_F7BANK1_FBIT22_Msk                                      (0x1UL << CAN_F7BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F7BANK1_FBIT22                                          CAN_F7BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F7BANK1_FBIT23_Pos                                      (23U)
#define CAN_F7BANK1_FBIT23_Msk                                      (0x1UL << CAN_F7BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F7BANK1_FBIT23                                          CAN_F7BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F7BANK1_FBIT24_Pos                                      (24U)
#define CAN_F7BANK1_FBIT24_Msk                                      (0x1UL << CAN_F7BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F7BANK1_FBIT24                                          CAN_F7BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F7BANK1_FBIT25_Pos                                      (25U)
#define CAN_F7BANK1_FBIT25_Msk                                      (0x1UL << CAN_F7BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F7BANK1_FBIT25                                          CAN_F7BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F7BANK1_FBIT26_Pos                                      (26U)
#define CAN_F7BANK1_FBIT26_Msk                                      (0x1UL << CAN_F7BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F7BANK1_FBIT26                                          CAN_F7BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F7BANK1_FBIT27_Pos                                      (27U)
#define CAN_F7BANK1_FBIT27_Msk                                      (0x1UL << CAN_F7BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F7BANK1_FBIT27                                          CAN_F7BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F7BANK1_FBIT28_Pos                                      (28U)
#define CAN_F7BANK1_FBIT28_Msk                                      (0x1UL << CAN_F7BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F7BANK1_FBIT28                                          CAN_F7BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F7BANK1_FBIT29_Pos                                      (29U)
#define CAN_F7BANK1_FBIT29_Msk                                      (0x1UL << CAN_F7BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F7BANK1_FBIT29                                          CAN_F7BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F7BANK1_FBIT30_Pos                                      (30U)
#define CAN_F7BANK1_FBIT30_Msk                                      (0x1UL << CAN_F7BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F7BANK1_FBIT30                                          CAN_F7BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F7BANK1_FBIT31_Pos                                      (31U)
#define CAN_F7BANK1_FBIT31_Msk                                      (0x1UL << CAN_F7BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F7BANK1_FBIT31                                          CAN_F7BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F8BANK1 register  *******************/
#define CAN_F8BANK1_FBIT0_Pos                                       (0U)
#define CAN_F8BANK1_FBIT0_Msk                                       (0x1UL << CAN_F8BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F8BANK1_FBIT0                                           CAN_F8BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F8BANK1_FBIT1_Pos                                       (1U)
#define CAN_F8BANK1_FBIT1_Msk                                       (0x1UL << CAN_F8BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F8BANK1_FBIT1                                           CAN_F8BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F8BANK1_FBIT2_Pos                                       (2U)
#define CAN_F8BANK1_FBIT2_Msk                                       (0x1UL << CAN_F8BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F8BANK1_FBIT2                                           CAN_F8BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F8BANK1_FBIT3_Pos                                       (3U)
#define CAN_F8BANK1_FBIT3_Msk                                       (0x1UL << CAN_F8BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F8BANK1_FBIT3                                           CAN_F8BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F8BANK1_FBIT4_Pos                                       (4U)
#define CAN_F8BANK1_FBIT4_Msk                                       (0x1UL << CAN_F8BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F8BANK1_FBIT4                                           CAN_F8BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F8BANK1_FBIT5_Pos                                       (5U)
#define CAN_F8BANK1_FBIT5_Msk                                       (0x1UL << CAN_F8BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F8BANK1_FBIT5                                           CAN_F8BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F8BANK1_FBIT6_Pos                                       (6U)
#define CAN_F8BANK1_FBIT6_Msk                                       (0x1UL << CAN_F8BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F8BANK1_FBIT6                                           CAN_F8BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F8BANK1_FBIT7_Pos                                       (7U)
#define CAN_F8BANK1_FBIT7_Msk                                       (0x1UL << CAN_F8BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F8BANK1_FBIT7                                           CAN_F8BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F8BANK1_FBIT8_Pos                                       (8U)
#define CAN_F8BANK1_FBIT8_Msk                                       (0x1UL << CAN_F8BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F8BANK1_FBIT8                                           CAN_F8BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F8BANK1_FBIT9_Pos                                       (9U)
#define CAN_F8BANK1_FBIT9_Msk                                       (0x1UL << CAN_F8BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F8BANK1_FBIT9                                           CAN_F8BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F8BANK1_FBIT10_Pos                                      (10U)
#define CAN_F8BANK1_FBIT10_Msk                                      (0x1UL << CAN_F8BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F8BANK1_FBIT10                                          CAN_F8BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F8BANK1_FBIT11_Pos                                      (11U)
#define CAN_F8BANK1_FBIT11_Msk                                      (0x1UL << CAN_F8BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F8BANK1_FBIT11                                          CAN_F8BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F8BANK1_FBIT12_Pos                                      (12U)
#define CAN_F8BANK1_FBIT12_Msk                                      (0x1UL << CAN_F8BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F8BANK1_FBIT12                                          CAN_F8BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F8BANK1_FBIT13_Pos                                      (13U)
#define CAN_F8BANK1_FBIT13_Msk                                      (0x1UL << CAN_F8BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F8BANK1_FBIT13                                          CAN_F8BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F8BANK1_FBIT14_Pos                                      (14U)
#define CAN_F8BANK1_FBIT14_Msk                                      (0x1UL << CAN_F8BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F8BANK1_FBIT14                                          CAN_F8BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F8BANK1_FBIT15_Pos                                      (15U)
#define CAN_F8BANK1_FBIT15_Msk                                      (0x1UL << CAN_F8BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F8BANK1_FBIT15                                          CAN_F8BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F8BANK1_FBIT16_Pos                                      (16U)
#define CAN_F8BANK1_FBIT16_Msk                                      (0x1UL << CAN_F8BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F8BANK1_FBIT16                                          CAN_F8BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F8BANK1_FBIT17_Pos                                      (17U)
#define CAN_F8BANK1_FBIT17_Msk                                      (0x1UL << CAN_F8BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F8BANK1_FBIT17                                          CAN_F8BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F8BANK1_FBIT18_Pos                                      (18U)
#define CAN_F8BANK1_FBIT18_Msk                                      (0x1UL << CAN_F8BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F8BANK1_FBIT18                                          CAN_F8BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F8BANK1_FBIT19_Pos                                      (19U)
#define CAN_F8BANK1_FBIT19_Msk                                      (0x1UL << CAN_F8BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F8BANK1_FBIT19                                          CAN_F8BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F8BANK1_FBIT20_Pos                                      (20U)
#define CAN_F8BANK1_FBIT20_Msk                                      (0x1UL << CAN_F8BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F8BANK1_FBIT20                                          CAN_F8BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F8BANK1_FBIT21_Pos                                      (21U)
#define CAN_F8BANK1_FBIT21_Msk                                      (0x1UL << CAN_F8BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F8BANK1_FBIT21                                          CAN_F8BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F8BANK1_FBIT22_Pos                                      (22U)
#define CAN_F8BANK1_FBIT22_Msk                                      (0x1UL << CAN_F8BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F8BANK1_FBIT22                                          CAN_F8BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F8BANK1_FBIT23_Pos                                      (23U)
#define CAN_F8BANK1_FBIT23_Msk                                      (0x1UL << CAN_F8BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F8BANK1_FBIT23                                          CAN_F8BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F8BANK1_FBIT24_Pos                                      (24U)
#define CAN_F8BANK1_FBIT24_Msk                                      (0x1UL << CAN_F8BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F8BANK1_FBIT24                                          CAN_F8BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F8BANK1_FBIT25_Pos                                      (25U)
#define CAN_F8BANK1_FBIT25_Msk                                      (0x1UL << CAN_F8BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F8BANK1_FBIT25                                          CAN_F8BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F8BANK1_FBIT26_Pos                                      (26U)
#define CAN_F8BANK1_FBIT26_Msk                                      (0x1UL << CAN_F8BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F8BANK1_FBIT26                                          CAN_F8BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F8BANK1_FBIT27_Pos                                      (27U)
#define CAN_F8BANK1_FBIT27_Msk                                      (0x1UL << CAN_F8BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F8BANK1_FBIT27                                          CAN_F8BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F8BANK1_FBIT28_Pos                                      (28U)
#define CAN_F8BANK1_FBIT28_Msk                                      (0x1UL << CAN_F8BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F8BANK1_FBIT28                                          CAN_F8BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F8BANK1_FBIT29_Pos                                      (29U)
#define CAN_F8BANK1_FBIT29_Msk                                      (0x1UL << CAN_F8BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F8BANK1_FBIT29                                          CAN_F8BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F8BANK1_FBIT30_Pos                                      (30U)
#define CAN_F8BANK1_FBIT30_Msk                                      (0x1UL << CAN_F8BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F8BANK1_FBIT30                                          CAN_F8BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F8BANK1_FBIT31_Pos                                      (31U)
#define CAN_F8BANK1_FBIT31_Msk                                      (0x1UL << CAN_F8BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F8BANK1_FBIT31                                          CAN_F8BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F9BANK1 register  *******************/
#define CAN_F9BANK1_FBIT0_Pos                                       (0U)
#define CAN_F9BANK1_FBIT0_Msk                                       (0x1UL << CAN_F9BANK1_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F9BANK1_FBIT0                                           CAN_F9BANK1_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F9BANK1_FBIT1_Pos                                       (1U)
#define CAN_F9BANK1_FBIT1_Msk                                       (0x1UL << CAN_F9BANK1_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F9BANK1_FBIT1                                           CAN_F9BANK1_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F9BANK1_FBIT2_Pos                                       (2U)
#define CAN_F9BANK1_FBIT2_Msk                                       (0x1UL << CAN_F9BANK1_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F9BANK1_FBIT2                                           CAN_F9BANK1_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F9BANK1_FBIT3_Pos                                       (3U)
#define CAN_F9BANK1_FBIT3_Msk                                       (0x1UL << CAN_F9BANK1_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F9BANK1_FBIT3                                           CAN_F9BANK1_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F9BANK1_FBIT4_Pos                                       (4U)
#define CAN_F9BANK1_FBIT4_Msk                                       (0x1UL << CAN_F9BANK1_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F9BANK1_FBIT4                                           CAN_F9BANK1_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F9BANK1_FBIT5_Pos                                       (5U)
#define CAN_F9BANK1_FBIT5_Msk                                       (0x1UL << CAN_F9BANK1_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F9BANK1_FBIT5                                           CAN_F9BANK1_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F9BANK1_FBIT6_Pos                                       (6U)
#define CAN_F9BANK1_FBIT6_Msk                                       (0x1UL << CAN_F9BANK1_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F9BANK1_FBIT6                                           CAN_F9BANK1_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F9BANK1_FBIT7_Pos                                       (7U)
#define CAN_F9BANK1_FBIT7_Msk                                       (0x1UL << CAN_F9BANK1_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F9BANK1_FBIT7                                           CAN_F9BANK1_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F9BANK1_FBIT8_Pos                                       (8U)
#define CAN_F9BANK1_FBIT8_Msk                                       (0x1UL << CAN_F9BANK1_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F9BANK1_FBIT8                                           CAN_F9BANK1_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F9BANK1_FBIT9_Pos                                       (9U)
#define CAN_F9BANK1_FBIT9_Msk                                       (0x1UL << CAN_F9BANK1_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F9BANK1_FBIT9                                           CAN_F9BANK1_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F9BANK1_FBIT10_Pos                                      (10U)
#define CAN_F9BANK1_FBIT10_Msk                                      (0x1UL << CAN_F9BANK1_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F9BANK1_FBIT10                                          CAN_F9BANK1_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F9BANK1_FBIT11_Pos                                      (11U)
#define CAN_F9BANK1_FBIT11_Msk                                      (0x1UL << CAN_F9BANK1_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F9BANK1_FBIT11                                          CAN_F9BANK1_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F9BANK1_FBIT12_Pos                                      (12U)
#define CAN_F9BANK1_FBIT12_Msk                                      (0x1UL << CAN_F9BANK1_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F9BANK1_FBIT12                                          CAN_F9BANK1_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F9BANK1_FBIT13_Pos                                      (13U)
#define CAN_F9BANK1_FBIT13_Msk                                      (0x1UL << CAN_F9BANK1_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F9BANK1_FBIT13                                          CAN_F9BANK1_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F9BANK1_FBIT14_Pos                                      (14U)
#define CAN_F9BANK1_FBIT14_Msk                                      (0x1UL << CAN_F9BANK1_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F9BANK1_FBIT14                                          CAN_F9BANK1_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F9BANK1_FBIT15_Pos                                      (15U)
#define CAN_F9BANK1_FBIT15_Msk                                      (0x1UL << CAN_F9BANK1_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F9BANK1_FBIT15                                          CAN_F9BANK1_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F9BANK1_FBIT16_Pos                                      (16U)
#define CAN_F9BANK1_FBIT16_Msk                                      (0x1UL << CAN_F9BANK1_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F9BANK1_FBIT16                                          CAN_F9BANK1_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F9BANK1_FBIT17_Pos                                      (17U)
#define CAN_F9BANK1_FBIT17_Msk                                      (0x1UL << CAN_F9BANK1_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F9BANK1_FBIT17                                          CAN_F9BANK1_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F9BANK1_FBIT18_Pos                                      (18U)
#define CAN_F9BANK1_FBIT18_Msk                                      (0x1UL << CAN_F9BANK1_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F9BANK1_FBIT18                                          CAN_F9BANK1_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F9BANK1_FBIT19_Pos                                      (19U)
#define CAN_F9BANK1_FBIT19_Msk                                      (0x1UL << CAN_F9BANK1_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F9BANK1_FBIT19                                          CAN_F9BANK1_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F9BANK1_FBIT20_Pos                                      (20U)
#define CAN_F9BANK1_FBIT20_Msk                                      (0x1UL << CAN_F9BANK1_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F9BANK1_FBIT20                                          CAN_F9BANK1_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F9BANK1_FBIT21_Pos                                      (21U)
#define CAN_F9BANK1_FBIT21_Msk                                      (0x1UL << CAN_F9BANK1_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F9BANK1_FBIT21                                          CAN_F9BANK1_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F9BANK1_FBIT22_Pos                                      (22U)
#define CAN_F9BANK1_FBIT22_Msk                                      (0x1UL << CAN_F9BANK1_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F9BANK1_FBIT22                                          CAN_F9BANK1_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F9BANK1_FBIT23_Pos                                      (23U)
#define CAN_F9BANK1_FBIT23_Msk                                      (0x1UL << CAN_F9BANK1_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F9BANK1_FBIT23                                          CAN_F9BANK1_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F9BANK1_FBIT24_Pos                                      (24U)
#define CAN_F9BANK1_FBIT24_Msk                                      (0x1UL << CAN_F9BANK1_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F9BANK1_FBIT24                                          CAN_F9BANK1_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F9BANK1_FBIT25_Pos                                      (25U)
#define CAN_F9BANK1_FBIT25_Msk                                      (0x1UL << CAN_F9BANK1_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F9BANK1_FBIT25                                          CAN_F9BANK1_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F9BANK1_FBIT26_Pos                                      (26U)
#define CAN_F9BANK1_FBIT26_Msk                                      (0x1UL << CAN_F9BANK1_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F9BANK1_FBIT26                                          CAN_F9BANK1_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F9BANK1_FBIT27_Pos                                      (27U)
#define CAN_F9BANK1_FBIT27_Msk                                      (0x1UL << CAN_F9BANK1_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F9BANK1_FBIT27                                          CAN_F9BANK1_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F9BANK1_FBIT28_Pos                                      (28U)
#define CAN_F9BANK1_FBIT28_Msk                                      (0x1UL << CAN_F9BANK1_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F9BANK1_FBIT28                                          CAN_F9BANK1_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F9BANK1_FBIT29_Pos                                      (29U)
#define CAN_F9BANK1_FBIT29_Msk                                      (0x1UL << CAN_F9BANK1_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F9BANK1_FBIT29                                          CAN_F9BANK1_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F9BANK1_FBIT30_Pos                                      (30U)
#define CAN_F9BANK1_FBIT30_Msk                                      (0x1UL << CAN_F9BANK1_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F9BANK1_FBIT30                                          CAN_F9BANK1_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F9BANK1_FBIT31_Pos                                      (31U)
#define CAN_F9BANK1_FBIT31_Msk                                      (0x1UL << CAN_F9BANK1_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F9BANK1_FBIT31                                          CAN_F9BANK1_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F10BANK1 register  ******************/
#define CAN_F10BANK1_FBIT0_Pos                                      (0U)
#define CAN_F10BANK1_FBIT0_Msk                                      (0x1UL << CAN_F10BANK1_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F10BANK1_FBIT0                                          CAN_F10BANK1_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F10BANK1_FBIT1_Pos                                      (1U)
#define CAN_F10BANK1_FBIT1_Msk                                      (0x1UL << CAN_F10BANK1_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F10BANK1_FBIT1                                          CAN_F10BANK1_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F10BANK1_FBIT2_Pos                                      (2U)
#define CAN_F10BANK1_FBIT2_Msk                                      (0x1UL << CAN_F10BANK1_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F10BANK1_FBIT2                                          CAN_F10BANK1_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F10BANK1_FBIT3_Pos                                      (3U)
#define CAN_F10BANK1_FBIT3_Msk                                      (0x1UL << CAN_F10BANK1_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F10BANK1_FBIT3                                          CAN_F10BANK1_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F10BANK1_FBIT4_Pos                                      (4U)
#define CAN_F10BANK1_FBIT4_Msk                                      (0x1UL << CAN_F10BANK1_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F10BANK1_FBIT4                                          CAN_F10BANK1_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F10BANK1_FBIT5_Pos                                      (5U)
#define CAN_F10BANK1_FBIT5_Msk                                      (0x1UL << CAN_F10BANK1_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F10BANK1_FBIT5                                          CAN_F10BANK1_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F10BANK1_FBIT6_Pos                                      (6U)
#define CAN_F10BANK1_FBIT6_Msk                                      (0x1UL << CAN_F10BANK1_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F10BANK1_FBIT6                                          CAN_F10BANK1_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F10BANK1_FBIT7_Pos                                      (7U)
#define CAN_F10BANK1_FBIT7_Msk                                      (0x1UL << CAN_F10BANK1_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F10BANK1_FBIT7                                          CAN_F10BANK1_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F10BANK1_FBIT8_Pos                                      (8U)
#define CAN_F10BANK1_FBIT8_Msk                                      (0x1UL << CAN_F10BANK1_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F10BANK1_FBIT8                                          CAN_F10BANK1_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F10BANK1_FBIT9_Pos                                      (9U)
#define CAN_F10BANK1_FBIT9_Msk                                      (0x1UL << CAN_F10BANK1_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F10BANK1_FBIT9                                          CAN_F10BANK1_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F10BANK1_FBIT10_Pos                                     (10U)
#define CAN_F10BANK1_FBIT10_Msk                                     (0x1UL << CAN_F10BANK1_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F10BANK1_FBIT10                                         CAN_F10BANK1_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F10BANK1_FBIT11_Pos                                     (11U)
#define CAN_F10BANK1_FBIT11_Msk                                     (0x1UL << CAN_F10BANK1_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F10BANK1_FBIT11                                         CAN_F10BANK1_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F10BANK1_FBIT12_Pos                                     (12U)
#define CAN_F10BANK1_FBIT12_Msk                                     (0x1UL << CAN_F10BANK1_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F10BANK1_FBIT12                                         CAN_F10BANK1_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F10BANK1_FBIT13_Pos                                     (13U)
#define CAN_F10BANK1_FBIT13_Msk                                     (0x1UL << CAN_F10BANK1_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F10BANK1_FBIT13                                         CAN_F10BANK1_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F10BANK1_FBIT14_Pos                                     (14U)
#define CAN_F10BANK1_FBIT14_Msk                                     (0x1UL << CAN_F10BANK1_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F10BANK1_FBIT14                                         CAN_F10BANK1_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F10BANK1_FBIT15_Pos                                     (15U)
#define CAN_F10BANK1_FBIT15_Msk                                     (0x1UL << CAN_F10BANK1_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F10BANK1_FBIT15                                         CAN_F10BANK1_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F10BANK1_FBIT16_Pos                                     (16U)
#define CAN_F10BANK1_FBIT16_Msk                                     (0x1UL << CAN_F10BANK1_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F10BANK1_FBIT16                                         CAN_F10BANK1_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F10BANK1_FBIT17_Pos                                     (17U)
#define CAN_F10BANK1_FBIT17_Msk                                     (0x1UL << CAN_F10BANK1_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F10BANK1_FBIT17                                         CAN_F10BANK1_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F10BANK1_FBIT18_Pos                                     (18U)
#define CAN_F10BANK1_FBIT18_Msk                                     (0x1UL << CAN_F10BANK1_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F10BANK1_FBIT18                                         CAN_F10BANK1_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F10BANK1_FBIT19_Pos                                     (19U)
#define CAN_F10BANK1_FBIT19_Msk                                     (0x1UL << CAN_F10BANK1_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F10BANK1_FBIT19                                         CAN_F10BANK1_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F10BANK1_FBIT20_Pos                                     (20U)
#define CAN_F10BANK1_FBIT20_Msk                                     (0x1UL << CAN_F10BANK1_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F10BANK1_FBIT20                                         CAN_F10BANK1_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F10BANK1_FBIT21_Pos                                     (21U)
#define CAN_F10BANK1_FBIT21_Msk                                     (0x1UL << CAN_F10BANK1_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F10BANK1_FBIT21                                         CAN_F10BANK1_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F10BANK1_FBIT22_Pos                                     (22U)
#define CAN_F10BANK1_FBIT22_Msk                                     (0x1UL << CAN_F10BANK1_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F10BANK1_FBIT22                                         CAN_F10BANK1_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F10BANK1_FBIT23_Pos                                     (23U)
#define CAN_F10BANK1_FBIT23_Msk                                     (0x1UL << CAN_F10BANK1_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F10BANK1_FBIT23                                         CAN_F10BANK1_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F10BANK1_FBIT24_Pos                                     (24U)
#define CAN_F10BANK1_FBIT24_Msk                                     (0x1UL << CAN_F10BANK1_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F10BANK1_FBIT24                                         CAN_F10BANK1_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F10BANK1_FBIT25_Pos                                     (25U)
#define CAN_F10BANK1_FBIT25_Msk                                     (0x1UL << CAN_F10BANK1_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F10BANK1_FBIT25                                         CAN_F10BANK1_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F10BANK1_FBIT26_Pos                                     (26U)
#define CAN_F10BANK1_FBIT26_Msk                                     (0x1UL << CAN_F10BANK1_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F10BANK1_FBIT26                                         CAN_F10BANK1_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F10BANK1_FBIT27_Pos                                     (27U)
#define CAN_F10BANK1_FBIT27_Msk                                     (0x1UL << CAN_F10BANK1_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F10BANK1_FBIT27                                         CAN_F10BANK1_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F10BANK1_FBIT28_Pos                                     (28U)
#define CAN_F10BANK1_FBIT28_Msk                                     (0x1UL << CAN_F10BANK1_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F10BANK1_FBIT28                                         CAN_F10BANK1_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F10BANK1_FBIT29_Pos                                     (29U)
#define CAN_F10BANK1_FBIT29_Msk                                     (0x1UL << CAN_F10BANK1_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F10BANK1_FBIT29                                         CAN_F10BANK1_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F10BANK1_FBIT30_Pos                                     (30U)
#define CAN_F10BANK1_FBIT30_Msk                                     (0x1UL << CAN_F10BANK1_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F10BANK1_FBIT30                                         CAN_F10BANK1_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F10BANK1_FBIT31_Pos                                     (31U)
#define CAN_F10BANK1_FBIT31_Msk                                     (0x1UL << CAN_F10BANK1_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F10BANK1_FBIT31                                         CAN_F10BANK1_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F11BANK1 register  ******************/
#define CAN_F11BANK1_FBIT0_Pos                                      (0U)
#define CAN_F11BANK1_FBIT0_Msk                                      (0x1UL << CAN_F11BANK1_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F11BANK1_FBIT0                                          CAN_F11BANK1_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F11BANK1_FBIT1_Pos                                      (1U)
#define CAN_F11BANK1_FBIT1_Msk                                      (0x1UL << CAN_F11BANK1_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F11BANK1_FBIT1                                          CAN_F11BANK1_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F11BANK1_FBIT2_Pos                                      (2U)
#define CAN_F11BANK1_FBIT2_Msk                                      (0x1UL << CAN_F11BANK1_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F11BANK1_FBIT2                                          CAN_F11BANK1_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F11BANK1_FBIT3_Pos                                      (3U)
#define CAN_F11BANK1_FBIT3_Msk                                      (0x1UL << CAN_F11BANK1_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F11BANK1_FBIT3                                          CAN_F11BANK1_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F11BANK1_FBIT4_Pos                                      (4U)
#define CAN_F11BANK1_FBIT4_Msk                                      (0x1UL << CAN_F11BANK1_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F11BANK1_FBIT4                                          CAN_F11BANK1_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F11BANK1_FBIT5_Pos                                      (5U)
#define CAN_F11BANK1_FBIT5_Msk                                      (0x1UL << CAN_F11BANK1_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F11BANK1_FBIT5                                          CAN_F11BANK1_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F11BANK1_FBIT6_Pos                                      (6U)
#define CAN_F11BANK1_FBIT6_Msk                                      (0x1UL << CAN_F11BANK1_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F11BANK1_FBIT6                                          CAN_F11BANK1_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F11BANK1_FBIT7_Pos                                      (7U)
#define CAN_F11BANK1_FBIT7_Msk                                      (0x1UL << CAN_F11BANK1_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F11BANK1_FBIT7                                          CAN_F11BANK1_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F11BANK1_FBIT8_Pos                                      (8U)
#define CAN_F11BANK1_FBIT8_Msk                                      (0x1UL << CAN_F11BANK1_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F11BANK1_FBIT8                                          CAN_F11BANK1_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F11BANK1_FBIT9_Pos                                      (9U)
#define CAN_F11BANK1_FBIT9_Msk                                      (0x1UL << CAN_F11BANK1_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F11BANK1_FBIT9                                          CAN_F11BANK1_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F11BANK1_FBIT10_Pos                                     (10U)
#define CAN_F11BANK1_FBIT10_Msk                                     (0x1UL << CAN_F11BANK1_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F11BANK1_FBIT10                                         CAN_F11BANK1_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F11BANK1_FBIT11_Pos                                     (11U)
#define CAN_F11BANK1_FBIT11_Msk                                     (0x1UL << CAN_F11BANK1_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F11BANK1_FBIT11                                         CAN_F11BANK1_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F11BANK1_FBIT12_Pos                                     (12U)
#define CAN_F11BANK1_FBIT12_Msk                                     (0x1UL << CAN_F11BANK1_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F11BANK1_FBIT12                                         CAN_F11BANK1_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F11BANK1_FBIT13_Pos                                     (13U)
#define CAN_F11BANK1_FBIT13_Msk                                     (0x1UL << CAN_F11BANK1_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F11BANK1_FBIT13                                         CAN_F11BANK1_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F11BANK1_FBIT14_Pos                                     (14U)
#define CAN_F11BANK1_FBIT14_Msk                                     (0x1UL << CAN_F11BANK1_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F11BANK1_FBIT14                                         CAN_F11BANK1_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F11BANK1_FBIT15_Pos                                     (15U)
#define CAN_F11BANK1_FBIT15_Msk                                     (0x1UL << CAN_F11BANK1_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F11BANK1_FBIT15                                         CAN_F11BANK1_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F11BANK1_FBIT16_Pos                                     (16U)
#define CAN_F11BANK1_FBIT16_Msk                                     (0x1UL << CAN_F11BANK1_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F11BANK1_FBIT16                                         CAN_F11BANK1_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F11BANK1_FBIT17_Pos                                     (17U)
#define CAN_F11BANK1_FBIT17_Msk                                     (0x1UL << CAN_F11BANK1_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F11BANK1_FBIT17                                         CAN_F11BANK1_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F11BANK1_FBIT18_Pos                                     (18U)
#define CAN_F11BANK1_FBIT18_Msk                                     (0x1UL << CAN_F11BANK1_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F11BANK1_FBIT18                                         CAN_F11BANK1_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F11BANK1_FBIT19_Pos                                     (19U)
#define CAN_F11BANK1_FBIT19_Msk                                     (0x1UL << CAN_F11BANK1_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F11BANK1_FBIT19                                         CAN_F11BANK1_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F11BANK1_FBIT20_Pos                                     (20U)
#define CAN_F11BANK1_FBIT20_Msk                                     (0x1UL << CAN_F11BANK1_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F11BANK1_FBIT20                                         CAN_F11BANK1_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F11BANK1_FBIT21_Pos                                     (21U)
#define CAN_F11BANK1_FBIT21_Msk                                     (0x1UL << CAN_F11BANK1_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F11BANK1_FBIT21                                         CAN_F11BANK1_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F11BANK1_FBIT22_Pos                                     (22U)
#define CAN_F11BANK1_FBIT22_Msk                                     (0x1UL << CAN_F11BANK1_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F11BANK1_FBIT22                                         CAN_F11BANK1_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F11BANK1_FBIT23_Pos                                     (23U)
#define CAN_F11BANK1_FBIT23_Msk                                     (0x1UL << CAN_F11BANK1_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F11BANK1_FBIT23                                         CAN_F11BANK1_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F11BANK1_FBIT24_Pos                                     (24U)
#define CAN_F11BANK1_FBIT24_Msk                                     (0x1UL << CAN_F11BANK1_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F11BANK1_FBIT24                                         CAN_F11BANK1_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F11BANK1_FBIT25_Pos                                     (25U)
#define CAN_F11BANK1_FBIT25_Msk                                     (0x1UL << CAN_F11BANK1_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F11BANK1_FBIT25                                         CAN_F11BANK1_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F11BANK1_FBIT26_Pos                                     (26U)
#define CAN_F11BANK1_FBIT26_Msk                                     (0x1UL << CAN_F11BANK1_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F11BANK1_FBIT26                                         CAN_F11BANK1_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F11BANK1_FBIT27_Pos                                     (27U)
#define CAN_F11BANK1_FBIT27_Msk                                     (0x1UL << CAN_F11BANK1_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F11BANK1_FBIT27                                         CAN_F11BANK1_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F11BANK1_FBIT28_Pos                                     (28U)
#define CAN_F11BANK1_FBIT28_Msk                                     (0x1UL << CAN_F11BANK1_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F11BANK1_FBIT28                                         CAN_F11BANK1_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F11BANK1_FBIT29_Pos                                     (29U)
#define CAN_F11BANK1_FBIT29_Msk                                     (0x1UL << CAN_F11BANK1_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F11BANK1_FBIT29                                         CAN_F11BANK1_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F11BANK1_FBIT30_Pos                                     (30U)
#define CAN_F11BANK1_FBIT30_Msk                                     (0x1UL << CAN_F11BANK1_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F11BANK1_FBIT30                                         CAN_F11BANK1_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F11BANK1_FBIT31_Pos                                     (31U)
#define CAN_F11BANK1_FBIT31_Msk                                     (0x1UL << CAN_F11BANK1_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F11BANK1_FBIT31                                         CAN_F11BANK1_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F12BANK1 register  ******************/
#define CAN_F12BANK1_FBIT0_Pos                                      (0U)
#define CAN_F12BANK1_FBIT0_Msk                                      (0x1UL << CAN_F12BANK1_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F12BANK1_FBIT0                                          CAN_F12BANK1_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F12BANK1_FBIT1_Pos                                      (1U)
#define CAN_F12BANK1_FBIT1_Msk                                      (0x1UL << CAN_F12BANK1_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F12BANK1_FBIT1                                          CAN_F12BANK1_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F12BANK1_FBIT2_Pos                                      (2U)
#define CAN_F12BANK1_FBIT2_Msk                                      (0x1UL << CAN_F12BANK1_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F12BANK1_FBIT2                                          CAN_F12BANK1_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F12BANK1_FBIT3_Pos                                      (3U)
#define CAN_F12BANK1_FBIT3_Msk                                      (0x1UL << CAN_F12BANK1_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F12BANK1_FBIT3                                          CAN_F12BANK1_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F12BANK1_FBIT4_Pos                                      (4U)
#define CAN_F12BANK1_FBIT4_Msk                                      (0x1UL << CAN_F12BANK1_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F12BANK1_FBIT4                                          CAN_F12BANK1_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F12BANK1_FBIT5_Pos                                      (5U)
#define CAN_F12BANK1_FBIT5_Msk                                      (0x1UL << CAN_F12BANK1_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F12BANK1_FBIT5                                          CAN_F12BANK1_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F12BANK1_FBIT6_Pos                                      (6U)
#define CAN_F12BANK1_FBIT6_Msk                                      (0x1UL << CAN_F12BANK1_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F12BANK1_FBIT6                                          CAN_F12BANK1_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F12BANK1_FBIT7_Pos                                      (7U)
#define CAN_F12BANK1_FBIT7_Msk                                      (0x1UL << CAN_F12BANK1_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F12BANK1_FBIT7                                          CAN_F12BANK1_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F12BANK1_FBIT8_Pos                                      (8U)
#define CAN_F12BANK1_FBIT8_Msk                                      (0x1UL << CAN_F12BANK1_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F12BANK1_FBIT8                                          CAN_F12BANK1_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F12BANK1_FBIT9_Pos                                      (9U)
#define CAN_F12BANK1_FBIT9_Msk                                      (0x1UL << CAN_F12BANK1_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F12BANK1_FBIT9                                          CAN_F12BANK1_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F12BANK1_FBIT10_Pos                                     (10U)
#define CAN_F12BANK1_FBIT10_Msk                                     (0x1UL << CAN_F12BANK1_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F12BANK1_FBIT10                                         CAN_F12BANK1_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F12BANK1_FBIT11_Pos                                     (11U)
#define CAN_F12BANK1_FBIT11_Msk                                     (0x1UL << CAN_F12BANK1_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F12BANK1_FBIT11                                         CAN_F12BANK1_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F12BANK1_FBIT12_Pos                                     (12U)
#define CAN_F12BANK1_FBIT12_Msk                                     (0x1UL << CAN_F12BANK1_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F12BANK1_FBIT12                                         CAN_F12BANK1_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F12BANK1_FBIT13_Pos                                     (13U)
#define CAN_F12BANK1_FBIT13_Msk                                     (0x1UL << CAN_F12BANK1_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F12BANK1_FBIT13                                         CAN_F12BANK1_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F12BANK1_FBIT14_Pos                                     (14U)
#define CAN_F12BANK1_FBIT14_Msk                                     (0x1UL << CAN_F12BANK1_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F12BANK1_FBIT14                                         CAN_F12BANK1_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F12BANK1_FBIT15_Pos                                     (15U)
#define CAN_F12BANK1_FBIT15_Msk                                     (0x1UL << CAN_F12BANK1_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F12BANK1_FBIT15                                         CAN_F12BANK1_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F12BANK1_FBIT16_Pos                                     (16U)
#define CAN_F12BANK1_FBIT16_Msk                                     (0x1UL << CAN_F12BANK1_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F12BANK1_FBIT16                                         CAN_F12BANK1_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F12BANK1_FBIT17_Pos                                     (17U)
#define CAN_F12BANK1_FBIT17_Msk                                     (0x1UL << CAN_F12BANK1_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F12BANK1_FBIT17                                         CAN_F12BANK1_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F12BANK1_FBIT18_Pos                                     (18U)
#define CAN_F12BANK1_FBIT18_Msk                                     (0x1UL << CAN_F12BANK1_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F12BANK1_FBIT18                                         CAN_F12BANK1_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F12BANK1_FBIT19_Pos                                     (19U)
#define CAN_F12BANK1_FBIT19_Msk                                     (0x1UL << CAN_F12BANK1_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F12BANK1_FBIT19                                         CAN_F12BANK1_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F12BANK1_FBIT20_Pos                                     (20U)
#define CAN_F12BANK1_FBIT20_Msk                                     (0x1UL << CAN_F12BANK1_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F12BANK1_FBIT20                                         CAN_F12BANK1_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F12BANK1_FBIT21_Pos                                     (21U)
#define CAN_F12BANK1_FBIT21_Msk                                     (0x1UL << CAN_F12BANK1_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F12BANK1_FBIT21                                         CAN_F12BANK1_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F12BANK1_FBIT22_Pos                                     (22U)
#define CAN_F12BANK1_FBIT22_Msk                                     (0x1UL << CAN_F12BANK1_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F12BANK1_FBIT22                                         CAN_F12BANK1_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F12BANK1_FBIT23_Pos                                     (23U)
#define CAN_F12BANK1_FBIT23_Msk                                     (0x1UL << CAN_F12BANK1_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F12BANK1_FBIT23                                         CAN_F12BANK1_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F12BANK1_FBIT24_Pos                                     (24U)
#define CAN_F12BANK1_FBIT24_Msk                                     (0x1UL << CAN_F12BANK1_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F12BANK1_FBIT24                                         CAN_F12BANK1_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F12BANK1_FBIT25_Pos                                     (25U)
#define CAN_F12BANK1_FBIT25_Msk                                     (0x1UL << CAN_F12BANK1_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F12BANK1_FBIT25                                         CAN_F12BANK1_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F12BANK1_FBIT26_Pos                                     (26U)
#define CAN_F12BANK1_FBIT26_Msk                                     (0x1UL << CAN_F12BANK1_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F12BANK1_FBIT26                                         CAN_F12BANK1_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F12BANK1_FBIT27_Pos                                     (27U)
#define CAN_F12BANK1_FBIT27_Msk                                     (0x1UL << CAN_F12BANK1_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F12BANK1_FBIT27                                         CAN_F12BANK1_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F12BANK1_FBIT28_Pos                                     (28U)
#define CAN_F12BANK1_FBIT28_Msk                                     (0x1UL << CAN_F12BANK1_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F12BANK1_FBIT28                                         CAN_F12BANK1_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F12BANK1_FBIT29_Pos                                     (29U)
#define CAN_F12BANK1_FBIT29_Msk                                     (0x1UL << CAN_F12BANK1_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F12BANK1_FBIT29                                         CAN_F12BANK1_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F12BANK1_FBIT30_Pos                                     (30U)
#define CAN_F12BANK1_FBIT30_Msk                                     (0x1UL << CAN_F12BANK1_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F12BANK1_FBIT30                                         CAN_F12BANK1_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F12BANK1_FBIT31_Pos                                     (31U)
#define CAN_F12BANK1_FBIT31_Msk                                     (0x1UL << CAN_F12BANK1_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F12BANK1_FBIT31                                         CAN_F12BANK1_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F13BANK1 register  ******************/
#define CAN_F13BANK1_FBIT0_Pos                                      (0U)
#define CAN_F13BANK1_FBIT0_Msk                                      (0x1UL << CAN_F13BANK1_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F13BANK1_FBIT0                                          CAN_F13BANK1_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F13BANK1_FBIT1_Pos                                      (1U)
#define CAN_F13BANK1_FBIT1_Msk                                      (0x1UL << CAN_F13BANK1_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F13BANK1_FBIT1                                          CAN_F13BANK1_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F13BANK1_FBIT2_Pos                                      (2U)
#define CAN_F13BANK1_FBIT2_Msk                                      (0x1UL << CAN_F13BANK1_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F13BANK1_FBIT2                                          CAN_F13BANK1_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F13BANK1_FBIT3_Pos                                      (3U)
#define CAN_F13BANK1_FBIT3_Msk                                      (0x1UL << CAN_F13BANK1_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F13BANK1_FBIT3                                          CAN_F13BANK1_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F13BANK1_FBIT4_Pos                                      (4U)
#define CAN_F13BANK1_FBIT4_Msk                                      (0x1UL << CAN_F13BANK1_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F13BANK1_FBIT4                                          CAN_F13BANK1_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F13BANK1_FBIT5_Pos                                      (5U)
#define CAN_F13BANK1_FBIT5_Msk                                      (0x1UL << CAN_F13BANK1_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F13BANK1_FBIT5                                          CAN_F13BANK1_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F13BANK1_FBIT6_Pos                                      (6U)
#define CAN_F13BANK1_FBIT6_Msk                                      (0x1UL << CAN_F13BANK1_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F13BANK1_FBIT6                                          CAN_F13BANK1_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F13BANK1_FBIT7_Pos                                      (7U)
#define CAN_F13BANK1_FBIT7_Msk                                      (0x1UL << CAN_F13BANK1_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F13BANK1_FBIT7                                          CAN_F13BANK1_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F13BANK1_FBIT8_Pos                                      (8U)
#define CAN_F13BANK1_FBIT8_Msk                                      (0x1UL << CAN_F13BANK1_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F13BANK1_FBIT8                                          CAN_F13BANK1_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F13BANK1_FBIT9_Pos                                      (9U)
#define CAN_F13BANK1_FBIT9_Msk                                      (0x1UL << CAN_F13BANK1_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F13BANK1_FBIT9                                          CAN_F13BANK1_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F13BANK1_FBIT10_Pos                                     (10U)
#define CAN_F13BANK1_FBIT10_Msk                                     (0x1UL << CAN_F13BANK1_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F13BANK1_FBIT10                                         CAN_F13BANK1_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F13BANK1_FBIT11_Pos                                     (11U)
#define CAN_F13BANK1_FBIT11_Msk                                     (0x1UL << CAN_F13BANK1_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F13BANK1_FBIT11                                         CAN_F13BANK1_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F13BANK1_FBIT12_Pos                                     (12U)
#define CAN_F13BANK1_FBIT12_Msk                                     (0x1UL << CAN_F13BANK1_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F13BANK1_FBIT12                                         CAN_F13BANK1_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F13BANK1_FBIT13_Pos                                     (13U)
#define CAN_F13BANK1_FBIT13_Msk                                     (0x1UL << CAN_F13BANK1_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F13BANK1_FBIT13                                         CAN_F13BANK1_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F13BANK1_FBIT14_Pos                                     (14U)
#define CAN_F13BANK1_FBIT14_Msk                                     (0x1UL << CAN_F13BANK1_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F13BANK1_FBIT14                                         CAN_F13BANK1_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F13BANK1_FBIT15_Pos                                     (15U)
#define CAN_F13BANK1_FBIT15_Msk                                     (0x1UL << CAN_F13BANK1_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F13BANK1_FBIT15                                         CAN_F13BANK1_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F13BANK1_FBIT16_Pos                                     (16U)
#define CAN_F13BANK1_FBIT16_Msk                                     (0x1UL << CAN_F13BANK1_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F13BANK1_FBIT16                                         CAN_F13BANK1_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F13BANK1_FBIT17_Pos                                     (17U)
#define CAN_F13BANK1_FBIT17_Msk                                     (0x1UL << CAN_F13BANK1_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F13BANK1_FBIT17                                         CAN_F13BANK1_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F13BANK1_FBIT18_Pos                                     (18U)
#define CAN_F13BANK1_FBIT18_Msk                                     (0x1UL << CAN_F13BANK1_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F13BANK1_FBIT18                                         CAN_F13BANK1_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F13BANK1_FBIT19_Pos                                     (19U)
#define CAN_F13BANK1_FBIT19_Msk                                     (0x1UL << CAN_F13BANK1_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F13BANK1_FBIT19                                         CAN_F13BANK1_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F13BANK1_FBIT20_Pos                                     (20U)
#define CAN_F13BANK1_FBIT20_Msk                                     (0x1UL << CAN_F13BANK1_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F13BANK1_FBIT20                                         CAN_F13BANK1_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F13BANK1_FBIT21_Pos                                     (21U)
#define CAN_F13BANK1_FBIT21_Msk                                     (0x1UL << CAN_F13BANK1_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F13BANK1_FBIT21                                         CAN_F13BANK1_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F13BANK1_FBIT22_Pos                                     (22U)
#define CAN_F13BANK1_FBIT22_Msk                                     (0x1UL << CAN_F13BANK1_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F13BANK1_FBIT22                                         CAN_F13BANK1_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F13BANK1_FBIT23_Pos                                     (23U)
#define CAN_F13BANK1_FBIT23_Msk                                     (0x1UL << CAN_F13BANK1_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F13BANK1_FBIT23                                         CAN_F13BANK1_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F13BANK1_FBIT24_Pos                                     (24U)
#define CAN_F13BANK1_FBIT24_Msk                                     (0x1UL << CAN_F13BANK1_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F13BANK1_FBIT24                                         CAN_F13BANK1_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F13BANK1_FBIT25_Pos                                     (25U)
#define CAN_F13BANK1_FBIT25_Msk                                     (0x1UL << CAN_F13BANK1_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F13BANK1_FBIT25                                         CAN_F13BANK1_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F13BANK1_FBIT26_Pos                                     (26U)
#define CAN_F13BANK1_FBIT26_Msk                                     (0x1UL << CAN_F13BANK1_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F13BANK1_FBIT26                                         CAN_F13BANK1_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F13BANK1_FBIT27_Pos                                     (27U)
#define CAN_F13BANK1_FBIT27_Msk                                     (0x1UL << CAN_F13BANK1_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F13BANK1_FBIT27                                         CAN_F13BANK1_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F13BANK1_FBIT28_Pos                                     (28U)
#define CAN_F13BANK1_FBIT28_Msk                                     (0x1UL << CAN_F13BANK1_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F13BANK1_FBIT28                                         CAN_F13BANK1_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F13BANK1_FBIT29_Pos                                     (29U)
#define CAN_F13BANK1_FBIT29_Msk                                     (0x1UL << CAN_F13BANK1_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F13BANK1_FBIT29                                         CAN_F13BANK1_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F13BANK1_FBIT30_Pos                                     (30U)
#define CAN_F13BANK1_FBIT30_Msk                                     (0x1UL << CAN_F13BANK1_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F13BANK1_FBIT30                                         CAN_F13BANK1_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F13BANK1_FBIT31_Pos                                     (31U)
#define CAN_F13BANK1_FBIT31_Msk                                     (0x1UL << CAN_F13BANK1_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F13BANK1_FBIT31                                         CAN_F13BANK1_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F0BANK2 register  *******************/
#define CAN_F0BANK2_FBIT0_Pos                                       (0U)
#define CAN_F0BANK2_FBIT0_Msk                                       (0x1UL << CAN_F0BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F0BANK2_FBIT0                                           CAN_F0BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F0BANK2_FBIT1_Pos                                       (1U)
#define CAN_F0BANK2_FBIT1_Msk                                       (0x1UL << CAN_F0BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F0BANK2_FBIT1                                           CAN_F0BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F0BANK2_FBIT2_Pos                                       (2U)
#define CAN_F0BANK2_FBIT2_Msk                                       (0x1UL << CAN_F0BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F0BANK2_FBIT2                                           CAN_F0BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F0BANK2_FBIT3_Pos                                       (3U)
#define CAN_F0BANK2_FBIT3_Msk                                       (0x1UL << CAN_F0BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F0BANK2_FBIT3                                           CAN_F0BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F0BANK2_FBIT4_Pos                                       (4U)
#define CAN_F0BANK2_FBIT4_Msk                                       (0x1UL << CAN_F0BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F0BANK2_FBIT4                                           CAN_F0BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F0BANK2_FBIT5_Pos                                       (5U)
#define CAN_F0BANK2_FBIT5_Msk                                       (0x1UL << CAN_F0BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F0BANK2_FBIT5                                           CAN_F0BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F0BANK2_FBIT6_Pos                                       (6U)
#define CAN_F0BANK2_FBIT6_Msk                                       (0x1UL << CAN_F0BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F0BANK2_FBIT6                                           CAN_F0BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F0BANK2_FBIT7_Pos                                       (7U)
#define CAN_F0BANK2_FBIT7_Msk                                       (0x1UL << CAN_F0BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F0BANK2_FBIT7                                           CAN_F0BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F0BANK2_FBIT8_Pos                                       (8U)
#define CAN_F0BANK2_FBIT8_Msk                                       (0x1UL << CAN_F0BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F0BANK2_FBIT8                                           CAN_F0BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F0BANK2_FBIT9_Pos                                       (9U)
#define CAN_F0BANK2_FBIT9_Msk                                       (0x1UL << CAN_F0BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F0BANK2_FBIT9                                           CAN_F0BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F0BANK2_FBIT10_Pos                                      (10U)
#define CAN_F0BANK2_FBIT10_Msk                                      (0x1UL << CAN_F0BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F0BANK2_FBIT10                                          CAN_F0BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F0BANK2_FBIT11_Pos                                      (11U)
#define CAN_F0BANK2_FBIT11_Msk                                      (0x1UL << CAN_F0BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F0BANK2_FBIT11                                          CAN_F0BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F0BANK2_FBIT12_Pos                                      (12U)
#define CAN_F0BANK2_FBIT12_Msk                                      (0x1UL << CAN_F0BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F0BANK2_FBIT12                                          CAN_F0BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F0BANK2_FBIT13_Pos                                      (13U)
#define CAN_F0BANK2_FBIT13_Msk                                      (0x1UL << CAN_F0BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F0BANK2_FBIT13                                          CAN_F0BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F0BANK2_FBIT14_Pos                                      (14U)
#define CAN_F0BANK2_FBIT14_Msk                                      (0x1UL << CAN_F0BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F0BANK2_FBIT14                                          CAN_F0BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F0BANK2_FBIT15_Pos                                      (15U)
#define CAN_F0BANK2_FBIT15_Msk                                      (0x1UL << CAN_F0BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F0BANK2_FBIT15                                          CAN_F0BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F0BANK2_FBIT16_Pos                                      (16U)
#define CAN_F0BANK2_FBIT16_Msk                                      (0x1UL << CAN_F0BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F0BANK2_FBIT16                                          CAN_F0BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F0BANK2_FBIT17_Pos                                      (17U)
#define CAN_F0BANK2_FBIT17_Msk                                      (0x1UL << CAN_F0BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F0BANK2_FBIT17                                          CAN_F0BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F0BANK2_FBIT18_Pos                                      (18U)
#define CAN_F0BANK2_FBIT18_Msk                                      (0x1UL << CAN_F0BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F0BANK2_FBIT18                                          CAN_F0BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F0BANK2_FBIT19_Pos                                      (19U)
#define CAN_F0BANK2_FBIT19_Msk                                      (0x1UL << CAN_F0BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F0BANK2_FBIT19                                          CAN_F0BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F0BANK2_FBIT20_Pos                                      (20U)
#define CAN_F0BANK2_FBIT20_Msk                                      (0x1UL << CAN_F0BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F0BANK2_FBIT20                                          CAN_F0BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F0BANK2_FBIT21_Pos                                      (21U)
#define CAN_F0BANK2_FBIT21_Msk                                      (0x1UL << CAN_F0BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F0BANK2_FBIT21                                          CAN_F0BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F0BANK2_FBIT22_Pos                                      (22U)
#define CAN_F0BANK2_FBIT22_Msk                                      (0x1UL << CAN_F0BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F0BANK2_FBIT22                                          CAN_F0BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F0BANK2_FBIT23_Pos                                      (23U)
#define CAN_F0BANK2_FBIT23_Msk                                      (0x1UL << CAN_F0BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F0BANK2_FBIT23                                          CAN_F0BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F0BANK2_FBIT24_Pos                                      (24U)
#define CAN_F0BANK2_FBIT24_Msk                                      (0x1UL << CAN_F0BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F0BANK2_FBIT24                                          CAN_F0BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F0BANK2_FBIT25_Pos                                      (25U)
#define CAN_F0BANK2_FBIT25_Msk                                      (0x1UL << CAN_F0BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F0BANK2_FBIT25                                          CAN_F0BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F0BANK2_FBIT26_Pos                                      (26U)
#define CAN_F0BANK2_FBIT26_Msk                                      (0x1UL << CAN_F0BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F0BANK2_FBIT26                                          CAN_F0BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F0BANK2_FBIT27_Pos                                      (27U)
#define CAN_F0BANK2_FBIT27_Msk                                      (0x1UL << CAN_F0BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F0BANK2_FBIT27                                          CAN_F0BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F0BANK2_FBIT28_Pos                                      (28U)
#define CAN_F0BANK2_FBIT28_Msk                                      (0x1UL << CAN_F0BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F0BANK2_FBIT28                                          CAN_F0BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F0BANK2_FBIT29_Pos                                      (29U)
#define CAN_F0BANK2_FBIT29_Msk                                      (0x1UL << CAN_F0BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F0BANK2_FBIT29                                          CAN_F0BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F0BANK2_FBIT30_Pos                                      (30U)
#define CAN_F0BANK2_FBIT30_Msk                                      (0x1UL << CAN_F0BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F0BANK2_FBIT30                                          CAN_F0BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F0BANK2_FBIT31_Pos                                      (31U)
#define CAN_F0BANK2_FBIT31_Msk                                      (0x1UL << CAN_F0BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F0BANK2_FBIT31                                          CAN_F0BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F1BANK2 register  *******************/
#define CAN_F1BANK2_FBIT0_Pos                                       (0U)
#define CAN_F1BANK2_FBIT0_Msk                                       (0x1UL << CAN_F1BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F1BANK2_FBIT0                                           CAN_F1BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F1BANK2_FBIT1_Pos                                       (1U)
#define CAN_F1BANK2_FBIT1_Msk                                       (0x1UL << CAN_F1BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F1BANK2_FBIT1                                           CAN_F1BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F1BANK2_FBIT2_Pos                                       (2U)
#define CAN_F1BANK2_FBIT2_Msk                                       (0x1UL << CAN_F1BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F1BANK2_FBIT2                                           CAN_F1BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F1BANK2_FBIT3_Pos                                       (3U)
#define CAN_F1BANK2_FBIT3_Msk                                       (0x1UL << CAN_F1BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F1BANK2_FBIT3                                           CAN_F1BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F1BANK2_FBIT4_Pos                                       (4U)
#define CAN_F1BANK2_FBIT4_Msk                                       (0x1UL << CAN_F1BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F1BANK2_FBIT4                                           CAN_F1BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F1BANK2_FBIT5_Pos                                       (5U)
#define CAN_F1BANK2_FBIT5_Msk                                       (0x1UL << CAN_F1BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F1BANK2_FBIT5                                           CAN_F1BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F1BANK2_FBIT6_Pos                                       (6U)
#define CAN_F1BANK2_FBIT6_Msk                                       (0x1UL << CAN_F1BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F1BANK2_FBIT6                                           CAN_F1BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F1BANK2_FBIT7_Pos                                       (7U)
#define CAN_F1BANK2_FBIT7_Msk                                       (0x1UL << CAN_F1BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F1BANK2_FBIT7                                           CAN_F1BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F1BANK2_FBIT8_Pos                                       (8U)
#define CAN_F1BANK2_FBIT8_Msk                                       (0x1UL << CAN_F1BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F1BANK2_FBIT8                                           CAN_F1BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F1BANK2_FBIT9_Pos                                       (9U)
#define CAN_F1BANK2_FBIT9_Msk                                       (0x1UL << CAN_F1BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F1BANK2_FBIT9                                           CAN_F1BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F1BANK2_FBIT10_Pos                                      (10U)
#define CAN_F1BANK2_FBIT10_Msk                                      (0x1UL << CAN_F1BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F1BANK2_FBIT10                                          CAN_F1BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F1BANK2_FBIT11_Pos                                      (11U)
#define CAN_F1BANK2_FBIT11_Msk                                      (0x1UL << CAN_F1BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F1BANK2_FBIT11                                          CAN_F1BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F1BANK2_FBIT12_Pos                                      (12U)
#define CAN_F1BANK2_FBIT12_Msk                                      (0x1UL << CAN_F1BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F1BANK2_FBIT12                                          CAN_F1BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F1BANK2_FBIT13_Pos                                      (13U)
#define CAN_F1BANK2_FBIT13_Msk                                      (0x1UL << CAN_F1BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F1BANK2_FBIT13                                          CAN_F1BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F1BANK2_FBIT14_Pos                                      (14U)
#define CAN_F1BANK2_FBIT14_Msk                                      (0x1UL << CAN_F1BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F1BANK2_FBIT14                                          CAN_F1BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F1BANK2_FBIT15_Pos                                      (15U)
#define CAN_F1BANK2_FBIT15_Msk                                      (0x1UL << CAN_F1BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F1BANK2_FBIT15                                          CAN_F1BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F1BANK2_FBIT16_Pos                                      (16U)
#define CAN_F1BANK2_FBIT16_Msk                                      (0x1UL << CAN_F1BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F1BANK2_FBIT16                                          CAN_F1BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F1BANK2_FBIT17_Pos                                      (17U)
#define CAN_F1BANK2_FBIT17_Msk                                      (0x1UL << CAN_F1BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F1BANK2_FBIT17                                          CAN_F1BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F1BANK2_FBIT18_Pos                                      (18U)
#define CAN_F1BANK2_FBIT18_Msk                                      (0x1UL << CAN_F1BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F1BANK2_FBIT18                                          CAN_F1BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F1BANK2_FBIT19_Pos                                      (19U)
#define CAN_F1BANK2_FBIT19_Msk                                      (0x1UL << CAN_F1BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F1BANK2_FBIT19                                          CAN_F1BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F1BANK2_FBIT20_Pos                                      (20U)
#define CAN_F1BANK2_FBIT20_Msk                                      (0x1UL << CAN_F1BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F1BANK2_FBIT20                                          CAN_F1BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F1BANK2_FBIT21_Pos                                      (21U)
#define CAN_F1BANK2_FBIT21_Msk                                      (0x1UL << CAN_F1BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F1BANK2_FBIT21                                          CAN_F1BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F1BANK2_FBIT22_Pos                                      (22U)
#define CAN_F1BANK2_FBIT22_Msk                                      (0x1UL << CAN_F1BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F1BANK2_FBIT22                                          CAN_F1BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F1BANK2_FBIT23_Pos                                      (23U)
#define CAN_F1BANK2_FBIT23_Msk                                      (0x1UL << CAN_F1BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F1BANK2_FBIT23                                          CAN_F1BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F1BANK2_FBIT24_Pos                                      (24U)
#define CAN_F1BANK2_FBIT24_Msk                                      (0x1UL << CAN_F1BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F1BANK2_FBIT24                                          CAN_F1BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F1BANK2_FBIT25_Pos                                      (25U)
#define CAN_F1BANK2_FBIT25_Msk                                      (0x1UL << CAN_F1BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F1BANK2_FBIT25                                          CAN_F1BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F1BANK2_FBIT26_Pos                                      (26U)
#define CAN_F1BANK2_FBIT26_Msk                                      (0x1UL << CAN_F1BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F1BANK2_FBIT26                                          CAN_F1BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F1BANK2_FBIT27_Pos                                      (27U)
#define CAN_F1BANK2_FBIT27_Msk                                      (0x1UL << CAN_F1BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F1BANK2_FBIT27                                          CAN_F1BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F1BANK2_FBIT28_Pos                                      (28U)
#define CAN_F1BANK2_FBIT28_Msk                                      (0x1UL << CAN_F1BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F1BANK2_FBIT28                                          CAN_F1BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F1BANK2_FBIT29_Pos                                      (29U)
#define CAN_F1BANK2_FBIT29_Msk                                      (0x1UL << CAN_F1BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F1BANK2_FBIT29                                          CAN_F1BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F1BANK2_FBIT30_Pos                                      (30U)
#define CAN_F1BANK2_FBIT30_Msk                                      (0x1UL << CAN_F1BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F1BANK2_FBIT30                                          CAN_F1BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F1BANK2_FBIT31_Pos                                      (31U)
#define CAN_F1BANK2_FBIT31_Msk                                      (0x1UL << CAN_F1BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F1BANK2_FBIT31                                          CAN_F1BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F2BANK2 register  *******************/
#define CAN_F2BANK2_FBIT0_Pos                                       (0U)
#define CAN_F2BANK2_FBIT0_Msk                                       (0x1UL << CAN_F2BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F2BANK2_FBIT0                                           CAN_F2BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F2BANK2_FBIT1_Pos                                       (1U)
#define CAN_F2BANK2_FBIT1_Msk                                       (0x1UL << CAN_F2BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F2BANK2_FBIT1                                           CAN_F2BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F2BANK2_FBIT2_Pos                                       (2U)
#define CAN_F2BANK2_FBIT2_Msk                                       (0x1UL << CAN_F2BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F2BANK2_FBIT2                                           CAN_F2BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F2BANK2_FBIT3_Pos                                       (3U)
#define CAN_F2BANK2_FBIT3_Msk                                       (0x1UL << CAN_F2BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F2BANK2_FBIT3                                           CAN_F2BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F2BANK2_FBIT4_Pos                                       (4U)
#define CAN_F2BANK2_FBIT4_Msk                                       (0x1UL << CAN_F2BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F2BANK2_FBIT4                                           CAN_F2BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F2BANK2_FBIT5_Pos                                       (5U)
#define CAN_F2BANK2_FBIT5_Msk                                       (0x1UL << CAN_F2BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F2BANK2_FBIT5                                           CAN_F2BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F2BANK2_FBIT6_Pos                                       (6U)
#define CAN_F2BANK2_FBIT6_Msk                                       (0x1UL << CAN_F2BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F2BANK2_FBIT6                                           CAN_F2BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F2BANK2_FBIT7_Pos                                       (7U)
#define CAN_F2BANK2_FBIT7_Msk                                       (0x1UL << CAN_F2BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F2BANK2_FBIT7                                           CAN_F2BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F2BANK2_FBIT8_Pos                                       (8U)
#define CAN_F2BANK2_FBIT8_Msk                                       (0x1UL << CAN_F2BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F2BANK2_FBIT8                                           CAN_F2BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F2BANK2_FBIT9_Pos                                       (9U)
#define CAN_F2BANK2_FBIT9_Msk                                       (0x1UL << CAN_F2BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F2BANK2_FBIT9                                           CAN_F2BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F2BANK2_FBIT10_Pos                                      (10U)
#define CAN_F2BANK2_FBIT10_Msk                                      (0x1UL << CAN_F2BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F2BANK2_FBIT10                                          CAN_F2BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F2BANK2_FBIT11_Pos                                      (11U)
#define CAN_F2BANK2_FBIT11_Msk                                      (0x1UL << CAN_F2BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F2BANK2_FBIT11                                          CAN_F2BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F2BANK2_FBIT12_Pos                                      (12U)
#define CAN_F2BANK2_FBIT12_Msk                                      (0x1UL << CAN_F2BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F2BANK2_FBIT12                                          CAN_F2BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F2BANK2_FBIT13_Pos                                      (13U)
#define CAN_F2BANK2_FBIT13_Msk                                      (0x1UL << CAN_F2BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F2BANK2_FBIT13                                          CAN_F2BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F2BANK2_FBIT14_Pos                                      (14U)
#define CAN_F2BANK2_FBIT14_Msk                                      (0x1UL << CAN_F2BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F2BANK2_FBIT14                                          CAN_F2BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F2BANK2_FBIT15_Pos                                      (15U)
#define CAN_F2BANK2_FBIT15_Msk                                      (0x1UL << CAN_F2BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F2BANK2_FBIT15                                          CAN_F2BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F2BANK2_FBIT16_Pos                                      (16U)
#define CAN_F2BANK2_FBIT16_Msk                                      (0x1UL << CAN_F2BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F2BANK2_FBIT16                                          CAN_F2BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F2BANK2_FBIT17_Pos                                      (17U)
#define CAN_F2BANK2_FBIT17_Msk                                      (0x1UL << CAN_F2BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F2BANK2_FBIT17                                          CAN_F2BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F2BANK2_FBIT18_Pos                                      (18U)
#define CAN_F2BANK2_FBIT18_Msk                                      (0x1UL << CAN_F2BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F2BANK2_FBIT18                                          CAN_F2BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F2BANK2_FBIT19_Pos                                      (19U)
#define CAN_F2BANK2_FBIT19_Msk                                      (0x1UL << CAN_F2BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F2BANK2_FBIT19                                          CAN_F2BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F2BANK2_FBIT20_Pos                                      (20U)
#define CAN_F2BANK2_FBIT20_Msk                                      (0x1UL << CAN_F2BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F2BANK2_FBIT20                                          CAN_F2BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F2BANK2_FBIT21_Pos                                      (21U)
#define CAN_F2BANK2_FBIT21_Msk                                      (0x1UL << CAN_F2BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F2BANK2_FBIT21                                          CAN_F2BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F2BANK2_FBIT22_Pos                                      (22U)
#define CAN_F2BANK2_FBIT22_Msk                                      (0x1UL << CAN_F2BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F2BANK2_FBIT22                                          CAN_F2BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F2BANK2_FBIT23_Pos                                      (23U)
#define CAN_F2BANK2_FBIT23_Msk                                      (0x1UL << CAN_F2BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F2BANK2_FBIT23                                          CAN_F2BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F2BANK2_FBIT24_Pos                                      (24U)
#define CAN_F2BANK2_FBIT24_Msk                                      (0x1UL << CAN_F2BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F2BANK2_FBIT24                                          CAN_F2BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F2BANK2_FBIT25_Pos                                      (25U)
#define CAN_F2BANK2_FBIT25_Msk                                      (0x1UL << CAN_F2BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F2BANK2_FBIT25                                          CAN_F2BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F2BANK2_FBIT26_Pos                                      (26U)
#define CAN_F2BANK2_FBIT26_Msk                                      (0x1UL << CAN_F2BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F2BANK2_FBIT26                                          CAN_F2BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F2BANK2_FBIT27_Pos                                      (27U)
#define CAN_F2BANK2_FBIT27_Msk                                      (0x1UL << CAN_F2BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F2BANK2_FBIT27                                          CAN_F2BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F2BANK2_FBIT28_Pos                                      (28U)
#define CAN_F2BANK2_FBIT28_Msk                                      (0x1UL << CAN_F2BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F2BANK2_FBIT28                                          CAN_F2BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F2BANK2_FBIT29_Pos                                      (29U)
#define CAN_F2BANK2_FBIT29_Msk                                      (0x1UL << CAN_F2BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F2BANK2_FBIT29                                          CAN_F2BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F2BANK2_FBIT30_Pos                                      (30U)
#define CAN_F2BANK2_FBIT30_Msk                                      (0x1UL << CAN_F2BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F2BANK2_FBIT30                                          CAN_F2BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F2BANK2_FBIT31_Pos                                      (31U)
#define CAN_F2BANK2_FBIT31_Msk                                      (0x1UL << CAN_F2BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F2BANK2_FBIT31                                          CAN_F2BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F3BANK2 register  *******************/
#define CAN_F3BANK2_FBIT0_Pos                                       (0U)
#define CAN_F3BANK2_FBIT0_Msk                                       (0x1UL << CAN_F3BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F3BANK2_FBIT0                                           CAN_F3BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F3BANK2_FBIT1_Pos                                       (1U)
#define CAN_F3BANK2_FBIT1_Msk                                       (0x1UL << CAN_F3BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F3BANK2_FBIT1                                           CAN_F3BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F3BANK2_FBIT2_Pos                                       (2U)
#define CAN_F3BANK2_FBIT2_Msk                                       (0x1UL << CAN_F3BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F3BANK2_FBIT2                                           CAN_F3BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F3BANK2_FBIT3_Pos                                       (3U)
#define CAN_F3BANK2_FBIT3_Msk                                       (0x1UL << CAN_F3BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F3BANK2_FBIT3                                           CAN_F3BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F3BANK2_FBIT4_Pos                                       (4U)
#define CAN_F3BANK2_FBIT4_Msk                                       (0x1UL << CAN_F3BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F3BANK2_FBIT4                                           CAN_F3BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F3BANK2_FBIT5_Pos                                       (5U)
#define CAN_F3BANK2_FBIT5_Msk                                       (0x1UL << CAN_F3BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F3BANK2_FBIT5                                           CAN_F3BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F3BANK2_FBIT6_Pos                                       (6U)
#define CAN_F3BANK2_FBIT6_Msk                                       (0x1UL << CAN_F3BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F3BANK2_FBIT6                                           CAN_F3BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F3BANK2_FBIT7_Pos                                       (7U)
#define CAN_F3BANK2_FBIT7_Msk                                       (0x1UL << CAN_F3BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F3BANK2_FBIT7                                           CAN_F3BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F3BANK2_FBIT8_Pos                                       (8U)
#define CAN_F3BANK2_FBIT8_Msk                                       (0x1UL << CAN_F3BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F3BANK2_FBIT8                                           CAN_F3BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F3BANK2_FBIT9_Pos                                       (9U)
#define CAN_F3BANK2_FBIT9_Msk                                       (0x1UL << CAN_F3BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F3BANK2_FBIT9                                           CAN_F3BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F3BANK2_FBIT10_Pos                                      (10U)
#define CAN_F3BANK2_FBIT10_Msk                                      (0x1UL << CAN_F3BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F3BANK2_FBIT10                                          CAN_F3BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F3BANK2_FBIT11_Pos                                      (11U)
#define CAN_F3BANK2_FBIT11_Msk                                      (0x1UL << CAN_F3BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F3BANK2_FBIT11                                          CAN_F3BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F3BANK2_FBIT12_Pos                                      (12U)
#define CAN_F3BANK2_FBIT12_Msk                                      (0x1UL << CAN_F3BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F3BANK2_FBIT12                                          CAN_F3BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F3BANK2_FBIT13_Pos                                      (13U)
#define CAN_F3BANK2_FBIT13_Msk                                      (0x1UL << CAN_F3BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F3BANK2_FBIT13                                          CAN_F3BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F3BANK2_FBIT14_Pos                                      (14U)
#define CAN_F3BANK2_FBIT14_Msk                                      (0x1UL << CAN_F3BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F3BANK2_FBIT14                                          CAN_F3BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F3BANK2_FBIT15_Pos                                      (15U)
#define CAN_F3BANK2_FBIT15_Msk                                      (0x1UL << CAN_F3BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F3BANK2_FBIT15                                          CAN_F3BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F3BANK2_FBIT16_Pos                                      (16U)
#define CAN_F3BANK2_FBIT16_Msk                                      (0x1UL << CAN_F3BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F3BANK2_FBIT16                                          CAN_F3BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F3BANK2_FBIT17_Pos                                      (17U)
#define CAN_F3BANK2_FBIT17_Msk                                      (0x1UL << CAN_F3BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F3BANK2_FBIT17                                          CAN_F3BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F3BANK2_FBIT18_Pos                                      (18U)
#define CAN_F3BANK2_FBIT18_Msk                                      (0x1UL << CAN_F3BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F3BANK2_FBIT18                                          CAN_F3BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F3BANK2_FBIT19_Pos                                      (19U)
#define CAN_F3BANK2_FBIT19_Msk                                      (0x1UL << CAN_F3BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F3BANK2_FBIT19                                          CAN_F3BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F3BANK2_FBIT20_Pos                                      (20U)
#define CAN_F3BANK2_FBIT20_Msk                                      (0x1UL << CAN_F3BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F3BANK2_FBIT20                                          CAN_F3BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F3BANK2_FBIT21_Pos                                      (21U)
#define CAN_F3BANK2_FBIT21_Msk                                      (0x1UL << CAN_F3BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F3BANK2_FBIT21                                          CAN_F3BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F3BANK2_FBIT22_Pos                                      (22U)
#define CAN_F3BANK2_FBIT22_Msk                                      (0x1UL << CAN_F3BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F3BANK2_FBIT22                                          CAN_F3BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F3BANK2_FBIT23_Pos                                      (23U)
#define CAN_F3BANK2_FBIT23_Msk                                      (0x1UL << CAN_F3BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F3BANK2_FBIT23                                          CAN_F3BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F3BANK2_FBIT24_Pos                                      (24U)
#define CAN_F3BANK2_FBIT24_Msk                                      (0x1UL << CAN_F3BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F3BANK2_FBIT24                                          CAN_F3BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F3BANK2_FBIT25_Pos                                      (25U)
#define CAN_F3BANK2_FBIT25_Msk                                      (0x1UL << CAN_F3BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F3BANK2_FBIT25                                          CAN_F3BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F3BANK2_FBIT26_Pos                                      (26U)
#define CAN_F3BANK2_FBIT26_Msk                                      (0x1UL << CAN_F3BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F3BANK2_FBIT26                                          CAN_F3BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F3BANK2_FBIT27_Pos                                      (27U)
#define CAN_F3BANK2_FBIT27_Msk                                      (0x1UL << CAN_F3BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F3BANK2_FBIT27                                          CAN_F3BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F3BANK2_FBIT28_Pos                                      (28U)
#define CAN_F3BANK2_FBIT28_Msk                                      (0x1UL << CAN_F3BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F3BANK2_FBIT28                                          CAN_F3BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F3BANK2_FBIT29_Pos                                      (29U)
#define CAN_F3BANK2_FBIT29_Msk                                      (0x1UL << CAN_F3BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F3BANK2_FBIT29                                          CAN_F3BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F3BANK2_FBIT30_Pos                                      (30U)
#define CAN_F3BANK2_FBIT30_Msk                                      (0x1UL << CAN_F3BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F3BANK2_FBIT30                                          CAN_F3BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F3BANK2_FBIT31_Pos                                      (31U)
#define CAN_F3BANK2_FBIT31_Msk                                      (0x1UL << CAN_F3BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F3BANK2_FBIT31                                          CAN_F3BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F4BANK2 register  *******************/
#define CAN_F4BANK2_FBIT0_Pos                                       (0U)
#define CAN_F4BANK2_FBIT0_Msk                                       (0x1UL << CAN_F4BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F4BANK2_FBIT0                                           CAN_F4BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F4BANK2_FBIT1_Pos                                       (1U)
#define CAN_F4BANK2_FBIT1_Msk                                       (0x1UL << CAN_F4BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F4BANK2_FBIT1                                           CAN_F4BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F4BANK2_FBIT2_Pos                                       (2U)
#define CAN_F4BANK2_FBIT2_Msk                                       (0x1UL << CAN_F4BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F4BANK2_FBIT2                                           CAN_F4BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F4BANK2_FBIT3_Pos                                       (3U)
#define CAN_F4BANK2_FBIT3_Msk                                       (0x1UL << CAN_F4BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F4BANK2_FBIT3                                           CAN_F4BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F4BANK2_FBIT4_Pos                                       (4U)
#define CAN_F4BANK2_FBIT4_Msk                                       (0x1UL << CAN_F4BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F4BANK2_FBIT4                                           CAN_F4BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F4BANK2_FBIT5_Pos                                       (5U)
#define CAN_F4BANK2_FBIT5_Msk                                       (0x1UL << CAN_F4BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F4BANK2_FBIT5                                           CAN_F4BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F4BANK2_FBIT6_Pos                                       (6U)
#define CAN_F4BANK2_FBIT6_Msk                                       (0x1UL << CAN_F4BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F4BANK2_FBIT6                                           CAN_F4BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F4BANK2_FBIT7_Pos                                       (7U)
#define CAN_F4BANK2_FBIT7_Msk                                       (0x1UL << CAN_F4BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F4BANK2_FBIT7                                           CAN_F4BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F4BANK2_FBIT8_Pos                                       (8U)
#define CAN_F4BANK2_FBIT8_Msk                                       (0x1UL << CAN_F4BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F4BANK2_FBIT8                                           CAN_F4BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F4BANK2_FBIT9_Pos                                       (9U)
#define CAN_F4BANK2_FBIT9_Msk                                       (0x1UL << CAN_F4BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F4BANK2_FBIT9                                           CAN_F4BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F4BANK2_FBIT10_Pos                                      (10U)
#define CAN_F4BANK2_FBIT10_Msk                                      (0x1UL << CAN_F4BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F4BANK2_FBIT10                                          CAN_F4BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F4BANK2_FBIT11_Pos                                      (11U)
#define CAN_F4BANK2_FBIT11_Msk                                      (0x1UL << CAN_F4BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F4BANK2_FBIT11                                          CAN_F4BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F4BANK2_FBIT12_Pos                                      (12U)
#define CAN_F4BANK2_FBIT12_Msk                                      (0x1UL << CAN_F4BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F4BANK2_FBIT12                                          CAN_F4BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F4BANK2_FBIT13_Pos                                      (13U)
#define CAN_F4BANK2_FBIT13_Msk                                      (0x1UL << CAN_F4BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F4BANK2_FBIT13                                          CAN_F4BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F4BANK2_FBIT14_Pos                                      (14U)
#define CAN_F4BANK2_FBIT14_Msk                                      (0x1UL << CAN_F4BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F4BANK2_FBIT14                                          CAN_F4BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F4BANK2_FBIT15_Pos                                      (15U)
#define CAN_F4BANK2_FBIT15_Msk                                      (0x1UL << CAN_F4BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F4BANK2_FBIT15                                          CAN_F4BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F4BANK2_FBIT16_Pos                                      (16U)
#define CAN_F4BANK2_FBIT16_Msk                                      (0x1UL << CAN_F4BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F4BANK2_FBIT16                                          CAN_F4BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F4BANK2_FBIT17_Pos                                      (17U)
#define CAN_F4BANK2_FBIT17_Msk                                      (0x1UL << CAN_F4BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F4BANK2_FBIT17                                          CAN_F4BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F4BANK2_FBIT18_Pos                                      (18U)
#define CAN_F4BANK2_FBIT18_Msk                                      (0x1UL << CAN_F4BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F4BANK2_FBIT18                                          CAN_F4BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F4BANK2_FBIT19_Pos                                      (19U)
#define CAN_F4BANK2_FBIT19_Msk                                      (0x1UL << CAN_F4BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F4BANK2_FBIT19                                          CAN_F4BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F4BANK2_FBIT20_Pos                                      (20U)
#define CAN_F4BANK2_FBIT20_Msk                                      (0x1UL << CAN_F4BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F4BANK2_FBIT20                                          CAN_F4BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F4BANK2_FBIT21_Pos                                      (21U)
#define CAN_F4BANK2_FBIT21_Msk                                      (0x1UL << CAN_F4BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F4BANK2_FBIT21                                          CAN_F4BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F4BANK2_FBIT22_Pos                                      (22U)
#define CAN_F4BANK2_FBIT22_Msk                                      (0x1UL << CAN_F4BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F4BANK2_FBIT22                                          CAN_F4BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F4BANK2_FBIT23_Pos                                      (23U)
#define CAN_F4BANK2_FBIT23_Msk                                      (0x1UL << CAN_F4BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F4BANK2_FBIT23                                          CAN_F4BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F4BANK2_FBIT24_Pos                                      (24U)
#define CAN_F4BANK2_FBIT24_Msk                                      (0x1UL << CAN_F4BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F4BANK2_FBIT24                                          CAN_F4BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F4BANK2_FBIT25_Pos                                      (25U)
#define CAN_F4BANK2_FBIT25_Msk                                      (0x1UL << CAN_F4BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F4BANK2_FBIT25                                          CAN_F4BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F4BANK2_FBIT26_Pos                                      (26U)
#define CAN_F4BANK2_FBIT26_Msk                                      (0x1UL << CAN_F4BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F4BANK2_FBIT26                                          CAN_F4BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F4BANK2_FBIT27_Pos                                      (27U)
#define CAN_F4BANK2_FBIT27_Msk                                      (0x1UL << CAN_F4BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F4BANK2_FBIT27                                          CAN_F4BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F4BANK2_FBIT28_Pos                                      (28U)
#define CAN_F4BANK2_FBIT28_Msk                                      (0x1UL << CAN_F4BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F4BANK2_FBIT28                                          CAN_F4BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F4BANK2_FBIT29_Pos                                      (29U)
#define CAN_F4BANK2_FBIT29_Msk                                      (0x1UL << CAN_F4BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F4BANK2_FBIT29                                          CAN_F4BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F4BANK2_FBIT30_Pos                                      (30U)
#define CAN_F4BANK2_FBIT30_Msk                                      (0x1UL << CAN_F4BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F4BANK2_FBIT30                                          CAN_F4BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F4BANK2_FBIT31_Pos                                      (31U)
#define CAN_F4BANK2_FBIT31_Msk                                      (0x1UL << CAN_F4BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F4BANK2_FBIT31                                          CAN_F4BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F5BANK2 register  *******************/
#define CAN_F5BANK2_FBIT0_Pos                                       (0U)
#define CAN_F5BANK2_FBIT0_Msk                                       (0x1UL << CAN_F5BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F5BANK2_FBIT0                                           CAN_F5BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F5BANK2_FBIT1_Pos                                       (1U)
#define CAN_F5BANK2_FBIT1_Msk                                       (0x1UL << CAN_F5BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F5BANK2_FBIT1                                           CAN_F5BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F5BANK2_FBIT2_Pos                                       (2U)
#define CAN_F5BANK2_FBIT2_Msk                                       (0x1UL << CAN_F5BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F5BANK2_FBIT2                                           CAN_F5BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F5BANK2_FBIT3_Pos                                       (3U)
#define CAN_F5BANK2_FBIT3_Msk                                       (0x1UL << CAN_F5BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F5BANK2_FBIT3                                           CAN_F5BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F5BANK2_FBIT4_Pos                                       (4U)
#define CAN_F5BANK2_FBIT4_Msk                                       (0x1UL << CAN_F5BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F5BANK2_FBIT4                                           CAN_F5BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F5BANK2_FBIT5_Pos                                       (5U)
#define CAN_F5BANK2_FBIT5_Msk                                       (0x1UL << CAN_F5BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F5BANK2_FBIT5                                           CAN_F5BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F5BANK2_FBIT6_Pos                                       (6U)
#define CAN_F5BANK2_FBIT6_Msk                                       (0x1UL << CAN_F5BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F5BANK2_FBIT6                                           CAN_F5BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F5BANK2_FBIT7_Pos                                       (7U)
#define CAN_F5BANK2_FBIT7_Msk                                       (0x1UL << CAN_F5BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F5BANK2_FBIT7                                           CAN_F5BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F5BANK2_FBIT8_Pos                                       (8U)
#define CAN_F5BANK2_FBIT8_Msk                                       (0x1UL << CAN_F5BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F5BANK2_FBIT8                                           CAN_F5BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F5BANK2_FBIT9_Pos                                       (9U)
#define CAN_F5BANK2_FBIT9_Msk                                       (0x1UL << CAN_F5BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F5BANK2_FBIT9                                           CAN_F5BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F5BANK2_FBIT10_Pos                                      (10U)
#define CAN_F5BANK2_FBIT10_Msk                                      (0x1UL << CAN_F5BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F5BANK2_FBIT10                                          CAN_F5BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F5BANK2_FBIT11_Pos                                      (11U)
#define CAN_F5BANK2_FBIT11_Msk                                      (0x1UL << CAN_F5BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F5BANK2_FBIT11                                          CAN_F5BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F5BANK2_FBIT12_Pos                                      (12U)
#define CAN_F5BANK2_FBIT12_Msk                                      (0x1UL << CAN_F5BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F5BANK2_FBIT12                                          CAN_F5BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F5BANK2_FBIT13_Pos                                      (13U)
#define CAN_F5BANK2_FBIT13_Msk                                      (0x1UL << CAN_F5BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F5BANK2_FBIT13                                          CAN_F5BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F5BANK2_FBIT14_Pos                                      (14U)
#define CAN_F5BANK2_FBIT14_Msk                                      (0x1UL << CAN_F5BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F5BANK2_FBIT14                                          CAN_F5BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F5BANK2_FBIT15_Pos                                      (15U)
#define CAN_F5BANK2_FBIT15_Msk                                      (0x1UL << CAN_F5BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F5BANK2_FBIT15                                          CAN_F5BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F5BANK2_FBIT16_Pos                                      (16U)
#define CAN_F5BANK2_FBIT16_Msk                                      (0x1UL << CAN_F5BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F5BANK2_FBIT16                                          CAN_F5BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F5BANK2_FBIT17_Pos                                      (17U)
#define CAN_F5BANK2_FBIT17_Msk                                      (0x1UL << CAN_F5BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F5BANK2_FBIT17                                          CAN_F5BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F5BANK2_FBIT18_Pos                                      (18U)
#define CAN_F5BANK2_FBIT18_Msk                                      (0x1UL << CAN_F5BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F5BANK2_FBIT18                                          CAN_F5BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F5BANK2_FBIT19_Pos                                      (19U)
#define CAN_F5BANK2_FBIT19_Msk                                      (0x1UL << CAN_F5BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F5BANK2_FBIT19                                          CAN_F5BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F5BANK2_FBIT20_Pos                                      (20U)
#define CAN_F5BANK2_FBIT20_Msk                                      (0x1UL << CAN_F5BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F5BANK2_FBIT20                                          CAN_F5BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F5BANK2_FBIT21_Pos                                      (21U)
#define CAN_F5BANK2_FBIT21_Msk                                      (0x1UL << CAN_F5BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F5BANK2_FBIT21                                          CAN_F5BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F5BANK2_FBIT22_Pos                                      (22U)
#define CAN_F5BANK2_FBIT22_Msk                                      (0x1UL << CAN_F5BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F5BANK2_FBIT22                                          CAN_F5BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F5BANK2_FBIT23_Pos                                      (23U)
#define CAN_F5BANK2_FBIT23_Msk                                      (0x1UL << CAN_F5BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F5BANK2_FBIT23                                          CAN_F5BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F5BANK2_FBIT24_Pos                                      (24U)
#define CAN_F5BANK2_FBIT24_Msk                                      (0x1UL << CAN_F5BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F5BANK2_FBIT24                                          CAN_F5BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F5BANK2_FBIT25_Pos                                      (25U)
#define CAN_F5BANK2_FBIT25_Msk                                      (0x1UL << CAN_F5BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F5BANK2_FBIT25                                          CAN_F5BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F5BANK2_FBIT26_Pos                                      (26U)
#define CAN_F5BANK2_FBIT26_Msk                                      (0x1UL << CAN_F5BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F5BANK2_FBIT26                                          CAN_F5BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F5BANK2_FBIT27_Pos                                      (27U)
#define CAN_F5BANK2_FBIT27_Msk                                      (0x1UL << CAN_F5BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F5BANK2_FBIT27                                          CAN_F5BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F5BANK2_FBIT28_Pos                                      (28U)
#define CAN_F5BANK2_FBIT28_Msk                                      (0x1UL << CAN_F5BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F5BANK2_FBIT28                                          CAN_F5BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F5BANK2_FBIT29_Pos                                      (29U)
#define CAN_F5BANK2_FBIT29_Msk                                      (0x1UL << CAN_F5BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F5BANK2_FBIT29                                          CAN_F5BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F5BANK2_FBIT30_Pos                                      (30U)
#define CAN_F5BANK2_FBIT30_Msk                                      (0x1UL << CAN_F5BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F5BANK2_FBIT30                                          CAN_F5BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F5BANK2_FBIT31_Pos                                      (31U)
#define CAN_F5BANK2_FBIT31_Msk                                      (0x1UL << CAN_F5BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F5BANK2_FBIT31                                          CAN_F5BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F6BANK2 register  *******************/
#define CAN_F6BANK2_FBIT0_Pos                                       (0U)
#define CAN_F6BANK2_FBIT0_Msk                                       (0x1UL << CAN_F6BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F6BANK2_FBIT0                                           CAN_F6BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F6BANK2_FBIT1_Pos                                       (1U)
#define CAN_F6BANK2_FBIT1_Msk                                       (0x1UL << CAN_F6BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F6BANK2_FBIT1                                           CAN_F6BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F6BANK2_FBIT2_Pos                                       (2U)
#define CAN_F6BANK2_FBIT2_Msk                                       (0x1UL << CAN_F6BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F6BANK2_FBIT2                                           CAN_F6BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F6BANK2_FBIT3_Pos                                       (3U)
#define CAN_F6BANK2_FBIT3_Msk                                       (0x1UL << CAN_F6BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F6BANK2_FBIT3                                           CAN_F6BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F6BANK2_FBIT4_Pos                                       (4U)
#define CAN_F6BANK2_FBIT4_Msk                                       (0x1UL << CAN_F6BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F6BANK2_FBIT4                                           CAN_F6BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F6BANK2_FBIT5_Pos                                       (5U)
#define CAN_F6BANK2_FBIT5_Msk                                       (0x1UL << CAN_F6BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F6BANK2_FBIT5                                           CAN_F6BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F6BANK2_FBIT6_Pos                                       (6U)
#define CAN_F6BANK2_FBIT6_Msk                                       (0x1UL << CAN_F6BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F6BANK2_FBIT6                                           CAN_F6BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F6BANK2_FBIT7_Pos                                       (7U)
#define CAN_F6BANK2_FBIT7_Msk                                       (0x1UL << CAN_F6BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F6BANK2_FBIT7                                           CAN_F6BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F6BANK2_FBIT8_Pos                                       (8U)
#define CAN_F6BANK2_FBIT8_Msk                                       (0x1UL << CAN_F6BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F6BANK2_FBIT8                                           CAN_F6BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F6BANK2_FBIT9_Pos                                       (9U)
#define CAN_F6BANK2_FBIT9_Msk                                       (0x1UL << CAN_F6BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F6BANK2_FBIT9                                           CAN_F6BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F6BANK2_FBIT10_Pos                                      (10U)
#define CAN_F6BANK2_FBIT10_Msk                                      (0x1UL << CAN_F6BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F6BANK2_FBIT10                                          CAN_F6BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F6BANK2_FBIT11_Pos                                      (11U)
#define CAN_F6BANK2_FBIT11_Msk                                      (0x1UL << CAN_F6BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F6BANK2_FBIT11                                          CAN_F6BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F6BANK2_FBIT12_Pos                                      (12U)
#define CAN_F6BANK2_FBIT12_Msk                                      (0x1UL << CAN_F6BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F6BANK2_FBIT12                                          CAN_F6BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F6BANK2_FBIT13_Pos                                      (13U)
#define CAN_F6BANK2_FBIT13_Msk                                      (0x1UL << CAN_F6BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F6BANK2_FBIT13                                          CAN_F6BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F6BANK2_FBIT14_Pos                                      (14U)
#define CAN_F6BANK2_FBIT14_Msk                                      (0x1UL << CAN_F6BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F6BANK2_FBIT14                                          CAN_F6BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F6BANK2_FBIT15_Pos                                      (15U)
#define CAN_F6BANK2_FBIT15_Msk                                      (0x1UL << CAN_F6BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F6BANK2_FBIT15                                          CAN_F6BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F6BANK2_FBIT16_Pos                                      (16U)
#define CAN_F6BANK2_FBIT16_Msk                                      (0x1UL << CAN_F6BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F6BANK2_FBIT16                                          CAN_F6BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F6BANK2_FBIT17_Pos                                      (17U)
#define CAN_F6BANK2_FBIT17_Msk                                      (0x1UL << CAN_F6BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F6BANK2_FBIT17                                          CAN_F6BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F6BANK2_FBIT18_Pos                                      (18U)
#define CAN_F6BANK2_FBIT18_Msk                                      (0x1UL << CAN_F6BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F6BANK2_FBIT18                                          CAN_F6BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F6BANK2_FBIT19_Pos                                      (19U)
#define CAN_F6BANK2_FBIT19_Msk                                      (0x1UL << CAN_F6BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F6BANK2_FBIT19                                          CAN_F6BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F6BANK2_FBIT20_Pos                                      (20U)
#define CAN_F6BANK2_FBIT20_Msk                                      (0x1UL << CAN_F6BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F6BANK2_FBIT20                                          CAN_F6BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F6BANK2_FBIT21_Pos                                      (21U)
#define CAN_F6BANK2_FBIT21_Msk                                      (0x1UL << CAN_F6BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F6BANK2_FBIT21                                          CAN_F6BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F6BANK2_FBIT22_Pos                                      (22U)
#define CAN_F6BANK2_FBIT22_Msk                                      (0x1UL << CAN_F6BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F6BANK2_FBIT22                                          CAN_F6BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F6BANK2_FBIT23_Pos                                      (23U)
#define CAN_F6BANK2_FBIT23_Msk                                      (0x1UL << CAN_F6BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F6BANK2_FBIT23                                          CAN_F6BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F6BANK2_FBIT24_Pos                                      (24U)
#define CAN_F6BANK2_FBIT24_Msk                                      (0x1UL << CAN_F6BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F6BANK2_FBIT24                                          CAN_F6BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F6BANK2_FBIT25_Pos                                      (25U)
#define CAN_F6BANK2_FBIT25_Msk                                      (0x1UL << CAN_F6BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F6BANK2_FBIT25                                          CAN_F6BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F6BANK2_FBIT26_Pos                                      (26U)
#define CAN_F6BANK2_FBIT26_Msk                                      (0x1UL << CAN_F6BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F6BANK2_FBIT26                                          CAN_F6BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F6BANK2_FBIT27_Pos                                      (27U)
#define CAN_F6BANK2_FBIT27_Msk                                      (0x1UL << CAN_F6BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F6BANK2_FBIT27                                          CAN_F6BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F6BANK2_FBIT28_Pos                                      (28U)
#define CAN_F6BANK2_FBIT28_Msk                                      (0x1UL << CAN_F6BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F6BANK2_FBIT28                                          CAN_F6BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F6BANK2_FBIT29_Pos                                      (29U)
#define CAN_F6BANK2_FBIT29_Msk                                      (0x1UL << CAN_F6BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F6BANK2_FBIT29                                          CAN_F6BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F6BANK2_FBIT30_Pos                                      (30U)
#define CAN_F6BANK2_FBIT30_Msk                                      (0x1UL << CAN_F6BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F6BANK2_FBIT30                                          CAN_F6BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F6BANK2_FBIT31_Pos                                      (31U)
#define CAN_F6BANK2_FBIT31_Msk                                      (0x1UL << CAN_F6BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F6BANK2_FBIT31                                          CAN_F6BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F7BANK2 register  *******************/
#define CAN_F7BANK2_FBIT0_Pos                                       (0U)
#define CAN_F7BANK2_FBIT0_Msk                                       (0x1UL << CAN_F7BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F7BANK2_FBIT0                                           CAN_F7BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F7BANK2_FBIT1_Pos                                       (1U)
#define CAN_F7BANK2_FBIT1_Msk                                       (0x1UL << CAN_F7BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F7BANK2_FBIT1                                           CAN_F7BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F7BANK2_FBIT2_Pos                                       (2U)
#define CAN_F7BANK2_FBIT2_Msk                                       (0x1UL << CAN_F7BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F7BANK2_FBIT2                                           CAN_F7BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F7BANK2_FBIT3_Pos                                       (3U)
#define CAN_F7BANK2_FBIT3_Msk                                       (0x1UL << CAN_F7BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F7BANK2_FBIT3                                           CAN_F7BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F7BANK2_FBIT4_Pos                                       (4U)
#define CAN_F7BANK2_FBIT4_Msk                                       (0x1UL << CAN_F7BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F7BANK2_FBIT4                                           CAN_F7BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F7BANK2_FBIT5_Pos                                       (5U)
#define CAN_F7BANK2_FBIT5_Msk                                       (0x1UL << CAN_F7BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F7BANK2_FBIT5                                           CAN_F7BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F7BANK2_FBIT6_Pos                                       (6U)
#define CAN_F7BANK2_FBIT6_Msk                                       (0x1UL << CAN_F7BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F7BANK2_FBIT6                                           CAN_F7BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F7BANK2_FBIT7_Pos                                       (7U)
#define CAN_F7BANK2_FBIT7_Msk                                       (0x1UL << CAN_F7BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F7BANK2_FBIT7                                           CAN_F7BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F7BANK2_FBIT8_Pos                                       (8U)
#define CAN_F7BANK2_FBIT8_Msk                                       (0x1UL << CAN_F7BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F7BANK2_FBIT8                                           CAN_F7BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F7BANK2_FBIT9_Pos                                       (9U)
#define CAN_F7BANK2_FBIT9_Msk                                       (0x1UL << CAN_F7BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F7BANK2_FBIT9                                           CAN_F7BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F7BANK2_FBIT10_Pos                                      (10U)
#define CAN_F7BANK2_FBIT10_Msk                                      (0x1UL << CAN_F7BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F7BANK2_FBIT10                                          CAN_F7BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F7BANK2_FBIT11_Pos                                      (11U)
#define CAN_F7BANK2_FBIT11_Msk                                      (0x1UL << CAN_F7BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F7BANK2_FBIT11                                          CAN_F7BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F7BANK2_FBIT12_Pos                                      (12U)
#define CAN_F7BANK2_FBIT12_Msk                                      (0x1UL << CAN_F7BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F7BANK2_FBIT12                                          CAN_F7BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F7BANK2_FBIT13_Pos                                      (13U)
#define CAN_F7BANK2_FBIT13_Msk                                      (0x1UL << CAN_F7BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F7BANK2_FBIT13                                          CAN_F7BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F7BANK2_FBIT14_Pos                                      (14U)
#define CAN_F7BANK2_FBIT14_Msk                                      (0x1UL << CAN_F7BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F7BANK2_FBIT14                                          CAN_F7BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F7BANK2_FBIT15_Pos                                      (15U)
#define CAN_F7BANK2_FBIT15_Msk                                      (0x1UL << CAN_F7BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F7BANK2_FBIT15                                          CAN_F7BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F7BANK2_FBIT16_Pos                                      (16U)
#define CAN_F7BANK2_FBIT16_Msk                                      (0x1UL << CAN_F7BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F7BANK2_FBIT16                                          CAN_F7BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F7BANK2_FBIT17_Pos                                      (17U)
#define CAN_F7BANK2_FBIT17_Msk                                      (0x1UL << CAN_F7BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F7BANK2_FBIT17                                          CAN_F7BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F7BANK2_FBIT18_Pos                                      (18U)
#define CAN_F7BANK2_FBIT18_Msk                                      (0x1UL << CAN_F7BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F7BANK2_FBIT18                                          CAN_F7BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F7BANK2_FBIT19_Pos                                      (19U)
#define CAN_F7BANK2_FBIT19_Msk                                      (0x1UL << CAN_F7BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F7BANK2_FBIT19                                          CAN_F7BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F7BANK2_FBIT20_Pos                                      (20U)
#define CAN_F7BANK2_FBIT20_Msk                                      (0x1UL << CAN_F7BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F7BANK2_FBIT20                                          CAN_F7BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F7BANK2_FBIT21_Pos                                      (21U)
#define CAN_F7BANK2_FBIT21_Msk                                      (0x1UL << CAN_F7BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F7BANK2_FBIT21                                          CAN_F7BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F7BANK2_FBIT22_Pos                                      (22U)
#define CAN_F7BANK2_FBIT22_Msk                                      (0x1UL << CAN_F7BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F7BANK2_FBIT22                                          CAN_F7BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F7BANK2_FBIT23_Pos                                      (23U)
#define CAN_F7BANK2_FBIT23_Msk                                      (0x1UL << CAN_F7BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F7BANK2_FBIT23                                          CAN_F7BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F7BANK2_FBIT24_Pos                                      (24U)
#define CAN_F7BANK2_FBIT24_Msk                                      (0x1UL << CAN_F7BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F7BANK2_FBIT24                                          CAN_F7BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F7BANK2_FBIT25_Pos                                      (25U)
#define CAN_F7BANK2_FBIT25_Msk                                      (0x1UL << CAN_F7BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F7BANK2_FBIT25                                          CAN_F7BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F7BANK2_FBIT26_Pos                                      (26U)
#define CAN_F7BANK2_FBIT26_Msk                                      (0x1UL << CAN_F7BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F7BANK2_FBIT26                                          CAN_F7BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F7BANK2_FBIT27_Pos                                      (27U)
#define CAN_F7BANK2_FBIT27_Msk                                      (0x1UL << CAN_F7BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F7BANK2_FBIT27                                          CAN_F7BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F7BANK2_FBIT28_Pos                                      (28U)
#define CAN_F7BANK2_FBIT28_Msk                                      (0x1UL << CAN_F7BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F7BANK2_FBIT28                                          CAN_F7BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F7BANK2_FBIT29_Pos                                      (29U)
#define CAN_F7BANK2_FBIT29_Msk                                      (0x1UL << CAN_F7BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F7BANK2_FBIT29                                          CAN_F7BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F7BANK2_FBIT30_Pos                                      (30U)
#define CAN_F7BANK2_FBIT30_Msk                                      (0x1UL << CAN_F7BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F7BANK2_FBIT30                                          CAN_F7BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F7BANK2_FBIT31_Pos                                      (31U)
#define CAN_F7BANK2_FBIT31_Msk                                      (0x1UL << CAN_F7BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F7BANK2_FBIT31                                          CAN_F7BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F8BANK2 register  *******************/
#define CAN_F8BANK2_FBIT0_Pos                                       (0U)
#define CAN_F8BANK2_FBIT0_Msk                                       (0x1UL << CAN_F8BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F8BANK2_FBIT0                                           CAN_F8BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F8BANK2_FBIT1_Pos                                       (1U)
#define CAN_F8BANK2_FBIT1_Msk                                       (0x1UL << CAN_F8BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F8BANK2_FBIT1                                           CAN_F8BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F8BANK2_FBIT2_Pos                                       (2U)
#define CAN_F8BANK2_FBIT2_Msk                                       (0x1UL << CAN_F8BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F8BANK2_FBIT2                                           CAN_F8BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F8BANK2_FBIT3_Pos                                       (3U)
#define CAN_F8BANK2_FBIT3_Msk                                       (0x1UL << CAN_F8BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F8BANK2_FBIT3                                           CAN_F8BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F8BANK2_FBIT4_Pos                                       (4U)
#define CAN_F8BANK2_FBIT4_Msk                                       (0x1UL << CAN_F8BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F8BANK2_FBIT4                                           CAN_F8BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F8BANK2_FBIT5_Pos                                       (5U)
#define CAN_F8BANK2_FBIT5_Msk                                       (0x1UL << CAN_F8BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F8BANK2_FBIT5                                           CAN_F8BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F8BANK2_FBIT6_Pos                                       (6U)
#define CAN_F8BANK2_FBIT6_Msk                                       (0x1UL << CAN_F8BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F8BANK2_FBIT6                                           CAN_F8BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F8BANK2_FBIT7_Pos                                       (7U)
#define CAN_F8BANK2_FBIT7_Msk                                       (0x1UL << CAN_F8BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F8BANK2_FBIT7                                           CAN_F8BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F8BANK2_FBIT8_Pos                                       (8U)
#define CAN_F8BANK2_FBIT8_Msk                                       (0x1UL << CAN_F8BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F8BANK2_FBIT8                                           CAN_F8BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F8BANK2_FBIT9_Pos                                       (9U)
#define CAN_F8BANK2_FBIT9_Msk                                       (0x1UL << CAN_F8BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F8BANK2_FBIT9                                           CAN_F8BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F8BANK2_FBIT10_Pos                                      (10U)
#define CAN_F8BANK2_FBIT10_Msk                                      (0x1UL << CAN_F8BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F8BANK2_FBIT10                                          CAN_F8BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F8BANK2_FBIT11_Pos                                      (11U)
#define CAN_F8BANK2_FBIT11_Msk                                      (0x1UL << CAN_F8BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F8BANK2_FBIT11                                          CAN_F8BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F8BANK2_FBIT12_Pos                                      (12U)
#define CAN_F8BANK2_FBIT12_Msk                                      (0x1UL << CAN_F8BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F8BANK2_FBIT12                                          CAN_F8BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F8BANK2_FBIT13_Pos                                      (13U)
#define CAN_F8BANK2_FBIT13_Msk                                      (0x1UL << CAN_F8BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F8BANK2_FBIT13                                          CAN_F8BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F8BANK2_FBIT14_Pos                                      (14U)
#define CAN_F8BANK2_FBIT14_Msk                                      (0x1UL << CAN_F8BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F8BANK2_FBIT14                                          CAN_F8BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F8BANK2_FBIT15_Pos                                      (15U)
#define CAN_F8BANK2_FBIT15_Msk                                      (0x1UL << CAN_F8BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F8BANK2_FBIT15                                          CAN_F8BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F8BANK2_FBIT16_Pos                                      (16U)
#define CAN_F8BANK2_FBIT16_Msk                                      (0x1UL << CAN_F8BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F8BANK2_FBIT16                                          CAN_F8BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F8BANK2_FBIT17_Pos                                      (17U)
#define CAN_F8BANK2_FBIT17_Msk                                      (0x1UL << CAN_F8BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F8BANK2_FBIT17                                          CAN_F8BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F8BANK2_FBIT18_Pos                                      (18U)
#define CAN_F8BANK2_FBIT18_Msk                                      (0x1UL << CAN_F8BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F8BANK2_FBIT18                                          CAN_F8BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F8BANK2_FBIT19_Pos                                      (19U)
#define CAN_F8BANK2_FBIT19_Msk                                      (0x1UL << CAN_F8BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F8BANK2_FBIT19                                          CAN_F8BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F8BANK2_FBIT20_Pos                                      (20U)
#define CAN_F8BANK2_FBIT20_Msk                                      (0x1UL << CAN_F8BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F8BANK2_FBIT20                                          CAN_F8BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F8BANK2_FBIT21_Pos                                      (21U)
#define CAN_F8BANK2_FBIT21_Msk                                      (0x1UL << CAN_F8BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F8BANK2_FBIT21                                          CAN_F8BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F8BANK2_FBIT22_Pos                                      (22U)
#define CAN_F8BANK2_FBIT22_Msk                                      (0x1UL << CAN_F8BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F8BANK2_FBIT22                                          CAN_F8BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F8BANK2_FBIT23_Pos                                      (23U)
#define CAN_F8BANK2_FBIT23_Msk                                      (0x1UL << CAN_F8BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F8BANK2_FBIT23                                          CAN_F8BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F8BANK2_FBIT24_Pos                                      (24U)
#define CAN_F8BANK2_FBIT24_Msk                                      (0x1UL << CAN_F8BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F8BANK2_FBIT24                                          CAN_F8BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F8BANK2_FBIT25_Pos                                      (25U)
#define CAN_F8BANK2_FBIT25_Msk                                      (0x1UL << CAN_F8BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F8BANK2_FBIT25                                          CAN_F8BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F8BANK2_FBIT26_Pos                                      (26U)
#define CAN_F8BANK2_FBIT26_Msk                                      (0x1UL << CAN_F8BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F8BANK2_FBIT26                                          CAN_F8BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F8BANK2_FBIT27_Pos                                      (27U)
#define CAN_F8BANK2_FBIT27_Msk                                      (0x1UL << CAN_F8BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F8BANK2_FBIT27                                          CAN_F8BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F8BANK2_FBIT28_Pos                                      (28U)
#define CAN_F8BANK2_FBIT28_Msk                                      (0x1UL << CAN_F8BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F8BANK2_FBIT28                                          CAN_F8BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F8BANK2_FBIT29_Pos                                      (29U)
#define CAN_F8BANK2_FBIT29_Msk                                      (0x1UL << CAN_F8BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F8BANK2_FBIT29                                          CAN_F8BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F8BANK2_FBIT30_Pos                                      (30U)
#define CAN_F8BANK2_FBIT30_Msk                                      (0x1UL << CAN_F8BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F8BANK2_FBIT30                                          CAN_F8BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F8BANK2_FBIT31_Pos                                      (31U)
#define CAN_F8BANK2_FBIT31_Msk                                      (0x1UL << CAN_F8BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F8BANK2_FBIT31                                          CAN_F8BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F9BANK2 register  *******************/
#define CAN_F9BANK2_FBIT0_Pos                                       (0U)
#define CAN_F9BANK2_FBIT0_Msk                                       (0x1UL << CAN_F9BANK2_FBIT0_Pos)                               /*!< 0x00000001 */
#define CAN_F9BANK2_FBIT0                                           CAN_F9BANK2_FBIT0_Msk                                          /*!<Filter bit 0 */
#define CAN_F9BANK2_FBIT1_Pos                                       (1U)
#define CAN_F9BANK2_FBIT1_Msk                                       (0x1UL << CAN_F9BANK2_FBIT1_Pos)                               /*!< 0x00000002 */
#define CAN_F9BANK2_FBIT1                                           CAN_F9BANK2_FBIT1_Msk                                          /*!<Filter bit 1 */
#define CAN_F9BANK2_FBIT2_Pos                                       (2U)
#define CAN_F9BANK2_FBIT2_Msk                                       (0x1UL << CAN_F9BANK2_FBIT2_Pos)                               /*!< 0x00000004 */
#define CAN_F9BANK2_FBIT2                                           CAN_F9BANK2_FBIT2_Msk                                          /*!<Filter bit 2 */
#define CAN_F9BANK2_FBIT3_Pos                                       (3U)
#define CAN_F9BANK2_FBIT3_Msk                                       (0x1UL << CAN_F9BANK2_FBIT3_Pos)                               /*!< 0x00000008 */
#define CAN_F9BANK2_FBIT3                                           CAN_F9BANK2_FBIT3_Msk                                          /*!<Filter bit 3 */
#define CAN_F9BANK2_FBIT4_Pos                                       (4U)
#define CAN_F9BANK2_FBIT4_Msk                                       (0x1UL << CAN_F9BANK2_FBIT4_Pos)                               /*!< 0x00000010 */
#define CAN_F9BANK2_FBIT4                                           CAN_F9BANK2_FBIT4_Msk                                          /*!<Filter bit 4 */
#define CAN_F9BANK2_FBIT5_Pos                                       (5U)
#define CAN_F9BANK2_FBIT5_Msk                                       (0x1UL << CAN_F9BANK2_FBIT5_Pos)                               /*!< 0x00000020 */
#define CAN_F9BANK2_FBIT5                                           CAN_F9BANK2_FBIT5_Msk                                          /*!<Filter bit 5 */
#define CAN_F9BANK2_FBIT6_Pos                                       (6U)
#define CAN_F9BANK2_FBIT6_Msk                                       (0x1UL << CAN_F9BANK2_FBIT6_Pos)                               /*!< 0x00000040 */
#define CAN_F9BANK2_FBIT6                                           CAN_F9BANK2_FBIT6_Msk                                          /*!<Filter bit 6 */
#define CAN_F9BANK2_FBIT7_Pos                                       (7U)
#define CAN_F9BANK2_FBIT7_Msk                                       (0x1UL << CAN_F9BANK2_FBIT7_Pos)                               /*!< 0x00000080 */
#define CAN_F9BANK2_FBIT7                                           CAN_F9BANK2_FBIT7_Msk                                          /*!<Filter bit 7 */
#define CAN_F9BANK2_FBIT8_Pos                                       (8U)
#define CAN_F9BANK2_FBIT8_Msk                                       (0x1UL << CAN_F9BANK2_FBIT8_Pos)                               /*!< 0x00000100 */
#define CAN_F9BANK2_FBIT8                                           CAN_F9BANK2_FBIT8_Msk                                          /*!<Filter bit 8 */
#define CAN_F9BANK2_FBIT9_Pos                                       (9U)
#define CAN_F9BANK2_FBIT9_Msk                                       (0x1UL << CAN_F9BANK2_FBIT9_Pos)                               /*!< 0x00000200 */
#define CAN_F9BANK2_FBIT9                                           CAN_F9BANK2_FBIT9_Msk                                          /*!<Filter bit 9 */
#define CAN_F9BANK2_FBIT10_Pos                                      (10U)
#define CAN_F9BANK2_FBIT10_Msk                                      (0x1UL << CAN_F9BANK2_FBIT10_Pos)                              /*!< 0x00000400 */
#define CAN_F9BANK2_FBIT10                                          CAN_F9BANK2_FBIT10_Msk                                         /*!<Filter bit 10 */
#define CAN_F9BANK2_FBIT11_Pos                                      (11U)
#define CAN_F9BANK2_FBIT11_Msk                                      (0x1UL << CAN_F9BANK2_FBIT11_Pos)                              /*!< 0x00000800 */
#define CAN_F9BANK2_FBIT11                                          CAN_F9BANK2_FBIT11_Msk                                         /*!<Filter bit 11 */
#define CAN_F9BANK2_FBIT12_Pos                                      (12U)
#define CAN_F9BANK2_FBIT12_Msk                                      (0x1UL << CAN_F9BANK2_FBIT12_Pos)                              /*!< 0x00001000 */
#define CAN_F9BANK2_FBIT12                                          CAN_F9BANK2_FBIT12_Msk                                         /*!<Filter bit 12 */
#define CAN_F9BANK2_FBIT13_Pos                                      (13U)
#define CAN_F9BANK2_FBIT13_Msk                                      (0x1UL << CAN_F9BANK2_FBIT13_Pos)                              /*!< 0x00002000 */
#define CAN_F9BANK2_FBIT13                                          CAN_F9BANK2_FBIT13_Msk                                         /*!<Filter bit 13 */
#define CAN_F9BANK2_FBIT14_Pos                                      (14U)
#define CAN_F9BANK2_FBIT14_Msk                                      (0x1UL << CAN_F9BANK2_FBIT14_Pos)                              /*!< 0x00004000 */
#define CAN_F9BANK2_FBIT14                                          CAN_F9BANK2_FBIT14_Msk                                         /*!<Filter bit 14 */
#define CAN_F9BANK2_FBIT15_Pos                                      (15U)
#define CAN_F9BANK2_FBIT15_Msk                                      (0x1UL << CAN_F9BANK2_FBIT15_Pos)                              /*!< 0x00008000 */
#define CAN_F9BANK2_FBIT15                                          CAN_F9BANK2_FBIT15_Msk                                         /*!<Filter bit 15 */
#define CAN_F9BANK2_FBIT16_Pos                                      (16U)
#define CAN_F9BANK2_FBIT16_Msk                                      (0x1UL << CAN_F9BANK2_FBIT16_Pos)                              /*!< 0x00010000 */
#define CAN_F9BANK2_FBIT16                                          CAN_F9BANK2_FBIT16_Msk                                         /*!<Filter bit 16 */
#define CAN_F9BANK2_FBIT17_Pos                                      (17U)
#define CAN_F9BANK2_FBIT17_Msk                                      (0x1UL << CAN_F9BANK2_FBIT17_Pos)                              /*!< 0x00020000 */
#define CAN_F9BANK2_FBIT17                                          CAN_F9BANK2_FBIT17_Msk                                         /*!<Filter bit 17 */
#define CAN_F9BANK2_FBIT18_Pos                                      (18U)
#define CAN_F9BANK2_FBIT18_Msk                                      (0x1UL << CAN_F9BANK2_FBIT18_Pos)                              /*!< 0x00040000 */
#define CAN_F9BANK2_FBIT18                                          CAN_F9BANK2_FBIT18_Msk                                         /*!<Filter bit 18 */
#define CAN_F9BANK2_FBIT19_Pos                                      (19U)
#define CAN_F9BANK2_FBIT19_Msk                                      (0x1UL << CAN_F9BANK2_FBIT19_Pos)                              /*!< 0x00080000 */
#define CAN_F9BANK2_FBIT19                                          CAN_F9BANK2_FBIT19_Msk                                         /*!<Filter bit 19 */
#define CAN_F9BANK2_FBIT20_Pos                                      (20U)
#define CAN_F9BANK2_FBIT20_Msk                                      (0x1UL << CAN_F9BANK2_FBIT20_Pos)                              /*!< 0x00100000 */
#define CAN_F9BANK2_FBIT20                                          CAN_F9BANK2_FBIT20_Msk                                         /*!<Filter bit 20 */
#define CAN_F9BANK2_FBIT21_Pos                                      (21U)
#define CAN_F9BANK2_FBIT21_Msk                                      (0x1UL << CAN_F9BANK2_FBIT21_Pos)                              /*!< 0x00200000 */
#define CAN_F9BANK2_FBIT21                                          CAN_F9BANK2_FBIT21_Msk                                         /*!<Filter bit 21 */
#define CAN_F9BANK2_FBIT22_Pos                                      (22U)
#define CAN_F9BANK2_FBIT22_Msk                                      (0x1UL << CAN_F9BANK2_FBIT22_Pos)                              /*!< 0x00400000 */
#define CAN_F9BANK2_FBIT22                                          CAN_F9BANK2_FBIT22_Msk                                         /*!<Filter bit 22 */
#define CAN_F9BANK2_FBIT23_Pos                                      (23U)
#define CAN_F9BANK2_FBIT23_Msk                                      (0x1UL << CAN_F9BANK2_FBIT23_Pos)                              /*!< 0x00800000 */
#define CAN_F9BANK2_FBIT23                                          CAN_F9BANK2_FBIT23_Msk                                         /*!<Filter bit 23 */
#define CAN_F9BANK2_FBIT24_Pos                                      (24U)
#define CAN_F9BANK2_FBIT24_Msk                                      (0x1UL << CAN_F9BANK2_FBIT24_Pos)                              /*!< 0x01000000 */
#define CAN_F9BANK2_FBIT24                                          CAN_F9BANK2_FBIT24_Msk                                         /*!<Filter bit 24 */
#define CAN_F9BANK2_FBIT25_Pos                                      (25U)
#define CAN_F9BANK2_FBIT25_Msk                                      (0x1UL << CAN_F9BANK2_FBIT25_Pos)                              /*!< 0x02000000 */
#define CAN_F9BANK2_FBIT25                                          CAN_F9BANK2_FBIT25_Msk                                         /*!<Filter bit 25 */
#define CAN_F9BANK2_FBIT26_Pos                                      (26U)
#define CAN_F9BANK2_FBIT26_Msk                                      (0x1UL << CAN_F9BANK2_FBIT26_Pos)                              /*!< 0x04000000 */
#define CAN_F9BANK2_FBIT26                                          CAN_F9BANK2_FBIT26_Msk                                         /*!<Filter bit 26 */
#define CAN_F9BANK2_FBIT27_Pos                                      (27U)
#define CAN_F9BANK2_FBIT27_Msk                                      (0x1UL << CAN_F9BANK2_FBIT27_Pos)                              /*!< 0x08000000 */
#define CAN_F9BANK2_FBIT27                                          CAN_F9BANK2_FBIT27_Msk                                         /*!<Filter bit 27 */
#define CAN_F9BANK2_FBIT28_Pos                                      (28U)
#define CAN_F9BANK2_FBIT28_Msk                                      (0x1UL << CAN_F9BANK2_FBIT28_Pos)                              /*!< 0x10000000 */
#define CAN_F9BANK2_FBIT28                                          CAN_F9BANK2_FBIT28_Msk                                         /*!<Filter bit 28 */
#define CAN_F9BANK2_FBIT29_Pos                                      (29U)
#define CAN_F9BANK2_FBIT29_Msk                                      (0x1UL << CAN_F9BANK2_FBIT29_Pos)                              /*!< 0x20000000 */
#define CAN_F9BANK2_FBIT29                                          CAN_F9BANK2_FBIT29_Msk                                         /*!<Filter bit 29 */
#define CAN_F9BANK2_FBIT30_Pos                                      (30U)
#define CAN_F9BANK2_FBIT30_Msk                                      (0x1UL << CAN_F9BANK2_FBIT30_Pos)                              /*!< 0x40000000 */
#define CAN_F9BANK2_FBIT30                                          CAN_F9BANK2_FBIT30_Msk                                         /*!<Filter bit 30 */
#define CAN_F9BANK2_FBIT31_Pos                                      (31U)
#define CAN_F9BANK2_FBIT31_Msk                                      (0x1UL << CAN_F9BANK2_FBIT31_Pos)                              /*!< 0x80000000 */
#define CAN_F9BANK2_FBIT31                                          CAN_F9BANK2_FBIT31_Msk                                         /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F10BANK2 register  ******************/
#define CAN_F10BANK2_FBIT0_Pos                                      (0U)
#define CAN_F10BANK2_FBIT0_Msk                                      (0x1UL << CAN_F10BANK2_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F10BANK2_FBIT0                                          CAN_F10BANK2_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F10BANK2_FBIT1_Pos                                      (1U)
#define CAN_F10BANK2_FBIT1_Msk                                      (0x1UL << CAN_F10BANK2_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F10BANK2_FBIT1                                          CAN_F10BANK2_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F10BANK2_FBIT2_Pos                                      (2U)
#define CAN_F10BANK2_FBIT2_Msk                                      (0x1UL << CAN_F10BANK2_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F10BANK2_FBIT2                                          CAN_F10BANK2_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F10BANK2_FBIT3_Pos                                      (3U)
#define CAN_F10BANK2_FBIT3_Msk                                      (0x1UL << CAN_F10BANK2_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F10BANK2_FBIT3                                          CAN_F10BANK2_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F10BANK2_FBIT4_Pos                                      (4U)
#define CAN_F10BANK2_FBIT4_Msk                                      (0x1UL << CAN_F10BANK2_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F10BANK2_FBIT4                                          CAN_F10BANK2_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F10BANK2_FBIT5_Pos                                      (5U)
#define CAN_F10BANK2_FBIT5_Msk                                      (0x1UL << CAN_F10BANK2_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F10BANK2_FBIT5                                          CAN_F10BANK2_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F10BANK2_FBIT6_Pos                                      (6U)
#define CAN_F10BANK2_FBIT6_Msk                                      (0x1UL << CAN_F10BANK2_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F10BANK2_FBIT6                                          CAN_F10BANK2_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F10BANK2_FBIT7_Pos                                      (7U)
#define CAN_F10BANK2_FBIT7_Msk                                      (0x1UL << CAN_F10BANK2_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F10BANK2_FBIT7                                          CAN_F10BANK2_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F10BANK2_FBIT8_Pos                                      (8U)
#define CAN_F10BANK2_FBIT8_Msk                                      (0x1UL << CAN_F10BANK2_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F10BANK2_FBIT8                                          CAN_F10BANK2_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F10BANK2_FBIT9_Pos                                      (9U)
#define CAN_F10BANK2_FBIT9_Msk                                      (0x1UL << CAN_F10BANK2_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F10BANK2_FBIT9                                          CAN_F10BANK2_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F10BANK2_FBIT10_Pos                                     (10U)
#define CAN_F10BANK2_FBIT10_Msk                                     (0x1UL << CAN_F10BANK2_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F10BANK2_FBIT10                                         CAN_F10BANK2_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F10BANK2_FBIT11_Pos                                     (11U)
#define CAN_F10BANK2_FBIT11_Msk                                     (0x1UL << CAN_F10BANK2_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F10BANK2_FBIT11                                         CAN_F10BANK2_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F10BANK2_FBIT12_Pos                                     (12U)
#define CAN_F10BANK2_FBIT12_Msk                                     (0x1UL << CAN_F10BANK2_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F10BANK2_FBIT12                                         CAN_F10BANK2_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F10BANK2_FBIT13_Pos                                     (13U)
#define CAN_F10BANK2_FBIT13_Msk                                     (0x1UL << CAN_F10BANK2_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F10BANK2_FBIT13                                         CAN_F10BANK2_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F10BANK2_FBIT14_Pos                                     (14U)
#define CAN_F10BANK2_FBIT14_Msk                                     (0x1UL << CAN_F10BANK2_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F10BANK2_FBIT14                                         CAN_F10BANK2_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F10BANK2_FBIT15_Pos                                     (15U)
#define CAN_F10BANK2_FBIT15_Msk                                     (0x1UL << CAN_F10BANK2_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F10BANK2_FBIT15                                         CAN_F10BANK2_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F10BANK2_FBIT16_Pos                                     (16U)
#define CAN_F10BANK2_FBIT16_Msk                                     (0x1UL << CAN_F10BANK2_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F10BANK2_FBIT16                                         CAN_F10BANK2_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F10BANK2_FBIT17_Pos                                     (17U)
#define CAN_F10BANK2_FBIT17_Msk                                     (0x1UL << CAN_F10BANK2_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F10BANK2_FBIT17                                         CAN_F10BANK2_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F10BANK2_FBIT18_Pos                                     (18U)
#define CAN_F10BANK2_FBIT18_Msk                                     (0x1UL << CAN_F10BANK2_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F10BANK2_FBIT18                                         CAN_F10BANK2_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F10BANK2_FBIT19_Pos                                     (19U)
#define CAN_F10BANK2_FBIT19_Msk                                     (0x1UL << CAN_F10BANK2_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F10BANK2_FBIT19                                         CAN_F10BANK2_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F10BANK2_FBIT20_Pos                                     (20U)
#define CAN_F10BANK2_FBIT20_Msk                                     (0x1UL << CAN_F10BANK2_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F10BANK2_FBIT20                                         CAN_F10BANK2_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F10BANK2_FBIT21_Pos                                     (21U)
#define CAN_F10BANK2_FBIT21_Msk                                     (0x1UL << CAN_F10BANK2_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F10BANK2_FBIT21                                         CAN_F10BANK2_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F10BANK2_FBIT22_Pos                                     (22U)
#define CAN_F10BANK2_FBIT22_Msk                                     (0x1UL << CAN_F10BANK2_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F10BANK2_FBIT22                                         CAN_F10BANK2_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F10BANK2_FBIT23_Pos                                     (23U)
#define CAN_F10BANK2_FBIT23_Msk                                     (0x1UL << CAN_F10BANK2_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F10BANK2_FBIT23                                         CAN_F10BANK2_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F10BANK2_FBIT24_Pos                                     (24U)
#define CAN_F10BANK2_FBIT24_Msk                                     (0x1UL << CAN_F10BANK2_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F10BANK2_FBIT24                                         CAN_F10BANK2_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F10BANK2_FBIT25_Pos                                     (25U)
#define CAN_F10BANK2_FBIT25_Msk                                     (0x1UL << CAN_F10BANK2_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F10BANK2_FBIT25                                         CAN_F10BANK2_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F10BANK2_FBIT26_Pos                                     (26U)
#define CAN_F10BANK2_FBIT26_Msk                                     (0x1UL << CAN_F10BANK2_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F10BANK2_FBIT26                                         CAN_F10BANK2_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F10BANK2_FBIT27_Pos                                     (27U)
#define CAN_F10BANK2_FBIT27_Msk                                     (0x1UL << CAN_F10BANK2_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F10BANK2_FBIT27                                         CAN_F10BANK2_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F10BANK2_FBIT28_Pos                                     (28U)
#define CAN_F10BANK2_FBIT28_Msk                                     (0x1UL << CAN_F10BANK2_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F10BANK2_FBIT28                                         CAN_F10BANK2_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F10BANK2_FBIT29_Pos                                     (29U)
#define CAN_F10BANK2_FBIT29_Msk                                     (0x1UL << CAN_F10BANK2_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F10BANK2_FBIT29                                         CAN_F10BANK2_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F10BANK2_FBIT30_Pos                                     (30U)
#define CAN_F10BANK2_FBIT30_Msk                                     (0x1UL << CAN_F10BANK2_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F10BANK2_FBIT30                                         CAN_F10BANK2_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F10BANK2_FBIT31_Pos                                     (31U)
#define CAN_F10BANK2_FBIT31_Msk                                     (0x1UL << CAN_F10BANK2_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F10BANK2_FBIT31                                         CAN_F10BANK2_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F11BANK2 register  ******************/
#define CAN_F11BANK2_FBIT0_Pos                                      (0U)
#define CAN_F11BANK2_FBIT0_Msk                                      (0x1UL << CAN_F11BANK2_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F11BANK2_FBIT0                                          CAN_F11BANK2_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F11BANK2_FBIT1_Pos                                      (1U)
#define CAN_F11BANK2_FBIT1_Msk                                      (0x1UL << CAN_F11BANK2_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F11BANK2_FBIT1                                          CAN_F11BANK2_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F11BANK2_FBIT2_Pos                                      (2U)
#define CAN_F11BANK2_FBIT2_Msk                                      (0x1UL << CAN_F11BANK2_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F11BANK2_FBIT2                                          CAN_F11BANK2_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F11BANK2_FBIT3_Pos                                      (3U)
#define CAN_F11BANK2_FBIT3_Msk                                      (0x1UL << CAN_F11BANK2_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F11BANK2_FBIT3                                          CAN_F11BANK2_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F11BANK2_FBIT4_Pos                                      (4U)
#define CAN_F11BANK2_FBIT4_Msk                                      (0x1UL << CAN_F11BANK2_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F11BANK2_FBIT4                                          CAN_F11BANK2_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F11BANK2_FBIT5_Pos                                      (5U)
#define CAN_F11BANK2_FBIT5_Msk                                      (0x1UL << CAN_F11BANK2_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F11BANK2_FBIT5                                          CAN_F11BANK2_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F11BANK2_FBIT6_Pos                                      (6U)
#define CAN_F11BANK2_FBIT6_Msk                                      (0x1UL << CAN_F11BANK2_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F11BANK2_FBIT6                                          CAN_F11BANK2_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F11BANK2_FBIT7_Pos                                      (7U)
#define CAN_F11BANK2_FBIT7_Msk                                      (0x1UL << CAN_F11BANK2_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F11BANK2_FBIT7                                          CAN_F11BANK2_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F11BANK2_FBIT8_Pos                                      (8U)
#define CAN_F11BANK2_FBIT8_Msk                                      (0x1UL << CAN_F11BANK2_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F11BANK2_FBIT8                                          CAN_F11BANK2_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F11BANK2_FBIT9_Pos                                      (9U)
#define CAN_F11BANK2_FBIT9_Msk                                      (0x1UL << CAN_F11BANK2_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F11BANK2_FBIT9                                          CAN_F11BANK2_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F11BANK2_FBIT10_Pos                                     (10U)
#define CAN_F11BANK2_FBIT10_Msk                                     (0x1UL << CAN_F11BANK2_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F11BANK2_FBIT10                                         CAN_F11BANK2_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F11BANK2_FBIT11_Pos                                     (11U)
#define CAN_F11BANK2_FBIT11_Msk                                     (0x1UL << CAN_F11BANK2_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F11BANK2_FBIT11                                         CAN_F11BANK2_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F11BANK2_FBIT12_Pos                                     (12U)
#define CAN_F11BANK2_FBIT12_Msk                                     (0x1UL << CAN_F11BANK2_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F11BANK2_FBIT12                                         CAN_F11BANK2_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F11BANK2_FBIT13_Pos                                     (13U)
#define CAN_F11BANK2_FBIT13_Msk                                     (0x1UL << CAN_F11BANK2_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F11BANK2_FBIT13                                         CAN_F11BANK2_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F11BANK2_FBIT14_Pos                                     (14U)
#define CAN_F11BANK2_FBIT14_Msk                                     (0x1UL << CAN_F11BANK2_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F11BANK2_FBIT14                                         CAN_F11BANK2_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F11BANK2_FBIT15_Pos                                     (15U)
#define CAN_F11BANK2_FBIT15_Msk                                     (0x1UL << CAN_F11BANK2_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F11BANK2_FBIT15                                         CAN_F11BANK2_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F11BANK2_FBIT16_Pos                                     (16U)
#define CAN_F11BANK2_FBIT16_Msk                                     (0x1UL << CAN_F11BANK2_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F11BANK2_FBIT16                                         CAN_F11BANK2_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F11BANK2_FBIT17_Pos                                     (17U)
#define CAN_F11BANK2_FBIT17_Msk                                     (0x1UL << CAN_F11BANK2_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F11BANK2_FBIT17                                         CAN_F11BANK2_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F11BANK2_FBIT18_Pos                                     (18U)
#define CAN_F11BANK2_FBIT18_Msk                                     (0x1UL << CAN_F11BANK2_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F11BANK2_FBIT18                                         CAN_F11BANK2_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F11BANK2_FBIT19_Pos                                     (19U)
#define CAN_F11BANK2_FBIT19_Msk                                     (0x1UL << CAN_F11BANK2_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F11BANK2_FBIT19                                         CAN_F11BANK2_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F11BANK2_FBIT20_Pos                                     (20U)
#define CAN_F11BANK2_FBIT20_Msk                                     (0x1UL << CAN_F11BANK2_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F11BANK2_FBIT20                                         CAN_F11BANK2_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F11BANK2_FBIT21_Pos                                     (21U)
#define CAN_F11BANK2_FBIT21_Msk                                     (0x1UL << CAN_F11BANK2_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F11BANK2_FBIT21                                         CAN_F11BANK2_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F11BANK2_FBIT22_Pos                                     (22U)
#define CAN_F11BANK2_FBIT22_Msk                                     (0x1UL << CAN_F11BANK2_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F11BANK2_FBIT22                                         CAN_F11BANK2_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F11BANK2_FBIT23_Pos                                     (23U)
#define CAN_F11BANK2_FBIT23_Msk                                     (0x1UL << CAN_F11BANK2_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F11BANK2_FBIT23                                         CAN_F11BANK2_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F11BANK2_FBIT24_Pos                                     (24U)
#define CAN_F11BANK2_FBIT24_Msk                                     (0x1UL << CAN_F11BANK2_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F11BANK2_FBIT24                                         CAN_F11BANK2_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F11BANK2_FBIT25_Pos                                     (25U)
#define CAN_F11BANK2_FBIT25_Msk                                     (0x1UL << CAN_F11BANK2_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F11BANK2_FBIT25                                         CAN_F11BANK2_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F11BANK2_FBIT26_Pos                                     (26U)
#define CAN_F11BANK2_FBIT26_Msk                                     (0x1UL << CAN_F11BANK2_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F11BANK2_FBIT26                                         CAN_F11BANK2_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F11BANK2_FBIT27_Pos                                     (27U)
#define CAN_F11BANK2_FBIT27_Msk                                     (0x1UL << CAN_F11BANK2_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F11BANK2_FBIT27                                         CAN_F11BANK2_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F11BANK2_FBIT28_Pos                                     (28U)
#define CAN_F11BANK2_FBIT28_Msk                                     (0x1UL << CAN_F11BANK2_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F11BANK2_FBIT28                                         CAN_F11BANK2_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F11BANK2_FBIT29_Pos                                     (29U)
#define CAN_F11BANK2_FBIT29_Msk                                     (0x1UL << CAN_F11BANK2_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F11BANK2_FBIT29                                         CAN_F11BANK2_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F11BANK2_FBIT30_Pos                                     (30U)
#define CAN_F11BANK2_FBIT30_Msk                                     (0x1UL << CAN_F11BANK2_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F11BANK2_FBIT30                                         CAN_F11BANK2_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F11BANK2_FBIT31_Pos                                     (31U)
#define CAN_F11BANK2_FBIT31_Msk                                     (0x1UL << CAN_F11BANK2_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F11BANK2_FBIT31                                         CAN_F11BANK2_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F12BANK2 register  ******************/
#define CAN_F12BANK2_FBIT0_Pos                                      (0U)
#define CAN_F12BANK2_FBIT0_Msk                                      (0x1UL << CAN_F12BANK2_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F12BANK2_FBIT0                                          CAN_F12BANK2_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F12BANK2_FBIT1_Pos                                      (1U)
#define CAN_F12BANK2_FBIT1_Msk                                      (0x1UL << CAN_F12BANK2_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F12BANK2_FBIT1                                          CAN_F12BANK2_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F12BANK2_FBIT2_Pos                                      (2U)
#define CAN_F12BANK2_FBIT2_Msk                                      (0x1UL << CAN_F12BANK2_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F12BANK2_FBIT2                                          CAN_F12BANK2_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F12BANK2_FBIT3_Pos                                      (3U)
#define CAN_F12BANK2_FBIT3_Msk                                      (0x1UL << CAN_F12BANK2_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F12BANK2_FBIT3                                          CAN_F12BANK2_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F12BANK2_FBIT4_Pos                                      (4U)
#define CAN_F12BANK2_FBIT4_Msk                                      (0x1UL << CAN_F12BANK2_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F12BANK2_FBIT4                                          CAN_F12BANK2_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F12BANK2_FBIT5_Pos                                      (5U)
#define CAN_F12BANK2_FBIT5_Msk                                      (0x1UL << CAN_F12BANK2_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F12BANK2_FBIT5                                          CAN_F12BANK2_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F12BANK2_FBIT6_Pos                                      (6U)
#define CAN_F12BANK2_FBIT6_Msk                                      (0x1UL << CAN_F12BANK2_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F12BANK2_FBIT6                                          CAN_F12BANK2_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F12BANK2_FBIT7_Pos                                      (7U)
#define CAN_F12BANK2_FBIT7_Msk                                      (0x1UL << CAN_F12BANK2_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F12BANK2_FBIT7                                          CAN_F12BANK2_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F12BANK2_FBIT8_Pos                                      (8U)
#define CAN_F12BANK2_FBIT8_Msk                                      (0x1UL << CAN_F12BANK2_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F12BANK2_FBIT8                                          CAN_F12BANK2_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F12BANK2_FBIT9_Pos                                      (9U)
#define CAN_F12BANK2_FBIT9_Msk                                      (0x1UL << CAN_F12BANK2_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F12BANK2_FBIT9                                          CAN_F12BANK2_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F12BANK2_FBIT10_Pos                                     (10U)
#define CAN_F12BANK2_FBIT10_Msk                                     (0x1UL << CAN_F12BANK2_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F12BANK2_FBIT10                                         CAN_F12BANK2_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F12BANK2_FBIT11_Pos                                     (11U)
#define CAN_F12BANK2_FBIT11_Msk                                     (0x1UL << CAN_F12BANK2_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F12BANK2_FBIT11                                         CAN_F12BANK2_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F12BANK2_FBIT12_Pos                                     (12U)
#define CAN_F12BANK2_FBIT12_Msk                                     (0x1UL << CAN_F12BANK2_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F12BANK2_FBIT12                                         CAN_F12BANK2_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F12BANK2_FBIT13_Pos                                     (13U)
#define CAN_F12BANK2_FBIT13_Msk                                     (0x1UL << CAN_F12BANK2_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F12BANK2_FBIT13                                         CAN_F12BANK2_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F12BANK2_FBIT14_Pos                                     (14U)
#define CAN_F12BANK2_FBIT14_Msk                                     (0x1UL << CAN_F12BANK2_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F12BANK2_FBIT14                                         CAN_F12BANK2_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F12BANK2_FBIT15_Pos                                     (15U)
#define CAN_F12BANK2_FBIT15_Msk                                     (0x1UL << CAN_F12BANK2_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F12BANK2_FBIT15                                         CAN_F12BANK2_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F12BANK2_FBIT16_Pos                                     (16U)
#define CAN_F12BANK2_FBIT16_Msk                                     (0x1UL << CAN_F12BANK2_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F12BANK2_FBIT16                                         CAN_F12BANK2_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F12BANK2_FBIT17_Pos                                     (17U)
#define CAN_F12BANK2_FBIT17_Msk                                     (0x1UL << CAN_F12BANK2_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F12BANK2_FBIT17                                         CAN_F12BANK2_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F12BANK2_FBIT18_Pos                                     (18U)
#define CAN_F12BANK2_FBIT18_Msk                                     (0x1UL << CAN_F12BANK2_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F12BANK2_FBIT18                                         CAN_F12BANK2_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F12BANK2_FBIT19_Pos                                     (19U)
#define CAN_F12BANK2_FBIT19_Msk                                     (0x1UL << CAN_F12BANK2_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F12BANK2_FBIT19                                         CAN_F12BANK2_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F12BANK2_FBIT20_Pos                                     (20U)
#define CAN_F12BANK2_FBIT20_Msk                                     (0x1UL << CAN_F12BANK2_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F12BANK2_FBIT20                                         CAN_F12BANK2_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F12BANK2_FBIT21_Pos                                     (21U)
#define CAN_F12BANK2_FBIT21_Msk                                     (0x1UL << CAN_F12BANK2_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F12BANK2_FBIT21                                         CAN_F12BANK2_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F12BANK2_FBIT22_Pos                                     (22U)
#define CAN_F12BANK2_FBIT22_Msk                                     (0x1UL << CAN_F12BANK2_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F12BANK2_FBIT22                                         CAN_F12BANK2_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F12BANK2_FBIT23_Pos                                     (23U)
#define CAN_F12BANK2_FBIT23_Msk                                     (0x1UL << CAN_F12BANK2_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F12BANK2_FBIT23                                         CAN_F12BANK2_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F12BANK2_FBIT24_Pos                                     (24U)
#define CAN_F12BANK2_FBIT24_Msk                                     (0x1UL << CAN_F12BANK2_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F12BANK2_FBIT24                                         CAN_F12BANK2_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F12BANK2_FBIT25_Pos                                     (25U)
#define CAN_F12BANK2_FBIT25_Msk                                     (0x1UL << CAN_F12BANK2_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F12BANK2_FBIT25                                         CAN_F12BANK2_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F12BANK2_FBIT26_Pos                                     (26U)
#define CAN_F12BANK2_FBIT26_Msk                                     (0x1UL << CAN_F12BANK2_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F12BANK2_FBIT26                                         CAN_F12BANK2_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F12BANK2_FBIT27_Pos                                     (27U)
#define CAN_F12BANK2_FBIT27_Msk                                     (0x1UL << CAN_F12BANK2_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F12BANK2_FBIT27                                         CAN_F12BANK2_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F12BANK2_FBIT28_Pos                                     (28U)
#define CAN_F12BANK2_FBIT28_Msk                                     (0x1UL << CAN_F12BANK2_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F12BANK2_FBIT28                                         CAN_F12BANK2_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F12BANK2_FBIT29_Pos                                     (29U)
#define CAN_F12BANK2_FBIT29_Msk                                     (0x1UL << CAN_F12BANK2_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F12BANK2_FBIT29                                         CAN_F12BANK2_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F12BANK2_FBIT30_Pos                                     (30U)
#define CAN_F12BANK2_FBIT30_Msk                                     (0x1UL << CAN_F12BANK2_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F12BANK2_FBIT30                                         CAN_F12BANK2_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F12BANK2_FBIT31_Pos                                     (31U)
#define CAN_F12BANK2_FBIT31_Msk                                     (0x1UL << CAN_F12BANK2_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F12BANK2_FBIT31                                         CAN_F12BANK2_FBIT31_Msk                                        /*!<Filter bit 31 */

/*******************  Bit definition for CAN_F13BANK2 register  ******************/
#define CAN_F13BANK2_FBIT0_Pos                                      (0U)
#define CAN_F13BANK2_FBIT0_Msk                                      (0x1UL << CAN_F13BANK2_FBIT0_Pos)                              /*!< 0x00000001 */
#define CAN_F13BANK2_FBIT0                                          CAN_F13BANK2_FBIT0_Msk                                         /*!<Filter bit 0 */
#define CAN_F13BANK2_FBIT1_Pos                                      (1U)
#define CAN_F13BANK2_FBIT1_Msk                                      (0x1UL << CAN_F13BANK2_FBIT1_Pos)                              /*!< 0x00000002 */
#define CAN_F13BANK2_FBIT1                                          CAN_F13BANK2_FBIT1_Msk                                         /*!<Filter bit 1 */
#define CAN_F13BANK2_FBIT2_Pos                                      (2U)
#define CAN_F13BANK2_FBIT2_Msk                                      (0x1UL << CAN_F13BANK2_FBIT2_Pos)                              /*!< 0x00000004 */
#define CAN_F13BANK2_FBIT2                                          CAN_F13BANK2_FBIT2_Msk                                         /*!<Filter bit 2 */
#define CAN_F13BANK2_FBIT3_Pos                                      (3U)
#define CAN_F13BANK2_FBIT3_Msk                                      (0x1UL << CAN_F13BANK2_FBIT3_Pos)                              /*!< 0x00000008 */
#define CAN_F13BANK2_FBIT3                                          CAN_F13BANK2_FBIT3_Msk                                         /*!<Filter bit 3 */
#define CAN_F13BANK2_FBIT4_Pos                                      (4U)
#define CAN_F13BANK2_FBIT4_Msk                                      (0x1UL << CAN_F13BANK2_FBIT4_Pos)                              /*!< 0x00000010 */
#define CAN_F13BANK2_FBIT4                                          CAN_F13BANK2_FBIT4_Msk                                         /*!<Filter bit 4 */
#define CAN_F13BANK2_FBIT5_Pos                                      (5U)
#define CAN_F13BANK2_FBIT5_Msk                                      (0x1UL << CAN_F13BANK2_FBIT5_Pos)                              /*!< 0x00000020 */
#define CAN_F13BANK2_FBIT5                                          CAN_F13BANK2_FBIT5_Msk                                         /*!<Filter bit 5 */
#define CAN_F13BANK2_FBIT6_Pos                                      (6U)
#define CAN_F13BANK2_FBIT6_Msk                                      (0x1UL << CAN_F13BANK2_FBIT6_Pos)                              /*!< 0x00000040 */
#define CAN_F13BANK2_FBIT6                                          CAN_F13BANK2_FBIT6_Msk                                         /*!<Filter bit 6 */
#define CAN_F13BANK2_FBIT7_Pos                                      (7U)
#define CAN_F13BANK2_FBIT7_Msk                                      (0x1UL << CAN_F13BANK2_FBIT7_Pos)                              /*!< 0x00000080 */
#define CAN_F13BANK2_FBIT7                                          CAN_F13BANK2_FBIT7_Msk                                         /*!<Filter bit 7 */
#define CAN_F13BANK2_FBIT8_Pos                                      (8U)
#define CAN_F13BANK2_FBIT8_Msk                                      (0x1UL << CAN_F13BANK2_FBIT8_Pos)                              /*!< 0x00000100 */
#define CAN_F13BANK2_FBIT8                                          CAN_F13BANK2_FBIT8_Msk                                         /*!<Filter bit 8 */
#define CAN_F13BANK2_FBIT9_Pos                                      (9U)
#define CAN_F13BANK2_FBIT9_Msk                                      (0x1UL << CAN_F13BANK2_FBIT9_Pos)                              /*!< 0x00000200 */
#define CAN_F13BANK2_FBIT9                                          CAN_F13BANK2_FBIT9_Msk                                         /*!<Filter bit 9 */
#define CAN_F13BANK2_FBIT10_Pos                                     (10U)
#define CAN_F13BANK2_FBIT10_Msk                                     (0x1UL << CAN_F13BANK2_FBIT10_Pos)                             /*!< 0x00000400 */
#define CAN_F13BANK2_FBIT10                                         CAN_F13BANK2_FBIT10_Msk                                        /*!<Filter bit 10 */
#define CAN_F13BANK2_FBIT11_Pos                                     (11U)
#define CAN_F13BANK2_FBIT11_Msk                                     (0x1UL << CAN_F13BANK2_FBIT11_Pos)                             /*!< 0x00000800 */
#define CAN_F13BANK2_FBIT11                                         CAN_F13BANK2_FBIT11_Msk                                        /*!<Filter bit 11 */
#define CAN_F13BANK2_FBIT12_Pos                                     (12U)
#define CAN_F13BANK2_FBIT12_Msk                                     (0x1UL << CAN_F13BANK2_FBIT12_Pos)                             /*!< 0x00001000 */
#define CAN_F13BANK2_FBIT12                                         CAN_F13BANK2_FBIT12_Msk                                        /*!<Filter bit 12 */
#define CAN_F13BANK2_FBIT13_Pos                                     (13U)
#define CAN_F13BANK2_FBIT13_Msk                                     (0x1UL << CAN_F13BANK2_FBIT13_Pos)                             /*!< 0x00002000 */
#define CAN_F13BANK2_FBIT13                                         CAN_F13BANK2_FBIT13_Msk                                        /*!<Filter bit 13 */
#define CAN_F13BANK2_FBIT14_Pos                                     (14U)
#define CAN_F13BANK2_FBIT14_Msk                                     (0x1UL << CAN_F13BANK2_FBIT14_Pos)                             /*!< 0x00004000 */
#define CAN_F13BANK2_FBIT14                                         CAN_F13BANK2_FBIT14_Msk                                        /*!<Filter bit 14 */
#define CAN_F13BANK2_FBIT15_Pos                                     (15U)
#define CAN_F13BANK2_FBIT15_Msk                                     (0x1UL << CAN_F13BANK2_FBIT15_Pos)                             /*!< 0x00008000 */
#define CAN_F13BANK2_FBIT15                                         CAN_F13BANK2_FBIT15_Msk                                        /*!<Filter bit 15 */
#define CAN_F13BANK2_FBIT16_Pos                                     (16U)
#define CAN_F13BANK2_FBIT16_Msk                                     (0x1UL << CAN_F13BANK2_FBIT16_Pos)                             /*!< 0x00010000 */
#define CAN_F13BANK2_FBIT16                                         CAN_F13BANK2_FBIT16_Msk                                        /*!<Filter bit 16 */
#define CAN_F13BANK2_FBIT17_Pos                                     (17U)
#define CAN_F13BANK2_FBIT17_Msk                                     (0x1UL << CAN_F13BANK2_FBIT17_Pos)                             /*!< 0x00020000 */
#define CAN_F13BANK2_FBIT17                                         CAN_F13BANK2_FBIT17_Msk                                        /*!<Filter bit 17 */
#define CAN_F13BANK2_FBIT18_Pos                                     (18U)
#define CAN_F13BANK2_FBIT18_Msk                                     (0x1UL << CAN_F13BANK2_FBIT18_Pos)                             /*!< 0x00040000 */
#define CAN_F13BANK2_FBIT18                                         CAN_F13BANK2_FBIT18_Msk                                        /*!<Filter bit 18 */
#define CAN_F13BANK2_FBIT19_Pos                                     (19U)
#define CAN_F13BANK2_FBIT19_Msk                                     (0x1UL << CAN_F13BANK2_FBIT19_Pos)                             /*!< 0x00080000 */
#define CAN_F13BANK2_FBIT19                                         CAN_F13BANK2_FBIT19_Msk                                        /*!<Filter bit 19 */
#define CAN_F13BANK2_FBIT20_Pos                                     (20U)
#define CAN_F13BANK2_FBIT20_Msk                                     (0x1UL << CAN_F13BANK2_FBIT20_Pos)                             /*!< 0x00100000 */
#define CAN_F13BANK2_FBIT20                                         CAN_F13BANK2_FBIT20_Msk                                        /*!<Filter bit 20 */
#define CAN_F13BANK2_FBIT21_Pos                                     (21U)
#define CAN_F13BANK2_FBIT21_Msk                                     (0x1UL << CAN_F13BANK2_FBIT21_Pos)                             /*!< 0x00200000 */
#define CAN_F13BANK2_FBIT21                                         CAN_F13BANK2_FBIT21_Msk                                        /*!<Filter bit 21 */
#define CAN_F13BANK2_FBIT22_Pos                                     (22U)
#define CAN_F13BANK2_FBIT22_Msk                                     (0x1UL << CAN_F13BANK2_FBIT22_Pos)                             /*!< 0x00400000 */
#define CAN_F13BANK2_FBIT22                                         CAN_F13BANK2_FBIT22_Msk                                        /*!<Filter bit 22 */
#define CAN_F13BANK2_FBIT23_Pos                                     (23U)
#define CAN_F13BANK2_FBIT23_Msk                                     (0x1UL << CAN_F13BANK2_FBIT23_Pos)                             /*!< 0x00800000 */
#define CAN_F13BANK2_FBIT23                                         CAN_F13BANK2_FBIT23_Msk                                        /*!<Filter bit 23 */
#define CAN_F13BANK2_FBIT24_Pos                                     (24U)
#define CAN_F13BANK2_FBIT24_Msk                                     (0x1UL << CAN_F13BANK2_FBIT24_Pos)                             /*!< 0x01000000 */
#define CAN_F13BANK2_FBIT24                                         CAN_F13BANK2_FBIT24_Msk                                        /*!<Filter bit 24 */
#define CAN_F13BANK2_FBIT25_Pos                                     (25U)
#define CAN_F13BANK2_FBIT25_Msk                                     (0x1UL << CAN_F13BANK2_FBIT25_Pos)                             /*!< 0x02000000 */
#define CAN_F13BANK2_FBIT25                                         CAN_F13BANK2_FBIT25_Msk                                        /*!<Filter bit 25 */
#define CAN_F13BANK2_FBIT26_Pos                                     (26U)
#define CAN_F13BANK2_FBIT26_Msk                                     (0x1UL << CAN_F13BANK2_FBIT26_Pos)                             /*!< 0x04000000 */
#define CAN_F13BANK2_FBIT26                                         CAN_F13BANK2_FBIT26_Msk                                        /*!<Filter bit 26 */
#define CAN_F13BANK2_FBIT27_Pos                                     (27U)
#define CAN_F13BANK2_FBIT27_Msk                                     (0x1UL << CAN_F13BANK2_FBIT27_Pos)                             /*!< 0x08000000 */
#define CAN_F13BANK2_FBIT27                                         CAN_F13BANK2_FBIT27_Msk                                        /*!<Filter bit 27 */
#define CAN_F13BANK2_FBIT28_Pos                                     (28U)
#define CAN_F13BANK2_FBIT28_Msk                                     (0x1UL << CAN_F13BANK2_FBIT28_Pos)                             /*!< 0x10000000 */
#define CAN_F13BANK2_FBIT28                                         CAN_F13BANK2_FBIT28_Msk                                        /*!<Filter bit 28 */
#define CAN_F13BANK2_FBIT29_Pos                                     (29U)
#define CAN_F13BANK2_FBIT29_Msk                                     (0x1UL << CAN_F13BANK2_FBIT29_Pos)                             /*!< 0x20000000 */
#define CAN_F13BANK2_FBIT29                                         CAN_F13BANK2_FBIT29_Msk                                        /*!<Filter bit 29 */
#define CAN_F13BANK2_FBIT30_Pos                                     (30U)
#define CAN_F13BANK2_FBIT30_Msk                                     (0x1UL << CAN_F13BANK2_FBIT30_Pos)                             /*!< 0x40000000 */
#define CAN_F13BANK2_FBIT30                                         CAN_F13BANK2_FBIT30_Msk                                        /*!<Filter bit 30 */
#define CAN_F13BANK2_FBIT31_Pos                                     (31U)
#define CAN_F13BANK2_FBIT31_Msk                                     (0x1UL << CAN_F13BANK2_FBIT31_Pos)                             /*!< 0x80000000 */
#define CAN_F13BANK2_FBIT31                                         CAN_F13BANK2_FBIT31_Msk                                        /*!<Filter bit 31 */

/******************************************************************************/
/*                                                                            */
/*                          CRC calculation unit                              */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for CRC_DATA register  *********************/
#define CRC_DATA_DATA_Pos                                           (0U)
#define CRC_DATA_DATA_Msk                                           (0xFFFFFFFFUL << CRC_DATA_DATA_Pos)                            /*!< 0xFFFFFFFF */
#define CRC_DATA_DATA                                               CRC_DATA_DATA_Msk                                              /*!< Data register bits */


/*******************  Bit definition for CRC_INDATA register  ********************/
#define CRC_INDATA_INDATA_Pos                                       (0U)
#define CRC_INDATA_INDATA_Msk                                       (0xFFUL << CRC_INDATA_INDATA_Pos)                              /*!< 0x000000FF */
#define CRC_INDATA_INDATA                                           CRC_INDATA_INDATA_Msk                                          /*!< General-purpose 8-bit data register bits */


/********************  Bit definition for CRC_CTRL register  ********************/
#define CRC_CTRL_RST_Pos                                            (0U)
#define CRC_CTRL_RST_Msk                                            (0x1UL << CRC_CTRL_RST_Pos)                                    /*!< 0x00000001 */
#define CRC_CTRL_RST                                                CRC_CTRL_RST_Msk                                               /*!< RESET bit */

/******************************************************************************/
/*                                                                            */
/*                             DMA Controller                                 */
/*                                                                            */
/******************************************************************************/
/********************  Bits definition for DMA_INTSTS register  *****************/
#define DMA_INTSTS_GINTFLG1_Pos                                     (0U)
#define DMA_INTSTS_GINTFLG1_Msk                                     (0x1UL << DMA_INTSTS_GINTFLG1_Pos)                             /*!< 0x00000001 */
#define DMA_INTSTS_GINTFLG1                                         DMA_INTSTS_GINTFLG1_Msk                                        /*!< Channel 1 Global interrupt flag */
#define DMA_INTSTS_TCFLG1_Pos                                       (1U)
#define DMA_INTSTS_TCFLG1_Msk                                       (0x1UL << DMA_INTSTS_TCFLG1_Pos)                               /*!< 0x00000002 */
#define DMA_INTSTS_TCFLG1                                           DMA_INTSTS_TCFLG1_Msk                                          /*!< Channel 1 Transfer complete flag */
#define DMA_INTSTS_HTFLG1_Pos                                       (2U)
#define DMA_INTSTS_HTFLG1_Msk                                       (0x1UL << DMA_INTSTS_HTFLG1_Pos)                               /*!< 0x00000004 */
#define DMA_INTSTS_HTFLG1                                           DMA_INTSTS_HTFLG1_Msk                                          /*!< Channel 1 Half transfer flag */
#define DMA_INTSTS_TERRFLG1_Pos                                     (3U)
#define DMA_INTSTS_TERRFLG1_Msk                                     (0x1UL << DMA_INTSTS_TERRFLG1_Pos)                             /*!< 0x00000008 */
#define DMA_INTSTS_TERRFLG1                                         DMA_INTSTS_TERRFLG1_Msk                                        /*!< Channel 1 Transfer error flag */
#define DMA_INTSTS_GINTFLG2_Pos                                     (4U)
#define DMA_INTSTS_GINTFLG2_Msk                                     (0x1UL << DMA_INTSTS_GINTFLG2_Pos)                             /*!< 0x00000010 */
#define DMA_INTSTS_GINTFLG2                                         DMA_INTSTS_GINTFLG2_Msk                                        /*!< Channel 2 Global interrupt flag */
#define DMA_INTSTS_TCFLG2_Pos                                       (5U)
#define DMA_INTSTS_TCFLG2_Msk                                       (0x1UL << DMA_INTSTS_TCFLG2_Pos)                               /*!< 0x00000020 */
#define DMA_INTSTS_TCFLG2                                           DMA_INTSTS_TCFLG2_Msk                                          /*!< Channel 2 Transfer complete flag */
#define DMA_INTSTS_HTFLG2_Pos                                       (6U)
#define DMA_INTSTS_HTFLG2_Msk                                       (0x1UL << DMA_INTSTS_HTFLG2_Pos)                               /*!< 0x00000040 */
#define DMA_INTSTS_HTFLG2                                           DMA_INTSTS_HTFLG2_Msk                                          /*!< Channel 2 Half transfer flag */
#define DMA_INTSTS_TERRFLG2_Pos                                     (7U)
#define DMA_INTSTS_TERRFLG2_Msk                                     (0x1UL << DMA_INTSTS_TERRFLG2_Pos)                             /*!< 0x00000080 */
#define DMA_INTSTS_TERRFLG2                                         DMA_INTSTS_TERRFLG2_Msk                                        /*!< Channel 2 Transfer error flag */
#define DMA_INTSTS_GINTFLG3_Pos                                     (8U)
#define DMA_INTSTS_GINTFLG3_Msk                                     (0x1UL << DMA_INTSTS_GINTFLG3_Pos)                             /*!< 0x00000100 */
#define DMA_INTSTS_GINTFLG3                                         DMA_INTSTS_GINTFLG3_Msk                                        /*!< Channel 3 Global interrupt flag */
#define DMA_INTSTS_TCFLG3_Pos                                       (9U)
#define DMA_INTSTS_TCFLG3_Msk                                       (0x1UL << DMA_INTSTS_TCFLG3_Pos)                               /*!< 0x00000200 */
#define DMA_INTSTS_TCFLG3                                           DMA_INTSTS_TCFLG3_Msk                                          /*!< Channel 3 Transfer complete flag */
#define DMA_INTSTS_HTFLG3_Pos                                       (10U)
#define DMA_INTSTS_HTFLG3_Msk                                       (0x1UL << DMA_INTSTS_HTFLG3_Pos)                               /*!< 0x00000400 */
#define DMA_INTSTS_HTFLG3                                           DMA_INTSTS_HTFLG3_Msk                                          /*!< Channel 3 Half transfer flag */
#define DMA_INTSTS_TERRFLG3_Pos                                     (11U)
#define DMA_INTSTS_TERRFLG3_Msk                                     (0x1UL << DMA_INTSTS_TERRFLG3_Pos)                             /*!< 0x00000800 */
#define DMA_INTSTS_TERRFLG3                                         DMA_INTSTS_TERRFLG3_Msk                                        /*!< Channel 3 Transfer error flag */
#define DMA_INTSTS_GINTFLG4_Pos                                     (12U)
#define DMA_INTSTS_GINTFLG4_Msk                                     (0x1UL << DMA_INTSTS_GINTFLG4_Pos)                             /*!< 0x00001000 */
#define DMA_INTSTS_GINTFLG4                                         DMA_INTSTS_GINTFLG4_Msk                                        /*!< Channel 4 Global interrupt flag */
#define DMA_INTSTS_TCFLG4_Pos                                       (13U)
#define DMA_INTSTS_TCFLG4_Msk                                       (0x1UL << DMA_INTSTS_TCFLG4_Pos)                               /*!< 0x00002000 */
#define DMA_INTSTS_TCFLG4                                           DMA_INTSTS_TCFLG4_Msk                                          /*!< Channel 4 Transfer complete flag */
#define DMA_INTSTS_HTFLG4_Pos                                       (14U)
#define DMA_INTSTS_HTFLG4_Msk                                       (0x1UL << DMA_INTSTS_HTFLG4_Pos)                               /*!< 0x00004000 */
#define DMA_INTSTS_HTFLG4                                           DMA_INTSTS_HTFLG4_Msk                                          /*!< Channel 4 Half transfer flag */
#define DMA_INTSTS_TERRFLG4_Pos                                     (15U)
#define DMA_INTSTS_TERRFLG4_Msk                                     (0x1UL << DMA_INTSTS_TERRFLG4_Pos)                             /*!< 0x00008000 */
#define DMA_INTSTS_TERRFLG4                                         DMA_INTSTS_TERRFLG4_Msk                                        /*!< Channel 4 Transfer error flag */
#define DMA_INTSTS_GINTFLG5_Pos                                     (16U)
#define DMA_INTSTS_GINTFLG5_Msk                                     (0x1UL << DMA_INTSTS_GINTFLG5_Pos)                             /*!< 0x00010000 */
#define DMA_INTSTS_GINTFLG5                                         DMA_INTSTS_GINTFLG5_Msk                                        /*!< Channel 5 Global interrupt flag */
#define DMA_INTSTS_TCFLG5_Pos                                       (17U)
#define DMA_INTSTS_TCFLG5_Msk                                       (0x1UL << DMA_INTSTS_TCFLG5_Pos)                               /*!< 0x00020000 */
#define DMA_INTSTS_TCFLG5                                           DMA_INTSTS_TCFLG5_Msk                                          /*!< Channel 5 Transfer complete flag */
#define DMA_INTSTS_HTFLG5_Pos                                       (18U)
#define DMA_INTSTS_HTFLG5_Msk                                       (0x1UL << DMA_INTSTS_HTFLG5_Pos)                               /*!< 0x00040000 */
#define DMA_INTSTS_HTFLG5                                           DMA_INTSTS_HTFLG5_Msk                                          /*!< Channel 5 Half transfer flag */
#define DMA_INTSTS_TERRFLG5_Pos                                     (19U)
#define DMA_INTSTS_TERRFLG5_Msk                                     (0x1UL << DMA_INTSTS_TERRFLG5_Pos)                             /*!< 0x00080000 */
#define DMA_INTSTS_TERRFLG5                                         DMA_INTSTS_TERRFLG5_Msk                                        /*!< Channel 5 Transfer error flag */
#define DMA_INTSTS_GINTFLG6_Pos                                     (20U)
#define DMA_INTSTS_GINTFLG6_Msk                                     (0x1UL << DMA_INTSTS_GINTFLG6_Pos)                             /*!< 0x00100000 */
#define DMA_INTSTS_GINTFLG6                                         DMA_INTSTS_GINTFLG6_Msk                                        /*!< Channel 6 Global interrupt flag */
#define DMA_INTSTS_TCFLG6_Pos                                       (21U)
#define DMA_INTSTS_TCFLG6_Msk                                       (0x1UL << DMA_INTSTS_TCFLG6_Pos)                               /*!< 0x00200000 */
#define DMA_INTSTS_TCFLG6                                           DMA_INTSTS_TCFLG6_Msk                                          /*!< Channel 6 Transfer complete flag */
#define DMA_INTSTS_HTFLG6_Pos                                       (22U)
#define DMA_INTSTS_HTFLG6_Msk                                       (0x1UL << DMA_INTSTS_HTFLG6_Pos)                               /*!< 0x00400000 */
#define DMA_INTSTS_HTFLG6                                           DMA_INTSTS_HTFLG6_Msk                                          /*!< Channel 6 Half transfer flag */
#define DMA_INTSTS_TERRFLG6_Pos                                     (23U)
#define DMA_INTSTS_TERRFLG6_Msk                                     (0x1UL << DMA_INTSTS_TERRFLG6_Pos)                             /*!< 0x00800000 */
#define DMA_INTSTS_TERRFLG6                                         DMA_INTSTS_TERRFLG6_Msk                                        /*!< Channel 6 Transfer error flag */
#define DMA_INTSTS_GINTFLG7_Pos                                     (24U)
#define DMA_INTSTS_GINTFLG7_Msk                                     (0x1UL << DMA_INTSTS_GINTFLG7_Pos)                             /*!< 0x01000000 */
#define DMA_INTSTS_GINTFLG7                                         DMA_INTSTS_GINTFLG7_Msk                                        /*!< Channel 7 Global interrupt flag */
#define DMA_INTSTS_TCFLG7_Pos                                       (25U)
#define DMA_INTSTS_TCFLG7_Msk                                       (0x1UL << DMA_INTSTS_TCFLG7_Pos)                               /*!< 0x02000000 */
#define DMA_INTSTS_TCFLG7                                           DMA_INTSTS_TCFLG7_Msk                                          /*!< Channel 7 Transfer complete flag */
#define DMA_INTSTS_HTFLG7_Pos                                       (26U)
#define DMA_INTSTS_HTFLG7_Msk                                       (0x1UL << DMA_INTSTS_HTFLG7_Pos)                               /*!< 0x04000000 */
#define DMA_INTSTS_HTFLG7                                           DMA_INTSTS_HTFLG7_Msk                                          /*!< Channel 7 Half transfer flag */
#define DMA_INTSTS_TERRFLG7_Pos                                     (27U)
#define DMA_INTSTS_TERRFLG7_Msk                                     (0x1UL << DMA_INTSTS_TERRFLG7_Pos)                             /*!< 0x08000000 */
#define DMA_INTSTS_TERRFLG7                                         DMA_INTSTS_TERRFLG7_Msk                                        /*!< Channel 7 Transfer error flag */

/*******************  Bits definition for DMA_INTFCLR register  *****************/
#define DMA_INTFCLR_GINTCLR1_Pos                                    (0U)
#define DMA_INTFCLR_GINTCLR1_Msk                                    (0x1UL << DMA_INTFCLR_GINTCLR1_Pos)                            /*!< 0x00000001 */
#define DMA_INTFCLR_GINTCLR1                                        DMA_INTFCLR_GINTCLR1_Msk                                       /*!< Channel 1 Global interrupt clear */
#define DMA_INTFCLR_TCCLR1_Pos                                      (1U)
#define DMA_INTFCLR_TCCLR1_Msk                                      (0x1UL << DMA_INTFCLR_TCCLR1_Pos)                              /*!< 0x00000002 */
#define DMA_INTFCLR_TCCLR1                                          DMA_INTFCLR_TCCLR1_Msk                                         /*!< Channel 1 Transfer complete clear */
#define DMA_INTFCLR_HTCLR1_Pos                                      (2U)
#define DMA_INTFCLR_HTCLR1_Msk                                      (0x1UL << DMA_INTFCLR_HTCLR1_Pos)                              /*!< 0x00000004 */
#define DMA_INTFCLR_HTCLR1                                          DMA_INTFCLR_HTCLR1_Msk                                         /*!< Channel 1 Half transfer clear */
#define DMA_INTFCLR_TERRCLR1_Pos                                    (3U)
#define DMA_INTFCLR_TERRCLR1_Msk                                    (0x1UL << DMA_INTFCLR_TERRCLR1_Pos)                            /*!< 0x00000008 */
#define DMA_INTFCLR_TERRCLR1                                        DMA_INTFCLR_TERRCLR1_Msk                                       /*!< Channel 1 Transfer error clear */
#define DMA_INTFCLR_GINTCLR2_Pos                                    (4U)
#define DMA_INTFCLR_GINTCLR2_Msk                                    (0x1UL << DMA_INTFCLR_GINTCLR2_Pos)                            /*!< 0x00000010 */
#define DMA_INTFCLR_GINTCLR2                                        DMA_INTFCLR_GINTCLR2_Msk                                       /*!< Channel 2 Global interrupt clear */
#define DMA_INTFCLR_TCCLR2_Pos                                      (5U)
#define DMA_INTFCLR_TCCLR2_Msk                                      (0x1UL << DMA_INTFCLR_TCCLR2_Pos)                              /*!< 0x00000020 */
#define DMA_INTFCLR_TCCLR2                                          DMA_INTFCLR_TCCLR2_Msk                                         /*!< Channel 2 Transfer complete clear */
#define DMA_INTFCLR_HTCLR2_Pos                                      (6U)
#define DMA_INTFCLR_HTCLR2_Msk                                      (0x1UL << DMA_INTFCLR_HTCLR2_Pos)                              /*!< 0x00000040 */
#define DMA_INTFCLR_HTCLR2                                          DMA_INTFCLR_HTCLR2_Msk                                         /*!< Channel 2 Half transfer clear */
#define DMA_INTFCLR_TERRCLR2_Pos                                    (7U)
#define DMA_INTFCLR_TERRCLR2_Msk                                    (0x1UL << DMA_INTFCLR_TERRCLR2_Pos)                            /*!< 0x00000080 */
#define DMA_INTFCLR_TERRCLR2                                        DMA_INTFCLR_TERRCLR2_Msk                                       /*!< Channel 2 Transfer error clear */
#define DMA_INTFCLR_GINTCLR3_Pos                                    (8U)
#define DMA_INTFCLR_GINTCLR3_Msk                                    (0x1UL << DMA_INTFCLR_GINTCLR3_Pos)                            /*!< 0x00000100 */
#define DMA_INTFCLR_GINTCLR3                                        DMA_INTFCLR_GINTCLR3_Msk                                       /*!< Channel 3 Global interrupt clear */
#define DMA_INTFCLR_TCCLR3_Pos                                      (9U)
#define DMA_INTFCLR_TCCLR3_Msk                                      (0x1UL << DMA_INTFCLR_TCCLR3_Pos)                              /*!< 0x00000200 */
#define DMA_INTFCLR_TCCLR3                                          DMA_INTFCLR_TCCLR3_Msk                                         /*!< Channel 3 Transfer complete clear */
#define DMA_INTFCLR_HTCLR3_Pos                                      (10U)
#define DMA_INTFCLR_HTCLR3_Msk                                      (0x1UL << DMA_INTFCLR_HTCLR3_Pos)                              /*!< 0x00000400 */
#define DMA_INTFCLR_HTCLR3                                          DMA_INTFCLR_HTCLR3_Msk                                         /*!< Channel 3 Half transfer clear */
#define DMA_INTFCLR_TERRCLR3_Pos                                    (11U)
#define DMA_INTFCLR_TERRCLR3_Msk                                    (0x1UL << DMA_INTFCLR_TERRCLR3_Pos)                            /*!< 0x00000800 */
#define DMA_INTFCLR_TERRCLR3                                        DMA_INTFCLR_TERRCLR3_Msk                                       /*!< Channel 3 Transfer error clear */
#define DMA_INTFCLR_GINTCLR4_Pos                                    (12U)
#define DMA_INTFCLR_GINTCLR4_Msk                                    (0x1UL << DMA_INTFCLR_GINTCLR4_Pos)                            /*!< 0x00001000 */
#define DMA_INTFCLR_GINTCLR4                                        DMA_INTFCLR_GINTCLR4_Msk                                       /*!< Channel 4 Global interrupt clear */
#define DMA_INTFCLR_TCCLR4_Pos                                      (13U)
#define DMA_INTFCLR_TCCLR4_Msk                                      (0x1UL << DMA_INTFCLR_TCCLR4_Pos)                              /*!< 0x00002000 */
#define DMA_INTFCLR_TCCLR4                                          DMA_INTFCLR_TCCLR4_Msk                                         /*!< Channel 4 Transfer complete clear */
#define DMA_INTFCLR_HTCLR4_Pos                                      (14U)
#define DMA_INTFCLR_HTCLR4_Msk                                      (0x1UL << DMA_INTFCLR_HTCLR4_Pos)                              /*!< 0x00004000 */
#define DMA_INTFCLR_HTCLR4                                          DMA_INTFCLR_HTCLR4_Msk                                         /*!< Channel 4 Half transfer clear */
#define DMA_INTFCLR_TERRCLR4_Pos                                    (15U)
#define DMA_INTFCLR_TERRCLR4_Msk                                    (0x1UL << DMA_INTFCLR_TERRCLR4_Pos)                            /*!< 0x00008000 */
#define DMA_INTFCLR_TERRCLR4                                        DMA_INTFCLR_TERRCLR4_Msk                                       /*!< Channel 4 Transfer error clear */
#define DMA_INTFCLR_GINTCLR5_Pos                                    (16U)
#define DMA_INTFCLR_GINTCLR5_Msk                                    (0x1UL << DMA_INTFCLR_GINTCLR5_Pos)                            /*!< 0x00010000 */
#define DMA_INTFCLR_GINTCLR5                                        DMA_INTFCLR_GINTCLR5_Msk                                       /*!< Channel 5 Global interrupt clear */
#define DMA_INTFCLR_TCCLR5_Pos                                      (17U)
#define DMA_INTFCLR_TCCLR5_Msk                                      (0x1UL << DMA_INTFCLR_TCCLR5_Pos)                              /*!< 0x00020000 */
#define DMA_INTFCLR_TCCLR5                                          DMA_INTFCLR_TCCLR5_Msk                                         /*!< Channel 5 Transfer complete clear */
#define DMA_INTFCLR_HTCLR5_Pos                                      (18U)
#define DMA_INTFCLR_HTCLR5_Msk                                      (0x1UL << DMA_INTFCLR_HTCLR5_Pos)                              /*!< 0x00040000 */
#define DMA_INTFCLR_HTCLR5                                          DMA_INTFCLR_HTCLR5_Msk                                         /*!< Channel 5 Half transfer clear */
#define DMA_INTFCLR_TERRCLR5_Pos                                    (19U)
#define DMA_INTFCLR_TERRCLR5_Msk                                    (0x1UL << DMA_INTFCLR_TERRCLR5_Pos)                            /*!< 0x00080000 */
#define DMA_INTFCLR_TERRCLR5                                        DMA_INTFCLR_TERRCLR5_Msk                                       /*!< Channel 5 Transfer error clear */
#define DMA_INTFCLR_GINTCLR6_Pos                                    (20U)
#define DMA_INTFCLR_GINTCLR6_Msk                                    (0x1UL << DMA_INTFCLR_GINTCLR6_Pos)                            /*!< 0x00100000 */
#define DMA_INTFCLR_GINTCLR6                                        DMA_INTFCLR_GINTCLR6_Msk                                       /*!< Channel 6 Global interrupt clear */
#define DMA_INTFCLR_TCCLR6_Pos                                      (21U)
#define DMA_INTFCLR_TCCLR6_Msk                                      (0x1UL << DMA_INTFCLR_TCCLR6_Pos)                              /*!< 0x00200000 */
#define DMA_INTFCLR_TCCLR6                                          DMA_INTFCLR_TCCLR6_Msk                                         /*!< Channel 6 Transfer complete clear */
#define DMA_INTFCLR_HTCLR6_Pos                                      (22U)
#define DMA_INTFCLR_HTCLR6_Msk                                      (0x1UL << DMA_INTFCLR_HTCLR6_Pos)                              /*!< 0x00400000 */
#define DMA_INTFCLR_HTCLR6                                          DMA_INTFCLR_HTCLR6_Msk                                         /*!< Channel 6 Half transfer clear */
#define DMA_INTFCLR_TERRCLR6_Pos                                    (23U)
#define DMA_INTFCLR_TERRCLR6_Msk                                    (0x1UL << DMA_INTFCLR_TERRCLR6_Pos)                            /*!< 0x00800000 */
#define DMA_INTFCLR_TERRCLR6                                        DMA_INTFCLR_TERRCLR6_Msk                                       /*!< Channel 6 Transfer error clear */
#define DMA_INTFCLR_GINTCLR7_Pos                                    (24U)
#define DMA_INTFCLR_GINTCLR7_Msk                                    (0x1UL << DMA_INTFCLR_GINTCLR7_Pos)                            /*!< 0x01000000 */
#define DMA_INTFCLR_GINTCLR7                                        DMA_INTFCLR_GINTCLR7_Msk                                       /*!< Channel 7 Global interrupt clear */
#define DMA_INTFCLR_TCCLR7_Pos                                      (25U)
#define DMA_INTFCLR_TCCLR7_Msk                                      (0x1UL << DMA_INTFCLR_TCCLR7_Pos)                              /*!< 0x02000000 */
#define DMA_INTFCLR_TCCLR7                                          DMA_INTFCLR_TCCLR7_Msk                                         /*!< Channel 7 Transfer complete clear */
#define DMA_INTFCLR_HTCLR7_Pos                                      (26U)
#define DMA_INTFCLR_HTCLR7_Msk                                      (0x1UL << DMA_INTFCLR_HTCLR7_Pos)                              /*!< 0x04000000 */
#define DMA_INTFCLR_HTCLR7                                          DMA_INTFCLR_HTCLR7_Msk                                         /*!< Channel 7 Half transfer clear */
#define DMA_INTFCLR_TERRCLR7_Pos                                    (27U)
#define DMA_INTFCLR_TERRCLR7_Msk                                    (0x1UL << DMA_INTFCLR_TERRCLR7_Pos)                            /*!< 0x08000000 */
#define DMA_INTFCLR_TERRCLR7                                        DMA_INTFCLR_TERRCLR7_Msk                                       /*!< Channel 7 Transfer error clear */

/*******************  Bits definition for DMA_CHCFG register  *******************/
#define DMA_CHCFG_CHEN_Pos                                          (0U)
#define DMA_CHCFG_CHEN_Msk                                          (0x1UL << DMA_CHCFG_CHEN_Pos)                                  /*!< 0x00000001 */
#define DMA_CHCFG_CHEN                                              DMA_CHCFG_CHEN_Msk                                             /*!< Channel enable */
#define DMA_CHCFG_TCINTEN_Pos                                       (1U)
#define DMA_CHCFG_TCINTEN_Msk                                       (0x1UL << DMA_CHCFG_TCINTEN_Pos)                               /*!< 0x00000002 */
#define DMA_CHCFG_TCINTEN                                           DMA_CHCFG_TCINTEN_Msk                                          /*!< Transfer complete interrupt enable */
#define DMA_CHCFG_HTINTEN_Pos                                       (2U)
#define DMA_CHCFG_HTINTEN_Msk                                       (0x1UL << DMA_CHCFG_HTINTEN_Pos)                               /*!< 0x00000004 */
#define DMA_CHCFG_HTINTEN                                           DMA_CHCFG_HTINTEN_Msk                                          /*!< Half transfer interrupt enable */
#define DMA_CHCFG_TERRINTEN_Pos                                     (3U)
#define DMA_CHCFG_TERRINTEN_Msk                                     (0x1UL << DMA_CHCFG_TERRINTEN_Pos)                             /*!< 0x00000008 */
#define DMA_CHCFG_TERRINTEN                                         DMA_CHCFG_TERRINTEN_Msk                                        /*!< Transfer error interrupt enable */
#define DMA_CHCFG_DIRCFG_Pos                                        (4U)
#define DMA_CHCFG_DIRCFG_Msk                                        (0x1UL << DMA_CHCFG_DIRCFG_Pos)                                /*!< 0x00000010 */
#define DMA_CHCFG_DIRCFG                                            DMA_CHCFG_DIRCFG_Msk                                           /*!< Data transfer direction configuration */
#define DMA_CHCFG_CIRMODE_Pos                                       (5U)
#define DMA_CHCFG_CIRMODE_Msk                                       (0x1UL << DMA_CHCFG_CIRMODE_Pos)                               /*!< 0x00000020 */
#define DMA_CHCFG_CIRMODE                                           DMA_CHCFG_CIRMODE_Msk                                          /*!< Circular mode */
#define DMA_CHCFG_PERIMODE_Pos                                      (6U)
#define DMA_CHCFG_PERIMODE_Msk                                      (0x1UL << DMA_CHCFG_PERIMODE_Pos)                              /*!< 0x00000040 */
#define DMA_CHCFG_PERIMODE                                          DMA_CHCFG_PERIMODE_Msk                                         /*!< Peripheral increment mode */
#define DMA_CHCFG_MIMODE_Pos                                        (7U)
#define DMA_CHCFG_MIMODE_Msk                                        (0x1UL << DMA_CHCFG_MIMODE_Pos)                                /*!< 0x00000080 */
#define DMA_CHCFG_MIMODE                                            DMA_CHCFG_MIMODE_Msk                                           /*!< Memory increment mode */
#define DMA_CHCFG_PERSIZE_Pos                                       (8U)
#define DMA_CHCFG_PERSIZE_Msk                                       (0x3UL << DMA_CHCFG_PERSIZE_Pos)                               /*!< 0x00000300 */
#define DMA_CHCFG_PERSIZE                                           DMA_CHCFG_PERSIZE_Msk                                          /*!< PERSIZE[1:0] bits (Peripheral size) */
#define DMA_CHCFG_PERSIZE_0                                         (0x1UL << DMA_CHCFG_PERSIZE_Pos)                               /*!< 0x00000100 */
#define DMA_CHCFG_PERSIZE_1                                         (0x2UL << DMA_CHCFG_PERSIZE_Pos)                               /*!< 0x00000200 */
#define DMA_CHCFG_MEMSIZE_Pos                                       (10U)
#define DMA_CHCFG_MEMSIZE_Msk                                       (0x3UL << DMA_CHCFG_MEMSIZE_Pos)                               /*!< 0x00000C00 */
#define DMA_CHCFG_MEMSIZE                                           DMA_CHCFG_MEMSIZE_Msk                                          /*!< MEMSIZE[1:0] bits (Memory size) */
#define DMA_CHCFG_MEMSIZE_0                                         (0x1UL << DMA_CHCFG_MEMSIZE_Pos)                               /*!< 0x00000400 */
#define DMA_CHCFG_MEMSIZE_1                                         (0x2UL << DMA_CHCFG_MEMSIZE_Pos)                               /*!< 0x00000800 */
#define DMA_CHCFG_CHPL_Pos                                          (12U)
#define DMA_CHCFG_CHPL_Msk                                          (0x3UL << DMA_CHCFG_CHPL_Pos)                                  /*!< 0x00003000 */
#define DMA_CHCFG_CHPL                                              DMA_CHCFG_CHPL_Msk                                             /*!< CHPL[1:0] bits(Channel priority level) */
#define DMA_CHCFG_CHPL_0                                            (0x1UL << DMA_CHCFG_CHPL_Pos)                                  /*!< 0x00001000 */
#define DMA_CHCFG_CHPL_1                                            (0x2UL << DMA_CHCFG_CHPL_Pos)                                  /*!< 0x00002000 */
#define DMA_CHCFG_M2MMODE_Pos                                       (14U)
#define DMA_CHCFG_M2MMODE_Msk                                       (0x1UL << DMA_CHCFG_M2MMODE_Pos)                               /*!< 0x00004000 */
#define DMA_CHCFG_M2MMODE                                           DMA_CHCFG_M2MMODE_Msk                                          /*!< Memory to memory mode */

/*******************  Bits definition for DMA_CHNDATAR register  *****************/
#define DMA_CHNDATAR_NDATAT_Pos                                    (0U)
#define DMA_CHNDATAR_NDATAT_Msk                                    (0xFFFFUL << DMA_CHNDATAR_NDATAT_Pos)                           /*!< 0x0000FFFF */
#define DMA_CHNDATAR_NDATAT                                        DMA_CHNDATAR_NDATAT_Msk                                         /*!< Number of data to transfer */

/*******************  Bits definition for DMA_CHPADDR register  *****************/
#define DMA_CHPADDR_PERADDR_Pos                                    (0U)
#define DMA_CHPADDR_PERADDR_Msk                                    (0xFFFFFFFFUL << DMA_CHPADDR_PERADDR_Pos)                       /*!< 0xFFFFFFFF */
#define DMA_CHPADDR_PERADDR                                        DMA_CHPADDR_PERADDR_Msk                                         /*!< Peripheral address */

/*******************  Bits definition for DMA_CHMADDR register  *****************/
#define DMA_CHMADDR_MEMADDR_Pos                                    (0U)
#define DMA_CHMADDR_MEMADDR_Msk                                    (0xFFFFFFFFUL << DMA_CHMADDR_MEMADDR_Pos)                       /*!< 0xFFFFFFFF */
#define DMA_CHMADDR_MEMADDR                                        DMA_CHMADDR_MEMADDR_Msk                                         /*!< Memory address */

/******************************************************************************/
/*                                                                            */
/*                    External Interrupt/Event Controller                     */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for EINT_IMASK register  *******************/
#define EINT_IMASK_IMASK0_Pos                                       (0U)
#define EINT_IMASK_IMASK0_Msk                                       (0x1UL << EINT_IMASK_IMASK0_Pos)                               /*!< 0x00000001 */
#define EINT_IMASK_IMASK0                                           EINT_IMASK_IMASK0_Msk                                          /*!< Interrupt Mask on line 0 */
#define EINT_IMASK_IMASK1_Pos                                       (1U)
#define EINT_IMASK_IMASK1_Msk                                       (0x1UL << EINT_IMASK_IMASK1_Pos)                               /*!< 0x00000002 */
#define EINT_IMASK_IMASK1                                           EINT_IMASK_IMASK1_Msk                                          /*!< Interrupt Mask on line 1 */
#define EINT_IMASK_IMASK2_Pos                                       (2U)
#define EINT_IMASK_IMASK2_Msk                                       (0x1UL << EINT_IMASK_IMASK2_Pos)                               /*!< 0x00000004 */
#define EINT_IMASK_IMASK2                                           EINT_IMASK_IMASK2_Msk                                          /*!< Interrupt Mask on line 2 */
#define EINT_IMASK_IMASK3_Pos                                       (3U)
#define EINT_IMASK_IMASK3_Msk                                       (0x1UL << EINT_IMASK_IMASK3_Pos)                               /*!< 0x00000008 */
#define EINT_IMASK_IMASK3                                           EINT_IMASK_IMASK3_Msk                                          /*!< Interrupt Mask on line 3 */
#define EINT_IMASK_IMASK4_Pos                                       (4U)
#define EINT_IMASK_IMASK4_Msk                                       (0x1UL << EINT_IMASK_IMASK4_Pos)                               /*!< 0x00000010 */
#define EINT_IMASK_IMASK4                                           EINT_IMASK_IMASK4_Msk                                          /*!< Interrupt Mask on line 4 */
#define EINT_IMASK_IMASK5_Pos                                       (5U)
#define EINT_IMASK_IMASK5_Msk                                       (0x1UL << EINT_IMASK_IMASK5_Pos)                               /*!< 0x00000020 */
#define EINT_IMASK_IMASK5                                           EINT_IMASK_IMASK5_Msk                                          /*!< Interrupt Mask on line 5 */
#define EINT_IMASK_IMASK6_Pos                                       (6U)
#define EINT_IMASK_IMASK6_Msk                                       (0x1UL << EINT_IMASK_IMASK6_Pos)                               /*!< 0x00000040 */
#define EINT_IMASK_IMASK6                                           EINT_IMASK_IMASK6_Msk                                          /*!< Interrupt Mask on line 6 */
#define EINT_IMASK_IMASK7_Pos                                       (7U)
#define EINT_IMASK_IMASK7_Msk                                       (0x1UL << EINT_IMASK_IMASK7_Pos)                               /*!< 0x00000080 */
#define EINT_IMASK_IMASK7                                           EINT_IMASK_IMASK7_Msk                                          /*!< Interrupt Mask on line 7 */
#define EINT_IMASK_IMASK8_Pos                                       (8U)
#define EINT_IMASK_IMASK8_Msk                                       (0x1UL << EINT_IMASK_IMASK8_Pos)                               /*!< 0x00000100 */
#define EINT_IMASK_IMASK8                                           EINT_IMASK_IMASK8_Msk                                          /*!< Interrupt Mask on line 8 */
#define EINT_IMASK_IMASK9_Pos                                       (9U)
#define EINT_IMASK_IMASK9_Msk                                       (0x1UL << EINT_IMASK_IMASK9_Pos)                               /*!< 0x00000200 */
#define EINT_IMASK_IMASK9                                           EINT_IMASK_IMASK9_Msk                                          /*!< Interrupt Mask on line 9 */
#define EINT_IMASK_IMASK10_Pos                                      (10U)
#define EINT_IMASK_IMASK10_Msk                                      (0x1UL << EINT_IMASK_IMASK10_Pos)                              /*!< 0x00000400 */
#define EINT_IMASK_IMASK10                                          EINT_IMASK_IMASK10_Msk                                         /*!< Interrupt Mask on line 10 */
#define EINT_IMASK_IMASK11_Pos                                      (11U)
#define EINT_IMASK_IMASK11_Msk                                      (0x1UL << EINT_IMASK_IMASK11_Pos)                              /*!< 0x00000800 */
#define EINT_IMASK_IMASK11                                          EINT_IMASK_IMASK11_Msk                                         /*!< Interrupt Mask on line 11 */
#define EINT_IMASK_IMASK12_Pos                                      (12U)
#define EINT_IMASK_IMASK12_Msk                                      (0x1UL << EINT_IMASK_IMASK12_Pos)                              /*!< 0x00001000 */
#define EINT_IMASK_IMASK12                                          EINT_IMASK_IMASK12_Msk                                         /*!< Interrupt Mask on line 12 */
#define EINT_IMASK_IMASK13_Pos                                      (13U)
#define EINT_IMASK_IMASK13_Msk                                      (0x1UL << EINT_IMASK_IMASK13_Pos)                              /*!< 0x00002000 */
#define EINT_IMASK_IMASK13                                          EINT_IMASK_IMASK13_Msk                                         /*!< Interrupt Mask on line 13 */
#define EINT_IMASK_IMASK14_Pos                                      (14U)
#define EINT_IMASK_IMASK14_Msk                                      (0x1UL << EINT_IMASK_IMASK14_Pos)                              /*!< 0x00004000 */
#define EINT_IMASK_IMASK14                                          EINT_IMASK_IMASK14_Msk                                         /*!< Interrupt Mask on line 14 */
#define EINT_IMASK_IMASK15_Pos                                      (15U)
#define EINT_IMASK_IMASK15_Msk                                      (0x1UL << EINT_IMASK_IMASK15_Pos)                              /*!< 0x00008000 */
#define EINT_IMASK_IMASK15                                          EINT_IMASK_IMASK15_Msk                                         /*!< Interrupt Mask on line 15 */
#define EINT_IMASK_IMASK16_Pos                                      (16U)
#define EINT_IMASK_IMASK16_Msk                                      (0x1UL << EINT_IMASK_IMASK16_Pos)                              /*!< 0x00010000 */
#define EINT_IMASK_IMASK16                                          EINT_IMASK_IMASK16_Msk                                         /*!< Interrupt Mask on line 16 */
#define EINT_IMASK_IMASK17_Pos                                      (17U)
#define EINT_IMASK_IMASK17_Msk                                      (0x1UL << EINT_IMASK_IMASK17_Pos)                              /*!< 0x00020000 */
#define EINT_IMASK_IMASK17                                          EINT_IMASK_IMASK17_Msk                                         /*!< Interrupt Mask on line 17 */
#define EINT_IMASK_IMASK18_Pos                                      (18U)
#define EINT_IMASK_IMASK18_Msk                                      (0x1UL << EINT_IMASK_IMASK18_Pos)                              /*!< 0x00040000 */
#define EINT_IMASK_IMASK18                                          EINT_IMASK_IMASK18_Msk                                         /*!< Interrupt Mask on line 18 */

/* Reference Defines */
#define EINT_IMASK_IM0                                              EINT_IMASK_IMASK0
#define EINT_IMASK_IM1                                              EINT_IMASK_IMASK1
#define EINT_IMASK_IM2                                              EINT_IMASK_IMASK2
#define EINT_IMASK_IM3                                              EINT_IMASK_IMASK3
#define EINT_IMASK_IM4                                              EINT_IMASK_IMASK4
#define EINT_IMASK_IM5                                              EINT_IMASK_IMASK5
#define EINT_IMASK_IM6                                              EINT_IMASK_IMASK6
#define EINT_IMASK_IM7                                              EINT_IMASK_IMASK7
#define EINT_IMASK_IM8                                              EINT_IMASK_IMASK8
#define EINT_IMASK_IM9                                              EINT_IMASK_IMASK9
#define EINT_IMASK_IM10                                             EINT_IMASK_IMASK10
#define EINT_IMASK_IM11                                             EINT_IMASK_IMASK11
#define EINT_IMASK_IM12                                             EINT_IMASK_IMASK12
#define EINT_IMASK_IM13                                             EINT_IMASK_IMASK13
#define EINT_IMASK_IM14                                             EINT_IMASK_IMASK14
#define EINT_IMASK_IM15                                             EINT_IMASK_IMASK15
#define EINT_IMASK_IM16                                             EINT_IMASK_IMASK16
#define EINT_IMASK_IM17                                             EINT_IMASK_IMASK17
#define EINT_IMASK_IM18                                             EINT_IMASK_IMASK18
#define EINT_IMASK_IM_Pos                                           (0U)
#define EINT_IMASK_IM_Msk                                           (0x7FFFFUL << EINT_IMASK_IM_Pos)                               /*!< 0x007FFFF */
#define EINT_IMASK_IM                                               EINT_IMASK_IM_Msk                                              /*!< Interrupt Mask All */

/*******************  Bit definition for EINT_EMASK register  *******************/
#define EINT_EMASK_EMASK0_Pos                                       (0U)
#define EINT_EMASK_EMASK0_Msk                                       (0x1UL << EINT_EMASK_EMASK0_Pos)                               /*!< 0x00000001 */
#define EINT_EMASK_EMASK0                                           EINT_EMASK_EMASK0_Msk                                          /*!< Event Mask on line 0 */
#define EINT_EMASK_EMASK1_Pos                                       (1U)
#define EINT_EMASK_EMASK1_Msk                                       (0x1UL << EINT_EMASK_EMASK1_Pos)                               /*!< 0x00000002 */
#define EINT_EMASK_EMASK1                                           EINT_EMASK_EMASK1_Msk                                          /*!< Event Mask on line 1 */
#define EINT_EMASK_EMASK2_Pos                                       (2U)
#define EINT_EMASK_EMASK2_Msk                                       (0x1UL << EINT_EMASK_EMASK2_Pos)                               /*!< 0x00000004 */
#define EINT_EMASK_EMASK2                                           EINT_EMASK_EMASK2_Msk                                          /*!< Event Mask on line 2 */
#define EINT_EMASK_EMASK3_Pos                                       (3U)
#define EINT_EMASK_EMASK3_Msk                                       (0x1UL << EINT_EMASK_EMASK3_Pos)                               /*!< 0x00000008 */
#define EINT_EMASK_EMASK3                                           EINT_EMASK_EMASK3_Msk                                          /*!< Event Mask on line 3 */
#define EINT_EMASK_EMASK4_Pos                                       (4U)
#define EINT_EMASK_EMASK4_Msk                                       (0x1UL << EINT_EMASK_EMASK4_Pos)                               /*!< 0x00000010 */
#define EINT_EMASK_EMASK4                                           EINT_EMASK_EMASK4_Msk                                          /*!< Event Mask on line 4 */
#define EINT_EMASK_EMASK5_Pos                                       (5U)
#define EINT_EMASK_EMASK5_Msk                                       (0x1UL << EINT_EMASK_EMASK5_Pos)                               /*!< 0x00000020 */
#define EINT_EMASK_EMASK5                                           EINT_EMASK_EMASK5_Msk                                          /*!< Event Mask on line 5 */
#define EINT_EMASK_EMASK6_Pos                                       (6U)
#define EINT_EMASK_EMASK6_Msk                                       (0x1UL << EINT_EMASK_EMASK6_Pos)                               /*!< 0x00000040 */
#define EINT_EMASK_EMASK6                                           EINT_EMASK_EMASK6_Msk                                          /*!< Event Mask on line 6 */
#define EINT_EMASK_EMASK7_Pos                                       (7U)
#define EINT_EMASK_EMASK7_Msk                                       (0x1UL << EINT_EMASK_EMASK7_Pos)                               /*!< 0x00000080 */
#define EINT_EMASK_EMASK7                                           EINT_EMASK_EMASK7_Msk                                          /*!< Event Mask on line 7 */
#define EINT_EMASK_EMASK8_Pos                                       (8U)
#define EINT_EMASK_EMASK8_Msk                                       (0x1UL << EINT_EMASK_EMASK8_Pos)                               /*!< 0x00000100 */
#define EINT_EMASK_EMASK8                                           EINT_EMASK_EMASK8_Msk                                          /*!< Event Mask on line 8 */
#define EINT_EMASK_EMASK9_Pos                                       (9U)
#define EINT_EMASK_EMASK9_Msk                                       (0x1UL << EINT_EMASK_EMASK9_Pos)                               /*!< 0x00000200 */
#define EINT_EMASK_EMASK9                                           EINT_EMASK_EMASK9_Msk                                          /*!< Event Mask on line 9 */
#define EINT_EMASK_EMASK10_Pos                                      (10U)
#define EINT_EMASK_EMASK10_Msk                                      (0x1UL << EINT_EMASK_EMASK10_Pos)                              /*!< 0x00000400 */
#define EINT_EMASK_EMASK10                                          EINT_EMASK_EMASK10_Msk                                         /*!< Event Mask on line 10 */
#define EINT_EMASK_EMASK11_Pos                                      (11U)
#define EINT_EMASK_EMASK11_Msk                                      (0x1UL << EINT_EMASK_EMASK11_Pos)                              /*!< 0x00000800 */
#define EINT_EMASK_EMASK11                                          EINT_EMASK_EMASK11_Msk                                         /*!< Event Mask on line 11 */
#define EINT_EMASK_EMASK12_Pos                                      (12U)
#define EINT_EMASK_EMASK12_Msk                                      (0x1UL << EINT_EMASK_EMASK12_Pos)                              /*!< 0x00001000 */
#define EINT_EMASK_EMASK12                                          EINT_EMASK_EMASK12_Msk                                         /*!< Event Mask on line 12 */
#define EINT_EMASK_EMASK13_Pos                                      (13U)
#define EINT_EMASK_EMASK13_Msk                                      (0x1UL << EINT_EMASK_EMASK13_Pos)                              /*!< 0x00002000 */
#define EINT_EMASK_EMASK13                                          EINT_EMASK_EMASK13_Msk                                         /*!< Event Mask on line 13 */
#define EINT_EMASK_EMASK14_Pos                                      (14U)
#define EINT_EMASK_EMASK14_Msk                                      (0x1UL << EINT_EMASK_EMASK14_Pos)                              /*!< 0x00004000 */
#define EINT_EMASK_EMASK14                                          EINT_EMASK_EMASK14_Msk                                         /*!< Event Mask on line 14 */
#define EINT_EMASK_EMASK15_Pos                                      (15U)
#define EINT_EMASK_EMASK15_Msk                                      (0x1UL << EINT_EMASK_EMASK15_Pos)                              /*!< 0x00008000 */
#define EINT_EMASK_EMASK15                                          EINT_EMASK_EMASK15_Msk                                         /*!< Event Mask on line 15 */
#define EINT_EMASK_EMASK16_Pos                                      (16U)
#define EINT_EMASK_EMASK16_Msk                                      (0x1UL << EINT_EMASK_EMASK16_Pos)                              /*!< 0x00010000 */
#define EINT_EMASK_EMASK16                                          EINT_EMASK_EMASK16_Msk                                         /*!< Event Mask on line 16 */
#define EINT_EMASK_EMASK17_Pos                                      (17U)
#define EINT_EMASK_EMASK17_Msk                                      (0x1UL << EINT_EMASK_EMASK17_Pos)                              /*!< 0x00020000 */
#define EINT_EMASK_EMASK17                                          EINT_EMASK_EMASK17_Msk                                         /*!< Event Mask on line 17 */
#define EINT_EMASK_EMASK18_Pos                                      (18U)
#define EINT_EMASK_EMASK18_Msk                                      (0x1UL << EINT_EMASK_EMASK18_Pos)                              /*!< 0x00040000 */
#define EINT_EMASK_EMASK18                                          EINT_EMASK_EMASK18_Msk                                         /*!< Event Mask on line 18 */

/* Reference Defines */
#define EINT_EMASK_EM0                                              EINT_EMASK_EMASK0
#define EINT_EMASK_EM1                                              EINT_EMASK_EMASK1
#define EINT_EMASK_EM2                                              EINT_EMASK_EMASK2
#define EINT_EMASK_EM3                                              EINT_EMASK_EMASK3
#define EINT_EMASK_EM4                                              EINT_EMASK_EMASK4
#define EINT_EMASK_EM5                                              EINT_EMASK_EMASK5
#define EINT_EMASK_EM6                                              EINT_EMASK_EMASK6
#define EINT_EMASK_EM7                                              EINT_EMASK_EMASK7
#define EINT_EMASK_EM8                                              EINT_EMASK_EMASK8
#define EINT_EMASK_EM9                                              EINT_EMASK_EMASK9
#define EINT_EMASK_EM10                                             EINT_EMASK_EMASK10
#define EINT_EMASK_EM11                                             EINT_EMASK_EMASK11
#define EINT_EMASK_EM12                                             EINT_EMASK_EMASK12
#define EINT_EMASK_EM13                                             EINT_EMASK_EMASK13
#define EINT_EMASK_EM14                                             EINT_EMASK_EMASK14
#define EINT_EMASK_EM15                                             EINT_EMASK_EMASK15
#define EINT_EMASK_EM16                                             EINT_EMASK_EMASK16
#define EINT_EMASK_EM17                                             EINT_EMASK_EMASK17
#define EINT_EMASK_EM18                                             EINT_EMASK_EMASK18

/******************  Bit definition for EINT_RTEN register  *******************/
#define EINT_RTEN_RTEN0_Pos                                         (0U)
#define EINT_RTEN_RTEN0_Msk                                         (0x1UL << EINT_RTEN_RTEN0_Pos)                                 /*!< 0x00000001 */
#define EINT_RTEN_RTEN0                                             EINT_RTEN_RTEN0_Msk                                            /*!< Rising trigger event configuration bit of line 0 */
#define EINT_RTEN_RTEN1_Pos                                         (1U)
#define EINT_RTEN_RTEN1_Msk                                         (0x1UL << EINT_RTEN_RTEN1_Pos)                                 /*!< 0x00000002 */
#define EINT_RTEN_RTEN1                                             EINT_RTEN_RTEN1_Msk                                            /*!< Rising trigger event configuration bit of line 1 */
#define EINT_RTEN_RTEN2_Pos                                         (2U)
#define EINT_RTEN_RTEN2_Msk                                         (0x1UL << EINT_RTEN_RTEN2_Pos)                                 /*!< 0x00000004 */
#define EINT_RTEN_RTEN2                                             EINT_RTEN_RTEN2_Msk                                            /*!< Rising trigger event configuration bit of line 2 */
#define EINT_RTEN_RTEN3_Pos                                         (3U)
#define EINT_RTEN_RTEN3_Msk                                         (0x1UL << EINT_RTEN_RTEN3_Pos)                                 /*!< 0x00000008 */
#define EINT_RTEN_RTEN3                                             EINT_RTEN_RTEN3_Msk                                            /*!< Rising trigger event configuration bit of line 3 */
#define EINT_RTEN_RTEN4_Pos                                         (4U)
#define EINT_RTEN_RTEN4_Msk                                         (0x1UL << EINT_RTEN_RTEN4_Pos)                                 /*!< 0x00000010 */
#define EINT_RTEN_RTEN4                                             EINT_RTEN_RTEN4_Msk                                            /*!< Rising trigger event configuration bit of line 4 */
#define EINT_RTEN_RTEN5_Pos                                         (5U)
#define EINT_RTEN_RTEN5_Msk                                         (0x1UL << EINT_RTEN_RTEN5_Pos)                                 /*!< 0x00000020 */
#define EINT_RTEN_RTEN5                                             EINT_RTEN_RTEN5_Msk                                            /*!< Rising trigger event configuration bit of line 5 */
#define EINT_RTEN_RTEN6_Pos                                         (6U)
#define EINT_RTEN_RTEN6_Msk                                         (0x1UL << EINT_RTEN_RTEN6_Pos)                                 /*!< 0x00000040 */
#define EINT_RTEN_RTEN6                                             EINT_RTEN_RTEN6_Msk                                            /*!< Rising trigger event configuration bit of line 6 */
#define EINT_RTEN_RTEN7_Pos                                         (7U)
#define EINT_RTEN_RTEN7_Msk                                         (0x1UL << EINT_RTEN_RTEN7_Pos)                                 /*!< 0x00000080 */
#define EINT_RTEN_RTEN7                                             EINT_RTEN_RTEN7_Msk                                            /*!< Rising trigger event configuration bit of line 7 */
#define EINT_RTEN_RTEN8_Pos                                         (8U)
#define EINT_RTEN_RTEN8_Msk                                         (0x1UL << EINT_RTEN_RTEN8_Pos)                                 /*!< 0x00000100 */
#define EINT_RTEN_RTEN8                                             EINT_RTEN_RTEN8_Msk                                            /*!< Rising trigger event configuration bit of line 8 */
#define EINT_RTEN_RTEN9_Pos                                         (9U)
#define EINT_RTEN_RTEN9_Msk                                         (0x1UL << EINT_RTEN_RTEN9_Pos)                                 /*!< 0x00000200 */
#define EINT_RTEN_RTEN9                                             EINT_RTEN_RTEN9_Msk                                            /*!< Rising trigger event configuration bit of line 9 */
#define EINT_RTEN_RTEN10_Pos                                        (10U)
#define EINT_RTEN_RTEN10_Msk                                        (0x1UL << EINT_RTEN_RTEN10_Pos)                                /*!< 0x00000400 */
#define EINT_RTEN_RTEN10                                            EINT_RTEN_RTEN10_Msk                                           /*!< Rising trigger event configuration bit of line 10 */
#define EINT_RTEN_RTEN11_Pos                                        (11U)
#define EINT_RTEN_RTEN11_Msk                                        (0x1UL << EINT_RTEN_RTEN11_Pos)                                /*!< 0x00000800 */
#define EINT_RTEN_RTEN11                                            EINT_RTEN_RTEN11_Msk                                           /*!< Rising trigger event configuration bit of line 11 */
#define EINT_RTEN_RTEN12_Pos                                        (12U)
#define EINT_RTEN_RTEN12_Msk                                        (0x1UL << EINT_RTEN_RTEN12_Pos)                                /*!< 0x00001000 */
#define EINT_RTEN_RTEN12                                            EINT_RTEN_RTEN12_Msk                                           /*!< Rising trigger event configuration bit of line 12 */
#define EINT_RTEN_RTEN13_Pos                                        (13U)
#define EINT_RTEN_RTEN13_Msk                                        (0x1UL << EINT_RTEN_RTEN13_Pos)                                /*!< 0x00002000 */
#define EINT_RTEN_RTEN13                                            EINT_RTEN_RTEN13_Msk                                           /*!< Rising trigger event configuration bit of line 13 */
#define EINT_RTEN_RTEN14_Pos                                        (14U)
#define EINT_RTEN_RTEN14_Msk                                        (0x1UL << EINT_RTEN_RTEN14_Pos)                                /*!< 0x00004000 */
#define EINT_RTEN_RTEN14                                            EINT_RTEN_RTEN14_Msk                                           /*!< Rising trigger event configuration bit of line 14 */
#define EINT_RTEN_RTEN15_Pos                                        (15U)
#define EINT_RTEN_RTEN15_Msk                                        (0x1UL << EINT_RTEN_RTEN15_Pos)                                /*!< 0x00008000 */
#define EINT_RTEN_RTEN15                                            EINT_RTEN_RTEN15_Msk                                           /*!< Rising trigger event configuration bit of line 15 */
#define EINT_RTEN_RTEN16_Pos                                        (16U)
#define EINT_RTEN_RTEN16_Msk                                        (0x1UL << EINT_RTEN_RTEN16_Pos)                                /*!< 0x00010000 */
#define EINT_RTEN_RTEN16                                            EINT_RTEN_RTEN16_Msk                                           /*!< Rising trigger event configuration bit of line 16 */
#define EINT_RTEN_RTEN17_Pos                                        (17U)
#define EINT_RTEN_RTEN17_Msk                                        (0x1UL << EINT_RTEN_RTEN17_Pos)                                /*!< 0x00020000 */
#define EINT_RTEN_RTEN17                                            EINT_RTEN_RTEN17_Msk                                           /*!< Rising trigger event configuration bit of line 17 */
#define EINT_RTEN_RTEN18_Pos                                        (18U)
#define EINT_RTEN_RTEN18_Msk                                        (0x1UL << EINT_RTEN_RTEN18_Pos)                                /*!< 0x00040000 */
#define EINT_RTEN_RTEN18                                            EINT_RTEN_RTEN18_Msk                                           /*!< Rising trigger event configuration bit of line 18 */

/******************  Bit definition for EINT_FTEN register  *******************/
#define EINT_FTEN_FTEN0_Pos                                         (0U)
#define EINT_FTEN_FTEN0_Msk                                         (0x1UL << EINT_FTEN_FTEN0_Pos)                                 /*!< 0x00000001 */
#define EINT_FTEN_FTEN0                                             EINT_FTEN_FTEN0_Msk                                            /*!< Falling trigger event configuration bit of line 0 */
#define EINT_FTEN_FTEN1_Pos                                         (1U)
#define EINT_FTEN_FTEN1_Msk                                         (0x1UL << EINT_FTEN_FTEN1_Pos)                                 /*!< 0x00000002 */
#define EINT_FTEN_FTEN1                                             EINT_FTEN_FTEN1_Msk                                            /*!< Falling trigger event configuration bit of line 1 */
#define EINT_FTEN_FTEN2_Pos                                         (2U)
#define EINT_FTEN_FTEN2_Msk                                         (0x1UL << EINT_FTEN_FTEN2_Pos)                                 /*!< 0x00000004 */
#define EINT_FTEN_FTEN2                                             EINT_FTEN_FTEN2_Msk                                            /*!< Falling trigger event configuration bit of line 2 */
#define EINT_FTEN_FTEN3_Pos                                         (3U)
#define EINT_FTEN_FTEN3_Msk                                         (0x1UL << EINT_FTEN_FTEN3_Pos)                                 /*!< 0x00000008 */
#define EINT_FTEN_FTEN3                                             EINT_FTEN_FTEN3_Msk                                            /*!< Falling trigger event configuration bit of line 3 */
#define EINT_FTEN_FTEN4_Pos                                         (4U)
#define EINT_FTEN_FTEN4_Msk                                         (0x1UL << EINT_FTEN_FTEN4_Pos)                                 /*!< 0x00000010 */
#define EINT_FTEN_FTEN4                                             EINT_FTEN_FTEN4_Msk                                            /*!< Falling trigger event configuration bit of line 4 */
#define EINT_FTEN_FTEN5_Pos                                         (5U)
#define EINT_FTEN_FTEN5_Msk                                         (0x1UL << EINT_FTEN_FTEN5_Pos)                                 /*!< 0x00000020 */
#define EINT_FTEN_FTEN5                                             EINT_FTEN_FTEN5_Msk                                            /*!< Falling trigger event configuration bit of line 5 */
#define EINT_FTEN_FTEN6_Pos                                         (6U)
#define EINT_FTEN_FTEN6_Msk                                         (0x1UL << EINT_FTEN_FTEN6_Pos)                                 /*!< 0x00000040 */
#define EINT_FTEN_FTEN6                                             EINT_FTEN_FTEN6_Msk                                            /*!< Falling trigger event configuration bit of line 6 */
#define EINT_FTEN_FTEN7_Pos                                         (7U)
#define EINT_FTEN_FTEN7_Msk                                         (0x1UL << EINT_FTEN_FTEN7_Pos)                                 /*!< 0x00000080 */
#define EINT_FTEN_FTEN7                                             EINT_FTEN_FTEN7_Msk                                            /*!< Falling trigger event configuration bit of line 7 */
#define EINT_FTEN_FTEN8_Pos                                         (8U)
#define EINT_FTEN_FTEN8_Msk                                         (0x1UL << EINT_FTEN_FTEN8_Pos)                                 /*!< 0x00000100 */
#define EINT_FTEN_FTEN8                                             EINT_FTEN_FTEN8_Msk                                            /*!< Falling trigger event configuration bit of line 8 */
#define EINT_FTEN_FTEN9_Pos                                         (9U)
#define EINT_FTEN_FTEN9_Msk                                         (0x1UL << EINT_FTEN_FTEN9_Pos)                                 /*!< 0x00000200 */
#define EINT_FTEN_FTEN9                                             EINT_FTEN_FTEN9_Msk                                            /*!< Falling trigger event configuration bit of line 9 */
#define EINT_FTEN_FTEN10_Pos                                        (10U)
#define EINT_FTEN_FTEN10_Msk                                        (0x1UL << EINT_FTEN_FTEN10_Pos)                                /*!< 0x00000400 */
#define EINT_FTEN_FTEN10                                            EINT_FTEN_FTEN10_Msk                                           /*!< Falling trigger event configuration bit of line 10 */
#define EINT_FTEN_FTEN11_Pos                                        (11U)
#define EINT_FTEN_FTEN11_Msk                                        (0x1UL << EINT_FTEN_FTEN11_Pos)                                /*!< 0x00000800 */
#define EINT_FTEN_FTEN11                                            EINT_FTEN_FTEN11_Msk                                           /*!< Falling trigger event configuration bit of line 11 */
#define EINT_FTEN_FTEN12_Pos                                        (12U)
#define EINT_FTEN_FTEN12_Msk                                        (0x1UL << EINT_FTEN_FTEN12_Pos)                                /*!< 0x00001000 */
#define EINT_FTEN_FTEN12                                            EINT_FTEN_FTEN12_Msk                                           /*!< Falling trigger event configuration bit of line 12 */
#define EINT_FTEN_FTEN13_Pos                                        (13U)
#define EINT_FTEN_FTEN13_Msk                                        (0x1UL << EINT_FTEN_FTEN13_Pos)                                /*!< 0x00002000 */
#define EINT_FTEN_FTEN13                                            EINT_FTEN_FTEN13_Msk                                           /*!< Falling trigger event configuration bit of line 13 */
#define EINT_FTEN_FTEN14_Pos                                        (14U)
#define EINT_FTEN_FTEN14_Msk                                        (0x1UL << EINT_FTEN_FTEN14_Pos)                                /*!< 0x00004000 */
#define EINT_FTEN_FTEN14                                            EINT_FTEN_FTEN14_Msk                                           /*!< Falling trigger event configuration bit of line 14 */
#define EINT_FTEN_FTEN15_Pos                                        (15U)
#define EINT_FTEN_FTEN15_Msk                                        (0x1UL << EINT_FTEN_FTEN15_Pos)                                /*!< 0x00008000 */
#define EINT_FTEN_FTEN15                                            EINT_FTEN_FTEN15_Msk                                           /*!< Falling trigger event configuration bit of line 15 */
#define EINT_FTEN_FTEN16_Pos                                        (16U)
#define EINT_FTEN_FTEN16_Msk                                        (0x1UL << EINT_FTEN_FTEN16_Pos)                                /*!< 0x00010000 */
#define EINT_FTEN_FTEN16                                            EINT_FTEN_FTEN16_Msk                                           /*!< Falling trigger event configuration bit of line 16 */
#define EINT_FTEN_FTEN17_Pos                                        (17U)
#define EINT_FTEN_FTEN17_Msk                                        (0x1UL << EINT_FTEN_FTEN17_Pos)                                /*!< 0x00020000 */
#define EINT_FTEN_FTEN17                                            EINT_FTEN_FTEN17_Msk                                           /*!< Falling trigger event configuration bit of line 17 */
#define EINT_FTEN_FTEN18_Pos                                        (18U)
#define EINT_FTEN_FTEN18_Msk                                        (0x1UL << EINT_FTEN_FTEN18_Pos)                                /*!< 0x00040000 */
#define EINT_FTEN_FTEN18                                            EINT_FTEN_FTEN18_Msk                                           /*!< Falling trigger event configuration bit of line 18 */

/******************  Bit definition for EINT_SWINTE register  ******************/
#define EINT_SWINTE_SWINTE0_Pos                                     (0U)
#define EINT_SWINTE_SWINTE0_Msk                                     (0x1UL << EINT_SWINTE_SWINTE0_Pos)                             /*!< 0x00000001 */
#define EINT_SWINTE_SWINTE0                                         EINT_SWINTE_SWINTE0_Msk                                        /*!< Software Interrupt on line 0 */
#define EINT_SWINTE_SWINTE1_Pos                                     (1U)
#define EINT_SWINTE_SWINTE1_Msk                                     (0x1UL << EINT_SWINTE_SWINTE1_Pos)                             /*!< 0x00000002 */
#define EINT_SWINTE_SWINTE1                                         EINT_SWINTE_SWINTE1_Msk                                        /*!< Software Interrupt on line 1 */
#define EINT_SWINTE_SWINTE2_Pos                                     (2U)
#define EINT_SWINTE_SWINTE2_Msk                                     (0x1UL << EINT_SWINTE_SWINTE2_Pos)                             /*!< 0x00000004 */
#define EINT_SWINTE_SWINTE2                                         EINT_SWINTE_SWINTE2_Msk                                        /*!< Software Interrupt on line 2 */
#define EINT_SWINTE_SWINTE3_Pos                                     (3U)
#define EINT_SWINTE_SWINTE3_Msk                                     (0x1UL << EINT_SWINTE_SWINTE3_Pos)                             /*!< 0x00000008 */
#define EINT_SWINTE_SWINTE3                                         EINT_SWINTE_SWINTE3_Msk                                        /*!< Software Interrupt on line 3 */
#define EINT_SWINTE_SWINTE4_Pos                                     (4U)
#define EINT_SWINTE_SWINTE4_Msk                                     (0x1UL << EINT_SWINTE_SWINTE4_Pos)                             /*!< 0x00000010 */
#define EINT_SWINTE_SWINTE4                                         EINT_SWINTE_SWINTE4_Msk                                        /*!< Software Interrupt on line 4 */
#define EINT_SWINTE_SWINTE5_Pos                                     (5U)
#define EINT_SWINTE_SWINTE5_Msk                                     (0x1UL << EINT_SWINTE_SWINTE5_Pos)                             /*!< 0x00000020 */
#define EINT_SWINTE_SWINTE5                                         EINT_SWINTE_SWINTE5_Msk                                        /*!< Software Interrupt on line 5 */
#define EINT_SWINTE_SWINTE6_Pos                                     (6U)
#define EINT_SWINTE_SWINTE6_Msk                                     (0x1UL << EINT_SWINTE_SWINTE6_Pos)                             /*!< 0x00000040 */
#define EINT_SWINTE_SWINTE6                                         EINT_SWINTE_SWINTE6_Msk                                        /*!< Software Interrupt on line 6 */
#define EINT_SWINTE_SWINTE7_Pos                                     (7U)
#define EINT_SWINTE_SWINTE7_Msk                                     (0x1UL << EINT_SWINTE_SWINTE7_Pos)                             /*!< 0x00000080 */
#define EINT_SWINTE_SWINTE7                                         EINT_SWINTE_SWINTE7_Msk                                        /*!< Software Interrupt on line 7 */
#define EINT_SWINTE_SWINTE8_Pos                                     (8U)
#define EINT_SWINTE_SWINTE8_Msk                                     (0x1UL << EINT_SWINTE_SWINTE8_Pos)                             /*!< 0x00000100 */
#define EINT_SWINTE_SWINTE8                                         EINT_SWINTE_SWINTE8_Msk                                        /*!< Software Interrupt on line 8 */
#define EINT_SWINTE_SWINTE9_Pos                                     (9U)
#define EINT_SWINTE_SWINTE9_Msk                                     (0x1UL << EINT_SWINTE_SWINTE9_Pos)                             /*!< 0x00000200 */
#define EINT_SWINTE_SWINTE9                                         EINT_SWINTE_SWINTE9_Msk                                        /*!< Software Interrupt on line 9 */
#define EINT_SWINTE_SWINTE10_Pos                                    (10U)
#define EINT_SWINTE_SWINTE10_Msk                                    (0x1UL << EINT_SWINTE_SWINTE10_Pos)                            /*!< 0x00000400 */
#define EINT_SWINTE_SWINTE10                                        EINT_SWINTE_SWINTE10_Msk                                       /*!< Software Interrupt on line 10 */
#define EINT_SWINTE_SWINTE11_Pos                                    (11U)
#define EINT_SWINTE_SWINTE11_Msk                                    (0x1UL << EINT_SWINTE_SWINTE11_Pos)                            /*!< 0x00000800 */
#define EINT_SWINTE_SWINTE11                                        EINT_SWINTE_SWINTE11_Msk                                       /*!< Software Interrupt on line 11 */
#define EINT_SWINTE_SWINTE12_Pos                                    (12U)
#define EINT_SWINTE_SWINTE12_Msk                                    (0x1UL << EINT_SWINTE_SWINTE12_Pos)                            /*!< 0x00001000 */
#define EINT_SWINTE_SWINTE12                                        EINT_SWINTE_SWINTE12_Msk                                       /*!< Software Interrupt on line 12 */
#define EINT_SWINTE_SWINTE13_Pos                                    (13U)
#define EINT_SWINTE_SWINTE13_Msk                                    (0x1UL << EINT_SWINTE_SWINTE13_Pos)                            /*!< 0x00002000 */
#define EINT_SWINTE_SWINTE13                                        EINT_SWINTE_SWINTE13_Msk                                       /*!< Software Interrupt on line 13 */
#define EINT_SWINTE_SWINTE14_Pos                                    (14U)
#define EINT_SWINTE_SWINTE14_Msk                                    (0x1UL << EINT_SWINTE_SWINTE14_Pos)                            /*!< 0x00004000 */
#define EINT_SWINTE_SWINTE14                                        EINT_SWINTE_SWINTE14_Msk                                       /*!< Software Interrupt on line 14 */
#define EINT_SWINTE_SWINTE15_Pos                                    (15U)
#define EINT_SWINTE_SWINTE15_Msk                                    (0x1UL << EINT_SWINTE_SWINTE15_Pos)                            /*!< 0x00008000 */
#define EINT_SWINTE_SWINTE15                                        EINT_SWINTE_SWINTE15_Msk                                       /*!< Software Interrupt on line 15 */
#define EINT_SWINTE_SWINTE16_Pos                                    (16U)
#define EINT_SWINTE_SWINTE16_Msk                                    (0x1UL << EINT_SWINTE_SWINTE16_Pos)                            /*!< 0x00010000 */
#define EINT_SWINTE_SWINTE16                                        EINT_SWINTE_SWINTE16_Msk                                       /*!< Software Interrupt on line 16 */
#define EINT_SWINTE_SWINTE17_Pos                                    (17U)
#define EINT_SWINTE_SWINTE17_Msk                                    (0x1UL << EINT_SWINTE_SWINTE17_Pos)                            /*!< 0x00020000 */
#define EINT_SWINTE_SWINTE17                                        EINT_SWINTE_SWINTE17_Msk                                       /*!< Software Interrupt on line 17 */
#define EINT_SWINTE_SWINTE18_Pos                                    (18U)
#define EINT_SWINTE_SWINTE18_Msk                                    (0x1UL << EINT_SWINTE_SWINTE18_Pos)                            /*!< 0x00040000 */
#define EINT_SWINTE_SWINTE18                                        EINT_SWINTE_SWINTE18_Msk                                       /*!< Software Interrupt on line 18 */

/*******************  Bit definition for EINT_IPEND register  ********************/
#define EINT_IPEND_IPEND0_Pos                                       (0U)
#define EINT_IPEND_IPEND0_Msk                                       (0x1UL << EINT_IPEND_IPEND0_Pos)                               /*!< 0x00000001 */
#define EINT_IPEND_IPEND0                                           EINT_IPEND_IPEND0_Msk                                          /*!< Pending bit for line 0 */
#define EINT_IPEND_IPEND1_Pos                                       (1U)
#define EINT_IPEND_IPEND1_Msk                                       (0x1UL << EINT_IPEND_IPEND1_Pos)                               /*!< 0x00000002 */
#define EINT_IPEND_IPEND1                                           EINT_IPEND_IPEND1_Msk                                          /*!< Pending bit for line 1 */
#define EINT_IPEND_IPEND2_Pos                                       (2U)
#define EINT_IPEND_IPEND2_Msk                                       (0x1UL << EINT_IPEND_IPEND2_Pos)                               /*!< 0x00000004 */
#define EINT_IPEND_IPEND2                                           EINT_IPEND_IPEND2_Msk                                          /*!< Pending bit for line 2 */
#define EINT_IPEND_IPEND3_Pos                                       (3U)
#define EINT_IPEND_IPEND3_Msk                                       (0x1UL << EINT_IPEND_IPEND3_Pos)                               /*!< 0x00000008 */
#define EINT_IPEND_IPEND3                                           EINT_IPEND_IPEND3_Msk                                          /*!< Pending bit for line 3 */
#define EINT_IPEND_IPEND4_Pos                                       (4U)
#define EINT_IPEND_IPEND4_Msk                                       (0x1UL << EINT_IPEND_IPEND4_Pos)                               /*!< 0x00000010 */
#define EINT_IPEND_IPEND4                                           EINT_IPEND_IPEND4_Msk                                          /*!< Pending bit for line 4 */
#define EINT_IPEND_IPEND5_Pos                                       (5U)
#define EINT_IPEND_IPEND5_Msk                                       (0x1UL << EINT_IPEND_IPEND5_Pos)                               /*!< 0x00000020 */
#define EINT_IPEND_IPEND5                                           EINT_IPEND_IPEND5_Msk                                          /*!< Pending bit for line 5 */
#define EINT_IPEND_IPEND6_Pos                                       (6U)
#define EINT_IPEND_IPEND6_Msk                                       (0x1UL << EINT_IPEND_IPEND6_Pos)                               /*!< 0x00000040 */
#define EINT_IPEND_IPEND6                                           EINT_IPEND_IPEND6_Msk                                          /*!< Pending bit for line 6 */
#define EINT_IPEND_IPEND7_Pos                                       (7U)
#define EINT_IPEND_IPEND7_Msk                                       (0x1UL << EINT_IPEND_IPEND7_Pos)                               /*!< 0x00000080 */
#define EINT_IPEND_IPEND7                                           EINT_IPEND_IPEND7_Msk                                          /*!< Pending bit for line 7 */
#define EINT_IPEND_IPEND8_Pos                                       (8U)
#define EINT_IPEND_IPEND8_Msk                                       (0x1UL << EINT_IPEND_IPEND8_Pos)                               /*!< 0x00000100 */
#define EINT_IPEND_IPEND8                                           EINT_IPEND_IPEND8_Msk                                          /*!< Pending bit for line 8 */
#define EINT_IPEND_IPEND9_Pos                                       (9U)
#define EINT_IPEND_IPEND9_Msk                                       (0x1UL << EINT_IPEND_IPEND9_Pos)                               /*!< 0x00000200 */
#define EINT_IPEND_IPEND9                                           EINT_IPEND_IPEND9_Msk                                          /*!< Pending bit for line 9 */
#define EINT_IPEND_IPEND10_Pos                                      (10U)
#define EINT_IPEND_IPEND10_Msk                                      (0x1UL << EINT_IPEND_IPEND10_Pos)                              /*!< 0x00000400 */
#define EINT_IPEND_IPEND10                                          EINT_IPEND_IPEND10_Msk                                         /*!< Pending bit for line 10 */
#define EINT_IPEND_IPEND11_Pos                                      (11U)
#define EINT_IPEND_IPEND11_Msk                                      (0x1UL << EINT_IPEND_IPEND11_Pos)                              /*!< 0x00000800 */
#define EINT_IPEND_IPEND11                                          EINT_IPEND_IPEND11_Msk                                         /*!< Pending bit for line 11 */
#define EINT_IPEND_IPEND12_Pos                                      (12U)
#define EINT_IPEND_IPEND12_Msk                                      (0x1UL << EINT_IPEND_IPEND12_Pos)                              /*!< 0x00001000 */
#define EINT_IPEND_IPEND12                                          EINT_IPEND_IPEND12_Msk                                         /*!< Pending bit for line 12 */
#define EINT_IPEND_IPEND13_Pos                                      (13U)
#define EINT_IPEND_IPEND13_Msk                                      (0x1UL << EINT_IPEND_IPEND13_Pos)                              /*!< 0x00002000 */
#define EINT_IPEND_IPEND13                                          EINT_IPEND_IPEND13_Msk                                         /*!< Pending bit for line 13 */
#define EINT_IPEND_IPEND14_Pos                                      (14U)
#define EINT_IPEND_IPEND14_Msk                                      (0x1UL << EINT_IPEND_IPEND14_Pos)                              /*!< 0x00004000 */
#define EINT_IPEND_IPEND14                                          EINT_IPEND_IPEND14_Msk                                         /*!< Pending bit for line 14 */
#define EINT_IPEND_IPEND15_Pos                                      (15U)
#define EINT_IPEND_IPEND15_Msk                                      (0x1UL << EINT_IPEND_IPEND15_Pos)                              /*!< 0x00008000 */
#define EINT_IPEND_IPEND15                                          EINT_IPEND_IPEND15_Msk                                         /*!< Pending bit for line 15 */
#define EINT_IPEND_IPEND16_Pos                                      (16U)
#define EINT_IPEND_IPEND16_Msk                                      (0x1UL << EINT_IPEND_IPEND16_Pos)                              /*!< 0x00010000 */
#define EINT_IPEND_IPEND16                                          EINT_IPEND_IPEND16_Msk                                         /*!< Pending bit for line 16 */
#define EINT_IPEND_IPEND17_Pos                                      (17U)
#define EINT_IPEND_IPEND17_Msk                                      (0x1UL << EINT_IPEND_IPEND17_Pos)                              /*!< 0x00020000 */
#define EINT_IPEND_IPEND17                                          EINT_IPEND_IPEND17_Msk                                         /*!< Pending bit for line 17 */
#define EINT_IPEND_IPEND18_Pos                                      (18U)
#define EINT_IPEND_IPEND18_Msk                                      (0x1UL << EINT_IPEND_IPEND18_Pos)                              /*!< 0x00040000 */
#define EINT_IPEND_IPEND18                                          EINT_IPEND_IPEND18_Msk                                         /*!< Pending bit for line 18 */

/******************************************************************************/
/*                                                                            */
/*                     FLASH and Option Bytes Registers                       */
/*                                                                            */
/******************************************************************************/
/*******************  Bits definition for FMC_CTRL1 register  *******************/
#define FLASH_CTRL1_WS02_Pos                                        (0U)
#define FLASH_CTRL1_WS02_Msk                                        (0x7UL << FLASH_CTRL1_WS02_Pos)                                /*!< 0x00000007 */
#define FLASH_CTRL1_WS02                                            FLASH_CTRL1_WS02_Msk                                           /*!<WS[2:0] bits (Latency) */
#define FLASH_CTRL1_WS34_Pos                                        (6U)
#define FLASH_CTRL1_WS34_Msk                                        (0x3UL << FLASH_CTRL1_WS34_Pos)                                /*!< 0x000000C0 */
#define FLASH_CTRL1_WS34                                            FLASH_CTRL1_WS34_Msk                                           /*!<WS[4:3] bits (Latency) */
#define FLASH_CTRL1_WS_0                                            (0x1UL << FLASH_CTRL1_WS02_Pos)                                /*!< 0x00000001 */
#define FLASH_CTRL1_WS_1                                            (0x2UL << FLASH_CTRL1_WS02_Pos)                                /*!< 0x00000002 */
#define FLASH_CTRL1_WS_2                                            (0x4UL << FLASH_CTRL1_WS02_Pos)                                /*!< 0x00000004 */
#define FLASH_CTRL1_WS_3                                            (0x1UL << FLASH_CTRL1_WS34_Pos)                                /*!< 0x00000040 */
#define FLASH_CTRL1_WS_4                                            (0x2UL << FLASH_CTRL1_WS34_Pos)                                /*!< 0x00000080 */
#define FLASH_CTRL_WS_0WS                                           0x00000000U
#define FLASH_CTRL_WS_1WS                                           0x00000001U
#define FLASH_CTRL_WS_2WS                                           0x00000002U
#define FLASH_CTRL_WS_3WS                                           0x00000003U
#define FLASH_CTRL_WS_4WS                                           0x00000004U
#define FLASH_CTRL_WS_5WS                                           0x00000005U
#define FLASH_CTRL_WS_6WS                                           0x00000006U
#define FLASH_CTRL_WS_7WS                                           0x00000007U
#define FLASH_CTRL_WS_8WS                                           0x00000040U
#define FLASH_CTRL_WS_9WS                                           0x00000041U
#define FLASH_CTRL_WS_10WS                                          0x00000042U
#define FLASH_CTRL_WS_11WS                                          0x00000043U
#define FLASH_CTRL_WS_12WS                                          0x00000044U
#define FLASH_CTRL_WS_13WS                                          0x00000045U
#define FLASH_CTRL_WS_14WS                                          0x00000046U
#define FLASH_CTRL_WS_15WS                                          0x00000047U
#define FLASH_CTRL_WS_16WS                                          0x00000080U
#define FLASH_CTRL_WS_17WS                                          0x00000081U
#define FLASH_CTRL_WS_18WS                                          0x00000082U
#define FLASH_CTRL_WS_19WS                                          0x00000083U
#define FLASH_CTRL_WS_20WS                                          0x00000084U
#define FLASH_CTRL_WS_21WS                                          0x00000085U
#define FLASH_CTRL_WS_22WS                                          0x00000086U
#define FLASH_CTRL_WS_23WS                                          0x00000087U
#define FLASH_CTRL_WS_24WS                                          0x000000C0U
#define FLASH_CTRL_WS_25WS                                          0x000000C1U
#define FLASH_CTRL_WS_26WS                                          0x000000C2U
#define FLASH_CTRL_WS_27WS                                          0x000000C3U
#define FLASH_CTRL_WS_28WS                                          0x000000C4U
#define FLASH_CTRL_WS_29WS                                          0x000000C5U
#define FLASH_CTRL_WS_30WS                                          0x000000C6U
#define FLASH_CTRL_WS_31WS                                          0x000000C7U
#define FLASH_CTRL1_HCAEN_Pos                                       (3U)
#define FLASH_CTRL1_HCAEN_Msk                                       (0x1UL << FLASH_CTRL1_HCAEN_Pos)                               /*!< 0x00000008 */
#define FLASH_CTRL1_HCAEN                                           FLASH_CTRL1_HCAEN_Msk                                          /*!<Flash half cycle access enable */
#define FLASH_CTRL1_PBEN_Pos                                        (4U)
#define FLASH_CTRL1_PBEN_Msk                                        (0x1UL << FLASH_CTRL1_PBEN_Pos)                                /*!< 0x00000010 */
#define FLASH_CTRL1_PBEN                                            FLASH_CTRL1_PBEN_Msk                                           /*!<Prefetch buffer enable */
#define FLASH_CTRL1_PBSF_Pos                                        (5U)
#define FLASH_CTRL1_PBSF_Msk                                        (0x1UL << FLASH_CTRL1_PBSF_Pos)                                /*!< 0x00000020 */
#define FLASH_CTRL1_PBSF                                            FLASH_CTRL1_PBSF_Msk                                           /*!<Prefetch buffer status */
#define FLASH_CTRL1_ICACHEEN_Pos                                    (8)
#define FLASH_CTRL1_ICACHEEN_Msk                                    (0x1UL << FLASH_CTRL1_ICACHEEN_Pos)                            /*!< 0x00000100 */
#define FLASH_CTRL1_ICACHEEN                                        FLASH_CTRL1_ICACHEEN_Msk                                       /*!<Instruction cache enable */
#define FLASH_CTRL1_DCACHEEN_Pos                                    (9U)
#define FLASH_CTRL1_DCACHEEN_Msk                                    (0x1UL << FLASH_CTRL1_DCACHEEN_Pos)                            /*!< 0x00000200 */
#define FLASH_CTRL1_DCACHEEN                                        FLASH_CTRL1_DCACHEEN_Msk                                       /*!<Data cache enable */
#define FLASH_CTRL1_ICACHERST_Pos                                   (10U)
#define FLASH_CTRL1_ICACHERST_Msk                                   (0x1UL << FLASH_CTRL1_ICACHERST_Pos)                           /*!< 0x00000400 */
#define FLASH_CTRL1_ICACHERST                                       FLASH_CTRL1_ICACHERST_Msk                                      /*!<Instruction cache reset */
#define FLASH_CTRL1_DCACHERST_Pos                                   (11U)
#define FLASH_CTRL1_DCACHERST_Msk                                   (0x1UL << FLASH_CTRL1_DCACHERST_Pos)                           /*!< 0x00000800 */
#define FLASH_CTRL1_DCACHERST                                       FLASH_CTRL1_DCACHERST_Msk                                      /*!<Data cache reset */
#define FLASH_CTRL1_PRFTB_Pos                                       (15)
#define FLASH_CTRL1_PRFTB_Msk                                       (0x1UL << FLASH_CTRL1_PRFTB_Pos)                               /*!< 0x00008000 */
#define FLASH_CTRL1_PRFTB                                           FLASH_CTRL1_PRFTB_Msk                                          /*!<Prefetch policy control */

/********************  Bit definition for FMC_KEY register  ******************/
#define FLASH_KEY_KEY_Pos                                           (0U)
#define FLASH_KEY_KEY_Msk                                           (0xFFFFFFFFUL << FLASH_KEY_KEY_Pos)                            /*!< 0xFFFFFFFF */
#define FLASH_KEY_KEY                                               FLASH_KEY_KEY_Msk                                              /*!<FPEC key */

/*******************  Bit definition for FMC_OPTKEYR register  ****************/
#define FLASH_OBKEY_OBKEY_Pos                                       (0U)
#define FLASH_OBKEY_OBKEY_Msk                                       (0xFFFFFFFFUL << FLASH_OBKEY_OBKEY_Pos)                        /*!< 0xFFFFFFFF */
#define FLASH_OBKEY_OBKEY                                           FLASH_OBKEY_OBKEY_Msk                                          /*!< Option Byte Key */

/*******************  Bits definition for FMC_STS register  *******************/
#define FLASH_STS_BUSYF_Pos                                         (0U)
#define FLASH_STS_BUSYF_Msk                                         (0x1UL << FLASH_STS_BUSYF_Pos)                                 /*!< 0x00000001 */
#define FLASH_STS_BUSYF                                             FLASH_STS_BUSYF_Msk                                            /*!<Busy flag */
#define FLASH_STS_PEF_Pos                                           (2U)
#define FLASH_STS_PEF_Msk                                           (0x1UL << FLASH_STS_PEF_Pos)                                   /*!< 0x00000004 */
#define FLASH_STS_PEF                                               FLASH_STS_PEF_Msk                                              /*!<Programming error flag */
#define FLASH_STS_WPEF_Pos                                          (4U)
#define FLASH_STS_WPEF_Msk                                          (0x1UL << FLASH_STS_WPEF_Pos)                                  /*!< 0x00000010 */
#define FLASH_STS_WPEF                                              FLASH_STS_WPEF_Msk                                             /*!<Write protection error flag */
#define FLASH_STS_OCF_Pos                                           (5U)
#define FLASH_STS_OCF_Msk                                           (0x1UL << FLASH_STS_OCF_Pos)                                   /*!< 0x00000020 */
#define FLASH_STS_OCF                                               FLASH_STS_OCF_Msk                                              /*!<Operation completed flag */

/*******************  Bits definition for FMC_CTRL2 register  *******************/
#define FLASH_CTRL2_PG_Pos                                          (0U)
#define FLASH_CTRL2_PG_Msk                                          (0x1UL << FLASH_CTRL2_PG_Pos)                                  /*!< 0x00000001 */
#define FLASH_CTRL2_PG                                              FLASH_CTRL2_PG_Msk                                             /*!<Programming */
#define FLASH_CTRL2_PAGEERA_Pos                                     (1U)
#define FLASH_CTRL2_PAGEERA_Msk                                     (0x1UL << FLASH_CTRL2_PAGEERA_Pos)                             /*!< 0x00000002 */
#define FLASH_CTRL2_PAGEERA                                         FLASH_CTRL2_PAGEERA_Msk                                        /*!<Page erase */
#define FLASH_CTRL2_MASSERA_Pos                                     (2U)
#define FLASH_CTRL2_MASSERA_Msk                                     (0x1UL << FLASH_CTRL2_MASSERA_Pos)                             /*!< 0x00000004 */
#define FLASH_CTRL2_MASSERA                                         FLASH_CTRL2_MASSERA_Msk                                        /*!<Mass erase */
#define FLASH_CTRL2_OBP_Pos                                         (4U)
#define FLASH_CTRL2_OBP_Msk                                         (0x1UL << FLASH_CTRL2_OBP_Pos)                                 /*!< 0x00000010 */
#define FLASH_CTRL2_OBP                                             FLASH_CTRL2_OBP_Msk                                            /*!<Option byte programming */
#define FLASH_CTRL2_OBE_Pos                                         (5U)
#define FLASH_CTRL2_OBE_Msk                                         (0x1UL << FLASH_CTRL2_OBE_Pos)                                 /*!< 0x00000020 */
#define FLASH_CTRL2_OBE                                             FLASH_CTRL2_OBE_Msk                                            /*!<Option byte erase */
#define FLASH_CTRL2_STA_Pos                                         (6U)
#define FLASH_CTRL2_STA_Msk                                         (0x1UL << FLASH_CTRL2_STA_Pos)                                 /*!< 0x00000040 */
#define FLASH_CTRL2_STA                                             FLASH_CTRL2_STA_Msk                                            /*!<Start */
#define FLASH_CTRL2_LOCK_Pos                                        (7U)
#define FLASH_CTRL2_LOCK_Msk                                        (0x1UL << FLASH_CTRL2_LOCK_Pos)                                /*!< 0x00000080 */
#define FLASH_CTRL2_LOCK                                            FLASH_CTRL2_LOCK_Msk                                           /*!<Lock */
#define FLASH_CTRL2_OBWEN_Pos                                       (9)
#define FLASH_CTRL2_OBWEN_Msk                                       (0x1UL << FLASH_CTRL2_OBWEN_Pos)                               /*!< 0x00000200 */
#define FLASH_CTRL2_OBWEN                                           FLASH_CTRL2_OBWEN_Msk                                          /*!<Option byte write enable */
#define FLASH_CTRL2_ERRIE_Pos                                       (10U)
#define FLASH_CTRL2_ERRIE_Msk                                       (0x1UL << FLASH_CTRL2_ERRIE_Pos)                               /*!< 0x00000400 */
#define FLASH_CTRL2_ERRIE                                           FLASH_CTRL2_ERRIE_Msk                                          /*!<Error interrupt enable */
#define FLASH_CTRL2_OCIE_Pos                                        (12U)
#define FLASH_CTRL2_OCIE_Msk                                        (0x1UL << FLASH_CTRL2_OCIE_Pos)                                /*!< 0x00001000 */
#define FLASH_CTRL2_OCIE                                            FLASH_CTRL2_OCIE_Msk                                           /*!<Operation completed interrupt enable */

/*******************  Bits definition for FMC_ADDR register  *******************/
#define FLASH_ADDR_ADDR_Pos                                        (0U)
#define FLASH_ADDR_ADDR_Msk                                        (0xFFFFFFFFUL << FLASH_ADDR_ADDR_Pos)                           /*!< 0xFFFFFFFF */
#define FLASH_ADDR_ADDR                                            FLASH_ADDR_ADDR_Msk                                             /*!<Flash address */

/*******************  Bits definition for FMC_OBCS register  *******************/
#define FLASH_OBCS_OBE_Pos                                          (0U)
#define FLASH_OBCS_OBE_Msk                                          (0x1UL << FLASH_OBCS_OBE_Pos)                                  /*!< 0x00000001 */
#define FLASH_OBCS_OBE                                              FLASH_OBCS_OBE_Msk                                             /*!<Option byte error */
#define FLASH_OBCS_READPROT_Pos                                     (1U)
#define FLASH_OBCS_READPROT_Msk                                     (0x1UL << FLASH_OBCS_READPROT_Pos)                             /*!< 0x00000002 */
#define FLASH_OBCS_READPROT                                         FLASH_OBCS_READPROT_Msk                                        /*!<Read protection */
#define FLASH_OBCS_WDT_SEL_Pos                                      (2U)
#define FLASH_OBCS_WDT_SEL_Msk                                      (0x1UL << FLASH_OBCS_WDT_SEL_Pos)                              /*!< 0x00000004 */
#define FLASH_OBCS_WDT_SEL                                          FLASH_OBCS_WDT_SEL_Msk                                         /*!<IWDT select */
#define FLASH_OBCS_nRST_STOP_Pos                                    (3U)
#define FLASH_OBCS_nRST_STOP_Msk                                    (0x1UL << FLASH_OBCS_nRST_STOP_Pos)                            /*!< 0x00000008 */
#define FLASH_OBCS_nRST_STOP                                        FLASH_OBCS_nRST_STOP_Msk                                       /*!<nRST_STOP */
#define FLASH_OBCS_nRST_STDBY_Pos                                   (4U)
#define FLASH_OBCS_nRST_STDBY_Msk                                   (0x1UL << FLASH_OBCS_nRST_STDBY_Pos)                           /*!< 0x00000010 */
#define FLASH_OBCS_nRST_STDBY                                       FLASH_OBCS_nRST_STDBY_Msk                                      /*!<nRST_STDBY */
#define FLASH_OBCS_nROM_SEL_Pos                                     (6)
#define FLASH_OBCS_nROM_SEL_Msk                                     (0x1UL << FLASH_OBCS_nROM_SEL_Pos)                             /*!< 0x00000040 */
#define FLASH_OBCS_nROM_SEL                                         FLASH_OBCS_nROM_SEL_Msk                                        /*!<nROM_SEL */
#define FLASH_OBCS_UOB_Pos                                          (2U)
#define FLASH_OBCS_UOB_Msk                                          (0x17UL << FLASH_OBCS_UOB_Pos)                                 /*!< 0x0000005C */
#define FLASH_OBCS_UOB                                              FLASH_OBCS_UOB_Msk                                             /*!<User option byte */
#define FLASH_OBCS_DATA0_Pos                                        (10U)
#define FLASH_OBCS_DATA0_Msk                                        (0xFFUL << FLASH_OBCS_DATA0_Pos)                               /*!< 0x0003FC00 */
#define FLASH_OBCS_DATA0                                            FLASH_OBCS_DATA0_Msk                                           /*!<User data 0 */
#define FLASH_OBCS_DATA1_Pos                                        (18U)
#define FLASH_OBCS_DATA1_Msk                                        (0xFFUL << FLASH_OBCS_DATA1_Pos)                               /*!< 0x03FC0000 */
#define FLASH_OBCS_DATA1                                            FLASH_OBCS_DATA1_Msk                                           /*!<User data 1 */

/*******************  Bits definition for FMC_WRTPROT register  *******************/
#define FLASH_WRTPROT_WRTPROT_Pos                                   (0U)
#define FLASH_WRTPROT_WRTPROT_Msk                                   (0xFFFFFFFFUL << FLASH_WRTPROT_WRTPROT_Pos)                    /*!< 0xFFFFFFFF */
#define FLASH_WRTPROT_WRTPROT                                       FLASH_WRTPROT_WRTPROT_Msk                                      /*!<Write protection */

/*******************  Bits definition for FMC_READPROT register  *******************/
#define FLASH_READPROT_READPROT_Pos                                 (0U)
#define FLASH_READPROT_READPROT_Msk                                 (0xFFUL << FLASH_READPROT_READPROT_Pos)                        /*!< 0x000000FF */
#define FLASH_READPROT_READPROT                                     FLASH_READPROT_READPROT_Msk                                    /*!<Read protection option byte */
#define FLASH_READPROT_nREADPROT_Pos                                (8U)
#define FLASH_READPROT_nREADPROT_Msk                                (0xFFUL << FLASH_READPROT_nREADPROT_Pos)                       /*!< 0x0000FF00 */
#define FLASH_READPROT_nREADPROT                                    FLASH_READPROT_nREADPROT_Msk                                   /*!<Read protection complemented option byte */

/*******************  Bits definition for FMC_UOB register  *******************/
#define FLASH_UOB_UOB_Pos                                           (16U)
#define FLASH_UOB_UOB_Msk                                           (0xFFUL << FLASH_UOB_UOB_Pos)                                  /*!< 0x00FF0000 */
#define FLASH_UOB_UOB                                               FLASH_UOB_UOB_Msk                                              /*!<User option byte */
#define FLASH_UOB_nUOB_Pos                                          (24U)
#define FLASH_UOB_nUOB_Msk                                          (0xFFUL << FLASH_UOB_nUOB_Pos)                                 /*!< 0xFF000000 */
#define FLASH_UOB_nUOB                                              FLASH_UOB_nUOB_Msk                                             /*!<User option byte complemented */

/*******************  Bits definition for FMC_DATA0 register  *******************/
#define FLASH_DATA0_DATA0_Pos                                       (0U)
#define FLASH_DATA0_DATA0_Msk                                       (0xFFUL << FLASH_DATA0_DATA0_Pos)                              /*!< 0x000000FF */
#define FLASH_DATA0_DATA0                                           FLASH_DATA0_DATA0_Msk                                          /*!<User data storage option byte */
#define FLASH_DATA0_nDATA0_Pos                                      (8U)
#define FLASH_DATA0_nDATA0_Msk                                      (0xFFUL << FLASH_DATA0_nDATA0_Pos)                             /*!< 0x0000FF00 */
#define FLASH_DATA0_nDATA0                                          FLASH_DATA0_nDATA0_Msk                                         /*!<User data storage complemented option byte */

/*******************  Bits definition for FMC_DATA1 register  *******************/
#define FLASH_DATA1_DATA1_Pos                                       (16U)
#define FLASH_DATA1_DATA1_Msk                                       (0xFFUL << FLASH_DATA1_DATA1_Pos)                              /*!< 0x00FF0000 */
#define FLASH_DATA1_DATA1                                           FLASH_DATA1_DATA1_Msk                                          /*!<User data storage option byte */
#define FLASH_DATA1_nDATA1_Pos                                      (24U)
#define FLASH_DATA1_nDATA1_Msk                                      (0xFFUL << FLASH_DATA1_nDATA1_Pos)                             /*!< 0xFF000000 */
#define FLASH_DATA1_nDATA1                                          FLASH_DATA1_nDATA1_Msk                                         /*!<User data storage complemented option byte */

/*******************  Bits definition for FMC_WRP0 register  *******************/
#define FLASH_WRP0_WRP0_Pos                                         (0U)
#define FLASH_WRP0_WRP0_Msk                                         (0xFFUL << FLASH_WRP0_WRP0_Pos)                                /*!< 0x000000FF */
#define FLASH_WRP0_WRP0                                             FLASH_WRP0_WRP0_Msk                                            /*!<Flash memory write protection option byte */
#define FLASH_WRP0_nWRP0_Pos                                        (8U)
#define FLASH_WRP0_nWRP0_Msk                                        (0xFFUL << FLASH_WRP0_nWRP0_Pos)                               /*!< 0x0000FF00 */
#define FLASH_WRP0_nWRP0                                            FLASH_WRP0_nWRP0_Msk                                           /*!<Flash memory write protection complemented option byte */

/*******************  Bits definition for FMC_WRP1 register  *******************/
#define FLASH_WRP1_WRP1_Pos                                         (16U)
#define FLASH_WRP1_WRP1_Msk                                         (0xFFUL << FLASH_WRP1_WRP1_Pos)                                /*!< 0x00FF0000 */
#define FLASH_WRP1_WRP1                                             FLASH_WRP1_WRP1_Msk                                            /*!<Flash memory write protection option byte */
#define FLASH_WRP1_nWRP1_Pos                                        (24U)
#define FLASH_WRP1_nWRP1_Msk                                        (0xFFUL << FLASH_WRP1_nWRP1_Pos)                               /*!< 0xFF000000 */
#define FLASH_WRP1_nWRP1                                            FLASH_WRP1_nWRP1_Msk                                           /*!<Flash memory write protection complemented option byte */

/*******************  Bits definition for FMC_WRP2 register  *******************/
#define FLASH_WRP2_WRP2_Pos                                         (0U)
#define FLASH_WRP2_WRP2_Msk                                         (0xFFUL << FLASH_WRP2_WRP2_Pos)                                /*!< 0x000000FF */
#define FLASH_WRP2_WRP2                                             FLASH_WRP2_WRP2_Msk                                            /*!<Flash memory write protection option byte */
#define FLASH_WRP2_nWRP2_Pos                                        (8U)
#define FLASH_WRP2_nWRP2_Msk                                        (0xFFUL << FLASH_WRP2_nWRP2_Pos)                               /*!< 0x0000FF00 */
#define FLASH_WRP2_nWRP2                                            FLASH_WRP2_nWRP2_Msk                                           /*!<Flash memory write protection complemented option byte */

/*******************  Bits definition for FMC_WRP3 register  *******************/
#define FLASH_WRP3_WRP3_Pos                                         (16U)
#define FLASH_WRP3_WRP3_Msk                                         (0xFFUL << FLASH_WRP3_WRP3_Pos)                                /*!< 0x00FF0000 */
#define FLASH_WRP3_WRP3                                             FLASH_WRP3_WRP3_Msk                                            /*!<Flash memory write protection option byte */
#define FLASH_WRP3_nWRP3_Pos                                        (24U)
#define FLASH_WRP3_nWRP3_Msk                                        (0xFFUL << FLASH_WRP3_nWRP3_Pos)                               /*!< 0xFF000000 */
#define FLASH_WRP3_nWRP3                                            FLASH_WRP3_nWRP3_Msk                                           /*!<Flash memory write protection complemented option byte */

/******************************************************************************/
/*                                                                            */
/*                            General Purpose I/O                             */
/*                                                                            */
/******************************************************************************/
/******************  Bits definition for GPIO_CFGLOW register  *****************/
#define GPIO_CFGLOW_MODE_Pos                                        (0U)
#define GPIO_CFGLOW_MODE_Msk                                        (0x33333333UL << GPIO_CFGLOW_MODE_Pos)                        /*!< 0x33333333 */
#define GPIO_CFGLOW_MODE                                            GPIO_CFGLOW_MODE_Msk                                           /*!<Port x mode bits */

#define GPIO_CFGLOW_MODE0_Pos                                       (0U)
#define GPIO_CFGLOW_MODE0_Msk                                       (0x3UL << GPIO_CFGLOW_MODE0_Pos)                               /*!< 0x00000003 */
#define GPIO_CFGLOW_MODE0                                           GPIO_CFGLOW_MODE0_Msk                                          /*!<MODE0[1:0] bits (Port x mode bits, pin 0) */
#define GPIO_CFGLOW_MODE0_0                                         (0x1UL << GPIO_CFGLOW_MODE0_Pos)                               /*!< 0x00000001 */
#define GPIO_CFGLOW_MODE0_1                                         (0x2UL << GPIO_CFGLOW_MODE0_Pos)                               /*!< 0x00000002 */

#define GPIO_CFGLOW_MODE1_Pos                                       (4U)
#define GPIO_CFGLOW_MODE1_Msk                                       (0x3UL << GPIO_CFGLOW_MODE1_Pos)                               /*!< 0x00000030 */
#define GPIO_CFGLOW_MODE1                                           GPIO_CFGLOW_MODE1_Msk                                          /*!<MODE1[1:0] bits (Port x mode bits, pin 1) */
#define GPIO_CFGLOW_MODE1_0                                         (0x1UL << GPIO_CFGLOW_MODE1_Pos)                               /*!< 0x00000010 */
#define GPIO_CFGLOW_MODE1_1                                         (0x2UL << GPIO_CFGLOW_MODE1_Pos)                               /*!< 0x00000020 */

#define GPIO_CFGLOW_MODE2_Pos                                       (8U)
#define GPIO_CFGLOW_MODE2_Msk                                       (0x3UL << GPIO_CFGLOW_MODE2_Pos)                               /*!< 0x00000300 */
#define GPIO_CFGLOW_MODE2                                           GPIO_CFGLOW_MODE2_Msk                                          /*!<MODE2[1:0] bits (Port x mode bits, pin 2) */
#define GPIO_CFGLOW_MODE2_0                                         (0x1UL << GPIO_CFGLOW_MODE2_Pos)                               /*!< 0x00000100 */
#define GPIO_CFGLOW_MODE2_1                                         (0x2UL << GPIO_CFGLOW_MODE2_Pos)                               /*!< 0x00000200 */

#define GPIO_CFGLOW_MODE3_Pos                                       (12U)
#define GPIO_CFGLOW_MODE3_Msk                                       (0x3UL << GPIO_CFGLOW_MODE3_Pos)                               /*!< 0x00003000 */
#define GPIO_CFGLOW_MODE3                                           GPIO_CFGLOW_MODE3_Msk                                          /*!<MODE3[1:0] bits (Port x mode bits, pin 3) */
#define GPIO_CFGLOW_MODE3_0                                         (0x1UL << GPIO_CFGLOW_MODE3_Pos)                               /*!< 0x00001000 */
#define GPIO_CFGLOW_MODE3_1                                         (0x2UL << GPIO_CFGLOW_MODE3_Pos)                               /*!< 0x00002000 */

#define GPIO_CFGLOW_MODE4_Pos                                       (16U)
#define GPIO_CFGLOW_MODE4_Msk                                       (0x3UL << GPIO_CFGLOW_MODE4_Pos)                               /*!< 0x00030000 */
#define GPIO_CFGLOW_MODE4                                           GPIO_CFGLOW_MODE4_Msk                                          /*!<MODE4[1:0] bits (Port x mode bits, pin 4) */
#define GPIO_CFGLOW_MODE4_0                                         (0x1UL << GPIO_CFGLOW_MODE4_Pos)                               /*!< 0x00010000 */
#define GPIO_CFGLOW_MODE4_1                                         (0x2UL << GPIO_CFGLOW_MODE4_Pos)                               /*!< 0x00020000 */

#define GPIO_CFGLOW_MODE5_Pos                                       (20U)
#define GPIO_CFGLOW_MODE5_Msk                                       (0x3UL << GPIO_CFGLOW_MODE5_Pos)                               /*!< 0x00300000 */
#define GPIO_CFGLOW_MODE5                                           GPIO_CFGLOW_MODE5_Msk                                          /*!<MODE5[1:0] bits (Port x mode bits, pin 5) */
#define GPIO_CFGLOW_MODE5_0                                         (0x1UL << GPIO_CFGLOW_MODE5_Pos)                               /*!< 0x00100000 */
#define GPIO_CFGLOW_MODE5_1                                         (0x2UL << GPIO_CFGLOW_MODE5_Pos)                               /*!< 0x00200000 */

#define GPIO_CFGLOW_MODE6_Pos                                       (24U)
#define GPIO_CFGLOW_MODE6_Msk                                       (0x3UL << GPIO_CFGLOW_MODE6_Pos)                               /*!< 0x03000000 */
#define GPIO_CFGLOW_MODE6                                           GPIO_CFGLOW_MODE6_Msk                                          /*!<MODE6[1:0] bits (Port x mode bits, pin 6) */
#define GPIO_CFGLOW_MODE6_0                                         (0x1UL << GPIO_CFGLOW_MODE6_Pos)                               /*!< 0x01000000 */
#define GPIO_CFGLOW_MODE6_1                                         (0x2UL << GPIO_CFGLOW_MODE6_Pos)                               /*!< 0x02000000 */

#define GPIO_CFGLOW_MODE7_Pos                                       (28U)
#define GPIO_CFGLOW_MODE7_Msk                                       (0x3UL << GPIO_CFGLOW_MODE7_Pos)                               /*!< 0x30000000 */
#define GPIO_CFGLOW_MODE7                                           GPIO_CFGLOW_MODE7_Msk                                          /*!<MODE7[1:0] bits (Port x mode bits, pin 7) */
#define GPIO_CFGLOW_MODE7_0                                         (0x1UL << GPIO_CFGLOW_MODE7_Pos)                               /*!< 0x10000000 */
#define GPIO_CFGLOW_MODE7_1                                         (0x2UL << GPIO_CFGLOW_MODE7_Pos)                               /*!< 0x20000000 */

#define GPIO_CFGLOW_CFG_Pos                                         (2U)
#define GPIO_CFGLOW_CFG_Msk                                         (0x33333333UL << GPIO_CFGLOW_CFG_Pos)                          /*!< 0xCCCCCCCC */
#define GPIO_CFGLOW_CFG                                             GPIO_CFGLOW_CFG_Msk                                            /*!<Port x configuration bits */

#define GPIO_CFGLOW_CFG0_Pos                                        (2U)
#define GPIO_CFGLOW_CFG0_Msk                                        (0x3UL << GPIO_CFGLOW_CFG0_Pos)                                /*!< 0x0000000C */
#define GPIO_CFGLOW_CFG0                                            GPIO_CFGLOW_CFG0_Msk                                           /*!<CFG0[1:0] bits (Port x configuration bits, pin 0) */
#define GPIO_CFGLOW_CFG0_0                                          (0x1UL << GPIO_CFGLOW_CFG0_Pos)                                /*!< 0x00000004 */
#define GPIO_CFGLOW_CFG0_1                                          (0x2UL << GPIO_CFGLOW_CFG0_Pos)                                /*!< 0x00000008 */

#define GPIO_CFGLOW_CFG1_Pos                                        (6U)
#define GPIO_CFGLOW_CFG1_Msk                                        (0x3UL << GPIO_CFGLOW_CFG1_Pos)                                /*!< 0x000000C0 */
#define GPIO_CFGLOW_CFG1                                            GPIO_CFGLOW_CFG1_Msk                                           /*!<CFG1[1:0] bits (Port x configuration bits, pin 1) */
#define GPIO_CFGLOW_CFG1_0                                          (0x1UL << GPIO_CFGLOW_CFG1_Pos)                                /*!< 0x00000040 */
#define GPIO_CFGLOW_CFG1_1                                          (0x2UL << GPIO_CFGLOW_CFG1_Pos)                                /*!< 0x00000080 */

#define GPIO_CFGLOW_CFG2_Pos                                        (10U)
#define GPIO_CFGLOW_CFG2_Msk                                        (0x3UL << GPIO_CFGLOW_CFG2_Pos)                                /*!< 0x00000C00 */
#define GPIO_CFGLOW_CFG2                                            GPIO_CFGLOW_CFG2_Msk                                           /*!<CFG2[1:0] bits (Port x configuration bits, pin 2) */
#define GPIO_CFGLOW_CFG2_0                                          (0x1UL << GPIO_CFGLOW_CFG2_Pos)                                /*!< 0x00000400 */
#define GPIO_CFGLOW_CFG2_1                                          (0x2UL << GPIO_CFGLOW_CFG2_Pos)                                /*!< 0x00000800 */

#define GPIO_CFGLOW_CFG3_Pos                                        (14U)
#define GPIO_CFGLOW_CFG3_Msk                                        (0x3UL << GPIO_CFGLOW_CFG3_Pos)                                /*!< 0x0000C000 */
#define GPIO_CFGLOW_CFG3                                            GPIO_CFGLOW_CFG3_Msk                                           /*!<CFG3[1:0] bits (Port x configuration bits, pin 3) */
#define GPIO_CFGLOW_CFG3_0                                          (0x1UL << GPIO_CFGLOW_CFG3_Pos)                                /*!< 0x00004000 */
#define GPIO_CFGLOW_CFG3_1                                          (0x2UL << GPIO_CFGLOW_CFG3_Pos)                                /*!< 0x00008000 */

#define GPIO_CFGLOW_CFG4_Pos                                        (18U)
#define GPIO_CFGLOW_CFG4_Msk                                        (0x3UL << GPIO_CFGLOW_CFG4_Pos)                                /*!< 0x000C0000 */
#define GPIO_CFGLOW_CFG4                                            GPIO_CFGLOW_CFG4_Msk                                           /*!<CFG4[1:0] bits (Port x configuration bits, pin 4) */
#define GPIO_CFGLOW_CFG4_0                                          (0x1UL << GPIO_CFGLOW_CFG4_Pos)                                /*!< 0x00040000 */
#define GPIO_CFGLOW_CFG4_1                                          (0x2UL << GPIO_CFGLOW_CFG4_Pos)                                /*!< 0x00080000 */

#define GPIO_CFGLOW_CFG5_Pos                                        (22U)
#define GPIO_CFGLOW_CFG5_Msk                                        (0x3UL << GPIO_CFGLOW_CFG5_Pos)                                /*!< 0x00C00000 */
#define GPIO_CFGLOW_CFG5                                            GPIO_CFGLOW_CFG5_Msk                                           /*!<CFG5[1:0] bits (Port x configuration bits, pin 5) */
#define GPIO_CFGLOW_CFG5_0                                          (0x1UL << GPIO_CFGLOW_CFG5_Pos)                                /*!< 0x00400000 */
#define GPIO_CFGLOW_CFG5_1                                          (0x2UL << GPIO_CFGLOW_CFG5_Pos)                                /*!< 0x00800000 */

#define GPIO_CFGLOW_CFG6_Pos                                        (26U)
#define GPIO_CFGLOW_CFG6_Msk                                        (0x3UL << GPIO_CFGLOW_CFG6_Pos)                                /*!< 0x0C000000 */
#define GPIO_CFGLOW_CFG6                                            GPIO_CFGLOW_CFG6_Msk                                           /*!<CFG6[1:0] bits (Port x configuration bits, pin 6) */
#define GPIO_CFGLOW_CFG6_0                                          (0x1UL << GPIO_CFGLOW_CFG6_Pos)                                /*!< 0x04000000 */
#define GPIO_CFGLOW_CFG6_1                                          (0x2UL << GPIO_CFGLOW_CFG6_Pos)                                /*!< 0x08000000 */

#define GPIO_CFGLOW_CFG7_Pos                                        (30U)
#define GPIO_CFGLOW_CFG7_Msk                                        (0x3UL << GPIO_CFGLOW_CFG7_Pos)                                /*!< 0xC0000000 */
#define GPIO_CFGLOW_CFG7                                            GPIO_CFGLOW_CFG7_Msk                                           /*!<CFG7[1:0] bits (Port x configuration bits, pin 7) */
#define GPIO_CFGLOW_CFG7_0                                          (0x1UL << GPIO_CFGLOW_CFG7_Pos)                                /*!< 0x40000000 */
#define GPIO_CFGLOW_CFG7_1                                          (0x2UL << GPIO_CFGLOW_CFG7_Pos)                                /*!< 0x80000000 */

/******************  Bits definition for GPIO_CFGHIG register  ******************/
#define GPIO_CFGHIG_MODE_Pos                                        (0U)
#define GPIO_CFGHIG_MODE_Msk                                        (0x33333333UL << GPIO_CFGHIG_MODE_Pos)                        /*!< 0x33333333 */
#define GPIO_CFGHIG_MODE                                            GPIO_CFGHIG_MODE_Msk                                           /*!<Port x mode bits */

#define GPIO_CFGHIG_MODE8_Pos                                       (0U)
#define GPIO_CFGHIG_MODE8_Msk                                       (0x3UL << GPIO_CFGHIG_MODE8_Pos)                               /*!< 0x00000003 */
#define GPIO_CFGHIG_MODE8                                           GPIO_CFGHIG_MODE8_Msk                                          /*!<MODE8[1:0] bits (Port x mode bits, pin 8) */
#define GPIO_CFGHIG_MODE8_0                                         (0x1UL << GPIO_CFGHIG_MODE8_Pos)                               /*!< 0x00000001 */
#define GPIO_CFGHIG_MODE8_1                                         (0x2UL << GPIO_CFGHIG_MODE8_Pos)                               /*!< 0x00000002 */

#define GPIO_CFGHIG_MODE9_Pos                                       (4U)
#define GPIO_CFGHIG_MODE9_Msk                                       (0x3UL << GPIO_CFGHIG_MODE9_Pos)                               /*!< 0x00000030 */
#define GPIO_CFGHIG_MODE9                                           GPIO_CFGHIG_MODE9_Msk                                          /*!<MODE9[1:0] bits (Port x mode bits, pin 9) */
#define GPIO_CFGHIG_MODE9_0                                         (0x1UL << GPIO_CFGHIG_MODE9_Pos)                               /*!< 0x00000010 */
#define GPIO_CFGHIG_MODE9_1                                         (0x2UL << GPIO_CFGHIG_MODE9_Pos)                               /*!< 0x00000020 */

#define GPIO_CFGHIG_MODE10_Pos                                      (8U)
#define GPIO_CFGHIG_MODE10_Msk                                      (0x3UL << GPIO_CFGHIG_MODE10_Pos)                              /*!< 0x00000300 */
#define GPIO_CFGHIG_MODE10                                          GPIO_CFGHIG_MODE10_Msk                                         /*!<MODE10[1:0] bits (Port x mode bits, pin 10) */
#define GPIO_CFGHIG_MODE10_0                                        (0x1UL << GPIO_CFGHIG_MODE10_Pos)                              /*!< 0x00000100 */
#define GPIO_CFGHIG_MODE10_1                                        (0x2UL << GPIO_CFGHIG_MODE10_Pos)                              /*!< 0x00000200 */

#define GPIO_CFGHIG_MODE11_Pos                                      (12U)
#define GPIO_CFGHIG_MODE11_Msk                                      (0x3UL << GPIO_CFGHIG_MODE11_Pos)                              /*!< 0x00003000 */
#define GPIO_CFGHIG_MODE11                                          GPIO_CFGHIG_MODE11_Msk                                         /*!<MODE11[1:0] bits (Port x mode bits, pin 11) */
#define GPIO_CFGHIG_MODE11_0                                        (0x1UL << GPIO_CFGHIG_MODE11_Pos)                              /*!< 0x00001000 */
#define GPIO_CFGHIG_MODE11_1                                        (0x2UL << GPIO_CFGHIG_MODE11_Pos)                              /*!< 0x00002000 */

#define GPIO_CFGHIG_MODE12_Pos                                      (16U)
#define GPIO_CFGHIG_MODE12_Msk                                      (0x3UL << GPIO_CFGHIG_MODE12_Pos)                              /*!< 0x00030000 */
#define GPIO_CFGHIG_MODE12                                          GPIO_CFGHIG_MODE12_Msk                                         /*!<MODE12[1:0] bits (Port x mode bits, pin 12) */
#define GPIO_CFGHIG_MODE12_0                                        (0x1UL << GPIO_CFGHIG_MODE12_Pos)                              /*!< 0x00010000 */
#define GPIO_CFGHIG_MODE12_1                                        (0x2UL << GPIO_CFGHIG_MODE12_Pos)                              /*!< 0x00020000 */

#define GPIO_CFGHIG_MODE13_Pos                                      (20U)
#define GPIO_CFGHIG_MODE13_Msk                                      (0x3UL << GPIO_CFGHIG_MODE13_Pos)                              /*!< 0x00300000 */
#define GPIO_CFGHIG_MODE13                                          GPIO_CFGHIG_MODE13_Msk                                         /*!<MODE13[1:0] bits (Port x mode bits, pin 13) */
#define GPIO_CFGHIG_MODE13_0                                        (0x1UL << GPIO_CFGHIG_MODE13_Pos)                              /*!< 0x00100000 */
#define GPIO_CFGHIG_MODE13_1                                        (0x2UL << GPIO_CFGHIG_MODE13_Pos)                              /*!< 0x00200000 */

#define GPIO_CFGHIG_MODE14_Pos                                      (24U)
#define GPIO_CFGHIG_MODE14_Msk                                      (0x3UL << GPIO_CFGHIG_MODE14_Pos)                              /*!< 0x03000000 */
#define GPIO_CFGHIG_MODE14                                          GPIO_CFGHIG_MODE14_Msk                                         /*!<MODE14[1:0] bits (Port x mode bits, pin 14) */
#define GPIO_CFGHIG_MODE14_0                                        (0x1UL << GPIO_CFGHIG_MODE14_Pos)                              /*!< 0x01000000 */
#define GPIO_CFGHIG_MODE14_1                                        (0x2UL << GPIO_CFGHIG_MODE14_Pos)                              /*!< 0x02000000 */

#define GPIO_CFGHIG_MODE15_Pos                                      (28U)
#define GPIO_CFGHIG_MODE15_Msk                                      (0x3UL << GPIO_CFGHIG_MODE15_Pos)                              /*!< 0x30000000 */
#define GPIO_CFGHIG_MODE15                                          GPIO_CFGHIG_MODE15_Msk                                         /*!<MODE15[1:0] bits (Port x mode bits, pin 15) */
#define GPIO_CFGHIG_MODE15_0                                        (0x1UL << GPIO_CFGHIG_MODE15_Pos)                              /*!< 0x10000000 */
#define GPIO_CFGHIG_MODE15_1                                        (0x2UL << GPIO_CFGHIG_MODE15_Pos)                              /*!< 0x20000000 */

#define GPIO_CFGHIG_CFG_Pos                                         (2U)
#define GPIO_CFGHIG_CFG_Msk                                         (0x33333333UL << GPIO_CFGHIG_CFG_Pos)                          /*!< 0xCCCCCCCC */
#define GPIO_CFGHIG_CFG                                             GPIO_CFGHIG_CFG_Msk                                            /*!<Port x configuration bits */

#define GPIO_CFGHIG_CFG8_Pos                                        (2U)
#define GPIO_CFGHIG_CFG8_Msk                                        (0x3UL << GPIO_CFGHIG_CFG8_Pos)                                /*!< 0x0000000C */
#define GPIO_CFGHIG_CFG8                                            GPIO_CFGHIG_CFG8_Msk                                           /*!<CFG8[1:0] bits (Port x configuration bits, pin 8) */
#define GPIO_CFGHIG_CFG8_0                                          (0x1UL << GPIO_CFGHIG_CFG8_Pos)                                /*!< 0x00000004 */
#define GPIO_CFGHIG_CFG8_1                                          (0x2UL << GPIO_CFGHIG_CFG8_Pos)                                /*!< 0x00000008 */

#define GPIO_CFGHIG_CFG9_Pos                                        (6U)
#define GPIO_CFGHIG_CFG9_Msk                                        (0x3UL << GPIO_CFGHIG_CFG9_Pos)                                /*!< 0x000000C0 */
#define GPIO_CFGHIG_CFG9                                            GPIO_CFGHIG_CFG9_Msk                                           /*!<CFG9[1:0] bits (Port x configuration bits, pin 9) */
#define GPIO_CFGHIG_CFG9_0                                          (0x1UL << GPIO_CFGHIG_CFG9_Pos)                                /*!< 0x00000040 */
#define GPIO_CFGHIG_CFG9_1                                          (0x2UL << GPIO_CFGHIG_CFG9_Pos)                                /*!< 0x00000080 */

#define GPIO_CFGHIG_CFG10_Pos                                       (10U)
#define GPIO_CFGHIG_CFG10_Msk                                       (0x3UL << GPIO_CFGHIG_CFG10_Pos)                               /*!< 0x00000C00 */
#define GPIO_CFGHIG_CFG10                                           GPIO_CFGHIG_CFG10_Msk                                          /*!<CFG10[1:0] bits (Port x configuration bits, pin 10) */
#define GPIO_CFGHIG_CFG10_0                                         (0x1UL << GPIO_CFGHIG_CFG10_Pos)                               /*!< 0x00000400 */
#define GPIO_CFGHIG_CFG10_1                                         (0x2UL << GPIO_CFGHIG_CFG10_Pos)                               /*!< 0x00000800 */

#define GPIO_CFGHIG_CFG11_Pos                                       (14U)
#define GPIO_CFGHIG_CFG11_Msk                                       (0x3UL << GPIO_CFGHIG_CFG11_Pos)                               /*!< 0x0000C000 */
#define GPIO_CFGHIG_CFG11                                           GPIO_CFGHIG_CFG11_Msk                                          /*!<CFG11[1:0] bits (Port x configuration bits, pin 11) */
#define GPIO_CFGHIG_CFG11_0                                         (0x1UL << GPIO_CFGHIG_CFG11_Pos)                               /*!< 0x00004000 */
#define GPIO_CFGHIG_CFG11_1                                         (0x2UL << GPIO_CFGHIG_CFG11_Pos)                               /*!< 0x00008000 */

#define GPIO_CFGHIG_CFG12_Pos                                       (18U)
#define GPIO_CFGHIG_CFG12_Msk                                       (0x3UL << GPIO_CFGHIG_CFG12_Pos)                               /*!< 0x000C0000 */
#define GPIO_CFGHIG_CFG12                                           GPIO_CFGHIG_CFG12_Msk                                          /*!<CFG12[1:0] bits (Port x configuration bits, pin 12) */
#define GPIO_CFGHIG_CFG12_0                                         (0x1UL << GPIO_CFGHIG_CFG12_Pos)                               /*!< 0x00010000 */
#define GPIO_CFGHIG_CFG12_1                                         (0x2UL << GPIO_CFGHIG_CFG12_Pos)                               /*!< 0x00020000 */

#define GPIO_CFGHIG_CFG13_Pos                                       (22U)
#define GPIO_CFGHIG_CFG13_Msk                                       (0x3UL << GPIO_CFGHIG_CFG13_Pos)                               /*!< 0x00C00000 */
#define GPIO_CFGHIG_CFG13                                           GPIO_CFGHIG_CFG13_Msk                                          /*!<CFG13[1:0] bits (Port x configuration bits, pin 13) */
#define GPIO_CFGHIG_CFG13_0                                         (0x1UL << GPIO_CFGHIG_CFG13_Pos)                               /*!< 0x00400000 */
#define GPIO_CFGHIG_CFG13_1                                         (0x2UL << GPIO_CFGHIG_CFG13_Pos)                               /*!< 0x00800000 */

#define GPIO_CFGHIG_CFG14_Pos                                       (26U)
#define GPIO_CFGHIG_CFG14_Msk                                       (0x3UL << GPIO_CFGHIG_CFG14_Pos)                               /*!< 0x0C000000 */
#define GPIO_CFGHIG_CFG14                                           GPIO_CFGHIG_CFG14_Msk                                          /*!<CFG14[1:0] bits (Port x configuration bits, pin 14) */
#define GPIO_CFGHIG_CFG14_0                                         (0x1UL << GPIO_CFGHIG_CFG14_Pos)                               /*!< 0x04000000 */
#define GPIO_CFGHIG_CFG14_1                                         (0x2UL << GPIO_CFGHIG_CFG14_Pos)                               /*!< 0x08000000 */

#define GPIO_CFGHIG_CFG15_Pos                                       (30U)
#define GPIO_CFGHIG_CFG15_Msk                                       (0x3UL << GPIO_CFGHIG_CFG15_Pos)                               /*!< 0xC0000000 */
#define GPIO_CFGHIG_CFG15                                           GPIO_CFGHIG_CFG15_Msk                                          /*!<CFG15[1:0] bits (Port x configuration bits, pin 15) */
#define GPIO_CFGHIG_CFG15_0                                         (0x1UL << GPIO_CFGHIG_CFG15_Pos)                               /*!< 0x40000000 */
#define GPIO_CFGHIG_CFG15_1                                         (0x2UL << GPIO_CFGHIG_CFG15_Pos)                               /*!< 0x80000000 */


/* Defines used for Pin Mask Initialization */
#define GPIO_PIN_MASK_POS                                            (8U)
#define GPIO_PIN_NB                                                  (16U)

/******************  Bits definition for GPIO_IDATA register  *******************/
#define GPIO_IDATA_IDATA0_Pos                                       (0U)
#define GPIO_IDATA_IDATA0_Msk                                       (0x1UL << GPIO_IDATA_IDATA0_Pos)                               /*!< 0x00000001 */
#define GPIO_IDATA_IDATA0                                           GPIO_IDATA_IDATA0_Msk                                          /*!< Port input data, bit 0 */
#define GPIO_IDATA_IDATA1_Pos                                       (1U)
#define GPIO_IDATA_IDATA1_Msk                                       (0x1UL << GPIO_IDATA_IDATA1_Pos)                               /*!< 0x00000002 */
#define GPIO_IDATA_IDATA1                                           GPIO_IDATA_IDATA1_Msk                                          /*!< Port input data, bit 1 */
#define GPIO_IDATA_IDATA2_Pos                                       (2U)
#define GPIO_IDATA_IDATA2_Msk                                       (0x1UL << GPIO_IDATA_IDATA2_Pos)                               /*!< 0x00000004 */
#define GPIO_IDATA_IDATA2                                           GPIO_IDATA_IDATA2_Msk                                          /*!< Port input data, bit 2 */
#define GPIO_IDATA_IDATA3_Pos                                       (3U)
#define GPIO_IDATA_IDATA3_Msk                                       (0x1UL << GPIO_IDATA_IDATA3_Pos)                               /*!< 0x00000008 */
#define GPIO_IDATA_IDATA3                                           GPIO_IDATA_IDATA3_Msk                                          /*!< Port input data, bit 3 */
#define GPIO_IDATA_IDATA4_Pos                                       (4U)
#define GPIO_IDATA_IDATA4_Msk                                       (0x1UL << GPIO_IDATA_IDATA4_Pos)                               /*!< 0x00000010 */
#define GPIO_IDATA_IDATA4                                           GPIO_IDATA_IDATA4_Msk                                          /*!< Port input data, bit 4 */
#define GPIO_IDATA_IDATA5_Pos                                       (5U)
#define GPIO_IDATA_IDATA5_Msk                                       (0x1UL << GPIO_IDATA_IDATA5_Pos)                               /*!< 0x00000020 */
#define GPIO_IDATA_IDATA5                                           GPIO_IDATA_IDATA5_Msk                                          /*!< Port input data, bit 5 */
#define GPIO_IDATA_IDATA6_Pos                                       (6U)
#define GPIO_IDATA_IDATA6_Msk                                       (0x1UL << GPIO_IDATA_IDATA6_Pos)                               /*!< 0x00000040 */
#define GPIO_IDATA_IDATA6                                           GPIO_IDATA_IDATA6_Msk                                          /*!< Port input data, bit 6 */
#define GPIO_IDATA_IDATA7_Pos                                       (7U)
#define GPIO_IDATA_IDATA7_Msk                                       (0x1UL << GPIO_IDATA_IDATA7_Pos)                               /*!< 0x00000080 */
#define GPIO_IDATA_IDATA7                                           GPIO_IDATA_IDATA7_Msk                                          /*!< Port input data, bit 7 */
#define GPIO_IDATA_IDATA8_Pos                                       (8U)
#define GPIO_IDATA_IDATA8_Msk                                       (0x1UL << GPIO_IDATA_IDATA8_Pos)                               /*!< 0x00000100 */
#define GPIO_IDATA_IDATA8                                           GPIO_IDATA_IDATA8_Msk                                          /*!< Port input data, bit 8 */
#define GPIO_IDATA_IDATA9_Pos                                       (9U)
#define GPIO_IDATA_IDATA9_Msk                                       (0x1UL << GPIO_IDATA_IDATA9_Pos)                               /*!< 0x00000200 */
#define GPIO_IDATA_IDATA9                                           GPIO_IDATA_IDATA9_Msk                                          /*!< Port input data, bit 9 */
#define GPIO_IDATA_IDATA10_Pos                                      (10U)
#define GPIO_IDATA_IDATA10_Msk                                      (0x1UL << GPIO_IDATA_IDATA10_Pos)                              /*!< 0x00000400 */
#define GPIO_IDATA_IDATA10                                          GPIO_IDATA_IDATA10_Msk                                         /*!< Port input data, bit 10 */
#define GPIO_IDATA_IDATA11_Pos                                      (11U)
#define GPIO_IDATA_IDATA11_Msk                                      (0x1UL << GPIO_IDATA_IDATA11_Pos)                              /*!< 0x00000800 */
#define GPIO_IDATA_IDATA11                                          GPIO_IDATA_IDATA11_Msk                                         /*!< Port input data, bit 11 */
#define GPIO_IDATA_IDATA12_Pos                                      (12U)
#define GPIO_IDATA_IDATA12_Msk                                      (0x1UL << GPIO_IDATA_IDATA12_Pos)                              /*!< 0x00001000 */
#define GPIO_IDATA_IDATA12                                          GPIO_IDATA_IDATA12_Msk                                         /*!< Port input data, bit 12 */
#define GPIO_IDATA_IDATA13_Pos                                      (13U)
#define GPIO_IDATA_IDATA13_Msk                                      (0x1UL << GPIO_IDATA_IDATA13_Pos)                              /*!< 0x00002000 */
#define GPIO_IDATA_IDATA13                                          GPIO_IDATA_IDATA13_Msk                                         /*!< Port input data, bit 13 */
#define GPIO_IDATA_IDATA14_Pos                                      (14U)
#define GPIO_IDATA_IDATA14_Msk                                      (0x1UL << GPIO_IDATA_IDATA14_Pos)                              /*!< 0x00004000 */
#define GPIO_IDATA_IDATA14                                          GPIO_IDATA_IDATA14_Msk                                         /*!< Port input data, bit 14 */
#define GPIO_IDATA_IDATA15_Pos                                      (15U)
#define GPIO_IDATA_IDATA15_Msk                                      (0x1UL << GPIO_IDATA_IDATA15_Pos)                              /*!< 0x00008000 */
#define GPIO_IDATA_IDATA15                                          GPIO_IDATA_IDATA15_Msk                                         /*!< Port input data, bit 15 */

/******************  Bits definition for GPIO_ODATA register  *******************/
#define GPIO_ODATA_ODATA0_Pos                                       (0U)
#define GPIO_ODATA_ODATA0_Msk                                       (0x1UL << GPIO_ODATA_ODATA0_Pos)                               /*!< 0x00000001 */
#define GPIO_ODATA_ODATA0                                           GPIO_ODATA_ODATA0_Msk                                          /*!< Port output data, bit 0 */
#define GPIO_ODATA_ODATA1_Pos                                       (1U)
#define GPIO_ODATA_ODATA1_Msk                                       (0x1UL << GPIO_ODATA_ODATA1_Pos)                               /*!< 0x00000002 */
#define GPIO_ODATA_ODATA1                                           GPIO_ODATA_ODATA1_Msk                                          /*!< Port output data, bit 1 */
#define GPIO_ODATA_ODATA2_Pos                                       (2U)
#define GPIO_ODATA_ODATA2_Msk                                       (0x1UL << GPIO_ODATA_ODATA2_Pos)                               /*!< 0x00000004 */
#define GPIO_ODATA_ODATA2                                           GPIO_ODATA_ODATA2_Msk                                          /*!< Port output data, bit 2 */
#define GPIO_ODATA_ODATA3_Pos                                       (3U)
#define GPIO_ODATA_ODATA3_Msk                                       (0x1UL << GPIO_ODATA_ODATA3_Pos)                               /*!< 0x00000008 */
#define GPIO_ODATA_ODATA3                                           GPIO_ODATA_ODATA3_Msk                                          /*!< Port output data, bit 3 */
#define GPIO_ODATA_ODATA4_Pos                                       (4U)
#define GPIO_ODATA_ODATA4_Msk                                       (0x1UL << GPIO_ODATA_ODATA4_Pos)                               /*!< 0x00000010 */
#define GPIO_ODATA_ODATA4                                           GPIO_ODATA_ODATA4_Msk                                          /*!< Port output data, bit 4 */
#define GPIO_ODATA_ODATA5_Pos                                       (5U)
#define GPIO_ODATA_ODATA5_Msk                                       (0x1UL << GPIO_ODATA_ODATA5_Pos)                               /*!< 0x00000020 */
#define GPIO_ODATA_ODATA5                                           GPIO_ODATA_ODATA5_Msk                                          /*!< Port output data, bit 5 */
#define GPIO_ODATA_ODATA6_Pos                                       (6U)
#define GPIO_ODATA_ODATA6_Msk                                       (0x1UL << GPIO_ODATA_ODATA6_Pos)                               /*!< 0x00000040 */
#define GPIO_ODATA_ODATA6                                           GPIO_ODATA_ODATA6_Msk                                          /*!< Port output data, bit 6 */
#define GPIO_ODATA_ODATA7_Pos                                       (7U)
#define GPIO_ODATA_ODATA7_Msk                                       (0x1UL << GPIO_ODATA_ODATA7_Pos)                               /*!< 0x00000080 */
#define GPIO_ODATA_ODATA7                                           GPIO_ODATA_ODATA7_Msk                                          /*!< Port output data, bit 7 */
#define GPIO_ODATA_ODATA8_Pos                                       (8U)
#define GPIO_ODATA_ODATA8_Msk                                       (0x1UL << GPIO_ODATA_ODATA8_Pos)                               /*!< 0x00000100 */
#define GPIO_ODATA_ODATA8                                           GPIO_ODATA_ODATA8_Msk                                          /*!< Port output data, bit 8 */
#define GPIO_ODATA_ODATA9_Pos                                       (9U)
#define GPIO_ODATA_ODATA9_Msk                                       (0x1UL << GPIO_ODATA_ODATA9_Pos)                               /*!< 0x00000200 */
#define GPIO_ODATA_ODATA9                                           GPIO_ODATA_ODATA9_Msk                                          /*!< Port output data, bit 9 */
#define GPIO_ODATA_ODATA10_Pos                                      (10U)
#define GPIO_ODATA_ODATA10_Msk                                      (0x1UL << GPIO_ODATA_ODATA10_Pos)                              /*!< 0x00000400 */
#define GPIO_ODATA_ODATA10                                          GPIO_ODATA_ODATA10_Msk                                         /*!< Port output data, bit 10 */
#define GPIO_ODATA_ODATA11_Pos                                      (11U)
#define GPIO_ODATA_ODATA11_Msk                                      (0x1UL << GPIO_ODATA_ODATA11_Pos)                              /*!< 0x00000800 */
#define GPIO_ODATA_ODATA11                                          GPIO_ODATA_ODATA11_Msk                                         /*!< Port output data, bit 11 */
#define GPIO_ODATA_ODATA12_Pos                                      (12U)
#define GPIO_ODATA_ODATA12_Msk                                      (0x1UL << GPIO_ODATA_ODATA12_Pos)                              /*!< 0x00001000 */
#define GPIO_ODATA_ODATA12                                          GPIO_ODATA_ODATA12_Msk                                         /*!< Port output data, bit 12 */
#define GPIO_ODATA_ODATA13_Pos                                      (13U)
#define GPIO_ODATA_ODATA13_Msk                                      (0x1UL << GPIO_ODATA_ODATA13_Pos)                              /*!< 0x00002000 */
#define GPIO_ODATA_ODATA13                                          GPIO_ODATA_ODATA13_Msk                                         /*!< Port output data, bit 13 */
#define GPIO_ODATA_ODATA14_Pos                                      (14U)
#define GPIO_ODATA_ODATA14_Msk                                      (0x1UL << GPIO_ODATA_ODATA14_Pos)                              /*!< 0x00004000 */
#define GPIO_ODATA_ODATA14                                          GPIO_ODATA_ODATA14_Msk                                         /*!< Port output data, bit 14 */
#define GPIO_ODATA_ODATA15_Pos                                      (15U)
#define GPIO_ODATA_ODATA15_Msk                                      (0x1UL << GPIO_ODATA_ODATA15_Pos)                              /*!< 0x00008000 */
#define GPIO_ODATA_ODATA15                                          GPIO_ODATA_ODATA15_Msk                                         /*!< Port output data, bit 15 */

/******************  Bits definition for GPIO_BSC register  *********************/
#define GPIO_BSC_BS0_Pos                                            (0U)
#define GPIO_BSC_BS0_Msk                                            (0x1UL << GPIO_BSC_BS0_Pos)                                    /*!< 0x00000001 */
#define GPIO_BSC_BS0                                                GPIO_BSC_BS0_Msk                                               /*!< Port x bit Set bit 0 */
#define GPIO_BSC_BS1_Pos                                            (1U)
#define GPIO_BSC_BS1_Msk                                            (0x1UL << GPIO_BSC_BS1_Pos)                                    /*!< 0x00000002 */
#define GPIO_BSC_BS1                                                GPIO_BSC_BS1_Msk                                               /*!< Port x bit Set bit 1 */
#define GPIO_BSC_BS2_Pos                                            (2U)
#define GPIO_BSC_BS2_Msk                                            (0x1UL << GPIO_BSC_BS2_Pos)                                    /*!< 0x00000004 */
#define GPIO_BSC_BS2                                                GPIO_BSC_BS2_Msk                                               /*!< Port x bit Set bit 2 */
#define GPIO_BSC_BS3_Pos                                            (3U)
#define GPIO_BSC_BS3_Msk                                            (0x1UL << GPIO_BSC_BS3_Pos)                                    /*!< 0x00000008 */
#define GPIO_BSC_BS3                                                GPIO_BSC_BS3_Msk                                               /*!< Port x bit Set bit 3 */
#define GPIO_BSC_BS4_Pos                                            (4U)
#define GPIO_BSC_BS4_Msk                                            (0x1UL << GPIO_BSC_BS4_Pos)                                    /*!< 0x00000010 */
#define GPIO_BSC_BS4                                                GPIO_BSC_BS4_Msk                                               /*!< Port x bit Set bit 4 */
#define GPIO_BSC_BS5_Pos                                            (5U)
#define GPIO_BSC_BS5_Msk                                            (0x1UL << GPIO_BSC_BS5_Pos)                                    /*!< 0x00000020 */
#define GPIO_BSC_BS5                                                GPIO_BSC_BS5_Msk                                               /*!< Port x bit Set bit 5 */
#define GPIO_BSC_BS6_Pos                                            (6U)
#define GPIO_BSC_BS6_Msk                                            (0x1UL << GPIO_BSC_BS6_Pos)                                    /*!< 0x00000040 */
#define GPIO_BSC_BS6                                                GPIO_BSC_BS6_Msk                                               /*!< Port x bit Set bit 6 */
#define GPIO_BSC_BS7_Pos                                            (7U)
#define GPIO_BSC_BS7_Msk                                            (0x1UL << GPIO_BSC_BS7_Pos)                                    /*!< 0x00000080 */
#define GPIO_BSC_BS7                                                GPIO_BSC_BS7_Msk                                               /*!< Port x bit Set bit 7 */
#define GPIO_BSC_BS8_Pos                                            (8U)
#define GPIO_BSC_BS8_Msk                                            (0x1UL << GPIO_BSC_BS8_Pos)                                    /*!< 0x00000100 */
#define GPIO_BSC_BS8                                                GPIO_BSC_BS8_Msk                                               /*!< Port x bit Set bit 8 */
#define GPIO_BSC_BS9_Pos                                            (9U)
#define GPIO_BSC_BS9_Msk                                            (0x1UL << GPIO_BSC_BS9_Pos)                                    /*!< 0x00000200 */
#define GPIO_BSC_BS9                                                GPIO_BSC_BS9_Msk                                               /*!< Port x bit Set bit 9 */
#define GPIO_BSC_BS10_Pos                                           (10U)
#define GPIO_BSC_BS10_Msk                                           (0x1UL << GPIO_BSC_BS10_Pos)                                   /*!< 0x00000400 */
#define GPIO_BSC_BS10                                               GPIO_BSC_BS10_Msk                                              /*!< Port x bit Set bit 10 */
#define GPIO_BSC_BS11_Pos                                           (11U)
#define GPIO_BSC_BS11_Msk                                           (0x1UL << GPIO_BSC_BS11_Pos)                                   /*!< 0x00000800 */
#define GPIO_BSC_BS11                                               GPIO_BSC_BS11_Msk                                              /*!< Port x bit Set bit 11 */
#define GPIO_BSC_BS12_Pos                                           (12U)
#define GPIO_BSC_BS12_Msk                                           (0x1UL << GPIO_BSC_BS12_Pos)                                   /*!< 0x00001000 */
#define GPIO_BSC_BS12                                               GPIO_BSC_BS12_Msk                                              /*!< Port x bit Set bit 12 */
#define GPIO_BSC_BS13_Pos                                           (13U)
#define GPIO_BSC_BS13_Msk                                           (0x1UL << GPIO_BSC_BS13_Pos)                                   /*!< 0x00002000 */
#define GPIO_BSC_BS13                                               GPIO_BSC_BS13_Msk                                              /*!< Port x bit Set bit 13 */
#define GPIO_BSC_BS14_Pos                                           (14U)
#define GPIO_BSC_BS14_Msk                                           (0x1UL << GPIO_BSC_BS14_Pos)                                   /*!< 0x00004000 */
#define GPIO_BSC_BS14                                               GPIO_BSC_BS14_Msk                                              /*!< Port x bit Set bit 14 */
#define GPIO_BSC_BS15_Pos                                           (15U)
#define GPIO_BSC_BS15_Msk                                           (0x1UL << GPIO_BSC_BS15_Pos)                                   /*!< 0x00008000 */
#define GPIO_BSC_BS15                                               GPIO_BSC_BS15_Msk                                              /*!< Port x bit Set bit 15 */

#define GPIO_BSC_BC0_Pos                                            (16U)
#define GPIO_BSC_BC0_Msk                                            (0x1UL << GPIO_BSC_BC0_Pos)                                    /*!< 0x00010000 */
#define GPIO_BSC_BC0                                                GPIO_BSC_BC0_Msk                                               /*!< Port x bit Clear bit 0 */
#define GPIO_BSC_BC1_Pos                                            (17U)
#define GPIO_BSC_BC1_Msk                                            (0x1UL << GPIO_BSC_BC1_Pos)                                    /*!< 0x00020000 */
#define GPIO_BSC_BC1                                                GPIO_BSC_BC1_Msk                                               /*!< Port x bit Clear bit 1 */
#define GPIO_BSC_BC2_Pos                                            (18U)
#define GPIO_BSC_BC2_Msk                                            (0x1UL << GPIO_BSC_BC2_Pos)                                    /*!< 0x00040000 */
#define GPIO_BSC_BC2                                                GPIO_BSC_BC2_Msk                                               /*!< Port x bit Clear bit 2 */
#define GPIO_BSC_BC3_Pos                                            (19U)
#define GPIO_BSC_BC3_Msk                                            (0x1UL << GPIO_BSC_BC3_Pos)                                    /*!< 0x00080000 */
#define GPIO_BSC_BC3                                                GPIO_BSC_BC3_Msk                                               /*!< Port x bit Clear bit 3 */
#define GPIO_BSC_BC4_Pos                                            (20U)
#define GPIO_BSC_BC4_Msk                                            (0x1UL << GPIO_BSC_BC4_Pos)                                    /*!< 0x00100000 */
#define GPIO_BSC_BC4                                                GPIO_BSC_BC4_Msk                                               /*!< Port x bit Clear bit 4 */
#define GPIO_BSC_BC5_Pos                                            (21U)
#define GPIO_BSC_BC5_Msk                                            (0x1UL << GPIO_BSC_BC5_Pos)                                    /*!< 0x00200000 */
#define GPIO_BSC_BC5                                                GPIO_BSC_BC5_Msk                                               /*!< Port x bit Clear bit 5 */
#define GPIO_BSC_BC6_Pos                                            (22U)
#define GPIO_BSC_BC6_Msk                                            (0x1UL << GPIO_BSC_BC6_Pos)                                    /*!< 0x00400000 */
#define GPIO_BSC_BC6                                                GPIO_BSC_BC6_Msk                                               /*!< Port x bit Clear bit 6 */
#define GPIO_BSC_BC7_Pos                                            (23U)
#define GPIO_BSC_BC7_Msk                                            (0x1UL << GPIO_BSC_BC7_Pos)                                    /*!< 0x00800000 */
#define GPIO_BSC_BC7                                                GPIO_BSC_BC7_Msk                                               /*!< Port x bit Clear bit 7 */
#define GPIO_BSC_BC8_Pos                                            (24U)
#define GPIO_BSC_BC8_Msk                                            (0x1UL << GPIO_BSC_BC8_Pos)                                    /*!< 0x01000000 */
#define GPIO_BSC_BC8                                                GPIO_BSC_BC8_Msk                                               /*!< Port x bit Clear bit 8 */
#define GPIO_BSC_BC9_Pos                                            (25U)
#define GPIO_BSC_BC9_Msk                                            (0x1UL << GPIO_BSC_BC9_Pos)                                    /*!< 0x02000000 */
#define GPIO_BSC_BC9                                                GPIO_BSC_BC9_Msk                                               /*!< Port x bit Clear bit 9 */
#define GPIO_BSC_BC10_Pos                                           (26U)
#define GPIO_BSC_BC10_Msk                                           (0x1UL << GPIO_BSC_BC10_Pos)                                   /*!< 0x04000000 */
#define GPIO_BSC_BC10                                               GPIO_BSC_BC10_Msk                                              /*!< Port x bit Clear bit 10 */
#define GPIO_BSC_BC11_Pos                                           (27U)
#define GPIO_BSC_BC11_Msk                                           (0x1UL << GPIO_BSC_BC11_Pos)                                   /*!< 0x08000000 */
#define GPIO_BSC_BC11                                               GPIO_BSC_BC11_Msk                                              /*!< Port x bit Clear bit 11 */
#define GPIO_BSC_BC12_Pos                                           (28U)
#define GPIO_BSC_BC12_Msk                                           (0x1UL << GPIO_BSC_BC12_Pos)                                   /*!< 0x10000000 */
#define GPIO_BSC_BC12                                               GPIO_BSC_BC12_Msk                                              /*!< Port x bit Clear bit 12 */
#define GPIO_BSC_BC13_Pos                                           (29U)
#define GPIO_BSC_BC13_Msk                                           (0x1UL << GPIO_BSC_BC13_Pos)                                   /*!< 0x20000000 */
#define GPIO_BSC_BC13                                               GPIO_BSC_BC13_Msk                                              /*!< Port x bit Clear bit 13 */
#define GPIO_BSC_BC14_Pos                                           (30U)
#define GPIO_BSC_BC14_Msk                                           (0x1UL << GPIO_BSC_BC14_Pos)                                   /*!< 0x40000000 */
#define GPIO_BSC_BC14                                               GPIO_BSC_BC14_Msk                                              /*!< Port x bit Clear bit 14 */
#define GPIO_BSC_BC15_Pos                                           (31U)
#define GPIO_BSC_BC15_Msk                                           (0x1UL << GPIO_BSC_BC15_Pos)                                   /*!< 0x80000000 */
#define GPIO_BSC_BC15                                               GPIO_BSC_BC15_Msk                                              /*!< Port x bit Clear bit 15 */

/******************  Bits definition for GPIO_LOCK register  *********************/
#define GPIO_LOCK_LOCK0_Pos                                         (0U)
#define GPIO_LOCK_LOCK0_Msk                                         (0x1UL << GPIO_LOCK_LOCK0_Pos)                                 /*!< 0x00000001 */
#define GPIO_LOCK_LOCK0                                             GPIO_LOCK_LOCK0_Msk                                            /*!< Port x Lock bit 0 */
#define GPIO_LOCK_LOCK1_Pos                                         (1U)
#define GPIO_LOCK_LOCK1_Msk                                         (0x1UL << GPIO_LOCK_LOCK1_Pos)                                 /*!< 0x00000002 */
#define GPIO_LOCK_LOCK1                                             GPIO_LOCK_LOCK1_Msk                                            /*!< Port x Lock bit 1 */
#define GPIO_LOCK_LOCK2_Pos                                         (2U)
#define GPIO_LOCK_LOCK2_Msk                                         (0x1UL << GPIO_LOCK_LOCK2_Pos)                                 /*!< 0x00000004 */
#define GPIO_LOCK_LOCK2                                             GPIO_LOCK_LOCK2_Msk                                            /*!< Port x Lock bit 2 */
#define GPIO_LOCK_LOCK3_Pos                                         (3U)
#define GPIO_LOCK_LOCK3_Msk                                         (0x1UL << GPIO_LOCK_LOCK3_Pos)                                 /*!< 0x00000008 */
#define GPIO_LOCK_LOCK3                                             GPIO_LOCK_LOCK3_Msk                                            /*!< Port x Lock bit 3 */
#define GPIO_LOCK_LOCK4_Pos                                         (4U)
#define GPIO_LOCK_LOCK4_Msk                                         (0x1UL << GPIO_LOCK_LOCK4_Pos)                                 /*!< 0x00000010 */
#define GPIO_LOCK_LOCK4                                             GPIO_LOCK_LOCK4_Msk                                            /*!< Port x Lock bit 4 */
#define GPIO_LOCK_LOCK5_Pos                                         (5U)
#define GPIO_LOCK_LOCK5_Msk                                         (0x1UL << GPIO_LOCK_LOCK5_Pos)                                 /*!< 0x00000020 */
#define GPIO_LOCK_LOCK5                                             GPIO_LOCK_LOCK5_Msk                                            /*!< Port x Lock bit 5 */
#define GPIO_LOCK_LOCK6_Pos                                         (6U)
#define GPIO_LOCK_LOCK6_Msk                                         (0x1UL << GPIO_LOCK_LOCK6_Pos)                                 /*!< 0x00000040 */
#define GPIO_LOCK_LOCK6                                             GPIO_LOCK_LOCK6_Msk                                            /*!< Port x Lock bit 6 */
#define GPIO_LOCK_LOCK7_Pos                                         (7U)
#define GPIO_LOCK_LOCK7_Msk                                         (0x1UL << GPIO_LOCK_LOCK7_Pos)                                 /*!< 0x00000080 */
#define GPIO_LOCK_LOCK7                                             GPIO_LOCK_LOCK7_Msk                                            /*!< Port x Lock bit 7 */
#define GPIO_LOCK_LOCK8_Pos                                         (8U)
#define GPIO_LOCK_LOCK8_Msk                                         (0x1UL << GPIO_LOCK_LOCK8_Pos)                                 /*!< 0x00000100 */
#define GPIO_LOCK_LOCK8                                             GPIO_LOCK_LOCK8_Msk                                            /*!< Port x Lock bit 8 */
#define GPIO_LOCK_LOCK9_Pos                                         (9U)
#define GPIO_LOCK_LOCK9_Msk                                         (0x1UL << GPIO_LOCK_LOCK9_Pos)                                 /*!< 0x00000200 */
#define GPIO_LOCK_LOCK9                                             GPIO_LOCK_LOCK9_Msk                                            /*!< Port x Lock bit 9 */
#define GPIO_LOCK_LOCK10_Pos                                        (10U)
#define GPIO_LOCK_LOCK10_Msk                                        (0x1UL << GPIO_LOCK_LOCK10_Pos)                                /*!< 0x00000400 */
#define GPIO_LOCK_LOCK10                                            GPIO_LOCK_LOCK10_Msk                                           /*!< Port x Lock bit 10 */
#define GPIO_LOCK_LOCK11_Pos                                        (11U)
#define GPIO_LOCK_LOCK11_Msk                                        (0x1UL << GPIO_LOCK_LOCK11_Pos)                                /*!< 0x00000800 */
#define GPIO_LOCK_LOCK11                                            GPIO_LOCK_LOCK11_Msk                                           /*!< Port x Lock bit 11 */
#define GPIO_LOCK_LOCK12_Pos                                        (12U)
#define GPIO_LOCK_LOCK12_Msk                                        (0x1UL << GPIO_LOCK_LOCK12_Pos)                                /*!< 0x00001000 */
#define GPIO_LOCK_LOCK12                                            GPIO_LOCK_LOCK12_Msk                                           /*!< Port x Lock bit 12 */
#define GPIO_LOCK_LOCK13_Pos                                        (13U)
#define GPIO_LOCK_LOCK13_Msk                                        (0x1UL << GPIO_LOCK_LOCK13_Pos)                                /*!< 0x00002000 */
#define GPIO_LOCK_LOCK13                                            GPIO_LOCK_LOCK13_Msk                                           /*!< Port x Lock bit 13 */
#define GPIO_LOCK_LOCK14_Pos                                        (14U)
#define GPIO_LOCK_LOCK14_Msk                                        (0x1UL << GPIO_LOCK_LOCK14_Pos)                                /*!< 0x00004000 */
#define GPIO_LOCK_LOCK14                                            GPIO_LOCK_LOCK14_Msk                                           /*!< Port x Lock bit 14 */
#define GPIO_LOCK_LOCK15_Pos                                        (15U)
#define GPIO_LOCK_LOCK15_Msk                                        (0x1UL << GPIO_LOCK_LOCK15_Pos)                                /*!< 0x00008000 */
#define GPIO_LOCK_LOCK15                                            GPIO_LOCK_LOCK15_Msk                                           /*!< Port x Lock bit 15 */
#define GPIO_LOCK_LOCKKEY_Pos                                       (16U)
#define GPIO_LOCK_LOCKKEY_Msk                                       (0x1UL << GPIO_LOCK_LOCKKEY_Pos)                               /*!< 0x00010000 */
#define GPIO_LOCK_LOCKKEY                                           GPIO_LOCK_LOCKKEY_Msk                                          /*!< Lock key */

/******************************************************************************/
/*                                                                            */
/*                      Alternate Function I/O Configuration                  */
/*                                                                            */
/******************************************************************************/
/*****************  Bits definition for AFIO_EVCTRL register  ******************/
#define AFIO_EVCTRL_PINSEL_Pos                                      (0U)
#define AFIO_EVCTRL_PINSEL_Msk                                      (0xFUL << AFIO_EVCTRL_PINSEL_Pos)                              /*!< 0x0000000F */
#define AFIO_EVCTRL_PINSEL                                          AFIO_EVCTRL_PINSEL_Msk                                         /*!< PINSEL[3:0] bits (Pin selection) */
#define AFIO_EVCTRL_PINSEL_0                                        (0x1UL << AFIO_EVCTRL_PINSEL_Pos)                              /*!< 0x00000001 */
#define AFIO_EVCTRL_PINSEL_1                                        (0x2UL << AFIO_EVCTRL_PINSEL_Pos)                              /*!< 0x00000002 */
#define AFIO_EVCTRL_PINSEL_2                                        (0x4UL << AFIO_EVCTRL_PINSEL_Pos)                              /*!< 0x00000004 */
#define AFIO_EVCTRL_PINSEL_3                                        (0x8UL << AFIO_EVCTRL_PINSEL_Pos)                              /*!< 0x00000008 */

/*!< PIN configuration */
#define AFIO_EVCTRL_PINSEL_PX0                                      0x00000000U                                                    /*!< Pin 0 selected */
#define AFIO_EVCTRL_PINSEL_PX1_Pos                                  (0U)
#define AFIO_EVCTRL_PINSEL_PX1_Msk                                  (0x1UL << AFIO_EVCTRL_PINSEL_PX1_Pos)                          /*!< 0x00000001 */
#define AFIO_EVCTRL_PINSEL_PX1                                      AFIO_EVCTRL_PINSEL_PX1_Msk                                     /*!< Pin 1 selected */
#define AFIO_EVCTRL_PINSEL_PX2_Pos                                  (1U)
#define AFIO_EVCTRL_PINSEL_PX2_Msk                                  (0x1UL << AFIO_EVCTRL_PINSEL_PX2_Pos)                          /*!< 0x00000002 */
#define AFIO_EVCTRL_PINSEL_PX2                                      AFIO_EVCTRL_PINSEL_PX2_Msk                                     /*!< Pin 2 selected */
#define AFIO_EVCTRL_PINSEL_PX3_Pos                                  (0U)
#define AFIO_EVCTRL_PINSEL_PX3_Msk                                  (0x3UL << AFIO_EVCTRL_PINSEL_PX3_Pos)                          /*!< 0x00000003 */
#define AFIO_EVCTRL_PINSEL_PX3                                      AFIO_EVCTRL_PINSEL_PX3_Msk                                     /*!< Pin 3 selected */
#define AFIO_EVCTRL_PINSEL_PX4_Pos                                  (2U)
#define AFIO_EVCTRL_PINSEL_PX4_Msk                                  (0x1UL << AFIO_EVCTRL_PINSEL_PX4_Pos)                          /*!< 0x00000004 */
#define AFIO_EVCTRL_PINSEL_PX4                                      AFIO_EVCTRL_PINSEL_PX4_Msk                                     /*!< Pin 4 selected */
#define AFIO_EVCTRL_PINSEL_PX5_Pos                                  (0U)
#define AFIO_EVCTRL_PINSEL_PX5_Msk                                  (0x5UL << AFIO_EVCTRL_PINSEL_PX5_Pos)                          /*!< 0x00000005 */
#define AFIO_EVCTRL_PINSEL_PX5                                      AFIO_EVCTRL_PINSEL_PX5_Msk                                     /*!< Pin 5 selected */
#define AFIO_EVCTRL_PINSEL_PX6_Pos                                  (1U)
#define AFIO_EVCTRL_PINSEL_PX6_Msk                                  (0x3UL << AFIO_EVCTRL_PINSEL_PX6_Pos)                          /*!< 0x00000006 */
#define AFIO_EVCTRL_PINSEL_PX6                                      AFIO_EVCTRL_PINSEL_PX6_Msk                                     /*!< Pin 6 selected */
#define AFIO_EVCTRL_PINSEL_PX7_Pos                                  (0U)
#define AFIO_EVCTRL_PINSEL_PX7_Msk                                  (0x7UL << AFIO_EVCTRL_PINSEL_PX7_Pos)                          /*!< 0x00000007 */
#define AFIO_EVCTRL_PINSEL_PX7                                      AFIO_EVCTRL_PINSEL_PX7_Msk                                     /*!< Pin 7 selected */
#define AFIO_EVCTRL_PINSEL_PX8_Pos                                  (3U)
#define AFIO_EVCTRL_PINSEL_PX8_Msk                                  (0x1UL << AFIO_EVCTRL_PINSEL_PX8_Pos)                          /*!< 0x00000008 */
#define AFIO_EVCTRL_PINSEL_PX8                                      AFIO_EVCTRL_PINSEL_PX8_Msk                                     /*!< Pin 8 selected */
#define AFIO_EVCTRL_PINSEL_PX9_Pos                                  (0U)
#define AFIO_EVCTRL_PINSEL_PX9_Msk                                  (0x9UL << AFIO_EVCTRL_PINSEL_PX9_Pos)                          /*!< 0x00000009 */
#define AFIO_EVCTRL_PINSEL_PX9                                      AFIO_EVCTRL_PINSEL_PX9_Msk                                     /*!< Pin 9 selected */
#define AFIO_EVCTRL_PINSEL_PX10_Pos                                 (1U)
#define AFIO_EVCTRL_PINSEL_PX10_Msk                                 (0x5UL << AFIO_EVCTRL_PINSEL_PX10_Pos)                         /*!< 0x0000000A */
#define AFIO_EVCTRL_PINSEL_PX10                                     AFIO_EVCTRL_PINSEL_PX10_Msk                                    /*!< Pin 10 selected */
#define AFIO_EVCTRL_PINSEL_PX11_Pos                                 (0U)
#define AFIO_EVCTRL_PINSEL_PX11_Msk                                 (0xBUL << AFIO_EVCTRL_PINSEL_PX11_Pos)                         /*!< 0x0000000B */
#define AFIO_EVCTRL_PINSEL_PX11                                     AFIO_EVCTRL_PINSEL_PX11_Msk                                    /*!< Pin 11 selected */
#define AFIO_EVCTRL_PINSEL_PX12_Pos                                 (2U)
#define AFIO_EVCTRL_PINSEL_PX12_Msk                                 (0x3UL << AFIO_EVCTRL_PINSEL_PX12_Pos)                         /*!< 0x0000000C */
#define AFIO_EVCTRL_PINSEL_PX12                                     AFIO_EVCTRL_PINSEL_PX12_Msk                                    /*!< Pin 12 selected */
#define AFIO_EVCTRL_PINSEL_PX13_Pos                                 (0U)
#define AFIO_EVCTRL_PINSEL_PX13_Msk                                 (0xDUL << AFIO_EVCTRL_PINSEL_PX13_Pos)                         /*!< 0x0000000D */
#define AFIO_EVCTRL_PINSEL_PX13                                     AFIO_EVCTRL_PINSEL_PX13_Msk                                    /*!< Pin 13 selected */
#define AFIO_EVCTRL_PINSEL_PX14_Pos                                 (1U)
#define AFIO_EVCTRL_PINSEL_PX14_Msk                                 (0x7UL << AFIO_EVCTRL_PINSEL_PX14_Pos)                         /*!< 0x0000000E */
#define AFIO_EVCTRL_PINSEL_PX14                                     AFIO_EVCTRL_PINSEL_PX14_Msk                                    /*!< Pin 14 selected */
#define AFIO_EVCTRL_PINSEL_PX15_Pos                                 (0U)
#define AFIO_EVCTRL_PINSEL_PX15_Msk                                 (0xFUL << AFIO_EVCTRL_PINSEL_PX15_Pos)                         /*!< 0x0000000F */
#define AFIO_EVCTRL_PINSEL_PX15                                     AFIO_EVCTRL_PINSEL_PX15_Msk                                    /*!< Pin 15 selected */

#define AFIO_EVCTRL_PORTSEL_Pos                                     (4U)
#define AFIO_EVCTRL_PORTSEL_Msk                                     (0x7UL << AFIO_EVCTRL_PORTSEL_Pos)                             /*!< 0x00000070 */
#define AFIO_EVCTRL_PORTSEL                                         AFIO_EVCTRL_PORTSEL_Msk                                        /*!< PORTSEL[2:0] bits (Port selection) */
#define AFIO_EVCTRL_PORTSEL_0                                       (0x1UL << AFIO_EVCTRL_PORTSEL_Pos)                             /*!< 0x00000010 */
#define AFIO_EVCTRL_PORTSEL_1                                       (0x2UL << AFIO_EVCTRL_PORTSEL_Pos)                             /*!< 0x00000020 */
#define AFIO_EVCTRL_PORTSEL_2                                       (0x4UL << AFIO_EVCTRL_PORTSEL_Pos)                             /*!< 0x00000040 */

/*!< PORT configuration */
#define AFIO_EVCTRL_PORTSEL_PA                                      0x00000000U                                                    /*!< Port A selected */
#define AFIO_EVCTRL_PORTSEL_PB_Pos                                  (4U)
#define AFIO_EVCTRL_PORTSEL_PB_Msk                                  (0x1UL << AFIO_EVCTRL_PORTSEL_PB_Pos)                          /*!< 0x00000010 */
#define AFIO_EVCTRL_PORTSEL_PB                                      AFIO_EVCTRL_PORTSEL_PB_Msk                                     /*!< Port B selected */
#define AFIO_EVCTRL_PORTSEL_PC_Pos                                  (5U)
#define AFIO_EVCTRL_PORTSEL_PC_Msk                                  (0x3UL << AFIO_EVCTRL_PORTSEL_PC_Pos)                          /*!< 0x00000030 */
#define AFIO_EVCTRL_PORTSEL_PC                                      AFIO_EVCTRL_PORTSEL_PC_Msk                                     /*!< Port C selected */
#define AFIO_EVCTRL_PORTSEL_PD_Pos                                  (4U)
#define AFIO_EVCTRL_PORTSEL_PD_Msk                                  (0x5UL << AFIO_EVCTRL_PORTSEL_PD_Pos)                          /*!< 0x00000050 */
#define AFIO_EVCTRL_PORTSEL_PD                                      AFIO_EVCTRL_PORTSEL_PD_Msk                                     /*!< Port D selected */

#define AFIO_EVCTRL_EVOEN_Pos                                       (7U)
#define AFIO_EVCTRL_EVOEN_Msk                                       (0x1UL << AFIO_EVCTRL_EVOEN_Pos)                               /*!< 0x00000080 */
#define AFIO_EVCTRL_EVOEN                                           AFIO_EVCTRL_EVOEN_Msk                                          /*!< Event Output Enable */

/*****************  Bits definition for AFIO_REMAP1 register  ******************/
#define AFIO_REMAP1_SPI1_RMP_Pos                                    (0U)
#define AFIO_REMAP1_SPI1_RMP_Msk                                    (0x1UL << AFIO_REMAP1_SPI1_RMP_Pos)                            /*!< 0x00000001 */
#define AFIO_REMAP1_SPI1_RMP                                        AFIO_REMAP1_SPI1_RMP_Msk                                       /*!< SPI1 remapping */
#define AFIO_REMAP1_I2C1_RMP_Pos                                    (1U)
#define AFIO_REMAP1_I2C1_RMP_Msk                                    (0x1UL << AFIO_REMAP1_I2C1_RMP_Pos)                            /*!< 0x00000002 */
#define AFIO_REMAP1_I2C1_RMP                                        AFIO_REMAP1_I2C1_RMP_Msk                                       /*!< I2C1 remapping */
#define AFIO_REMAP1_USART1_RMP_Pos                                  (2U)
#define AFIO_REMAP1_USART1_RMP_Msk                                  (0x1UL << AFIO_REMAP1_USART1_RMP_Pos)                          /*!< 0x00000004 */
#define AFIO_REMAP1_USART1_RMP                                      AFIO_REMAP1_USART1_RMP_Msk                                     /*!< USART1 remapping */
#define AFIO_REMAP1_USART2_RMP_Pos                                  (3U)
#define AFIO_REMAP1_USART2_RMP_Msk                                  (0x1UL << AFIO_REMAP1_USART2_RMP_Pos)                          /*!< 0x00000008 */
#define AFIO_REMAP1_USART2_RMP                                      AFIO_REMAP1_USART2_RMP_Msk                                     /*!< USART2 remapping */
#define AFIO_REMAP1_USART3_RMP_Pos                                  (4U)
#define AFIO_REMAP1_USART3_RMP_Msk                                  (0x3UL << AFIO_REMAP1_USART3_RMP_Pos)                          /*!< 0x00000030 */
#define AFIO_REMAP1_USART3_RMP                                      AFIO_REMAP1_USART3_RMP_Msk                                     /*!< USART3RMP[1:0] bits (USART3 remapping) */
#define AFIO_REMAP1_USART3_RMP_0                                    (0x1UL << AFIO_REMAP1_USART3_RMP_Pos)                          /*!< 0x00000010 */
#define AFIO_REMAP1_USART3_RMP_1                                    (0x2UL << AFIO_REMAP1_USART3_RMP_Pos)                          /*!< 0x00000020 */

/* USART3_RMP configuration */
#define AFIO_REMAP1_USART3_RMP_NOREMAP                              0x00000000U                                                    /*!< No remap (TX/PB10, RX/PB11, CK = PC12, CTS = PB13, RTS = PB14) */
#define AFIO_REMAP1_USART3_RMP_PARTIALREMAP_Pos                     (4U)
#define AFIO_REMAP1_USART3_RMP_PARTIALREMAP_Msk                     (0x1UL << AFIO_REMAP1_USART3_RMP_PARTIALREMAP_Pos)             /*!< 0x00000010 */
#define AFIO_REMAP1_USART3_RMP_PARTIALREMAP                         AFIO_REMAP1_USART3_RMP_PARTIALREMAP_Msk                        /*!< Partial remap (TX/PC10, RX/PC11, CK = PC12, CTS = PB13, RTS = PB14) */
#define AFIO_REMAP1_USART3_RMP_FULLREMAP_Pos                        (4U)
#define AFIO_REMAP1_USART3_RMP_FULLREMAP_Msk                        (0x3UL << AFIO_REMAP1_USART3_RMP_FULLREMAP_Pos)                /*!< 0x00000030 */
#define AFIO_REMAP1_USART3_RMP_FULLREMAP                            AFIO_REMAP1_USART3_RMP_FULLREMAP_Msk                           /*!< Full remap (Invalid) */

#define AFIO_REMAP1_TMR1_RMP_Pos                                    (6U)
#define AFIO_REMAP1_TMR1_RMP_Msk                                    (0x3UL << AFIO_REMAP1_TMR1_RMP_Pos)                            /*!< 0x000000C0 */
#define AFIO_REMAP1_TMR1_RMP                                        AFIO_REMAP1_TMR1_RMP_Msk                                       /*!< TMR1_RMP[1:0] bits (Timer1 remapping) */
#define AFIO_REMAP1_TMR1_RMP_0                                      (0x1UL << AFIO_REMAP1_TMR1_RMP_Pos)                            /*!< 0x00000040 */
#define AFIO_REMAP1_TMR1_RMP_1                                      (0x2UL << AFIO_REMAP1_TMR1_RMP_Pos)                            /*!< 0x00000080 */

/* TMR1_RMP configuration */
#define AFIO_REMAP1_TMR1_RMP_NOREMAP                                0x00000000U                                                    /*!< No remap (ETR/PA12, CH1/PA8, CH2/PA9, CH3/PA10, CH4/PA11, BKIN/PB12, CN1N/PB13, CH2N/PB14, CH3N/PB15) */
#define AFIO_REMAP1_TMR1_RMP_PARTIALREMAP_Pos                       (6U)
#define AFIO_REMAP1_TMR1_RMP_PARTIALREMAP_Msk                       (0x1UL << AFIO_REMAP1_TMR1_RMP_PARTIALREMAP_Pos)               /*!< 0x00000040 */
#define AFIO_REMAP1_TMR1_RMP_PARTIALREMAP                           AFIO_REMAP1_TMR1_RMP_PARTIALREMAP_Msk                          /*!< Partial remap (ETR/PA12, CH1/PA8, CH2/PA9, CH3/PA10, CH4/PA11, BKIN/PA6, CN1N/PA7, CH2N/PB0, CH3N/PB1) */
#define AFIO_REMAP1_TMR1_RMP_FULLREMAP_Pos                          (6U)
#define AFIO_REMAP1_TMR1_RMP_FULLREMAP_Msk                          (0x3UL << AFIO_REMAP1_TMR1_RMP_FULLREMAP_Pos)                  /*!< 0x000000C0 */
#define AFIO_REMAP1_TMR1_RMP_FULLREMAP                              AFIO_REMAP1_TMR1_RMP_FULLREMAP_Msk                             /*!< Full remap (Invalid) */

#define AFIO_REMAP1_TMR2_RMP_Pos                                    (8U)
#define AFIO_REMAP1_TMR2_RMP_Msk                                    (0x3UL << AFIO_REMAP1_TMR2_RMP_Pos)                            /*!< 0x00000300 */
#define AFIO_REMAP1_TMR2_RMP                                        AFIO_REMAP1_TMR2_RMP_Msk                                       /*!< TMR2_RMP[1:0] bits (Timer2 remapping) */
#define AFIO_REMAP1_TMR2_RMP_0                                      (0x1UL << AFIO_REMAP1_TMR2_RMP_Pos)                            /*!< 0x00000100 */
#define AFIO_REMAP1_TMR2_RMP_1                                      (0x2UL << AFIO_REMAP1_TMR2_RMP_Pos)                            /*!< 0x00000200 */

/* TMR2_RMP configuration */
#define AFIO_REMAP1_TMR2_RMP_NOREMAP                                0x00000000U                                                    /*!< No remap (CH1/ETR/PA0, CH2/PA1, CH3/PA2, CH4/PA3) */
#define AFIO_REMAP1_TMR2_RMP_PARTIALREMAP1_Pos                      (8U)
#define AFIO_REMAP1_TMR2_RMP_PARTIALREMAP1_Msk                      (0x1UL << AFIO_REMAP1_TMR2_RMP_PARTIALREMAP1_Pos)              /*!< 0x00000100 */
#define AFIO_REMAP1_TMR2_RMP_PARTIALREMAP1                          AFIO_REMAP1_TMR2_RMP_PARTIALREMAP1_Msk                         /*!< Partial remap (CH1/ETR/PA15, CH2/PB3, CH3/PA2, CH4/PA3) */
#define AFIO_REMAP1_TMR2_RMP_PARTIALREMAP2_Pos                      (9U)
#define AFIO_REMAP1_TMR2_RMP_PARTIALREMAP2_Msk                      (0x1UL << AFIO_REMAP1_TMR2_RMP_PARTIALREMAP2_Pos)              /*!< 0x00000200 */
#define AFIO_REMAP1_TMR2_RMP_PARTIALREMAP2                          AFIO_REMAP1_TMR2_RMP_PARTIALREMAP2_Msk                         /*!< Partial remap (CH1/ETR/PA0, CH2/PA1, CH3/PB10, CH4/PB11) */
#define AFIO_REMAP1_TMR2_RMP_FULLREMAP_Pos                          (8U)
#define AFIO_REMAP1_TMR2_RMP_FULLREMAP_Msk                          (0x3UL << AFIO_REMAP1_TMR2_RMP_FULLREMAP_Pos)                  /*!< 0x00000300 */
#define AFIO_REMAP1_TMR2_RMP_FULLREMAP                              AFIO_REMAP1_TMR2_RMP_FULLREMAP_Msk                             /*!< Full remap (CH1/ETR/PA15, CH2/PB3, CH3/PB10, CH4/PB11) */

#define AFIO_REMAP1_TMR3_RMP_Pos                                    (10U)
#define AFIO_REMAP1_TMR3_RMP_Msk                                    (0x3UL << AFIO_REMAP1_TMR3_RMP_Pos)                            /*!< 0x00000C00 */
#define AFIO_REMAP1_TMR3_RMP                                        AFIO_REMAP1_TMR3_RMP_Msk                                       /*!< TMR3_RMP[1:0] bits (Timer3 remapping) */
#define AFIO_REMAP1_TMR3_RMP_0                                      (0x1UL << AFIO_REMAP1_TMR3_RMP_Pos)                            /*!< 0x00000400 */
#define AFIO_REMAP1_TMR3_RMP_1                                      (0x2UL << AFIO_REMAP1_TMR3_RMP_Pos)                            /*!< 0x00000800 */

/* TMR3_RMP configuration */
#define AFIO_REMAP1_TMR3_RMP_NOREMAP                                0x00000000U                                                    /*!< No remap (CH1/PA6, CH2/PA7, CH3/PB0, CH4/PB1) */
#define AFIO_REMAP1_TMR3_RMP_PARTIALREMAP_Pos                       (11U)
#define AFIO_REMAP1_TMR3_RMP_PARTIALREMAP_Msk                       (0x1UL << AFIO_REMAP1_TMR3_RMP_PARTIALREMAP_Pos)               /*!< 0x00000800 */
#define AFIO_REMAP1_TMR3_RMP_PARTIALREMAP                           AFIO_REMAP1_TMR3_RMP_PARTIALREMAP_Msk                          /*!< Partial remap (CH1/PB4, CH2/PB5, CH3/PB0, CH4/PB1) */
#define AFIO_REMAP1_TMR3_RMP_FULLREMAP_Pos                          (10U)
#define AFIO_REMAP1_TMR3_RMP_FULLREMAP_Msk                          (0x3UL << AFIO_REMAP1_TMR3_RMP_FULLREMAP_Pos)                  /*!< 0x00000C00 */
#define AFIO_REMAP1_TMR3_RMP_FULLREMAP                              AFIO_REMAP1_TMR3_RMP_FULLREMAP_Msk                             /*!< Full remap (CH1/PC6, CH2/PC7, CH3/PC8, CH4/PC9) */

#define AFIO_REMAP1_TMR4_RMP_Pos                                    (12U)
#define AFIO_REMAP1_TMR4_RMP_Msk                                    (0x1UL << AFIO_REMAP1_TMR4_RMP_Pos)                            /*!< 0x00001000 */
#define AFIO_REMAP1_TMR4_RMP                                        AFIO_REMAP1_TMR4_RMP_Msk                                       /*!< TMR4_RMP bit (Timer4 remapping) */

#define AFIO_REMAP1_CAN1_RMP_Pos                                    (13U)
#define AFIO_REMAP1_CAN1_RMP_Msk                                    (0x3UL << AFIO_REMAP1_CAN1_RMP_Pos)                            /*!< 0x00006000 */
#define AFIO_REMAP1_CAN1_RMP                                        AFIO_REMAP1_CAN1_RMP_Msk                                       /*!< CAN1_RMP[1:0] bits (CAN1 remapping) */
#define AFIO_REMAP1_CAN1_RMP_0                                      (0x1UL << AFIO_REMAP1_CAN1_RMP_Pos)                            /*!< 0x00002000 */
#define AFIO_REMAP1_CAN1_RMP_1                                      (0x2UL << AFIO_REMAP1_CAN1_RMP_Pos)                            /*!< 0x00004000 */

/* CAN1_RMP configuration */
#define AFIO_REMAP1_CAN1_RMP_REMAP1                                 0x00000000U                                                    /*!< CAN1RX mapped to PA11, CAN1TX mapped to PA12 */
#define AFIO_REMAP1_CAN1_RMP_REMAP2_Pos                             (14U)
#define AFIO_REMAP1_CAN1_RMP_REMAP2_Msk                             (0x1UL << AFIO_REMAP1_CAN1_RMP_REMAP2_Pos)                     /*!< 0x00004000 */
#define AFIO_REMAP1_CAN1_RMP_REMAP2                                 AFIO_REMAP1_CAN1_RMP_REMAP2_Msk                                /*!< CAN1RX mapped to PB8, CAN1TX mapped to PB9 */
#define AFIO_REMAP1_CAN1_RMP_REMAP3_Pos                             (13U)
#define AFIO_REMAP1_CAN1_RMP_REMAP3_Msk                             (0x3UL << AFIO_REMAP1_CAN1_RMP_REMAP3_Pos)                     /*!< 0x00006000 */
#define AFIO_REMAP1_CAN1_RMP_REMAP3                                 AFIO_REMAP1_CAN1_RMP_REMAP3_Msk                                /*!< CAN1RX mapped to PD0, CAN1TX mapped to PD1 */

#define AFIO_REMAP1_PD01_RMP_Pos                                    (15U)
#define AFIO_REMAP1_PD01_RMP_Msk                                    (0x1UL << AFIO_REMAP1_PD01_RMP_Pos)                            /*!< 0x00008000 */
#define AFIO_REMAP1_PD01_RMP                                        AFIO_REMAP1_PD01_RMP_Msk                                       /*!< Port D0/Port D1 mapping on OSC_IN/OSC_OUT */
#define AFIO_REMAP1_TMR5CH4_IRMP_Pos                                (16U)
#define AFIO_REMAP1_TMR5CH4_IRMP_Msk                                (0x1UL << AFIO_REMAP1_TMR5CH4_IRMP_Pos)                        /*!< 0x00010000 */
#define AFIO_REMAP1_TMR5CH4_IRMP                                    AFIO_REMAP1_TMR5CH4_IRMP_Msk                                   /*!< TIM5 Channel4 Internal remap */
#define AFIO_REMAP1_ADC1_ETRGINJC_RMP_Pos                           (17U)
#define AFIO_REMAP1_ADC1_ETRGINJC_RMP_Msk                           (0x1UL << AFIO_REMAP1_ADC1_ETRGINJC_RMP_Pos)                   /*!< 0x00020000 */
#define AFIO_REMAP1_ADC1_ETRGINJC_RMP                               AFIO_REMAP1_ADC1_ETRGINJC_RMP_Msk                              /*!< ADC 1 External Trigger Injected Conversion remapping */
#define AFIO_REMAP1_ADC1_ETRGREGC_RMP_Pos                           (18U)
#define AFIO_REMAP1_ADC1_ETRGREGC_RMP_Msk                           (0x1UL << AFIO_REMAP1_ADC1_ETRGREGC_RMP_Pos)                   /*!< 0x00040000 */
#define AFIO_REMAP1_ADC1_ETRGREGC_RMP                               AFIO_REMAP1_ADC1_ETRGREGC_RMP_Msk                              /*!< ADC 1 External Trigger Regular Conversion remapping */
#define AFIO_REMAP1_ADC2_ETRGINJC_RMP_Pos                           (19U)
#define AFIO_REMAP1_ADC2_ETRGINJC_RMP_Msk                           (0x1UL << AFIO_REMAP1_ADC2_ETRGINJC_RMP_Pos)                   /*!< 0x00080000 */
#define AFIO_REMAP1_ADC2_ETRGINJC_RMP                               AFIO_REMAP1_ADC2_ETRGINJC_RMP_Msk                              /*!< ADC 2 External Trigger Injected Conversion remapping */
#define AFIO_REMAP1_ADC2_ETRGREGC_RMP_Pos                           (20U)
#define AFIO_REMAP1_ADC2_ETRGREGC_RMP_Msk                           (0x1UL << AFIO_REMAP1_ADC2_ETRGREGC_RMP_Pos)                   /*!< 0x00100000 */
#define AFIO_REMAP1_ADC2_ETRGREGC_RMP                               AFIO_REMAP1_ADC2_ETRGREGC_RMP_Msk                              /*!< ADC 2 External Trigger Regular Conversion remapping */
#define AFIO_REMAP1_CAN2_RMP_Pos                                    (22U)
#define AFIO_REMAP1_CAN2_RMP_Msk                                    (0x1UL << AFIO_REMAP1_CAN2_RMP_Pos)                            /*!< 0x00400000 */
#define AFIO_REMAP1_CAN2_RMP                                        AFIO_REMAP1_CAN2_RMP_Msk                                       /*!< CAN2RX mapped to PB5, CAN2TX mapped to PB6 */

/* SWJ_CFG configuration */
#define AFIO_REMAP1_SWJ_CFG_Pos                                     (24U)
#define AFIO_REMAP1_SWJ_CFG_Msk                                     (0x7UL << AFIO_REMAP1_SWJ_CFG_Pos)                             /*!< 0x07000000 */
#define AFIO_REMAP1_SWJ_CFG                                         AFIO_REMAP1_SWJ_CFG_Msk                                        /*!< SWJ_CFG[2:0] bits (Serial Wire JTAG configuration) */
#define AFIO_REMAP1_SWJ_CFG_0                                       (0x1UL << AFIO_REMAP1_SWJ_CFG_Pos)                             /*!< 0x01000000 */
#define AFIO_REMAP1_SWJ_CFG_1                                       (0x2UL << AFIO_REMAP1_SWJ_CFG_Pos)                             /*!< 0x02000000 */
#define AFIO_REMAP1_SWJ_CFG_2                                       (0x4UL << AFIO_REMAP1_SWJ_CFG_Pos)                             /*!< 0x04000000 */

#define AFIO_REMAP1_SWJ_CFG_RESET                                   0x00000000U                                                    /*!< Full SWJ (JTAG-DP + SW-DP) : Reset State */
#define AFIO_REMAP1_SWJ_CFG_NOJNTRST_Pos                            (24U)
#define AFIO_REMAP1_SWJ_CFG_NOJNTRST_Msk                            (0x1UL << AFIO_REMAP1_SWJ_CFG_NOJNTRST_Pos)                    /*!< 0x01000000 */
#define AFIO_REMAP1_SWJ_CFG_NOJNTRST                                AFIO_REMAP1_SWJ_CFG_NOJNTRST_Msk                               /*!< Full SWJ (JTAG-DP + SW-DP) but without JNTRST */
#define AFIO_REMAP1_SWJ_CFG_JTAGDISABLE_Pos                         (25U)
#define AFIO_REMAP1_SWJ_CFG_JTAGDISABLE_Msk                         (0x1UL << AFIO_REMAP1_SWJ_CFG_JTAGDISABLE_Pos)                 /*!< 0x02000000 */
#define AFIO_REMAP1_SWJ_CFG_JTAGDISABLE                             AFIO_REMAP1_SWJ_CFG_JTAGDISABLE_Msk                            /*!< JTAG-DP Disabled and SW-DP Enabled */
#define AFIO_REMAP1_SWJ_CFG_DISABLE_Pos                             (26U)
#define AFIO_REMAP1_SWJ_CFG_DISABLE_Msk                             (0x1UL << AFIO_REMAP1_SWJ_CFG_DISABLE_Pos)                     /*!< 0x04000000 */
#define AFIO_REMAP1_SWJ_CFG_DISABLE                                 AFIO_REMAP1_SWJ_CFG_DISABLE_Msk                                /*!< JTAG-DP Disabled and SW-DP Disabled */

/*****************  Bits definition for AFIO_EINTSEL1 register  ******************/
#define AFIO_EINTSEL1_EINT0_Pos                                     (0U)
#define AFIO_EINTSEL1_EINT0_Msk                                     (0xFUL << AFIO_EINTSEL1_EINT0_Pos)                             /*!< 0x0000000F */
#define AFIO_EINTSEL1_EINT0                                         AFIO_EINTSEL1_EINT0_Msk                                        /*!< EINT0[3:0] bits (External interrupt 0 configuration) */
#define AFIO_EINTSEL1_EINT1_Pos                                     (4U)
#define AFIO_EINTSEL1_EINT1_Msk                                     (0xFUL << AFIO_EINTSEL1_EINT1_Pos)                             /*!< 0x000000F0 */
#define AFIO_EINTSEL1_EINT1                                         AFIO_EINTSEL1_EINT1_Msk                                        /*!< EINT1[3:0] bits (External interrupt 1 configuration) */
#define AFIO_EINTSEL1_EINT2_Pos                                     (8U)
#define AFIO_EINTSEL1_EINT2_Msk                                     (0xFUL << AFIO_EINTSEL1_EINT2_Pos)                             /*!< 0x00000F00 */
#define AFIO_EINTSEL1_EINT2                                         AFIO_EINTSEL1_EINT2_Msk                                        /*!< EINT2[3:0] bits (External interrupt 2 configuration) */
#define AFIO_EINTSEL1_EINT3_Pos                                     (12U)
#define AFIO_EINTSEL1_EINT3_Msk                                     (0xFUL << AFIO_EINTSEL1_EINT3_Pos)                             /*!< 0x0000F000 */
#define AFIO_EINTSEL1_EINT3                                         AFIO_EINTSEL1_EINT3_Msk                                        /*!< EINT3[3:0] bits (External interrupt 3 configuration) */

/* EINT0 configuration */
#define AFIO_EINTSEL1_EINT0_PA                                      0x00000000U                                                    /*!< PA[0] pin */
#define AFIO_EINTSEL1_EINT0_PB_Pos                                  (0U)
#define AFIO_EINTSEL1_EINT0_PB_Msk                                  (0x1UL << AFIO_EINTSEL1_EINT0_PB_Pos)                          /*!< 0x00000001 */
#define AFIO_EINTSEL1_EINT0_PB                                      AFIO_EINTSEL1_EINT0_PB_Msk                                     /*!< PB[0] pin */
#define AFIO_EINTSEL1_EINT0_PC_Pos                                  (1U)
#define AFIO_EINTSEL1_EINT0_PC_Msk                                  (0x1UL << AFIO_EINTSEL1_EINT0_PC_Pos)                          /*!< 0x00000002 */
#define AFIO_EINTSEL1_EINT0_PC                                      AFIO_EINTSEL1_EINT0_PC_Msk                                     /*!< PC[0] pin */
#define AFIO_EINTSEL1_EINT0_PD_Pos                                  (0U)
#define AFIO_EINTSEL1_EINT0_PD_Msk                                  (0x3UL << AFIO_EINTSEL1_EINT0_PD_Pos)                          /*!< 0x00000003 */
#define AFIO_EINTSEL1_EINT0_PD                                      AFIO_EINTSEL1_EINT0_PD_Msk                                     /*!< PD[0] pin */

/* EINT1 configuration */
#define AFIO_EINTSEL1_EINT1_PA                                      0x00000000U                                                    /*!< PA[1] pin */
#define AFIO_EINTSEL1_EINT1_PB_Pos                                  (4U)
#define AFIO_EINTSEL1_EINT1_PB_Msk                                  (0x1UL << AFIO_EINTSEL1_EINT1_PB_Pos)                          /*!< 0x00000010 */
#define AFIO_EINTSEL1_EINT1_PB                                      AFIO_EINTSEL1_EINT1_PB_Msk                                     /*!< PB[1] pin */
#define AFIO_EINTSEL1_EINT1_PC_Pos                                  (5U)
#define AFIO_EINTSEL1_EINT1_PC_Msk                                  (0x1UL << AFIO_EINTSEL1_EINT1_PC_Pos)                          /*!< 0x00000020 */
#define AFIO_EINTSEL1_EINT1_PC                                      AFIO_EINTSEL1_EINT1_PC_Msk                                     /*!< PC[1] pin */
#define AFIO_EINTSEL1_EINT1_PD_Pos                                  (4U)
#define AFIO_EINTSEL1_EINT1_PD_Msk                                  (0x3UL << AFIO_EINTSEL1_EINT1_PD_Pos)                          /*!< 0x00000030 */
#define AFIO_EINTSEL1_EINT1_PD                                      AFIO_EINTSEL1_EINT1_PD_Msk                                     /*!< PD[1] pin */

/* EINT2 configuration */
#define AFIO_EINTSEL1_EINT2_PA                                      0x00000000U                                                    /*!< PA[2] pin */
#define AFIO_EINTSEL1_EINT2_PB_Pos                                  (8U)
#define AFIO_EINTSEL1_EINT2_PB_Msk                                  (0x1UL << AFIO_EINTSEL1_EINT2_PB_Pos)                          /*!< 0x00000100 */
#define AFIO_EINTSEL1_EINT2_PB                                      AFIO_EINTSEL1_EINT2_PB_Msk                                     /*!< PB[2] pin */
#define AFIO_EINTSEL1_EINT2_PC_Pos                                  (9U)
#define AFIO_EINTSEL1_EINT2_PC_Msk                                  (0x1UL << AFIO_EINTSEL1_EINT2_PC_Pos)                          /*!< 0x00000200 */
#define AFIO_EINTSEL1_EINT2_PC                                      AFIO_EINTSEL1_EINT2_PC_Msk                                     /*!< PC[2] pin */
#define AFIO_EINTSEL1_EINT2_PD_Pos                                  (8U)
#define AFIO_EINTSEL1_EINT2_PD_Msk                                  (0x3UL << AFIO_EINTSEL1_EINT2_PD_Pos)                          /*!< 0x00000300 */
#define AFIO_EINTSEL1_EINT2_PD                                      AFIO_EINTSEL1_EINT2_PD_Msk                                     /*!< PD[2] pin */

/* EINT3 configuration */
#define AFIO_EINTSEL1_EINT3_PA                                      0x00000000U                                                    /*!< PA[3] pin */
#define AFIO_EINTSEL1_EINT3_PB_Pos                                  (12U)
#define AFIO_EINTSEL1_EINT3_PB_Msk                                  (0x1UL << AFIO_EINTSEL1_EINT3_PB_Pos)                          /*!< 0x00001000 */
#define AFIO_EINTSEL1_EINT3_PB                                      AFIO_EINTSEL1_EINT3_PB_Msk                                     /*!< PB[3] pin */
#define AFIO_EINTSEL1_EINT3_PC_Pos                                  (13U)
#define AFIO_EINTSEL1_EINT3_PC_Msk                                  (0x1UL << AFIO_EINTSEL1_EINT3_PC_Pos)                          /*!< 0x00002000 */
#define AFIO_EINTSEL1_EINT3_PC                                      AFIO_EINTSEL1_EINT3_PC_Msk                                     /*!< PC[3] pin */
#define AFIO_EINTSEL1_EINT3_PD_Pos                                  (12U)
#define AFIO_EINTSEL1_EINT3_PD_Msk                                  (0x3UL << AFIO_EINTSEL1_EINT3_PD_Pos)                          /*!< 0x00003000 */
#define AFIO_EINTSEL1_EINT3_PD                                      AFIO_EINTSEL1_EINT3_PD_Msk                                     /*!< PD[3] pin */

/*****************  Bits definition for AFIO_EINTSEL2 register  ******************/
#define AFIO_EINTSEL2_EINT4_Pos                                     (0U)
#define AFIO_EINTSEL2_EINT4_Msk                                     (0xFUL << AFIO_EINTSEL2_EINT4_Pos)                             /*!< 0x0000000F */
#define AFIO_EINTSEL2_EINT4                                         AFIO_EINTSEL2_EINT4_Msk                                        /*!< EINT4[3:0] bits (External interrupt 4 configuration) */
#define AFIO_EINTSEL2_EINT5_Pos                                     (4U)
#define AFIO_EINTSEL2_EINT5_Msk                                     (0xFUL << AFIO_EINTSEL2_EINT5_Pos)                             /*!< 0x000000F0 */
#define AFIO_EINTSEL2_EINT5                                         AFIO_EINTSEL2_EINT5_Msk                                        /*!< EINT5[3:0] bits (External interrupt 5 configuration) */
#define AFIO_EINTSEL2_EINT6_Pos                                     (8U)
#define AFIO_EINTSEL2_EINT6_Msk                                     (0xFUL << AFIO_EINTSEL2_EINT6_Pos)                             /*!< 0x00000F00 */
#define AFIO_EINTSEL2_EINT6                                         AFIO_EINTSEL2_EINT6_Msk                                        /*!< EINT6[3:0] bits (External interrupt 6 configuration) */
#define AFIO_EINTSEL2_EINT7_Pos                                     (12U)
#define AFIO_EINTSEL2_EINT7_Msk                                     (0xFUL << AFIO_EINTSEL2_EINT7_Pos)                             /*!< 0x0000F000 */
#define AFIO_EINTSEL2_EINT7                                         AFIO_EINTSEL2_EINT7_Msk                                        /*!< EINT7[3:0] bits (External interrupt 7 configuration) */

/* EINT4 configuration */
#define AFIO_EINTSEL2_EINT4_PA                                      0x00000000U                                                    /*!< PA[4] pin */
#define AFIO_EINTSEL2_EINT4_PB_Pos                                  (0U)
#define AFIO_EINTSEL2_EINT4_PB_Msk                                  (0x1UL << AFIO_EINTSEL2_EINT4_PB_Pos)                          /*!< 0x00000001 */
#define AFIO_EINTSEL2_EINT4_PB                                      AFIO_EINTSEL2_EINT4_PB_Msk                                     /*!< PB[4] pin */
#define AFIO_EINTSEL2_EINT4_PC_Pos                                  (1U)
#define AFIO_EINTSEL2_EINT4_PC_Msk                                  (0x1UL << AFIO_EINTSEL2_EINT4_PC_Pos)                          /*!< 0x00000002 */
#define AFIO_EINTSEL2_EINT4_PC                                      AFIO_EINTSEL2_EINT4_PC_Msk                                     /*!< PC[4] pin */
#define AFIO_EINTSEL2_EINT4_PD_Pos                                  (0U)
#define AFIO_EINTSEL2_EINT4_PD_Msk                                  (0x3UL << AFIO_EINTSEL2_EINT4_PD_Pos)                          /*!< 0x00000003 */
#define AFIO_EINTSEL2_EINT4_PD                                      AFIO_EINTSEL2_EINT4_PD_Msk                                     /*!< PD[4] pin */

/* EINT5 configuration */
#define AFIO_EINTSEL2_EINT5_PA                                      0x00000000U                                                    /*!< PA[5] pin */
#define AFIO_EINTSEL2_EINT5_PB_Pos                                  (4U)
#define AFIO_EINTSEL2_EINT5_PB_Msk                                  (0x1UL << AFIO_EINTSEL2_EINT5_PB_Pos)                          /*!< 0x00000010 */
#define AFIO_EINTSEL2_EINT5_PB                                      AFIO_EINTSEL2_EINT5_PB_Msk                                     /*!< PB[5] pin */
#define AFIO_EINTSEL2_EINT5_PC_Pos                                  (5U)
#define AFIO_EINTSEL2_EINT5_PC_Msk                                  (0x1UL << AFIO_EINTSEL2_EINT5_PC_Pos)                          /*!< 0x00000020 */
#define AFIO_EINTSEL2_EINT5_PC                                      AFIO_EINTSEL2_EINT5_PC_Msk                                     /*!< PC[5] pin */
#define AFIO_EINTSEL2_EINT5_PD_Pos                                  (4U)
#define AFIO_EINTSEL2_EINT5_PD_Msk                                  (0x3UL << AFIO_EINTSEL2_EINT5_PD_Pos)                          /*!< 0x00000030 */
#define AFIO_EINTSEL2_EINT5_PD                                      AFIO_EINTSEL2_EINT5_PD_Msk                                     /*!< PD[5] pin */

/* EINT6 configuration */
#define AFIO_EINTSEL2_EINT6_PA                                      0x00000000U                                                    /*!< PA[6] pin */
#define AFIO_EINTSEL2_EINT6_PB_Pos                                  (8U)
#define AFIO_EINTSEL2_EINT6_PB_Msk                                  (0x1UL << AFIO_EINTSEL2_EINT6_PB_Pos)                          /*!< 0x00000100 */
#define AFIO_EINTSEL2_EINT6_PB                                      AFIO_EINTSEL2_EINT6_PB_Msk                                     /*!< PB[6] pin */
#define AFIO_EINTSEL2_EINT6_PC_Pos                                  (9U)
#define AFIO_EINTSEL2_EINT6_PC_Msk                                  (0x1UL << AFIO_EINTSEL2_EINT6_PC_Pos)                          /*!< 0x00000200 */
#define AFIO_EINTSEL2_EINT6_PC                                      AFIO_EINTSEL2_EINT6_PC_Msk                                     /*!< PC[6] pin */
#define AFIO_EINTSEL2_EINT6_PD_Pos                                  (8U)
#define AFIO_EINTSEL2_EINT6_PD_Msk                                  (0x3UL << AFIO_EINTSEL2_EINT6_PD_Pos)                          /*!< 0x00000300 */
#define AFIO_EINTSEL2_EINT6_PD                                      AFIO_EINTSEL2_EINT6_PD_Msk                                     /*!< PD[6] pin */

/* EINT7 configuration */
#define AFIO_EINTSEL2_EINT7_PA                                      0x00000000U                                                    /*!< PA[7] pin */
#define AFIO_EINTSEL2_EINT7_PB_Pos                                  (12U)
#define AFIO_EINTSEL2_EINT7_PB_Msk                                  (0x1UL << AFIO_EINTSEL2_EINT7_PB_Pos)                          /*!< 0x00001000 */
#define AFIO_EINTSEL2_EINT7_PB                                      AFIO_EINTSEL2_EINT7_PB_Msk                                     /*!< PB[7] pin */
#define AFIO_EINTSEL2_EINT7_PC_Pos                                  (13U)
#define AFIO_EINTSEL2_EINT7_PC_Msk                                  (0x1UL << AFIO_EINTSEL2_EINT7_PC_Pos)                          /*!< 0x00002000 */
#define AFIO_EINTSEL2_EINT7_PC                                      AFIO_EINTSEL2_EINT7_PC_Msk                                     /*!< PC[7] pin */
#define AFIO_EINTSEL2_EINT7_PD_Pos                                  (12U)
#define AFIO_EINTSEL2_EINT7_PD_Msk                                  (0x3UL << AFIO_EINTSEL2_EINT7_PD_Pos)                          /*!< 0x00003000 */
#define AFIO_EINTSEL2_EINT7_PD                                      AFIO_EINTSEL2_EINT7_PD_Msk                                     /*!< PD[7] pin */

/*****************  Bits definition for AFIO_EINTSEL3 register  ******************/
#define AFIO_EINTSEL3_EINT8_Pos                                     (0U)
#define AFIO_EINTSEL3_EINT8_Msk                                     (0xFUL << AFIO_EINTSEL3_EINT8_Pos)                             /*!< 0x0000000F */
#define AFIO_EINTSEL3_EINT8                                         AFIO_EINTSEL3_EINT8_Msk                                        /*!< EINT8[3:0] bits (External interrupt 8 configuration) */
#define AFIO_EINTSEL3_EINT9_Pos                                     (4U)
#define AFIO_EINTSEL3_EINT9_Msk                                     (0xFUL << AFIO_EINTSEL3_EINT9_Pos)                             /*!< 0x000000F0 */
#define AFIO_EINTSEL3_EINT9                                         AFIO_EINTSEL3_EINT9_Msk                                        /*!< EINT9[3:0] bits (External interrupt 9 configuration) */
#define AFIO_EINTSEL3_EINT10_Pos                                    (8U)
#define AFIO_EINTSEL3_EINT10_Msk                                    (0xFUL << AFIO_EINTSEL3_EINT10_Pos)                            /*!< 0x00000F00 */
#define AFIO_EINTSEL3_EINT10                                        AFIO_EINTSEL3_EINT10_Msk                                       /*!< EINT10[3:0] bits (External interrupt 10 configuration) */
#define AFIO_EINTSEL3_EINT11_Pos                                    (12U)
#define AFIO_EINTSEL3_EINT11_Msk                                    (0xFUL << AFIO_EINTSEL3_EINT11_Pos)                            /*!< 0x0000F000 */
#define AFIO_EINTSEL3_EINT11                                        AFIO_EINTSEL3_EINT11_Msk                                       /*!< EINT11[3:0] bits (External interrupt 11 configuration) */

/* EINT8 configuration */
#define AFIO_EINTSEL3_EINT8_PA                                      0x00000000U                                                    /*!< PA[8] pin */
#define AFIO_EINTSEL3_EINT8_PB_Pos                                  (0U)
#define AFIO_EINTSEL3_EINT8_PB_Msk                                  (0x1UL << AFIO_EINTSEL3_EINT8_PB_Pos)                          /*!< 0x00000001 */
#define AFIO_EINTSEL3_EINT8_PB                                      AFIO_EINTSEL3_EINT8_PB_Msk                                     /*!< PB[8] pin */
#define AFIO_EINTSEL3_EINT8_PC_Pos                                  (1U)
#define AFIO_EINTSEL3_EINT8_PC_Msk                                  (0x1UL << AFIO_EINTSEL3_EINT8_PC_Pos)                          /*!< 0x00000002 */
#define AFIO_EINTSEL3_EINT8_PC                                      AFIO_EINTSEL3_EINT8_PC_Msk                                     /*!< PC[8] pin */
#define AFIO_EINTSEL3_EINT8_PD_Pos                                  (0U)
#define AFIO_EINTSEL3_EINT8_PD_Msk                                  (0x3UL << AFIO_EINTSEL3_EINT8_PD_Pos)                          /*!< 0x00000003 */
#define AFIO_EINTSEL3_EINT8_PD                                      AFIO_EINTSEL3_EINT8_PD_Msk                                     /*!< PD[8] pin */

/* EINT9 configuration */
#define AFIO_EINTSEL3_EINT9_PA                                      0x00000000U                                                    /*!< PA[9] pin */
#define AFIO_EINTSEL3_EINT9_PB_Pos                                  (4U)
#define AFIO_EINTSEL3_EINT9_PB_Msk                                  (0x1UL << AFIO_EINTSEL3_EINT9_PB_Pos)                          /*!< 0x00000010 */
#define AFIO_EINTSEL3_EINT9_PB                                      AFIO_EINTSEL3_EINT9_PB_Msk                                     /*!< PB[9] pin */
#define AFIO_EINTSEL3_EINT9_PC_Pos                                  (5U)
#define AFIO_EINTSEL3_EINT9_PC_Msk                                  (0x1UL << AFIO_EINTSEL3_EINT9_PC_Pos)                          /*!< 0x00000020 */
#define AFIO_EINTSEL3_EINT9_PC                                      AFIO_EINTSEL3_EINT9_PC_Msk                                     /*!< PC[9] pin */
#define AFIO_EINTSEL3_EINT9_PD_Pos                                  (4U)
#define AFIO_EINTSEL3_EINT9_PD_Msk                                  (0x3UL << AFIO_EINTSEL3_EINT9_PD_Pos)                          /*!< 0x00000030 */
#define AFIO_EINTSEL3_EINT9_PD                                      AFIO_EINTSEL3_EINT9_PD_Msk                                     /*!< PD[9] pin */

/* EINT10 configuration */
#define AFIO_EINTSEL3_EINT10_PA                                     0x00000000U                                                    /*!< PA[10] pin */
#define AFIO_EINTSEL3_EINT10_PB_Pos                                 (8U)
#define AFIO_EINTSEL3_EINT10_PB_Msk                                 (0x1UL << AFIO_EINTSEL3_EINT10_PB_Pos)                         /*!< 0x00000100 */
#define AFIO_EINTSEL3_EINT10_PB                                     AFIO_EINTSEL3_EINT10_PB_Msk                                    /*!< PB[10] pin */
#define AFIO_EINTSEL3_EINT10_PC_Pos                                 (9U)
#define AFIO_EINTSEL3_EINT10_PC_Msk                                 (0x1UL << AFIO_EINTSEL3_EINT10_PC_Pos)                         /*!< 0x00000200 */
#define AFIO_EINTSEL3_EINT10_PC                                     AFIO_EINTSEL3_EINT10_PC_Msk                                    /*!< PC[10] pin */
#define AFIO_EINTSEL3_EINT10_PD_Pos                                 (8U)
#define AFIO_EINTSEL3_EINT10_PD_Msk                                 (0x3UL << AFIO_EINTSEL3_EINT10_PD_Pos)                         /*!< 0x00000300 */
#define AFIO_EINTSEL3_EINT10_PD                                     AFIO_EINTSEL3_EINT10_PD_Msk                                    /*!< PD[10] pin */

/* EINT11 configuration */
#define AFIO_EINTSEL3_EINT11_PA                                     0x00000000U                                                    /*!< PA[11] pin */
#define AFIO_EINTSEL3_EINT11_PB_Pos                                 (12U)
#define AFIO_EINTSEL3_EINT11_PB_Msk                                 (0x1UL << AFIO_EINTSEL3_EINT11_PB_Pos)                         /*!< 0x00001000 */
#define AFIO_EINTSEL3_EINT11_PB                                     AFIO_EINTSEL3_EINT11_PB_Msk                                    /*!< PB[11] pin */
#define AFIO_EINTSEL3_EINT11_PC_Pos                                 (13U)
#define AFIO_EINTSEL3_EINT11_PC_Msk                                 (0x1UL << AFIO_EINTSEL3_EINT11_PC_Pos)                         /*!< 0x00002000 */
#define AFIO_EINTSEL3_EINT11_PC                                     AFIO_EINTSEL3_EINT11_PC_Msk                                    /*!< PC[11] pin */
#define AFIO_EINTSEL3_EINT11_PD_Pos                                 (12U)
#define AFIO_EINTSEL3_EINT11_PD_Msk                                 (0x3UL << AFIO_EINTSEL3_EINT11_PD_Pos)                         /*!< 0x00003000 */
#define AFIO_EINTSEL3_EINT11_PD                                     AFIO_EINTSEL3_EINT11_PD_Msk                                    /*!< PD[11] pin */

/*****************  Bits definition for AFIO_EINTSEL4 register  ******************/
#define AFIO_EINTSEL4_EINT12_Pos                                    (0U)
#define AFIO_EINTSEL4_EINT12_Msk                                    (0xFUL << AFIO_EINTSEL4_EINT12_Pos)                            /*!< 0x0000000F */
#define AFIO_EINTSEL4_EINT12                                        AFIO_EINTSEL4_EINT12_Msk                                       /*!< EINT12[3:0] bits (External interrupt 12 configuration) */
#define AFIO_EINTSEL4_EINT13_Pos                                    (4U)
#define AFIO_EINTSEL4_EINT13_Msk                                    (0xFUL << AFIO_EINTSEL4_EINT13_Pos)                            /*!< 0x000000F0 */
#define AFIO_EINTSEL4_EINT13                                        AFIO_EINTSEL4_EINT13_Msk                                       /*!< EINT13[3:0] bits (External interrupt 13 configuration) */
#define AFIO_EINTSEL4_EINT14_Pos                                    (8U)
#define AFIO_EINTSEL4_EINT14_Msk                                    (0xFUL << AFIO_EINTSEL4_EINT14_Pos)                            /*!< 0x00000F00 */
#define AFIO_EINTSEL4_EINT14                                        AFIO_EINTSEL4_EINT14_Msk                                       /*!< EINT14[3:0] bits (External interrupt 14 configuration) */
#define AFIO_EINTSEL4_EINT15_Pos                                    (12U)
#define AFIO_EINTSEL4_EINT15_Msk                                    (0xFUL << AFIO_EINTSEL4_EINT15_Pos)                            /*!< 0x0000F000 */
#define AFIO_EINTSEL4_EINT15                                        AFIO_EINTSEL4_EINT15_Msk                                       /*!< EINT15[3:0] bits (External interrupt 15 configuration) */

/* EINT12 configuration */
#define AFIO_EINTSEL4_EINT12_PA                                     0x00000000U                                                    /*!< PA[12] pin */
#define AFIO_EINTSEL4_EINT12_PB_Pos                                 (0U)
#define AFIO_EINTSEL4_EINT12_PB_Msk                                 (0x1UL << AFIO_EINTSEL4_EINT12_PB_Pos)                         /*!< 0x00000001 */
#define AFIO_EINTSEL4_EINT12_PB                                     AFIO_EINTSEL4_EINT12_PB_Msk                                    /*!< PB[12] pin */
#define AFIO_EINTSEL4_EINT12_PC_Pos                                 (1U)
#define AFIO_EINTSEL4_EINT12_PC_Msk                                 (0x1UL << AFIO_EINTSEL4_EINT12_PC_Pos)                         /*!< 0x00000002 */
#define AFIO_EINTSEL4_EINT12_PC                                     AFIO_EINTSEL4_EINT12_PC_Msk                                    /*!< PC[12] pin */
#define AFIO_EINTSEL4_EINT12_PD_Pos                                 (0U)
#define AFIO_EINTSEL4_EINT12_PD_Msk                                 (0x3UL << AFIO_EINTSEL4_EINT12_PD_Pos)                         /*!< 0x00000003 */
#define AFIO_EINTSEL4_EINT12_PD                                     AFIO_EINTSEL4_EINT12_PD_Msk                                    /*!< PD[12] pin */

/* EINT13 configuration */
#define AFIO_EINTSEL4_EINT13_PA                                     0x00000000U                                                    /*!< PA[13] pin */
#define AFIO_EINTSEL4_EINT13_PB_Pos                                 (4U)
#define AFIO_EINTSEL4_EINT13_PB_Msk                                 (0x1UL << AFIO_EINTSEL4_EINT13_PB_Pos)                         /*!< 0x00000010 */
#define AFIO_EINTSEL4_EINT13_PB                                     AFIO_EINTSEL4_EINT13_PB_Msk                                    /*!< PB[13] pin */
#define AFIO_EINTSEL4_EINT13_PC_Pos                                 (5U)
#define AFIO_EINTSEL4_EINT13_PC_Msk                                 (0x1UL << AFIO_EINTSEL4_EINT13_PC_Pos)                         /*!< 0x00000020 */
#define AFIO_EINTSEL4_EINT13_PC                                     AFIO_EINTSEL4_EINT13_PC_Msk                                    /*!< PC[13] pin */
#define AFIO_EINTSEL4_EINT13_PD_Pos                                 (4U)
#define AFIO_EINTSEL4_EINT13_PD_Msk                                 (0x3UL << AFIO_EINTSEL4_EINT13_PD_Pos)                         /*!< 0x00000030 */
#define AFIO_EINTSEL4_EINT13_PD                                     AFIO_EINTSEL4_EINT13_PD_Msk                                    /*!< PD[13] pin */

/* EINT14 configuration */
#define AFIO_EINTSEL4_EINT14_PA                                     0x00000000U                                                    /*!< PA[14] pin */
#define AFIO_EINTSEL4_EINT14_PB_Pos                                 (8U)
#define AFIO_EINTSEL4_EINT14_PB_Msk                                 (0x1UL << AFIO_EINTSEL4_EINT14_PB_Pos)                         /*!< 0x00000100 */
#define AFIO_EINTSEL4_EINT14_PB                                     AFIO_EINTSEL4_EINT14_PB_Msk                                    /*!< PB[14] pin */
#define AFIO_EINTSEL4_EINT14_PC_Pos                                 (9U)
#define AFIO_EINTSEL4_EINT14_PC_Msk                                 (0x1UL << AFIO_EINTSEL4_EINT14_PC_Pos)                         /*!< 0x00000200 */
#define AFIO_EINTSEL4_EINT14_PC                                     AFIO_EINTSEL4_EINT14_PC_Msk                                    /*!< PC[14] pin */
#define AFIO_EINTSEL4_EINT14_PD_Pos                                 (8U)
#define AFIO_EINTSEL4_EINT14_PD_Msk                                 (0x3UL << AFIO_EINTSEL4_EINT14_PD_Pos)                         /*!< 0x00000300 */
#define AFIO_EINTSEL4_EINT14_PD                                     AFIO_EINTSEL4_EINT14_PD_Msk                                    /*!< PD[14] pin */

/* EINT15 configuration */
#define AFIO_EINTSEL4_EINT15_PA                                     0x00000000U                                                    /*!< PA[15] pin */
#define AFIO_EINTSEL4_EINT15_PB_Pos                                 (12U)
#define AFIO_EINTSEL4_EINT15_PB_Msk                                 (0x1UL << AFIO_EINTSEL4_EINT15_PB_Pos)                         /*!< 0x00001000 */
#define AFIO_EINTSEL4_EINT15_PB                                     AFIO_EINTSEL4_EINT15_PB_Msk                                    /*!< PB[15] pin */
#define AFIO_EINTSEL4_EINT15_PC_Pos                                 (13U)
#define AFIO_EINTSEL4_EINT15_PC_Msk                                 (0x1UL << AFIO_EINTSEL4_EINT15_PC_Pos)                         /*!< 0x00002000 */
#define AFIO_EINTSEL4_EINT15_PC                                     AFIO_EINTSEL4_EINT15_PC_Msk                                    /*!< PC[15] pin */
#define AFIO_EINTSEL4_EINT15_PD_Pos                                 (12U)
#define AFIO_EINTSEL4_EINT15_PD_Msk                                 (0x3UL << AFIO_EINTSEL4_EINT15_PD_Pos)                         /*!< 0x00003000 */
#define AFIO_EINTSEL4_EINT15_PD                                     AFIO_EINTSEL4_EINT15_PD_Msk                                    /*!< PD[15] pin */

/******************************************************************************/
/*                                                                            */
/*                      Inter-integrated Circuit Interface                    */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for I2C_CTRL1 register  ********************/
#define I2C_CTRL1_I2CEN_Pos                                         (0U)
#define I2C_CTRL1_I2CEN_Msk                                         (0x1UL << I2C_CTRL1_I2CEN_Pos)                                 /*!< 0x00000001 */
#define I2C_CTRL1_I2CEN                                             I2C_CTRL1_I2CEN_Msk                                            /*!<Peripheral Enable                             */
#define I2C_CTRL1_SMBEN_Pos                                         (1U)
#define I2C_CTRL1_SMBEN_Msk                                         (0x1UL << I2C_CTRL1_SMBEN_Pos)                                 /*!< 0x00000002 */
#define I2C_CTRL1_SMBEN                                             I2C_CTRL1_SMBEN_Msk                                            /*!<SMBus Mode                                    */
#define I2C_CTRL1_SMBTCFG_Pos                                       (3U)
#define I2C_CTRL1_SMBTCFG_Msk                                       (0x1UL << I2C_CTRL1_SMBTCFG_Pos)                               /*!< 0x00000008 */
#define I2C_CTRL1_SMBTCFG                                           I2C_CTRL1_SMBTCFG_Msk                                          /*!<SMBus Type                                    */
#define I2C_CTRL1_ARPEN_Pos                                         (4U)
#define I2C_CTRL1_ARPEN_Msk                                         (0x1UL << I2C_CTRL1_ARPEN_Pos)                                 /*!< 0x00000010 */
#define I2C_CTRL1_ARPEN                                             I2C_CTRL1_ARPEN_Msk                                            /*!<ARP Enable                                    */
#define I2C_CTRL1_PECEN_Pos                                         (5U)
#define I2C_CTRL1_PECEN_Msk                                         (0x1UL << I2C_CTRL1_PECEN_Pos)                                 /*!< 0x00000020 */
#define I2C_CTRL1_PECEN                                             I2C_CTRL1_PECEN_Msk                                            /*!<PEC Enable                                    */
#define I2C_CTRL1_SRBEN_Pos                                         (6U)
#define I2C_CTRL1_SRBEN_Msk                                         (0x1UL << I2C_CTRL1_SRBEN_Pos)                                 /*!< 0x00000040 */
#define I2C_CTRL1_SRBEN                                             I2C_CTRL1_SRBEN_Msk                                            /*!<General Call Enable                           */
#define I2C_CTRL1_CLKSTRETCHD_Pos                                   (7U)
#define I2C_CTRL1_CLKSTRETCHD_Msk                                   (0x1UL << I2C_CTRL1_CLKSTRETCHD_Pos)                           /*!< 0x00000080 */
#define I2C_CTRL1_CLKSTRETCHD                                       I2C_CTRL1_CLKSTRETCHD_Msk                                      /*!<Clock Stretching Disable (Slave mode)         */
#define I2C_CTRL1_START_Pos                                         (8U)
#define I2C_CTRL1_START_Msk                                         (0x1UL << I2C_CTRL1_START_Pos)                                 /*!< 0x00000100 */
#define I2C_CTRL1_START                                             I2C_CTRL1_START_Msk                                            /*!<Start Generation                              */
#define I2C_CTRL1_STOP_Pos                                          (9U)
#define I2C_CTRL1_STOP_Msk                                          (0x1UL << I2C_CTRL1_STOP_Pos)                                  /*!< 0x00000200 */
#define I2C_CTRL1_STOP                                              I2C_CTRL1_STOP_Msk                                             /*!<Stop Generation                               */
#define I2C_CTRL1_ACKEN_Pos                                         (10U)
#define I2C_CTRL1_ACKEN_Msk                                         (0x1UL << I2C_CTRL1_ACKEN_Pos)                                 /*!< 0x00000400 */
#define I2C_CTRL1_ACKEN                                             I2C_CTRL1_ACKEN_Msk                                            /*!<Acknowledge Enable                            */
#define I2C_CTRL1_ACKPOS_Pos                                        (11U)
#define I2C_CTRL1_ACKPOS_Msk                                        (0x1UL << I2C_CTRL1_ACKPOS_Pos)                                /*!< 0x00000800 */
#define I2C_CTRL1_ACKPOS                                            I2C_CTRL1_ACKPOS_Msk                                           /*!<Acknowledge/PEC Position (for data reception) */
#define I2C_CTRL1_PEC_Pos                                           (12U)
#define I2C_CTRL1_PEC_Msk                                           (0x1UL << I2C_CTRL1_PEC_Pos)                                   /*!< 0x00001000 */
#define I2C_CTRL1_PEC                                               I2C_CTRL1_PEC_Msk                                              /*!<Packet Error Checking                         */
#define I2C_CTRL1_ALERTEN_Pos                                       (13U)
#define I2C_CTRL1_ALERTEN_Msk                                       (0x1UL << I2C_CTRL1_ALERTEN_Pos)                               /*!< 0x00002000 */
#define I2C_CTRL1_ALERTEN                                           I2C_CTRL1_ALERTEN_Msk                                          /*!<SMBus Alert                                   */
#define I2C_CTRL1_SWRST_Pos                                         (15U)
#define I2C_CTRL1_SWRST_Msk                                         (0x1UL << I2C_CTRL1_SWRST_Pos)                                 /*!< 0x00008000 */
#define I2C_CTRL1_SWRST                                             I2C_CTRL1_SWRST_Msk                                            /*!<Software Reset                                */

/*******************  Bit definition for I2C_CTRL2 register  ********************/
#define I2C_CTRL2_CLKFCFG_Pos                                       (0U)
#define I2C_CTRL2_CLKFCFG_Msk                                       (0x3FUL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x0000003F */
#define I2C_CTRL2_CLKFCFG                                           I2C_CTRL2_CLKFCFG_Msk                                          /*!<FREQ[5:0] bits (Peripheral Clock Frequency)   */
#define I2C_CTRL2_CLKFCFG_0                                         (0x01UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000001 */
#define I2C_CTRL2_CLKFCFG_1                                         (0x02UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000002 */
#define I2C_CTRL2_CLKFCFG_2                                         (0x04UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000004 */
#define I2C_CTRL2_CLKFCFG_3                                         (0x08UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000008 */
#define I2C_CTRL2_CLKFCFG_4                                         (0x10UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000010 */
#define I2C_CTRL2_CLKFCFG_5                                         (0x20UL << I2C_CTRL2_CLKFCFG_Pos)                              /*!< 0x00000020 */

#define I2C_CTRL2_ERRIEN_Pos                                        (8U)
#define I2C_CTRL2_ERRIEN_Msk                                        (0x1UL << I2C_CTRL2_ERRIEN_Pos)                                /*!< 0x00000100 */
#define I2C_CTRL2_ERRIEN                                            I2C_CTRL2_ERRIEN_Msk                                           /*!<Error Interrupt Enable  */
#define I2C_CTRL2_EVIEN_Pos                                         (9U)
#define I2C_CTRL2_EVIEN_Msk                                         (0x1UL << I2C_CTRL2_EVIEN_Pos)                                 /*!< 0x00000200 */
#define I2C_CTRL2_EVIEN                                             I2C_CTRL2_EVIEN_Msk                                            /*!<Event Interrupt Enable  */
#define I2C_CTRL2_BUFIEN_Pos                                        (10U)
#define I2C_CTRL2_BUFIEN_Msk                                        (0x1UL << I2C_CTRL2_BUFIEN_Pos)                                /*!< 0x00000400 */
#define I2C_CTRL2_BUFIEN                                            I2C_CTRL2_BUFIEN_Msk                                           /*!<Buffer Interrupt Enable */
#define I2C_CTRL2_DMAEN_Pos                                         (11U)
#define I2C_CTRL2_DMAEN_Msk                                         (0x1UL << I2C_CTRL2_DMAEN_Pos)                                 /*!< 0x00000800 */
#define I2C_CTRL2_DMAEN                                             I2C_CTRL2_DMAEN_Msk                                            /*!<DMA Requests Enable     */
#define I2C_CTRL2_LTCFG_Pos                                         (12U)
#define I2C_CTRL2_LTCFG_Msk                                         (0x1UL << I2C_CTRL2_LTCFG_Pos)                                 /*!< 0x00001000 */
#define I2C_CTRL2_LTCFG                                             I2C_CTRL2_LTCFG_Msk                                            /*!<DMA Last Transfer       */

/*******************  Bit definition for I2C_SADDR1 register  *******************/
#define I2C_SADDR1_ADDR1_7                                          0x000000FEU                                                    /*!<Interface Address */
#define I2C_SADDR1_ADDR8_9                                          0x00000300U                                                    /*!<Interface Address */

#define I2C_SADDR1_ADDR0_Pos                                        (0U)
#define I2C_SADDR1_ADDR0_Msk                                        (0x1UL << I2C_SADDR1_ADDR0_Pos)                                /*!< 0x00000001 */
#define I2C_SADDR1_ADDR0                                            I2C_SADDR1_ADDR0_Msk                                           /*!<Bit 0 */
#define I2C_SADDR1_ADDR1_Pos                                        (1U)
#define I2C_SADDR1_ADDR1_Msk                                        (0x1UL << I2C_SADDR1_ADDR1_Pos)                                /*!< 0x00000002 */
#define I2C_SADDR1_ADDR1                                            I2C_SADDR1_ADDR1_Msk                                           /*!<Bit 1 */
#define I2C_SADDR1_ADDR2_Pos                                        (2U)
#define I2C_SADDR1_ADDR2_Msk                                        (0x1UL << I2C_SADDR1_ADDR2_Pos)                                /*!< 0x00000004 */
#define I2C_SADDR1_ADDR2                                            I2C_SADDR1_ADDR2_Msk                                           /*!<Bit 2 */
#define I2C_SADDR1_ADDR3_Pos                                        (3U)
#define I2C_SADDR1_ADDR3_Msk                                        (0x1UL << I2C_SADDR1_ADDR3_Pos)                                /*!< 0x00000008 */
#define I2C_SADDR1_ADDR3                                            I2C_SADDR1_ADDR3_Msk                                           /*!<Bit 3 */
#define I2C_SADDR1_ADDR4_Pos                                        (4U)
#define I2C_SADDR1_ADDR4_Msk                                        (0x1UL << I2C_SADDR1_ADDR4_Pos)                                /*!< 0x00000010 */
#define I2C_SADDR1_ADDR4                                            I2C_SADDR1_ADDR4_Msk                                           /*!<Bit 4 */
#define I2C_SADDR1_ADDR5_Pos                                        (5U)
#define I2C_SADDR1_ADDR5_Msk                                        (0x1UL << I2C_SADDR1_ADDR5_Pos)                                /*!< 0x00000020 */
#define I2C_SADDR1_ADDR5                                            I2C_SADDR1_ADDR5_Msk                                           /*!<Bit 5 */
#define I2C_SADDR1_ADDR6_Pos                                        (6U)
#define I2C_SADDR1_ADDR6_Msk                                        (0x1UL << I2C_SADDR1_ADDR6_Pos)                                /*!< 0x00000040 */
#define I2C_SADDR1_ADDR6                                            I2C_SADDR1_ADDR6_Msk                                           /*!<Bit 6 */
#define I2C_SADDR1_ADDR7_Pos                                        (7U)
#define I2C_SADDR1_ADDR7_Msk                                        (0x1UL << I2C_SADDR1_ADDR7_Pos)                                /*!< 0x00000080 */
#define I2C_SADDR1_ADDR7                                            I2C_SADDR1_ADDR7_Msk                                           /*!<Bit 7 */
#define I2C_SADDR1_ADDR8_Pos                                        (8U)
#define I2C_SADDR1_ADDR8_Msk                                        (0x1UL << I2C_SADDR1_ADDR8_Pos)                                /*!< 0x00000100 */
#define I2C_SADDR1_ADDR8                                            I2C_SADDR1_ADDR8_Msk                                           /*!<Bit 8 */
#define I2C_SADDR1_ADDR9_Pos                                        (9U)
#define I2C_SADDR1_ADDR9_Msk                                        (0x1UL << I2C_SADDR1_ADDR9_Pos)                                /*!< 0x00000200 */
#define I2C_SADDR1_ADDR9                                            I2C_SADDR1_ADDR9_Msk                                           /*!<Bit 9 */

#define I2C_SADDR1_ADDRLEN_Pos                                      (15U)
#define I2C_SADDR1_ADDRLEN_Msk                                      (0x1UL << I2C_SADDR1_ADDRLEN_Pos)                              /*!< 0x00008000 */
#define I2C_SADDR1_ADDRLEN                                          I2C_SADDR1_ADDRLEN_Msk                                         /*!<Addressing Mode (Slave mode) */

/*******************  Bit definition for I2C_SADDR2 register  *******************/
#define I2C_SADDR2_ADDRNUM_Pos                                      (0U)
#define I2C_SADDR2_ADDRNUM_Msk                                      (0x1UL << I2C_SADDR2_ADDRNUM_Pos)                              /*!< 0x00000001 */
#define I2C_SADDR2_ADDRNUM                                          I2C_SADDR2_ADDRNUM_Msk                                         /*!<Dual addressing mode enable */
#define I2C_SADDR2_ADDR2_Pos                                        (1U)
#define I2C_SADDR2_ADDR2_Msk                                        (0x7FUL << I2C_SADDR2_ADDR2_Pos)                               /*!< 0x000000FE */
#define I2C_SADDR2_ADDR2                                            I2C_SADDR2_ADDR2_Msk                                           /*!<Interface address           */

/********************  Bit definition for I2C_DATA register  ********************/
#define I2C_DATA_DATA_Pos                                           (0U)
#define I2C_DATA_DATA_Msk                                           (0xFFUL << I2C_DATA_DATA_Pos)                                  /*!< 0x000000FF */
#define I2C_DATA_DATA                                               I2C_DATA_DATA_Msk                                              /*!<8-bit Data Register         */

/*******************  Bit definition for I2C_STS1 register  ********************/
#define I2C_STS1_STARTFLG_Pos                                       (0U)
#define I2C_STS1_STARTFLG_Msk                                       (0x1UL << I2C_STS1_STARTFLG_Pos)                               /*!< 0x00000001 */
#define I2C_STS1_STARTFLG                                           I2C_STS1_STARTFLG_Msk                                          /*!<Start Bit (Master mode)                         */
#define I2C_STS1_ADDRFLG_Pos                                        (1U)
#define I2C_STS1_ADDRFLG_Msk                                        (0x1UL << I2C_STS1_ADDRFLG_Pos)                                /*!< 0x00000002 */
#define I2C_STS1_ADDRFLG                                            I2C_STS1_ADDRFLG_Msk                                           /*!<Address sent (master mode)/matched (slave mode) */
#define I2C_STS1_BTCFLG_Pos                                         (2U)
#define I2C_STS1_BTCFLG_Msk                                         (0x1UL << I2C_STS1_BTCFLG_Pos)                                 /*!< 0x00000004 */
#define I2C_STS1_BTCFLG                                             I2C_STS1_BTCFLG_Msk                                            /*!<Byte Transfer Finished                          */
#define I2C_STS1_ADDR10FLG_Pos                                      (3U)
#define I2C_STS1_ADDR10FLG_Msk                                      (0x1UL << I2C_STS1_ADDR10FLG_Pos)                              /*!< 0x00000008 */
#define I2C_STS1_ADDR10FLG                                          I2C_STS1_ADDR10FLG_Msk                                         /*!<10-bit header sent (Master mode)                */
#define I2C_STS1_STOPFLG_Pos                                        (4U)
#define I2C_STS1_STOPFLG_Msk                                        (0x1UL << I2C_STS1_STOPFLG_Pos)                                /*!< 0x00000010 */
#define I2C_STS1_STOPFLG                                            I2C_STS1_STOPFLG_Msk                                           /*!<Stop detection (Slave mode)                     */
#define I2C_STS1_RXBNEFLG_Pos                                       (6U)
#define I2C_STS1_RXBNEFLG_Msk                                       (0x1UL << I2C_STS1_RXBNEFLG_Pos)                               /*!< 0x00000040 */
#define I2C_STS1_RXBNEFLG                                           I2C_STS1_RXBNEFLG_Msk                                          /*!<Data Register not Empty (receivers)             */
#define I2C_STS1_TXBEFLG_Pos                                        (7U)
#define I2C_STS1_TXBEFLG_Msk                                        (0x1UL << I2C_STS1_TXBEFLG_Pos)                                /*!< 0x00000080 */
#define I2C_STS1_TXBEFLG                                            I2C_STS1_TXBEFLG_Msk                                           /*!<Data Register Empty (transmitters)              */
#define I2C_STS1_BERRFLG_Pos                                        (8U)
#define I2C_STS1_BERRFLG_Msk                                        (0x1UL << I2C_STS1_BERRFLG_Pos)                                /*!< 0x00000100 */
#define I2C_STS1_BERRFLG                                            I2C_STS1_BERRFLG_Msk                                           /*!<Bus Error                                       */
#define I2C_STS1_ALFLG_Pos                                          (9U)
#define I2C_STS1_ALFLG_Msk                                          (0x1UL << I2C_STS1_ALFLG_Pos)                                  /*!< 0x00000200 */
#define I2C_STS1_ALFLG                                              I2C_STS1_ALFLG_Msk                                             /*!<Arbitration Lost (master mode)                  */
#define I2C_STS1_AEFLG_Pos                                          (10U)
#define I2C_STS1_AEFLG_Msk                                          (0x1UL << I2C_STS1_AEFLG_Pos)                                  /*!< 0x00000400 */
#define I2C_STS1_AEFLG                                              I2C_STS1_AEFLG_Msk                                             /*!<Acknowledge Failure                             */
#define I2C_STS1_OVRURFLG_Pos                                       (11U)
#define I2C_STS1_OVRURFLG_Msk                                       (0x1UL << I2C_STS1_OVRURFLG_Pos)                               /*!< 0x00000800 */
#define I2C_STS1_OVRURFLG                                           I2C_STS1_OVRURFLG_Msk                                          /*!<Overrun/Underrun                                */
#define I2C_STS1_PECEFLG_Pos                                        (12U)
#define I2C_STS1_PECEFLG_Msk                                        (0x1UL << I2C_STS1_PECEFLG_Pos)                                /*!< 0x00001000 */
#define I2C_STS1_PECEFLG                                            I2C_STS1_PECEFLG_Msk                                           /*!<PEC Error in reception                          */
#define I2C_STS1_TTEFLG_Pos                                         (14U)
#define I2C_STS1_TTEFLG_Msk                                         (0x1UL << I2C_STS1_TTEFLG_Pos)                                 /*!< 0x00004000 */
#define I2C_STS1_TTEFLG                                             I2C_STS1_TTEFLG_Msk                                            /*!<Timeout or Tlow Error                           */
#define I2C_STS1_SMBALTFLG_Pos                                      (15U)
#define I2C_STS1_SMBALTFLG_Msk                                      (0x1UL << I2C_STS1_SMBALTFLG_Pos)                              /*!< 0x00008000 */
#define I2C_STS1_SMBALTFLG                                          I2C_STS1_SMBALTFLG_Msk                                         /*!<SMBus Alert                                     */

/*******************  Bit definition for I2C_STS2 register  ********************/
#define I2C_STS2_MSFLG_Pos                                          (0U)
#define I2C_STS2_MSFLG_Msk                                          (0x1UL << I2C_STS2_MSFLG_Pos)                                  /*!< 0x00000001 */
#define I2C_STS2_MSFLG                                              I2C_STS2_MSFLG_Msk                                             /*!<Master/Slave                                    */
#define I2C_STS2_BUSBSYFLG_Pos                                      (1U)
#define I2C_STS2_BUSBSYFLG_Msk                                      (0x1UL << I2C_STS2_BUSBSYFLG_Pos)                              /*!< 0x00000002 */
#define I2C_STS2_BUSBSYFLG                                          I2C_STS2_BUSBSYFLG_Msk                                         /*!<Bus Busy                                        */
#define I2C_STS2_TRFLG_Pos                                          (2U)
#define I2C_STS2_TRFLG_Msk                                          (0x1UL << I2C_STS2_TRFLG_Pos)                                  /*!< 0x00000004 */
#define I2C_STS2_TRFLG                                              I2C_STS2_TRFLG_Msk                                             /*!<Transmitter/Receiver                            */
#define I2C_STS2_GENCALLFLG_Pos                                     (4U)
#define I2C_STS2_GENCALLFLG_Msk                                     (0x1UL << I2C_STS2_GENCALLFLG_Pos)                             /*!< 0x00000010 */
#define I2C_STS2_GENCALLFLG                                         I2C_STS2_GENCALLFLG_Msk                                        /*!<General Call Address (Slave mode)               */
#define I2C_STS2_SMBDADDRFLG_Pos                                    (5U)
#define I2C_STS2_SMBDADDRFLG_Msk                                    (0x1UL << I2C_STS2_SMBDADDRFLG_Pos)                            /*!< 0x00000020 */
#define I2C_STS2_SMBDADDRFLG                                        I2C_STS2_SMBDADDRFLG_Msk                                       /*!<SMBus Device Default Address (Slave mode)       */
#define I2C_STS2_SMMHADDR_Pos                                       (6U)
#define I2C_STS2_SMMHADDR_Msk                                       (0x1UL << I2C_STS2_SMMHADDR_Pos)                               /*!< 0x00000040 */
#define I2C_STS2_SMMHADDR                                           I2C_STS2_SMMHADDR_Msk                                          /*!<SMBus Host Header (Slave mode)                  */
#define I2C_STS2_DUALADDRFLG_Pos                                    (7U)
#define I2C_STS2_DUALADDRFLG_Msk                                    (0x1UL << I2C_STS2_DUALADDRFLG_Pos)                            /*!< 0x00000080 */
#define I2C_STS2_DUALADDRFLG                                        I2C_STS2_DUALADDRFLG_Msk                                       /*!<Dual Flag (Slave mode)                          */
#define I2C_STS2_PECVALUE_Pos                                       (8U)
#define I2C_STS2_PECVALUE_Msk                                       (0xFFUL << I2C_STS2_PECVALUE_Pos)                              /*!< 0x0000FF00 */
#define I2C_STS2_PECVALUE                                           I2C_STS2_PECVALUE_Msk                                          /*!<Packet Error Checking Register                  */

/*******************  Bit definition for I2C_CLKCTRL register  ********************/
#define I2C_CLKCTRL_CLKS_Pos                                        (0U)
#define I2C_CLKCTRL_CLKS_Msk                                        (0xFFFUL << I2C_CLKCTRL_CLKS_Pos)                              /*!< 0x00000FFF */
#define I2C_CLKCTRL_CLKS                                            I2C_CLKCTRL_CLKS_Msk                                           /*!<Clock Control Register in Fast/Standard mode (Master mode) */
#define I2C_CLKCTRL_FDUTYCFG_Pos                                    (14U)
#define I2C_CLKCTRL_FDUTYCFG_Msk                                    (0x1UL << I2C_CLKCTRL_FDUTYCFG_Pos)                            /*!< 0x00004000 */
#define I2C_CLKCTRL_FDUTYCFG                                        I2C_CLKCTRL_FDUTYCFG_Msk                                       /*!<Fast Mode Duty Cycle                                       */
#define I2C_CLKCTRL_SPEEDCFG_Pos                                    (15U)
#define I2C_CLKCTRL_SPEEDCFG_Msk                                    (0x1UL << I2C_CLKCTRL_SPEEDCFG_Pos)                            /*!< 0x00008000 */
#define I2C_CLKCTRL_SPEEDCFG                                        I2C_CLKCTRL_SPEEDCFG_Msk                                       /*!<I2C Master Mode Selection                                  */

/******************  Bit definition for I2C_RISETMAX register  *******************/
#define I2C_RISETMAX_RISETMAX_Pos                                   (0U)
#define I2C_RISETMAX_RISETMAX_Msk                                   (0x3FUL << I2C_RISETMAX_RISETMAX_Pos)                          /*!< 0x0000003F */
#define I2C_RISETMAX_RISETMAX                                       I2C_RISETMAX_RISETMAX_Msk                                      /*!<Maximum Rise Time in Fast/Standard mode (Master mode) */


/******************************************************************************/
/*                                                                            */
/*                           Independent WATCHDOG                             */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for IWDT_KEY register  ********************/
#define IWDT_KEY_KEY_Pos                                            (0U)
#define IWDT_KEY_KEY_Msk                                            (0xFFFFUL << IWDT_KEY_KEY_Pos)                                 /*!< 0x0000FFFF */
#define IWDT_KEY_KEY                                                IWDT_KEY_KEY_Msk                                               /*!<Key value (write only, read 0000h)  */

/*******************  Bit definition for IWDT_PSC register  ********************/
#define IWDT_PSC_PSC_Pos                                            (0U)
#define IWDT_PSC_PSC_Msk                                            (0x7UL << IWDT_PSC_PSC_Pos)                                    /*!< 0x00000007 */
#define IWDT_PSC_PSC                                                IWDT_PSC_PSC_Msk                                               /*!<PR[2:0] (Prescaler divider)         */
#define IWDT_PSC_PSC_0                                              (0x1UL << IWDT_PSC_PSC_Pos)                                    /*!< 0x01 */
#define IWDT_PSC_PSC_1                                              (0x2UL << IWDT_PSC_PSC_Pos)                                    /*!< 0x02 */
#define IWDT_PSC_PSC_2                                              (0x4UL << IWDT_PSC_PSC_Pos)                                    /*!< 0x04 */

/*******************  Bit definition for IWDT_CNTRLD register  *******************/
#define IWDT_CNTRLD_CNTRLD_Pos                                      (0U)
#define IWDT_CNTRLD_CNTRLD_Msk                                      (0xFFFUL << IWDT_CNTRLD_CNTRLD_Pos)                            /*!< 0x00000FFF */
#define IWDT_CNTRLD_CNTRLD                                          IWDT_CNTRLD_CNTRLD_Msk                                         /*!<Watchdog counter reload value        */

/*******************  Bit definition for IWDT_STS register  ********************/
#define IWDT_STS_PSCUFLG_Pos                                        (0U)
#define IWDT_STS_PSCUFLG_Msk                                        (0x1UL << IWDT_STS_PSCUFLG_Pos)                                /*!< 0x00000001 */
#define IWDT_STS_PSCUFLG                                            IWDT_STS_PSCUFLG_Msk                                           /*!<Watchdog prescaler value update      */
#define IWDT_STS_CNTUFLG_Pos                                        (1U)
#define IWDT_STS_CNTUFLG_Msk                                        (0x1UL << IWDT_STS_CNTUFLG_Pos)                                /*!< 0x00000002 */
#define IWDT_STS_CNTUFLG                                            IWDT_STS_CNTUFLG_Msk                                           /*!<Watchdog counter reload value update */


/******************************************************************************/
/*                                                                            */
/*                             Power Control                                  */
/*                                                                            */
/******************************************************************************/
/********************  Bit definition for PMU_CTRL register  ********************/
#define PMU_CTRL_LPDSCFG_Pos                                        (0U)
#define PMU_CTRL_LPDSCFG_Msk                                        (0x1UL << PMU_CTRL_LPDSCFG_Pos)                                /*!< 0x00000001 */
#define PMU_CTRL_LPDSCFG                                            PMU_CTRL_LPDSCFG_Msk                                           /*!< Low-Power Deepsleep                 */
#define PMU_CTRL_PDDSCFG_Pos                                        (1U)
#define PMU_CTRL_PDDSCFG_Msk                                        (0x1UL << PMU_CTRL_PDDSCFG_Pos)                                /*!< 0x00000002 */
#define PMU_CTRL_PDDSCFG                                            PMU_CTRL_PDDSCFG_Msk                                           /*!< Power Down Deepsleep                */
#define PMU_CTRL_WUFLGCLR_Pos                                       (2U)
#define PMU_CTRL_WUFLGCLR_Msk                                       (0x1UL << PMU_CTRL_WUFLGCLR_Pos)                               /*!< 0x00000004 */
#define PMU_CTRL_WUFLGCLR                                           PMU_CTRL_WUFLGCLR_Msk                                          /*!< Clear Wakeup Flag                   */
#define PMU_CTRL_SBFLGCLR_Pos                                       (3U)
#define PMU_CTRL_SBFLGCLR_Msk                                       (0x1UL << PMU_CTRL_SBFLGCLR_Pos)                               /*!< 0x00000008 */
#define PMU_CTRL_SBFLGCLR                                           PMU_CTRL_SBFLGCLR_Msk                                          /*!< Clear Standby Flag                  */
#define PMU_CTRL_PVDEN_Pos                                          (4U)
#define PMU_CTRL_PVDEN_Msk                                          (0x1UL << PMU_CTRL_PVDEN_Pos)                                  /*!< 0x00000010 */
#define PMU_CTRL_PVDEN                                              PMU_CTRL_PVDEN_Msk                                             /*!< Power Voltage Detector Enable       */

#define PMU_CTRL_PLSEL_Pos                                          (5U)
#define PMU_CTRL_PLSEL_Msk                                          (0x7UL << PMU_CTRL_PLSEL_Pos)                                  /*!< 0x000000E0 */
#define PMU_CTRL_PLSEL                                              PMU_CTRL_PLSEL_Msk                                             /*!< PLS[2:0] bits (PVD Level Selection) */
#define PMU_CTRL_PLSEL_0                                            (0x1UL << PMU_CTRL_PLSEL_Pos)                                  /*!< 0x00000020 */
#define PMU_CTRL_PLSEL_1                                            (0x2UL << PMU_CTRL_PLSEL_Pos)                                  /*!< 0x00000040 */
#define PMU_CTRL_PLSEL_2                                            (0x4UL << PMU_CTRL_PLSEL_Pos)                                  /*!< 0x00000080 */

/*!< PVD level configuration */
#define PMU_CTRL_PLSEL_LEV0                                         0x00000000U                                                    /*!< PVD level 0 */
#define PMU_CTRL_PLSEL_LEV1                                         0x00000020U                                                    /*!< PVD level 1 */
#define PMU_CTRL_PLSEL_LEV2                                         0x00000040U                                                    /*!< PVD level 2 */
#define PMU_CTRL_PLSEL_LEV3                                         0x00000060U                                                    /*!< PVD level 3 */
#define PMU_CTRL_PLSEL_LEV4                                         0x00000080U                                                    /*!< PVD level 4 */
#define PMU_CTRL_PLSEL_LEV5                                         0x000000A0U                                                    /*!< PVD level 5 */
#define PMU_CTRL_PLSEL_LEV6                                         0x000000C0U                                                    /*!< PVD level 6 */
#define PMU_CTRL_PLSEL_LEV7                                         0x000000E0U                                                    /*!< PVD level 7 */
#define PMU_CTRL_BPWEN_Pos                                          (8U)
#define PMU_CTRL_BPWEN_Msk                                          (0x1UL << PMU_CTRL_BPWEN_Pos)                                  /*!< 0x00000100 */
#define PMU_CTRL_BPWEN                                              PMU_CTRL_BPWEN_Msk                                             /*!< Disable Backup Domain write protection                     */

/*******************  Bit definition for PMU_CSTS register  ********************/
#define PMU_CSTS_WUEFLG_Pos                                         (0U)
#define PMU_CSTS_WUEFLG_Msk                                         (0x1UL << PMU_CSTS_WUEFLG_Pos)                                 /*!< 0x00000001 */
#define PMU_CSTS_WUEFLG                                             PMU_CSTS_WUEFLG_Msk                                            /*!< Wakeup Flag                                      */
#define PMU_CSTS_SBFLG_Pos                                          (1U)
#define PMU_CSTS_SBFLG_Msk                                          (0x1UL << PMU_CSTS_SBFLG_Pos)                                  /*!< 0x00000002 */
#define PMU_CSTS_SBFLG                                              PMU_CSTS_SBFLG_Msk                                             /*!< Standby Flag                                     */
#define PMU_CSTS_PVDOFLG_Pos                                        (2U)
#define PMU_CSTS_PVDOFLG_Msk                                        (0x1UL << PMU_CSTS_PVDOFLG_Pos)                                /*!< 0x00000004 */
#define PMU_CSTS_PVDOFLG                                            PMU_CSTS_PVDOFLG_Msk                                           /*!< PVD Output                                       */
#define PMU_CSTS_WKUPCFG_Pos                                        (8U)
#define PMU_CSTS_WKUPCFG_Msk                                        (0x1UL << PMU_CSTS_WKUPCFG_Pos)                                /*!< 0x00000100 */
#define PMU_CSTS_WKUPCFG                                            PMU_CSTS_WKUPCFG_Msk                                           /*!< Enable WKUP pin                                  */

/******************************************************************************/
/*                                                                            */
/*                              Backup Registers                              */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for BAKPR_DATA1 register  *******************/
#define BAKPR_DATA1_DATA_Pos                                        (0U)
#define BAKPR_DATA1_DATA_Msk                                        (0xFFFFUL << BAKPR_DATA1_DATA_Pos)                             /*!< 0x0000FFFF */
#define BAKPR_DATA1_DATA                                            BAKPR_DATA1_DATA_Msk                                           /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA2 register  *******************/
#define BAKPR_DATA2_DATA_Pos                                        (0U)
#define BAKPR_DATA2_DATA_Msk                                        (0xFFFFUL << BAKPR_DATA2_DATA_Pos)                             /*!< 0x0000FFFF */
#define BAKPR_DATA2_DATA                                            BAKPR_DATA2_DATA_Msk                                           /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA3 register  *******************/
#define BAKPR_DATA3_DATA_Pos                                        (0U)
#define BAKPR_DATA3_DATA_Msk                                        (0xFFFFUL << BAKPR_DATA3_DATA_Pos)                             /*!< 0x0000FFFF */
#define BAKPR_DATA3_DATA                                            BAKPR_DATA3_DATA_Msk                                           /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA4 register  *******************/
#define BAKPR_DATA4_DATA_Pos                                        (0U)
#define BAKPR_DATA4_DATA_Msk                                        (0xFFFFUL << BAKPR_DATA4_DATA_Pos)                             /*!< 0x0000FFFF */
#define BAKPR_DATA4_DATA                                            BAKPR_DATA4_DATA_Msk                                           /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA5 register  *******************/
#define BAKPR_DATA5_DATA_Pos                                        (0U)
#define BAKPR_DATA5_DATA_Msk                                        (0xFFFFUL << BAKPR_DATA5_DATA_Pos)                             /*!< 0x0000FFFF */
#define BAKPR_DATA5_DATA                                            BAKPR_DATA5_DATA_Msk                                           /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA6 register  *******************/
#define BAKPR_DATA6_DATA_Pos                                        (0U)
#define BAKPR_DATA6_DATA_Msk                                        (0xFFFFUL << BAKPR_DATA6_DATA_Pos)                             /*!< 0x0000FFFF */
#define BAKPR_DATA6_DATA                                            BAKPR_DATA6_DATA_Msk                                           /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA7 register  *******************/
#define BAKPR_DATA7_DATA_Pos                                        (0U)
#define BAKPR_DATA7_DATA_Msk                                        (0xFFFFUL << BAKPR_DATA7_DATA_Pos)                             /*!< 0x0000FFFF */
#define BAKPR_DATA7_DATA                                            BAKPR_DATA7_DATA_Msk                                           /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA8 register  *******************/
#define BAKPR_DATA8_DATA_Pos                                        (0U)
#define BAKPR_DATA8_DATA_Msk                                        (0xFFFFUL << BAKPR_DATA8_DATA_Pos)                             /*!< 0x0000FFFF */
#define BAKPR_DATA8_DATA                                            BAKPR_DATA8_DATA_Msk                                           /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA9 register  *******************/
#define BAKPR_DATA9_DATA_Pos                                        (0U)
#define BAKPR_DATA9_DATA_Msk                                        (0xFFFFUL << BAKPR_DATA9_DATA_Pos)                             /*!< 0x0000FFFF */
#define BAKPR_DATA9_DATA                                            BAKPR_DATA9_DATA_Msk                                           /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA10 register  *******************/
#define BAKPR_DATA10_DATA_Pos                                       (0U)
#define BAKPR_DATA10_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA10_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA10_DATA                                           BAKPR_DATA10_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA11 register  *******************/
#define BAKPR_DATA11_DATA_Pos                                       (0U)
#define BAKPR_DATA11_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA11_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA11_DATA                                           BAKPR_DATA11_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA12 register  *******************/
#define BAKPR_DATA12_DATA_Pos                                       (0U)
#define BAKPR_DATA12_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA12_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA12_DATA                                           BAKPR_DATA12_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA13 register  *******************/
#define BAKPR_DATA13_DATA_Pos                                       (0U)
#define BAKPR_DATA13_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA13_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA13_DATA                                           BAKPR_DATA13_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA14 register  *******************/
#define BAKPR_DATA14_DATA_Pos                                       (0U)
#define BAKPR_DATA14_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA14_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA14_DATA                                           BAKPR_DATA14_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA15 register  *******************/
#define BAKPR_DATA15_DATA_Pos                                       (0U)
#define BAKPR_DATA15_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA15_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA15_DATA                                           BAKPR_DATA15_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA16 register  *******************/
#define BAKPR_DATA16_DATA_Pos                                       (0U)
#define BAKPR_DATA16_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA16_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA16_DATA                                           BAKPR_DATA16_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA17 register  *******************/
#define BAKPR_DATA17_DATA_Pos                                       (0U)
#define BAKPR_DATA17_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA17_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA17_DATA                                           BAKPR_DATA17_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA18 register  *******************/
#define BAKPR_DATA18_DATA_Pos                                       (0U)
#define BAKPR_DATA18_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA18_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA18_DATA                                           BAKPR_DATA18_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA19 register  *******************/
#define BAKPR_DATA19_DATA_Pos                                       (0U)
#define BAKPR_DATA19_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA19_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA19_DATA                                           BAKPR_DATA19_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA20 register  *******************/
#define BAKPR_DATA20_DATA_Pos                                       (0U)
#define BAKPR_DATA20_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA20_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA20_DATA                                           BAKPR_DATA20_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA21 register  *******************/
#define BAKPR_DATA21_DATA_Pos                                       (0U)
#define BAKPR_DATA21_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA21_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA21_DATA                                           BAKPR_DATA21_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA22 register  *******************/
#define BAKPR_DATA22_DATA_Pos                                       (0U)
#define BAKPR_DATA22_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA22_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA22_DATA                                           BAKPR_DATA22_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA23 register  *******************/
#define BAKPR_DATA23_DATA_Pos                                       (0U)
#define BAKPR_DATA23_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA23_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA23_DATA                                           BAKPR_DATA23_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA24 register  *******************/
#define BAKPR_DATA24_DATA_Pos                                       (0U)
#define BAKPR_DATA24_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA24_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA24_DATA                                           BAKPR_DATA24_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA25 register  *******************/
#define BAKPR_DATA25_DATA_Pos                                       (0U)
#define BAKPR_DATA25_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA25_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA25_DATA                                           BAKPR_DATA25_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA26 register  *******************/
#define BAKPR_DATA26_DATA_Pos                                       (0U)
#define BAKPR_DATA26_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA26_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA26_DATA                                           BAKPR_DATA26_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA27 register  *******************/
#define BAKPR_DATA27_DATA_Pos                                       (0U)
#define BAKPR_DATA27_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA27_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA27_DATA                                           BAKPR_DATA27_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA28 register  *******************/
#define BAKPR_DATA28_DATA_Pos                                       (0U)
#define BAKPR_DATA28_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA28_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA28_DATA                                           BAKPR_DATA28_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA29 register  *******************/
#define BAKPR_DATA29_DATA_Pos                                       (0U)
#define BAKPR_DATA29_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA29_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA29_DATA                                           BAKPR_DATA29_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA30 register  *******************/
#define BAKPR_DATA30_DATA_Pos                                       (0U)
#define BAKPR_DATA30_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA30_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA30_DATA                                           BAKPR_DATA30_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA31 register  *******************/
#define BAKPR_DATA31_DATA_Pos                                       (0U)
#define BAKPR_DATA31_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA31_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA31_DATA                                           BAKPR_DATA31_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA32 register  *******************/
#define BAKPR_DATA32_DATA_Pos                                       (0U)
#define BAKPR_DATA32_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA32_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA32_DATA                                           BAKPR_DATA32_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA33 register  *******************/
#define BAKPR_DATA33_DATA_Pos                                       (0U)
#define BAKPR_DATA33_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA33_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA33_DATA                                           BAKPR_DATA33_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA34 register  *******************/
#define BAKPR_DATA34_DATA_Pos                                       (0U)
#define BAKPR_DATA34_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA34_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA34_DATA                                           BAKPR_DATA34_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA35 register  *******************/
#define BAKPR_DATA35_DATA_Pos                                       (0U)
#define BAKPR_DATA35_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA35_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA35_DATA                                           BAKPR_DATA35_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA36 register  *******************/
#define BAKPR_DATA36_DATA_Pos                                       (0U)
#define BAKPR_DATA36_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA36_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA36_DATA                                           BAKPR_DATA36_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA37 register  *******************/
#define BAKPR_DATA37_DATA_Pos                                       (0U)
#define BAKPR_DATA37_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA37_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA37_DATA                                           BAKPR_DATA37_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA38 register  *******************/
#define BAKPR_DATA38_DATA_Pos                                       (0U)
#define BAKPR_DATA38_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA38_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA38_DATA                                           BAKPR_DATA38_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA39 register  *******************/
#define BAKPR_DATA39_DATA_Pos                                       (0U)
#define BAKPR_DATA39_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA39_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA39_DATA                                           BAKPR_DATA39_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA40 register  *******************/
#define BAKPR_DATA40_DATA_Pos                                       (0U)
#define BAKPR_DATA40_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA40_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA40_DATA                                           BAKPR_DATA40_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA41 register  *******************/
#define BAKPR_DATA41_DATA_Pos                                       (0U)
#define BAKPR_DATA41_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA41_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA41_DATA                                           BAKPR_DATA41_DATA_Msk                                          /*!< Backup data */

/*******************  Bit definition for BAKPR_DATA42 register  *******************/
#define BAKPR_DATA42_DATA_Pos                                       (0U)
#define BAKPR_DATA42_DATA_Msk                                       (0xFFFFUL << BAKPR_DATA42_DATA_Pos)                            /*!< 0x0000FFFF */
#define BAKPR_DATA42_DATA                                           BAKPR_DATA42_DATA_Msk                                          /*!< Backup data */

#define RTC_BKP_NUMBER   42

/*******************  Bit definition for BAKPR_CLKCAL register  *******************/
#define BAKPR_CLKCAL_CALVALUE_Pos                                   (0U)
#define BAKPR_CLKCAL_CALVALUE_Msk                                   (0x7FUL << BAKPR_CLKCAL_CALVALUE_Pos)                          /*!< 0x0000007F */
#define BAKPR_CLKCAL_CALVALUE                                       BAKPR_CLKCAL_CALVALUE_Msk                                      /*!< Calibration value */
#define BAKPR_CLKCAL_CALCOEN_Pos                                    (7U)
#define BAKPR_CLKCAL_CALCOEN_Msk                                    (0x1UL << BAKPR_CLKCAL_CALCOEN_Pos)                            /*!< 0x00000080 */
#define BAKPR_CLKCAL_CALCOEN                                        BAKPR_CLKCAL_CALCOEN_Msk                                       /*!< Calibration clock output enable */
#define BAKPR_CLKCAL_ASPOEN_Pos                                     (8U)
#define BAKPR_CLKCAL_ASPOEN_Msk                                     (0x1UL << BAKPR_CLKCAL_ASPOEN_Pos)                             /*!< 0x00000100 */
#define BAKPR_CLKCAL_ASPOEN                                         BAKPR_CLKCAL_ASPOEN_Msk                                        /*!< Alarm or Second pulse output enable */
#define BAKPR_CLKCAL_ASPOSEL_Pos                                    (9U)
#define BAKPR_CLKCAL_ASPOSEL_Msk                                    (0x1UL << BAKPR_CLKCAL_ASPOSEL_Pos)                            /*!< 0x00000200 */
#define BAKPR_CLKCAL_ASPOSEL                                        BAKPR_CLKCAL_ASPOSEL_Msk                                       /*!< Alarm or Second pulse output selection */

/*******************  Bit definition for BAKPR_CTRL register  *******************/
#define BAKPR_CTRL_TPFCFG_Pos                                       (0U)
#define BAKPR_CTRL_TPFCFG_Msk                                       (0x1UL << BAKPR_CTRL_TPFCFG_Pos)                               /*!< 0x00000001 */
#define BAKPR_CTRL_TPFCFG                                           BAKPR_CTRL_TPFCFG_Msk                                          /*!< Tamper Pin Function Configuration */
#define BAKPR_CTRL_TPALCFG_Pos                                      (1U)
#define BAKPR_CTRL_TPALCFG_Msk                                      (0x1UL << BAKPR_CTRL_TPALCFG_Pos)                              /*!< 0x00000002 */
#define BAKPR_CTRL_TPALCFG                                          BAKPR_CTRL_TPALCFG_Msk                                         /*!< Tamper Pin Active Level Configuration */

/*******************  Bit definition for BAKPR_CSTS register  *******************/
#define BAKPR_CSTS_TECLR_Pos                                        (0U)
#define BAKPR_CSTS_TECLR_Msk                                        (0x1UL << BAKPR_CSTS_TECLR_Pos)                                /*!< 0x00000001 */
#define BAKPR_CSTS_TECLR                                            BAKPR_CSTS_TECLR_Msk                                           /*!< Tamper Event Clear */
#define BAKPR_CSTS_TICLR_Pos                                        (1U)
#define BAKPR_CSTS_TICLR_Msk                                        (0x1UL << BAKPR_CSTS_TICLR_Pos)                                /*!< 0x00000002 */
#define BAKPR_CSTS_TICLR                                            BAKPR_CSTS_TICLR_Msk                                           /*!< Tamper Interrupt Clear */
#define BAKPR_CSTS_TPIEN_Pos                                        (2U)
#define BAKPR_CSTS_TPIEN_Msk                                        (0x1UL << BAKPR_CSTS_TPIEN_Pos)                                /*!< 0x00000004 */
#define BAKPR_CSTS_TPIEN                                            BAKPR_CSTS_TPIEN_Msk                                           /*!< Tamper Pin Interrupt Enable */
#define BAKPR_CSTS_TEFLG_Pos                                        (8U)
#define BAKPR_CSTS_TEFLG_Msk                                        (0x1UL << BAKPR_CSTS_TEFLG_Pos)                                /*!< 0x00000100 */
#define BAKPR_CSTS_TEFLG                                            BAKPR_CSTS_TEFLG_Msk                                           /*!< Tamper Event Flag */
#define BAKPR_CSTS_TIFLG_Pos                                        (9U)
#define BAKPR_CSTS_TIFLG_Msk                                        (0x1UL << BAKPR_CSTS_TIFLG_Pos)                                /*!< 0x00000200 */
#define BAKPR_CSTS_TIFLG                                            BAKPR_CSTS_TIFLG_Msk                                           /*!< Tamper Interrupt Flag */

/******************************************************************************/
/*                                                                            */
/*                         Reset and Clock Control                            */
/*                                                                            */
/******************************************************************************/
/********************  Bit definition for RCM_CTRL register  ********************/
#define RCM_CTRL_HSIEN_Pos                                          (0U)
#define RCM_CTRL_HSIEN_Msk                                          (0x1UL << RCM_CTRL_HSIEN_Pos)                                  /*!< 0x00000001 */
#define RCM_CTRL_HSIEN                                              RCM_CTRL_HSIEN_Msk
#define RCM_CTRL_HSIRDYFLG_Pos                                      (1U)
#define RCM_CTRL_HSIRDYFLG_Msk                                      (0x1UL << RCM_CTRL_HSIRDYFLG_Pos)                              /*!< 0x00000002 */
#define RCM_CTRL_HSIRDYFLG                                          RCM_CTRL_HSIRDYFLG_Msk

#define RCM_CTRL_HSITRM_Pos                                         (3U)
#define RCM_CTRL_HSITRM_Msk                                         (0x1FUL << RCM_CTRL_HSITRM_Pos)                                /*!< 0x000000F8 */
#define RCM_CTRL_HSITRM                                             RCM_CTRL_HSITRM_Msk
#define RCM_CTRL_HSITRM_0                                           (0x01UL << RCM_CTRL_HSITRM_Pos)                                /*!< 0x00000008 */
#define RCM_CTRL_HSITRM_1                                           (0x02UL << RCM_CTRL_HSITRM_Pos)                                /*!< 0x00000010 */
#define RCM_CTRL_HSITRM_2                                           (0x04UL << RCM_CTRL_HSITRM_Pos)                                /*!< 0x00000020 */
#define RCM_CTRL_HSITRM_3                                           (0x08UL << RCM_CTRL_HSITRM_Pos)                                /*!< 0x00000040 */
#define RCM_CTRL_HSITRM_4                                           (0x10UL << RCM_CTRL_HSITRM_Pos)                                /*!< 0x00000080 */

#define RCM_CTRL_HSICAL_Pos                                         (8U)
#define RCM_CTRL_HSICAL_Msk                                         (0xFFUL << RCM_CTRL_HSICAL_Pos)                                /*!< 0x0000FF00 */
#define RCM_CTRL_HSICAL                                             RCM_CTRL_HSICAL_Msk
#define RCM_CTRL_HSICAL_0                                           (0x01UL << RCM_CTRL_HSICAL_Pos)                                /*!< 0x00000100 */
#define RCM_CTRL_HSICAL_1                                           (0x02UL << RCM_CTRL_HSICAL_Pos)                                /*!< 0x00000200 */
#define RCM_CTRL_HSICAL_2                                           (0x04UL << RCM_CTRL_HSICAL_Pos)                                /*!< 0x00000400 */
#define RCM_CTRL_HSICAL_3                                           (0x08UL << RCM_CTRL_HSICAL_Pos)                                /*!< 0x00000800 */
#define RCM_CTRL_HSICAL_4                                           (0x10UL << RCM_CTRL_HSICAL_Pos)                                /*!< 0x00001000 */
#define RCM_CTRL_HSICAL_5                                           (0x20UL << RCM_CTRL_HSICAL_Pos)                                /*!< 0x00002000 */
#define RCM_CTRL_HSICAL_6                                           (0x40UL << RCM_CTRL_HSICAL_Pos)                                /*!< 0x00004000 */
#define RCM_CTRL_HSICAL_7                                           (0x80UL << RCM_CTRL_HSICAL_Pos)                                /*!< 0x00008000 */

#define RCM_CTRL_HSEEN_Pos                                          (16U)
#define RCM_CTRL_HSEEN_Msk                                          (0x1UL << RCM_CTRL_HSEEN_Pos)                                  /*!< 0x00010000 */
#define RCM_CTRL_HSEEN                                              RCM_CTRL_HSEEN_Msk
#define RCM_CTRL_HSERDYFLG_Pos                                      (17U)
#define RCM_CTRL_HSERDYFLG_Msk                                      (0x1UL << RCM_CTRL_HSERDYFLG_Pos)                              /*!< 0x00020000 */
#define RCM_CTRL_HSERDYFLG                                          RCM_CTRL_HSERDYFLG_Msk
#define RCM_CTRL_HSEBCFG_Pos                                        (18U)
#define RCM_CTRL_HSEBCFG_Msk                                        (0x1UL << RCM_CTRL_HSEBCFG_Pos)                                /*!< 0x00040000 */
#define RCM_CTRL_HSEBCFG                                            RCM_CTRL_HSEBCFG_Msk
#define RCM_CTRL_CSSEN_Pos                                          (19U)
#define RCM_CTRL_CSSEN_Msk                                          (0x1UL << RCM_CTRL_CSSEN_Pos)                                  /*!< 0x00080000 */
#define RCM_CTRL_CSSEN                                              RCM_CTRL_CSSEN_Msk
#define RCM_CTRL_PLL1EN_Pos                                         (24U)
#define RCM_CTRL_PLL1EN_Msk                                         (0x1UL << RCM_CTRL_PLL1EN_Pos)                                 /*!< 0x01000000 */
#define RCM_CTRL_PLL1EN                                             RCM_CTRL_PLL1EN_Msk
#define RCM_CTRL_PLL1RDYFLG_Pos                                     (25U)
#define RCM_CTRL_PLL1RDYFLG_Msk                                     (0x1UL << RCM_CTRL_PLL1RDYFLG_Pos)                             /*!< 0x02000000 */
#define RCM_CTRL_PLL1RDYFLG                                         RCM_CTRL_PLL1RDYFLG_Msk

/********************  Bit definition for RCM_CFG register  ******************/
/*!< SCLKSEL configuration */
#define RCM_CFG_SCLKSEL_Pos                                         (0U)
#define RCM_CFG_SCLKSEL_Msk                                         (0x3UL << RCM_CFG_SCLKSEL_Pos)                                 /*!< 0x00000003 */
#define RCM_CFG_SCLKSEL                                             RCM_CFG_SCLKSEL_Msk                                            /*!< SCLKSEL[1:0] bits (System clock Switch) */
#define RCM_CFG_SCLKSEL_0                                           (0x1UL << RCM_CFG_SCLKSEL_Pos)                                 /*!< 0x00000001 */
#define RCM_CFG_SCLKSEL_1                                           (0x2UL << RCM_CFG_SCLKSEL_Pos)                                 /*!< 0x00000002 */

#define RCM_CFG_SCLKSEL_HSI                                         0x00000000U                                                    /*!< HSI selected as system clock */
#define RCM_CFG_SCLKSEL_HSE                                         0x00000001U                                                    /*!< HSE selected as system clock */
#define RCM_CFG_SCLKSEL_PLL                                         0x00000002U                                                    /*!< PLL selected as system clock */

/*!< SCLKSELSTS configuration */
#define RCM_CFG_SCLKSELSTS_Pos                                      (2U)
#define RCM_CFG_SCLKSELSTS_Msk                                      (0x3UL << RCM_CFG_SCLKSELSTS_Pos)                              /*!< 0x0000000C */
#define RCM_CFG_SCLKSELSTS                                          RCM_CFG_SCLKSELSTS_Msk                                         /*!< SCLKSELSTS[1:0] bits (System Clock Switch Status) */
#define RCM_CFG_SCLKSELSTS_0                                        (0x1UL << RCM_CFG_SCLKSELSTS_Pos)                              /*!< 0x00000004 */
#define RCM_CFG_SCLKSELSTS_1                                        (0x2UL << RCM_CFG_SCLKSELSTS_Pos)                              /*!< 0x00000008 */

#define RCM_CFG_SCLKSELSTS_HSI                                      0x00000000U                                                    /*!< HSI oscillator used as system clock        */
#define RCM_CFG_SCLKSELSTS_HSE                                      0x00000004U                                                    /*!< HSE oscillator used as system clock        */
#define RCM_CFG_SCLKSELSTS_PLL                                      0x00000008U                                                    /*!< PLL used as system clock                   */

/*!< AHBPSC configuration */
#define RCM_CFG_AHBPSC_Pos                                          (4U)
#define RCM_CFG_AHBPSC_Msk                                          (0xFUL << RCM_CFG_AHBPSC_Pos)                                  /*!< 0x000000F0 */
#define RCM_CFG_AHBPSC                                              RCM_CFG_AHBPSC_Msk                                             /*!< AHBPSC[3:0] bits (AHB prescaler) */
#define RCM_CFG_AHBPSC_0                                            (0x1UL << RCM_CFG_AHBPSC_Pos)                                  /*!< 0x00000010 */
#define RCM_CFG_AHBPSC_1                                            (0x2UL << RCM_CFG_AHBPSC_Pos)                                  /*!< 0x00000020 */
#define RCM_CFG_AHBPSC_2                                            (0x4UL << RCM_CFG_AHBPSC_Pos)                                  /*!< 0x00000040 */
#define RCM_CFG_AHBPSC_3                                            (0x8UL << RCM_CFG_AHBPSC_Pos)                                  /*!< 0x00000080 */

#define RCM_CFG_AHBPSC_DIV1                                         0x00000000U                                                    /*!< SYSCLK not divided    */
#define RCM_CFG_AHBPSC_DIV2                                         0x00000080U                                                    /*!< SYSCLK divided by 2   */
#define RCM_CFG_AHBPSC_DIV4                                         0x00000090U                                                    /*!< SYSCLK divided by 4   */
#define RCM_CFG_AHBPSC_DIV8                                         0x000000A0U                                                    /*!< SYSCLK divided by 8   */
#define RCM_CFG_AHBPSC_DIV16                                        0x000000B0U                                                    /*!< SYSCLK divided by 16  */
#define RCM_CFG_AHBPSC_DIV64                                        0x000000C0U                                                    /*!< SYSCLK divided by 64  */
#define RCM_CFG_AHBPSC_DIV128                                       0x000000D0U                                                    /*!< SYSCLK divided by 128 */
#define RCM_CFG_AHBPSC_DIV256                                       0x000000E0U                                                    /*!< SYSCLK divided by 256 */
#define RCM_CFG_AHBPSC_DIV512                                       0x000000F0U                                                    /*!< SYSCLK divided by 512 */

/*!< APB1PSC configuration */
#define RCM_CFG_APB1PSC_Pos                                         (8U)
#define RCM_CFG_APB1PSC_Msk                                         (0x7UL << RCM_CFG_APB1PSC_Pos)                                 /*!< 0x00000700 */
#define RCM_CFG_APB1PSC                                             RCM_CFG_APB1PSC_Msk                                            /*!< APB1PSC[2:0] bits (APB1 prescaler) */
#define RCM_CFG_APB1PSC_0                                           (0x1UL << RCM_CFG_APB1PSC_Pos)                                 /*!< 0x00000100 */
#define RCM_CFG_APB1PSC_1                                           (0x2UL << RCM_CFG_APB1PSC_Pos)                                 /*!< 0x00000200 */
#define RCM_CFG_APB1PSC_2                                           (0x4UL << RCM_CFG_APB1PSC_Pos)                                 /*!< 0x00000400 */

#define RCM_CFG_APB1PSC_DIV1                                        0x00000000U                                                    /*!< HCLK not divided   */
#define RCM_CFG_APB1PSC_DIV2                                        0x00000400U                                                    /*!< HCLK divided by 2  */
#define RCM_CFG_APB1PSC_DIV4                                        0x00000500U                                                    /*!< HCLK divided by 4  */
#define RCM_CFG_APB1PSC_DIV8                                        0x00000600U                                                    /*!< HCLK divided by 8  */
#define RCM_CFG_APB1PSC_DIV16                                       0x00000700U                                                    /*!< HCLK divided by 16 */

/*!< APB2PSC configuration */
#define RCM_CFG_APB2PSC_Pos                                         (11U)
#define RCM_CFG_APB2PSC_Msk                                         (0x7UL << RCM_CFG_APB2PSC_Pos)                                 /*!< 0x00003800 */
#define RCM_CFG_APB2PSC                                             RCM_CFG_APB2PSC_Msk                                            /*!< APB2PSC[2:0] bits (APB2 prescaler) */
#define RCM_CFG_APB2PSC_0                                           (0x1UL << RCM_CFG_APB2PSC_Pos)                                 /*!< 0x00000800 */
#define RCM_CFG_APB2PSC_1                                           (0x2UL << RCM_CFG_APB2PSC_Pos)                                 /*!< 0x00001000 */
#define RCM_CFG_APB2PSC_2                                           (0x4UL << RCM_CFG_APB2PSC_Pos)                                 /*!< 0x00002000 */

#define RCM_CFG_APB2PSC_DIV1                                        0x00000000U                                                    /*!< HCLK not divided   */
#define RCM_CFG_APB2PSC_DIV2                                        0x00002000U                                                    /*!< HCLK divided by 2  */
#define RCM_CFG_APB2PSC_DIV4                                        0x00002800U                                                    /*!< HCLK divided by 4  */
#define RCM_CFG_APB2PSC_DIV8                                        0x00003000U                                                    /*!< HCLK divided by 8  */
#define RCM_CFG_APB2PSC_DIV16                                       0x00003800U                                                    /*!< HCLK divided by 16 */

/*!< ADCPSC configuration */
#define RCM_CFG_ADCPSC_Pos                                          (14U)
#define RCM_CFG_ADCPSC_Msk                                          (0x3UL << RCM_CFG_ADCPSC_Pos)                                  /*!< 0x0000C000 */
#define RCM_CFG_ADCPSC                                              RCM_CFG_ADCPSC_Msk                                             /*!< ADCPSC[1:0] bits (ADC prescaler) */
#define RCM_CFG_ADCPSC_0                                            (0x1UL << RCM_CFG_ADCPSC_Pos)                                  /*!< 0x00004000 */
#define RCM_CFG_ADCPSC_1                                            (0x2UL << RCM_CFG_ADCPSC_Pos)                                  /*!< 0x00008000 */

#define RCM_CFG_ADCPSC_DIV2                                         0x00000000U                                                    /*!< PCLK2 divided by 2 */
#define RCM_CFG_ADCPSC_DIV4                                         0x00004000U                                                    /*!< PCLK2 divided by 4 */
#define RCM_CFG_ADCPSC_DIV6                                         0x00008000U                                                    /*!< PCLK2 divided by 6 */
#define RCM_CFG_ADCPSC_DIV8                                         0x0000C000U                                                    /*!< PCLK2 divided by 8 */

#define RCM_CFG_PLLSRCSEL_Pos                                       (16U)
#define RCM_CFG_PLLSRCSEL_Msk                                       (0x1UL << RCM_CFG_PLLSRCSEL_Pos)                               /*!< 0x00010000 */
#define RCM_CFG_PLLSRCSEL                                           RCM_CFG_PLLSRCSEL_Msk                                          /*!< PLLSRCSEL bit (PLL clock source selection) */

#define RCM_CFG_PLLHSEPSC_Pos                                       (17U)
#define RCM_CFG_PLLHSEPSC_Msk                                       (0x1UL << RCM_CFG_PLLHSEPSC_Pos)                               /*!< 0x00020000 */
#define RCM_CFG_PLLHSEPSC                                           RCM_CFG_PLLHSEPSC_Msk                                          /*!< PLLHSEPSC bit (PLL HSE prescaler) */

#define RCM_CFG_PLLHSEPSC_HSE                                       0x00000000U                                                    /*!< HSE clock not divided for PLL entry */
#define RCM_CFG_PLLHSEPSC_HSE_DIV2                                  0x00020000U                                                    /*!< HSE clock divided by 2 for PLL entry */

/*!< PLLMULCFG configuration */
#define RCM_CFG_PLLMULCFG_Pos                                       (18U)
#define RCM_CFG_PLLMULCFG_Msk                                       (0xFUL << RCM_CFG_PLLMULCFG_Pos)                               /*!< 0x003C0000 */
#define RCM_CFG_PLLMULCFG                                           RCM_CFG_PLLMULCFG_Msk                                          /*!< PLLMULCFG[3:0] bits (PLL multiplication factor) */
#define RCM_CFG_PLLMULCFG_0                                         (0x1UL << RCM_CFG_PLLMULCFG_Pos)                               /*!< 0x00040000 */
#define RCM_CFG_PLLMULCFG_1                                         (0x2UL << RCM_CFG_PLLMULCFG_Pos)                               /*!< 0x00080000 */
#define RCM_CFG_PLLMULCFG_2                                         (0x4UL << RCM_CFG_PLLMULCFG_Pos)                               /*!< 0x00100000 */
#define RCM_CFG_PLLMULCFG_3                                         (0x8UL << RCM_CFG_PLLMULCFG_Pos)                               /*!< 0x00200000 */

#define RCM_CFG_PLLMULCFG2                                          0x00000000U                                                    /*!< PLL input clock * 2 */
#define RCM_CFG_PLLMULCFG3_Pos                                      (18U)
#define RCM_CFG_PLLMULCFG3_Msk                                      (0x1UL << RCM_CFG_PLLMULCFG3_Pos)                              /*!< 0x00040000 */
#define RCM_CFG_PLLMULCFG3                                          RCM_CFG_PLLMULCFG3_Msk                                         /*!< PLL input clock * 3 */
#define RCM_CFG_PLLMULCFG4_Pos                                      (19U)
#define RCM_CFG_PLLMULCFG4_Msk                                      (0x1UL << RCM_CFG_PLLMULCFG4_Pos)                              /*!< 0x00080000 */
#define RCM_CFG_PLLMULCFG4                                          RCM_CFG_PLLMULCFG4_Msk                                         /*!< PLL input clock * 4 */
#define RCM_CFG_PLLMULCFG5_Pos                                      (18U)
#define RCM_CFG_PLLMULCFG5_Msk                                      (0x3UL << RCM_CFG_PLLMULCFG5_Pos)                              /*!< 0x000C0000 */
#define RCM_CFG_PLLMULCFG5                                          RCM_CFG_PLLMULCFG5_Msk                                         /*!< PLL input clock * 5 */
#define RCM_CFG_PLLMULCFG6_Pos                                      (20U)
#define RCM_CFG_PLLMULCFG6_Msk                                      (0x1UL << RCM_CFG_PLLMULCFG6_Pos)                              /*!< 0x00100000 */
#define RCM_CFG_PLLMULCFG6                                          RCM_CFG_PLLMULCFG6_Msk                                         /*!< PLL input clock * 6 */
#define RCM_CFG_PLLMULCFG7_Pos                                      (18U)
#define RCM_CFG_PLLMULCFG7_Msk                                      (0x5UL << RCM_CFG_PLLMULCFG7_Pos)                              /*!< 0x00140000 */
#define RCM_CFG_PLLMULCFG7                                          RCM_CFG_PLLMULCFG7_Msk                                         /*!< PLL input clock * 7 */
#define RCM_CFG_PLLMULCFG8_Pos                                      (19U)
#define RCM_CFG_PLLMULCFG8_Msk                                      (0x3UL << RCM_CFG_PLLMULCFG8_Pos)                              /*!< 0x00180000 */
#define RCM_CFG_PLLMULCFG8                                          RCM_CFG_PLLMULCFG8_Msk                                         /*!< PLL input clock * 8 */
#define RCM_CFG_PLLMULCFG9_Pos                                      (18U)
#define RCM_CFG_PLLMULCFG9_Msk                                      (0x7UL << RCM_CFG_PLLMULCFG9_Pos)                              /*!< 0x001C0000 */
#define RCM_CFG_PLLMULCFG9                                          RCM_CFG_PLLMULCFG9_Msk                                         /*!< PLL input clock * 9 */
#define RCM_CFG_PLLMULCFG10_Pos                                     (21U)
#define RCM_CFG_PLLMULCFG10_Msk                                     (0x1UL << RCM_CFG_PLLMULCFG10_Pos)                             /*!< 0x00200000 */
#define RCM_CFG_PLLMULCFG10                                         RCM_CFG_PLLMULCFG10_Msk                                        /*!< PLL input clock * 10 */
#define RCM_CFG_PLLMULCFG11_Pos                                     (18U)
#define RCM_CFG_PLLMULCFG11_Msk                                     (0x9UL << RCM_CFG_PLLMULCFG11_Pos)                             /*!< 0x00240000 */
#define RCM_CFG_PLLMULCFG11                                         RCM_CFG_PLLMULCFG11_Msk                                        /*!< PLL input clock * 11 */
#define RCM_CFG_PLLMULCFG12_Pos                                     (19U)
#define RCM_CFG_PLLMULCFG12_Msk                                     (0x5UL << RCM_CFG_PLLMULCFG12_Pos)                             /*!< 0x00280000 */
#define RCM_CFG_PLLMULCFG12                                         RCM_CFG_PLLMULCFG12_Msk                                        /*!< PLL input clock * 12 */
#define RCM_CFG_PLLMULCFG13_Pos                                     (18U)
#define RCM_CFG_PLLMULCFG13_Msk                                     (0xBUL << RCM_CFG_PLLMULCFG13_Pos)                             /*!< 0x002C0000 */
#define RCM_CFG_PLLMULCFG13                                         RCM_CFG_PLLMULCFG13_Msk                                        /*!< PLL input clock * 13 */
#define RCM_CFG_PLLMULCFG14_Pos                                     (20U)
#define RCM_CFG_PLLMULCFG14_Msk                                     (0x3UL << RCM_CFG_PLLMULCFG14_Pos)                             /*!< 0x00300000 */
#define RCM_CFG_PLLMULCFG14                                         RCM_CFG_PLLMULCFG14_Msk                                        /*!< PLL input clock * 14 */
#define RCM_CFG_PLLMULCFG15_Pos                                     (18U)
#define RCM_CFG_PLLMULCFG15_Msk                                     (0xDUL << RCM_CFG_PLLMULCFG15_Pos)                             /*!< 0x00340000 */
#define RCM_CFG_PLLMULCFG15                                         RCM_CFG_PLLMULCFG15_Msk                                        /*!< PLL input clock * 15 */
#define RCM_CFG_PLLMULCFG16_Pos                                     (19U)
#define RCM_CFG_PLLMULCFG16_Msk                                     (0x7UL << RCM_CFG_PLLMULCFG16_Pos)                             /*!< 0x00380000 */
#define RCM_CFG_PLLMULCFG16                                         RCM_CFG_PLLMULCFG16_Msk                                        /*!< PLL input clock * 16 */

#define RCM_CFG_OTGFSPSC_Pos                                        (22U)
#define RCM_CFG_OTGFSPSC_Msk                                        (0x3UL << RCM_CFG_OTGFSPSC_Pos)                                /*!< 0x00C00000 */
#define RCM_CFG_OTGFSPSC                                            RCM_CFG_OTGFSPSC_Msk                                           /*!< OTGFSPSC[1:0] bits (USB OTG FS prescaler) */
#define RCM_CFG_OTGFSPSC_0                                          (0x1UL << RCM_CFG_OTGFSPSC_Pos)                                /*!< 0x00400000 */
#define RCM_CFG_OTGFSPSC_1                                          (0x2UL << RCM_CFG_OTGFSPSC_Pos)                                /*!< 0x00800000 */

#define RCM_CFG_OTGFSPSC_DIV_1_5                                    0x00000000U                                                    /*!< PLL clock divided by 1.5 */

/*!< MCOSEL configuration */
#define RCM_CFG_MCOSEL_Pos                                          (24U)
#define RCM_CFG_MCOSEL_Msk                                          (0x7UL << RCM_CFG_MCOSEL_Pos)                                  /*!< 0x07000000 */
#define RCM_CFG_MCOSEL                                              RCM_CFG_MCOSEL_Msk                                             /*!< MCOSEL[2:0] bits (Microcontroller Clock Output) */
#define RCM_CFG_MCOSEL_0                                            (0x1UL << RCM_CFG_MCOSEL_Pos)                                  /*!< 0x01000000 */
#define RCM_CFG_MCOSEL_1                                            (0x2UL << RCM_CFG_MCOSEL_Pos)                                  /*!< 0x02000000 */
#define RCM_CFG_MCOSEL_2                                            (0x4UL << RCM_CFG_MCOSEL_Pos)                                  /*!< 0x04000000 */

#define RCM_CFG_MCOSEL_NOCLOCK                                      0x00000000U                                                    /*!< No clock */
#define RCM_CFG_MCOSEL_SYSCLK                                       0x04000000U                                                    /*!< System clock selected */
#define RCM_CFG_MCOSEL_HSI                                          0x05000000U                                                    /*!< HSI oscillator clock selected */
#define RCM_CFG_MCOSEL_HSE                                          0x06000000U                                                    /*!< HSE oscillator clock selected */
#define RCM_CFG_MCOSEL_PLLCLK_DIV2                                  0x07000000U                                                    /*!< PLL clock divided by 2 */

/********************  Bit definition for RCM_INT register  *******************/
#define RCM_INT_LSIRDYFLG_Pos                                       (0U)
#define RCM_INT_LSIRDYFLG_Msk                                       (0x1UL << RCM_INT_LSIRDYFLG_Pos)                               /*!< 0x00000001 */
#define RCM_INT_LSIRDYFLG                                           RCM_INT_LSIRDYFLG_Msk
#define RCM_INT_LSERDYFLG_Pos                                       (1U)
#define RCM_INT_LSERDYFLG_Msk                                       (0x1UL << RCM_INT_LSERDYFLG_Pos)                               /*!< 0x00000002 */
#define RCM_INT_LSERDYFLG                                           RCM_INT_LSERDYFLG_Msk
#define RCM_INT_HSIRDYFLG_Pos                                       (2U)
#define RCM_INT_HSIRDYFLG_Msk                                       (0x1UL << RCM_INT_HSIRDYFLG_Pos)                               /*!< 0x00000004 */
#define RCM_INT_HSIRDYFLG                                           RCM_INT_HSIRDYFLG_Msk
#define RCM_INT_HSERDYFLG_Pos                                       (3U)
#define RCM_INT_HSERDYFLG_Msk                                       (0x1UL << RCM_INT_HSERDYFLG_Pos)                               /*!< 0x00000008 */
#define RCM_INT_HSERDYFLG                                           RCM_INT_HSERDYFLG_Msk
#define RCM_INT_PLL1RDYFLG_Pos                                      (4U)
#define RCM_INT_PLL1RDYFLG_Msk                                      (0x1UL << RCM_INT_PLL1RDYFLG_Pos)                              /*!< 0x00000010 */
#define RCM_INT_PLL1RDYFLG                                          RCM_INT_PLL1RDYFLG_Msk

#define RCM_INT_CSSFLG_Pos                                          (7U)
#define RCM_INT_CSSFLG_Msk                                          (0x1UL << RCM_INT_CSSFLG_Pos)                                  /*!< 0x00000080 */
#define RCM_INT_CSSFLG                                              RCM_INT_CSSFLG_Msk
#define RCM_INT_LSIRDYEN_Pos                                        (8U)
#define RCM_INT_LSIRDYEN_Msk                                        (0x1UL << RCM_INT_LSIRDYEN_Pos)                                /*!< 0x00000100 */
#define RCM_INT_LSIRDYEN                                            RCM_INT_LSIRDYEN_Msk
#define RCM_INT_LSERDYEN_Pos                                        (9U)
#define RCM_INT_LSERDYEN_Msk                                        (0x1UL << RCM_INT_LSERDYEN_Pos)                                /*!< 0x00000200 */
#define RCM_INT_LSERDYEN                                            RCM_INT_LSERDYEN_Msk
#define RCM_INT_HSIRDYEN_Pos                                        (10U)
#define RCM_INT_HSIRDYEN_Msk                                        (0x1UL << RCM_INT_HSIRDYEN_Pos)                                /*!< 0x00000400 */
#define RCM_INT_HSIRDYEN                                            RCM_INT_HSIRDYEN_Msk
#define RCM_INT_HSERDYEN_Pos                                        (11U)
#define RCM_INT_HSERDYEN_Msk                                        (0x1UL << RCM_INT_HSERDYEN_Pos)                                /*!< 0x00000800 */
#define RCM_INT_HSERDYEN                                            RCM_INT_HSERDYEN_Msk
#define RCM_INT_PLL1RDYEN_Pos                                       (12U)
#define RCM_INT_PLL1RDYEN_Msk                                       (0x1UL << RCM_INT_PLL1RDYEN_Pos)                               /*!< 0x00001000 */
#define RCM_INT_PLL1RDYEN                                           RCM_INT_PLL1RDYEN_Msk

#define RCM_INT_LSIRDYCLR_Pos                                       (16U)
#define RCM_INT_LSIRDYCLR_Msk                                       (0x1UL << RCM_INT_LSIRDYCLR_Pos)                               /*!< 0x00010000 */
#define RCM_INT_LSIRDYCLR                                           RCM_INT_LSIRDYCLR_Msk
#define RCM_INT_LSERDYCLR_Pos                                       (17U)
#define RCM_INT_LSERDYCLR_Msk                                       (0x1UL << RCM_INT_LSERDYCLR_Pos)                               /*!< 0x00020000 */
#define RCM_INT_LSERDYCLR                                           RCM_INT_LSERDYCLR_Msk
#define RCM_INT_HSIRDYCLR_Pos                                       (18U)
#define RCM_INT_HSIRDYCLR_Msk                                       (0x1UL << RCM_INT_HSIRDYCLR_Pos)                               /*!< 0x00040000 */
#define RCM_INT_HSIRDYCLR                                           RCM_INT_HSIRDYCLR_Msk
#define RCM_INT_HSERDYCLR_Pos                                       (19U)
#define RCM_INT_HSERDYCLR_Msk                                       (0x1UL << RCM_INT_HSERDYCLR_Pos)                               /*!< 0x00080000 */
#define RCM_INT_HSERDYCLR                                           RCM_INT_HSERDYCLR_Msk
#define RCM_INT_PLL1RDYCLR_Pos                                      (20U)
#define RCM_INT_PLL1RDYCLR_Msk                                      (0x1UL << RCM_INT_PLL1RDYCLR_Pos)                              /*!< 0x00100000 */
#define RCM_INT_PLL1RDYCLR                                          RCM_INT_PLL1RDYCLR_Msk

#define RCM_INT_CSSCLR_Pos                                          (23U)
#define RCM_INT_CSSCLR_Msk                                          (0x1UL << RCM_INT_CSSCLR_Pos)                                  /*!< 0x00800000 */
#define RCM_INT_CSSCLR                                              RCM_INT_CSSCLR_Msk

/*******************  Bit definition for RCM_AHB1RST register  ******************/
#define RCM_AHB1RST_OTGFSRST_Pos                                    (9U)
#define RCM_AHB1RST_OTGFSRST_Msk                                    (0x1UL << RCM_AHB1RST_OTGFSRST_Pos)                            /*!< 0x00000200 */
#define RCM_AHB1RST_OTGFSRST                                        RCM_AHB1RST_OTGFSRST_Msk                                       /*!< USB OTG FS reset */

/*******************  Bit definition for RCM_APB2RST register  ******************/
#define RCM_APB2RST_AFIORST_Pos                                     (0U)
#define RCM_APB2RST_AFIORST_Msk                                     (0x1UL << RCM_APB2RST_AFIORST_Pos)                             /*!< 0x00000001 */
#define RCM_APB2RST_AFIORST                                         RCM_APB2RST_AFIORST_Msk                                        /*!< Alternate Function I/O reset */
#define RCM_APB2RST_PARST_Pos                                       (2U)
#define RCM_APB2RST_PARST_Msk                                       (0x1UL << RCM_APB2RST_PARST_Pos)                               /*!< 0x00000004 */
#define RCM_APB2RST_PARST                                           RCM_APB2RST_PARST_Msk                                          /*!< I/O port A reset */
#define RCM_APB2RST_PBRST_Pos                                       (3U)
#define RCM_APB2RST_PBRST_Msk                                       (0x1UL << RCM_APB2RST_PBRST_Pos)                               /*!< 0x00000008 */
#define RCM_APB2RST_PBRST                                           RCM_APB2RST_PBRST_Msk                                          /*!< I/O port B reset */
#define RCM_APB2RST_PCRST_Pos                                       (4U)
#define RCM_APB2RST_PCRST_Msk                                       (0x1UL << RCM_APB2RST_PCRST_Pos)                               /*!< 0x00000010 */
#define RCM_APB2RST_PCRST                                           RCM_APB2RST_PCRST_Msk                                          /*!< I/O port C reset */
#define RCM_APB2RST_PDRST_Pos                                       (5U)
#define RCM_APB2RST_PDRST_Msk                                       (0x1UL << RCM_APB2RST_PDRST_Pos)                               /*!< 0x00000020 */
#define RCM_APB2RST_PDRST                                           RCM_APB2RST_PDRST_Msk                                          /*!< I/O port D reset */
#define RCM_APB2RST_ADC1RST_Pos                                     (9U)
#define RCM_APB2RST_ADC1RST_Msk                                     (0x1UL << RCM_APB2RST_ADC1RST_Pos)                             /*!< 0x00000200 */
#define RCM_APB2RST_ADC1RST                                         RCM_APB2RST_ADC1RST_Msk                                        /*!< ADC 1 interface reset */
#define RCM_APB2RST_ADC2RST_Pos                                     (10U)
#define RCM_APB2RST_ADC2RST_Msk                                     (0x1UL << RCM_APB2RST_ADC2RST_Pos)                             /*!< 0x00000400 */
#define RCM_APB2RST_ADC2RST                                         RCM_APB2RST_ADC2RST_Msk                                        /*!< ADC 2 interface reset */
#define RCM_APB2RST_TMR1RST_Pos                                     (11U)
#define RCM_APB2RST_TMR1RST_Msk                                     (0x1UL << RCM_APB2RST_TMR1RST_Pos)                             /*!< 0x00000800 */
#define RCM_APB2RST_TMR1RST                                         RCM_APB2RST_TMR1RST_Msk                                        /*!< Timer 1 reset */
#define RCM_APB2RST_SPI1RST_Pos                                     (12U)
#define RCM_APB2RST_SPI1RST_Msk                                     (0x1UL << RCM_APB2RST_SPI1RST_Pos)                             /*!< 0x00001000 */
#define RCM_APB2RST_SPI1RST                                         RCM_APB2RST_SPI1RST_Msk                                        /*!< SPI 1 reset */
#define RCM_APB2RST_TMR8RST_Pos                                     (13U)
#define RCM_APB2RST_TMR8RST_Msk                                     (0x1UL << RCM_APB2RST_TMR8RST_Pos)                             /*!< 0x00002000 */
#define RCM_APB2RST_TMR8RST                                         RCM_APB2RST_TMR8RST_Msk                                        /*!< Timer 8 reset */
#define RCM_APB2RST_USART1RST_Pos                                   (14U)
#define RCM_APB2RST_USART1RST_Msk                                   (0x1UL << RCM_APB2RST_USART1RST_Pos)                           /*!< 0x00004000 */
#define RCM_APB2RST_USART1RST                                       RCM_APB2RST_USART1RST_Msk                                      /*!< USART 1 reset */

/*******************  Bit definition for RCM_APB1RST register  ******************/
#define RCM_APB1RST_TMR2RST_Pos                                     (0U)
#define RCM_APB1RST_TMR2RST_Msk                                     (0x1UL << RCM_APB1RST_TMR2RST_Pos)                             /*!< 0x00000001 */
#define RCM_APB1RST_TMR2RST                                         RCM_APB1RST_TMR2RST_Msk                                        /*!< Timer 2 reset */
#define RCM_APB1RST_TMR3RST_Pos                                     (1U)
#define RCM_APB1RST_TMR3RST_Msk                                     (0x1UL << RCM_APB1RST_TMR3RST_Pos)                             /*!< 0x00000002 */
#define RCM_APB1RST_TMR3RST                                         RCM_APB1RST_TMR3RST_Msk                                        /*!< Timer 3 reset */
#define RCM_APB1RST_TMR4RST_Pos                                     (2U)
#define RCM_APB1RST_TMR4RST_Msk                                     (0x1UL << RCM_APB1RST_TMR4RST_Pos)                             /*!< 0x00000004 */
#define RCM_APB1RST_TMR4RST                                         RCM_APB1RST_TMR4RST_Msk                                        /*!< Timer 4 reset */
#define RCM_APB1RST_TMR5RST_Pos                                     (3U)
#define RCM_APB1RST_TMR5RST_Msk                                     (0x1UL << RCM_APB1RST_TMR5RST_Pos)                             /*!< 0x00000008 */
#define RCM_APB1RST_TMR5RST                                         RCM_APB1RST_TMR5RST_Msk                                        /*!< Timer 5 reset */
#define RCM_APB1RST_WWDTRST_Pos                                     (11U)
#define RCM_APB1RST_WWDTRST_Msk                                     (0x1UL << RCM_APB1RST_WWDTRST_Pos)                             /*!< 0x00000800 */
#define RCM_APB1RST_WWDTRST                                         RCM_APB1RST_WWDTRST_Msk                                        /*!< Window Watchdog reset */
#define RCM_APB1RST_SPI2RST_Pos                                     (14U)
#define RCM_APB1RST_SPI2RST_Msk                                     (0x1UL << RCM_APB1RST_SPI2RST_Pos)                             /*!< 0x00004000 */
#define RCM_APB1RST_SPI2RST                                         RCM_APB1RST_SPI2RST_Msk                                        /*!< SPI 2 reset */
#define RCM_APB1RST_USART2RST_Pos                                   (17U)
#define RCM_APB1RST_USART2RST_Msk                                   (0x1UL << RCM_APB1RST_USART2RST_Pos)                           /*!< 0x00020000 */
#define RCM_APB1RST_USART2RST                                       RCM_APB1RST_USART2RST_Msk                                      /*!< USART 2 reset */
#define RCM_APB1RST_USART3RST_Pos                                   (18U)
#define RCM_APB1RST_USART3RST_Msk                                   (0x1UL << RCM_APB1RST_USART3RST_Pos)                           /*!< 0x00040000 */
#define RCM_APB1RST_USART3RST                                       RCM_APB1RST_USART3RST_Msk                                      /*!< USART 3 reset */
#define RCM_APB1RST_UART4RST_Pos                                    (19U)
#define RCM_APB1RST_UART4RST_Msk                                    (0x1UL << RCM_APB1RST_UART4RST_Pos)                            /*!< 0x00080000 */
#define RCM_APB1RST_UART4RST                                        RCM_APB1RST_UART4RST_Msk                                       /*!< UART 4 reset */
#define RCM_APB1RST_I2C1RST_Pos                                     (21U)
#define RCM_APB1RST_I2C1RST_Msk                                     (0x1UL << RCM_APB1RST_I2C1RST_Pos)                             /*!< 0x00200000 */
#define RCM_APB1RST_I2C1RST                                         RCM_APB1RST_I2C1RST_Msk                                        /*!< I2C 1 reset */
#define RCM_APB1RST_CAN1RST_Pos                                     (25U)
#define RCM_APB1RST_CAN1RST_Msk                                     (0x1UL << RCM_APB1RST_CAN1RST_Pos)                             /*!< 0x02000000 */
#define RCM_APB1RST_CAN1RST                                         RCM_APB1RST_CAN1RST_Msk                                        /*!< CAN1 reset */
#define RCM_APB1RST_CAN2RST_Pos                                     (26U)
#define RCM_APB1RST_CAN2RST_Msk                                     (0x1UL << RCM_APB1RST_CAN2RST_Pos)                             /*!< 0x04000000 */
#define RCM_APB1RST_CAN2RST                                         RCM_APB1RST_CAN2RST_Msk                                        /*!< CAN2 reset */
#define RCM_APB1RST_BAKPRST_Pos                                     (27U)
#define RCM_APB1RST_BAKPRST_Msk                                     (0x1UL << RCM_APB1RST_BAKPRST_Pos)                             /*!< 0x08000000 */
#define RCM_APB1RST_BAKPRST                                         RCM_APB1RST_BAKPRST_Msk                                        /*!< Backup interface reset */
#define RCM_APB1RST_PMURST_Pos                                      (28U)
#define RCM_APB1RST_PMURST_Msk                                      (0x1UL << RCM_APB1RST_PMURST_Pos)                              /*!< 0x10000000 */
#define RCM_APB1RST_PMURST                                          RCM_APB1RST_PMURST_Msk                                         /*!< Power interface reset */

/********************  Bit definition for RCM_AHB1CLKEN register  ***************/
#define RCM_AHB1CLKEN_DMA1EN_Pos                                    (0U)
#define RCM_AHB1CLKEN_DMA1EN_Msk                                    (0x1UL << RCM_AHB1CLKEN_DMA1EN_Pos)                            /*!< 0x00000001 */
#define RCM_AHB1CLKEN_DMA1EN                                        RCM_AHB1CLKEN_DMA1EN_Msk                                       /*!< DMA1 clock enable */
#define RCM_AHB1CLKEN_DMA2EN_Pos                                    (1U)
#define RCM_AHB1CLKEN_DMA2EN_Msk                                    (0x1UL << RCM_AHB1CLKEN_DMA2EN_Pos)                            /*!< 0x00000002 */
#define RCM_AHB1CLKEN_DMA2EN                                        RCM_AHB1CLKEN_DMA2EN_Msk                                       /*!< DMA2 clock enable */
#define RCM_AHB1CLKEN_SRAMEN_Pos                                    (2U)
#define RCM_AHB1CLKEN_SRAMEN_Msk                                    (0x1UL << RCM_AHB1CLKEN_SRAMEN_Pos)                            /*!< 0x00000004 */
#define RCM_AHB1CLKEN_SRAMEN                                        RCM_AHB1CLKEN_SRAMEN_Msk                                       /*!< SRAM interface clock enable */
#define RCM_AHB1CLKEN_FMCEN_Pos                                     (4U)
#define RCM_AHB1CLKEN_FMCEN_Msk                                     (0x1UL << RCM_AHB1CLKEN_FMCEN_Pos)                             /*!< 0x00000010 */
#define RCM_AHB1CLKEN_FMCEN                                         RCM_AHB1CLKEN_FMCEN_Msk                                        /*!< FMC clock enable */
#define RCM_AHB1CLKEN_CRCEN_Pos                                     (6U)
#define RCM_AHB1CLKEN_CRCEN_Msk                                     (0x1UL << RCM_AHB1CLKEN_CRCEN_Pos)                             /*!< 0x00000040 */
#define RCM_AHB1CLKEN_CRCEN                                         RCM_AHB1CLKEN_CRCEN_Msk                                        /*!< CRC clock enable */
#define RCM_AHB1CLKEN_OTGFSEN_Pos                                   (9U)
#define RCM_AHB1CLKEN_OTGFSEN_Msk                                   (0x1UL << RCM_AHB1CLKEN_OTGFSEN_Pos)                           /*!< 0x00000200 */
#define RCM_AHB1CLKEN_OTGFSEN                                       RCM_AHB1CLKEN_OTGFSEN_Msk                                      /*!< OTG FS clock enable */

/********************  Bit definition for RCM_APB2CLKEN register  ***************/
#define RCM_APB2CLKEN_AFIOEN_Pos                                    (0U)
#define RCM_APB2CLKEN_AFIOEN_Msk                                    (0x1UL << RCM_APB2CLKEN_AFIOEN_Pos)                            /*!< 0x00000001 */
#define RCM_APB2CLKEN_AFIOEN                                        RCM_APB2CLKEN_AFIOEN_Msk                                       /*!< Alternate Function I/O clock enable */
#define RCM_APB2CLKEN_PAEN_Pos                                      (2U)
#define RCM_APB2CLKEN_PAEN_Msk                                      (0x1UL << RCM_APB2CLKEN_PAEN_Pos)                              /*!< 0x00000004 */
#define RCM_APB2CLKEN_PAEN                                          RCM_APB2CLKEN_PAEN_Msk                                         /*!< I/O port A clock enable */
#define RCM_APB2CLKEN_PBEN_Pos                                      (3U)
#define RCM_APB2CLKEN_PBEN_Msk                                      (0x1UL << RCM_APB2CLKEN_PBEN_Pos)                              /*!< 0x00000008 */
#define RCM_APB2CLKEN_PBEN                                          RCM_APB2CLKEN_PBEN_Msk                                         /*!< I/O port B clock enable */
#define RCM_APB2CLKEN_PCEN_Pos                                      (4U)
#define RCM_APB2CLKEN_PCEN_Msk                                      (0x1UL << RCM_APB2CLKEN_PCEN_Pos)                              /*!< 0x00000010 */
#define RCM_APB2CLKEN_PCEN                                          RCM_APB2CLKEN_PCEN_Msk                                         /*!< I/O port C clock enable */
#define RCM_APB2CLKEN_PDEN_Pos                                      (5U)
#define RCM_APB2CLKEN_PDEN_Msk                                      (0x1UL << RCM_APB2CLKEN_PDEN_Pos)                              /*!< 0x00000020 */
#define RCM_APB2CLKEN_PDEN                                          RCM_APB2CLKEN_PDEN_Msk                                         /*!< I/O port D clock enable */
#define RCM_APB2CLKEN_ADC1EN_Pos                                    (9U)
#define RCM_APB2CLKEN_ADC1EN_Msk                                    (0x1UL << RCM_APB2CLKEN_ADC1EN_Pos)                            /*!< 0x00000200 */
#define RCM_APB2CLKEN_ADC1EN                                        RCM_APB2CLKEN_ADC1EN_Msk                                       /*!< ADC 1 interface clock enable */
#define RCM_APB2CLKEN_ADC2EN_Pos                                    (10U)
#define RCM_APB2CLKEN_ADC2EN_Msk                                    (0x1UL << RCM_APB2CLKEN_ADC2EN_Pos)                            /*!< 0x00000400 */
#define RCM_APB2CLKEN_ADC2EN                                        RCM_APB2CLKEN_ADC2EN_Msk                                       /*!< ADC 2 interface clock enable */
#define RCM_APB2CLKEN_TMR1EN_Pos                                    (11U)
#define RCM_APB2CLKEN_TMR1EN_Msk                                    (0x1UL << RCM_APB2CLKEN_TMR1EN_Pos)                            /*!< 0x00000800 */
#define RCM_APB2CLKEN_TMR1EN                                        RCM_APB2CLKEN_TMR1EN_Msk                                       /*!< Timer 1 clock enable */
#define RCM_APB2CLKEN_SPI1EN_Pos                                    (12U)
#define RCM_APB2CLKEN_SPI1EN_Msk                                    (0x1UL << RCM_APB2CLKEN_SPI1EN_Pos)                            /*!< 0x00001000 */
#define RCM_APB2CLKEN_SPI1EN                                        RCM_APB2CLKEN_SPI1EN_Msk                                       /*!< SPI 1 clock enable */
#define RCM_APB2CLKEN_TMR8EN_Pos                                    (13U)
#define RCM_APB2CLKEN_TMR8EN_Msk                                    (0x1UL << RCM_APB2CLKEN_TMR8EN_Pos)                            /*!< 0x00002000 */
#define RCM_APB2CLKEN_TMR8EN                                        RCM_APB2CLKEN_TMR8EN_Msk                                       /*!< Timer 8 clock enable */
#define RCM_APB2CLKEN_USART1EN_Pos                                  (14U)
#define RCM_APB2CLKEN_USART1EN_Msk                                  (0x1UL << RCM_APB2CLKEN_USART1EN_Pos)                          /*!< 0x00004000 */
#define RCM_APB2CLKEN_USART1EN                                      RCM_APB2CLKEN_USART1EN_Msk                                     /*!< USART 1 clock enable */

/********************  Bit definition for RCM_APB1CLKEN register  ***************/
#define RCM_APB1CLKEN_TMR2EN_Pos                                    (0U)
#define RCM_APB1CLKEN_TMR2EN_Msk                                    (0x1UL << RCM_APB1CLKEN_TMR2EN_Pos)                            /*!< 0x00000001 */
#define RCM_APB1CLKEN_TMR2EN                                        RCM_APB1CLKEN_TMR2EN_Msk                                       /*!< Timer 2 clock enable */
#define RCM_APB1CLKEN_TMR3EN_Pos                                    (1U)
#define RCM_APB1CLKEN_TMR3EN_Msk                                    (0x1UL << RCM_APB1CLKEN_TMR3EN_Pos)                            /*!< 0x00000002 */
#define RCM_APB1CLKEN_TMR3EN                                        RCM_APB1CLKEN_TMR3EN_Msk                                       /*!< Timer 3 clock enable */
#define RCM_APB1CLKEN_TMR4EN_Pos                                    (2U)
#define RCM_APB1CLKEN_TMR4EN_Msk                                    (0x1UL << RCM_APB1CLKEN_TMR4EN_Pos)                            /*!< 0x00000004 */
#define RCM_APB1CLKEN_TMR4EN                                        RCM_APB1CLKEN_TMR4EN_Msk                                       /*!< Timer 4 clock enable */
#define RCM_APB1CLKEN_TMR5EN_Pos                                    (3U)
#define RCM_APB1CLKEN_TMR5EN_Msk                                    (0x1UL << RCM_APB1CLKEN_TMR5EN_Pos)                            /*!< 0x00000008 */
#define RCM_APB1CLKEN_TMR5EN                                        RCM_APB1CLKEN_TMR5EN_Msk                                       /*!< Timer 5 clock enable */
#define RCM_APB1CLKEN_WWDTEN_Pos                                    (11U)
#define RCM_APB1CLKEN_WWDTEN_Msk                                    (0x1UL << RCM_APB1CLKEN_WWDTEN_Pos)                            /*!< 0x00000800 */
#define RCM_APB1CLKEN_WWDTEN                                        RCM_APB1CLKEN_WWDTEN_Msk                                       /*!< Window Watchdog clock enable */
#define RCM_APB1CLKEN_SPI2EN_Pos                                    (14U)
#define RCM_APB1CLKEN_SPI2EN_Msk                                    (0x1UL << RCM_APB1CLKEN_SPI2EN_Pos)                            /*!< 0x00004000 */
#define RCM_APB1CLKEN_SPI2EN                                        RCM_APB1CLKEN_SPI2EN_Msk                                       /*!< SPI 2 clock enable */
#define RCM_APB1CLKEN_USART2EN_Pos                                  (17U)
#define RCM_APB1CLKEN_USART2EN_Msk                                  (0x1UL << RCM_APB1CLKEN_USART2EN_Pos)                          /*!< 0x00020000 */
#define RCM_APB1CLKEN_USART2EN                                      RCM_APB1CLKEN_USART2EN_Msk                                     /*!< USART 2 clock enable */
#define RCM_APB1CLKEN_USART3EN_Pos                                  (18U)
#define RCM_APB1CLKEN_USART3EN_Msk                                  (0x1UL << RCM_APB1CLKEN_USART3EN_Pos)                          /*!< 0x00040000 */
#define RCM_APB1CLKEN_USART3EN                                      RCM_APB1CLKEN_USART3EN_Msk                                     /*!< USART 3 clock enable */
#define RCM_APB1CLKEN_UART4EN_Pos                                   (19U)
#define RCM_APB1CLKEN_UART4EN_Msk                                   (0x1UL << RCM_APB1CLKEN_UART4EN_Pos)                           /*!< 0x00080000 */
#define RCM_APB1CLKEN_UART4EN                                       RCM_APB1CLKEN_UART4EN_Msk                                      /*!< UART 4 clock enable */
#define RCM_APB1CLKEN_I2C1EN_Pos                                    (21U)
#define RCM_APB1CLKEN_I2C1EN_Msk                                    (0x1UL << RCM_APB1CLKEN_I2C1EN_Pos)                            /*!< 0x00200000 */
#define RCM_APB1CLKEN_I2C1EN                                        RCM_APB1CLKEN_I2C1EN_Msk                                       /*!< I2C 1 clock enable */
#define RCM_APB1CLKEN_CAN1EN_Pos                                    (25U)
#define RCM_APB1CLKEN_CAN1EN_Msk                                    (0x1UL << RCM_APB1CLKEN_CAN1EN_Pos)                            /*!< 0x02000000 */
#define RCM_APB1CLKEN_CAN1EN                                        RCM_APB1CLKEN_CAN1EN_Msk                                       /*!< CAN1 clock enable */
#define RCM_APB1CLKEN_CAN2EN_Pos                                    (26U)
#define RCM_APB1CLKEN_CAN2EN_Msk                                    (0x1UL << RCM_APB1CLKEN_CAN2EN_Pos)                            /*!< 0x04000000 */
#define RCM_APB1CLKEN_CAN2EN                                        RCM_APB1CLKEN_CAN2EN_Msk                                       /*!< CAN2 clock enable */
#define RCM_APB1CLKEN_BAKPEN_Pos                                    (27U)
#define RCM_APB1CLKEN_BAKPEN_Msk                                    (0x1UL << RCM_APB1CLKEN_BAKPEN_Pos)                            /*!< 0x08000000 */
#define RCM_APB1CLKEN_BAKPEN                                        RCM_APB1CLKEN_BAKPEN_Msk                                       /*!< Backup interface clock enable */
#define RCM_APB1CLKEN_PMUEN_Pos                                     (28U)
#define RCM_APB1CLKEN_PMUEN_Msk                                     (0x1UL << RCM_APB1CLKEN_PMUEN_Pos)                             /*!< 0x10000000 */
#define RCM_APB1CLKEN_PMUEN                                         RCM_APB1CLKEN_PMUEN_Msk                                        /*!< Power interface clock enable */

/********************  Bit definition for RCM_BDCTRL register  ******************/
#define RCM_BDCTRL_LSEEN_Pos                                        (0U)
#define RCM_BDCTRL_LSEEN_Msk                                        (0x1UL << RCM_BDCTRL_LSEEN_Pos)                                /*!< 0x00000001 */
#define RCM_BDCTRL_LSEEN                                            RCM_BDCTRL_LSEEN_Msk
#define RCM_BDCTRL_LSERDYFLG_Pos                                    (1U)
#define RCM_BDCTRL_LSERDYFLG_Msk                                    (0x1UL << RCM_BDCTRL_LSERDYFLG_Pos)                            /*!< 0x00000002 */
#define RCM_BDCTRL_LSERDYFLG                                        RCM_BDCTRL_LSERDYFLG_Msk
#define RCM_BDCTRL_LSEBCFG_Pos                                      (2U)
#define RCM_BDCTRL_LSEBCFG_Msk                                      (0x1UL << RCM_BDCTRL_LSEBCFG_Pos)                              /*!< 0x00000004 */
#define RCM_BDCTRL_LSEBCFG                                          RCM_BDCTRL_LSEBCFG_Msk

#define RCM_BDCTRL_RTCSRCSEL_Pos                                    (8U)
#define RCM_BDCTRL_RTCSRCSEL_Msk                                    (0x3UL << RCM_BDCTRL_RTCSRCSEL_Pos)                            /*!< 0x00000300 */
#define RCM_BDCTRL_RTCSRCSEL                                        RCM_BDCTRL_RTCSRCSEL_Msk
#define RCM_BDCTRL_RTCSRCSEL_0                                      (0x1UL << RCM_BDCTRL_RTCSRCSEL_Pos)                            /*!< 0x00000100 */
#define RCM_BDCTRL_RTCSRCSEL_1                                      (0x2UL << RCM_BDCTRL_RTCSRCSEL_Pos)                            /*!< 0x00000200 */

#define RCM_BDCTRL_RTCCLKEN_Pos                                     (15U)
#define RCM_BDCTRL_RTCCLKEN_Msk                                     (0x1UL << RCM_BDCTRL_RTCCLKEN_Pos)                             /*!< 0x00008000 */
#define RCM_BDCTRL_RTCCLKEN                                         RCM_BDCTRL_RTCCLKEN_Msk
#define RCM_BDCTRL_BDRST_Pos                                        (16U)
#define RCM_BDCTRL_BDRST_Msk                                        (0x1UL << RCM_BDCTRL_BDRST_Pos)                                /*!< 0x00010000 */
#define RCM_BDCTRL_BDRST                                            RCM_BDCTRL_BDRST_Msk

/********************  Bit definition for RCM_CSTS register  *******************/
#define RCM_CSTS_LSIEN_Pos                                          (0U)
#define RCM_CSTS_LSIEN_Msk                                          (0x1UL << RCM_CSTS_LSIEN_Pos)                                  /*!< 0x00000001 */
#define RCM_CSTS_LSIEN                                              RCM_CSTS_LSIEN_Msk
#define RCM_CSTS_LSIRDYFLG_Pos                                      (1U)
#define RCM_CSTS_LSIRDYFLG_Msk                                      (0x1UL << RCM_CSTS_LSIRDYFLG_Pos)                              /*!< 0x00000002 */
#define RCM_CSTS_LSIRDYFLG                                          RCM_CSTS_LSIRDYFLG_Msk
#define RCM_CSTS_RSTFLGCLR_Pos                                      (24U)
#define RCM_CSTS_RSTFLGCLR_Msk                                      (0x1UL << RCM_CSTS_RSTFLGCLR_Pos)                              /*!< 0x01000000 */
#define RCM_CSTS_RSTFLGCLR                                          RCM_CSTS_RSTFLGCLR_Msk
#define RCM_CSTS_PINRSTFLG_Pos                                      (26U)
#define RCM_CSTS_PINRSTFLG_Msk                                      (0x1UL << RCM_CSTS_PINRSTFLG_Pos)                              /*!< 0x04000000 */
#define RCM_CSTS_PINRSTFLG                                          RCM_CSTS_PINRSTFLG_Msk
#define RCM_CSTS_PODRSTFLG_Pos                                      (27U)
#define RCM_CSTS_PODRSTFLG_Msk                                      (0x1UL << RCM_CSTS_PODRSTFLG_Pos)                              /*!< 0x08000000 */
#define RCM_CSTS_PODRSTFLG                                          RCM_CSTS_PODRSTFLG_Msk
#define RCM_CSTS_SWRSTFLG_Pos                                       (28U)
#define RCM_CSTS_SWRSTFLG_Msk                                       (0x1UL << RCM_CSTS_SWRSTFLG_Pos)                               /*!< 0x10000000 */
#define RCM_CSTS_SWRSTFLG                                           RCM_CSTS_SWRSTFLG_Msk
#define RCM_CSTS_IWDTRSTFLG_Pos                                     (29U)
#define RCM_CSTS_IWDTRSTFLG_Msk                                     (0x1UL << RCM_CSTS_IWDTRSTFLG_Pos)                             /*!< 0x20000000 */
#define RCM_CSTS_IWDTRSTFLG                                         RCM_CSTS_IWDTRSTFLG_Msk
#define RCM_CSTS_WWDTRSTFLG_Pos                                     (30U)
#define RCM_CSTS_WWDTRSTFLG_Msk                                     (0x1UL << RCM_CSTS_WWDTRSTFLG_Pos)                             /*!< 0x40000000 */
#define RCM_CSTS_WWDTRSTFLG                                         RCM_CSTS_WWDTRSTFLG_Msk
#define RCM_CSTS_LPWRRSTFLG_Pos                                     (31U)
#define RCM_CSTS_LPWRRSTFLG_Msk                                     (0x1UL << RCM_CSTS_LPWRRSTFLG_Pos)                             /*!< 0x80000000 */
#define RCM_CSTS_LPWRRSTFLG                                         RCM_CSTS_LPWRRSTFLG_Msk
/* Legacy defines */
#define RCM_CSTS_PADRSTF                                            RCM_CSTS_PINRSTFLG
#define RCM_CSTS_WDGRSTF                                            RCM_CSTS_IWDTRSTFLG

/******************************************************************************/
/*                                                                            */
/*                           Real-Time Clock (RTC)                            */
/*                                                                            */
/******************************************************************************/
#define RTC_ALARM_SUPPORT
#define RTC_COUNT_SUPPORT
/********************  Bits definition for RTC_CTRL register  *****************/
#define RTC_CTRL_SECIEN_Pos                                         (0U)
#define RTC_CTRL_SECIEN_Msk                                         (0x1UL << RTC_CTRL_SECIEN_Pos)                                 /*!< 0x00000001 */
#define RTC_CTRL_SECIEN                                             RTC_CTRL_SECIEN_Msk                                            /*!< Second interrupt enable */
#define RTC_CTRL_ALRIEN_Pos                                         (1U)
#define RTC_CTRL_ALRIEN_Msk                                         (0x1UL << RTC_CTRL_ALRIEN_Pos)                                 /*!< 0x00000002 */
#define RTC_CTRL_ALRIEN                                             RTC_CTRL_ALRIEN_Msk                                            /*!< Alarm interrupt enable */
#define RTC_CTRL_OVRIEN_Pos                                         (2U)
#define RTC_CTRL_OVRIEN_Msk                                         (0x1UL << RTC_CTRL_OVRIEN_Pos)                                 /*!< 0x00000004 */
#define RTC_CTRL_OVRIEN                                             RTC_CTRL_OVRIEN_Msk                                            /*!< Overflow interrupt enable */

/********************  Bits definition for RTC_CSTS register  *****************/
#define RTC_CSTS_SECFLG_Pos                                         (0U)
#define RTC_CSTS_SECFLG_Msk                                         (0x1UL << RTC_CSTS_SECFLG_Pos)                                 /*!< 0x00000001 */
#define RTC_CSTS_SECFLG                                             RTC_CSTS_SECFLG_Msk                                            /*!< Second flag */
#define RTC_CSTS_ALRFLG_Pos                                         (1U)
#define RTC_CSTS_ALRFLG_Msk                                         (0x1UL << RTC_CSTS_ALRFLG_Pos)                                 /*!< 0x00000002 */
#define RTC_CSTS_ALRFLG                                             RTC_CSTS_ALRFLG_Msk                                            /*!< Alarm flag */
#define RTC_CSTS_OVRFLG_Pos                                         (2U)
#define RTC_CSTS_OVRFLG_Msk                                         (0x1UL << RTC_CSTS_OVRFLG_Pos)                                 /*!< 0x00000004 */
#define RTC_CSTS_OVRFLG                                             RTC_CSTS_OVRFLG_Msk                                            /*!< Overflow flag */
#define RTC_CSTS_RSYNCFLG_Pos                                       (3U)
#define RTC_CSTS_RSYNCFLG_Msk                                       (0x1UL << RTC_CSTS_RSYNCFLG_Pos)                               /*!< 0x00000008 */
#define RTC_CSTS_RSYNCFLG                                           RTC_CSTS_RSYNCFLG_Msk                                          /*!< Registers synchronized flag */
#define RTC_CSTS_CFGMFLG_Pos                                        (4U)
#define RTC_CSTS_CFGMFLG_Msk                                        (0x1UL << RTC_CSTS_CFGMFLG_Pos)                                /*!< 0x00000010 */
#define RTC_CSTS_CFGMFLG                                            RTC_CSTS_CFGMFLG_Msk                                           /*!< Configuration mode enable flag */
#define RTC_CSTS_OCFLG_Pos                                          (5U)
#define RTC_CSTS_OCFLG_Msk                                          (0x1UL << RTC_CSTS_OCFLG_Pos)                                  /*!< 0x00000020 */
#define RTC_CSTS_OCFLG                                              RTC_CSTS_OCFLG_Msk                                             /*!< RTC operation complete flag */

/*******************  Bits definition for RTC_PSCRLDH register  *****************/
#define RTC_PSCRLDH_PSCRLDH_Pos                                     (0U)
#define RTC_PSCRLDH_PSCRLDH_Msk                                     (0xFUL << RTC_PSCRLDH_PSCRLDH_Pos)                             /*!< 0x0000000F */
#define RTC_PSCRLDH_PSCRLDH                                         RTC_PSCRLDH_PSCRLDH_Msk                                        /*!< RTC prescaler reload value high */

/*******************  Bits definition for RTC_PSCRLDL register  *****************/
#define RTC_PSCRLDL_PSCRLDL_Pos                                     (0U)
#define RTC_PSCRLDL_PSCRLDL_Msk                                     (0xFFFFUL << RTC_PSCRLDL_PSCRLDL_Pos)                          /*!< 0x0000FFFF */
#define RTC_PSCRLDL_PSCRLDL                                         RTC_PSCRLDL_PSCRLDL_Msk                                        /*!< RTC prescaler reload value low */

/*******************  Bits definition for RTC_PSCH register  *****************/
#define RTC_PSCH_PSCH_Pos                                           (0U)
#define RTC_PSCH_PSCH_Msk                                           (0xFUL << RTC_PSCH_PSCH_Pos)                                   /*!< 0x0000000F */
#define RTC_PSCH_PSCH                                               RTC_PSCH_PSCH_Msk                                              /*!< RTC clock divider high */

/*******************  Bits definition for RTC_PSCL register  *****************/
#define RTC_PSCL_PSCL_Pos                                           (0U)
#define RTC_PSCL_PSCL_Msk                                           (0xFFFFUL << RTC_PSCL_PSCL_Pos)                                /*!< 0x0000FFFF */
#define RTC_PSCL_PSCL                                               RTC_PSCL_PSCL_Msk                                              /*!< RTC clock divider low */

/*******************  Bits definition for RTC_CNTH register  *****************/
#define RTC_CNTH_CNTH_Pos                                           (0U)
#define RTC_CNTH_CNTH_Msk                                           (0xFFFFUL << RTC_CNTH_CNTH_Pos)                                /*!< 0x0000FFFF */
#define RTC_CNTH_CNTH                                               RTC_CNTH_CNTH_Msk                                              /*!< RTC counter high */

/*******************  Bits definition for RTC_CNTL register  *****************/
#define RTC_CNTL_CNTL_Pos                                           (0U)
#define RTC_CNTL_CNTL_Msk                                           (0xFFFFUL << RTC_CNTL_CNTL_Pos)                                /*!< 0x0000FFFF */
#define RTC_CNTL_CNTL                                               RTC_CNTL_CNTL_Msk                                              /*!< RTC counter low */

/*******************  Bits definition for RTC_ALRH register  *****************/
#define RTC_ALRH_ALRH_Pos                                           (0U)
#define RTC_ALRH_ALRH_Msk                                           (0xFFFFUL << RTC_ALRH_ALRH_Pos)                                /*!< 0x0000FFFF */
#define RTC_ALRH_ALRH                                               RTC_ALRH_ALRH_Msk                                              /*!< RTC alarm high */

/*******************  Bits definition for RTC_ALRL register  *****************/
#define RTC_ALRL_ALRL_Pos                                           (0U)
#define RTC_ALRL_ALRL_Msk                                           (0xFFFFUL << RTC_ALRL_ALRL_Pos)                                /*!< 0x0000FFFF */
#define RTC_ALRL_ALRL                                               RTC_ALRL_ALRL_Msk                                              /*!< RTC alarm low */

/******************************************************************************/
/*                                                                            */
/*                        Serial Peripheral Interface                         */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for SPI_CTRL1 register  ********************/
#define SPI_CTRL1_CPHA_Pos                                          (0U)
#define SPI_CTRL1_CPHA_Msk                                          (0x1UL << SPI_CTRL1_CPHA_Pos)                                  /*!< 0x00000001 */
#define SPI_CTRL1_CPHA                                              SPI_CTRL1_CPHA_Msk                                             /*!<Clock Phase      */
#define SPI_CTRL1_CPOL_Pos                                          (1U)
#define SPI_CTRL1_CPOL_Msk                                          (0x1UL << SPI_CTRL1_CPOL_Pos)                                  /*!< 0x00000002 */
#define SPI_CTRL1_CPOL                                              SPI_CTRL1_CPOL_Msk                                             /*!<Clock Polarity   */
#define SPI_CTRL1_MSMCFG_Pos                                        (2U)
#define SPI_CTRL1_MSMCFG_Msk                                        (0x1UL << SPI_CTRL1_MSMCFG_Pos)                                /*!< 0x00000004 */
#define SPI_CTRL1_MSMCFG                                            SPI_CTRL1_MSMCFG_Msk                                           /*!<Master Selection */

#define SPI_CTRL1_BRSEL_Pos                                         (3U)
#define SPI_CTRL1_BRSEL_Msk                                         (0x7UL << SPI_CTRL1_BRSEL_Pos)                                 /*!< 0x00000038 */
#define SPI_CTRL1_BRSEL                                             SPI_CTRL1_BRSEL_Msk                                            /*!<BR[2:0] bits (Baud Rate Control) */
#define SPI_CTRL1_BRSEL_0                                           (0x1UL << SPI_CTRL1_BRSEL_Pos)                                 /*!< 0x00000008 */
#define SPI_CTRL1_BRSEL_1                                           (0x2UL << SPI_CTRL1_BRSEL_Pos)                                 /*!< 0x00000010 */
#define SPI_CTRL1_BRSEL_2                                           (0x4UL << SPI_CTRL1_BRSEL_Pos)                                 /*!< 0x00000020 */

#define SPI_CTRL1_SPIEN_Pos                                         (6U)
#define SPI_CTRL1_SPIEN_Msk                                         (0x1UL << SPI_CTRL1_SPIEN_Pos)                                 /*!< 0x00000040 */
#define SPI_CTRL1_SPIEN                                             SPI_CTRL1_SPIEN_Msk                                            /*!<SPI Enable                          */
#define SPI_CTRL1_LSBSEL_Pos                                        (7U)
#define SPI_CTRL1_LSBSEL_Msk                                        (0x1UL << SPI_CTRL1_LSBSEL_Pos)                                /*!< 0x00000080 */
#define SPI_CTRL1_LSBSEL                                            SPI_CTRL1_LSBSEL_Msk                                           /*!<Frame Format                        */
#define SPI_CTRL1_ISSEL_Pos                                         (8U)
#define SPI_CTRL1_ISSEL_Msk                                         (0x1UL << SPI_CTRL1_ISSEL_Pos)                                 /*!< 0x00000100 */
#define SPI_CTRL1_ISSEL                                             SPI_CTRL1_ISSEL_Msk                                            /*!<Internal slave select               */
#define SPI_CTRL1_SSEN_Pos                                          (9U)
#define SPI_CTRL1_SSEN_Msk                                          (0x1UL << SPI_CTRL1_SSEN_Pos)                                  /*!< 0x00000200 */
#define SPI_CTRL1_SSEN                                              SPI_CTRL1_SSEN_Msk                                             /*!<Software slave management           */
#define SPI_CTRL1_RXOMEN_Pos                                        (10U)
#define SPI_CTRL1_RXOMEN_Msk                                        (0x1UL << SPI_CTRL1_RXOMEN_Pos)                                /*!< 0x00000400 */
#define SPI_CTRL1_RXOMEN                                            SPI_CTRL1_RXOMEN_Msk                                           /*!<Receive only                        */
#define SPI_CTRL1_DFLSEL_Pos                                        (11U)
#define SPI_CTRL1_DFLSEL_Msk                                        (0x1UL << SPI_CTRL1_DFLSEL_Pos)                                /*!< 0x00000800 */
#define SPI_CTRL1_DFLSEL                                            SPI_CTRL1_DFLSEL_Msk                                           /*!<Data Frame Format                   */
#define SPI_CTRL1_CRCNXT_Pos                                        (12U)
#define SPI_CTRL1_CRCNXT_Msk                                        (0x1UL << SPI_CTRL1_CRCNXT_Pos)                                /*!< 0x00001000 */
#define SPI_CTRL1_CRCNXT                                            SPI_CTRL1_CRCNXT_Msk                                           /*!<Transmit CRC next                   */
#define SPI_CTRL1_CRCEN_Pos                                         (13U)
#define SPI_CTRL1_CRCEN_Msk                                         (0x1UL << SPI_CTRL1_CRCEN_Pos)                                 /*!< 0x00002000 */
#define SPI_CTRL1_CRCEN                                             SPI_CTRL1_CRCEN_Msk                                            /*!<Hardware CRC calculation enable     */
#define SPI_CTRL1_BMOEN_Pos                                         (14U)
#define SPI_CTRL1_BMOEN_Msk                                         (0x1UL << SPI_CTRL1_BMOEN_Pos)                                 /*!< 0x00004000 */
#define SPI_CTRL1_BMOEN                                             SPI_CTRL1_BMOEN_Msk                                            /*!<Output enable in bidirectional mode */
#define SPI_CTRL1_BMEN_Pos                                          (15U)
#define SPI_CTRL1_BMEN_Msk                                          (0x1UL << SPI_CTRL1_BMEN_Pos)                                  /*!< 0x00008000 */
#define SPI_CTRL1_BMEN                                              SPI_CTRL1_BMEN_Msk                                             /*!<Bidirectional data mode enable      */

/*******************  Bit definition for SPI_CTRL2 register  ********************/
#define SPI_CTRL2_RXDEN_Pos                                         (0U)
#define SPI_CTRL2_RXDEN_Msk                                         (0x1UL << SPI_CTRL2_RXDEN_Pos)                                 /*!< 0x00000001 */
#define SPI_CTRL2_RXDEN                                             SPI_CTRL2_RXDEN_Msk                                            /*!<Rx Buffer DMA Enable                 */
#define SPI_CTRL2_TXDEN_Pos                                         (1U)
#define SPI_CTRL2_TXDEN_Msk                                         (0x1UL << SPI_CTRL2_TXDEN_Pos)                                 /*!< 0x00000002 */
#define SPI_CTRL2_TXDEN                                             SPI_CTRL2_TXDEN_Msk                                            /*!<Tx Buffer DMA Enable                 */
#define SPI_CTRL2_SSOEN_Pos                                         (2U)
#define SPI_CTRL2_SSOEN_Msk                                         (0x1UL << SPI_CTRL2_SSOEN_Pos)                                 /*!< 0x00000004 */
#define SPI_CTRL2_SSOEN                                             SPI_CTRL2_SSOEN_Msk                                            /*!<SS Output Enable                     */
#define SPI_CTRL2_ERRIEN_Pos                                        (5U)
#define SPI_CTRL2_ERRIEN_Msk                                        (0x1UL << SPI_CTRL2_ERRIEN_Pos)                                /*!< 0x00000020 */
#define SPI_CTRL2_ERRIEN                                            SPI_CTRL2_ERRIEN_Msk                                           /*!<Error Interrupt Enable               */
#define SPI_CTRL2_RXBNEIEN_Pos                                      (6U)
#define SPI_CTRL2_RXBNEIEN_Msk                                      (0x1UL << SPI_CTRL2_RXBNEIEN_Pos)                              /*!< 0x00000040 */
#define SPI_CTRL2_RXBNEIEN                                          SPI_CTRL2_RXBNEIEN_Msk                                         /*!<RX buffer Not Empty Interrupt Enable */
#define SPI_CTRL2_TXBEIEN_Pos                                       (7U)
#define SPI_CTRL2_TXBEIEN_Msk                                       (0x1UL << SPI_CTRL2_TXBEIEN_Pos)                               /*!< 0x00000080 */
#define SPI_CTRL2_TXBEIEN                                           SPI_CTRL2_TXBEIEN_Msk                                          /*!<Tx buffer Empty Interrupt Enable     */

/********************  Bit definition for SPI_STS register  ********************/
#define SPI_STS_RXBNEFLG_Pos                                        (0U)
#define SPI_STS_RXBNEFLG_Msk                                        (0x1UL << SPI_STS_RXBNEFLG_Pos)                                /*!< 0x00000001 */
#define SPI_STS_RXBNEFLG                                            SPI_STS_RXBNEFLG_Msk                                           /*!<Receive buffer Not Empty */
#define SPI_STS_TXBEFLG_Pos                                         (1U)
#define SPI_STS_TXBEFLG_Msk                                         (0x1UL << SPI_STS_TXBEFLG_Pos)                                 /*!< 0x00000002 */
#define SPI_STS_TXBEFLG                                             SPI_STS_TXBEFLG_Msk                                            /*!<Transmit buffer Empty    */
#define SPI_STS_SCHDIR_Pos                                          (2U)
#define SPI_STS_SCHDIR_Msk                                          (0x1UL << SPI_STS_SCHDIR_Pos)                                  /*!< 0x00000004 */
#define SPI_STS_SCHDIR                                              SPI_STS_SCHDIR_Msk                                             /*!<Channel side             */
#define SPI_STS_UDRFLG_Pos                                          (3U)
#define SPI_STS_UDRFLG_Msk                                          (0x1UL << SPI_STS_UDRFLG_Pos)                                  /*!< 0x00000008 */
#define SPI_STS_UDRFLG                                              SPI_STS_UDRFLG_Msk                                             /*!<Underrun flag            */
#define SPI_STS_CRCEFLG_Pos                                         (4U)
#define SPI_STS_CRCEFLG_Msk                                         (0x1UL << SPI_STS_CRCEFLG_Pos)                                 /*!< 0x00000010 */
#define SPI_STS_CRCEFLG                                             SPI_STS_CRCEFLG_Msk                                            /*!<CRC Error flag           */
#define SPI_STS_MEFLG_Pos                                           (5U)
#define SPI_STS_MEFLG_Msk                                           (0x1UL << SPI_STS_MEFLG_Pos)                                   /*!< 0x00000020 */
#define SPI_STS_MEFLG                                               SPI_STS_MEFLG_Msk                                              /*!<Mode fault               */
#define SPI_STS_OVRFLG_Pos                                          (6U)
#define SPI_STS_OVRFLG_Msk                                          (0x1UL << SPI_STS_OVRFLG_Pos)                                  /*!< 0x00000040 */
#define SPI_STS_OVRFLG                                              SPI_STS_OVRFLG_Msk                                             /*!<Overrun flag             */
#define SPI_STS_BSYFLG_Pos                                          (7U)
#define SPI_STS_BSYFLG_Msk                                          (0x1UL << SPI_STS_BSYFLG_Pos)                                  /*!< 0x00000080 */
#define SPI_STS_BSYFLG                                              SPI_STS_BSYFLG_Msk                                             /*!<Busy flag                */

/********************  Bit definition for SPI_DATA register  ********************/
#define SPI_DATA_DATA_Pos                                           (0U)
#define SPI_DATA_DATA_Msk                                           (0xFFFFUL << SPI_DATA_DATA_Pos)                                /*!< 0x0000FFFF */
#define SPI_DATA_DATA                                               SPI_DATA_DATA_Msk                                              /*!<Data Register           */

/*******************  Bit definition for SPI_CRCPOLY register  ******************/
#define SPI_CRCPOLY_CRCPOLY_Pos                                     (0U)
#define SPI_CRCPOLY_CRCPOLY_Msk                                     (0xFFFFUL << SPI_CRCPOLY_CRCPOLY_Pos)                          /*!< 0x0000FFFF */
#define SPI_CRCPOLY_CRCPOLY                                         SPI_CRCPOLY_CRCPOLY_Msk                                        /*!<CRC polynomial register */

/******************  Bit definition for SPI_RXCRC register  ******************/
#define SPI_RXCRC_RXCRC_Pos                                         (0U)
#define SPI_RXCRC_RXCRC_Msk                                         (0xFFFFUL << SPI_RXCRC_RXCRC_Pos)                              /*!< 0x0000FFFF */
#define SPI_RXCRC_RXCRC                                             SPI_RXCRC_RXCRC_Msk                                            /*!<Rx CRC Register         */

/******************  Bit definition for SPI_TXCRC register  ******************/
#define SPI_TXCRC_TXCRC_Pos                                         (0U)
#define SPI_TXCRC_TXCRC_Msk                                         (0xFFFFUL << SPI_TXCRC_TXCRC_Pos)                              /*!< 0x0000FFFF */
#define SPI_TXCRC_TXCRC                                             SPI_TXCRC_TXCRC_Msk                                            /*!<Tx CRC Register         */

/******************  Bit definition for SPI_I2SCFG register  *****************/
#define SPI_I2SCFG_CHLEN_Pos                                        (0U)
#define SPI_I2SCFG_CHLEN_Msk                                        (0x1UL << SPI_I2SCFG_CHLEN_Pos)                                /*!< 0x00000001 */
#define SPI_I2SCFG_CHLEN                                            SPI_I2SCFG_CHLEN_Msk                                           /*!<Channel length (number of bits per audio channel) */

#define SPI_I2SCFG_DATALEN_Pos                                      (1U)
#define SPI_I2SCFG_DATALEN_Msk                                      (0x3UL << SPI_I2SCFG_DATALEN_Pos)                              /*!< 0x00000006 */
#define SPI_I2SCFG_DATALEN                                          SPI_I2SCFG_DATALEN_Msk                                         /*!<DATLEN[1:0] bits (Data length to be transferred)  */
#define SPI_I2SCFG_DATALEN_0                                        (0x1UL << SPI_I2SCFG_DATALEN_Pos)                              /*!< 0x00000002 */
#define SPI_I2SCFG_DATALEN_1                                        (0x2UL << SPI_I2SCFG_DATALEN_Pos)                              /*!< 0x00000004 */

#define SPI_I2SCFG_CPOL_Pos                                         (3U)
#define SPI_I2SCFG_CPOL_Msk                                         (0x1UL << SPI_I2SCFG_CPOL_Pos)                                 /*!< 0x00000008 */
#define SPI_I2SCFG_CPOL                                             SPI_I2SCFG_CPOL_Msk                                            /*!<steady state clock polarity               */

#define SPI_I2SCFG_I2SSSEL_Pos                                      (4U)
#define SPI_I2SCFG_I2SSSEL_Msk                                      (0x3UL << SPI_I2SCFG_I2SSSEL_Pos)                              /*!< 0x00000030 */
#define SPI_I2SCFG_I2SSSEL                                          SPI_I2SCFG_I2SSSEL_Msk                                         /*!<I2SSTD[1:0] bits (I2S standard selection) */
#define SPI_I2SCFG_I2SSSEL_0                                        (0x1UL << SPI_I2SCFG_I2SSSEL_Pos)                              /*!< 0x00000010 */
#define SPI_I2SCFG_I2SSSEL_1                                        (0x2UL << SPI_I2SCFG_I2SSSEL_Pos)                              /*!< 0x00000020 */

#define SPI_I2SCFG_PFSSEL_Pos                                       (7U)
#define SPI_I2SCFG_PFSSEL_Msk                                       (0x1UL << SPI_I2SCFG_PFSSEL_Pos)                               /*!< 0x00000080 */
#define SPI_I2SCFG_PFSSEL                                           SPI_I2SCFG_PFSSEL_Msk                                          /*!<PCM frame synchronization                 */

#define SPI_I2SCFG_I2SMOD_Pos                                       (8U)
#define SPI_I2SCFG_I2SMOD_Msk                                       (0x3UL << SPI_I2SCFG_I2SMOD_Pos)                               /*!< 0x00000300 */
#define SPI_I2SCFG_I2SMOD                                           SPI_I2SCFG_I2SMOD_Msk                                          /*!<I2SCFG[1:0] bits (I2S configuration mode) */
#define SPI_I2SCFG_I2SMOD_0                                         (0x1UL << SPI_I2SCFG_I2SMOD_Pos)                               /*!< 0x00000100 */
#define SPI_I2SCFG_I2SMOD_1                                         (0x2UL << SPI_I2SCFG_I2SMOD_Pos)                               /*!< 0x00000200 */

#define SPI_I2SCFG_I2SEN_Pos                                        (10U)
#define SPI_I2SCFG_I2SEN_Msk                                        (0x1UL << SPI_I2SCFG_I2SEN_Pos)                                /*!< 0x00000400 */
#define SPI_I2SCFG_I2SEN                                            SPI_I2SCFG_I2SEN_Msk                                           /*!<I2S Enable         */
#define SPI_I2SCFG_MODESEL_Pos                                      (11U)
#define SPI_I2SCFG_MODESEL_Msk                                      (0x1UL << SPI_I2SCFG_MODESEL_Pos)                              /*!< 0x00000800 */
#define SPI_I2SCFG_MODESEL                                          SPI_I2SCFG_MODESEL_Msk                                         /*!<I2S mode selection */

/******************  Bit definition for SPI_I2SPSC register  *******************/
#define SPI_I2SPSC_I2SPSC_Pos                                       (0U)
#define SPI_I2SPSC_I2SPSC_Msk                                       (0xFFUL << SPI_I2SPSC_I2SPSC_Pos)                              /*!< 0x000000FF */
#define SPI_I2SPSC_I2SPSC                                           SPI_I2SPSC_I2SPSC_Msk                                          /*!<I2S Linear prescaler         */
#define SPI_I2SPSC_ODDPS_Pos                                        (8U)
#define SPI_I2SPSC_ODDPS_Msk                                        (0x1UL << SPI_I2SPSC_ODDPS_Pos)                                /*!< 0x00000100 */
#define SPI_I2SPSC_ODDPS                                            SPI_I2SPSC_ODDPS_Msk                                           /*!<Odd factor for the prescaler */
#define SPI_I2SPSC_MCOEN_Pos                                        (9U)
#define SPI_I2SPSC_MCOEN_Msk                                        (0x1UL << SPI_I2SPSC_MCOEN_Pos)                                /*!< 0x00000200 */
#define SPI_I2SPSC_MCOEN                                            SPI_I2SPSC_MCOEN_Msk                                           /*!<Master Clock Output Enable   */

/******************************************************************************/
/*                                                                            */
/*                                    TMR                                     */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for TMR_CTRL1 register  ********************/
#define TMR_CTRL1_CNTEN_Pos                                         (0U)
#define TMR_CTRL1_CNTEN_Msk                                         (0x1UL << TMR_CTRL1_CNTEN_Pos)                                 /*!< 0x00000001 */
#define TMR_CTRL1_CNTEN                                             TMR_CTRL1_CNTEN_Msk                                            /*!<Counter enable        */
#define TMR_CTRL1_UD_Pos                                            (1U)
#define TMR_CTRL1_UD_Msk                                            (0x1UL << TMR_CTRL1_UD_Pos)                                    /*!< 0x00000002 */
#define TMR_CTRL1_UD                                                TMR_CTRL1_UD_Msk                                               /*!<Update disable        */
#define TMR_CTRL1_URSSEL_Pos                                        (2U)
#define TMR_CTRL1_URSSEL_Msk                                        (0x1UL << TMR_CTRL1_URSSEL_Pos)                                /*!< 0x00000004 */
#define TMR_CTRL1_URSSEL                                            TMR_CTRL1_URSSEL_Msk                                           /*!<Update request source */
#define TMR_CTRL1_SPMEN_Pos                                         (3U)
#define TMR_CTRL1_SPMEN_Msk                                         (0x1UL << TMR_CTRL1_SPMEN_Pos)                                 /*!< 0x00000008 */
#define TMR_CTRL1_SPMEN                                             TMR_CTRL1_SPMEN_Msk                                            /*!<One pulse mode        */
#define TMR_CTRL1_CNTDIR_Pos                                        (4U)
#define TMR_CTRL1_CNTDIR_Msk                                        (0x1UL << TMR_CTRL1_CNTDIR_Pos)                                /*!< 0x00000010 */
#define TMR_CTRL1_CNTDIR                                            TMR_CTRL1_CNTDIR_Msk                                           /*!<Direction             */

#define TMR_CTRL1_CAMSEL_Pos                                        (5U)
#define TMR_CTRL1_CAMSEL_Msk                                        (0x3UL << TMR_CTRL1_CAMSEL_Pos)                                /*!< 0x00000060 */
#define TMR_CTRL1_CAMSEL                                            TMR_CTRL1_CAMSEL_Msk                                           /*!<CMS[1:0] bits (Center-aligned mode selection) */
#define TMR_CTRL1_CAMSEL_0                                          (0x1UL << TMR_CTRL1_CAMSEL_Pos)                                /*!< 0x0020 */
#define TMR_CTRL1_CAMSEL_1                                          (0x2UL << TMR_CTRL1_CAMSEL_Pos)                                /*!< 0x0040 */

#define TMR_CTRL1_ARPEN_Pos                                         (7U)
#define TMR_CTRL1_ARPEN_Msk                                         (0x1UL << TMR_CTRL1_ARPEN_Pos)                                 /*!< 0x00000080 */
#define TMR_CTRL1_ARPEN                                             TMR_CTRL1_ARPEN_Msk                                            /*!<Auto-reload preload enable     */

#define TMR_CTRL1_CLKDIV_Pos                                        (8U)
#define TMR_CTRL1_CLKDIV_Msk                                        (0x3UL << TMR_CTRL1_CLKDIV_Pos)                                /*!< 0x00000300 */
#define TMR_CTRL1_CLKDIV                                            TMR_CTRL1_CLKDIV_Msk                                           /*!<CKD[1:0] bits (clock division) */
#define TMR_CTRL1_CLKDIV_0                                          (0x1UL << TMR_CTRL1_CLKDIV_Pos)                                /*!< 0x0100 */
#define TMR_CTRL1_CLKDIV_1                                          (0x2UL << TMR_CTRL1_CLKDIV_Pos)                                /*!< 0x0200 */

/*******************  Bit definition for TMR_CTRL2 register  ********************/
#define TMR_CTRL2_CCPEN_Pos                                         (0U)
#define TMR_CTRL2_CCPEN_Msk                                         (0x1UL << TMR_CTRL2_CCPEN_Pos)                                 /*!< 0x00000001 */
#define TMR_CTRL2_CCPEN                                             TMR_CTRL2_CCPEN_Msk                                            /*!<Capture/Compare Preloaded Control        */
#define TMR_CTRL2_CCUSEL_Pos                                        (2U)
#define TMR_CTRL2_CCUSEL_Msk                                        (0x1UL << TMR_CTRL2_CCUSEL_Pos)                                /*!< 0x00000004 */
#define TMR_CTRL2_CCUSEL                                            TMR_CTRL2_CCUSEL_Msk                                           /*!<Capture/Compare Control Update Selection */
#define TMR_CTRL2_CCDSEL_Pos                                        (3U)
#define TMR_CTRL2_CCDSEL_Msk                                        (0x1UL << TMR_CTRL2_CCDSEL_Pos)                                /*!< 0x00000008 */
#define TMR_CTRL2_CCDSEL                                            TMR_CTRL2_CCDSEL_Msk                                           /*!<Capture/Compare DMA Selection            */

#define TMR_CTRL2_MMSEL_Pos                                         (4U)
#define TMR_CTRL2_MMSEL_Msk                                         (0x7UL << TMR_CTRL2_MMSEL_Pos)                                 /*!< 0x00000070 */
#define TMR_CTRL2_MMSEL                                             TMR_CTRL2_MMSEL_Msk                                            /*!<MMS[2:0] bits (Master Mode Selection) */
#define TMR_CTRL2_MMSEL_0                                           (0x1UL << TMR_CTRL2_MMSEL_Pos)                                 /*!< 0x0010 */
#define TMR_CTRL2_MMSEL_1                                           (0x2UL << TMR_CTRL2_MMSEL_Pos)                                 /*!< 0x0020 */
#define TMR_CTRL2_MMSEL_2                                           (0x4UL << TMR_CTRL2_MMSEL_Pos)                                 /*!< 0x0040 */

#define TMR_CTRL2_TI1SEL_Pos                                        (7U)
#define TMR_CTRL2_TI1SEL_Msk                                        (0x1UL << TMR_CTRL2_TI1SEL_Pos)                                /*!< 0x00000080 */
#define TMR_CTRL2_TI1SEL                                            TMR_CTRL2_TI1SEL_Msk                                           /*!<TI1 Selection */
#define TMR_CTRL2_OC1OIS_Pos                                        (8U)
#define TMR_CTRL2_OC1OIS_Msk                                        (0x1UL << TMR_CTRL2_OC1OIS_Pos)                                /*!< 0x00000100 */
#define TMR_CTRL2_OC1OIS                                            TMR_CTRL2_OC1OIS_Msk                                           /*!<Output Idle state 1 (OC1 output)  */
#define TMR_CTRL2_OC1NOIS_Pos                                       (9U)
#define TMR_CTRL2_OC1NOIS_Msk                                       (0x1UL << TMR_CTRL2_OC1NOIS_Pos)                               /*!< 0x00000200 */
#define TMR_CTRL2_OC1NOIS                                           TMR_CTRL2_OC1NOIS_Msk                                          /*!<Output Idle state 1 (OC1N output) */
#define TMR_CTRL2_OC2OIS_Pos                                        (10U)
#define TMR_CTRL2_OC2OIS_Msk                                        (0x1UL << TMR_CTRL2_OC2OIS_Pos)                                /*!< 0x00000400 */
#define TMR_CTRL2_OC2OIS                                            TMR_CTRL2_OC2OIS_Msk                                           /*!<Output Idle state 2 (OC2 output)  */
#define TMR_CTRL2_OC2NOIS_Pos                                       (11U)
#define TMR_CTRL2_OC2NOIS_Msk                                       (0x1UL << TMR_CTRL2_OC2NOIS_Pos)                               /*!< 0x00000800 */
#define TMR_CTRL2_OC2NOIS                                           TMR_CTRL2_OC2NOIS_Msk                                          /*!<Output Idle state 2 (OC2N output) */
#define TMR_CTRL2_OC3OIS_Pos                                        (12U)
#define TMR_CTRL2_OC3OIS_Msk                                        (0x1UL << TMR_CTRL2_OC3OIS_Pos)                                /*!< 0x00001000 */
#define TMR_CTRL2_OC3OIS                                            TMR_CTRL2_OC3OIS_Msk                                           /*!<Output Idle state 3 (OC3 output)  */
#define TMR_CTRL2_OC3NOIS_Pos                                       (13U)
#define TMR_CTRL2_OC3NOIS_Msk                                       (0x1UL << TMR_CTRL2_OC3NOIS_Pos)                               /*!< 0x00002000 */
#define TMR_CTRL2_OC3NOIS                                           TMR_CTRL2_OC3NOIS_Msk                                          /*!<Output Idle state 3 (OC3N output) */
#define TMR_CTRL2_OC4OIS_Pos                                        (14U)
#define TMR_CTRL2_OC4OIS_Msk                                        (0x1UL << TMR_CTRL2_OC4OIS_Pos)                                /*!< 0x00004000 */
#define TMR_CTRL2_OC4OIS                                            TMR_CTRL2_OC4OIS_Msk                                           /*!<Output Idle state 4 (OC4 output)  */

/*******************  Bit definition for TMR_SMCTRL register  *******************/
#define TMR_SMCTRL_SMFSEL_Pos                                       (0U)
#define TMR_SMCTRL_SMFSEL_Msk                                       (0x7UL << TMR_SMCTRL_SMFSEL_Pos)                               /*!< 0x00000007 */
#define TMR_SMCTRL_SMFSEL                                           TMR_SMCTRL_SMFSEL_Msk                                          /*!<SMS[2:0] bits (Slave mode selection)    */
#define TMR_SMCTRL_SMFSEL_0                                         (0x1UL << TMR_SMCTRL_SMFSEL_Pos)                               /*!< 0x0001 */
#define TMR_SMCTRL_SMFSEL_1                                         (0x2UL << TMR_SMCTRL_SMFSEL_Pos)                               /*!< 0x0002 */
#define TMR_SMCTRL_SMFSEL_2                                         (0x4UL << TMR_SMCTRL_SMFSEL_Pos)                               /*!< 0x0004 */

#define TMR_SMCTRL_TRGSEL_Pos                                       (4U)
#define TMR_SMCTRL_TRGSEL_Msk                                       (0x7UL << TMR_SMCTRL_TRGSEL_Pos)                               /*!< 0x00000070 */
#define TMR_SMCTRL_TRGSEL                                           TMR_SMCTRL_TRGSEL_Msk                                          /*!<TS[2:0] bits (Trigger selection)        */
#define TMR_SMCTRL_TRGSEL_0                                         (0x1UL << TMR_SMCTRL_TRGSEL_Pos)                               /*!< 0x0010 */
#define TMR_SMCTRL_TRGSEL_1                                         (0x2UL << TMR_SMCTRL_TRGSEL_Pos)                               /*!< 0x0020 */
#define TMR_SMCTRL_TRGSEL_2                                         (0x4UL << TMR_SMCTRL_TRGSEL_Pos)                               /*!< 0x0040 */

#define TMR_SMCTRL_MSMEN_Pos                                        (7U)
#define TMR_SMCTRL_MSMEN_Msk                                        (0x1UL << TMR_SMCTRL_MSMEN_Pos)                                /*!< 0x00000080 */
#define TMR_SMCTRL_MSMEN                                            TMR_SMCTRL_MSMEN_Msk                                           /*!<Master/slave mode                       */

#define TMR_SMCTRL_ETFCFG_Pos                                       (8U)
#define TMR_SMCTRL_ETFCFG_Msk                                       (0xFUL << TMR_SMCTRL_ETFCFG_Pos)                               /*!< 0x00000F00 */
#define TMR_SMCTRL_ETFCFG                                           TMR_SMCTRL_ETFCFG_Msk                                          /*!<ETF[3:0] bits (External trigger filter) */
#define TMR_SMCTRL_ETFCFG_0                                         (0x1UL << TMR_SMCTRL_ETFCFG_Pos)                               /*!< 0x0100 */
#define TMR_SMCTRL_ETFCFG_1                                         (0x2UL << TMR_SMCTRL_ETFCFG_Pos)                               /*!< 0x0200 */
#define TMR_SMCTRL_ETFCFG_2                                         (0x4UL << TMR_SMCTRL_ETFCFG_Pos)                               /*!< 0x0400 */
#define TMR_SMCTRL_ETFCFG_3                                         (0x8UL << TMR_SMCTRL_ETFCFG_Pos)                               /*!< 0x0800 */

#define TMR_SMCTRL_ETPCFG_Pos                                       (12U)
#define TMR_SMCTRL_ETPCFG_Msk                                       (0x3UL << TMR_SMCTRL_ETPCFG_Pos)                               /*!< 0x00003000 */
#define TMR_SMCTRL_ETPCFG                                           TMR_SMCTRL_ETPCFG_Msk                                          /*!<ETPS[1:0] bits (External trigger prescaler) */
#define TMR_SMCTRL_ETPCFG_0                                         (0x1UL << TMR_SMCTRL_ETPCFG_Pos)                               /*!< 0x1000 */
#define TMR_SMCTRL_ETPCFG_1                                         (0x2UL << TMR_SMCTRL_ETPCFG_Pos)                               /*!< 0x2000 */

#define TMR_SMCTRL_ECEN_Pos                                         (14U)
#define TMR_SMCTRL_ECEN_Msk                                         (0x1UL << TMR_SMCTRL_ECEN_Pos)                                 /*!< 0x00004000 */
#define TMR_SMCTRL_ECEN                                             TMR_SMCTRL_ECEN_Msk                                            /*!<External clock enable     */
#define TMR_SMCTRL_ETPOL_Pos                                        (15U)
#define TMR_SMCTRL_ETPOL_Msk                                        (0x1UL << TMR_SMCTRL_ETPOL_Pos)                                /*!< 0x00008000 */
#define TMR_SMCTRL_ETPOL                                            TMR_SMCTRL_ETPOL_Msk                                           /*!<External trigger polarity */

/*******************  Bit definition for TMR_DIEN register  *******************/
#define TMR_DIEN_UIEN_Pos                                           (0U)
#define TMR_DIEN_UIEN_Msk                                           (0x1UL << TMR_DIEN_UIEN_Pos)                                   /*!< 0x00000001 */
#define TMR_DIEN_UIEN                                               TMR_DIEN_UIEN_Msk                                              /*!<Update interrupt enable */
#define TMR_DIEN_CC1IEN_Pos                                         (1U)
#define TMR_DIEN_CC1IEN_Msk                                         (0x1UL << TMR_DIEN_CC1IEN_Pos)                                 /*!< 0x00000002 */
#define TMR_DIEN_CC1IEN                                             TMR_DIEN_CC1IEN_Msk                                            /*!<Capture/Compare 1 interrupt enable   */
#define TMR_DIEN_CC2IEN_Pos                                         (2U)
#define TMR_DIEN_CC2IEN_Msk                                         (0x1UL << TMR_DIEN_CC2IEN_Pos)                                 /*!< 0x00000004 */
#define TMR_DIEN_CC2IEN                                             TMR_DIEN_CC2IEN_Msk                                            /*!<Capture/Compare 2 interrupt enable   */
#define TMR_DIEN_CC3IEN_Pos                                         (3U)
#define TMR_DIEN_CC3IEN_Msk                                         (0x1UL << TMR_DIEN_CC3IEN_Pos)                                 /*!< 0x00000008 */
#define TMR_DIEN_CC3IEN                                             TMR_DIEN_CC3IEN_Msk                                            /*!<Capture/Compare 3 interrupt enable   */
#define TMR_DIEN_CC4IEN_Pos                                         (4U)
#define TMR_DIEN_CC4IEN_Msk                                         (0x1UL << TMR_DIEN_CC4IEN_Pos)                                 /*!< 0x00000010 */
#define TMR_DIEN_CC4IEN                                             TMR_DIEN_CC4IEN_Msk                                            /*!<Capture/Compare 4 interrupt enable   */
#define TMR_DIEN_COMIEN_Pos                                         (5U)
#define TMR_DIEN_COMIEN_Msk                                         (0x1UL << TMR_DIEN_COMIEN_Pos)                                 /*!< 0x00000020 */
#define TMR_DIEN_COMIEN                                             TMR_DIEN_COMIEN_Msk                                            /*!<COM interrupt enable                 */
#define TMR_DIEN_TRGIEN_Pos                                         (6U)
#define TMR_DIEN_TRGIEN_Msk                                         (0x1UL << TMR_DIEN_TRGIEN_Pos)                                 /*!< 0x00000040 */
#define TMR_DIEN_TRGIEN                                             TMR_DIEN_TRGIEN_Msk                                            /*!<Trigger interrupt enable             */
#define TMR_DIEN_BRKIEN_Pos                                         (7U)
#define TMR_DIEN_BRKIEN_Msk                                         (0x1UL << TMR_DIEN_BRKIEN_Pos)                                 /*!< 0x00000080 */
#define TMR_DIEN_BRKIEN                                             TMR_DIEN_BRKIEN_Msk                                            /*!<Break interrupt enable               */
#define TMR_DIEN_UDIEN_Pos                                          (8U)
#define TMR_DIEN_UDIEN_Msk                                          (0x1UL << TMR_DIEN_UDIEN_Pos)                                  /*!< 0x00000100 */
#define TMR_DIEN_UDIEN                                              TMR_DIEN_UDIEN_Msk                                             /*!<Update DMA request enable            */
#define TMR_DIEN_CC1DEN_Pos                                         (9U)
#define TMR_DIEN_CC1DEN_Msk                                         (0x1UL << TMR_DIEN_CC1DEN_Pos)                                 /*!< 0x00000200 */
#define TMR_DIEN_CC1DEN                                             TMR_DIEN_CC1DEN_Msk                                            /*!<Capture/Compare 1 DMA request enable */
#define TMR_DIEN_CC2DEN_Pos                                         (10U)
#define TMR_DIEN_CC2DEN_Msk                                         (0x1UL << TMR_DIEN_CC2DEN_Pos)                                 /*!< 0x00000400 */
#define TMR_DIEN_CC2DEN                                             TMR_DIEN_CC2DEN_Msk                                            /*!<Capture/Compare 2 DMA request enable */
#define TMR_DIEN_CC3DEN_Pos                                         (11U)
#define TMR_DIEN_CC3DEN_Msk                                         (0x1UL << TMR_DIEN_CC3DEN_Pos)                                 /*!< 0x00000800 */
#define TMR_DIEN_CC3DEN                                             TMR_DIEN_CC3DEN_Msk                                            /*!<Capture/Compare 3 DMA request enable */
#define TMR_DIEN_CC4DEN_Pos                                         (12U)
#define TMR_DIEN_CC4DEN_Msk                                         (0x1UL << TMR_DIEN_CC4DEN_Pos)                                 /*!< 0x00001000 */
#define TMR_DIEN_CC4DEN                                             TMR_DIEN_CC4DEN_Msk                                            /*!<Capture/Compare 4 DMA request enable */
#define TMR_DIEN_COMDEN_Pos                                         (13U)
#define TMR_DIEN_COMDEN_Msk                                         (0x1UL << TMR_DIEN_COMDEN_Pos)                                 /*!< 0x00002000 */
#define TMR_DIEN_COMDEN                                             TMR_DIEN_COMDEN_Msk                                            /*!<COM DMA request enable               */
#define TMR_DIEN_TRGDEN_Pos                                         (14U)
#define TMR_DIEN_TRGDEN_Msk                                         (0x1UL << TMR_DIEN_TRGDEN_Pos)                                 /*!< 0x00004000 */
#define TMR_DIEN_TRGDEN                                             TMR_DIEN_TRGDEN_Msk                                            /*!<Trigger DMA request enable           */

/********************  Bit definition for TMR_STS register  ********************/
#define TMR_STS_UIFLG_Pos                                           (0U)
#define TMR_STS_UIFLG_Msk                                           (0x1UL << TMR_STS_UIFLG_Pos)                                   /*!< 0x00000001 */
#define TMR_STS_UIFLG                                               TMR_STS_UIFLG_Msk                                              /*!<Update interrupt Flag              */
#define TMR_STS_CC1IFLG_Pos                                         (1U)
#define TMR_STS_CC1IFLG_Msk                                         (0x1UL << TMR_STS_CC1IFLG_Pos)                                 /*!< 0x00000002 */
#define TMR_STS_CC1IFLG                                             TMR_STS_CC1IFLG_Msk                                            /*!<Capture/Compare 1 interrupt Flag   */
#define TMR_STS_CC2IFLG_Pos                                         (2U)
#define TMR_STS_CC2IFLG_Msk                                         (0x1UL << TMR_STS_CC2IFLG_Pos)                                 /*!< 0x00000004 */
#define TMR_STS_CC2IFLG                                             TMR_STS_CC2IFLG_Msk                                            /*!<Capture/Compare 2 interrupt Flag   */
#define TMR_STS_CC3IFLG_Pos                                         (3U)
#define TMR_STS_CC3IFLG_Msk                                         (0x1UL << TMR_STS_CC3IFLG_Pos)                                 /*!< 0x00000008 */
#define TMR_STS_CC3IFLG                                             TMR_STS_CC3IFLG_Msk                                            /*!<Capture/Compare 3 interrupt Flag   */
#define TMR_STS_CC4IFLG_Pos                                         (4U)
#define TMR_STS_CC4IFLG_Msk                                         (0x1UL << TMR_STS_CC4IFLG_Pos)                                 /*!< 0x00000010 */
#define TMR_STS_CC4IFLG                                             TMR_STS_CC4IFLG_Msk                                            /*!<Capture/Compare 4 interrupt Flag   */
#define TMR_STS_COMIFLG_Pos                                         (5U)
#define TMR_STS_COMIFLG_Msk                                         (0x1UL << TMR_STS_COMIFLG_Pos)                                 /*!< 0x00000020 */
#define TMR_STS_COMIFLG                                             TMR_STS_COMIFLG_Msk                                            /*!<COM interrupt Flag                 */
#define TMR_STS_TRGIFLG_Pos                                         (6U)
#define TMR_STS_TRGIFLG_Msk                                         (0x1UL << TMR_STS_TRGIFLG_Pos)                                 /*!< 0x00000040 */
#define TMR_STS_TRGIFLG                                             TMR_STS_TRGIFLG_Msk                                            /*!<Trigger interrupt Flag             */
#define TMR_STS_BRKIFLG_Pos                                         (7U)
#define TMR_STS_BRKIFLG_Msk                                         (0x1UL << TMR_STS_BRKIFLG_Pos)                                 /*!< 0x00000080 */
#define TMR_STS_BRKIFLG                                             TMR_STS_BRKIFLG_Msk                                            /*!<Break interrupt Flag               */
#define TMR_STS_CC1RCFLG_Pos                                        (9U)
#define TMR_STS_CC1RCFLG_Msk                                        (0x1UL << TMR_STS_CC1RCFLG_Pos)                                /*!< 0x00000200 */
#define TMR_STS_CC1RCFLG                                            TMR_STS_CC1RCFLG_Msk                                           /*!<Capture/Compare 1 Overcapture Flag */
#define TMR_STS_CC2RCFLG_Pos                                        (10U)
#define TMR_STS_CC2RCFLG_Msk                                        (0x1UL << TMR_STS_CC2RCFLG_Pos)                                /*!< 0x00000400 */
#define TMR_STS_CC2RCFLG                                            TMR_STS_CC2RCFLG_Msk                                           /*!<Capture/Compare 2 Overcapture Flag */
#define TMR_STS_CC3RCFLG_Pos                                        (11U)
#define TMR_STS_CC3RCFLG_Msk                                        (0x1UL << TMR_STS_CC3RCFLG_Pos)                                /*!< 0x00000800 */
#define TMR_STS_CC3RCFLG                                            TMR_STS_CC3RCFLG_Msk                                           /*!<Capture/Compare 3 Overcapture Flag */
#define TMR_STS_CC4RCFLG_Pos                                        (12U)
#define TMR_STS_CC4RCFLG_Msk                                        (0x1UL << TMR_STS_CC4RCFLG_Pos)                                /*!< 0x00001000 */
#define TMR_STS_CC4RCFLG                                            TMR_STS_CC4RCFLG_Msk                                           /*!<Capture/Compare 4 Overcapture Flag */

/*******************  Bit definition for TMR_CEG register  ********************/
#define TMR_CEG_UEG_Pos                                             (0U)
#define TMR_CEG_UEG_Msk                                             (0x1UL << TMR_CEG_UEG_Pos)                                     /*!< 0x00000001 */
#define TMR_CEG_UEG                                                 TMR_CEG_UEG_Msk                                                /*!<Update Generation                         */
#define TMR_CEG_CC1EG_Pos                                           (1U)
#define TMR_CEG_CC1EG_Msk                                           (0x1UL << TMR_CEG_CC1EG_Pos)                                   /*!< 0x00000002 */
#define TMR_CEG_CC1EG                                               TMR_CEG_CC1EG_Msk                                              /*!<Capture/Compare 1 Generation              */
#define TMR_CEG_CC2EG_Pos                                           (2U)
#define TMR_CEG_CC2EG_Msk                                           (0x1UL << TMR_CEG_CC2EG_Pos)                                   /*!< 0x00000004 */
#define TMR_CEG_CC2EG                                               TMR_CEG_CC2EG_Msk                                              /*!<Capture/Compare 2 Generation              */
#define TMR_CEG_CC3EG_Pos                                           (3U)
#define TMR_CEG_CC3EG_Msk                                           (0x1UL << TMR_CEG_CC3EG_Pos)                                   /*!< 0x00000008 */
#define TMR_CEG_CC3EG                                               TMR_CEG_CC3EG_Msk                                              /*!<Capture/Compare 3 Generation              */
#define TMR_CEG_CC4EG_Pos                                           (4U)
#define TMR_CEG_CC4EG_Msk                                           (0x1UL << TMR_CEG_CC4EG_Pos)                                   /*!< 0x00000010 */
#define TMR_CEG_CC4EG                                               TMR_CEG_CC4EG_Msk                                              /*!<Capture/Compare 4 Generation              */
#define TMR_CEG_COMG_Pos                                            (5U)
#define TMR_CEG_COMG_Msk                                            (0x1UL << TMR_CEG_COMG_Pos)                                    /*!< 0x00000020 */
#define TMR_CEG_COMG                                                TMR_CEG_COMG_Msk                                               /*!<Capture/Compare Control Update Generation */
#define TMR_CEG_TEG_Pos                                             (6U)
#define TMR_CEG_TEG_Msk                                             (0x1UL << TMR_CEG_TEG_Pos)                                     /*!< 0x00000040 */
#define TMR_CEG_TEG                                                 TMR_CEG_TEG_Msk                                                /*!<Trigger Generation                        */
#define TMR_CEG_BEG_Pos                                             (7U)
#define TMR_CEG_BEG_Msk                                             (0x1UL << TMR_CEG_BEG_Pos)                                     /*!< 0x00000080 */
#define TMR_CEG_BEG                                                 TMR_CEG_BEG_Msk                                                /*!<Break Generation                          */

/******************  Bit definition for TMR_CCM1 register  *******************/
#define TMR_CCM1_CC1SEL_Pos                                         (0U)
#define TMR_CCM1_CC1SEL_Msk                                         (0x3UL << TMR_CCM1_CC1SEL_Pos)                                 /*!< 0x00000003 */
#define TMR_CCM1_CC1SEL                                             TMR_CCM1_CC1SEL_Msk                                            /*!<CC1S[1:0] bits (Capture/Compare 1 Selection) */
#define TMR_CCM1_CC1SEL_0                                           (0x1UL << TMR_CCM1_CC1SEL_Pos)                                 /*!< 0x0001 */
#define TMR_CCM1_CC1SEL_1                                           (0x2UL << TMR_CCM1_CC1SEL_Pos)                                 /*!< 0x0002 */

#define TMR_CCM1_OC1FEN_Pos                                         (2U)
#define TMR_CCM1_OC1FEN_Msk                                         (0x1UL << TMR_CCM1_OC1FEN_Pos)                                 /*!< 0x00000004 */
#define TMR_CCM1_OC1FEN                                             TMR_CCM1_OC1FEN_Msk                                            /*!<Output Compare 1 Fast enable                 */
#define TMR_CCM1_OC1PEN_Pos                                         (3U)
#define TMR_CCM1_OC1PEN_Msk                                         (0x1UL << TMR_CCM1_OC1PEN_Pos)                                 /*!< 0x00000008 */
#define TMR_CCM1_OC1PEN                                             TMR_CCM1_OC1PEN_Msk                                            /*!<Output Compare 1 Preload enable              */

#define TMR_CCM1_OC1MOD_Pos                                         (4U)
#define TMR_CCM1_OC1MOD_Msk                                         (0x7UL << TMR_CCM1_OC1MOD_Pos)                                 /*!< 0x00000070 */
#define TMR_CCM1_OC1MOD                                             TMR_CCM1_OC1MOD_Msk                                            /*!<OC1M[2:0] bits (Output Compare 1 Mode)       */
#define TMR_CCM1_OC1MOD_0                                           (0x1UL << TMR_CCM1_OC1MOD_Pos)                                 /*!< 0x0010 */
#define TMR_CCM1_OC1MOD_1                                           (0x2UL << TMR_CCM1_OC1MOD_Pos)                                 /*!< 0x0020 */
#define TMR_CCM1_OC1MOD_2                                           (0x4UL << TMR_CCM1_OC1MOD_Pos)                                 /*!< 0x0040 */

#define TMR_CCM1_OC1CEN_Pos                                         (7U)
#define TMR_CCM1_OC1CEN_Msk                                         (0x1UL << TMR_CCM1_OC1CEN_Pos)                                 /*!< 0x00000080 */
#define TMR_CCM1_OC1CEN                                             TMR_CCM1_OC1CEN_Msk                                            /*!<Output Compare 1Clear Enable                 */

#define TMR_CCM1_CC2SEL_Pos                                         (8U)
#define TMR_CCM1_CC2SEL_Msk                                         (0x3UL << TMR_CCM1_CC2SEL_Pos)                                 /*!< 0x00000300 */
#define TMR_CCM1_CC2SEL                                             TMR_CCM1_CC2SEL_Msk                                            /*!<CC2S[1:0] bits (Capture/Compare 2 Selection) */
#define TMR_CCM1_CC2SEL_0                                           (0x1UL << TMR_CCM1_CC2SEL_Pos)                                 /*!< 0x0100 */
#define TMR_CCM1_CC2SEL_1                                           (0x2UL << TMR_CCM1_CC2SEL_Pos)                                 /*!< 0x0200 */

#define TMR_CCM1_OC2FEN_Pos                                         (10U)
#define TMR_CCM1_OC2FEN_Msk                                         (0x1UL << TMR_CCM1_OC2FEN_Pos)                                 /*!< 0x00000400 */
#define TMR_CCM1_OC2FEN                                             TMR_CCM1_OC2FEN_Msk                                            /*!<Output Compare 2 Fast enable                 */
#define TMR_CCM1_OC2PEN_Pos                                         (11U)
#define TMR_CCM1_OC2PEN_Msk                                         (0x1UL << TMR_CCM1_OC2PEN_Pos)                                 /*!< 0x00000800 */
#define TMR_CCM1_OC2PEN                                             TMR_CCM1_OC2PEN_Msk                                            /*!<Output Compare 2 Preload enable              */

#define TMR_CCM1_OC2MOD_Pos                                         (12U)
#define TMR_CCM1_OC2MOD_Msk                                         (0x7UL << TMR_CCM1_OC2MOD_Pos)                                 /*!< 0x00007000 */
#define TMR_CCM1_OC2MOD                                             TMR_CCM1_OC2MOD_Msk                                            /*!<OC2M[2:0] bits (Output Compare 2 Mode)       */
#define TMR_CCM1_OC2MOD_0                                           (0x1UL << TMR_CCM1_OC2MOD_Pos)                                 /*!< 0x1000 */
#define TMR_CCM1_OC2MOD_1                                           (0x2UL << TMR_CCM1_OC2MOD_Pos)                                 /*!< 0x2000 */
#define TMR_CCM1_OC2MOD_2                                           (0x4UL << TMR_CCM1_OC2MOD_Pos)                                 /*!< 0x4000 */

#define TMR_CCM1_OC2CEN_Pos                                         (15U)
#define TMR_CCM1_OC2CEN_Msk                                         (0x1UL << TMR_CCM1_OC2CEN_Pos)                                 /*!< 0x00008000 */
#define TMR_CCM1_OC2CEN                                             TMR_CCM1_OC2CEN_Msk                                            /*!<Output Compare 2 Clear Enable */

/*----------------------------------------------------------------------------*/

#define TMR_CCM1_IC1PSC_Pos                                         (2U)
#define TMR_CCM1_IC1PSC_Msk                                         (0x3UL << TMR_CCM1_IC1PSC_Pos)                                 /*!< 0x0000000C */
#define TMR_CCM1_IC1PSC                                             TMR_CCM1_IC1PSC_Msk                                            /*!<IC1PSC[1:0] bits (Input Capture 1 Prescaler) */
#define TMR_CCM1_IC1PSC_0                                           (0x1UL << TMR_CCM1_IC1PSC_Pos)                                 /*!< 0x0004 */
#define TMR_CCM1_IC1PSC_1                                           (0x2UL << TMR_CCM1_IC1PSC_Pos)                                 /*!< 0x0008 */

#define TMR_CCM1_IC1F_Pos                                           (4U)
#define TMR_CCM1_IC1F_Msk                                           (0xFUL << TMR_CCM1_IC1F_Pos)                                   /*!< 0x000000F0 */
#define TMR_CCM1_IC1F                                               TMR_CCM1_IC1F_Msk                                              /*!<IC1F[3:0] bits (Input Capture 1 Filter)      */
#define TMR_CCM1_IC1F_0                                             (0x1UL << TMR_CCM1_IC1F_Pos)                                   /*!< 0x0010 */
#define TMR_CCM1_IC1F_1                                             (0x2UL << TMR_CCM1_IC1F_Pos)                                   /*!< 0x0020 */
#define TMR_CCM1_IC1F_2                                             (0x4UL << TMR_CCM1_IC1F_Pos)                                   /*!< 0x0040 */
#define TMR_CCM1_IC1F_3                                             (0x8UL << TMR_CCM1_IC1F_Pos)                                   /*!< 0x0080 */

#define TMR_CCM1_IC2PSC_Pos                                         (10U)
#define TMR_CCM1_IC2PSC_Msk                                         (0x3UL << TMR_CCM1_IC2PSC_Pos)                                 /*!< 0x00000C00 */
#define TMR_CCM1_IC2PSC                                             TMR_CCM1_IC2PSC_Msk                                            /*!<IC2PSC[1:0] bits (Input Capture 2 Prescaler)  */
#define TMR_CCM1_IC2PSC_0                                           (0x1UL << TMR_CCM1_IC2PSC_Pos)                                 /*!< 0x0400 */
#define TMR_CCM1_IC2PSC_1                                           (0x2UL << TMR_CCM1_IC2PSC_Pos)                                 /*!< 0x0800 */

#define TMR_CCM1_IC2F_Pos                                           (12U)
#define TMR_CCM1_IC2F_Msk                                           (0xFUL << TMR_CCM1_IC2F_Pos)                                   /*!< 0x0000F000 */
#define TMR_CCM1_IC2F                                               TMR_CCM1_IC2F_Msk                                              /*!<IC2F[3:0] bits (Input Capture 2 Filter)       */
#define TMR_CCM1_IC2F_0                                             (0x1UL << TMR_CCM1_IC2F_Pos)                                   /*!< 0x1000 */
#define TMR_CCM1_IC2F_1                                             (0x2UL << TMR_CCM1_IC2F_Pos)                                   /*!< 0x2000 */
#define TMR_CCM1_IC2F_2                                             (0x4UL << TMR_CCM1_IC2F_Pos)                                   /*!< 0x4000 */
#define TMR_CCM1_IC2F_3                                             (0x8UL << TMR_CCM1_IC2F_Pos)                                   /*!< 0x8000 */

/******************  Bit definition for TMR_CCM2 register  *******************/
#define TMR_CCM2_CC3SEL_Pos                                         (0U)
#define TMR_CCM2_CC3SEL_Msk                                         (0x3UL << TMR_CCM2_CC3SEL_Pos)                                 /*!< 0x00000003 */
#define TMR_CCM2_CC3SEL                                             TMR_CCM2_CC3SEL_Msk                                            /*!<CC3S[1:0] bits (Capture/Compare 3 Selection)  */
#define TMR_CCM2_CC3SEL_0                                           (0x1UL << TMR_CCM2_CC3SEL_Pos)                                 /*!< 0x0001 */
#define TMR_CCM2_CC3SEL_1                                           (0x2UL << TMR_CCM2_CC3SEL_Pos)                                 /*!< 0x0002 */

#define TMR_CCM2_OC3FEN_Pos                                         (2U)
#define TMR_CCM2_OC3FEN_Msk                                         (0x1UL << TMR_CCM2_OC3FEN_Pos)                                 /*!< 0x00000004 */
#define TMR_CCM2_OC3FEN                                             TMR_CCM2_OC3FEN_Msk                                            /*!<Output Compare 3 Fast enable           */
#define TMR_CCM2_OC3PEN_Pos                                         (3U)
#define TMR_CCM2_OC3PEN_Msk                                         (0x1UL << TMR_CCM2_OC3PEN_Pos)                                 /*!< 0x00000008 */
#define TMR_CCM2_OC3PEN                                             TMR_CCM2_OC3PEN_Msk                                            /*!<Output Compare 3 Preload enable        */

#define TMR_CCM2_OC3MOD_Pos                                         (4U)
#define TMR_CCM2_OC3MOD_Msk                                         (0x7UL << TMR_CCM2_OC3MOD_Pos)                                 /*!< 0x00000070 */
#define TMR_CCM2_OC3MOD                                             TMR_CCM2_OC3MOD_Msk                                            /*!<OC3M[2:0] bits (Output Compare 3 Mode) */
#define TMR_CCM2_OC3MOD_0                                           (0x1UL << TMR_CCM2_OC3MOD_Pos)                                 /*!< 0x0010 */
#define TMR_CCM2_OC3MOD_1                                           (0x2UL << TMR_CCM2_OC3MOD_Pos)                                 /*!< 0x0020 */
#define TMR_CCM2_OC3MOD_2                                           (0x4UL << TMR_CCM2_OC3MOD_Pos)                                 /*!< 0x0040 */

#define TMR_CCM2_OC3CEN_Pos                                         (7U)
#define TMR_CCM2_OC3CEN_Msk                                         (0x1UL << TMR_CCM2_OC3CEN_Pos)                                 /*!< 0x00000080 */
#define TMR_CCM2_OC3CEN                                             TMR_CCM2_OC3CEN_Msk                                            /*!<Output Compare 3 Clear Enable */

#define TMR_CCM2_CC4SEL_Pos                                         (8U)
#define TMR_CCM2_CC4SEL_Msk                                         (0x3UL << TMR_CCM2_CC4SEL_Pos)                                 /*!< 0x00000300 */
#define TMR_CCM2_CC4SEL                                             TMR_CCM2_CC4SEL_Msk                                            /*!<CC4S[1:0] bits (Capture/Compare 4 Selection) */
#define TMR_CCM2_CC4SEL_0                                           (0x1UL << TMR_CCM2_CC4SEL_Pos)                                 /*!< 0x0100 */
#define TMR_CCM2_CC4SEL_1                                           (0x2UL << TMR_CCM2_CC4SEL_Pos)                                 /*!< 0x0200 */

#define TMR_CCM2_OC4FEN_Pos                                         (10U)
#define TMR_CCM2_OC4FEN_Msk                                         (0x1UL << TMR_CCM2_OC4FEN_Pos)                                 /*!< 0x00000400 */
#define TMR_CCM2_OC4FEN                                             TMR_CCM2_OC4FEN_Msk                                            /*!<Output Compare 4 Fast enable    */
#define TMR_CCM2_OC4PEN_Pos                                         (11U)
#define TMR_CCM2_OC4PEN_Msk                                         (0x1UL << TMR_CCM2_OC4PEN_Pos)                                 /*!< 0x00000800 */
#define TMR_CCM2_OC4PEN                                             TMR_CCM2_OC4PEN_Msk                                            /*!<Output Compare 4 Preload enable */

#define TMR_CCM2_OC4MOD_Pos                                         (12U)
#define TMR_CCM2_OC4MOD_Msk                                         (0x7UL << TMR_CCM2_OC4MOD_Pos)                                 /*!< 0x00007000 */
#define TMR_CCM2_OC4MOD                                             TMR_CCM2_OC4MOD_Msk                                            /*!<OC4M[2:0] bits (Output Compare 4 Mode) */
#define TMR_CCM2_OC4MOD_0                                           (0x1UL << TMR_CCM2_OC4MOD_Pos)                                 /*!< 0x1000 */
#define TMR_CCM2_OC4MOD_1                                           (0x2UL << TMR_CCM2_OC4MOD_Pos)                                 /*!< 0x2000 */
#define TMR_CCM2_OC4MOD_2                                           (0x4UL << TMR_CCM2_OC4MOD_Pos)                                 /*!< 0x4000 */

#define TMR_CCM2_OC4CEN_Pos                                         (15U)
#define TMR_CCM2_OC4CEN_Msk                                         (0x1UL << TMR_CCM2_OC4CEN_Pos)                                 /*!< 0x00008000 */
#define TMR_CCM2_OC4CEN                                             TMR_CCM2_OC4CEN_Msk                                            /*!<Output Compare 4 Clear Enable */

/*----------------------------------------------------------------------------*/

#define TMR_CCM2_IC3PSC_Pos                                         (2U)
#define TMR_CCM2_IC3PSC_Msk                                         (0x3UL << TMR_CCM2_IC3PSC_Pos)                                 /*!< 0x0000000C */
#define TMR_CCM2_IC3PSC                                             TMR_CCM2_IC3PSC_Msk                                            /*!<IC3PSC[1:0] bits (Input Capture 3 Prescaler) */
#define TMR_CCM2_IC3PSC_0                                           (0x1UL << TMR_CCM2_IC3PSC_Pos)                                 /*!< 0x0004 */
#define TMR_CCM2_IC3PSC_1                                           (0x2UL << TMR_CCM2_IC3PSC_Pos)                                 /*!< 0x0008 */

#define TMR_CCM2_IC3F_Pos                                           (4U)
#define TMR_CCM2_IC3F_Msk                                           (0xFUL << TMR_CCM2_IC3F_Pos)                                   /*!< 0x000000F0 */
#define TMR_CCM2_IC3F                                               TMR_CCM2_IC3F_Msk                                              /*!<IC3F[3:0] bits (Input Capture 3 Filter) */
#define TMR_CCM2_IC3F_0                                             (0x1UL << TMR_CCM2_IC3F_Pos)                                   /*!< 0x0010 */
#define TMR_CCM2_IC3F_1                                             (0x2UL << TMR_CCM2_IC3F_Pos)                                   /*!< 0x0020 */
#define TMR_CCM2_IC3F_2                                             (0x4UL << TMR_CCM2_IC3F_Pos)                                   /*!< 0x0040 */
#define TMR_CCM2_IC3F_3                                             (0x8UL << TMR_CCM2_IC3F_Pos)                                   /*!< 0x0080 */

#define TMR_CCM2_IC4PSC_Pos                                         (10U)
#define TMR_CCM2_IC4PSC_Msk                                         (0x3UL << TMR_CCM2_IC4PSC_Pos)                                 /*!< 0x00000C00 */
#define TMR_CCM2_IC4PSC                                             TMR_CCM2_IC4PSC_Msk                                            /*!<IC4PSC[1:0] bits (Input Capture 4 Prescaler) */
#define TMR_CCM2_IC4PSC_0                                           (0x1UL << TMR_CCM2_IC4PSC_Pos)                                 /*!< 0x0400 */
#define TMR_CCM2_IC4PSC_1                                           (0x2UL << TMR_CCM2_IC4PSC_Pos)                                 /*!< 0x0800 */

#define TMR_CCM2_IC4F_Pos                                           (12U)
#define TMR_CCM2_IC4F_Msk                                           (0xFUL << TMR_CCM2_IC4F_Pos)                                   /*!< 0x0000F000 */
#define TMR_CCM2_IC4F                                               TMR_CCM2_IC4F_Msk                                              /*!<IC4F[3:0] bits (Input Capture 4 Filter) */
#define TMR_CCM2_IC4F_0                                             (0x1UL << TMR_CCM2_IC4F_Pos)                                   /*!< 0x1000 */
#define TMR_CCM2_IC4F_1                                             (0x2UL << TMR_CCM2_IC4F_Pos)                                   /*!< 0x2000 */
#define TMR_CCM2_IC4F_2                                             (0x4UL << TMR_CCM2_IC4F_Pos)                                   /*!< 0x4000 */
#define TMR_CCM2_IC4F_3                                             (0x8UL << TMR_CCM2_IC4F_Pos)                                   /*!< 0x8000 */

/*******************  Bit definition for TMR_CCEN register  *******************/
#define TMR_CCEN_CC1EN_Pos                                          (0U)
#define TMR_CCEN_CC1EN_Msk                                          (0x1UL << TMR_CCEN_CC1EN_Pos)                                  /*!< 0x00000001 */
#define TMR_CCEN_CC1EN                                              TMR_CCEN_CC1EN_Msk                                             /*!<Capture/Compare 1 output enable                 */
#define TMR_CCEN_CC1POL_Pos                                         (1U)
#define TMR_CCEN_CC1POL_Msk                                         (0x1UL << TMR_CCEN_CC1POL_Pos)                                 /*!< 0x00000002 */
#define TMR_CCEN_CC1POL                                             TMR_CCEN_CC1POL_Msk                                            /*!<Capture/Compare 1 output Polarity               */
#define TMR_CCEN_CC1NEN_Pos                                         (2U)
#define TMR_CCEN_CC1NEN_Msk                                         (0x1UL << TMR_CCEN_CC1NEN_Pos)                                 /*!< 0x00000004 */
#define TMR_CCEN_CC1NEN                                             TMR_CCEN_CC1NEN_Msk                                            /*!<Capture/Compare 1 Complementary output enable   */
#define TMR_CCEN_CC1NPOL_Pos                                        (3U)
#define TMR_CCEN_CC1NPOL_Msk                                        (0x1UL << TMR_CCEN_CC1NPOL_Pos)                                /*!< 0x00000008 */
#define TMR_CCEN_CC1NPOL                                            TMR_CCEN_CC1NPOL_Msk                                           /*!<Capture/Compare 1 Complementary output Polarity */
#define TMR_CCEN_CC2EN_Pos                                          (4U)
#define TMR_CCEN_CC2EN_Msk                                          (0x1UL << TMR_CCEN_CC2EN_Pos)                                  /*!< 0x00000010 */
#define TMR_CCEN_CC2EN                                              TMR_CCEN_CC2EN_Msk                                             /*!<Capture/Compare 2 output enable                 */
#define TMR_CCEN_CC2POL_Pos                                         (5U)
#define TMR_CCEN_CC2POL_Msk                                         (0x1UL << TMR_CCEN_CC2POL_Pos)                                 /*!< 0x00000020 */
#define TMR_CCEN_CC2POL                                             TMR_CCEN_CC2POL_Msk                                            /*!<Capture/Compare 2 output Polarity               */
#define TMR_CCEN_CC2NEN_Pos                                         (6U)
#define TMR_CCEN_CC2NEN_Msk                                         (0x1UL << TMR_CCEN_CC2NEN_Pos)                                 /*!< 0x00000040 */
#define TMR_CCEN_CC2NEN                                             TMR_CCEN_CC2NEN_Msk                                            /*!<Capture/Compare 2 Complementary output enable   */
#define TMR_CCEN_CC2NPOL_Pos                                        (7U)
#define TMR_CCEN_CC2NPOL_Msk                                        (0x1UL << TMR_CCEN_CC2NPOL_Pos)                                /*!< 0x00000080 */
#define TMR_CCEN_CC2NPOL                                            TMR_CCEN_CC2NPOL_Msk                                           /*!<Capture/Compare 2 Complementary output Polarity */
#define TMR_CCEN_CC3EN_Pos                                          (8U)
#define TMR_CCEN_CC3EN_Msk                                          (0x1UL << TMR_CCEN_CC3EN_Pos)                                  /*!< 0x00000100 */
#define TMR_CCEN_CC3EN                                              TMR_CCEN_CC3EN_Msk                                             /*!<Capture/Compare 3 output enable                 */
#define TMR_CCEN_CC3POL_Pos                                         (9U)
#define TMR_CCEN_CC3POL_Msk                                         (0x1UL << TMR_CCEN_CC3POL_Pos)                                 /*!< 0x00000200 */
#define TMR_CCEN_CC3POL                                             TMR_CCEN_CC3POL_Msk                                            /*!<Capture/Compare 3 output Polarity               */
#define TMR_CCEN_CC3NEN_Pos                                         (10U)
#define TMR_CCEN_CC3NEN_Msk                                         (0x1UL << TMR_CCEN_CC3NEN_Pos)                                 /*!< 0x00000400 */
#define TMR_CCEN_CC3NEN                                             TMR_CCEN_CC3NEN_Msk                                            /*!<Capture/Compare 3 Complementary output enable   */
#define TMR_CCEN_CC3NPOL_Pos                                        (11U)
#define TMR_CCEN_CC3NPOL_Msk                                        (0x1UL << TMR_CCEN_CC3NPOL_Pos)                                /*!< 0x00000800 */
#define TMR_CCEN_CC3NPOL                                            TMR_CCEN_CC3NPOL_Msk                                           /*!<Capture/Compare 3 Complementary output Polarity */
#define TMR_CCEN_CC4EN_Pos                                          (12U)
#define TMR_CCEN_CC4EN_Msk                                          (0x1UL << TMR_CCEN_CC4EN_Pos)                                  /*!< 0x00001000 */
#define TMR_CCEN_CC4EN                                              TMR_CCEN_CC4EN_Msk                                             /*!<Capture/Compare 4 output enable                 */
#define TMR_CCEN_CC4POL_Pos                                         (13U)
#define TMR_CCEN_CC4POL_Msk                                         (0x1UL << TMR_CCEN_CC4POL_Pos)                                 /*!< 0x00002000 */
#define TMR_CCEN_CC4POL                                             TMR_CCEN_CC4POL_Msk                                            /*!<Capture/Compare 4 output Polarity               */
#define TMR_CCEN_CC4NPOL_Pos                                        (15U)
#define TMR_CCEN_CC4NPOL_Msk                                        (0x1UL << TMR_CCEN_CC4NPOL_Pos)                                /*!< 0x00008000 */
#define TMR_CCEN_CC4NPOL                                            TMR_CCEN_CC4NPOL_Msk                                           /*!<Capture/Compare 4 Complementary output Polarity */

/*******************  Bit definition for TMR_CNT register  ********************/
#define TMR_CNT_CNT_Pos                                             (0U)
#define TMR_CNT_CNT_Msk                                             (0xFFFFFFFFUL << TMR_CNT_CNT_Pos)                              /*!< 0xFFFFFFFF */
#define TMR_CNT_CNT                                                 TMR_CNT_CNT_Msk                                                /*!<Counter Value            */

/*******************  Bit definition for TMR_PSC register  ********************/
#define TMR_PSC_PSC_Pos                                             (0U)
#define TMR_PSC_PSC_Msk                                             (0xFFFFUL << TMR_PSC_PSC_Pos)                                  /*!< 0x0000FFFF */
#define TMR_PSC_PSC                                                 TMR_PSC_PSC_Msk                                                /*!<Prescaler Value          */

/*******************  Bit definition for TMR_AUTORLD register  ********************/
#define TMR_AUTORLD_AUTORLD_Pos                                     (0U)
#define TMR_AUTORLD_AUTORLD_Msk                                     (0xFFFFFFFFUL << TMR_AUTORLD_AUTORLD_Pos)                      /*!< 0xFFFFFFFF */
#define TMR_AUTORLD_AUTORLD                                         TMR_AUTORLD_AUTORLD_Msk                                        /*!<actual auto-reload Value */

/*******************  Bit definition for TMR_REPCNT register  ********************/
#define TMR_REPCNT_REPCNT_Pos                                       (0U)
#define TMR_REPCNT_REPCNT_Msk                                       (0xFFUL << TMR_REPCNT_REPCNT_Pos)                              /*!< 0x000000FF */
#define TMR_REPCNT_REPCNT                                           TMR_REPCNT_REPCNT_Msk                                          /*!<Repetition Counter Value */

/*******************  Bit definition for TMR_CC1 register  *******************/
#define TMR_CC1_CC1_Pos                                             (0U)
#define TMR_CC1_CC1_Msk                                             (0xFFFFUL << TMR_CC1_CC1_Pos)                                  /*!< 0x0000FFFF */
#define TMR_CC1_CC1                                                 TMR_CC1_CC1_Msk                                                /*!<Capture/Compare 1 Value  */

/*******************  Bit definition for TMR_CC2 register  *******************/
#define TMR_CC2_CC2_Pos                                             (0U)
#define TMR_CC2_CC2_Msk                                             (0xFFFFUL << TMR_CC2_CC2_Pos)                                  /*!< 0x0000FFFF */
#define TMR_CC2_CC2                                                 TMR_CC2_CC2_Msk                                                /*!<Capture/Compare 2 Value  */

/*******************  Bit definition for TMR_CC3 register  *******************/
#define TMR_CC3_CC3_Pos                                             (0U)
#define TMR_CC3_CC3_Msk                                             (0xFFFFUL << TMR_CC3_CC3_Pos)                                  /*!< 0x0000FFFF */
#define TMR_CC3_CC3                                                 TMR_CC3_CC3_Msk                                                /*!<Capture/Compare 3 Value  */

/*******************  Bit definition for TMR_CC4 register  *******************/
#define TMR_CC4_CC4_Pos                                             (0U)
#define TMR_CC4_CC4_Msk                                             (0xFFFFUL << TMR_CC4_CC4_Pos)                                  /*!< 0x0000FFFF */
#define TMR_CC4_CC4                                                 TMR_CC4_CC4_Msk                                                /*!<Capture/Compare 4 Value  */

/*******************  Bit definition for TMR_BDT register  *******************/
#define TMR_BDT_DTS_Pos                                             (0U)
#define TMR_BDT_DTS_Msk                                             (0xFFUL << TMR_BDT_DTS_Pos)                                    /*!< 0x000000FF */
#define TMR_BDT_DTS                                                 TMR_BDT_DTS_Msk                                                /*!<DTG[0:7] bits (Dead-Time Generator set-up) */
#define TMR_BDT_DTS_0                                               (0x01UL << TMR_BDT_DTS_Pos)                                    /*!< 0x0001 */
#define TMR_BDT_DTS_1                                               (0x02UL << TMR_BDT_DTS_Pos)                                    /*!< 0x0002 */
#define TMR_BDT_DTS_2                                               (0x04UL << TMR_BDT_DTS_Pos)                                    /*!< 0x0004 */
#define TMR_BDT_DTS_3                                               (0x08UL << TMR_BDT_DTS_Pos)                                    /*!< 0x0008 */
#define TMR_BDT_DTS_4                                               (0x10UL << TMR_BDT_DTS_Pos)                                    /*!< 0x0010 */
#define TMR_BDT_DTS_5                                               (0x20UL << TMR_BDT_DTS_Pos)                                    /*!< 0x0020 */
#define TMR_BDT_DTS_6                                               (0x40UL << TMR_BDT_DTS_Pos)                                    /*!< 0x0040 */
#define TMR_BDT_DTS_7                                               (0x80UL << TMR_BDT_DTS_Pos)                                    /*!< 0x0080 */

#define TMR_BDT_LOCKCFG_Pos                                         (8U)
#define TMR_BDT_LOCKCFG_Msk                                         (0x3UL << TMR_BDT_LOCKCFG_Pos)                                 /*!< 0x00000300 */
#define TMR_BDT_LOCKCFG                                             TMR_BDT_LOCKCFG_Msk                                            /*!<LOCK[1:0] bits (Lock Configuration) */
#define TMR_BDT_LOCKCFG_0                                           (0x1UL << TMR_BDT_LOCKCFG_Pos)                                 /*!< 0x0100 */
#define TMR_BDT_LOCKCFG_1                                           (0x2UL << TMR_BDT_LOCKCFG_Pos)                                 /*!< 0x0200 */

#define TMR_BDT_IMOS_Pos                                            (10U)
#define TMR_BDT_IMOS_Msk                                            (0x1UL << TMR_BDT_IMOS_Pos)                                    /*!< 0x00000400 */
#define TMR_BDT_IMOS                                                TMR_BDT_IMOS_Msk                                               /*!<Off-State Selection for Idle mode */
#define TMR_BDT_RMOS_Pos                                            (11U)
#define TMR_BDT_RMOS_Msk                                            (0x1UL << TMR_BDT_RMOS_Pos)                                    /*!< 0x00000800 */
#define TMR_BDT_RMOS                                                TMR_BDT_RMOS_Msk                                               /*!<Off-State Selection for Run mode  */
#define TMR_BDT_BRKEN_Pos                                           (12U)
#define TMR_BDT_BRKEN_Msk                                           (0x1UL << TMR_BDT_BRKEN_Pos)                                   /*!< 0x00001000 */
#define TMR_BDT_BRKEN                                               TMR_BDT_BRKEN_Msk                                              /*!<Break enable                      */
#define TMR_BDT_BRKPOL_Pos                                          (13U)
#define TMR_BDT_BRKPOL_Msk                                          (0x1UL << TMR_BDT_BRKPOL_Pos)                                  /*!< 0x00002000 */
#define TMR_BDT_BRKPOL                                              TMR_BDT_BRKPOL_Msk                                             /*!<Break Polarity                    */
#define TMR_BDT_AOEN_Pos                                            (14U)
#define TMR_BDT_AOEN_Msk                                            (0x1UL << TMR_BDT_AOEN_Pos)                                    /*!< 0x00004000 */
#define TMR_BDT_AOEN                                                TMR_BDT_AOEN_Msk                                               /*!<Automatic Output enable           */
#define TMR_BDT_MOEN_Pos                                            (15U)
#define TMR_BDT_MOEN_Msk                                            (0x1UL << TMR_BDT_MOEN_Pos)                                    /*!< 0x00008000 */
#define TMR_BDT_MOEN                                                TMR_BDT_MOEN_Msk                                               /*!<Main Output enable                */

/*******************  Bit definition for TMR_DCTRL register  ********************/
#define TMR_DCTRL_DBADDR_Pos                                        (0U)
#define TMR_DCTRL_DBADDR_Msk                                        (0x1FUL << TMR_DCTRL_DBADDR_Pos)                               /*!< 0x0000001F */
#define TMR_DCTRL_DBADDR                                            TMR_DCTRL_DBADDR_Msk                                           /*!<DBA[4:0] bits (DMA Base Address) */
#define TMR_DCTRL_DBADDR_0                                          (0x01UL << TMR_DCTRL_DBADDR_Pos)                               /*!< 0x0001 */
#define TMR_DCTRL_DBADDR_1                                          (0x02UL << TMR_DCTRL_DBADDR_Pos)                               /*!< 0x0002 */
#define TMR_DCTRL_DBADDR_2                                          (0x04UL << TMR_DCTRL_DBADDR_Pos)                               /*!< 0x0004 */
#define TMR_DCTRL_DBADDR_3                                          (0x08UL << TMR_DCTRL_DBADDR_Pos)                               /*!< 0x0008 */
#define TMR_DCTRL_DBADDR_4                                          (0x10UL << TMR_DCTRL_DBADDR_Pos)                               /*!< 0x0010 */

#define TMR_DCTRL_DBLEN_Pos                                         (8U)
#define TMR_DCTRL_DBLEN_Msk                                         (0x1FUL << TMR_DCTRL_DBLEN_Pos)                                /*!< 0x00001F00 */
#define TMR_DCTRL_DBLEN                                             TMR_DCTRL_DBLEN_Msk                                            /*!<DBL[4:0] bits (DMA Burst Length) */
#define TMR_DCTRL_DBLEN_0                                           (0x01UL << TMR_DCTRL_DBLEN_Pos)                                /*!< 0x0100 */
#define TMR_DCTRL_DBLEN_1                                           (0x02UL << TMR_DCTRL_DBLEN_Pos)                                /*!< 0x0200 */
#define TMR_DCTRL_DBLEN_2                                           (0x04UL << TMR_DCTRL_DBLEN_Pos)                                /*!< 0x0400 */
#define TMR_DCTRL_DBLEN_3                                           (0x08UL << TMR_DCTRL_DBLEN_Pos)                                /*!< 0x0800 */
#define TMR_DCTRL_DBLEN_4                                           (0x10UL << TMR_DCTRL_DBLEN_Pos)                                /*!< 0x1000 */

/*******************  Bit definition for TMR_DMADDR register  *******************/
#define TMR_DMADDR_DMADDR_Pos                                       (0U)
#define TMR_DMADDR_DMADDR_Msk                                       (0xFFFFUL << TMR_DMADDR_DMADDR_Pos)                            /*!< 0x0000FFFF */
#define TMR_DMADDR_DMADDR                                           TMR_DMADDR_DMADDR_Msk                                          /*!<DMA register for burst accesses                    */

/*******************  Bit definition for TMR_OPT register  *********************/
#define TMR_OPT_RMPSEL_Pos                                           (10U)
#define TMR_OPT_RMPSEL_Msk                                           (0x1UL << TMR_OPT_RMPSEL_Pos)                                 /*!< 0x00000400 */
#define TMR_OPT_RMPSEL                                               TMR_OPT_RMPSEL_Msk                                            /*!<ITR1_RMP bits (TMR2 Internal trigger 1 remap) */

/******************************************************************************/
/*                                                                            */
/*         Universal Synchronous Asynchronous Receiver Transmitter            */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for USART_STS register  *******************/
#define USART_STS_PEFLG_Pos                                         (0U)
#define USART_STS_PEFLG_Msk                                         (0x1UL << USART_STS_PEFLG_Pos)                                 /*!< 0x00000001 */
#define USART_STS_PEFLG                                             USART_STS_PEFLG_Msk                                            /*!<Parity Error                 */
#define USART_STS_FEFLG_Pos                                         (1U)
#define USART_STS_FEFLG_Msk                                         (0x1UL << USART_STS_FEFLG_Pos)                                 /*!< 0x00000002 */
#define USART_STS_FEFLG                                             USART_STS_FEFLG_Msk                                            /*!<Framing Error                */
#define USART_STS_NEFLG_Pos                                         (2U)
#define USART_STS_NEFLG_Msk                                         (0x1UL << USART_STS_NEFLG_Pos)                                 /*!< 0x00000004 */
#define USART_STS_NEFLG                                             USART_STS_NEFLG_Msk                                            /*!<Noise Error Flag             */
#define USART_STS_OVREFLG_Pos                                       (3U)
#define USART_STS_OVREFLG_Msk                                       (0x1UL << USART_STS_OVREFLG_Pos)                               /*!< 0x00000008 */
#define USART_STS_OVREFLG                                           USART_STS_OVREFLG_Msk                                          /*!<OverRun Error                */
#define USART_STS_IDLEFLG_Pos                                       (4U)
#define USART_STS_IDLEFLG_Msk                                       (0x1UL << USART_STS_IDLEFLG_Pos)                               /*!< 0x00000010 */
#define USART_STS_IDLEFLG                                           USART_STS_IDLEFLG_Msk                                          /*!<IDLE line detected           */
#define USART_STS_RXBNEFLG_Pos                                      (5U)
#define USART_STS_RXBNEFLG_Msk                                      (0x1UL << USART_STS_RXBNEFLG_Pos)                              /*!< 0x00000020 */
#define USART_STS_RXBNEFLG                                          USART_STS_RXBNEFLG_Msk                                         /*!<Read Data Register Not Empty */
#define USART_STS_TXCFLG_Pos                                        (6U)
#define USART_STS_TXCFLG_Msk                                        (0x1UL << USART_STS_TXCFLG_Pos)                                /*!< 0x00000040 */
#define USART_STS_TXCFLG                                            USART_STS_TXCFLG_Msk                                           /*!<Transmission Complete        */
#define USART_STS_TXBEFLG_Pos                                       (7U)
#define USART_STS_TXBEFLG_Msk                                       (0x1UL << USART_STS_TXBEFLG_Pos)                               /*!< 0x00000080 */
#define USART_STS_TXBEFLG                                           USART_STS_TXBEFLG_Msk                                          /*!<Transmit Data Register Empty */
#define USART_STS_LBDFLG_Pos                                        (8U)
#define USART_STS_LBDFLG_Msk                                        (0x1UL << USART_STS_LBDFLG_Pos)                                /*!< 0x00000100 */
#define USART_STS_LBDFLG                                            USART_STS_LBDFLG_Msk                                           /*!<LIN Break Detection Flag     */
#define USART_STS_CTSFLG_Pos                                        (9U)
#define USART_STS_CTSFLG_Msk                                        (0x1UL << USART_STS_CTSFLG_Pos)                                /*!< 0x00000200 */
#define USART_STS_CTSFLG                                            USART_STS_CTSFLG_Msk                                           /*!<CTS Flag                     */

/*******************  Bit definition for USART_DATA register  *******************/
#define USART_DATA_DATA_Pos                                         (0U)
#define USART_DATA_DATA_Msk                                         (0x1FFUL << USART_DATA_DATA_Pos)                               /*!< 0x000001FF */
#define USART_DATA_DATA                                             USART_DATA_DATA_Msk                                            /*!<Data value */

/******************  Bit definition for USART_BR register  *******************/
#define USART_BR_FBR_Pos                                            (0U)
#define USART_BR_FBR_Msk                                            (0xFUL << USART_BR_FBR_Pos)                                    /*!< 0x0000000F */
#define USART_BR_FBR                                                USART_BR_FBR_Msk                                               /*!<Fraction of USARTDIV */
#define USART_BR_IBR_Pos                                            (4U)
#define USART_BR_IBR_Msk                                            (0xFFFUL << USART_BR_IBR_Pos)                                  /*!< 0x0000FFF0 */
#define USART_BR_IBR                                                USART_BR_IBR_Msk                                               /*!<Mantissa of USARTDIV */

/******************  Bit definition for USART_CTRL1 register  *******************/
#define USART_CTRL1_TXBF_Pos                                        (0U)
#define USART_CTRL1_TXBF_Msk                                        (0x1UL << USART_CTRL1_TXBF_Pos)                                /*!< 0x00000001 */
#define USART_CTRL1_TXBF                                            USART_CTRL1_TXBF_Msk                                           /*!<Send Break                             */
#define USART_CTRL1_RXMUTEEN_Pos                                    (1U)
#define USART_CTRL1_RXMUTEEN_Msk                                    (0x1UL << USART_CTRL1_RXMUTEEN_Pos)                            /*!< 0x00000002 */
#define USART_CTRL1_RXMUTEEN                                        USART_CTRL1_RXMUTEEN_Msk                                       /*!<Receiver wakeup                        */
#define USART_CTRL1_RXEN_Pos                                        (2U)
#define USART_CTRL1_RXEN_Msk                                        (0x1UL << USART_CTRL1_RXEN_Pos)                                /*!< 0x00000004 */
#define USART_CTRL1_RXEN                                            USART_CTRL1_RXEN_Msk                                           /*!<Receiver Enable                        */
#define USART_CTRL1_TXEN_Pos                                        (3U)
#define USART_CTRL1_TXEN_Msk                                        (0x1UL << USART_CTRL1_TXEN_Pos)                                /*!< 0x00000008 */
#define USART_CTRL1_TXEN                                            USART_CTRL1_TXEN_Msk                                           /*!<Transmitter Enable                     */
#define USART_CTRL1_IDLEIEN_Pos                                     (4U)
#define USART_CTRL1_IDLEIEN_Msk                                     (0x1UL << USART_CTRL1_IDLEIEN_Pos)                             /*!< 0x00000010 */
#define USART_CTRL1_IDLEIEN                                         USART_CTRL1_IDLEIEN_Msk                                        /*!<IDLE Interrupt Enable                  */
#define USART_CTRL1_RXBNEIEN_Pos                                    (5U)
#define USART_CTRL1_RXBNEIEN_Msk                                    (0x1UL << USART_CTRL1_RXBNEIEN_Pos)                            /*!< 0x00000020 */
#define USART_CTRL1_RXBNEIEN                                        USART_CTRL1_RXBNEIEN_Msk                                       /*!<RXNE Interrupt Enable                  */
#define USART_CTRL1_TXCIEN_Pos                                      (6U)
#define USART_CTRL1_TXCIEN_Msk                                      (0x1UL << USART_CTRL1_TXCIEN_Pos)                              /*!< 0x00000040 */
#define USART_CTRL1_TXCIEN                                          USART_CTRL1_TXCIEN_Msk                                         /*!<Transmission Complete Interrupt Enable */
#define USART_CTRL1_TXBEIEN_Pos                                     (7U)
#define USART_CTRL1_TXBEIEN_Msk                                     (0x1UL << USART_CTRL1_TXBEIEN_Pos)                             /*!< 0x00000080 */
#define USART_CTRL1_TXBEIEN                                         USART_CTRL1_TXBEIEN_Msk                                        /*!<TXE Interrupt Enable                   */
#define USART_CTRL1_PEIEN_Pos                                       (8U)
#define USART_CTRL1_PEIEN_Msk                                       (0x1UL << USART_CTRL1_PEIEN_Pos)                               /*!< 0x00000100 */
#define USART_CTRL1_PEIEN                                           USART_CTRL1_PEIEN_Msk                                          /*!<PE Interrupt Enable                    */
#define USART_CTRL1_PCFG_Pos                                        (9U)
#define USART_CTRL1_PCFG_Msk                                        (0x1UL << USART_CTRL1_PCFG_Pos)                                /*!< 0x00000200 */
#define USART_CTRL1_PCFG                                            USART_CTRL1_PCFG_Msk                                           /*!<Parity Selection                       */
#define USART_CTRL1_PCEN_Pos                                        (10U)
#define USART_CTRL1_PCEN_Msk                                        (0x1UL << USART_CTRL1_PCEN_Pos)                                /*!< 0x00000400 */
#define USART_CTRL1_PCEN                                            USART_CTRL1_PCEN_Msk                                           /*!<Parity Control Enable                  */
#define USART_CTRL1_WUPMCFG_Pos                                     (11U)
#define USART_CTRL1_WUPMCFG_Msk                                     (0x1UL << USART_CTRL1_WUPMCFG_Pos)                             /*!< 0x00000800 */
#define USART_CTRL1_WUPMCFG                                         USART_CTRL1_WUPMCFG_Msk                                        /*!<Wakeup method                          */
#define USART_CTRL1_DBLCFG_Pos                                      (12U)
#define USART_CTRL1_DBLCFG_Msk                                      (0x1UL << USART_CTRL1_DBLCFG_Pos)                              /*!< 0x00001000 */
#define USART_CTRL1_DBLCFG                                          USART_CTRL1_DBLCFG_Msk                                         /*!<Word length                            */
#define USART_CTRL1_UEN_Pos                                         (13U)
#define USART_CTRL1_UEN_Msk                                         (0x1UL << USART_CTRL1_UEN_Pos)                                 /*!< 0x00002000 */
#define USART_CTRL1_UEN                                             USART_CTRL1_UEN_Msk                                            /*!<USART Enable                           */

/******************  Bit definition for USART_CTRL2 register  *******************/
#define USART_CTRL2_ADDR_Pos                                        (0U)
#define USART_CTRL2_ADDR_Msk                                        (0xFUL << USART_CTRL2_ADDR_Pos)                                /*!< 0x0000000F */
#define USART_CTRL2_ADDR                                            USART_CTRL2_ADDR_Msk                                           /*!<Address of the USART node            */
#define USART_CTRL2_LBDLCFG_Pos                                     (5U)
#define USART_CTRL2_LBDLCFG_Msk                             