/*!
 * @file        bsp_flash.c
 *
 * @brief       Include flash operation
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "bsp_flash.h"
#include "usb_host_user.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/*!
 * @brief       Unlocks Flash for write access
 *
 * @param       None
 *
 * @retval      None
 *
 */
void FLASH_FlashInit(void)
{
    /* Unlock the flash memory */
    DAL_FLASH_Unlock();
}

/*!
 * @brief       Get readout protection status
 *
 * @param       None
 *
 * @retval      Readout protection status
 *
 */
uint8_t FLASH_ReadOutProtectionStatus(void)
{
    FLASH_OBProgramInitTypeDef OBProgram_InitStruct;
    uint8_t errSta = RESET;
    uint32_t obRDPlevel = 0;

    OBProgram_InitStruct.RDPLevel = obRDPlevel;

    DAL_FLASHEx_OBGetConfig(&OBProgram_InitStruct);

    if (obRDPlevel == SET)
    {
        errSta = SET;
    }

    return errSta;
}

/*!
 * @brief       Get the flash sector number of the address
 *
 * @param       None
 *
 * @retval      The flash sector number of the address
 *
 */
static uint32_t FLASH_GetSectorNumber(uint32_t address)
{
    uint32_t flashSector = FLASH_SECTOR_0;

    /* Flash sector less than 11 sector */
    if (address < FLASH_SECTOR_1_ADDR && address >= FLASH_SECTOR_0_ADDR)
    {
        flashSector = FLASH_SECTOR_0;
    }
    else if (address < FLASH_SECTOR_2_ADDR && address >= FLASH_SECTOR_1_ADDR)
    {
        flashSector = FLASH_SECTOR_1;
    }
    else if (address < FLASH_SECTOR_3_ADDR && address >= FLASH_SECTOR_2_ADDR)
    {
        flashSector = FLASH_SECTOR_2;
    }
    else if (address < FLASH_SECTOR_4_ADDR && address >= FLASH_SECTOR_3_ADDR)
    {
        flashSector = FLASH_SECTOR_3;
    }
    else if (address < FLASH_SECTOR_5_ADDR && address >= FLASH_SECTOR_4_ADDR)
    {
        flashSector = FLASH_SECTOR_4;
    }
    else if (address < FLASH_SECTOR_6_ADDR && address >= FLASH_SECTOR_5_ADDR)
    {
        flashSector = FLASH_SECTOR_5;
    }
    else if (address < FLASH_SECTOR_7_ADDR && address >= FLASH_SECTOR_6_ADDR)
    {
        flashSector = FLASH_SECTOR_6;
    }
    else if (address < FLASH_SECTOR_8_ADDR && address >= FLASH_SECTOR_7_ADDR)
    {
        flashSector = FLASH_SECTOR_7;
    }
    else if (address < FLASH_SECTOR_9_ADDR && address >= FLASH_SECTOR_8_ADDR)
    {
        flashSector = FLASH_SECTOR_8;
    }
    else if (address < FLASH_SECTOR_10_ADDR && address >= FLASH_SECTOR_9_ADDR)
    {
        flashSector = FLASH_SECTOR_9;
    }
    else if (address < FLASH_SECTOR_11_ADDR && address >= FLASH_SECTOR_10_ADDR)
    {
        flashSector = FLASH_SECTOR_10;
    }
    else/*(address < FLASH_END_ADDR && address >= ADDR_FLASH_SECTOR_11)*/
    {
        flashSector = FLASH_SECTOR_11;
    }

    return flashSector;
}

/*!
 * @brief       Erase of all user flash area
 *
 * @param       address: application address
 *
 * @retval      flash operation status
 *
 */
uint8_t FLASH_EraseSectors(uint32_t address)
{
    FMC_OP_T errSta = FMC_OP_OK;
    FLASH_EraseInitTypeDef Erase_InitStruct = {0};

    uint32_t startSector;
    uint32_t nbOfSector;
    uint32_t sectorError;

    if ( address <= (uint32_t) USER_FLASH_LAST_PAGE_ADDRESS)
    {
        startSector = FLASH_GetSectorNumber(address);
        nbOfSector = FLASH_GetSectorNumber(USER_FLASH_LAST_PAGE_ADDRESS) - startSector + 1;

        Erase_InitStruct.TypeErase = FLASH_TYPEERASE_SECTORS;
        Erase_InitStruct.Sector = startSector;
        Erase_InitStruct.NbSectors = nbOfSector;
        Erase_InitStruct.VoltageRange = FLASH_VOLTAGE_RANGE_3;
        if (DAL_FLASHEx_Erase(&Erase_InitStruct, &sectorError) != DAL_OK)
        {
            errSta = FMC_OP_ERROR;
        }
    }
    else
    {
        errSta = FMC_OP_ERROR;
    }

    return errSta;
}

/*!
 * @brief       Programs a word at a specified address
 *
 * @param       address: specifies the address to be programmed
 *
 * @param       data: specifies the data to be programmed
 *
 * @retval      flash status
 *
 */
uint8_t FLASH_ProgramWord(uint32_t address, uint32_t data)
{
    FMC_OP_T errSta = FMC_OP_OK;

    if ( address <= (uint32_t) USER_FLASH_LAST_PAGE_ADDRESS)
    {
        if (DAL_FLASH_Program(FLASH_TYPEPROGRAM_WORD, address, data) != DAL_OK)
        {
            errSta = FMC_OP_ERROR;
        }
    }
    else
    {
        errSta = FMC_OP_ERROR;
    }

    return errSta;
}
