/**
 * @file        usbd_board.c
 *
 * @brief       This file provides firmware functions to USB board
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "usbd_board.h"

/* Private includes *******************************************************/
#include "usbd_core.h"
#include "apm32f4xx_device_cfg.h"

/* Private macro **********************************************************/
#define USBD_FS2_RX_FIFO_SIZE                128
#define USBD_FS2_TX_FIFO_0_SIZE              64
#define USBD_FS2_TX_FIFO_1_SIZE              128

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
PCD_HandleTypeDef husbDevice;
/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* Private functions ******************************************************/

/**
 * @brief  Setup stage callback
 *
 * @param  hpcd: PCD handle
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_SetupStageCallback(PCD_HandleTypeDef *hpcd)
#else
void DAL_PCD_SetupStageCallback(PCD_HandleTypeDef *hpcd)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_SetupStage((USBD_INFO_T*)hpcd->pData, (uint8_t *)hpcd->Setup);
}

/**
 * @brief  Data Out stage callback
 *
 * @param  hpcd: PCD handle
 *
 * @param  epnum: Endpoint number
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_DataOutStageCallback(PCD_HandleTypeDef *hpcd, uint8_t epnum)
#else
void DAL_PCD_DataOutStageCallback(PCD_HandleTypeDef *hpcd, uint8_t epnum)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_DataOutStage((USBD_INFO_T*)hpcd->pData, epnum, hpcd->OUT_ep[epnum].xfer_buff);
}

/**
 * @brief  Data In stage callback
 *
 * @param  hpcd: PCD handle
 *
 * @param  epnum: Endpoint number
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_DataInStageCallback(PCD_HandleTypeDef *hpcd, uint8_t epnum)
#else
void DAL_PCD_DataInStageCallback(PCD_HandleTypeDef *hpcd, uint8_t epnum)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_DataInStage((USBD_INFO_T*)hpcd->pData, epnum, hpcd->IN_ep[epnum].xfer_buff);
}

/**
 * @brief  SOF callback
 *
 * @param  hpcd: PCD handle
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_SOFCallback(PCD_HandleTypeDef *hpcd)
#else
void DAL_PCD_SOFCallback(PCD_HandleTypeDef *hpcd)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_HandleSOF((USBD_INFO_T*)hpcd->pData);
}

/**
 * @brief  Reset callback
 *
 * @param  hpcd: PCD handle
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_ResetCallback(PCD_HandleTypeDef *hpcd)
#else
void DAL_PCD_ResetCallback(PCD_HandleTypeDef *hpcd)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_DEVICE_SPEED_T speed = USBD_DEVICE_SPEED_FS;

    if (hpcd->Init.speed == PCD_SPEED_HIGH)
    {
        speed = USBD_DEVICE_SPEED_HS;
    }
    else if (hpcd->Init.speed == PCD_SPEED_FULL)
    {
        speed = USBD_DEVICE_SPEED_FS;
    }
    else
    {
        Error_Handler();
    }

    /* Set USB core speed */
    USBD_SetSpeed((USBD_INFO_T*)hpcd->pData, speed);

    /* Reset Device. */
    USBD_Reset((USBD_INFO_T*)hpcd->pData);
}

/**
 * @brief  Suspend callback
 *
 * @param  hpcd: PCD handle
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_SuspendCallback(PCD_HandleTypeDef *hpcd)
#else
void DAL_PCD_SuspendCallback(PCD_HandleTypeDef *hpcd)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    /* USB core enters in suspend mode */
    USBD_Suspend((USBD_INFO_T*)hpcd->pData);

    if ((hpcd->Init.phy_itface == USB_OTG_EMBEDDED_PHY) && \
        (hpcd->Init.speed == PCD_SPEED_HIGH))
    {
        /* Embedded HS PHY can not stop clock */
    }
    else
    {
        __DAL_PCD_GATE_PHYCLOCK(hpcd);
    }

    /* Enter in STOP mode. */
    if (hpcd->Init.low_power_enable)
    {
        BOARD_LED_Off(LED3);
        /* Set SLEEPDEEP bit and SleepOnExit of Cortex System Control Register. */
        SCB->SCR |= (uint32_t)((uint32_t)(SCB_SCR_SLEEPDEEP_Msk | SCB_SCR_SLEEPONEXIT_Msk));
    }
}

/**
 * @brief  Resume callback
 *
 * @param  hpcd: PCD handle
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_ResumeCallback(PCD_HandleTypeDef *hpcd)
#else
void DAL_PCD_ResumeCallback(PCD_HandleTypeDef *hpcd)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_Resume((USBD_INFO_T*)hpcd->pData);
}

/**
 * @brief  ISOOUTIncomplete callback
 *
 * @param  hpcd: PCD handle
 *
 * @param  epnum: Endpoint number
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_ISOOUTIncompleteCallback(PCD_HandleTypeDef *hpcd, uint8_t epnum)
#else
void DAL_PCD_ISOOUTIncompleteCallback(PCD_HandleTypeDef *hpcd, uint8_t epnum)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_IsoOutInComplete((USBD_INFO_T*)hpcd->pData, epnum);
}

/**
 * @brief  ISOINIncomplete callback
 *
 * @param  hpcd: PCD handle
 *
 * @param  epnum: Endpoint number
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_ISOINIncompleteCallback(PCD_HandleTypeDef *hpcd, uint8_t epnum)
#else
void DAL_PCD_ISOINIncompleteCallback(PCD_HandleTypeDef *hpcd, uint8_t epnum)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_IsoInInComplete((USBD_INFO_T*)hpcd->pData, epnum);
}

/**
 * @brief  Connect callback
 *
 * @param  hpcd: PCD handle
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_ConnectCallback(PCD_HandleTypeDef *hpcd)
#else
void DAL_PCD_ConnectCallback(PCD_HandleTypeDef *hpcd)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_Connect((USBD_INFO_T*)hpcd->pData);
}

/**
 * @brief  Disconnect callback
 *
 * @param  hpcd: PCD handle
 *
 * @retval None
 */
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
static void PCD_DisconnectCallback(PCD_HandleTypeDef *hpcd)
#else
void DAL_PCD_DisconnectCallback(PCD_HandleTypeDef *hpcd)
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */
{
    USBD_Disconnect((USBD_INFO_T*)hpcd->pData);
}

/* External functions *****************************************************/

/**
 * @brief   Init USB hardware
 *
 * @param   usbInfo: USB core information
 *
 * @retval  None
 */
void USBD_HardwareInit(USBD_INFO_T* usbInfo)
{
    if (usbInfo->devSpeed == USBD_SPEED_FS2)
    {
        /* Link data */
        husbDevice.pData                    = usbInfo;
        usbInfo->dataPoint                  = &husbDevice;

        husbDevice.Instance                 = USB_OTG_FS2;
        husbDevice.Init.dev_endpoints       = 4;
        husbDevice.Init.speed               = PCD_SPEED_FULL;
        husbDevice.Init.dma_enable          = DISABLE;
        husbDevice.Init.phy_itface          = PCD_PHY_EMBEDDED;
        husbDevice.Init.Sof_enable          = DISABLE;
        husbDevice.Init.low_power_enable    = ENABLE;
        husbDevice.Init.lpm_enable          = DISABLE;
        husbDevice.Init.vbus_sensing_enable = DISABLE;
        husbDevice.Init.use_dedicated_ep1   = DISABLE;
        if (DAL_PCD_Init(&husbDevice) != DAL_OK)
        {
            Error_Handler();
        }
        
#if (USE_DAL_PCD_REGISTER_CALLBACKS == 1U)
        /* Register USB PCD CallBacks */
        DAL_PCD_RegisterCallback(&husbDevice, DAL_PCD_SOF_CB_ID, PCD_SOFCallback);
        DAL_PCD_RegisterCallback(&husbDevice, DAL_PCD_SETUPSTAGE_CB_ID, PCD_SetupStageCallback);
        DAL_PCD_RegisterCallback(&husbDevice, DAL_PCD_RESET_CB_ID, PCD_ResetCallback);
        DAL_PCD_RegisterCallback(&husbDevice, DAL_PCD_SUSPEND_CB_ID, PCD_SuspendCallback);
        DAL_PCD_RegisterCallback(&husbDevice, DAL_PCD_RESUME_CB_ID, PCD_ResumeCallback);
        DAL_PCD_RegisterCallback(&husbDevice, DAL_PCD_CONNECT_CB_ID, PCD_ConnectCallback);
        DAL_PCD_RegisterCallback(&husbDevice, DAL_PCD_DISCONNECT_CB_ID, PCD_DisconnectCallback);

        DAL_PCD_RegisterDataOutStageCallback(&husbDevice, PCD_DataOutStageCallback);
        DAL_PCD_RegisterDataInStageCallback(&husbDevice, PCD_DataInStageCallback);
        DAL_PCD_RegisterIsoOutIncpltCallback(&husbDevice, PCD_ISOOUTIncompleteCallback);
        DAL_PCD_RegisterIsoInIncpltCallback(&husbDevice, PCD_ISOINIncompleteCallback);
#endif /* USE_DAL_PCD_REGISTER_CALLBACKS */

        DAL_PCDEx_SetRxFiFo(&husbDevice, USBD_FS2_RX_FIFO_SIZE);
        DAL_PCDEx_SetTxFiFo(&husbDevice, 0, USBD_FS2_TX_FIFO_0_SIZE);
        DAL_PCDEx_SetTxFiFo(&husbDevice, 1, USBD_FS2_TX_FIFO_1_SIZE);

        /* Start USB device core */
        USBD_StartCallback(usbInfo);
    }
}

/**
 * @brief   Reset USB hardware
 *
 * @param   usbInfo:usb handler information
 *
 * @retval  None
 */
void USBD_HardwareReset(USBD_INFO_T* usbInfo)
{
    DAL_PCD_DeInit(usbInfo->dataPoint);
}

/**
 * @brief   USB device start event callback
 *
 * @param   usbInfo: USB core information
 *
 * @retval  None
 */
void USBD_StartCallback(USBD_INFO_T* usbInfo)
{
    DAL_PCD_Start(usbInfo->dataPoint);
}

/**
 * @brief   USB device stop event callback
 *
 * @param   usbInfo : USB core information
 *
 * @retval  None
 */
void USBD_StopCallback(USBD_INFO_T* usbInfo)
{
    DAL_PCD_Stop(usbInfo->dataPoint);
}

/**
 * @brief   USB device open EP callback
 *
 * @param   usbInfo : USB core information
 *
 * @param   epAddr: endpoint address
 *
 * @param   epType: endpoint type
 *
 * @param   epMps: endpoint maxinum of packet size
 *
 * @retval  None
 */
void USBD_EP_OpenCallback(USBD_INFO_T* usbInfo, uint8_t epAddr, \
                          uint8_t epType, uint16_t epMps)
{
    DAL_PCD_EP_Open(usbInfo->dataPoint, epAddr, epMps, epType);
}

/**
 * @brief   USB device close EP callback
 *
 * @param   usbInfo : USB core information
 *
 * @param   epAddr: endpoint address
 *
 * @retval  None
 */
void USBD_EP_CloseCallback(USBD_INFO_T* usbInfo, uint8_t epAddr)
{
    DAL_PCD_EP_Close(usbInfo->dataPoint, epAddr);
}

/**
 * @brief   USB device flush EP handler callback
 *
 * @param   usbInfo : USB core information
 *
 * @param   epAddr : endpoint address
 *
 * @retval  usb device status
 */
USBD_STA_T USBD_EP_FlushCallback(USBD_INFO_T* usbInfo, uint8_t epAddr)
{
    USBD_STA_T usbStatus = USBD_OK;

    DAL_PCD_EP_Flush(usbInfo->dataPoint, epAddr);

    return usbStatus;
}

/**
 * @brief   USB device set EP on stall status callback
 *
 * @param   usbInfo : USB core information
 *
 * @param   epAddr: endpoint address
 *
 * @retval  usb device status
 */
USBD_STA_T USBD_EP_StallCallback(USBD_INFO_T* usbInfo, uint8_t epAddr)
{
    USBD_STA_T usbStatus = USBD_OK;

    DAL_PCD_EP_SetStall(usbInfo->dataPoint, epAddr);

    return usbStatus;
}

/**
 * @brief   USB device clear EP stall status callback
 *
 * @param   usbInfo : USB core information
 *
 * @param   epAddr: endpoint address
 *
 * @retval  usb device status
 */
USBD_STA_T USBD_EP_ClearStallCallback(USBD_INFO_T* usbInfo, uint8_t epAddr)
{
    USBD_STA_T usbStatus = USBD_OK;

    DAL_PCD_EP_ClrStall(usbInfo->dataPoint, epAddr);

    return usbStatus;
}

/**
 * @brief   USB device read EP stall status callback
 *
 * @param   usbInfo : USB core information
 *
 * @param   epAddr: endpoint address
 *
 * @retval  Stall status
 */
uint8_t USBD_EP_ReadStallStatusCallback(USBD_INFO_T* usbInfo, uint8_t epAddr)
{
    return (DAL_PCD_EP_ReadStallStatus(usbInfo->dataPoint, epAddr));
}

/**
 * @brief   USB device set device address handler callback
 *
 * @param   usbInfo : USB core information
 *
 * @param   address : address
 *
 * @retval  usb device status
 */
USBD_STA_T USBD_SetDevAddressCallback(USBD_INFO_T* usbInfo, uint8_t address)
{
    USBD_STA_T usbStatus = USBD_OK;

    DAL_PCD_SetAddress(usbInfo->dataPoint, address);

    return usbStatus;
}

/**
 * @brief   USB device read EP last receive data size callback
 *
 * @param   usbInfo : USB core information
 *
 * @param   epAddr: endpoint address
 *
 * @retval  size of last receive data
 */
uint32_t USBD_EP_ReadRxDataLenCallback(USBD_INFO_T* usbInfo, uint8_t epAddr)
{
    return DAL_PCD_EP_GetRxCount(usbInfo->dataPoint, epAddr);
}

/**
 * @brief     USB device EP transfer handler callback
 *
 * @param     usbInfo : USB core information
 *
 * @param     epAddr : endpoint address
 *
 * @param     buffer : data buffer
 *
 * @param     length : length of data
 *
 * @retval    usb device status
 */
USBD_STA_T USBD_EP_TransferCallback(USBD_INFO_T* usbInfo, uint8_t epAddr, \
                                    uint8_t* buffer, uint32_t length)
{
    USBD_STA_T usbStatus = USBD_OK;

    DAL_PCD_EP_Transmit(usbInfo->dataPoint, epAddr, buffer, length);

    return usbStatus;
}

/**
 * @brief     USB device EP receive handler callback
 *
 * @param     usbInfo : USB core information
 *
 * @param     epAddr : endpoint address
 *
 * @param     buffer : data buffer
 *
 * @param     length : length of data
 *
 * @retval    usb device status
 */
USBD_STA_T USBD_EP_ReceiveCallback(USBD_INFO_T* usbInfo, uint8_t epAddr, \
                                   uint8_t* buffer, uint32_t length)
{
    USBD_STA_T usbStatus = USBD_OK;

    DAL_PCD_EP_Receive(usbInfo->dataPoint, epAddr, buffer, length);

    return usbStatus;
}

#ifdef USBD_HS_TESTMODE_ENABLE
/*!
 * @brief     USB device set test mode handler callback
 *
 * @param     usbInfo : usb handler information
 *
 * @param     testMode : test mode
 *
 * @retval    usb device status
 */
USBD_STA_T USBD_SetTestModeCallback(USBD_INFO_T* usbInfo, uint8_t testMode)
{
    USBD_STA_T usbStatus = USBD_OK;

    DAL_PCD_SetTestMode(usbInfo->dataPoint, testMode);

    return usbStatus;
}
#endif /* USBD_HS_TESTMODE_ENABLE */
