/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "usbd_hid.h"

#include <string.h>

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/**
 * @brief User application state
 */
typedef enum
{
    USER_APP_INIT,
    USER_APP_RUN,
} USER_APP_STATE_T;

/* Private variables ******************************************************/

/* Private function prototypes ********************************************/

/* External variables *****************************************************/
extern USBD_INFO_T gUsbDeviceFS;
extern PCD_HandleTypeDef husbDevice;
__IO uint8_t wakeupEnable = 0;
/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Infinite loop */
    while (1)
    {
        USB_DevUserApplication();
    }
}

/**
 * @brief       User application
 *
 * @param       None
 *
 * @retval      None
 */
void USB_DevUserApplication(void)
{
    static uint8_t userAppState = USER_APP_INIT;
    static uint8_t interval = 0;
    static int8_t report[4] = { 0 };

    switch (userAppState)
    {
        case USER_APP_INIT:
            interval = USBD_HID_ReadInterval(&gUsbDeviceFS);

            report[0] = 0;
            report[1] = 0;
            report[2] = 0;
            report[3] = 0;
            userAppState = USER_APP_RUN;
            break;

        case USER_APP_RUN:
            if (wakeupEnable)
            {
                wakeupEnable = 0;

                /* Activate the remote wakeup */
                DAL_PCD_ActivateRemoteWakeup(&husbDevice);

                DAL_Delay(10);

                /* Deactivate the remote wakeup */
                DAL_PCD_DeActivateRemoteWakeup(&husbDevice);

                /* Change state */
                ((USBD_INFO_T *)husbDevice.pData)->devState = ((USBD_INFO_T *)husbDevice.pData)->preDevState;

                /* Clear the remote wakeup status */
                ((USBD_INFO_T *)husbDevice.pData)->devRemoteWakeUpStatus = 0;
            }
            if (!BOARD_BUTTON_GetState(BUTTON_KEY3))
            {
                DAL_Delay(10);
                if (!BOARD_BUTTON_GetState(BUTTON_KEY3))
                {
                    report[1] = 5;
                    USBD_HID_TxReport(&gUsbDeviceFS, (uint8_t*)report, 4);
                    
                    while(!BOARD_BUTTON_GetState(BUTTON_KEY3));
                }
            }

            DAL_Delay(interval);
            break;
    }
}

/**
 * @brief  EINT line detection callbacks
 *
 * @param  GPIO_Pin Specifies the pins connected EINT line
 *
 * @retval None
 */
void DAL_GPIO_EINT_Callback(uint16_t GPIO_Pin)
{
    if (GPIO_Pin == KEY2_BUTTON_PIN)
    {
        if ((((USBD_INFO_T *)husbDevice.pData)->devRemoteWakeUpStatus == 1) && \
            (((USBD_INFO_T *)husbDevice.pData)->devState == USBD_DEV_SUSPEND))
        {
            if ((&husbDevice)->Init.low_power_enable == ENABLE)
            {
                /* Reset SLEEPDEEP bit and SLEEPONEXIT SCR */
                SCB->SCR &= (uint32_t)~((uint32_t)(SCB_SCR_SLEEPDEEP_Msk | SCB_SCR_SLEEPONEXIT_Msk));
                DAL_SysClkConfig_STOP();

                /* Ungate PHY clock */
                __DAL_PCD_UNGATE_PHYCLOCK(&husbDevice);
            }

            wakeupEnable = 1;
        }
    }
}
