/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";

__IO uint32_t captureIndex = 0U;
uint32_t captureVal[2] = {0};
uint32_t captureFreq = 0U;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/
extern TMR_HandleTypeDef htmr5;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Start TMR5 channel 1 in input capture mode */
    if (DAL_TMR_IC_Start_IT(&htmr5, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Infinite loop */
    while (1)
    {
        LOGI(tag, "Frequency: %d Hz\r\n", captureFreq);
        DAL_Delay(500U);
    }
}

/**
 * @brief  Input Capture callback in non-blocking mode
 *
 * @param  htmr TMR IC handle
 *
 * @retval None
 */
void DAL_TMR_IC_CaptureCallback(TMR_HandleTypeDef *htmr)
{
    uint32_t tempValue = 0U;

    if (htmr->Channel == DAL_TMR_ACTIVE_CHANNEL_1)
    {
        captureVal[captureIndex] = DAL_TMR_ReadCapturedValue(htmr, TMR_CHANNEL_1);
        
        if (captureIndex != 0U)
        {
            captureIndex = 0U;
            
            /* Capture computation */
            if (captureVal[1] > captureVal[0])
            {
                tempValue = captureVal[1] - captureVal[0];
            }
            else if (captureVal[1] < captureVal[0])
            {
                tempValue = 0xFFFFFFFFU - captureVal[0] + captureVal[1];
            }
            else
            {
                tempValue = 0U;
            }

            /* Frequency computation */
            captureFreq = (DAL_RCM_GetPCLK1Freq() * 2U) / tempValue;
        }
        else
        {
            captureIndex++;
        }
    }
}
