/**
 * @file        apm32f4xx_qspi_cfg.c
 *
 * @brief       This file provides configuration support for QSPI
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_qspi_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
QSPI_HandleTypeDef hqspi;
DMA_HandleTypeDef hdma_qspiTx;
DMA_HandleTypeDef hdma_qspiRx;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the QSPI MSP
 * 
 * @param  hqspi: QSPI handle
 * 
 * @retval None
 */
void DAL_QSPI_MspInit(QSPI_HandleTypeDef *hqspi)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0U};

    if (hqspi->Instance == QSPI)
    {
        /* Enable GPIO clock */
        __DAL_RCM_GPIOB_CLK_ENABLE();
        __DAL_RCM_GPIOC_CLK_ENABLE();
        __DAL_RCM_GPIOE_CLK_ENABLE();

        /* Enable peripheral clock */
        __DAL_RCM_QSPI_CLK_ENABLE();

        /* Enable DMA clock */
        __DAL_RCM_DMA2_CLK_ENABLE();

        /* QSPI GPIO Configuration */
        GPIO_InitStruct.Pin       = QSPI_IO0_PIN;
        GPIO_InitStruct.Mode      = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull      = GPIO_NOPULL;
        GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_VERY_HIGH;
        GPIO_InitStruct.Alternate = QSPI_IO0_AF;
        DAL_GPIO_Init(QSPI_IO0_GPIO_PORT, &GPIO_InitStruct);

        GPIO_InitStruct.Pin       = QSPI_IO1_PIN;
        GPIO_InitStruct.Mode      = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull      = GPIO_NOPULL;
        GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_VERY_HIGH;
        GPIO_InitStruct.Alternate = QSPI_IO1_AF;
        DAL_GPIO_Init(QSPI_IO1_GPIO_PORT, &GPIO_InitStruct);

        GPIO_InitStruct.Pin       = QSPI_IO2_PIN;
        GPIO_InitStruct.Mode      = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull      = GPIO_NOPULL;
        GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_VERY_HIGH;
        GPIO_InitStruct.Alternate = QSPI_IO2_AF;
        DAL_GPIO_Init(QSPI_IO2_GPIO_PORT, &GPIO_InitStruct);

        GPIO_InitStruct.Pin       = QSPI_IO3_PIN;
        GPIO_InitStruct.Mode      = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull      = GPIO_NOPULL;
        GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_VERY_HIGH;
        GPIO_InitStruct.Alternate = QSPI_IO3_AF;
        DAL_GPIO_Init(QSPI_IO3_GPIO_PORT, &GPIO_InitStruct);

        GPIO_InitStruct.Pin       = QSPI_CLK_PIN;
        GPIO_InitStruct.Mode      = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull      = GPIO_NOPULL;
        GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_VERY_HIGH;
        GPIO_InitStruct.Alternate = QSPI_CLK_AF;
        DAL_GPIO_Init(QSPI_CLK_GPIO_PORT, &GPIO_InitStruct);

        GPIO_InitStruct.Pin       = QSPI_CS_PIN;
        GPIO_InitStruct.Mode      = GPIO_MODE_OUTPUT_PP;
        GPIO_InitStruct.Pull      = GPIO_NOPULL;
        GPIO_InitStruct.Speed     = GPIO_SPEED_FREQ_VERY_HIGH;
        DAL_GPIO_Init(QSPI_CS_GPIO_PORT, &GPIO_InitStruct);

        /* Disable QSPI CS */
        DAL_GPIO_WritePin(QSPI_CS_GPIO_PORT, QSPI_CS_PIN, GPIO_PIN_SET);

        /* Configure QSPI TX DMA parameters */
        hdma_qspiTx.Instance                 = DMA2_Stream7;
        hdma_qspiTx.Init.Channel             = DMA_CHANNEL_3;
        hdma_qspiTx.Init.Direction           = DMA_MEMORY_TO_PERIPH;
        hdma_qspiTx.Init.PeriphInc           = DMA_PINC_DISABLE;
        hdma_qspiTx.Init.MemInc              = DMA_MINC_ENABLE;
        hdma_qspiTx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
        hdma_qspiTx.Init.MemDataAlignment    = DMA_MDATAALIGN_BYTE;
        hdma_qspiTx.Init.Mode                = DMA_NORMAL;
        hdma_qspiTx.Init.Priority            = DMA_PRIORITY_LOW;
        hdma_qspiTx.Init.FIFOMode            = DMA_FIFOMODE_DISABLE;
        hdma_qspiTx.Init.FIFOThreshold       = DMA_FIFO_THRESHOLD_FULL;
        hdma_qspiTx.Init.PeriphBurst         = DMA_PBURST_INC4;
        hdma_qspiTx.Init.MemBurst            = DMA_MBURST_INC4;
        if (DAL_DMA_Init(&hdma_qspiTx) != DAL_OK)
        {
            Error_Handler();
        }

        /* Link DMA handle */
        __DAL_LINKDMA(hqspi, hdmatx, hdma_qspiTx);

        /* Configure QSPI RX DMA parameters */
        hdma_qspiRx.Instance                 = DMA2_Stream4;
        hdma_qspiRx.Init.Channel             = DMA_CHANNEL_3;
        hdma_qspiRx.Init.Direction           = DMA_PERIPH_TO_MEMORY;
        hdma_qspiRx.Init.PeriphInc           = DMA_PINC_DISABLE;
        hdma_qspiRx.Init.MemInc              = DMA_MINC_ENABLE;
        hdma_qspiRx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
        hdma_qspiRx.Init.MemDataAlignment    = DMA_MDATAALIGN_BYTE;
        hdma_qspiRx.Init.Mode                = DMA_NORMAL;
        hdma_qspiRx.Init.Priority            = DMA_PRIORITY_LOW;
        hdma_qspiRx.Init.FIFOMode            = DMA_FIFOMODE_DISABLE;
        hdma_qspiRx.Init.FIFOThreshold       = DMA_FIFO_THRESHOLD_FULL;
        hdma_qspiRx.Init.PeriphBurst         = DMA_PBURST_INC4;
        hdma_qspiRx.Init.MemBurst            = DMA_MBURST_INC4;
        if (DAL_DMA_Init(&hdma_qspiRx) != DAL_OK)
        {
            Error_Handler();
        }

        /* Link DMA handle */
        __DAL_LINKDMA(hqspi, hdmarx, hdma_qspiRx);

        /* Disable QSPI interrupts */
        __DAL_QSPI_DISABLE_IT(hqspi, QSPI_IT_TFE \
                                     | QSPI_IT_TFO \
                                     | QSPI_IT_RFU \
                                     | QSPI_IT_RFO \
                                     | QSPI_IT_RFF);

        /* QSPI interrupt Init */
        DAL_NVIC_SetPriority(QSPI_IRQn, 1U, 0U);
        DAL_NVIC_EnableIRQ(QSPI_IRQn);

        /* Enable QSPI Tx DMA interrupt */
        DAL_NVIC_SetPriority(DMA2_Stream7_IRQn, 2U, 0U);
        DAL_NVIC_EnableIRQ(DMA2_Stream7_IRQn);

        /* Enable QSPI Rx DMA interrupt */
        DAL_NVIC_SetPriority(DMA2_Stream4_IRQn, 3U, 0U);
        DAL_NVIC_EnableIRQ(DMA2_Stream4_IRQn);
    }
}

/**
 * @brief  DeInitializes the QSPI MSP
 * 
 * @param  hqspi: QSPI handle
 * 
 * @retval None
 */
void DAL_QSPI_MspDeInit(QSPI_HandleTypeDef *hqspi)
{
    if (hqspi->Instance == QSPI)
    {
        /* Disable peripheral clock */
        __DAL_RCM_QSPI_FORCE_RESET();
        __DAL_RCM_QSPI_RELEASE_RESET();

        /* QSPI GPIO Configuration */
        DAL_GPIO_DeInit(QSPI_IO0_GPIO_PORT, QSPI_IO0_PIN);
        DAL_GPIO_DeInit(QSPI_IO1_GPIO_PORT, QSPI_IO1_PIN);
        DAL_GPIO_DeInit(QSPI_IO2_GPIO_PORT, QSPI_IO2_PIN);
        DAL_GPIO_DeInit(QSPI_IO3_GPIO_PORT, QSPI_IO3_PIN);
        DAL_GPIO_DeInit(QSPI_CLK_GPIO_PORT, QSPI_CLK_PIN);
        DAL_GPIO_DeInit(QSPI_CS_GPIO_PORT, QSPI_CS_PIN);

        /* DeInitialize QSPI DMA */
        DAL_DMA_DeInit(&hdma_qspiTx);
        DAL_DMA_DeInit(&hdma_qspiRx);

        /* Disable QSPI interrupt */
        DAL_NVIC_DisableIRQ(QSPI_IRQn);
        DAL_NVIC_DisableIRQ(DMA2_Stream7_IRQn);
        DAL_NVIC_DisableIRQ(DMA2_Stream4_IRQn);
    }
}

/**
 * @brief   QSPI configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_QSPI_Config(void)
{
    hqspi.Instance                  = QSPI;
    hqspi.Init.ClockPrescaler       = 12U;
    hqspi.Init.ClockStretch         = ENABLE;
    hqspi.Init.TxFifoThreshold      = 0U;
    hqspi.Init.TxFifoLevel          = 0U;
    hqspi.Init.RxFifoThreshold      = 1U;
    hqspi.Init.ChipSelectToggle     = QSPI_CS_TOGGLE_DISABLE;
    if (DAL_QSPI_Init(&hqspi) != DAL_OK)
    {
        Error_Handler();
    }
}
