/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "flash.h"
#include <string.h>
#include <stdbool.h>

/* Private macro **********************************************************/
#define BUFFER_SIZE         256U

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
uint8_t txBuffer[BUFFER_SIZE] = {0x00U};
uint8_t rxBuffer[BUFFER_SIZE] = {0x00U};

/* Private function prototypes ********************************************/
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size);

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    uint32_t i = 0;

    /* Device configuration */
    DAL_DeviceConfig();

    /* Init flash */
    FLASH_Init();

    /* Prepare data */
    for (i = 0; i < BUFFER_SIZE; i++)
    {
        if ((i % 2) == 0)
        {
            txBuffer[i] = 0xAA;
        }
        else
        {
            txBuffer[i] = 0x55;
        }
    }

    /* Erase sector */
    FLASH_EraseSector(0x1000U);

    /* Write data */
    FLASH_WriteData(0x1000U, txBuffer, BUFFER_SIZE);

    /* Read data */
    FLASH_ReadData(0x1000U, rxBuffer, BUFFER_SIZE);

    /* Compare data */
    if (BufferCmp((uint8_t*)txBuffer, (uint8_t*)rxBuffer, BUFFER_SIZE) != true)
    {
        BOARD_LED_On(LED3);
        Error_Handler();
    }

    /* Infinite loop */
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500U);
    }
}

/* Private functions ******************************************************/

/**
 * @brief   Buffer compare
 *
 * @param   buf1: First buffer to be compared
 *
 * @param   buf2: Second buffer to be compared
 *
 * @param   size: Buffer size
 *
 * @retval  Comparison result(ture or false)
 */
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size)
{
    uint32_t i;

    bool status = true;

    for (i = 0U; i < size; i++)
    {
        if (buf1[i] != buf2[i])
        {
            status = false;
            break;
        }
    }

    return status;
}
