/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";
__IO uint16_t adc1ConvValue = 0U;
__IO uint16_t dacValue = 0U;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/
extern DAC_HandleTypeDef hdac;
extern ADC_HandleTypeDef hadc1;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Start DAC channel 1 output */
    DAL_DAC_Start(&hdac, DAC_CHANNEL_1);

    /* Strat ADC conversation */
    if (DAL_ADC_Start_IT(&hadc1) != DAL_OK)
    {
        Error_Handler();
    }
    
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500U);
    }
}

/**
 * @brief  Regular conversion complete callback in non blocking mode
 *
 * @param  hadc pointer to a ADC_HandleTypeDef structure that contains
 *         the configuration information for the specified ADC
 *
 * @retval None
 */
void DAL_ADC_ConvCpltCallback(ADC_HandleTypeDef* hadc)
{
    uint16_t voltage;

    /* Generate analog signal */
    DAL_DAC_SetValue(&hdac, DAC_CHANNEL_1, DAC_ALIGN_12B_R, dacValue);

    if (dacValue >= 0x0FFFU)
    {
        dacValue = 0U;
    }

    dacValue += 0x7F;

    /* Get the converted value */
    adc1ConvValue = DAL_ADC_GetValue(hadc);

    voltage = (adc1ConvValue * 3300U) / 4095U;

    LOGI(tag, "DAC CH1 Value: %d mV\r\n",voltage);

    /* Strat ADC Conversation */
    if (DAL_ADC_Start_IT(&hadc1) != DAL_OK)
    {
        Error_Handler();
    }
}
