/**
 * @file        apm32f4xx_sd_cfg.c
 *
 * @brief       This file provides configuration support for SD card
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_sd_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
SD_HandleTypeDef hsd1;
DMA_HandleTypeDef hdma_sdio_tx;
DMA_HandleTypeDef hdma_sdio_rx;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief   Initializes the SD MSP
 *
 * @param   hsd: Pointer to SD handle
 *
 * @retval  None
 */
void DAL_SD_MspInit(SD_HandleTypeDef *hsd)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (hsd->Instance == SDIO)
    {
        /* Enable peripheral clock */
        __DAL_RCM_SDIO_CLK_ENABLE();

        /* Enable DMA clock */
        __DAL_RCM_DMA2_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOC_CLK_ENABLE();
        __DAL_RCM_GPIOD_CLK_ENABLE();

        /* SDIO pin configuration 
           SDIO D0 ----> PC8
           SDIO D1 ----> PC9
           SDIO D2 ----> PC10
           SDIO D3 ----> PC11
           SDIO CK ----> PC12
           SDIO CMD ---> PD2
        */
        GPIO_InitStruct.Pin         = GPIO_PIN_8 | GPIO_PIN_9 | GPIO_PIN_10 | \
                                      GPIO_PIN_11 | GPIO_PIN_12;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF11_SDIO;
        DAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_2;
        DAL_GPIO_Init(GPIOD, &GPIO_InitStruct);

        /* Configure SDIO TX DMA */
        hdma_sdio_tx.Instance                   = DMA2_Stream6;
        hdma_sdio_tx.Init.Channel               = DMA_CHANNEL_4;
        hdma_sdio_tx.Init.Direction             = DMA_MEMORY_TO_PERIPH;
        hdma_sdio_tx.Init.PeriphInc             = DMA_PINC_DISABLE;
        hdma_sdio_tx.Init.MemInc                = DMA_MINC_ENABLE;
        hdma_sdio_tx.Init.PeriphDataAlignment   = DMA_PDATAALIGN_WORD;
        hdma_sdio_tx.Init.MemDataAlignment      = DMA_MDATAALIGN_WORD;
        hdma_sdio_tx.Init.Mode                  = DMA_PFCTRL;
        hdma_sdio_tx.Init.Priority              = DMA_PRIORITY_VERY_HIGH;
        hdma_sdio_tx.Init.FIFOMode              = DMA_FIFOMODE_ENABLE;
        hdma_sdio_tx.Init.FIFOThreshold         = DMA_FIFO_THRESHOLD_FULL;
        hdma_sdio_tx.Init.MemBurst              = DMA_MBURST_INC4;
        hdma_sdio_tx.Init.PeriphBurst           = DMA_PBURST_INC4;
        if (DAL_DMA_Init(&hdma_sdio_tx) != DAL_OK)
        {
            Error_Handler();
        }

        __DAL_LINKDMA(hsd,hdmatx,hdma_sdio_tx);
        
        /* Configure SDIO RX DMA */
        hdma_sdio_rx.Instance                   = DMA2_Stream3;
        hdma_sdio_rx.Init.Channel               = DMA_CHANNEL_4;
        hdma_sdio_rx.Init.Direction             = DMA_PERIPH_TO_MEMORY;
        hdma_sdio_rx.Init.PeriphInc             = DMA_PINC_DISABLE;
        hdma_sdio_rx.Init.MemInc                = DMA_MINC_ENABLE;
        hdma_sdio_rx.Init.PeriphDataAlignment   = DMA_PDATAALIGN_WORD;
        hdma_sdio_rx.Init.MemDataAlignment      = DMA_MDATAALIGN_WORD;
        hdma_sdio_rx.Init.Mode                  = DMA_PFCTRL;
        hdma_sdio_rx.Init.Priority              = DMA_PRIORITY_VERY_HIGH;
        hdma_sdio_rx.Init.FIFOMode              = DMA_FIFOMODE_ENABLE;
        hdma_sdio_rx.Init.FIFOThreshold         = DMA_FIFO_THRESHOLD_FULL;
        hdma_sdio_rx.Init.MemBurst              = DMA_MBURST_INC4;
        hdma_sdio_rx.Init.PeriphBurst           = DMA_PBURST_INC4;
        if (DAL_DMA_Init(&hdma_sdio_rx) != DAL_OK)
        {
            Error_Handler();
        }

        __DAL_LINKDMA(hsd,hdmarx,hdma_sdio_rx);
    }
}

/**
 * @brief   De-Initialize SD MSP
 *
 * @param   hsd: Pointer to SD handle
 *
 * @retval  None
 */
void DAL_SD_MspDeInit(SD_HandleTypeDef *hsd)
{
    if (hsd->Instance == SDIO)
    {
        /* Disable peripheral clock */
        __DAL_RCM_SDIO_CLK_DISABLE();

        /* SDIO GPIO configuration */
        DAL_GPIO_DeInit(GPIOC, GPIO_PIN_8 | GPIO_PIN_9 | GPIO_PIN_10 | \
                               GPIO_PIN_11 | GPIO_PIN_12);
        DAL_GPIO_DeInit(GPIOD, GPIO_PIN_2);

        /* DeInit SDIO DMA */
        DAL_DMA_DeInit(hsd->hdmatx);
        DAL_DMA_DeInit(hsd->hdmarx);

        /* Disable peripheral interrupt */
        DAL_NVIC_DisableIRQ(SDIO_IRQn);
        DAL_NVIC_DisableIRQ(DMA2_Stream3_IRQn);
        DAL_NVIC_DisableIRQ(DMA2_Stream6_IRQn);
    }
}

/**
 * @brief   SD configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_SD_Config(void)
{
    /* Configure the the SDIO */
    hsd1.Instance                    = SDIO;
    hsd1.Init.ClockDiv               = SDIO_TRANSFER_CLK_DIV;
    hsd1.Init.ClockEdge              = SDIO_CLOCK_EDGE_RISING;
    hsd1.Init.ClockBypass            = SDIO_CLOCK_BYPASS_DISABLE;
    hsd1.Init.ClockPowerSave         = SDIO_CLOCK_POWER_SAVE_DISABLE;
    hsd1.Init.BusWide                = SDIO_BUS_WIDE_1B;
    hsd1.Init.HardwareFlowControl    = SDIO_HARDWARE_FLOW_CONTROL_DISABLE;
}
