/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr3;
TMR_HandleTypeDef htmr4;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR Base MSP
 *
 * @param  htmr TMR Base handle
 *
 * @retval None
 */
void DAL_TMR_Base_MspInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR3)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR3_CLK_ENABLE();
    }

    if (htmr->Instance == TMR4)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR4_CLK_ENABLE();
    }
}

/**
 * @brief  DeInitializes TMR Base MSP
 *
 * @param  htmr TMR Base handle
 *
 * @retval None
 */
void DAL_TMR_Base_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR3)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR3_CLK_DISABLE();

        /* Disable peripheral interrupt */
        DAL_NVIC_DisableIRQ(TMR3_IRQn);
    }

    if (htmr->Instance == TMR4)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR4_CLK_DISABLE();

        /* Disable peripheral interrupt */
        DAL_NVIC_DisableIRQ(TMR4_IRQn);
    }
}

/**
 * @brief   TMR3 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR3_Config(void)
{
    TMR_SlaveConfigTypeDef Slave_ConfigStruct = {0};

    htmr3.Instance                  = TMR3;
    htmr3.Init.Prescaler            = 0U;
    htmr3.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr3.Init.Period               = 0xFFFF;
    htmr3.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr3.Init.RepetitionCounter    = 0U;
    htmr3.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_ENABLE;
    if (DAL_TMR_Base_Init(&htmr3) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure slave TMR */
    Slave_ConfigStruct.SlaveMode     = TMR_SLAVEMODE_EXTERNAL1;
    Slave_ConfigStruct.InputTrigger  = TMR_TS_ITR3;
    if (DAL_TMR_SlaveConfigSynchro(&htmr3, &Slave_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}

/**
 * @brief   TMR4 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR4_Config(void)
{
    TMR_MasterConfigTypeDef Master_ConfigStruct = {0};

    htmr4.Instance                  = TMR4;
    htmr4.Init.Prescaler            = 167U;
    htmr4.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr4.Init.Period               = 0xFFFFU;
    htmr4.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr4.Init.RepetitionCounter    = 0U;
    htmr4.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_ENABLE;
    if (DAL_TMR_Base_Init(&htmr4) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure master TMR */
    Master_ConfigStruct.MasterOutputTrigger = TMR_TRGO_UPDATE;
    Master_ConfigStruct.MasterSlaveMode     = TMR_MASTERSLAVEMODE_ENABLE;
    if (DAL_TMREx_MasterConfigSynchronization(&htmr4,&Master_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}
