/**
 * @file        apm32f4xx_timebase_cfg.c
 *
 * @brief       This file provides configuration support for timebase
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_dal.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr6;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief This function configures the source of the time base.
 *        The time source is configured  to have 1ms time base with a dedicated
 *        Tick interrupt priority
 *
 * @param TickPriority Tick interrupt priority
 *
 * @retval DAL status
 */
DAL_StatusTypeDef DAL_InitTick(uint32_t TickPriority)
{
    RCM_ClkInitTypeDef RCM_ClkInitStruct = {0};
    uint32_t tmrClock;
    uint32_t APB1Prescaler = 0U;
    uint32_t prescalerValue = 0U;
    uint32_t flashLatency;
    DAL_StatusTypeDef status = DAL_OK;

    /* Enable TMR6 clock */
    __DAL_RCM_TMR6_CLK_ENABLE();

    /* Get clock configuration */
    DAL_RCM_GetClockConfig(&RCM_ClkInitStruct, &flashLatency);

    /* Get APB1 prescaler */
    APB1Prescaler = RCM_ClkInitStruct.APB1CLKDivider;

    /* Compute TMR6 clock */
    if (APB1Prescaler == RCM_HCLK_DIV1)
    {
        tmrClock = DAL_RCM_GetPCLK1Freq();
    }
    else
    {
        tmrClock = 2 * DAL_RCM_GetPCLK1Freq();
    }

    /* Compute the prescaler value to have TMR6 counter clock equal to 1MHz */
    prescalerValue = (uint32_t) ((tmrClock / 1000000U) - 1U);

    /* Initialize TMR6 */
    htmr6.Instance = TMR6;

    /* Initialize TMR6 peripheral as follow:
    Period = [(TMR6CLK/1000) - 1]. to have a (1/1000) s time base
    Prescaler = (tmrClock / 1000000 - 1) to have a 1MHz counter clock
    */
    htmr6.Init.Period               = (1000000U / 1000U) - 1U;
    htmr6.Init.Prescaler            = prescalerValue;
    htmr6.Init.ClockDivision        = 0U;
    htmr6.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr6.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    status = DAL_TMR_Base_Init(&htmr6);
    if (status == DAL_OK)
    {
        /* Start the TMR time Base generation in interrupt mode */
        status = DAL_TMR_Base_Start_IT(&htmr6);
        if (status == DAL_OK)
        {
            /* Enable the TMR6 global Interrupt */
            DAL_NVIC_EnableIRQ(TMR6_DAC_IRQn);

            if (TickPriority < (1UL << __NVIC_PRIO_BITS))
            {
            /* Enable the TMR6 global Interrupt */
            DAL_NVIC_SetPriority(TMR6_DAC_IRQn, TickPriority, 0);
            uwTickPrio = TickPriority;
            }
            else
            {
            status = DAL_ERROR;
            }
        }
    }

    return status;
}

/**
 * @brief  Suspend Tick increment
 *
 * @note   Disable the tick increment by disabling TMR update interrupt
 *
 * @retval None
 */
void DAL_SuspendTick(void)
{
    /* Disable TMR6 update Interrupt */
    __DAL_TMR_DISABLE_IT(&htmr6, TMR_IT_UPDATE);
}

/**
  * @brief  Resume Tick increment.
  * @note   Enable the tick increment by Enabling TMR update interrupt.
  * @retval None
  */
void DAL_ResumeTick(void)
{
    /* Enable TMR6 Update interrupt */
    __DAL_TMR_ENABLE_IT(&htmr6, TMR_IT_UPDATE);
}

/**
  * @brief  Period elapsed callback in non blocking mode
  * @note   This function is called  when TMR interrupt took place, inside
  * DAL_TMR_IRQHandler(). It makes a direct call to DAL_IncTick() to increment
  * a global variable "uwTick" used as application time base.
  * @param  htmr  TMR handle
  * @retval None
  */
void DAL_TMR_PeriodElapsedCallback(TMR_HandleTypeDef *htmr)
{
    DAL_IncTick();
}

/**
 * @brief  This function handles TMR interrupt request.
 * @retval None
 */
void TMR6_DAC_IRQHandler(void)
{
    DAL_TMR_IRQHandler(&htmr6);
}
