/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"
#include <string.h>
#include <stdbool.h>

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";
uint16_t txBuffer[BUFFERSIZE] =
{
    0x0102U, 0x0304U, 0x0506U, 0x0708U, 0x090AU, 0x0B0CU,
    0x0D0EU, 0x0F10U, 0x1112U, 0x1314U, 0x1516U, 0x1718U,
    0x191AU, 0x1B1CU, 0x1D1EU, 0x1F20U, 0x2122U, 0x2324U,
    0x2526U, 0x2728U, 0x292AU, 0x2B2CU, 0x2D2EU, 0x2F30U,
    0x3132U, 0x3334U, 0x3536U, 0x3738U, 0x393AU, 0x3B3CU,
    0x3D3EU, 0x3F40U, 0x4142U, 0x4344U, 0x4546U, 0x4748U,
};

uint16_t rxBuffer[BUFFERSIZE];
__IO uint32_t txCnt = 0U;
__IO uint32_t rxCnt = 0U;

/* Private function prototypes ********************************************/
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size);

/* External variables *****************************************************/
extern I2S_HandleTypeDef hi2s2;
extern I2S_HandleTypeDef hi2s3;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Enable TXE and ERR interrupt */
    __DAL_I2S_ENABLE_IT(&hi2s3, (I2S_IT_TXE | I2S_IT_ERR));
    __DAL_I2S_ENABLE_IT(&hi2s2, (I2S_IT_RXNE | I2S_IT_ERR));

    __DAL_I2S_ENABLE(&hi2s2);
    __DAL_I2S_ENABLE(&hi2s3);

    /* Wait for the end of I2S send */
    while (rxCnt < BUFFERSIZE)
    {
        
    }
    rxCnt = 0U;

    __DAL_I2S_DISABLE(&hi2s2);
    __DAL_I2S_DISABLE(&hi2s3);

    if (BufferCmp((uint8_t*)txBuffer, (uint8_t*)rxBuffer, BUFFERSIZE) != true)
    {
        BOARD_LED_On(LED3);
        LOGE(tag, "I2S Interrupt Communication Error\r\n");
        Error_Handler();
    }

    /* Infinite loop */
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500U);
    }
}

/**
 * @brief   Buffer compare
 *
 * @param   buf1: First buffer to be compared
 *
 * @param   buf2: Second buffer to be compared
 *
 * @param   size: Buffer size
 *
 * @retval  Comparison result(ture or false)
 */
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size)
{
    uint32_t i;

    bool status = true;

    for (i = 0U; i < size; i++)
    {
        if (buf1[i] != buf2[i])
        {
            status = false;
            break;
        }
    }

    return status;
}
