/**
 * @file        apm32f4xx_adc_cfg.c
 *
 * @brief       This file provides configuration support for ADC
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_adc_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
ADC_HandleTypeDef hadc1;
ADC_HandleTypeDef hadc2;
DMA_HandleTypeDef hdma_adc;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the ADC MSP
 *
 * @param  hadc pointer to a ADC_HandleTypeDef structure that contains
 *         the configuration information for the specified ADC. 
 *
 * @retval None
 */
void DAL_ADC_MspInit(ADC_HandleTypeDef* hadc)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0U};

    __DAL_RCM_ADC1_CLK_ENABLE();

    __DAL_RCM_ADC2_CLK_ENABLE();

    __DAL_RCM_GPIOA_CLK_ENABLE();

    /* ADC1 pin configuration */
    GPIO_InitStruct.Pin     = GPIO_PIN_0 | GPIO_PIN_1 | GPIO_PIN_2;
    GPIO_InitStruct.Mode    = GPIO_MODE_ANALOG;
    GPIO_InitStruct.Pull    = GPIO_NOPULL;
    DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

    /* ADC DMA Init */
    hdma_adc.Instance                  = DMA2_Stream0;
    hdma_adc.Init.Channel              = DMA_CHANNEL_0;
    hdma_adc.Init.Direction            = DMA_PERIPH_TO_MEMORY;
    hdma_adc.Init.PeriphInc            = DMA_PINC_DISABLE;
    hdma_adc.Init.MemInc               = DMA_MINC_ENABLE;
    hdma_adc.Init.PeriphDataAlignment  = DMA_PDATAALIGN_HALFWORD;
    hdma_adc.Init.MemDataAlignment     = DMA_MDATAALIGN_HALFWORD;
    hdma_adc.Init.Mode                 = DMA_CIRCULAR;
    hdma_adc.Init.Priority             = DMA_PRIORITY_HIGH;
    hdma_adc.Init.FIFOMode             = DMA_FIFOMODE_DISABLE;
    hdma_adc.Init.FIFOThreshold        = DMA_FIFO_THRESHOLD_HALFFULL;
    hdma_adc.Init.MemBurst             = DMA_MBURST_SINGLE;
    hdma_adc.Init.PeriphBurst          = DMA_PBURST_SINGLE;
    if (DAL_DMA_Init(&hdma_adc) != DAL_OK)
    {
        Error_Handler();
    }

    __DAL_LINKDMA(hadc,DMA_Handle,hdma_adc);

    /* Enable DMA interrupt */
    DAL_NVIC_SetPriority(DMA2_Stream0_IRQn, 0U, 0U);
    DAL_NVIC_EnableIRQ(DMA2_Stream0_IRQn);
}

/**
 * @brief  DeInitializes the ADC MSP
 *
 * @param  hadc pointer to a ADC_HandleTypeDef structure that contains
 *         the configuration information for the specified ADC
 *
 * @retval None
 */
void DAL_ADC_MspDeInit(ADC_HandleTypeDef* hadc)
{
    /* Reset peripheral clock */
    __DAL_RCM_ADC_FORCE_RESET();
    __DAL_RCM_ADC_RELEASE_RESET();

    /* ADC GPIO configuration */
    DAL_GPIO_DeInit(GPIOA, GPIO_PIN_0 | GPIO_PIN_1 | GPIO_PIN_2);

    /* Disable peripheral DMA */
    DAL_DMA_DeInit(hadc->DMA_Handle);

    /* Disable peripheral interrupt */
    DAL_NVIC_DisableIRQ(DMA2_Stream0_IRQn);
}

/**
 * @brief   ADC configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_ADC_Config(void)
{
    ADC_ChannelConfTypeDef Channel_ConfigStruct = {0};
    ADC_MultiModeTypeDef MultiMode_ConfigStruct = {0};

    /* Configure the ADC1 */
    hadc1.Instance                      = ADC1;
    hadc1.Init.ClockPrescaler           = ADC_CLOCK_SYNC_PCLK_DIV4;
    hadc1.Init.Resolution               = ADC_RESOLUTION_12B;
    hadc1.Init.ScanConvMode             = ENABLE;
    hadc1.Init.ContinuousConvMode       = ENABLE;
    hadc1.Init.DiscontinuousConvMode    = DISABLE;
    hadc1.Init.ExternalTrigConvEdge     = ADC_EXTERNALTRIGCONVEDGE_NONE;
    hadc1.Init.ExternalTrigConv         = ADC_SOFTWARE_START;
    hadc1.Init.DataAlign                = ADC_DATAALIGN_RIGHT;
    hadc1.Init.NbrOfConversion          = 2U;
    hadc1.Init.DMAContinuousRequests    = ENABLE;
    hadc1.Init.EOCSelection             = ADC_EOC_SINGLE_CONV;
    if (DAL_ADC_Init(&hadc1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure for the selected ADC regular channel */
    Channel_ConfigStruct.Channel        = ADC_CHANNEL_0;
    Channel_ConfigStruct.Rank           = 1U;
    Channel_ConfigStruct.SamplingTime   = ADC_SAMPLETIME_3CYCLES;
    if (DAL_ADC_ConfigChannel(&hadc1, &Channel_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }

    Channel_ConfigStruct.Channel        = ADC_CHANNEL_1;
    Channel_ConfigStruct.Rank           = 2U;
    Channel_ConfigStruct.SamplingTime   = ADC_SAMPLETIME_3CYCLES;
    if (DAL_ADC_ConfigChannel(&hadc1, &Channel_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure the the ADC2 */
    hadc2.Instance                      = ADC2;
    hadc2.Init.ClockPrescaler           = ADC_CLOCK_SYNC_PCLK_DIV4;
    hadc2.Init.Resolution               = ADC_RESOLUTION_12B;
    hadc2.Init.ScanConvMode             = ENABLE;
    hadc2.Init.ContinuousConvMode       = ENABLE;
    hadc2.Init.DiscontinuousConvMode    = DISABLE;
    hadc2.Init.ExternalTrigConvEdge     = ADC_EXTERNALTRIGCONVEDGE_NONE;
    hadc2.Init.ExternalTrigConv         = ADC_SOFTWARE_START;
    hadc2.Init.DataAlign                = ADC_DATAALIGN_RIGHT;
    hadc2.Init.NbrOfConversion          = 2U;
    hadc2.Init.DMAContinuousRequests    = ENABLE;
    hadc2.Init.EOCSelection             = ADC_EOC_SINGLE_CONV;
    if (DAL_ADC_Init(&hadc2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure for the selected ADC regular channel */
    Channel_ConfigStruct.Channel        = ADC_CHANNEL_1;
    Channel_ConfigStruct.Rank           = 1U;
    Channel_ConfigStruct.SamplingTime   = ADC_SAMPLETIME_3CYCLES;
    if (DAL_ADC_ConfigChannel(&hadc2, &Channel_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }

    Channel_ConfigStruct.Channel        = ADC_CHANNEL_2;
    Channel_ConfigStruct.Rank           = 2U;
    Channel_ConfigStruct.SamplingTime   = ADC_SAMPLETIME_3CYCLES;
    if (DAL_ADC_ConfigChannel(&hadc2, &Channel_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure ADC multi mode */
    MultiMode_ConfigStruct.Mode             = ADC_DUALMODE_REGSIMULT;
    MultiMode_ConfigStruct.DMAAccessMode    = ADC_DMAACCESSMODE_1;
    MultiMode_ConfigStruct.TwoSamplingDelay = ADC_TWOSAMPLINGDELAY_5CYCLES;
    if (DAL_ADCEx_MultiModeConfigChannel(&hadc1, &MultiMode_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}
