/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include <string.h>
#include <stdbool.h>

/* Private macro **********************************************************/

/* Define firmware to switch between 1st board or 2nd board */
#define IS_FIRST_BOARD

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static char txBuffer[] = "UART_TwoBoardsDMA\r\n";
static char rxBuffer[50U];
static volatile uint32_t uartEventFlag = 0U;

/* Private function prototypes ********************************************/
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size);

/* External variables *****************************************************/
extern UART_HandleTypeDef huart1;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

#if defined(IS_FIRST_BOARD)
    if (DAL_UART_Transmit_DMA(&huart1, (uint8_t *)txBuffer, strlen(txBuffer)) != DAL_OK)
    {
        Error_Handler();
    }

    /* Wait for the end of UART send */
    while (uartEventFlag != 1U)
    {
        
    }
    uartEventFlag = 0U;

    if (DAL_UART_Receive_DMA(&huart1, (uint8_t *)rxBuffer, strlen(txBuffer)) != DAL_OK)
    {
        Error_Handler();
    }

#else
    if (DAL_UART_Receive_DMA(&huart1, (uint8_t *)rxBuffer, strlen(txBuffer)) != DAL_OK)
    {
        Error_Handler();
    }

    /* Wait for the end of UART send */
    while (uartEventFlag != 1U)
    {
        
    }
    uartEventFlag = 0U;

    if (DAL_UART_Transmit_DMA(&huart1, (uint8_t *)txBuffer, strlen(txBuffer)) != DAL_OK)
    {
        Error_Handler();
    }
#endif

    /* Wait for the end of UART send */
    while (uartEventFlag != 1U)
    {
        
    }
    uartEventFlag = 0U;

    if (BufferCmp((uint8_t*)txBuffer, (uint8_t*)rxBuffer, strlen(txBuffer)) != true)
    {
        BOARD_LED_On(LED3);
        Error_Handler();
    }

    /* Infinite loop */
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500U);
    }
}

/**
 * @brief  Tx Transfer completed callbacks
 *
 * @param  huart  Pointer to a UART_HandleTypeDef structure that contains
 *                the configuration information for the specified UART module
 *
 * @retval None
 */
void DAL_UART_TxCpltCallback(UART_HandleTypeDef *huart)
{
    uartEventFlag = 1U;
}

/**
 * @brief  Rx Transfer completed callbacks
 *
 * @param  huart  Pointer to a UART_HandleTypeDef structure that contains
 *                the configuration information for the specified UART module
 *
 * @retval None
 */
void DAL_UART_RxCpltCallback(UART_HandleTypeDef *huart)
{
    uartEventFlag = 1U;
}

/**
 * @brief  UART error callbacks
 *
 * @param  huart  Pointer to a UART_HandleTypeDef structure that contains
 *                the configuration information for the specified UART module
 *
 * @retval None
 */
void DAL_UART_ErrorCallback(UART_HandleTypeDef *huart)
{
    BOARD_LED_On(LED3);
}

/* Private functions ******************************************************/

/**
 * @brief   Buffer compare
 *
 * @param   buf1: First buffer to be compared
 *
 * @param   buf2: Second buffer to be compared
 *
 * @param   size: Buffer size
 *
 * @retval  Comparison result(ture or false)
 */
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size)
{
    uint32_t i;

    bool status = true;

    for (i = 0U; i < size; i++)
    {
        if (buf1[i] != buf2[i])
        {
            status = false;
            break;
        }
    }

    return status;
}
