/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr1;
DMA_HandleTypeDef hdma_tmr1;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR PWM MSP
 *
 * @param  htmr TMR PWM handle
 *
 * @retval None
 */
void DAL_TMR_PWM_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR1)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR1_CLK_ENABLE();

        /* Enable DMA clock */
        __DAL_RCM_DMA2_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();
        __DAL_RCM_GPIOB_CLK_ENABLE();
        __DAL_RCM_GPIOE_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF1_TMR1;

        /* TMR channel 1  : PA8
           TMR channel 1N : PA7
           TMR channel 2  : PE11
           TMR channel 2N : PB14
           TMR channel 3  : PE13
           TMR channel 3N : PB15
           TMR channel 4  : PE14
        */
        GPIO_InitStruct.Pin         = GPIO_PIN_7 | GPIO_PIN_8;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_14 | GPIO_PIN_15;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_11 | GPIO_PIN_13 | GPIO_PIN_14;
        DAL_GPIO_Init(GPIOE, &GPIO_InitStruct);

        /* Configure DMA */
        hdma_tmr1.Instance                  = DMA2_Stream5;
        hdma_tmr1.Init.Channel              = DMA_CHANNEL_6;
        hdma_tmr1.Init.Direction            = DMA_MEMORY_TO_PERIPH;
        hdma_tmr1.Init.PeriphInc            = DMA_PINC_DISABLE;
        hdma_tmr1.Init.MemInc               = DMA_MINC_ENABLE;
        hdma_tmr1.Init.PeriphDataAlignment  = DMA_PDATAALIGN_WORD;
        hdma_tmr1.Init.MemDataAlignment     = DMA_MDATAALIGN_WORD;
        hdma_tmr1.Init.Mode                 = DMA_NORMAL;
        hdma_tmr1.Init.Priority             = DMA_PRIORITY_HIGH;
        hdma_tmr1.Init.FIFOMode             = DMA_FIFOMODE_ENABLE;
        hdma_tmr1.Init.FIFOThreshold        = DMA_FIFO_THRESHOLD_FULL;
        hdma_tmr1.Init.MemBurst             = DMA_MBURST_SINGLE;
        hdma_tmr1.Init.PeriphBurst          = DMA_PBURST_SINGLE;

        __DAL_LINKDMA(htmr, hdma[TMR_DMA_ID_UPDATE], hdma_tmr1);

        /* DMA with TMR update */
        if (DAL_DMA_Init(htmr->hdma[TMR_DMA_ID_UPDATE]) != DAL_OK)
        {
            Error_Handler();
        }
    }
}

/**
 * @brief  DeInitializes TMR PWM MSP
 *
 * @param  htmr TMR PWM handle
 *
 * @retval None
 */
void DAL_TMR_PWM_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR1)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR1_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_7 | GPIO_PIN_8);
        DAL_GPIO_DeInit(GPIOB, GPIO_PIN_14 | GPIO_PIN_15);
        DAL_GPIO_DeInit(GPIOE, GPIO_PIN_11 | GPIO_PIN_13 | GPIO_PIN_14);

        /* De-init peripheral DMA */
        DAL_DMA_DeInit(htmr->hdma[TMR_DMA_ID_UPDATE]);
    }
}

/**
 * @brief   TMR1 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR1_Config(void)
{
    TMR_OC_InitTypeDef OC_ConfigStruct = {0};

    htmr1.Instance                  = TMR1;
    htmr1.Init.Prescaler            = 167U;
    htmr1.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr1.Init.Period               = 9999U;
    htmr1.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr1.Init.RepetitionCounter    = 0U;
    htmr1.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_PWM_Init(&htmr1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure output channel 1 */
    OC_ConfigStruct.OCMode          = TMR_OCMODE_PWM1;
    OC_ConfigStruct.OCPolarity      = TMR_OCPOLARITY_HIGH;
    OC_ConfigStruct.OCNPolarity     = TMR_OCNPOLARITY_HIGH;
    OC_ConfigStruct.OCIdleState     = TMR_OCIDLESTATE_SET;
    OC_ConfigStruct.OCNIdleState    = TMR_OCNIDLESTATE_SET;
    OC_ConfigStruct.OCFastMode      = TMR_OCFAST_DISABLE;
    OC_ConfigStruct.Pulse           = 5000U;
    if (DAL_TMR_PWM_ConfigChannel(&htmr1, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    if (DAL_TMR_PWM_ConfigChannel(&htmr1, &OC_ConfigStruct, TMR_CHANNEL_2) != DAL_OK)
    {
        Error_Handler();
    }

    if (DAL_TMR_PWM_ConfigChannel(&htmr1, &OC_ConfigStruct, TMR_CHANNEL_3) != DAL_OK)
    {
        Error_Handler();
    }

    if (DAL_TMR_PWM_ConfigChannel(&htmr1, &OC_ConfigStruct, TMR_CHANNEL_4) != DAL_OK)
    {
        Error_Handler();
    }
}
