/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr5;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR Input Capture MSP
 *
 * @param  htmr TMR Input Capture handle
 *
 * @retval None
 */
void DAL_TMR_IC_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR5)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR5_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF2_TMR5;

        /* TMR channel 2 */
        GPIO_InitStruct.Pin         = GPIO_PIN_1;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }
}

/**
 * @brief  DeInitializes TMR Input Capture MSP
 *
 * @param  htmr TMR handle
 *
 * @retval None
 */
void DAL_TMR_IC_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR5)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR5_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_1);
    }
}

/**
 * @brief   TMR5 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR5_Config(void)
{
    TMR_IC_InitTypeDef IC_ConfigStruct = {0};
    TMR_SlaveConfigTypeDef Slave_ConfigStruct = {0};

    htmr5.Instance                  = TMR5;
    htmr5.Init.Prescaler            = 0U;
    htmr5.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr5.Init.Period               = 0xFFFFFFFFU; /* Set 32 bits counter */
    htmr5.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr5.Init.RepetitionCounter    = 0U;
    htmr5.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_IC_Init(&htmr5) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure input channel 1 */
    IC_ConfigStruct.ICPolarity  = TMR_ICPOLARITY_FALLING;
    IC_ConfigStruct.ICSelection = TMR_ICSELECTION_INDIRECTTI;
    IC_ConfigStruct.ICPrescaler = TMR_ICPSC_DIV1;
    IC_ConfigStruct.ICFilter    = 0U;
    if (DAL_TMR_IC_ConfigChannel(&htmr5, &IC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure input channel 2 */
    IC_ConfigStruct.ICPolarity  = TMR_ICPOLARITY_RISING;
    IC_ConfigStruct.ICSelection = TMR_ICSELECTION_DIRECTTI;
    IC_ConfigStruct.ICPrescaler = TMR_ICPSC_DIV1;
    IC_ConfigStruct.ICFilter    = 0U;
    if (DAL_TMR_IC_ConfigChannel(&htmr5, &IC_ConfigStruct, TMR_CHANNEL_2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure slave TMR */
    Slave_ConfigStruct.SlaveMode     = TMR_SLAVEMODE_RESET;
    Slave_ConfigStruct.InputTrigger  = TMR_TS_TI2FP2;
    if (DAL_TMR_SlaveConfigSynchro(&htmr5, &Slave_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}
