/**
 * @file        cameraif.c
 *
 * @brief       This file provides configuration support for camera
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "cameraif.h"

/* Private includes *******************************************************/
#include "board_ov2640.h"
#include "apm32f4xx_device_cfg.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
CAMERA_INFO_T cameraInfo;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief   Camera configuration
 *
 * @param   format: Camera resolution format
 *
 * @retval  Camera status
 */
uint32_t CAMERA_Config(uint32_t format)
{
    uint32_t status = CAMERA_STATUS_OK;

    /* Power on OV2640 */
#if defined(CAMERA_POWER_DOWN)
    DAL_GPIO_WritePin(CAMERA_PWDN_GPIO_PORT, CAMERA_PWDN_GPIO_PIN, GPIO_PIN_RESET);
    DAL_Delay(10U);
#endif /* CAMERA_POWER_DOWN */

    /* Reset OV2640 */
#if defined(CAMERA_HARDWARE_RESET)
    DAL_GPIO_WritePin(CAMERA_RESET_GPIO_PORT, CAMERA_RESET_GPIO_PIN, GPIO_PIN_RESET);
    DAL_Delay(10U);
    DAL_GPIO_WritePin(CAMERA_RESET_GPIO_PORT, CAMERA_RESET_GPIO_PIN, GPIO_PIN_SET);
    DAL_Delay(10U);
#endif /* CAMERA_HARDWARE_RESET */

    /* Init DCI */
    if (DAL_DCI_Config() == DAL_OK)
    {
        /* Check OV2640 PID */
        if (OV2640_ReadID(CAMERA_SCCB_ADDRESS) == OV2640_PID)
        {
            /* Init OV2640 */
            OV2640_Init(CAMERA_SCCB_ADDRESS, format);
        }

        cameraInfo.format = format;
    }
    else
    {
        status = CAMERA_STATUS_ERROR;
    }

    return status;
}

/**
 * @brief   Initializes SCCB
 *
 * @param   None
 *
 * @retval  0 if OK, -1 if ERROR
 */
int32_t OV2640_SCCB_InitCallback(void)
{
    DAL_I2C1_Config();

    return 0;
}

/**
 * @brief   OV2640 delay
 *
 * @param   delay: Delay in ms
 *
 * @retval  None
 */
void OV2640_DelayCallback(uint32_t delay)
{
    DAL_Delay(delay);
}

/**
 * @brief   Read a register
 *
 * @param   addr: I2C address
 *
 * @param   regAddr: Register address
 *
 * @param   regVal: Pointer to the register value
 *
 * @retval  0 if OK, -1 if ERROR
 */
int32_t OV2640_SCCB_ReadRegCallback(uint8_t addr, uint8_t regAddr, uint8_t *regVal)
{
    I2C1_ReadData(addr, regAddr, regVal);

    return 0;
}

/**
 * @brief   Write a value to a register
 *
 * @param   addr: I2C address
 *
 * @param   regAddr: Register address
 *
 * @param   regVal: Value to be written
 *
 * @retval  0 if OK, -1 if ERROR
 */
int32_t OV2640_SCCB_WriteRegCallback(uint8_t addr, uint8_t regAddr, uint8_t regVal)
{
    I2C1_WriteData(addr, regAddr, regVal);

    return 0;
}
