/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"
#include <string.h>

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";

/* Plaintext */
uint32_t plaintext[8] =
{
    0x6BC1BEE2U, 0x2E409F96U, 0xE93D7E11U, 0x7393172AU,
    0xAE2D8A57U, 0x1E03AC9CU, 0x9EB76FACU, 0x45AF8E51U
};

/* Expected ECB Ciphertext */
uint32_t ciphertextECB[8] =
{
    0x714772F3U, 0x39841D34U, 0x267FCC4BU, 0xD2949CC3U,
    0xEE11C22AU, 0x576A3038U, 0x76183F99U, 0xC0B6DE87U
};

/* Expected CBC Ciphertext */
uint32_t ciphertextCBC[8] =
{
    0x2079C3D5U, 0x3AA763E1U, 0x93B79E25U, 0x69AB5262U,
    0x51657048U, 0x1F25B50FU ,0x73C0BDA8U, 0x5C8E0DA7U
}; 

/* Encrypted text buffer */
static uint32_t encryptedText[8] = {0};

/* Decrypted text buffer */
static uint32_t decryptedText[8] = {0};

/* Private function prototypes ********************************************/

/* External variables *****************************************************/
extern CRYP_HandleTypeDef hcryp;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* TDES mode ECB */
    DAL_CRYP_TDESECBConfig();

    /* Encrypt the plaintext message*/
    DAL_CRYP_Encrypt(&hcryp, plaintext, 8U, encryptedText, 0xFFU);

    /* Compare with expected result */
    if(memcmp(encryptedText, ciphertextECB, 32U) != 0U)
    {
        LOGI(tag, "TDES ECB encryption error\r\n");
    }
    else
    {
        LOGI(tag, "TDES ECB encryption passed\r\n");
    }

    /* Decrypt the plaintext message */
    DAL_CRYP_Decrypt(&hcryp, ciphertextECB , 8U, decryptedText, 0xFFU);

    /*compare with expected result */
    if(memcmp(decryptedText, plaintext, 32U) != 0U)
    {
        LOGI(tag, "TDES ECB decryption error\r\n");
    }
    else
    {
        LOGI(tag, "TDES ECB decryption passed\r\n");
    }

    /* TDES mode CBC */
    DAL_CRYP_TDESCBCConfig();
    
    /* Encrypt the plaintext message*/
    DAL_CRYP_Encrypt(&hcryp, plaintext, 8U, encryptedText, 0xFFU);

    /* Compare with expected result */
    if(memcmp(encryptedText, ciphertextCBC, 32U) != 0U)
    {
        LOGI(tag, "TDES CBC encryption error\r\n");
    }
    else
    {
        LOGI(tag, "TDES CBC encryption passed\r\n");
    }
    
    /* Decrypt the plaintext message */
    DAL_CRYP_Decrypt(&hcryp, ciphertextCBC , 8U, decryptedText, 0xFFU);

    /*compare with expected result */
    if(memcmp(decryptedText, plaintext, 32U) != 0U)
    {
        LOGI(tag, "TDES CBC decryption error\r\n");
    }
    else
    {
        LOGI(tag, "TDES CBC decryption passed\r\n");
    }

    /* Infinite loop */
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500U);
    }
}
