/**
 * @file        apm32f4xx_can_cfg.c
 *
 * @brief       This file provides configuration support for CAN
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_can_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
CAN_HandleTypeDef hcan2;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the CAN MSP
 *
 * @param  hcan pointer to a CAN_HandleTypeDef structure that contains
 *         the configuration information for the specified CAN
 *
 * @retval None
 */
void DAL_CAN_MspInit(CAN_HandleTypeDef *hcan)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    /* Enable peripheral clock */
    __DAL_RCM_CAN1_CLK_ENABLE();
    __DAL_RCM_CAN2_CLK_ENABLE();

    /* Enable peripheral GPIO clock */
    __DAL_RCM_GPIOB_CLK_ENABLE();

    /* CAN2 TX and RX pin configuration */
    GPIO_InitStruct.Pin         = GPIO_PIN_5 | GPIO_PIN_6;
    GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
    GPIO_InitStruct.Pull        = GPIO_PULLUP;
    GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_HIGH;
    GPIO_InitStruct.Alternate   = GPIO_AF9_CAN2;
    DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
}

/**
 * @brief  DeInitializes the CAN MSP
 *
 * @param  hcan pointer to a CAN_HandleTypeDef structure that contains
 *         the configuration information for the specified CAN
 *
 * @retval None
 */
void DAL_CAN_MspDeInit(CAN_HandleTypeDef *hcan)
{
    __DAL_RCM_CAN2_FORCE_RESET();
    __DAL_RCM_CAN2_RELEASE_RESET();

    /* CAN GPIO configuration */
    DAL_GPIO_DeInit(GPIOB, GPIO_PIN_5 | GPIO_PIN_6);
}

/**
 * @brief   CAN1 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_CAN2_Config(void)
{
    CAN_FilterTypeDef  Filter_ConfigStruct = {0};

    /* Configure the CAN2 */
    /* Baudrate = (APB1Clock * 1000) / (Prescaler * (TimeSeg1 + TimeSeg2 + 3)) = 125kbps */
    hcan2.Instance                  = CAN2;
    hcan2.Init.Prescaler            = 56U;
    hcan2.Init.Mode                 = CAN_MODE_LOOPBACK;
    hcan2.Init.SyncJumpWidth        = CAN_SJW_1TQ;
    hcan2.Init.TimeSeg1             = CAN_BS1_2TQ;
    hcan2.Init.TimeSeg2             = CAN_BS2_3TQ;
    hcan2.Init.AutoBusOff           = DISABLE;
    hcan2.Init.AutoWakeUp           = DISABLE;
    hcan2.Init.AutoRetransmission   = DISABLE;
    hcan2.Init.ReceiveFifoLocked    = DISABLE;
    hcan2.Init.TransmitFifoPriority = DISABLE;
    if (DAL_CAN_Init(&hcan2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure the CAN Filter */
    Filter_ConfigStruct.FilterBank              = 14U;
    Filter_ConfigStruct.FilterMode              = CAN_FILTERMODE_IDMASK;
    Filter_ConfigStruct.FilterScale             = CAN_FILTERSCALE_32BIT;
    Filter_ConfigStruct.FilterIdHigh            = 0x0000U;
    Filter_ConfigStruct.FilterIdLow             = 0x0000U;
    Filter_ConfigStruct.FilterMaskIdHigh        = 0x0000U;
    Filter_ConfigStruct.FilterMaskIdLow         = 0x0000U;
    Filter_ConfigStruct.FilterFIFOAssignment    = CAN_RX_FIFO0;
    Filter_ConfigStruct.FilterActivation        = ENABLE;
    Filter_ConfigStruct.SlaveStartFilterBank    = 14U;
    if (DAL_CAN_ConfigFilter(&hcan2, &Filter_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}
