/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include <string.h>
#include <stdbool.h>

/* Private macro **********************************************************/

/* Private typedef ********************************************************/
/**
 * @brief   RS485 transfer status
 */
typedef enum
{
    RS485_STATUS_TX = 0U,   /*!< RS485 transfer status: TX */
    RS485_STATUS_RX = 1U,   /*!< RS485 transfer status: RX */
} RS485_StatusTypeDef;

/* Private variables ******************************************************/
static char txBuffer[] = "UART_RS485\r\n";
static char rxBuffer[50U];
static volatile uint32_t uartEventFlag = 0U;

/* Private function prototypes ********************************************/
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size);
static void RS485_ConfigTxRx(RS485_StatusTypeDef status);

/* External variables *****************************************************/
extern UART_HandleTypeDef huart3;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Configure RS485 transfer status */
    RS485_ConfigTxRx(RS485_STATUS_TX);

    if (DAL_UART_Transmit_IT(&huart3, (uint8_t *)txBuffer, strlen(txBuffer)) != DAL_OK)
    {
        Error_Handler();
    }

    /* Wait for the end of UART send */
    while (uartEventFlag != 1U)
    {
        
    }
    uartEventFlag = 0U;

    if (DAL_UART_Receive_IT(&huart3, (uint8_t *)rxBuffer, strlen(txBuffer)) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure RS485 transfer status */
    RS485_ConfigTxRx(RS485_STATUS_RX);

    /* Wait for the end of UART send */
    while (uartEventFlag != 1U)
    {
        
    }
    uartEventFlag = 0U;

    if (BufferCmp((uint8_t*)txBuffer, (uint8_t*)rxBuffer, strlen(txBuffer)) != true)
    {
        BOARD_LED_On(LED3);
        Error_Handler();
    }

    /* Infinite loop */
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500U);
    }
}

/**
 * @brief  Tx Transfer completed callbacks
 *
 * @param  huart  Pointer to a UART_HandleTypeDef structure that contains
 *                the configuration information for the specified UART module
 *
 * @retval None
 */
void DAL_UART_TxCpltCallback(UART_HandleTypeDef *huart)
{
    uartEventFlag = 1U;
}

/**
 * @brief  Rx Transfer completed callbacks
 *
 * @param  huart  Pointer to a UART_HandleTypeDef structure that contains
 *                the configuration information for the specified UART module
 *
 * @retval None
 */
void DAL_UART_RxCpltCallback(UART_HandleTypeDef *huart)
{
    uartEventFlag = 1U;
}

/**
 * @brief  UART error callbacks
 *
 * @param  huart  Pointer to a UART_HandleTypeDef structure that contains
 *                the configuration information for the specified UART module
 *
 * @retval None
 */
void DAL_UART_ErrorCallback(UART_HandleTypeDef *huart)
{
    BOARD_LED_On(LED3);
}

/* Private functions ******************************************************/

/**
 * @brief   Configures the RS485 transfer status
 *
 * @param   status : RS485 transfer status
 *          This parameter can be one of the following values:
 *           @arg RS485_STATUS_TX : RS485 transfer status: TX
 *           @arg RS485_STATUS_RX : RS485 transfer status: RX
 *
 * @retval  None
 */
static void RS485_ConfigTxRx(RS485_StatusTypeDef status)
{
    DAL_Delay(1000U);

    if (status == RS485_STATUS_TX)
    {
        /* Enable RS485 TX */
        DAL_GPIO_WritePin(RS485_EN_GPIO_PORT, RS485_EN_GPIO_PIN, GPIO_PIN_SET);
    }
    else
    {
        /* Enable RS485 RX */
        DAL_GPIO_WritePin(RS485_EN_GPIO_PORT, RS485_EN_GPIO_PIN, GPIO_PIN_RESET);
    }
}

/**
 * @brief   Buffer compare
 *
 * @param   buf1: First buffer to be compared
 *
 * @param   buf2: Second buffer to be compared
 *
 * @param   size: Buffer size
 *
 * @retval  Comparison result(ture or false)
 */
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size)
{
    uint8_t i;

    bool status = true;

    for (i = 0U; i < size; i++)
    {
        if (buf1[i] != buf2[i])
        {
            status = false;
            break;
        }
    }

    return status;
}
