/**
 * @file        flash.c
 *
 * @brief       This file provides application support for flash
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "flash.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"

/* Private macro **********************************************************/
#define FLASH_TIMEOUT_DEFAULT_VALUE        5000U
#define FLASH_ERASE_TIMEOUT                100000U

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
FLASH_INFO_T flashInfo;

/* Private function prototypes ********************************************/
static uint32_t FLASH_Config(void);
static uint8_t FLASH_TransferByte(uint8_t data);

/* External variables *****************************************************/
extern SPI_HandleTypeDef hspi1;

/* External functions *****************************************************/

/**
 * @brief   Initializes flash
 *
 * @param   None
 *
 * @retval  Flash status
 */
uint32_t FLASH_Init(void)
{
    /* Configure flash */
    FLASH_Config();

    /* Read W25QXX device ID */
    FLASH_ReadDeviceID();
    if (flashInfo.deviceID != W25Q16_DEVICE_ID)
    {
        return FLASH_STATUS_ERROR;
    }
    else
    {
        flashInfo.sectorSize = 256;
    }

    return FLASH_STATUS_OK;
}

/**
 * @brief   De-initializes flash
 * 
 * @param   None
 * 
 * @retval  Flash status
 */
uint32_t FLASH_DeInit(void)
{
    return FLASH_STATUS_OK;
}

/**
 * @brief   Write enable
 * 
 * @param   None
 * 
 * @retval  Flash status
 */
uint32_t FLASH_WriteEnable(void)
{
    FLASH_ChipSelect(ENABLE);

    FLASH_TransferByte(W25QXX_INS_WRITE_ENABLE);

    FLASH_ChipSelect(DISABLE);

    return FLASH_STATUS_OK;
}

/**
 * @brief   Read status register
 * 
 * @param   reg: register
 *          This parameter can be one of the following values:
 *           @arg W25QXX_STATUS_REG1: status register 1
 *           @arg W25QXX_STATUS_REG2: status register 2
 *           @arg W25QXX_STATUS_REG3: status register 3
 * 
 * @retval  Register value
 */
uint8_t FLASH_ReadSR(uint32_t reg)
{
    uint8_t cmd;
    uint8_t status = 0;

    switch (reg)
    {
        case W25QXX_STATUS_REG1:
            cmd = W25QXX_INS_READ_STATUS_REG1;
            break;

        case W25QXX_STATUS_REG2:
            cmd = W25QXX_INS_READ_STATUS_REG2;
            break;

        case W25QXX_STATUS_REG3:
            cmd = W25QXX_INS_READ_STATUS_REG3;
            break;

        default:
            return 0;
    }

    FLASH_ChipSelect(ENABLE);

    FLASH_TransferByte(cmd);
    status = FLASH_TransferByte(0xFF);

    FLASH_ChipSelect(DISABLE);

    return status;
}

/**
 * @brief   Write status register
 * 
 * @param   reg: register
 *          This parameter can be one of the following values:
 *           @arg W25QXX_STATUS_REG1: status register 1
 *           @arg W25QXX_STATUS_REG2: status register 2
 *           @arg W25QXX_STATUS_REG3: status register 3
 * 
 * @param   sr: status register value
 * 
 * @retval  Flash status
 */
uint32_t FLASH_WriteSR(uint32_t reg, uint8_t sr)
{
    uint32_t status = FLASH_STATUS_OK;
    uint8_t cmd;

    switch (reg)
    {
        case W25QXX_STATUS_REG1:
            cmd = W25QXX_INS_WRITE_STATUS_REG1;
            break;

        case W25QXX_STATUS_REG2:
            cmd = W25QXX_INS_WRITE_STATUS_REG2;
            break;

        case W25QXX_STATUS_REG3:
            cmd = W25QXX_INS_WRITE_STATUS_REG3;
            break;

        default:
            return FLASH_STATUS_ERROR;
    }

    /* Enable write */
    FLASH_WriteEnable();
    status = FLASH_WaitForWriteReady(FLASH_TIMEOUT_DEFAULT_VALUE);
    if (status != FLASH_STATUS_OK)
    {
        return status;
    }

    /* Wait for flash ready */
    status = FLASH_WaitForReady(FLASH_TIMEOUT_DEFAULT_VALUE);
    if (status != FLASH_STATUS_OK)
    {
        return status;
    }

    /* Erase sector */
    FLASH_ChipSelect(ENABLE);

    FLASH_TransferByte(cmd);
    FLASH_TransferByte(sr);

    FLASH_ChipSelect(DISABLE);

    /* Wait for flash ready */
    status = FLASH_WaitForReady(FLASH_ERASE_TIMEOUT);

    return FLASH_STATUS_OK;
}

/**
 * @brief   Wait for flash ready
 * 
 * @param   timeout: timeout
 * 
 * @retval  Flash status
 */
uint32_t FLASH_WaitForReady(uint32_t timeout)
{
    uint32_t status = FLASH_STATUS_OK;
    uint32_t sr1;

    /* Wait for flash ready */
    while (1)
    {
        sr1 = FLASH_ReadSR(W25QXX_STATUS_REG1);
        if ((sr1 & W25QXX_SR1_BUSY) != W25QXX_SR1_BUSY)
        {
            break;
        }

        if (timeout == 0U)
        {
            status = FLASH_STATUS_TIMEOUT;
            break;
        }
        timeout--;
    }

    return status;
}

/**
 * @brief  Wait for flash ready to write
 * 
 * @param  timeout: timeout
 * 
 * @retval Flash status
 */
uint32_t FLASH_WaitForWriteReady(uint32_t timeout)
{
    uint32_t status = FLASH_STATUS_OK;
    uint32_t sr1;

    /* Wait for flash ready */
    while (1)
    {
        sr1 = FLASH_ReadSR(W25QXX_STATUS_REG1);
        if ((sr1 & W25QXX_SR1_WEL) == W25QXX_SR1_WEL)
        {
            break;
        }

        if (timeout == 0U)
        {
            status = FLASH_STATUS_TIMEOUT;
            break;
        }
        timeout--;
    }

    return status;
}

/**
 * @brief  Read device ID
 * 
 * @param  None
 * 
 * @retval Flash status
 */
uint32_t FLASH_ReadDeviceID(void)
{
    uint8_t temp[2];

    FLASH_ChipSelect(ENABLE);

    FLASH_TransferByte(W25QXX_INS_MANUFACTURER_ID);
    FLASH_TransferByte(0x00);
    FLASH_TransferByte(0x00);
    FLASH_TransferByte(0x00);

    temp[0] = FLASH_TransferByte(0xFF);
    temp[1] = FLASH_TransferByte(0xFF);

    FLASH_ChipSelect(DISABLE);

    flashInfo.deviceID = temp[0] << 8 | temp[1];

    return FLASH_STATUS_OK;
}

/**
 * @brief   Erase flash sector
 * 
 * @param   addr: address
 * 
 * @retval  Flash status
 */
uint32_t FLASH_EraseSector(uint32_t addr)
{
    uint32_t status = FLASH_STATUS_OK;
    uint8_t data[4];

    data[0] = W25QXX_INS_SECTOR_ERASE;
    data[1] = (addr >> 16) & 0xFF;
    data[2] = (addr >> 8) & 0xFF;
    data[3] = addr & 0xFF;

    /* Enable write */
    FLASH_WriteEnable();
    status = FLASH_WaitForWriteReady(FLASH_TIMEOUT_DEFAULT_VALUE);
    if (status != FLASH_STATUS_OK)
    {
        return status;
    }

    /* Wait for flash ready */
    status = FLASH_WaitForReady(FLASH_TIMEOUT_DEFAULT_VALUE);
    if (status != FLASH_STATUS_OK)
    {
        return status;
    }

    /* Erase sector */
    FLASH_ChipSelect(ENABLE);

    FLASH_TransferByte(data[0]);
    FLASH_TransferByte(data[1]);
    FLASH_TransferByte(data[2]);
    FLASH_TransferByte(data[3]);

    FLASH_ChipSelect(DISABLE);

    /* Wait for flash ready */
    status = FLASH_WaitForReady(FLASH_ERASE_TIMEOUT);

    return status;
}

/**
 * @brief   FLASH read data
 * 
 * @param   addr: address
 * 
 * @param   data: pointer to data
 * 
 * @param   len: length of data
 * 
 * @retval  Flash status
 */
uint32_t FLASH_ReadData(uint32_t addr, uint8_t *data, uint32_t len)
{
    uint32_t status = FLASH_STATUS_OK;
    uint32_t i;

    FLASH_ChipSelect(ENABLE);

    FLASH_TransferByte(W25QXX_INS_READ_DATA);
    FLASH_TransferByte((addr >> 16) & 0xFF);
    FLASH_TransferByte((addr >> 8) & 0xFF);
    FLASH_TransferByte(addr & 0xFF);

    for (i = 0; i < len; i++)
    {
        data[i] = FLASH_TransferByte(0xFF);
    }

    FLASH_ChipSelect(DISABLE);

    return status;
}

/**
 * @brief   FLASH write data
 * 
 * @param   addr: address
 * 
 * @param   data: pointer to data
 * 
 * @param   len: length of data
 * 
 * @retval  Flash status
 */
uint32_t FLASH_WriteData(uint32_t addr, uint8_t *data, uint32_t len)
{
    uint32_t status = FLASH_STATUS_OK;
    uint32_t i;

    /* Enable write */
    FLASH_WriteEnable();
    status = FLASH_WaitForWriteReady(FLASH_TIMEOUT_DEFAULT_VALUE);
    if (status != FLASH_STATUS_OK)
    {
        return status;
    }

    /* Wait for flash ready */
    status = FLASH_WaitForReady(FLASH_TIMEOUT_DEFAULT_VALUE);
    if (status != FLASH_STATUS_OK)
    {
        return status;
    }

    /* Write data */
    FLASH_ChipSelect(ENABLE);

    FLASH_TransferByte(W25QXX_INS_PAGE_PROGRAM);
    FLASH_TransferByte((addr >> 16) & 0xFF);
    FLASH_TransferByte((addr >> 8) & 0xFF);
    FLASH_TransferByte(addr & 0xFF);

    for (i = 0; i < len; i++)
    {
        FLASH_TransferByte(data[i]);
    }

    FLASH_ChipSelect(DISABLE);

    /* Wait for flash ready */
    status = FLASH_WaitForReady(FLASH_TIMEOUT_DEFAULT_VALUE);

    return status;
}

/**
 * @brief   FLASH delay
 *
 * @param   delay: delay time
 * 
 * @retval  None
 */
void FLASH_Delay(uint32_t delay)
{
    DAL_Delay(delay);
}

/**
 * @brief   FLASH chip select
 * 
 * @param   state: chip select state
 *          This parameter can be one of the following values:
 *          @arg ENABLE: chip select enable
 *          @arg DISABLE: chip select disable
 * 
 * @retval  None
 */
void FLASH_ChipSelect(FunctionalState state)
{
    if (state == ENABLE)
    {
        DAL_GPIO_WritePin(SPI_CS_GPIO_PORT, SPI_CS_GPIO_PIN, GPIO_PIN_RESET);
    }
    else
    {
        DAL_GPIO_WritePin(SPI_CS_GPIO_PORT, SPI_CS_GPIO_PIN, GPIO_PIN_SET);
    }
}

/**
 * @brief   FLASH transfer byte
 * 
 * @param   data: pointer to data
 * 
 * @retval  None
 */
static uint8_t FLASH_TransferByte(uint8_t data)
{
    uint8_t rxData;

    DAL_SPI_TransmitReceive(&hspi1, &data, &rxData, 1, 1000);

    return rxData;
}

/**
 * @brief   Configures flash
 *
 * @param   None
 *
 * @retval  Flash status
 */
static uint32_t FLASH_Config(void)
{
    uint32_t status = FLASH_STATUS_OK;

    /* Initialize W25QXX */
    DAL_SPI1_Config();

    return status;
}
