/**
 * @file        flash.h
 *
 * @brief       This file provides application support for flash
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Define to prevent recursive inclusion */
#ifndef FLASH_H
#define FLASH_H

#ifdef __cplusplus
  extern "C" {
#endif

/* Includes ***************************************************************/
#include "apm32f4xx_dal.h"

/* Exported macro *********************************************************/

/**
 * @brief Flash_Status FLASH status
 */
#define FLASH_STATUS_OK                         0x00U
#define FLASH_STATUS_ERROR                      0x01U
#define FLASH_STATUS_TIMEOUT                    0x02U

/**
 * @brief W25QXX_Status W25QXX status
 */
#define W25QXX_STATUS_INIT                      0x00U
#define W25QXX_STATUS_READY                     0x01U
#define W25QXX_STATUS_ERROR                     0x02U
#define W25QXX_STATUS_OK                        0x03U
#define W25QXX_STATUS_TIMEOUT                   0x04U

/**
 * @brief W25QXX_Instruction_Mode W25QXX instruction mode
 */
#define W25QXX_STANDARD_INS_ADDR_MODE           0x00U
#define W25QXX_STANDARD_INS_MODE                0x01U
#define W25QXX_FRF_INS_ADDR_MODE                0x02U

/**
 * @brief W25QXX_Device_ID W25QXX device ID
 */
#define W25Q16_DEVICE_ID                        0xEF14U
#define W25Q32_DEVICE_ID                        0xEF15U
#define W25Q64_DEVICE_ID                        0xEF16U
#define W25Q128_DEVICE_ID                       0xEF17U
#define W25Q256_DEVICE_ID                       0xEF18U

/**
 * @brief W25QXX_Status_Register_Number W25QXX status register number
 */
#define W25QXX_STATUS_REG1                      0x01U
#define W25QXX_STATUS_REG2                      0x02U
#define W25QXX_STATUS_REG3                      0x03U

/**
 * @brief W25QXX_Status_Register1 W25QXX status register 1
 */
#define W25QXX_SR1_BUSY                         0x01U
#define W25QXX_SR1_WEL                          0x02U
#define W25QXX_SR1_BP0                          0x04U
#define W25QXX_SR1_BP1                          0x08U
#define W25QXX_SR1_BP2                          0x10U
#define W25QXX_SR1_TB                           0x20U
#define W25QXX_SR1_SEC                          0x40U
#define W25QXX_SR1_SRP0                         0x80U

/**
 * @brief W25QXX_Status_Register2 W25QXX status register 2
 */
#define W25QXX_SR2_SRL                          0x01U
#define W25QXX_SR2_QE                           0x02U
#define W25QXX_SR2_LB1                          0x08U
#define W25QXX_SR2_LB2                          0x10U
#define W25QXX_SR2_LB3                          0x20U
#define W25QXX_SR2_CMP                          0x40U
#define W25QXX_SR2_SUS                          0x80U

/**
 * @brief W25QXX_Instruction_Table1 W25QXX instruction table 1
 */
#define W25QXX_INS_WRITE_ENABLE                 0x06U
#define W25QXX_INS_WRITE_DISABLE                0x04U
#define W25QXX_INS_READ_STATUS_REG1             0x05U
#define W25QXX_INS_READ_STATUS_REG2             0x35U
#define W25QXX_INS_READ_STATUS_REG3             0x15U
#define W25QXX_INS_WRITE_STATUS_REG1            0x01U
#define W25QXX_INS_WRITE_STATUS_REG2            0x31U
#define W25QXX_INS_WRITE_STATUS_REG3            0x11U
#define W25QXX_INS_READ_DATA                    0x03U
#define W25QXX_INS_FAST_READ_DATA               0x0BU
#define W25QXX_INS_PAGE_PROGRAM                 0x02U
#define W25QXX_INS_BLOCK_ERASE                  0xD8U
#define W25QXX_INS_SECTOR_ERASE                 0x20U
#define W25QXX_INS_CHIP_ERASE                   0xC7U
#define W25QXX_INS_POWER_DOWN                   0xB9U
#define W25QXX_INS_RELEASE_POWER_DOWN           0xABU
#define W25QXX_INS_DEVICE_ID                    0xABU
#define W25QXX_INS_MANUFACTURER_ID              0x90U
#define W25QXX_INS_JEDEC_ID                     0x9FU
#define W25QXX_INS_UNIQUE_ID                    0x4BU
#define W25QXX_INS_READ_SFDP_REG                0x5AU
#define W25QXX_INS_ERASE_SECURITY_REG           0x44U
#define W25QXX_INS_PROGRAM_SECURITY_REG         0x42U
#define W25QXX_INS_READ_SECURITY_REG            0x48U
#define W25QXX_INS_GLOBAL_BLOCK_LOCK            0x7EU
#define W25QXX_INS_GLOBAL_BLOCK_UNLOCK          0x98U
#define W25QXX_INS_READ_BLOCK_LOCK              0x3DU
#define W25QXX_INS_INDIVIDUAL_BLOCK_LOCK        0x36U
#define W25QXX_INS_INDIVIDUAL_BLOCK_UNLOCK      0x39U
#define W25QXX_INS_ERASE_PROGRAM_SUSPEND        0x75U
#define W25QXX_INS_ERASE_PROGRAM_RESUME         0x7AU
#define W25QXX_INS_ENABLE_RESET                 0x66U
#define W25QXX_INS_RESET_DEVICE                 0x99U

/**
 * @brief W25QXX_Instruction_Table2 W25QXX instruction table 2
 */
#define W25QXX_INS_FAST_READ_DUAL               0x3BU
#define W25QXX_INS_FAST_READ_DUAL_IO            0xBBU
#define W25QXX_INS_FAST_READ_QUAD               0x6BU
#define W25QXX_INS_FAST_READ_QUAD_IO            0xEBU
#define W25QXX_INS_MANUFACTURER_ID_DUAL         0x92U
#define W25QXX_INS_MANUFACTURER_ID_QUAD         0x94U
#define W25QXX_INS_PAGE_PROGRAM_QUAD            0x32U

/* Exported typedef *******************************************************/

/**
 * @brief Flash_Informations FLASH informations
 */
typedef struct
{
    uint16_t deviceID;      /*!< Device ID */
    uint32_t sectorSize;    /*!< Sector size */
} FLASH_INFO_T;

/* Exported function prototypes *******************************************/
uint32_t FLASH_Init(void);
uint32_t FLASH_DeInit(void);
void FLASH_ChipSelect(FunctionalState state);
uint32_t FLASH_WriteEnable(void);
uint32_t FLASH_ReadDeviceID(void);
uint32_t FLASH_EraseSector(uint32_t addr);
uint32_t FLASH_ReadData(uint32_t addr, uint8_t *data, uint32_t len);
uint32_t FLASH_WriteData(uint32_t addr, uint8_t *data, uint32_t len);

uint32_t FLASH_WaitForReady(uint32_t timeout);
uint32_t FLASH_WaitForWriteReady(uint32_t timeout);

#ifdef __cplusplus
}
#endif

#endif /* FLASH_H */
