/**
 * @file        fatfs.c
 *
 * @brief       Fatfs program body
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "fatfs.h"

/* Private includes *******************************************************/
#include <string.h>
#include <stdio.h>

#include "log.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "fatfs";

/* Workspace of logic disk */
FATFS* fs[FF_VOLUMES];
UINT byteReadWriteNum;
/* fatfs read buffer */
BYTE readBuffer[1024] = {0};
/* fatfs write buffer */
BYTE writeBuffer[] = "www.geehy.com";

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
* @brief    Write a file to fatfs
*
* @param    file: point to the file
*
* @param    path: Disk path
*
* @retva    None
*/
void FATFS_WriteFile(FIL* file, const TCHAR *path)
{
    FRESULT status;
    char filePath[32];

    /* Write file */
    /* Open or create file */
    LOGI(tag, "****** Write File ******\r\n");
    sprintf(filePath, "%sapm32.txt", path);
    status = f_open(file, filePath, FA_CREATE_ALWAYS | FA_WRITE);
    if (status == FR_OK)
    {
        LOGI(tag, "Open or create file success\r\n");
        LOGI(tag, "Write data to file\r\n");

        /* Write data */
        status = f_write(file, writeBuffer, sizeof(writeBuffer), &byteReadWriteNum);
        if (status == FR_OK)
        {
            LOGI(tag, "Write %d bytes data to file\r\n", byteReadWriteNum);
            LOGI(tag, "The data written to the file is \"%s\"\r\n", writeBuffer);
        }
        else
        {
            LOGE(tag, "Write data to file fail\r\n");
        }

        /* Close file */
        f_close(file);
    }
    else
    {
        LOGE(tag, "Open or create file fail, status is %d\r\n", status);
    }

    /* Close file */
    f_close(file);
}

/**
* @brief    Read a file from fatfs
*
* @param    file: point to the file.
*
* @param    path: Disk path
*
* @retval   None
*
*/
void FATFS_ReadFile(FIL* file, const TCHAR *path)
{
    FRESULT status;
    char filePath[64];
    
    /* Read file */
    /* Open file */
    LOGI(tag, "****** Read File ******\r\n");
    sprintf(filePath, "%sapm32.txt", path);
    status = f_open(file, filePath, FA_OPEN_EXISTING | FA_READ);
    if (status == FR_OK)
    {
        LOGI(tag, "Open file success.\r\n");
        status = f_read(file, readBuffer, sizeof(readBuffer), &byteReadWriteNum);
        if (status == FR_OK)
        {
            LOGI(tag, "Read %d bytes data from file\r\n", byteReadWriteNum);
            LOGI(tag, "The data read from the file is \"%s\"\r\n", readBuffer);
        }
        else
        {
            LOGE(tag, "Read data from file fail\r\n");
        }
    }
    else
    {
        LOGE(tag, "Open file fail, status is %d\r\n", status);
    }

    /* Close file */
    f_close(file);
}

/**
 * @brief  Get Drive Path
 * 
 * @param  driveNum: Drive number
 *         This parameter can be one of the following values:
 *          @arg USBH_DISK
 *          @arg SD_DISK
 *          @arg SRAM_DISK
 *          @arg SDRAM_DISK
 *          @arg USER_DISK
 * 
 * @retval char*: Drive path
 */
char* FATFS_GetDrivePath(BYTE driveNum)
{
    static char path[3];

    if (driveNum < FF_VOLUMES)
    {
        path[0] = '0' + driveNum;
        path[1] = ':';
        path[2] = '\0';
    }
    else
    {
        path[0] = '\0';
    }

    return path;
}

/**
 * @brief  Get time from RTC
 * 
 * @retval DWORD: Time
 */
DWORD get_fattime(void)
{
    return 0;
}
