/**
 * @file        apm32f4xx_sdram_cfg.c
 *
 * @brief       This file provides configuration support for SDRAM
 *
 * @version     V1.0.1
 *
 * @date        2024-09-13
 *
 * @attention
 *
 *  Copyright (C) 2023-2024 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_sdram_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
SDRAM_HandleTypeDef hsdram1;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/


/**
  * @brief  SDRAM MSP Init.
  * @param  hsdram pointer to a SDRAM_HandleTypeDef structure that contains
  *                the configuration information for SDRAM module.
  * @retval None
  */
void DAL_SDRAM_MspInit(SDRAM_HandleTypeDef *hsdram)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};
    RCM_PeriphCLKInitTypeDef PeriphCLK_InitStruct = {0};

    if(hsdram->Instance == DMC_SDRAM_DEVICE)
    {
        /* Enable peripheral clock */
        PeriphCLK_InitStruct.PeriphClockSelection   = RCM_PERIPHCLK_SDRAM;
        PeriphCLK_InitStruct.SDRAMClockDivision     = RCM_SDRAM_DIV_4;
        DAL_RCMEx_PeriphCLKConfig(&PeriphCLK_InitStruct);

        __DAL_RCM_EMMC_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();
        __DAL_RCM_GPIOD_CLK_ENABLE();
        __DAL_RCM_GPIOF_CLK_ENABLE();
        __DAL_RCM_GPIOH_CLK_ENABLE();
        __DAL_RCM_GPIOG_CLK_ENABLE();
        __DAL_RCM_GPIOI_CLK_ENABLE();

        /* SDRAM pin configuration 
           A0   ----> PF1
           A1   ----> PF2
           A2   ----> PF3
           A3   ----> PF4
           A4   ----> PF6
           A5   ----> PF7
           A6   ----> PF8
           A7   ----> PF9
           A8   ----> PF10
           A9   ----> PH3
           A10  ----> PF0
           D0   ----> PG3
           D1   ----> PG4
           D2   ----> PG5
           D3   ----> PG6
           D4   ----> PG8
           D5   ----> PH13
           D6   ----> PH15
           D7   ----> PI3
           D8   ----> PH8
           D9   ----> PH10
           D10  ----> PD10
           D11  ----> PD12
           D12  ----> PD13
           D13  ----> PD14
           D14  ----> PD15
           D15  ----> PG2
           BA   ----> PI11
           CKE  ----> PA3
           CLK  ----> PG1
           LDQM ----> PG15
           UNQM ----> PF11
           NWE  ----> PI7
           NCAS ----> PI8
           NRAS ----> PI9
           NCS  ----> PI10
        */
        GPIO_InitStruct.Pin         = GPIO_PIN_10 | GPIO_PIN_12 | \
                                      GPIO_PIN_13 | GPIO_PIN_14 | \
                                      GPIO_PIN_15;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_NOPULL;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_VERY_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF12_DMC;
        DAL_GPIO_Init(GPIOD, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_0 | GPIO_PIN_1 | \
                                      GPIO_PIN_2 | GPIO_PIN_3 | \
                                      GPIO_PIN_4 | GPIO_PIN_6 | \
                                      GPIO_PIN_7 | GPIO_PIN_8 | \
                                      GPIO_PIN_9 | GPIO_PIN_10 | \
                                      GPIO_PIN_11;
        DAL_GPIO_Init(GPIOF, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_1 | GPIO_PIN_2 | \
                                      GPIO_PIN_3 | GPIO_PIN_4 | \
                                      GPIO_PIN_5 | GPIO_PIN_6 | \
                                      GPIO_PIN_8 | GPIO_PIN_15;
        DAL_GPIO_Init(GPIOG, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_3 | GPIO_PIN_8 | \
                                      GPIO_PIN_10 | GPIO_PIN_13 | \
                                      GPIO_PIN_15;
        DAL_GPIO_Init(GPIOH, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_3;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_3 | GPIO_PIN_7 | \
                                      GPIO_PIN_8 | GPIO_PIN_9 | \
                                      GPIO_PIN_10 | GPIO_PIN_11;
        DAL_GPIO_Init(GPIOI, &GPIO_InitStruct);
    }
}

/**
  * @brief  SDRAM MSP DeInit.
  * @param  hsdram pointer to a SDRAM_HandleTypeDef structure that contains
  *                the configuration information for SDRAM module.
  * @retval None
  */
void DAL_SDRAM_MspDeInit(SDRAM_HandleTypeDef *hsdram)
{
    if(hsdram->Instance == DMC_SDRAM_DEVICE)
    {
        /* Disable peripheral clock */
        __DAL_RCM_EMMC_CLK_DISABLE();

        /* SDRAM GPIO configuration */
        DAL_GPIO_DeInit(GPIOD, GPIO_PIN_10 | GPIO_PIN_12 | GPIO_PIN_13 | \
                               GPIO_PIN_14 | GPIO_PIN_15);

        DAL_GPIO_DeInit(GPIOF, GPIO_PIN_0 | GPIO_PIN_1 | GPIO_PIN_2 | \
                               GPIO_PIN_3 | GPIO_PIN_4 | GPIO_PIN_6 | \
                               GPIO_PIN_7 | GPIO_PIN_8 | GPIO_PIN_9 | \
                               GPIO_PIN_10 | GPIO_PIN_11);

        DAL_GPIO_DeInit(GPIOG, GPIO_PIN_1 | GPIO_PIN_2 | GPIO_PIN_3 | \
                               GPIO_PIN_4 | GPIO_PIN_5 | GPIO_PIN_6 | \
                               GPIO_PIN_8 | GPIO_PIN_15);

        DAL_GPIO_DeInit(GPIOH, GPIO_PIN_3 | GPIO_PIN_8 | GPIO_PIN_10 | \
                               GPIO_PIN_13 | GPIO_PIN_15);

        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_3);

        DAL_GPIO_DeInit(GPIOI, GPIO_PIN_3 | GPIO_PIN_7 | GPIO_PIN_8 | \
                               GPIO_PIN_9 | GPIO_PIN_10 | GPIO_PIN_11);
    }
}

/**
 * @brief   SDRAM configuration
 *
 * @param   None
 *
 * @retval  None
 */
void SDRAM_Config(void)
{
    DMC_SDRAM_TimingTypeDef Timing_ConfigStruct = {0};

    /* Init the SDRAM */
    hsdram1.Instance                            = DMC_SDRAM_DEVICE;
    hsdram1.Init.BankWidth                      = DMC_SDRAM_BANK_WIDTH_BITS_NUM_1;
    hsdram1.Init.ColumnBitsNumber               = DMC_SDRAM_COLUMN_BITS_NUM_8;
    hsdram1.Init.RowBitsNumber                  = DMC_SDRAM_ROW_BITS_NUM_11;
    hsdram1.Init.MemoryDataWidth                = DMC_SDRAM_MEM_BUS_WIDTH_16;
    hsdram1.Init.ClockPhase                     = DMC_SDRAM_CLK_PHASE_REVERSE;
    hsdram1.Init.RDDelay                        = DMC_SDRAM_RD_DELAY_DISABLE;
    hsdram1.Init.RDDelayClk                     = 7U;
    hsdram1.Init.WritePipe                      = DMC_SDRAM_WRITE_PIPE_ENABLE;
    hsdram1.Init.AccelerateMode                 = DMC_SDRAM_ACCELERATE_MODE_ENABLE;
    hsdram1.Init.WRAPBurstType                  = DMC_SDRAM_WRAP_BURST_INC8;
    hsdram1.Init.SelfRefreshMode                = DMC_SDRAM_SELF_REFRESH_DISABLE;
    hsdram1.Init.PowerDownMode                  = DMC_SDRAM_POWER_DOWN_DISABLE;
    hsdram1.Init.RefreshTypeEnterSelfRefresh    = DMC_SDRAM_REFRESH_TYPE_ROW_ONE;
    hsdram1.Init.RefreshTypeExitSelfRefresh     = DMC_SDRAM_REFRESH_TYPE_ROW_ONE;
    hsdram1.Init.RegisterInsertNumber           = 1U;
    hsdram1.Init.OpenBankNumber                 = 2U;

    /* Configure the SDRAM timing */
    Timing_ConfigStruct.CASLatency              = DMC_SDRAM_CAS_LATENCY_3;
    Timing_ConfigStruct.RASTime                 = 2U;
    Timing_ConfigStruct.RASToCASDelay           = 1U;
    Timing_ConfigStruct.PrechargeMode           = DMC_SDRAM_PRECHARGE_MODE_DELAY;
    Timing_ConfigStruct.PrechargePeriod         = 1U;
    Timing_ConfigStruct.AutoRefreshTime         = 10U;
    Timing_ConfigStruct.AutoRefreshNumber       = 8U;
    Timing_ConfigStruct.WriteRecoveryTime       = 2U;
    Timing_ConfigStruct.XSRTime                 = 3U;
    Timing_ConfigStruct.ActiveCommandPeriod     = 1U;
    Timing_ConfigStruct.RefreshPeriod           = 761U;
    Timing_ConfigStruct.StableTime              = 20000U;
    if (DAL_SDRAM_Init(&hsdram1, &Timing_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}

/**
 * @brief   Writes an mount of data to the SDRAM memory in polling mode
 *
 * @param   startAddr: Start address from where data is to be written
 *
 * @param   buffer: Pointer to the data buffer
 *
 * @param   dataSize: Size of data to write
 *
 * @retval  SDRAM status
 */
uint8_t SDRAM_WriteData(uint32_t startAddr, uint32_t *buffer, uint32_t dataSize)
{
    if (DAL_SDRAM_Write_32b(&hsdram1, (uint32_t *)startAddr, buffer, dataSize) != DAL_OK)
    {
        return SDRAM_ERROR;
    }
    else
    {
        return SDRAM_OK;
    }
}

/**
 * @brief   Reads an mount of data to the SDRAM memory in polling mode
 *
 * @param   startAddr: Start address from where data is to be read
 *
 * @param   buffer: Pointer to the data buffer
 *
 * @param   dataSize: Size of data to read
 *
 * @retval  SDRAM status
 */
uint8_t SDRAM_ReadData(uint32_t startAddr, uint32_t *buffer, uint32_t dataSize)
{
    if (DAL_SDRAM_Read_32b(&hsdram1, (uint32_t *)startAddr, buffer, dataSize) != DAL_OK)
    {
        return SDRAM_ERROR;
    }
    else
    {
        return SDRAM_OK;
    }
}

/**
 * @brief   Writes an mount of data to the SDRAM memory in DMA mode
 *
 * @param   startAddr: Start address from where data is to be written
 *
 * @param   buffer: Pointer to the data buffer
 *
 * @param   dataSize: Size of data to write
 *
 * @retval  SDRAM status
 */
uint8_t SDRAM_WriteDataDMA(uint32_t startAddr, uint32_t *buffer, uint32_t dataSize)
{
    if (DAL_SDRAM_Write_DMA(&hsdram1, (uint32_t *)startAddr, buffer, dataSize) != DAL_OK)
    {
        return SDRAM_ERROR;
    }
    else
    {
        return SDRAM_OK;
    }
}

/**
 * @brief   Reads an mount of data to the SDRAM memory in DMA mode
 *
 * @param   startAddr: Start address from where data is to be read
 *
 * @param   buffer: Pointer to the data buffer
 *
 * @param   dataSize: Size of data to read
 *
 * @retval  SDRAM status
 */
uint8_t SDRAM_ReadDataDMA(uint32_t startAddr, uint32_t *buffer, uint32_t dataSize)
{
    if (DAL_SDRAM_Read_DMA(&hsdram1, (uint32_t *)startAddr, buffer, dataSize) != DAL_OK)
    {
        return SDRAM_ERROR;
    }
    else
    {
        return SDRAM_OK;
    }
}
