/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr2;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR Input Capture MSP
 *
 * @param  htmr TMR Input Capture handle
 *
 * @retval None
 */
void DAL_TMR_IC_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR2)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR2_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_HIGH;

        /* TMR channel 2 */
        GPIO_InitStruct.Pin         = GPIO_PIN_1;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* TMR interrupt */
        DAL_NVIC_SetPriority(TMR2_IRQn, 1U, 0U);
        DAL_NVIC_EnableIRQ(TMR2_IRQn);
    }
}

/**
 * @brief  DeInitializes TMR Input Capture MSP
 *
 * @param  htmr TMR handle
 *
 * @retval None
 */
void DAL_TMR_IC_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR2)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR2_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_1);
    }
}

/**
 * @brief   TMR2 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR2_Config(void)
{
    TMR_IC_InitTypeDef IC_ConfigStruct = {0};

    htmr2.Instance                  = TMR2;
    htmr2.Init.Prescaler            = 0U;
    htmr2.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr2.Init.Period               = 0xFFFFFFFFU; /* Set 32 bits counter */
    htmr2.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr2.Init.RepetitionCounter    = 0U;
    htmr2.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_IC_Init(&htmr2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure input channel */
    IC_ConfigStruct.ICPolarity  = TMR_ICPOLARITY_RISING;
    IC_ConfigStruct.ICSelection = TMR_ICSELECTION_DIRECTTI;
    IC_ConfigStruct.ICPrescaler = TMR_ICPSC_DIV1;
    IC_ConfigStruct.ICFilter    = 0U;
    if (DAL_TMR_IC_ConfigChannel(&htmr2, &IC_ConfigStruct, TMR_CHANNEL_2) != DAL_OK)
    {
        Error_Handler();
    }
}
