/**
 * @file        main.c
 *
 * @brief       Include flash operation
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "bsp_flash.h"
#include "usb_host_user.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/*!
 * @brief       Unlocks Flash for write access
 *
 * @param       None
 *
 * @retval      None
 *
 */
void FLASH_FlashInit(void)
{
    /* Unlock the flash memory */
    DAL_FLASH_Unlock();

    /* Clear all FMC flags */
    __DAL_FLASH_CLEAR_FLAG(FLASH_FLAG_EOP | FLASH_FLAG_PGERR | FLASH_FLAG_WRPERR);
}

/*!
 * @brief       Get readout protection status
 *
 * @param       None
 *
 * @retval      Readout protection status
 *
 */
uint8_t FLASH_ReadOutProtectionStatus(void)
{
    uint8_t errSta = RESET;

    if (READ_BIT(FLASH->OBCS, FLASH_OBCS_READPROT) == FLASH_OBCS_READPROT)
    {
        errSta = SET;
    }

    return errSta;
}

/*!
 * @brief       Erase of all user flash area
 *
 * @param       address: application address
 *
 * @retval      flash operation status
 *
 */
uint8_t FLASH_ErasePages(uint32_t address)
{
    FMC_OP_T errSta = FMC_OP_OK;
    uint32_t pageError = 0U;
    FLASH_EraseInitTypeDef Erase_InitStruct = {0};

    /* Erase sector */
    Erase_InitStruct.TypeErase      = FLASH_TYPEERASE_PAGES;
    Erase_InitStruct.PageAddress    = address;
    Erase_InitStruct.NbPages        = (USER_FLASH_END_ADDRESS - address + 1) / FLASH_PAGE_SIZE;
    if (DAL_FLASHEx_Erase(&Erase_InitStruct, &pageError) != DAL_OK)
    {
        errSta = FMC_OP_ERROR;
    }

    return errSta;
}

/*!
 * @brief       Programs a word at a specified address
 *
 * @param       address: specifies the address to be programmed
 *
 * @param       data: specifies the data to be programmed
 *
 * @retval      flash status
 *
 */
uint8_t FLASH_ProgramWord(uint32_t address, uint32_t data)
{
    FMC_OP_T errSta = FMC_OP_OK;

    if (DAL_FLASH_Program(FLASH_TYPEPROGRAM_WORD, address, data) != DAL_OK)
    {
        errSta = FMC_OP_ERROR;
    }

    return errSta;
}
