/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2024-12-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include <string.h>
#include <stdbool.h>
#include "log.h"

/* Private macro **********************************************************/
#define Smartcard_T0_Protocol       0x00    /* T0 PROTOCOL */
#define Smartcard_Setup_Lenght      20
#define Smartcard_Hist_Lenght       20
#define Smartcard_Receive_Timeout   0x4000  /* direction to reader */

/* Private typedef ********************************************************/
/**
 * @brief    Smartcard ATR structure definition
 */
typedef  struct
{
  uint8_t TS;                           /* Bit Convention */
  uint8_t T0;                           /* High Nibble = N. of setup byte; low nibble = N. of historical byte */
  uint8_t T[Smartcard_Setup_Lenght];    /* Setup array */
  uint8_t H[Smartcard_Hist_Lenght];     /* Historical array */
  uint8_t Tlenght;                      /* Setup array dimension */
  uint8_t Hlenght;                      /* Historical array dimension */
} Smartcard_ATR_T;

/* Private variables ******************************************************/
volatile uint32_t tick = 0;
uint8_t SmartcardData = 0;
uint8_t receivedData = 0;
Smartcard_ATR_T  Smartcard_A2RStruct;

__IO uint8_t ReceivedBuffer[50]=
{
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

/* Private function prototypes ********************************************/
uint8_t Smartcard_decode_Answer2reset(__IO uint8_t* cardBuff);
uint32_t SmartcardTest(void);

/* External variables *****************************************************/
extern UART_HandleTypeDef huart1;
extern SMARTCARD_HandleTypeDef hsc3;
/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Output a message on Hyperterminal using printf function */
    LOG_Print("\r\nThis example provides a basic communication USART3 Smartcard mode.\r\n");
    LOG_Print(">> SystemCoreClock: %d\r\n",SystemCoreClock);

    if (SmartcardTest() == SUCCESS)
    {
        BOARD_LED_On(LED3);
        LOG_Print(">> Read Smartcard ATR response passed.\r\n");
    }
    else
    {
        BOARD_LED_Off(LED3);
        LOG_Print(">> Read Smartcard ATR response failed.\r\n");
    }

    /* Infinite loop */
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500);
    }
}

/* Private functions ******************************************************/
/*!
 * @brief       Test the inserted card is ISO7816-3 T=0
 *
 * @param       None
 *
 * @retval      None
 *
 * @note
 */
uint32_t SmartcardTest(void)
{
    uint32_t Counter = 0;

    /* Set Smartcard Reset Pin to Low */
    DAL_GPIO_WritePin(GPIOB, GPIO_PIN_11, GPIO_PIN_RESET);

    /* Set Smartcard Reset Pin to High */
    DAL_GPIO_WritePin(GPIOB, GPIO_PIN_11, GPIO_PIN_SET);

    /* Read Smartcard ATR response */
    for (uint16_t i = 0; i < 40; i++, Counter = 0)
    {
        DAL_SMARTCARD_Receive(&hsc3, &SmartcardData, 1, 0x4000);
        ReceivedBuffer[i] = SmartcardData;
    }

    Counter = 0;

    for (uint16_t i = 0; i < 40; i++)
    {
        if ( ReceivedBuffer[i] == 0)
        {
            Counter ++;
        }
    }

    if (Counter == 40)
    {
        /* The data received by the Smartcard is abnormal */
        return ERROR;
    }

    /* Decode ATR, Test if the inserted card is ISO7816-3 T=0 compatible */
    if (Smartcard_decode_Answer2reset(ReceivedBuffer) == 0)
    {
        /* Inserted card is ISO7816-3 T=0 compatible */
        return SUCCESS;
    }
    else
    {
        /* Inserted Smartcard is not ISO7816-3 T=0 compatible */
        return ERROR;
    }
}

/*!
 * @brief       Decode the Card ATR Response.
 *
 * @param       card: pointer to the buffer containing the Card ATR.
 *
 * @retval      Card protocol
 *
 */
uint8_t Smartcard_decode_Answer2reset(__IO uint8_t* cardBuff)
{
    uint32_t i = 0, flag = 0, buf = 0, protocol = 0;

    Smartcard_A2RStruct.TS = 0;
    Smartcard_A2RStruct.T0 = 0;

    for (i = 0; i < Smartcard_Setup_Lenght; i++)
    {
        Smartcard_A2RStruct.T[i] = 0;
    }

    for (i = 0; i < Smartcard_Hist_Lenght; i++)
    {
        Smartcard_A2RStruct.H[i] = 0;
    }

    Smartcard_A2RStruct.Tlenght = 0;
    Smartcard_A2RStruct.Hlenght = 0;

    Smartcard_A2RStruct.TS = cardBuff[0]; /* INITIAL CHARACTER */
    Smartcard_A2RStruct.T0 = cardBuff[1]; /* FORMAT CHARACTER */

    Smartcard_A2RStruct.Hlenght = Smartcard_A2RStruct.T0 & 0x0F;

    if ((Smartcard_A2RStruct.T0 & 0x80) == 0x80)
    {
        flag = 1;
    }

    for (i = 0; i < 4; i++)
    {
        Smartcard_A2RStruct.Tlenght = Smartcard_A2RStruct.Tlenght + \
                                      (((Smartcard_A2RStruct.T0 & 0xF0) >> (4 + i)) & 0x1);
    }

    for (i = 0; i < Smartcard_A2RStruct.Tlenght; i++)
    {
        Smartcard_A2RStruct.T[i] = cardBuff[i + 2];
    }

    protocol = Smartcard_A2RStruct.T[Smartcard_A2RStruct.Tlenght - 1] & 0x0F;

    while (flag)
    {
        if ((Smartcard_A2RStruct.T[Smartcard_A2RStruct.Tlenght-1] & 0x80)== 0x80)
        {
            flag = 1;
        }
        else
        {
            flag = 0;
        }

        buf = Smartcard_A2RStruct.Tlenght;
        Smartcard_A2RStruct.Tlenght = 0;

        for (i = 0; i < 4; i++)
        {
            Smartcard_A2RStruct.Tlenght = Smartcard_A2RStruct.Tlenght + \
                                          (((Smartcard_A2RStruct.T[buf - 1] & 0xF0) >> (4 + i)) & 0x1);
        }

        for (i = 0; i < Smartcard_A2RStruct.Tlenght; i++)
        {
            Smartcard_A2RStruct.T[buf + i] = cardBuff[i + 2 + buf];
        }

        Smartcard_A2RStruct.Tlenght += buf;
    }

    for (i = 0; i < Smartcard_A2RStruct.Hlenght; i++)
    {
        Smartcard_A2RStruct.H[i] = cardBuff[i + 2 + Smartcard_A2RStruct.Tlenght];
    }

    return ((uint8_t)protocol);
}


/*!
 * @brief       This function handles Smartcard USART global interrupt request.
 *
 * @param       None
 *
 * @retval      None
 *
 */
void DAL_SMARTCARD_ErrorCallback(SMARTCARD_HandleTypeDef *hsc)
{
    if (hsc->Instance == USART3)
    {
        if (DAL_SMARTCARD_GetError(hsc) & DAL_SMARTCARD_ERROR_PE)
        {
            DAL_SMARTCARD_Receive_IT(hsc, &receivedData, 1); 
        }
    }
}

/*!
 * @brief       This function handles Smartcard USART global interrupt request.
 *
 * @param       None
 *
 * @retval      None
 *
 */
void DAL_SMARTCARD_RxCpltCallback(SMARTCARD_HandleTypeDef *hsc)
{
    if (hsc->Instance == USART3)
    {
        __DAL_SMARTCARD_DISABLE_IT(hsc, SMARTCARD_IT_RXNE);
    }
}
