/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr8;
DMA_HandleTypeDef hdma_tmr8;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR PWM MSP
 *
 * @param  htmr TMR PWM handle
 *
 * @retval None
 */
void DAL_TMR_PWM_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR8)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR8_CLK_ENABLE();

        /* Enable DMA clock */
        __DAL_RCM_DMA2_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();
        __DAL_RCM_GPIOC_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FAST;

        /* TMR channel 1  : PC6
           TMR channel 1N : PA7
        */
        GPIO_InitStruct.Pin         = GPIO_PIN_6;
        DAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_7;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* Configure DMA */
        hdma_tmr8.Instance                  = DMA2_Channel3;
        hdma_tmr8.Init.Direction            = DMA_MEMORY_TO_PERIPH;
        hdma_tmr8.Init.PeriphInc            = DMA_PINC_DISABLE;
        hdma_tmr8.Init.MemInc               = DMA_MINC_ENABLE;
        hdma_tmr8.Init.PeriphDataAlignment  = DMA_PDATAALIGN_WORD;
        hdma_tmr8.Init.MemDataAlignment     = DMA_MDATAALIGN_WORD;
        hdma_tmr8.Init.Mode                 = DMA_CIRCULAR;
        hdma_tmr8.Init.Priority             = DMA_PRIORITY_HIGH;

        __DAL_LINKDMA(htmr, hdma[TMR_DMA_ID_CC1], hdma_tmr8);

        /* DMA with TMR update */
        if (DAL_DMA_Init(htmr->hdma[TMR_DMA_ID_CC1]) != DAL_OK)
        {
            Error_Handler();
        }

        DAL_NVIC_SetPriority(DMA2_Channel2_IRQn, 1U, 0U);
        DAL_NVIC_EnableIRQ(DMA2_Channel2_IRQn);
    }
}

/**
 * @brief  DeInitializes TMR PWM MSP
 *
 * @param  htmr TMR PWM handle
 *
 * @retval None
 */
void DAL_TMR_PWM_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR8)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR8_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_7);
        DAL_GPIO_DeInit(GPIOC, GPIO_PIN_6);

        /* De-init peripheral DMA */
        DAL_DMA_DeInit(htmr->hdma[TMR_DMA_ID_CC1]);
    }
}

/**
 * @brief   TMR8 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR8_Config(void)
{
    TMR_OC_InitTypeDef OC_ConfigStruct = {0};

    htmr8.Instance                  = TMR8;
    htmr8.Init.Prescaler            = 119U;
    htmr8.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr8.Init.Period               = 999U;
    htmr8.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr8.Init.RepetitionCounter    = 6U;
    htmr8.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_PWM_Init(&htmr8) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure output channel 1 */
    OC_ConfigStruct.OCMode          = TMR_OCMODE_PWM1;
    OC_ConfigStruct.OCPolarity      = TMR_OCPOLARITY_HIGH;
    OC_ConfigStruct.OCNPolarity     = TMR_OCNPOLARITY_HIGH;
    OC_ConfigStruct.OCIdleState     = TMR_OCIDLESTATE_SET;
    OC_ConfigStruct.OCNIdleState    = TMR_OCNIDLESTATE_SET;
    OC_ConfigStruct.OCFastMode      = TMR_OCFAST_DISABLE;
    OC_ConfigStruct.Pulse           = 499U;
    if (DAL_TMR_PWM_ConfigChannel(&htmr8, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }
}
