/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/
#define TMR2_PERIOD     1000U

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr2;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR Output Compare MSP
 *
 * @param  htmr TMR Output Compare handle
 *
 * @retval None
 */
void DAL_TMR_OC_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR2)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR2_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_HIGH;

        /* TMR channel 1 */
        GPIO_InitStruct.Pin         = GPIO_PIN_0;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* TMR channel 2 */
        GPIO_InitStruct.Pin         = GPIO_PIN_1;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* TMR channel 3 */
        GPIO_InitStruct.Pin         = GPIO_PIN_2;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* TMR channel 4 */
        GPIO_InitStruct.Pin         = GPIO_PIN_3;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }
}

/**
 * @brief  DeInitializes TMR Output Compare MSP
 *
 * @param  htmr TMR Output Compare handle
 *
 * @retval None
 */
void DAL_TMR_OC_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR2)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR2_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_0 | GPIO_PIN_1 | \
                               GPIO_PIN_2 | GPIO_PIN_3);
    }
}

/**
 * @brief   TMR2 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR2_Config(void)
{
    TMR_OC_InitTypeDef OC_ConfigStruct = {0};

    htmr2.Instance                  = TMR2;
    htmr2.Init.Prescaler            = 119U;
    htmr2.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr2.Init.Period               = TMR2_PERIOD - 1U;
    htmr2.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr2.Init.RepetitionCounter    = 0U;
    htmr2.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_OC_Init(&htmr2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure channel */
    OC_ConfigStruct.OCMode          = TMR_OCMODE_ACTIVE;
    OC_ConfigStruct.OCPolarity      = TMR_OCPOLARITY_HIGH;

    /* Set the pulse value for channel 1, 800 / 1MHz = 800us */
    OC_ConfigStruct.Pulse           = 800U;
    if (DAL_TMR_OC_ConfigChannel(&htmr2, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Set the pulse value for channel 2, 600 / 1MHz = 600us */
    OC_ConfigStruct.Pulse           = 600U;
    if (DAL_TMR_OC_ConfigChannel(&htmr2, &OC_ConfigStruct, TMR_CHANNEL_2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Set the pulse value for channel 3, 400/1MHz = 400us */
    OC_ConfigStruct.Pulse           = 400U;
    if (DAL_TMR_OC_ConfigChannel(&htmr2, &OC_ConfigStruct, TMR_CHANNEL_3) != DAL_OK)
    {
        Error_Handler();
    }

    /* Set the pulse value for channel 4, 200/1MHz = 200us */
    OC_ConfigStruct.Pulse           = 200U;
    if (DAL_TMR_OC_ConfigChannel(&htmr2, &OC_ConfigStruct, TMR_CHANNEL_4) != DAL_OK)
    {
        Error_Handler();
    }
}
