/**
 * @file        apm32f4xx_i2c_cfg.c
 *
 * @brief       This file provides configuration support for I2C
 *
 * @version     V1.0.1
 *
 * @date        2025-03-10
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_i2c_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
I2C_HandleTypeDef hi2c1;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initialize the I2C MSP
 *
 * @param  hi2c Pointer to a I2C_HandleTypeDef structure that contains
 *         the configuration information for the specified I2C
 *
 * @retval None
 */
void DAL_I2C_MspInit(I2C_HandleTypeDef *hi2c)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0U};

    if(hi2c->Instance == I2C1)
    {
        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOB_CLK_ENABLE();

        /* SCL pin configuration */
        GPIO_InitStruct.Pin         = GPIO_PIN_6;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_OD;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FAST;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

        /* SDA pin configuration */
        GPIO_InitStruct.Pin         = GPIO_PIN_7;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

        /* Enable peripheral clock */
        __DAL_RCM_I2C1_CLK_ENABLE();
    }
}

/**
 * @brief  DeInitialize the I2C MSP
 *
 * @param  hi2c Pointer to a I2C_HandleTypeDef structure that contains
 *         the configuration information for the specified I2C
 *
 * @retval None
 */
void DAL_I2C_MspDeInit(I2C_HandleTypeDef *hi2c)
{
    if(hi2c->Instance == I2C1)
    {
        /* Reset peripheral clock */
        __DAL_RCM_I2C1_FORCE_RESET();
        __DAL_RCM_I2C1_RELEASE_RESET();

        /* I2C GPIO configuration */
        DAL_GPIO_DeInit(GPIOB, GPIO_PIN_6 | GPIO_PIN_7);
    }
}

/**
 * @brief   I2C1 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_I2C1_Config(void)
{
    hi2c1.Instance              = I2C1;
    hi2c1.Init.ClockSpeed       = 400000U;
    hi2c1.Init.DutyCycle        = I2C_DUTYCYCLE_16_9;
    hi2c1.Init.OwnAddress1      = I2C_SALVE_ADDRESS;
    hi2c1.Init.OwnAddress2      = 0xFEU;
    hi2c1.Init.AddressingMode   = I2C_ADDRESSINGMODE_10BIT;
    hi2c1.Init.DualAddressMode  = I2C_DUALADDRESS_DISABLE;
    hi2c1.Init.GeneralCallMode  = I2C_GENERALCALL_DISABLE;
    hi2c1.Init.NoStretchMode    = I2C_NOSTRETCH_DISABLE;
    if (DAL_I2C_Init(&hi2c1) != DAL_OK)
    {
        Error_Handler();
    }
}
