/**
 * @file        apm32f4xx_rtc_cfg.c
 *
 * @brief       This file provides configuration support for RTC
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_rtc_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
RTC_HandleTypeDef hrtc;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
  * @brief  Initializes the RTC MSP.
  *
  * @param  hrtc pointer to a RTC_HandleTypeDef structure that contains
  *                the configuration information for RTC.
  *
  * @retval None
  */
void DAL_RTC_MspInit(RTC_HandleTypeDef *hrtc)
{
    if (hrtc->Instance == RTC)
    {
        DAL_PMU_EnableBkUpAccess();

        /* Enable backup register clock */
        __DAL_RCM_BAKP_CLK_ENABLE();

        /* Enable peripheral clock */
        __DAL_RCM_RTC_ENABLE();

        /* Enable RTC interrupt */
        DAL_NVIC_SetPriority(TAMPER_IRQn, 0x0F, 0);
        DAL_NVIC_EnableIRQ(TAMPER_IRQn);
    }
}

/**
  * @brief  DeInitializes the RTC MSP.
  *
  * @param  hrtc pointer to a RTC_HandleTypeDef structure that contains
  *                the configuration information for RTC.
  *
  * @retval None
  */
void DAL_RTC_MspDeInit(RTC_HandleTypeDef *hrtc)
{
    if (hrtc->Instance == RTC)
    {
        /* Disable peripheral clock */
        __DAL_RCM_RTC_DISABLE();

        /* Disable RTC interrupt */
        DAL_NVIC_DisableIRQ(TAMPER_IRQn);
    }
}

/**
 * @brief   RTC configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_RTC_Config(void)
{
    RTC_TamperTypeDef TamperStruct = {0};

    /* Configure the RTC */
    hrtc.Instance = RTC;
    hrtc.Init.AsynchPrediv = RTC_AUTO_1_SECOND;
    hrtc.Init.OutPut = RTC_OUTPUTSOURCE_NONE;
    if (DAL_RTC_Init(&hrtc) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure the RTC tamper */
    TamperStruct.Tamper = RTC_TAMPER_1;
    TamperStruct.Trigger = RTC_TAMPERTRIGGER_LOWLEVEL;
    if (DAL_RTCEx_SetTamper_IT(&hrtc, &TamperStruct) != DAL_OK)
    {
        Error_Handler();
    }

    /* Clear the Tamper interrupt pending bit */
    __DAL_RTC_TAMPER_CLEAR_FLAG(&hrtc,RTC_FLAG_TAMP1F);
}
