/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";

__IO uint16_t adc1ConvValue = 0U;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/
extern ADC_HandleTypeDef hadc1;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Run the ADC calibration */
    if (DAL_ADCEx_Calibration_Start(&hadc1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Strat ADC conversation */
    if (DAL_ADC_Start_IT(&hadc1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Infinite loop */
    while (1)
    {
    }
}

/**
 * @brief  Regular conversion complete callback in non blocking mode
 *
 * @param  hadc pointer to a ADC_HandleTypeDef structure that contains
 *         the configuration information for the specified ADC
 *
 * @retval None
 */
void DAL_ADC_ConvCpltCallback(ADC_HandleTypeDef* hadc)
{
    float voltage = 0.0;
    float temperature = 0.0;
    
    /* Get the converted value */
    adc1ConvValue = DAL_ADC_GetValue(hadc);

    voltage = (float)adc1ConvValue * 3.3f / 4095U;

    /**
     * According to actual test data of multiple bathes of chips,
     * V28 is adopted instead of V25 for this example. And 1.4822v is the voltage for 28 degrees celsius
     */
    temperature = (voltage - 1.4822f) / 0.0024f + 28U;
    
    LOGI(tag, "Temperature Value: %.3f degrees celsius\r\n",temperature);

    /* Strat ADC Conversation */
    if (DAL_ADC_Start_IT(&hadc1) != DAL_OK)
    {
        Error_Handler();
    }
}
