/*!
 * @file        main.c
 *
 * @brief       Main program
 *
 * @version     V1.0.0
 *
 * @date        2026-01-31
 *
 * @attention
 *
 *  Copyright (C) 2026 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes */
#include "user_config.h"
#include "board.h"
#include <stdio.h>

/** @addtogroup APM32F445_Examples
  @{
  */

/** @addtogroup PDU_PeriodicInterrupt
  @{
  */

/** @defgroup PDU_PeriodicInterrupt_Macros Macros
  @{
*/

/* PDU instance */
#define PDU0_INSTANCE           (0U)

/* PDU timeout, measured in microseconds */
#define PDU_TIMEOUT_US          (500000UL)


/**@} end of group PDU_PeriodicInterrupt_Macros*/

/** @defgroup PDU_PeriodicInterrupt_Functions Functions
  @{
  */

bool PDU_CalculateIntTimerValue(
    const PDU_TIMER_CONFIG_T *const pduTimerConfig,
    const uint32_t us,
    uint16_t *const intTimerValue);

/*!
 * @brief   Main function
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    uint16_t pduIntTimerValue;

    /* Initialize clock */
    CLOCK_SYS_Init(&g_clockConfig);

    /* Initialize pins */
    PINS_Init(NUM_OF_CONFIGURED_PINS0, g_pinsConfig);

    /* Initialize LEDs */
    LED_Init();
    LED_On(LED_GREEN);

    /* Initialize Buttons */
    BTN_Init();

    /* Initialize UART */
    COM_Init();


    PDU_Init(PDU0_INSTANCE, &g_pduTimerConfig);

    /* Calculate the required value of the PDU timer counter for the specified timeout */
    if (!PDU_CalculateIntTimerValue(&g_pduTimerConfig, PDU_TIMEOUT_US, &pduIntTimerValue))
    {
        goto end;
    }

    /* Set the period of the counter */
    PDU_ConfigPeriodCountValue(PDU0_INSTANCE, pduIntTimerValue);

    /* Set the delay value to schedule the PDU interrupt */
    PDU_ConfigTimerIntDelayValue(PDU0_INSTANCE, pduIntTimerValue);

    /* Enable PDU */
    PDU_Enable(PDU0_INSTANCE);

    /* Executes the command of loading values */
    PDU_LoadValuesCmd(PDU0_INSTANCE);

    /* Software trigger the pdu counter */
    PDU_EnableSoftTrigger(PDU0_INSTANCE);

    while (1)
    {
    }

end:
    PDU_DeInit(PDU0_INSTANCE);
    LED_On(LED_RED);
    return 0;
}


/*!
 * @brief Calculate the timing value of an interrupt generated by the PDU.
 *
 * @param pduTimerConfig: pointer to the PDU configuration struct
 * @param us            : pdu interrupt interval, measured in microseconds
 * @param intTimerValue : pointer to set the calculated value

 * @retval true: successful.
 *         false: fail.
 */
bool PDU_CalculateIntTimerValue(
    const PDU_TIMER_CONFIG_T *const pduTimerConfig,
    const uint32_t us,
    uint16_t *const intTimerValue)
{
    uint32_t tempIntTimerValue   = 0;
    uint8_t  clkPscDiv        = 0;
    uint8_t  clkPscMultFactor = 0;
    uint32_t pduClkFreq       = 0;
    bool     retStatus = false;

    /* Get the PDU prescaler divider */
    clkPscDiv = (1 << pduTimerConfig->clkPscDiv);

     /* Get the frequency of the PDU clock source */
    CLOCK_SYS_ReadFreq(CORE_CLK, &pduClkFreq);

    /* Measured in microseconds */
    pduClkFreq /= 1000000;

    /* Get the PDU prescaler multiplier factor */
    switch (pduTimerConfig->clkPscMultFactor)
    {
        case PDU_CLK_PSCMULT_FACTOR_AS_1:
            clkPscMultFactor    =   1U;
            break;
        case PDU_CLK_PSCMULT_FACTOR_AS_10:
            clkPscMultFactor    =   10U;
            break;
        case PDU_CLK_PSCMULT_FACTOR_AS_20:
            clkPscMultFactor    =   20U;
            break;
        case PDU_CLK_PSCMULT_FACTOR_AS_40:
            clkPscMultFactor    =   40U;
            break;
        default:
            clkPscMultFactor    =   1U;
            break;
    }

    /* Calculate the PDU timer interrupt value */
    tempIntTimerValue = (pduClkFreq * us) / (clkPscDiv * clkPscMultFactor);

    /* 0 < tempIntTimerValue <= 0xFFFFU */
    if((tempIntTimerValue > 0) && (tempIntTimerValue <= 0xFFFFU))
    {
        retStatus = true;
        (*intTimerValue) = (uint16_t)tempIntTimerValue;
    }
    else
    {
        retStatus = false;
        (*intTimerValue) = 0U;
    }

    return retStatus;
}

/*!
 * @brief   PDU IRQ handler.
 *
 * @param   None
 *
 * @retval  None
 */
void PDU0_IRQHandler(void)
{
    if (PDU_ReadTimerIntFlag(PDU0_INSTANCE) != false)
    {
        /* Clear PDU Interrupt flag */
        PDU_ClearTimerIntFlag(PDU0_INSTANCE);

        /* Toggle green LED */
        LED_Toggle(LED_GREEN);

        printf("PDU timing 500ms.\r\n");
    }
}

/**@} end of group PDU_PeriodicInterrupt_Functions */
/**@} end of group PDU_PeriodicInterrupt */
/**@} end of group Examples */
